/*------------------------------------------------
    file memory.c

 Wiener filter restoration

This file does the basic stuff

memory allocation
file i/o
header updating

----------------------------------------------------*/
#include  <stdio.h>
#include  <stdlib.h>
#include  <string.h>
#include  <math.h>
#include "memory.h"


static int bVerbose = 1 ;

void *memGet ( int iSize ) ;
void memFree ( void *pAddr ) ;

/*-------------------------------------------------------
  alloc1di       allocate 1-dim array of integers
  -------------------------------------------------------*/
int *alloc1di ( int iDim )
{
  int *pdata ;

  pdata = (int *)memGet ( iDim * sizeof ( int ) ) ;
  if ( pdata == NULL )
    {
      printf ("alloc1di: memory allocation problem!"  ) ;
      exit (1) ;
    }

  return ( pdata ) ;
}

void free1di ( int *pdata )
{
  memFree ( (void *)pdata ) ;
}


/*-------------------------------------------------------
  alloc1df       allocate 1-dim array of Float numbers
  -------------------------------------------------------*/
float *alloc1df ( int iDim )
{
  float *pdata ;

  pdata = (float *)memGet ( iDim * sizeof ( float ) ) ;
  if ( pdata == NULL )
    {
      printf ("alloc1df: memory allocation problem!"  ) ;
      exit (1) ;
    }
  return ( pdata ) ;
}

void free1df ( float *pdata )
{
  memFree ( (void *)pdata ) ;
}

/*-------------------------------------------------------
  alloc1dd       allocate 1-dim array of doubles
  -------------------------------------------------------*/
double *alloc1dd ( int iDim )
{
  double *pdata ;

  pdata = (double *)memGet ( iDim * sizeof ( double ) ) ;
  if ( pdata == NULL )
    {
      printf ("alloc1dd: memory allocation problem!"  ) ;
      exit (1) ;
    }
  return ( pdata ) ;
}

void free1dd ( double *pdata )
{
  memFree ( (void *)pdata ) ;
}


/*-------------------------------------------------------
  alloc2di       allocate 2-dim array of int numbers

  value[i][j]  where 0 <= i <= iRow-1
                     0 <= j <= iCol-1
  -------------------------------------------------------*/
int **alloc2di ( int iRow , int iCol )
{
  int i ;
  int **pdata ;

  /* Allocate pointer to rows */

  pdata = (int **) memGet ( iRow * sizeof ( int * ) ) ;
  if ( pdata == NULL )
    {
      printf ( "alloc2di: memory allocation problem 1\n" ) ;
      exit (1) ;
    }

  /* Allocate rows and set pointer to them. */

  pdata[0] = (int *) memGet ( iCol * iRow * sizeof ( int ) ) ;
  if ( pdata[0] == NULL )
    {
      printf ( "alloc2df: memory allocation problem! 2\n"  ) ;
      exit (1) ;
    }

  for ( i = 1 ; i < iRow ; i++ )
    pdata[i] = pdata[0] + i*iCol ;

  return ( pdata ) ;
}

/*-------------------------------------------------------
  alloc2df       allocate 2-dim array of Float numbers

  value[i][j]  where 0 <= i <= iRow-1
                     0 <= j <= iCol-1
  -------------------------------------------------------*/
float **alloc2df ( int iRow , int iCol )
{
  int i ;
  float **pdata ;

  /* Allocate pointer to rows */

  pdata = (float **) memGet ( iRow * sizeof ( float * ) ) ;
  if ( pdata == NULL )
    {
      printf ( "alloc2df: memory allocation problem 1\n" ) ;
      exit (1) ;
    }

  /* Allocate rows and set pointer to them. */

  pdata[0] = (float *) memGet ( iCol * iRow * sizeof ( float ) ) ;
  if ( pdata[0] == NULL )
    {
      printf ( "alloc2df: memory allocation problem! 2\n"  ) ;
      exit (1) ;
    }

  for ( i = 1 ; i < iRow ; i++ )
    pdata[i] = pdata[0] + i*iCol ;

  return ( pdata ) ;
}

/*-------------------------------------------------------
  alloc3df       allocate 3-dim array of Float numbers

  value[i][j][k]  where 0 <= i <= i1-1
                        0 <= j <= i2-1
                        0 <= k <= i3-1
  -------------------------------------------------------*/
float ***alloc3df ( int i1 , int i2, int i3 )
{
  int i ;
  float ***pdata ;

  /* Allocate pointer to Frames */

  pdata = (float ***) memGet ( i1 * sizeof ( float **) ) ;
  if ( pdata == NULL )
    {
      printf ( "alloc3df: memory allocation problem 1\n" ) ;
      exit (1) ;
    }

  for ( i = 0 ; i < i1 ; i++ )
    pdata[i] = alloc2df ( i2, i3 ) ;

  return ( pdata ) ;
}

float ***alloc3df_old_And_Correct_but_not_simple ( int i1 , int i2, int i3 )
{
  int i, j ;
  float ***pdata ;

  /* Allocate pointer to Frames */

  pdata = (float ***) memGet ( i1 * sizeof ( float **) ) ;
  if ( pdata == NULL )
    {
      printf ( "alloc3df: memory allocation problem 1\n" ) ;
      exit (1) ;
    }

  /* Allocate pointer to rows */

  pdata[0] = (float **) memGet ( i1 * i2 * sizeof ( float *) ) ;
  if ( pdata == NULL )
    {
      printf ( "alloc3df: memory allocation problem 1\n" ) ;
      exit (1) ;
    }

  /* Allocate pointer to data points */

  pdata[0][0] = (float *) memGet ( i1 * i2 * i3 * sizeof (float)) ;
  if ( pdata == NULL )
    {
      printf ( "alloc3df: memory allocation problem 1\n" ) ;
      exit (1) ;
    }

  /* Initialize the "intermediate" pointers */

  for ( i = 0 ; i < i1 ; i++ )
    {
      pdata[i] = pdata[0] + i*i2 ;
      for ( j = 0 ; j < i2 ; j++ )
	pdata[i][j] = pdata[0][0] + ( i*i2 + j) * i3 ;
    }
  
  return ( pdata ) ;
}

void free2di ( int **pdata )
{
  memFree ( (void *)pdata[0] ) ;
  memFree ( (void *)pdata ) ;
}

void free2df ( float **pdata )
{
  memFree ( (void *)pdata[0] ) ;
  memFree ( (void *)pdata ) ;
}

void free3df ( float ***pdata )
{
  memFree ( (void *)pdata[0][0] ) ;
  memFree ( (void *)pdata[0] ) ;
  memFree ( (void *)pdata ) ;
}

/*-------------------------------------------------------
  alloc2dd       allocate 2-dim array of double floats

  value[i][j]  where 0 <= i <= iRow-1
                     0 <= j <= iCol-1
  -------------------------------------------------------*/
double **alloc2dd ( int iRow , int iCol )
{
  int i ;
  double **pdata ;

  /* Allocate pointer to rows */

  pdata = (double **) memGet ( iRow * sizeof ( double * ) ) ;
  if ( pdata == NULL )
    {
      printf ( "alloc2dd: memory allocation problem 1\n" ) ;
      exit (1) ;
    }

  /* Allocate rows and set pointer to them. */

  pdata[0] = (double *) memGet ( iCol * iRow * sizeof ( double ) ) ;
  if ( pdata[0] == NULL )
    {
      printf ( "alloc2dd: memory allocation problem!"  ) ;
      exit (1) ;
    }

  for ( i = 1 ; i < iRow ; i++ )
    pdata[i] = pdata[0] + i*iCol ;

  return ( pdata ) ;
}

void free2dd ( double **pdata )
{
  memFree ( (void *)pdata[0] ) ;
  memFree ( (void *)pdata ) ;
}

/*-----------------------------------------------------
  memGet

  wrapper for malloc to provide debugging information
  -----------------------------------------------------*/
void *memGet ( int iSize )
{
  void *pAddr ;

  pAddr = malloc ( iSize ) ;
  if ( pAddr == NULL )
    {
      printf ( "memGet: memory for %d bytes not available\n", 
	       iSize  ) ;
      exit (1) ;
    }
  if ( bVerbose )
    printf ( "memGet:  %12d bytes at addr = %12o\n", iSize, (unsigned int)pAddr ) ;

  return ( pAddr ) ;
}

/*-----------------------------------------------------
  memFree

  wrapper for free to provide debugging information
  -----------------------------------------------------*/
void memFree ( void *pAddr )
{

  if (bVerbose) 
    printf ( "memFree: free memory at addr = %12o\n", (unsigned int)pAddr ) ;

  free ( pAddr ) ;
}


/*------------------------------------------------
  memory allocation functions for compatability
  with "Numerical Recipes in C"

  ------------------------------------------------*/

int *ivector ( int nl, int nh ) 
{
  int *v ;
  v = (int *)memGet( (nh-nl+2)*sizeof(int) ) ;
  if ( v==NULL )
      nrerror("allocation failure in ivector()\n" ) ;
  return ( v-nl+1 ) ;
}

void free_ivector ( int *v, int nl, int nh )
{
  memFree ( (void *)(v+nl-1) ) ;
}





