/*
  npoi.cpp

  The is the main program for simulating NPOI performance.

  invoked classes

  atmosphere: wrapper for the atmosphere simulation.  Takes time in
  seconds, returns atmospheric delay in micrometers.  One delay value
  for each station.

  combiner

  fdl


  sineWave is a test class that returns the delay associated with sine wave.

  The combiner class returns the intensities for the integration
  ending at the specified time.

  The fdl class implemented one delay line.

  ================================================================*/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "memory.h"
#include "config.hpp"
#include "../atmos/npoi.h"
#include "fdl.hpp"
#include "atmosphere.hpp"
#include "sineWave.hpp"
#include "combiner.hpp"
#include "fringeMath.hpp"
#include "fsm.hpp"

#include "display.hpp"
#include "fringeSpectrum.hpp"

static int bVerbose = 0 ;

/** ============================================================
    main

    ============================================================*/
int main ( void )
{
  int bTest = 0 ;       /* test mode.  This does atmosphere phase power spectra */
  config  *pConfig ;    /* configuration class */
  atmosphere *pAtm ;    /* atmosphere model */
  fdl **pFDL ;          /* array of fdl's */
  fsm **pFSM ;          /* fast steering mirrors */
  combiner *pBC ;       /* beam combiner model */
  fringeMath *pMath ;   /* Math functions for fringe detection */
  display *pDisplay[4]; /* graphical display functions */
  fringeSpectrum *pFS ; /* fringe spectrum class */
  int **piData ;        /* storage for raw fringe data */
  float fStroke ;       /* unit for stroke amplitudes (microns) */
  float fFDLTimeDelay = 0.004 ;  /* Time delay for fdl servo */

  int i, iWaves, iChan, iStations, iApPoints, iBins, iCnt ;
  int *piBSign, *piBFreq, *piQuad, *piModAmp ;
  float *pfDelay, *pfPiston, *pfFDL, *pfTip, *pfTilt ;
  double dfTime, dfStartTime ; 
  float fApDia, *pfStnX, *pfStnY, *pfApX, *pfApY ;
  float fDeltaTime = 0.002 ;
  float *pfDeltaV, *pfSNR, fCounts ;
  float *pfWave ;
  float *pfFreq=NULL, *pfGDAmp=NULL ;
  FILE *pFFDL, *pFbins, *pFTracker ;

  printf ( "npoi: begins\n" ) ;

  /*====================================================================
    Part I.  Configuration

    All the configuration variables

    The numbers presented here determine how the simulation works.  At
    some point they should be moved into configuration files.

    Changing the numbers in this section is all you should have to do
    to change the performance of the simulation.
    ====================================================================*/

  /* options are NPOI or TEST */

  pConfig = new config ( TEST ) ;

  iBins  = pConfig->getBins()  ;  /* number of bins per stroke */
  iWaves = pConfig->getWaves() ;  /* number of wavelength in spectrograph */

  /* The number of stations and X and Y offsets for each station from
     the array center.  Distances are in meters. */

  iStations = pConfig->getStations() ;
  pfStnX = alloc1df ( iStations ) ;
  pfStnY = alloc1df ( iStations ) ;

  pConfig->getStationList ( pfStnX, pfStnY ) ;

  /* The aperture diameters */

  fApDia  = pConfig->getApSize() ;

  /* The number of points at which the wavefront is sampled on each
     aperture and the offsets for those points in units of the
     aperture diameter.  A value of 0.5 is at the edge of the aperture */

  iApPoints = pConfig->getNumApPts() ; /* Number of points in each aperture's wavefront */
  pfApX = alloc1df ( iApPoints ) ;
  pfApY = alloc1df ( iApPoints ) ;
  pConfig->getApPtList ( pfApX, pfApY ) ;

  /* The length of the delay modualtion for each delay line.  The
     stroke is measured in integral units of fStroke, where fStroke is
     in microns.  One value for each FDL = one value for each station.  */

  fStroke   = pConfig->getStroke() ;
  piModAmp = alloc1di ( iStations ) ;
  pConfig->getModAmps( piModAmp ) ;

  /* The mapping of the delay lines into the beam combiner inputs.
     The order of the inputs does not matter in this implementation,
     but which delay lines contribute does matter.  For example, in
     the NPOI combiner there are three implementations of the
     combiner, one for each spectrograph.  Each of the combiners takes
     light from four of the six delay lines.  */

  piQuad   = alloc1di ( iStations ) ;
  i = 0 ;
  while ( i < iStations )
    piQuad[i] = i++ ;

  printf ( "part 1 ends:  Configuraton  complete\n" ) ;


  /*==============================================
    Part Ia.  Write configuration files for the atmosphere simulator.

    This section forces agreement between this code and the atmosphere
    simulator.

    It has also been moved to the atmosphere class where it belongs.
  ==============================================*/
  /*==============================================
    Part II.  Initialization

    all the configuration variables
    allocate memory
    initialize all variables

    iStations and iApPoints must be consistent
    with the data in files geometry and aperture.

    ==============================================*/
  /* open files to store results */

  pFFDL     = fopen ( "FDLData.dat",     "w" ) ;
  pFbins     = fopen ( "FringeData.dat", "w" ) ;
  pFTracker = fopen ( "TrackerData.dat", "w" ) ;
  if (( pFFDL == NULL )||( pFbins == NULL )||( pFTracker == NULL ))
    {
      printf ( " cannot open output files\n" ) ;
      exit(1) ;
    }
  else
    printf ( "output files opened sucessfully\n" ) ;

  pfWave   = alloc1df ( iWaves ) ;
  piBSign  = alloc1di ( iStations ) ;
  piBFreq  = alloc1di ( iStations ) ;
  pfSNR    = alloc1df ( iStations ) ;
  pfDelay  = alloc1df ( iStations ) ;
  pfDeltaV = alloc1df ( iStations ) ;
  pfTip    = alloc1df ( iStations ) ;
  pfTilt   = alloc1df ( iStations ) ;

  /* Set the start time to be large enough so that no requested time
     ever be negative. */

  dfStartTime = 0.1 ;

  /* create the necessary instances of each class.  */

  pAtm  = new atmosphere( pConfig ) ;

  /* test code */

  if ( bTest ) 
    {
      pAtm->testAtmos() ;
      exit(1) ;
    }

  pFDL = (fdl **)malloc( iStations * sizeof( fdl *) ) ;
  pFSM = (fsm **)malloc( iStations * sizeof( fsm *) ) ;
  if (( pFDL == NULL )||( pFSM == NULL ))
    {
      printf ( " cannot allocated memory for pFDL or pFSM\n" ) ;
      exit(1) ;
    }

  /* Set the length of each stroke, in wavelengths.  Initialize all
     the delay lines to be only a few microns off the fringe at the
     start of the process.  This is so I don't have to work out the
     fringe search algorithms yet.  */

  dfTime = dfStartTime ;
  for ( i = 0 ; i < iStations ; i++ )
    {
      if ( bVerbose )
	printf ( "initializing FDL%2d\n", i ) ;
      pFDL[i]  = new fdl(piModAmp[i]*fStroke, 
		  	 pAtm->piston( piQuad[i], dfTime )+float(2*i), fFDLTimeDelay ) ;
      pFSM[i]  = new fsm(0.0) ;
    }
  if ( bVerbose )
    printf ( "initializing combiner\n" ) ;
  pBC = new combiner (  pConfig, pFDL, pAtm, piQuad ) ;
  pConfig->getWavelengths ( pfWave ) ;

  pMath = new fringeMath ( iBins, 6, iWaves, pfWave ) ;

  /* set storage space for the data */

  piData = alloc2di( iWaves, iBins ) ;

  /* Initial positions for the servos */

  pfPiston = alloc1df ( iStations ) ;
  pfFDL    = alloc1df ( iStations ) ;
  for ( i = 0 ; i < iStations ; i++ )
    {
      pfPiston[i] = 0. ;
      pfFDL[i] = 0. ;
    }

  /* Set up the mapping from the fringe frequencies to the FDLs that
     have to be servoed.  The fringe tracking servo keeps FDL[0] at a
     constant position.  The other FDLs are controlled using only a
     single baseline.  The direction of a baseline is defined as
     running from a smaller modulation amplitude to a larger
     modulation amplitude.  If we reverse the direction of the
     baseline, we have to negate the delay.  */

  for ( i = 1 ; i < iStations ; i++ )
    {
      piBSign[i] = -1 ;
      piBFreq[i]  = piModAmp[i] - piModAmp[0] ;
      if ( piBFreq[i] < 0 )
	{
	  piBSign[i] *= -1 ;
	  piBFreq[i]  *= -1 ;
	}
    }

  /* Set up the graphical displays  */

  pDisplay[0] = new display(GRAY_SCALE, iWaves, 64) ;
  pDisplay[1] = new display(CHART, 5, 0.0, 1.0, -3., 3. ) ;
  pDisplay[2] = new display( XY ) ;
  pDisplay[3] = new display(CHART, 6, 0.0, 1.0, -10., 10. ) ;

  /* Set up the plot titles */

  pDisplay[0]->setTitles( "wavelength channel", "Bin count", "Frame" ) ;
  pDisplay[1]->setTitles( "Time (seconds)", "Delay (microns)", "Delay Lines" ) ;
  pDisplay[2]->setTitles( "Delay(microns)", "Power", "Group Delay Tracking Signal" ) ;
  pDisplay[3]->setTitles( "Time (seconds)", "Delay (microns)", "Atmospheric Piston" ) ;

  /* set autorange on XY plot */

  pDisplay[2]->range( 0., 0., 0., 0. ) ;


  pFS = new fringeSpectrum() ;

  printf ( "data storage space allocated\n" ) ;

  /*=================================================================*/
  /*=================================================================*/
  /*=================================================================
    Start the System running
    =================================================================*/
  /*=================================================================*/
  /*=================================================================*/

  if ( bVerbose )
    printf ( " npoi:  starting simulation loop over times \n" ) ;
  while ( dfTime < 10. )
    {
      /*=======================================================
	Calculation section.  Get the data available to the
	system at this time 
	=======================================================*/

      if ( bVerbose )
	printf ( "NPOI: Starting getData.  For time %6.3f\n", dfTime ) ;
      pBC->getData( (float)dfTime, piData ) ;

      /*=======================================================
	Servo Calculations.  Get the appropriate error signals
	for the hardware
	=======================================================*/

      if ( bVerbose )
	printf ( "NPOI: Starting calcXYN for time %6.3f\n", dfTime ) ;
      fCounts = pMath->calcXYN( piData ) ;

      /* run the FDL servo */

      pfDelay[0]= 0. ;
      pfDeltaV[0] = 0. ;

      for ( i = 1 ; i < iStations ; i++ )
	{
	  pfDelay[i]  = piBSign[i]*pMath->calcGD (piBFreq[i], pfSNR+i ) ;
	  if ( bVerbose )
	    printf ( "NPOI: Starting FDL servo %2d for time %6.3f\n", i, dfTime ) ;
	  pfDeltaV[i] = pFDL[i]->servo( (float)dfTime, pfDelay[i] ) ;
	}

      /*=======================================================
	Display.
	=======================================================*/

      /* Plot the bin data as a gray scale Display */

      pDisplay[0]->displayFrame( piData ) ;

      /* Plot the fringe spectrum */

      pFS->calc ( piData ) ;
      pFS->showFS () ;

      /* Plot the group delay for one frequency */

      iCnt = pMath->getSpectrum( 4, &pfFreq, &pfGDAmp ) ;
      if ( bVerbose )
	{
	  printf ( "ready to call plot function.  %d values are \n", iCnt ) ;
	  for ( i = 0 ; i < iCnt ; i++ )
	    printf ( " i = %3d  Freq = %6.3f  Amp = %9.3f\n",
		     i, pfFreq[i], pfGDAmp[i] ) ;
	}
      pDisplay[2]->plotXY( iCnt, pfFreq, pfGDAmp ) ;


      /* Write Bin Data to Disk  */

      for ( iChan = 0; iChan < iWaves ; iChan++ )
	{
	  fprintf ( pFbins,  "%6.3f %6.3f  ", dfTime, pfWave[iChan] ) ;
	  for ( i = 0 ; i < iBins ; i++ )
	    fprintf ( pFbins, "%3d", piData[iChan][i] ) ;
	  fprintf ( pFbins, "\n" ) ;
	}
      fprintf ( pFbins, "\n" ) ;
      fflush ( pFbins ) ;

      /* Plot the fringe tracking data */

      pDisplay[1]->chart ( 0, (float)dfTime, pfDelay[1] ) ;
      pDisplay[1]->chart ( 1, (float)dfTime, pfDelay[2] ) ;
      pDisplay[1]->chart ( 2, (float)dfTime, pfDelay[3] ) ;

      /* and the raw atmosphere  */

      for ( i = 0 ; i < iStations ; i++ )
	{
	  pfFDL[i] = pFDL[i]->readMetrology(dfTime+0.75*fDeltaTime) ;
	  pfPiston[i] = pAtm->piston(i, dfTime+fDeltaTime) ;

	  pDisplay[3]->chart ( i, (float)dfTime, pfPiston[i] ) ;
	}

      fprintf ( pFFDL,     "%6.3f   ", dfTime ) ;
      for ( i = 0 ; i < iStations ; i++ )
	fprintf ( pFFDL, " %7.3f", pfFDL[i] ) ;

      fprintf ( pFFDL, "\n" ) ;
      fflush ( pFFDL ) ;

      /* Fringe tracker performance data */

      fprintf ( pFTracker, "T=%7.3f ", dfTime ) ;
      fprintf ( pFTracker, "I0=%4d SNRs= ", int(fCounts) ) ;

      /* The data stored as SNR is really N^2V^2 whereas
	 the SNR is more like (NV^2)^.5.  */

      for ( i = 1 ; i < iStations ; i++ )
	{
	  pfSNR[i] = sqrt(pfSNR[i]/fCounts) ;
	  fprintf ( pFTracker, "%4.0f", pfSNR[i] ) ;
	}
      fprintf ( pFTracker, " atmos= " ) ;
      for ( i = 0 ; i < iStations ; i++ )
	fprintf ( pFTracker, " %7.3f", pfPiston[i] ) ;

      fprintf ( pFTracker, " FDL-ATM= " ) ;
      for ( i = 0 ; i < iStations ; i++ )
	fprintf ( pFTracker, " %7.3f", pfFDL[i] - pfPiston[i] ) ;

      fprintf ( pFTracker, " GD = " );
      for ( i = 1 ; i < iStations ; i++ )
	fprintf ( pFTracker, " %7.3f", pfDelay[i] ) ;
      fprintf ( pFTracker, " um\n" ) ;
      fflush ( pFTracker ) ;

      dfTime += fDeltaTime ;

      
      printf ( "Time = %6.3f   Tip/tilt=", dfTime ) ;
      for ( i = 0 ; i < iStations ; i++ )
	printf ( " %7.3f/%7.3f", pfTip[i], pfTilt[i] ) ;
      printf ( "\n" ) ;
    }

  fclose ( pFFDL ) ;

return 0 ;
}



