/* =======================================================================
   fitData.C

   Fit the fringe spectra

   2002-08-20 David Mozurkewich -- function created
   
=======================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <libgen.h>
#include <math.h>
#include <string.h>
extern "C" {
#include <cpgplot.h>
}

#define PI  3.1415926535897932385
#define	NK	320	/* Number of points to interpolate  */


typedef struct{
  float fA ;
  float fB0 ;
  float fB1 ;
  float fk1 ;
  int iPeaks ;
  float *pfFreq ;
  float *pfA ;
} FIT ;

/*  =========== global variables ==================*/

static float pfK[NK], pfData[NK], pfFit[NK], pfX[65], pfRaw[65] ;
static char pcScanName[200] ;
static int iChan, iSpec ;
static float fWaveln, fI0 ;
static char *base(char *pcName) ;
static char pcBuf[8000] ;
static FILE *pFin, *pFout ;
static int iNumFreqs ;      // number of fringe frequencies
static int iInterp ;     // interpolation factor  1 -> only indep points
static float fUseFreq = 0. ;

char *pcPlotDev = "/xserv" ;
int iPlotDev ;

/*================= forward function definitions ===================== */

int readData( FILE *pF )  ;
void plotData (int iPts, float *pfX, float *pfY ) ;
float sinc ( float fX ) ;
float sincsq ( float fX ) ;
void doFit ( int iPts, float *pfX, float *pfData) ;
void fit ( FIT *pFit, int iPts, float *pfX, float *pfData ) ;
float fitB1( int iPts, int iTerp, 
	     float *pfX, float *pfFit, float *pfData, FIT *pFit ) ;
float fitB0( int iPts, int iTerp, 
	     float *pfX, float *pfFit, float *pfData, FIT *pFit ) ;
void fitA( int iPts, int iTerp, 
	   float *pfX, float *pfFit, float *pfData, FIT *pFit ) ;
float fitFreq( float *pfData, float *pfFit, float fk ) ;

/* from file plot2.C */

void setLabels ( char *pcX, char *pcY, char *pcTop ) ;
void plot2( int iPts, int iTerp, float *pfX, float *pfY, float *pfFit ) ;


/* ******************************************************************** */
int main(int iArgs, char **pcArg)
{
  /* =========================================================
     part 1.  User interface.  Get user supplied parameters
     and store them where they can be accessed where needed.
     ==========================================================*/

  if ( iArgs < 2 )
    {
      printf ( "usage:  \"%s fileName\"\n", pcArg[0] ) ;
      exit(1) ;
    }
  /* =========================================================
     Part 2.  Initialize

     read wavelength list from wavelength file
     create an instance of the fringeSpectra class 

     Create one plot device.
     =========================================================*/

  pFin  = fopen ( pcArg[1], "r" ) ;
  pFout = fopen ( "Results", "a" ) ;
  if ( pFin != 0 )
    printf ( " file \"%s\" opened\n", pcArg[1] ) ;
  else
    exit(1) ;

  if ( iArgs > 2 )
    fUseFreq = atof( pcArg[2] ) ;
  printf ( "using fringe frequency of %6.2f\n", fUseFreq ) ;

  /* ==========================================================
     Part 3.  Process Data
     ==========================================================*/

  iPlotDev = cpgopen(pcPlotDev);
  while ( readData(pFin) )
    {
      plotData(iNumFreqs, pfK, pfData) ;
      sprintf(pcBuf, "%s, Sp%1d Ch%2.2d,   %5.1f nm",
	      base(pcScanName), iSpec, iChan, fWaveln );
      setLabels ( "Fringe Frequency", "Squared Amp V\\u2\\d", pcBuf ) ;
      doFit ( iNumFreqs, pfK, pfData) ;
    }
  return (0) ;
}

int readData( FILE *pF ) 
{
  int iRet =  0 ;
  int i ;

  if ( strlen(fgets( pcScanName, 200, pF )) < 10 )
    return ( iRet ) ;
  pcScanName[strlen(pcScanName)-1] = '\0' ;
  printf ( "Scan name = \"%s\"\n", pcScanName ) ;
  printf ( "truncted  = \"%s\"\n", base(pcScanName) ) ;
  if ( strlen(fgets( pcBuf, 8000, pF )) < 10 ) 
       return ( iRet ) ;
  sscanf(pcBuf+ 6, "%d", &iNumFreqs ) ;
  sscanf(pcBuf+21, "%d", &iInterp ) ;
  printf ( " spectrum has %2d frequencies with %2d points per indep freq\n",
	   iNumFreqs, iInterp ) ;
  if ( strlen(fgets( pcBuf, 8000, pF )) < 10 ) 
       return ( iRet ) ;
  printf ( "line read = \"%s\"\n", pcBuf ) ;
  sscanf(pcBuf+ 3, "%d", &iSpec ) ;
  printf ( " spectrograph number %2d\n", iSpec ) ;
  sscanf(pcBuf+ 9, "%d", &iChan ) ;
  printf ( " spectral channel = %2d\n", iChan ) ;
  sscanf(pcBuf+13, "%f", &fWaveln ) ;
  printf ( "wavelength = %9.3f\n", fWaveln ) ;
  sscanf(pcBuf+25, "%f", &fI0 ) ;
  printf ( "Counts per frame = %6.2f\n", fI0 ) ;
  if ( strlen(fgets( pcBuf, 8000, pF )) < 10 ) 
       return ( iRet ) ;

  for ( i = 0 ; i < iNumFreqs ; i++ )
    {
      sscanf(pcBuf+9*i, "%f", &pfData[i] ) ;
      pfK[i] = float(i)/float(iInterp) ;
    }
  return ( 1 ) ;
}

/** ==============================================================
    sinc(x)  =    sin(pi X ) / ( pi x )
    ==============================================================*/
float sinc ( float fX )
{
  if ( fX == 0. )
    return ( 1.0 ) ;
  else
    return ( sin(PI*fX)/(PI*fX ) ) ;
}
/** ==============================================================
    sincsq(x)  =    sinc^2(X)
    ==============================================================*/
float sincsq ( float fX )
{
  float f ;
  f = sinc(fX) ;
  return ( f*f ) ;
}



void plotData (int iPts, float *pfX, float *pfY )
{
  float fMax ;
  int iK ;

  fMax = 0. ; 
  for ( iK = 0 ; iK < iPts ; iK++ )
    if ( pfY[iK] > fMax )
      fMax = pfY[iK] ;

  cpgslct( iPlotDev ) ;
  cpgenv(0, pfX[iPts-1], 0, 1.1*fMax, 0, 0);
  sprintf(pcBuf, "%s, Sp%1d Ch%2.2d,   %5.1f nm",
	  base(pcScanName), iSpec, iChan, fWaveln );
  cpglab("Fringe Frequency", "V\\u2", pcBuf);
  cpgsch(1.0);
  cpgsci(1);
  cpgline(iPts, pfX, pfY);
}

/** ===============================================
    char *base
    truncate file name to just data star and time
    ================================================*/
char *base( char *pcName )
{
  int i=0, iLen ;
  iLen = strlen(pcName) ;
  while ( i < iLen )
    {
      if ( (pcName[i]>='0')&&(pcName[i]<='9'))
	break ;
      i++ ;
    }
  printf ( "i=%d  iLen = %d\n", i, iLen ) ;
  if ( i == strlen(pcName) )
    return (pcName) ;
  else
    return (pcName+i) ;
}

/** ==============================================================
    doFit  fit the fringe spectrum assuming iPeaks fringes present

    ==============================================================*/
void doFit ( int iPts, float *pfX, float *pfData )
{
  FIT *pFit ;
  int iPeaks = 1 ;
  float pfFreq[] = {6.0} ;
  float pfA[6] = {0., 0., 0., 0., 0., 0. } ;
  int i, iIter  ;
  float pfFit[320] ;
  float fX, fXX ;

  pFit = new FIT ;
  pFit->iPeaks = iPeaks ;
  pFit->pfFreq = pfFreq ;
  pFit->pfA = pfA ;

  /* Enter initial guesses for the fit parameters */

  if ( fUseFreq )
    pfFreq[0] = fUseFreq ;

  pFit->fB0 = pfData[iPts-1] ;               // white noise level
  pFit->fB1 = pfData[iInterp] - pFit->fB0 ;        // red noise level
  pFit->fk1 = 12. ;                            // red noise decay "time"
  pFit->fA  = pfData[0]-pFit->fB0-pFit->fB1 ; // zero frequency peak
  fit( pFit, iPts, pfX, pfFit ) ;
  fitA( iPts, iInterp, pfX, pfFit, pfData, pFit ) ;

  /* put a Fit through the data */
  /*
  printf ( "first  values\n" ) ;
  printf ( " fA    = %6.3f\n", pFit->fA ) ;
  printf ( " fB0   = %6.3f\n", pFit->fB0 ) ;
  printf ( " fB1   = %6.3f\n", pFit->fB1 ) ;
  printf ( " fk1   = %6.3f\n", pFit->fk1 ) ;
  for ( i = 0 ; i < pFit->iPeaks ; i++ )
    printf ( " fA[%1d] = %7.4f\n", int(pFit->pfFreq[i]), pFit->pfA[i] ) ;
  */
  fit( pFit, iPts, pfX, pfFit ) ;

  /*  Improve the fit  */

  for ( iIter = 0 ; iIter < 30 ; iIter++ )
    {
      /*
      plot2( iPts, iInterp, fX, pfData, pfFit ) ;
      fit( pFit, iPts, pfX, pfFit ) ;
      */
      pFit->fB1 += fitB1(iPts, iInterp, pfX, pfFit, pfData, pFit ) ;
      fit( pFit, iPts, pfX, pfFit ) ;
      pFit->fB0 += fitB0(iPts, iInterp, pfX, pfFit, pfData, pFit ) ;
      fit( pFit, iPts, pfX, pfFit ) ;
      pFit->fA  += pfData[0] - pfFit[0] ;
      fitA( iPts, iInterp, pfX, pfFit, pfData, pFit ) ;
      
      /* fit the first fringe freq */
      if ( !(iIter%4) )
	pFit->pfFreq[0] += fitFreq( pfData, pfFit, pFit->pfFreq[0] ) ;
      fit( pFit, iPts, pfX, pfFit ) ;
    }
  printf ( " fA    = %6.3f\n", pFit->fA ) ;
  printf ( " fB0   = %6.3f\n", pFit->fB0 ) ;
  printf ( " fB1   = %6.3f\n", pFit->fB1 ) ;
  printf ( " fk1   = %6.3f\n", pFit->fk1 ) ;
  for ( i = 0 ; i < pFit->iPeaks ; i++)
    printf ( " fA[%4.2f] = %7.4f\n", pFit->pfFreq[i], pFit->pfA[i] ) ;

  fprintf ( pFout, "%s", base(pcScanName) ) ;
  fprintf ( pFout, "%6.1f %6.2f", fWaveln, fI0 ) ;
  fprintf ( pFout, " fA = %6.3f", pFit->fA ) ;
  fprintf ( pFout, " fB0 = %6.3f", pFit->fB0 ) ;
  fprintf ( pFout, " fB1 = %6.3f", pFit->fB1 ) ;
  fprintf ( pFout, " fk1 = %6.3f", pFit->fk1 ) ;
  for ( i = 0 ; i < pFit->iPeaks ; i++)
    fprintf ( pFout, " fA[%4.2f] = %7.4f",
	      pFit->pfFreq[i], pFit->pfA[i] ) ;
  fprintf ( pFout, "\n" ) ;
  fflush ( pFout ) ;

  plot2( iPts, iInterp, pfX, pfData, pfFit ) ;

  /* Calculate mean and rms.  After "plot2", pfFit
     contains the residuals.  */

  fX = 0. ;
  fXX = 0. ;
  for ( i = 0 ; i < iPts ; i+=iInterp )
    {
      fX  += pfFit[i] ;
      fXX += pfFit[i]*pfFit[i] ;
    }
  fX  *= float(iInterp)/float(iPts) ;
  fXX *= float(iInterp)/float(iPts) ;
  printf ( " mean residual = %6.4f\n", fX ) ;
  printf ( " standard dev  = %6.4f\n", sqrt(fXX-fX*fX) ) ;     

  delete [] pFit ;
}

void fitA( int iPts, int iTerp, 
	     float *pfX, float *pfFit, float *pfData, FIT *pFit )
{
  int i ;

  for ( i = 0 ; i < pFit->iPeaks ; i++ )
    pFit->pfA[i] += pfData[int(float(iTerp)*pFit->pfFreq[i])]
      - pfFit[int(float(iTerp)*pFit->pfFreq[i])] ;
}

float fitB0( int iPts, int iTerp, 
	     float *pfX, float *pfFit, float *pfData, FIT *pFit )
{
  int i, iCount=0 ;
  float fY = 0. ;
  for ( i = iTerp; i < iPts ; i+=iTerp )
    {
      fY += pfData[i] - pfFit[i] ;
      iCount++ ;
    }
  return ( fY/float(iCount) ) ;
}

float fitB1( int iPts, int iTerp, 
	     float *pfX, float *pfFit, float *pfData, FIT *pFit )
{
  int i, iCount=0 ;
  float fY = 0. ;
  for ( i = iTerp; i < iPts ; i+=iTerp )
    {
      if ( pfX[i] < pFit->fk1 )
	{
	  fY += pfData[i] - pfFit[i] ;
	  iCount++ ;
	}
    }
  return ( 1.6*fY/float(iCount) ) ;
}

void fit ( FIT *pFit, int iPts, float *pfX, float *pfFit )
{
  int i, k ;
  for ( i = 0 ; i < iPts ; i++ )
    {
      pfFit[i] = pFit->fA*sincsq(pfX[i])
	+ pFit->fB0
	+ pFit->fB1 * exp(-pfX[i]/pFit->fk1) ;
      for ( k = 0 ; k < pFit->iPeaks ; k++ )
	pfFit[i] +=  pFit->pfA[k]*sincsq(pfX[i]-pFit->pfFreq[k]) ;
    }
}


float fitFreq( float *pfData, float *pfFit, float fk )
{
  int i1, i2, j ;
  float fDelta ;

  /* find the two half power points */

  i1 = int(0.5+float(iInterp)*(fk-0.5)) ;
  i2 = int(0.5+float(iInterp)*(fk+0.5)) ;

  printf ( "interp = %3d\n", iInterp ) ;
  printf( "fitFreq: begins.  freq = %6.3f at pixel %3d\n",
	  fk, int(fk*float(iInterp)) ) ;
  for ( j = i1 ; j < i2+1 ; j++ )
    printf ( " i=%d  Data = %9.6f  fit = %9.6f\n",
	     j, pfData[j], pfFit[j] ) ;
  printf ( "interpolation values are at pixels %d nd %d\n", i1, i2 ) ;
  fDelta = (PI*PI/16.)*(pfData[i2]-pfFit[i2]-pfData[i1]+pfFit[i1])  ;
  printf ( " delta Freq = %9.6f\n", fDelta ) ;
  return ( fDelta ) ;
}
