/** =======================================================================
    fringeSpectra.C    fringe spectrum class.

    calculates the fringe spectrum for a scan.
    user specifies the spectrograph spectral channel

    2002-07-02 written by Dan Driscoll

    2002-08-19 modified by mozurkewich
    replaced X,Y calc by fringeParam  by internal calc
    calculate all independent frequencies
    exact interpolation with the sinc^2 formulae
    should be several times faster
    Plot full range and interpolated low frequencies.
    do interpolation better as a learning exercize.
    ========================================================================*/

#include <stdio.h>
#include <libgen.h>
#include <math.h>
#include <vector.h>
#include "iter.h"

#include <npoi_container.h>
#include <npoi_frame.h>

#include "wavelength.h"
#include "configure.h"

#include <npoi_config.h>

#include "fringeSpectra.h"

extern "C" {
#include <cpgplot.h>
}

/** ======================================================================
    fringeSpectra::fringeSpectra(NPOI_CONFIG *conf, float kStart,
    int kN, float kInc)
    Constructor, zeros the whole structure and copies over the
    appropriate values from the configuration structure.

    iInterp number of freq / number of independent freqs
    kN  number of frequencies
    ======================================================================*/
fringeSpectra::fringeSpectra(configure *pC, int iInt, int kN )
{

  iNumFreqs = kN ;   // number of fringe frequencies calculated
  iInterp   = iInt ; // Interpolation factor  1 -> no interpolation

  pConf = pC ;
  iSpec = 0;
  iChan = 0;

  nFrame = pC->getNumFrames() ;
  /*
  if ( nFrame > 1000 )
    nFrame = 1000 ;
  */
  nBin   = pC->getNumBins() ;

  pdfSpectrum = new double[iNumFreqs] ;
  spectra = new float[iNumFreqs] ;
}

/** ======================================================================
    fringeSpectra::~fringeSpectra()   destructor
    =======================================================================*/
fringeSpectra::~fringeSpectra()
{
  delete [] pdfSpectrum ;
  delete [] spectra ;

}

/* =======================================================================
   fringeSpectra::calculate(NPOI_FRAME )
    Perform fringe spectra calculation.

    This is a calculation directly from the raw bin counts since the
    X,Y values wshen stored in the npoi_container are only for the
    low, integer fringe frequencies.  Here, we calculate all the
    fringe frequencies, then interpolate between them.

    This function calculates the spectrum at all the integer
    frequencies.
    ======================================================================*/
void fringeSpectra::calculate(NPOI_FRAME *data)
{
  int iFrame, iK ;
  float fCounts ;

  if ( pdfSpectrum == NULL )
    {
      printf ( "ERROR: pdfSpectrum does not exists " ) ;
      printf ( "at start of fringeSpectra:calculate!\n" ) ;
      exit(1) ;
    }

  printf ( "calculation covers %d frames of data\n", nFrame) ;
  printf ( " fringeSpectra::calculate doing %d frequencies\n", nBin ) ;

  /* Calculate the mean signal strength first */
  /* fCounts will be the mean number of counts per bin
     dfI0 is the mean number of counts per frame
     both are averaged over all the frames.  */

  dfI0 = 0. ;
  for(iFrame = 0; iFrame < nFrame; iFrame++)
    dfI0 += calcI0( data[iFrame].photonCounts[iSpec][iChan] ) ;
  dfI0 /= float(nFrame) ;
  fCounts = dfI0 / float(nBin) ;

  /* now do the bin power spectrum.  If fCounts=0 on calling calc1,
     the mean counts for that frame are subtracted before doing the 
     fourier transforms.  */

  for ( iK = 0; iK < iNumFreqs ; iK++ )
    pdfSpectrum[iK] = 0. ;
  for(iFrame = 0; iFrame < nFrame; iFrame++)
    calc1( nBin, data[iFrame].photonCounts[iSpec][iChan], fCounts ) ;

  /* Normalize the result.  This code DOES NOT subtract off the bias. */

  printf ( "mean counts per frame = %f\n", dfI0 ) ;
  for ( iK = 0; iK < iNumFreqs ; iK++ )
    {
      pdfSpectrum[iK] /= double (nFrame) ;
      pdfSpectrum[iK] = 4.0*pdfSpectrum[iK] / ( dfI0*dfI0 ) ;
    }
}

/* =======================================================================
   fringeSpectra::calcI0(frame)
    Returns the mean number of counts per frame
    in this frame for this spectral channel.
    ======================================================================*/
float fringeSpectra::calcI0( unsigned char  *piBins)
{
  int iBin ;
  float fCounts ;

  fCounts = 0. ;
  for ( iBin = 0 ; iBin < nBin ; iBin++ )
    fCounts += float(piBins[iBin]) ;

  return ( fCounts) ;
}

/* =======================================================================
   fringeSpectra::calc1(void)
    Perform fringe spectra calculation.

    This is a calculation directly from the raw bin counts sinc the 
    X,Y values stored in the npoi_container are only for the low, interger
    fringe frequencies.  Here, we calculate all the fringe frequencies
    and interpolate between them.

    This function increments the spectrum at all the integer
    frequencies.

    if fMeanCouns == 0, the mean signal is determined and subtracted
    otherwise, the value of fMeanCounts is used as the mean counts per bin
    ======================================================================*/
void fringeSpectra::calc1(int iNumBins, 
			  unsigned char  *piBins, float fMeanCounts)
{
  int iBin, iK;
  double fX, fY, fCounts, fFreq ;

  if ( fMeanCounts )
    fCounts = fMeanCounts ;
  else
    {
      fCounts = 0. ;
      for ( iBin = 0 ; iBin < iNumBins ; iBin++ )
	fCounts += float(piBins[iBin]) ;
      fCounts /= float(iNumBins) ;
    }

  for ( iK = 0 ; iK < iNumFreqs ; iK++ )
    {
      fX = 0. ;
      fY = 0. ;
      for ( iBin = 0 ; iBin < iNumBins ; iBin++ )
	{
	  fFreq = 2.*PI*float(iK*iBin)/float(iInterp*iNumBins) ;
	  fX += (double(piBins[iBin])-fCounts)* cos(fFreq) ;
	  fY += (double(piBins[iBin])-fCounts)* sin(fFreq) ;
	}
      pdfSpectrum[iK] += fX*fX + fY*fY ;
    }
}

/** =====================================================
    fringeSpectra::save

    save fringe power spectra.  file format ascii
    input data file name
    number fringe freq     delta between frequencies
    spectrograph, channel,  wavelength,  mean flux  
    power at each freq starting with k=0
    ======================================================*/
void fringeSpectra::save( void ) 
{
  FILE *pF ;
  int i ;

  pF = pConf->getOutputStream() ;
  fprintf ( pF, "%s\n", pConf->getFileName() ) ;
  fprintf ( pF, "nKs = %4d  INTERP = %6d\n", iNumFreqs, iInterp ) ;
  fprintf ( pF, "SP=%2d CH=%2d %6.1f  I0=%8.3f\n", iSpec, iChan,
	    pConf->getWaveln(iSpec, iChan), dfI0 ) ;
  for ( i = 0 ; i < iNumFreqs ; i++ )
    fprintf ( pF, "%9.6f", pdfSpectrum[i] ) ;
  fprintf ( pF, "\n" ) ;
  fflush ( pF ) ;
}

/** ==============================================================
    sinc(x)  =    sin(pi X ) / ( pi x )
    ==============================================================*/
float fringeSpectra::sinc ( float fX )
{
  if ( fX == 0. )
    return ( 1.0 ) ;
  else
    return ( sin(PI*fX)/(PI*fX ) ) ;
}

/** ============================================
    fringeSpectra::plotsFS

    make plots of both the low (raw) and
    high (interpolated) resolution fringe spectra
    ==============================================*/
void fringeSpectra::plotFS(void)
{
  char pcBuf[BUFSIZ] ;
  static int piDevice[2] ;
  float fMax, *pfX=NULL ;
  int iK ;

  piDevice[0] = cpgopen("?");

  /* low resolution first */

  fMax = 0. ; 
  for ( iK = 0 ; iK < nBin ; iK++ )
    {
      spectra[iK] = pdfSpectrum[iK] ;
      if ( pdfSpectrum[iK] > fMax )
	fMax = pdfSpectrum[iK] ;
    }
  printf ( "max value of %d plotted points is %f\n", 64, fMax ) ;
  if ( fMax <= 0. )
    fMax = 0.5 ;

  pfX = new float[nBin+1];
  for(iK = 0; iK < nBin+1; iK++)
    pfX[iK] = iK ;

  cpgenv(0, nBin/2, 0, 1.1*fMax, 0, 0);
  sprintf(pcBuf, "File %s, spec %2d chan %2d %5.1f",
	  basename(pConf->getFileName()), iSpec, iChan,
	  pConf->getWaveln(iSpec,iChan) );
  cpglab("Fringe Frequency", "V\\u2", pcBuf);
  cpgsch(0.6);
  cpgsci(1);
  cpgline(nBin/2+1, pfX, spectra);

  printf ( "plots complete.  Closing devices\n" ) ;

  delete [] pfX ;
  cpgclos();
  printf ( "device 2 closed\n" ) ;
  cpgslct(piDevice[0]) ;
  cpgclos();
  printf ( "device 1 closed\n" ) ;

}

