/* ******************************************************************** */
/* gdscrollf.C								*/
/* 2002-07-30 Dan Driscoll						*/
/* ******************************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <assert.h>
#include <curses.h>
#include <getopt.h>
#include <npoi_container.h>

extern "C" {
#include <cpgplot.h>
}

#include "minmax.h"
#include "groupDelayMerit.h"
#include "wavelength.h"
#include "histobin.h"
#include "plotter.h"

#define STARTD	-8000.0
#define INCD	20.0
#define ND	 800
#define ENDD	INCD * ND + STARTD
#define NBIN	20
#define ALPHA	-5e-7
#define SF	1.1
#define CHARSCALE 1.3

#define WHITE 1
#define RED 2

groupDelayMerit *calculateGD(NPOI_CONTAINER *, wavelength *, int, float, int, int);
void interact(groupDelayMerit *, int, int, int, int, int, float, float, float, float,
		unsigned long, unsigned long);

/* ******************************************************************** */
int main(int argc, char **argv)
{
  NPOI_CONTAINER *d;
  wavelength *wave;
  groupDelayMerit *gd;
  int iSpec, nFrame;
  unsigned long	first_ts;
  unsigned long	delta_ts;
  float		k;
  float		maxG;
  float		maxDD;
  float		mindWdG;
  float		maxdWdG;
  int		statWin0;
  int		statWin1;
  int		statWin2;
  int		plotWin;
  int		XYsmooth;
  int		GDsmooth;

  /* check for proper user input */
  if(argc < 7)
    {
      fprintf(stderr, "Usage: %s INFILE WAVEFILE SPEC K XYSMOOTH GDSMOOTH\n", argv[0]);
      exit(1);
    }

  /* read the data from the input file */

  printf ( "loading data from file \"%s\"\n", argv[1] ) ;
  d = new NPOI_CONTAINER(argv[1]);
  if(d->config == NULL)
    {
      fprintf(stderr, "No data loaded\n");
      exit(1);
    }
  nFrame = d->config->nFrame;
  printf ( "all data loaded.  %4d frames\n", nFrame ) ;
  if ( nFrame > 2000 )
    {
      nFrame = 2000 ;
      printf ( "using first %4d frames\n", nFrame ) ;
    }

  /* get the wavelength list from the user-specified file.  wave
     contains center wavelengths, edge wavelengths, bandpasses and
     sorting indices.  */

  wave = new wavelength(d->config);
  wave->load(argv[2]);
  printf ( "wavelengths loaded\n" ) ;

  /* and the other user-supplied parameters */

  iSpec = atoi(argv[3]);   // spectrograph number.  starting with zero, sorry, I have to change that
  k = atof(argv[4]);       // fringe frequency
  XYsmooth = atoi(argv[5]);// smoothing to do on the bin counts
  GDsmooth = atoi(argv[6]); // and on the group-delay function
  printf ( "smoothing set:  XY = %2d points,  GD = %2d frames\n", 
	   XYsmooth, GDsmooth ) ;

  /* This is the first time stamp and total time covered by the observations */

  first_ts = d->config->firstTS;
  delta_ts = d->config->lastTS - first_ts;

  /* ======================================================================
     the plotting is initialized in the main program, but controlled 
     from function "interact".  Is this a good idea? */

  /* Open the plotting window.  There should only be one.
     Set it to have four panels.   The two left panels will have group delay
     and "white light" delay as a function of time for 500 points.
     The two right panels will have G(d), amplitude and phase for the selected time.  */

  statWin0 = cpgopen("/xserv");
  cpgsch(CHARSCALE);
  cpgsubp(2, 2);
  cpgask(0);

  printf ( "ready to run calculateGD\n" ) ;
  gd = calculateGD(d, wave, iSpec, k, XYsmooth, GDsmooth);
  printf ( " gd calculations done.\n" ) ;

  interact(gd, statWin0, statWin1, statWin2, plotWin, nFrame,
	   maxG, mindWdG, maxdWdG, maxDD, first_ts, delta_ts);

  cpgclos();

  delete d;

  exit(0);
}

/** =======================================================================
    Calculate the group delay function for each time stamp
    for the chosen spectrograph and fringe frequency 

    *d contains the data
    wave contains the list of wavelengths
    iSpec, k are the spectrograph and fringe frequency

    XYsmooth averaging time for the complex Visibility
    GDsmooth averaging time for the group delay calc

    The total av time, in sample times, is the product of the two.

    The important stuff calculate in here seems to be

    gd[iFrame].fringe[iWave].{X,Y,I0} fringe parameters averaged to XYsmooth
    gd[iFrame].{R,I,G,phase} real, imag, power and phase of function G
    
    ========================================================================*/
groupDelayMerit *calculateGD(NPOI_CONTAINER *d, wavelength *wave,
			     int iSpec, float k, int XYsmooth, int GDsmooth)
{
  groupDelayMerit *gd ;
  fringeParam **fp ;
  float maxG ;
  int iFrame, nFrame, i, iD, iChan, nChan ;

  nFrame = d->config->nFrame ;
  nChan = d->config->nChan ;
  if ( nFrame > 2000 )
    {
      nFrame = 2000 ;
      printf ( "gdMerit: using first %d frames\n", nFrame ) ;
    }

  /* gd and fp provide storage for the group delay merit function and 
     calculation */

  printf ( "calling groupDelayMerit\n" ) ;
  gd = new groupDelayMerit[nFrame](d->config, STARTD, ND, INCD);
  printf ( "group delay merit done\n" ) ;

  /* set aside storage for the group delay calculation.
     fringeParam calculates X,Y,I0 (I0= old N) for a specific
     fringe frequency, wavelength and spectrograph.
     
     ********all wavelengths should be treated as a group.*********
     */

  fp = new (fringeParam *)[nFrame];
  printf ( "fringeparam done\n" ) ;
  for(iFrame = 0; iFrame < nFrame; iFrame++)
    fp[iFrame] = new fringeParam[nChan](d->config);

  /* and do the calculation */

  for(iFrame = 0; iFrame < nFrame; iFrame++)
    for(iChan = 0; iChan < nChan; iChan++)
      {
	fp[iFrame][iChan].iSpec = iSpec;
	fp[iFrame][iChan].iChan = iChan;
	fp[iFrame][iChan].k = k;

	fp[iFrame][iChan].calculate(&(d->frames[iFrame]));
      }
  printf ( "Fringe parameter calculation done!\n" ) ;

  /* Smooth the raw visibilities.  Note that the time stamp
     corresponds to the start of the averaging cycle */

  /* also note that this is a little wrong.  only independent
     X,Y averages should be included in the data product.  */

  for(iFrame = 0; iFrame < nFrame; iFrame++)
    for(i = 1; i < XYsmooth; i++)
      if(iFrame + i < nFrame)
	{
	  fp[iFrame][iChan].X  += fp[iFrame + i][iChan].X ;
	  fp[iFrame][iChan].Y  += fp[iFrame + i][iChan].Y ;
	  fp[iFrame][iChan].I0 += fp[iFrame + i][iChan].I0 ;
	}

  printf ( "XYN smoothing done\n" ) ;

  /* now calculate the group delay function G */

  for(iFrame = 0; iFrame < nFrame; iFrame++)
    {
      gd[iFrame].k = k;
      gd[iFrame].iSpec = iSpec;

      gd[iFrame].wave = wave;

      gd[iFrame].fringe = fp[iFrame];
      gd[iFrame].calculate();
      if ( !(iFrame%500) )
	printf ( "calculate and find max %3d\n", iFrame) ;
      gd[iFrame].findMaxG();

      gd[iFrame].wave = NULL;
    }
  printf ( "Group Delay Calculation done\n" ) ;


  for(iFrame = 0; iFrame < nFrame; iFrame++)
    {
      if(iFrame == 0 || gd[iFrame].maxG > maxG)
	maxG = gd[iFrame].maxG;
    }

  for(iFrame = 0; iFrame < nFrame; iFrame++)
    {
      if(GDsmooth)
	{
	  for(i = 1; i < GDsmooth; i++)
	    if(iFrame + i < nFrame)
	      for(iD = 0; iD < gd[iFrame].d->n; iD++)
		{
		  gd[iFrame].R[iD] += gd[iFrame + i].R[iD];
		  gd[iFrame].I[iD] += gd[iFrame + i].I[iD];
		}

	  for(iD = 0; iD < gd[iFrame].d->n; iD++)
	    {
	      gd[iFrame].G[iD] =	gd[iFrame].R[iD] * gd[iFrame].R[iD] +
		gd[iFrame].I[iD] * gd[iFrame].I[iD];
	      gd[iFrame].phase[iD] =	atan2(gd[iFrame].I[iD],
					      gd[iFrame].R[iD]);
	    }

	  gd[iFrame].findMaxG();
	}
    }

  for(iFrame = 0; iFrame < nFrame; iFrame++)
    {
      gd[iFrame].gd = gd;
      gd[iFrame].nGD = nFrame;
      gd[iFrame].iGD = iFrame;

      gd[iFrame].meritStepThreshold = 20;

      gd[iFrame].weightPower = 1;
      gd[iFrame].weightStep = 2;
      gd[iFrame].weightDeriv = 1;

      gd[iFrame].meritCalc();
    }
  
  delete [] fp;

  return gd;
}

/* ******************************************************************** */
void interact(groupDelayMerit *gd, int statWin0, int statWin1, int statWin2,
	      int plotWin, int nFrame, float maxG, float mindWdG, float maxdWdG,
	      float maxDD, unsigned long first_ts, unsigned long delta_ts)
{
  int iTimeFlag  = 0 ;
  char pcTitle[500] ;
  int iNumFrames = 200 ;
  int iFirstFrame = 0 ;
  float fMax ;
  char		bQuit = 0;
  int		c;
  int iD = 0, iLast = 0 ;
  unsigned long	iTS;
  unsigned long ulFirstTS ;
  int i, iFrame ;
  float pfX[ND] ;
  char buf[BUFSIZ] ;


  ulFirstTS = gd[0].time ;
  
  /* These switch the terminal into interactive mode so that
     single characters can be read */

  (void)initscr();
  (void)keypad(stdscr, TRUE);
  (void)cbreak();
  (void)noecho();

  /* display the help menu.  just once for now */

  printw("Controls: LEFT/RIGHT -> Select frame\n");
  printw("          PAGEUP     -> Forward 50 frames\n");
  printw("          PAGEDOWN   -> Backward 50 frames\n");
  printw("          t          -> toggle time axis(point to time)\n" ) ;
  printw("          q          -> Quit\n");

  while ( 1 )  {

    /* make all the plots */

  cpgslct(statWin0);
  cpgbbuf();

  /* for now, redraw everything every time. IT'S FAST ENOUGH */

  /* plot 1.  group delay versus time */

  if ( iTimeFlag )
    for ( i = 0 ; i < iNumFrames ; i++ )
      pfX[i] = 0.002 * float(gd[iFirstFrame+i].time - ulFirstTS) ;
  else
    for ( i = 0 ; i < iNumFrames ; i++ )
      pfX[i] = iFirstFrame + i ;

  fMax = 0. ;
  for ( i = 0 ; i < iNumFrames ; i++ )
    {
      iFrame = iFirstFrame + i ;
      if ( fMax < gd[iFrame].maxD0p ) 
	fMax = gd[iFrame].maxD0p ;
    }
  cpgpanl(1, 1);
  cpgeras() ;
  cpgsci(WHITE) ;
  cpgswin(pfX[0], pfX[iNumFrames-1], STARTD, ENDD );
  cpgbox  ( "BCNST", 0.0, 0, "BCNSTV", 0.0, 0 ) ;
  sprintf ( pcTitle, "Start Time = %10ld", gd[iFirstFrame].time ) ;
  if ( iTimeFlag )
    cpglab("Seconds", "d\\dW", "White light delay");
  else
    cpglab("FRAME NUMBER", "d\\dW", "d at max G");
  for ( i = 0 ; i < iNumFrames ; i++ )
    cpgpt1(pfX[i], gd[iFirstFrame+i].maxD0p, -1);

  /* mark the current point in red with tick marks on the axes to make
     the point more visible */

  cpgsci(RED) ;
  cpgpt1( pfX[iD], gd[iFirstFrame+iD].maxD0p, -1 ) ;
  drawtick_right(gd[iFirstFrame+iLast].maxD0p, gd[iFirstFrame+iD].maxD0p ) ;
  drawtick_top(pfX[iLast], pfX[iD]) ;

    /* plot 2.  GD amplitude versus time */
  
  cpgsci(WHITE) ;
  fMax = 0. ;
  for ( iFrame = iFirstFrame ; iFrame < iFirstFrame+iNumFrames ; iFrame++ )
    if ( fMax < gd[iFrame].maxG ) 
      fMax = gd[iFrame].maxG ;

  cpgpanl(1, 2);
  cpgeras() ;
  cpgswin(pfX[0], pfX[iNumFrames-1], 0., fMax );
  cpgbox  ( "BCNST", 0.0, 0, "BCNSTV", 0.0, 0 ) ;
  if ( iTimeFlag )
    cpglab("SECONDS", "max G", "Group Delay Peak") ;
  else
    cpglab("Frame number", "max G", "Group Delay Peak") ;
  for ( i = 0 ; i < iNumFrames ; i++ )
    cpgpt1( pfX[i], gd[iFirstFrame+i].maxG, -1 ) ;

  /* mark the current point in red with tick marks on the axes to make
     the point more visible */

  cpgsci(RED) ;
  cpgpt1( pfX[iD], gd[iFirstFrame+iD].maxG, -1 ) ;
  drawtick_right(gd[iFirstFrame+iLast].maxG, gd[iFirstFrame+iD].maxG ) ;
  drawtick_top(pfX[iLast], pfX[iD]) ;

  /* plot 3.  group delay phase versus wavelength */

    cpgpanl(2, 1);
    cpgeras() ;
    cpgsci(WHITE) ;
    cpgswin(STARTD, ENDD, -PI, PI);
    cpgbox  ( "BCNST", 0.0, 0, "BCNSTV", 0.0, 0 ) ;
    cpgsci(1);
    sprintf(buf, "FRAME[%5d] TIME IN SCAN=%7.3f seconds", iFirstFrame+iD, 
	    0.002 * float(gd[iFirstFrame+iD].time - ulFirstTS) );
    cpglab("d", "Phase", buf);

    for ( i = 0 ; i < ND ; i++ )
      pfX[i] = STARTD + (ENDD-STARTD)*float(i)/float(ND-1) ;
    plot_phase(ND, pfX, gd[iD].phase);

    cpgsci(2);
    cpgmove(gd[iD].maxD0p, -PI);
    cpgdraw(gd[iD].maxD0p, PI);
    cpgmove(STARTD, gd[iD].phaseAtMaxD0p);
    cpgdraw(ENDD, gd[iD].phaseAtMaxD0p);

    cpgsci(4);
    cpgmove(gd[iD].maxD, -PI);
    cpgdraw(gd[iD].maxD, PI);
    cpgmove(STARTD, gd[iD].phaseAtMaxD);
    cpgdraw(ENDD, gd[iD].phaseAtMaxD);

    cpgsci(1);

    /* plot 4.  group delay */

  fMax = 0. ;
  for ( i = 0 ; i < ND ; i++ )

    cpgpanl(2, 2);
    cpgeras() ;
    cpgswin(STARTD, ENDD, 0, gd[iD].maxG ) ;
    cpgbox  ( "BCNST", 0.0, 0, "BCNSTV", 0.0, 0 ) ;
    cpgsci(1);
    cpglab("Delay (nm)", "Power G(d)", "\0");
    cpgline( ND, pfX, gd[iD].G ) ;

    cpgebuf();
    cpgsci(1);

    /* **************************************************** */
    /*
    drawtick_right(gd[iLast].maxG0p, gd[iD].maxG0p);

    */


    /* ask for user input */

    iLast = iD;
    c = getch();
    switch(c)
      {
      case 'q' :
      case 'Q' :
	bQuit = 1;
	break;
      case 'p' :
      case 'P' :
      case KEY_LEFT :
	iD--;
	break;
      case 'n' :
      case 'N' :
      case KEY_RIGHT :
	iD++;
	break;
      case KEY_PPAGE :
	iD += 50;
	break;
      case KEY_NPAGE :
	iD -= 50;
	break;
      case 't' :
      case 'T' :
	iTimeFlag = !iTimeFlag ;
	break ;
      default :
	break;
      }
    if ( bQuit )
      break ;

    /* Make sure we are within bounds.  This needs a little tweaking
       to make sure all is well when nFrames/iNumFrames is not an
       integer.  There can also be problems if iNumFrames < 50   */

    if(iD < 0)
      {
	if ( iFirstFrame >= iNumFrames )
	  {
	    iFirstFrame -= iNumFrames ;
	    iD += iNumFrames ;
	  }
	else
	  iD = 0 ;
      }
    if(iD >= iNumFrames)
      {
	if ( (iFirstFrame+iNumFrames )< nFrame )
	  {
	    iFirstFrame += iNumFrames ;
	    iD -= iNumFrames ;
	  }
	else
	  iD = iNumFrames-1 ;
      }
    iTS = gd[iLast].time - first_ts;
  }
  
  endwin();
}

