/* ******************************************************************** */
/** groupDelay.C							*/
/** 2002-07-03 Dan Driscoll						*/
/* ******************************************************************** */

#include <stdio.h>
#include <math.h>
#include "groupDelay.h"

/** =====================================================================
    groupDelay::groupDelay(float dStart, int dN, float dInc)
    Constructor, zeros whole structure.

    dStart is the first delay
    dN is the number of delays to calculate
    dInc is the interval between delays
    ======================================================================*/
groupDelay::groupDelay(float dStart, int dN, float dInc)
{
  d = new iter(dStart, dN, dInc);

  R = new float[d->n];
  I = new float[d->n];
  G = new float[d->n];
  phase = new float[d->n];

  k = 0;
  iSpec = 0;

  nChan = 0;
  nBin = 0;

  wave = NULL;
  fringe = NULL;

  maxG = 0;
  maxD = 0;
  phaseAtMaxD = 0;
}

/* ******************************************************************** */
/** groupDelay::groupDelay(NPOI_CONFIG *conf, float dStart, int dN,
    float dInc)
    Constructor, zeros whole structure and sets appropriate values from
    conf structure.							*/
/* ******************************************************************** */
groupDelay::groupDelay(NPOI_CONFIG *conf, float dStart, int dN, float dInc)
{
  d = new iter(dStart, dN, dInc);

  R = new float[d->n];
  I = new float[d->n];
  G = new float[d->n];
  phase = new float[d->n];

  iSpec = 0;

  nChan = conf->nChan;
  nBin = conf->nBin;

  wave = NULL;
  fringe = NULL;

  maxG = 0;
  maxD = 0;
  phaseAtMaxD = 0;
}

/** ========================================================
    groupDelay::~groupDelay()
    Destructor.
    ========================================================*/
groupDelay::~groupDelay()
{
  if(d)		{ delete d;		}
  if(wave)	{ delete wave;		}
  if(fringe)	{ delete fringe;	}
  if(R)		{ delete [] R;		}
  if(I)		{ delete [] I;		}
  if(phase)	{ delete [] phase;	}
}

/** ========================================================
    groupDelay::calculate()

    Calculate G(d) for one frame at each of the grid values
    R, I are the real and imaginary parts
    G is the power (square of amplitude)
    phase is the phase.

    This version assumes that X, Y, and I0 have already
    been calculated.
    ========================================================*/
void groupDelay::calculate()
{
  int iChan, iD ;
  float fD, fArg, re, im ;

  d->reset();
  while(d->next())
    {
      iD = d->idx;
      fD = d->val;

      time = fringe[0].time;

      R[iD] = 0;
      I[iD] = 0;

      for(iChan = 0; iChan < nChan; iChan++)
	{
	  fArg = 2 * PI * fD / wave->center[iSpec][iChan];

	  re = cos(fArg);
	  im = sin(fArg);

	  R[iD] += ((re * fringe[iChan].X) - (im * fringe[iChan].Y)) ;
	  I[iD] += ((re * fringe[iChan].Y) - (im * fringe[iChan].X)) ;
	}

      G[iD] = R[iD] * R[iD] + I[iD] * I[iD];
      phase[iD] = atan2(I[iD], R[iD]);
    }

  findMaxG();
}

/** ========================================================
    groupDelay::calculate(NPOI_FRAME *data)

    Calculate G(d) for one frame at each of the grid values
    R, I are the real and imaginary parts
    G is the power (square of amplitude)
    phase is the phase.

    This version also calculates X, Y, and I0 for that frame
    ========================================================*/
void groupDelay::calculate(NPOI_FRAME *data)
{
  int	iChan;

  if(!fringe)
    {
      fringe = new fringeParam[nChan]();

      for(iChan = 0; iChan < nChan; iChan++)
	{
	  fringe[iChan].iSpec = iSpec;
	  fringe[iChan].iChan = iChan;
	  fringe[iChan].k = k;

	  fringe[iChan].nBin = nBin;
	  fringe[iChan].calculate(data);
	}
    }

  if(!wave) {
    fprintf(stderr, "groupDelay.wave not filled\n");

    return;
  }

  calculate();
}

/** ========================================================
    groupDelay::findMaxG()

    Finds the maximum value of G(d) for one frame
    the function must already be calculated
    ========================================================*/
void groupDelay::findMaxG()
{
  int	iD;
  int	tiD;

  d->reset();
  while(d->next())
    {
      iD = d->idx;
      if(iD == 0 || G[iD] > maxG)
	{
	  maxG = G[iD];
	  maxD = d->val;
	  phaseAtMaxD = phase[iD];
	  tiD = iD;
	}
    }
}
