/* ******************************************************************** */
/** photonPix.C								*/
/** 2002-07-29 Dan Driscoll						*/
/* ******************************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <libgen.h>
#include <unistd.h>
#include "wavelength.h"
#include "minmax.h"
#include <npoi_container.h>

extern "C" {
#include <cpgplot.h>
}

#define	HPIXELS	640
#define VPIXELS	640
#define AR 3		/* Aspect ratio -> 3:1				*/

wavelength *buildWavenumber(wavelength *, int);
int **countPhotons(NPOI_CONTAINER *, int, int, int);

/* ******************************************************************** */
int main(int argc, char **argv) {
	NPOI_CONTAINER	*d;
	wavelength	*wave;
	int		iFrame;
	int		iSpec;
	int		iChan;
	int		nChan;
	int		iBin;
	int		nBin;
	int		intr;
	int		idx;
	int		nidx;
	wavelength	*wavenumber;
	float		min_wn_bound;
	float		max_wn_bound;
	float		wn_per_pixel;
	float		bn_per_pixel;
	int		**raw_cts;
	int		iWN;
	float		out_cts[HPIXELS * VPIXELS];
	float		TR[6] = {0, 1, 0, 0, 0, 1};
	float		*hist_cts;
	float		maxHist;
	float		hSf;

	if(argc < 8) {
		fprintf(stderr, "Usage: %s INFILE WAVEFILE PIXELDEVICE HISTDEVICE",
				argv[0]);
		fprintf(stderr, " FRAME SPEC INTEGRATION\n");
		exit(1);
	}

	d = new NPOI_CONTAINER(argv[1]);
	if(d->config == NULL) {
		fprintf(stderr, "No data loaded\n");
		exit(1);
	}

	nChan = d->config->nChan;
	nBin = d->config->nBin;

	wave = new wavelength(d->config);
	wave->load(argv[2]);

	iFrame = atoi(argv[5]);
	iSpec = atoi(argv[6]);

	intr = atoi(argv[7]);

	wavenumber = buildWavenumber(wave, iSpec);

	min_wn_bound = findMin(wave->nChan, wavenumber->red[iSpec]);
	max_wn_bound = findMax(wave->nChan, wavenumber->blue[iSpec]);

	wn_per_pixel = (max_wn_bound - min_wn_bound) / (HPIXELS * 1.0);
	bn_per_pixel = (d->config->nBin) / (VPIXELS * 1.0);

	raw_cts = countPhotons(d, iFrame, iSpec, intr);

	for(iBin = 0; iBin < HPIXELS * VPIXELS; iBin++) {
		out_cts[iBin] = 0;
	}

	TR[0] = 0;
	TR[1] = 1.0 / HPIXELS;
	TR[2] = 0;
	TR[3] = 0;
	TR[4] = 0;
	TR[5] = AR / (VPIXELS * 1.0);

	for(iChan = 0; iChan < nChan; iChan++) {
		for(iWN =
			(int)((wavenumber->red[iSpec][iChan] - min_wn_bound) / wn_per_pixel);
			iWN < (wavenumber->blue[iSpec][iChan] - min_wn_bound) / wn_per_pixel;
			iWN++)
		{
			for(iBin = 0; iBin < VPIXELS; iBin++) {
				out_cts[iBin * HPIXELS + iWN] =
					raw_cts[iChan][(int)(iBin * bn_per_pixel)];
			}
		}
	}

	cpgopen(argv[3]);
	cpgenv(0, 1, 0, AR, 1, -1);
	cpglab("Wavelength", "Bin", "Photon Counts");
	cpggray(out_cts, HPIXELS, VPIXELS, 1, HPIXELS, 1, VPIXELS,
			findMax(HPIXELS * VPIXELS, out_cts), 0, TR);
	cpgaxis("N", 0, 0, 0, AR, 1, 65, 4, 0, 0.5, 0, 0, -0.7, 90);

	cpgclos();

	cpgopen(argv[4]);
	cpgenv(0, 1, 0, AR, 1, -1);
	cpglab("Wavelength", "Photons Per Frame", "Photon Counts");

	hist_cts = new float[nChan];

	for(iChan = 0; iChan < nChan; iChan++) {
		hist_cts[iChan] = 0;

		for(iBin = 0; iBin < nBin; iBin++) {
			hist_cts[iChan] += raw_cts[iChan][iBin];
		}

		hist_cts[iChan] /= intr;
	}

	maxHist = findMax(nChan, hist_cts);

	cpgaxis("N", 0, 0, 0, AR, 0, maxHist, 5, 0, 0.5, 0, 0, -0.7, 0);

	hSf = (max_wn_bound - min_wn_bound);

	cpgmove(0, 0);
	for(iChan = 0; iChan < nChan; iChan++) {
		idx = wave->rMap[iSpec][nChan - iChan - 1];
		nidx = wave->rMap[iSpec][nChan - iChan - 2];

		hist_cts[idx] /= maxHist;

		cpgdraw((wavenumber->red[iSpec][idx] - min_wn_bound) / hSf, hist_cts[idx]);
		cpgdraw((wavenumber->blue[iSpec][idx] - min_wn_bound) / hSf, hist_cts[idx]);

		if((iChan != nChan - 1) &&
			(wavenumber->blue[iSpec][idx] !=
			 wavenumber->red[iSpec][nidx]))
		{
			cpgdraw((wavenumber->blue[iSpec][idx] - min_wn_bound) / hSf, 0);
			cpgmove((wavenumber->red[iSpec][nidx] - min_wn_bound) / hSf, 0);
		}
	}

	cpgdraw((wavenumber->blue[iSpec][idx] - min_wn_bound) / hSf, 0);

	cpgclos();

	delete d;

	exit(0);
}

/* ******************************************************************** */
wavelength *buildWavenumber(wavelength *wave, int iSpec) {
	wavelength	*new_wave;
	int		iChan;

	new_wave = new wavelength();

	new_wave->nSpec = wave->nSpec;
	new_wave->nChan = wave->nChan;

	new_wave->buildData();

	for(iChan = 0; iChan < wave->nChan; iChan++) {
		new_wave->red[iSpec][iChan] = 1000 / wave->red[iSpec][iChan];
		new_wave->blue[iSpec][iChan] = 1000 / wave->blue[iSpec][iChan];
		new_wave->center[iSpec][iChan] = 1000 / wave->center[iSpec][iChan];
	}

	return new_wave;
}

/* ******************************************************************** */
int **countPhotons(NPOI_CONTAINER *d, int iFrame, int iSpec, int intr) {
	int	**cts;
	int	iChan;
	int	iBin;
	int	nChan;
	int	nBin;
	int	nFrame;
	int	i;

	nChan = d->config->nChan;
	nBin = d->config->nBin;
	nFrame = d->config->nFrame;

	cts = new (int *)[nChan];
	for(iChan = 0; iChan < nChan; iChan++) {
		cts[iChan] = new int[nBin];

		for(iBin = 0; iBin < nBin; iBin++) {
			cts[iChan][iBin] = 0;

			for(i = iFrame; i < ((iFrame + intr < nFrame) ? iFrame + intr :
					nFrame); i++)
			{
				cts[iChan][iBin] +=
					d->frames[i].photonCounts[iSpec][iChan][iBin];
			}
		}
	}

	return cts;
}
