/* ******************************************************************** */
/* npoi_config.C							*/
/* 2002-08-05  Dan Driscoll						*/
/* ******************************************************************** */

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "npoi_config.h"
#include "npoi_order.h"
#include "npoi_io.h"

/* ******************************************************************** */
/** NPOI_CONFIG::NPOI_CONFIG(nSid, nBas, nSpec, nChan) [all uint32]
   Constructor for one NPOI_CONFIG, it will allocate all necessary
   substructures.							*/
/* ******************************************************************** */
NPOI_CONFIG::NPOI_CONFIG(
		uint32 enSid,
		uint32 enBas,
		uint32 enSpec,
		uint32 enChan)
{
	init(enSid, enBas, enSpec, enChan);
}

/* ******************************************************************** */
/** NPOI_CONFIG::NPOI_CONFIG(nSid, nBas, nFdl, nTriple, nSpec, nChan,
   nBin) [all uint32]							*/
/* ******************************************************************** */
NPOI_CONFIG::NPOI_CONFIG(
		uint32 enSid,
		uint32 enBas,
		uint32 enFdl,
		uint32 enTriple,
		uint32 enSpec,
		uint32 enChan,
		uint32 enBin)
{
	init(enSid, enBas, enSpec, enChan);

	nFdl	= enFdl;
	nTriple	= enTriple;
	nBin	= enBin;
}

/* ******************************************************************** */
/** NPOI_CONFIG::NPOI_CONFIG(FILE *fp)
   Constructor that uses the first few bytes of the file to determine
   how to configure the structure.  File pointer gets rewound.		*/
/* ******************************************************************** */
NPOI_CONFIG::NPOI_CONFIG(FILE *fp) {
	uint32	enSid;
	uint32	enBas;
	uint32	enSpec;
	uint32	enChan;
	int	seek;
	int	offset;
	char	*buf;


	seek = ftell(fp);

	fread(&formatVersion, sizeof(formatVersion), 1, fp);
	if(formatVersion != FORMAT_VERSION) {
		formatVersion = -1;
		fprintf(stderr, "NPOI_CONFIG::load(): Unknown format encountered\n");
		fprintf(stderr, "  Expected %hhu, got %hhu\n", FORMAT_VERSION,
				formatVersion);
		fseek(fp, seek, SEEK_SET);
		return;
	}

	fread(&native, sizeof(native), 1, fp);

	offset =
		sizeof(starID) +
		sizeof(starName) +
		sizeof(date) +
		sizeof(fringeFlag) +

		sizeof(catalogRA) +
		sizeof(catalogDec) +
		sizeof(apparentRA) +
		sizeof(apparentDec) +

		sizeof(firstTS) +
		sizeof(lastTS) +

		sizeof(scanID) +

		sizeof(nFrame);

	/* ************************************************************ */
	/* Use this method because fseek() won't work on .gz files	*/
	/* ************************************************************ */
	buf = new char[offset];
	fread(buf, offset, 1, fp);
	delete [] buf;

	fread(&enSid, sizeof(nSid), 1, fp);
	fread(&enBas, sizeof(nBas), 1, fp);

	offset = sizeof(nFdl) + sizeof(nTriple);
	/* ************************************************************ */
	/* Again							*/
	/* ************************************************************ */
	buf = new char[offset];
	fread(buf, offset, 1, fp);
	delete [] buf;

	fread(&enSpec, sizeof(nSpec), 1, fp);
	fread(&enChan, sizeof(nChan), 1, fp);

	enSid  = hostl(enSid, native);
	enBas  = hostl(enBas, native);
	enSpec = hostl(enSpec, native);
	enChan = hostl(enChan, native);

	fseek(fp, seek, SEEK_SET);

	init(enSid, enBas, enSpec, enChan);
}

/* ******************************************************************** */
/** NPOI_CONFIG::NPOI_CONFIG(const NPOI_CONFIG &enConfig)		*/
/* ******************************************************************** */
NPOI_CONFIG::NPOI_CONFIG(const NPOI_CONFIG &enConfig) {
	OUTPUT_BEAM	*c_ob;
	OUTPUT_BEAM	*e_ob;
	uint32		iSpec;

	if(!enConfig.native) {
		fprintf(stderr,
			"Cannot use non-native copy constructor!\n");
		assert(0);
		exit(1);
	}

	init(enConfig.nSid, enConfig.nBas, enConfig.nSpec,
			enConfig.nChan);

	formatVersion = enConfig.formatVersion;
	native = enConfig.native;

	memcpy(starID, enConfig.starID, DEFSTRLEN);
	memcpy(starName, enConfig.starName, DEFSTRLEN);
	date = enConfig.date;

	fringeFlag = enConfig.fringeFlag;

	catalogRA = enConfig.catalogRA;
	catalogDec = enConfig.catalogDec;
	apparentRA = enConfig.apparentRA;
	apparentDec = enConfig.apparentDec;

	firstTS = enConfig.firstTS;
	lastTS = enConfig.lastTS;

	scanID = enConfig.scanID;

	nFrame = enConfig.nFrame;
	nSid = enConfig.nSid;
	nBas = enConfig.nBas;
	nFdl = enConfig.nFdl;
	nTriple = enConfig.nTriple;
	nSpec = enConfig.nSpec;
	nChan = enConfig.nChan;
	nBin = enConfig.nBin;

	proGainX = enConfig.proGainX;
	proGainY = enConfig.proGainY;
	intGainX = enConfig.intGainX;
	intGainY = enConfig.intGainY;
	derGainX = enConfig.derGainX;
	derGainY = enConfig.derGainY;

	filterGainX = enConfig.filterGainX;
	filterGainY = enConfig.filterGainY;
	lowPassA = enConfig.lowPassA;

	sidGainX = enConfig.sidGainX;
	sidGainY = enConfig.sidGainY;

	latitude = enConfig.latitude;
	longitude = enConfig.longitude;
	altitude = enConfig.altitude;
	earthRadius = enConfig.earthRadius;
	J2 = enConfig.J2;
	tdtMinusUTC = enConfig.tdtMinusUTC;
	ut1MinusUTC = enConfig.ut1MinusUTC;

	instrCohInt = enConfig.instrCohInt;

	beamCombinerID = enConfig.beamCombinerID;
	nPlate = enConfig.nPlate;

	darkTime = enConfig.darkTime;
	darkNumIntegrations = enConfig.darkNumIntegrations;

	for(iSpec = 0; iSpec < nSpec; iSpec++) {
		memcpy(darkCounts[iSpec], enConfig.darkCounts[iSpec],
				nChan * sizeof(int32));
	}

	memcpy(inputBeams, enConfig.inputBeams,
			nSid * sizeof(INPUT_BEAM));

	for(iSpec = 0; iSpec < nSpec; iSpec++) {
		c_ob = &(outputBeams[iSpec]);
		e_ob = &(enConfig.outputBeams[iSpec]);

		c_ob->bcOutputID = e_ob->bcOutputID;
		c_ob->nFringeFreq = e_ob->nFringeFreq;
		memcpy(c_ob->inputPair, e_ob->inputPair, 2 * nBas *
				sizeof(int32));
		memcpy(c_ob->fringeMod, e_ob->fringeMod,
				nBas * sizeof(uint32));

		c_ob->apdArrayID = e_ob->apdArrayID;
		c_ob->nSpecChan = e_ob->nSpecChan;
		
		memcpy(c_ob->specChan, e_ob->specChan,
				nChan * sizeof(SPEC_CHAN));
	}
}

/* ******************************************************************** */
/** NPOI_CONFIG::~NPOI_CONFIG()
   Desctructor for one NPOI_CONFIG.					*/
/* ******************************************************************** */
NPOI_CONFIG::~NPOI_CONFIG() {
	uint32	iSpec;

	for(iSpec = 0; iSpec < hostl(nSpec, native); iSpec++) {
		delete [] darkCounts[iSpec];
	}
	delete [] darkCounts;

	delete [] inputBeams;

	for(iSpec = 0; iSpec < hostl(nSpec, native); iSpec++) {
		delete [] outputBeams[iSpec].inputPair;
		delete [] outputBeams[iSpec].fringeMod;
		delete [] outputBeams[iSpec].specChan;
	}
	delete [] outputBeams;

	darkCounts = NULL;
	inputBeams = NULL;
	outputBeams = NULL;
}

/* ******************************************************************** */
/** NPOI_CONFIG::swap()
   Swap byte order for all elements within memory.			*/
/* ******************************************************************** */
void NPOI_CONFIG::swap() {
	uint32		iSid;
	uint32		iBas;
	uint32		iSpec;
	uint32		iChan;
	OUTPUT_BEAM	*c_ob;
	INPUT_BEAM	*c_ib;

	swapl(&date, native);
	swapd(&catalogRA, native);
	swapd(&catalogDec, native);
	swapd(&apparentRA, native);
	swapd(&apparentDec, native);

	swapl(&firstTS, native);
	swapl(&lastTS, native);

	swapl(&scanID, native);

	swapd(&proGainX, native);
	swapd(&proGainY, native);
	swapd(&intGainX, native);
	swapd(&intGainY, native);
	swapd(&derGainX, native);
	swapd(&derGainY, native);
	swapd(&filterGainX, native);
	swapd(&filterGainY, native);
	swapd(&lowPassA, native);
	swapd(&sidGainX, native);
	swapd(&sidGainY, native);

	swapd(&latitude, native);
	swapd(&longitude, native);
	swapd(&altitude, native);
	swapd(&earthRadius, native);
	swapd(&J2, native);
	swapd(&tdtMinusUTC, native);
	swapd(&ut1MinusUTC, native);
	swapd(&instrCohInt, native);

	swapl(&beamCombinerID, native);
	swapl(&nPlate, native);
	swapl(&darkTime, native);
	swapl(&darkNumIntegrations, native);

	for(iSpec = 0; iSpec < hostl(nSpec, native); iSpec++) {
		c_ob = &(outputBeams[iSpec]);

		for(iChan = 0; iChan < hostl(nChan, native); iChan++) {
			swapl(&(darkCounts[iSpec][iChan]), native);
			swapd(&(c_ob->specChan[iChan].wavelength), native);
			swapd(&(c_ob->specChan[iChan].wavelengthErr), native);
			swapd(&(c_ob->specChan[iChan].chanWidth), native);
			swapd(&(c_ob->specChan[iChan].chanWidthErr), native);
		}

		swapl(&(c_ob->bcOutputID), native);
		swapl(&(c_ob->nFringeFreq), native);

		for(iBas = 0; iBas < hostl(nBas, native); iBas++) {
			swapl(&(c_ob->inputPair[iBas][0]), native);
			swapl(&(c_ob->inputPair[iBas][1]), native);
			swapl(&(c_ob->fringeMod[iBas]), native);
		}

		swapl(&(c_ob->apdArrayID), native);
		swapl(&(c_ob->nSpecChan), native);
	}

	for(iSid = 0; iSid < hostl(nSid, native); iSid++) {
		c_ib = &(inputBeams[iSid]);

		swapd(&(c_ib->stationCoord[0]), native);
		swapd(&(c_ib->stationCoord[1]), native);
		swapd(&(c_ib->stationCoord[2]), native);
		swapd(&(c_ib->stationCoord[3]), native);

		swapl(&(c_ib->sidID), native);
		swapl(&(c_ib->fdlTankID), native);

		swapd(&(c_ib->stroke), native);

		swapl(&(c_ib->quadCellID), native);

		swapl(&(c_ib->bcInputID), native);
	}

	swapl(&nFrame, native);
	swapl(&nSid, native);
	swapl(&nBas, native);
	swapl(&nFdl, native);
	swapl(&nTriple, native);
	swapl(&nSpec, native);
	swapl(&nChan, native);
	swapl(&nBin, native);

	native = !native;
}

/* ******************************************************************** */
/** NPOI_CONFIG::save(FILE *fp)
   Save this config into the file at this point.
   File pointer stays advanced.						*/
/* ******************************************************************** */
void NPOI_CONFIG::save(FILE *fp) {
	uint32		iSpec;
	OUTPUT_BEAM	*c_ob;

	if(native) {
		swap();
	}

	fwrite(&formatVersion, sizeof(formatVersion), 1, fp);
	fwrite(&native, sizeof(native), 1, fp);

	fwrite(starID, sizeof(starID), 1, fp);
	fwrite(starName, sizeof(starName), 1, fp);
	fwritel(&date, fp);

	fwrite(&fringeFlag, sizeof(fringeFlag), 1, fp);

	fwrited(&catalogRA, fp);
	fwrited(&catalogDec, fp);
	fwrited(&apparentRA, fp);
	fwrited(&apparentDec, fp);

	fwritel(&firstTS, fp);
	fwritel(&lastTS, fp);

	fwritel(&scanID, fp);
	fwritel(&nFrame, fp);
	fwritel(&nSid, fp);
	fwritel(&nBas, fp);
	fwritel(&nFdl, fp);
	fwritel(&nTriple, fp);
	fwritel(&nSpec, fp);
	fwritel(&nChan, fp);
	fwritel(&nBin, fp);

	fwrited(&proGainX, fp);
	fwrited(&proGainY, fp);
	fwrited(&intGainX, fp);
	fwrited(&intGainY, fp);
	fwrited(&derGainX, fp);
	fwrited(&derGainY, fp);
	fwrited(&filterGainX, fp);
	fwrited(&filterGainY, fp);
	fwrited(&lowPassA, fp);
	fwrited(&sidGainX, fp);
	fwrited(&sidGainY, fp);

	fwrited(&latitude, fp);
	fwrited(&longitude, fp);
	fwrited(&altitude, fp);
	fwrited(&earthRadius, fp);
	fwrited(&J2, fp);
	fwrited(&tdtMinusUTC, fp);
	fwrited(&ut1MinusUTC, fp);
	
	fwrited(&instrCohInt, fp);

	fwritel(&beamCombinerID, fp);
	fwritel(&nPlate, fp);
	fwritel(&darkTime, fp);
	fwritel(&darkNumIntegrations, fp);

	for(iSpec = 0; iSpec < hostl(nSpec, native); iSpec++) {
		fwrite(darkCounts[iSpec], sizeof(int32), hostl(nChan, native), fp);
	}

	fwrite(inputBeams, sizeof(INPUT_BEAM), hostl(nSid, native), fp);

	for(iSpec = 0; iSpec < hostl(nSpec, native); iSpec++) {
		c_ob = &(outputBeams[iSpec]);

		fwritel(&(c_ob->bcOutputID), fp);
		fwritel(&(c_ob->nFringeFreq), fp);

		fwritel(&(c_ob->apdArrayID), fp);
		fwritel(&(c_ob->nSpecChan), fp);

		fwrite(c_ob->inputPair, sizeof(int32[2]), hostl(nBas, native), fp);
		fwrite(c_ob->fringeMod, sizeof(uint32), hostl(nBas, native), fp);

		fwrite(c_ob->specChan, sizeof(SPEC_CHAN), hostl(nChan, native), fp);
	}
}

/* ******************************************************************** */
/** NPOI_CONFIG::load(FILE *fp)
   Load this config from the file at this point.
   File pointer stays advanced						*/
/* ******************************************************************** */
void NPOI_CONFIG::load(FILE *fp) {
	uint32		iSpec;
	OUTPUT_BEAM	*c_ob;

	fread(&formatVersion, sizeof(formatVersion), 1, fp);
	if(formatVersion != FORMAT_VERSION) {
		formatVersion = -1;
		fprintf(stderr, "NPOI_CONFIG::load(): Unknown format encountered\n");
		return;
	}

	fread(&native, sizeof(native), 1, fp);

	fread(starID, sizeof(starID), 1, fp);
	fread(starName, sizeof(starName), 1, fp);
	freadl(&date, fp);

	fread(&fringeFlag, sizeof(fringeFlag), 1, fp);

	freadd(&catalogRA, fp);
	freadd(&catalogDec, fp);
	freadd(&apparentRA, fp);
	freadd(&apparentDec, fp);

	freadl(&firstTS, fp);
	freadl(&lastTS, fp);

	freadl(&scanID, fp);
	freadl(&nFrame, fp);
	freadl(&nSid, fp);
	freadl(&nBas, fp);
	freadl(&nFdl, fp);
	freadl(&nTriple, fp);
	freadl(&nSpec, fp);
	freadl(&nChan, fp);
	freadl(&nBin, fp);

	freadd(&proGainX, fp);
	freadd(&proGainY, fp);
	freadd(&intGainX, fp);
	freadd(&intGainY, fp);
	freadd(&derGainX, fp);
	freadd(&derGainY, fp);
	freadd(&filterGainX, fp);
	freadd(&filterGainY, fp);
	freadd(&lowPassA, fp);
	freadd(&sidGainX, fp);
	freadd(&sidGainY, fp);

	freadd(&latitude, fp);
	freadd(&longitude, fp);
	freadd(&altitude, fp);
	freadd(&earthRadius, fp);
	freadd(&J2, fp);
	freadd(&tdtMinusUTC, fp);
	freadd(&ut1MinusUTC, fp);
	
	freadd(&instrCohInt, fp);

	freadl(&beamCombinerID, fp);
	freadl(&nPlate, fp);
	freadl(&darkTime, fp);
	freadl(&darkNumIntegrations, fp);

	for(iSpec = 0; iSpec < hostl(nSpec, native); iSpec++) {
		fread(darkCounts[iSpec], sizeof(int32), hostl(nChan, native), fp);
	}

	fread(inputBeams, sizeof(INPUT_BEAM), hostl(nSid, native), fp);

	for(iSpec = 0; iSpec < hostl(nSpec, native); iSpec++) {
		c_ob = &(outputBeams[iSpec]);

		freadl(&(c_ob->bcOutputID), fp);
		freadl(&(c_ob->nFringeFreq), fp);

		freadl(&(c_ob->apdArrayID), fp);
		freadl(&(c_ob->nSpecChan), fp);

		fread(c_ob->inputPair, sizeof(int32[2]), hostl(nBas, native), fp);
		fread(c_ob->fringeMod, sizeof(uint32), hostl(nBas, native), fp);

		fread(c_ob->specChan, sizeof(SPEC_CHAN), hostl(nChan, native), fp);
	}

	if(!native) {
		swap();
	}
}

/* ******************************************************************** */
/** NPOI_CONFIG::init()							*/
/* ******************************************************************** */
void NPOI_CONFIG::init(
		uint32 enSid,
		uint32 enBas,
		uint32 enSpec,
		uint32 enChan)
{
	uint32	iSpec;

	nSid  = enSid;
	nBas  = enBas;
	nSpec = enSpec;
	nChan = enChan;

	/* ************************************************************ */
	/** Set pertinent flags						*/
	/* ************************************************************ */
	formatVersion = FORMAT_VERSION;
	native = 1;

	/* ************************************************************ */
	/** Zero all other variables					*/
	/* ************************************************************ */
	starID[0] = '\0';
	starName[0] = '\0';

	date = 0;

	fringeFlag = 0;

	catalogRA = catalogDec = 0;
	apparentRA = apparentDec = 0;

	firstTS = lastTS = 0;

	scanID = nFrame = nFdl = nTriple = nBin = 0;

	proGainX = proGainY = intGainX = intGainY = 0;
	derGainX = derGainY = filterGainX = filterGainY = 0;
	lowPassA = sidGainX = sidGainY = 0;

	latitude = longitude = altitude = earthRadius = J2 = 0;
	tdtMinusUTC = ut1MinusUTC = instrCohInt = 0;

	beamCombinerID = nPlate = 0;

	darkTime = darkNumIntegrations = 0;

	/* ************************************************************ */
	/** Allocate and zero structure space				*/
	/* ************************************************************ */
	darkCounts = new (int32 *)[nSpec];
	for(iSpec = 0; iSpec < nSpec; iSpec++) {
		darkCounts[iSpec] = new int32[nChan];
		memset(darkCounts[iSpec], 0, nChan * sizeof(int32));
	}

	inputBeams = new INPUT_BEAM[nSid];
	memset(inputBeams, 0, nSid * sizeof(INPUT_BEAM));

	outputBeams = new OUTPUT_BEAM[nSpec];
	memset(outputBeams, 0, nSpec * sizeof(OUTPUT_BEAM));

	for(iSpec = 0; iSpec < nSpec; iSpec++) {
		outputBeams[iSpec].inputPair = new int32[nBas][2];
		memset(outputBeams[iSpec].inputPair, 0,
				nBas * sizeof(int32[2]));

		outputBeams[iSpec].fringeMod = new uint32[nBas];
		memset(outputBeams[iSpec].fringeMod, 0,
				nBas * sizeof(uint32));

		outputBeams[iSpec].specChan = new SPEC_CHAN[nChan];
		memset(outputBeams[iSpec].specChan, 0,
				nChan * sizeof(SPEC_CHAN));
	}
}

/* ******************************************************************** */
/** NPOI_CONFIG::buildFilename(char *outdir, char *suffix)		*/
/* ******************************************************************** */
char *NPOI_CONFIG::buildFilename(char *outdir, char *suffix) {
	char	*buf;
	uint32	tFirstTS;
	char	tSuffix[1];

	/* ************************************************************ */
	/* Workaround to allow calling buildFilename(outdir, NULL)	*/
	/* ************************************************************ */
	if(suffix == NULL) {
		suffix = tSuffix;
		suffix[0] = '\0';
	}

	tFirstTS = hostl(firstTS, native);

	buf = new char[strlen(outdir) + strlen(starID) + 1 +
			2 + 1 + 2 + 1 + 2 + 1 + strlen(suffix) + 1];

	sprintf(buf, "%s/%s-%2.2luh%2.2lum%2.2lus%s", outdir,
		starID,
		(tFirstTS / (3600 * 1000)),
		(tFirstTS / (60 * 1000)) % 60,
		(tFirstTS / 1000) % 60,
		suffix);

	return buf;
}
