/* ******************************************************************** */
/* npoi_container.C							*/
/* 2002-08-05  Dan Driscoll						*/
/* ******************************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "npoi_container.h"
#include "npoi_order.h"

/* ******************************************************************** */
/** NPOI_CONTAINER::NPOI_CONTAINER()					*/
/* ******************************************************************** */
NPOI_CONTAINER::NPOI_CONTAINER() {
	config = NULL;
}

/* ******************************************************************** */
/** NPOI_CONTAINER::NPOI_CONTAINER(FILE *fp)				*/
/* ******************************************************************** */
NPOI_CONTAINER::NPOI_CONTAINER(FILE *fp) {
	config = NULL;

	load(fp);
}

/* ******************************************************************** */
/** NPOI_CONTAINER::NPOI_CONTAINER(char *filename)			*/
/* ******************************************************************** */
NPOI_CONTAINER::NPOI_CONTAINER(char *filename) {
	config = NULL;

	load(filename);
}

/* ******************************************************************** */
/** NPOI_CONTAINER::NPOI_CONTAINER(NPOI_CONFIG *enConfig,
   NPOI_FRAME *enFrame)							*/
/* ******************************************************************** */
NPOI_CONTAINER::NPOI_CONTAINER(NPOI_CONFIG *enConfig, NPOI_FRAME *enFrame)
{
	config = enConfig;
	frames = enFrame;
}

/* ******************************************************************** */
/** NPOI_CONTAINER::NPOI_CONTAINER(NPOI_CONFIG *enConfig)		*/
/* ******************************************************************** */
NPOI_CONTAINER::NPOI_CONTAINER(NPOI_CONFIG *enConfig) {
	config = new NPOI_CONFIG(*enConfig);
}

/* ******************************************************************** */
/** NPOI_CONTAINER::~NPOI_CONTAINER()					*/
/* ******************************************************************** */
NPOI_CONTAINER::~NPOI_CONTAINER() {
	if(frames) {
		uint32	iFrame;
		for(iFrame = 0; iFrame < hostl(config->nFrame,
					config->native); iFrame++)
		{
			frames[iFrame].deleteArrays();
		}

		delete [] frames;
		frames = NULL;
	}

	if(config) {
		delete config;
		config = NULL;
	}
}

/* ******************************************************************** */
/* NPOI_CONTAINER::save(FILE *fp)					*/
/* ******************************************************************** */
void NPOI_CONTAINER::save(FILE *fp) {
	uint32	iFrame;

	config->save(fp);

	for(iFrame = 0; iFrame < hostl(config->nFrame, config->native); iFrame++) {
		frames[iFrame].save(fp);
	}
}

/* ******************************************************************** */
/* NPOI_CONTAINER::load(FILE *fp)					*/
/* ******************************************************************** */
void NPOI_CONTAINER::load(FILE *fp) {
	long int	iFrame;
	long int	nFrame;

	if(!config) {
		config = new NPOI_CONFIG(fp);

		config->load(fp);
	}

	nFrame = hostl(config->nFrame, config->native);

	frames = new NPOI_FRAME[nFrame](config);

	for(iFrame = 0; iFrame < nFrame; iFrame++) {
		frames[iFrame].load(fp);
	}
}

/* ******************************************************************** */
/* NPOI_CONTAINER::save(char *filename)					*/
/* ******************************************************************** */
void NPOI_CONTAINER::save(char *filename) {
	FILE	*fp;
	char	buf[BUFSIZ];

	if(strstr(filename, ".gz") == filename + strlen(filename) - 3) {
		sprintf(buf, "gzip > %s", filename);
		fp = popen(buf, "w");
	} else {
		fp = fopen(filename, "w");
	}

	if(fp == NULL) {
		fprintf(stderr, "Could not open file\n");
		return;
	}

	save(fp);

	fclose(fp);
}

/* ******************************************************************** */
/* NPOI_CONTAINER::load(char *filename)					*/
/* ******************************************************************** */
void NPOI_CONTAINER::load(char *filename) {
	FILE	*fp;
	char	buf[BUFSIZ];
	char	gzFlag;

	if(strstr(filename, ".gz") == filename + strlen(filename) - 3) {
		gzFlag = 1;
	} else {
		gzFlag = 0;
	}

	if(gzFlag) {
		sprintf(buf, "zcat %s", filename);
		fp = popen(buf, "r");

		/* **************************************************** */
		/** WARNING!  HACK BELOW!
		   It turns out that you can't really rewind a popen
		   file descriptor.  So...this is a cheat to allow us
		   to read a config, rewind, then read again.		*/
		/* **************************************************** */

		config = new NPOI_CONFIG(fp);

		pclose(fp);
		fp = popen(buf, "r");

		config->load(fp);
	} else {
		fp = fopen(filename, "r");
	}

	if(fp == NULL) {
		fprintf(stderr, "Could not open file\n");
		return;
	}

	load(fp);

	if(gzFlag) {
		pclose(fp);
	} else {
		fclose(fp);
	}
}

/* ******************************************************************** */
/* NPOI_CONTAINER::sortFrames						*/
/* ******************************************************************** */
void NPOI_CONTAINER::sortFrames() {
	if(!config->native) {
		fprintf(stderr, "Cannot sort, not in native format\n");
		return;
	}

	if(!config->nFrame || !frames) {
		fprintf(stderr, "Cannot sort, No data\n");
		return;
	}

	qsort(frames, hostl(config->nFrame, config->native),
			sizeof(NPOI_FRAME), sortFrameFunc);

	updateTS();
}

/* ******************************************************************** */
/* NPOI_CONTAINER::updateTS						*/
/* ******************************************************************** */
void NPOI_CONTAINER::updateTS() {
	uint32	iFrame;
	uint32	maxTS = 0;
	uint32	minTS = 0;

	for(iFrame = 0; iFrame < config->nFrame; iFrame++) {
		if(iFrame == 0 || frames[iFrame].fringeTime < minTS) {
			minTS = frames[iFrame].fringeTime;
		}

		if(iFrame == 0 || frames[iFrame].fringeTime > maxTS) {
			maxTS = frames[iFrame].fringeTime;
		}
	}

	config->firstTS = minTS;
	config->lastTS = maxTS;
}

/* ******************************************************************** */
/* sortFrameFunc							*/
/* ******************************************************************** */
int sortFrameFunc(const void *p0, const void *p1) {
	NPOI_FRAME *q0 = (NPOI_FRAME *)p0;
	NPOI_FRAME *q1 = (NPOI_FRAME *)p1;

	if(q0->fringeTime == q1->fringeTime) {
		return 0;
	} else if(q0->fringeTime < q1->fringeTime) {
		return -1;
	} else {
		return 1;
	}
}
