/* ******************************************************************** */
/* npoi_frame.C								*/
/* 2002-08-02  Dan Driscoll  driscoll@nrl.navy.mil			*/
/* ******************************************************************** */

#include <stdio.h>
#include <iostream.h>
#include <string.h>
#include "packetdefs.h"
#include "npoi_frame.h"
#include "npoi_config.h"
#include "npoi_order.h"
#include "npoi_io.h"

/* ******************************************************************** */
/** NPOI_FRAME::NPOI_FRAME()						*/
/* ******************************************************************** */
NPOI_FRAME::NPOI_FRAME() {
	nSid = nFdl = nSpec = nChan = nBin = 0;

	formatVersion = FORMAT_VERSION;
	native = 1;

	fringeTime = fdlTime = natTime = 0;

	photonCounts = NULL;
	fringeConDelayEstimate = NULL;
	fdlPosition = NULL;
	natCounts = NULL;
}

/* ******************************************************************** */
/** NPOI_FRAME::NPOI_FRAME(nSid, nFdl, nSpec, nChan, nBin) [all uint32]
   Constructor for one NPOI_FRAME, it will allocate all necessary
   substructures.							*/
/* ******************************************************************** */
NPOI_FRAME::NPOI_FRAME(
		uint32 enSid,
		uint32 enFdl,
		uint32 enSpec,
		uint32 enChan,
		uint32 enBin)
{
	uint32	iSpec;
	uint32	iChan;

	nSid  = enSid;
	nFdl  = enFdl;
	nSpec = enSpec;
	nChan = enChan;
	nBin  = enBin;

	formatVersion = FORMAT_VERSION;
	native = 1;

	fringeTime = fdlTime = natTime = 0;

	photonCounts = new (uint8 **)[nSpec];
	for(iSpec = 0; iSpec < nSpec; iSpec++) {
		photonCounts[iSpec] = new (uint8 *)[nChan];
		for(iChan = 0; iChan < nChan; iChan++) {
			photonCounts[iSpec][iChan] = new uint8[nBin];
			memset(photonCounts[iSpec][iChan], 0,
				nBin * sizeof(uint8));
		}
	}

	fringeConDelayEstimate = new FRINGE_CON_DELAY_ESTIMATE[nSpec];
	memset(fringeConDelayEstimate, 0,
			nSpec * sizeof(FRINGE_CON_DELAY_ESTIMATE));

	fdlPosition = new FDL_POSITION[nFdl];
	memset(fdlPosition, 0, nFdl * sizeof(FDL_POSITION));

	natCounts = new NAT_COUNTS[nSid];
	memset(natCounts, 0, nSid * sizeof(NAT_COUNTS));
}

/* ******************************************************************** */
/** NPOI_FRAME::NPOI_FRAME(NPOI_CONFIG *config)
   Constructor for one NPOI_FRAME, it will allocate all necessary
   structures.								*/
/* ******************************************************************** */
NPOI_FRAME::NPOI_FRAME(NPOI_CONFIG *config) {
	uint32	iSpec;
	uint32	iChan;

	nSid  = hostl(config->nSid, config->native);
	nFdl  = hostl(config->nFdl, config->native);
	nSpec = hostl(config->nSpec, config->native);
	nChan = hostl(config->nChan, config->native);
	nBin  = hostl(config->nBin, config->native);

	formatVersion = FORMAT_VERSION;
	native = 1;

	fringeTime = fdlTime = natTime = 0;

	photonCounts = new (uint8 **)[nSpec];
	for(iSpec = 0; iSpec < nSpec; iSpec++) {
		photonCounts[iSpec] = new (uint8 *)[nChan];
		for(iChan = 0; iChan < nChan; iChan++) {
			photonCounts[iSpec][iChan] = new uint8[nBin];
			memset(photonCounts[iSpec][iChan], 0,
				nBin * sizeof(uint8));
		}
	}

	fringeConDelayEstimate = new FRINGE_CON_DELAY_ESTIMATE[nSpec];
	memset(fringeConDelayEstimate, 0,
			nSpec * sizeof(FRINGE_CON_DELAY_ESTIMATE));

	fdlPosition = new FDL_POSITION[nFdl];
	memset(fdlPosition, 0, nFdl * sizeof(FDL_POSITION));

	natCounts = new NAT_COUNTS[nSid];
	memset(natCounts, 0, nSid * sizeof(NAT_COUNTS));
}

/* ******************************************************************** */
/** NPOI_FRAME::NPOI_FRAME(const NPOI_FRAME &old)
   Copies all bits of the old structure then allocates new memory for
   the rest.								*/
/* ******************************************************************** */
NPOI_FRAME::NPOI_FRAME(const NPOI_FRAME &old) {
	uint32	iSpec;
	uint32	iChan;

	nSid = old.nSid;
	nFdl = old.nFdl;
	nSpec = old.nSpec;
	nChan = old.nChan;
	nBin = old.nBin;

	native = old.native;

	fringeTime = old.fringeTime;
	fdlTime = old.fdlTime;
	natTime = old.natTime;

	photonCounts = new (uint8 **)[hostl(nSpec, native)];
	for(iSpec = 0; iSpec < hostl(nSpec, native); iSpec++) {
		photonCounts[iSpec] = new (uint8 *)[hostl(nChan, native)];
		for(iChan = 0; iChan < hostl(nChan, native); iChan++) {
			photonCounts[iSpec][iChan] = new uint8[hostl(nBin, native)];
			memcpy(photonCounts[iSpec][iChan], old.photonCounts[iSpec][iChan],
					hostl(nBin, native) * sizeof(uint8));
		}
	}

	fringeConDelayEstimate = new FRINGE_CON_DELAY_ESTIMATE[hostl(nSpec, native)];
	memcpy(fringeConDelayEstimate, old.fringeConDelayEstimate,
			hostl(nSpec, native) * sizeof(FRINGE_CON_DELAY_ESTIMATE));

	fdlPosition = new FDL_POSITION[hostl(nFdl, native)];
	memcpy(fdlPosition, old.fdlPosition, hostl(nFdl, native) * sizeof(FDL_POSITION));

	natCounts = new NAT_COUNTS[hostl(nSid, native)];
	memcpy(natCounts, old.natCounts, hostl(nSid, native) * sizeof(NAT_COUNTS));
}

/* ******************************************************************** */
/** NPOI_FRAME::~NPOI_FRAME()
   Destructor for one NPOI_FRAME.					*/
/* ******************************************************************** */
NPOI_FRAME::~NPOI_FRAME() {
	deleteArrays();
}

/* ******************************************************************** */
/** NPOI_FRAME::deleteArrays()
   Delete all arrays under this frame.					*/
/* ******************************************************************** */
void NPOI_FRAME::deleteArrays() {
	uint32	iSpec;
	uint32	iChan;

	if(photonCounts) {
		for(iSpec = 0; iSpec < nSpec; iSpec++) {
			for(iChan = 0; iChan < nChan; iChan++) {
				if(iChan == nChan || iSpec == nSpec) {
					fprintf(stderr, "ERR!\n");
					fflush(stderr);
				}
				delete [] photonCounts[iSpec][iChan];
			}
			delete [] photonCounts[iSpec];
		}
		delete [] photonCounts;
	}

	if(fringeConDelayEstimate) delete [] fringeConDelayEstimate;
	if(fdlPosition) delete [] fdlPosition;
	if(natCounts) delete [] natCounts;

	unlink();
}

/* ******************************************************************** */
/** NPOI_FRAME::swap()
   Swap byte order for all elements within the file.			*/
/* ******************************************************************** */
void NPOI_FRAME::swap() {
	uint32	iSid;
	uint32	iFdl;
	uint32	iSpec;

	swapl(&fringeTime, native);
	swapl(&fdlTime, native);
	swapl(&natTime, native);

	for(iSpec = 0; iSpec < nSpec; iSpec++) {
		swaps(&(fringeConDelayEstimate[iSpec].val), native);
		swaps(&(fringeConDelayEstimate[iSpec].err), native);
	}

	for(iFdl = 0; iFdl < nFdl; iFdl++) {
		swapl(&(fdlPosition[iFdl].laserLo), native);
		swapl(&(fdlPosition[iFdl].laserHi), native);
		swapl(&(fdlPosition[iFdl].jitter), native);
		swapd(&(fdlPosition[iFdl].adfLaser), native);
	}

	for(iSid = 0; iSid < nSid; iSid++) {
		swaps(&(natCounts[iSid].quadA), native);
		swaps(&(natCounts[iSid].quadB), native);
		swaps(&(natCounts[iSid].quadC), native);
		swaps(&(natCounts[iSid].quadD), native);
		swaps(&(natCounts[iSid].deltaX), native);
		swaps(&(natCounts[iSid].deltaY), native);
	}

	native = !native;
}

/* ******************************************************************** */
/** NPOI_FRAME::save(FILE *fp)
   Save this frame into the file at this point.
   File pointer stays advanced.						*/
/* ******************************************************************** */
void NPOI_FRAME::save(FILE *fp) {
	uint32	iSpec;
	uint32	iChan;

	if(native) {
		swap();
	}

	fwrite(&formatVersion, sizeof(formatVersion), 1, fp);
	fwrite(&native, sizeof(native), 1, fp);

	fwritel(&fringeTime, fp);
	fwritel(&fdlTime, fp);
	fwritel(&natTime, fp);

	for(iSpec = 0; iSpec < nSpec; iSpec++) {
		for(iChan = 0; iChan < nChan; iChan++) {
			fwrite(photonCounts[iSpec][iChan], sizeof(uint8), nBin, fp);
		}
	}

	fwrite(fringeConDelayEstimate,
			sizeof(FRINGE_CON_DELAY_ESTIMATE), nSpec, fp);

	fwrite(fdlPosition, sizeof(FDL_POSITION), nFdl, fp);
	fwrite(natCounts, sizeof(NAT_COUNTS), nSid, fp);
}

/* ******************************************************************** */
/** NPOI_FRAME::load(FILE *fp)
   Load this frame from the file at this point.
   File pointer stays advanced.						*/
/* ******************************************************************** */
void NPOI_FRAME::load(FILE *fp) {
	uint32	iSpec;
	uint32	iChan;

	fread(&formatVersion, sizeof(formatVersion), 1, fp);
	if(formatVersion != FORMAT_VERSION) {
		formatVersion = -1;
		fprintf(stderr, "NPOI_FRAME::load(): Unknown format encountered\n");
		fprintf(stderr, "  Expected %hhu, got %hhu\n", FORMAT_VERSION,
				formatVersion);
		return;
	}

	fread(&native, sizeof(native), 1, fp);

	freadl(&fringeTime, fp);
	freadl(&fdlTime, fp);
	freadl(&natTime, fp);

	for(iSpec = 0; iSpec < nSpec; iSpec++) {
		for(iChan = 0; iChan < nChan; iChan++) {
			fread(photonCounts[iSpec][iChan], sizeof(uint8), nBin, fp);
		}
	}

	fread(fringeConDelayEstimate,
			sizeof(FRINGE_CON_DELAY_ESTIMATE), nSpec, fp);

	fread(fdlPosition, sizeof(FDL_POSITION), nFdl, fp);
	fread(natCounts, sizeof(NAT_COUNTS), nSid, fp);

	if(!native) {
		swap();
	}
}

/* ******************************************************************** */
/** NPOI_FRAME::calculateADFLaser(uint32 laserHi, uint32 laserLo)	*/
/* ******************************************************************** */
double NPOI_FRAME::calculateADFLaser(uint32 laserHi, uint32 laserLo) {
	return	(0x000000FF - laserHi) * FDLHI_MET_UNITS +
		(0xFFFFFFFF - laserLo) * FDLLO_MET_UNITS;
}

/* ******************************************************************** */
/** NPOI_FRAME::calculateADFLasers()					*/
/* ******************************************************************** */
void NPOI_FRAME::calculateADFLasers() {
	uint32	iFdl;

	for(iFdl = 0; iFdl < nFdl; iFdl++) {
		fdlPosition[iFdl].adfLaser =
			calculateADFLaser(
				fdlPosition[iFdl].laserHi,
				fdlPosition[iFdl].laserLo);
	}
}

/* ******************************************************************** */
/** NPOI_FRAME::operator < (NPOI_FRAME &rhs)				*/
/* ******************************************************************** */
bool NPOI_FRAME::operator < (const NPOI_FRAME &rhs) const {
	return fringeTime < rhs.fringeTime;
}

/* ******************************************************************** */
/** NPOI_FRAME::buildTimeStamp()					*/
/* ******************************************************************** */
char *NPOI_FRAME::buildTimeStamp() {
	char	*buf;

	buf = new char[16];

	sprintf(buf, "%2.2luh%2.2lum%2.2lus%4.4luus",
			(fringeTime / (3600 * 1000)),
			(fringeTime / (60 * 1000)) % 60,
			(fringeTime / 1000) % 60,
			fringeTime % 1000);

	return buf;
}

/* ******************************************************************** */
/** NPOI_FRAME::unlink()
   Delete all pointers so that a memcpy of this object takes
   responsibility of them.						*/
/* ******************************************************************** */
void NPOI_FRAME::unlink() {
	photonCounts = NULL;
	fringeConDelayEstimate = NULL;
	fdlPosition = NULL;
	natCounts = NULL;
}
