/* ******************************************************************** */
/* npoi_import.C							*/
/* 2002-08-06  Dan Driscoll						*/
/* ******************************************************************** */

#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <libgen.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "npoi_container.h"
#include "npoi_import.h"
#include "npoi_import_support.h"
#include "datacon.h"
#include "packetdefs.h"
#include "packettype.h"
#include "sys_config.h"

extern "C" {
#include "pktprint.h"
#include "swap.h"
}

/* ******************************************************************** */
/* importFringe								*/
/* ******************************************************************** */
void importFringe(char *indir, char *outdir) {
	FILE			**fringe_fd;
	char			**fringe_names;
	int			fringe_n;
	void			*data;
	NPOI_CONTAINER		*c_d = NULL;
	NPOI_CONFIG		*sysconfig;
	int			c_f;
	DATAconHeader		t_header;
	int			i;
	int			cFrame;
	int			*scans;
	int			**nFrames;
	int			iScan;

	sysconfig = initConfig();

	searchDirectoryForData(indir, FS_FRINGE, &fringe_names,
			&fringe_fd, &fringe_n);

	countFramesInFiles(fringe_fd, &nFrames, &scans, fringe_n);

	for(c_f = 0; c_f < fringe_n; c_f++) {
		iScan = 0;

		fprintf(stderr, "File: %s\n", fringe_names[c_f]);

		while(fread(&t_header, sizeof(DATAconHeader), 1,
					fringe_fd[c_f]))
		{
			/* ******************************************** */
			/* Fix packet					*/
			/* ******************************************** */
			t_header.dchType	= swaplong(t_header.dchType);
			t_header.dchLength	= swaplong(t_header.dchLength);
			t_header.dchTime	= swaplong(t_header.dchTime);
			t_header.dchDay		= swaplong(t_header.dchDay);

			print_packet(t_header, 0);

			if(t_header.dchLength < sizeof(DATAconHeader)) {
				fprintf(stderr, "Data length [%lu] too short\n",
						t_header.dchLength);

				/* ************************************ */
				/* If we're not done with files, save	*/
				/* current star and go to next file	*/
				/* ************************************ */
				if(c_f < fringe_n - 1) {
					if(c_d) {
						saveStar(c_d, outdir);

						fprintf(stderr,
						  "Saving current star\n");

						delete c_d;

						c_d = NULL;
					}

					fprintf(stderr,
						"Skipping rest of file\n");
					c_f++;

					continue;
				} else {
					fprintf(stderr,
						"Ending prematurely\n");

					return;
				}
			}

			/* ******************************************** */
			/* Read data					*/
			/* ******************************************** */
			data = (void *)malloc(t_header.dchLength -
					sizeof(DATAconHeader));

			fread(data, t_header.dchLength - sizeof(DATAconHeader),
					1, fringe_fd[c_f]);

			/* ******************************************** */
			/* Check to build new star			*/
			/* ******************************************** */
			if((PACKET_TYPE_ENUM)(t_header.dchType) ==
					pktSCAN_START_VERSION_1 ||
			   (PACKET_TYPE_ENUM)(t_header.dchType) ==
			   		pktSCAN_START_VERSION_2)
			{
				/* ************************************ */
				/* Save old star ...			*/
				/* ************************************ */
				if(c_d) {
					saveStar(c_d, outdir);

					delete c_d;
				}

				c_d = new NPOI_CONTAINER(sysconfig);
				c_d->config->nFrame = nFrames[c_f][iScan];
				c_d->frames =
					new NPOI_FRAME[nFrames[c_f][iScan]](sysconfig);
				iScan++;
				cFrame = -VAR_NUM_REC;
			}

			processPacket(&t_header, data, c_d, sysconfig, &cFrame);

			free(data);
		}
	}

	if(c_d) {
		saveStar(c_d, outdir);

		delete c_d;
	}

	for(i = 0; i < fringe_n; i++) {
		fclose(fringe_fd[i]);
		free(fringe_names[i]);
	}

	free(fringe_fd);
	free(fringe_names);

	exit(0);
}

/* ******************************************************************** */
/* processPacket							*/
/* ******************************************************************** */
void processPacket(
		DATAconHeader	*p_header,
		void		*data,
		NPOI_CONTAINER	*cstar,
		NPOI_CONFIG	*sysconfig,
		int		*ecFrame)
{
	uint32		iSpec;
	uint32		iChan;
	uint32		iFrame;
	uint32		idFrame;
	uint32		iSid;
	uint32		iFdl;
	uint32		iBas;
	unsigned long	idx;
	void		*dbuf;
	NPOI_CONFIG	*c_c;
	NPOI_FRAME	*c_r;
	int		cFrame;

	cFrame = *ecFrame;

	/* ************************************************************ */
	/* cFrame is an index which follows the total number of frames	*/
	/* read.  cFrame indexes the first frame of the most recent	*/
	/* RECORD read into memory.  Since each record is 500 frames	*/
	/* long, the cFrame is usually = nFrame - ~500.			*/
	/* ************************************************************ */

	c_c = cstar ? cstar->config : NULL;

	dbuf = data;

	switch((PACKET_TYPE_ENUM)p_header->dchType) {
		default:
			return;

		/* **************************************************** */
		case pktSCAN_START_VERSION_1: {
			SCAN_START_BODY_VERSION_1 *d =
				(SCAN_START_BODY_VERSION_1 *)dbuf;

			fprintf(stderr, "Scan: %3lu Star: %s\n",
					swaplong(d->ScanID), d->StarID);

			strncpy(c_c->starID, d->StarID, DEFSTRLEN);
			strncpy(c_c->starName, d->StarName, DEFSTRLEN);

			c_c->fringeFlag = swaplong(d->uFringeFlag);
			/* ******************************************** */
			/* They _read_ Apparent, but mean "catalog"	*/
			/* ******************************************** */
			c_c->catalogRA = swapdouble(d->ApparentRA);
			c_c->catalogDec = swapdouble(d->ApparentDec);

			/* ******************************************** */
			/* Fix for bad ScanID				*/
			/* ******************************************** */
			c_c->scanID = swaplong(d->ScanID);
		} break;

		/* **************************************************** */
		case pktSCAN_START_VERSION_2: {
			SCAN_START_BODY_VERSION_2 *d =
				(SCAN_START_BODY_VERSION_2 *)dbuf;

			strncpy(c_c->starID, d->StarID, DEFSTRLEN);
			strncpy(c_c->starName, d->StarName, DEFSTRLEN);

			c_c->fringeFlag = swaplong(d->uFringeFlag);
			/* ******************************************** */
			/* They read Apparent and mean "apparent"	*/
			/* ******************************************** */
			c_c->apparentRA = swapdouble(d->ApparentRA);
			c_c->apparentDec = swapdouble(d->ApparentDec);

			/* ******************************************** */
			/* Fix for bad ScanID				*/
			/* ******************************************** */
			c_c->scanID = swaplong(d->ScanID);
		} break;

		/* **************************************************** */
		case pktSTAR_ACQUIRED: {
			STAR_ACQ_BODY *d = (STAR_ACQ_BODY *)dbuf;

			if(!c_c->scanID) {
				c_c->scanID = swaplong(d->ScanID);
			} else if((unsigned long int)swaplong(d->ScanID)
					!= c_c->scanID)
			{
				fprintf(stderr,
				  "Scan [%lu] doesn't match expected [%lu]\n",
				  	swaplong(d->ScanID), c_c->scanID);
			}

			if(!(swaplong(d->Acquired))) {
				fprintf(stderr,
				  "Scan [%lu] not acquired: [%*.*s]\n",
					swaplong(d->ScanID),
					DEFSTRLEN, DEFSTRLEN, d->azStatus);
			}
		} break;

		/* **************************************************** */
		case pktSCAN_END: {
			SCAN_END_BODY *d = (SCAN_END_BODY *)dbuf;

			if(!c_c->scanID) {
				c_c->scanID = swaplong(d->ScanID);
			} else if((unsigned long int)swaplong(d->ScanID)
					!= c_c->scanID)
			{
				fprintf(stderr,
				  "Scan [%lu] doesn't match expected [%lu]\n",
				  	swaplong(d->ScanID), c_c->scanID);
			}
			fprintf(stderr,
				"Scan terminated with status [%*.*s]\n",
				DEFSTRLEN, DEFSTRLEN, d->azStatus);
		} break;

		/* **************************************************** */
		case pktFRINGE_DATA_VERSION_1: {
			FRINGE_DATA_BODY_VERSION_1 *d =
				(FRINGE_DATA_BODY_VERSION_1 *)dbuf;

			cFrame += VAR_NUM_REC;

			for(idFrame = 0; idFrame < VAR_NUM_REC; idFrame++) {
				iFrame = idFrame + cFrame;

				c_r = &(cstar->frames[iFrame]);

				/* ************************************ */
				/* Copy over photonCounts		*/
				/* ************************************ */
				for(iSpec = 0; iSpec < c_c->nSpec; iSpec++) {
					for(iChan = 0; iChan < c_c->nChan; iChan++) {
						memcpy(
						c_r->photonCounts[iSpec][iChan],
						  (*d)[idFrame].PhotonCounts[iSpec][iChan],
							c_c->nBin * sizeof(uint8));
					}
				}

				/* ************************************ */
				/* Copy over fringe_con_delay_estimates	*/
				/* ************************************ */
				for(iSpec = 0; iSpec < c_c->nSpec; iSpec++) {
					c_r->fringeConDelayEstimate[iSpec].val =
					  swapshort(
				  (*d)[idFrame].FringeConDelayEstimate[iSpec].Value);

					c_r->fringeConDelayEstimate[iSpec].err =
					  swapshort(
				  (*d)[idFrame].FringeConDelayEstimate[iSpec].Error);
				}
			}
		} break;

		/* **************************************************** */
		/* Identical to previous except fringe_time is set	*/
		/* **************************************************** */
		case pktFRINGE_DATA_VERSION_2: {
			FRINGE_DATA_BODY_VERSION_2 *d =
				(FRINGE_DATA_BODY_VERSION_2 *)dbuf;

			cFrame += VAR_NUM_REC;

			for(idFrame = 0; idFrame < VAR_NUM_REC; idFrame++) {
				iFrame = idFrame + cFrame;

				c_r = &(cstar->frames[iFrame]);

				c_r->fringeTime = swaplong((*d)[idFrame].uTime);

				for(iSpec = 0; iSpec < c_c->nSpec; iSpec++) {
					/* **************************** */
					/* Copy over photonCounts	*/
					/* **************************** */
					for(iChan = 0; iChan < c_c->nChan; iChan++) {
						memcpy(
						c_r->photonCounts[iSpec][iChan],
						  (*d)[idFrame].PhotonCounts[iSpec][iChan],
							c_c->nBin * sizeof(uint8));
					}

				  /* ********************************** */
				  /* Copy over fringeConDelayEstimates	*/
				  /* ********************************** */
					
				  c_r->fringeConDelayEstimate[iSpec].val =
				    swapshort(
				    (*d)[idFrame].FringeConDelayEstimate[iSpec].Value);

				  c_r->fringeConDelayEstimate[iSpec].err =
				    swapshort(
				    (*d)[idFrame].FringeConDelayEstimate[iSpec].Error);
				}
			}
		} break;

		/* **************************************************** */
		/* Same as above except without fringe_con_...		*/
		/* **************************************************** */
		case pktFRINGE_BG: {
			FRINGE_BG_BODY *d = (FRINGE_BG_BODY *)dbuf;

			cFrame += VAR_NUM_REC;

			for(idFrame = 0; idFrame < VAR_NUM_REC; idFrame++) {
				iFrame = idFrame + cFrame;

				c_r = &(cstar->frames[iFrame]);

				c_r->fringeTime = swaplong((*d)[idFrame].TimeStamp);

				for(iSpec = 0; iSpec < c_c->nSpec; iSpec++) {
					/* **************************** */
					/* Copy over photonCounts	*/
					/* **************************** */
					for(iChan = 0; iChan < c_c->nChan; iChan++) {
						memcpy(
						c_r->photonCounts[iSpec][iChan],
						  (*d)[idFrame].PhotonCounts[iSpec][iChan],
							c_c->nBin * sizeof(uint8));
					}
				}
			}
		} break;

		/* **************************************************** */
		case pktFRINGE_DARK: {
			FRINGE_PHOT_BODY *d = (FRINGE_PHOT_BODY *)dbuf;

			c_c->darkTime = swaplong(d->uTime);
			c_c->darkNumIntegrations =
				swaplong(d->uNumIntegrations);

			for(iSpec = 0; iSpec < c_c->nSpec; iSpec++) {
				for(iChan = 0; iChan < c_c->nChan; iChan++) {
					c_c->darkCounts[iSpec][iChan] =
						swaplong(d->iCounts[iSpec][iChan]);
				}
			}
		} break;

		/* **************************************************** */
		case pktNAT_COUNTS:
		case pktNAT_BG:
		case pktNAT_DARK: {
			NAT_QUAD_COUNT_BODY *d = (NAT_QUAD_COUNT_BODY *)dbuf;

			for(iFrame = cFrame; iFrame < c_c->nFrame; iFrame++) {
				idFrame = iFrame - cFrame;
				NAT_QUAD_COUNT_RECORD *r = &(d->Records[iFrame]);

				c_r = &(cstar->frames[iFrame]);

				c_r->natTime = swaplong(r->uTime);

				for(iSid = 0; iSid < c_c->nSid; iSid++) {
					NAT_QUAD_COUNTS *q =
						&(r->NatQuadCounts[idFrame]);

					c_r->natCounts[iSid].quadA =
						swapshort(q->uQuadA);
					c_r->natCounts[iSid].quadB =
						swapshort(q->uQuadB);
					c_r->natCounts[iSid].quadC =
						swapshort(q->uQuadC);
					c_r->natCounts[iSid].quadD =
						swapshort(q->uQuadD);
				}
			}
		} break;

		/* **************************************************** */
		case pktNAT_PIEZO_SIGNAL: {
			NAT_DAC_OFFSET *d = (NAT_DAC_OFFSET *)dbuf;

			for(iFrame = cFrame; iFrame < c_c->nFrame; iFrame++) {
				idFrame = iFrame - cFrame;
				c_r = &(cstar->frames[iFrame]);

				for(iSid = 0; iSid < c_c->nSid; iSid++) {
					idx = (idFrame * c_c->nSid) + iSid;

					c_r->natCounts[iSid].deltaX = 
						swapshort(d[idx].DeltaX);

					c_r->natCounts[iSid].deltaY = 
						swapshort(d[idx].DeltaY);
				}
			}
		} break;

		/* **************************************************** */
		case pktNAT_SERVO_PARAMS: {
			NAT_SERVO_PARAMS_BODY *d =
				(NAT_SERVO_PARAMS_BODY *)dbuf;

			c_c->proGainX = swapdouble(d->proportionalTermGainX);
			c_c->proGainY = swapdouble(d->proportionalTermGainY);
			c_c->intGainX = swapdouble(d->integralTermGainX);
			c_c->intGainY = swapdouble(d->integralTermGainY);
			c_c->derGainX = swapdouble(d->derivativeTermGainX);
			c_c->derGainY = swapdouble(d->derivativeTermGainY);
			c_c->filterGainX = swapdouble(d->filterGainX);
			c_c->filterGainY = swapdouble(d->filterGainY);
			c_c->lowPassA = swapdouble(d->lopassA);
			c_c->sidGainX = swapdouble(d->sidGainX);
			c_c->sidGainY = swapdouble(d->sidGainY);
		} break;

		/* **************************************************** */
		case pktFDL_POSITION: {
			FDL_POSITION_BODY *d = (FDL_POSITION_BODY *)dbuf;
			FDL_POSITION_RECORD	*r;
			FDL_POSITION		*c_f;

			for(iFrame = cFrame; iFrame < cFrame + VAR_NUM_REC; iFrame++) {
				r = &(d->FDLRecords[iFrame - cFrame]);
				c_r = &(cstar->frames[iFrame]);

				/* ************************************ */
				/* Assume uTime is same across all	*/
				/* siderostats				*/
				/* ************************************ */
				c_r->fdlTime =
					swaplong(r->FDLPositions[0].uTime);

				for(iFdl = 0; iFdl < c_c->nFdl; iFdl++) {
					FDL_DATUM *q = &(r->FDLPositions[iFdl]);
					c_f = &(c_r->fdlPosition[iFdl]);

					c_f->laserLo = swaplong(q->uLaserLo);
					c_f->laserHi = swaplong(q->uLaserHi);
					c_f->jitter = swaplong(q->iJitter);
				}

				c_r->calculateADFLasers();
			}
		} break;

		/* **************************************************** */
		case pktSYS_CONFIG: {
			SYS_CONFIG_C_STRUCT *d = (SYS_CONFIG_C_STRUCT *)dbuf;
			INPUT_BEAM		*c_ib;
			OUTPUT_BEAM		*c_ob;

			/* ******************************************** */
			/* HEY!  pktSYS_CONFIG actually contains	*/
			/* ASCII text instead of this structure.  Oops.	*/
			/* ******************************************** */
			break;

			c_c = sysconfig;

			c_c->latitude = swapdouble(d->Latitude);
			c_c->longitude = swapdouble(d->Longitude);
			c_c->altitude = swapdouble(d->Altitude);
			c_c->earthRadius = swapdouble(d->EarthRadius);
			c_c->J2 = swapdouble(d->J2);
			c_c->tdtMinusUTC = swapdouble(d->TDTminusUTC);
			c_c->ut1MinusUTC = swapdouble(d->UT1minusUTC);
			c_c->instrCohInt = swapdouble(d->InstrCohInt);

			c_c->beamCombinerID = swaplong(d->BeamCombinerID);
			c_c->nPlate = swaplong(d->NumPlate);

			for(iSid = 0; iSid < c_c->nSid; iSid++) {
				c_ib = &(c_c->inputBeams[iSid]);
				strncpy(c_ib->stationID,
						d->InputBeam.StationID[iSid],
						MAX_IDSTR);

				c_ib->stationCoord[0] =
				  swapdouble(d->InputBeam.StationCoord[iSid][0]);
				c_ib->stationCoord[1] =
				  swapdouble(d->InputBeam.StationCoord[iSid][1]);
				c_ib->stationCoord[2] =
				  swapdouble(d->InputBeam.StationCoord[iSid][2]);
				c_ib->stationCoord[3] =
				  swapdouble(d->InputBeam.StationCoord[iSid][3]);

				c_ib->sidID =
					swaplong(d->InputBeam.SiderostatID[iSid]);

				c_ib->fdlTankID =
					swaplong(d->InputBeam.FDLTankID[iSid]);

				c_ib->stroke = 
					swapdouble(d->InputBeam.Stroke[iSid]);

				c_ib->quadCellID = 
					swaplong(d->InputBeam.QuadCellID[iSid]);

				c_ib->bcInputID = 
					swaplong(d->InputBeam.BCInputID[iSid]);
			}

			for(iSpec = 0; iSpec < c_c->nSpec; iSpec++) {
				c_ob = &(c_c->outputBeams[iSpec]);

				c_ob->bcOutputID =
					swaplong(d->OutputBeam.BCOutputID[iSpec]);

				c_ob->nFringeFreq = swaplong(
						d->OutputBeam.NumFringeFreq[iSpec]);

				for(iBas = 0; iBas < c_c->nBas; iBas++) {
					c_ob->inputPair[iBas][0] = swaplong(
					  d->OutputBeam.InputPair[iSpec][iBas][0]);

					c_ob->inputPair[iBas][1] = swaplong(
					  d->OutputBeam.InputPair[iSpec][iBas][1]);

					c_ob->fringeMod[iBas] = swaplong(
					  d->OutputBeam.FringeMod[iSpec][iBas]);
				}

				c_ob->apdArrayID = swaplong(
						d->OutputBeam.APDArrayID[iBas]);

				c_ob->nSpecChan = swaplong(
						d->OutputBeam.NumSpecChan[iBas]);

				for(iChan = 0; iChan < c_c->nChan; iChan++) {
					c_ob->specChan[iChan].wavelength = swapdouble(
					    d->OutputBeam.Wavelength[iSpec][iChan]);

					c_ob->specChan[iChan].wavelengthErr = swapdouble(
					    d->OutputBeam.WavelengthErr[iSpec][iChan]);

					c_ob->specChan[iChan].chanWidth = swapdouble(
					    d->OutputBeam.ChanWidth[iSpec][iChan]);

					c_ob->specChan[iChan].chanWidthErr = swapdouble(
				  	    d->OutputBeam.ChanWidthErr[iSpec][iChan]);
				}
			}
		} break;
	}

	*ecFrame = cFrame;
}

/* ******************************************************************** */
/* searchDirectoryForData						*/
/* ******************************************************************** */
void searchDirectoryForData(
		char		*dirname,
		char		*search_string,
		char		***names,
		FILE		***data,
		int		*n_data)
{
	DIR		*dir;
	struct dirent	*curr_file;
	struct stat	file_stat;
	char		buf[BUFSIZ];
	int		i;

	*data = NULL;
	*names = NULL;
	*n_data = 0;

	if((dir = opendir(dirname)) == NULL) {
		fprintf(stderr, "Could not open %s: %s\n", dirname,
				strerror(errno));

		exit(1);
	}

	while((curr_file = readdir(dir))) {
		sprintf(buf, "%s/%s", dirname, curr_file->d_name);

		stat(buf, &file_stat);

		if(S_ISREG(file_stat.st_mode) && strstr(curr_file->d_name, search_string))
		{
			(*n_data)++;

			if(!(*data)) {
				*data = (FILE **)malloc(
					(*n_data) * sizeof(*data));
			} else {
				*data = (FILE **)realloc(*data,
					(*n_data) * sizeof(*data));
			}

			if(!(*names)) {
				*names = (char **)malloc(
					(*n_data) * sizeof(*names));
			} else {
				*names = (char **)realloc(*names,
					(*n_data) * sizeof(*names));
			}

			(*names)[(*n_data) - 1] = (char *)malloc(
							strlen(buf) + 1);
			strcpy((*names)[(*n_data) - 1], buf);

			if(((*data)[(*n_data) - 1] = fopen(buf, "r")) ==
					NULL)
			{
				fprintf(stderr, "Cannot open %s: %s\n",
						buf, strerror(errno));
				exit(1);
			}
		}
	}

	fprintf(stderr, "Found %d input file(s):\n", *n_data);

	for(i = 0 ; i < *n_data; i++) {
		fprintf(stderr, " [%s]\n", (*names)[i]);
	}
}

/* ******************************************************************** */
/* importAlign								*/
/* ******************************************************************** */
void importAlign(char *indir, char *outdir) {
	FILE			**align_fd;
	FILE			**data_fd;
	char			**align_names;
	char			**data_names;
	int			align_n;
	int			data_n;
	void			*data;
	NPOI_CONFIG		*sysconfig;
	int			c_f;
	DATAconHeader		t_header;
	int			i;
	alignStruct		*align = NULL;
	int			alignLen = 0;
	uint32			seekPos;

	sysconfig = initConfig();

	searchDirectoryForData(indir, FS_ALIGN, &align_names,
			&align_fd, &align_n);

	searchDirectoryForData(outdir, ".gz", &data_names,
			&data_fd, &data_n);

	for(i = 0; i < data_n; i++) {
		fclose(data_fd[i]);
	}

	for(c_f = 0; c_f < align_n; c_f++) {
		fprintf(stderr, "File: %s\n", align_names[c_f]);
		while(fread(&t_header, sizeof(DATAconHeader), 1,
					align_fd[c_f]))
		{
			/* ******************************************** */
			/* Fix packet					*/
			/* ******************************************** */
			t_header.dchType	= swaplong(t_header.dchType);
			t_header.dchLength	= swaplong(t_header.dchLength);
			t_header.dchTime	= swaplong(t_header.dchTime);
			t_header.dchDay		= swaplong(t_header.dchDay);

			print_packet(t_header, 0);

			if(t_header.dchLength < sizeof(DATAconHeader)) {
				fprintf(stderr, "Data length [%lu] too short\n",
						t_header.dchLength);
				return;
			}

			/* ******************************************** */
			/* Read data					*/
			/* ******************************************** */
			data = (void *)malloc(t_header.dchLength -
					sizeof(DATAconHeader));

			seekPos = ftell(align_fd[c_f]);
			fread(data, t_header.dchLength - sizeof(DATAconHeader),
					1, align_fd[c_f]);

			processAlign(&t_header, data, &align, &alignLen,
					c_f, seekPos);

			free(data);
		}
	}

	qsort(align, alignLen, sizeof(alignStruct), sortAlignFunc);

	packDataFiles(data_names, data_n, align, alignLen, align_fd);

	for(i = 0; i < align_n; i++) {
		fclose(align_fd[i]);
		free(align_names[i]);
	}

	free(align_fd);
	free(align_names);

	for(i = 0; i < data_n; i++) {
		free(data_names[i]);
	}

	free(data_fd);
	free(data_names);

	exit(0);
}

/* ******************************************************************** */
/* processAlign								*/
/* ******************************************************************** */
void processAlign(
		DATAconHeader	*t_header,
		void		*data,
		alignStruct	**align,
		int		*alignLen,
		char		c_f,
		uint32		seekPos)
{
	alignStruct	*al;
	int		len;
	int		olen;
	int		i;

	al = *align;

	len = *alignLen;

	switch((PACKET_TYPE_ENUM)t_header->dchType) {
		default:
			return;

		case pktNAT_COUNTS:
		case pktNAT_BG:
		case pktNAT_DARK: {
			NAT_QUAD_COUNT_BODY *d = (NAT_QUAD_COUNT_BODY *)data;

			olen = len;
			len += VAR_NUM_REC;

			al = (alignStruct *)realloc(al, len * sizeof(alignStruct));

			for(i = 0; i < VAR_NUM_REC; i++) {
				al[olen + i].time = swaplong(d->Records[i].uTime);
				al[olen + i].seek =
					seekPos + (char *)&(d->Records[i]) - (char *)d;
				al[olen + i].fd = c_f;
			}
			break; }
	}

	*align = al;
	*alignLen = len;
}

/* ******************************************************************** */
/* sortAlignFunc							*/
/* ******************************************************************** */
int sortAlignFunc(const void *p0, const void *p1) {
	alignStruct *q0 = (alignStruct *)p0;
	alignStruct *q1 = (alignStruct *)p1;

	if(q0->time == q1->time) {
		return 0;
	} else if(q0->time < q1->time) {
		return -1;
	} else {
		return 1;
	}
}

/* ******************************************************************** */
/* packDataFiles							*/
/* ******************************************************************** */
void packDataFiles(char **data, int nData, alignStruct *align, int nAlign, FILE **fds) {
	int			iData;
	NPOI_CONTAINER		*d;
	int			fTS;
	int			iAlign;
	uint32			iFrame;
	uint32			iSid;
	NAT_QUAD_COUNT_RECORD	counts;

	for(iData = 0; iData < nData; iData++) {
		d = new NPOI_CONTAINER(data[iData]);
		printf("[%s] ", basename(data[iData]));
		fflush(stdout);

		assert(d->config->nSid == VAR_NUM_SID);

		if(d->config->nFrame == 0 ||
				(fTS = findFirstInstance(align, nAlign,
				d->config->firstTS)) == -1)
		{
			delete d;

			printf(" SKIP\n");
		} else {
			iAlign = fTS;
			iFrame = 0;

			while(iFrame < d->config->nFrame) {
				while(align[iAlign].time < d->frames[iFrame].fringeTime &&
						iAlign < nAlign)
					iAlign++;

				if(align[iAlign].time >= d->frames[iFrame].fringeTime) {
					fseek(fds[align[iAlign].fd], align[iAlign].seek,
						SEEK_SET);

					fread(&counts, sizeof(NAT_QUAD_COUNT_RECORD), 1,
						fds[align[iAlign].fd]);

					d->frames[iFrame].natTime = swaplong(counts.uTime);
					for(iSid = 0; iSid < d->config->nSid; iSid++) {
					d->frames[iFrame].natCounts[iSid].quadA =
						swaplong(counts.NatQuadCounts[iSid].uQuadA);
					d->frames[iFrame].natCounts[iSid].quadB =
						swaplong(counts.NatQuadCounts[iSid].uQuadB);
					d->frames[iFrame].natCounts[iSid].quadC =
						swaplong(counts.NatQuadCounts[iSid].uQuadC);
					d->frames[iFrame].natCounts[iSid].quadD =
						swaplong(counts.NatQuadCounts[iSid].uQuadD);
					d->frames[iFrame].natCounts[iSid].deltaX = 0;
					d->frames[iFrame].natCounts[iSid].deltaY = 0;
					}
					if((iFrame % 500) == 0) {
						putchar('.');
						fflush(stdout);
					}
				}

				iFrame++;
			}

			printf(" #");
			fflush(stdout);

			d->save(data[iData]);

			printf("#\n");
			fflush(stdout);

			delete d;
		}

	}
}

/* ******************************************************************** */
/* findFirstInstance							*/
/* ******************************************************************** */
int findFirstInstance(alignStruct *align, int nAlign, uint32 target) {
	int	i;

	i = nAlign - 1;

	while(i > 0 && align[i].time > target)
		i--;

	if(align[i].time <= target) {
		return i;
	} else {
		return -1;
	}
}

/* ******************************************************************** */
/* countFramesInFiles							*/
/* ******************************************************************** */
void countFramesInFiles(FILE **fds, int ***eCounts, int **eScans, int nFile) {
	int			iFile;
	int			**counts;
	int			*scans;
	DATAconHeader		t_header;
	PACKET_TYPE_ENUM	headerType;

	*eCounts = NULL;
	*eScans = NULL;

	scans = (int *)calloc(nFile, sizeof(int));
	counts = (int **)calloc(nFile, sizeof(int *));

	for(iFile = 0; iFile < nFile; iFile++) {
		fseek(fds[iFile], 0, SEEK_SET);

		while(fread(&t_header, sizeof(DATAconHeader), 1, fds[iFile])) {
			/* ******************************************** */
			/* Fix packet					*/
			/* ******************************************** */
			t_header.dchType	= swaplong(t_header.dchType);
			t_header.dchLength	= swaplong(t_header.dchLength);
			t_header.dchTime	= swaplong(t_header.dchTime);
			t_header.dchDay		= swaplong(t_header.dchDay);

			if(t_header.dchLength < sizeof(DATAconHeader)) {
				fprintf(stderr, "Data length [%lu] too short\n",
						t_header.dchLength);
				return;
			}

			/* ******************************************** */
			/* Read data					*/
			/* ******************************************** */
			fseek(fds[iFile], t_header.dchLength - sizeof(DATAconHeader),
					SEEK_CUR);

			/* ******************************************** */
			/* Check to build new star			*/
			/* ******************************************** */
			headerType = (PACKET_TYPE_ENUM)(t_header.dchType);
			if(headerType == pktSCAN_START_VERSION_1 ||
			   		headerType == pktSCAN_START_VERSION_2)
			{
				scans[iFile]++;
				counts[iFile] = (int *)realloc(counts[iFile],
						scans[iFile] * sizeof(int));
				counts[iFile][scans[iFile] - 1] = 0;
			} else if(headerType == pktFRINGE_BG ||
				headerType == pktFRINGE_DATA_VERSION_1 ||
				headerType == pktFRINGE_DATA_VERSION_2)
			{
				counts[iFile][scans[iFile] - 1] += VAR_NUM_REC;
			}
		}

		fseek(fds[iFile], 0, SEEK_SET);

		printf("Found %d scans\n", scans[iFile]);
	}

	*eCounts = counts;
	*eScans = scans;
}
