/* packetdefs.h - typedefs for packet bodies with a compile-time definition */

#ifndef _PACKETDEF_H
#define _PACKETDEF_H

#include "defs.h"

/*******************************************************************************

In this file we attempt to provide definitions for all the packets
produced by the NPOI embedded systems in such a way that this file can
be included in the C-programs that create or read them. This is not
possible in all cases, especially variable-length packets. In this
case we provide a nominal definition.

In some cases, variable array dimensions are replaced by #defines of nominal
values of these parameters, whose names are prefixed by VAR_ to indicate that 
they are really variables and not constants. However, these array dimension
must match those of the raw data packets. The latter are currently not
communicated by means of the SYS_CONFIG packet.

Another commonly-occurring variable-length packet is the text packet, and
here we either define string templates that could be used in a sprintf() call
to produce this packet or merely use the typedef TEXT_BODY and explain in
comments the contents of the text.

*******************************************************************************/

#define TEXT_BODY char             	/* All text-packets have this type */
#define DEFSTRLEN 80               	/* Default string length */

#define VAR_NUM_SID 6            	/* Number of siderostats allocated */

#define VAR_NUM_BAS 6			/* Number of baseline in one output */

#define VAR_NUM_FDL 6           	/* Number of delay lines allocated */
#define VAR_NUM_TRPLE 512		/* Number of possible triples */

/* 
 *18-Mar-2002 TAP: These are the values of the current system.
 */
#define VAR_NUM_SPEC 2  		/* Number of spectrometers allocated */
#define VAR_NUM_CHAN 16     		/* Number of channels allocated */
#define NUM_BIN 64                      /* Number of sample bins */
#define VAR_NUM_REC 500          	/* Number of 2 ms samples allocated */

/*------------------------------------------------------------------------------

             D A T A c o n   P A C K E T S

------------------------------------------------------------------------------*/

/* pktFILE_HEADER=0x00000000 */

#define FILE_HEADER_TEMPLATE \
"DATACON_VERSION=%s\n"                /* Format = nn.nn.nn */ \
"FILE_ID=%s\n"                        /* Format = siteYYYYMMDDHHMM.raw */ \
"DISK_SEQUENCE_ID=%d\n"               /* 1 = first disk in session */ \
"HEADER_SEQUENCE_ID=%d\n"             /* 1 = first copy of header on disk */ \
"SITE=anderson_mesa\n" \
"FILE_DATE=%s\n"                      /* Format = YYYY-MM-DD */ \
"FILE_START_TIME=%s\n"                /* Format = HH:MM:SS   */ \
"TAPE_LOGICAL_RECORD_SIZE=%d\n"       /* Tape records are this  many bytes \
					 long (typically 10k) */ \
"TAPE_FILE_MARK_INTERVAL=%d\n"        /* To make accessing raw data on tape \
					 more simple, the tape dump of the \
					 raw file is split into chunks \
					 delineated by tape file-marks. \
					 Each chunk except perhaps the last is\
					 this many tape records long \
					 (Typically 100Mb, i.e. 10000 \
					 records). */


/* pktPACKET_SYMBOL_TABLE=0x00010000 */

#define PACKET_SYMBOL_TABLE "\
; This is a packet symbol table\n\
; It was last modified on 24-Jun-1994 by DFB\n\
\n\
0x00000000 FILE_HEADER               ; Header packet identifying the dataset\n\
\n\
0x00010000 PACKET_SYMBOL_TABLE       ; This packet\n\
0x00010001 PACKET_DIRECTORY          ; List of packets on a disk \n\
\n\
0x00020000 PAD                       ; Dummy packet\n\
\n\
0x00030000 END_OF_DISK               ; Obvious\n\
0x00030001 END_OF_SESSION            ; Last disk in observing session\n\
\n\
0x000a0000 SYS_CONFIG                ; Essential instrument setup info\n\
0x000a0001 SYS_LOG                   ; Dump of system logs\n\
\n\
0x000b0000 SCAN_START_VERSION_1      ; Start slewing to a star, cat. pos.\n\
0x000b0001 STAR_ACQUIRED             ; Acquisition success flag\n\
0x000b0002 SCAN_START_VERSION_2      ; Start slewing to a star, app. pos.\n\
0x000b0003 SCAN_END                  ; End of observation of star\n\
0x000b0004 SCAN_OPERATOR_COMMENT     ; Comments on a specific scan\n\
\n\
0x000c0001 FRINGE_DATA_VERSION_1     ; Photon counts from FRINGEcon\n\
0x000c0002 FRINGE_DATA_VERSION_2     ; Photon counts from FRINGEcon\n\
0x000c0004 FRINGE_BG                 ; Fringe detector background counts\n\
0x000c0005 FRINGE_DARK               ; Dark counts (shutter closed)\n\
0x000c0006 FRINGECON_SERVO_PARAMS    ; Servo parameters\n\
\n\
0x000d0001 NAT_COUNTS                ; Quad-cell counts\n\
0x000d0002 NAT_PIEZO_SIGNAL          ; Obvious\n\
0x000d0003 NAT_BG                    ; Quad cell background counts\n\
0x000d0004 NAT_DARK                  ; Quad cell counts, shutter closed\n\
0x000d0005 NAT_SERVO_PARAMS          ; Servo parameters\n\
\n\
0x000e0001 FDL_POSITION              ; Delay line metrology counts\n\
0x000e0002 FDL_JITTER                ; High-frequency rms error\n\
0x000e0003 FDL_STATUS                ; Current performance of delay lines\n\
\n\
0x000f0001 WEATHER                   ; Info from weather station\n\
0x000f0002 BRAD_CONFIG               ; Configuration of environmental sensors\n\
0x000f0003 BRAD_DATA                 ; Raw environmental sensor data\n\
\n\
0x00100001 SIDMET		; Obvious\n\
0x00100002 OPTANCH		; Obvious\n\
0x00100003 PIER2PIER		; Obvious\n\
0x00100004 EXTCATEYE		; Obvious\n\
0x00100005 PLATEEXP		; Obvious\n\
0x00100006 METAIRTEMP		; Obvious\n\
0x00100007 METSOLIDTMP		; Obvious\n\
0x00100008 METPRESSURE		; Obvious\n\
0x00100009 METHUMIDITY		; Obvious\n\
0x0010000a LABAIRTEMP		; Obvious\n\
0x0010000b LABSOLIDTMP		; Obvious\n\
0x0010000c LABPRESS		; Obvious\n\
0x0010000d LABHUM		; Obvious\n\
0x0010000e DLPRESS		; Obvious\n\
0x0010000f FBPRESS		; Obvious\n\
0x00100010 WXAIRTEMP		; Obvious\n\
0x00100011 WXPRESSURE		; Obvious\n\
0x00100012 WXHUMIDITY		; Obvious\n\
0x00100013 WINDSPEED		; Obvious\n\
0x00100014 WINDDIR		; Obvious\n\
\n\
0x00110001 WASA_IMAGE                ; Compressed image of acquisition field\n\
0x00110002 SID_MOTOR_COUNTS          ; Obvious\n\
0x00110003 SID_MODEL                 ; Current pointing model\n\
\n\
0x0e100f09 SYNC                      ; Signature packet for synchronisation\n\
"

/* pktPACKET_DIRECTORY=0x00010001 */

       /****Undefined at present****/

/* pktPAD=0x00020000 */

typedef void PAD_BODY;

/* pktEND_OF_DISK=0x00030000 */

#define END_OF_DISK_BODY "END OF DISK";

/* pktEND_OF_SESSION=0x00030001 */

#define END_OF_SESSION_BODY "TH-TH-THAT'S ALL FOLKS!";

/* pktSYS_SETUP=0x000a0000 */

       /****Undefined at present****/

/* pktSYS_LOG=0x000a0001 */

/*

   This is a generic packet that contains a dump of a log file on argus.
   It is a text packet whose first line is the name of the log file and
   the date and time the file was last written to, e.g.

    seqcon_error.log 1994-Jun-30 20:35:33\n
    ...dump of file....
    \0

*/

/*------------------------------------------------------------------------------

             S C A N   P A C K E T S

------------------------------------------------------------------------------*/

/* pktSCAN_START_VERSION_1=0x000b0000 */

typedef struct {
   int32 ScanID;		/* ID for scan, unique to session */
   char StarID[DEFSTRLEN];	/* e.g. HIC1202 - official catalog entry */
   char StarName[DEFSTRLEN];	/* e.g. alpha Her - colloquial name
				   chosen by user */
   uint8 uFringeFlag;
   uint8 uPad0;
   uint8 uPad1;
   uint8 uPad2;
   double ApparentRA;		/* Catalog pos.!, Unit = HOURS */
   double ApparentDec;		/* Catalog pos.!, Unit = DEGREES */
} SCAN_START_BODY_VERSION_1;

/* pktSCAN_START_VERSION_2=0x000b0002 */

typedef struct {
   int32 ScanID;		/* ID for scan, unique to session */
   char StarID[DEFSTRLEN];	/* e.g. HIC1202 - official catalog entry */
   char StarName[DEFSTRLEN];	/* e.g. alpha Her - colloquial name
				   chosen by user */
   uint8 uFringeFlag;
   uint8 uPad0;
   uint8 uPad1;
   uint8 uPad2;
   double ApparentRA;		/* Unit = HOURS */
   double ApparentDec;		/* Unit = DEGREES */
} SCAN_START_BODY_VERSION_2;


/* pktSTAR_ACQUIRED=0x000b0001 */

typedef struct {
  int32 ScanID;
  int32 Acquired;                   /* FALSE = acquire failed */
  char azStatus[DEFSTRLEN];        /* Text string describing acquire status */
} STAR_ACQ_BODY;

/* pktSCAN_END=0x000b0003 */

typedef struct {
  int ScanID;
  char azStatus[DEFSTRLEN];       /* Possible values: "Normal termination",
				     "User hit ABORT", etc. */
} SCAN_END_BODY;

/* pktSCAN_COMMENT=0x000b0004 */

/* 
   Text packet, comments intended to apply to a particular scan. The idea
   is that any summary of the scans for a night would include this comment
   field, as distinct from the general log which is a separate document
   and applies to the night as a whole
*/

/*------------------------------------------------------------------------------

             F r i n g e c o n   P A C K E T S

------------------------------------------------------------------------------*/

/* pktFRINGE_DATA_VERSION_1=0x000c0001 */

typedef struct {
  uint8 PhotonCounts[VAR_NUM_SPEC][VAR_NUM_CHAN][NUM_BIN]; 
  struct {
    int16 Value;                 /* Net delay in 0.01 micron units */
    int16 Error;                 /* Rms error in 0.01 micron units */
  } FringeConDelayEstimate[VAR_NUM_SPEC];
} FRINGE_DATA_BODY_VERSION_1[VAR_NUM_REC];

/* pktFRINGE_DATA_VERSION_2=0x000c0002 */

typedef struct {
  uint8 PhotonCounts[VAR_NUM_SPEC][VAR_NUM_CHAN][NUM_BIN]; 
  struct {
    int16 Value;                /* Net delay in 0.01 micron units */
    int16 Error;                /* Rms error in 0.01 micron units */
  } FringeConDelayEstimate[VAR_NUM_SPEC];
  uint32 uTime ;         	/* UT time in 100 microsecond units */
} FRINGE_DATA_BODY_VERSION_2[VAR_NUM_REC];

/* pktFRINGE_BG=0x000c0004 */
/* Currently the 6-way fringe packet */

typedef struct {
  uint8  PhotonCounts[VAR_NUM_SPEC][VAR_NUM_CHAN][NUM_BIN];
  uint32 TimeStamp;
} FRINGE_BG_BODY[VAR_NUM_REC];

/* pktFRINGE_DARK=0x000c0005 */
/* Count rate on blank sky */

typedef struct {
  uint32 uTime;
  uint32 uNumIntegrations;
  int32 iCounts[VAR_NUM_SPEC][VAR_NUM_CHAN];
} FRINGE_PHOT_BODY;

/* pktFRINGECON_SERVO_PARAMS=0x000c0006 */
/* Undefined at present */


/*------------------------------------------------------------------------------

             N A T   P A C K E T S

------------------------------------------------------------------------------*/

/* pktNAT_COUNTS=0x000d0001 */

typedef struct {
  uint16 uQuadA;     		/* Lower right */
  uint16 uQuadB;     		/* Lower left */
  uint16 uQuadC;     		/* Upper left */
  uint16 uQuadD;     		/* Upper right */
} NAT_QUAD_COUNTS;

typedef struct NAT_QUAD_COUNT_RECORD {
  NAT_QUAD_COUNTS NatQuadCounts[VAR_NUM_SID];
  uint32 uTime;	/* UT time in ms */
} NAT_QUAD_COUNT_RECORD;

typedef struct NAT_QUAD_COUNT_BODY {
  NAT_QUAD_COUNT_RECORD Records[VAR_NUM_REC];
} NAT_QUAD_COUNT_BODY;

/* pktNAT_PIEZO_SIGNAL=0x000d0002 */

typedef struct {
  int16 DeltaX;
  int16 DeltaY;
} NAT_DAC_OFFSET;

typedef NAT_DAC_OFFSET NAT_PIEZO_BODY[VAR_NUM_REC][VAR_NUM_SID];

/* pktNAT_BG=0x000d0003 */

/*******************************************************************

  Same structure as NAT_COUNTS

********************************************************************/

/* pktNAT_DARK=0x000d0004 */

/*******************************************************************

  Same structure as NAT_COUNTS

********************************************************************/

/* pktNAT_SERVO_PARAMS=0x000d0005 */

typedef struct
{
    double proportionalTermGainX;
    double proportionalTermGainY;
    double integralTermGainX;
    double integralTermGainY;
    double derivativeTermGainX;
    double derivativeTermGainY;
    
    double filterGainX;
    double filterGainY;
    double lopassA;

    double sidGainX;
    double sidGainY;
} NAT_SERVO_PARAMS_BODY;                  /* One packet per SIDcon */


/*------------------------------------------------------------------------------

             F D L   P A C K E T S

------------------------------------------------------------------------------*/

/* pktFDL_POSITION=0x000e0001 */

typedef struct FDL_DATUM {
  uint32 uLaserLo;      	/* lo 32 bits of laser counter */
  uint32 uLaserHi;      	/* hi 32 bits of laser counter */
  uint32 uTime;         	/* UT time in 100 microsecond units */
  int32 iJitter;            	/* current jitter, not filtered */
} FDL_DATUM;

typedef struct FDL_POSITION_RECORD {
  FDL_DATUM FDLPositions[VAR_NUM_FDL];
} FDL_POSITION_RECORD;

typedef struct FDL_POSITION_BODY {
  FDL_POSITION_RECORD FDLRecords[VAR_NUM_REC];
} FDL_POSITION_BODY;

#define LAMBDA_HENE (632.9914e-9)	/* HeNe wavelength in vacuum */
#define FDL_MET_UNITS (LAMBDA_HENE/512)   

/*  
These convert the low and high bytes seperately.  Although the absolute 
accuracy is not important - it corresponds to an error in the baseline -
the ratio of the high and low conversions must have full accuracy, or there 
will be a jump in the observed delay every time the high part changes 
*/

#define FDLLO_MET_UNITS (LAMBDA_HENE/512)
#define TWO232 (4294967296)		/* 2^32 */
/*
#define FDLHI_MET_UNITS (FDLLO_MET_UNITS*TWO232)
*/
#define TWO223 (8388608)
#define FDLHI_MET_UNITS (LAMBDA_HENE*TWO223)

/* pktFDL_JITTER=0x000e0002 */

                 /*  Undefined at present  */

/* pktFDL_STATUS=0x000e0003 */

                 /*  Undefined at present  */


/*------------------------------------------------------------------------------

             E N V I R O N M E N T   P A C K E T S

------------------------------------------------------------------------------*/

/* pktWEATHER=0x000f0001 */

/***********************************************************************

This packet is a dump of the output of the weather console. Here is
a typical dump:

CAPRICORN II WEATHER REPORT
INSIDE TEMP  82.4 DEGREES F
OUTSIDE TEMP  76.2 DEGREES F
WIND CHILL TEMP  74.8 DEGREES F
HIGH TEMP  82.0 DEGREES F
LOW TEMP  58.8 DEGREES F
WIND DIRECTION W   
WIND SPEED   3 MPH
HIGH WIND SPEED  23 MPH
PEAK WIND GUST  31 MPH
BAROMETER  798 MILLIBAR
HIGH BAROMETER  799 MILLIBAR
LOW BAROMETER  795 MILLIBAR
WIND ALARM +SILENT+ >  20 MPH
TEMP ALARM +SILENT+ <  70 DEG F


It should be noted that this packet should conform to the standard for
a text packet, notably the CR-LF line terminators output by the weather
station should be translated to LF-only.

**************************************************************************/

/* pktBRAD_CONFIG=0x000f0002 */

/***********************************************************************

A text packet which shows the configuration of the brad sensors, one line
for each sensor, one packet per SIDcon and one from aligncon. The instance
number in the packet header is used to indicate which system is being referred
to. The first instance number is 0 for ALIGNcon and 1 for SIDcon. The second
instance number indicates which SIDcon if it is SIDcon.

The format of the config records is:

CHAIN ADDRESS TYPE UNIT SCALE OFFSET SAMPLE_INTERVAL \
   LOCATION_CODE LOCX LOCY LOCZ LOCERRX LOCERRY LOCERRZ

Where:

CHAIN is the BRAD sensor chain number [0-3]
ADDRESS is the address of the BRAD board within the chain [0-31]
TYPE is the type of sensor: one of AirTemp, SolidTemp, Pressure, Humidity
UNIT is the physical unit of the measurement once converted with 
     SCALE and OFFSET, one of DegreeC, Millibar, Percent. 
SCALE and OFFSET convert the raw ADC data to physical units the formula is
     Physical = (RawADC*SCALE) + OFFSET
     A zero value for SCALE means that the given BRAD board is out of service.
SAMPLE_INTERVAL is the sampling interval of that BRAD board in milliseconds
     In fact, as presently defined, this value must be the same for all
     BRADs on a given subsystem.
LOCATION_CODE is an ASCII code denoting the location/purpose of the sensor:
     one of Lab, SidHut, SidCoop, MetPlate, FDLTank, FeedBeam, Outdoors.
LOCX, Y, Z Coordinates of probe with respect to the array center, in meters.
     Z is up, Y is North, X is East.
LOCERRX, Y, Z Estimated errors in the above coords.
  
**************************************************************************/

/* pktBRAD_DATA=0x000f0003 */

#define MAX_BRAD_CHAIN 4    /* Max number of BRAD chains */
#define MAX_BRAD_ADDR 32    /* Max number of BRAD boards on a chain */

typedef uint16 BRAD_DATA_BODY[VAR_NUM_REC][MAX_BRAD_CHAIN][MAX_BRAD_ADDR];

/*------------------------------------------------------------------------------

             M E T R O L O G Y   P A C K E T S

------------------------------------------------------------------------------*/

/* pktCONST_TERM_METROLOGY=0x00100001 */

                 /*  Undefined at present  */

/* pktBEAM_COM_DATA=0x00100002 */

                 /*  Undefined at present  */

/* pktSID_METROLOGY=0x00100003 */

                 /*  Undefined at present  */

/* pktP2P_METROLOGY=0x00100004 */

                 /*  Undefined at present  */

/* pktOAC_METROLOGY=0x00100005 */

                 /*  Undefined at present  */

/* pktTABLE_METROLOGY=0x00100006 */

                 /*  Undefined at present  */

/* pktAUX_CATSEYE_METROLOGY=0x00100007 */

                 /*  Undefined at present  */




/*------------------------------------------------------------------------------

             M I S C   S I D E R O S T A T   P A C K E T S

  ----------------------------------------------------------------------------*/
/* pktWASA_IMAGE=0x00110001 */

typedef struct {
  double dfXPosition;           /* X centroid position */
  double dfYPosition;           /* Y centroid position */
  double dfTotalBrightness;     /* total brightness of pixels in the
				   bounding rectangle */
} WASA_STAR_ENTRY;

typedef WASA_STAR_ENTRY WASA_IMAGE_BODY[VAR_NUM_REC];


/* pktSID_MOTOR_COUNTS=0x00110002 */

typedef struct {
  int32 iAzMotorCounts;      /* motor angle from SMIM on azimuth axis */
  int32 iElMotorCounts;      /* motor angle from SMIM on elevation axis */
  uint32 uTime;              /* UT time in ms */
} SID_MOTOR_COUNTS_REC;

typedef struct SID_MOTOR_COUNTS_BODY {

  SID_MOTOR_COUNTS_REC Records[VAR_NUM_REC];

  uint8 uSIDcon;        /* Which SIDcon */
  uint8 uint8Pad0;          /* Some Pads */
  uint8 uint8Pad1;
  uint8 uint8Pad2;

} SID_MOTOR_COUNTS_BODY;

/*typedef SID_MOTOR_COUNTS_REC SID_MOTOR_COUNTS_BODY[VAR_NUM_REC];*/

/* pktSID_MODEL=0x00110003 */

/* This is taken directly from SIDcalc.h in ~buchanan/projects/SIDcon/ai */

typedef struct SID_MODEL
{
    double dfFeedBeamAz;      /* feed beam azimuth - 0 = North, 90 = West */
    double dfFeedBeamAzErr;
    double dfFeedBeamEl;      /* feed beam elevation - 0 = Horizon, 90 = Vertical */
    double dfFeedBeamElErr;
    double dfSidAz;           /* siderostat azimuth - 0 = North, 90 = West */
    double dfSidAzErr;
    double dfSidEl;           /* siderostat elevation - 0 = Horizon, 90 = Vertical */  
    double dfSidElErr;
    double dfAxisOffset;       /* error in angle between axis, 0 = orthogonal */
    double dfAxisOffsetErr;
    double dfZeroAz;          /* zero point on azimuth axis */
    double dfZeroAzErr;
    double dfZeroEl;          /* zero point on the elevation axis */
    double dfZeroElErr;
    double dfMirrorOffset;     /* tilt of mirror in its cell, 0 = perpendicular to axis */
    double dfMirrorOffsetErr;

    double dfLatitude;            /* site latitude - 90 degrees */
    double dfLongitude;           /* site longitude */
    double dfRefractionIndex;     /* site atmospheric refraction index */
    int iTDTOffset;               /* the dynamical time offset in seconds */

    uint8 uSIDcon;
    uint8 uint8Pad0;
    uint8 uint8Pad1;
    uint8 uint8Pad2;

} SID_MODEL;

/*
typedef struct SID_MODEL_FILE
{
    SID_MODEL Model;

    double dfLatitude;
    double dfLongitude;
    double dfRefractionIndex;
    int iTDTOffset;

} SID_MODEL_BODY;
*/

/* pktSYNC=0x0e100f09 */

#define SYNC_BODY \
"NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN\n" \
"PPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPP\n" \
"OOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOO\n" \
"IIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIII\n"

#endif /* _PACKETDEF_H */
