/**************************************************************************
scanpkt.c

description:
------------

Scan for data packets in a DATAcon file, and print the timestamps
and other information from the body of the packet.

This program is based on pktlist.c written by Dave Buscher

functions: 
----------

modification history:
--------------------
11-Apr-1993  DFB pktlist.c created
15-Mar-2002  TAP First version of scanpkt.c
22-May-2002  TAP Last changes


***********************************************************************/

/* includes */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <unistd.h>
#include "defs.h"
#include "datacon.h"
#include "swap.h"
#include "sys_config.h"

/* global variable declarations */

#define MAXSTR  500             /* For sizing strings -- DON'T use BUFSIZ! */
#define STREQP(a, b) (*(a) == *(b) && strcmp((a), (b)) == 0)

int quiet = 0;                          /* Suppress messages */
int binarymode = 0;                     /* Whether to dump output as text
                                           or in binary directory format */
char *progname;
char *inname;                           /* filename for messages etc. */

#ifdef UTZOOERR
extern char *mkprogname();
#else
#define mkprogname(a) (a)
#endif

/*
 - main - parse arguments and handle options
 */
int main(int argc, char *argv[])
{
        int c;
        int errflg = 0;
        FILE *in;
        struct stat statbuf;
        extern int optind;
        extern char *optarg;
        void process();

        progname = mkprogname(argv[0]);
        while ((c = getopt(argc, argv, "bqh")) != EOF)
                switch (c) {
                case 'b':       /* Binary mode. */
		  binarymode++;
		  break;
                case 'q':       /* Debugging. */
                        quiet++;
                        break;
                case 'h':
                default:
                        errflg++;
                        break;
                }
        if (errflg || optind >= argc) {
                fprintf(stderr, "Usage: %s ", progname);
                fprintf(stderr, "[-bqh] file ...\n");
                fprintf(stderr,
			"Use the file name '-' to refer to standard input\n");
                fprintf(stderr,
			"Flags: b = binary output, q = quiet, h = help\n");
                exit(2);
        }
	for (; optind < argc; optind++)
	  if (STREQP(argv[optind], "-"))
	    process(stdin, "-");
	  else {
	    in = fopen(argv[optind], "r");
	    if (in == NULL){
	      printf("Can't open `%s'\n",argv[optind]);
	      exit(1);
	    }
	    if (fstat(fileno(in), &statbuf) < 0){
	      printf("Can't fstat `%s'\n", argv[optind]);
	      exit(1);
	    }
	    if ((statbuf.st_mode & S_IFMT) == S_IFDIR){
	      printf("`%s' is directory!\n", argv[optind]);
	      exit(1);
	    }
	    process(in, argv[optind]);
	    (void) fclose(in);
	  }
        exit(0);
}

/*
 - process - process input file
 */
void
process(FILE *in, char *name)
{
  typedef struct {
    uint8 PhotonCounts[VAR_NUM_SPEC][VAR_NUM_CHAN][NUM_BIN];
    uint32 TimeStamp;
  }FRINGE_BODY;
  
  DATAconHeader CurrentHeader,TempHeader;
  FRINGE_BODY FringeBody[VAR_NUM_REC];
  FDL_POSITION_RECORD FdlBody[VAR_NUM_REC];
  NAT_QUAD_COUNT_BODY NatBody;
  SCAN_START_BODY_VERSION_1 StartBody;
  STAR_ACQ_BODY AcquiredBody;
  int iPacket, iType, iHour, iMin, iuSec, iSec; 
  unsigned long iByteTotal=0;
  unsigned long iDay, jDay, iTime;
  char azReason[MAXSTR];
  float fTime,fSec;
  FILE *Logfile, *DirFile;
  char	config[100000];

  if (binarymode) Logfile = stderr; else Logfile = stdout;
  if (binarymode) DirFile = fopen("packet.dir","wb");

  if (!quiet) fprintf(Logfile, "File %s:\n\n", name);
  
  for (iPacket = 0; iPacket < 10000000; iPacket++) {
    fread(&TempHeader, sizeof(TempHeader), 1, in);
    if (ferror(in)) {
      strcpy(azReason, "File reading error");
      break;
    }
    if (feof(in)) {
      strcpy(azReason, "End of file (no end-of-disk packet)");
      break;
    }
    CurrentHeader.dchDay = swaplong(TempHeader.dchDay);
    CurrentHeader.dchTime = swaplong(TempHeader.dchTime);
    CurrentHeader.dchType = swaplong(TempHeader.dchType);
    CurrentHeader.dchLength = swaplong(TempHeader.dchLength);


/* Test sanity of current header */

    if (   CurrentHeader.dchLength < sizeof(DATAconHeader)
	|| CurrentHeader.dchLength > 2000000 ) {
      sprintf(azReason, "Bad packet header: %8x %6d %6d",
	      CurrentHeader.dchType,
	      CurrentHeader.dchLength - sizeof(DATAconHeader),
	      CurrentHeader.dchTime);
      break;
    }

    if (iPacket == 0) iDay = CurrentHeader.dchDay;

    if (binarymode){
      fwrite(&(CurrentHeader.dchType),sizeof(uint32),1, DirFile);
      fwrite(&(CurrentHeader.dchLength),sizeof(uint32),1, DirFile);
      fwrite(&(CurrentHeader.dchTime),sizeof(uint32),1, DirFile);
    } else {
      iType = CurrentHeader.dchType;
      jDay = CurrentHeader.dchDay;
      if (jDay == 0) jDay = iDay;
      /*
      fTime=(float)(CurrentHeader.dchTime+(jDay-iDay)*86400000)/3600000;
      */
      fTime=(float)(CurrentHeader.dchTime)/3600000;
      iHour=(int)fTime;
      iMin=(int)((fTime-iHour)*60);
      fSec=(fTime-iHour-iMin/60.0)*3600;
      switch(CurrentHeader.dchType){
	      case pktSYS_CONFIG :
		      fread(config,
			(CurrentHeader.dchLength - sizeof(DATAconHeader)), 1, in);
#if 0
		      printf("################\n");
		      printf("%*s\n", CurrentHeader.dchLength - sizeof(DATAconHeader),
				      (char *)config);
		      printf("################\n");
#endif
		      fseek(in, -(CurrentHeader.dchLength - sizeof(DATAconHeader)),
				      SEEK_CUR);
		      break;
        case pktSCAN_START_VERSION_1:
		fread(&StartBody,
		      (CurrentHeader.dchLength - sizeof(DATAconHeader)),1,in);
                printf(" SCAN_START_1  Scan: %o Star: %s\n",
		       swaplong(StartBody.ScanID),StartBody.StarID);
		fseek(in,-sizeof(StartBody),SEEK_CUR);
		break;
        case pktSCAN_START_VERSION_2:
                printf(" SCAN_START_2\n");
		break;
        case pktSTAR_ACQUIRED:
		fread(&AcquiredBody,
		      (CurrentHeader.dchLength - sizeof(DATAconHeader)),1,in);
                printf(" STAR_ACQUIRED Scan: %o\n",
		       swaplong(AcquiredBody.ScanID));
		fseek(in,-sizeof(AcquiredBody),SEEK_CUR);
		break;
        case pktSCAN_END:
                printf(" SCAN_END\n");
		break;
        case pktFRINGE_DATA_VERSION_1:
                printf(" FRINGE_DATA_1\n");
		break;
        case pktFRINGE_DATA_VERSION_2:
                printf(" FRINGE_DATA_2\n");
		break;
        case pktFRINGE_BG:
                 printf(" FRINGE_BG: 6-station packet\n");
		 printf("%8x %6d %6d %6d %2.2dh%2.2dm%4.1fs\n",
			CurrentHeader.dchType,
			CurrentHeader.dchLength - sizeof(DATAconHeader),
			CurrentHeader.dchTime,CurrentHeader.dchDay,
			iHour,iMin,fSec);
		fread(FringeBody,
		      (CurrentHeader.dchLength - sizeof(DATAconHeader)),1,in);
		fTime=(float)(swaplong(FringeBody[0].TimeStamp))/3600000;
		iHour=(int)fTime;
		iMin=(int)((fTime-iHour)*60);
		fSec=(fTime-iHour-iMin/60.0)*3600;
		printf("%d First fringe frame timestamp: %2.2dh%2.2dm%7.4fs\n",
		       iPacket,iHour,iMin,fSec);

		/* Move the file pointer back so we can skip to next header */
		fseek(in,-sizeof(FringeBody),SEEK_CUR);

		break;
        case pktFRINGE_DARK:
                printf(" FRINGE_DARK\n");
		break;
        case pktFDL_POSITION:
	  /*
                printf(" FDL_POSITION\n");
	  */
		fread(FdlBody,
		      (CurrentHeader.dchLength - sizeof(DATAconHeader)),1,in);
		/* The FDL time stamps appear to be in units of 50 usec. */
		fTime=(float)(swaplong(FdlBody[0].FDLPositions[0].uTime))/72000000;
		iHour=(int)fTime;
		iMin=(int)((fTime-iHour)*60);
		fSec=(fTime-iHour-iMin/60.0)*3600;
		printf("%d First FDL frame timestamp:    %2.2dh%2.2dm%7.4fs\n\n",
		       iPacket,iHour,iMin,fSec);

		/* Move the file pointer back so we can skip to next header */
		fseek(in,-sizeof(FdlBody),SEEK_CUR);

		break;
        case pktFDL_JITTER:
                printf(" FDL_JITTER\n");
		break;
        case pktFDL_STATUS:
                printf(" FDL_STATUS\n");
		break;
	case pktNAT_COUNTS:
		fread(&(NatBody),
			CurrentHeader.dchLength - sizeof(DATAconHeader),
			1, in);
		iTime=swaplong(NatBody.Records[0].uTime);
		iuSec = iTime % 1000;
		iTime -= iuSec;

		iSec = (iTime / 1000) % 60;
		iTime -= iSec * 1000;

		iMin = (iTime / (60 * 1000)) % 60;
		iTime -= iMin * 60 * 1000;

		iHour = (iTime / (3600 * 1000));
		iTime -= iHour * 3600 * 1000;

		printf("%4.4d First NAT frame timestamp: %2.2dh%2.2dm%2.2ds%4.4d -- %u %lu\n",
			iPacket, iHour, iMin, iSec, iuSec,
			swaplong(NatBody.Records[0].uTime), iTime);
		fseek(in, -sizeof(NatBody), SEEK_CUR);

	default:
		break;
      }
      iByteTotal+=CurrentHeader.dchLength;
    }
    if (CurrentHeader.dchType == pktEND_OF_DISK) {
      strcpy(azReason, " End of disk packet");
      break;
    }

/* Next Packet */

    fseek(in,
	  CurrentHeader.dchLength - sizeof(DATAconHeader),
	  SEEK_CUR);
  }
  if (!quiet) {
    fprintf(Logfile," Read %d packets.\n Termination reason = %s\n", 
		      iPacket, azReason);
    fprintf(Logfile," Total number of bytes in packets: %ld\n",iByteTotal);
    }
}
