/***********************************************************************
 * Edge: Determines the (Theta, Phi) values for the edge of the outline
 * of the star along with r, given the angle, Theta' in the projected
 * X-Y plane (S->W).  All you need to give it besides Theta' is the
 * scaled angular velocity and the inclination.
 **********************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "rochechi2.h"

int Edge(Roche * R, float ThP, float * Th, float * Phi, float * rout) {

  float incl=R->fIncl;
  float th, r, sth, sth2;
  float w=R->fwOmega, w2=w*w, si=sin(incl), ci=cos(incl); 
  //int IfWrite=0; // For debugging

  /* Force 0 <= ThP < 2PI */
  while (ThP < 0.) {
    ThP += (float)TWOPI;
  } 
  while (ThP >= (float)TWOPI) {
    ThP -= (float)TWOPI;
  }

  /* Incl = 90 => theta = 180-theta' */
  if (fabs((fabs(incl) - (float)PI2)) < 1.e-5) {
    th = (float)PI - ThP;
    sth = sin(th);
    sth2 = sth*sth;
    r = FractRadius(w2, sth2);
    *Th = th;
    *Phi = (float)PI2;
    if (ThP > (float)PI) {
      *Phi += (float)PI;
      *Th = -th;
    }
    *rout = r;
    return EXIT_SUCCESS;
  }

  /* Incl = 0 => phi = theta', theta = 90 */
  if (fabs(incl) < 1.e-5) {
    th = (float)PI2;
    *Th = th;
    *Phi = ThP;
    *rout = FractRadius(w2, 1.);
    return EXIT_SUCCESS;
  }

  /* ThP=+/-90 case for all inclinations */
  if (fabs(fabs(ThP) - (float)PI2) < 1.e-5) {
    *Th = (float)PI2;
    *Phi = (float)PI2;
    if (ThP < 0.) *Phi += (float)TWOPI;
    *rout = FractRadius(w2, 1.);
    return EXIT_SUCCESS;
  }
    
  /* Finally, ThP=+/-270 case for all inclinations */
  if (fabs(fabs(ThP) - 3.*(float)PI2) < 1.e-5) {
    *Th = (float)PI2;
    *Phi = (float)PI2;
    if (ThP > 0.) *Phi += (float)PI;
    *rout = FractRadius(w2, 1.);
    return EXIT_SUCCESS;
  }

  /* else try just iterating */
  {float tanth0=0., tanth=1., a, eight27= 8./27., tanthp=tan(ThP), 
    onema, tanph=1., tanph0=0., cph;
  int ic=0;

  r=FractRadius(w2, 1.); // Start with the max r for this w
  if (fabs(w2-1.) < 1.e-3) r=1.49; // But beware of the singularity

  /*
  IfWrite = 0;
  //if (fabs(ThP*(float)DEGRAD-5.) < .1) IfWrite = 1;

  if (IfWrite) {
    printf("In Edge: ");
    printf("ThP=%1.6g rnot=%1.6g\n", ThP*(float)DEGRAD, r);
  } // end IfWrite
  */

  if (fabs(incl) > (float)PI2/2.) {
    /* if |inclination| > 45, converge on theta (phi => 90 as i => 90) */
    while (fabs(tanth0-tanth)/(fabs(tanth)+1.) > 1.e-5 && ic < 30) {
      tanth0 = tanth;
      a = eight27 * w2 * r * r * r;
      onema = 1. - a;
      tanph = tanthp * (1. - a * si * si) / ci;
      cph = sqrt(1. / (1. + tanph * tanph));
      if ((float)PI2 < ThP && 3.*(float)PI2 > ThP) cph = -cph;
      tanth = -ci / (onema * cph * si);
      sth2 = 1. / (1.+1. / (tanth * tanth));
      r = FractRadius(w2, sth2);

      /*
      if (IfWrite) {
	printf("i.gt.45, 1-a=%1.6g tanth=%1.6g tanphi\n",
	       onema, tanth, tanph);
      } // end IfWrite
       */
      
      ic++;
    } // and iterate
  }
  else {
    /* if inclination <= 45, converge on theta (phi => 90 as i => 90) */
    while (fabs(tanph0-tanph)/(fabs(tanph)+1.) > 1.e-5 && ic < 30) {
      tanph0 = tanph;
      a = eight27 * w2 * r * r * r;
      onema = 1. - a;
      tanph = tanthp * (1. - a * si*si)/ci;
      cph = sqrt(1. / (1. + tanph * tanph));
      if ((float)PI2 < ThP && 3.*(float)PI2 > ThP) cph = -cph;
      tanth = -ci / (onema * cph * si);
      sth2 = 1. / (1.+1. / (tanth * tanth));
      r = FractRadius(w2, sth2);

      /*
      if (IfWrite) {
	printf("i.le.45, 1-a=%1.6g tanth=%1.6g tanphi=%1.6g\n", 
                onema, tanth, tanph);
      } // end IfWrite
       */
      
      ic++;
    } // and iterate
  }

  //printf("ic=%1i tanph=%1.6g tanth=%1.6g r=%1.6g\n", 
  //ic, tanph, tanth, r);
  if (ic >= 30) {
    printf("Edge didn't converge, Th'=%1.6g incl=%1.6g", 
	   ThP*(float)DEGRAD, incl*(float)DEGRAD);
    if (fabs(incl) > (float)PI2/2.) {
      printf(" tanth=%1.6g |tanth0-tanth|/(|tanth|+1)=%1.6g", 
	     tanth, fabs(tanth0-tanth)/(fabs(tanth)+1.));
    }
    else {
      printf(" tanph=%1.6g |tanph0-tanph|/(|tanph|+1)=%1.6g", 
	     tanph, fabs(tanph0-tanph)/(fabs(tanph)+1.));
    }
    printf("\n");
  }
  *Th = atan(tanth);
  if (*Th < 0.) *Th += (float)PI;
  if (ThP == 0. || ThP == (float)PI) { // then Phi = ThP!
    *Phi = ThP;
  }
  else {
    *Phi = atan(tanph);
  }
  if (*Phi < 0.) *Phi += (float)PI;
  if (ThP > (float)PI) {
    *Phi += (float)PI; // and right hemisphere
  }
  *rout = r;
  } // scope iterating variables

  //return EXIT_FAILURE;
  return EXIT_SUCCESS;
}
