/***********************************************************************
 * FractRadius: 
 * Calculates the radius as a function of the co-latitude for a Roche
 * Spheroid.  The variables are small Omega and sin theta, both squared.
 * 
 * Note: This gives the radius in units of the Polar Breakup Radius
 * which is less than 1, even when w=0.
 *
 * Note: Following Hardorp and Scholz, have suppressed the shrinkage of
 * the polar axis with omega.  It's =1 now.  For good.
 *
 * It converges slowly near critical so I've added logic that starts
 * from the high side (Xo=1.5) using 4/27-A as the driving parameter.
************************************************************************/
#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "rochechi2.h"

float FractRadius (float w2, float s2) {

  float dy, dydx;
  const float four27=4./27., B=1., Acut=.125; // Acut=1/8!
  float Xo = 1.; // 1.0 le Xo le 1.5!
  //float scalepole = 0.1087 ; // Approximately keeps the model at const lum.
  //float scalepole = 0.; // No it didn't!  This is much better!!
  //float B = (1.-scalepole)/(1.-scalepole*w2);
  float A = four27*w2*s2;
  float D = four27-A; // For starting with Xo=1.5
  float a, b, c;
  float dx = Xo;
  int ic=0;

  /*  Use a second order expansion to get the first step. Split the
   *  interval into low (A <= Acut) and high (A > Acut) rotation *
   *  latitude.  For the latter, expand around D = 4/27 - A, Xo=3/2.
   *
   *  Acut is choosen to be the critical A for the Xo=1 expansion
   *  (discriminate goes to zero at A=1/8).  It could probably be
   *  choosen a little lower, but it has no effect so long as we stay in
   *  single precision.
   */

  if (A <= Acut) {
    Xo=1.;
    a=1.+A;
    b=(.5-A)/a;
    c=(1-B/a);
    dx = b-sqrt(b*b-c);
    //printf("a=%f b=%f c=%f dx=%f ",a,b,c,dx);
    Xo += dx;
    //printf("Xo=%f ",Xo);
    if (dx < .004) return Xo;
  }

  if (A > Acut) {
    Xo=1.5;
    b=2.25*D;
    a=B-b;
    dx = Xo*(b-sqrt(b))/a;
    //printf("a=%f b=%f dx=%f ",a,b,dx);
    Xo += dx;
    //printf("Xo=%f ",Xo);
  }

  while (fabs(dx/Xo) > 1.e-4 && ic <= 20) {
    dy = (1./Xo+A*Xo*Xo-B);
    dydx = 1./Xo/Xo-2.*A*Xo;
    dx = dy/dydx;
    Xo += dx;
    //printf("\n  ic=%1i dy=%1.6g dydx=%1.6g dx=%1.6g Xo=%1.6g ",
    //ic, dy, dydx, dx, Xo);
    ic++;
  }
  //printf("\nFR A=%f D=%f ic=%i Xo=%f\n",A,D,ic,Xo);
  if (ic > 20)printf("FR: ic > 20 without converging! Continuing.\n");
  return Xo;
}
