/***********************************************************************
 * RThetaPhifromXY: Calculates the underlying (r, theta, phi)
 * coordinates of a Roche Spheroid, given the coordinates of a point on
 * the surface as projected on the sky.  Assumes the dimensionless
 * angular velocity and the inclination are given.

 * Three cases are considered: 
 *
 *   1) First we check the "critical" radius, rc=sqrt(X'*X'+Y'*Y').  If
 *   that is less than the polar radius then it can never be reached and
 *   simple iterations work well.
 *
 *   2) If rc > rp but starting with rc and iterating once moves the
 *   result in the opposite direction than starting with re and
 *   iterating once, then a simple "divide by 2" search scheme works
 *   really well.
 *
 *   3) However, if in 2) iterating from rc and re both move in the same
 *   direction (this only happens when you're really near the edge) then
 *   we go to a Newton-Raphson procedure.  Here the critical ingredient
 *   is the starting guess.  For that we use the edge values just beyond
 *   (X', Y') as seen from the subsolar point (ie, we use Edge with
 *   these values of (X', Y') defining theta').
 *
 *   Actually, 3) is more complicated than that.  At, literally, the
 *   edge, the first derivative vanishes.  Further, there's no clear way
 *   to signal that you want the front side solution versus the back
 *   side.  This is all taken care of if you do a second order expansion
 *   for the Newton-Raphson.  Then, at the edge you still get a solution
 *   AND you get to choose which branch you want.  You also continue to
 *   include the linear term and, as you move away from the edge you
 *   correctly allow for it to grow, eventually dominating the quadratic
 *   term and bringing you back to the usual first order Newton-Raphson.
 *   These are very nice - actually critical - properties.
 *
 *   Added "IW" in case calling program wants diagnostic output at
 *   a specific point.
 *
 *   Fixed bug in setting radical sign in 2nd-order Newton-Raphson
 *   12/22/03
 ************************************************************************/
#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "roche.h"

/*
int RThetaPhifromXY(float X, float Y, Roche * R, float * rout, 
		     float * Th, float * Phi, int IW) {
*/

int RThetaPhifromXY(float X, float Y, Roche * R, float * rout, 
		     float * Th, float * Phi) {
		     
  float th, phi, sth2, sth, cth2, cth, sphi, cphi, incl=R->fIncl, 
    si=sin(incl),ci=cos(incl), w2=R->fwOmega*R->fwOmega;
  int ic=0, ifsuccessfail=EXIT_SUCCESS;
  float r=0, r0, rc2=X*X+Y*Y, rc=sqrt(rc2), arg;
  float rl0, rl1, drl, rs0, rs1, drs, rm0, rm1, drm, adr;
  float thp, dxp, drdth, d2rdth2, r2, dth, tthc=.2962963, a;
  float thedge, phiedge, redge, rad, Ars, Brs, Acoeff, Bcoeff, Ccoeff;
  float r2c2s2, frdsc, drdth2;
  int IfWrite, IW=0;

  IfWrite = 0;
  if (IW) printf("RTHPhifromXY: X=%g Y=%g\n",X,Y);

  rl0=FractRadius(w2, 1.); // equatorial radius
  r0=rl0;
  rs0=FractRadius(w2, 0.); // polar radius


  /* rc is accessible. Check whether we can do a simple divide by 2
     search */
  if (rs0 <= rc) {
    if (IW) printf("rc is accessible: rc=%g rs0=%g\n",rc,rs0);

    cth=(-X*si+sqrt(rl0*rl0-rc2)*ci)/rl0;
    sth2=1.-cth*cth;
    rl1=FractRadius(w2,sth2);
    drl=rl0-rl1;
    if (IW) printf("rl0=%1.6g rl1=%1.6g drl=%1.6g\n", rl0, rl1, drl);
    adr=fabs(drl);

    /* Found the solution? */
    if (adr <= 1.e-5) {
      sth = sqrt(sth2);
      sphi = Y/rl0/sth;
      if (R->fIncl < (float)PI2) {
	cphi = (X+rl0*cth*si)/(rl0*sth*ci);
	*Phi = (float) atan2(sphi,cphi);
      }
      else { // in case i = 90deg
	*Phi = asin(sphi);
      }
      *Th = (float) acos(cth);
      *rout = rl0;
      return EXIT_SUCCESS;
    }
  
    rs0=rc;
    arg=0.;
    cth=(-X*si+sqrt(arg)*ci)/rs0;
    sth2=1.-cth*cth;
    rs1=FractRadius(w2,sth2);
    drs=rs0-rs1;
    /* Found the solution? */
    if (fabs(drs) <= 1.e-5) {
      sth = sqrt(sth2);
      sphi = Y/rs0/sth;
      if (R->fIncl < (float)PI2) {
	cphi = (X+rs0*cth*si)/(rs0*sth*ci);
	*Phi = (float) atan2(sphi,cphi);
      }
      else { // in case i = 90deg
	*Phi = asin(sphi);
      }
      *Th = (float) acos(cth);
      *rout = rs0;
      return EXIT_SUCCESS;
    }
  

    if (IW) printf("rs0=%1.6g rs1=%1.6g drs=%1.6g", rs0, rs1, drs);
    if (IW) printf(" dlr*dsr=%g\n",drl*drs);
 
    if (drl*drs < 0.) {
      /* Signs change, divide by 2 will work */
      if (IfWrite) printf("/2: dr switches signs\n");
      ic=0;
      while (adr > 1.e-5 && ic < 40) {
      
	/* Now search by halving the r interval */
	rm0=(rl0 + rs0)/2.;
	arg=rm0*rm0-rc2;
	if (arg < 0.) {
	  rm0=rc;
	  arg=0.;
	}
	cth=(-X*si+sqrt(arg)*ci)/rm0;
	sth2=1.-cth*cth;
	rm1=FractRadius(w2,sth2);
	drm=rm0-rm1;
	if (IfWrite) printf("ic=%2i rm0=%1.6g drm=%1.6g\n", ic, rm0, drm);

	if (drl*drs < 0.) {
	  /* dr switches signs over rl<->rs */
	  if (drm*drl < 0.) {
	    rs0=rm0;
	    drs=drm;
	    if (fabs(drm) < adr) {
	      adr = fabs(drm);
	      r=rm1;
	    }
	  }
	  else {
	    rl0=rm0;
	    drl=drm;
	    if (fabs(drm) < adr) {
	      adr = fabs(drm);
	      r=rm1;
	    }
	  }
	}

	/* move same direction, have to look at smallest dr */
	if (drl*drs > 0.) {
	  if (IfWrite) printf("/2: dr doesn't switch signs\n");
	  if (fabs(drl) < fabs(drs)) {
	    rs0=rm0;
	    drs=drm;
	    if (fabs(drm) < adr) {
	      adr = fabs(drm);
	      r=rm1;
	    }
	  }
	  else {
	    rl0=rm0;
	    drl=drm;
	    if (fabs(drm) < adr) {
	      adr = fabs(drm);
	      r=rm1;
	    }
	  }
	}
	r0=rm0;
	ic++;
      }
      if (ic >= 40) {
	printf("RThPh<-XY /2: ic=%2i X=%1.6g Y=%1.6g", ic, X, Y);
	printf(" r=%1.6g adr=%1.6g rc=%1.6g\n", r, adr, rc);
      }
    }


    /* Doesn't switch signs.  /2 often doesn't work.  Use N-R with Edge
       values to start. */
    else {

      thp=atan2(Y, X);
      if (Edge(R, thp, &thedge, &phiedge, &redge) == EXIT_FAILURE) 
	return EXIT_FAILURE;

      /*
      //if (Y > .14 && fabs(X) > 1.32 && fabs(X) < 1.34 ) {
      //if (Y < -.14 && fabs(X) > 1.13 && fabs(X) < 1.16 ) {
	printf("same dir, Edge: rE=%1.6g thE=%1.6g phE=%1.6g", redge, 
	       (float)DEGRAD * thedge, (float)DEGRAD * phiedge);
	printf(" X'=%1.6g Y'=%1.6g\n", X, Y);
	printf("      Th'=%1.6g X'e=%1.6g Y'e=%1.6g\n", thp*(float)DEGRAD, 
	       redge*(sin(thedge)*cos(phiedge)*ci-cos(thedge)*si),
	       redge*sin(thedge)*sin(phiedge));
      //}
      */

      if (thp < 0.) thp += (float)TWOPI;

      th = thedge;
      phi = phiedge;
      r = redge;
      sth=sin(th);
      sth2 = sth*sth;
      ic=0;
      dxp=1.;
      if (thp >= (float)PI2 && thp < (float)(TWOPI - PI2)) {
	/* negative radical*/
	while (ic < 40 && (fabs(dxp) > 1.e-5)) {
	  cth = cos(th);
	  cth2 = cth*cth;
	  r2 = r*r ;
	  a = tthc*w2*r*r2;
	  drdth = a*r*sth*cth/(1.-a*sth2);
	  rad = -sqrt(r2*sth2 - Y*Y);
	  dxp = rad*ci-r*cth*si-X;
	  Ccoeff=dxp;
	  Ars = 2.*r*sth*(drdth*sth+r*cth);
	  Bcoeff = .5*Ars/rad*ci - (drdth*cth-r*sth)*si;
	  r2c2s2=r2*(cth2-sth2);
	  frdsc=4.*r*drdth*sth*cth;
	  drdth2 = drdth*drdth;
	  d2rdth2 = (drdth2 *(2.+a*sth2) + a*(frdsc + r2c2s2))/r/(1.-a*sth2);
	  Brs = (r*d2rdth2 + drdth2)*sth2 + frdsc + r2c2s2;
	  Acoeff = (Brs/rad/2. - Ars*Ars/rad/rad/rad/8.)*ci 
	    -(d2rdth2/2.*cth - drdth*sth - r/2.*cth)*si;
	  /* Always go to smaller theta, that's the near side */
	  dth = (-Bcoeff -sqrt(Bcoeff*Bcoeff - 4.*Acoeff*Ccoeff))/2./Acoeff;
	  /*
	  //if (Y < -.14 && fabs(X) > 1.13 && fabs(X) < 1.16 ) {
	    printf("ic=%2i A=%1.6g B=%1.6g C=%1.6g rad=%1.6g dth=%1.6g\n", 
	           ic, Acoeff, Bcoeff, Ccoeff, rad, (float)DEGRAD * dth);
	    printf("    a=%1.6g drdth=%1.6g d2rdth2=%1.6g",a,drdth,d2rdth2);
	    printf(" Ars=%1.6g Brs=%1.6g\n", Ars, Brs);
	    printf("    4r(drdth)sc=%1.6g r2c2s2=%1.6g\n", frdsc, r2c2s2);
	    //}
	  */
	  th += dth;
	  sth = sin(th);
	  sth2 = sth*sth;
	  r = FractRadius(w2, sth2);
	  ic++;
	}
	sphi = Y/(r*sth);
	cphi = rad/(r*sth);
	phi = atan2(sphi, cphi);
      }

      else {
	/* Or the other side */
	while (ic < 40 && (fabs(dxp) > 1.e-5)) {
	  cth = cos(th);
	  cth2 = cth*cth;
	  r2 = r*r ;
	  a = tthc*w2*r*r2;
	  drdth = a*r*sth*cth/(1.-a*sth2);
	  rad = sqrt(r2*sth2 - Y*Y);
	  dxp = rad*ci-r*cth*si-X; 
	  Ccoeff=dxp;
	  Ars = 2.*r*sth*(drdth*sth+r*cth);
	  Bcoeff = .5*Ars/rad*ci - (drdth*cth-r*sth)*si;
	  r2c2s2=r2*(cth2-sth2);
	  frdsc=4.*r*drdth*sth*cth;
	  drdth2 = drdth*drdth;
	  d2rdth2 = (drdth2 *(2.+a*sth2) + a*(frdsc + r2c2s2))/r/(1.-a*sth2);
	  Brs = (r*d2rdth2 + drdth2)*sth2 + frdsc + r2c2s2;
	  Acoeff = (Brs/rad/2. - Ars*Ars/rad/rad/rad/8.)*ci 
	    -(d2rdth2/2.*cth - drdth*sth - r/2.*cth)*si;
	  /* Always go to smaller theta, that's the near side */
	  dth = (-Bcoeff +sqrt(Bcoeff*Bcoeff - 4.*Acoeff*Ccoeff))/2./Acoeff;
	  /*
	  //if (Y < -.14 && fabs(X) > 1.14 && fabs(X) < 1.16 ) {
	    printf("ic=%2i drdth=%1.6g d2rdth2=%1.6g",ic,drdth,d2rdth2);
	    printf(" A=%1.6g B=%1.6g C=%1.6g\n",Acoeff,Bcoeff,Ccoeff);
	    printf("     dth=%1.6g rad=%1.6g Ars=%1.6g Brs=%1.6g\n", 
	           (float)DEGRAD * dth, rad, Ars, Brs);
	    printf("    4r(drdth)sc=%1.6g r2c2s2=%1.6g\n",frdsc,r2c2s2);
	    //}
	  */
	  th += dth;
	  sth = sin(th);
	  sth2 = sth*sth;
	  r = FractRadius(w2, sth2);
	  ic++;
	}
	sphi = Y/(r*sth);
	cphi = rad/(r*sth);
	phi = atan2(sphi, cphi);
      }

      if (ic >= 10) {
	printf("RThPh<-XY NR: ic=%2i X=%1.6g Y=%1.6g dx=%1.6g", 
	       ic, X, Y, dxp);
      }
      /*
	printf("RThPhXY: ic=%2i X=%1.6g Y=%1.6g dxp=%1.6g", ic, X, Y, dxp);
	printf(" th=%1.6g phi=%1.6g r=%g\n",
	       th*(float)DEGRAD,phi*(float)DEGRAD,r);
      */
      *rout = r;
      *Th = th;
      *Phi = phi;

      return EXIT_SUCCESS;
    }
  }


  /* Hand over to straight iteration */
  else {
    if (IfWrite) printf("rc inaccessible, straight iteration\n");
    while (fabs(r-r0) > 1.e-5 && ic < 100) {
      r=r0;
      arg = r*r-rc2;
      if (arg < 0.) arg=0.;
      cth = (-X*si+sqrt(arg)*ci)/r; // This should be cos_theta
      cth2 = cth*cth;
      sth2 = 1.-cth2;
      r0 = FractRadius (w2, sth2);
      if (IfWrite) {
	printf("RThetaPhi<-XY: arg=0, X=%1.6g Y=%1.6g rc=%1.6g", X, Y, rc); 
	printf(" cth=%1.6g r=%1.6g r0=%1.6g dr=%1.6g\n", cth, r, r0, r-r0);
      }
      if (r0 < rc) {
	r0=rc;
      }
      r0=.4*r0+.6*r; // Bring it in slowly
      ic++;
    }
    if (ic >= 100) {
      printf("RThetaPhifromXY Didn't converge: r - r0=%1.6g\n", r-r0);
      ifsuccessfail = EXIT_FAILURE;
    }
    if (IfWrite) printf("RThPhiXY: Straight iterate, ic=%2i\n", ic);
  }

  sth = sqrt(sth2);
  sphi = Y/r0/sth;
  if (R->fIncl < (float)PI2) {
    cphi = (X+r0*cth*si)/(r0*sth*ci);
    *Phi = (float) atan2(sphi,cphi);
  }
  else { // in case i = 90deg
    *Phi = asin(sphi);
  }
  *Th = (float) acos(cth);
  *rout = r0;
  if (IfWrite) printf("Inside RThetaPhiXY: Phi=%1.6g Th=%1.6g r=%1.6g\n",
		      *Phi,*Th,*rout); 

  return ifsuccessfail;
}
