/* This calculates the properties of Roche Spheroids. 
 * 
 * Version 1.1: Includes the transformation of the baselines from the
 * ground to the star.  Nominally for Hour Angle = 0, but it can be set
 * for anything.
 *
 * Version 1.2: Adds in a computation of the UV plane coverage.  Since a
 * range of Hour Angles is required, this adds a do loop and does the
 * observables for all them (from a fixed set entered at the beginning
 * of Roche.c).  Fixed a bug in RThetaPhifromXY that blew up near the
 * equator.  Added a calculation of the apparent major and minor axes of
 * the inclined disk (in milliarcseconds).  Adds a PA offset to the
 * image drawn in Picture.c, and switching to angular (mas) scales.
 *
 * Version 1.3: Splits up the surface brightness calculation (after
 * Christian complained) into two parts.  First, the structure of the
 * surface that depends only on the geometry (location of pixels and the
 * values of (Teff, log_g, mu) associated with them) resolved along and
 * perpendicular to each baseline is calculated (Surface.c) (that is
 * fairly time consuming).  Then the strip brightness distribution can
 * be calculated for each wavelength fairly quickly.  This required a
 * major rewrite of StripBright.c.  The geometry calculation at somewhat
 * finer resolution will be used to calculate the projected area, the
 * Visual flux and B-V color (without reference to the baselines) (not
 * yet implemented).
 *
 * Version 1.4: Reworks the graphical output.  Switches to a scheme that
 * does one inclination and one triangle per plot page.  That allows the
 * triple amplitude to be added along with graphics showing the UV
 * coverage and the outline of the object on the sky, indicating PA(s)
 * and inclination.  Up to 6 PA's can be plotted on each plot.  We now
 * allow this to be plotted against wavelength or channel number, your
 * choice. 
 * 
 * A plotting switch has been included in the plot parameter file.  It
 * binary encodes which combination of plots you want.  See the header
 * file for how to set the code.  But in short you can get just a
 * summary plot, described above and/or full page versions of each of
 * the visibilities and/or phases.
 *
 * Further on Version 1.4, we decided to move the read in value for the
 * disk scaling factor, fAngDiam_pB, to the inclinations file, since the
 * major axis of the projected image is always so strongly dependent on
 * rotation near breakup. From now on the first two options in the
 * program - to print out the main parameters of the rotating star and
 * to look at renderings of the star - will be run off the inclinations
 * file as soon as the request for a single wavelength is seen. 09/10/03
 *
 * Fixed PA error (wasn't converting to radians!). Also, this version
 * now forces RpB=1 for all omega (see comments in FractRadius). 09/02/04
 *
 * Forces the correct wavelength index for access to the limb darkening
 * array.  Bug pointed out by Christian Hummel 26/02/04
 */

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <math.h>

#include "cpgplot.h"
#include "roche.h"

Roche R; // Model and derived parameters
LimbDark LD; // Limb Darkening data
Observatory O; // Observatory parameters
Disk D[IBMAX][IPAMAX]; // Stellar disks, for each (baseline,PA)
Plot P; // Information for the plotting subroutine
Scans S; // Storage for the data

int main(int iargc, char *acargv[]) {

  float Incl[10], W[10], DpBmas[10], ThPA; 
  float Bright[ISTRIPMAX],X[ISTRIPMAX],Uniform[ISTRIPMAX];
  float Vis[6][15][32], phase[6][15][32], Vis2[6][15][32]; 
  float ReV[6][15][32], ImV[6][15][32], phiClose, amplitudeClose;
  int IfWrite, IfRotate=0, iBaseline, iIncl, iInclmax=10, imagedev;
  int i, icp, inl, j, j1Wave, j2Wave, iHA, ib;
  char YN[]=" ";
  char name[10];
  char sIncl[5], sPlotFile[40];

  /* To calculate Visibilities */
  float cmasnmm=1.e9/(3.6e6*(float)DEGRAD); // cycles/mas*nm/m
  float Re=0., Im=0., ReC, ImC, u, xi, delta, dsincu;
  float sum=0., sumu=0.;

  int iTri=0, iB1, iB2, iPAg, iPA;
  //int IfWave=0; // whether to plot against wavelength (=1) or channel #

  /* 
   * There are two arguments in the call list.  The first is the model
   * name (just the name, we add the ".dat" later).  There the program
   * picks up the Mass (Msun); Radius (Rsun), the radius at zero
   * rotation; TeffpB (K), the effective Temperature the pole would have
   * at breakup; AngDiam_pB, the polar (smallest) angular diameter you
   * would get at breakup; and two parameters describing the limb
   * darkening set: LDLaw, the limb darkening law to use and Waveset,
   * which NPOI wavelength set to use.  The limb darkening file name is
   * constructed, in part, from these.
   *
   * The second command line parameter is an index.  If the index is
   * positive and <= indexWaveMax (=32) it does that many channels
   * starting with the longest wavelength.  If the index is negative it
   * does the wavelength for that (positive) index only.  An index of
   * zero does 500nm. An index whose absolute value is greater than
   * indexWaveMax is assumed to be a wavelength in nanometers.
   *
   * We either read in the NPOI wavelengths with the file containing the
   * bandpass averaged limb darkening coefficients if the absolute value
   * of the index is <= 32 or, if a single wavelength is called out, we
   * read in the limb darkening file for monochromatic wavelengths and
   * interpolate values from that.
   */

  if (InitRoche(&R, &LD, name, iargc, acargv, &j1Wave, &j2Wave) 
      == EXIT_FAILURE) return EXIT_FAILURE;

  /* Read in the Inclinations and angular velocities, wOmega. */
  if (ReadInclinations(&R, iInclmax, Incl, W, DpBmas, &iIncl) == 
      EXIT_FAILURE) return EXIT_FAILURE;
    
  /* 
   * For a single wavelength, just do model parameters and maybe look at
   * a rendering.
   */ 
  if (j2Wave - j1Wave <= 1) {

    for (i=0; i<iIncl; i++) {
      R.fIncl = Incl[i];
      R.fwOmega = W[i];
      R.fAngDiam_pB = DpBmas[i];
      printf("\ni=%1i Incl=%1.6g W=%1.6g DpB=%1.6g\n", i, 
	     Incl[i]*(float)DEGRAD, W[i], DpBmas[i]);
      MainVar(&R, &LD, 1);
    }    

    printf ("Want to draw a spheroid? (Y/N): ");
    scanf("%s",YN);
    getchar();
    if (toupper(YN[0]) == 'Y') {
      printf("Position Angle of the axis (N->E) (deg): ");
      scanf("%f",&ThPA);
      getchar();
      ThPA = (float)RADDEG*ThPA;
      if (imagedev = cpgopen("?") < 1) return EXIT_FAILURE;
      //printf("Image Device index=%1i\n", imagedev);
      for (i=0; i<iIncl; i++) {
	R.fIncl = Incl[i];
	R.fwOmega = W[i];
	R.fAngDiam_pB = DpBmas[i];
	//if (i > 0) cpgpage();
	MainVar(&R, &LD, 0);
	/* Draw a tilted spheroid */
	if (Picture(&R, &LD, ThPA, name, imagedev, IfRotate) == 
	    EXIT_FAILURE) return EXIT_FAILURE;
      }
      cpgclos();
    }
  }

  /* 
   * Next do the detailed Visibility/Phase plots for the model
   * parameters as a function of Position Angle (within the plots) and
   * inclination (as read in from the "inclinations" file) for a given
   * array geometry.  All this done over a range of Hour Angles.
   */

  else {
    P.j1Wave=j1Wave; // Need these lots of places
    P.j2Wave=j2Wave;

    //printf( "\nj2 > j1+1: j1=%1i j2=%1i\n\n", j1Wave, j2Wave); 

    /* Read in the Baselines and Triangles from a file.  The latter are
       for Closure Phases. */

    printf("\n");
    if (ReadBaselines(&O, &R) == EXIT_FAILURE) return EXIT_FAILURE;
    iBaseline=O.iBaselines;

    printf("\nMs=%1.6g Ro=%1.6g Teff(pB)=%1.6g\n", R.fMass, R.fRNot, 
	   R.fTeffpB);
    
    /* Read in the observations (Scans). */
    if (ReadScans(&S) == EXIT_FAILURE) return EXIT_FAILURE;

    /* Initialize Plotting Routine */
    //if (InitPlot(&R, &O, &LD, &S, &P, IfWave) == EXIT_FAILURE) //old
    if (InitPlot(&R, &O, &LD, &S, &P) == EXIT_FAILURE) return EXIT_FAILURE;

    /* Do over (inclination, omega)'s */
    for (inl = 0; inl < iIncl; inl++) {
      P.iInl = inl; // for plot routine
      R.fIncl = Incl[inl];
      R.fwOmega = W[inl];
      R.fAngDiam_pB = DpBmas[inl];
      MainVar(&R, &LD, 0); // calculate all the physical variables
      /* Open the window for the complicated plot */

      /* Need to set up file name, if writing a file */
      if (P.IfScreen) {
	if (cpgopen(" ") < 1) return EXIT_FAILURE; // to screen
      }
      else {
	strcpy(sIncl,"");
	if (iIncl > 1) sprintf(sIncl,"%g",(float)DEGRAD*Incl[inl]);
	strcpy(sPlotFile,P.sBaseFile);
	strcat(sPlotFile,"_");
	strcat(sPlotFile,sIncl);
	strcat(sPlotFile,P.sExtension);
	strcat(sPlotFile,P.sDevice);
	strcpy(P.sIncl,sIncl);
	strcpy(P.sPlotFile,sPlotFile);
	printf("Made up sPlotFile=%s\n",sPlotFile);
	if (cpgopen(sPlotFile) < 1) return EXIT_FAILURE; // to file
      }

      /* Over Observations (Hour Angles) */
      //printf("Start HA loop. iNScans=%i\n",S.iNScans);
      for (iHA = 0; iHA < S.iNScans; iHA++) {
	P.iHA = iHA; // for plot routine 

	/* Transform to Baseline projected on the stellar disk */
	//printf("iHA=%i Call GetUVW\n",iHA);
	if (GetUVW(&O, &R, S.afHA[iHA]) == EXIT_FAILURE) return EXIT_FAILURE;
	/*
	printf("iBaselines=%i\n",O.iBaselines);
	  for (i=0; i<O.iBaselines; i++) {
	  printf("iHA=%i iBl=%i U=%g Th=%g\n",
	  iHA,i,O.fUTheta[i][0],(float)DEGRAD*O.fUTheta[i][1]);
	  }
	  //printf("\n");
	*/

	/* 
	 * There will be P.iPAnot groups of Position Angles, each of
	 * which needs to be plotted on separate pages.  This will allow
	 * us to plot 6, say, PAs on a plot and do several of them,
	 * showing the whole range.  
	 */
	for (iPAg = 0; iPAg < P.iPAnot; iPAg++) {
	  P.iPAg = iPAg; // for plot routine
	  //printf("iPAg=%i iPAnot=%i\n",iPAg, P.iPAnot);

	  /* 
	   * Do the Disk pretabulations first.
	   *
	   * There will be P.iPAdel different angles graphed on a PGPLOT
	   * page, starting at P.afPAnot[i] and incremented by P.fPAdel.
	   */
	  for (iPA = 0; iPA < P.iPAdel; iPA++) {

	    //printf("\n**********iPA=%i inl=%i iHA=%i\n", iPA, inl, iHA);
	    /* Set up Disk map for the baselines at these HA's*/
	    for (ib=0; ib<O.iBaselines; ib++) {
	      //printf("ib=%i iBaselines=%i\n",ib,O.iBaselines);
	      D[ib][iPA].iB = ISTRIPMAX;
	      /*
		if (ib == 2 && iPA == 5 && inl == 0) {
		D[ib][iPA].iB = -ISTRIPMAX;
		printf("Baseline: %s ib=%i\n",
		O.sBaselineNames[ib][0],ib);
		}
	      */
	      //printf("Baseline: %s ib=%i\n",O.sBaselineNames[ib][0],ib);

	      if (Surface(&R, &D[ib][iPA], O.fUTheta[ib][1]
			  -(P.afPAnot[iPAg]+P.fPAdel*iPA)*(float)RADDEG) 
		  == EXIT_FAILURE) return EXIT_FAILURE;
	      if (ib > 0 && D[ib][iPA].iB < 0) D[ib][iPA].iB = -D[ib][iPA].iB; 
	      if (D[ib][iPA].iB < 0) D[ib][iPA].iB = -D[ib][iPA].iB; 
	      //printf("ib=%i - Exit Surface\n",ib);

	      /* Do over wavelengths */
	      for (j=j1Wave; j<j2Wave; j++) {
		R.fWave=LD.afWaves[j];
		LD.iWave = j; // correct bug found by Christian!!

		/* Get Strip Brightness Distribution at the wavelength. */
		/* Setting D.iB < 0 causes a lot of printing
		  if (ib == 2 && inl == 0 && iPA == 5) 
		  D[ib][iPA].iB = -ISTRIPMAX;
		  else D[ib][iPA].iB = ISTRIPMAX;
		*/

		if (StripBright(&LD, &D[ib][iPA], R.fWave, X, Bright, 
				Uniform) == EXIT_FAILURE) 
		  return EXIT_FAILURE;
		//printf("ib=%i - Exit StripBright\n",ib);
		/* clear write */ 
		if (D[ib][iPA].iB < 0) D[ib][iPA].iB = -D[ib][iPA].iB; 
		IfWrite = 0;
		/*
		  if (iB1 == 0 && inl == 0 && iPA == 0 && j == j1Wave) 
		  IfWrite=1;
		*/
		if (IfWrite) {
		  printf("Strip Brightness Distribution:\n");
		} // end IfWrite
		sum=0.;
		sumu=0.;
		for (i = 0; i < D[ib][iPA].iB; i++) {
		  sum += Bright[i];
		  sumu += Uniform[i];
		}
		for (i = 0; i < D[ib][iPA].iB; i++) {
		  Bright[i] /= sum;
		  if (IfWrite) {
		    printf("i=%1i X=%1.6g B=%1.6g Uniform=%1.6g\n", 
			   i, X[i], Bright[i], Uniform[i]/sumu);
		  } // end IfWrite
		}
		//if (IfWrite == 1 && j > 2) return EXIT_FAILURE;
		
		/* Get Complex Visibility for Baseline ib */
		Re=0.; 
		Im=0.; 
		u=cmasnmm*O.fUTheta[ib][0]/R.fWave;
		delta = fabs(X[1]-X[0])*R.fAngDiam_pB/2.;
		if (IfWrite) {
		  printf("Cycles/mas*nm/m=%1.6g u=%1.6g delta=%1.6g\n",
			 cmasnmm, u, delta); 
		} // end IfWrite
		
		for (i = 0; i < D[ib][iPA].iB; i++) {
		  if (fabs(delta*u) > 0.) {
		    dsincu = sin((float)PI*delta*u)/((float)PI*delta*u);
		  }
		  else {
		    dsincu = 1;
		  }
		  xi=(float)TWOPI*R.fAngDiam_pB/2*X[i];
		  Re += Bright[i] * cos(xi*u) * dsincu;
		  Im += Bright[i] * sin(xi*u) * dsincu;
		} 
		Vis2[iPA][ib][j] = Re*Re+Im*Im;
		Vis[iPA][ib][j] = sqrt(Re*Re+Im*Im);
		phase[iPA][ib][j] = atan2(Im, Re) * (float)DEGRAD;
		ReV[iPA][ib][j] = Re;
		ImV[iPA][ib][j] = Im;
		/*
		printf("j=%i Wave=%g Re=%g Im=%g phi=%g\n",
		       j,R.fWave,Re,Im,phase[iPA][ib][j]);
		*/

		/* remove any 360-degree discontinuities */
		/*
		*/
		if (j > j1Wave) {
		  if (fabs(fabs(phase[iPA][ib][j]-phase[iPA][ib][j-1])-360.)
		      < 1.) {
		    if (phase[iPA][ib][j-1] > 0.) phase[iPA][ib][j] +=360.;
		    if (phase[iPA][ib][j-1] < 0.) phase[iPA][ib][j] -=360.;
		  }		     
		}
		/*
		if (phase[iPA][ib][j] < -180.1) phase[iPA][ib][j] += 360.;
		else if (phase[iPA][ib][j] > 180.1)
		  phase[iPA][ib][j] -= 360.;
		*/

		/*
		//if (iPA == 0 && iHA == 0 && inl == 0) {
		if (j == j1Wave && iPA == 0 && iHA == 0 && inl == 0) {
		//if ((j == j1Wave || j == j2Wave-1) && iPA == 0) {
		printf("j=%2i Wave=%1.6g Visibility=%1.6g Phase=%1.6g\n", 
		j,LD.afWaves[j],Vis[iPA][ib][j],phase[iPA][ib][j]);
		printf("     Re=%1.6g Im=%1.6g\n", Re, Im);
		}
		*/
	      }
	    }
	  }

	  /*NOW DO PLOTS*/

	  /* Do all Triangles */
	  /* iTri currently forced =0 */
	  P.iTri = iTri; // for plot routine
	  //printf("iTri=%2i\n",iTri);
	    
	  /* Set up the plot: 3 V^2 & Phases, & 3-Phi + 3-Amp 
	   * all as functions of: PA &, waves 
	   */
	  for (iPA = 0; iPA < P.iPAdel; iPA++) {

	    /* Do around the triangle for closure phase */
	    //if (iPA == 0) printf("Phase closure phases:");
	    for (j = j1Wave; j < j2Wave; j++) {
	      //if ((((j-j1Wave)/4)*4 == j-j1Wave) && iPA == 0) printf("\n");
	      ReC=1.;
	      ImC=0.;
	      phiClose=0.;
	      amplitudeClose=1.;
	      //printf("j=%i Wave=%g\n",j,LD.afWaves[j]);
	      for (icp = 0; icp < 3; icp++) {
		iB1=O.iTriangleBaselines[iTri][icp];
		/* iB2 indicates the direction along that baseline.  + for
		 * the direction assumed in the baselines, - for the other
		 * way.  Need this for triangle closure.  The sense is
		 * encoded in the sign of iTriangleBaselines.  Therefore
		 * you can't use index=0, see ReadBaselines.
		 */
		iB2=1;
		if (iB1 < 0) {
		  iB1 = -iB1;
		  iB2 = -1;
		}
		iB1--;
		P.aiB1[icp]=iB1; // for use in plots
		P.aiB2[icp]=iB2;
		P.afV2[iPA][icp][j] = Vis2[iPA][iB1][j];
		P.afV[iPA][icp][j] = Vis[iPA][iB1][j];
		P.afphase[iPA][icp][j] = phase[iPA][iB1][j];
		if (iB2 < 0) P.afphase[iPA][icp][j] += 180.;
		if (P.afphase[iPA][icp][j] < -90.)P.afphase[iPA][icp][j]+=360.;
		if (P.afphase[iPA][icp][j] > 270.)P.afphase[iPA][icp][j]-=360.;
		//amplitudeClose *= Vis[iPA][iB1][j];
		Re = (ReC*ReV[iPA][iB1][j]-ImC*ImV[iPA][iB1][j])*iB2;
		Im = (ReC*ImV[iPA][iB1][j]+ImC*ReV[iPA][iB1][j])*iB2;
		ReC = Re;
		ImC = Im;
		/**********
		phiClose += P.afphase[iPA][icp][j];
		if (phiClose > 180.) phiClose -= 360.;
		if (phiClose < -180.) phiClose += 360.;
		printf("  iB2=%i phi[%i]=%g Sphi=%g Mphi=%g ReC=%g ImC=%g\n",
		       iB2,iB1,P.afphase[iPA][icp][j], phiClose,
		       (float)DEGRAD*atan2(ImC,ReC), ReC, ImC);
		**********/
	      }
	      phiClose = atan2(ImC,ReC)*(float)DEGRAD;
	      amplitudeClose = sqrt(ReC*ReC+ImC*ImC);
	      P.afCphase[iPA][j] = phiClose;
	      if (phiClose < -90.) P.afCphase[iPA][j] += 360.;
	      P.afCamp[iPA][j] = amplitudeClose;
	    }

	  }
	  if (PlotTriangle(&R, &LD, &O, &P, &S) == EXIT_FAILURE) 
	    return EXIT_FAILURE;

	} // iPAg: PA group
      } // iHA
      cpgclos(); // close that file
    } // Incl
  } // else {
  return EXIT_SUCCESS ;
}
