/* This code was modified for use in OYSTER by C. A. Hummel */

/* This calculates the properties of Roche Spheroids. 
 * 
 * Version 1.1: Includes the transformation of the baselines from the
 * ground to the star.  Nominally for Hour Angle = 0, but it can be set
 * for anything.
 *
 * Version 1.2: Adds in a computation of the UV plane coverage.  Since a
 * range of Hour Angles is required, this adds a do loop and does the
 * observables for all them (from a fixed set entered at the beginning
 * of Roche.c).  Fixed a bug in RThetaPhifromXY that blew up near the
 * equator.  Added a calculation of the apparent major and minor axes of
 * the inclined disk (in milliarcseconds).  Adds a PA offset to the
 * image drawn in Picture.c, and switching to angular (mas) scales.
 *
 * Version 1.3: Splits up the surface brightness calculation (after
 * Christian complained) into two parts.  First, the structure of the
 * surface that depends only on the geometry (location of pixels and the
 * values of (Teff, log_g, mu) associated with them) resolved along and
 * perpendicular to each baseline is calculated (Surface.c) (that is
 * fairly time consuming).  Then the strip brightness distribution can
 * be calculated for each wavelength fairly quickly.  This required a
 * major rewrite of StripBright.c.  The geometry calculation at somewhat
 * finer resolution will be used to calculate the projected area, the
 * Visual flux and B-V color (without reference to the baselines) (not
 * yet implemented).
 *
 * Version 1.4: Reworks the graphical output.  Switches to a scheme that
 * does one inclination and one triangle per plot page.  That allows the
 * triple amplitude to be added along with graphics showing the UV
 * coverage and the outline of the object on the sky, indicating PA(s)
 * and inclination.  Up to 6 PA's can be plotted on each plot.  We now
 * allow this to be plotted against wavelength or channel number, your
 * choice. 
 * 
 * A plotting switch has been included in the plot parameter file.  It
 * binary encodes which combination of plots you want.  See the header
 * file for how to set the code.  But in short you can get just a
 * summary plot, described above and/or full page versions of each of
 * the visibilities and/or phases.
 *
 * Further on Version 1.4, we decided to move the read in value for the
 * disk scaling factor, fAngDiam_pB, to the inclinations file, since the
 * major axis of the projected image is always so strongly dependent on
 * rotation near breakup. From now on the first two options in the
 * program - to print out the main parameters of the rotating star and
 * to look at renderings of the star - will be run off the inclinations
 * file as soon as the request for a single wavelength is seen. 10/9/03
 *
 * Fixed PA error (wasn't converting to radians!). Also, this version
 * now forces RpB=1 for all omega (see comments in FractRadius). 2/9/04
 *
 * Forces the correct wavelength index for access to the limb darkening
 * array.  Bug pointed out by Christian Hummel 26/02/04
 */

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <math.h>

#include "roche.h"

Roche R; // Model and derived parameters
static LimbDark LD; // Limb Darkening data
Observatory O; // Observatory parameters
Disk D; // Stellar disk

int RocheVis(float *pfRocheParms, int *piN, float *pfLam, 
	     float *pfReVis, float *pfImVis, float *pfFlux, 
	     float *pfUm, float *pfVm,
	     int *piLDinit, 
	     int *piWavesMax, int *piTeffMax, int *piLogGMax, int *paiLogG, 
	     float *pafLD, float *pafLogG, float *pafWaves, float *pafTeff) {

  float Incl[10], W[10], DpBmas[10], ThPA; 
  float Bright[ISTRIPMAX],X[ISTRIPMAX],Uniform[ISTRIPMAX];
  float Vis[6][15][32], phase[6][15][32], Vis2[6][15][32]; 
  float ReV[6][15][32], ImV[6][15][32], phiClose, amplitudeClose;
  int IfWrite, IfRotate=0, iBaseline, iIncl, iInclmax=10, imagedev;
  int i, icp, inl, j, j1Wave, j2Wave, l, iHA, ib;
  char YN[]=" ";
  char name[10];
  char sIncl[5], sPlotFile[40];

  /* To calculate Visibilities */
  float cmasnmm=1.e9/(3.6e6*(float)DEGRAD); // cycles/mas*nm/m
  float Re=0., Im=0., ReC, ImC, u, xi, delta, dsincu;
  float sum=0., sumu=0.;

  int iTri=0, iB1, iB2, iPAg, iPA;

  float fDeltaLambda;

  /* 
   * There are two arguments in the call list.  The first is the model
   * name (just the name, we add the ".dat" later).  There the program
   * picks up the Mass (Msun); Radius (Rsun), the radius at zero
   * rotation; TeffpB (K), the effective Temperature the pole would have
   * at breakup; AngDiam_pB, the polar (smallest) angular diameter you
   * would get at breakup; and two parameters describing the limb
   * darkening set: LDLaw, the limb darkening law to use and Waveset,
   * which NPOI wavelength set to use.  The limb darkening file name is
   * constructed, in part, from these.
   *
   * The second command line parameter is an index.  If the index is
   * positive and <= indexWaveMax (=32) it does that many channels
   * starting with the longest wavelength.  If the index is negative it
   * does the wavelength for that (positive) index only.  An index of
   * zero does 500nm. An index whose absolute value is greater than
   * indexWaveMax is assumed to be a wavelength in nanometers.
   *
   * We either read in the NPOI wavelengths with the file containing the
   * bandpass averaged limb darkening coefficients if the absolute value
   * of the index is <= 32 or, if a single wavelength is called out, we
   * read in the limb darkening file for monochromatic wavelengths and
   * interpolate values from that.
   */

  if (InitRoche1(&R, &LD, name, pfRocheParms, piN, pfLam, &j1Wave, &j2Wave,
	         piLDinit, piWavesMax, piTeffMax, piLogGMax, paiLogG, 
		 pafLD,pafLogG, pafWaves,pafTeff) 
      	== EXIT_FAILURE) return EXIT_FAILURE;

  R.fWave=700;
  MainVar(&R, &LD, 0); // calculate all the physical variables

  /* Set up Disk map for the baselines at these HA's*/
  D.iB = ISTRIPMAX;

  if (Surface(&R, &D, PI2-atan2(pfVm[0],pfUm[0]) - R.fPA*(float)RADDEG) 
	== EXIT_FAILURE) return EXIT_FAILURE;
  if (D.iB < 0) D.iB = -D.iB; 

  /* Loop over wavelengths */
  for (j=j1Wave; j<j2Wave; j++) {
	R.fWave=pfLam[j];
	LD.iWave=-1;
	fDeltaLambda=10;
	for (l=0; l<LD.indexWaveMax; l++) {
		if (fabs(pfLam[j]-LD.afWaves[l]) < fDeltaLambda) {
			LD.iWave = l;
			fDeltaLambda = fabs(pfLam[j]-LD.afWaves[l]);
		}
	}
	if (LD.iWave < 0) {printf("No match for %f\n",R.fWave);exit(0);}
		
	/* Get Strip Brightness Distribution at the wavelength. */

	if (StripBright(&LD, &D, R.fWave, X, Bright, Uniform) 
		== EXIT_FAILURE) return EXIT_FAILURE;
	if (D.iB < 0) D.iB = -D.iB; 
	sum=0.;
	sumu=0.;
	for (i = 0; i < D.iB; i++) {
	  sum += Bright[i];
	  sumu += Uniform[i];
	}
	for (i = 0; i < D.iB; i++) {
	  Bright[i] /= sum;
	}
		
	/* Get Complex Visibility */
	Re=0.; 
	Im=0.; 
	u=cmasnmm*sqrt(pfUm[j]*pfUm[j]+pfVm[j]*pfVm[j])/R.fWave;
	delta = fabs(X[1]-X[0])*R.fAngDiam_pB/2.;
		
	for (i = 0; i < D.iB; i++) {
	  if (fabs(delta*u) > 0.) {
	    dsincu = sin((float)PI*delta*u)/((float)PI*delta*u);
	  }
	  else {
	    dsincu = 1;
	  }
	  xi=(float)TWOPI*R.fAngDiam_pB/2*X[i];
	  Re += Bright[i] * cos(xi*u) * dsincu;
	  Im += Bright[i] * sin(xi*u) * dsincu;
	} 
	pfReVis[j] = Re;
	pfImVis[j] = Im;
	pfFlux[j] = sum;
  }

  return EXIT_SUCCESS ;
}
