/***********************************************************************
 * StripBright: Determines the weighted intensity along a specific look
 * direction (baseline) at some angle, Theta, to N-S (measured N->E).
 * Return two vectors, weighted brightness and distance of the center
 * of that strip from the subSolar point in units of the Polar Breakup
 * Radius.  
 *
 * We also return the Strip Brightness distribution of a uniformily
 * illuminated disk of the same shape along that direction.
 *
 * All the relevant parameters about the Roche model come in through
 * (Roche) R.
 *
 * This has been rewritten in Version 1.3 to split off the mapping of
 * the surface into pixels along and perpendicular to the baseline (now
 * in Surface.c) from the calculation of the emission from each pixel
 * and subsequent calculation of the strip brightness distribution, done
 * here.  This allows the surface geometry and hence Teff, log_g and mu
 * for each (x,y)) to be defined separately from the intensity
 * distributions.  The latter are done at many wavelengths and are time
 * consuming.
 *
 * The (Teff, log_g, mu) map is now brought in through a Disk structure
 * precalculated (in Surface.c) for each projected baseline. 18/12/03
 *
 * The original calculation used zeroth order integration (steps) over
 * the surface, which requires way too many points to be accurate.
 * This goes to trapizoidal integration, ie, it assumes a linear
 * behavior over an interval.  For integrations perpendicular to the
 * baseline, that is essentially Simpson's rule.  For the complex
 * Visibility, we need the F.T. of a linear function, defined at the
 * end points.  This is pretty much the same logic as before, that
 * aside.  
 *
 * Note that the points along the baseline direction (x) assume zero
 * intensity at the two ends, because of the zero height of the cord
 * perpendicular to the baseline.  8/3/04
 **********************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "rochechi2.h"

int StripBright(LimbDark * LD, Disk * D, float lam, float * X, 
		float * Weight, float * Uniform) {

  int IfWrite;
  int i, j, iB;

  /* For gravity/temperature calculation */
  float hclk = 1.43878e7/lam; // units: nm*K
  float logg, tEff, hclkt, eMiss, eMiss1;
  float lam5=lam*lam*lam*lam*lam;
  float thc2ol5 = 1.191062e23/lam5; // 2hc^2/lam^5
  float SurfB, mu;

  float dy, ysum, area, Area; int iNy;
  float dx;

  iB = D->iB;
  IfWrite = 0;
  if (iB < 0) {
    iB = -iB;
    IfWrite = 1;
  }

  dx = fabs(D->fdx); // increment
  if (IfWrite) {
    printf("\nStripB: dx=%1.6g lam=%1.6g\n", dx,lam);
  } // end IfWrite
  Area=0.;

  /* Do, one strip at a time */
  for (i = 0; i < abs(iB); i++) {
    ysum=0.; 
    area=0.;
    //IfWrite = 0;
    //if (i == 1) IfWrite = 1;
    X[i] = D->afX[i];
    if (IfWrite) {
      printf("i=%2i x=%1.6g\n", i, X[i]);
    }

    /* Sum over Y */
    iNy = D->iNy[i];
    dy = fabs(D->fdy[i]);
    if (IfWrite) {
      printf(" iNy=%2i dy=%1.6g\n", iNy, dy);
    } // end IfWrite
    for (j = 0; j < iNy; j++) {
      //y = D->afY[i][j];
      if (IfWrite && i == ISTRIPMAX-1) printf("j=%i y[i][j]=%1.6g\n",j,
					      D->afY[i][j]);
      if (IfWrite) printf("y[%i][%i]=%1.6g\n",i,j,D->afY[i][j]);
      /* Those are along the baseline. */

      tEff = D->afTLgMu[i][j][0];
      hclkt = hclk/tEff;
      logg = D->afTLgMu[i][j][1];
      mu = D->afTLgMu[i][j][2];
      //if (IfWrite && i == ISTRIPMAX-1) 
      if (IfWrite) 
	printf("Teff=%1.6g log_g=%1.6g mu=%1.6g\n",tEff,logg,mu);
      if (SurfBright(LD, tEff, logg, mu, &SurfB) 
	  == EXIT_FAILURE) return EXIT_FAILURE;

      eMiss = thc2ol5/(exp(hclkt)-1)*SurfB; 
      //if (i == 2) {
      //printf("gEff=%1.6g tEff=%1.6g hclkt=%1.6g",gEff, tEff, hclkt);
      //printf(" eMiss=%1.6g dx=%1.6g dy=%1.6g\n", eMiss, dx, dy);
      //}
      
      /* Easy, huh? */

      //ysum += eMiss*fabs(dx*dy);

      if (i == 0 || i == abs(iB)-1) {
	ysum += eMiss * fabs(dy);
	area += fabs(dy);
      }

      if (j > 0) {
	ysum += (eMiss + eMiss1)*fabs(dy)/2.;
	area += fabs(dy);
      }
      eMiss1 = eMiss;
      //area += fabs(dx*dy);
      if (IfWrite && i == ISTRIPMAX-1) {
	printf("j=%i dx=%1.6g dy=%1.6g eMiss=%1.6g ysum=%1.6g area=%1.6g\n",
	       j, dx, dy, eMiss, ysum, area); 
      }
    }
    Weight[i] = ysum;
    Uniform[i] = area;
    //Area += area;
    Area += area*dx;
    if (IfWrite) {
      printf("StripB: i=%i X[i]=%1.6g Weight[i]=%1.6g Uniform[i]=%1.6g\n",
	     i,X[i],Weight[i],Uniform[i]);
    }
  }
  //IfWrite=1;
  if (IfWrite) {
    printf("StripB: Area=%1.6g dx=%g\nUniform:",Area,dx);
    area=0;
    ysum=0;
    for (i=0; i<abs(iB); i++) {
      if ((i/2)*2 == i) printf("\n");
      printf("i=%2i Uniform=%7.5g Weight=%7.5g ",i,Uniform[i],Weight[i]);
      area += Uniform[i];
      ysum += Weight[i];
    }
    printf("\nSum Uniform =%g area=%g W/eMiss=%g\n",area,area*dx,ysum/eMiss*dx);
  }
  //if (j >= 2) return EXIT_FAILURE;

  //return EXIT_FAILURE;
  return EXIT_SUCCESS;
}
