/***********************************************************************
 * This divides the projected outline of a Roche spheroid into an array
 * of pixels along and perpendicular to the (baseline) direction, Theta
 * (measured East from North). "iB" is the resolution along the
 * baseline.  The resolution perpendicular for each strip is set so that
 * the pixel size is about square.  
 *
 * Once the pixels are defined, the values of Teff, log_g and mu are
 * evaluated at their centers. 
 *
 * This is cannibalized extensively from the original StripBright.c
 * subroutine. 
 *
 * If iB < 0, forces a complete dump! 
 **********************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "roche.h"

int Surface(Roche * R, Disk * D, float Theta) {

  float cTh = cos(Theta), sTh = sin(Theta), incl=R->fIncl, ci = cos(incl),
    si = sin(incl);
  int IfWrite;
  int i, j, iB;
  int iYfound=0, j1, j2; 
  int OutNum=72;

  /* For gravity/temperature calculation */
  float gpb= GRAV*R->fMass*MSUN/R->fRpB/R->fRpB/RSUN/RSUN;
  float ttc = 0.2962963, tts = 0.087791495, fact, w = R->fwOmega, w2=w*w;
  //float hck = 1.43878e7; // units: nm*K
  float gEff, tEff; // hclkt, eMiss;
  float xp, yp, thout, phiout, rout2;
  float dlrdth, mu; // SurfBp, SurfB, 

  float Outthp[72], Outxp[72], Outyp[72], delth = 360./OutNum*(float)RADDEG;
  float OutX[72], OutY[72];
  float y, dy, ynot, ysum, area; int iNy;
  float th, phi, rout, sth, sth2, cth, cth2,sph, cph;
  float xmin=0., xmax=0.; int ixmin=-1, ixmax=-1;
  float y1, y2, wtp, wtm; int iy1, iy2;
  float x, dx, xnot;

  iB = D->iB;
  if (iB < 0) IfWrite=1;
  IfWrite = 0;
  /* Get the star's outline.  Every 5 degrees. */
  if (IfWrite) {
    printf("Surface for Theta=%1.6g", Theta*(float)DEGRAD);
    printf(" cos(Theta)=%1.6g sin(Theta)=%1.6g\n", cTh, sTh);
  } // end IfWrite

  for (i=0; i < OutNum; i++) {
    if (Edge(R, Outthp[i]=delth*i, &th, &phi, &rout) == EXIT_FAILURE)
      return EXIT_FAILURE;
    sth = sin(th);
    cth = cos(th);
    sph = sin(phi);
    cph = cos(phi);
    Outxp[i] = rout*(sth*cph*ci-cth*si);
    Outyp[i] = rout*sth*sph;
    OutX[i] = -Outxp[i]*cTh - Outyp[i]*sTh;
    OutY[i] =  Outxp[i]*sTh - Outyp[i]*cTh;
    if (IfWrite) {
      printf("Th'=%1.6g X'=%1.6g", Outthp[i]*(float)DEGRAD, Outxp[i]); 
      printf(" Y'=%1.6g Th=%1.6g", Outyp[i], th*(float)DEGRAD);
      printf(" Phi=%1.6g R=%1.6g", phi*(float)DEGRAD, rout);
      printf(" X=%1.6g Y=%1.6g\n", OutX[i],OutY[i]);
    } // end IfWrite
    /*
    if (IfWrite) {
      CkTh = atan2(OutY[i], OutX[i]);
      if (CkTh < 0.) CkTh = CkTh + (float)TWOPI;
      printf("    CkTh=%1.6g", CkTh*(float)DEGRAD);
      printf(" X=%1.6g Y=%1.6g\n", OutX[i], OutY[i]);
    } // end IfWrite
    */
  } 

  /* Get Min/Max of Outline */
  for (i = 0; i < OutNum; i++) {
    if (OutX[i] > xmax) {
      xmax = OutX[i];
      ixmax = i;
    }
    if (OutX[i] < xmin) {
      xmin = OutX[i];
      ixmin = i;
    }
  }

  IfWrite = 0;
  if (iB < 0) IfWrite=1;
  if (IfWrite) {
    printf("xmin=%1.6g xmax=%1.6g\n", xmin, xmax);
  } // end IfWrite
  IfWrite=0;

  dx = (xmax-xmin)/abs(iB); // increment
  D->fdx = dx;
  xnot = xmin+dx/2;
  if (IfWrite) {
    printf("Find strips: dx=%1.6g xnot=%1.6g\n", dx, xnot);
  } // end IfWrite

  /* Do, one strip at a time */
  for (i = 0; i < abs(iB); i++) {
    ysum=0.; 
    area=0.;
    iYfound=0;
    //IfWrite = 0;
    //if (i == 1) IfWrite = 1;
    x = xnot + dx*i;
    //X[i] = x;
    D->afX[i] = x;
    if (IfWrite) {
      printf("i=%2i x=%1.6g\n", i, x);
    }

    /* Find the two Y intercepts */
    j=0;
    y1=0.;
    y2=0.;
    while (j < OutNum && iYfound < 2) {
      j1=j;
      j2=j+1;
      if (j == OutNum-1) j2=0;
      //if (j != 0) IfWrite = 0;
      if ((OutX[j1] <= x && OutX[j2] > x) || (OutX[j1] >= x && OutX[j2] < x)) {
	if (IfWrite) {
	  printf("j=%2i j1=%2i j2=%2i\n", j, j1, j2);
	  printf(" OutX[j1]=%1.6g OutX[j2]=%1.6g", OutX[j1], OutX[j2]);
	  printf(" X[j]=%1.6g X[j2]=%1.6g\n", OutX[j], OutX[j2]);
	} // end IfWrite
	if (iYfound == 0) {
	  wtp=(x-OutX[j2])/(OutX[j1]-OutX[j2]);
	  wtm=1.-wtp;
	  iy1=j1;
	  y1 = OutY[j1]*wtp + OutY[j2]*wtm;
	  if (IfWrite) {
	    printf(" iYfound=%2i iy1=%2i y1=%1.6g", iYfound, iy1, y1);
	    printf(" j1=%1i j2=%1i wtp=%1.6g wtm=%1.6g\n",j1,j2,wtp,wtm);
	  }
	  iYfound++;
	}
	else if (iYfound == 1) {
	  wtp=(x-OutX[j2])/(OutX[j1]-OutX[j2]);
	  wtm=1.-wtp;
	  iy2=j1;
	  y2 = OutY[j1]*wtp + OutY[j2]*wtm;
	  if (IfWrite) {
	    printf(" iYfound=%2i iy2=%2i y2=%1.6g", iYfound, iy2, y2);
	    printf(" j1=%1i j2=%1i wtp=%1.6g wtm=%1.6g\n",j1,j2,wtp,wtm);
	  }
	  iYfound++;
	}
      }
      j++;
    }

    //IfWrite = 0;
    if (IfWrite) {
      printf("i=%2i x=%1.6g iY1=%2i Y1=%1.6g iY2=%2i Y2=%1.6g\n", 
	     i, x, iy1, y1, iy2, y2);
    } // end IfWrite

    /* Over Y */
    iNy = 1 + (int) fabs((y2-y1)/dx); // keep dy about the size of dx
    D->iNy[i] = iNy;
    dy = (y2-y1)/iNy;
    D->fdy[i] = dy;
    ynot = y1 + dy/2.;
    if (IfWrite) {
      printf(" iNy=%2i dy=%1.6g ynot=%1.6g\n", iNy, dy, ynot);
    } // end IfWrite
    for (j = 0; j < iNy; j++) {
      y = ynot + dy*j;
      D->afY[i][j] = y;
      /* Those are along the baseline. */

      /* Fundamental Plane */
      xp = -x*cTh + y*sTh;
      yp = -x*sTh - y*cTh;

      /* Then get body coordinates */
      /*
      if (D->iB < 0 && i==0 && j==iNy-1) IfWrite=1;
      if (RThetaPhifromXY(xp, yp, R, &rout, &thout, &phiout, IfWrite) 
	  == EXIT_FAILURE) {
      */
      if (RThetaPhifromXY(xp, yp, R, &rout, &thout, &phiout) 
	  == EXIT_FAILURE) {
	printf("i=%2i x=%1.6g j=%2i y=%1.6g cosTh=%1.6g", i, x, j, y, cTh);
	printf(" sinTh=%1.6g xp=%1.6g yp=%1.6g\n", sTh, xp, yp);
	return EXIT_FAILURE;
      }
      if (IfWrite == 1) {
	printf("i=%2i x=%1.6g j=%2i y=%1.6g cosTh=%1.6g", i, x, j, y, cTh);
	printf(" sinTh=%1.6g xp=%1.6g yp=%1.6g\n", sTh, xp, yp); 
	printf("rout=%g thout=%g phiout=%g\n",rout,(float)DEGRAD*thout,
	       (float)DEGRAD*phiout);
      }

      sth = sin(thout);
      sth2 = sth*sth;
      cth = cos(thout);
      cth2 = cth*cth;
      cph = cos(phiout);
      rout2 = rout*rout;
      if (D->iB < 0 && i == 0 && IfWrite == 1) {
	printf("sth=%1.6g sth2=%1.6g cth=%1.6g cth2=%1.6g cph=%1.6g rout2=%5.6g\n",
	       sth,sth2,cth,cth2,cph,rout2);
      }
      IfWrite=0;

      /* Now can get local effective gravity and effective temperature */
      fact = (ttc*w2*rout*sth2-1/rout2);
      gEff = sqrt(fact*fact + tts*w2*w2*rout2*sth2*cth2); // scaled by gpb
      tEff = R->fTeffpB*sqrt(sqrt(gEff));
      D->afTLgMu[i][j][0] = tEff;
      D->afTLgMu[i][j][1] = log10(gEff*gpb);
      dlrdth=(ttc*w2*rout2*rout*sth*cth)/(1.-ttc*w2*rout2*rout*sth2);
      mu=(dlrdth*(sth*ci-cth*cph*si)+cth*ci+sth*cph*si)/
	sqrt(dlrdth*dlrdth+1.);
      if (mu < 0.) mu=0.; // fix any noise in the numerical procedures
      D->afTLgMu[i][j][2] = mu;
      if (D->iB < 0 && i == 0 && IfWrite == 1) {
	printf("fact=%g gEff=%g tEff=%g dlrdth=%g mu=%g\n",
	       fact,gEff,tEff,dlrdth,mu);
      }
    }
  }

  if (iB < 0)  IfWrite = 1;

  /* Write out results */
  if (IfWrite) {
    printf("Surface:\n  i  iY     X    j    y      Teff   logg     mu\n");
    //for (i=0; i<abs(iB); i++) {
    for (i=0; i<1; i++) {
      printf("%3i%3i %8.5f\n",i,D->iNy[i], D->afX[i]);
      for (j=0; j<D->iNy[i]; j++) {
	printf("                %2i %8.5f %5.0f  %5.2f %6.4f\n",j,D->afY[i][j],
	     D->afTLgMu[i][j][0],D->afTLgMu[i][j][1],D->afTLgMu[i][j][2]);
      }
    }
  }
  return EXIT_SUCCESS;
}
