/* This calculates the complex Visibility of anything presented as a
 * strip brightness distribution.  In this case we assume the geometry
 * has been previously calculated in Surface and summarized in the Disk
 * structure.  The call to StripBright then evaluates the strip
 * brightness distribution at this wavelength and we use that
 * distribution to calculate the Visibility.
 *
 * The thing here is that the strip brightness distribution is
 * represented as piecewise linear, which is significantly more
 * accurate than a simple step function which we originally used.  We
 * should be able to use substantially fewer points across the Roche
 * disk to get an accurate rendering of the Visibility function.
 */

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <math.h>

#include "rochechi2.h"

int Visibility(Roche * R, LimbDark * LD, Disk * D, float U, float * Re,
	       float * Im, float * B, int IfWrite) {


  int i;
  float Bright[ISTRIPMAX],X[ISTRIPMAX],Uniform[ISTRIPMAX];
  /* To calculate Visibilities */
  float cmasnmm=1.e9/(3.6e6*(float)DEGRAD); // cycles/mas*nm/m
  float u, xi, delta, pdu, dsincu, dcosu, cxu, sxu, smcpdu;
  float sum=0., sumu=0., hbpha, hbmha;

  //if (D->iB < 0) D->iB = -D->iB; 
  //D->iB = -D->iB;
  if (StripBright(LD, D, R->fWave, X, Bright, Uniform) == EXIT_FAILURE) 
    return EXIT_FAILURE;
  //printf("Out of StripBright\n");

  sum=0.;
  sumu=0.;
  for (i = 0; i < D->iB; i++) {
    sum += Bright[i];
    sumu += Uniform[i];
  }
  for (i = 0; i < D->iB; i++) Bright[i] /= sum;
  /* CAH */
  /* The flux is returned, but has to be multiplied with strip width */
  *B = sum*fabs(X[D->iB-1]-X[0])/(D->iB-1);
		
  /* Get Complex Visibility for Baseline ib */
  *Re=0.; 
  *Im=0.; 
  u=cmasnmm*U/R->fWave;
  delta = fabs(X[D->iB-1]-X[0])/(D->iB-1)*R->fAngDiam_pB/2.;
  pdu = (float)PI*delta*u;
		
  if (fabs(pdu) > 0.) {
    dsincu = sin(pdu)/pdu;
    dcosu = cos(pdu);
    smcpdu = (dsincu-dcosu)/pdu;
  }
  else {
    dsincu = 1.;
    dcosu = 1.;
    smcpdu = 0;
  }

  for (i = 1; i < D->iB; i++) {
    xi=(float)PI*R->fAngDiam_pB/2*(X[i]+X[i-1]);
    cxu = cos(xi*u);
    sxu = sin(xi*u);

    hbpha = (Bright[i] + Bright[i-1])/2.;
    hbmha = (Bright[i] - Bright[i-1])/2.;

    *Re += hbpha*dsincu*cxu - hbmha*smcpdu*sxu;
    *Im -= hbpha*dsincu*sxu + hbmha*smcpdu*cxu;
  } 

  return EXIT_SUCCESS;
}
