/* Structures for evaluating the Roche Spheroids. */

typedef struct {
  char sName[9]; // Name of Star
  float fRA; // Degrees
  float fDec; // Degrees
  float fEpoch; // For coordinates, probably not important
  float fwOmega; // Fractional Angular Rotation Velocity (unit: fOmegaB)
  float fRNot; // Stellar Radius at wOmega=0 (unit: RSun)
  float fMass; // Stellar Mass (unit: MSun)
  float fRp; // Actual Polar Radius (unit: RSun)
  float fRpB; // Breakup Polar Radius (unit: RSun)
  float fReB; // Breakup Equatorial Radius (unit: RSun)
  float fRe; // Actual Equatorial Radius (unit: RSun)
  float flogGp; // Actual Polar log_g
  float flogGpB; // Polar Breakup log_g
  float flogGe; // Actual Equatorial log_g
  float fOmegaB; // Breakup Angular Velocity
  float fOmega; // Angular Velocity = wOMEGA*fOmegaB
  float fVeqB; // Equatorial Breakup Velocity = fOmegaB*fReB
  float fVeq; // Actual Equatorial Velocity = fOmega*fRe
  float fVol; // Volume of Spheroid (unit: fRpB^3)
  float feqArea; // Area of Cross Section, Equator on (unit: fRpB^2)
  float fArea; // Area of Cross Section, this look-angle (unit: fRpB^2)
  float fIncl; // Polar axis inclination angle (0 = pole on)
  float fPA; // Position angle of axis on sky
  float fTeffpB; // Polar breakup Teff(K)
  float fTeffp; // Teff(K) at pole
  float fTeffe; // Teff(K) at equator
  float fAngDiam_pB; // Polar Angular Diameter for Breakup (unit: mas)
  float fAngDiam_Xp; // Angular Diameter along minor axis (unit: mas)
  float fAngDiam_Yp; // Angular Diameter along major axis (equator, unit: mas)
  float fWave; // wavelength(unit: nm)
} Roche;

/*********************************************************************/
/* For the Limbdarkening law and grid */
typedef struct {
  char sLDLaw[10]; // The Limb Darkening Law (ie, Lin, Sqrt, ...)
  char sInstrument[10]; // The interferometer (ie, Mark3, NPOI, ...)
  char sWaveset[10]; // Wavelength set (ie, Reg, Mon, ...)
  int iWave; // Lambda index in current use
  int indexWaveMax; // Largest useable wavelength index (+1)
  float fWaveWtMinus; // interpolate wavelength for "Mon" LD files
  int iTeffMax; // Number of Teffs
  int aiLogG[61]; // how many log_g's at each Teff
  float afLD[3][61][11][832]; // X, (Y), and I_1 for [Teff, log_g, & lambda]
  float afLogG[11]; // 5.0, 4.5, ..., 0.0, max (but all start at 5.0)
  float afWaves[832]; // wavelength set
  float afTeff[61]; // Teff set
} LimbDark;

/*********************************************************************/
/* For the Limbdarkening law and grid */
typedef struct {
  char sLDLaw[10]; // The Limb Darkening Law (ie, Lin, Sqrt, ...)
  char sInstrument[10]; // The interferometer (ie, Mark3, NPOI, ...)
  char sWaveset[10]; // Wavelength set (ie, Reg, Mon, ...)
  int iWave; // Lambda index in current use
  int indexWaveMax; // Largest useable wavelength index (+1)
  float fWaveWtMinus; // interpolate wavelength for "Mon" LD files
  int iTeffMax; // Number of Teffs
  int aiLogG[61]; // how many log_g's at each Teff
  float afLD[3][61][11][32]; // X, (Y), and I_1 for [Teff, log_g, & lambda]
  float afLogG[11]; // 5.0, 4.5, ..., 0.0, max (but all start at 5.0)
  float afWaves[32]; // wavelength set
  float afTeff[61]; // Teff set
} LimbDark0;

/*********************************************************************/
/* Observatory parameters, Siderostat locations, Baselines and Triangles*/
typedef struct {
  float fLat; // degrees and decimal fractions
  float fLong; // likewise, west of Greenwich is negative
  float fAlt; // may be useful sometime

  int iBaselines; // number of baselines
  /* Baseline names: [0]: Ap0-Ap1 and [1]: Ap1-Ap0. IBMAX=15 */
  char sBaselineNames[15][2][9]; 
  int iBaselineIndex[15][2]; // Index of aperture pair
  /* Ap[1]-Ap[0] (x,y,z) for each pair. x + to East, y + to North, z +
     up, meters*/
  float fBxyz[15][3];  // baselines in horizon coordinates (meters)
  float fBAzi[15][2]; // |B| and azimuth (m, deg N->E) on ground.

  /* Next is the rotation matrix that takes the Baseline (Bx,By,Bz) in
   * local coordinates and transforms it to the (X,Y,Z) system
   * (Thompson, Moran & Swensen 1ed, p 86). Later the rotation matrix
   * for the observation (a function of Hour angle and declination)
   * transforms that to the (u,v,w) coordinates in the star's reference
   * frame. */
  float afRotateBase[3][3]; 
  float fXYZ[15][3]; // baselines in the equatorial plane, see above
  float fUVW[15][3]; // Baselines projected on star (meters)
  float fUTheta[15][2]; // |UVW| and p.a. of baseline (m, deg N->E)
  char sUThNames[15][2][30]; // Add |UVW| and theta to baseline names

  int iTriangles; // number of triangles
  /* Baseline indices for the legs of the triangles.  Sign indicates the
   * order with "-" meaning backwards along the baseline (use Baseline
   * name index = "1").  IMPORTANT: To encode sign, the index here is
   * the baseline index + 1. */
  int iTriangleBaselines[10][3]; // baseline index + 1 (- means inverted)
  char sTriangleNames[10][28];
} Observatory;


/*********************************************************************/
/* The stellar DISK projections as a function of baseline.  X is along
 * the baseline, Y perpendicular.  (Teff, log_g, mu) are evaluated at
 * each point. */
typedef struct {
  /* ISTRIPMAX=>20, ISTRIPMAX+1=>21 in dimensions*/
  int iB; // number of strips along baseline
  float fdx; // resolution along baseline
  float fdy[20]; // resolution perpendicular to baseline
  int iNy[21]; // how many y's in this row (no more than ISTRIPMAX+1)
  float afX[20]; // X at center of pixels for a baseline (unit: RpB)
  float afY[20][21]; // Y at center of each pixel, perp to baseline
  float afTLgMu[20][21][3]; // (Teff, log_g, mu)=>3 for each pixel
} Disk;


/*********************************************************************/
/* Information needed for the plots.  Keeps calling lists civilized. */
typedef struct {

  /* 
   * "IfPlot" Binary encodes Plots wanted: 1/0 = yes/no.  
   * 1 = Summary page, small scale plots and parameters
   * 2 = HiRes: Closure Phase and Closure Amplitudes
   * 4 = HiRes: V^2 for each baseline
   * 8 = HiRes: Phases for each baseline
   * IfPlot = logical and of these
   */
  int IfPlot;
  int iPlotSum; // Summary page
  int iPlotClose; // Closure phase and amplitude
  int iPlotV2; // Visibility squared's for each baseline
  int iPlotPhi; // Phases for each baseline

  /* Current index values - avoids humungous call lists */
  int iInl; // inclination (value already stored in R.fIncl
  int iPAg; // Position Angle group
  int iHA; // Hour Angle (eventually: which scan)
  int iTri; // Triangle

  /* Wavelength range */
  int j1Wave;
  int j2Wave;

  /* X scale - preset to wavelength or channel number */
  float afAbcissa[32]; // X scale. Plots will use a subset of it
  float fAbcmin; // Actual X range
  float fAbcmax;
  char sAbLabel[9]; // Label for abcissas: "//gl(nm)" or "Channels"

  /* Plot file names */
  int IfScreen; // True if only want screen display
  char sBaseFile[20]; // Name given with the device and extension removed
  char sIncl[10]; // The inclination for plots in this file
  char sExtension[10]; // Store the extension (ie, ".ps") 
  char sDevice[10]; // Store the device identification (ie, \VCPS)  
  char sPlotFile[40]; // Reconstruct it here with an inclination if needed

  /* Range of Position Angles to be covered. Read in from "plot.dat" */
  int iPAnot; // <=6, how many plot groups
  int iPAdel; // <=6 how many plotted in a group
  float afPAnot[6]; // Starting PA for each plotted group
  float fPAdel; // increments from afPAnot
  float afXOutline[73]; // Outline of the model at PA=0.
  float afYOutline[73]; // Tabulated every 5-deg. (X,Y) in mas.

  /* Store UV coverage plot data */
  float fUVmax[2]; //2=[U,V], to set up plot boundaries
  float fUVmin[2]; 
  float fSaveUV[100][15][2]; //UV components [IOBSMAX][IBMAX][U,V] 

  /* Which baselines used in this this triangle and which direction */
  float aiB1[3]; // baseline index
  float aiB2[3]; // baseline sense (and can use to get name: 1=>0, -1=>1)

  /* Outline of the model for PA=0 (changes with inclination). */
  int iOutline; // number of points around the outline (=73 for 5deg)
  float afxol[73]; // in units of R_pB
  float afyol[73]; //

  /* Quantities to plot */
  float afV2[6][3][32]; // V^2[IPAMAX][3-Bs][waves]
  float afV[6][3][32]; // |V|[IPAMAX][3-Bs][waves]
  float afphase[6][3][32]; // Visibility phase[IPAMAX][3-Bs][waves]
  float afCphase[6][32]; // Triangle Closure phase[IPAMAX][waves]
  float afCamp[6][32]; // Triangle Closure amplitude[IPAMAX][waves]
  
} Plot;

/*********************************************************************/
/* Observations (=Scans) */
typedef struct {
  int iNScans; // number of different scans
  float afHA[100]; // Hour Angle for each scan
  char sID[100][20]; // ID (ie, date) for each scan
} Scans;

/*********************************************************************/
/* Prototypes */

float FractRadius(float w2, float s2);
int Volume(Roche * R);

/* Save for debugging
int RThetaPhifromXY(float X, float Y, Roche * R, float * r, 
		    float * Th, float * Phi, int IW);
*/

int RThetaPhifromXY(float X, float Y, Roche * R, float * r, 
		    float * Th, float * Phi);
int Edge(Roche * R, float ThP, float * Th, float * Phi, float * rout);
int Surface(Roche * R, Disk * D, float Theta);
int StripBright(LimbDark * LD, Disk * D, float lam, float * X, 
		float * Weight, float * Uniform);
int LatLong(Roche * R, float dPh, float ThPA, float rscale);
int Picture(Roche * R, LimbDark * LD, float ThPA, char * name, int imagedev, 
	    int IfRotate);
int MainVar(Roche * R, LimbDark * LD, int IfWrite);
int ReadModel(Roche * R, LimbDark * LD, char * name, char * sModelfile);
int InitRoche(Roche * R, LimbDark * LD, char * name, int iargc, 
	      char * acargv[], int * j1Wave, int * j2Wave);
int ReadBaselines(Observatory * O, Roche * R);
int ReadInclinations(Roche * R, int iInclmax, float * Incl, float * W,
		     float * DpBmas, int * iIncl);
int ReadLimbDark(LimbDark * LD);
int SurfBright(LimbDark * LD, float Teff, float glog, float mu, 
	       float * SBright);
int GetUVW(Observatory * O, Roche * R, float HA);
/*
int InitPlot(Roche * R, Observatory * O, LimbDark * LD, Scans * S, 
	     Plot * P, int IfWave);
*/
int InitPlot(Roche * R, Observatory * O, LimbDark * LD, Scans * S, 
	     Plot * P);
int PlotTriangle(Roche * R, LimbDark * LD, Observatory * O, Plot * P, 
		 Scans * S);
int ReadScans(Scans * S);
//int VMag(Roche * R, LimbDark * LD, float * Vmag, float * BmV); // soon

/* Defines */

#define CLIGHT 29979245800. // cm/s 
#define TWOPI 6.28318530717958647692
#define PI 3.14159265358979323846
#define PI2 1.57079632679489661923 // PI/2
#define PI4 0.78539816339744830962 // PI/4
#define DEGRAD 180. / PI
#define RADDEG PI / 180.
#define GRAV 6.6720e-8 // dyn cm^2 g^-2
#define SIGMA 5.66956e-5 // as in L=SIGMA*Teff^4*R^2
#define RSUN 6.9599e10 // in cm
#define MSUN 1.989e33 // in gm
#define LSUN 3.826e33 // erg/s
#define AU 1.495979e13 // cm
#define IAPMAX 6 // maximum number of siderostates (6 for NPOI)
#define IBMAX 15 // maximum number of baselines (15 for 6 apertures)
#define ITRIMAX 10 // maximum number of triangles (10 for 6 apertures)
#define ISTRIPMAX 20 // resolution for strip brightness calculation (Disks)
#define IFLUXMAX 20 // resolution for Visual flux and B-V calculation (Flux)
#define IOBSMAX 100 // maximum number of Observations (HAs) per run
#define IPAMAX 6 // maximum number of Position Angles per group
#define IPA_GRPMAX 6 // maximum number of PA groups
