/* Structures for evaluating the Roche Spheroids. */

typedef struct {
  char sName[20]; // Name of Star
  float fRA; // Degrees
  float fDec; // Degrees
  float fEpoch; // For coordinates, probably not important
  float fwOmega; // Fractional Angular Rotation Velocity (unit: fOmegaB)
  float fRNot; // Stellar Radius at wOmega=0 (unit: RSun)
  float fMass; // Stellar Mass (unit: MSun)
  float fRp; // Actual Polar Radius (unit: RSun)
  float fRpB; // Breakup Polar Radius (unit: RSun)
  float fReB; // Breakup Equatorial Radius (unit: RSun)
  float fRe; // Actual Equatorial Radius (unit: RSun)
  float flogGp; // Actual Polar log_g
  float flogGpB; // Polar Breakup log_g
  float flogGe; // Actual Equatorial log_g
  float fOmegaB; // Breakup Angular Velocity
  float fOmega; // Angular Velocity = wOMEGA*fOmegaB
  float fVeqB; // Equatorial Breakup Velocity = fOmegaB*fReB
  float fVeq; // Actual Equatorial Velocity = fOmega*fRe
  float fVol; // Volume of Spheroid (unit: fRpB^3)
  float feqArea; // Area of Cross Section, Equator on (unit: fRpB^2)
  float fArea; // Area of Cross Section, this look-angle (unit: fRpB^2)
  float fIncl; // Polar axis inclination angle (0 = pole on)
  float fPA; // Position angle of axis on sky (added back by CAH)
  float fTeffpB; // Polar breakup Teff(K)
  float fTeffp; // Teff(K) at pole
  float fTeffe; // Teff(K) at equator
  float fAngDiam_pB; // Polar Angular Diameter for Breakup (unit: mas)
  float fAngDiam_Xp; // Angular Diameter along minor axis (unit: mas)
  float fAngDiam_Yp; // Angular Diameter along major axis (equator, unit: mas)
  float fWave; // wavelength(unit: nm)
} Roche;

/*********************************************************************/
/* For the Limbdarkening law and grid */
typedef struct {
  char sLDLaw[10]; // The Limb Darkening Law (ie, Lin, Sqrt, ...)
  char sInstrument[10]; // The interferometer (ie, Mark3, NPOI, ...)
  char sWaveset[10]; // Wavelength set (ie, Reg, Mon, ...)
  char sPath[30]; // path (including back slashes) to the files
  int iWave; // Lambda index in current use
  int indexWaveMax; // Largest wavelength index (+1) in this array
  float fWaveWtMinus; // interpolate wavelength for "Mon" LD files
  int iTeffMax; // Number of Teffs
  int aiLogG[61]; // how many log_g's at each Teff
  float afLD[3][61][11][832]; // X, (Y), and I_1 for [Teff, log_g, & lambda]
  float afLogG[11]; // 5.0, 4.5, ..., 0.0, max (but all start at 5.0)
  float afWaves[832]; // Nominal wavelength set
  float afEffWaves[832]; // Effective wavelength set
  float afTeff[61]; // Teff set
} LimbDark;

/*********************************************************************/
/* For the Limbdarkening law and grid */
typedef struct {
  char sLDLaw[10]; // The Limb Darkening Law (ie, Lin, Sqrt, ...)
  char sInstrument[10]; // The interferometer (ie, Mark3, NPOI, ...)
  char sWaveset[10]; // Wavelength set (ie, Reg, Mon, ...)
  char sPath[30]; // path (including back slashes) to the files
  int iWave; // Lambda index in current use
  int indexWaveMax; // Largest wavelength index (+1) in this array
  float fWaveWtMinus; // interpolate wavelength for "Mon" LD files
  int iTeffMax; // Number of Teffs
  int aiLogG[61]; // how many log_g's at each Teff
  float afLD[3][61][11][32]; // X, (Y), and I_1 for [Teff, log_g, & lambda]
  float afLogG[11]; // 5.0, 4.5, ..., 0.0, max (but all start at 5.0)
  float afWaves[32]; // Nominal wavelength set
  float afEffWaves[32]; // Effective wavelength set
  float afTeff[61]; // Teff set
} LimbDark0;

/*********************************************************************/
/* Observatory parameters, Siderostat locations, Baselines and Triangles*/
typedef struct {
  char sObsFileName[50]; // Name of the (flat) file for baseline info 
  float fLat; // degrees and decimal fractions
  float fLong; // likewise, west of Greenwich is negative
  float fAlt; // may be useful sometime

  int iBaselines; // number of baselines
  /* Baseline names: [0]: Ap0-Ap1 and [1]: Ap1-Ap0. IBMAX=15 */
  char sBaselineNames[15][2][9]; 
  int iBaselineIndex[15][2]; // Index of aperture pair
  /* Ap[1]-Ap[0] (x,y,z) for each pair. x + to East, y + to North, z +
     up, meters*/
  float fBxyz[15][3];  // baselines in horizon coordinates (meters)
  float fBAzi[15][2]; // |B| and azimuth (m, deg N->E) on ground.

  /* Next is the rotation matrix that takes the Baseline (Bx,By,Bz) in
   * local coordinates and transforms it to the (X,Y,Z) system
   * (Thompson, Moran & Swensen 1ed, p 86). Later the rotation matrix
   * for the observation (a function of Hour angle and declination)
   * transforms that to the (u,v,w) coordinates in the star's reference
   * frame. */
  float afRotateBase[3][3]; 
  float fXYZ[15][3]; // baselines in the equatorial plane, see above
  float fUVW[15][3]; // Baselines projected on star (meters)
  float fUTheta[15][2]; // |UVW| and p.a. of baseline (m, deg N->E)
  char sUThNames[15][2][30]; // Add |UVW| and theta to baseline names

  int iTriangles; // maximum index used out of ITRIMAX possible
  int aiIfTri[10]; // if that triangle in use (=1) or not (=0)
  /* Baseline indices for the legs of the triangles.  Sign indicates the
   * direction with "-" meaning backwards along the baseline (use
   * Baseline name index = "1").  IMPORTANT: To encode sign, the index
   * here is the baseline index + 1. */
  int iTriangleBaselines[10][3]; // baseline index + 1 (- means reverse)
  char sTriangleNames[10][28];
} Observatory;


/*********************************************************************/
/* The stellar DISK projections as a function of baseline.  X is along
 * the baseline, Y perpendicular.  (Teff, log_g, mu) are evaluated at
 * each point. 
 *
 * For first order integrations, the quantities are evaluated at the
 * edges of the pixels, not the centers*/
typedef struct {
  /* ISTRIPMAX=>100, ISTRIPMAX+1=>101 in dimensions*/
  int iB; // number of strips along baseline (# of points for 1st ord integ)
  float fdx; // resolution along baseline
  float fdy[100]; // resolution perpendicular to baseline
  int iNy[100]; // how many y's in this row (no more than ISTRIPMAX+1)
  float afX[100]; // X at center of pixels for a baseline (unit: RpB)
  float afY[100][101]; // Y at center of each pixel, perp to baseline
  float afTLgMu[100][101][3]; // (Teff, log_g, mu)=>3 for each pixel
} Disk;


/*********************************************************************/
/* Observations (=Scans) */
typedef struct {

  int iNScans; // number of different scans

  char sPathFileName[50]; // where the data are
  char sFileName[30]; // Name of Data file, used as page title
  char sID[100][20]; // ID (ie, date) for each scan

  float afHA[100]; // Hour Angle for each scan

  int ajnWave[2]; // j1Wave & j2Wave, the wavelength index range for these data
  int aiNWaves[100]; // number of channels actually used in each scan
  int aiWaveIndex[100][32]; // Index to LD file for this wavelength
  float afWaves[100][32]; // Wavelengths of channels used in each scan

  /* iIfTri: 0: separate baselines only 
             1: triangles: triple phases, triple amplitudes
	     2: triple phase but baselines separately 
  */
  int iIfTri; 

  int iNTri; // number of triangles (iIfTri >= 1)
  int aiTri[10]; // indices of the different triangles in each scan
  float afTAmp[100][32][10]; // Triple amplitudes (scan,wave,triangle)
  float afTAmpErr[100][32][10]; // "     "      errors
  int aiTAFlag[100][32][10]; // Bad Data flag (either Amp/error) =0 if good
  float afTPh[100][32][10]; // Closure phases
  float afTPhErr[100][32][10]; // and errors
  int aiTPFlag[100][32][10]; // Bad Data flag

  int iNBaselines; // number of baselines 
  /* 8 combinations/spectrograph, 3 spectrographs - some redundancy */
  int aiBln[24]; // indices of the different baselines in each scan
  float afBVissq[100][32][24]; // Baseline V^2 (scan,wave,baseline)
  float afBVerr[100][32][24]; // Baseline V^2 err
  int aiBFlag[100][32][24]; // Bad Data flag 
} Scans;


/*********************************************************************/
/* Data Sets */
typedef struct {
  char sObjectFile[50]; // ie, FlatFiles/Stars/Vega.dat

  int iNDataSets; // number of data sets (scans)
  char sScanFile[10][50]; // full name (path/name.ext) for the data sets
  int iFormat[10]; // data file format: 0=old NPOI, 1=2Tri NPOI, 2=PTI
  int iIndexLD[10]; // use which Limbdarkening file
  int iIndexBL[10]; // use which Baselines file
  int iIfTri[10]; // individual baselines or triangles

  int iNLimbDarkFiles;
  char sLDLaw[10][10]; // The Limb Darkening Law (ie, Lin, Sqrt, ...)  
  char sInstrument[10][10]; // The interferometer (ie, Mark3, NPOI, ...)
  char sWaveset[10][10]; // Wavelength set (ie, Reg, Mon, ...)
  char sLimbDarkPath[10][30]; // path for Limbdarkening files

  int iNBaselineFiles;
  char sBaselinesFile[10][50]; // full name for baseline (observatory) files
  int aiBlnIndex[10][24];
} DataSets;


/*********************************************************************/
/* Prototypes */

float FractRadius(float w2, float s2);
int Volume(Roche * R);
int RThetaPhifromXY(float X, float Y, Roche * R, float * r, 
		    float * Th, float * Phi);
int Edge(Roche * R, float ThP, float * Th, float * Phi, float * rout);
int Surface(Roche * R, Disk * D, float Theta);
int StripBright(LimbDark * LD, Disk * D, float lam, float * X, 
		float * Weight, float * Uniform);
int LatLong(Roche * R, float dPh, float ThPA, float rscale);
int Picture(Roche * R, LimbDark * LD, float ThPA, char * name, int imagedev, 
	    int IfRotate);
int MainVar(Roche * R, LimbDark * LD, int IfWrite);
int ReadModelChi2(Roche * R, DataSets * DS); // okay
int ReadDataSets(DataSets * DS);
int ReadBaselinesChi2(Observatory * O);
int ReadParameters(int iImax, float * Incl, int * iIncl, int iWmax, float * W,
		   int * iW, int iDpBmax, float * DpBmas, int * iDpB, 
		   int iPAmax, float * PA, int * iPA);
int ReadLimbDarkChi2(LimbDark * LD);
int ReadScansChi2(LimbDark * LD, Scans * S, int iFormat);
int ReadScans2Tri(LimbDark * LD, Scans * S);
int ReadScansPTI(LimbDark * LD, Scans * S);
int ReadScansAltair6Way(LimbDark * LD, Scans * S);
int SurfBright(LimbDark * LD, float Teff, float glog, float mu, 
	       float * SBright);
int GetUVW(Observatory * O, Roche * R, float HA);
int Visibility(Roche * R, LimbDark * LD, Disk * D, float U, float * Re, 
	       float * Im, float * F, int IfWrite);
/*
int InitPlot(Roche * R, Observatory * O, LimbDark * LD, Scans * S, 
	     Plot * P);
int PlotTriangle(Roche * R, LimbDark * LD, Observatory * O, Plot * P, 
		 Scans * S);
*/
//int VMag(Roche * R, LimbDark * LD, float * Vmag, float * BmV); // soon

/* Defines */

#define CLIGHT 29979245800. // cm/s 
#define TWOPI 6.28318530717958647692
#define PI 3.14159265358979323846
#define PI2 1.57079632679489661923 // PI/2
#define PI4 0.78539816339744830962 // PI/4
#define DEGRAD 180. / PI
#define RADDEG PI / 180.
#define GRAV 6.6720e-8 // dyn cm^2 g^-2
#define SIGMA 5.66956e-5 // as in L=SIGMA*Teff^4*R^2
#define RSUN 6.9599e10 // in cm
#define MSUN 1.989e33 // in gm
#define LSUN 3.826e33 // erg/s
#define AU 1.495979e13 // cm
#define IAPMAX 6 // maximum number of siderostates (6 for NPOI)
#define IBMAX 15 // maximum number of baselines (15 for 6 apertures)
#define ITRIMAX 10 // maximum number of triangles (10 for 6 apertures)
#define ISTRIPMAX 50 // resolution for strip brightness calculation (Disks)
#define IFLUXMAX 20 // resolution for Visual flux and B-V calculation (Flux)
#define IOBSMAX 100 // maximum number of Observations (HAs) per run
#define IDATASETS 5 // maximum number of independent data sets
#define IOBSERVATORIES 2 // maximum number of observatories/configurations
#define ILIMBDARKSETS 2 // maximum number of limbdarkening sets
