/***********************************************************************
 * This implements a simple first order Newton-Raphson iteration to
 * find (Theta,Phi) for a given (x',y') (the primes denote the
 * fundamental plane after inclination is accounted for, but with the
 * pole pointed North).  The first guess comes in from outside and is
 * iterated to at least 5 digits.  A flag is returned if it doesn't
 * converge in a relatively small number of iterations.  21/12/04
 *
 * Returns: = N ( >= 1) if converged in N (< Nmax) iterations
 *          = 0 exceeded Nmax iterations (convergence reported)
 *          = -N if blew up after N iterations
 **********************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "roche.h"

int XYxNR(float w, float incl, float X, float Y, float * Th, float * Ph, int IW) {

  float theta=*Th, phi=*Ph, r, x, y, w2 = w*w;
  float sth, s2, cth, sph, cph, si=sin(incl), ci=cos(incl);
  float a1, a2, b1, b2, dx, dy, dth, dph, drdth, yor, xor, det, a;
  int i, iMax=20;

  sth = sin(theta);
  s2 = sth*sth;
  r = FractRadius(w2,s2);
  cth = cos(theta);
  sph = sin(phi);
  cph = cos(phi);
  xor = (sth*cph*ci-cth*si);
  x = r*xor;
  yor = sth*sph;
  y = r*yor;
  if (IW) {
    printf("XY-NR: Th-in=%7.2f Phi-in=%7.2f r=%8.5f X=%8.5f Y=%8.5f\n",
	   (float)DEGRAD*(*Th),(float)DEGRAD*(*Ph),r,X,Y);
  }

  /* Always make sure it isn't trivial */
  dx = X-x;
  dy = Y-y;
  if (fabs(dx)+fabs(dy) < 1.e-6) {
    printf("Nothing to do!\n");
    printf("XY-NR: X=%8.5f x=%8.5f Y=%8.5f y=%8.5f\n",X,x,Y,y);
    return 1;
  }

  for (i=0; i<iMax; i++) {
    /*
    if (IW) {
      printf("XY-NR: i=%2i r=%8.5f x=%8.5f y=%8.5f dx=%8.5f dy=%8.5f\n",
	     i,r,x,y,dx,dy);
    }
    */
    a = 8./27.*w2*r*r*r;
    drdth = 0.;
    if (1.-a*s2 > 1.e-5) drdth = a*r*sth*cth/(1.-a*s2); 
    a1 = drdth*xor+r*(cth*cph*ci + sth*si);
    a2 = -y*ci;
    b1 = drdth*yor + r*cth*sph;
    b2 = r*sth*cph;
    det = a1*b2-a2*b1;
    if (det == 0) {
      printf("XY-NR: determinant singular i=%i\n",i);
      printf("       a1=%g a2=%g b1=%g b2=%g det=%g\n",a1,a2,b1,b2,det);
      return -(i+1);
    }
    dth = (b2*dx - a2*dy)/det;
    dph = (-b1*dx + a1*dy)/det;
    /*
    if (IW) {
      printf("XY-NR: theta=%7.2f phi=%7.2f a=%7.5f drdth=%6.3f\n",
	     (float)DEGRAD*theta,(float)DEGRAD*phi,a,drdth);
      printf("       a1=%g a2=%g b1=%g b2=%g det=%g\n",a1,a2,b1,b2,det);
      printf("       dth=%7.3f dphi=%7.3f\n",(float)DEGRAD*dth,
	     (float)DEGRAD*dph);
    }
    */
    theta += dth;
    phi += dph;
    sth = sin(theta);
    s2 = sth*sth;
    r = FractRadius(w2,s2);
    cth = cos(theta);
    sph = sin(phi);
    cph = cos(phi);
    xor = (sth*cph*ci-cth*si);
    x = r*xor;
    yor = sth*sph;
    y = r*yor;
    dx = X-x;
    dy = Y-y;
    if (fabs(dx)+fabs(dy) < 1.e-5) {
      *Th = theta;
      *Ph = phi;
      return i+1;
    }
  }

  //printf("XY-NR: Did not converge dx'=%8.2g dy'=%8.2g\n",dx,dy);
  *Th = theta;
  *Ph = phi;
  return 0;
}
