FUNCTION uv2proj, uvec, proj, sz_proj
;+NAME/ONE LINE DESCRIPTION OF ROUTINE:
;    uv2proj - Converts unit vectors to projection (screen) coordinates.
;
;DESCRIPTION:
;    This IDL function converts an array of array of unit vectors to an
;    array of projection screen coordinates.  The input unit vector array
;    has dimensions (n,3) where n is the number of vectors to be converted.
;    The output array has dimensions (n,2) and for a 512 by 256 
;    projection, the numbers in this array range from 0 to 511 for the
;    elements in the first row and 0 to 255 for elements in the second.
;    The coordinate system for the output is the same as that of the 
;    input.  This routine is used by DRAW_GRID_LINE to create the
;    meridians and parallels for the projections.
;
;CALLING SEQUENCE:
;    ij = uv2proj, uvec, proj, sz_proj
;
;ARGUMENTS (I = input, O = output, [] = optional):
;    uvec        I     flt arr             Input unit vector array
;    proj        I     string              projection string ('A','S','M')
;    sz_proj     I     lon arr             SIZE(projection array)
;    ij          O     int/long arr        Projection screen coord. array
;
;WARNINGS:
;    Only works for Aitoff, Global Sinusoidal, Molweide and 
;    Polar projections.
;    This is NOT a fully supported user program.
;
;EXAMPLE: 
;
; To convert from a unit vector "uv" to the screen coordinates of an 
; Aitoff projection (stored in the IDL array, 'proj_image'):
;
; proj_ij = uv2proj(uv,'a',SIZE(proj_image))
;
;#
;COMMON BLOCKS:
;    None
;
;PROCEDURE (AND OTHER PROGRAMMING NOTES):
;    Uses CASE statement to direct to proper code for each projection.
;    The Mollweide projection must use a lookup interpolation table
;    generated by moll_sph_xy to perform the conversion.  Also, it
;    must do the conversion on a single element because of peculiarities
;    with the "end points", lat = 0 and 90 degrees.
;
;PERTINENT ALGORITHMS, LIBRARY CALLS, ETC.:
;    Subroutines called: moll_sph_xy
;
;    Uses coordinate transformation formuli that can be found in any 
;    complete book on mapping/projections.
;
;MODIFICATION HISTORY
;    Written by J.M. Gales, Applied Research Corp.   Apr 92
;
;  SPR 12144  Add grid line code for polar projection  J.M. Gales
;  16-MAR-95
;
;.TITLE
;Routine UV2PROJ
;-
;
COMMON mollweide, phi, theta

lon = ATAN(uvec(*,1),uvec(*,0))
lat = ASIN(uvec(*,2))

half_l = sz_proj(1)/2
half_h = sz_proj(2)/2


CASE proj OF

'A' :	BEGIN

	den = SQRT(1 + COS(lat)*COS(lon/2))

	proj_x = half_l - FIX(half_l * (COS(lat)*SIN(lon/2) / den))
	proj_y = half_h + FIX(half_h * (SIN(lat) / den))

	END


'S' :	BEGIN

	proj_x = half_l - FIX(half_l * lon * COS(lat) / !pi)
	proj_y = half_h + FIX(half_h * lat / (!pi/2))

	END


'M' :	BEGIN

	deg_unit = 0.25 * !dpi / 180
	IF (N_ELEMENTS(phi) EQ 0) THEN moll_sph_xy
	proj_x = LONARR(N_ELEMENTS(lat))
	proj_y = LONARR(N_ELEMENTS(lat))

	FOR i=0l,N_ELEMENTS(lat)-1 DO BEGIN

	IF ((uvec(i,2) NE 0) AND (ABS(uvec(i,2)) NE 1.0)) THEN BEGIN

	elem = FIX(ABS(lat(i)) / deg_unit)
	del_theta = theta(elem+1) - theta(elem)
	del_phi = phi(elem+1) - phi(elem)
	theta0 = theta(elem) + (del_theta/del_phi) * (ABS(lat(i)) - phi(elem))

	theta0 = theta0 * lat(i) / ABS(lat(i))

	ENDIF 

	IF ((uvec(i,2) EQ 0) OR (ABS(uvec(i,2)) EQ 1.0)) THEN theta0 = lat(i)

	proj_x(i) = half_l - FIX(half_l * lon(i) * COS(theta0) / !pi)
	proj_y(i) = half_h + FIX(half_h * SIN(theta0))

	ENDFOR

	END


'P' :	BEGIN

	fac1 = SQRT(1-SIN(ABS(lat)))
	sgn = lat / ABS(lat)
	fac2 = 1 - (sgn / 2)

	proj_x = fac2 * half_l - sgn * FIX(0.5 * half_l * fac1 * SIN(lon))
	proj_y = half_h -       FIX(       half_h * fac1 * COS(lon))

	END

ENDCASE

RETURN,[[FIX(proj_x)],[FIX(proj_y)]]
END
;DISCLAIMER:
;
;This software was written at the Cosmology Data Analysis Center in
;support of the Cosmic Background Explorer (COBE) Project under NASA
;contract number NAS5-30750.
;
;This software may be used, copied, modified or redistributed so long
;as it is not sold and this disclaimer is distributed along with the
;software.  If you modify the software please indicate your
;modifications in a prominent place in the source code.  
;
;All routines are provided "as is" without any express or implied
;warranties whatsoever.  All routines are distributed without guarantee
;of support.  If errors are found in this code it is requested that you
;contact us by sending email to the address below to report the errors
;but we make no claims regarding timely fixes.  This software has been 
;used for analysis of COBE data but has not been validated and has not 
;been used to create validated data sets of any type.
;
;Please send bug reports to CGIS@ZWICKY.GSFC.NASA.GOV.


