/***************************************************************************
 *
 *                 amdlib Yorick plugin addons
 *         Copyright 2008-2013 F. Millour and B. Valat
 *                       fmillour@oca.eu
 *
 ***************************************************************************
 *
 * Please note that this script is distributed under the GPL licence,
 * available at http://www.gnu.org/licenses/gpl.txt
 *
 * Please ACKNOWLEDGE the use of this script for any use in a publication
 * (please cite Millour, F.; Valat, B. et al. 2008, SPIE, 7013)
 *
 * Parts of this code have been transferred to amdlib version 3.0
 * (in development), with the agreement of the authors.
 *
 ***************************************************************************
 *
 * HOW TO INSTALL:
 *
 * - First, you need the amdlib software. It is freely
 * downloadable from the JMMC website:
 * http://www.jmmc.fr/data_processing_amber.htm
 *
 * - Copy the "amdlibPipeline.i" script to your "~/Yorick" directory
 * (that you have to create if it does not yet exist).
 *
 * - Once amdlib is launched via its yorick interface, type-in:
 * include,"amdlibPipeline.i"
 * help,"amdlibPipeline.i" to get instructions on how it works
 *
 * - enjoy !
 *
 ***************************************************************************/

/*************************************************************/
/* Test amdlib installation **********************************/
/*************************************************************/

if(is_void(amdlib))
{
    write,"No amdlib pre-loaded, trying to load it";
    fh = popen("ls "+Y_SITE+"i/*amdlib*",0);
    dat = [];
    while(line=rdline(fh))
        grow,dat,line;
    close,fh;

    if(numberof(dat)>0)
    {
        write,"You have amdlib v>=2.0 installed, loading plugin ...";
        require,"amdlib.i";
    }
    else
    {
        write,"\n\
 ******************************************************\n\
 *                                                    *\n\
 *   Please install first the amdlib yorick plugin    *\n\
 *                 to use this script                 *\n\
 *          http://amber.obs.ujf-grenoble.fr          *\n\
 *               section data processing              *\n\
 *      article \"AMBER data reduction software\"     *\n\
 *                                                    *\n\
 ******************************************************\n";
        error;
    }
}

// Configure amdlib properly
amdlibSetPreference,"errorEnable",1;
amdlibSetPreference,"debugEnable",1;
amdlibSetPreference,"fileChooser","new";
amdlibSetPreference,"graphicsDpi",70;

/*************************************************************/
/* Use web tool from yoco ************************************/
/*************************************************************/

webTool = yocoWEB_TOOL;

/*************************************************************/
/** Includes *************************************************/
/*************************************************************/

yocoLogInfo,"#include \"amdlibPipeline.i\"";
colors= ["red","green","blue","magenta","cyan","yellow","black"];
require, "bessel.i";

/***************************************************************************/

func amdlibPipeline(void)
    /* DOCUMENT amdlibPipeline

       DESCRIPTION
       ********************
       *** EXPERIMENTAL ***
       ********************

       AMBER data reduction pipeline
       
       See the "ADVISED REDUCTION STEPS" section at the end of this help for
       more details on how to use this script with AMBER data.


       CAUTIONS
       *************************************************
       * Please install first the amdlib yorick plugin *
       * to use this script: ***************************
       * http://amber.obs.ujf-grenoble.fr **************
       * section data processing ***********************
       * article "AMBER data reduction software" *******
       * Please also make sure you have the wget tool: *
       * http://www.gnu.org/software/wget **************
       *************************************************


       AUTHORS
       see the authors list at the beginning of this script

       FUNCTIONS
       - ComputePistonAIPS              : New implementation of piston
       - ComputePistonIterative         : iterative piston measurement
       - ComputePistonPhase             : finite difference piston measurement
       - LSperiod                       : Lomb-Scargle periodogram
       - Rydberg                        : 
       - amdlibBinOiDatas               : 
       - amdlibChangeSpectralShift      : 
       - amdlibCheckSpectralShift       : 
       - amdlibCheckSpectralShiftOiData : 
       - amdlibComputeAtt               : 
       - amdlibConvertScienceToCalib    : TEST ONLY. !!! DO NOT USE !!! For
       tests only: allow to switch between SCIENCE and CALIB keywords on a
       series of files. MODIFIES THE RAW FILES !!!
       - amdlibCorrectAllCoherenceLength: Same as following but applied to
       many files
       - amdlibCorrectCoherenceLength   : Correct raw visibility files from
       the coherence length visibility loss. Useful for low resolution data
       only.
       - amdlibDumpBinTable             : dump a bin table from one fits
       file to another
       - amdlibGetJitterByInterspectrum : TEST ONLY. get the jitter effect
       - amdlibListAllStars             : 
       - amdlibListStars                : 
       - amdlibLoadSpectrum             : 
       - amdlibLoadSpectrums            : 
       - amdlibPipeline                 : Tentative data reduction pipeline.
       - amdlibPlotDataSeries           : From a series of data files
       (OI DATA), plots visibility versus wavelength for all files.
       - amdlibPlotUV                   : 
       - amdlibPlotUV_Night             : Plot the UV coverage for all stars
       in a night
       - amdlibPlotV2fSpFreq            : Plot visibility versus spatial
       frequency
       - amdlibPlotV2fUVFreq            : Plot a 2D-map of visibility versus
       spatial frequency
       - amdlibShowSpectrum             : 
       - applyWlenSolutionLR            : 
       - applyWlenSolutionMR            : 
       - averageDataFMi                 : 
       - calibrateSpectrumHR            : 
       - calibrateSpectrumMR            : 
       - calibrateSpectrumMR_airmass    : 
       - changeAMBERNamesToDVD          : 
       - changeAMBERNamesToOrig         : 
       - changeAllAMBERNamesToDVD       : 
       - changeVISIRNamesToDVD          : 
       - checkBadPixelMap               : 
       - comparData                     : 
       - computeBIPM_Density            : 
       - computeChromaticPiston         : Computes piston on a small
       lambda-patch in a sliding window to get an estimation of the chromatic
       piston
       - computeCompressibility         : 
       - computeDiffPhasePiston         : From an OI data AVG, computes a
       chromatic piston coming from the differential phase
       - computeIndexWetAirCiddor       : 
       - computeP2vmPhase               : From a P2VM file, get the Fourier
       phase of the fringes on all 3 baselines
       - computeP2vmPiston              : Used with computeP2vmPhase,
       allow to compute a chromatic piston from a P2VM reference file
       - computePhiChromDryAirCiddor    : 
       - computePhiChromDryAirVannier   : 
       - computePhiChromWetAirCiddor    : 
       - computePhiChromWetAirMathar    : 
       - computePhiDiffChromFromOIFile  : 
       - computeWlenSolutionHR          : 
       - computeWlenSolutionLR          : 
       - computeWlenSolutionMR          : 
       - computeWsolAndSpecType         : 
       - correctAllChromOPD             : 
       - correctChromOPD                : 
       - divideAllByTemplate            : 
       - divideByTemplate               : 
       - filterOiFile                   : Filter (i.e. select wavelength,
       bases, etc.) an AMBER OI fits file
       - filtrIris                      : filter the IRIS fringes. Useful in HR
       - filtreArray                    : used in filtrIris
       - fitFac                         : 
       - fitWsol                        : 
       - fixDateObs                     : 
       - gausf                          : used in filtreArray
       - getAllJitter                   : TEST ONLY. get the jitter effect
       - getJitter                      : TEST ONLY. get the jitter effect
       - getSpecsForSpectralShift       : 
       - getTemplateSpec                : 
       - getWlenSol                     : 
       - get_color                      : 
       - interpolateBadPixels           : interpolate the pixels set to 0
       (AMBER convention for bad pixels)
       - interpolateBadPixels3          : 
       - lineSpec                       : 
       - mag2CorrMag                    : 
       - mag2CorrMagUD                  : 
       - mergeSpectrum                  : 
       - normalizeSpec                  : 
       - plotChromOPD                   : plot chromatic OPD
       - plotContBoxes                  : 
       - plotKeyFTime                   : 
       - plotP2vmPiston                 : plot P2vm piston
       - plotTfVal                      : 
       - readOI_FITS_DIT                : Reads the Detector Integration
       Time from an OI fits file
       - readOI_FITS_MJD                : Reads the Mean Julian Day from
       an OI fits file
       - readOI_FITS_UV                 : Reads UV coordinated from an
       OI fits file
       - readOI_FITS_WLEN               : Reads the wavelength table
       - readOI_FITS_star_name          : Reads the star name in an OI fits
       file
       - readableError                  : 
       - sortAMBERFiles                 : Sort files by DIT, etc.
       - testChromDry                   : 
       - testChromWet                   : 


       ******************************************************
       *** ADVISED REDUCTION STEPS **************************
       ******************************************************

       // Start with a clean raw data directory which contains the data of a
       // whole night (not only the content of your DVD, but also the result
       // of a ftp transfer from the ESO archive of all the calibration data)
       // The command ls *_PRODUCT* *LOG* *CAL* should give no result
       // ((re)move files if necessary)

       -----------------------------------------------------------
       // Have a look to the stars observed that night (select a directory)
       > amdlibListStars

       -----------------------------------------------------------
       // Valid for LR and MR: Compute P2VMs, using an emission line file
       // for spectral reshifting
       > [amdlib] amdlibComputeP2vm, specCalShifts=amdlibChangeSpectralShift();
       ****** OR ******
       // Valid for HR: The same but with a filtering of the spurious IRIS
       // fringes
       > [amdlib] amdlibComputeP2vm, specCalShifts=amdlibChangeSpectralShift(filtrIris=1);
       ****** OR ******
       // If nothing to declare, rune default
       > [amdlib] amdlibComputeAllP2vm

       -----------------------------------------------------------
       // [amdlib] Compute all OI Data. Answer "NO" to all questions
       > amdlibComputeAllOiData,splitBands=1

       -----------------------------------------------------------
       // Plot the UV coverage obtained on the science stars
       > amdlibPlotUV_Night

       -----------------------------------------------------------
       // Produce a calibration stars diameters file.
       // You need an Internet connexion for this step.
       > amdlibGetCalibzSize

       -----------------------------------------------------------
       // Eventually, edit and fill-in the missing information
       // in the file named ~/amdlibCalibDatabase.txt

       -----------------------------------------------------------
       // Sort the files by instrument mode, FINITO use, BCD, etc.
       > sortAMBERFiles, keywords=["ESO OCS OBS MODE","ESO OCS OBS SPECCONF", "ESO DET DIT", "ESO DEL FT STATUS", "ESO INS OPTI7 NAME"],keyDec=[0,0,1,0,0];

       -----------------------------------------------------------
       // [amdlib] Eventually, append raw oi data files to increase SNR
       > amdlibAppendAllOiData

       -----------------------------------------------------------
       // [amdlib] Compute frame selection by fringe SNR.
       // This is suitable for MR and HR data
       > amdlibPerformAllFrameSelection, selCriterion="snr", selType="percent", selValue=20;
       ****** OR ******
       // [amdlib] Compute frame selection, first selecting by flux, then by piston,
       // and then by fringe SNR. This is suitable for LR data
       > amdlibPerformAllFrameSelection,
       selCriterion=["flux", "piston", "snr"],
       selType=["threshold", "threshold", "percent"],
       selValue=[3, 10000, 20];

       -----------------------------------------------------------
       // Compute the wavelength solution, only on the AVG files

       // MR:
       > computeWlenSolutionMR
       // MR (alternate, with correlations):
       > computeWsolAndSpecType

       // LR:
       > shift = stretch = [];
       > computeWlenSolutionLR, shift, stretch

       // HR not yet done !

       -----------------------------------------------------------
       // Apply the wavelength solution to a series of files

       // MR:
       applyWlenSolutionMR // a coefficitent file is asked and then the OI data files to apply the correction.

       // LR:
       applyWlenSolutionLR, shift, stretch;
       // Here the shift and stretch found with computeWlenSolutionLR are used.

       // HR not yet done !

       // -----------------------------------------------------------
       // Correct Chromatic OPD effect on differential phase, using ambient data
       correctAllChromOPD
       
       // -----------------------------------------------------------
       divideAllByTemplate

       // -----------------------------------------------------------

       // // Calibrate the spectrum, for each science/calibrator pair.
       // // For now only the Br
       Gamma line problem is solved by this script
       // (see Hanson et al. 1996)
       // MR:
       calibrateSpectrumMR
       
       // Compute a series of "star-free" spectra for spectral calibration
       divideAllByTemplate
       // MR (alternate, takes into account airmass in the computation):
       calibrateSpectrumMR_airmass
       // LR, HR not yet done !

       -----------------------------------------------------------
       // If observations are in LR, Correct for the coherence length loss, 
       > amdlibCorrectAllCoherenceLength

       -----------------------------------------------------------
       // Compute the transfer function for all the night
       > amdlibComputeTransFunc;

       -----------------------------------------------------------
       // Plot the transfer function vs time for a given spectral region

       // K band
       > amdlibPlotTransFuncTime,[2.1,2.2]*1e-6;

       // H band
       > amdlibPlotTransFuncTime,[1.55,1.65]*1e-6;

       -----------------------------------------------------------
       // Plot the transfer function vs wavelength for all the night 
       > amdlibPlotTransFuncWlen;

       -----------------------------------------------------------
       // Calibrate the data 
       > amdlibCalibrateOiDataFMi;

       -----------------------------------------------------------
       // Eventually, bin several calibrated files
       > amdlibBinOiDatas

       -----------------------------------------------------------
       // [amdlib] Plot the result
       > amdlibShowOiData

       -----------------------------------------------------------
       // Plot the result versus spatial frequency
       > amdlibPlotV2fSpFreq

       -----------------------------------------------------------
       // Plot a 2D-map of squared visibilities versus spatial frequency
       > amdlibPlotV2fUVFreq


       
       computeWsolAndSpecType
       applyWlenSolutionMR
       // a coefficitent file is asked and then the OI data files
       // to apply the correction.
       correctAllChromOPD
       divideAllByTemplate
       calibrateSpectrumMR_airmass
       

       SEE ALSO
       yocoFile, amdlib
    */
{
    version = "1."+strpart(strtok("$Revision: 692 $",":")(2),2:-2);
    if (am_subroutine())
    {
        write, format="package version: %s\n", version;
        help, amdlibPipeline;
    } 
    return version;
}

/***************************************************************************/ 

func readableError(msg)
    /* DOCUMENT readableError(msg)

       DESCRIPTION
       Issue a readable error with a GUI

       PARAMETERS
       - msg: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(msg))
    {
        write,"\n ***********************************************************";
        write,"NO OFFICIAL ERROR MESSAGE SO FAR.\n Record your last function call, the files you used,\n and the line and file mentionned below, and contact us\n with these informations";
        write,"***********************************************************\n";
        error;
    }
    else
    {
        write,"\n ******************************************************";
        write,"ERROR:",msg;
        write,"******************************************************\n";
        error;
    }

}

/*************************************************************/
/** Acknowledgement ******************************************/
/*************************************************************/

write,
    "******************************************************\n\
 *                                                    *\n\
 *                   AMDLIB-PIPELINE                  *"
    v="";
v = "Version: "+amdlibPipeline();
len=54;
len2= strlen(v);
strt =len/2-(len2+1.1)/2+2;
stp = len/2+(len2-1.1)/2+2;
count=1;
str="*";
for(k=1;k<=strt-1;k++)
    str=str+" ";
str=str+v;
for(k=stp;k<=len-1;k++)
    str=str+" ";
str=str+"*";
write,str;
write,
    "*                                                    *\n\
 *             amdlib Yorick plugin addons            *\n\
 *                                                    *\n\
 *               by F. Millour, B. Valat              *\n\
 *                   fmillour@oca.eu                  *\n\
 *                                                    *\n\
 * - Please ACKNOWLEDGE the use of this script (or    *\n\
 *   parts of it) in the case it is of any use in a   *\n\
 *   publication. The corresponding proceeding        *\n\
 *   article is the following:                        *\n\
 *       Millour, F. et al. 2008, SPIE, 7013          *\n\
 * - Please note that this script is distributed      *\n\
 *   under the GPL licence, available at              *\n\
 *   http://www.gnu.org/licenses/gpl.txt              *\n\
 * - This script is meant to be used for functions    *\n\
 *   prototyping ONLY. Use it at your own risks!      *";
write,
    "*                                                    *\n\
 ******************************************************\n\
\n\
For help about use, just type \"help,amdlibPipeline\"\n";

/***************************************************************************/

// Obsolete functions redefined with new ones here
_sendVizQuery           = _yocoCdsSendVizierQuery;
_yocoCdsReadVizierQuery = _yocoCdsSendVizierQuery;
getSED                  = yocoCdsGetSED;
vizQuery                = yocoCdsVizierQuery;
amdlibStarDiamQuery     = yocoCdsQueryStarDiameter;
amdlibGetCalibzSize     = amdlibSearchAllStarDiameters;

/***************************************************************************/

func readOI_FITS_UV(file)
    /* DOCUMENT readOI_FITS_UV(file)
       
       DESCRIPTION
       Reads the UV information in an OI FITS file
       
       SEE ALSO: 
    */
{
    status = cfitsio_typeinit_status();
    fh = cfitsio_typeinit_fitsfile();
    __ffopen, fh, string(file), int(READONLY), status;
    if(!status)
    {
        cfitsio_goto_hdu,fh,"OI_ARRAY"; // go to a HDU
        table = cfitsio_read_bintable(fh,titles); // read the binary table
        object = where(titles=="TEL_NAME");
        if(numberof(object)==1)
            objname = (*table(object(1)));
        nTels = numberof(objname);
        nBases = nTels * (nTels-1)/2;

        cfitsio_goto_hdu,fh,"OI_VIS"; // go to a HDU
        table = cfitsio_read_bintable(fh,titles); // read the binary table
        cfitsio_close,fh; // close the file

        object = where(titles=="UCOORD");
        if(numberof(object)==1)
            uCoord = (*table(object(1)))(1:nBases);
        object = where(titles=="VCOORD");
        if(numberof(object)==1)
            vCoord = (*table(object(1)))(1:nBases);
    }
    else
        readableError,status;
    __ffclos, fh, status;

    uvCoord = transpose(grow(array(uCoord,1),array(vCoord,1)));
    return uvCoord;
}

/***************************************************************************/

func readOI_FITS_star_name(file)
    /* DOCUMENT readOI_FITS_star_name(file)

       DESCRIPTION
       Gets the star name from an OI FITS file, as it is not stored in a
       fits keyword but in a binary table column.

       PARAMETERS
       - file: the oifits file to read

       RESULTS
       returns the star name

       EXAMPLES
       file = "/home/blah/OIFITS/dummyTest.oifits";
       readOI_FITS_star_name(file);

       SEE ALSO:
    */
{
    // status = cfitsio_typeinit_status();
    //fh     = cfitsio_typeinit_fitsfile();
    fh = __ffopen( string(file), int(READONLY));
    if(is_void(fh))
        return [];
    
    cfitsio_goto_hdu,fh,"OI_TARGET"; // go to a HDU
    table = cfitsio_read_bintable(fh,titles); // read the binary table
    cfitsio_close,fh; // close the file
    
    object = where(titles=="TARGET");
    if(numberof(object)==1)
        objname = (*table(object(1)))(1);

    return objname;
}

/***************************************************************************/

func readOI_FITS_WLEN(file, &wlen, &bandwidth)
    /* DOCUMENT readOI_FITS_WLEN(file, &wlen, &bandwidth)

       DESCRIPTION
       reads the wavelength binary table of an OI fits file

       PARAMETERS
       - file     : the oifits file to read
       - wlen     : the result wavelength table
       - bandwidth: the result bandwidth table

       EXAMPLES
       file = "/home/blah/OIFITS/dummyTest.oifits";
       readOI_FITS_WLEN(file);

       SEE ALSO
    */
{
    status = cfitsio_typeinit_status();
    fh = cfitsio_typeinit_fitsfile();
    __ffopen, fh, string(file), int(READONLY), status;
    if(!status)
    {
        cfitsio_goto_hdu,fh,"OI_WAVELENGTH"; // go to a HDU
        table = cfitsio_read_bintable(fh,titles); // read the binary table

        wlen = *table(where(titles=="EFF_WAVE")(1));
        bandwidth = *table(where(titles=="EFF_BAND")(1));
    }
    else
        readableError,status;
    cfitsio_close,fh; // close the file

    return wlen;
}

/***************************************************************************/

func readOI_FITS_MJD(file)
    /* DOCUMENT readOI_FITS_MJD(file)

       DESCRIPTION
       Reads the Mean Julian Day from an OI fits file 

       PARAMETERS
       - file: 

       RESULTS 

       CAUTIONS
       This function looks only at the table OI_VIS.

       EXAMPLES 

       SEE ALSO:
    */
{
    status = cfitsio_typeinit_status();
    fh = cfitsio_typeinit_fitsfile();
    __ffopen, fh, string(file), int(READONLY), status;
    if(!status)
    {
        cfitsio_goto_hdu,fh,"OI_VIS"; // go to a HDU
        table = cfitsio_read_bintable(fh,titles); // read the binary table
        cfitsio_close,fh; // close the file

        object = where(titles=="MJD");
        if(numberof(object)==1)
            objname = (*table(object(1)))(1);
    }
    return objname;
}

/***************************************************************************/

func readOI_FITS_DIT(file)
    /* DOCUMENT readOI_FITS_DIT(file)

       DESCRIPTION
       Reads the DIT (Detector Integration Time) from an OI FITS file

       SEE ALSO:
    */
{
    // status = STATUS_INIT;
    // fh = cfitsio_TTYPE_type(TSTRING);
    // _ffopen, &fh, string(file), int(READONLY), &status;
    fh = cfitsio_open_file(file, READONLY);
    if(!status)
    {
        cfitsio_goto_hdu,fh,"OI_VIS"; // go to a HDU
        table = cfitsio_read_bintable(fh,titles); // read the binary table
        cfitsio_close,fh; // close the file

        object = where(titles=="INT_TIME");
        if(numberof(object)==1)
            objname = (*table(object(1)))(1);
    }
    return objname;
}

/***************************************************************************/

overPlotStarNames = amdlibOverPlotLabels

    /***************************************************************************/

    func amdlibCorrectAllCoherenceLength(void, rawOiDir=, avgDir=, p2vmDir=, outputDir=, overwrite=, useP2VM=, useDiffPhi=)
    /* DOCUMENT amdlibCorrectAllCoherenceLength(rawOiDir=, avgDir=, p2vmDir=, outputDir=, overwrite=, useP2VM=, useDiffPhi=)

       DESCRIPTION
       (Supposedly) correct all coherence lengths in a set of files

       SEE ALSO
    */
{
    yocoLogTrace,"amdlibCorrectAllCoherenceLength()";

    if(is_void(overwrite))
        overwrite=0;

    if(is_void(useP2VM))
        useP2VM=1;

    /* Read the window size */
    local graphicsDpi;
    if (amdlibCheckGraphicsDpi(graphicsDpi) == 0)
    {
        yocoError, "Check function's parameters";
        return 0;
    }

    /* Select directory containing files to be treated */
    message = "Choose the raw data directory you want to treat"; 
    if (amdlibFileChooser(message, rawOiDir) == 0)
    {
        return 0;
    }

    /* Select directory containing files to be treated */
    message = "Choose the avg data directory you want to treat"; 
    if (amdlibFileChooser(message, avgDir) == 0)
    {
        return 0;
    }

    /* Select directory containing files to be treated */
    message = "Choose the p2vm data directory you want to treat"; 
    if (amdlibFileChooser(message, p2vmDir) == 0)
    {
        return 0;
    }

    /* Check outputDir */
    if ( !amdlibCheckProductDir(outputDir, 1, overwrite, suffix="COR", up=1, inputDir=avgDir) )
    {
        yocoError, "Could not construct 'outputDir'";
        return 0;
    }

    if (!open(outputDir, "r", 1))
    {
        /* If outputDir does not exist, obviously force overwrite */
        overwrite = 1;
        /* Create the product dir if not existing */
        mkdir, outputDir;
    }

    yocoLogInfo, "Coherence length-corrected directory is: "+outputDir;




    /*** Find the raw oidata files ***/
    /* Find the default logFile name */
    if (amdlibGetLogName(rawOiDir+"/", logFile) == 0)
    {
        yocoError, "Could not get log file name", , 1;
        return 0;
    }
    logFile = rawOiDir + "/" + string(logFile);

    /* Read the log of the rawOiDir */
    if (!amdlibReadLog(logFile, OBSsTab, titles))
    {
        yocoError, "Could not read log file", , 1;
        return 0;
    }

    rawtabFiles = OBSsTab(where(titles=="fileName"), );
    rawtabIds   = OBSsTab(where(titles=="P2VM_ID"), );
    rawproCatg  = OBSsTab(where(titles=="pro_catg"), );

    /* Getting observing files index */
    RAWFilesNr = where(strmatch(rawproCatg,"CALIB_REDUCED")|
                       strmatch(rawproCatg,"SCIENCE_REDUCED"));

    if (numberof(RAWFilesNr) == 0)
    {
        yocoLogWarning, "No observing data, unable to compute visibilities !";
    }

    /* Geting observing files and its corresponding data */
    RAWFiles = rawtabFiles(RAWFilesNr);
    RAWIds   = rawtabIds(RAWFilesNr); 
    nbObs = numberof(RAWFilesNr);



    /*** Find the avg files ***/
    /* Find the default logFile name */
    if (amdlibGetLogName(avgDir+"/", logFile) == 0)
    {
        yocoError, "Could not get log file name", , 1;
        return 0;
    }
    logFile = avgDir + "/" + string(logFile);

    /* Read the log of the avgDir */
    if (!amdlibReadLog(logFile, OBSsTab, titles))
    {
        yocoError, "Could not read log file", , 1;
        return 0;
    }

    avgtabFiles = OBSsTab(where(titles=="fileName"), );
    avgtabIds   = OBSsTab(where(titles=="P2VM_ID"), );
    avgproCatg  = OBSsTab(where(titles=="pro_catg"), );

    /* Getting observing files index */
    AVGFilesNr = where(strmatch(avgproCatg,"CALIB_AVERAGED")|
                       strmatch(avgproCatg,"SCIENCE_AVERAGED"));

    if (numberof(AVGFilesNr) == 0)
    {
        yocoLogWarning, "No observing data, unable to compute visibilities !";
    }

    /* Getting observing files and its corresponding data */
    AVGFiles = avgtabFiles(AVGFilesNr);
    AVGIds   = avgtabIds(AVGFilesNr); 

    /* Getting observing files index */
    SELFilesNr = where(strmatch(avgproCatg,"FRAME_SELECTION"));

    if (numberof(SELFilesNr) == 0)
    {
        yocoLogWarning, "No observing data, unable to compute visibilities !";
    }

    /* Getting observing files and its corresponding data */
    SELFiles = avgtabFiles(SELFilesNr);
    SELIds   = avgtabIds(SELFilesNr); 


    /*** Read P2vm files names ***/
    /* Find the default logFile name */
    if (amdlibGetLogName(p2vmDir+"/", logFile) == 0)
    {
        yocoError, "Could not get log file name", , 1;
        return 0;
    }
    logFile = p2vmDir + "/" + string(logFile);

    /* Read the log of the rawOiDir */
    if (!amdlibReadLog(logFile, P2VMsTab, titles))
    {
        yocoError, "Could not read log file", , 1;
        return 0;
    }

    p2vmtabFiles = P2VMsTab(where(titles=="fileName"), );
    p2vmtabIds   = P2VMsTab(where(titles=="P2VM_ID"), );
    p2vmproCatg  = P2VMsTab(where(titles=="pro_catg"), );

    /* Getting observing files index */
    P2VMFilesNr = where(strmatch(p2vmproCatg,"P2VM_REDUCED"));

    if (numberof(P2VMFilesNr) == 0)
    {
        yocoError, "No P2VM, unable to correct visibilities !";
    }

    /* Geting observing files and its corresponding data */
    P2VMFiles = p2vmtabFiles(P2VMFilesNr);
    P2VMIds   = p2vmtabIds(P2VMFilesNr);




    

    /* Extract the names and extentions */
    yocoFileSplitName, RAWFiles, , RAWNames, RAWsuffix;

    // Build up output files names
    inputOiFitsFiles = rawOiDir+"/"+RAWFiles;
    
    /* AVG file determination for use with the observing files */
    usedAVGFiles = [];
    for (iObs = 1; iObs <= nbObs; iObs++)
    {
        raw_name = yocoStrReplace(RAWFiles(iObs),"RAW","");
        usedAvgNr = where(raw_name == yocoStrReplace(AVGFiles,"AVG",""));
        grow, usedAVGFiles, AVGFiles(usedAvgNr);
    }
    
    /* Split the initial filenames */
    yocoFileSplitName, usedAVGFiles, obsDir, obsFile, obsSuffix;
    /* Construct the output filenames by changing
       the RAW into COR, or simply add _COR to the name */
    obsFile = obsFile + ["_AVG",""](strmatch(obsFile,"_AVG")+1);
    obsFile = yocoStrReplace(obsFile, "AVG","AVG_COR"); 
    outputOiFitsFiles = swrite(format="%s/%s%s", outputDir, 
                               obsFile, obsSuffix);
    
    /* SEL file determination for use with the observing files */
    usedSELFiles = [];
    for (iObs = 1; iObs <= nbObs; iObs++)
    {
        raw_name = yocoStrReplace(RAWFiles(iObs),"RAW","");
        usedSelNr = where(raw_name == yocoStrReplace(SELFiles,"SEL",""));
        grow, usedSELFiles, SELFiles(usedSelNr);
    }
    
    /* P2VM determination for use with the observing files */
    if (useP2VM == 1)
    {
        usedP2VMFiles = [];
        for (iObs = 1; iObs <= nbObs; iObs++)
        {
            /* Found the associated Obs files */
            correspondingP2VM = where(RAWIds(iObs)==P2VMIds);
            grow, usedP2VMFiles, P2VMFiles(correspondingP2VM);
        }
    }
    else
    {
        usedP2VMFiles = array("", nbObs);
        yocoLogInfo, "You decided not to use P2VMs.";
    }

    for (iObs = 1; iObs <= nbObs; iObs++)
    {
        // /* Check if produced file already exist if yes,
        // skip this file expect if overwrite==1 */
        // if ( yocoTypeIsFile(outputOiFitsFiles(iObs)) && (overwrite==0) )
        // {
        // yocoLogInfo, "Skipped because the following file already exist:",
        // outputOiFitsFiles(iObs);
        // continue;
        // }

        // yfvysd()
        if(!is_void(RAWFiles(iObs))&
           !is_void(usedP2VMFiles(iObs))&
           !is_void(usedAVGFiles(iObs))&
           !is_void(usedSELFiles(iObs))&
           !is_void(outputOiFitsFiles(iObs)))
            amdlibCorrectCoherenceLength,
                inputRawOiFile  = rawOiDir + RAWFiles(iObs),
                inputP2vmFile   = p2vmDir  + usedP2VMFiles(iObs),
                inputAvgOiFile  = avgDir   + usedAVGFiles(iObs),
                inputSelFile    = avgDir   + usedSELFiles(iObs),
                outputAvgOiFile = outputOiFitsFiles(iObs),
                overwrite       = overwrite,
                useDiffPhi      = useDiffPhi;
    }

    /* Change permissions */
    if (yocoSystem("chmod -R ug+rw " + outputDir) != 0)
    {
        return 0;
    }

    yocoLogInfo, "amdlibCorrectAllCoherenceLength done\n"; 
    return 1;
}

    /***************************************************************************/

    func amdlibCorrectCoherenceLength(inputAvgOiFile=, inputRawOiFile=, inputSelFile=, inputP2vmFile=, outputAvgOiFile=, pistonOffsets=, overwrite=, useDiffPhi=)
        /* DOCUMENT amdlibCorrectCoherenceLength(inputAvgOiFile=, inputRawOiFile=, inputSelFile=, inputP2vmFile=, outputAvgOiFile=, pistonOffsets=, overwrite=, useDiffPhi=)

           DESCRIPTION
           (Supposedly) correct coherence length from an AMBER OI DATA file

           SEE ALSO 
        */
    {
        if(is_void(overwrite))
            overwrite=0;

        if(is_void(useDiffPhi))
            useDiffPhi=0;

        message = "Please choose AVG OI file"; 
        if (amdlibFileChooser(message, inputAvgOiFile) == 0)
        {
            return 0;
        }

        message = "Please choose SEL file"; 
        if (amdlibFileChooser(message, inputSelFile) == 0)
        {
            return 0;
        }

        message = "Please choose RAW OI file"; 
        if (amdlibFileChooser(message, inputRawOiFile) == 0)
        {
            return 0;
        }

        if(is_void(inputSelFile))
        {
            testSelFile = yocoStrReplace(inputAvgOiFile,
                                         ["AVG"],
                                         ["SEL"]);
            if (open(testSelFile,"r",1))
            {
                if (_amdlibGetKwdVals(testSelFile,"ESO PRO CATG")(1)==
                    "FRAME_SELECTION")
                {
                    inputSelFile = testSelFile;
                }
                else
                {
                    yocoLogWarning,"The selection file is corrupted !";
                    return 0;
                }
            } 
            else
            {
                yocoLogWarning,"No selection file !";
                return 0;
            }
        }

        if(is_void(inputRawOiFile))
        {
            testRawFile = yocoStrReplace(inputAvgOiFile,
                                         ["AVG"],
                                         ["RAW"]);
            if (open(testRawFile,"r",1))
            {
                if ((_amdlibGetKwdVals(testRawFile,"ESO PRO CATG")(1)==
                     "CALIB_REDUCED")||
                    (_amdlibGetKwdVals(testRawFile,"ESO PRO CATG")(1)==
                     "SCIENCE_REDUCED"))
                {
                    inputRawOiFile = testRawFile;
                }
                else
                {
                    yocoLogWarning,"The raw file is corrupted !";
                    return 0;
                }
            } 
            else
            {
                yocoLogWarning,"No raw file !";
                return 0;
            }
        }

        message = "Please choose P2VM file"; 
        if (amdlibFileChooser(message, inputP2vmFile) == 0)
        {
            yocoLogWarning,"You did not select a P2VM file, proceeding without ...";
            inputP2vmFile = [];
        } 


        if(is_void(outputAvgOiFile))
            outputAvgOiFile = yocoStrReplace(inputAvgOiFile,
                                             ["AVG"],
                                             ["AVG_TEST"]);

        // Test if gzipped
        yocoFileSplitName,inputAvgOiFile,dir,name,ext;
        if (strmatch(ext,".gz"))
        {
            ext = sum(yocoStrSplit(ext,".gz"));
            inputAvgOiFile = dir+name+ext;
            recompressFile = "yes";

            yocoFileSplitName,outputAvgOiFile,dir2,name2;
            outputAvgOiFile = dir2+name2+ext;
        }

        yocoLogInfo,"Output file is",outputAvgOiFile;

        /* Check if produced file already exist if yes,
           skip this file expect if overwrite==1 */
        if ( yocoTypeIsFile(outputAvgOiFile) && !overwrite)
        {
            yocoLogInfo, "Skipped because the following file already exist:",
                outputAvgOiFile;
            return 0;
        }
        else if(yocoTypeIsFile(outputAvgOiFile) && overwrite)
        {
            yocoLogInfo, "File " + outputAvgOiFile + " is in the way, removing it";
            remove,outputAvgOiFile;
        }


        // Read the science star OI_VIS file
        yocoLogInfo, "reading file "+ inputRawOiFile;

        /* DUMBLY READING TWICE THE FILE */
        _amdlibReadVis, inoiWave, inamberPhot, inoiVis, inoiVis2, inoiVis3,
            inamberOpd, inamberInsCfg, inoiArray, inoiTarget,
            inputOiFile=inputRawOiFile;

        if(!amdlibLoadOiData(wlen, bandWidth, time, fringeSNR,
                             sqVis, sqVisErr, uvCoord,
                             bandFlag, pistonOPDArray,
                             sigmaPistonArray, diffVisAmp,
                             diffVisAmpErr, diffVisPhase,
                             diffVisPhaseErr, vis3Amp,
                             vis3AmpErr, vis3Phase, vis3PhaseErr,
                             cpxVis, cpxVisErr, visCovRI,
                             uv1Coord, uv2Coord, fluxSumPiPj,
                             fluxRatPiPj, fluxProdPiPj,
                             inputOiFile=inputRawOiFile))
        {
            yocoGuiErrorBox, "reading "+inputRawOiFile;//+inputSciFiles;
        }

        selection = amdlibLoadFrameSelection(inputSelFile=inputSelFile);


        OpD = pistonOPDArray;
        nbFrames = dimsof(sqVis)(4);
        nbWlen = dimsof(sqVis)(3);
        nbBases = dimsof(sqVis)(2);

        MJD = (*inoiVis.table).dateObsMJD;
        targetId = (*inoiVis.table).targetId;
        time = (*inoiVis.table).time;
        DIT = (*inoiVis.table).expTime;
        stations = (*inoiVis.table).stationIndex;

        if (nbBases == 3)
        {
            stations3 = (*inoiVis3.table).stationIndex;
            U1 = transpose(array(uv1Coord.u,1));
            V1 = transpose(array(uv1Coord.v,1));
            U2 = transpose(array(uv2Coord.u,1));
            V2 = transpose(array(uv2Coord.v,1));
        }

        U = uvCoord.u;
        V = uvCoord.v;

        R = wlen / (bandWidth+(bandWidth==0)) / 2.0;
        // write,"Resolution",R;
        Lc = wlen * R;
        // write,"Lc",Lc
        sigma1 = Lc / (sqrt(2 * log(2))) / 4.0;

        /* Treatment itself */
        if(is_void(pistonOffsets))
            pistonOffsets = array(0.0,nbBases,nbWlen);

        if(useDiffPhi==1)
        {
            diffPhasePistons = computeDiffPhasePiston(inputOiFile=inputAvgOiFile);
            pistonOffsets = pistonOffsets + diffPhasePistons;
        }

        katt = array(1.0,nbBases,nbWlen);
        if(!is_void(p2vmFileName))
        {
            p2vmPiston = computeP2vmPiston(pwlen, p2vmFile=p2vmFileName);
            pidx = where((pwlen<=max(wlen)+avg(bandWidth)/10)&
                         (pwlen>=min(wlen)-avg(bandWidth)/10));
            kpst = p2vmPiston(,pidx);
            pistonOffsets = pistonOffsets - kpst;

            for(l=1;l<=nbBases;l++)
            {
                for(k=1;k<=nbWlen;k++)
                {
                    katt(l,k) = sigma1(k) * sqrt(2 * pi) *
                        yocoMathGauss(kpst(l,k), [sigma1(k),0.0]);
                }
            }
        }

        pouet()


    
            kBand = where(bandFlag)(1);
        pst = (OpD(..,kBand)(,-,) * yocoAstroSI.nm/yocoAstroSI.m + pistonOffsets(,,-));
        selection = selection(..,kBand);

        attenuationFact = array(0.0,dimsof(sqVis));
        for(l=1;l<=nbBases;l++)
        {
            for(k=1;k<=nbWlen;k++)
            {
                attenuationFact(l,k,) = katt(k,l) * sigma1(k) * sqrt(2 * pi) *
                    yocoMathGauss(pst(l,k,), [sigma1(k),0.0]);
            }
        }

        att = array(1.0,nbBases,nbWlen);
        for(l=1;l<=nbBases;l++)
        {
            for(k=1;k<=nbWlen;k++)
            {
                selFrames= where(selection(l,));
                att(l,k) = sum((attenuationFact*fluxProdPiPj)(l,k,selFrames)) /
                    sum(fluxProdPiPj(l,k,selFrames));
            }
        }

        /* DUMBLY READING TWICE THE FILE */
        _amdlibReadVis, inavgoiWave, inavgamberPhot, inavgoiVis, inavgoiVis2, inavgoiVis3,
            inavgamberOpd, inavgamberInsCfg, inavgoiArray, inavgoiTarget,inavgSpectrum,
            inputOiFile=inputAvgOiFile;

        if(!amdlibLoadOiData(avgwlen, avgbandWidth, avgtime, avgfringeSNR,
                             avgsqVis, avgsqVisErr, avguvCoord,
                             avgbandFlag, avgpistonOPDArray,
                             avgsigmaPistonArray, avgdiffVisAmp,
                             avgdiffVisAmpErr, avgdiffVisPhase,
                             avgdiffVisPhaseErr, avgvis3Amp,
                             avgvis3AmpErr, avgvis3Phase, avgvis3PhaseErr,
                             avgcpxVis, avgcpxVisErr, avgvisCovRI,
                             avguv1Coord, avguv2Coord, avgfluxSumPiPj,
                             avgfluxRatPiPj, avgfluxProdPiPj,
                             inputOiFile=inputAvgOiFile))
        {
            yocoGuiErrorBox, "reading "+inputAvgOiFile;//+inputSciFiles;
        }

        avgnbFrames = dimsof(sqVis)(4);
        avgnbWlen = dimsof(sqVis)(3);
        avgnbBases = dimsof(sqVis)(2);

        avgMJD = (*inavgoiVis.table).dateObsMJD;
        avgtargetId = (*inavgoiVis.table).targetId;
        avgtime = (*inavgoiVis.table).time;
        avgDIT = (*inavgoiVis.table).expTime;
        avgstations = (*inavgoiVis.table).stationIndex;

        if (avgnbBases == 3)
        {
            avgstations3 = (*inavgoiVis3.table).stationIndex;
            avgU1 = transpose(array(avguv1Coord.u,1));
            avgV1 = transpose(array(avguv1Coord.v,1));
            avgU2 = transpose(array(avguv2Coord.u,1));
            avgV2 = transpose(array(avguv2Coord.v,1));
        }

        avgU = avguvCoord.u;
        avgV = avguvCoord.v;

        /* Get back structures from the yorick arrays. */
        if(avgnbBases == 3)
        {
            avgnbClos = 1;
            avgnbTels = 3;
        }
        else if(avgnbBases == 1)
        {
            avgnbClos = 0;
            avgnbTels = 2;
        }

        att = array(att,1);

        amdlibSetOiVis2, newVis2, avgsqVis/att, avgsqVisErr,
            avgtargetId, avgtime, avgMJD, avgDIT, avgU, avgV, avgstations;

        yocoLogInfo, "writing file "+outputAvgOiFile;
        status = amdlibWriteOiFile(outputAvgOiFile,
                                   pointer(nil), &inavgoiArray, 
                                   &inavgoiTarget, &inavgoiWave,
                                   &inavgamberPhot, &inavgoiVis, 
                                   &newVis2, &inavgoiVis3, &inavgamberOpd,
                                   &inavgSpectrum);

        if (status == amdlibFAILURE)
        {
            yocoGuiErrorBox,"Could not save result file\n";
        }

        // dump bin tables
        yocoLogInfo, "dump tables";
        _amdlibDumpAllKeywords, 1, inputAvgOiFile, outputAvgOiFile;

        // Write additional column
        outFh = cfitsio_open(outputAvgOiFile, "a", overwrite=1);
        hduz = cfitsio_list(outFh);
        cfitsio_goto_hdu,outFh,"OI_VIS2";
        cfitsio_insert_cols,outFh,1,"ATT_FACT",pr1(dimsof(att)(3))+"D";
        cfitsio_write_col,outFh,1,transpose(att);
        cfitsio_close,outFh;


        // Rezip file if needed
        if (recompressFile=="yes")
        {
            yocoLogInfo,"Recompressing file";
            if (yocoSystem("gzip -f "+outputAvgOiFile) != 0)
            {
                return 0;
            }
        }

        return 1;
    }

/***************************************************************************/

func amdlibComputeAtt(inputOiFile=, p2vmFileName=, avgOiFile=, outputFileName=, pistonOffsets=)
    /* DOCUMENT amdlibComputeAtt(inputOiFile=, p2vmFileName=, avgOiFile=, outputFileName=, pistonOffsets=)

       DESCRIPTION
       Compute the visibility loss in a file

       SEE ALSO 
    */
{
    message = "Please choose RAW OI file"; 
    if (amdlibFileChooser(message, inputOiFile) == 0)
    {
        return 0;
    }

    // Guess what is the selection file and load it
    // selFile = yocoStrReplace(inputOiFile,"RAW","SEL");
    // selData = amdlibLoadFrameSelection(inputSelFile=selFile);

    // Guess what is the averaged file and load it
    avgOiFile = yocoStrReplace(inputOiFile,"RAW","AVG");
    message = "Please choose AVG OI file"; 
    if (amdlibFileChooser(message, avgOiFile) == 0)
    {
        return 0;
    }


    // Guess what is the associated P2VM and load it
    yocoFileSplitName, inputOiFile, dir, name, ext;
    amdlibGetLogName,dir,logName;
    amdlibReadLog, dir+logName, valuesTab, titles, outputLog;

    tabFiles = valuesTab(where(titles=="fileName"), );
    tabDates = valuesTab(where(titles=="date_obs"), );
    tabNr = valuesTab(where(titles=="calib_nr"), );
    tabTypes = valuesTab(where(titles=="obs_catg"), );
    OBJTypes = valuesTab(where(titles=="obs_type"), );
    tabIds = valuesTab(where(titles=="P2VM_ID"), );
    tabOBsNr = valuesTab(where(titles=="OB_nr"), );
    tabTplNr = valuesTab(where(titles=="Template_nr"), );
    tabDit = valuesTab(where(titles=="DIT"), );
    proCatg = valuesTab(where(titles=="pro_catg"), );

    //get P2VM indexes
    P2vmIndexes = where(proCatg == "P2VM_REDUCED");
    P2VMFiles = dir + tabFiles(, P2vmIndexes);
    P2VMIds = tabIds(, P2vmIndexes);

    // Get P2vm which index match the file
    fileIndex = where(strmatch(tabFiles,name+ext));
    fileId = tabIds(fileIndex);
    theP2VMIdx = where(P2VMIds == fileId);
    p2vmFileName = P2VMFiles(theP2VMIdx);

    if(numberof(p2vmFileName)!=1)
        readableError;
    else
        p2vmFileName = p2vmFileName(1);

    message = "Please choose P2VM file"; 
    if (amdlibFileChooser(message, p2vmFileName) == 0)
    {
        return 0;
    }

    // Read the science star OI_VIS file
    yocoLogInfo, "reading file "+ inputOiFile;

    if(is_void(outputFileName))
        outputFileName = yocoStrReplace(inputOiFile,"RAW","COR");

    // Test if gzipped
    yocoFileSplitName,outputFileName,dir,name,ext;
    if (strmatch(ext,".gz"))
    {
        ext = sum(yocoStrSplit(ext,".gz"));
        recompressFile = "yes";
    }
    outputFileName = dir+name+ext;


    /* DUMBLY READING TWICE THE FILE */
    _amdlibReadVis, inoiWave, inamberPhot, inoiVis, inoiVis2, inoiVis3,
        inamberOpd, inamberInsCfg, inoiArray, inoiTarget,
        inputOiFile=inputOiFile;

    if(!amdlibLoadOiData(wlen, bandWidth, time, fringeSNR,
                         sqVis, sqVisErr, uvCoord,
                         bandFlag, pistonOPDArray,
                         sigmaPistonArray, diffVisAmp,
                         diffVisAmpErr, diffVisPhase,
                         diffVisPhaseErr, vis3Amp,
                         vis3AmpErr, vis3Phase, vis3PhaseErr,
                         cpxVis, cpxVisErr, visCovRI,
                         uv1Coord, uv2Coord, fluxSumPiPj,
                         fluxRatPiPj, fluxProdPiPj,
                         inputOiFile=inputOiFile))
    {
        yocoGuiErrorBox, "reading "+inputOiFile;//+inputSciFiles;
    }

    OpD = pistonOPDArray;
    nbFrames = dimsof(sqVis)(4);
    nbWlen = dimsof(sqVis)(3);
    nbBases = dimsof(sqVis)(2);

    MJD = (*inoiVis.table).dateObsMJD;
    targetId = (*inoiVis.table).targetId;
    time = (*inoiVis.table).time;
    DIT = (*inoiVis.table).expTime;
    stations = (*inoiVis.table).stationIndex;

    if (nbBases == 3)
    {
        stations3 = (*inoiVis3.table).stationIndex;
        U1 = transpose(array(uv1Coord.u,1));
        V1 = transpose(array(uv1Coord.v,1));
        U2 = transpose(array(uv2Coord.u,1));
        V2 = transpose(array(uv2Coord.v,1));
    }

    U = uvCoord.u;
    V = uvCoord.v;


    R = wlen / bandWidth / 2.0;
    Lc = wlen * R;
    sigma1 = Lc / (sqrt(2 * log(2))) / 4//2.75;

        /* Treatment itself */
        if(is_void(pistonOffsets))
            pistonOffsets = array(0.0,nbBases,nbWlen);

    if(!is_void(avgOiFile))
    {
        diffPhasePistons = computeDiffPhasePiston(inputOiFile=avgOiFile);
        pistonOffsets = pistonOffsets + diffPhasePistons;
    }

    if(!is_void(p2vmFileName))
    {
        p2vmPiston = computeP2vmPiston(pwlen, p2vmFile=p2vmFileName);
        pidx = where((pwlen<=max(wlen)+avg(bandWidth)/4)&
                     (pwlen>=min(wlen)-avg(bandWidth)/4));
        kpst = p2vmPiston(,pidx);
        pistonOffsets = pistonOffsets - kpst;

        write,"piston P2vm",kpst(,avg);

        katt = array(0.0,nbBases,nbWlen);
        for(l=1;l<=nbBases;l++)
        {
            for(k=1;k<=nbWlen;k++)
            {
                katt(l,k) = sigma1(k) * sqrt(2 * pi) *
                    yocoMathGauss(kpst(l,k), [sigma1(k),0.0]);
            }
        }
        write,"attenuation P2vm",katt(,avg);

    }


    kBand = where(bandFlag)(1);
    pst = OpD(,,kBand)(,-,) * yocoAstroSI.nm/yocoAstroSI.m + pistonOffsets(,,-);

    attenuationFact = array(0.0,dimsof(sqVis));
    for(l=1;l<=nbBases;l++)
    {
        for(k=1;k<=nbWlen;k++)
        {
            attenuationFact(l,k,) = katt(l,k) * sigma1(k) * sqrt(2 * pi) *
                yocoMathGauss(pst(l,k,), [sigma1(k),0.0]);
        }
    } 

    // winkill;

    for(l=1;l<=nbBases;l++)
    {
        yocoNmCreate,l,nbWlen,width=650,height=850,wait=1;
        sel = where(fringeSNR(l,,kBand)>1)
            for(k=1;k<=nbWlen;k++)
            {
                x = OpD(l,sel,kBand);
                idx = sort(x);
                x = x(idx)
                y = sqVis(l,k,sel)(idx);
                plsys,k;
                plg,y,x,type="none",marker='\1';
                pldj,0,0,0,1,type="dash";
                // require,"lmfit.i";
                // a = [0.7,4e4,0];
                // lmfit,fitaGauss,x,a,y;
                // plg,fitaGauss(x,a),x,color="green";
                // pldj,a(3),0,a(3),1,color="green";
                pldj,-pistonOffsets(l,k)*yocoAstroSI.m/yocoAstroSI.nm,
                0,-pistonOffsets(l,k)*yocoAstroSI.m/yocoAstroSI.nm,
                1,color="red";
                plg,attenuationFact(l,k,sel)(idx),x,color="red";
                plg,attenuationFact(l,k,sel)(idx)/2,x,color="red";
                limits,-Lc(k)*yocoAstroSI.m/yocoAstroSI.nm/2,
                Lc(k)*yocoAstroSI.m/yocoAstroSI.nm/2,0,1;

            }
    }

    fsd()

        kLambda = 10;
    //winkill;
    window,4,style=HOME+"/Yorick/3horiz.gs",height=500,width=500;
    window,5,style=HOME+"/Yorick/3horiz.gs",height=500,width=500;
    window,6,style=HOME+"/Yorick/3horiz.gs",height=500,width=500;
    for(kBase = 1; kBase<=nbBases;kBase++)
    {

        // sel = where(selData(kBase,,kBand));

        a=sqVis(kBase,kLambda,)(avg);
        r=sqVis(kBase,kLambda,)(rms);
        a2=(sqVis(kBase,kLambda,) / attenuationFact(kBase,kLambda,))(avg);
        r2=(sqVis(kBase,kLambda,) / attenuationFact(kBase,kLambda,))(rms);
        write,a,r,r/a*100;
        write,a2,r2,r2/a2*100;

        window,1;
        plsys,kBase;
        plg,sqVis(kBase,kLambda,),time(1,)-min(time(1,));
        plg,attenuationFact(kBase,kLambda,)^2,time(1,)-min(time(1,)),color="red";

        limits,-0.05,1.05,-0.05,1.05;

        window,2;
        plsys,kBase;
        plg,sqVis(kBase,kLambda,) / attenuationFact(kBase,kLambda,)^2,time(1,)-min(time(1,));
        limits,-0.05,1.05,-0.05,1.05;

        window,3;
        plsys,kBase;
        plg,sqVis(kBase,kLambda,), attenuationFact(kBase,kLambda,)^2,type="none",marker='\2';


        pldj,-0.1,-0.1,1.1,1.1;
        limits,-0.05,1.05,-0.05,1.05;
        if(kBase==2)
            xytitles,,"Squared visibility";
        if(kBase==3)
            xytitles,"Attenuation from OPD";
    }
    kjshdf()

        window,3;
    hcps,HOME+name+"_OPD_Att.ps";

    // Compute the attenuation from the jitter effect !
}

/***************************************************************************/

func amdlibDumpBinTable(hdu, inputFitsFile, outputFitsFile)
    /* DOCUMENT amdlibDumpBinTable(hdu, inputFitsFile, outputFitsFile)

       DESCRIPTION
       This function dumps a binary table from a source file to a target file
       
       PARAMETERS
       - hdu           : Number of hdu or string of the hdu name
       - inputFitsFile : The source file to read
       - outputFitsFile: The destination file to append or write
       
       SEE ALSO 
    */
{
    local k;
    
    // open source file
    inFh = cfitsio_open(inputFitsFile, "r");
    
    // Test if gzipped
    yocoFileSplitName,outputFitsFile,dir,name,ext;
    if (strmatch(ext,".gz"))
    {
        if (yocoSystem("gunzip -f "+outputFitsFile) != 0)
        {
            return 0;
        }
        ext = sum(yocoStrSplit(ext,".gz"));
        recompressFile = "yes";
    }
    outputFitsFile = dir+name+ext;

    // open destination file
    outFh = cfitsio_open(outputFitsFile, "a", overwrite=1);

    hduz = cfitsio_list(inFh);
    for(k=1;k<=numberof(hdu);k++)
    {
        // go to a HDU
        cfitsio_goto_hdu,inFh,hdu(k);
        if(typeof(hdu(k))=="int" || typeof(hdu(k))=="long")
            hduName = hduz(hdu(k));
        else
            hduName = hdu(k);

        table = cfitsio_read_bintable(inFh,titles); // read the binary table
        cfitsio_add_bintable, outFh, table, titles,,hduName; // Write bin table

        _amdlibDumpAllKeywords,hdu(k), inFh, outFh;
    }

    // close files
    cfitsio_close_file, inFh;
    cfitsio_close_file, outFh;

    // Rezip file if needed
    if (recompressFile=="yes")
    {
        yocoLogInfo,"Recompressing file";
        if (yocoSystem("gzip -f "+outputFitsFile) != 0)
        {
            return 0;
        }
    }
}

/***************************************************************************/

func getJitter(CalDiam=, fileName=)
    /* DOCUMENT getJitter(CalDiam=, fileName=)

       DESCRIPTION
       calculate the Jitter mean taking into account the diameter of
       the observed star.

       PARAMETERS
       - CalDiam : 
       - fileName: 
       If no fileName is put the software will ask on.
       need of a CalDiam in mas!

       RESULTS

       CAUTIONS
       need of a CalDiam in mas!

       EXAMPLES

       SEE ALSO:
    */
{
    if(is_void(fileName))
    {
        amdlibFileChooser("Please choose an OiRaw data of a calibrator",
                          fileName);
    }
    amdlibLoadOiData(wlen, bandWidth, time, fringeSNR, sqVis, sqVisErr,
                     uvCoord, bandFlag, pistonOPDArray, sigmaPistonArray,
                     diffVisAmp, diffVisAmpErr, diffVisPhase, diffVisPhaseErr,
                     vis3Amp, vis3AmpErr, vis3Phase, vis3PhaseErr,
                     cpxVis, cpxVisErr, visCovRI, uv1Coord,
                     uv2Coord, fluxSumPiPj, fluxRatPiPj, fluxProdPiPj,
                     spectrum, spectrumErr,
                     fileName=fileName);

    vis2Cal = (amdlibComputeVisUniformDisk(uvCoord.u, uvCoord.v,
                                           CalDiam*mas2rad,
                                           wlen)^2); 


    nbFrames = dimsof(sqVis)(4);
    nbBases = dimsof(sqVis)(2);
    sigmaG = array(0.0,nbBases,nbFrames);

    for(k=1;k<=nbFrames;k++)
    {
        InstantaneousTransFunc = sqVis(,,k)/vis2Cal;
        for(l=1;l<=nbBases;l++)
        {
            idxPlus = where((InstantaneousTransFunc(l,)>0)&
                            (InstantaneousTransFunc(l,)<1));
            if (numberof(idxPlus)==0)
            {
                sigmaG(l,k)=0;
            }
            else
            {
                bloup = InstantaneousTransFunc(l,idxPlus);

                sigmaG(l,k)=(sqrt(-log10(bloup)*wlen(idxPlus)^2/(4*pi^2)))(avg);
            }
        }
    }

    //winkill;
    //yocoPlotHistogramme,sigmaG(1,);


    // f= open(fileName+"jitter_corrected.txt", "w");
    //write,f,"BASE1","BASE2","BASE3";
    //write,f,sigmaG(1,),sigmaG(2,),sigmaG(3,);
    //close,f;
    return sigmaG;
}

/***************************************************************************/

func getAllJitter(void)
    /* DOCUMENT getAllJitter

       DESCRIPTION
       Call the getJitter function on every data

       PARAMETERS
       - void : 

       RESULTS


       CAUTIONS
       under construction... Do not take account J and H band!!!
       As the jitter shouldn't be chromatique K band should be enought

       EXAMPLES

       SEE ALSO:
    */
{
    // read the data on the calibrator
    calibzFile=[];
    amdlibFileChooser("Please choose a calibration file", calibzFile);
    vals = yocoFileReadAscii(calibzFile);
    calNames = vals(1,2:);
    calDiamz = yocoStr2Double(vals(3,2:));
    calDiamzErr = yocoStr2Double(vals(4,2:));

    //define the directory
    inputDir=[];
    amdlibFileChooser,"Choose the product directory (OI_DATA)", inputDir;
    File=lsdir(inputDir);
    test=File(where(strmatch(File,"OIDATA_RAW_K.fits")));

    if(max(strmatch(File,"_LOG.txt"))==1)
    {
        logFile=File(where(strmatch(File,"_LOG.txt")));
    }
    else
    {
        amdlibCreateLog(directory=inputDir,
                        outputFile=inputDir+"PRODUCT_LOG.txt");
        logFile= inputDir+"PRODUCT_LOG.txt";
    }
    amdlibReadLog,logFile(1), valuesTab, colTitles;

    // loop on every interesting file
    nbrFile=dimsof(test)(2);
    // test of the object (science star, a calibrator star or a wrong
    // format??)
    newFileList = [];

    for(fileNumber=1;fileNumber<=nbrFile;fileNumber++)
    {
        studiedfileName=test(fileNumber);
        if(strmatch(test(fileNumber),".txt")(1)==0)
        {
            objType=valuesTab(where(strmatch(valuesTab,studiedfileName))+15);
            if(strmatch(objType,"CALIB")(1)==1)
            {
                grow, newFileList, test(fileNumber);
            }
        }
    }
    nbrCalFile=dimsof(newFileList)(2);
    jittermean=array(0.0, nbrCalFile, 3);
    jitterrms=array(0.0,nbrCalFile, 3);

    // calcul the jitter on each interesting file
    for(fileNumber=1;fileNumber<=nbrCalFile;fileNumber++)
    {
        studiedfileName=newFileList(fileNumber);
        objName=valuesTab(where(strmatch(valuesTab,studiedfileName))+8);
        CalStarDiam=calDiamz(where(strmatch(calNames, objName(1))));
        sigmaG=getJitter(CalDiam=CalStarDiam,
                         fileName=inputDir+studiedfileName);
        f= open(inputDir+studiedfileName+"jitter_corrected.txt", "w");
        write,f,"BASE1","BASE2","BASE3";
        write,f,sigmaG(1,),sigmaG(2,),sigmaG(3,);
        close,f;

        //calcul the mean and the rms
        for (iter=1;iter<=3;iter++)
        {
            jittermean(fileNumber, iter)=sigmaG(iter,)(avg);
            jitterrms(fileNumber, iter)=sigmaG(iter,)(rms);
        }
    }

    window, 1;
    plg, jittermean(,1);
    window, 2;
    plg, jitterrms(,1);

    // save the avg and rms in a file

    f= open(inputDir+"K_jittermean_rms.txt", "w");
    write, f,"FileName", "Jitter_mean", "Jitter_mean",
        "Jitter_mean","jitter_rms","jitter_rms","jitter_rms";
    write,f,"FileName ", "B1","B2", "B3","B1", "B2","B3";
    for(i=0;i<=dimsof(newFileList)(2);i++)
    {
        write,f, newFileList(i), pr1(jittermean(i,1)),
            pr1(jittermean(i,2)), pr1(jittermean(i,3)), pr1(jitterrms(i,1)),
            pr1(jitterrms(i,2)), pr1(jitterrms(i,3));
    }
    close,f;
} 

/***************************************************************************/

func amdlibGetJitterByInterspectrum(inputOiFile=, outputFileName=, pistonOffsets=)
    /* DOCUMENT amdlibGetJitterByInterspectrum(inputOiFile=, outputFileName=, pistonOffsets=)

       DESCRIPTION
       Calibrate an OI_DATA file from a science star with another from
       a calibrator

       PARAMETERS
       - inputOiFile   : 
       - outputFileName: name of the file where resulting IO-FITS file 
       - pistonOffsets : 
       is stored

       RETURN VALUE
       a CAL_OI_DATA file.

       CAUTION
       The case of an over resolved calibration star with non-zero closure phase 
       is NOT taken into account


       SEE ALSO 
    */
{
    message = "Please choose your Raw data night directory"; 
    if (amdlibFileChooser(message, inputOiFile) == 0)
    {
        return 0;
    }

    yocoFileSplitName,inputOiFile,dir,fil,ext;

    // Read the science star OI_VIS file
    write, "reading file "+ inputOiFile;

    if(is_void(outputFileName))
        outputFileName = yocoStrReplace(inputOiFile,"RAW","RAWCORR");


    /* DUMBLY READING TWICE THE FILE */
    _amdlibReadVis,oiWave, amberPhot, oiVis, oiVis2, oiVis3, amberOpd, amberInsCfg, oiArray, oiTarget, inputOiFile=inputOiFile;

    if(!amdlibLoadOiData(wlen, bandWidth, time, fringeSNR,
                         sqVis, sqVisErr, uvCoord,
                         bandFlag, pistonOPDArray,
                         sigmaPistonArray, diffVisAmp,
                         diffVisAmpErr, diffVisPhase,
                         diffVisPhaseErr, vis3Amp,
                         vis3AmpErr, vis3Phase, vis3PhaseErr,
                         cpxVis, cpxVisErr, visCovRI,
                         uv1Coord, uv2Coord, fluxSumPiPj,
                         fluxRatPiPj, fluxProdPiPj,
                         inputOiFile=inputOiFile))

        yocoGuiErrorBox, "reading "+inputOiFile;//+inputSciFiles;

    nbFramesOut = dimsof(sqVis)(4)-1;
    nbWlen = dimsof(sqVis)(3);
    nbBases = dimsof(sqVis)(2);

    pistonOPDArray = pistonOPDArray*nano2unit;

    cf = sqVis * fluxProdPiPj;
    cxf = cpxVis * signedSqrt(fluxProdPiPj);

    sqVisOut = (cf(,,:-1:) + cf(,,2::)) / (fluxProdPiPj(,,:-1:) + fluxProdPiPj(,,2::));
    pistonOut = (pistonOPDArray(,:-1:,) + pistonOPDArray(,2::,)) / 2.0;

    timeOut = (time(1,:-1)+time(1,2:))/2;

    diffPist = pistonOPDArray(,:-1:,) - pistonOPDArray(,2::,);

    frameDif = cxf(,,:-1:) * conj(cxf(,,2::));

    pistW = array(0.0,nbBases,nbFramesOut);
    for(k=1;k<=nbFramesOut;k++)
    {
        for(j=1;j<=nbBases;j++)
        {
            //pistW(j,k) = ComputePistonPhase(frameDif(j,,k),wlen);
            pistW(j,k) = ComputePistonAIPS(frameDif(j,,k),wlen);
        }
    }

    phaseDif = atan(frameDif.im,frameDif.re);

    kLambda = 10;
    lambda = wlen(kLambda);

    wkll;
    window,1,style=HOME+"/Yorick/3horiz.gs",height=500,width=500;
    for(kBase = 1; kBase<=nbBases; kBase++)
    {
        plsys,kBase;
        plg,sqVisOut(kBase,kLambda,),(pistW(kBase,)),type="none",marker='\2';
        limits,,,0,1;
    }


    lcj90()

        dpist = abs(phaseDif)*lambda / (2*pi);
    window,2,style=HOME+"/Yorick/3horiz.gs",height=500,width=500;
    for(kBase = 1; kBase<=nbBases; kBase++)
    {
        plsys,kBase;
        plg,sqVisOut(kBase,kLambda,),dpist(kBase,kLambda,),type="none",marker='\2';
    }

    window,3,style=HOME+"/Yorick/3horiz.gs",height=500,width=500;
    window,4,style=HOME+"/Yorick/3horiz.gs",height=500,width=500;
    window,5,style=HOME+"/Yorick/3horiz.gs",height=500,width=500;
    for(kBase = 1; kBase<=nbBases; kBase++)
    {
        magicFact = 0.1;
        att1 = exp(- (2 * pi^2 * (magicFact*diffPist(kBase,,3))^2) / (lambda^2));
        att2 = exp(- (2 * pi^2 * (magicFact*dpist(kBase,kLambda,))^2) / (lambda^2));

        window,3;
        plsys,kBase;
        plg,sqVisOut(kBase,kLambda,),att1^2,type="none",marker='\2';
        pldj,-0.1,-0.1,1.1,1.1;
        limits,-0.05,1.05,-0.05,1.05;
        if(kBase==2)
            xytitles,,"Squared visibility";
        if(kBase==3)
            xytitles,"Attenuation from jitter";

        frg1 = fringeSNR(kBase,:-1:,3);
        frg2 = fringeSNR(kBase,2::,3);
        thr = 3;
        sel = where((frg1>thr)&(frg2>thr));
        if(numberof(sel)!=0)
        {
            plg,sqVisOut(kBase,kLambda,sel), att1(sel)^2,type="none",marker='\2',color="red";
        }

        window,4;
        plsys,kBase;
        plg,sqVisOut(kBase,kLambda,),att2^2,type="none",marker='\2';
        pldj,-0.1,-0.1,1.1,1.1;
        limits,-0.05,1.05,-0.05,1.05;

        window,5;
        plsys,kBase;
        //plg,sqVisOut(kBase,kLambda,) / att1^2,timeOut-min(timeOut);
        plg,sqVisOut(kBase,kLambda,),timeOut-min(timeOut),color="red";
        limits,,,-0.05,1.05;

        write,(sqVisOut(kBase,kLambda,sel) / att1(sel)^2)(avg);
    }

    window,3;
    hcps,"~/"+fil+"jitter1.ps";

    window,4;
    hcps,"~/"+fil+"jitter2.ps";

    return 0;
    feo();


    /* Get back structures from the yorick arrays. */

    amdlibSetWave, newWave,
        wlen * yocoAstroSI.m/yocoAstroSI.nm,
        bandWidth * yocoAstroSI.m/yocoAstroSI.nm;

    amdlibSetOiVis, newVis, cpxVis, cpxVisErr, visCovRI, 
        diffVisAmp, diffVisAmpErr, 
        diffVisPhase, diffVisPhaseErr,
        fringeSNR, targetId, time, MJD, DIT, U, V, stations;

    amdlibSetOiVis2, newVis2, sqVis_corr, sqVisErr,
        targetId, time, MJD, DIT, U, V, stations;

    if (nbBases == 3)
    {
        amdlibSetOiVis3, newTriple3,
            vis3Amp, vis3AmpErr, vis3Phase, vis3PhaseErr,
            targetId, time, MJD, DIT, U1, U2, V1, V2, stations3;
    }

    write, "writing file "+outputFileName;
    status = amdlibWriteOiFile(outputFileName,
                               pointer(nil), &oiArray, 
                               &oiTarget, &newWave, &amberPhot, &newVis, 
                               &newVis2, &newTriple3, &amberOpd, pointer(nil));


    if (status == amdlibFAILURE)
    {
        yocoGuiErrorBox,"Could not save result file\n";
    }

    // dump bin tables
    write, "dump bin table";
    amdlibDumpAllKeywords, 1, inputOiFile, outputFileName;
    amdlibDumpBinTable, ["AMBER_DATA","AMBER_SPECTRUM"],
        inputOiFile, outputFileName;

    return 1;
}

/***************************************************************************/

func amdlibConvertScienceToCalib(nightDir=)
    /* DOCUMENT amdlibConvertScienceToCalib(nightDir=)

       !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
       !!! USE THIS SCRIPT WITH CAUTION !!!
       !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

       DESCRIPTION
       Change the fits files header in the case you have data with mismatched
       types for the stars (eg. all stars are tagged as "science"). This is a
       wrap-over of a problem encountered in the beginning of AMBER operation,
       so you are unlikely needing to use this script.

       PARAMETERS
       - nightDir: The night directory. Can be raw data or OI DATA files
       directory. It is strongly advised to operate only on OI DATA files
       because it can corrupt your data!

       RESULTS
       updated headers, and log file.

       CAUTIONS
       !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
       !!! USE THIS SCRIPT WITH CAUTION !!!
       !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

       EXAMPLES
       > amdlibConvertScienceToCalib
       choose a product directory and follow instructions

       SEE ALSO:
    */
{
    /* Select directory containing files to be treated */
    message = "Choose the product directory (OI_DATA)"; 
    if (amdlibFileChooser(message, nightDir) == 0)
    {
        return 0;
    }

    if(is_void(verbose))
        verbose=1;

    // Read log file & get usefule keywords
    logFile = [];
    if(is_void(logFile))
    {
        amdlibGetLogName, nightDir, logFile;
        logFile = nightDir + string(logFile);
    }
    logTable = titles = [];

    amdlibReadLog, logFile, logTable, titles;

    tabFiles = logTable(where(titles=="fileName"), );
    tabDates = logTable(where(titles=="date_obs"), );
    tabNr    = logTable(where(titles=="calib_nr"), );
    proTypes = logTable(where(titles=="pro_catg"), );
    tabTypes = logTable(where(titles=="obs_catg"), );
    OBJTypes = logTable(where(titles=="obs_type"), );
    tabIds   = logTable(where(titles=="P2VM_ID"), );
    tabOBsNr = logTable(where(titles=="OB_nr"), );
    tabTplNr = logTable(where(titles=="Template_nr"), );
    tabNames = logTable(where(titles=="object_name"), );

    calNames = yocoStrRemoveMultiple(tabNames);
    calNames = calNames(where(calNames!="-"));

    nCals = numberof(calNames);
    len   = max(strlen(calNames));

    // Select only OBJECT files
    rightOnes = where(OBJTypes=="OBJECT");

    // Define colors according to the file type ("SCIENCE" or "CALIB")
    kolorz = array("",nCals);
    for(k=1;k<=nCals;k++)
    {
        types = tabTypes(rightOnes)(where(tabNames(rightOnes)==calNames(k)));

        if(is_void(types))
            kolorz(k)="undefined";
        else
        {
            types = types(where(types!="-"));

            if(types(1) == "SCIENCE")
                kolorz(k)="science";
            else if(types(1) == "CALIB")
                kolorz(k)="calib";
        }
    }

    // The actual colors are defined there
    calib     = char(int([255,0,0]));
    science   = char(int([0,255,0]));
    undefined = char(int([0,0,0]));

    // Buildup buttons to plot a list of star names with colors
    buttons = array(Button(),nCals);
    for(k=1;k<=nCals;k++)
    {
        buttons(k).y    = 0.85-(k-1)/20.0;
        buttons(k).dx   = 0.006*len;
        buttons(k).x    = 0.4;
        buttons(k).dy   = 0.02;
        buttons(k).text = yocoStrReplace(calNames(k));
    }
    grow,buttons,Button(x=0.4,
                        y= 0.85-(k)/20.0,
                        dx=0.02,dy=0.02,text="OK");

    pixLetWidth  = 12;
    pixLetHeight = 45;

    // Plot interface
    winkill,1;
    window,1,width=600,height=800;
    pltitle,
        "Click multiple times on star name to change type of files, then click \"OK\"\n"+
        "Green = SCIENCE \n"+
        " Red = CALIB \n"+
        "Black = undefined \n";
    stop=0;
    
    while(stop==0)
    {
        plsys,0;

        for(k=1;k<=nCals;k++)
        {
            if(kolorz(k)=="calib")
                useColor = calib;
            else if(kolorz(k)=="science")
                useColor = science;
            else if(kolorz(k)=="undefined")
                useColor = undefined;
            else
                useColor=[];
            Y = [buttons(k).y+buttons(k).dy,
                 buttons(k).y+buttons(k).dy,
                 buttons(k).y-buttons(k).dy,
                 buttons(k).y-buttons(k).dy];
            X = [buttons(k).x-buttons(k).dx,
                 buttons(k).x+buttons(k).dx,
                 buttons(k).x+buttons(k).dx,
                 buttons(k).x-buttons(k).dx];
            // Plot the coloured rectangle
            plfp,array(useColor,1),Y,X,4;
        }
        // Plot the buttons
        button_plot,buttons;

        // Mouse interaction
        result= mouse(0, 0);
        x = result(1);
        y = result(2);

        for(k=1;k<=nCals+1;k++)
        {
            // Test buttons to change eventually the corresponding color
            if(button_test(buttons(k), x, y))
            {
                if(buttons(k).text=="OK")
                {
                    stop=1;
                    break;
                }
                if(kolorz(k)=="calib")
                    kolorz(k)="science";
                else if(kolorz(k)=="science")
                    kolorz(k)="undefined";
                else if(kolorz(k)=="undefined")
                    kolorz(k)="calib";
            }
        }
    }

    // Kill interface window
    winkill,1;

    // Loop on files to change type
    filesToChange = tabFiles(rightOnes);
    tabToChange   = tabNames(rightOnes);
    nFiles        = numberof(filesToChange);
    for(k=1;k<=nFiles;k++)
    {
        write,filesToChange(k);
        yocoFileSplitName,filesToChange(k),dir,fil,ext;

        inFh = cfitsio_open(nightDir+filesToChange(k), "r");
        cfitsio_goto_hdu,inFh,1;
        value = cfitsio_get(inFh, ["HIERARCH ESO DPR CATG",
                                   "HIERARCH ESO PRO CATG"]);
        cfitsio_close,inFh;

        Nkz = numberof(value);
        KZ = ["HIERARCH ESO DPR CATG",
              "HIERARCH ESO PRO CATG"];
        KZ = KZ(1:Nkz);

        idx = where(tabToChange(k) == calNames);
        if(numberof(idx)!=0)
        {
            idx = idx(1);
        }
        else
            continue;
        col = kolorz(idx);

        for(l=1;l<=numberof(value);l++)
        {
            if(col=="calib")
            {
                key = "CALIB";
            }
            else if(col=="science")
            {
                key = "SCIENCE";
            }
            else if(col=="undefined")
            {
                key = "UNDEFINED";
            }
            valz = yocoStrReplace(value,value(1),key);
        }
        for(g=1;g<=Nkz;g++)
            write,calNames(idx),value(g);
        if(anyof(valz!=value))
        {
            for(g=1;g<=Nkz;g++)
                write,"Change to:",valz(g);
            if(strmatch(ext,".gz"))
            {
                ext = yocoStrReplace(ext,".gz","");
                system,"gunzip "+nightDir+filesToChange(k);
                filesToChange(k) = dir+fil+ext;
            }
            outFh = cfitsio_open(nightDir+filesToChange(k), "a", overwrite=1);
            cfitsio_goto_hdu,outFh,1;
            cfitsio_set, outFh, KZ, valz,
                "By amdlibConvertScienceToCalib";
            cfitsio_close,outFh;
            system,"gzip "+nightDir+filesToChange(k);
        }
    }
    winkill,1;
    // Create log again since types have changed
    amdlibCreateLog,inputDir=nightDir,overwrite=1;
} 

/***************************************************************************/

func plotTfVal(nightDir=, keyword=, keyDec=)
    /* DOCUMENT plotTfVal(nightDir=, keyword=, keyDec=)

       DESCRIPTION

       PARAMETERS
       - nightDir: 
       - keyword : 
       - keyDec  : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{

    for(l=1;l<=3;l++)
    {
        window,l;
    }

    if(is_void(keyword))
        keyword="ESO ISS AMBI FWHM END";

    if(is_void(keyDec))
        keyDec=1;

    amdlibLoadTransFunc, orig, trans;

    keys = _amdlibGetKwdVals(orig.file, keyword, valuesTab);

    if(keyDec == 1)
        keys = yocoStr2Double(keys);

    nf = numberof(FILENAMES);
    for(k=1;k<=nf;k++)
    {
        for(l=1;l<=3;l++)
        {
            window,l;
            tmp = (*trans.vis2(k))(l,avg,1);
            plg,tmp,keys(k),type="none";
        }
    }
} 

/***************************************************************************/

func plotKeyFTime(nightDir=, keyword=)
    /* DOCUMENT plotKeyFTime(nightDir=, keyword=)

       DESCRIPTION

       PARAMETERS
       - nightDir: 
       - keyword : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES
       seeing : "ESO ISS AMBI FWHM END" "ESO ISS AMBI FWHM START"
       coherence time : "ESO ISS AMBI FWHM END"

       SEE ALSO
    */
{

    if(is_void(keyword))
        keyword="ESO ISS AMBI FWHM END";

    amdlibGetLogName, nightDir, logName;
    amdlibReadLog, logName, valuesTab, colTitles;

    files = valuesTab(where(colTitles=="fileName"),);
    object = valuesTab(where(colTitles=="object_name"),);
    MJD = yocoStr2Double(valuesTab(where(colTitles=="MJD-OBS"),))(1,);

    keys = yocoStr2Double(_amdlibGetKwdVals(files, keyword, valuesTab))(,1);

    idx = where(keys!=0);
    MJD = MJD(idx);
    keys = keys(idx);
    object = object(idx);

    winkill,1;
    window,1,style="amdlibLargeBox.gs",width=800,height=700;
    plg, keys, 24*(MJD-min(MJD)), type="none", marker='\2';
    overPlotStarNames,object,24*(MJD-min(MJD)),MIN=0,MAX=1.5*max(keys),yText=1.45*max(keys);
    limits,,,0;
} 


/***************************************************************************/

struct oidata
/* DOCUMENT oidata

   DESCRIPTION
   Simplified OI DATA structure, to store AMBER datas

   PARAMETERS
   string name;
   string file;
   double time;

   pointer vis2;
   pointer vis2Err;

   pointer phi;
   pointer phiErr;

   pointer clos;
   pointer closErr;

   pointer wlen;
   pointer band;

   pointer uv;
   pointer uv1;
   pointer uv2;

   SEE ALSO:
*/
{
    string name;
    string file;
    double time;

    pointer vis2;
    pointer vis2Err;

    pointer dPhi;
    pointer dPhiErr;

    pointer dVis;
    pointer dVisErr;

    pointer clos;
    pointer closErr;

    pointer spec;
    pointer specErr;

    pointer wlen;
    pointer band;

    pointer uv;
    pointer uv1;
    pointer uv2;
};

/***************************************************************************/

/***************************************************************************/

func amdlibPlotDataSeries(files=, verbose=, log=)
    /* DOCUMENT amdlibPlotDataSeries(files=, verbose=, log=)

       DESCRIPTION
       Plot a data series

       PARAMETERS
       - files  : 
       - verbose: 
       - log    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    /* Select directory containing files to be treated */
    message = "Choose the OI_DATA files to plot"; 
    if (amdlibFileChooser(message, files) == 0)
    {
        return 0;
    }

    winkill,iWin;

    for(iWin=1;iWin<=3;iWin++)
        window,iWin,width=638,height=500,style="amdlibLargeBox.gs";

    N = numberof(files);

    for(k=1;k<=N;k++)
    {
        amdlibLoadOiData,wlen,bandWidth,time,fringeSNR,sqVis,sqVisErr,
            uvCoord,bandFlag,pistonOPDArray,
            sigmaPistonArray,diffVisAmp,
            diffVisAmpErr,diffVisPhase,diffVisPhaseErr,
            vis3Amp,vis3AmpErr,vis3Phase,
            vis3PhaseErr,cpxVis,cpxVisErr,visCovRI,
            uv1Coord,uv2Coord,fluxSumPiPj,
            fluxRatPiPj,fluxProdPiPj,spectrum,spectrumErr,
            inputOiFile=files(k);

        nbBases = dimsof(sqVis)(2);
        nbWlen = dimsof(sqVis)(3);
        nbFrames = dimsof(sqVis)(4);

        if(nbFrames !=1)
            readableError,"Sorry ! more than 1 frame not accepted ! run amdlibPerformFrameSelection first !";

        for(iWin=1;iWin<=nbBases;iWin++)
        {
            window,iWin;
            v2 = sqVis(iWin,,1);
            v2Err = sqVisErr(iWin,,1);
            yocoPlotWithErrBars, v2,v2Err,wlen,color=colors(k%7);
        }
    }

    for(iWin=1;iWin<=3;iWin++)
    {
        window,iWin;
        xytitles,"Wavelength","V^2";
        limits,,,0,1;
        if(log==1)
        {
            logxy,0,1;
            limits;
            limits,,,,1;
        }

        hcps,"./DATA_SERIE_BASE_"+pr1(iWin)+".ps";
    }
}

/***************************************************************************/

func amdlibBinOiDatas(files=, verbose=, outName=)
    /* DOCUMENT amdlibBinOiDatas(files=, verbose=, outName=)

       DESCRIPTION
       "Bin" OI DATAs i.e. average them and compute additional error related to the data scatter.
       Should be used with the greatest care !

       PARAMETERS
       - files  : 
       - verbose: 
       - outName: 

       SEE ALSO
    */
{
    message = "Choose the OI_DATA files to merge"; 
    if (amdlibFileChooser(message, files) == 0)
    {
        return 0;
    }

    N = numberof(files);

    SQVIS = SQVISERR = DIFFVISAMP = DIFFVISAMPERR =
        DIFFVISPHASE = DIFFVISPHASEERR = VIS3AMP = VIS3AMPERR =
        VIS3PHASE = VIS3PHASEERR = UV1COORD = UV2COORD =
        SPECTRUM = SPECTRUMERR = UVCOORD =
        MJD = DIT = TARGETID = STATIONS = STATIONS3 = TIME = [];

    for(k=1;k<=N;k++)
    {

        /* DUMBLY READING TWICE THE FILE */
        _amdlibReadVis, inoiWave, inamberPhot, inoiVis, inoiVis2, inoiVis3,
            inamberOpd, inamberInsCfg, inoiArray, inoiTarget,
            inputOiFile=files(k);

        amdlibLoadOiData,wlen,bandWidth,time,fringeSNR,sqVis,sqVisErr,
            uvCoord,bandFlag,pistonOPDArray,
            sigmaPistonArray,diffVisAmp,
            diffVisAmpErr,diffVisPhase,diffVisPhaseErr,
            vis3Amp,vis3AmpErr,vis3Phase,
            vis3PhaseErr,cpxVis,cpxVisErr,visCovRI,
            uv1Coord,uv2Coord,fluxSumPiPj,
            fluxRatPiPj,fluxProdPiPj,spectrum,spectrumErr,
            inputOiFile=files(k);

        nbBases = dimsof(sqVis)(2);
        nbWlen = dimsof(sqVis)(3);
        nbFrames = dimsof(sqVis)(4);

        mjd = (*inoiVis.table).dateObsMJD;
        targetId = (*inoiVis.table).targetId;
        time = (*inoiVis.table).time;
        dit = (*inoiVis.table).expTime;
        stations = (*inoiVis.table).stationIndex;

        if((nbBases==1)&&(nbWlen==1))
        {
            sqVis = reform(sqVis,[3,nbFrames,1,1]);
            sqVisErr = reform(sqVisErr,[3,nbFrames,1,1]);
            uvCoord = reform(uvCoord,[2,nbFrames,1]);
            diffVisAmp = reform(diffVisAmp,[3,nbFrames,1,1]);
            diffVisAmpErr = reform(diffVisAmpErr,[3,nbFrames,1,1]);
            diffVisPhase = reform(diffVisPhase,[3,nbFrames,1,1]);
            diffVisPhaseErr = reform(diffVisPhaseErr,[3,nbFrames,1,1]);
            time = reform(time,[2,nbFrames,1]);
            mjd = reform(mjd,[2,nbFrames,1]);
            targetId = reform(targetId,[2,nbFrames,1]);
            dit = reform(dit,[2,nbFrames,1]);
            stations = reform(stations,[3,2,nbFrames,1]);

            nbBases = nbFrames;
            nbWlen = 1;
            nbFrames = 1;
        }

        if (nbBases == 3)
        {
            stations3 = (*inoiVis3.table).stationIndex;
            grow,VIS3AMP,vis3Amp;
            grow,VIS3AMPERR,vis3AmpErr;
            grow,VIS3PHASE,vis3Phase;
            grow,VIS3PHASEERR,vis3PhaseErr;
            grow,UV1COORD,uv1Coord;
            grow,UV2COORD,uv2Coord;
        }

        grow,SQVIS,sqVis;
        grow,SQVISERR,sqVisErr;
        grow,UVCOORD,uvCoord;
        grow,DIFFVISAMP,diffVisAmp;
        grow,DIFFVISAMPERR,diffVisAmpErr;
        grow,DIFFVISPHASE,diffVisPhase;
        grow,DIFFVISPHASEERR,diffVisPhaseErr;
        // grow,SPECTRUM,transpose(spectrum);
        // grow,SPECTRUMERR,transpose(spectrumErr);
        grow,MJD,mjd;
        grow,DIT,dit;
        grow,TARGETID,targetId;
        grow,STATIONS,stations;
        grow,STATIONS3,stations3;
        grow,TIME,time;

    }

    NF = dimsof(SQVIS)(0);

    DIFAMP = DIFFVISAMP(,,avg);
    DIFAMPER = DIFFVISAMPERR(,,avg);
    DIFPHI = DIFFVISPHASE(,,avg);
    DIFPHIER = DIFFVISPHASEERR(,,avg);
    TARG = TARGETID(,,1);
    TIM = TIME(,avg);
    JD = MJD(,avg);
    EXP = DIT(,sum);
    U = (UVCOORD.u)(,avg);
    V = (UVCOORD.v)(,avg);
    STAT = STATIONS(,,1);
    SQV = SQVIS(,,avg);
    SQVER = 4* (SQVIS(,,rms)/sqrt(NF)+SQVISERR(,,avg));
    if (nbBases == 3)
    {
        V3AMP = VIS3AMP(,,avg);
        V3AMPER = VIS3AMPERR(,,avg);
        V3PHI = VIS3PHASE(,,avg);
        V3PHIER = VIS3PHASEERR(,,avg);
        U1 = (UV1COORD.u)(avg);
        U2 = (UV2COORD.u)(avg);
        V1 = (UV1COORD.v)(avg);
        V2 = (UV2COORD.v)(avg);
        STAT3 = STATIONS3(,,1);
    }

    nbBases = dimsof(SQVIS)(2);
    nbWlen = dimsof(SQVIS)(3);
    nbFrames = 1;

    amdlibSetWave, newWave, wlen*yocoAstroSI.m/yocoAstroSI.nm, bandWidth*yocoAstroSI.m/yocoAstroSI.nm;

    cpxDummyArray = array(complex,nbBases,nbWlen,nbFrames);
    dummyArray = array(0.0,nbBases,nbWlen,nbFrames);

    amdlibSetOiVis, newVis,
        cpxDummyArray,cpxDummyArray,dummyArray,
        array(DIFAMP,nbFrames),array(DIFAMPER,nbFrames), 
        array(DIFPHI,nbFrames), array(DIFPHIER,nbFrames),
        array(0.0,nbBases,nbFrames),
        array(TARG,nbFrames), array(TIM,nbFrames),
        array(JD,nbFrames), array(EXP,nbFrames),
        array(U,nbFrames), array(V,nbFrames), array(STAT,nbFrames);

    amdlibSetOiVis2, newVis2,
        array(SQV,1), array(SQVER,1),
        array(TARG,nbFrames), array(TIM,nbFrames),
        array(JD,nbFrames), array(EXP,nbFrames),
        array(U,nbFrames), array(V,nbFrames), array(STAT,nbFrames);

    if (nbBases == 3)
    {
        amdlibSetOiVis3, newTriple3,
            array(V3AMP,1),array(V3AMPER,1),
            array(V3PHI,1),array(V3PHIER,1),
            array(TARG,nbFrames), array(TIM,nbFrames),
            array(JD,nbFrames), array(EXP,nbFrames),
            array(U1,nbFrames), array(U2,nbFrames),
            array(V1,nbFrames), array(V2,nbFrames), STAT3;
    }

    if(is_void(outName))
    {
        stringStart = files(1);
        stringEnd = files(0);
        w = 1;
        while(strmatch(strpart(stringStart,1:w),strpart(stringEnd,1:w)))
            w++;

        outName = yocoStrSplit(stringStart,"OIDATA")(1) +
            "-" +
            strpart(stringEnd,w:);
    }

    write, "writing file "+outName;
    status = amdlibWriteOiFile(outName,
                               pointer(nil), &inoiArray, &inoiTarget,
                               &newWave, &newPhot, &newVis, 
                               &newVis2, &newTriple3, &newPiston,
                               pointer(nil));

    /* Copy all keywords that could be useful for use afterwards */
    if(_amdlibDumpAllKeywords( 1, files(1), outName) == 0)
        yocoError, "Could not write OI files", errMsg, 3;

    // amdlibDumpBinTable, ["OI_ARRAY","OI_TARGET"], files(1), outName;
} 

/***************************************************************************/

func interpolateBadPixels(Img0, typeI=, badVal=)
    /* DOCUMENT interpolateBadPixels(Img0, typeI=, badVal=)

       DESCRIPTION
       Interpolate the pixels in an image set to zero (as a convention for marking pixels as "bad".
       4 different ways exist.
       - 1st way is by using growing index starting from 1
       - 2nd way is the reverse as 1st. When big holes exist in the image, these two methods gives
       typical diagonal stripes, with a different direction.
       - 3rd way is using a random index, minimizing the effect of the previous diagonal stripes.
       Instead, the interpolated region will appear as "patchy".
       - 4the way is the same as 3rd except the it takes pixels farer to interpolate in order to
       improve SNR (prototype). "crosses" patterns may appear by using this way.

       PARAMETERS
       - Img0  : the input image, where bad pixels are set to zero
       - typeI : the way the pixels are interpolated, described in DESCRIPTION
       - badVal: 

       RESULTS
       an image with interpolated regions.

       CAUTIONS
       many 

       EXAMPLES
       img = random(100,100);
       img(25:75,25:75)=0;
       window,1;
       pli,img;
       window,2;
       pli,interpolateBadPixels(img,typeI=1)
       window,3;
       pli,interpolateBadPixels(img,typeI=2)
       window,4;
       pli,interpolateBadPixels(img,typeI=3)
       window,5;
       pli,interpolateBadPixels(img,typeI=4)

       SEE ALSO:
    */
{
    local Img1;

    if(is_void(typeI))
        typeI=1;

    Img1 = Img0;

    nX = dimsof(Img1)(2);

    nY = dimsof(Img1)(3);

    if(is_void(badVal))
        badVal=0.0;

    bads = where2(Img1==badVal);

    if(typeI==1)
    {
        while(numberof(bads)!=0)
        {
            if(numberof(bads)==storeOldBads)
                break;
            storeOldBads = numberof(bads);
            N = numberof(bads(1,));
            for(i=1;i<=N;i++)
            {
                count = 0;
                if(bads(1,i)+1<=nX)
                {
                    img10 = Img1(bads(1,i)+1,bads(2,i));
                    if(img10 != 0)
                    {
                        Img1(bads(1,i),bads(2,i)) += img10;
                        count++;
                    }
                }
                if(bads(2,i)+1<=nY)
                {
                    img01 = Img1(bads(1,i),bads(2,i)+1);
                    if(img01 != 0)
                    {
                        Img1(bads(1,i),bads(2,i)) += img01;
                        count++;
                    }
                }
                if(bads(1,i)-1>0)
                {
                    img_10 = Img1(bads(1,i)-1,bads(2,i));
                    if(img_10 != 0)
                    {
                        Img1(bads(1,i),bads(2,i)) += img_10;
                        count++;
                    }
                }
                if(bads(2,i)-1>0)
                {
                    img0_1 = Img1(bads(1,i),bads(2,i)-1);
                    if(img0_1 != 0)
                    {
                        Img1(bads(1,i),bads(2,i)) += img0_1;
                        count++;
                    }
                }
                Img1(bads(1,i),bads(2,i))/=double(count+(count==0));
            }
            bads = where2(Img1==badVal);
        }
    }
    else if(typeI==2)
    {
        while(numberof(bads)!=0)
        {
            N = numberof(bads(1,));
            for(i=N;i>=1;i--)
            {
                count = 0;
                if(bads(1,i)+1<=nX)
                {
                    img10 = Img1(bads(1,i)+1,bads(2,i));
                    if(img10 != 0)
                    {
                        Img1(bads(1,i),bads(2,i)) += img10;
                        count++;
                    }
                }
                if(bads(2,i)+1<=nY)
                {
                    img01 = Img1(bads(1,i),bads(2,i)+1);
                    if(img01 != 0)
                    {
                        Img1(bads(1,i),bads(2,i)) += img01;
                        count++;
                    }
                }
                if(bads(1,i)-1>0)
                {
                    img_10 = Img1(bads(1,i)-1,bads(2,i));
                    if(img_10 != 0)
                    {
                        Img1(bads(1,i),bads(2,i)) += img_10;
                        count++;
                    }
                }
                if(bads(2,i)-1>0)
                {
                    img0_1 = Img1(bads(1,i),bads(2,i)-1);
                    if(img0_1 != 0)
                    {
                        Img1(bads(1,i),bads(2,i)) += img0_1;
                        count++;
                    }

                }
                Img1(bads(1,i),bads(2,i))/=double(count+(count==0));
            }
            bads = where2(Img1==badVal);
        }
    }
    else if(typeI==3)
    {
        randomize;
        while(numberof(bads)!=0)
        {
            N = numberof(bads(1,));
            i=int(random(N)*N)+1;

            for(k=1;k<=N;k++)
            {
                if(Img1(bads(1,i(k)),bads(2,i(k)))==0)
                {
                    count = 0;
                    if(bads(1,i(k))+1<=nX)
                    {
                        img10 = Img1(bads(1,i(k))+1,bads(2,i(k)));
                        if(img10 != 0)
                        {
                            Img1(bads(1,i(k)),bads(2,i(k))) += img10;
                            count++;
                        }
                    }
                    if(bads(2,i(k))+1<=nY)
                    {
                        img01 = Img1(bads(1,i(k)),bads(2,i(k))+1);
                        if(img01 != 0)
                        {
                            Img1(bads(1,i(k)),bads(2,i(k))) += img01;
                            count++;
                        }
                    }
                    if(bads(1,i(k))-1>0)
                    {
                        img_10 = Img1(bads(1,i(k))-1,bads(2,i(k)));
                        if(img_10 != 0)
                        {
                            Img1(bads(1,i(k)),bads(2,i(k))) += img_10;
                            count++;
                        }
                    }
                    if(bads(2,i(k))-1>0)
                    {
                        img0_1 = Img1(bads(1,i(k)),bads(2,i(k))-1);
                        if(img0_1 != 0)
                        {
                            Img1(bads(1,i(k)),bads(2,i(k))) += img0_1;
                            count++;
                        }
                    }
                    Img1(bads(1,i(k)),bads(2,i(k)))/=double(count+(count==0));
                }
            }
            bads = where2(Img1==badVal);
        }
    }
    else if(typeI==4)
    {
        stop = 10;
        randomize;
        while(numberof(bads)!=0)
        {
            N = numberof(bads(1,));
            i=int(random(N)*N)+1;

            for(k=1;k<=N;k++)
            {
                if(Img1(bads(1,i(k)),bads(2,i(k)))==0)
                {
                    count = 0;
                    img10 = kb = 0;
                    while((img10 == 0)&&(kb != stop))
                    {
                        kb++;
                        if(bads(1,i(k))+kb<=nX)
                        {
                            img10 = Img1(bads(1,i(k))+kb,bads(2,i(k)));
                            if(img10 != 0)
                            {
                                wt = 1.0/kb;
                                Img1(bads(1,i(k)),bads(2,i(k))) += wt*img10;
                                count+=wt;
                            }
                        }
                    }
                    img01 = kb = 0;
                    while((img01 == 0)&&(kb != stop))
                    {
                        kb++;
                        if(bads(2,i(k))+kb<=nY)
                        {
                            img01 = Img1(bads(1,i(k)),bads(2,i(k))+kb);
                            if(img01 != 0)
                            {
                                wt = 1.0/kb;
                                Img1(bads(1,i(k)),bads(2,i(k))) += wt*img01;
                                count+=wt;
                            }
                        }
                    }
                    img_10 = kb = 0;
                    while((img_10 == 0)&&(kb != stop))
                    {
                        kb++;
                        if(bads(1,i(k))-kb>0)
                        {
                            img_10 = Img1(bads(1,i(k))-kb,bads(2,i(k)));
                            if(img_10 != 0)
                            {
                                wt = 1.0/kb;
                                Img1(bads(1,i(k)),bads(2,i(k))) += wt*img_10;
                                count+=wt;
                            }
                        }
                    }
                    img0_1 = kb = 0;
                    while((img0_1 == 0)&&(kb != stop))
                    {
                        kb++;
                        if(bads(2,i(k))-kb>0)
                        {
                            img0_1 = Img1(bads(1,i(k)),bads(2,i(k))-kb);
                            if(img0_1 != 0)
                            {
                                wt = 1.0/kb;
                                Img1(bads(1,i(k)),bads(2,i(k))) += wt*img0_1;
                                count+=wt;
                            }
                        }
                    }
                    Img1(bads(1,i(k)),bads(2,i(k)))/=double(count+(count==0));
                }
            }
            bads = where2(Img1==badVal);
        }
    }
    return Img1;
}

/***************************************************************************/

func interpolateBadPixels3(Img0, badPixMap, typeI=)
    /* DOCUMENT interpolateBadPixels3(Img0, badPixMap, typeI=)

       SEE ALSO:
    */
{
    local Img1, badPix;

    badPix = badPixMap;

    if(is_void(typeI))
        typeI=1;

    Img1 = Img0;

    nX = dimsof(Img1)(2);

    nY = dimsof(Img1)(3);

    if(is_void(badVal))
        badVal=1.0;

    bads = where2(badPix==badVal);

    if(typeI==1)
    {
        while(numberof(bads)!=0)
        {
            if(numberof(bads)==storeOldBads)
                break;

            storeOldBads = numberof(bads);
            N = numberof(bads(1,));
            for(i=1;i<=N;i++)
            {
                Img1(bads(1,i),bads(2,i),) = 0;
                count = 0;
                if(bads(1,i)+1<=nX)
                {
                    if(badPix(bads(1,i)+1,bads(2,i))!=badVal)
                    {
                        img10 = Img1(bads(1,i)+1,bads(2,i),);
                        Img1(bads(1,i),bads(2,i),) += img10;
                        count++;
                    }
                }
                if(bads(2,i)+1<=nY)
                {
                    if(badPix(bads(1,i),bads(2,i)+1)!=badVal)
                    {
                        img01 = Img1(bads(1,i),bads(2,i)+1,);
                        Img1(bads(1,i),bads(2,i),) += img01;
                        count++;
                    }
                }
                if(bads(1,i)-1>0)
                {
                    if(badPix(bads(1,i)-1,bads(2,i))!=badVal)
                    {
                        img_10 = Img1(bads(1,i)-1,bads(2,i),);
                        Img1(bads(1,i),bads(2,i),) += img_10;
                        count++;
                    }
                }
                if(bads(2,i)-1>0)
                {
                    if(badPix(bads(1,i),bads(2,i)-1)!=badVal)
                    {
                        img0_1 = Img1(bads(1,i),bads(2,i)-1,);
                        Img1(bads(1,i),bads(2,i),) += img0_1;
                        count++;
                    }
                }
                Img1(bads(1,i),bads(2,i),)/=double(count+(count==0));
                badPix(bads(1,i),bads(2,i)) = 0;
            }
            bads = where2(badPix==badVal);
        }
    }
    else if(typeI==3)
    {
        randomize;
        while(numberof(bads)!=0)
        {
            N = numberof(bads(1,));
            i=int(random(N)*N)+1;

            for(k=1;k<=N;k++)
            {
                if(bads(1,i)+1<=nX)
                {
                    if(badPix(bads(1,i(k))+1,bads(2,i(k)))!=badVal)
                    {
                        count = 0;
                        if(bads(1,i(k))+1<=nX)
                        {
                            img10 = Img1(bads(1,i(k))+1,bads(2,i(k)),);
                            if(anyof(img10) != 0)
                            {
                                Img1(bads(1,i(k)),bads(2,i(k)),where(img10!=0)) += img10(where(img10!=0));
                                badPix(bads(1,i(k))+1,bads(2,i(k))) = 1;
                                count++;
                            }
                        }
                        if(bads(2,i(k))+1<=nY)
                        {
                            img01 = Img1(bads(1,i(k)),bads(2,i(k))+1,);
                            if(anyof(img01) != 0)
                            {
                                Img1(bads(1,i(k)),bads(2,i(k)),where(img01!=0)) += img01(where(img01!=0));
                                badPix(bads(1,i(k))+1,bads(2,i(k))) = 1;
                                count++;
                            }
                        }
                        if(bads(1,i(k))-1>0)
                        {
                            img_10 = Img1(bads(1,i(k))-1,bads(2,i(k)),);
                            if(anyof(img_10) != 0)
                            {
                                Img1(bads(1,i(k)),bads(2,i(k)),where(img_10!=0)) += img_10(where(img_10!=0));
                                badPix(bads(1,i(k))+1,bads(2,i(k))) = 1;
                                count++;
                            }
                        }
                        if(bads(2,i(k))-1>0)
                        {
                            img0_1 = Img1(bads(1,i(k)),bads(2,i(k))-1,);
                            if(anyof(img0_1) != 0)
                            {
                                Img1(bads(1,i(k)),bads(2,i(k)),where(img0_1!=0)) += img0_1(where(img0_1!=0));
                                badPix(bads(1,i(k))+1,bads(2,i(k))) = 1;
                                count++;
                            }
                        }
                        Img1(bads(1,i(k)),bads(2,i(k)),)/=double(count+(count==0));
                    }
                }
                bads = where2(badPix==badVal);
            }
        }
    }
    return Img1;
}

/*************************************************************/

func computeP2vmPhase(&wlen, &freq, p2vmFile=, plot=)
    /* DOCUMENT computeP2vmPhase(&wlen, &freq, p2vmFile=, plot=)

       DESCRIPTION
       Computes the P2VM fringes phase (i.e. the internal instrumental phase)

       PARAMETERS
       - wlen    : 
       - freq    : 
       - p2vmFile: 
       - plot    : 

       RESULTS
       A phase array

       CAUTIONS
       many here also

       EXAMPLES 

       SEE ALSO:
    */
{
    if(is_void(plot))
        plot=0;

    if(plot==1)
    {
        winkill;
    }

    amdlibLoadP2vm,mat, phot, Vk, wlen, flag, calibPhase, nbTels, keyNames, keyValues, inputP2vmFile=p2vmFile;

    nbBases = dimsof(mat)(2)/2;
    nbWlen  = dimsof(mat)(4);
    nbPix   = dimsof(mat)(3);

    padding = 8192;

    useMat = transpose(mat(1:nbBases,,),[2,3]);
    useVk2 = transpose(Vk,[1,3])(sum,,);
    useVk  = interpolateBadPixels(useVk2);

    for(k=1;k<=nbBases;k++)
    {
        useMat(k,,) = interpolateBadPixels(useMat(k,,));
    }
    useMat = useMat - useMat(,,avg)(,,-);
    corrMat = useMat / (useVk(-,,)+(useVk(-,,)==0));

    // corrMatPad = grow(corrMat(,,:nbPix/2),
    // array(0.0,nbBases,nbWlen,padding-nbPix),
    // corrMat(,,nbPix/2+1:));
    corrMatPad = roll(grow(array(0.0,nbBases,nbWlen,padding/2-nbPix/2),
                           corrMat,
                           array(0.0,nbBases,nbWlen,padding/2-nbPix/2)),
                      [0,0,padding/2]);

    if(plot==1)
    {
        for(k=1;k<=nbBases;k++)
        {
            window,k,height=500,width=500;
            pli,corrMatPad(k,,);
        }
    }

    fftMat = fft(corrMatPad,[0,0,1]);
    modMat = abs(fftMat);
    modMatNorm = modMat / (modMat(,,max)(,,-)+(modMat(,,max)(,,-)==0));

    fringFreq = modMat(,,:padding/2)(,,mxx);

    if(plot==1)
    {
        for(k=1;k<=nbBases;k++)
        {
            window,k+3,height=500,width=500;
            pli,modMatNorm(k,,);
            plg,fringFreq(k,),indgen(nbWlen)-0.5,color="red";
        }
    }

    phiMod = atan(fftMat.im,fftMat.re); 

    if(plot==1)
    {
        for(k=1;k<=nbBases;k++)
        {
            window,k+6,height=500,width=500;
            pli,phiMod(k,,);
            plg,fringFreq(k,),indgen(nbWlen)-0.5,color="red";
        }
    }

    phaseP2vm = array(0.0,dimsof(fringFreq));

    for(j=1;j<=nbBases;j++)
        for(l=1;l<=nbWlen;l++)
        {
            phaseP2vm(j,l) = phiMod(j,l,fringFreq(j,l));
        }

    if(plot==1)
    {
        window,10,height=500,width=500;
        for(k=1;k<=nbBases;k++)
        {
            plg,phaseP2vm(k,),wlen,color=colors(k);
        }
    }

    freq = fringFreq/double(padding)*nbPix;

    return phaseP2vm;
}

/*************************************************************/

func computeChromaticPiston(wlen, corrFlux, plot=, poly=)
    /* DOCUMENT computeChromaticPiston(wlen, corrFlux, plot=, poly=)

       DESCRIPTION
       Compute a piston locally of a given wavelength.

       PARAMETERS
       - wlen    : 
       - corrFlux: 
       - plot    : 
       - poly    : 

       SEE ALSO:
    */
{
    if(is_void(plot))
        plot=0;

    if(plot==1)
    {
        winkill;
    }

    nbWlen = dimsof(corrFlux)(3);
    nbBases = dimsof(corrFlux)(2);

    pist = array(0.0,dimsof(corrFlux));
    phi = atan(corrFlux.im,corrFlux.re);

    nbElemPist = 4;

    for(k=1;k<=nbBases;k++)
    {
        for(l=1;l<=nbWlen;l++)
        {
            corrToComp = wlenToComp = [];
            for(m=1;m<=nbElemPist;m++)
            {
                if((l+m-1-nbElemPist/2<=nbWlen)&&
                   (l+m-1-nbElemPist/2>=1))
                {
                    if((corrFlux(k,l+m-1-nbElemPist/2)!=0)&&
                       (phi(k,l+m-1-nbElemPist/2)!=0))
                    {
                        grow,corrToComp,corrFlux(k,l+m-1-nbElemPist/2);
                        grow,wlenToComp,wlen(l+m-1-nbElemPist/2);
                    }
                }
            }
            if(numberof(wlenToComp)>2)
                // pist(k,l) = -ComputePistonPhase(corrToComp,wlenToComp);
                // pist(k,l) = -ComputePistonIterative(corrToComp,wlenToComp);
                pist(k,l) = -ComputePistonAIPS(corrToComp,wlenToComp);
        }
    }

    phi0 = where(phi==0);
    if(numberof(phi0)!=0)
        pist(phi0)=0.0;

    if(plot==1)
    {
        window,1,height=500,width=500;
        for(k=1;k<=nbBases;k++)
            plg,pist(k,)*yocoAstroSI.m/yocoAstroSI.mum,wlen,color=colors(k);
        xytitles,"wavelength (m)","Piston (!mm)";
        hcps,"~/p2vmPiston.ps";

        window,2,height=500,width=500;
        for(k=1;k<=nbBases;k++)
            plg,phi(k,),wlen,color=colors(k);
        xytitles,"wavelength (m)","Phase (rad)";
        hcps,"~/p2vmPhase.ps";
    }

    return pist;
}

/*************************************************************/

func computeDiffPhasePiston(&wlen, inputOiFile=)
    /* DOCUMENT computeDiffPhasePiston(&wlen, inputOiFile=)

       DESCRIPTION
       Compute a local piston versus wavelength from differential phase data information.

       PARAMETERS
       - wlen       : 
       - inputOiFile: 

       SEE ALSO:
    */
{
    amdlibLoadOiData, qwlen, qbandWidth, qtime, qfringeSNR, qsqVis, qsqVisErr,
        quvCoord, qbandFlag, qpistonOPDArray, qsigmaPistonArray, qdiffVisAmp,
        qdiffVisAmpErr, qdiffVisPhase, qdiffVisPhaseErr, qvis3Amp, qvis3AmpErr,
        qvis3Phase, qvis3PhaseErr, qcpxVis, qcpxVisErr, qvisCovRI, quv1Coord,
        quv2Coord, qfluxSumPiPj, qfluxRatPiPj, qfluxProdPiPj, qspectrum,
        qspectrumErr, inputOiFile=inputOiFile;

    cFlux = exp(1i*qdiffVisPhase(,,1));

    pist = computeChromaticPiston(qwlen, cFlux);

    return pist;
}

/*************************************************************/

func computeP2vmPiston(&pwlen, p2vmFile=)
    /* DOCUMENT computeP2vmPiston(&pwlen, p2vmFile=)

       DESCRIPTION
       From a P2VM file, computes the piston using the phase computed by the
       FT method

       PARAMETERS
       - pwlen   : 
       - p2vmFile:

       SEE ALSO:
    */
{
    P2vmPhase = computeP2vmPhase(pwlen,p2vmFile=p2vmFile);
    P2vmCorrFlux = exp(1i*P2vmPhase);
    pwlen = pwlen*1e-9;

    pist = computeChromaticPiston(pwlen, P2vmCorrFlux);

    return pist;
}

/*************************************************************/

func plotP2vmPiston(&pwlen, p2vmFile=)
    /* DOCUMENT plotP2vmPiston(&pwlen, p2vmFile=)

       DESCRIPTION
       Plots the P2VM piston

       PARAMETERS
       - pwlen   : 
       - p2vmFile: 

       SEE ALSO:
    */
{
    winkill;
    pist = computeP2vmPiston(pwlen, p2vmFile=p2vmFile);
    window,1;
    for(k=1;k<=3;k++)
        plg,pist(k,)*yocoAstroSI.m/yocoAstroSI.mum,pwlen*yocoAstroSI.m/yocoAstroSI.mum;
    xytitles,"Wavelength (!mm)", "Piston (!mm)";

    window,2;

}

/*************************************************************/

func ComputePistonAIPS(corrFlux, wlen, plot=)
    /* DOCUMENT ComputePistonAIPS(corrFlux, wlen, plot=)

       DESCRIPTION 
       Computes the piston using methods inspired from the AIPS software,
       and from K. Tristram.

       SEE ALSO:
    */
{
    if(numberof(wlen)<=1)
        return  0;
    
    nWlen = numberof(wlen);

    // Get wave-number first
    sigma = 1.0/wlen;

    // Put the complex coherent flux into a zero-padded array
    nPad0 = 5*nWlen;
    nP = 2;
    while(nP^2<=nPad0)
        nP*=2;
    nPad = nP^2;
    padArray = array(complex,nPad);
    N        = numberof(corrFlux);
    padArray(1:N) = corrFlux;

    // Compute the spectral density of the coherent flux
    DSP  = roll(abs(fft(padArray))^2);
    fx   = 1.0/sigma(dif)(avg);
    pist = span(-fx/2,fx/2,nPad+1);

    // Get the peak of the coherent flux: this is the OPD
    jMax   = DSP(mxx);
    piston = pist(jMax);

    if(plot==1)
    {
        window,3;
        fma;
        plg,DSP,pist;
    }

    return piston;
}

/*************************************************************/

func ComputePistonPhase(corrFlux, wlen, &errorPist, &falseDetection, threshold=, verbose=)
    /* DOCUMENT ComputePistonPhase(corrFlux, wlen, &errorPist, &falseDetection, threshold=, verbose=)

       DESCRIPTION 
       Computes the piston from the "F. Millour, PhD thesis 2006" method using
       successive incremental differential phase measurements.

       SEE ALSO:
    */
{
    if(is_void(verbose))
        verbose=1;
    // Set fringes detection threshold to be 1 radian by default
    if(is_void(threshold))
        threshold=1.0;

    // Compute wave number from wavelength 
    sigma = 1/wlen;
    xi = 2 * pi * sigma;

    // Compute successive interspectra by shifting coherent flux array
    // by 1 spectral channel
    interSpectra = corrFlux(2:) * conj(corrFlux(:-1));

    // Get number of wavelength -1
    nbW = numberof(interSpectra);
    oddOnes= where(indgen(nbW)%2);

    // Keep only odd interspectra to avoid repetition and noise problems
    interSpectra = interSpectra(oddOnes); 

    // Compute differential phase, i.e. phase of the interspectra
    phiavg = atan(interSpectra(avg).im, interSpectra(avg).re);

    // Keep only relevant wave numbers
    goods = where(xi(dif)!=0);

    // Compute piston, 1st approximation
    piston = (phiavg/(xi(dif)(goods)))(avg);

    // Correct correlated flux from piston
    newCorrFlux = corrFlux * exp(-1i*xi*piston);

    // Compute phase from this corrected correlated flux
    phiPlus = atan(newCorrFlux(avg).im, newCorrFlux(avg).re);

    // Add the fringe fraction correction to the piston
    piston += phiPlus / (2*xi(avg));

    // Compute piston-free correlated flux
    newCorrFlux = corrFlux * exp(-1i*xi*piston);

    // Compute residual
    residual = atan(newCorrFlux.im, newCorrFlux.re)(rms);

    // Test if piston computation failed
    if( residual > threshold )
    {
        // Print a warning
        if(verbose)
            yocoLogWarning, "data set with more than 1rad of rms";

        // Compute piston using a different method, more reliable
        // respective to noise, but less accurate
        interSpectra = newCorrFlux(2:) * conj(newCorrFlux(:-1));

        // Average interspectrum on all wavelengths
        intavg = interSpectra(avg);

        // Get differential phase
        phiavg = atan(intavg.im, intavg.re);

        // Correct piston with this new estimate
        piston += phiavg/(xi(dif)(avg));

        // Compute again piston-free correlated flux
        newCorrFlux = corrFlux * exp(-1i*xi*piston);
    }

    // Test for false detection
    if(residual > threshold)
    {
        falseDetection = 1;
        errorPist = 1e99;
    }
    else
    {
        // Compute piston error
        falseDetection = 0;
        errorPist = residual / (2*xi(avg));
    }

    return piston;
}

/************************************************************************/

func ComputePistonIterative(corrFlux, wlen, bandwidth, sigma2CorrFlux=, plot=, pist0=, debug=)
    /* DOCUMENT ComputePistonIterative(corrFlux, wlen, bandwidth, sigma2CorrFlux=, plot=, pist0=, debug=)

       DESCRIPTION

       PARAMETERS
       - corrFlux      : 
       - wlen          : 
       - bandwidth     : 
       - sigma2CorrFlux: 
       - plot          : 
       - pist0         : 
       - debug         : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    LAMBDA=1.0;

    if(debug)
        write,"Iterative_2_regul";
    debug=0;
    if(debug)
    {
        window, 5, wait=1;
        fma;
    }

    ds = (1./wlen);
    waveNr = 2*pi*ds;

    if(is_void(pist0))
    {
        opd0 = ComputePistonPhase(corrFlux, wlen,verbose=0);
        // nx = 2*numberof(wlen);
        // if(is_void(bandwidth))
        // R = wlen(avg) / (wlen(dif)(avg) * 2);
        // else
        // R = (wlen / bandwidth)(avg)/2;
        // totoPist = span(-R*wlen(avg), R*wlen(avg), 2*nx);

        // fazorr = exp( -1i * waveNr(, -) * totoPist(-, ));
        // cVisC = corrFlux(, -) * fazorr;
        // cVisCN = cVisC * conj(cVisC(avg, ))(-, );
        // LAMBDA = 10000;
        // chi2 = ((cVisCN.im^2)(avg, )/(cVisCN.re^2)(avg, ) + LAMBDA/((cVisCN.im(dif,)^2)(avg,) + (cVisCN.re(dif,)^2)(avg,)))/(1+2.0*LAMBDA);
        // opd0 = totoPist(chi2(mnx));
    }
    else
        opd0 = pist0;

    // pouet()

    if(is_void(sigma2CorrFlux))
        sigma2CorrFlux = array(1.0+1i, dimsof(corrFlux));

    maxIter = 50;
    amdlibPRECISION = 1e-5;
    // Methode dichotomique

    coefGrow = 1.5;
    coefShrink = 2.;

    Dopd = 0.0;
    // opd = 0.0;
    for(N=0; N<maxIter; N++)
    { /* start iterations */

        fazor0 = exp( -1i * waveNr * opd0);
        cVisC0 = corrFlux * fazor0;
        cVisCN0 = cVisC0 * conj(cVisC0(avg));
        chi20 = ((cVisCN0.im^2)(avg, )/(cVisCN0.re^2)(avg, ) + LAMBDA/((cVisCN0.im(dif,)^2)(avg,) + (cVisCN0.re(dif,)^2)(avg,)))/(1+2.0*LAMBDA);
        // chi20 = (abs((corrFlux * fazor0))^2)(sum);

        opd1 = opd0+Dopd;
        fazor1 = exp( -1i * waveNr * opd1);
        cVisC1 = corrFlux * fazor1;
        cVisCN1 = cVisC1 * conj(cVisC1(avg));
        chi21 = ((cVisCN1.im^2)(avg, )/(cVisCN1.re^2)(avg, ) + LAMBDA/((cVisCN1.im(dif,)^2)(avg,) + (cVisCN1.re(dif,)^2)(avg,)))/(1+2.0*LAMBDA);
        // chi21 = (abs((corrFlux * fazor1))^2)(sum);

        opd2 = opd0-Dopd;
        fazor2 = exp( -1i * waveNr * opd2);
        cVisC2 = corrFlux * fazor2;
        cVisCN2 = cVisC2 * conj(cVisC2(avg));
        chi22 = ((cVisCN2.im^2)(avg, )/(cVisCN2.re^2)(avg, ) + LAMBDA/((cVisCN2.im(dif,)^2)(avg,) + (cVisCN2.re(dif,)^2)(avg,)))/(1+2.0*LAMBDA);
        // chi22 = (abs((corrFlux * fazor2))^2)(sum);

        if((chi21 >= chi20)&&(chi20 > chi22))
        {
            opd0 = opd0 - Dopd / coefShrink;
            Dopd = Dopd * coefGrow;
        }
        else if((chi21 <= chi20)&&(chi20 < chi22))
        {
            opd0 = opd0 + Dopd / coefShrink;
            Dopd = Dopd * coefGrow;
        }
        else if((chi21 > chi20)&&(chi20 < chi22))
        {
            Dopd = Dopd / coefShrink;
        }
        else
        {
            r = random();
            if(Dopd == 0)
                Dopd = (r-0.5)*wlen(1);
            if(r>0.5)
                opd0 -= Dopd / coefShrink;
            else
                opd0 += Dopd / coefShrink;
        }

        if ((100 * (opd0==0) + Dopd/(opd0+(opd0==0)))^2 < amdlibPRECISION*amdlibPRECISION)
        {
            if(debug)
            {
                write, "stopped at", N, opd0, abs(opd0*chi20);
                window, 5;
                fma; 
                plg, chi2/chi2(max), totoPist;
                plg, [chi20, chi21, chi22]/chi2(max), [opd0, opd1, opd2], type="none", marker='\2', color="red", width=100;
                logxy, , 1;
                window, 6;
                fma;
                plg, atan(corrFlux.im, corrFlux.re), wlen;
                plg, modulo(2*pi*opd0/wlen, 2*pi)-pi, wlen, color="red";
                // plg, 2*pi*opd0/wlen, wlen, color="red"

                window, 7;
                fma;
                cpxCor = corrFlux * exp(-2i*pi*opd0/wlen);
                cpxCor = cpxCor * conj(cpxCor(avg))
                    plg, atan(cpxCor.im, cpxCor.re), wlen;
                pause, 300;
            }
            break;
        }
    } /* end iterations */
    piston = opd0;
    sigmaPist = abs(opd0 * chi20);
    return piston;
}

/************************************************************************/

func getSpecsForSpectralShift(&specPhot, &specIntf, &wlen, &photoOffsets, inputRawFile=, inputBiasFile=, inputP2vmFile=, inputP2vmCalFileList=, inputRawCalFile=, inputBiasCalFile=)
    /* DOCUMENT getSpecsForSpectralShift(&specPhot, &specIntf, &wlen, &photoOffsets, inputRawFile=, inputBiasFile=, inputP2vmFile=, inputP2vmCalFileList=, inputRawCalFile=, inputBiasCalFile=)

       DESCRIPTION 

       PARAMETERS
       - specPhot            : 
       - specIntf            : 
       - wlen                : 
       - photoOffsets        : 
       - inputRawFile        : 
       - inputBiasFile       : 
       - inputP2vmFile       : 
       - inputP2vmCalFileList: 
       - inputRawCalFile     : 
       - inputBiasCalFile    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    if (!amdlibCheckBadPixelFile() || !amdlibCheckFlatFieldFile())
    {
        amdlibError, "Check bad pixel file and flat field file are loaded";
        return 0;
    }

    // Load science raw data file (mandatory)
    if (amdlibFileChooser("RAW SCIENCE DATA FILE\n(right-click to exit)", inputRawFile) == 0)
    {
        yocoLogError,"Sorry, raw data file mandatory !";
        return 0;
    }

    // Load science pixel bias map
    if (amdlibFileChooser("SCIENCE PIXEL BIAS MAP\n(right-click to ignore)", inputBiasFile) == 0)
    {
        inputBiasFile = [];
    }

    _amdlibGetKwdVals, inputRawFile, "MJD-OBS",mjd, gui=0;
    write,"MJD", mjd ;

    // If no P2VM is given, set offsets to 0
    offsets = [0.0,0.0,0.0];
    nbFiles = numberof(inputRawFile);
    
    Phot = Interf = [];
    for(k=1;k<=nbFiles;k++)
    {
        if(is_void(inputBiasFile))
        {
            if (amdlibLoadRawData(Frames,Wlen,Dark,Phottmp,Interftmp,
                                  inputRawFile=inputRawFile(k),
                                  inputBiasFile=inputBiasFile,
                                  calibrateRaw=0,convertToScience=0,
                                  photoOffset = offsets) == 0)
            {
                return 0;
            }
            Phottmp = Phottmp(,avg,..)
                }
        else
        {
            // Load science raw data file
            if (amdlibLoadRawData(Frames,Wlen,Dark,Phottmp,Interftmp,
                                  inputRawFile=inputRawFile(k),
                                  inputBiasFile=inputBiasFile,
                                  calibrateRaw=1,convertToScience=1,
                                  photoOffset = offsets) == 0)
            {
                return 0;
            }
        }
        grow, Phot, Phottmp;
        grow, Interf, Interftmp;
    }
    wWlen  = where(Wlen!=0);
    Wlen   = Wlen(wWlen);
    Phot   = Phot(,wWlen,);
    Interf = Interf(,wWlen,);

    // Get arrays dimensions
    nbTel   = dimsof(Phot)(2);
    nbBases = nbTel*(nbTel-1)/2;
    nbWlen  = dimsof(Phot)(3);

    // Calibrate spectra 
    fot  = Phot(, , sum);
    intf = Interf(sum, , sum);

    fot  = fot/fot(,avg)(,-);
    intf = intf/intf(avg);

    // Return values
    photoOffsets = offsets;
    specPhot = fot;
    specIntf = intf;

    // Tell the operation succeeded
    return 1;
}

/************************************************************************/

func amdlibChangeSpectralShift(void, inputRawFile=, inputBiasFile=, inputP2vmFile=, inputP2vmCalFileList=, inputRawCalFile=, inputBiasCalFile=, filtrIris=)
    /* DOCUMENT amdlibChangeSpectralShift(inputRawFile=, inputBiasFile=, inputP2vmFile=, inputP2vmCalFileList=, inputRawCalFile=, inputBiasCalFile=, filtrIris=)

       DESCRIPTION
       This function allows to set manually the spectral shift between the
       interferometric beam and the photometric beams. This step is one of
       the most importants in AMBER data reduction, so please do it very
       carefully.

       PARAMETERS
       - inputRawFile        : MANDATORY raw data file from a science star
       - inputBiasFile       : dark file associated to the science star
       - inputP2vmFile       : P2VM file. This is where the shifts are read so it is strongly recommended to use one
       - inputP2vmCalFileList: Series of calibration files of a P2VM
       - inputRawCalFile     : Calibration star data file
       - inputBiasCalFile    : Calibration star dark file
       - filtrIris           : 

       FILES
       You will need the following raw data files (all mandatory)
       o a Flat Field map
       o a bad pixel map
       o a star raw data file, preferably with a visible emission
       or absorption line
       o a pixel bias map (i.e. a dark)
       o the P2VM acquisition files just before the observation.

       RESULTS
       You get an interactive window to change the shifts on all 3 telescopes.
       When you are done, click on "OK".

       CAUTIONS
       The P2VM files will be updated (i.e.the files will be changed by this
       program)

       EXAMPLES
       type amdlibChangeSpectralShift and follow instructions.

       SEE ALSO
       amdlibEditSpectralShift, amdlibCheckSpectralShift
    */
{
    winkill;

    // Load calibrated spectra
    if(!getSpecsForSpectralShift(fot, intf, wlen, offsets,
                                 inputRawFile=inputRawFile,
                                 inputBiasFile=inputBiasFile,
                                 inputP2vmFile=inputP2vmFile,
                                 inputP2vmCalFileList=inputP2vmCalFileList,
                                 inputRawCalFile=inputRawCalFile,
                                 inputBiasCalFile=inputBiasCalFile))
    {
        return 0;
    }

    // Get arrays
    nbTel = dimsof(fot)(2);
    shifts = offsets;

    if(filtrIris==1)
    {
        intf = filtreArray(intf,plot=0);
        for (i=1; i<=nbTel; i++)
        {
            fot(i,) = filtreArray(fot(i,),plot=0);
        } 
    }

    fig2ch = array(0, dimsof(shifts));


    yocoNmCreate,1,1,nbTel, width=850, height=650, style="amdlibLargeBox.gs",landscape=1;
    // for (i=1; i<=nbTel; i++)
    // {
    //     window, i, wait=1, width=650, height=450, style="amdlibLargeBox.gs";
    // }

    while (1==1)
    {
        window,1;
        fma;
        
        idx = indgen(numberof(Wlen));
        for (i=1; i<=nbTel; i++)
        {
            //crop the arrays to not contaminate them with discontinuities
            crop1 = int(offsets(i)-shifts(i)+1.5);
            if (crop1<=0)
                crop1=1;
            crop2 = int(offsets(i)-shifts(i)-0.5);
            if (crop2>0)
                crop2=0;
            range = crop1:crop2;

            newFot = _amdlibSubPixelShift(fot(i, ), offsets(i)-shifts(i))(range);
            chi2 = sum((newFot - intf(range))^2);

            //plot updated spectra
            plsys, i;
            plg, intf(range), idx(range), width=4;
            plg, newFot, idx(range), color=colors(i);
            plt,"!c^2^: "+pr1(chi2),numberof(idx(range))/2,0.2*max(newFot),height=24,justify="CH",tosys=i;
        }

        // Get GUI information from user
        result = yocoGuiChangeNumber(shifts, fig2ch, once=1, kill=0, win=0);

        if (result=="ok")
        {
            // return shifts
            write,"Shifts are:",shifts;
            return shifts;
        }
    }
}

/************************************************************************/

func amdlibCheckSpectralShiftOiData(inputOiFile=, filtrIris=)
    /* DOCUMENT amdlibCheckSpectralShiftOiData(inputOiFile=, filtrIris=)

       DESCRIPTION

       PARAMETERS
       - inputOiFile: 
       - filtrIris  : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    winkill;

    if (amdlibFileChooser("RAW SCIENCE DATA FILE\nEnter an AMBER file for shift estimation \n(right-click to exit)", inputOiFile) == 0)
    {
        return 0;
    }

    if (amdlibLoadOiData(wlen, bandWidth, time, fringeSNR, sqVis, sqVisErr,
                         uvCoord, bandFlag, pistonOPDArray,
                         sigmaPistonArray, diffVisAmp,
                         diffVisAmpErr, diffVisPhase, diffVisPhaseErr,
                         vis3Amp, vis3AmpErr,
                         vis3Phase, vis3PhaseErr, cpxVis, cpxVisErr,
                         visCovRI, uv1Coord,
                         uv2Coord, fluxSumPiPj, fluxRatPiPj, fluxProdPiPj,
                         spectrum, spectrumErr,
                         inputOiFile=inputOiFile) == 0)
    {
        return 0;
    }

    nbTel = dimsof(spectrum)(2);

    if(filtrIris==1)
    {
        for (i=1; i<=nbTel; i++)
        {
            spectrum(i,) = filtreArray(spectrum(i,),plot=0);
        } 
    }

    window, 1, wait=1,width=550,height=400,style="amdlibLargeBox.gs";
    for (i=1;i<=nbTel;i++)
    {
        plg, spectrum(i,) / spectrum(i,avg), ,color=colors(i);
    }

}

/************************************************************************/

func amdlibCheckSpectralShift(void, inputRawFile=, inputBiasFile=, inputP2vmFile=, inputP2vmCalFileList=, inputRawCalFile=, inputBiasCalFile=, filtrIris=)
    /* DOCUMENT amdlibCheckSpectralShift(inputRawFile=, inputBiasFile=, inputP2vmFile=, inputP2vmCalFileList=, inputRawCalFile=, inputBiasCalFile=, filtrIris=)

       DESCRIPTION
       This function allows to check the spectral shift between the
       interferometric beam and the photometric beams written in the raw data
       P2VM files header. This step is one of the most importants in AMBER
       data reduction. Check the spectral shift very carefully.

       PARAMETERS
       - inputRawFile        : a star raw data file, preferably with a visible emission or absorption line
       - inputBiasFile       : a pixel bias map (i.e. a dark)
       - inputP2vmFile       : a P2VM file
       - inputP2vmCalFileList: the P2VM acquisition files just before the observation, with an up-to-date spectral shift in the header.
       - inputRawCalFile     : a calibration star raw data file, preferably with a flat or feature-free spectrum
       - inputBiasCalFile    : a pixel bias map (i.e. a dark)
       - filtrIris           : 

       FILES
       You will need the following raw data files (all mandatory)
       o a Flat Field map
       o a bad pixel map
       o a star raw data file, preferably with a visible emission
       or absorption line
       o a pixel bias map (i.e. a dark)
       o the P2VM acquisition files just before the observation, with an
       up-to-date spectral shift in the header.

       RESULTS
       You get an overview of the spectral shift. If all is OK then all the
       spectra should look superimposed. If a shift remain then proceed
       [again] to amdlibChangeSpectralShift

       EXAMPLES
       type amdlibCheckSpectralShift and follow instructions.

       SEE ALSO
       amdlibEditSpectralShift, amdlibChangeSpectralShift
    */
{
    winkill;

    // Load calibrated spectra
    if(!getSpecsForSpectralShift(fot, intf, wlen, offsets,
                                 inputRawFile=inputRawFile,
                                 inputBiasFile=inputBiasFile,
                                 inputP2vmFile=inputP2vmFile,
                                 inputP2vmCalFileList=inputP2vmCalFileList,
                                 inputRawCalFile=inputRawCalFile,
                                 inputBiasCalFile=inputBiasCalFile))
    {
        return 0;
    }

    nbTel = dimsof(fot)(2);

    if(filtrIris==1)
    {
        intf = filtreArray(intf,plot=0);
        for (i=1; i<=nbTel; i++)
        {
            fot(i,) = filtreArray(fot(i,),plot=0);
        } 
    }

    // Plot spectra
    for (i=1;i<=nbTel;i++)
    {
        window, i, wait=1,width=550,height=400,style="amdlibLargeBox.gs";
        plg, intf, width=4;
        plg, fot(i, ), color=colors(i);
    }
}

/************************************************************************/

func _amdlibSubPixelShift(tab_in, dpix)
    /* DOCUMENT _amdlibSubPixelShift(tab_in, dpix)
       Shifts an array of a non integer number of pixels.

       SEE ALSO: roll, fft, array
    */
{
    // Get number of pixels, i.e. the first dimension of the array
    npix=dimsof(tab_in)(2);

    // Define x as a c-index array
    x = indgen(npix)-1;

    // Get the integer part (ip) and fractionnal part (fp)
    // of the offset to apply to tab_in
    ip_dpix=int(dpix);
    fp_dpix=dpix-ip_dpix;

    if (fp_dpix!=0.)
    {
        // Get the values of the first and last points to substract
        // the affine part of the signal
        m1 = tab_in(1);
        m2 = tab_in(0);

        // The value to substract to the signal before the shift
        dr = (m2-m1) / float(npix-1) * x + m1;

        // The value to add again to the periodized signal after the shift
        drc = (m2-m1) / float(npix-1) * (x+fp_dpix) + m1;

        iPix = span (-npix/2,npix/2,npix);

        phi = (2. * pi * fp_dpix / float(npix)) * iPix;

        // The phasor to apply to the fft of the periodized signal
        // to shift it of fp_dpix
        phasor = roll(exp(1i*phi));

        tab_in_periodized = double(tab_in-dr);

        fft_directe = fft(tab_in_periodized,-1);

        phased_fft = phasor*fft_directe;

        tab_decal_periodized = fft(phased_fft ,+1);

        tab_decal = (1.0/npix)*double(tab_decal_periodized) + drc;

        // The shift itself
        tab_out = tab_decal;
    }
    else
        tab_out=tab_in;

    // The integer part of the shift, made by the built-in function roll
    tab_out=roll(tab_out,ip_dpix);

    return tab_out;
}

/************************************************************************/

func sortAMBERFiles(directory=, keywords=, keyDec=, symbolic=, cropVal=)
    /* DOCUMENT sortAMBERFiles(directory=, keywords=, keyDec=, symbolic=, cropVal=)

       DESCRIPTION 
       Creates subdirectories with the names specified by "keyDec" and links
       (or copy) the files to these subdirectories.

       EXAMPLES

       // Sort files by star name
       > sortAMBERFiles

       // Move the files instead of creating symbolic links
       sortAMBERFiles, keywords="HIERARCH ESO DET DIT", keyDec=1, symbolic=0

       // Sort files by night
       sortAMBERFiles, keywords="DATE-OBS", cropVal=10

       // Sort files by instrument mode
       sortAMBERFiles, keywords="HIERARCH ESO INS MODE"

       // Sort files by exposure time
       sortAMBERFiles, keywords="HIERARCH ESO DET DIT", keyDec=1

       // Sort files by FINITO status (ON/OFF)
       sortAMBERFiles, keywords="HIERARCH ESO DEL FT STATUS"

       // Sort files by BCD status (IN/OUT)
       sortAMBERFiles, keywords="HIERARCH ESO INS OPTI7 NAME"

       // Sort the files by all the above at once (by default)
       sortAMBERFiles, keywords=["HIERARCH ESO OCS OBS MODE","HIERARCH ESO OCS OBS SPECCONF", "HIERARCH ESO DET DIT", "HIERARCH ESO DEL FT STATUS", "HIERARCH ESO INS OPTI7 NAME"],keyDec=[0,0,1,0,0];

       SEE ALSO:
    */
{
    if(is_void(symbolic))
        symbolic = 1;

    //By default, sort by instrument name, spectral configuration, dit, finito status, and BCD status
    if(is_void(keywords))
        keywords = ["HIERARCH ESO OCS OBS MODE",
                    "HIERARCH ESO OCS OBS SPECCONF",
                    "HIERARCH ESO DET DIT",
                    "HIERARCH ESO DEL FT STATUS",
                    "HIERARCH ESO INS OPTI7 NAME"];
        
    // Default for keyDec linked to the keywords
    if(is_void(keyDec))
        keyDec=[0,0,1,0,0];

    if(amdlibFileChooser("DIRECTORY\nPlease choose which directory you want to use \n(right-click to exit)", directory) == 0)
        return 0;

    amdlibGetLogName, directory, logFile;
    logFile = directory + string(logFile);

    // Read the log
    amdlibReadLog,logFile,logTable,titles;

    fileName = lsdir(directory);
    nFiles = numberof(fileName);
    objnames = _amdlibGetKwdVals(fileName, keywords, valuesTab);
    
    // Treat the case of decimal keywords
    if(anyof(keyDec==1))
    {
        wk = where(keyDec==1);
        nk = numberof(wk);
        for(l=1;l<=nk;l++)
        {
            for(k=1;k<=nFiles;k++)
            {
                if(objnames(k,wk(l))!="-")
                    objnames(k,wk(l)) = pr1(yocoStr2Double(objnames(k,wk(l))));
            }
        }
    }

    // move the files
    write,"Moving files...";
    for(k=1;k<=nFiles;k++)
    {
        if(anyof(objnames(k,)=="-"))
            continue;
        
        dirToMove = sum(objnames(k,:-1)+"_")+objnames(k,0);
        if(!open(dirToMove,"r",1))
        {
            write,"directory "+dirToMove+" does not exist. Creating...";
            mkdir,dirToMove;
        }

        if(symbolic==1)
        {
            if(strmatch(fileName(k),".gz"))
                yocoError,"Symbolic links will work only on non-compressed files. Uncompress first your files or consider using the symbolic=0 option";
            else
                system,"ln -s "+
                    directory+fileName(k) +
                    " "+
                    directory+dirToMove+"/"+fileName(k);
        }
        else
            rename, directory+fileName(k),
                directory+dirToMove+
                "/"+fileName(k);
    }
} 

/***************************************************************************/

func amdlibPlotUV_Night(nightDir=, plotCals=, width=, win=, star=, plotName=, color=, kill=, plotLabels=)
    /* DOCUMENT amdlibPlotUV_Night(nightDir=, plotCals=, width=, win=, star=, plotName=, color=, kill=, plotLabels=)

       DESCRIPTION
       From a night directory, search for all available UV data and plots
       the corresponding UV points for each star observed.

       PARAMETERS
       - nightDir  : 
       - plotCals  : 
       - width     : 
       - win       : 
       - star      : 
       - plotName  : 
       - color     : 
       - kill      : 
       - plotLabels: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(kill))
        kill=1;

    if(is_void(plotLabels))
        plotLabels = 1;

    if(is_void(plotName))
        plotLabels = 1;

    if(is_void(win))
        win = 1;
    /* Select directory containing files to be treated */
    message = "Choose the product directory (OI_DATA)"; 
    if (amdlibFileChooser(message, nightDir) == 0)
    {
        return 0;
    }

    if(is_void(verbose))
        verbose=1;

    if(is_void(plotCals))
        plotCals = 0;

    logFile = [];
    if(is_void(logFile))
    {
        amdlibGetLogName, nightDir, logFile;
        logFile = nightDir + string(logFile);
    }

    here = get_cwd(".");
    nightDirReal = cd(nightDir);
    cd, here;
    yocoLogTest, "Real name of directory is:", nightDirReal;
    
    nightName = yocoStrSplit(nightDirReal,"/")(-1);
    
    logTable = titles = [];
    amdlibReadLog, logFile, logTable, titles;

    tabFiles = logTable(where(titles=="fileName"), );
    tabDates = logTable(where(titles=="date_obs"), );
    tabNr    = logTable(where(titles=="calib_nr"), );
    proTypes = logTable(where(titles=="pro_catg"), );
    tabTypes = logTable(where(titles=="obs_catg"), );
    OBJTypes = logTable(where(titles=="obs_type"), );
    tabIds   = logTable(where(titles=="P2VM_ID"), );
    tabOBsNr = logTable(where(titles=="OB_nr"), );
    tabTplNr = logTable(where(titles=="Template_nr"), );
    tabNames = logTable(where(titles=="object_name"), );

    starNames = yocoStrRemoveMultiple(tabNames);
    starNames = starNames(where(starNames!="-"));

    id = namesPlotted = [];
    if(!is_void(star))
    {
        h=1;
        while(h <= numberof(star))
        {
            grow, id, where(starNames==star(h));
            h++;
        }
    }
    else
    {
        id = indgen(numberof(starNames));
    }

    nStars = numberof(starNames);
    if(nStars==0)
    {
        yocoLogWarning,"!!! SORRY !!! No stars observed in this night !!!";
        return 0;
    }

    len = max(strlen(starNames));

    rightOnes = where(OBJTypes=="OBJECT");
    if(numberof(rightOnes)==0)
    {
        yocoLogWarning,"!!! SORRY !!! No stars observed in this night !!!";
        return 0;
    }

    kolorz = array("",nStars);
    for(k=1;k<=nStars;k++)
    {
        types = tabTypes(rightOnes)(where(tabNames(rightOnes)==starNames(k)));

        if(is_void(types))
            kolorz(k)="undefined";
        else
        {
            types = types(where(types!="-"));

            if(types(1) == "SCIENCE")
                kolorz(k)="science";
            else if(types(1) == "CALIB")
                kolorz(k)="calib";
        }
    }

    if(plotCals==0)
        toUse = where(kolorz=="science");
    else if(!is_void(star))
    {
        toUse = id;
    }
    else
    {
        toUse = indgen(nStars);
        for(k=1;k<=nStars;k++)
            kolorz(k)="science";
    }

    nStars = numberof(toUse);
    if(nStars==0)
    {
        write,"!!! SORRY !!! No stars observed in this night !!!";
        return 0;
    }

    // Define the window size with an elegant shape & minimum sub-windows
    sqWinSize1 = sqrt(nStars);
    fracWin = 1.0;
    sqX = int(max(1,floor(sqWinSize1/sqrt(fracWin))));
    sqY = int(max(1,floor(sqWinSize1*sqrt(fracWin))));
    
    while(sqX*sqY<nStars)
        sqX++;

    // Create the window with correct dimensions
    if(kill==1)
    {
        winkill,win;
        yocoNmCreate,win,sqX,sqY,square=1,wait=1, width=800,height=600, fx=,fy=,landscape=1,dx=0,dy=0;
    }
    liMax = 0;

    w = 0;
    for(k=1;k<=nStars;k++)
    {
        if(anyof(id==toUse(k)))
        {
            w++;
            types = tabTypes(rightOnes)(where(tabNames(rightOnes)==
                                              starNames((toUse)(k))));
            files = tabFiles(rightOnes)(where(tabNames(rightOnes)==
                                              starNames((toUse)(k))));
            nFiles = numberof(files);
            plsys,w;

            if(!is_void(types))
            {
                types = types(where(types!="-"));

                if(types(1) == "SCIENCE")
                {
                    write,"plotting "+yocoStrReplace(starNames(toUse(k)));
                    grow,namesPlotted,starNames(toUse(k));

                    tliMax = 0;
                    liMin = 1e99;
                    paMax = -1e99;
                    paMin = 1e99;
                    for(l=1;l<=nFiles;l++)
                    {
                        uvTable = [];
                        if(is_void(color))
                            col = colors(l%7);
                        else if(color=="angle")
                            col="angle";
                        else
                            col=color;

                        amdlibPlotUVCoordinates,uvTable,
                            fileName=nightDir+files(l),
                            noTitle=1,noLabels=1,color=col,width=width;

                        Bmax  = max(abs(uvTable(1,),uvTable(2,)));
                        Bmin  = min(abs(uvTable(1,),uvTable(2,)));
                        PAmax = max((atan(uvTable(1,),uvTable(2,))));
                        PAmin = min((atan(uvTable(1,),uvTable(2,))));

                        if(Bmax>liMax)
                            liMax = Bmax;
                        if(Bmax>tliMax)
                            tliMax = Bmax;
                        if(Bmin<liMin)
                            liMin = Bmin;
                        if(PAmax>paMax)
                            paMax = PAmax;
                        if(PAmin<paMin)
                            paMin = PAmin;
                    }
                    if(paMin<0)
                        paMin = paMin+pi;
                    if(paMax<0)
                        paMax = paMax+pi;

                    write, "Range of baselines (m) :", liMin, tliMax;
                    write, "Range of position angles (degrees):", rad2deg*paMin, rad2deg*paMax;

                    limits,square=1;
                }
                else
                {
                    if(plotCals==1)
                    {
                        for(l=1;l<=nFiles;l++)
                        {
                            uvTable = [];
                            amdlibPlotUVCoordinates,uvTable,fileName=nightDir+files(l),
                                noTitle=1,noLabels=1,color=colors(l%7), width=width;
                            Bmax = max(abs(uvTable(1,),uvTable(2,)));
                            if(Bmax>liMax)
                                liMax = Bmax;
                        }
                    }
                }
            }
        }
    }

    w = 0;
    for(k=1;k<=nStars;k++)
    {
        if(anyof(id==toUse(k)))
        {
            w++;
            plsys,w;
            if(plotName==1)
                plt,yocoStrReplace(starNames((toUse)(k))),liMax*0.9,liMax*0.9,
                    tosys=1,justify="LN";

            limits,liMax*1.1,-liMax*1.1,-liMax*1.1,liMax*1.1;
            yocoAstroPlotNorthEast,-liMax,-liMax,0.2*liMax;
        }
    }

    if(plotLabels==1)
        yocoNmXytitles,"U (m)","V (m)",[0.02,0.02];
    write,"Saving ps file to ",nightDir+"/"+nightName+sum("_"+namesPlotted)+"_UV_PLANE.ps";
    hcps,nightDir+"/"+nightName+sum("_"+namesPlotted)+"_UV_PLANE.ps";
  
}


/***************************************************************************/

func amdlibPlotUV(nightDir=, plotCals=, width=, win=, star=, plotName=, color=, kill=, plotLabels=)
    /* DOCUMENT amdlibPlotUV(nightDir=, plotCals=, width=, win=, star=, plotName=, color=, kill=, plotLabels=)

       DESCRIPTION
       From a night directory, search for all available UV data and plots
       the corresponding UV points for each star observed.

       PARAMETERS
       - nightDir  : 
       - plotCals  : 
       - width     : 
       - win       : 
       - star      : 
       - plotName  : 
       - color     : 
       - kill      : 
       - plotLabels: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(kill))
        kill=1;

    if(is_void(plotLabels))
        plotLabels = 1;

    if(is_void(plotName))
        plotLabels = 1;

    if(is_void(win))
        win = 1;
    /* Select directory containing files to be treated */
    message = "Choose the product directory (OI_DATA)"; 
    if (amdlibFileChooser(message, nightDir) == 0)
    {
        return 0;
    }

    if(is_void(verbose))
        verbose=1;

    if(is_void(plotCals))
        plotCals = 0;

    logFile = [];
    if(is_void(logFile))
    {
        amdlibGetLogName, nightDir, logFile;
        logFile = nightDir + string(logFile);
    }

    nightName = yocoStrSplit(nightDir,"/")(-1);

    logTable = titles = [];

    amdlibReadLog, logFile, logTable, titles;

    tabFiles = logTable(where(titles=="fileName"), );
    tabDates = logTable(where(titles=="date_obs"), );
    tabNr    = logTable(where(titles=="calib_nr"), );
    proTypes = logTable(where(titles=="pro_catg"), );
    tabTypes = logTable(where(titles=="obs_catg"), );
    OBJTypes = logTable(where(titles=="obs_type"), );
    tabIds   = logTable(where(titles=="P2VM_ID"), );
    tabOBsNr = logTable(where(titles=="OB_nr"), );
    tabTplNr = logTable(where(titles=="Template_nr"), );
    tabNames = logTable(where(titles=="object_name"), );

    starNames = yocoStrRemoveMultiple(tabNames);
    starNames = starNames(where(starNames!="-"));

    id = [];
    if(!is_void(star))
    {
        h=1;
        while(h <= numberof(star))
        {
            grow, id, where(starNames==star(h));
            h++;
        }
    }
    else
    {
        id = indgen(numberof(starNames));
    }

    nStars = numberof(starNames);
    if(nStars==0)
    {
        write,"!!! SORRY !!! No stars observed in this night !!!";
        return 0;
    }

    len = max(strlen(starNames));

    rightOnes = where(OBJTypes=="OBJECT");
    if(numberof(rightOnes)==0)
    {
        write,"!!! SORRY !!! No stars observed in this night !!!";
        return 0;
    }

    kolorz = array("",nStars);
    for(k=1;k<=nStars;k++)
    {
        types = tabTypes(rightOnes)(where(tabNames(rightOnes)==starNames(k)));

        if(is_void(types))
            kolorz(k)="undefined";
        else
        {
            types = types(where(types!="-"));

            if(types(1) == "SCIENCE")
                kolorz(k)="science";
            else if(types(1) == "CALIB")
                kolorz(k)="calib";
        }
    }

    if(plotCals==0)
        toUse = where(kolorz=="science");
    else
    {
        toUse = indgen(nStars);
        for(k=1;k<=nStars;k++)
            kolorz(k)="science";
    }

    nStars = numberof(toUse);
    if(nStars==0)
    {
        write,"!!! SORRY !!! No stars observed in this night !!!";
        return 0;
    }

    vpSize = int(ceil(sqrt(nStars)));

    if(!is_void(star))
        vpSize = 1;

    if((vpSize^2-nStars) > ((vpSize+1)*(vpSize-1)-nStars))
    {
        vpSize1 = vpSize+1;
        vpSize2 = vpSize-1;
    }
    else
    {
        vpSize1 = vpSize2 = vpSize;
    }

    if(kill==1)
    {
        winkill,win;
        yocoNmCreate,win,vpSize,vpSize,square=1,wait=1, width=838,height=1825, fx=1,fy=1;
    }
    liMax = 0;

    w = 0;
    for(k=1;k<=nStars;k++)
    {
        if(anyof(id==k))
        {
            w++;
            types = tabTypes(rightOnes)(where(tabNames(rightOnes)==
                                              starNames((toUse)(k))));
            files = tabFiles(rightOnes)(where(tabNames(rightOnes)==
                                              starNames((toUse)(k))));
            nFiles = numberof(files);
            plsys,w;

            if(!is_void(types))
            {
                types = types(where(types!="-"));

                if(types(1) == "SCIENCE")
                {
                    write,"plotting "+yocoStrReplace(where(starNames(toUse==k)));
                    tliMax = 0;
                    liMin = 1e99;
                    paMax = -1e99;
                    paMin = 1e99;
                    for(l=1;l<=nFiles;l++)
                    {
                        uvTable = [];
                        if(is_void(color))
                            col = colors(l%7);
                        else if(color=="angle")
                            col="angle";
                        else
                            col=color;

                        amdlibPlotUVCoordinates,uvTable,
                            fileName=nightDir+files(l),
                            noTitle=1,noLabels=1,color=col,width=width;

                        Bmax  = max(abs(uvTable(1,),uvTable(2,)));
                        Bmin  = min(abs(uvTable(1,),uvTable(2,)));
                        PAmax = max((atan(uvTable(1,),uvTable(2,))));
                        PAmin = min((atan(uvTable(1,),uvTable(2,))));

                        if(Bmax>liMax)
                            liMax = Bmax;
                        if(Bmax>tliMax)
                            tliMax = Bmax;
                        if(Bmin<liMin)
                            liMin = Bmin;
                        if(PAmax>paMax)
                            paMax = PAmax;
                        if(PAmin<paMin)
                            paMin = PAmin;
                    }
                    if(paMin<0)
                        paMin = paMin+pi;
                    if(paMax<0)
                        paMax = paMax+pi;

                    write, "Range of baselines (m) :", liMin, tliMax;
                    write, "Range of position angles (degrees):", rad2deg*paMin, rad2deg*paMax;

                    limits,square=1;
                }
                else
                {
                    if(plotCals==1)
                    {
                        for(l=1;l<=nFiles;l++)
                        {
                            uvTable = [];
                            amdlibPlotUVCoordinates,uvTable,fileName=nightDir+files(l),
                                noTitle=1,noLabels=1,color=colors(l%7), width=width;
                            Bmax = max(abs(uvTable(1,),uvTable(2,)));
                            if(Bmax>liMax)
                                liMax = Bmax;
                        }
                    }
                }
            }
        }
    }

    w = 0;
    for(k=1;k<=nStars;k++)
    {
        if(anyof(id==k))
        {
            w++;
            plsys,w;
            if(plotName==1)
                plt,yocoStrReplace(starNames(were(toUse==k))),0,liMax*0.7,
                    tosys=1,justify="CN";

            limits,-liMax*1.1,liMax*1.1,-liMax*1.1,liMax*1.1;
        }
    }

    if(plotLabels==1)
        yocoNmXytitles,"U (m)","V (m)",[0.02,0.02];
    hcps,nightDir+"/"+nightName+"_UV_PLANE.ps";

    
    stopskjha()
        }

/***************************************************************************/

func LSperiod(x, y, ofac, hifac, &px, &py, &nout, &jmax, &prob)
    /* DOCUMENT LSperiod(x, y, ofac, hifac, &px, &py, &nout, &jmax, &prob)

       DESCRIPTION
       Implements the Lomb-Scargle periodogram of unevenly spaced datasets,
       as described in Numerical recipes page 688. This is meant to be
       used on datasets with less than 60 to 100 datapoints

       PARAMETERS
       - x    : (unevenly spaced) ordinates
       - y    : values of the periodic function
       - ofac : oversampling parameter (typical value is 4)
       - hifac: maximum frequency in Nyquist frequency units
       - px   : power spectrum ordinate
       - py   : power spectrum
       - nout : number of elemens in p and py
       - jmax : py(jmax) is the maximum element in py
       - prob : estimate of the significance of py(jmax)

       RETURN VALUES
       Returns the Lomb-Scargle periodogram, i.e. the power spectrum of
       the discrete Fourier Transform of an unevenly spaced dataset.

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    n=numberof(x);

    wi = wpi = wpr = wr = array(0.0,n);

    nout = int(0.5*ofac*hifac*n);
    px = array(0.0,nout);
    py = array(0.0,nout);

    ave = y(avg);
    var = avg((ave-y)^2);

    if (var == 0.0)
        readableError,"zero variance in period";

    xmax = max(x);
    xmin = min(x);

    xdif = xmax - xmin;
    xave = 0.5*(xmax+xmin);
    pymax=0.0;
    pnow=1.0 / (xdif * ofac);

    arg = 2 * pi * ((x - xave) * pnow);
    wpr = -2.0*sin(0.5*arg)^2;
    wpi = sin(arg);
    wr = cos(arg);
    wi = wpi;

    for (i=1;i<=nout;i++)
    {
        px(i) = pnow;
        sumsh = sum(wr * wi);
        sumc = sum( (wr - wi) * (wr + wi) );

        wtau = 0.5*atan(2.0*sumsh,sumc);
        swtau = sin(wtau);
        cwtau = cos(wtau);
        sums = sumc = sumsy = sumcy = 0.0;

        s = wi;
        c = wr;
        ss = s*cwtau - c*swtau;
        cc = c*cwtau + s*swtau;
        sums = sum(ss*ss);
        sumc = sum(cc*cc);
        yy = y - ave;
        sumsy = sum(yy*ss);
        sumcy = sum(yy*cc);
        wtemp = wr;
        wr = (wtemp*wpr - wi*wpi) + wr;
        wi = (wi*wpr + wtemp*wpi) + wi;

        py(i)=0.5*(sumcy*sumcy/(sumc+(sumc==0))+
                   sumsy*sumsy/(sums+(sums==0)))/var;
        if (py(i) >= pymax)
        {
            pymax = py(i);
            jmax=i;
        }
        pnow += 1.0/(ofac*xdif);
    }
    expy=exp(-pymax);
    effm=2.0*nout/ofac;
    prob=effm*expy;
    if (prob > 0.01)
        prob=1.0-(1.0-expy)^effm;
} 

/***************************************************************************/

func amdlibListStars(&allStars, &allTypes, nightDir=)
    /* DOCUMENT amdlibListStars(&allStars, &allTypes, nightDir=)

       DESCRIPTION

       PARAMETERS
       - allStars: 
       - allTypes: 
       - nightDir: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    /* Select directory containing files to be treated */
    message = "Choose the product directory (OI_DATA)"; 
    if (amdlibFileChooser(message, nightDir) == 0)
    {
        return 0;
    }

    if(is_void(verbose))
        verbose=1;

    logFile = [];
    if(is_void(logFile))
    {
        amdlibGetLogName, nightDir, logFile;
        logFile = nightDir + string(logFile);
    }

    logTable = titles = [];

    amdlibReadLog, logFile, logTable, titles;
    if(is_void(titles))
    {
        write,"No log, exiting...";
        return 0;
    }

    tabFiles = logTable(where(titles=="fileName"), );
    tabDates = logTable(where(titles=="date_obs"), );
    tabNr    = logTable(where(titles=="calib_nr"), );
    proTypes = logTable(where(titles=="pro_catg"), );
    tabTypes = logTable(where(titles=="obs_catg"), );
    OBJTypes = logTable(where(titles=="obs_type"), );
    tabIds   = logTable(where(titles=="P2VM_ID"), );
    tabOBsNr = logTable(where(titles=="OB_nr"), );
    tabTplNr = logTable(where(titles=="Template_nr"), );
    tabNames = logTable(where(titles=="object_name"), );
    insModes =logTable(where(titles=="insMode"), );

    oiFitsFiles = where((insModes=="DUMMY")| // Compatibility with samSuffit
                        (proTypes=="SCIENCE_CALIBRATED")|
                        (proTypes=="SCIENCE_REDUCED")|
                        (proTypes=="CALIB_REDUCED")|
                        (proTypes=="TRANSFER_FUNCTION"));

    for(k=1;k<=numberof(oiFitsFiles);k++)
    {
        nam = readOI_FITS_star_name(tabFiles(oiFitsFiles)(k));
        if(!is_void(nam))
        {
            tabNames(oiFitsFiles(k)) = nam;
            OBJTypes(oiFitsFiles(k)) = "OBJECT";
        }
    }

    allStars = yocoStrRemoveMultiple(tabNames, order);
    newOrder = where(allStars!="-");

    if(numberof(allStars)==0)
    {
        write,"No stars observed during this night. Exiting...";
        return 0;
    }

    nCals = numberof(allStars);
    len = max(strlen(allStars));

    rightOnes = where(OBJTypes=="OBJECT");

    if(numberof(rightOnes)==0)
    {
        write,"No stars observed during this night. Exiting...";
        return 0;
    }

    allTypes = array("",nCals);
    for(k=1;k<=nCals;k++)
    {
        allTypesTmp = tabTypes(rightOnes)(where(tabNames(rightOnes)==allStars(k)));

        if((numberof(allTypesTmp)==0)||(is_void(allTypesTmp)))
            allTypes(k)="undefined";
        else
        {
            // allTypesTmp = allTypesTmp(where(allTypesTmp!="-"));
            allTypes(k)=allTypesTmp(1);
        }
    }
    defined = where(allTypes!="undefined");
    allStars = allStars(defined);
    allTypes = allTypes(defined);
    write,allStars,allTypes;

    return allStars;
} 

/***************************************************************************/

func amdlibListAllStars(nightDirs=)
    /* DOCUMENT amdlibListAllStars(nightDirs=)

       DESCRIPTION
       Lists the star names of a series of input directories

       PARAMETERS
       - nightDirs: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    /* Select directory containing files to be treated */
    message = "Choose the directories for which you want to build a stars database"; 
    if (amdlibFileChooser(message, nightDirs) == 0)
    {
        return 0;
    }

    currentDir = yocoStrrtok(nightDirs(1),"/")(1);

    if(is_void(verbose))
        verbose=1;

    nDirs = numberof(nightDirs);
    stars = stars2 = stars3 = types = [];

    for(k=1;k<=nDirs;k++)
    {
        write,"Looking for "+nightDirs(k);
        amdlibListStars, allStars, allTypes, nightDir=nightDirs(k)+"/";
        if(numberof(allStars)!=0)
        {
            bla = yocoStrReplace(strcase(1,allStars),["-","_"],["",""]);
            bla2 = allStars;
            bla3 = allTypes;
            grow,stars,&bla ;
            grow,stars2,bla ;
            grow,stars3,bla2;
            grow,types,bla3;
        }
    }

    starList  = yocoStrRemoveMultiple(stars2,order);
    starNames = stars3(order);
    starTypes = types(order);
    N = numberof(starList);
    fh = open(currentDir+"starsList.txt","w");
    for(j=1;j<=N;j++)
    {
        write,fh,starNames(j)+" "+starTypes(j);
        for(k=1;k<=nDirs;k++)
        {
            if(numberof(*stars(k))!=0)
            {
                if(anyof(strmatch(*stars(k), starList(j), 0)))
                {
                    write,fh," ",nightDirs(k);
                }
            }
        }
    }
    close,fh;
    cd,"..";
}


/***************************************************************************/

func amdlibPlotV2fSpFreq(void, inputOiFile=, kill=, win=, skipLeft=, skipRight=, plotErr=, color=, posAng=, xtitle=, ytitle=, freqFact=, sqroot=, sizebar=, xm=, xM=, ym=, yM=)
    /* DOCUMENT amdlibPlotV2fSpFreq(inputOiFile=, kill=, win=, skipLeft=, skipRight=, plotErr=, color=, posAng=, xtitle=, ytitle=, freqFact=, sqroot=, sizebar=, xm=, xM=, ym=, yM=)

       DESCRIPTION
       Plots squared Visibilities in function of spatial frequencies

       PARAMETERS
       - inputOiFile: An averaged OI data file
       - kill       : 
       - win        : 
       - skipLeft   : 
       - skipRight  : 
       - plotErr    : 
       - color      : 
       - posAng     : 
       - xtitle     : 
       - ytitle     : 
       - freqFact   : 
       - sqroot     : 
       - sizebar    : 
       - xm         : 
       - xM         : 
       - ym         : 
       - yM         : 

       RESULTS
       The visibilities are plotted in function of spatial frequencies,
       i.e. B/lambda. This script is useful for checking calibration stars
       visibilities

       EXAMPLES
       type
       amdlibPlotV2fSpFreq
       and browse to an AVG OI DATA file

       SEE ALSO:
    */
{
    yocoLogTrace,"amdlibPlotV2fSpFreq()";
    if (!is_void(void)) yocoLogWarning,"void should be void !";

    if(is_void(freqFact))
        freqFact = 1.0;

    /* Read the default window size */
    amdlibCheckGraphicsDpi, graphicsDpi;

    // legend offset
    legendOffset = [0.02,0.02];

    if (is_void(win))
        win = 1;

    if (is_void(kill))
        kill=1;
    /* All the Yorick opened windows are killed */
    if (kill==1)
        winkill;

    /* If the function plotRawData, is called without argument, a file browser
     * popups to choose the correct file */
    message = "OI FITS FILE\nEnter the AMBER OI-FITS file to plot";
    if (amdlibFileChooser(message, inputOiFile) == 0)
    {
        return 0;
    }

    window, win;

    if(kill==1)
        window, win, style="amdlibLargeBox.gs", width=638*graphicsDpi/75,
            height=425*graphicsDpi/75;
    else
        window,win;

    nFiles = numberof(inputOiFile);
    for(kFile=1;kFile<=nFiles;kFile++)
    {

        wlen = bandWidth = time = fringeSNR = sqVis = sqVisErr =
            uvCoord = bandFlag = pistonOPDArray = sigmaPistonArray =
            diffVisAmp = diffVisAmpErr = diffVisPhase = diffVisPhaseErr =
            vis3Amp = vis3AmpErr = vis3Phase = vis3PhaseErr = cpxVis =
            cpxVisErr = visCovRI = uv1Coord = uv2Coord = fluxSumPiPj =
            fluxRatPiPj = fluxProdPiPj = spectrum = spectrumErr = [];

        status = amdlibLoadOiData(wlen, bandWidth, time, fringeSNR,
                                  sqVis, sqVisErr, uvCoord, bandFlag,
                                  pistonOPDArray, sigmaPistonArray,
                                  diffVisAmp, diffVisAmpErr, diffVisPhase,
                                  diffVisPhaseErr, vis3Amp, vis3AmpErr,
                                  vis3Phase, vis3PhaseErr, cpxVis,
                                  cpxVisErr, visCovRI, uv1Coord, uv2Coord,
                                  fluxSumPiPj, fluxRatPiPj, fluxProdPiPj,
                                  spectrum, spectrumErr,
                                  inputOiFile=inputOiFile(kFile));

        /* Get the dimensions of the arrays */ 
        nbBases = dimsof(cpxVis)(2);
        nbWlen = dimsof(cpxVis)(3);
        nbFrames = dimsof(cpxVis)(4);

        dat = sqVis;
        datErr = sqVisErr;
        if(sqroot==1)
        {
            dat = sqrt(sqVis);
            datErr = sqVisErr/(2*sqrt(sqVis));
        }


        // dat = diffVisPhase;
        // datErr = diffVisPhaseErr;

        if (nbBases == 1)
        {
            nbTel = 2;
        }
        if (nbBases == 3)
        {
            nbTel = 3;
        }
        // fma;

        uvTable = transpose(grow(array(uvCoord.u,1),
                                 array(uvCoord.v,1)),[1,2,3]);
        R = abs(uvTable(1,),uvTable(2,));
        PA = atan(uvTable(2,),uvTable(1,));
        if(!is_void(posAng))
        {
            R = R *sin(PA+posAng);
        }

        spFreq = R(,-,) / wlen(skipLeft:skipRight)(-,,-) * freqFact;
        for (iBase=1; iBase <= nbBases; iBase++)
        {
            for(kFrame=1;kFrame<=nbFrames;kFrame++)
            {
                if(plotErr==1)
                {
                    if(is_void(color))
                        color=(-iBase-4);

                    yocoPlotWithErrBars,dat(iBase,skipLeft:skipRight,kFrame),
                        datErr(iBase,skipLeft:skipRight,kFrame),
                        spFreq(iBase,skipLeft:skipRight,kFrame),
                        color=color,sizebar=sizebar//2*max(spFreq)/numberof(spFreq);

                        yocoPlotWithErrBars,dat(iBase,skipLeft:skipRight,kFrame),
                        datErr(iBase,skipLeft:skipRight,kFrame),
                        -spFreq(iBase,skipLeft:skipRight,kFrame),
                        color=color,sizebar=sizebar//r=2*max(spFreq)/numberof(spFreq);
                        }
                else
                {
                    if(is_void(color))
                        color=(-iBase-4);

                    if(numberof(dat(iBase,skipLeft:skipRight,kFrame))==1)
                    {
                        plg,dat(iBase,skipLeft:skipRight,kFrame),
                            spFreq(iBase,skipLeft:skipRight,kFrame),
                            color=color, marks=1,marker='\2', type="none";

                        plg,dat(iBase,skipLeft:skipRight,kFrame),
                            -spFreq(iBase,skipLeft:skipRight,kFrame),
                            color=color, marks=1,marker='\2', type="none";
                    }
                    else
                    {
                        plg,dat(iBase,skipLeft:skipRight,kFrame),
                            spFreq(iBase,skipLeft:skipRight,kFrame),
                            color=color;

                        plg,dat(iBase,skipLeft:skipRight,kFrame),
                            -spFreq(iBase,skipLeft:skipRight,kFrame),
                            color=color;
                    }
                }
            }
        }
        if(is_void(xm))
            xm = 0;
        if(is_void(xM))
            xM = [];
        if(is_void(ym))
            ym = -0.05;
        if(is_void(yM))
            yM = 1.05;

        limits,xm,xM,ym,yM;
    }

    if(is_void(xtitle))
        xtitle = "Spatial Frequency (cycles/rad)";
    if(is_void(ytitle))
        ytitle = "Squared Visibilities";

    xytitles,xtitle,ytitle;

    hcps,"~/Visibilities_vs_Spatial_Frequencies.ps";
    yocoLogTrace,"amdlibPlotV2fSpFreq done";
}

/***************************************************************************/

func amdlibPlotV2fUVFreq(void, inputOiFile=, kill=, win=, skipLeft=, skipRight=, cmin=, cmax=, telDiam=)
    /* DOCUMENT amdlibPlotV2fUVFreq(inputOiFile=, kill=, win=, skipLeft=, skipRight=, cmin=, cmax=, telDiam=)

       DESCRIPTION
       Plots squared Visibilities as a function of spatial frequencies

       PARAMETERS
       - inputOiFile: An averaged OI data file
       - kill       : 
       - win        : 
       - skipLeft   : 
       - skipRight  : 
       - cmin       : 
       - cmax       : 
       - telDiam    : 

       RESULTS
       The visibilities are plotted in function of spatial frequencies,
       i.e. B/lambda. This script is useful for checking calibration stars
       visibilities

       EXAMPLES
       type
       amdlibPlotV2fSpFreq
       and browse to an AVG OI DATA file

       SEE ALSO:
    */
{
    yocoLogTrace,"amdlibPlotV2fSpFreq()";
    if (!is_void(void)) yocoLogWarning,"void should be void !";

    /* Read the default window size */
    amdlibCheckGraphicsDpi, graphicsDpi;

    // legend offset
    legendOffset = [0.02,0.02];

    if (is_void(win))
        win = 1;

    if (is_void(cmin))
        cmin = 0;

    if (is_void(cmax))
        cmax = 1;

    if (is_void(skipLeft))
        skipLeft = 1;

    if (is_void(skipRight))
        skipRight = 0;

    if (is_void(kill))
        kill=1;

    /* All the Yorick opened windows are killed */
    if (kill==1)
        winkill;

    /* If the function plotRawData, is called without argument, a file browser
     * popups to choose the correct file */
    message = "OI FITS FILE\nEnter the AMBER OI-FITS file to plot";
    if (amdlibFileChooser(message, inputOiFile) == 0)
    {
        return 0;
    }
    wlen = bandWidth = time = fringeSNR = sqVis = sqVisErr =
        uvCoord = bandFlag = pistonOPDArray = sigmaPistonArray =
        diffVisAmp = diffVisAmpErr = diffVisPhase = diffVisPhaseErr =
        vis3Amp = vis3AmpErr = vis3Phase = vis3PhaseErr = cpxVis =
        cpxVisErr = visCovRI = uv1Coord = uv2Coord = fluxSumPiPj =
        fluxRatPiPj = fluxProdPiPj = spectrum = spectrumErr = [];

    status = amdlibLoadOiData(wlen, bandWidth, time, fringeSNR,
                              sqVis, sqVisErr, uvCoord, bandFlag,
                              pistonOPDArray, sigmaPistonArray,
                              diffVisAmp, diffVisAmpErr, diffVisPhase,
                              diffVisPhaseErr, vis3Amp, vis3AmpErr,
                              vis3Phase, vis3PhaseErr, cpxVis,
                              cpxVisErr, visCovRI, uv1Coord, uv2Coord,
                              fluxSumPiPj, fluxRatPiPj, fluxProdPiPj,
                              spectrum, spectrumErr, inputOiFile=inputOiFile);

    /* Get the dimensions of the arrays */ 
    nbBases = dimsof(cpxVis)(2);
    nbWlen = dimsof(cpxVis)(3);
    nbFrames = dimsof(cpxVis)(4);

    if (nbBases == 1)
    {
        nbTel = 2;
    }
    if (nbBases == 3)
    {
        nbTel = 3;
    }

    if(kill==1)
        window, win, width=425*graphicsDpi/75,
            height=425*graphicsDpi/75;
    else
        window,win;
    // fma;

    uvTable = transpose(grow(array(uvCoord.u,1),
                             array(uvCoord.v,1)),[1,2,3]);
    R = sqrt((uvTable(1,)^2)+(uvTable(2,)^2));

    UTdiam = 2;
    nPointsPerCircle = 30;
    param = span(-pi, pi, nPointsPerCircle);

    for (iBase=1; iBase <= nbBases; iBase++)
    {
        sv = sqVis(iBase,skipLeft:skipRight,1);
        toto = digitize(sv,span(cmin,cmax,255));

        for(iWlen=1;iWlen<=nbWlen+skipRight-skipLeft-1;iWlen++)
        {
            colorz = array(array(char(toto(iWlen)),3),1);
            plg, -(UTdiam/2*sin(param)+uvTable(1,iBase,1))/wlen(iWlen),
                -( UTdiam/2*cos(param)+uvTable(2,iBase,1))/wlen(iWlen), 
                color=[64,64,64];
            plg, (UTdiam/2*sin(param)+uvTable(1,iBase,1))/wlen(iWlen),
                ( UTdiam/2*cos(param)+uvTable(2,iBase,1))/wlen(iWlen), 
                color=[64,64,64];

            plfp, colorz,
                (UTdiam/2*sin(param)+uvTable(1,iBase,1))/wlen(iWlen),
                ( UTdiam/2*cos(param)+uvTable(2,iBase,1))/wlen(iWlen), 
                nPointsPerCircle,cmin=cmin,cmax=cmax;
            plfp, colorz,
                -(UTdiam/2*sin(param)+uvTable(1,iBase,1))/wlen(iWlen),
                -( UTdiam/2*cos(param)+uvTable(2,iBase,1))/wlen(iWlen), 
                nPointsPerCircle,cmin=cmin,cmax=cmax;
        }
    }
    //limits,0,,-0.1,1.1;

    yocoLogTrace,"amdlibPlotV2fSpFreq done";
    //sdh()
}

/***************************************************************************/

func changeAMBERNamesToOrig(workDir=, ext=)
    /* DOCUMENT changeAMBERNamesToOrig(workDir=, ext=)

       DESCRIPTION 
       From an AMBER file whose name is of the form
       AMBER.YYYY-MM-DDThh:mm:ss.xxx.fits, changes its name to
       AMBER_XXXXXX_XXXXX_XXXX.fits, which is more related to the
       contained data in the file.

       CAUTIONS 
       !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
       !!! PLEASE NOTE THAT THIS OPERATION IS IRREVERSIBLE !!!
       !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

       You can use changeNamesReverse to do the contrary but it does not
       changes back all the files names.

       SEE ALSO:changeNamesReverse,changeNamesForward
    */
{
    if(amdlibFileChooser("Please choose which directory you want to use \n(right-click to exit)",workDir) == 0)
        return 0;

    logName = [];
    amdlibGetLogName, workDir, logName;
    amdlibReadLog, workDir+"/"+logName, valuesTab, colTitles;

    files = valuesTab(where(colTitles=="fileName")(1),);
    origNames = valuesTab(where(colTitles=="originalFileName")(1),);
    nfiles = numberof(files);

    for(i=1;i<=numberof(origNames);i++)
    {
        yocoFileSplitName,files(i),d,f,ext;
        yocoFileSplitName,origNames(i),d,ori;
        horrigueName = ori + ext;

        yocoGuiProgressBox,i,kill=(i<=1),
            barTitle="Renaming files...",
            minAmount=0,maxAmount=nfiles,
            minLabel="0",maxLabel=pr1(nfiles)+" files", win=7,
            dpi=graphicsDpi;

        if(horrigueName != "-")
            if(files(i) != horrigueName)
            {
                write,"renaming "+files(i)+" to "+horrigueName;
                rename,files(i),horrigueName;
            }
    }
}

/************************************************************************/

func changeAMBERNamesToDVD(workDir=, ext=)
    /* DOCUMENT changeAMBERNamesToDVD(workDir=, ext=)

       DESCRIPTION 
       From an AMBER file whose name is of the form
       AMBER_XXXXXX_XXXXX_XXXX.fits, changes its name to
       AMBER.YYYY-MM-DDThh:mm:ss.xxx.fits which is the name of the file
       in the ESO archive.
       Please note that this does not work for P2VM files.

       SEE ALSO:changeNamesReverse,changeNamesForward
    */
{
    if(amdlibFileChooser("Please choose which directory you want to use \n(right-click to exit)",workDir) == 0)
        return 0;

    logName = [];
    amdlibGetLogName, workDir, logName;
    amdlibReadLog, workDir+"/"+logName, valuesTab, colTitles;

    files   = valuesTab(where(colTitles=="fileName")(1),);
    dateObs = valuesTab(where(colTitles=="date_obs")(1),);
    proctg  = valuesTab(where(colTitles=="pro_catg")(1),);

    trail = yocoStrReplace(strtok(dateObs,"."),":","_");
    nfiles = numberof(files);

    for(i=1;i<=nfiles;i++)
    {
        if (proctg(i)!="P2VM_REDUCED")
            if (_amdlibGetKwdVals(files(i),"INSTRUME")(1)==
                "AMBER")
            {
                yocoGuiProgressBox,i,kill=(i<=1),
                    barTitle="Renaming files...",
                    minAmount=0,maxAmount=nfiles,
                    minLabel="0",maxLabel=pr1(nfiles)+" files", win=7,
                    dpi=graphicsDpi;
            
                yocoFileSplitName,files(i),d,f,ext;
                archiveName = "AMBER." + trail(1,i) + "." +
                    swrite(format="%03d",int(yocoStr2Double(trail(2,i)(1))/10)) + ext;
                if(dateObs(i)!="-")
                {
                    if(files(i)!=archiveName)
                    {
                        write,"renaming "+files(i)+" to "+archiveName;
                        rename, workDir+"/"+files(i), workDir+"/"+archiveName;
                    }
                }
            }
            else
            {
                write,files(i)+" Not an AMBER file, skipping...";
            }
    }
} 


/************************************************************************/

func changeVISIRNamesToDVD(workDir=, ext=)
    /* DOCUMENT changeVISIRNamesToDVD(workDir=, ext=)

       DESCRIPTION 
       From an VISIR file whose name is of the form
       VISIR_XXXXXX_XXXXX_XXXX.fits, changes its name to
       VISIR.YYYY-MM-DDThh:mm:ss.xxx.fits which is the name of the file
       in the ESO archive.
       Please note that this does not work for P2VM files.

       SEE ALSO:changeNamesReverse,changeNamesForward
    */
{
    if(amdlibFileChooser("Please choose which directory you want to use \n(right-click to exit)",workDir) == 0)
        return 0;

    logName = [];
    amdlibGetLogName, workDir, logName;
    amdlibReadLog, workDir+"/"+logName, valuesTab, colTitles;

    files   = valuesTab(where(colTitles=="fileName")(1),);
    dateObs = valuesTab(where(colTitles=="date_obs")(1),);
    proctg  = valuesTab(where(colTitles=="pro_catg")(1),);

    trail = yocoStrReplace(strtok(dateObs,"."),":","_");
    nfiles = numberof(files);

    for(i=1;i<=nfiles;i++)
    {
        if (proctg(i)!="P2VM_REDUCED")
            if (_amdlibGetKwdVals(files(i),"INSTRUME")(1)==
                "VISIR")
            {
                yocoGuiProgressBox,i,kill=(i<=1),
                    barTitle="Renaming files...",
                    minAmount=0,maxAmount=nfiles,
                    minLabel="0",maxLabel=pr1(nfiles)+" files", win=7,
                    dpi=graphicsDpi;
            
                yocoFileSplitName,files(i),d,f,ext;
                archiveName = "VISIR." + trail(1,i) + "." +
                    swrite(format="%03d",int(yocoStr2Double(trail(2,i)(1))/10)) + ext;
                if(dateObs(i)!="-")
                {
                    if(files(i)!=archiveName)
                    {
                        write,"renaming "+files(i)+" to "+archiveName;
                        rename, workDir+"/"+files(i), workDir+"/"+archiveName;
                    }
                }
            }
            else
            {
                write,files(i)+" Not an VISIR file, skipping...";
            }
    }
} 


/***************************************************************************/

func changeAllAMBERNamesToDVD(nightDirs=)
    /* DOCUMENT changeAllAMBERNamesToDVD(nightDirs=)

       DESCRIPTION

       PARAMETERS
       - nightDirs: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    /* Select directory containing files to be treated */
    message = "Choose the directories for which you want to build a stars database"; 
    if (amdlibFileChooser(message, nightDirs) == 0)
    {
        return 0;
    }

    currentDir = sum(yocoStrSplit(nightDirs(1),"/")(:-1)+"/");

    if(is_void(verbose))
        verbose=1;

    nDirs = numberof(nightDirs);
    stars = stars2 = [];
    for(k=1;k<=nDirs;k++)
    {
        write,nightDirs(k);
        changeAMBERNamesToDVD,workDir=nightDirs(k);
    }
}

/***************************************************************************/

func averageDataFMi(inputRawOiFile=, errors=)
    /* DOCUMENT averageDataFMi(inputRawOiFile=, errors=)

       DESCRIPTION

       PARAMETERS
       - inputRawOiFile: 
       - errors        : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    inputOiFile = outputFileName = [];
    /* Select file to be treated */
    message = "Choose the file you want to average"; 
    if (amdlibFileChooser(message, inputOiFile) == 0)
    {
        return 0;
    }

    if(is_void(outputFileName))
        outputFileName = yocoStrReplace(inputOiFile,["RAW_COR","RAW"],["RAW2_COR","AVG_COR"]);

    // Test if gzipped
    yocoFileSplitName,outputFileName,dir,name,ext;
    if (strmatch(ext,".gz"))
    {
        ext = sum(yocoStrSplit(ext,".gz"));
        recompressFile = "yes";
    }
    outputFileName = dir+name+ext;


    /* DUMBLY READING TWICE THE FILE */
    _amdlibReadVis, inoiWave, inamberPhot, inoiVis, inoiVis2, inoiVis3,
        inamberOpd, inamberInsCfg, inoiArray, inoiTarget,
        inputOiFile=inputOiFile;

    if(!amdlibLoadOiData(wlen, bandWidth, time, fringeSNR,
                         sqVis, sqVisErr, uvCoord,
                         bandFlag, pistonOPDArray,
                         sigmaPistonArray, diffVisAmp,
                         diffVisAmpErr, diffVisPhase,
                         diffVisPhaseErr, vis3Amp,
                         vis3AmpErr, vis3Phase, vis3PhaseErr,
                         cpxVis, cpxVisErr, visCovRI,
                         uv1Coord, uv2Coord, fluxSumPiPj,
                         fluxRatPiPj, fluxProdPiPj,
                         inputOiFile=inputOiFile))
    {
        yocoGuiErrorBox, "reading "+inputOiFile;//+inputSciFiles;
    }

    nbFrames = dimsof(sqVis)(4);
    nbWlen = dimsof(sqVis)(3);
    nbBases = dimsof(sqVis)(2);

    MJD = (*inoiVis.table).dateObsMJD;
    targetId = (*inoiVis.table).targetId;
    time = (*inoiVis.table).time;
    DIT = (*inoiVis.table).expTime;
    stations = (*inoiVis.table).stationIndex;

    if (nbBases == 3)
    {
        stations3 = (*inoiVis3.table).stationIndex;
        U1 = transpose(array(uv1Coord.u,1));
        V1 = transpose(array(uv1Coord.v,1));
        U2 = transpose(array(uv2Coord.u,1));
        V2 = transpose(array(uv2Coord.v,1));
    }

    U = uvCoord.u;
    V = uvCoord.v;

    // Read the additional column if it exists
    inFh = cfitsio_open(inputOiFile, "r");
    hduz = cfitsio_list(inFh);
    cfitsio_goto_hdu,inFh,"OI_VIS";
    table = cfitsio_read_bintable(inFh,titles); // read the binary table
    cfitsio_goto_hdu,inFh,"OI_VIS2";
    table2 = cfitsio_read_bintable(inFh,titles2); // read the binary table
    cfitsio_goto_hdu,inFh,"OI_VIS3";
    table3 = cfitsio_read_bintable(inFh,titles3); // read the binary table
    cfitsio_close,inFh;

    if(anyof(titles=="ATT_FACT"))
        att = transpose(reform(*table(where(titles=="ATT_FACT")(1)),
                               [3,nbWlen,nbBases,nbFrames]),[1,2]);
    else
        att = array(1.0,nbBases,nbWlen,nbFrames);

    F2 = sqVis * fluxProdPiPj;
    v2Avg = array(F2(..,sum) / (att2 * fluxProdPiPj)(..,sum),1);
    v2AvgErr = array((F2 / (att2 * fluxProdPiPj))(..,rms),1);


    if(anyof(titles2=="ATT_FACT"))
        att2 = transpose(reform(*table(where(titles2=="ATT_FACT")(1)),
                                [3,nbWlen,nbBases,nbFrames]),[1,2]);
    else
        att2 = array(1.0,nbBases,nbWlen,nbFrames);

    F2 = sqVis * fluxProdPiPj;
    v2Avg = array(F2(..,sum) / (att2 * fluxProdPiPj)(..,sum),1);
    v2AvgErr = array((F2 / (att2 * fluxProdPiPj))(..,rms),1);

    if(anyof(titles3=="ATT_FACT"))
        att2 = transpose(reform(*table(where(titles3=="ATT_FACT")(1)),
                                [3,nbWlen,nbBases,nbFrames]),[1,2]);
    else
        att2 = array(1.0,nbBases,nbWlen,nbFrames);

    F2 = sqVis * fluxProdPiPj;
    v2Avg = array(F2(..,sum) / (att2 * fluxProdPiPj)(..,sum),1);
    v2AvgErr = array((F2 / (att2 * fluxProdPiPj))(..,rms),1); 




    amdlibSetWave, newWave, wlen*yocoAstroSI.m/yocoAstroSI.nm, bandWidth*yocoAstroSI.m/yocoAstroSI.nm;

    // amdlibSetOiVis, newVis, cpxVis, cpxVisErr, visCovRI, 
    // diffVisAmp, diffVisAmpErr, 
    // diffVisPhase, diffVisPhaseErr,
    // fringeSNR, targetId, time, MJD, DIT, U, V, stations;

    // pouet()
    amdlibSetOiVis2, newVis2, v2Avg, v2AvgErr,
        targetId, time, MJD, DIT, U, V, stations;

    // if (nbBases == 3)
    // {
    // amdlibSetOiVis3, newTriple3,
    // vis3Amp, vis3AmpErr, vis3Phase, vis3PhaseErr,
    // targetId, time, MJD, DIT, U1, U2, V1, V2, stations3;
    // }

    // amdlibSetPhotometry, newPhot, 
    // fluxSumPiPj, fluxRatPiPj, fluxProdPiPj;

    // amdlibSetPiston, newPiston, bandFlag,
    // pistonOPDArray, sigmaPistonArray;

    write, "writing file "+outputFileName;
    // status = amdlibWriteOiFile(outputFileName,
    // &obsInsCfg, &oiArray, 
    // &oiTarget, &newWave, &newPhot, &newVis, 
    // &newVis2, &newTriple3, &newPiston,
    // pointer(nil));
    status = amdlibWriteOiFile(outputFileName,
                               pointer(nil), &inoiArray, 
                               &inoiTarget, &newWave,
                               pointer(nil), pointer(nil), 
                               &newVis2, pointer(nil), pointer(nil),
                               pointer(nil));

    if (status == amdlibFAILURE)
    {
        yocoGuiErrorBox,"Could not save result file\n";
    }

    // dump bin tables
    write, "dump bin table";
    _amdlibDumpAllKeywords, 1, inputOiFile, outputFileName;

} 

/***************************************************************************/

func filterOiFile(inputOiFile=, outputOiFile=, wlenRange=, keepBase=, wstretch=, woffset=, avgWlen=, nSigma=, crl=, crr=)
    /* DOCUMENT filterOiFile(inputOiFile=, outputOiFile=, wlenRange=, keepBase=, wstretch=, woffset=, avgWlen=, crl=, crr=)

       DESCRIPTION

       PARAMETERS
       - inputOiFile : 
       - outputOiFile: 
       - wlenRange   : 
       - keepBase    : 
       - wstretch    : 
       - woffset     : 
       - avgWlen     : 
       - crl         : 
       - crr         : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES
       To keep only the H band
       filterOiFile, woffset=-0.1e-6, wlenRange=[1.55,1.9]*1e-6
       To keep only the K band
       filterOiFile, woffset=-0.1e-6, wlenRange=[1.97,2.45]*1e-6

       SEE ALSO
    */
{
    message = "Please choose OI file"; 
    if (amdlibFileChooser(message, inputOiFile) == 0)
    {
        return 0;
    }

    //Stretch and offset if any
    if(is_void(wstretch))
        wstretch = 1.0;
    
    if(is_void(crl))
        crl = 1;
    
    if(is_void(crr))
        crr = -1;

    if(is_void(woffset))
        woffset = 0.0;

    if(is_void(wlenRange))
        wlenRange = [0,1e99];

    if(is_void(keepBase))
        keepBase = indgen(3);

    nFiles = numberof(inputOiFile);
    for(kFile=1;kFile<=nFiles;kFile++)
    {
        /* DUMBLY READING TWICE THE FILE */
        _amdlibReadVis, inoiWave, inamberPhot, inoiVis, inoiVis2, inoiVis3,
            inamberOpd, inamberInsCfg, inoiArray, inoiTarget, inSpectrum, 
            inputOiFile=inputOiFile(kFile);

        if(!amdlibLoadOiData(wlen, bandWidth, time, fringeSNR,
                             sqVis, sqVisErr, uvCoord,
                             bandFlag, pistonOPDArray,
                             sigmaPistonArray, diffVisAmp,
                             diffVisAmpErr, diffVisPhase,
                             diffVisPhaseErr, vis3Amp,
                             vis3AmpErr, vis3Phase, vis3PhaseErr,
                             cpxVis, cpxVisErr, visCovRI,
                             uv1Coord, uv2Coord, fluxSumPiPj,
                             fluxRatPiPj, fluxProdPiPj,
                             inputOiFile=inputOiFile(kFile)))
        {
            yocoGuiErrorBox, "reading "+inputOiFile(kFile);//+inputSciFiles;
        }

        /* Apply stretch and offset if any */
        wlen = wstretch * (wlen-avg(wlen)) + avg(wlen) + woffset;

        nbFrames = dimsof(sqVis)(4);
        nbWlen = dimsof(sqVis)(3);
        nbBases = dimsof(sqVis)(2);

        if(numberof(keepBase)>nbBases)
            keepBas = [1];
        else
            keepBas = keepBase;

        MJD      = (*inoiVis.table).dateObsMJD;
        targetId = (*inoiVis.table).targetId;
        time     = (*inoiVis.table).time;
        DIT      = (*inoiVis.table).expTime;
        stations = (*inoiVis.table).stationIndex;

        idx = array(1,numberof(wlen));
        idx(:crl) = 0;
        if(crr!=0)
            idx(crr:) = 0;
        
        wlenIdx = where(idx&(wlen>wlenRange(1))&(wlen<wlenRange(2)));

        // das()
            
        if (avgWlen == 1)
        {
            wlen      = array(wlen(wlenIdx)(avg),1);
            bandWidth = array(bandWidth(wlenIdx)(avg),1);
            
            cpxVis          = cpxVis(,wlenIdx,)(,avg,)(,-,);
            cpxVisErr       = cpxVisErr(,wlenIdx,)(,avg,)(,-,);
            visCovRI        = visCovRI(,wlenIdx,)(,avg,)(,-,);
            diffVisAmp      = diffVisAmp(,wlenIdx,)(,avg,)(,-,);
            diffVisAmpErr   = diffVisAmpErr(,wlenIdx,)(,avg,)(,-,);
            diffVisPhase    = diffVisPhase(,wlenIdx,)(,avg,)(,-,);
            diffVisPhaseErr = diffVisPhaseErr(,wlenIdx,)(,avg,)(,-,);
            sqVis           = sqVis(,wlenIdx,)(,avg,)(,-,);
            sqVisErr        = sqVisErr(,wlenIdx,)(,avg,)(,-,);
            if (numberof(keepBas) == 3)
            {
                vis3Amp = vis3Amp(,wlenIdx,)(,avg,)(,-,);
                vis3AmpErr = vis3AmpErr(,wlenIdx,)(,avg,)(,-,);
                vis3Phase = vis3Phase(,wlenIdx,)(,avg,)(,-,);
                vis3PhaseErr = vis3PhaseErr(,wlenIdx,)(,avg,)(,-,);
            }
            
            wlenIdx = [];
        }

        
        if (nbBases == 3)
        {
            stations3 = (*inoiVis3.table).stationIndex;
            U1 = transpose(array(uv1Coord.u,1));
            V1 = transpose(array(uv1Coord.v,1));
            U2 = transpose(array(uv2Coord.u,1));
            V2 = transpose(array(uv2Coord.v,1));
        }

        U = uvCoord.u;
        V = uvCoord.v;

        if(numberof(wlenIdx)==0)
        {
            yocoLogWarning,"Nothing to filter! Exiting...";
            return 0;
        }

        amdlibSetWave, newWave,
            wlen(wlenIdx)*yocoAstroSI.m/yocoAstroSI.nm,
            bandWidth(wlenIdx)*yocoAstroSI.m/yocoAstroSI.nm;

        amdlibSetOiVis, newVis,
            cpxVis(keepBas,wlenIdx,), cpxVisErr(keepBas,wlenIdx,),
            visCovRI(keepBas,wlenIdx,), 
            diffVisAmp(keepBas,wlenIdx,), diffVisAmpErr(keepBas,wlenIdx,), 
            diffVisPhase(keepBas,wlenIdx,), diffVisPhaseErr(keepBas,wlenIdx,),
            fringeSNR(keepBas,), targetId(keepBas,), time(keepBas,),
            MJD(keepBas,), DIT(keepBas,), U(keepBas,), V(keepBas,),
            stations(,keepBas,);

        amdlibSetOiVis2, newVis2,
            sqVis(keepBas,wlenIdx,), sqVisErr(keepBas,wlenIdx,),
            targetId(keepBas,), time(keepBas,), MJD(keepBas,), DIT(keepBas,),
            U(keepBas,), V(keepBas,), stations(,keepBas,);


        if (numberof(keepBas) == 3)
        {
            amdlibSetOiVis3, newTriple3,
                vis3Amp(,wlenIdx,), vis3AmpErr(,wlenIdx,),
                vis3Phase(,wlenIdx,), vis3PhaseErr(,wlenIdx,),
                targetId, time, MJD, DIT, U1, U2, V1, V2, stations3;
        }
        else
            newTriple3 = pointer(nil);

        if(is_void(outputOiFile))
        {
            yocoFileSplitName,inputOiFile(kFile),d,f,e;
            outputFileName = d+"/FILT/"+f+
                yocoStrReplace(sum(swrite(wlenRange))," ","-")+
                yocoStrReplace(sum(swrite(keepBas))," ","-")+e;

            if(!open(d+"/FILT","r",1))
                mkdir,d+"/FILT";
        }
        else
            outputFileName = outputOiFile(kFile);

        yocoFileSplitName, outputFileName, d, f, e;
        outputFileTmp = d+f+"_TMP"+e;

        write, "writing file "+outputFileName;

        status = amdlibWriteOiFile(outputFileTmp,
                                   pointer(nil), &inoiArray, 
                                   &inoiTarget, &newWave,
                                   &inamberPhot, &newVis, 
                                   &newVis2, &newTriple3, &inamberOpd,
                                   &inSpectrum);
        
        if (status == amdlibFAILURE)
        {
            yocoGuiErrorBox,"Could not save result file\n";
        }
        
        write, "dump tables";
        _amdlibDumpAllKeywords, 1, inputOiFile(kFile), outputFileTmp;
        rename, outputFileTmp, outputFileName;
    }
}

/*************************************************************/

func mag2CorrMag(mag, band, vis)
    /* DOCUMENT mag2CorrMag(mag, band, vis)

       DESCRIPTION

       PARAMETERS
       - mag : 
       - band: 
       - vis : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    flux = yocoAstroMagToFlux(mag, band);
    corrFlux = flux*vis;
    corrMag = yocoAstroFluxToMag(corrFlux, band);
    write,"Corr flux",corrFlux,"corr mag",corrMag;
    return corrMag;
}

/*************************************************************/

func mag2CorrMagUD(mag, band, diam, baseline, unitSys=)
    /* DOCUMENT mag2CorrMagUD(mag, band, diam, baseline, unitSys=)

       DESCRIPTION

       PARAMETERS
       - mag     : 
       - band    : 
       - diam    : 
       - baseline: 
       - unitSys : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    Utable = array(0.0,dimsof(baseline));
    Vtable = baseline;
    bd = yocoAstroExtractBand(band, unitSys);
    lambda = bd.wl;
    
    vis = abs(visUniformDisk(Utable, Vtable, lambda, time, diam, x0, y0));
    return mag2CorrMag(mag, band, vis);
}


/*************************************************************/

func plotChromOPD(file)
    /* DOCUMENT plotChromOPD(file)

       DESCRIPTION

       PARAMETERS
       - file: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    include,"chg_bbox.i";
    amdlibFileChooser,,file;

    phiDiffChrom = computePhiDiffChromFromOIFile(file);

    amdlibLoadOiData, wlen, bandWidth, time, fringeSNR, sqVis, sqVisErr, uvCoord, bandFlag, pistonOPDArray, sigmaPistonArray, diffVisAmp, diffVisAmpErr, diffVisPhase, diffVisPhaseErr, vis3Amp, vis3AmpErr, vis3Phase, vis3PhaseErr, cpxVis, cpxVisErr, visCovRI, uv1Coord, uv2Coord, fluxSumPiPj, fluxRatPiPj, PiMultPj, spectrum, spectrumErr, inputOiFile=file;
    diffVisPhaseErr = diffVisPhaseErr+(diffVisPhaseErr==0);
    nbBases = dimsof(sqVis)(2);

    wkll;
    for(k=1;k<=nbBases;k++)
    {
        window,k,width=500,height=400;
        yocoPlotWithErrBars,diffVisPhase(k,,1),diffVisPhaseErr(k,,1),wlen,color=-k-4;
        plg,phiDiffChrom(k,),wlen;
        xytitles,"Wavelength (m)","Phase (rad)";

        W = exp(1i*diffVisPhase(k,,1)) * exp(-1i*phiDiffChrom(k,));
        phi = atan(W.im,W.re);
        // window,k+3,width=500,height=400;
        yocoPlotWithErrBars,phi-1,diffVisPhaseErr(k,,1),wlen,color=-k-4;
        limits,,,-1-pi/4,pi/4;
        hcps,"~/phiDiffChrom_Match_"+pr1(k)+".ps";
        require,"chg_bbox.i";
        chg_bbox,"~/phiDiffChrom_Match_"+pr1(k)+".ps";
    }
}

/*************************************************************/

func correctAllChromOPD(void, inputDir=, outputDir=, wet=)
    /* DOCUMENT correctAllChromOPD(inputDir=, outputDir=, wet=)

       DESCRIPTION

       PARAMETERS
       - inputDir : 
       - outputDir: 
       - wet      : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{

    /* Select directory containing files to be treated,
       check if the parameter given by the user is valid */
    message = "Choose the directory you want to treat";
    if ( !( is_void(inputDir) ||
            yocoTypeIsStringScalar(inputDir) ||
            ( yocoTypeIsScalar(inputDir) && inputDir==0 ) ) )
    {
        yocoError, "Check 'inputDir' parameter", 
            "Should be void or a scalar string";
        return 0;
    }
    else if (amdlibFileChooser(message, inputDir) == 0)
    {
        return 0;
    }

    /* create outputDir if not given by the user */
    if(is_void(outputDir))
    {
        /* Found the default for product dir */
        if (amdlibGetProductDir(inputDir+"/", outputDir, suffix="CHROMOPD",up=1) == 0)
        {
            yocoError, "Could not construct outputDir", , 1;
            return 0;
        }
        outputDir = string(outputDir);
    }

    /* Test the outputDir keyword */
    if ( !yocoTypeIsScalar(outputDir) ||
         !yocoTypeIsStringScalar(outputDir) ||
         anyof(outputDir==string(0)) ||
         anyof(outputDir=="") )
    {
        yocoError, "Check 'outputDir' parameter", 
            "Should be void or a scalar string";
        return 0;
    }

    if(!open(outputDir, "r", 1))
    {
        write,"Creating",outputDir,"..."
            /* If outputDir does not exist, obviously force overwrite */
            overwrite = 1;
        /* Create the product dir if not existing */
        mkdir, outputDir;
    }

    /* Find the default logFile name */
    if (amdlibGetLogName(inputDir+"/", logFile) == 0)
    {
        yocoError, "Could not get log file name", , 1;
        return 0;
    }
    logFile = inputDir + "/" + string(logFile);

    /* Read the log of the inputDir */
    if (!amdlibReadLog(logFile, OBSsTab, titles))
    {
        yocoError, "Could not read log file", , 1;
        return 0;
    }

    /* Get useful columns */
    tabFiles = OBSsTab(where(titles=="fileName"), );
    tabDates = OBSsTab(where(titles=="date_obs"), );
    tabNr    = OBSsTab(where(titles=="calib_nr"), );
    tabTypes = OBSsTab(where(titles=="obs_catg"), );
    OBJTypes = OBSsTab(where(titles=="obs_type"), );

    tabIds     = OBSsTab(where(titles=="P2VM_ID"), );
    tabOBsNr   = OBSsTab(where(titles=="OB_nr"), );
    tabTplNr   = OBSsTab(where(titles=="Template_nr"), );
    tabProCatg = OBSsTab(where(titles=="pro_catg"), );

    /* Find AVG OI DATA files */
    AVGIdx = where((tabProCatg == "CALIB_AVERAGED")|
                   (tabProCatg == "SCIENCE_AVERAGED"));
    NAVGI = numberof(AVGIdx);

    if (NAVGI == 0)
    {
        yocoLogInfo,"No relevant raw OI DATA files !";
    }
    for (k=1; k <= NAVGI; k++)
    {
        /* Update the progress bar */
        yocoGuiProgressBox, k, kill=(k<=1),
            barTitle="Computing...",
            minAmount=0,maxAmount=NAVGI,
            minLabel="0",maxLabel=pr1(NAVGI)+" files", win=7,
            dpi=graphicsDpi;
        
        /* replace by the real name */
        here = get_cwd(".");
        inputDir = cd(inputDir);
        cd,here;
    
        yocoFileSplitName,tabFiles(AVGIdx)(k), d,f,e;
        correctChromOPD(inOiFile=inputDir+"/"+tabFiles(AVGIdx)(k),
                        outOiFile=outputDir+"/"+f+"_CHROMOPD"+e, wet=wet);

    }
}

/*************************************************************/

func correctChromOPD(inOiFile=, outOiFile=, signs=, wet=)
    /* DOCUMENT correctChromOPD(inOiFile=, outOiFile=, signs=, wet=)

       DESCRIPTION

       PARAMETERS
       - inOiFile : 
       - outOiFile: 
       - signs    : 
       - wet      : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(signs))
        signs=[1,1,-1];

    /* Select directory containing files to be treated,
       check if the parameter given by the user is valid */
    message = "Choose the directory you want to treat";
    if ( !( is_void(inOiFile) ||
            yocoTypeIsStringScalar(inOiFile) ||
            ( yocoTypeIsScalar(inOiFile) && inOiFile==0 ) ) )
    {
        yocoError, "Check 'inOiFile' parameter", 
            "Should be void or a scalar string";
        return 0;
    }
    else if (amdlibFileChooser(message, inOiFile) == 0)
    {
        return 0;
    }

    yocoFileSplitName,inOiFile, inputDir, f, e; 

    /* create outputDir if not given by the user */
    if(is_void(outputDir))
    {
        /* Found the default for product dir */
        if (amdlibGetProductDir(inputDir+"/", outputDir, suffix="CHROMOPD") == 0)
        {
            yocoError, "Could not construct outputDir", , 1;
            return 0;
        }
        outputDir = inputDir+"/"+string(outputDir);
    }

    /* Test the outputDir keyword */
    if ( !yocoTypeIsScalar(outputDir) ||
         !yocoTypeIsStringScalar(outputDir) ||
         anyof(outputDir==string(0)) ||
         anyof(outputDir=="") )
    {
        yocoError, "Check 'outputDir' parameter", 
            "Should be void or a scalar string";
        return 0;
    }
    else if (!open(outputDir, "r", 1))
    {
        /* If outputDir does not exist, obviously force overwrite */
        overwrite = 1;
        /* Create the product dir if not existing */
        mkdir, outputDir;
    }

    phiDiffChrom = -computePhiDiffChromFromOIFile(inOiFile);

    /* DUMBLY READING TWICE THE FILE */
    _amdlibReadVis, inoiWave, inamberPhot, inoiVis, inoiVis2, inoiVis3,
        inamberOpd, inamberInsCfg, inoiArray, inoiTarget,inSpectrum,
        inputOiFile=inOiFile;

    MJD = (*inoiVis.table).dateObsMJD;
    targetId = (*inoiVis.table).targetId;
    time = (*inoiVis.table).time;
    DIT = (*inoiVis.table).expTime;
    stations = (*inoiVis.table).stationIndex;

    if (nbBases == 3)
    {
        stations3 = (*inoiVis3.table).stationIndex;
        U1 = transpose(array(uv1Coord.u,1));
        V1 = transpose(array(uv1Coord.v,1));
        U2 = transpose(array(uv2Coord.u,1));
        V2 = transpose(array(uv2Coord.v,1));
    } 
    U = uvCoord.u;
    V = uvCoord.v;

    // Read the OI FITS file
    amdlibLoadOiData,
        wlen, bandWidth, time, fringeSNR, sqVis, sqVisErr, uvCoord,
        bandFlag, pistonOPDArray, sigmaPistonArray, diffVisAmp,
        diffVisAmpErr, diffVisPhase, diffVisPhaseErr, vis3Amp, vis3AmpErr,
        vis3Phase, vis3PhaseErr, cpxVis, cpxVisErr, visCovRI, uv1Coord,
        uv2Coord, fluxSumPiPj, fluxRatPiPj, PiMultPj, spectrum,
        spectrumErr, inputOiFile=inOiFile;

    diffVisPhaseErr = diffVisPhaseErr+(diffVisPhaseErr==0);

    nbBases = dimsof(sqVis)(2);
    nbWlen = dimsof(sqVis)(3);
    nbFrames = dimsof(sqVis)(4);

    phiToCorrect = array(phiDiffChrom,nbFrames);
    corrFlux = corrFluxCorr = exp(1i*diffVisPhase);

    // remove residual piston
    for(k=1;k<=nbBases;k++)
    {
        // Correct from dry air phase curvature
        corrFluxCorr(k,,1) = corrFlux(k,,1)*exp(-1i*phiToCorrect(k,,1));

        // Correct from residual piston
        pist = ComputePistonAIPS(corrFluxCorr(k,,1),wlen);
        corrFluxCorr(k,,1) = corrFluxCorr(k,,1)*exp(-2i*pi*pist/wlen);

        // remove any residual offset
        corrFluxCorr(k,,1) =
            corrFluxCorr(k,,1) *
            conj(corrFluxCorr(k,avg,1)(,-,));
    }
    diffVisPhaseCorr = atan(corrFluxCorr.im,corrFluxCorr.re);

    yocoFileSplitName,inOiFile,d,f,ext;
    if(is_void(outOiFile))
        outOiFile = outputDir+f+"_CHROMOPD"+ext;

    if (strmatch(ext,".gz"))
    {
        ext = sum(yocoStrSplit(ext,".gz"));
        recompressFile = "yes";
        yocoFileSplitName,outOiFile,dir2,name2;
        outOiFile = dir2+name2+ext;
    }
    write,"Output file is",outOiFile;

    amdlibSetOiVis, newVis, cpxVis, cpxVisErr, visCovRI, 
        diffVisAmp, diffVisAmpErr, 
        diffVisPhaseCorr, diffVisPhaseErr,
        fringeSNR, targetId, time, MJD, DIT, U, V, stations;

    write, "writing file "+outOiFile;
    status = amdlibWriteOiFile(outOiFile,
                               pointer(nil), &inoiArray, 
                               &inoiTarget, &inoiWave, &inamberPhot, &newVis, 
                               &inoiVis2, &inoiVis3, &inamberOpd,
                               &inSpectrum);

    _amdlibDumpAllKeywords, 1, inOiFile, outOiFile;

    outFh = cfitsio_open(outOiFile, "a", overwrite=1);
    cfitsio_goto_hdu,outFh,1;
    cfitsio_set, outFh, "ESO PRO REC3 ID", "correctChromOPD",
        "By F. Millour";
    cfitsio_close_file, outFh;

    // Rezip file if needed
    if (recompressFile=="yes")
    {
        yocoLogInfo,"Recompressing file";
        if (yocoSystem("gzip -f "+outOiFile) != 0)
        {
            return 0;
        }
    }
}

/*************************************************************/

func computePhiChromWetAirMathar(wlen, P, T, H, Bsinz, &n)
    /* DOCUMENT computePhiChromWetAirMathar(wlen, P, T, H, Bsinz, &n)

       DESCRIPTION
       Compute the differential phase bias introduced by the chromatic
       differential OPD coming from humid air, using the formulae from
       Mathar (2007, journal of optics A, 9, 470)

       PARAMETERS
       - wlen : 
       - P    : 
       - T    : 
       - H    : 
       - Bsinz: 
       - n    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    /*** wlen en m ***/
    sigma = 1.0 / wlen * yocoAstroSI.cm / yocoAstroSI.m;
    sigmaref = 1e4 / 2.25;

    // T en Kelvins
    Tref = 273.15 + 17.5;

    // Pref en Pascals
    Pref = 75000;

    // Href in percentage of humidity
    Href = 10;

    Cjref = [0.200192e-3,
             0.113474e-9,
             -0.424595e-14,
             0.100957e-16,
             -0.293315e-20,
             0.307228e-24
             ];

    CjT = [0.588625e-1,
           -0.385766e-7,
           0.888019e-10,
           -0.567650e-13,
           0.166615e-16,
           -0.174845e-20
           ];

    CjTT = [-3.01513,
            0.406167e-3,
            -0.514544e-6,
            0.343161e-9,
            -0.101189e-12,
            0.106749e-16
            ];

    CjH = [-0.103945e-7,
           0.136858e-11,
           -0.171039e-14,
           0.112908e-17,
           -0.329925e-21,
           0.344747e-25
           ];

    CjHH = [0.573256e-12,
            0.186367e-16,
            -0.228150e-19,
            0.150947e-22,
            -0.441214e-26,
            0.461209e-30
            ];

    CjP = [0.267085e-8,
           0.135941e-14,
           0.135295e-18,
           0.818218e-23,
           -0.222957e-26,
           0.249964e-30
           ];

    CjPP = [0.609186e-17,
            0.519024e-23,
            -0.419477e-27,
            0.434120e-30,
            -0.122445e-33,
            0.134816e-37
            ];

    CjTH = [0.497859e-4,
            -0.661752e-8,
            0.832034e-11,
            -0.551793e-14,
            0.161899e-17,
            -0.169901e-21
            ];

    CjTP = [0.779176e-6,
            0.396499e-12,
            0.395114e-16,
            0.233587e-20,
            -0.636441e-24,
            0.716868e-28
            ];

    CjHP = [-0.206567e-15,
            0.106141e-20,
            -0.149982e-23,
            0.984046e-27,
            -0.288266e-30,
            0.299105e-34
            ];

    nm1 = array(0,numberof(wlen));
    for(j=1;j<=6;j++)
    {
        Cj = Cjref(j) +
            CjT(j) * (1/T - 1/Tref) +
            CjTT(j) * (1/T - 1/Tref)^2 +
            CjH(j) * (H - Href) +
            CjHH(j) * (H - Href)^2 +
            CjP(j) * (P - Pref) +
            CjPP(j) * (P - Pref)^2 +
            CjTH(j) * (1/T - 1/Tref) * (H - Href) +
            CjTP(j) * (1/T - 1/Tref) * (P - Pref) +
            CjHP(j) * (H - Href) * (H - Href);

        write,Cj;
        nm1 += Cj * (sigma - sigmaref) ^ double(j-1);
    }
    n = 1 + nm1;

    phi_OPDchroma = 2 * pi * n * Bsinz / wlen;
    return phi_OPDchroma;
}

/*************************************************************/

func computePhiChromDryAirVannier(wlen, P, T, Bsinz, &n)
    /* DOCUMENT computePhiChromDryAirVannier(wlen, P, T, Bsinz, &n)

       DESCRIPTION
       Compute the differential phase bias introduced by the chromatic
       differential OPD coming from dry air, using the formulae from Ciddor
       (1996, applied optics, 35, 1566) and ideas from M. Vannier

       PARAMETERS
       - wlen : 
       - P    : 
       - T    : 
       - Bsinz: 
       - n    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    //**************** Loi standard: 
    k0 = 238.0185;
    k1 = 5792105.0;
    k2 = 57.362;
    k3 = 167917.0;

    // T donné en Kelvins, transformé en Celsius
    T = T - 273.15;

    // P donné Pascals, transformé en mmHg
    P = P / yocoAstroSI.mmHg / yocoAstroSI.Pa;

    /*** Loi standard, wlen en microns ***/
    sigma2 = 1.0 / (wlen * 1e6) ^ 2;
    // Ciddor 1996, eq. 1
    n_as = 1+1e-8 * (k1 / (k0 - sigma2) + k3 / (k2 - sigma2));

    /*** Indice instantane, T en degres celsius et P en mmHg ***/
    // Ciddor, eq. ??? 
    n = 1 + (n_as-1) * P * (1 + (1.049 - 0.0157 * T) * 1e-6 * P) /
        (720.883 * (1 + 0.009661 * T));

    phi_OPDchroma = 2 * pi * n * Bsinz / wlen;
    return phi_OPDchroma;
}

/*************************************************************/

func computeCompressibility(P, T, xw)
    /* DOCUMENT computeCompressibility(P, T, xw)

       DESCRIPTION

       PARAMETERS
       - P : 
       - T : 
       - xw: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // P in pascals
    // T in Kelvins
    // xw is the molar fraction of water vapor in moist air

    // Coefficients to compute Z
    a0 = 1.58123e-6;
    a1 = -2.9331e-8;
    a2 = 1.1043e-10;
    b0 = 5.707e-6;
    b1 = -2.051e-8;
    c0 = 1.9898e-4;
    c1 = -2.376e-6;
    d = 1.83e-11;
    e = -0.765e-8;

    // t in Celsius
    t = T - 273.15;

    // Compute compressibility of gas, from Ciddor 1996, eq. 12
    Z = 1 - (P / T) *
        (a0 +
         a1 * t +
         a2 * t^2 +
         (b0 + b1 * t) * xw +
         (c0 + c1 * t) * xw^2) +
        (P / T) ^ 2 * (d + e * xw^2);

    return Z;
}

/*************************************************************/

func computeBIPM_Density(P, T, xw, xc)
    /* DOCUMENT computeBIPM_Density(P, T, xw, xc)

       DESCRIPTION

       PARAMETERS
       - P : 
       - T : 
       - xw: 
       - xc: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // P in pascals
    // T in Kelvins
    // xw is the molar fraction of water vapor in moist air
    // xc ppm of CO2

    // gas constant,
    R = 8.314510;

    // Mw is the molar mass of water vapor
    Mw = 0.018015;

    // molar mass of dry air containing xc ppm of CO2
    Ma = 1e-3 * (28.9635 + 12.011e-6 * (xc - 400));

    // Compressibility of gas
    Z = computeCompressibility(P, T, xw);

    // BIPM density, Ciddor 1996 eq. 4
    rho = (P * Ma / (Z * R * T)) *
        (1 - xw * (1 - Mw / Ma));

    return rho;
}

/*************************************************************/

func computeIndexWetAirCiddor(wlen, P, T, H, Bsinz, xc, &n_prop, &n_gprop)
    /* DOCUMENT computeIndexWetAirCiddor(wlen, P, T, H, Bsinz, xc, &n_prop, &n_gprop)

       DESCRIPTION

       PARAMETERS
       - wlen   : 
       - P      : 
       - T      : 
       - H      : 
       - Bsinz  : 
       - xc     : 
       - n_prop : 
       - n_gprop: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{

    //h is the fractionnal humidity value
    h = H / 100.0;

    // Sigma is in microns^-1
    sigma = 1.0 / (wlen * yocoAstroSI.m / yocoAstroSI.mum);
    sigma2 = sigma * sigma;
    sigma4 = sigma2 * sigma2;
    sigma6 = sigma4 * sigma2;

    /*** Temperatures in Kelvin ! ***/
    T2 = T^2;

    // Following the steps from Ciddor 1996 in Appendix B
    /************ Point 1 ************/
    // Find svp (see below eq. 4)
    A = 1.2378847e-5;
    B = -1.9121316e-2;
    C = 33.93711047;
    D = -6.3431645e3;
    svp = exp(A * T2 + B * T + C + D / T);

    // find pw
    pw = h * svp;

    // find f (see below eq. 4)
    t = T - 273.15;
    t2 = t^2;
    alpha = 1.00062;
    beta = 3.14e-8;
    gamma = 5.6e-7;
    f = alpha + beta * P + gamma * t2;

    // find xw
    xw = f * h * svp / P;

    /************ Point 2 ************/
    // Ciddor 1996, eq. 1
    k0 = 238.0185;
    k1 = 5792105.0;
    k2 = 57.362;
    k3 = 167917.0;
    n_as = 1 + 1e-8 * (k1 / (k0 - sigma2) + k3 / (k2 - sigma2));

    // Ciddor 1996, eq. 2
    n_axs = 1 + (n_as - 1)*(1 + 0.534e-6 * (xc - 450));
    n_gasx = 1 + 1e-8 * (k1 * (k0 + sigma2) / (k0 - sigma2)^2 +
                         k3 * (k2 + sigma2) / (k2 - sigma2)^2) *
        (1 + 0.534e-6 * (xc - 450));

    /************ Point 3 ************/
    // Find Ma (done in the function computeBIPM_Density)

    /************ Point 4 ************/

    // Find Za (done in point 6)
    // Ciddor 1996, eq. 12
    Pa = 101325;
    Ta = 15 + 273.15;
    xwa = 0;

    /************ Point 5 ************/
    // Find Zw (done in point 7)
    Pw = 1333;
    Tw = 20 + 273.15;
    xww = 1;

    /************ Point 6 ************/
    // compute rho_axs
    rho_axs = computeBIPM_Density(Pa, Ta, xwa, xc);

    // compute rho_ws
    rho_ws = computeBIPM_Density(Pw, Tw, xww, xc);

    /************ Point 7 ************/
    // Compressibility of moist air under experimental conditions
    // (eq. 12) Done in point 8 through computeBIPM_Density

    /************ Point 8 ************/
    // Density of the dry component of the moist air
    //rho_a = P * Ma * (1 - xw) / (Z * R * T);
    rho_a = computeBIPM_Density(P, T, xw, xc);

    /************ Point 9 ************/
    // Density of the water vapour component of the moist air
    //rho_w = P * Mw * xw / (Z * R * T);
    rho_w = computeBIPM_Density(pw, T, 1, xc);

    // Ciddor 1996, eq. 3
    //**************** vapeur d'eau standard:
    cf = 1.022;
    w0 = 295.235;
    w1 = 2.6422;
    w2 = -0.032380;
    w3 = 0.004028;
    n_ws = 1 + 1e-8 * cf * (w0 + w1 * sigma2 + w2 * sigma4 + w3 * sigma6);
    n_gws = 1 + 1e-8 * cf * (w0 + 3 * w1 * sigma2 + 5 * w2 * sigma4 + 7 * w3 * sigma6);

    /************ Point 10 ************/
    // Ciddor 1996, eq. 5
    n_prop = 1 + (rho_a / rho_axs) * (n_axs - 1) +
        (rho_w / rho_ws) * (n_ws - 1);

    n_gprop = 1 + (rho_a / rho_axs) * (n_gasx - 1) +
        (rho_w / rho_ws) * (n_gws - 1);
}

func computePhiChromWetAirCiddor(wlen, P, T, H, Bsinz, &n_prop)
    /* DOCUMENT computePhiChromWetAirCiddor(wlen, P, T, H, Bsinz, &n_prop)

       DESCRIPTION
       Compute the differential phase bias introduced by the chromatic
       differential OPD coming from dry air, using the formulae from Ciddor
       (1996, applied optics, 35, 1566)

       PARAMETERS
       - wlen  : in meters
       - P     : pression in pascals
       - T     : temperature in Kelvins
       - H     : relative humidity in percents
       - Bsinz : pathlength of air
       - n_prop: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // Here xc is the standard CO2 content in ppm
    xc = 450;

    // CO2 content in Chile as of 31 dec. 2008. See
    // http://www.esrl.noaa.gov/gmd/ccgg/carbontracker/co2weather.php?type=glb#imagetable
    // for details
    xc = 384;

    computeIndexWetAirCiddor, wlen, P, T, H, Bsinz, xc, n_prop, n_gprop;

    //fsdf()
    phi_OPDchroma = 2 * pi * (n_prop) * Bsinz / wlen;

    return phi_OPDchroma;
}

func computePhiChromDryAirCiddor(wlen, P, T, Bsinz, &n_prop)
    /* DOCUMENT computePhiChromDryAirCiddor(wlen, P, T, Bsinz, &n_prop)

       DESCRIPTION

       PARAMETERS
       - wlen  : 
       - P     : 
       - T     : 
       - Bsinz : 
       - n_prop: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    return computePhiChromWetAirCiddor(wlen, P, T, 0, Bsinz, n_prop);
}

func testChromWet(void)
    /* DOCUMENT testChromWet

       DESCRIPTION

       PARAMETERS
       - : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    T = 15.+273.15;
    P = 1013.24*100;
    H = 100.0;
    Bsinz = 20.0;

    wlen = span(1.3,2.5,100)*1e-6;
    toto = computePhiChromWetAirMathar(wlen, P, T, H, Bsinz, n_Mathar);
    toto = computePhiChromWetAirCiddor(wlen, P, T, H, Bsinz, n_Ciddor);
    toto = computePhiChromDryAirCiddor(wlen, P, T, Bsinz, n_Ciddor_dry);
    toto = computePhiChromDryAirVannier(wlen, P, T, Bsinz, n_Vannier);

    wkll;
    window,1,width=600,height=600;
    plg,n_Ciddor,wlen,color="blue";
    plg,n_Ciddor_dry,wlen,color="magenta";
    // plg,n_Vannier,wlen,color="red";
    plg,n_Mathar,wlen;

    

    // pouet()
}
//testChromWet

func testChromDry(void)
    /* DOCUMENT testChromDry

       DESCRIPTION

       PARAMETERS
       - : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    T = [20, 20, 20, 10, 30]+273.15;
    P = [80, 100, 120, 100, 100]*1000;
    H = 0.0;
    Bsinz = 0;
    wlen = 633e-9;
    toto = computePhiChromWetAirCiddor(wlen, P, T, H, Bsinz, n_Ciddor);
    //toto = computePhiChromDryAirVannier(wlen, P, T, Bsinz, n_Ciddor);
    write,T-273.15, P / 1000, 1e8*(n_Ciddor-1);

    write,"--------------------------------";
    T = [19.526, 19.517, 19.173, 19.173, 19.188, 19.189, 19.532, 19.534, 19.534]+273.15;
    P = [102094.8, 102096.8, 102993.0, 103006.0, 102918.8, 102927.8, 103603.2, 103596.2, 103599.2];

    t = T// - 273.15;
        A = 1.2378847e-5;
    B = -1.9121316e-2;
    C = 33.93711047;
    D = -6.3431645e3;
    svp = exp(A * t^2 + B * t + C + D / t);
    pw = [1065, 1065, 641, 642, 706, 708, 986, 962, 951];
    H = pw / svp * 100;
    xc = [510, 510, 450, 440, 450, 440, 600, 600, 610]
        toto = computeIndexWetAirCiddor(wlen, P, T, H, Bsinz, xc, n_prop, n_gprop)
        write,T-273.15, P, pw, xc, 1e8*(n_prop-1);

    write,"--------------------------------";
    T = [20, 20, 40, 40, 50, 50]+273.15
        P = [80, 120, 80, 120, 80, 120]*1000;
    H = [75, 75, 75, 75, 100, 100];
    toto = computePhiChromWetAirCiddor(wlen, P, T, H, Bsinz, n_Ciddor);
    write,T-273.15, P / 1000, H, 1e8*(n_Ciddor-1);
}
//testChromDry
/*************************************************************/

func computePhiDiffChromFromOIFile(oiFile, &att, corrType=)
    /* DOCUMENT computePhiDiffChromFromOIFile(oiFile, &att, corrType=)

       DESCRIPTION
       Compute the differential phase bias introduced by the chromatic
       differential OPD coming from dry air on an AMBER file. Some ideas
       thanks to J.-B. Lebouquin

       PARAMETERS
       - oiFile  : 
       - att     : 
       - corrType: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(corrType))
        // corrType = "wetCiddor";
        corrType = "wetMathar";

    // Load file
    amdlibFileChooser,"Load averaged OI file",oiFile;

    amdlibLoadOiData, wlen, bandWidth, time, fringeSNR, sqVis, sqVisErr, uvCoord, bandFlag, pistonOPDArray, sigmaPistonArray, diffVisAmp, diffVisAmpErr, diffVisPhase, diffVisPhaseErr, vis3Amp, vis3AmpErr, vis3Phase, vis3PhaseErr, cpxVis, cpxVisErr, visCovRI, uv1Coord, uv2Coord, fluxSumPiPj, fluxRatPiPj, PiMultPj, spectrum, spectrumErr, inputOiFile=oiFile;

    nbBases = dimsof(sqVis)(2);
    diffVisPhaseErr = diffVisPhaseErr+(diffVisPhaseErr==0);

    /* get important keywords from the file: pressure,
       temperature, humidity and azimut */
    vals = yocoStr2Double(_amdlibGetKwdVals(oiFile, ["ESO ISS AMBI TEMP",
                                                     "ESO ISS AMBI PRES",
                                                     "ESO ISS AMBI RHUM",
                                                     "ESO ISS AZ",
                                                     "ESO DEL DLT1 OPL START",
                                                     "ESO DEL DLT1 OPL END",
                                                     "ESO DEL DLT2 OPL START",
                                                     "ESO DEL DLT2 OPL END",
                                                     "ESO DEL DLT3 OPL START",
                                                     "ESO DEL DLT3 OPL END",
                                                     "ESO DEL REF OPL",
                                                     "ESO ISS CONF A1L",
                                                     "ESO ISS CONF A2L",
                                                     "ESO ISS CONF A3L",
                                                     "ESO ISS TEMP LAB1",
                                                     "ESO ISS TEMP LAB2",
                                                     "ESO ISS TEMP LAB3",
                                                     "ESO ISS TEMP LAB4",
                                                     "ESO ISS TEMP TUN1",
                                                     "ESO ISS TEMP TUN2",
                                                     "ESO ISS TEMP TUN3",
                                                     "ESO ISS TEMP TUN4"], gui=0));

    // For dry air dispersion
    // Get temperature in Kelvin
    temp = vals(1) + 273.15;
    //temp = vals(15) + 273.15; // temp in lab
    //temp = vals(16) + 273.15; // Temp in lab
    //temp = vals(17) + 273.15; // Temp in lab eastern wall
    //temp = vals(18) + 273.15; // Temp in lab
    //temp = vals(19) + 273.15; // Temp in tunnel A west
    //temp = vals(20) + 273.15; // Temp in tunnel centre
    //temp = vals(21) + 273.15; // Temp in tunnel centre
    //temp = vals(22) + 273.15; // Temp in tunnel L east

    // Convert pressure from millibars to pascals
    pres = vals(2) * yocoAstroSI.mbar / yocoAstroSI.Pa;
    // Convert azimut from degrees to radians
    azimut = vals(4)*yocoAstroSI.deg;
    // relative humidity in percents (?)
    relHum = vals(3);

    // Average position of delay lines
    A1L = vals(12);
    A2L = vals(13);
    A3L = vals(14);

    // Starting and stopping points of delay lines
    DLT1_START = vals(5);
    DLT2_START = vals(7);
    DLT3_START = vals(9);
    DLT1_END = vals(6);
    DLT2_END = vals(8);
    DLT3_END = vals(10);

    // Delay introduced, per telescope
    OPL1 = A1L + 0.5*( DLT1_START + DLT1_END);
    OPL2 = A2L + 0.5*( DLT2_START + DLT2_END);
    OPL3 = A3L + 0.5*( DLT3_START + DLT3_END);

    // The real path of air difference between the telescopes
    Bsinz = -[OPL2-OPL1,
              OPL3-OPL2,
              OPL3-OPL1];

    // get wave number (of later use)
    sigma = 1.0/wlen;

    nbWlen = numberof(wlen);
    phiDiffChrom = array(0.0, nbBases, nbWlen);

    for(k=1;k<=nbBases;k++)
    {
        /* compute chromatic phase */
        if(corrType=="wetCiddor")
            phiChrom = computePhiChromWetAirCiddor(wlen, pres, temp, relHum, Bsinz(k));
        else if(corrType=="wetMathar")
            phiChrom = computePhiChromWetAirMathar(wlen, pres, temp, relHum, Bsinz(k));
        else if(corrType=="dryCiddor")
            phiChrom = computePhiChromDryAirCiddor(wlen, pres, temp, Bsinz(k));
        else if(corrType=="dryVannier")
            phiChrom = computePhiChromDryAirVannier(wlen, pres, temp, Bsinz(k));

        /* remove slope in wave number */
        //pist = ComputePistonAIPS(frameDif(j,,k),wlen);

        res = yocoMathPolyFit(1, sigma, phiChrom);

        /* remove slope in wave number */
        res2 = yocoMathPolyFit(1, sigma, diffVisPhase(k,,1),
                               1.0/(diffVisPhaseErr(k,,1)^2));
        res2 = yocoMathPolyFit(1, sigma, diffVisPhase(k,,1));
        diffVisPhase(k,,1) = diffVisPhase(k,,1) - yocoMathPoly(sigma, res2);

        phiDiffChrom(k,) = phiChrom - yocoMathPoly(sigma, res);
    }

    corrFlux = exp(1i*phiDiffChrom);
    chroPist = computeChromaticPiston(wlen, corrFlux);
    att = cos(phiDiffChrom);
        
    return phiDiffChrom;
}

/*************************************************************/

func gausf(x, params)
    /* DOCUMENT gausf(x, params)

       DESCRIPTION

       PARAMETERS
       - x     : 
       - params: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    return params(-1)*sqrt(2*pi)*params(1)*yocoMathGauss(x,params(:-2))+params(0);
}

/*************************************************************/

func filtreArray(inArray, freqFilt=, color=, plot=)
    /* DOCUMENT filtreArray(inArray, freqFilt=, color=, plot=)

       DESCRIPTION

       PARAMETERS
       - inArray : 
       - freqFilt: 
       - color   : 
       - plot    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    N = numberof(inArray);

    if(is_void(freqFilt))
        freqFilt = int(N/3.8);

    ffs = fft(inArray);

    if(plot==1)
    {
        window,1,width=400,height=400;
        plg,inArray,color=color;
    }

    x = indgen(N/2);
    y = abs(ffs(:N/2)) * (1-sqrt(2*pi)*10.*yocoMathGauss(x,[10,0]));

    mod = abs(ffs);
    phi = atan(ffs.im,ffs.re);

    cen = freqFilt;
    mac = y(freqFilt);
    lar = 3;
    off = median(y);

    par = [lar, cen, mac-off, off];

    if(plot==1)
    {
        window,2;
        plg,gausf(x,par),x,color=color;
        plg,y,x,color=color;
    }

    require,"lmfit.i";
    r = lmfit(gausf, x, par, y, 1.0);

    if(plot==1)
    {
        plg,gausf(x,par),x,color=color;
    }

    fun = sqrt(2*pi)*par(1)*yocoMathGauss(x,par(:2));
    use = where(fun>0.00000001);
    start = use(1);
    end = use(0);

    modmod = mod;
    x2 = indgen(N);

    modmod = mod - 1*par(3)*sqrt(2*pi)*par(1)*yocoMathGauss(x2,par(:2))-
        1*par(3)*sqrt(2*pi)*par(1)*yocoMathGauss(N-x2+1+N%2,par(:2));

    hyx = max(int(log(N)),1);

    sstart = start-hyx;
    if(sstart<1)
        asfklj()
            // sstart=1;

            eend = end+hyx;
    if(eend>N)
        asfklj()
            // eend=N;

            modmod(start:end) = interp([avg(modmod(sstart:start)),
                                        avg(modmod(end:eend))],
                                       [start,end],indgen(end-start+1)+start-1);

    start2 = N-end+1+N%2;
    end2 = N-start+1+N%2;

    sstart2 = start2-hyx;
    if(sstart2<1)
        asfklj()
            // sstart2=1;

            eend2 = end2+hyx;
    if(eend2>N)
        asfklj()
            // eend2=N;

            modmod(start2:end2) = interp([avg(modmod(sstart2:start2)),
                                          avg(modmod(end2:eend2))],
                                         [start2,end2],indgen(end2-start2+1)+start2-1);


    if(plot==1)
    {
        window,3;
        plg,modmod,color=color;
        plg,mod,color=color;
        logxy,0,1;
    }

    ffs2 = modmod*exp(1i*phi);

    outArray = 1.0/N*fft(ffs2,-1).re;

    if(plot==1)
    {
        window,1;
        plg,outArray;
    }

    return outArray;

}

/*************************************************************/

func filtrIris(oiFile=)
    /* DOCUMENT filtrIris(oiFile=)

       DESCRIPTION

       PARAMETERS
       - oiFile: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    winkill;

    // Load file
    amdlibFileChooser,"Load averaged OI file",oiFile;

    /* DUMBLY READING TWICE THE FILE */
    _amdlibReadVis, inoiWave, inamberPhot, inoiVis, inoiVis2, inoiVis3,
        inamberOpd, inamberInsCfg, inoiArray, inoiTarget,
        inputOiFile=oiFile;

    amdlibLoadOiData, wlen, bandWidth, time, fringeSNR, sqVis, sqVisErr, uvCoord, bandFlag, pistonOPDArray, sigmaPistonArray, diffVisAmp, diffVisAmpErr, diffVisPhase, diffVisPhaseErr, vis3Amp, vis3AmpErr, vis3Phase, vis3PhaseErr, cpxVis, cpxVisErr, visCovRI, uv1Coord, uv2Coord, fluxSumPiPj, fluxRatPiPj, PiMultPj, spectrum, spectrumErr, inputOiFile=oiFile;

    nbBases = dimsof(sqVis)(2);
    nbWlen = dimsof(sqVis)(3);
    nbFrames = dimsof(sqVis)(4);

    MJD = (*inoiVis.table).dateObsMJD;
    targetId = (*inoiVis.table).targetId;
    time = (*inoiVis.table).time;
    DIT = (*inoiVis.table).expTime;
    stations = (*inoiVis.table).stationIndex;

    if (nbBases == 3)
    {
        stations3 = (*inoiVis3.table).stationIndex;
        U1 = transpose(array(uv1Coord.u,1));
        V1 = transpose(array(uv1Coord.v,1));
        U2 = transpose(array(uv2Coord.u,1));
        V2 = transpose(array(uv2Coord.v,1));
    }

    U = uvCoord.u;
    V = uvCoord.v;

    dat = spectrum;
    dat = sqVis(..,1);
    dat = diffVisPhase(..,1);

    for(k=1;k<=nbBases;k++)
    {
        if(!is_void(spectrum))
            spectrum(k,..) = filtreArray(spectrum(k,..),color=colors(k));
        sqVis(k,,1) = filtreArray(sqVis(k,,1),color=colors(k));
        diffVisPhase(k,,1) = filtreArray(diffVisPhase(k,,1),color=colors(k));
    }

    // if (nbBases == 3)
    // {
    // vis3Phase(1,,1) =filtreArray(vis3Phase(1,,1),color=colors(k));
    // }

    if(!is_void(spectrum))
        amdlibSetSpec, newSpectrum, spectrum, spectrumErr;

    amdlibSetOiVis, newVis, cpxVis, cpxVisErr, visCovRI, 
        diffVisAmp, diffVisAmpErr, 
        diffVisPhase, diffVisPhaseErr,
        fringeSNR, targetId, time, MJD, DIT, U, V, stations;

    amdlibSetOiVis2, newVis2, sqVis, sqVisErr,
        targetId, time, MJD, DIT, U, V, stations;

    if (nbBases == 3)
    {
        amdlibSetOiVis3, newTriple3,
            vis3Amp, vis3AmpErr, vis3Phase, vis3PhaseErr,
            targetId, time, MJD, DIT, U1, U2, V1, V2, stations3;
    }

    if(!open("oldAVG","r",1))
        mkdir,"oldAVG";

    yocoFileSplitName,oiFile,d,f,e;

    if(!open("oldAVG/"+f+e,"r",1))
        system,"cp "+f+e+" "+d+"/oldAVG/"+f+e;

    outputFileName = oiFile;

    write, "writing file "+outputFileName;
    status = amdlibWriteOiFile(outputFileName,
                               pointer(nil), &inoiArray, 
                               &inoiTarget, &inoiWave, &newPhot, &newVis, 
                               &newVis2, &newTriple3, &newPiston,
                               &newSpectrum);

    /* Copy all keywords that could be useful for use afterwards */
    _amdlibDumpAllKeywords, 1, d+"/oldAVG/"+f+e, outputFileName;

}

/*************************************************************/

func Rydberg(n, nPrim, &lineName)
    /* DOCUMENT Rydberg(n, nPrim, &lineName)

       DESCRIPTION

       PARAMETERS
       - n       : 
       - nPrim   : 
       - lineName: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    R = 1.097373156853955e7;
    sigma = R * (1.0 / nPrim^2 - 1.0 / n^2);

    if(nPrim==1)
        pref = "Ly";
    else if(nPrim==2)
        pref = "H";
    else if(nPrim==3)
        pref = "Pa";
    else if(nPrim==4)
        pref = "Br";
    else if(nPrim==5)
        pref = "Pf";
    else if(nPrim==6)
        pref = "Hu";

    star = ["_!a", "_!b", "_!g", "_!d", "_!e", "_!z", "_!h", "_!t", "_!i", "_!k", "_!l", "_!m", "_!n", "_!x", "_!o"];
    
    lineName = array(string,numberof(n));
    N        = int(n(where(n<32666))-nPrim);
    for(k=1;k<=numberof(N);k++)
    {
        if(N(k)<6)
            lineName(k) = pref+star(N(k));
        else if((N(k)>=6)&&(N(k)<=1000))
            lineName(k) = pref+"_"+pr1(N(k));
    }
    if(numberof(where(n<32666))!=0)
        lineName(where(n>32666)) = pref+"_oo";
    
    return 1/sigma;
}

nLines = 15;

///////////// Lyman series
Wlen_H_Lyman = Rydberg(grow(double(indgen(nLines)),1e100)+1, 1, Name_H_Lyman);

//////////// Balmer series
Wlen_H_Balmer = Rydberg(grow(double(indgen(nLines)),1e100)+2, 2, Name_H_Balmer);
// Wlen_H_Balmer = [656, 486, 434, 410, 397, 356]*yocoAstroSI.nm;
// Name_H_Balmer = "H"+["_!a", "_!b", "_!g", "_!d", "_!e", "_oo"];

//////////// Paschen series
Wlen_H_Paschen = Rydberg(grow(double(indgen(nLines)),1e100)+3, 3, Name_H_Paschen);
// Wlen_H_Paschen = [1874.5, 1281.4, 1093.5, 1004.6, 954.3, 922.6, 901.2, 886.0, 874.8, 866.2, 820.1]*yocoAstroSI.nm;
// Name_H_Paschen = "Pa"+["_!a", "_!b", "_!g", "_!d", "_!e", "_6", "_7","_8", "_9" ,"_10", "_oo"];

//////////// Brackett series
Wlen_H_Brackett = Rydberg(grow(double(indgen(nLines)),1e100)+4, 4, Name_H_Brackett);
//Wlen_H_Brackett = [4052.5, 2625.9, 2166.1, 1945.1, 1818.1, 1458.0]*yocoAstroSI.nm;
//Name_H_Brackett = "Br"+["_!a", "_!b", "_!g", "_!d", "_!e", "_oo"];

//////////// Pfund series
Wlen_H_Pfund = Rydberg(grow(double(indgen(nLines)),1e100)+5, 5, Name_H_Pfund);
// Wlen_H_Pfund = [7476, 4664, 3749, 3304, 3046, 2279]*yocoAstroSI.nm;
// Name_H_Pfund = "Pf"+["_!a", "_!b", "_!g", "_!d", "_!e", "_oo"];

//////////// Humphreys series
Wlen_H_Humphreys = Rydberg(grow(double(indgen(nLines)),1e100)+6, 6, Name_H_Humphreys);
// Wlen_H_Humphreys = [12368, 7503, 5905, 5129, 4673, 4374, 4171, 4021, 3908, 3819, 3749, 3282]*yocoAstroSI.nm;
// Name_H_Humphreys = "Hu"+["_!a", "_!b", "_!g", "_!d", "_!e", "_6", "_7","_8", "_9" ,"_10", "_11", "_oo"]; 


/*************************************************************/

func calibrateSpectrumMR(sciFiles=, calFiles=, outputSpecFile=)
    /* DOCUMENT calibrateSpectrumMR(sciFiles=, calFiles=, outputSpecFile=)

       DESCRIPTION

       PARAMETERS
       - sciFiles      : 
       - calFiles      : 
       - outputSpecFile: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    require,"lmfit.i";
    if(amdlibFileChooser("Choose Calibration OI files", calFiles) == 0)
        return 0;
    if(amdlibFileChooser("Choose Science OI files", sciFiles) == 0)
        return 0;

    wkll;

    // Load data from calibrator star
    sspectrum = sspectrumErr = cspectrum = cspectrumErr = [];
    for(k=1;k<=numberof(calFiles);k++)
    {
        amdlibLoadOiData, cwlen,bandWidth,time,fringeSNR,sqVis,sqVisErr,
            uvCoord,bandFlag,pistonOPDArray,sigmaPistonArray,diffVisAmp,
            diffVisAmpErr,diffVisPhase,diffVisPhaseErr,vis3Amp,vis3AmpErr,
            vis3Phase, vis3PhaseErr,cpxVis,cpxVisErr,visCovRI,
            uv1Coord,uv2Coord,fluxSumPiPj,
            fluxRatPiPj,PiMultPj,cspectru,cspectrumEr,inputOiFile=calFiles(k);
        grow,cspectrum,array(cspectru,1);
        grow,cspectrumErr,array(cspectrumEr,1);
    }

    // Load data from science star
    for(k=1;k<=numberof(sciFiles);k++)
    {
        amdlibLoadOiData, swlen,bandWidth,time,fringeSNR,sqVis,sqVisErr,
            uvCoord,bandFlag,pistonOPDArray,sigmaPistonArray,diffVisAmp,
            diffVisAmpErr,diffVisPhase,diffVisPhaseErr,vis3Amp,vis3AmpErr,
            vis3Phase, vis3PhaseErr,cpxVis,cpxVisErr,visCovRI,
            uv1Coord,uv2Coord,fluxSumPiPj,
            fluxRatPiPj,PiMultPj,sspectru,sspectrumEr,inputOiFile=sciFiles(k);
        grow,sspectrum,array(sspectru,1);
        grow,sspectrumErr,array(sspectrumEr,1);
    }

    // Plave here the wavelength offset to have the proper wavelength scale

    cspectrumvg = cspectrum(sum,,sum);
    cspectrumvgE = sqrt((cspectrumErr^2)(sum,,sum))/avg(cspectrumvg);
    cspectrumvg = cspectrumvg/median(cspectrumvg); 

    sspectrumvg = sspectrum(sum,,sum);
    sspectrumvgE = sqrt((sspectrumErr^2)(sum,,sum))/avg(sspectrumvg);
    sspectrumvg = sspectrumvg/median(sspectrumvg);

    HWlen = grow(Wlen_H_Lyman,
                 Wlen_H_Balmer,
                 Wlen_H_Paschen,
                 Wlen_H_Brackett,
                 Wlen_H_Pfund,
                 Wlen_H_Humphreys);
    HNames = grow(Name_H_Lyman,
                  Name_H_Balmer,
                  Name_H_Paschen,
                  Name_H_Brackett,
                  Name_H_Pfund,
                  Name_H_Humphreys);

    idx = where((HWlen>min(cwlen))&
                (HWlen<max(cwlen)));
    if(numberof(idx)!=0)
    {
        HWlen = HWlen(idx);
        HNames = HNames(idx);
        idx = sort(HWlen);
        HWlen = HWlen(idx);
        HNames = HNames(idx);

        window,1;
        nFeat = numberof(HWlen);
        for(k=1;k<=nFeat;k++)
        {
            startagain=1;
            while(startagain==1)
            {
                fma;
                newIdx = where((cwlen<HWlen(k)+0.03e-6)&
                               (cwlen>HWlen(k)-0.03e-6));
                newDat = cspectrumvg(newIdx);
                newDat = newDat/avg(newDat);
                newWlen = cwlen(newIdx);
                plg,newDat,newWlen, color="blue";
                pltitle,"Please select the continuum level";
                pldj,HWlen(k),0,HWlen(k),2;
                plt,HNames(k)+", "+pr1(HWlen(k)*1e6)+"!mm",
                    HWlen(k)+0.001e-6,
                    min(newDat)+(min(newDat)-avg(newDat))*0.01,tosys=1;

                limits,HWlen(k)-0.02e-6,HWlen(k)+0.02e-6,
                    min(newDat)+(min(newDat)-avg(newDat))*0.1,
                    max(newDat)+(max(newDat)-avg(newDat))*0.1;

                result= mouse(1, 0, "Please select the continuum level");
                if(result(10)==1)
                {
                    cont = result(2);
                    fma;
                    plg,newDat,newWlen, color="blue";
                    pldj,HWlen(k),0,HWlen(k),2;
                    plt,HNames(k)+", "+pr1(HWlen(k)*1e6)+"!mm",
                        HWlen(k)+0.001e-6,
                        min(newDat)+(min(newDat)-avg(newDat))*0.01,tosys=1;
                    pldj,min(newWlen),cont,max(newWlen),cont;

                    pltitle,"Please select the "+HNames(k) +
                        " line bottom";
                    result= mouse(1, 2,
                                  "Please select the "+HNames(k) +
                                  " line bottom");
                    if(result(10)==1)
                    {
                        wlenLine = HWlen(k);
                        deep = result(2);
                        pldj,min(newWlen),deep,max(newWlen),deep;

                        pltitle,"Please select the "+HNames(k) +
                            " line width";
                        result= mouse(1, 2,
                                      "Please select the "+HNames(k) +
                                      " line width");
                        if(result(10)==1)
                        {
                            wd = abs(result(3)-result(1))/4;
                            //wd = 0.001e-6;
                            params = [wd, wd, wlenLine, cont, deep-cont];
                            plg,lineSpec(newWlen, params), newWlen,color="red";

                            res = lmfit(lineSpec, newWlen, params, newDat);
                            plg,lineSpec(newWlen, params), newWlen,color="magenta";
                            result= mouse(1, 0,
                                          "Left-click OK\nmiddle-click start again\nright-click cancel");
                            if(result(10)==1)
                            {
                                newCont = 1.0;
                                newDeep = (deep-cont)/cont;
                                params(3) = newCont;
                                params(4) = newDeep;
                                cspectrumvg(newIdx) = cspectrumvg(newIdx)/lineSpec(newWlen, params);

                                fma;
                                plg,cspectrumvg,cwlen, color="blue";
                                plg,sspectrumvg,swlen, color="green";
                                pause,1000;
                                startagain=0;
                            }
                            else if(result(10)==3)
                            {
                                startagain=0;
                            }
                        }
                    }

                }
                else if(result(10)==3)
                {
                    startagain=0;
                }
            } 
        }
    }

    window,2;
    calSpec = sspectrumvg/cspectrumvg;
    calSpecErr = sqrt(
                      (sspectrumvgE^2*cspectrumvg^2 +
                       cspectrumvgE^2*sspectrumvg^2)/
                      (cspectrumvg^4));
    yocoPlotWithErrBars,calSpec,calSpecErr,cwlen, color="black";

    sciFiles = sciFiles(sort(sciFiles));

    /* Define the default for the outputSpecFile from the
       first and last files. By default, create an output files with
       all similar part of the file names */
    if(is_void(outputSpecFile))
    {
        yocoFileSplitName,sciFiles(1),d,f1,e;
        yocoFileSplitName,sciFiles(0),d,f2,e;

        stringStart = f1;
        stringEnd = f2;
        w = 1;
        while((strmatch(strpart(stringStart,1:w),
                        strpart(stringEnd,1:w)))&&
              (w<=strlen(f1)))
        {
            w++;
        }

        /////////////////////////////
        // Same for calibration files
        
        yocoFileSplitName,calFiles(1),d,f3,e;
        yocoFileSplitName,calFiles(0),d,f4,e;

        stringStart2 = f3;
        stringEnd2 = f4;
        w2 = 1;
        while((strmatch(strpart(stringStart2,1:w2),
                        strpart(stringEnd2,1:w2)))&&
              (w2<=strlen(f1)))
        {
            w2++;
        }

        outputSpecFile = yocoStrSplit(stringStart,"OIDATA")(1) +
            "-" +
            strpart(stringEnd,w:) + "-" +
            yocoStrSplit(stringStart2,"OIDATA")(1) +
            "-" +
            strpart(stringEnd2,w2:) ;
    }

    yocoFileSplitName,outputSpecFile,d,f,e;
    outFile = d+f+"_SPECTRUM.txt";
    write, "Writing the file "+outFile;
    fh = open(outFile,"w");
    write,fh,"Spectrum","SpectrumErr","Wlen";
    write,fh,calSpec,calSpecErr,cwlen;
    close,fh;
}

/*************************************************************/

func calibrateSpectrumHR(sciFiles=, calFiles=, outputSpecFile=)
    /* DOCUMENT calibrateSpectrumHR(sciFiles=, calFiles=, outputSpecFile=)

       DESCRIPTION

       PARAMETERS
       - sciFiles      : 
       - calFiles      : 
       - outputSpecFile: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    require,"lmfit.i";
    if(amdlibFileChooser("Choose Calibration OI files", calFiles) == 0)
        return 0;
    if(amdlibFileChooser("Choose Science OI files", sciFiles) == 0)
        return 0;

    wkll;

    // Load data from calibrator star
    sspectrum = sspectrumErr = cspectrum = cspectrumErr = [];
    for(k=1;k<=numberof(calFiles);k++)
    {
        amdlibLoadOiData, cwlen,bandWidth,time,fringeSNR,sqVis,sqVisErr,
            uvCoord,bandFlag,pistonOPDArray,sigmaPistonArray,diffVisAmp,
            diffVisAmpErr,diffVisPhase,diffVisPhaseErr,vis3Amp,vis3AmpErr,
            vis3Phase, vis3PhaseErr,cpxVis,cpxVisErr,visCovRI,
            uv1Coord,uv2Coord,fluxSumPiPj,
            fluxRatPiPj,PiMultPj,cspectru,cspectrumEr,inputOiFile=calFiles(k);
        grow,cspectrum,array(cspectru,1);
        grow,cspectrumErr,array(cspectrumEr,1);
    }

    // Load data from science star
    for(k=1;k<=numberof(sciFiles);k++)
    {
        amdlibLoadOiData, swlen,bandWidth,time,fringeSNR,sqVis,sqVisErr,
            uvCoord,bandFlag,pistonOPDArray,sigmaPistonArray,diffVisAmp,
            diffVisAmpErr,diffVisPhase,diffVisPhaseErr,vis3Amp,vis3AmpErr,
            vis3Phase, vis3PhaseErr,cpxVis,cpxVisErr,visCovRI,
            uv1Coord,uv2Coord,fluxSumPiPj,
            fluxRatPiPj,PiMultPj,sspectru,sspectrumEr,inputOiFile=sciFiles(k);
        grow,sspectrum,array(sspectru,1);
        grow,sspectrumErr,array(sspectrumEr,1);
    }

    // Plave here the wavelength offset to have the proper wavelength scale

    cspectrumvg = cspectrum(sum,,sum);
    cspectrumvgE = sqrt((cspectrumErr^2)(sum,,sum))/avg(cspectrumvg);
    cspectrumvg = cspectrumvg/median(cspectrumvg); 

    sspectrumvg = sspectrum(sum,,sum);
    sspectrumvgE = sqrt((sspectrumErr^2)(sum,,sum))/avg(sspectrumvg);
    sspectrumvg = sspectrumvg/median(sspectrumvg);

    HWlen = grow(Wlen_H_Lyman,
                 Wlen_H_Balmer,
                 Wlen_H_Paschen,
                 Wlen_H_Brackett,
                 Wlen_H_Pfund,
                 Wlen_H_Humphreys);
    HNames = grow(Name_H_Lyman,
                  Name_H_Balmer,
                  Name_H_Paschen,
                  Name_H_Brackett,
                  Name_H_Pfund,
                  Name_H_Humphreys);

    idx = where((HWlen>min(cwlen))&
                (HWlen<max(cwlen)));
    if(numberof(idx)!=0)
    {
        HWlen = HWlen(idx);
        HNames = HNames(idx);
        idx = sort(HWlen);
        HWlen = HWlen(idx);
        HNames = HNames(idx);

        window,1;
        nFeat = numberof(HWlen);
        for(k=1;k<=nFeat;k++)
        {
            startagain=1;
            while(startagain==1)
            {
                fma;
                newIdx = where((cwlen<HWlen(k)+0.03e-6)&
                               (cwlen>HWlen(k)-0.03e-6));
                newDat = cspectrumvg(newIdx);
                newDat = newDat/avg(newDat);
                newWlen = cwlen(newIdx);
                plg,newDat,newWlen, color="blue";
                pldj,HWlen(k),0,HWlen(k),2;
                plt,HNames(k)+", "+pr1(HWlen(k)*1e6)+"!mm",
                    HWlen(k)+0.001e-6,
                    min(newDat)+(min(newDat)-avg(newDat))*0.01,tosys=1;

                limits,HWlen(k)-0.02e-6,HWlen(k)+0.02e-6,
                    min(newDat)+(min(newDat)-avg(newDat))*0.1,
                    max(newDat)+(max(newDat)-avg(newDat))*0.1;

                pltitle,"Please select the region of interest";
                result= mouse(1, 1, "Please select the region of interest");
                if(result(10)==1)
                {
                    lambda1 = result(1);
                    lambda2 = result(3);

                    idx = where((newWlen>lambda1)&(newWlen<lambda2));

                    pltitle,"Please select the continuum level";
                    result= mouse(1, 0, "Please select the continuum level");
                    if(result(10)==1)
                    {
                        cont = result(2);
                        fma;
                        plg,newDat,newWlen, color="blue";
                        pldj,HWlen(k),0,HWlen(k),2;
                        plt,HNames(k)+", "+pr1(HWlen(k)*1e6)+"!mm",
                            HWlen(k)+0.001e-6,
                            min(newDat)+(min(newDat)-avg(newDat))*0.01,tosys=1;
                        pldj,min(newWlen),cont,max(newWlen),cont;

                        pltitle,"Please select the "+HNames(k) +
                            " line bottom";
                        result= mouse(1, 2,
                                      "Please select the "+HNames(k) +
                                      " line bottom");
                        if(result(10)==1)
                        {
                            wlenLine = HWlen(k);
                            deep = result(2);
                            pldj,min(newWlen),deep,max(newWlen),deep;

                            pltitle,"Please select the "+HNames(k) +
                                " line width";
                            result= mouse(1, 2,
                                          "Please select the "+HNames(k) +
                                          " line width");
                            if(result(10)==1)
                            {
                                wd = abs(result(3)-result(1))/4;
                                //wd = 0.001e-6;
                                params = [wd, wd, wlenLine, cont, deep-cont];
                                plg,lineSpec(newWlen, params), newWlen,color="red";

                                res = lmfit(lineSpec, newWlen(idx), params, newDat(idx));
                                plg,lineSpec(newWlen, params), newWlen,color="magenta";
                                result= mouse(1, 0,
                                              "Left-click OK\nmiddle-click start again\nright-click cancel");
                                if(result(10)==1)
                                {
                                    newCont = 1.0;
                                    deep = params(4)+params(5);
                                    newDeep = (deep-cont)/cont;
                                    params(4) = newCont;
                                    params(5) = newDeep;
                                    cspectrumvg(newIdx) = cspectrumvg(newIdx)/lineSpec(newWlen, params);

                                    fma;
                                    plg,cspectrumvg,cwlen, color="blue";
                                    plg,sspectrumvg,swlen, color="green";
                                    pause,1000;
                                    startagain=0;
                                }
                                else if(result(10)==3)
                                {
                                    startagain=0;
                                }
                            }
                        }
                    }

                }
                else if(result(10)==3)
                {
                    startagain=0;
                }
            } 
        }
    }

    window,2;
    calSpec = sspectrumvg/cspectrumvg;
    calSpecErr = sqrt(
                      (sspectrumvgE^2*cspectrumvg^2 +
                       cspectrumvgE^2*sspectrumvg^2)/
                      (cspectrumvg^4));
    yocoPlotWithErrBars,calSpec,calSpecErr,cwlen, color="black";

    sciFiles = sciFiles(sort(sciFiles));

    /* Define the default for the outputSpecFile from the
       first and last files. By default, create an output files with
       all similar part of the file names */
    if(is_void(outputSpecFile))
    {
        yocoFileSplitName,sciFiles(1),d,f1,e;
        yocoFileSplitName,sciFiles(0),d,f2,e;

        stringStart = f1;
        stringEnd = f2;
        w = 1;
        while((strmatch(strpart(stringStart,1:w),
                        strpart(stringEnd,1:w)))&&
              (w<=strlen(f1)))
        {
            w++;
        }

        outputSpecFile = yocoStrSplit(stringStart,"OIDATA")(1) +
            "-" +
            strpart(stringEnd,w:);
    }

    yocoFileSplitName,outputSpecFile,d,f,e;
    outFile = d+f+"_SPECTRUM.txt";
    write, "Writing the file "+outFile;
    fh = open(outFile,"w");
    write,fh,"Spectrum","SpectrumErr","Wlen";
    write,fh,calSpec,calSpecErr,cwlen;
    close,fh;
}

/*************************************************************/

func computeWlenSolutionHR(calFiles=, inputP2vmFile=, outputCoefFile=, maxGrade=, filtrIris=)
    /* DOCUMENT computeWlenSolutionHR(calFiles=, inputP2vmFile=, outputCoefFile=, maxGrade=, filtrIris=)

       DESCRIPTION

       PARAMETERS
       - calFiles      : 
       - inputP2vmFile : 
       - outputCoefFile: 
       - maxGrade      : 
       - filtrIris     : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // Get reference spectrum from Kitt Peak
    require,"Kitt_Peak_Transmission.i";
    require,"lmfit.i";
    wkll;

    if(is_void(maxGrade))
        maxGrade = 3;

    if(amdlibFileChooser("Choose Calib OI files", calFiles) == 0)
        return 0;

    // if(is_void(inputP2vmFile))
    // {
    //         p2vmLoaded = 0;}
    // else
    // Load P2VM file
    if (amdlibLoadP2vm( mat, p2vmphot, p2vmVk, p2vmwlen, p2vmflag,
                        p2vmcalibPhase, p2vmnbTels, p2vmkeyNames,
                        p2vmkeyValues, inputP2vmFile=inputP2vmFile) == 0)
    {
        // p2vmLoaded = 0;
    }
    else
    {
        p2vmLoaded = 1;
        p2vmwlen = p2vmwlen*1e-9;
            
        // Get P2VM dimensions
        nbP2vmWlen = numberof(p2vmwlen);
        nbP2vmBases = dimsof(mat)(2)/2;
            
        // Build Photometric & interferometric array from the Phot array
        newPhot = array(0.0,nbP2vmWlen,p2vmnbTels+1,p2vmnbTels+2*nbP2vmBases);
        for(k=1;k<=p2vmnbTels;k++)
            newPhot(,k,k) = p2vmphot(,k,1);
        tel1 = [1,2,1];
        tel2 = [2,3,3];
        for(k=1;k<=2*nbP2vmBases;k++)
        {
            l = (k+1)/2;
            newPhot(,tel1(l),k+p2vmnbTels) = p2vmphot(,1,k+1);
            newPhot(,tel2(l),k+p2vmnbTels) = p2vmphot(,2,k+1);
            newPhot(,0,k+p2vmnbTels) = p2vmphot(avg,3,k+1);
        }

        // Construct calibration spectra
        p2vmSpec = transpose(newPhot(,1:p2vmnbTels,sum));
        p2vmIntf = newPhot(,0,sum);
    }

    // Load data from calibrator star
    sspectrum = sspectrumErr = cspectrum = cspectrumErr = [];
    for(k=1;k<=numberof(calFiles);k++)
    {
        amdlibLoadOiData, cwlen,bandWidth,time,fringeSNR,sqVis,sqVisErr,
            uvCoord,bandFlag,pistonOPDArray,sigmaPistonArray,diffVisAmp,
            diffVisAmpErr,diffVisPhase,diffVisPhaseErr,vis3Amp,vis3AmpErr,
            vis3Phase, vis3PhaseErr,cpxVis,cpxVisErr,visCovRI,
            uv1Coord,uv2Coord,fluxSumPiPj,
            fluxRatPiPj,PiMultPj,cspectru,cspectrumEr,inputOiFile=calFiles(k);
        grow,cspectrum,array(cspectru,1);
        grow,cspectrumErr,array(cspectrumEr,1);
    }

    if(filtrIris==1)
    {
        nbTel = dimsof(cspectrum)(2);
        for (i=1; i<=nbTel; i++)
        {
            cspectrum(i,) = filtreArray(cspectrum(i,),plot=0);
        } 
    }
        
    
    if(p2vmLoaded==1)
    {
        // Check for wavelength match between P2VM and science star data
        intv = abs(p2vmwlen(dif))(avg)/2.;
        idxStart = where((min(cwlen)-intv<=p2vmwlen)&
                         (min(cwlen)+intv>=p2vmwlen))(1);
        idxStop = where((max(cwlen)-intv<=p2vmwlen)&
                        (max(cwlen)+intv>=p2vmwlen))(0);

        // If wavelength order is reverse, then reverse indexes
        if (idxStart>idxStop)
        {
            idxStart2 = idxStop;
            idxStop = idxStart;
            idxStart = idxStart2;
        }

        // Reshape calibration spectrum to the right number of wavelengths
        p2vmfot  = p2vmSpec(,idxStart:idxStop);
        p2vmintf = p2vmIntf(idxStart:idxStop);

        cspectrum2 = cspectrum / p2vmfot(..,-);
    }

    // Place here the wavelength offset to have the proper wavelength scale

    cspectrumvg = cspectrum(sum,,sum);
    cspectrumvgE = sqrt((cspectrumErr^2)(sum,,sum))/avg(cspectrumvg);
    cspectrumvg = cspectrumvg/median(cspectrumvg);
    
    cspectrumvg2 = cspectrum2(sum,,sum);
    cspectrumvgE2 = sqrt((cspectrumErr^2)(sum,,sum))/avg(cspectrumvg2);
    cspectrumvg2 = cspectrumvg2/median(cspectrumvg2);

    yocoNmCreate,1,width=800,height=600,landscape=1;
    fma;
    
    plg,cspectrumvg,cwlen, color="blue";
    plg,cspectrumvg2,cwlen, color="green";
    plg,kittPeakTrans_12000,kittPeakWlen_12000,color="red";
    // for(k=1;k<=3;k++)
    //     plg,p2vmSpec(k,)/p2vmSpec(k,avg),p2vmwlen;
    // plg,p2vmIntf/p2vmIntf(avg),p2vmwlen;

    limits,min(p2vmwlen), max(p2vmwlen);

    result = mouse(1, 2, "Drag your mouse of the global offset");
    offset = result(3)-result(1);


    
    HWlen = grow(Wlen_H_Lyman,
                 Wlen_H_Balmer,
                 Wlen_H_Paschen,
                 Wlen_H_Brackett,
                 Wlen_H_Pfund,
                 Wlen_H_Humphreys);
    HNames = grow(Name_H_Lyman,
                  Name_H_Balmer,
                  Name_H_Paschen,
                  Name_H_Brackett,
                  Name_H_Pfund,
                  Name_H_Humphreys);

    dat = yocoFileReadAscii(Y_SITE+"i/gemini_Catalog_K.txt");

    atmWlen = yocoStr2Double(dat(1,));
    atmStrg = yocoStr2Double(dat(4,));
    atmNames = array("Atm. line",numberof(atmWlen));

    idx = where((atmWlen>min(cwlen)-0.02e-6)&
                (atmWlen<max(cwlen)+0.02e-6));
    if(numberof(idx)!=0)
    {
        atmWlen  = atmWlen(idx);
        atmNames = atmNames(idx);
        atmStrg  = atmStrg(idx);
        idx      = sort(atmStrg);
        atmWlen  = atmWlen(idx);
        atmNames = atmNames(idx);
        atmStrg  = atmStrg(idx);

        nFeat  = numberof(atmWlen);
        mdat   = rdat = [];
        //offset = 0;
        for(k=1;k<=nFeat;k++)
        {
            if(atmStrg(k)<=maxGrade)
            {
                fma;
                newIdx = where((cwlen-offset<atmWlen(k)+0.03e-6)&
                               (cwlen-offset>atmWlen(k)-0.03e-6));
                newDat = cspectrumvg(newIdx);
                newDat = newDat/median(newDat);
                newWlen = cwlen(newIdx)-offset;

                newIdxG = where((kittPeakWlen_12000<atmWlen(k)+0.03e-6)&
                                (kittPeakWlen_12000>atmWlen(k)-0.03e-6));
                newGem = kittPeakTrans_12000(newIdxG);
                newGem = newGem/median(newGem);
                newGemWlen = kittPeakWlen_12000(newIdxG);

                plg,newDat,newWlen, color="blue";
                plg,newGem,newGemWlen,color="red";

                pldj,atmWlen(k),0,atmWlen(k),2;
                plt,atmNames(k)+", "+pr1(atmWlen(k)*1e6)+"!mm\nGrade: "+
                    pr1(atmStrg(k)),
                    atmWlen(k)+0.001e-6,
                    min(newDat)+(min(newDat)-avg(newDat))*0.01,
                    tosys=1;

                for(l=1;l<=numberof(atmWlen);l++)
                {
                    if((atmWlen(l)>min(newWlen))&&((atmWlen(l)<max(newWlen))))
                        pldj,atmWlen(l),0,atmWlen(l),2,type="dash";
                }

                limits,atmWlen(k)-0.03e-6,atmWlen(k)+0.03e-6,
                    min(newDat)+(min(newDat)-avg(newDat))*0.1,
                    max(newDat)+(max(newDat)-avg(newDat))*0.1;
                pltitle,"Please select the underlined feature\nright-click to ignore";
                result = mouse(1, 0, "Please select the underlined feature\nright-click to ignore");
                if(result(10)==1)
                {
                    grow, mdat, result(1)+offset;
                    grow, rdat, atmWlen(k);
                }
                if(numberof(mdat)>=1)
                {
                    offset = avg(mdat)-avg(rdat);
                }
            }
        }
    }

    idx = where((HWlen>min(cwlen)-0.02e-6)&
                (HWlen<max(cwlen)+0.02e-6));
    if(numberof(idx)!=0)
    {
        HWlen = HWlen(idx);
        HNames = HNames(idx);
        idx = sort(HWlen);
        HWlen = HWlen(idx);
        HNames = HNames(idx);

        nFeat = numberof(HWlen);
        for(k=1;k<=nFeat;k++)
        {
            fma;
            newIdx = where((cwlen-offset<HWlen(k)+0.03e-6)&
                           (cwlen-offset>HWlen(k)-0.03e-6));
            newDat = cspectrumvg(newIdx);
            newDat = newDat/avg(newDat)
                newWlen = cwlen(newIdx)-offset;
            plg,newDat,newWlen, color="blue";
            plg,kittPeakTrans_12000,kittPeakWlen_12000,color="red";
            pldj,HWlen(k),0,HWlen(k),2;
            plt,HNames(k)+", "+pr1(HWlen(k)*1e6)+"!mm",
                HWlen(k)+0.001e-6,0.2,tosys=1;
            limits,HWlen(k)-0.03e-6,HWlen(k)+0.03e-6,0,max(newDat)*1.2;
            result = mouse(1, 0,
                           "Please select the underlined feature\nright-click to ignore");
            if(result(10)==1)
            {
                grow, mdat, result(1)+offset;
                grow, rdat, HWlen(k);
            }
            if(numberof(mdat)>=1)
            {
                offset = avg(mdat)-avg(rdat);
            }
        }
    }

    if(numberof(mdat)==0)
        readableError, "At least one reference wavelength should be identified !!";
    else if(numberof(mdat)==1)
        coef = [-offset, 1, 0];
    else if(numberof(mdat)<=6)
        coef = grow(yocoMathPolyFit(1, mdat, rdat),0);
    else
        coef = yocoMathPolyFit(2, mdat, rdat);

    cwlen2 = yocoMathPoly(cwlen, coef);

    fma;
    plg,cspectrumvg, cwlen2, color="blue";
    plg,kittPeakTrans_12000,kittPeakWlen_12000,color="red";
    limits,min(cwlen2),max(cwlen2), 0, max(cspectrumvg);

    window,3;
    plg, rdat, mdat, type="none",marker='\2';
    x = span(min(mdat)*0.99,max(mdat)*1.01,100);
    plg, yocoMathPoly(x, coef), x;


    write," pouet" ;
    /* Define the default for the outputOiFile from the
       first and last files. By default, create an output files with
       all similar part of the file names */
    if(is_void(outputCoefFile))
    {
        yocoFileSplitName,calFiles(1),d,f1,e;
        yocoFileSplitName,calFiles(0),d,f2,e;

        stringStart = f1;
        stringEnd   = f2;
        w = 1;
        while((strmatch(strpart(stringStart,1:w),
                        strpart(stringEnd,1:w)))&&
              (w<=strlen(f1)))
        {
            w++;
        }
        outputCoefFile = yocoStrSplit(stringStart,"OIDATA")(1) +
            "-" +
            strpart(stringEnd,w:);
    }
    yocoFileSplitName,outputCoefFile,d,f,e;
    outFile = d+f+"_COEFF.txt";
    write, "Writing the file "+outFile;
    fh = open(outFile,"w");
    write,fh,"#Wavelength solution coefficients";
    write,fh,coef;
    close,fh;

    return coef;
}

/***************************************************************************/

func fitWsol(x, a, &obsWlen, &chi2, plot=)
    /* DOCUMENT fitWsol(x, a, &obsWlen, &chi2, plot=)

       DESCRIPTION
       Residuals computation between refSpec, an array representing a reference curve and obsSpec, a supposedly observed curve. The parameters influence the abscissa coordinate (here a wavelength solution given by a polynomial).

       PARAMETERS
       - x      : Input pixel coordinates (indgen(...))
       - a      : the polynomial parameters
       - obsWlen: The wavelength table referring to obsSpec
       - chi2   : The output Chi squared
       - plot   : optional plot

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    local wlen, spectrum, obsWlen;
    extern obsSpec, refSpec, refWlen;

    obsWlen = yocoMathPoly(x,a);
    chi2    = comparData(obsSpec, obsWlen, refSpec, refWlen, chiSp,
                         fac=fac, plot=plot);

    return chiSp;
}

/***************************************************************************/

func fitFac(x, a, y, plot=)
    /* DOCUMENT fitFac(x, a, y, plot=)

       DESCRIPTION
       Residuals computation between refSpec, an array representing a reference curve and obsSpec, a supposedly observed curve. The parameters influence the abscissa coordinate (here a wavelength solution given by a polynomial).

       PARAMETERS
       - x   : Input pixel coordinates (indgen(...))
       - a   : the polynomial parameters
       - y   : 
       - plot: optional plot

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    local wlen, spectrum;
    extern obsSpec, obsWlen, refSpec, refWlen;

    fac     = abs(a);

    chi2    = comparData(obsSpec, obsWlen, refSpec, refWlen, chiSp,
                         fac=fac, plot=plot);
    return chiSp;
}


/***************************************************************************/

func getWlenSol(wlen)
    /* DOCUMENT getWlenSol(wlen)

       DESCRIPTION
       Fits a polynomial to a given wavelength solution (to be used as first guess).

       PARAMETERS
       - wlen: input wavelength table

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    local wlen;
    fit = yocoMathPolyFit(5, indgen(numberof(wlen)), wlen);
    return fit;
}

/***************************************************************************/

func getTemplateSpec(obsWlen, refSpec, refWlen, &spp, &kpsp, fac=)
    /* DOCUMENT getTemplateSpec(obsWlen, refSpec, refWlen, &spp, &kpsp, fac=)

       DESCRIPTION

       PARAMETERS
       - obsWlen: 
       - refSpec: 
       - refWlen: 
       - spp    : 
       - kpsp   : 
       - fac    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    local wlen, spectrum;

    if(is_void(fac))
        fac=0.0;

    wnz = where(refSpec!=0);
    refSpec = refSpec(wnz);
    refWlen = refWlen(wnz);
    
    spp = interp(refSpec, refWlen, obsWlen);
    if(allof(spp==0))
    {
        return 1e99;
    }
    
    require,"Kitt_Peak_Transmission.i";
    kpsp = interp(kittPeakTrans_1500,kittPeakWlen_1500, obsWlen) + fac;
    kpsp = kpsp/max(kpsp);
    //kpsp = interp(geminiTrans,geminiWlen-decalA,wlen);
    
    tplSpec = (spp*kpsp)/(spp*kpsp)(avg);
    //tplSpec = spp/spp(avg)*kpsp/kpsp(avg);
    
    return tplSpec;
}

/***************************************************************************/

func comparData(obsSpec, obsWlen, refSpec, refWlen, &calsp2, &calsp, fac=, plot=)
    /* DOCUMENT comparData(obsSpec, obsWlen, refSpec, refWlen, &calsp2, &calsp, fac=, plot=)

       DESCRIPTION

       PARAMETERS
       - obsSpec: 
       - obsWlen: 
       - refSpec: 
       - refWlen: 
       - calsp2 : 
       - calsp  : 
       - fac    : 
       - plot   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    local wlen, spectrum;
    local fi1, fi2;
    if(dimsof(obsSpec)(1)==1)
        obsSpec = transpose([obsSpec]);
    
    if(is_void(plot))
        plot=0;
    
    if(is_void(fac))
        fac=0;
    
    chi2=0;
    calsp = calsp2 = [];

    if(plot==1)
    {
        yocoNmCreate,2,2,2,landscape=1,height=600,width=800,fx=1,dy=0.1;
        fma;
    }
    
    for(k=1;k<=dimsof(obsSpec)(2);k++)
    {
        wwlen   = indgen(numberof(obsWlen));
        mySpec  = getTemplateSpec(obsWlen, refSpec, refWlen, spp, kpsp, fac=fac);
        theSpec = obsSpec(k,)/obsSpec(k,avg);
        
        if(plot==1)
        {
            // winkill;
            // window,1,height=600,width=800;
            plsys,1;
            limits;
            plg,theSpec,obsWlen*1e6;
            plg,mySpec,obsWlen*1e6,color="green";
            plg,spp/spp(avg),obsWlen*1e6,color="red";
            // plg,spp/spp(avg)*kpsp/kpsp(avg),obsWlen,color="green";
            plg,kpsp/kpsp(avg),obsWlen*1e6,color="blue";
            fi1 = yocoMathPolyFit(5,wwlen,mySpec);
            fi2 = yocoMathPolyFit(5,wwlen,theSpec);
            //plg,yocoMathPoly(wwlen,fi1),obsWlen*1e6,color="red";
            plg,yocoMathPoly(wwlen,fi2),obsWlen*1e6,color="black";
            pltitle,"Black: observed, Red: template, Blue: atmospheric,\nGreen: both";
            logxy,0,0;
        }
        
        calsp2tmp = mySpec / theSpec ;
        fit       = yocoMathPolyFit(5,obsWlen-avg(obsWlen),calsp2tmp);
        grow, calsp2, [calsp2tmp / yocoMathPoly(obsWlen-avg(obsWlen), fit)];
        
        calsptmp    = theSpec / spp;
        grow, calsp, [calsptmp/avg(calsptmp)];
        
        if(plot==1)
        {
            csp = theSpec / mySpec;
            csp = csp/avg(csp);
            plsys,2;
            limits;
            plg,csp,obsWlen*1e6;
            limits,,,0;
            pltitle,"Cleaned spectrum";
            
            sp2 = theSpec/spp;
            sp2 = sp2/avg(sp2);
            
            sp1 = theSpec/kpsp;
            sp1 = sp1/avg(sp1);
            
            plsys,3;
            plg,sp1,obsWlen*1e6;
            plg,spp/spp(avg)-0.5,obsWlen*1e6,color="red";
            pltitle,"Spectrum, atmosphere removed\nRed: template";
            limits,,,0;
            
            plsys,4;        
            plg,sp2,obsWlen*1e6;
            plg,kpsp/kpsp(avg)-0.5,obsWlen*1e6,color="red";
            pltitle,"Spectrum, template removed\nRed: atmosphere";
            limits,,,0;
        }
        
        // chi2 = sum(calsp(wwlen)^2) + sqrt(numberof(calsp))*max(abs(calsp))^2;
        chi2 += sum((calsp2(wwlen)-1)^2);
    }
    
    calsp = transpose(calsp);
    calsp2 = transpose(calsp2);
    
    return chi2;
}

/***************************************************************************/

func computeWsolAndSpecType(&template, &CHI2, &outputWlen, &outputSpec, &outputSpecErr, &outputTemplate, inputFile=, outputCoefFile=, noWsol=, overwrite=, check=, specType=, N=, kill=)
    /* DOCUMENT computeWsolAndSpecType(&template, &CHI2, &outputWlen, &outputSpec, &outputSpecErr, &outputTemplate, inputFile=, outputCoefFile=, noWsol=, overwrite=, check=, specType=, N=, kill=)

       DESCRIPTION
       Computes wavelength solution and spectral type of a given star

       PARAMETERS
       - template      : 
       - CHI2          : 
       - outputWlen    : 
       - outputSpec    : 
       - outputSpecErr : 
       - outputTemplate: 
       - inputFile     : 
       - outputCoefFile: 
       - noWsol        : 
       - overwrite     : 
       - check         : 
       - specType      : 
       - N             : 
       - kill          : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    extern wsolDej, TEMPLATE_SPECTRA;
    local wlen, spectrum;

    if(is_void(kill))
        kill=0;
    
    if(kill==1)
        winkill;

    // Select file
    if(amdlibFileChooser("Choose Calib OI files for spectral calibration",
                         inputFile) == 0)
        return 0;

    // Set intial values
    if(is_void(noWsol))
        noWsol=0;

    // Set intial values
    if(is_void(check))
        check=1;

    // Set intial values
    if(is_void(N))
        N=5;

    // Load templates catalogs
    if(!wsolDej)
    {
        yocoLogInfo,"Loading spectra catlog...";
        TEMPLATE_SPECTRA = [];
        require,"templates_struct.i";
        require,"load_UVKLIB.i";
        // require,specDataDir+"load_Kurucz.i";
        require,"load_IRTF.i";
        require,"load_Hanson1996.i";
        require,"load_Hanson2005.i";
        wsolDej=1;
    }
        
    // Load atmospheric data
    require,"Kitt_Peak_Transmission.i";
    //require,"gemini_Transmission.i";
    
    files = TEMPLATE_SPECTRA.name;
    CHI2 = CHI22 = WOF = WSOL = [];   

    // Load spectrum files
    yocoLogInfo,"Loading data file...";
    amdlibLoadSpectrums, spectrum, wlen, spectrumErr, band, specFile=inputFile;
    if(strmatch(inputFile(1),"fits"))
        starName = readOI_FITS_star_name(inputFile(1));
    else
        starName = "My Star";

    // Match with input given spectral types
    if(!is_void(specType))
    {
        spTypes = TEMPLATE_SPECTRA.specType;
        // Sort out the good spectral types
        gs = where(strmatch(strpart(spTypes,1:1),specType,1));
    }
    else
        gs = indgen(numberof(files));
    
    yocoLogInfo,"Testing templates vs data...";
    for(iFile=1;iFile<=numberof(gs);iFile++)
        //kFile = 1;
    {
        kFile = gs(iFile);
        wsol  = getWlenSol(wlen);
        
        yocoFileSplitName,files(kFile),di,fi,ex;
        dat    = TEMPLATE_SPECTRA(kFile);

        refWlen = *dat.wlen;
        refSpec = *dat.flux;

        obsSpec = spectrum;
        if(dimsof(obsSpec)(1)==1)
            obsSpec = transpose([obsSpec]);
        obsWlen = wlen;
    
        if(noWsol==0)
        {
            // Compute wavelength solution given the template spectrum
            intv    = max(obsWlen) - min(obsWlen);
            ws0     = wsol(1);
            minChi2 = 1e99;
            for(kOff=-0.02e-6;kOff<=0.02e-6;kOff+=0.01e-7)
            {
                x = indgen(numberof(obsWlen));
                wsol(1) = ws0 + kOff;
                toto = fitWsol(x, wsol, obsWl, cshi2, plot=0);
                if(cshi2<minChi2)
                {
                    minChi2 = cshi2;
                    wowo = kOff;
                }
                
                grow, CHI22, cshi2;
                grow, WOF, kOff;
            }
            
            yocoNmCreate, 2, 2,2, landscape=1,wait=1;
            fma;
            plsys,1;
            plg, CHI22, WOF;
            logxy,,1;
            yocoPlotVertLine, wowo, color="red";
            
            plsys,2;
            plg,refSpec,refWlen;
            limits,min(obsWlen),max(obsWlen);
            
            plsys,3;
            plg,obsSpec(1,),obsWlen;
            
            wsol(1) = ws0 + wowo;

            // Fit wavelength solution
            require,"lmfit.i";
            lmfit, fitWsol, indgen(numberof(obsWlen)), wsol,
                array(0.0,3, numberof(obsWlen));
            
            obsWlen = yocoMathPoly(indgen(numberof(wlen)), wsol);
        }
        
        chi2 = comparData(obsSpec, obsWlen, refSpec, refWlen);
        
        grow,CHI2,chi2;
        grow,WSOL,array(wsol,1);
    }

    theChi2  = CHI2(mnx);
    kFile    = gs(theChi2);
    template = files(kFile);

    allOK = 0;
    while(allOK==0)
    {
        kFile   = where(files==template)(1);
        iFile   = where(gs==kFile)(1);
        wsol    = WSOL(,iFile);
        obsWlen = yocoMathPoly(indgen(numberof(wlen)), wsol);
        
        yocoFileSplitName,files(kFile),d,fi,e;
        dat    = TEMPLATE_SPECTRA(kFile);
        
        irwlen = *dat.wlen;
        irflux = *dat.flux;
        irerr  = *dat.fluxErr;

        // Fit relative intensity of atmospheric lines
        require,"lmfit.i";
        fac = 0.18;
        //lmfit, fitFac, spectrum, fac, obsWlen;
        chi2 = comparData(spectrum, obsWlen, irflux, irwlen, csp, calsp, plot=1, fac=fac);

        write,"----------------------";
        write,kFile,fi,", Type:",dat.specType,", Chi2:",CHI2(iFile), ", Catalog:",dat.catalog;
        
        coef = yocoMathPolyFit(N, wlen, obsWlen);

        write,coef;

        if(check==1)
        {
            // Plot a GUI to check template spectrum is OK & allow
            // for multiple choices.
            window,5,width=600,height=800;
            fma;
            
            //FIXME:
            nNames  = 17;
            nyn     = 4;
            namnyn  = min(nNames*nyn,numberof(CHI2));

            idx     = gs(sort(CHI2))(1:namnyn);
            spTypes = TEMPLATE_SPECTRA(idx).specType;
            
            
            buttons = array(Button(),namnyn);
            for(k=1;k<=namnyn;k++)
            {
                wdt = 0.08;
                buttons(k).y    = 0.95-((k-1)%nNames)/20.0;
                buttons(k).dx   = wdt;
                buttons(k).x    = 0.4-nyn*wdt*1.2+wdt+
                    int(nyn*double((k-1)/double(nNames*nyn)))*2*0.08*1.2;
                buttons(k).dy   = 0.02;
                buttons(k).text = yocoStrReplace(spTypes(k));
            }
            
            grow,buttons,[Button(text="Keep", dx = 0.08, dy = 0.02,
                                 x=0.4-3*0.08,y=0.1),
                          Button(text="Discard", dx = 0.08, dy = 0.02,
                                 x=0.4+3*0.08,y=0.1)];
            nBut = numberof(buttons);
            
            button_plot,buttons;
            
            prompt = "Please verify spectrum and confirm using the file\nor select proper spectral type\n"+starName;
            plt,prompt,0.4,1.0,tosys=0,justify="CH";
            
            res = mouse(0, 1, prompt);
            for(k=1;k<=nBut;k++)
            {
                test = button_test(buttons(k),res(1),res(2));
                if(test==1)
                {
                    if(buttons(k).text == "Discard")
                    {
                        yocoLogWarning,"Discarding file...";
                        return 0;
                    }
                    else if(buttons(k).text == "Keep")
                    {
                        yocoLogInfo,"That file is OK...";
                        allOK=1;
                    }
                    else 
                    {
                        // Otherwise redefine template
                        theOne   = idx(k);
                        template = TEMPLATE_SPECTRA(theOne).name;
                    }
                }
                else
                    continue;
            }
        }
        else
            break;
    }

    // Set output values
    outputSpec     = calsp;
    outputSpecErr  = spectrum /spectrum(,avg)(,-)^2 / calsp * spectrumErr;
    outputWlen     = wlen;
    outputTemplate = template;

    /* Define the default for the outputOiFile from the
       first and last files. By default, create an output files with
       all similar part of the file names */
    if(is_void(outputCoefFile))
    {
        yocoFileSplitName,inputFile(1),d1,f1,e;
        yocoFileSplitName,inputFile(0),d2,f2,e;
        
        stringStart = f1;
        stringEnd = f2;
        w = 1;
        while((strmatch(strpart(stringStart,1:w),
                        strpart(stringEnd,1:w)))&&
              (w<=strlen(f1)))
        {
            w++;
        }
        outputCoefFile = d1+yocoStrSplit(stringStart,"OIDATA")(1) +
            "-" +
            strpart(stringEnd,w:);
    }
    yocoFileSplitName,outputCoefFile,d,f,e;
    outFile = d+f+"_COEFF.txt";
    write, "Writing the file "+outFile;
    
    if(open(outFile,"r",1) && (!overwrite))
    {
        yocoLogWarning,"File " + f+e + " already exists, skipping";
        return 0;
    }
    
    if(noWsol==0)
    {
        fh = open(outFile,"w");
        write,fh,"#Wavelength_solution_coefficients, template="+files(kFile);
        write,fh,coef,linesize=999;
        close,fh;
    }
    
    return coef;
}

/***************************************************************************/

func divideByTemplate(inputFile=, outputFile=, template=, check=, kill=)
    /* DOCUMENT divideByTemplate(inputFile=, outputFile=, template=, check=, kill=)

       DESCRIPTION

       PARAMETERS
       - inputFile : 
       - outputFile: 
       - template  : 
       - check     : 
       - kill      : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    local wlen, spectrum, template, CHI2;
    extern wsolDej, TEMPLATE_SPECTRA;

    // Select file
    if(amdlibFileChooser("Choose Calib OI files for spectral calibration", inputFile) == 0)
        return 0;

    yocoFileSplitName,inputFile,inputDir,fil,ext;

    /* Check outputDir */
    if ( !amdlibCheckProductDir(outputDir, 1, overwrite, suffix="SPC", up=1, inputDir=inputDir) )
    {
        yocoError, "Could not construct 'outputDir'";
        return 0;
    }

    /* Verbose output directory */
    yocoLogInfo, "Product directory is:",outputDir;
    
    // Get best possible template spectrum
    computeWsolAndSpecType, template0, cchi2, outputWlen, outputSpec, outputSpecErr, template, inputFile=inputFile, outputCoefFile=, noWsol=1, overwrite=0, check=1, kill=kill;
        
    /* Construct output OI file name.
       Use the same rule as amdlibComputeTransferFunction */
    yocoFileSplitName,inputFile,caldir,calname,calext;
    if(is_void(outputFile))
        outputFile = outputDir + yocoStrReplace(calname,"AVG","SPC")+calext;
        
    /* Check if produced file already exist if yes,
       skip this file expect if overwrite==1 */
    if ( yocoTypeIsFile(outputFile) && (overwrite==0) )
    {
        yocoLogInfo,"Skipped because the following file already exist:",
            outputFile;
        return 0;
    }
    
    // Read the structure data in prder to put it back to new file as is
    _amdlibReadVis, inoiWave, inamberPhot, inoiVis, inoiVis2, inoiVis3,
        inamberOpd, inamberInsCfg, inoiArray, inoiTarget, inSpectrum, 
        inputOiFile=inputFile;
    
    
    amdlibSetSpec, newSpectrum, outputSpec, outputSpecErr;
    
    write, "writing file "+outputFile;
    status = amdlibWriteOiFile(outputFile,
                               &inamberInsCfg, &inoiArray, 
                               &inoiTarget, &inoiWave, &inamberPhot, &inoiVis, 
                               &inoiVis2, &inoiVis3, &inamberOpd, &newSpectrum);

    if (status == amdlibFAILURE)
    {
        yocoGuiErrorBox,"Could not save result file\n";
    }

    write, "Dump tables...";
    _amdlibDumpAllKeywords, 1, inputFile, outputFile;

    kFile = where(TEMPLATE_SPECTRA.name==template)(1);
    dat    = TEMPLATE_SPECTRA(kFile);
    
    write, "Add keywords...";
    fh = cfitsio_open(outputFile,"a");
    cfitsio_goto_hdu,fh,1;
    cfitsio_write_key, fh, "PIPELINE SPEC TEMPLATE", dat.name;
    cfitsio_write_key, fh, "PIPELINE SPEC TYPE", dat.specType;
    cfitsio_write_key, fh, "PIPELINE SPEC CATALOG", dat.catalog;

    cfitsio_close,fh;
}

/***************************************************************************/

func divideAllByTemplate(inputDir=, outputDir=, logFile=, overwrite=)
    /* DOCUMENT divideAllByTemplate(inputDir=, outputDir=, logFile=, overwrite=)

       DESCRIPTION
       Night transfer function computation on a bunch of files contained in a
       directory.

       PARAMETERS
       - inputDir : Data directory name
       - outputDir: 
       - logFile  : 
       - overwrite: 

       RESULTS

       EXAMPLES

       SEE ALSO:
    */
{
    local wlen, spectrum;
    yocoLogInfo, "divideAllByTemplate()";

    winkill;
    
    /* Check simples parameters */
    if ( !amdlibCheckOverwrite(overwrite)       ||
         !amdlibCheckGraphicsDpi(graphicsDpi)  )
    {
        yocoError, "Check function's parameters";
        return 0;
    }

    /* Select directory containing files to be treated */
    message = "Choose the input directory (OIDATA_AVG)";
    if ( !amdlibFileChooser(message, inputDir, 3)  ||
         !amdlibCheckDir(inputDir, 0) )
    {
        yocoError,"'inputDir' not specified correctly.";
        return 0;
    }
    
    /* Verbose output */
    yocoLogInfo, "Computing all atmosphere spectra for directory"+
        " (overwrite="+pr1(overwrite)+") :", inputDir;
    
    /* Check outputDir */
    if ( !amdlibCheckProductDir(outputDir, 1, overwrite, suffix="SPC", up=1, inputDir=inputDir) )
    {
        yocoError, "Could not construct 'outputDir'";
        return 0;
    }

    /* Verbose output */
    yocoLogInfo, "Product directory is:",outputDir;

    /* If existing but overwrite is void, ask the user */
    if (is_void(overwrite))
    {
        overwrite = yocoGuiInfoBox("Directory " +
                                   yocoStrrtok(outputDir,"/")(2) + 
                                   " already exists. Re-compute all spectra ?\n",
                                   butText=["YES","NO!"],butReturn=[1,0],
                                   dpi=graphicsDpi);
    }
    
    /* Read the log of the inputDir */
    if ( !amdlibReadLogFromDir(inputDir, logTable, titles) )
    {
        yocoError, "Could not read logfile of 'inputDir'";
        return 0;
    }

    /* Extract the info from the Log */
    tabFiles = logTable(where(titles=="fileName"), );
    tabNames = logTable(where(titles=="object_name"), );
    proTypes = logTable(where(titles=="pro_catg"), );

    /* Get the calibration files only */
    calibObs   = where(proTypes=="CALIB_AVERAGED");
    calibFiles = tabFiles(calibObs);
    calibNames = tabNames(calibObs);
    nCalFiles  = numberof(calibFiles);

    /* Check if nothing has to be done */
    if ( nCalFiles < 1 )
    {
        yocoLogWarning, "No OIDATA_AVG files to be processed into OIDATA_SPC... exit!";
        return 1;
    }
    
    /* Loop on files */
    for(kFile=1;kFile<=nCalFiles;kFile++)
    {
        inputOiFile=inputDir+calibFiles(kFile);

        /* Verbose output */
        yocoLogInfo, "divideAllByTemplate is now working in file:",
            inputOiFile;
                
        /* Construct output OI file name.
           Use the same rule as amdlibComputeTransferFunction */
        yocoFileSplitName,inputOiFile,caldir,calname,calext;
        outputOiFile = outputDir + yocoStrReplace(calname,"AVG","SPC")+calext;
        
        /* Check if produced file already exist if yes,
           skip this file expect if overwrite==1 */
        if ( yocoTypeIsFile(outputOiFile) && (overwrite==0) )
        {
            yocoLogInfo,"Skipped because the following file already exist:",
                outputOiFile;
            continue;
        }

        divideByTemplate, inputFile=inputOiFile, outputFile=outputOiFile,check=1, kill=(kFile==1);

    }
    /* End loop on files */

    /* Verbose output */
    yocoLogInfo, " done";
    winkill;

    
    return 1;
}

/***************************************************************************/

func calibrateSpectrumMR_airmass(inputDir=, outputDir=, logFile=, overwrite=)
    /* DOCUMENT calibrateSpectrumMR_airmass(inputDir=, outputDir=, logFile=, overwrite=)

       DESCRIPTION
       Computation of calibrated spectra

       PARAMETERS
       - inputDir : Data directory name
       - outputDir: 
       - logFile  : 
       - overwrite: 

       RESULTS

       EXAMPLES

       SEE ALSO:
    */
{
    local wlen, spectrum, transDir;
    yocoLogInfo, "divideAllByTemplate()";

    winkill,63;
    winkill,62;

    /* Check simples parameters */
    if ( !amdlibCheckOverwrite(overwrite)       ||
         !amdlibCheckGraphicsDpi(graphicsDpi)  )
    {
        yocoError, "Check function's parameters";
        return 0;
    }

    /* Select directory containing files to be treated */
    message = "Choose the input directory (OIDATA_AVG)";
    if ( !amdlibFileChooser(message, inputDir, 3)  ||
         !amdlibCheckDir(inputDir, 0) )
    {
        yocoError,"'inputDir' not specified correctly.";
        return 0;
    }
    
    /* Verbose output */
    yocoLogInfo, "Computing all atmosphere spectra for directory"+
        " (overwrite="+pr1(overwrite)+") :", inputDir;
    
    /* Check outputDir */
    if ( !amdlibCheckProductDir(outputDir, 1, overwrite, suffix="SPC", up=1, inputDir=inputDir) )
    {
        yocoError, "Could not construct 'outputDir'";
        return 0;
    }

    /* Verbose output */
    yocoLogInfo, "Product directory is:",outputDir;

    /* If existing but overwrite is void, ask the user */
    if (is_void(overwrite))
    {
        overwrite = yocoGuiInfoBox("Directory " +
                                   yocoStrrtok(outputDir,"/")(2) + 
                                   " already exists. Re-compute all spectra ?\n",
                                   butText=["YES","NO!"],butReturn=[1,0],
                                   dpi=graphicsDpi);
    }
    
    /* Read the log of the inputDir */
    if ( !amdlibReadLogFromDir(inputDir, logTable, titles) )
    {
        yocoError, "Could not read logfile of 'inputDir'";
        return 0;
    }

    /* Extract the info from the Log */
    tabFiles = logTable(where(titles=="fileName"), );
    tabNames = logTable(where(titles=="object_name"), );
    proTypes = logTable(where(titles=="pro_catg"), );

    /* Get the scienceration files only */
    scienceObs   = where(proTypes=="SCIENCE_AVERAGED");
    scienceFiles = tabFiles(scienceObs);
    scienceNames = tabNames(scienceObs);
    nSciFiles    = numberof(scienceFiles);

    transDir = [];
    /* For amdlib3, check for the transfer function directory */
    if ( !amdlibCheckProductDir(transDir, 0, suffix="SPC", up=1, inputDir=inputDir) )
    {
        yocoError, "Could not find directory with the OIDATA_SPC files", , 1;
        return 0;
    }
    
    /* Read the log of the inputDir */
    if ( !amdlibReadLogFromDir(transDir, logTable2, titles2) )
    {
        yocoError, "Could not read logfile of 'inputDir'";
        return 0;
    }

    /* Extract the info from the Log */
    tabFiles = logTable2(where(titles=="fileName"), );
    tabNames = logTable2(where(titles=="object_name"), );
    proTypes = logTable2(where(titles=="pro_catg"), );

    /* Get the calibration files only */
    calibObs   = where(proTypes=="CALIB_AVERAGED");
    calibFiles = tabFiles(calibObs);
    calibNames = tabNames(calibObs);
    nCalFiles  = numberof(calibFiles);
    
        
    /* Check if nothing has to be done */
    if ( nCalFiles < 1 )
    {
        yocoLogWarning, "No OIDATA_AVG files to be processed into OIDATA_SPC... exit!";
        return 1;
    }
    
    /* Load calibration files */
    CSPC = CSPCE = CAIRM = CWL = CBW = [];
    for(kFile=1;kFile<=nCalFiles;kFile++)
    {
        inputOiFile=transDir+calibFiles(kFile);
        
        amdlibLoadOiData,wlen,bandWidth,time,fringeSNR,sqVis,sqVisErr,
            uvCoord,bandFlag,pistonOPDArray,sigmaPistonArray,diffVisAmp,
            diffVisAmpErr,diffVisPhase,diffVisPhaseErr,vis3Amp,vis3AmpErr,vis3Phase,
            vis3PhaseErr,cpxVis,cpxVisErr,visCovRI,uv1Coord,uv2Coord,fluxSumPiPj,
            fluxRatPiPj,PiMultPj,spectrum,spectrumErr,inputOiFile=inputOiFile;
        
        _amdlibGetKwdVals, inputOiFile, "ESO ISS AIRM START",vals, gui=0;

        // Store spectrum into arrays
        cspc  = spectrum    / spectrum(,avg)(,-);
        cspce = spectrumErr / spectrum(,avg)(,-);
        airm  = yocoStr2Double(vals)(1);

        grow, CAIRM, airm;
        grow, CSPC,  &cspc;
        grow, CSPCE, &cspce;
        grow, CWL,   &wlen;
        grow, CBW,   &bandWidth
        
            yocoNmCreate,63,2,2,landscape=1,width=800,height=600;
        plsys,1;
        plg,cspc(1,) - 1 + airm;//,wlen;
        
        plsys,2;
        yocoPlotWithErrBars,
            cspc(1,min)/cspc(1,avg),cspce(1,min)/cspc(1,avg),airm,
            type="none", marks=1,marker='\2';
    }
    /* End loop on files */

    // Merge spectra to get a continuously matching intensity
    mergeSpectrum, CSPC, CWL, CSPCE, CBW,,,,, CSPC, CWL, CSPCE, CBW;
    
    /* Load science files */
    SSPC = SSPCE = SAIRM = SWL = [];
    for(kFile=1;kFile<=nSciFiles;kFile++)
    {
        inputOiFile = inputDir + scienceFiles(kFile);
        
        amdlibLoadOiData,wlen,bandWidth,time,fringeSNR,sqVis,sqVisErr,
            uvCoord,bandFlag,pistonOPDArray,sigmaPistonArray,diffVisAmp,
            diffVisAmpErr,diffVisPhase,diffVisPhaseErr,vis3Amp,vis3AmpErr,vis3Phase,
            vis3PhaseErr,cpxVis,cpxVisErr,visCovRI,uv1Coord,uv2Coord,fluxSumPiPj,
            fluxRatPiPj,PiMultPj,spectrum,spectrumErr,inputOiFile=inputOiFile;
        
        nbFrames = dimsof(sqVis)(4);
        nbWlen   = dimsof(sqVis)(3);
        nbBases  = dimsof(sqVis)(2);

        // Read airmass from science file
        _amdlibGetKwdVals, inputOiFile, "ESO ISS AIRM START",vals, gui=0;
        
        sspc  = spectrum; // spectrum(,avg)(,-);
        sspce = spectrumErr; // spectrum(,avg)(,-);
        airm = yocoStr2Double(vals)(1);

        theSpec = theSpecErr = theCSpec = array(0.0,dimsof(sspc));
        for(ib=1;ib<=nbBases;ib++)
        {
            for(wl=1;wl<=nbWlen;wl++)
            {
                theVal = theAirm = theErr = [];
                for(iFile=1;iFile<=nCalFiles;iFile++)
                {
                    // Match wavelengths
                    cwlen = *CWL(iFile);
                    if(anyof(cwlen == wlen(wl)))
                    {
                        wmatch = where(cwlen == wlen(wl))(1);
                        grow,theVal, (*CSPC(iFile))(ib,wmatch);
                        grow,theErr, (*CSPCE(iFile))(ib,wmatch);
                        grow,theAirm,(CAIRM(iFile));
                    }
                }

                write,numberof(theVal)

                    if(numberof(theVal)>1)
                        fit = yocoMathPolyFit(1, theAirm, theVal);
                    else
                        fit = avg(theVal);
                
                x = span(min(theAirm),max(theAirm),100);
                
                window,62;
                fma;
                plg,theVal,theAirm,type="none", marks=1,marker='\2',color="red";
                plg,yocoMathPoly(x,fit),x;

                if((min(theAirm)<airm)&(max(theAirm)>airm))
                    theCSpec(ib, wl) = yocoMathPoly(airm, fit);
                else if(min(theAirm)>airm)
                    theCSpec(ib, wl) = theVal(theAirm(mnx));
                else if(max(theAirm)>airm)
                    theCSpec(ib, wl) = theVal(theAirm(mxx));

                if(numberof(theVal)>1)
                    airToUse = yocoMathPoly(airm,fit);
                else
                    airToUse = airm;
                //theSpec(ib, wl)  = sspc(ib, wl) / yocoMathPoly(airm,fit);

                intv = abs(theAirm - airm);
                theSpec(ib, wl)  = sspc(ib, wl) / theVal(where(intv == min(intv)));
                reums = (theVal - yocoMathPoly(theAirm,fit))(rms);
                // Variance d'un quotient de deux variables aléatoires vu sur 
                // http://www.les-mathematiques.net/phorum/read.php?12,359583
                theSpecErr(ib, wl) = sqrt(
                                          sspce(ib,wl)^2 / airToUse^2 +
                                          sspc(ib,wl)^2 / airToUse^4 * reums^2);
                
            }
        }

        grow, SAIRM, airm;
        grow, SSPC,  &theSpec;
        grow, SSPCE, &theSpecErr;
        grow, SWL,   &wlen;
        
        yocoFileSplitName,inputOiFile,d,f,e;
        outFile = d+f+"_SPECTRUM.txt";
        write, "Writing the file "+outFile;
        fh = open(outFile,"w");
        write,fh,"Spectrum","SpectrumErr","Wlen","Spectrum","SpectrumErr","Spectrum","SpectrumErr";
        write,fh,
            theSpec(1,),theSpecErr(1,),wlen,
            theSpec(2,),theSpecErr(2,),
            theSpec(3,),theSpecErr(3,);
        close,fh;

        window,63;
        plsys,4;
        //plg,theSpec(1,) - 1 + airm,color="red";
        yocoPlotWithErrBars,theSpec(1,),theSpecErr(1,),color="blue";
        //plg,sspc(1,min)/sspc(1,avg),airm,type="none", marks=1,marker='\2',color="red";
    }
    /* End loop on files */
    
    /* Verbose output */
    yocoLogInfo, " done";
    return 1;
}

/*************************************************************/

func computeWlenSolutionMR(calFiles=, inputP2vmFile=, outputCoefFile=, maxGrade=)
    /* DOCUMENT computeWlenSolutionMR(calFiles=, inputP2vmFile=, outputCoefFile=, maxGrade=)

       DESCRIPTION

       PARAMETERS
       - calFiles      : 
       - inputP2vmFile : 
       - outputCoefFile: 
       - maxGrade      : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // Get reference spectrum from Gemini
    require,"gemini_Transmission.i";
    require,"lmfit.i";
    wkll;

    if(is_void(maxGrade))
        maxGrade = 3;

    if(amdlibFileChooser("Choose Calib OI files", calFiles) == 0)
        return 0;

    // if(is_void(inputP2vmFile))
    // {
    //         p2vmLoaded = 0;}
    // else
    // Load P2VM file
    if (amdlibLoadP2vm( mat, p2vmphot, p2vmVk, p2vmwlen, p2vmflag,
                        p2vmcalibPhase, p2vmnbTels, p2vmkeyNames,
                        p2vmkeyValues, inputP2vmFile=inputP2vmFile) == 0)
    {
        // p2vmLoaded = 0;
    }
    else
    {
        p2vmLoaded = 1;
        p2vmwlen = p2vmwlen*1e-9;
            
        // Get P2VM dimensions
        nbP2vmWlen = numberof(p2vmwlen);
        nbP2vmBases = dimsof(mat)(2)/2;
            
        // Build Photometric & interferometric array from the Phot array
        newPhot = array(0.0,nbP2vmWlen,p2vmnbTels+1,p2vmnbTels+2*nbP2vmBases);
        for(k=1;k<=p2vmnbTels;k++)
            newPhot(,k,k) = p2vmphot(,k,1);
        tel1 = [1,2,1];
        tel2 = [2,3,3];
        for(k=1;k<=2*nbP2vmBases;k++)
        {
            l = (k+1)/2;
            newPhot(,tel1(l),k+p2vmnbTels) = p2vmphot(,1,k+1);
            newPhot(,tel2(l),k+p2vmnbTels) = p2vmphot(,2,k+1);
            newPhot(,0,k+p2vmnbTels) = p2vmphot(avg,3,k+1);
        }

        // Construct calibration spectra
        p2vmSpec = transpose(newPhot(,1:p2vmnbTels,sum));
        p2vmIntf = newPhot(,0,sum);
    }

    // Load data from calibrator star
    sspectrum = sspectrumErr = cspectrum = cspectrumErr = [];
    for(k=1;k<=numberof(calFiles);k++)
    {
        amdlibLoadOiData, cwlen,bandWidth,time,fringeSNR,sqVis,sqVisErr,
            uvCoord,bandFlag,pistonOPDArray,sigmaPistonArray,diffVisAmp,
            diffVisAmpErr,diffVisPhase,diffVisPhaseErr,vis3Amp,vis3AmpErr,
            vis3Phase, vis3PhaseErr,cpxVis,cpxVisErr,visCovRI,
            uv1Coord,uv2Coord,fluxSumPiPj,
            fluxRatPiPj,PiMultPj,cspectru,cspectrumEr,inputOiFile=calFiles(k);
        grow,cspectrum,array(cspectru,1);
        grow,cspectrumErr,array(cspectrumEr,1);
    }

    
    if(p2vmLoaded==1)
    {
        // Check for wavelength match between P2VM and science star data
        intv = abs(p2vmwlen(dif))(avg)/2.;
        idxStart = where((min(cwlen)-intv<=p2vmwlen)&
                         (min(cwlen)+intv>=p2vmwlen))(1);
        idxStop = where((max(cwlen)-intv<=p2vmwlen)&
                        (max(cwlen)+intv>=p2vmwlen))(0);

        // If wavelength order is reverse, then reverse indexes
        if (idxStart>idxStop)
        {
            idxStart2 = idxStop;
            idxStop = idxStart;
            idxStart = idxStart2;
        }

        // Reshape calibration spectrum to the right number of wavelengths
        p2vmfot  = p2vmSpec(,idxStart:idxStop);
        p2vmintf = p2vmIntf(idxStart:idxStop);

        cspectrum2 = cspectrum / p2vmfot(..,-);
    }

    // Place here the wavelength offset to have the proper wavelength scale

    cspectrumvg = cspectrum(sum,,sum);
    cspectrumvgE = sqrt((cspectrumErr^2)(sum,,sum))/avg(cspectrumvg);
    cspectrumvg = cspectrumvg/median(cspectrumvg);
    
    cspectrumvg2 = cspectrum2(sum,,sum);
    cspectrumvgE2 = sqrt((cspectrumErr^2)(sum,,sum))/avg(cspectrumvg2);
    cspectrumvg2 = cspectrumvg2/median(cspectrumvg2);

    yocoNmCreate,1,width=800,height=600,landscape=1;
    fma;
    
    plg,cspectrumvg,cwlen, color="blue";
    plg,cspectrumvg2,cwlen, color="green";
    plg,geminiTrans,geminiWlen,color="red";
    // for(k=1;k<=3;k++)
    //     plg,p2vmSpec(k,)/p2vmSpec(k,avg),p2vmwlen;
    // plg,p2vmIntf/p2vmIntf(avg),p2vmwlen;

    limits,min(p2vmwlen), max(p2vmwlen);

    result = mouse(1, 2, "Drag your mouse of the global offset");
    offset = result(3)-result(1);


    
    HWlen = grow(Wlen_H_Lyman,
                 Wlen_H_Balmer,
                 Wlen_H_Paschen,
                 Wlen_H_Brackett,
                 Wlen_H_Pfund,
                 Wlen_H_Humphreys);
    HNames = grow(Name_H_Lyman,
                  Name_H_Balmer,
                  Name_H_Paschen,
                  Name_H_Brackett,
                  Name_H_Pfund,
                  Name_H_Humphreys);

    dat = yocoFileReadAscii(Y_SITE+"i/gemini_Catalog_K.txt");

    atmWlen = yocoStr2Double(dat(1,));
    atmStrg = yocoStr2Double(dat(4,));
    atmNames = array("Atm. line",numberof(atmWlen));

    idx = where((atmWlen>min(cwlen)-0.02e-6)&
                (atmWlen<max(cwlen)+0.02e-6));
    if(numberof(idx)!=0)
    {
        atmWlen  = atmWlen(idx);
        atmNames = atmNames(idx);
        atmStrg  = atmStrg(idx);
        idx      = sort(atmStrg);
        atmWlen  = atmWlen(idx);
        atmNames = atmNames(idx);
        atmStrg  = atmStrg(idx);

        nFeat  = numberof(atmWlen);
        mdat   = rdat = [];
        //offset = 0;
        for(k=1;k<=nFeat;k++)
        {
            if(atmStrg(k)<=maxGrade)
            {
                fma;
                newIdx = where((cwlen-offset<atmWlen(k)+0.03e-6)&
                               (cwlen-offset>atmWlen(k)-0.03e-6));
                newDat  = cspectrumvg(newIdx);
                newDat  = newDat/median(newDat);
                newWlen = cwlen(newIdx)-offset;

                newIdxG = where((geminiWlen<atmWlen(k)+0.03e-6)&
                                (geminiWlen>atmWlen(k)-0.03e-6));
                newGem     = geminiTrans(newIdxG);
                newGem     = newGem/median(newGem);
                newGemWlen = geminiWlen(newIdxG);

                plg,newDat,newWlen, color="blue";
                plg,newGem,newGemWlen,color="red";

                pldj,atmWlen(k),0,atmWlen(k),2;
                plt,atmNames(k)+", "+pr1(atmWlen(k)*1e6)+"!mm\nGrade: "+
                    pr1(atmStrg(k)),
                    atmWlen(k)+0.001e-6,
                    min(newDat)+(min(newDat)-avg(newDat))*0.01,
                    tosys=1;

                for(l=1;l<=numberof(atmWlen);l++)
                {
                    if((atmWlen(l)>min(newWlen))&&((atmWlen(l)<max(newWlen))))
                        pldj,atmWlen(l),0,atmWlen(l),2,type="dash";
                }

                limits,atmWlen(k)-0.03e-6,atmWlen(k)+0.03e-6,
                    min(newDat)+(min(newDat)-avg(newDat))*0.1,
                    max(newDat)+(max(newDat)-avg(newDat))*0.1;
                pltitle,"Please select the underlined feature\nright-click to ignore";
                result = mouse(1, 0, "Please select the underlined feature\nright-click to ignore");
                if(result(10)==1)
                {
                    grow, mdat, result(1)+offset;
                    grow, rdat, atmWlen(k);
                }
                if(numberof(mdat)>=1)
                {
                    offset = avg(mdat)-avg(rdat);
                }
            }
        }
    }

    idx = where((HWlen>min(cwlen)-0.02e-6)&
                (HWlen<max(cwlen)+0.02e-6));
    if(numberof(idx)!=0)
    {
        HWlen = HWlen(idx);
        HNames = HNames(idx);
        idx = sort(HWlen);
        HWlen = HWlen(idx);
        HNames = HNames(idx);

        nFeat = numberof(HWlen);
        for(k=1;k<=nFeat;k++)
        {
            fma;
            newIdx = where((cwlen-offset<HWlen(k)+0.03e-6)&
                           (cwlen-offset>HWlen(k)-0.03e-6));
            newDat = cspectrumvg(newIdx);
            newDat = newDat/avg(newDat)
                newWlen = cwlen(newIdx)-offset;
            plg,newDat,newWlen, color="blue";
            plg,geminiTrans,geminiWlen,color="red";
            pldj,HWlen(k),0,HWlen(k),2;
            plt,HNames(k)+", "+pr1(HWlen(k)*1e6)+"!mm",
                HWlen(k)+0.001e-6,0.2,tosys=1;
            limits,HWlen(k)-0.03e-6,HWlen(k)+0.03e-6,0,max(newDat)*1.2;
            result = mouse(1, 0,
                           "Please select the underlined feature\nright-click to ignore");
            if(result(10)==1)
            {
                grow, mdat, result(1)+offset;
                grow, rdat, HWlen(k);
            }
            if(numberof(mdat)>=1)
            {
                offset = avg(mdat)-avg(rdat);
            }
        }
    }

    if(numberof(mdat)==0)
        readableError, "At least one reference wavelength should be identified !!";
    else if(numberof(mdat)==1)
        coef = [-offset, 1, 0];
    else if(numberof(mdat)<=6)
        coef = grow(yocoMathPolyFit(1, mdat, rdat),0);
    else
        coef = yocoMathPolyFit(2, mdat, rdat);

    cwlen2 = yocoMathPoly(cwlen, coef);

    fma;
    plg,cspectrumvg, cwlen2, color="blue";
    plg,geminiTrans,geminiWlen,color="red";
    limits,min(cwlen2),max(cwlen2), 0, max(cspectrumvg);

    window,3;
    plg, rdat, mdat, type="none",marker='\2';
    x = span(min(mdat)*0.99,max(mdat)*1.01,100);
    plg, yocoMathPoly(x, coef), x;

    write," pouet" ;
    /* Define the default for the outputOiFile from the
       first and last files. By default, create an output files with
       all similar part of the file names */
    if(is_void(outputCoefFile))
    {
        yocoFileSplitName,calFiles(1),d,f1,e;
        yocoFileSplitName,calFiles(0),d,f2,e;

        stringStart = f1;
        stringEnd = f2;
        w = 1;
        while((strmatch(strpart(stringStart,1:w),
                        strpart(stringEnd,1:w)))&&
              (w<=strlen(f1)))
        {
            w++;
        }
        outputCoefFile = yocoStrSplit(stringStart,"OIDATA")(1) +
            "-" +
            strpart(stringEnd,w:);
    }
    yocoFileSplitName,outputCoefFile,d,f,e;
    outFile = d+f+"_COEFF.txt";
    write, "Writing the file "+outFile;
    fh = open(outFile,"w");
    write,fh,"#Wavelength solution coefficients";
    write,fh,coef;
    close,fh;

    return coef;
}

/*************************************************************/

func computeWlenSolutionLR(&shift, &stretch, calFiles=, inputP2vmFile=, outputCoefFile=, maxGrade=)
    /* DOCUMENT computeWlenSolutionLR(&shift, &stretch, calFiles=, inputP2vmFile=, outputCoefFile=, maxGrade=)

       DESCRIPTION

       PARAMETERS
       - shift         : 
       - stretch       : 
       - calFiles      : 
       - inputP2vmFile : 
       - outputCoefFile: 
       - maxGrade      : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    require,"gemini_Transmission.i";
    require,"lmfit.i";
    wkll;

    if(is_void(maxGrade))
        maxGrade = 3;

    if(amdlibFileChooser("Choose Calib OI files", calFiles) == 0)
        readableError, "OI files MUST be selected for this script!";

    // Load P2VM file
    if (amdlibLoadP2vm( mat, p2vmphot, p2vmVk, p2vmwlen, p2vmflag,
                        p2vmcalibPhase, p2vmnbTels, p2vmkeyNames,
                        p2vmkeyValues, inputP2vmFile=inputP2vmFile) == 0)
    {
        readableError, "P2VM MUST be selected for this script!";
    }
    else
    {
        p2vmLoaded = 1;
        p2vmwlen = p2vmwlen*1e-9;

        // Get P2VM dimensions
        nbP2vmWlen = numberof(p2vmwlen);
        nbP2vmBases = dimsof(mat)(2)/2;

        // Build Photometric & interferometric array from the Phot array
        newPhot = array(0.0,nbP2vmWlen,p2vmnbTels+1,p2vmnbTels+2*nbP2vmBases);
        for(k=1;k<=p2vmnbTels;k++)
            newPhot(,k,k) = p2vmphot(,k,1);
        tel1 = [1,2,1];
        tel2 = [2,3,3];
        for(k=1;k<=2*nbP2vmBases;k++)
        {
            l = (k+1)/2;
            newPhot(,tel1(l),k+p2vmnbTels) = p2vmphot(,1,k+1);
            newPhot(,tel2(l),k+p2vmnbTels) = p2vmphot(,2,k+1);
            newPhot(,0,k+p2vmnbTels) = p2vmphot(avg,3,k+1);
        }

        // Construct calibration spectra
        p2vmSpec = transpose(newPhot(,1:p2vmnbTels,sum));
        p2vmIntf = newPhot(,0,sum);
    }

    // Load data from calibrator star
    cspectrum = cwlen = p2vmfot = pwlen = [];
    for(k=1;k<=numberof(calFiles);k++)
    {
        amdlibLoadOiData, wlen,bandWidth,time,fringeSNR,sqVis,sqVisErr,
            uvCoord,bandFlag,pistonOPDArray,sigmaPistonArray,diffVisAmp,
            diffVisAmpErr,diffVisPhase,diffVisPhaseErr,vis3Amp,vis3AmpErr,
            vis3Phase, vis3PhaseErr,cpxVis,cpxVisErr,visCovRI,
            uv1Coord,uv2Coord,fluxSumPiPj,
            fluxRatPiPj,PiMultPj,cspectru,cspectrumEr,inputOiFile=calFiles(k);

        grow,cspectrum,cspectru/cspectru(,avg);
        grow,cspectrumErr,cspectrumEr;
        grow,cwlen,wlen;

        pwlentmp = idx = [];
        for(l=1;l<=numberof(wlen);l++)
        {
            test = where((wlen(l)-0.00001e-6<=p2vmwlen)&
                         (wlen(l)+0.00001e-6>=p2vmwlen))(1);
            if(numberof(test)!=0)
            {
                grow, pwlentmp, p2vmwlen(test);
                grow, idx, test;
            }
        }

        // Reshape calibration spectrum to the right number of wavelengths
        grow, p2vmfot, p2vmSpec(,idx)/p2vmSpec(,idx)(,avg);
        grow, pwlen, pwlentmp;

    }

    cspectrumN = cspectrum / p2vmfot(..,-);

    srt = sort(cwlen);
    cwlen = cwlen(srt);
    p2vmfot = p2vmfot(,srt);
    cspectrum = cspectrum(,srt);
    cspectrumN = cspectrumN(,srt);


    mw = min(cwlen);
    Mw = max(cwlen);

    interv = [mw-0.2*(Mw-mw), Mw+0.2*(Mw-mw)];
    idx = where((geminiWlen>interv(1))&(geminiWlen<interv(2)));
    resol = 35;
    gresol = geminiWlen(idx)(avg)/geminiWlen(idx)(dif)(avg)/2;
    fac = gresol/resol;
    sgemTrans = gaussm(geminiTrans(idx)/geminiTrans(idx)(avg),fac);

    wlenPlot = cwlen;
    shift = 0.0;
    stretch = 1.0;
    vals = [];

    while (1==1)
    {
        if(!is_void(vals))
        {
            shift = vals(1);
            stretch = vals(2);
        }

        window,1,width=600,height=600;
        fma;

        wlenPlot = stretch * (cwlen - avg(cwlen)) + avg(cwlen) + shift;

        for(k=1;k<=3;k++)
            plg,cspectrumN(k,,1)/cspectrumN(k,avg,1),wlenPlot;
        plg,sgemTrans,geminiWlen(idx),color="red";


        limits, interv(1),interv(2);

        vals = [shift, stretch];

        // Get GUI information from user
        result = yocoGuiChangeNumber(vals, fig2ch, once=1, kill=0, win=0);

        if (result=="ok")
        {
            // return shifts
            write,"Shift:",vals(1),"Stretch:",vals(2);
            return shifts;
        }
    }

}

/*************************************************************/

func applyWlenSolutionMR(inputOiFiles=, coeffFile=)
    /* DOCUMENT applyWlenSolutionMR(inputOiFiles=, coeffFile=)

       DESCRIPTION

       PARAMETERS
       - inputOiFiles: 
       - coeffFile   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    local wlen;
    if(amdlibFileChooser("Choose a Coefficients file", coeffFile) == 0)
    {
        coef = computeWlenSolutionMR(calFiles=inputOiFiles);
    }
    else
    {
        // load coeff file
        dat  = yocoFileReadAscii(coeffFile, nbCols=1000);
        coef = yocoStr2Double(dat(,2));

        wc = where(coef!=0);
        coef = coef(:wc(0));
    }

    message = "Please choose OI files to apply change in wavelength solution"; 
    if (amdlibFileChooser(message, inputOiFiles) == 0)
    {
        readableError, "Input OI files are mandatory for this step. Please start again";
    }

    for(kFile=1;kFile<=numberof(inputOiFiles); kFile++)
    {
        /* DUMBLY READING TWICE THE FILE */
        _amdlibReadVis, inoiWave, inamberPhot, inoiVis, inoiVis2, inoiVis3,
            inamberOpd, inamberInsCfg, inoiArray, inoiTarget, inSpectrum, 
            inputOiFile=inputOiFiles(kFile);

        if(!amdlibLoadOiData(wlen, bandWidth, time, fringeSNR,
                             sqVis, sqVisErr, uvCoord,
                             bandFlag, pistonOPDArray,
                             sigmaPistonArray, diffVisAmp,
                             diffVisAmpErr, diffVisPhase,
                             diffVisPhaseErr, vis3Amp,
                             vis3AmpErr, vis3Phase, vis3PhaseErr,
                             cpxVis, cpxVisErr, visCovRI,
                             uv1Coord, uv2Coord, fluxSumPiPj,
                             fluxRatPiPj, fluxProdPiPj,
                             inputOiFile=inputOiFiles(kFile)))
        {
            yocoGuiErrorBox, "reading "+inputOiFiles(kFile);//+inputSciFiles;
        }

        wlen = yocoMathPoly(wlen, coef);

        nbFrames = dimsof(sqVis)(4);
        nbWlen   = dimsof(sqVis)(3);
        nbBases  = dimsof(sqVis)(2);

        if(numberof(keepBase)>nbBases)
            keepBas = [1];
        else
            keepBas = keepBase;

        MJD      = (*inoiVis.table).dateObsMJD;
        targetId = (*inoiVis.table).targetId;
        time     = (*inoiVis.table).time;
        DIT      = (*inoiVis.table).expTime;
        stations = (*inoiVis.table).stationIndex;

        if (nbBases == 3)
        {
            stations3 = (*inoiVis3.table).stationIndex;
            U1 = transpose(array(uv1Coord.u,1));
            V1 = transpose(array(uv1Coord.v,1));
            U2 = transpose(array(uv2Coord.u,1));
            V2 = transpose(array(uv2Coord.v,1));
        }

        U = uvCoord.u;
        V = uvCoord.v;

        amdlibSetWave, newWave,
            wlen*yocoAstroSI.m/yocoAstroSI.nm,
            bandWidth*yocoAstroSI.m/yocoAstroSI.nm;

        yocoFileSplitName,inputOiFiles(kFile),d,f,e;
        
        if (amdlibGetProductDir(d+"/", outputDir, suffix="WSOL",up=1) == 0)
        {
            yocoError, "Could not construct outputDir", , 1;
            return 0;
        }
        outputDir = string(outputDir);

        if(!open(outputDir, "r", 1))
        {
            write,"Creating",outputDir,"..."
                /* If outputDir does not exist, obviously force overwrite */
                overwrite = 1;
            /* Create the product dir if not existing */
            mkdir, outputDir;
        }
        
        outputFileName = outputDir+f+e;

        write, "re-writing file "+outputFileName;

        status = amdlibWriteOiFile(outputFileName,
                                   pointer(nil), &inoiArray, 
                                   &inoiTarget, &newWave,
                                   &inamberPhot, &inoiVis, 
                                   &inoiVis2, &inoiVis3, &inamberOpd,
                                   &inSpectrum);

        if (status == amdlibFAILURE)
        {
            yocoGuiErrorBox,"Could not save result file\n";
        }

        write, "dump tables";
        _amdlibDumpAllKeywords, 1, inputOiFiles(kFile), outputFileName;
    }
}

/*************************************************************/

func applyWlenSolutionLR(shift, stretch, inputOiFiles=)
    /* DOCUMENT applyWlenSolutionLR(shift, stretch, inputOiFiles=)

       DESCRIPTION

       PARAMETERS
       - shift       : 
       - stretch     : 
       - inputOiFiles: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(shift))
        readableError, "shift is mandatory. Input zero if needed.";

    if(is_void(stretch))
        readableError, "stretch is mandatory. Input zero if needed.";

    message = "Please choose OI files to apply change in wavelength solution"; 
    if (amdlibFileChooser(message, inputOiFiles) == 0)
    {
        return 0;
    }

    filterOiFile,
        inputOiFile = inputOiFiles,
        outputOiFile = inputOiFiles,
        wstretch = stretch,
        woffset = shift;
}

/*************************************************************/

func lineSpec(x, params)
    /* DOCUMENT lineSpec(x, params)

       DESCRIPTION
       Returns a gaussian function, normalized so that the integral is equal to 1

       PARAMETERS
       - x     : is the abscissa
       - params: is an array of parameters for this function :
       o sigma = params(1)
       o sigma = params(1)
       o offset = params(2)

       CAUTIONS
       The parameters are given in an array so that fitting procedures
       (LMfit or curvefit) can use the function directly without modifications

       EXAMPLES 
       > x = span(0,10,100);
       > sigma=1;
       > offset=5;
       > params=[sigma,offset];
       > plg,yocoMathGauss(x,params),x;

       SEE ALSO
       porte, gauss, lorentz, voigt, raie, sinc
    */
{
    sigma = double(params(1));
    gamma = double(params(2));
    offset = double(params(3));
    cont = double(params(4));
    if(numberof(params)==5)
        fact = double(params(5));
    else
    {
        fact = double(params(4));
        cont=1.0;
    }
        
    // G = cont + fact * yocoMathVoigt(x, [sigma, gamma, offset]);
    G = cont + fact * sigma*sqrt(2*pi)*yocoMathGauss(x, [sigma, offset]);

    return G;
}

/*************************************************************/

func amdlibLoadSpectrums(&spec, &wlen, &err, &band, specFile=, woffset=)
    /* DOCUMENT amdlibLoadSpectrums(&spec, &wlen, &err, &band, specFile=, woffset=)

       DESCRIPTION

       PARAMETERS
       - spec    : 
       - wlen    : 
       - err     : 
       - band    : 
       - specFile: 
       - woffset : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    
    SPEC = WLEN = ERR = BAND = [];
    for(k=1;k<=numberof(specFile);k++)
    {
        amdlibLoadSpectrum,spec, wlen, err, band, specFile=specFile(k), woffset=woffset;
        
        grow,SPEC,&spec;
        grow,WLEN,&wlen;
        grow,ERR,&err;
        grow,BAND,&band;
    }

    // Merge spectra of different forms
    mergeSpectrum,SPEC, WLEN, ERR, BAND, spec, wlen, err, band;

}

/*************************************************************/

func amdlibLoadSpectrum(&spec, &wlen, &err, &band, specFile=, woffset=)
    /* DOCUMENT amdlibLoadSpectrum(&spec, &wlen, &err, &band, specFile=, woffset=)

       DESCRIPTION

       PARAMETERS
       - spec    : 
       - wlen    : 
       - err     : 
       - band    : 
       - specFile: 
       - woffset : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    wlen = band = spec = err = [];
    if(is_void(woffset))
        woffset = 0;

    if(amdlibFileChooser("Choose Science OI files", specFile) == 0)
        return 0;

    if(strmatch(specFile,".fits"))
    {
        // Go directly to spectrum table
        
        fh    = cfitsio_open(specFile,"r");        // open a existing file
        cfitsio_goto_hdu,fh,"AMBER_SPECTRUM";            // go to a HDU
        table = cfitsio_read_bintable(fh,titles);    // read the binary table
        cfitsio_close,fh;                            // close the file

        wlen  = *table(where(titles=="EFF_WAVE")(1))+woffset;
        band  = *table(where(titles=="EFF_BAND")(1));
        spec  = *table(where(titles=="SPECTRUM")(1));
        // if(numberof(where(titles=="INTERF_SPECTRUM"))!=0)
        // {
        //     specI = *table(where(titles=="INTERF_SPECTRUM")(1));
        //     spec  = transpose(grow([specI],transpose(spec)));
        // }
        err   = *table(where(titles=="SPECTRUM_ERROR")(1));
    }
    else
    {
        dat = yocoFileReadAscii(specFile);
        titles = dat(,1);

        // Find spectrum
        wspec = where(strmatch(titles,"spec",1)&!strmatch(titles,"err",1));
        if(numberof(wspec)==1)
            wspec = wspec(1);

        // Find spectrum error
        werr  = where(strmatch(titles,"err",1));
        if(numberof(werr)==1)
            werr = werr(1);

        // Find wavelengths table
        wwlen = where(strmatch(titles,"wlen",1))(1);
        if(numberof(wwlen)==1)
            wwlen = wwlen(1);

        // Find bandwidth
        wbnd  = where(strmatch(titles,"band",1));
        if(numberof(wbnd)==1)
            wbnd = wbnd(1);

        spec = yocoStr2Double(dat(wspec,2:));
        wlen = yocoStr2Double(dat(wwlen,2:))+woffset;
        if(numberof(werr)!=0)
            err = yocoStr2Double(dat(werr,2:));
        else
            err = array(0.0,dimsof(spec));
                
        if(numberof(wbnd)!=0)
            band = yocoStr2Double(dat(wbnd,2:));
        else
            band = array(0.0,dimsof(wlen));
    }

}

/*************************************************************/

func mergeSpectrum(spec, wlen, err, band, &SPEc, &WLEn, &ERr, &BANd, &pSPEc, &pWLEn, &pERr, &pBANd)
    /* DOCUMENT mergeSpectrum(spec, wlen, err, band, &SPEc, &WLEn, &ERr, &BANd, &pSPEc, &pWLEn, &pERr, &pBANd)

       DESCRIPTION
       Merge spectra

       PARAMETERS
       - spec : 
       - wlen : 
       - err  : 
       - band : 
       - SPEc : 
       - WLEn : 
       - ERr  : 
       - BANd : 
       - pSPEc: 
       - pWLEn: 
       - pERr : 
       - pBANd: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    yocoLogInfo,"Merging spectra...";

    // Case where there is only one spectrum as input: do nothing
    if(numberof(spec)==1)
    {
        SPEc = *spec(1);
        WLEn = *wlen(1);
        ERr  = *err(1);
        BANd = *band(1);

        yocoLogInfo,"Nothing to merge, exiting...";
        return 1;
    }
    
    // Initialize first spectrum
    SPEc = *spec(1);
    WLEn = *wlen(1);
    ERr  = *err(1);
    BANd = *band(1);
    
    // Buildup pointers array
    pSPEc = &SPEc;
    pERr  = &ERr;
    pWLEn = &WLEn;
    pBANd = &BANd;
    
    nbSpec = numberof(spec);
    
    // idx is a counter to check all pairs of regions until there is
    // an adjacent one to the work region
    alldone = 0;
    idx = indgen(nbSpec-1)+1;

    // Loop on all possible pairs of regions
    count=0;
    while(alldone==0)
    {
        // Increment pair count
        count++;
        
        // Get current region index
        k  = idx(1);

        // Get current spectrum, wavelength, error, and bandwidth.
        s2 = *spec(k);
        w2 = *wlen(k);
        e2 = *err(k);
        b2 = *band(k);

        // Match main region wavelengths with the current wavelength
        nM=M=M2 = [];
        for(w=1;w<=numberof(w2);w++)
        {
            match = where(w2(w)==WLEn);
            if(numberof(match)!=0)
                grow,M,match(1);
        }

        // Match current region wavelengths with the main wavelength
        for(w=1;w<=numberof(WLEn);w++)
        {
            match = where(WLEn(w)==w2);
            if(numberof(match)!=0)
                grow,M2,match(1);
        }

        // Get the non-matching wavelengths
        toto     = array(1,numberof(w2));
        toto(M2) = 0;
        nM       = where(toto);
        
        // In the case of no match, put current region at the end of idx,
        // without re-normalizing
        if(is_void(M))
        {
            idx = grow(idx(2:),idx(1));

            // If count is larger than the possible number of combinations
            // (approximated here by square), then exit doing nothing
            if(count > (nbSpec-1)^2)
            {
                yocoLogInfo,"No matching wavelengths, exiting doing nothing...";            
                return 1;
            }
            continue;
        }
        // Case where some wavelengths match
        else
        {
            // Get conversion factor to apply to appending spectrum;
            val1 = median(SPEc(,M),0);
            val2 = median(s2(,M2),0);
            fac  = val1/val2;

            // Buildup pointers array
            grow,pSPEc,&(fac*s2);
            grow,pERr, &(fac*e2);
            grow,pWLEn,&w2;
            grow,pBANd,&b2;
            
            // add the new spectrum data to intersecting wavelengths, with a
            // weighted average
            SPEc(,M) = (1./ERr(,M)*SPEc(,M) + 1./(fac*e2(,M2))*fac*s2(,M2)) /
                (1./ERr(,M) + 1./(fac*e2(,M2)));
            ERr(,M)  = sqrt( (ERr(,M) + (fac*e2(,M2))) / 2. / 
                             (1./ERr(,M) + 1./(fac*e2(,M2))));
            
            if(numberof(nM)!=0)
            {
                // add the new spectrum data to intersecting wavelengths
                grow,SPEc,fac(,-)*s2(,nM);
                grow,ERr, fac(,-)*e2(,nM);
                grow,WLEn,w2(nM);
                grow,BANd,b2(nM);
            }
        }
        
        // End loop if there is nothing left to do
        if(numberof(idx)==1)
            break;
        
        // Remove current index from the todo list
        idx = idx(2:);
    }
    
    if(pointers!=1)
    {
        // Resort wavelengths
        wh   = sort(WLEn);
        WLEn = WLEn(wh);
        SPEc = SPEc(,wh);
        BANd = BANd(wh);
        ERr  = ERr(,wh);
    }
}

/*************************************************************/

func normalizeSpec(flux, lambda, &continuum, channelRefMat=, plagesCont=, contType=, contParams=)
    /* DOCUMENT normalizeSpec(flux, lambda, &continuum, channelRefMat=, plagesCont=, contType=, contParams=)

       DESCRIPTION
       Computes "instrumental" spectrum from the input flux. An optionnal
       matrix allows one to use versatile definition of the normalization of
       the spectrum

       PARAMETERS
       - flux         : 
       - lambda       : 
       - continuum    : 
       - channelRefMat: 
       - plagesCont   : 
       - contType     : 
       - contParams   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(contParams))
        contParams = 2;
    if(is_void(contType))
        contType="polynom";
    
    nbWlen = numberof(lambda);

    isCont = array(0,nbWlen);
    for(k=1;k<=dimsof(plagesCont)(3);k++)
    {
        isCont = isCont |
            (min(plagesCont(,k))<=lambda)&
            (max(plagesCont(,k))>=lambda);
    }
    wc = where(isCont);
    
    // Compute continuum
    if(contType="polynom")
        cont = yocoMathPolyFit(contParams,lambda(wc)-avg(lambda(wc)),flux(wc));
    continuum = yocoMathPoly(lambda-avg(lambda(wc)),cont);
    
    spec = flux / continuum;
    return spec;
}

func plotContBoxes(spec, wlen, plagesCont)
    /* DOCUMENT plotContBoxes(spec, wlen, plagesCont)

       DESCRIPTION

       PARAMETERS
       - spec      : 
       - wlen      : 
       - plagesCont: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // Plot continuum boxes
    isCont = array(0,nbWlen);
    for(k=1;k<=dimsof(plagesCont)(3);k++)
    {
        isCont = isCont |
            (min(plagesCont(,k))<=wlen)&
            (max(plagesCont(,k))>=wlen);
    }
    wc = where(isCont);
    mx = max(spec(,wc));
    mn = min(spec(,wc));
    
    for(k=1;k<=dimsof(plagesCont)(3);k++)
    {
        if(anyof((plagesCont(,k)<max(wlen))&(plagesCont(,k)>min(wlen))))
        {
            p1=plagesCont(1,k);
            p2=plagesCont(2,k);
            plg,[mn,mx,mx,mn,mn]+offset,
                [p1,p1,p2,p2,p1]*1e6,type="dash";
        }
    }
}

/*******************************************************/

func amdlibShowSpectrum(&spec, &err, &wlen, &continuum, &smed, specFile=, offset=, factor=, kill=, color=, merge=, normalize=, multiple=, filtrIris=, plagesCont=, contParams=, contType=, woffset=, msize=)
    /* DOCUMENT amdlibShowSpectrum(&spec, &err, &wlen, &continuum, &smed, specFile=, offset=, factor=, kill=, color=, merge=, normalize=, multiple=, filtrIris=, plagesCont=, contParams=, contType=, woffset=, msize=)

       DESCRIPTION
       Display spectrum from ascii file or AMBER oifits file

       PARAMETERS
       - spec      : 
       - err       : 
       - wlen      : 
       - continuum : 
       - smed      : 
       - specFile  : 
       - offset    : 
       - factor    : 
       - kill      : 
       - color     : 
       - merge     : 
       - normalize : 
       - multiple  : 
       - filtrIris : 
       - plagesCont: 
       - contParams: 
       - contType  : 
       - woffset   : 
       - msize     : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    require,"lmfit.i";

    if(amdlibFileChooser("Choose Science OI files", specFile) == 0)
        write,"No file given as input. Considering table input";
    
    if(is_void(offset))
        offset = 0.;
    if(is_void(factor))
        factor = 1.0;
    if(is_void(multiple))
        multiple=0;

    if(kill==1)
        winkill;

    // window,1,width=600,height=600;

    maxSpec=0;

    if(is_void(spec))
        amdlibLoadSpectrums, spec, wlen, err, band,
            specFile=specFile, woffset=woffset;

      
    nbDims = dimsof(spec)(1);
    if(nbDims>1)
    {
        nbTel = dimsof(spec)(2);
        spec2=spec;
    }
    else
    {
        nbTel = 1;
        spec2 = transpose([spec]);
        if(!is_void(err))
            err = transpose([err]);
    }
    
    // Addup spectra before normalizing
    if((multiple==0)&&(dimsof(spec)(1)!=1))
    {
        spec2 = transpose([median(spec2,1)]);
        if(!is_void(err))
            err  = transpose([sqrt((err^2)(sum,) / dimsof(err)(2))]);
        nbTel = 1;
    }
        
    if(is_void(plagesCont))
    {
        // normalize by median
        smed = median(spec2,0);
        spec2 = spec2 / smed(,-);
        if(!is_void(err))
            err  = err / smed(,-);
    }
    else
    {
        CONT=[];
        for(k=1;k<=nbTel;k++)
        {
            spec2(k,) = normalizeSpec(spec2(k,),
                                      wlen,
                                      continuum,
                                      plagesCont=plagesCont,
                                      contParams=contParams,
                                      contType=contType);
            grow,CONT,[continuum];
        }
        CONT = transpose(CONT);
        if(!is_void(err))
            err  = err / CONT;
    }
    
    // Remove spurious IRIS fringes from spectrum
    if(filtrIris==1)
    {
        for (i=1; i<=nbTel; i++)
        {
            spec2(i,) = filtreArray(spec2(i,),plot=0);
        } 
    }
    
    if(is_void(err))
    {
        if(dimsof(spec)(1)==1)
            plg,spec2(*)*factor+offset,wlen*1e6, color=color;
        else if(dimsof(spec)(1)==2)
            pla,transpose(spec2)*factor+offset(k),wlen*1e6, color=color;
    }
    else
    {
        if(nbTel==1)
        {
            yocoPlotWithErrBars,spec2(*)*factor+offset,err(*), wlen*1e6, color=color,msize=msize;
        }
        else if(nbTel>1)
        {
            for(kt=1;kt<=dimsof(spec)(2);kt++)
                yocoPlotWithErrBars,spec2(kt,)*factor+offset,err(kt,), wlen*1e6, color=color,msize=msize;
        }
    }

    if(!is_void(plagesCont))
    {
        plotContBoxes, spec2, wlen, plagesCont;
    }

    maxSpec = max(maxSpec, max(spec2));

    spec = spec2;
        
    limits,,,0;
    xytitles,"Wavelength (!mm)","Intensity";
    hcps,"~/Spectrum.ps";
}

/*************************************************************/

func fixDateObs(file)
    /* DOCUMENT fixDateObs(file)

       DESCRIPTION

       PARAMETERS
       - file: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    /* Select file to be treated */
    message = "Choose the OI DATA"; 
    if (amdlibFileChooser(message, file) == 0)
    {
        return 0;
    }

    fh = cfitsio_open(file,"a"); // open a existing file
    nhdu = cfitsio_get_num_hdus(fh);
    for(k=1;k<=nhdu;k++)
    {
        cfitsio_goto_hdu,fh,k;
        cfitsio_write_comment,fh,"Data file from the letter Millour et al. 2009, A&A, 506, 49.";
        cfitsio_write_comment,fh,"Please cite this article if you use these datasets in a publication.";
        date_obs = cfitsio_get(fh,"DATE-OBS",comments); // read a key
        if(!is_void(date_obs))
        {
            if(date_obs!="")
            {
                storedateobs = date_obs;
            }
            else
            {
                if(is_void(storedateobs))
                {
                    yocoFileSplitName,file,,storedateobs;
                    storedateobs = strpart(storedateobs,1:10);
                }

                //storedateobs = "2006-02-18T03:19:49.0745"
                cfitsio_update_key, fh, "DATE-OBS",
                    storedateobs,"Date of observation";
                // cfitsio_write_key, fh, "DATE-OBS", storedateobs; // add a card in this HDU
            }
            print, date_obs;
        }
    } // list all the HDU type
    cfitsio_close, fh; 

}
//amdlibPlotV2fSpFreq,kill=0,posAng=-4*deg2rad,freqFact=1000*mas2rad,color="red",plotErr=1,xtitle="Spatial Frequency (cycles/as)"

/************************************************************************/

func checkBadPixelMap(badPixelFile, overFile, iFrame)
    /* DOCUMENT checkBadPixelMap(badPixelFile, overFile, iFrame)
  
       DESCRIPTION 
       Allows to edit and modify a bad pixel map in order to remove bad pixels that were not taken into account in the badPixels procedure.

       FILES

       OPTIONS

       RESULTS

       CAUTIONS 

       EXAMPLES 

       SEE ALSO: fyorick, fyorickHL, showRawData, showVis, showP2VM
    */
{
    yocoGuiWinkill;
    winkill;
    winkill;
    winkill;
    nButtons = 6;
    Buttons = array(Button, nButtons);

    Buttons.x = [0.15, 0.25, 0.35, 0.45, 0.55, 0.65];
    Buttons.y = array(0.34, nButtons);
    Buttons.dx = array(0.03, nButtons);
    Buttons.dy = array(0.01, nButtons);
    Buttons.text = ["PX-CH", "FRM+", "FRM-", "IMG", "QUIT", "CUT"];
    Buttons.font = array("courier", nButtons);
    Buttons.height = array(0, nButtons);
    Buttons.width = array(0, nButtons);

    if(amdlibFileChooser("RAW DATA FILE\nPlease enter a file to superimpose to the badPixels\n(right-click if you don't have one)", overFile)==1)
    {
        if(overFile == 0)
            overFile = "";
      
        amdlibFileChooser, "BIAS FILE\nPlease enter a bias file", biasFile;
        if(biasFile == 0)
            biasFile = "";
    }

    if(amdlibFileChooser("BAD PIXEL MAP\nplease enter a file for computing\n(right-click to exit)", badPixelFile) == 0)
        return 0;
  
    if(is_void(iFrame))
        iFrame = 1;

    badPixel = cfitsRead(badPixelFile);
    badPixel = badPixel(, , 1);

    minBad = min(badPixel);
    maxBad = max(badPixel);

    if(overFile!="")
    {
        amdlibLoadRawData(rawData, wlen, dark, phot, interf, bandwidth, inputRawFile=overFile, inputBiasFile=biasFile, firstFrame=, nbFrames=, calibrateRaw=1, convertToScience=0, photoOffset=);
      
        keys = _amdlibGetKwdVals(overFile, ["HIERARCH ESO DET WIN AMNC0",
                                            "HIERARCH ESO DET WIN AMNC1",
                                            "HIERARCH ESO DET WIN AMNC2",
                                            "HIERARCH ESO DET WIN AMNC3",
                                            "HIERARCH ESO DET WIN AMNC4",
                                            "HIERARCH ESO DET WIN AMNR0",
                                            "HIERARCH ESO DET WIN AMNR1",
                                            "HIERARCH ESO DET WIN AMNR2",
                                            "HIERARCH ESO DET WIN AMSC0",
                                            "HIERARCH ESO DET WIN AMSC1",
                                            "HIERARCH ESO DET WIN AMSC2",
                                            "HIERARCH ESO DET WIN AMSC3",
                                            "HIERARCH ESO DET WIN AMSC4",
                                            "HIERARCH ESO DET WIN AMSR0",
                                            "HIERARCH ESO DET WIN AMSR1",
                                            "HIERARCH ESO DET WIN AMSR2",
                                            "HIERARCH ESO DET NROW",
                                            "HIERARCH ESO DET NTEL"]);
        keyVals = yocoStr2Long(keys)(1,);
        width = keyVals(1:5);
        height = keyVals(6:8);
        xoff = keyVals(9:13);
        yoff = keyVals(14:16);
        nrow = keyVals(17);
        ntel = keyVals(18);
        nbFrames = dimsof(rawData)(0);
    }      

    jeVeuxSortir = 0;
    imgPlot = 1;
    zoom = 1.0;
    size = dimsof(badPixel)(2:);
    center = size/2;

    wkll; 
    window, 1, dpi=150, height=1200, width=1200;
    // palette, "stern.gp";
    palette, "heat.gp";

    title = "BadPixelFile : "+yocoStrReplace(yocoStrrtok(badPixelFile, "/")(2))+"\nFile : "+yocoStrReplace(yocoStrrtok(overFile, "/")(2))+" at Frame : ";

    while(jeVeuxSortir == 0)
    {
        window, 1;
        fma;
        if(NO_CRITE==0)
            cRite;

        button_plot, Buttons;

        pltitle, title+pr1(iFrame);
        pli, badPixel, cmin=minBad, cmax=maxBad;

        if((imgPlot == 1)&&(overFile!=""))
        {
            for(iTel=1;iTel<=ntel+2;iTel++)
            {
                for(jRow=1;jRow<=nrow;jRow++)
                {
                    if(iTel==1)
                        idx1 = 1;
                    else
                        idx1 = sum(width(1:iTel-1))+1;
                    idx2 = sum(width(1:iTel));
                    
                    if(jRow==1)
                        idy1 = 1;
                    else
                        idy1 = sum(height(1:jRow-1))+1;
                    idy2 = sum(height(1:jRow));
                    
                    plotData = rawData(idx1:idx2, idy1:idy2, iFrame);
                    
                    window,1;
                    pli, plotData, xoff(iTel), yoff(jRow), xoff(iTel)+width(iTel), yoff(jRow)+height(jRow);
                    
                }
            }
        }

        psfile = "/tmp/checkBadPixel.ps";
        if((verbose==1)&&(traceonce==0))
            yocoLogInfo, "Saving amber BadPixelMap in the file "+psfile;
        hcps, psfile;
        system, "chmod a+rw "+psfile;

        limits, center(1)-size(1)/(2*zoom), center(1)+size(1)/(2*zoom), center(2)-size(2)/(2*zoom), center(2)+size(2)/(2*zoom);

        result=mouse(-1, 1, "");
        xmouse1 = result(1);
        ymouse1 = result(2);
        xmouse2 = result(3);
        ymouse2 = result(4);
        if(xmouse2<xmouse1)
        {
            xmouse2 = result(1);
            xmouse1 = result(3);
        }
        if(ymouse2<ymouse1)
        {
            ymouse2 = result(2);
            ymouse1 = result(4);
        }
        xmouseNDC = result(7);
        ymouseNDC = result(8);
        systemWindow = result(9);
        whichButton = result(10);
        modifier = result(11);

        // write, xmouse1, ymouse1, systemWindow, whichButton;

        if(button_test(Buttons(5), xmouseNDC, ymouseNDC))
        {
            fma;
            if(NO_CRITE==0)
                cRite;
            pli, badPixel;
            newBadPixelFile = "/tmp/BadPixelMap-"+"Hadjara"+".fits";
            write, "Writing modified badPixelFile to " + newBadPixelFile;
            cfitsWrite, newBadPixelFile, badPixel;
            return 1;
        }
        else if((button_test(Buttons(1), xmouseNDC, ymouseNDC))||(modifier==4))
        {
            if(modifier!=4)
                result=mouse(-1, 0, "");
            badPixel(int(xmouse1)+1:int(xmouse2)+1, 
                     int(ymouse1)+1:int(ymouse2)+1) =
                1 - badPixel(int(xmouse1)+1:int(xmouse2)+1, 
                             int(ymouse1)+1:int(ymouse2)+1);
        }
        else if(button_test(Buttons(2), xmouseNDC, ymouseNDC))
            iFrame = (iFrame+1)%nbFrames;
        else if(button_test(Buttons(3), xmouseNDC, ymouseNDC))
            iFrame = (iFrame-1)%nbFrames;
        else if(button_test(Buttons(4), xmouseNDC, ymouseNDC))
            imgPlot = 1-imgPlot;
        else if(button_test(Buttons(6), xmouseNDC, ymouseNDC))
        {
            result=mouse(-1, 0, "");
            xmouse = result(3);
            ymouse = result(4);

            window, 2, dpi=75;

            fma;
            if(NO_CRITE==0)
                cRite;

            if((imgPlot == 1)&&(overFile!=""))
            {
                for(iTel=1;iTel<=ntel+2;iTel++)
                {
                    for(jRow=1;jRow<=nrow;jRow++)
                    {
                        if(iTel==1)
                            idx1 = 1;
                        else
                            idx1 = sum(width(1:iTel-1))+1;
                        idx2 = sum(width(1:iTel));
                    
                        if(jRow==1)
                            idy1 = 1;
                        else
                            idy1 = sum(height(1:jRow-1))+1;
                        idy2 = sum(height(1:jRow));
                    
                        write, xoff(iTel), width(iTel), yoff(jRow), height(jRow);
                        plotData = rawData(idx1:idx2, idy1:idy2, iFrame);
                    
                        window,1;
                        pli, plotData, xoff(iTel), yoff(jRow), xoff(iTel)+width(iTel), yoff(jRow)+height(jRow);
                        write, plotData(avg, int(ymouse)-y0, iFrame);
                    }
                }
            } 
        } 
        else if(whichButton == 1)
        {
            center = [xmouse1, ymouse1];
            zoom *= 2;
        }
        else if(whichButton == 3)
        {
            center = [xmouse1, ymouse1];
            zoom /= 2;
        }
        else if(whichButton == 2)
        {
            center = [xmouse1, ymouse1];
        }
    }
}

/***************************************************************************/

func get_color(intensite, phase)
    /* DOCUMENT get_color(intensite, phase)
       provide [r,g,b] arrays compatible with the color= option
       
       SEE ALSO:
    */
{
    if(is_void(phase))
        phase = 0.0;
  
    if(is_void(intensite))
        intensite = 1.0;
  
    if(numberof(where(intensite>1)) != 0)
        intensite(where(intensite>1)) = 0;
    
    col = array(double,3,dimsof(phase));

    phi = 6*(phase%(2*pi))/(2*pi);
  
    col(1,) = 255 * intensite * ((phi <= 1.5) * (phi >= 0.) * 1.0 +
                                 (phi <= 3.) * (phi >= 1.5) * (3. - phi)/1.5 +
                                 (phi <= 6.) * (phi > 5.) * 1.0 +
                                 (phi <= 5.) * (phi > 4.) * (phi - 4.));
  
    col(2,) = 255 * intensite * ((phi <= 1.5) * phi / 1.5 +
                                 (phi <= 3.) * (phi > 1.5) *  1. +
                                 (phi <= 4.) * (phi > 3.) * (4. - phi));
    
    col(3,) = 255 * intensite * ((phi <= 3.) * (phi > 2.) * (phi - 2) +
                                 (phi <= 5.) * (phi > 3.) * 1.0 +
                                 (phi <= 6.) * (phi > 5.) * (6. - phi));
  
    return int(col);
}
