/***************************************************************************
 *                                                                         *
 *                                AMBER++                                  *
 *               FFT-based data reduction soft for AMBER                   *
 *                                                                         *
 *                      Copyright 2011, F. Millour                         *
 *                            fmillour@oca.eu                              *
 *                                                                         *
 ***************************************************************************
 *
 * This file contains yorick scripts to reduce AMBER datasets using FFTs
 *
 * Please note that this script is distributed under the GPL licence,
 * available at http://www.gnu.org/licenses/gpl.txt
 *
 * Please ACKNOWLEDGE the use of this script for any use in a publication
 * by citing Millour et al. (in prep.)
 *
 * This is a sub-script. Pease consider reading amplAMBER.i for more info.
 *
 * Bugs should be logged in the "KNOWNBUGS" file
 * Missing features about this code should be logged in the "ROADMAP" file
 * New features should be logged in the "CHANGELOG" file
 *
 * $Id: amplData.i 703 2019-11-14 15:36:48Z fmillour $
 * $URL: https://svn.oca.eu/codes/AMBERpl/trunk/yorick/amplData.i $
 *
 ***************************************************************************/

yocoLogInfo,"#include,\"amplData.i\"";

// this causes some kind of loop...
// require,"amplAMBER.i"

if(is_void(amplAMBER))
    amplAMBER = "13.6";

func amplData(void)
    /* DOCUMENT amplData

       DESCRIPTION
       Loads AMBER dataset header and fills a structure
       with the relevant information

       FUNCTIONS
       - amplCfitsio_add_bintable : 
       - amplCfitsio_write_col    : 
       - amplCfitsio_write_column : 
       - amplData                 : This script
       - amplGrowData             : 
       - amplInterpolateBadPixels3: interpolate bad pixels on an image cube
       (x,y,t)
       - amplLoadAllRawDatas      : 
       - amplLoadCalib            : loads a calibration matrix
       - amplLoadData             : loads AMBER++ data
       - amplLoadDataHdr          : loads fits file header and stores it into
       an AMBER++ structure
       - amplLoadOiData           : loads OI data and stores it into a relevant
       AMBER++ structure
       - amplLoadOiDatas          : 
       - amplLoadRawData          : Loads raw dataset
       - amplLoadSciFiles         : 
       - amplResampleFringes      : reasmples fringes to straight line
       - amplSaveCalib            : saves AMBER++ calibration file
       - amplSaveData             : saves AMBER++ data structure
       - amplSaveOiData           : Saves OI dataset
       - amplShowAll              : 
       - amplShowOiData           : 
       - amplShowUV               : 
       - amplSubtractContinuum    : subtracts photometry
       - cfitsio_list_hdus        : amplCfitsio listing of HDU names
       - getKey                   : gets a particular key from log content

       SEE ALSO amdlibLoadRawData
    */
{
    extern MajorVersion;
    ver = yocoStr2Double(pr1(MajorVersion)+"."+
                         strpart(strtok("$Rev: 703 $",":")(2),2:-2));
    if (am_subroutine())
    {
        yocoLogInfo, format="script version: %1.3f\n", ver;
        yocoLogInfo,"Type \"help, amplAMBER\" to get the package version";
        help, amplData;
    }
    return ver;
}

/***************************************************************************/

func amplLoadDataHdr(&dataHdr, inputFile=, quiet=)
    /* DOCUMENT amplLoadDataHdr(&dataHdr, inputFile=, quiet=)

       DESCRIPTION
       Loads AMBER  dataset header and fills a structure
       with the relevant information

       PARAMETERS
       - dataHdr  : 
       - inputFile: the input  file(s)
       - quiet    : 

       SEE ALSO amdlibLoadData
    */
{
    if(!quiet)
        yocoLogInfo,"Loading data header...";
    // if(is_void(dataHdr))
    dataHdr = amplHeaderStruct();

    dataHdr.file = inputFile;
    
    /* Check the inputFile */
    message = "DATA FILE\n Enter the FITS data file to read header";
    if (amdlibFileChooser(message, inputFile) == 0)
    {
        yocoError,"'inputFile' not specified correctly.";
        return 0;
    }
    
    /* Getting the inputFile without the filepath */
    yocoFileSplitName, inputFile, directory, file, suffix;

    _amdlibReadFitsHeader, inputFile, keyName, keyValue, keyComment;
        
    // keyValue = keyComment = keyName = [];
    // for(k=1;k<=numberof(inputFile);k++)
    // {
    //     fh    = cfitsio_open(inputFile,"r");      // open a existing file
    //     kV = cfitsio_get(fh,"*",kC, kN, point=1); // read a key
    //     cfitsio_close,fh;                            // close the file
    //     grow,keyValue,  array(kV,1);
    //     grow,keyComment,array(kC,1);
    //     grow,keyName,   array(kN,1);
    // }
   
    // Dump all keywords
    dataHdr.keyName = &keyName;
    dataHdr.keyVal  = &keyValue;
    dataHdr.keyCom  = &keyComment;
    
    // Get OBs information
    if(!is_void(getKey(keyValue, keyName, "HIERARCH ESO OBS NAME")))
    {
        dataHdr.obName  = getKey(keyValue, keyName, "HIERARCH ESO OBS NAME")(1);
    }
    if(!is_void(getKey(keyValue, keyName, "HIERARCH ESO OBS TPLNO")))
    {
        dataHdr.tplNr   = getKey(keyValue, keyName, "HIERARCH ESO OBS TPLNO")(1);
    }
    if(!is_void(getKey(keyValue, keyName, "HIERARCH ESO OBS ID")))
    {
        dataHdr.obId    = getKey(keyValue, keyName, "HIERARCH ESO OBS ID")(1);
    }
    if(!is_void(getKey(keyValue, keyName, "HIERARCH ESO OCS P2VM ID")))
    {
        dataHdr.p2vmId  = getKey(keyValue, keyName, "HIERARCH ESO OCS P2VM ID")(1);
    }
   
    // Get equinox of observation
    dataHdr.EQUINOX = double(getKey(keyValue, keyName, "EQUINOX"))(1);
    //dataHdr.RAEP0   = double(getKey(keyValue, keyName, "RA"))(1);
    //dataHdr.DECEP0  = double(getKey(keyValue, keyName, "DEC"))(1);
   
    // Fill AMBER-specific information
    dataHdr.catg    = string(getKey(keyValue, keyName, "HIERARCH ESO DPR CATG"))(1);
    dataHdr.bcd     = string(getKey(keyValue, keyName, "HIERARCH ESO INS OPTI7 NAME"))(1);
    dataHdr.tracker = string(getKey(keyValue, keyName, "HIERARCH ESO DEL FT STATUS"))(1);
    
    // Fill dit & ndit
    dataHdr.dit     = double(getKey(keyValue, keyName, "HIERARCH ESO DET DIT"))(1);
    dataHdr.ndit    = long(getKey(keyValue, keyName, "HIERARCH ESO DET NDIT"))(1);
    
    // Fill unknown values
    dataHdr.VELTYP = "UNKNOWN";
    dataHdr.VELDEF = "OPTICAL";

    // Get instrument name
    if(!is_void(getKey(keyValue, keyName, "INSTRUME")))
        dataHdr.INSNAME = getKey(keyValue, keyName, "INSTRUME")(1);

    // Get array name
    tmp = getKey(keyValue, keyName, "TELESCOP");
    if(anyof(tmp))
    {
        dataHdr.ARRNAME = tmp(1);
        if(anyof(dataHdr.ARRNAME != getKey(keyValue, keyName, "TELESCOP")))
            error,"Array configuration mismatch in the input files";
    }
    
    // Get array geometry information, if any
    fh = cfitsio_open(inputFile, "r");
    hduz = cfitsio_list(fh);
    for(k=1;k<=numberof(hduz);k++)
    {
        // go to the HDU
        cfitsio_goto_hdu, fh, k;
        hduName = cfitsio_get(fh, "EXTNAME");
        if((hduName=="ARRAY_GEOMETRY")||(hduName=="OI_ARRAY"))
        {
            keyVals = cfitsio_get(fh, ["ARRAYX", "ARRAYY", "ARRAYZ"], com, nam);
            if(!is_void(keyVals))
            {
                if(is_string(keyVals))
                    keyVals = yocoStr2Double(keyVals);
                dataHdr.ARRAYXYZ = keyVals;
            }
            
            keyVals2 = cfitsio_get(fh, ["FRAME"], com, nam);
            if(!is_void(keyVals2))
                dataHdr.FRAME = keyVals2(1);
            
            // read the binary table
            table = cfitsio_read_bintable(fh,
                                          ["TEL_NAME","STA_NAME","STA_INDEX",
                                           "DIAMETER", "STAXYZ"]);
            dataHdr.TEL_NAME = table(1);
            dataHdr.STA_NAME = table(2);
            dataHdr.STA_INDEX = table(3);
            dataHdr.DIAMETER = table(4);
            dataHdr.STAXYZ = table(5);
        }
    }
    cfitsio_close,fh;
    
    // Get instrument mode
    dataHdr.INS_MODE = string(getKey(keyValue, keyName,
                                     "HIERARCH ESO INS MODE"))(1);
    if(anyof(dataHdr.INS_MODE!=string(getKey(keyValue, keyName,
                                             "HIERARCH ESO INS MODE"))))
        error,"Instrument mode mismatch in the input files";

    // Get stations and telescopes names
    // S1 = vals(1,6);
    // S2 = vals(1,7);
    // S3 = vals(1,8);
    // T1 = vals(1,45);
    // T2 = vals(1,46);
    // T3 = vals(1,47);
    // dataHdr.TEL_NAME = &[T1,T2,T3];
    // dataHdr.STA_NAME = &[S1,S2,S3];
    
    // Get date
    //if(!is_void((tmp=getKey(keyValue, keyName, "DATE-OBS"))))
    //  dataHdr.DATE_OBS = tmp(1);
                                        
    
    // Get DIT
    DIT = double(getKey(keyValue, keyName, "HIERARCH ESO DET DIT"));
    if(anyof(avg(DIT)!=DIT))
        error,"DIT mismatch! You should check your input files!";
    dataHdr.INT_TIME = DIT = DIT(1);

    // Get temperature in Kelvin
    temp = double(getKey(keyValue, keyName, "HIERARCH ESO ISS AMBI TEMP"))(1) + 273.15; // Observatory ambient temperature
    dataHdr.tempOut = avg(temp);

    //temp = double(getKey(keyValue, keyName, "HIERARCH ESO ISS TEMP LAB1")(1) + 273.15; // temp in lab
    //temp = double(getKey(keyValue, keyName, "HIERARCH ESO ISS TEMP LAB2")(1) + 273.15; // Temp in lab
    //temp = double(getKey(keyValue, keyName, "HIERARCH ESO ISS TEMP LAB3")(1) + 273.15; // Temp in lab eastern wall
    //temp = double(getKey(keyValue, keyName, "HIERARCH ESO ISS TEMP LAB4")(1) + 273.15; // Temp in lab
    //temp = double(getKey(keyValue, keyName, "HIERARCH ESO ISS TEMP TUN1")(1) + 273.15; // Temp in tunnel A west
    temp = double(getKey(keyValue, keyName, "HIERARCH ESO ISS TEMP TUN2"))(1) + 273.15; // Temp in tunnel centre
    //temp = double(getKey(keyValue, keyName, "HIERARCH ESO ISS TEMP TUN3")(1) + 273.15; // Temp in tunnel centre
    //temp = double(getKey(keyValue, keyName, "HIERARCH ESO ISS TEMP TUN4")(1) + 273.15; // Temp in tunnel L east
    dataHdr.tempIn = avg(temp);

    // Convert pressure from millibars to pascals
    dataHdr.press = double(getKey(keyValue, keyName, "HIERARCH ESO ISS AMBI PRES"))(1) * yocoAstroSI.mbar / yocoAstroSI.Pa;

    // relative humidity in percents
    dataHdr.watVap = double(getKey(keyValue, keyName, "HIERARCH ESO ISS AMBI RHUM"))(1);

    // UV coordinates
    R12s = double(getKey(keyValue, keyName, "HIERARCH ESO ISS PBL12 START"))(1);
    R12e = double(getKey(keyValue, keyName, "HIERARCH ESO ISS PBL12 END"))(1);
    R23s = double(getKey(keyValue, keyName, "HIERARCH ESO ISS PBL23 START"))(1);
    R23e = double(getKey(keyValue, keyName, "HIERARCH ESO ISS PBL23 END"))(1);
    R13s = double(getKey(keyValue, keyName, "HIERARCH ESO ISS PBL13 START"))(1);
    R13e = double(getKey(keyValue, keyName, "HIERARCH ESO ISS PBL13 END"))(1);
    
    A12s = double(getKey(keyValue, keyName, "HIERARCH ESO ISS PBLA12 START"))(1);
    A12e = double(getKey(keyValue, keyName, "HIERARCH ESO ISS PBLA12 END"))(1);
    A23s = double(getKey(keyValue, keyName, "HIERARCH ESO ISS PBLA23 START"))(1);
    A23e = double(getKey(keyValue, keyName, "HIERARCH ESO ISS PBLA23 END"))(1);
    A13s = double(getKey(keyValue, keyName, "HIERARCH ESO ISS PBLA13 START"))(1);
    A13e = double(getKey(keyValue, keyName, "HIERARCH ESO ISS PBLA13 END"))(1);
    
    if(R12s!=0)
    {
        dataHdr.UCOORD = &([span(R12s,R12e,dataHdr.ndit),
                            span(R23s,R23e,dataHdr.ndit),
                            span(R13s,R13e,dataHdr.ndit)] *
                           sin([span(A12s,A12e,dataHdr.ndit),
                                span(A23s,A23e,dataHdr.ndit),
                                span(A13s,A13e,dataHdr.ndit)]*deg2rad));
        dataHdr.VCOORD = &([span(R12s,R12e,dataHdr.ndit),
                            span(R23s,R23e,dataHdr.ndit),
                            span(R13s,R13e,dataHdr.ndit)] *
                           cos([span(A12s,A12e,dataHdr.ndit),
                                span(A23s,A23e,dataHdr.ndit),
                                span(A13s,A13e,dataHdr.ndit)]*deg2rad));
        dataHdr.U1COORD = &((*dataHdr.UCOORD)(,1));
        dataHdr.V1COORD = &((*dataHdr.VCOORD)(,1));
        dataHdr.U2COORD = &((*dataHdr.UCOORD)(,2));
        dataHdr.V2COORD = &((*dataHdr.VCOORD)(,2));
    }
    
    // Average position of delay lines
    A1L = double(getKey(keyValue, keyName, "HIERARCH ESO ISS CONF A1L"));
    A2L = double(getKey(keyValue, keyName, "HIERARCH ESO ISS CONF A2L"));
    A3L = double(getKey(keyValue, keyName, "HIERARCH ESO ISS CONF A3L"));
    // Starting and stopping points of delay lines
    DLT1_START = double(getKey(keyValue, keyName,
                               "HIERARCH ESO DEL DLT1 OPL START"));
    DLT2_START = double(getKey(keyValue, keyName,
                               "HIERARCH ESO DEL DLT2 OPL START"));
    DLT3_START = double(getKey(keyValue, keyName,
                               "HIERARCH ESO DEL DLT3 OPL START"));
    DLT1_END   = double(getKey(keyValue, keyName,
                               "HIERARCH ESO DEL DLT1 OPL END"));
    DLT2_END   = double(getKey(keyValue, keyName,
                               "HIERARCH ESO DEL DLT2 OPL END"));
    DLT3_END   = double(getKey(keyValue, keyName,
                               "HIERARCH ESO DEL DLT3 OPL END"));
    // Delay introduced, per telescope
    OPL1 = A1L + 0.5*( DLT1_START + DLT1_END);
    OPL2 = A2L + 0.5*( DLT2_START + DLT2_END);
    OPL3 = A3L + 0.5*( DLT3_START + DLT3_END);
    // The real path of air difference between the telescopes
    Bsinz = -[avg(OPL2-OPL1),
              avg(OPL3-OPL2),
              avg(OPL3-OPL1)];
    dataHdr.OPL = &Bsinz;

    // Get MJD
    MJD = double(getKey(keyValue, keyName, "MJD-OBS"));
    dataHdr.MJD = avg(MJD);

    // Compute hours, minute, seconds
    hour   = (avg(MJD)-floor(avg(MJD)))*24;
    dataHdr.hour = floor(hour);
    minute = (hour-floor(hour))*60;
    dataHdr.minute = floor(minute);
    second = (minute - floor(minute))*60;
    dataHdr.second = second;

    // Get the instrument mode
    resolution = string(getKey(keyValue, keyName, "HIERARCH ESO INS MODE"));
    if(anyof(resolution(1)!=resolution))
        error,"Instrument mode mismatch in the input files";
    dataHdr.specMode = resolution(1);
    
    target = string(getKey(keyValue, keyName, "HIERARCH ESO OBS TARG NAME"));
    // Check target name is the same in all input files
    if(anyof(target(1)!=target))
        error,"Target name mismatch in the input files";
    dataHdr.TARGET = target(1);

    // store AO-related values
    AO1_STREHL = double(getKey(keyValue, keyName,
                               "HIERARCH ESO COU AO1 STREHL_MEAN"));
    AO2_STREHL = double(getKey(keyValue, keyName,
                               "HIERARCH ESO COU AO2 STREHL_MEAN"));
    AO3_STREHL = double(getKey(keyValue, keyName,
                               "HIERARCH ESO COU AO3 STREHL_MEAN"));
    AO1_WFE    = double(getKey(keyValue, keyName,
                               "HIERARCH ESO COU AO1 WFE_MEAN"));
    AO2_WFE    = double(getKey(keyValue, keyName,
                               "HIERARCH ESO COU AO2 WFE_MEAN"));
    AO3_WFE    = double(getKey(keyValue, keyName,
                               "HIERARCH ESO COU AO3 WFE_MEAN"));

    if(numberof(AO1_STREHL)>=1)
    {
        dataHdr.AO_STREHL= &([avg(AO1_STREHL),avg(AO2_STREHL),avg(AO3_STREHL)]);
        dataHdr.AO_WFE   = &([avg(AO1_WFE),   avg(AO2_WFE),   avg(AO3_WFE)   ]);
    }
}

/***************************************************************************/

func getKey(kVal, kNam, myKey)
    /* DOCUMENT getKey(kVal, kNam, myKey)

       DESCRIPTION
       Retrieve the content of a key according to the log file

       PARAMETERS
       - kVal : 
       - kNam : 
       - myKey: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(kVal))
        return [];
    
    theVals = kVal(where(kNam==myKey));
    returnVal = [];
    for(k=1;k<=numberof(theVals);k++)
    {
        // Check if key is numeric
        if(yocoStrReplace(theVals(k),
                          ["1", "2", "3", "4", "5",
                           "6", "7", "8", "9", "0",
                           ".", "-", "e"],
                          ["","","","","","","","","","","","",""])=="")
        {
            dbl = yocoStr2Double(theVals(k));
            // Check if key is a long or a double
            lng = yocoStr2Long(theVals(k));
            if(dbl==lng)
                grow, returnVal, lng;
            else
                grow, returnVal, dbl;
        }
        else
            grow, returnVal, theVals(k);
    }
    return returnVal;
}

/***************************************************************************/

func amplLoadRawData(&rawDataStruct, inputRawFile=, inputBiasFile=, inputBPM=, inputFFM=, specCalShifts=, firstFrame=, nbFrames=, plot=, quiet=)
    /* DOCUMENT amplLoadRawData(&rawDataStruct, inputRawFile=, inputBiasFile=, inputBPM=, inputFFM=, specCalShifts=, firstFrame=, nbFrames=, plot=, quiet=)

       DESCRIPTION
       Loads AMBER raw datasets, apply cosmetics, and
       store them into a data structure

       PARAMETERS
       - rawDataStruct: the output data structure
       - inputRawFile : the input raw file(s)
       - inputBiasFile: the input bias file (dark)
       - inputBPM     : bad pixel file
       - inputFFM     : flat field file
       - specCalShifts: spec cal shifts to apply (default values applies)
       - firstFrame   : 
       - nbFrames     : 
       - plot         : 
       - quiet        : 

       SEE ALSO amdlibLoadRawData
    */
{
    if(!quiet)
        yocoLogInfo,"Loading raw data...";
    if(is_void(rawDataStruct))
        rawDataStruct = amplDataStruct();
    
    rawDataStruct.dataType = "Raw Data";
    
    /* Check the inputRawFile */
    message = "RAW DATA FILE\n Enter the AMBER raw data file to show";
    if (amdlibFileChooser(message, inputRawFile, 2) == 0)
    {
        yocoError,"'inputRawFile' not specified correctly.";
        return 0;
    }
    
    /* Check the bias file */
    message = "BIAS FILE\nEnter a DARK or a SKY file";
    if (amdlibFileChooser(message, inputBiasFile, 3) == 0)
    {
        yocoError,"'inputBiasFile' not specified correctly.";
        return 0;
    }
    
    /* Check flatFieldFile and badPixelFile */
    if ( !amdlibCheckBadPixelFile() ||
         !amdlibCheckFlatFieldFile()  )
        return 0;

    if(is_void(firstFrame))
        firstFrame = 1;
    else
        firstFrame = int(firstFrame);

    if(!is_void(nbFrames))
        nbFrames = int(nbFrames);
    
    
    /* Getting the inputRawFile without the filepath */
    yocoFileSplitName, inputRawFile, directory, file, suffix;

    hdr = [];
    // Read the data file header
    amplLoadDataHdr, hdr, inputFile=inputRawFile(1), quiet=quiet;
    rawDataStruct.hdr = hdr;

    // pompage amdlibComputeP2vm (écrit par bibi)
    // fixed shifts and dates of changes
    FIX_SHIFTS_LR  = [[3,   3,    1],
                      [3.4, 3.4, -0.2],
                      [5,   4,   0]];
    FIX_SHIFTS_MR  = [[0.873, 2.064,  0.472],
                      [2.000, 2.576, -0.263],
                      [4.008, 2.850, -1.319]];
    FIX_SHIFTS_HR  = [[-1, 1,  0],
                      [ 1, 2, -2],
                      [ 0, 2, -2]];
    FIX_SHIFTS_MJD = [0,54353, 54526,100000000];
    
    /* find the time interval */
    period = digitize(avg(rawDataStruct.hdr.MJD), FIX_SHIFTS_MJD) - 1;
    
    /* find the spectral mode */
    if(strmatch(rawDataStruct.hdr.specMode, "Low_JHK"))
    {
        if(!quiet)
            yocoLogInfo,"Using LR spectral shifts";
        specCalShifts = FIX_SHIFTS_LR(,period);
    }
    else if(strmatch(rawDataStruct.hdr.specMode, "Medium"))
    {
        if(!quiet)
            yocoLogInfo,"Using MR spectral shifts";
        specCalShifts = FIX_SHIFTS_MR(,period);
    }
    else if(strmatch(rawDataStruct.hdr.specMode, "High"))
    {
        if(!quiet)
            yocoLogInfo,"Using HR spectral shifts";
        specCalShifts = FIX_SHIFTS_HR(,period);
    }
    else
    {
        yocoError,"Cannot read the Spectral Mode of these observations","Please check the input files";
        return 0;
    }
    
    /* keep only integer */
    specCalShifts = lround(specCalShifts); 
    
    /* some verbose */
    if(!quiet)
        yocoLogInfo,"This observation was made between (mjd): "+
            pr1(FIX_SHIFTS_MJD( [period, period+1] ));
    
    /* verbose */
    if(!quiet)
        yocoLogInfo, "The spectral resolution is "+rawDataStruct.hdr.specMode;
    if(!quiet)
        yocoLogInfo, "The used spectral shifts are (pixels): "+pr1(specCalShifts);
    // End of pompage amdlibComputeP2vm
    
    // Loop on number of input files for loading
    dark = phot = interf = time = [];
    totFrames = 0;
    for(kFile=1;kFile<=numberof(inputRawFile);kFile++)
    {
        // Use the AMBER load function
        errMod = yocoErrorGet();
        yocoErrorSet,0;
        status = amdlibLoadAndCalRawData(framesT, wlenT, darkT, photT, interfT,
                                         inputRawFile=inputRawFile(kFile),
                                         inputBiasFile=inputBiasFile,
                                         p2vmFile=p2vmFile,
                                         firstFrame=firstFrame,
                                         nbFrames=nbFrames,
                                         calibrateRaw=1,
                                         convertToScience=0);
        if (status == 0)
        {
            yocoError, "Could not load and calibrate raw data";
            return 0;
        }
        yocoErrorSet,errMod;
        
        // Get the TIME column from the raw data file
        fh   = cfitsio_open(inputRawFile(kFile), "r");
        hduz = cfitsio_list(fh);
        for(k=1;k<=numberof(hduz);k++)
        {
            // go to the HDU
            cfitsio_goto_hdu, fh, k;
            hduName = cfitsio_get(fh, "EXTNAME");
            if((hduName=="IMAGING_DATA"))
            {
                // write,"First frame",firstFrame;//,nbFrames;
                // read the binary table                
                TIME = cfitsio_read_column(fh, "TIME", ttype, anynul,
                                           frow=firstFrame, nrows=nbFrames);
                // FIX: TIME array sometimes not same dimension when zero frames...
                TIME = TIME(where(TIME!=0));
            }
        }
        cfitsio_close,fh;
                    
        // Apply spectral shifts
        photT2 = photT;
        for(kBase=1;kBase<=3;kBase++)
        {
            photT2(kBase,..) = roll(photT(kBase,..),[0,-specCalShifts(kBase),0]);
        }

        // Crop the arrays with the right dimensions each
        minS = min(specCalShifts);
        if(minS<0)
        {
            darkT   =   darkT( ,-minS:,);
            photT   =  photT2(,,-minS:,);
            interfT = interfT( ,-minS:,);
            wlenT   =   wlenT(  -minS:);
        }
        else if(minS>=0)
        {
            darkT   =   darkT( ,:-minS,);
            photT   =  photT2(,,:-minS,);
            interfT = interfT( ,:-minS,);
            wlenT   =   wlenT(  :-minS);
        }
        
        maxS = max(specCalShifts);
        if(maxS>=0)
        {
            darkT   =   darkT( ,:-maxS,);
            photT   =   photT(,,:-maxS,);
            interfT = interfT( ,:-maxS,);
            wlenT   =   wlenT(  :-maxS);
        }
        else if(maxS<0)
        {
            darkT   =   darkT( ,-maxS:,);
            photT   =   photT(,,-maxS:,);
            interfT = interfT( ,-maxS:,);
            wlenT   =   wlenT(  -maxS:);
        }

        // Sort the wavelengths in the right direction
        if(anyof(wlenT != wlenT(sort(wlenT))))
        {
            yocoLogWarning,"Resorting wavelengths!";
            darkT   =   darkT( ,::-1,);
            photT   =   photT(,,::-1,);
            interfT = interfT( ,::-1,);
            wlenT   =   wlenT(  ::-1);
        }
                    
        nbFrames  = dimsof(interfT)(0);
        
        // Append ("Grow") all exposures together
        if(kFile==1)
        {
            dark   = darkT;
            phot   = photT;
            interf = interfT;
            time   = TIME;//rawDataStruct.hdr.MJD*3600*24 + (indgen(nbFrames)-1) * rawDataStruct.hdr.INT_TIME;
        }
        else if(totFrames + nbFrames >= dimsof(interf)(0))
        {
            grow, interf, array(0.0, dimsof(interf));
            grow,   phot, array(0.0, dimsof(phot));
            grow,   dark, array(0.0, dimsof(dark));
            grow,   time, array(0.0, dimsof(time));
        }
        
        dark(..,totFrames+1:totFrames+nbFrames)   = amplInterpolateBadPixels3(darkT);
        phot(..,totFrames+1:totFrames+nbFrames)   = amplInterpolateBadPixels3(photT);
        interf(..,totFrames+1:totFrames+nbFrames) = amplInterpolateBadPixels3(interfT);
        //        time(totFrames+1:totFrames+nbFrames)      = rawDataStruct.hdr.MJD + (indgen(nbFrames)-1) * rawDataStruct.hdr.INT_TIME;
        //
        // bug-fix -- time should be in seconds
        //
        time(totFrames+1:totFrames+nbFrames)      = TIME(1:nbFrames);//rawDataStruct.hdr.MJD*3600*24 + (indgen(nbFrames)-1) * rawDataStruct.hdr.INT_TIME;
        totFrames += nbFrames;
    }
    
    dark   =   dark(..,:totFrames);
    phot   =   phot(..,:totFrames);
    interf = interf(..,:totFrames);
    time   =   time(   :totFrames);
    
    dimDk = dimsof(dark);
    dimP  = dimsof(phot);
    dimIn = dimsof(interf);
    
    nbPix                  = dimIn(2);
    
    rawDataStruct.hdr.ndit = totFrames;
    rawDataStruct.X        = &indgen(nbPix);
    rawDataStruct.Xlab     = "Pixels";
    rawDataStruct.Y        = &(wlenT*1e-9); // Wavelength in meters
    rawDataStruct.Ylab     = "Wavelength";
    rawDataStruct.Z        = &time; // Time in seconds
    rawDataStruct.Zlab     = "Time";
    rawDataStruct.dark     = &dark;
    rawDataStruct.phot     = &phot;
    rawDataStruct.fringes  = &interf;

    if(plot==1)
    {
        window,1,wait=1, width=400, height=400;
        fma;
        pltitle, "Average of interf. channel over frames";
        pli,(*rawDataStruct.fringes)(..,avg);
        hcps,HOME+rawDataStruct.hdr.TARGET(1)+"_MJD"+
            pr1(int(avg(rawDataStruct.hdr.MJD)))+"_"+
            pr1(lround(rawDataStruct.hdr.hour))+"h"+
            pr1(lround(rawDataStruct.hdr.minute))+"m_avgFlux.ps";

        window,2;
        fma;
        // ??? Difference avec window 24 ? Fond deja soustrait ? 
        for(k=1;k<=3;k++)
        {
            plg, phot(k,avg,,avg) / phot(k,avg,avg,avg), wlen, color=colors(k);
        }
        plg, interf(avg,,avg) / interf(avg,avg,avg), wlen;
    }
    return 1;
}

/***************************************************************************/

func amplSubtractContinuum(calDataStruct, rawDataStruct, &subDataStruct, subtractionType=, avgFrame=, avgWlen=, useFring=, plot=, quiet=)
    /* DOCUMENT amplSubtractContinuum(calDataStruct, rawDataStruct, &subDataStruct, subtractionType=, avgFrame=, avgWlen=, useFring=, plot=, quiet=)

       DESCRIPTION
       Subtracts the continuum,
       computes readout noise from the dark pixels,
       collapses the photometric channels.

       PARAMETERS
       - calDataStruct  : ampl P2VM info structure (from amplGetP2vmInfo)
       - rawDataStruct  : 
       - subDataStruct  : 
       - subtractionType: 
       - avgFrame       : 
       - avgWlen        : 
       - useFring       : 
       - plot           : 
       - quiet          : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(!quiet)
        yocoLogInfo,"Subtracting continuum...";

    // 2 types of subtraction here! 2nd one ("Dirty") is really dirty... 
    if(is_void(subtractionType))
        subtractionType="clean";
    
    if(is_void(avgFrame))
        avgFrame = 0;
            
    if(is_void(avgWlen))
        avgWlen = 0;
            
    if(is_void(useFring))
        useFring = 1;

    if(subtractionType=="clean")
    {
        // Get relevant data information
        subDataStruct = rawDataStruct;
        fringes       = *rawDataStruct.fringes;
        phot          = *rawDataStruct.phot;
        wlen          = *rawDataStruct.Y;
        nbWlen        = numberof(wlen);
        nbBases       = dimsof(phot)(2);
        nbFrames      = dimsof(fringes)(0);
        nbPix         = dimsof(fringes)(2);
        
        // Retrieve Kappa matrix information
        kappa     = *calDataStruct.kappaMat;
        kappaFlux = *calDataStruct.kappaFlux;
        kappaWlen = *calDataStruct.wlen;
        beamShape = *calDataStruct.beamShape;
        nbKlen    = numberof(wlen);

        // This is if the interferometric channel has to be used for
        // photometric estimate
        if(useFring != 1)
            kappa = kappa(,:3,);
        
        // Match P2vm wavelength and obs wlen
        start = where(kappaWlen >= min(wlen))(1);
        stop  = where(kappaWlen <= max(wlen))(0);
        if(numberof(start)==0)
            start = 1;
        if(numberof(stop)==0)
            stop = 0;
        
        // Resize kappa matrix and other useful infos
        kappa = kappa(start:stop,..);
        kappaFlux = kappaFlux(start:stop,..);
        beamShape = beamShape(,start:stop,);

        // Inverse the Kappa matrix, averaged over all wavelengths
        kappavg  = kappa(avg,,) / kappaFlux(avg,)(-,);
        // Singular values decomposition
        svKappa  = SVdec(kappavg, u, vt);
        // Generalized inverse
        invKappa = (transpose(vt)(,+) * diag(1.0/svKappa)(+,))(,+) *
            transpose(u)(+,);

        // Get the photometry and compute the estimated flux in the
        // interferometric channel
        instantPhot = transpose(phot(,avg,avg,));
        // Add the interferometric channel for estimation
        if(useFring == 1)
            instantPhot = grow(instantPhot, fringes(avg,avg,,-));
        // Compute estimate
        instantFlux = invKappa(..,+) * instantPhot(..,+);
        // (optionally) time-average photometry
        // instantFlux = boxcar(instantFlux, avgFrame);
        
        // Inverse the Kappa matrix, wavelength-by-wavelength
        // This is to get the average flux of the target
        // (for better SNR on the subtraction)
        invKappa2 = array(0.0,dimsof(transpose(kappa, [2,3])));
        for(k=1;k<=nbKlen;k++)
        {
            kapptmp2  = kappa(k,..) / kappaFlux(k,)(-,);
            svKappa2  = SVdec(kapptmp2, u, vt);
            invKappa2(k,..) = (transpose(vt)(,+) * diag(1.0/svKappa2)(+,))(,+) * transpose(u)(+,);
        }
        instantPhot2 = transpose(phot(,avg,..), [1,3,2]);
        if(useFring == 1)
            instantPhot2 = grow(instantPhot2, fringes(avg,..,-));
        
        instantFlux2 = array(0.0, nbWlen, nbBases, nbFrames);
        for(k=1;k<=nbKlen;k++)
        {
            instantFlux2(k,..) = invKappa2(k,..,+) * instantPhot2(k,..,+);
        }

        avgSpec = instantFlux2(,avg,avg);
        avgSpec = avgSpec / avg(avgSpec);

        // FIXME: here something's wrong!
        //PHOT = transpose(avgSpec(-,,-,-) * beamShape(,,,-) * instantFlux(-,-,,),[3,1,2]);
        PHOT = transpose(instantFlux2(-,..) * beamShape(..,-),[3,1,2]);
    
        subFlux = (fringes - PHOT(sum,..));
        
        subDataStruct.fringes = &subFlux;
        subDataStruct.phot    = &PHOT;

        if(plot==1)
        {
            yocoNmCreate,24,2,2,landscape=1,width=800,height=600,wait=1;
            fma;
            plsys,1;
            pltitle, "Photometry from photo and interfero channels";
            
            for(k=1;k<=3;k++)
                plg, PHOT(k,avg,,avg) / PHOT(k,avg,avg,avg)+k/100, wlen, color=colors(k);
            plg, fringes(avg,,avg) / fringes(avg,avg,avg), wlen;
            plg,PHOT(sum,avg,,avg) / PHOT(sum,avg,avg,avg), wlen,color="magenta";

            plsys,3;
            plg, fringes(avg,,avg) / fringes(avg,avg,avg) - PHOT(sum,avg,,avg) / PHOT(sum,avg,avg,avg), wlen;

            plsys,2;
            pltitle, "Photometry from raw data";
            
            for(k=1;k<=3;k++)
                plg, phot(k,avg,,avg) / PHOT(k,avg,avg,avg)+k/100, wlen, color=colors(k);
            
        }
    }
    else if(subtractionType="dirty")
    {
        if(!quiet)
            yocoLogInfo,"Subtracting continuum 0...";
        
        // The "dirty way" is simply averaging the interferometric beam!
        subDataStruct = rawDataStruct;
        flux         = *rawDataStruct.fringes;
        beamShape    = flux(..,avg);
        avgBeamShape = beamShape / avg(beamShape);
        avgFlux      = flux(avg,avg,);

        subFlux  = (flux - avgBeamShape(,,-) * avgFlux(-,-,));
    
        subDataStruct.fringes = &subFlux;
    }
    else
        error;
}

/***************************************************************************/

func amplResampleFringes(&calDataStruct, &rawDataStruct, plot=, fitPol=, sampWlen=, sampPix=, quiet=)
    /* DOCUMENT amplResampleFringes(&calDataStruct, &rawDataStruct, plot=, fitPol=, sampWlen=, sampPix=, quiet=)

       DESCRIPTION
       Resamples fringes according to wave-number (vertical direction)
       and fringe-sampling (horizontal direction)

       PARAMETERS
       - calDataStruct: input P2VM structure
       - rawDataStruct: raw data structure
       - plot         : plot flag
       - fitPol       :
       - sampWlen     :
       - sampPix      : 
       - quiet        : 

       SEE ALSO amplSubtractContinuum, amplAMBER
    */
{
    if(!quiet)
        yocoLogInfo, "Resampling fringes...";

    // Get relevant data information
    fringes       = *rawDataStruct.fringes;
    phot          = *rawDataStruct.phot;
    dark          = *rawDataStruct.dark;
    wlen          = *rawDataStruct.Y;
    nbWlen        = numberof(wlen);

    // Define here upsampling factor to not loose signal
    if(is_void(sampWlen))
    {
        // First get non-zero wavelengths
        wlen2    = wlen(where(wlen!=0));
        // Define wavelength interval
        interv   = max(wlen2)-min(wlen2);
        // Get differential wavelength
        delta    = wlen2(dif);
        // Compute the ratio between the widest and narrowest spectral channels
        fact     = max(delta) / min(delta);
        // Compute the new number of spectral pixels to use for resampling
        nToUse   = interv / min(delta);
        // Compute the upsampling factor to use
        sampWlen = fact * nToUse / nbWlen;

        if(!quiet)
            yocoLogInfo,"No sampWlen set. Using default: "+pr1(sampWlen);
    }

    // Get arrays dimensions
    nbFrames = dimsof(fringes)(0);
    nbPix    = dimsof(fringes)(2);
    nbPid    = dimsof(dark)(2);
    nbTels   = dimsof(phot)(2);

    // Get relevant calibration information
    peakPos    = *calDataStruct.freqPeak;
    peakWlen   = *calDataStruct.wlen;
    sigma      = 1.0 / wlen;
    nbPeakWlen = numberof(peakWlen);

    // Match P2vm wavelength and obs wlen
    start = where(peakWlen >= min(wlen))(1);
    stop  = where(peakWlen <= max(wlen))(0);
    if(numberof(start)==0)
        start = 1;
    if(numberof(stop)==0)
        stop = 0;

    // Fit a polynom
    if(fitPol==1)
    {
        theFit = pol = polC = [];
        for(k=1;k<=3;k++)
        {
            thePeakPos = peakPos(,k);
            grow, theFit, array(yocoMathPolyFit(4,peakWlen,thePeakPos),1);
            grow, pol,    array(yocoMathPoly(wlen, theFit(,k)), 1);
            grow, polC,   array(yocoMathPoly(peakWlen, theFit(,k)), 1);
        }
        popol  = pol(,avg)/max(pol(,avg))   * nbPix;
        popolC = polC(,avg)/max(polC(,avg)) * nbPix;
    }
    else
    {
        pol    = peakPos(start:stop,);
        polC   = peakPos;
        popol  = pol(,avg)  / max(pol(,avg))  * nbPix;
        popolC = polC(,avg) / max(polC(,avg)) * nbPix;
    }
    
    if(is_void(sampPix))
    {
        sampPix = max(popol) / min(popol);
	// FIXME: sampPix temporarily set to 1.0, awaiting for further investigation of the problem
        sampPix = 1.0;
        if(!quiet)
            yocoLogInfo,"No sampPix set. Using default: "+pr1(sampPix);
    }
    
    // Plot intermediate results
    if(plot==1)
    {
        window,2, wait=1;
        fma;
        pltitle, "Peak frequency after resampling (amplResampleFringes).\n Color=raw waveband";
        for(k=1;k<=3;k++)
        {
            plg,peakPos(,k),peakWlen;
            plg,pol(,k),wlen,color="red";
        }
        window,3, wait=1;
        fma;
        for(k=1;k<=3;k++)
        {
            // pltitle, "Peak frequency after resampling (amplResampleFringes).\n Color=raw waveband";
            plg, pol(,k) / pol(,avg), wlen,color=colors(k);
        }
    }
    
    newNbWlen = int(ceil(sampWlen * nbWlen));
    newNbPix  = int(ceil(sampPix * nbPix));

    // Make new wavelength as such as sigma is evenly sampled
    newPol   = span(-max(popol), max(popol), newNbPix);
    newPold  = span(-max(popol), max(popol), nbPid);
    newSigma = span( sigma(1), sigma(0), newNbWlen);
    newWlen  = 1.0/newSigma;
    
    YY   = transpose(array(interp(indgen(nbWlen),sigma,newSigma),newNbPix));
    ppol = interp(nbPix^2/popol,sigma,newSigma);
    frc  = (ppol-nbPix)/2;
    XX   = span(1-frc,ppol-frc,newNbPix);
    
    YYd   = transpose(array(interp(indgen(nbWlen),sigma,newSigma),nbPid));
    XXd   = array(indgen(nbPid),newNbWlen);

    if(plot==1)
    {
        Y = transpose(array(sigma, nbPix));
        X = span(-popol, popol, nbPix);
        
        window,4, wait=1;
        pltitle, "Raw fringes, 1st frame";
        plf,fringes(,,1), Y, X;
    }

    elapsed = elapsed2 = array(0.0,3);
    timer,elapsed;
    
    newFringes = array(0.0, newNbPix, newNbWlen, nbFrames);
    newDark    = array(0.0, nbPid, newNbWlen, nbFrames);
    newPhot    = array(0.0, nbTels, newNbPix, newNbWlen, nbFrames);
        
    for(k=1;k<=nbFrames;k++)
    {
        newFringes(..,k) = bilinear(fringes(..,k), XX, YY);
        newDark(..,k)    = bilinear(dark(..,k), XXd, YYd);

        for(l=1;l<=nbTels;l++)
        {
            newPhot(l, .., k) = bilinear(phot(l, .., k), XX, YY);
        }
    }

    if(plot==1)
    {
        window,5, wait=1;
        pltitle, "Interpolated fringes (amplResampleFringes), 1st frame";
        pli,newFringes(,::-1,1);
    }
    
    // timer,elapsed2;
    // write,elapsed2-elapsed;

    newFreqPeak  = pol  / popol  * newNbPix;
    newFreqPeakC = polC / popolC * newNbPix;

    if(!quiet)
        yocoLogInfo,newFreqPeakC(avg) / newNbPix;
    
    rawDataStruct.fringes   = &newFringes;
    rawDataStruct.phot      = &newPhot;
    rawDataStruct.dark      = &newDark;
    rawDataStruct.Y         = &newWlen(start2:stop2);
    *calDataStruct.freqPeak = newFreqPeakC;
    
}

/***************************************************************************/

func amplInterpolateBadPixels3(Img0, typeI=, quiet=)
    /* DOCUMENT amplInterpolateBadPixels3(Img0, typeI=, quiet=)

       SEE ALSO:
    */
{
    if(!quiet)
        yocoLogInfo,"Interpolating bad pixels...";
    local Img1, badPix;

    Img1 = Img0;

    nX = dimsof(Img1)(2);
    nY = dimsof(Img1)(3);

    badVal=0.0;
    bads = where2(Img1==badVal);
    
    randomize;
    kStop=0;
    while((numberof(bads)!=0)&&(kStop<2*sqrt(nX*nY)))
    {
        kStop++;
        N = numberof(bads(1,));
        i=lround(random(N)*N)+1;
        
        for(kBad=1;kBad<=N;kBad++)
        {
            if(i(kBad)<=N)
            {
                if(avg(Img1(bads(1,i(kBad)),bads(2,i(kBad)),))==0)
                {
                    count = 0;
                    if(bads(1,i(kBad))+1<=nX)
                    {
                        img10 = Img1(bads(1,i(kBad))+1,bads(2,i(kBad)),);
                        if(anyof(img10 != 0))
                        {
                            Img1(bads(1,i(kBad)),bads(2,i(kBad)),) += img10;
                            count++;
                        }
                    }
                    if(bads(2,i(kBad))+1<=nY)
                    {
                        img01 = Img1(bads(1,i(kBad)),bads(2,i(kBad))+1,);
                        if(anyof(img01 != 0))
                        {
                            Img1(bads(1,i(kBad)),bads(2,i(kBad)),) += img01;
                            count++;
                        }
                    }
                    if(bads(1,i(kBad))-1>0)
                    {
                        img_10 = Img1(bads(1,i(kBad))-1,bads(2,i(kBad)),);
                        if(anyof(img_10 != 0))
                        {
                            Img1(bads(1,i(kBad)),bads(2,i(kBad)),) += img_10;
                            count++;
                        }
                    }
                    if(bads(2,i(kBad))-1>0)
                    {
                        img0_1 = Img1(bads(1,i(kBad)),bads(2,i(kBad))-1,);
                        if(anyof(img0_1 != 0))
                        {
                            Img1(bads(1,i(kBad)),bads(2,i(kBad)),) += img0_1;
                            count++;
                        }
                    }
                    Img1(bads(1,i(kBad)),bads(2,i(kBad)),)/=double(count+(count==0));
                }
            }
        }
        bads = where2(Img1==badVal);
    }
        
    return Img1;
}

/***************************************************************************/

func amplSaveCalib(outFile, calib, overwrite=, quiet=)
    /* DOCUMENT amplSaveCalib(outFile, calib, overwrite=, quiet=)

       DESCRIPTION

       PARAMETERS
       - outFile  : 
       - calib    : 
       - overwrite: 
       - quiet    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(open(outFile, "r", 1))
    {
        if(overwrite==1)
        {
            if(!quiet)
                yocoLogInfo, "File exists. Erasing existing file...";
            remove, outFile;
        }
        else
        {
            if(!quiet)
                yocoLogWarning, "File is in the way! Skipping...";
            return 0;
        }
    }

    if(!quiet)
        yocoLogInfo,"Saving calibration file "+outFile;
       
    fh = cfitsio_open(outFile,"w");
    cfitsio_set, fh, "SIMPLE", char(1);
    cfitsio_set, fh, ["BITPIX", "NAXIS"], [16, 0];
    cfitsio_set, fh, "EXTEND", char(1);
    cfitsio_set, fh, "AMPL VERSION", amplAMBER();
    cfitsio_set, fh, "ESO OCS P2VM ID", calib.p2vmId;
    //cfitsio_write_image, fh, [1];
    cfitsio_set, fh, ["DATATYPE", "ESO DPR TYPE"], ["CALIB", "P2VM"];
    cfitsio_add_image, fh, *calib.wlen, "WLEN";
    cfitsio_add_image, fh, *calib.freqPeak, "FREQPEAK";
    if(!is_void(*calib.frgContMat))
        cfitsio_add_image, fh, *calib.frgContMat, "FRGCONTMAT";
    cfitsio_add_image, fh, *calib.kappaMat, "KAPPAMAT";
    cfitsio_add_image, fh, *calib.kappaFlux, "KAPPAFLUX";
    cfitsio_add_image, fh, *calib.beamShape, "BEAMSHAPE";
    
    cfitsio_close,fh;
}

/***************************************************************************/

func amplSaveData(outFile, theData, overwrite=, append=, quiet=)
    /* DOCUMENT amplSaveData(outFile, theData, overwrite=, append=, quiet=)

       DESCRIPTION

       PARAMETERS
       - outFile  : 
       - theData  : 
       - overwrite: 
       - append   : 
       - quiet    : 

       SEE ALSO
    */
{
    if(open(outFile, "r", 1))
    {
        if((overwrite==1)&&(!append))
        {
            if(!quiet)
                yocoLogInfo, "File exists. Erasing existing file...";
            remove, outFile;
        }
        else if(append==1)
        {
            if(!quiet)
                yocoLogInfo, "File exists. Appending file...";
        }
        else
        {
            yocoLogWarning, "File is in the way! Skipping...";
            return 0;
        }
    }

    if(!quiet)
        yocoLogInfo,"Saving data file "+outFile;

    if(append==1)
    {
        // Open file with append mode
        fh = cfitsio_open(outFile,"a");

        dims=dimsof(*theData.fringes)(1);
        
        // Write axes
        if(dims==1)
            amplCfitsio_add_bintable, fh, theData.X, "X",,"X", append=append;
        else if(dims==2)
            amplCfitsio_add_bintable, fh, theData.Y, "Y",,"Y", append=append;
        else if(dims==3)
            amplCfitsio_add_bintable, fh, theData.Z, "Z",,"Z", append=append;
        else if(dims==4)
            amplCfitsio_add_bintable, fh, theData.T, "T",,"T", append=append;
    }
    else
    {
        // Open file
        fh = cfitsio_open(outFile,"w");
        if(!quiet)
            yocoLogInfo,"Writing keywords";
        // Set basic keywords
        cfitsio_set, fh, "SIMPLE", char(1);
        cfitsio_set, fh, ["BITPIX", "NAXIS"], [16, 0];
        cfitsio_set, fh, "EXTEND", char(1);
        cfitsio_set, fh, "AMPL VERSION", amplAMBER();

        // Cleanup keywords from bad stuff
        bads = ["SIMPLE", "BITPIX", "NAXIS", "EXTEND", "END", string(nil)];
        kn = *theData.hdr.keyName;
        kv = *theData.hdr.keyVal;
        kc = *theData.hdr.keyCom;
        for(k=1;k<=numberof(bads);k++)
        {
            if(anyof(kn==bads(k)))
            {
                wk = where(kn!=bads(k));
                kn = kn(wk);
                kv = kv(wk);
                kc = kc(wk);
            }
        }
        
        // Write all original keywords
        cfitsio_set, fh, kn,kv,kc;
        
        // Write additional ampl keywords
        cfitsio_set, fh,
            ["AMPL DATATYPE", "AMPL XLAB", "AMPL YLAB", "AMPL ZLAB", "AMPL TLAB"],
            [theData.dataType, theData.Xlab, theData.Ylab,  theData.Zlab, theData.Tlab],,["", theData.Xunit, theData.Yunit, theData.Zunit, theData.Tunit];
        cfitsio_set, fh,
            ["AMPL REALUPSAMPLEX", "AMPL REALUPSAMPLEY"],
            [theData.realUpsampleX, theData.realUpsampleY];
        cfitsio_set, fh, "AMPL VERSION", amplAMBER();
    
        //////////////////////////////////////////////
        // Write (optional) OI_ARRAY table first,
        //////////////////////////////////////////////
        if((theData.hdr.ARRNAME != string(nil))&&
           (!is_void(*theData.hdr.STA_NAME))&&
           (!is_void(*theData.hdr.STA_INDEX))&&
           (!is_void(*theData.hdr.DIAMETER))&&
           (!is_void(*theData.hdr.STAXYZ)))
        {
            if(!quiet)
                yocoLogInfo,"Writing OI_ARRAY";
            amplCfitsio_add_bintable, fh,
                [&swrite(*theData.hdr.TEL_NAME,format="%-16s"), &swrite(*theData.hdr.STA_NAME,format="%-16s"), theData.hdr.STA_INDEX, theData.hdr.DIAMETER, theData.hdr.STAXYZ],
                ["TEL_NAME", "STA_NAME", "STA_INDEX", "DIAMETER", "STAXYZ"],
                ["", "", "", "m", "m"], "OI_ARRAY";
        
            // Write the OIFITS revision and instrument name
            cfitsio_set, fh, "AMPL VERSION", amplAMBER();
            cfitsio_set, fh,
                ["ARRNAME", "FRAME"],
                [theData.hdr.ARRNAME, theData.hdr.FRAME],
                ["Array name, for cross-referencing", "Coordinate frame"],
                ["", ""];
            cfitsio_set, fh, "OI_REVN", int(1),
                "Revision number of the table definition", "";
            cfitsio_set, fh,
                ["ARRAYX", "ARRAYY", "ARRAYZ"],
                [theData.hdr.ARRAYXYZ(1), theData.hdr.ARRAYXYZ(2), theData.hdr.ARRAYXYZ(3)],
                ["Coordinate frame", "Array center coordinates","Array center coordinates","Array center coordinates"],
                ["m", "m", "m"];
        }

        if(!quiet)
            yocoLogInfo,"Writing mask";
        // Write mask information
        if(!is_void(*theData.mask))
            cfitsio_add_image, fh, *theData.mask, "MASK";
        if(!is_void(*theData.maskRef))
            cfitsio_add_image, fh, *theData.maskRef, "MASKREF";
        if(!is_void(*theData.maskWlen))
            cfitsio_add_image, fh, *theData.maskWlen, "MASKWLEN";

        // Write AMBER++-specific keywords
        cfitsio_set, fh,
            ["ESO PRO DID", "ESO PRO CATG", "ESO PRO TYPE", "ESO OCS DRS VERSION"],
            ["AMBER++", "SCIENCE_REDUCED", theData.dataType, "AMBER++, revision "+pr1(amplAMBER())];

        if(!quiet)
            yocoLogInfo,"Writing axes";
        // Write axes
        if(!is_void(*theData.X))
            amplCfitsio_add_bintable, fh, theData.X, "X",,"X";
        if(!is_void(*theData.Y))
            amplCfitsio_add_bintable, fh, theData.Y, "Y",,"Y";
        if(!is_void(*theData.Z))
            amplCfitsio_add_bintable, fh, theData.Z, "Z",,"Z";
        if(!is_void(*theData.T))
            amplCfitsio_add_bintable, fh, theData.T, "T",,"T";

    }
    
    // Write the data
    if(!quiet)
        yocoLogInfo,"Writing interferometry data";
    amplCfitsio_add_bintable, fh, theData.fringes, "FRINGES",,"FRINGES", append=append;
    
    if(!quiet)
        yocoLogInfo,"Writing photometry data";
    amplCfitsio_add_bintable, fh, theData.dark, "DARK",,"DARK", append=append;
    amplCfitsio_add_bintable, fh, theData.phot, "PHOT",,"PHOT", append=append;
   
    cfitsio_close,fh;
}

/***************************************************************************/

func amplLoadCalib(&cal, inputFile=, quiet=)
    /* DOCUMENT amplLoadCalib(&cal, inputFile=, quiet=)

       DESCRIPTION

       PARAMETERS
       - cal      : 
       - inputFile: 
       - quiet    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if (amdlibFileChooser(message, inputFile, 2) == 0)
    {
        yocoError,"'inputFile' not specified correctly.";
        return 0;
    }

    if(!quiet)
        yocoLogInfo,"Loading calibration file "+inputFile+"...";
    
    cal = amplCalibStruct();
    
    fh = cfitsio_open(inputFile,"r");
    
    vals = cfitsio_get(fh, ["ESO OCS P2VM ID"]);
    cal.p2vmId = vals(1);
     
    ext = cfitsio_list_hdus(fh);
    if(anyof(ext=="WLEN"))
    {
        cfitsio_goto_hdu,fh,"WLEN"; 
        cal.wlen = &cfitsio_read_image(fh);
    }
    if(anyof(ext=="FREQPEAK"))
    {
        cfitsio_goto_hdu,fh,"FREQPEAK"; 
        cal.freqPeak   = &cfitsio_read_image(fh);
    }
    if(anyof(ext=="KAPPAMAT"))
    {
        cfitsio_goto_hdu,fh,"KAPPAMAT"; 
        cal.kappaMat   = &cfitsio_read_image(fh);
    }
    if(anyof(ext=="KAPPAFLUX"))
    {
        cfitsio_goto_hdu,fh,"KAPPAFLUX"; 
        cal.kappaFlux   = &cfitsio_read_image(fh);
    }
    if(anyof(ext=="BEAMSHAPE"))
    {
        cfitsio_goto_hdu,fh,"BEAMSHAPE"; 
        cal.beamShape   = &cfitsio_read_image(fh);
    }
    cfitsio_close,fh;
    return cal;
}

/***************************************************************************/

func amplLoadData(void, inputFile=, quiet=)
    /* DOCUMENT amplLoadData(inputFile=, quiet=)

       DESCRIPTION

       PARAMETERS
       - inputFile: 
       - quiet    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if (amdlibFileChooser(message, inputFile, 2) == 0)
    {
        yocoError,"'inputFile' not specified correctly.";
        return 0;
    }

    if(!quiet)
        yocoLogInfo,"Loading data file "+inputFile+"...";
    
    dat = amplDataStruct();
    
    amplLoadDataHdr, hdr, inputFile=inputFile, quiet=quiet;
    dat.hdr = hdr;
    fh = cfitsio_open(inputFile,"r");

    
    vals = cfitsio_get(fh, ["AMPL DATATYPE", "AMPL XLAB", "AMPL YLAB", "AMPL ZLAB", "AMPL TLAB"]);
    dat.dataType = vals(1);
    dat.Xlab = vals(2);
    dat.Ylab = vals(3);
    dat.Zlab = vals(4);
    dat.Tlab = vals(5);
   
    tmpX = cfitsio_get(fh, "AMPL REALUPSAMPLEX");
    if(typeof(tmpX)=="string")
        tmpX = yocoStr2Double(tmpX)
            dat.realUpsampleX = tmpX;
   
    tmpY = cfitsio_get(fh, "AMPL REALUPSAMPLEY");
    if(typeof(tmpY)=="string")
        tmpY = yocoStr2Double(tmpY);
    dat.realUpsampleY = tmpY;
   
    ext = cfitsio_list_hdus(fh);
    if(anyof(ext=="FRINGES"))
    {
        cfitsio_goto_hdu,fh,"FRINGES"; 
        dat.fringes   = cfitsio_read_bintable(fh)(1);
    }
    if(anyof(ext=="X"))
    {
        cfitsio_goto_hdu,fh,"X"; 
        dat.X   = cfitsio_read_bintable(fh)(1);
    }
    if(anyof(ext=="Y"))
    {
        cfitsio_goto_hdu,fh,"Y"; 
        dat.Y   = cfitsio_read_bintable(fh)(1);
    }
    if(anyof(ext=="Z"))
    {
        cfitsio_goto_hdu,fh,"Z"; 
        dat.Z   = cfitsio_read_bintable(fh)(1);
    }
    if(anyof(ext=="T"))
    {
        cfitsio_goto_hdu,fh,"T"; 
        dat.T   = cfitsio_read_bintable(fh)(1);
    }
    if(anyof(ext=="DARK"))
    {
        cfitsio_goto_hdu,fh,"DARK"; 
        dat.dark   = cfitsio_read_bintable(fh)(1);
    }
    if(anyof(ext=="PHOT"))
    {
        cfitsio_goto_hdu,fh,"PHOT"; 
        dat.phot   = cfitsio_read_bintable(fh)(1);
    }
    if(anyof(ext=="MASK"))
    {
        cfitsio_goto_hdu,fh,"MASK"; 
        dat.mask   = &cfitsio_read_image(fh);
    }
    if(anyof(ext=="MASKREF"))
    {
        cfitsio_goto_hdu,fh,"MASKREF"; 
        dat.maskRef   = &cfitsio_read_image(fh);
    }
    if(anyof(ext=="MASKWLEN"))
    {
        cfitsio_goto_hdu,fh,"MASKWLEN"; 
        dat.maskWlen   = &cfitsio_read_image(fh);
    }
    cfitsio_close,fh;
    return dat;
}

/***************************************************************************/

func amplGrowData(inDat, addDat, &outDat)
    /* DOCUMENT amplGrowData(inDat, addDat, &outDat)

       DESCRIPTION

       PARAMETERS
       - inDat : 
       - addDat: 
       - outDat: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(inDat))
    {
        outDat = addDat;
        return 1;
    }

    outDat         = inDat;
    outDat.fringes = &grow(*inDat.fringes, *addDat.fringes);
    outDat.dark    = &grow(*inDat.dark,    *addDat.dark);
    outDat.phot    = &grow(*inDat.phot,    *addDat.phot);
    if(outDat.Xlab=="Time")
        outDat.X       = &grow(*inDat.X,       *addDat.X);
    else if(outDat.Ylab=="Time")
        outDat.Y       = &grow(*inDat.Y,       *addDat.Y);
    else if(outDat.Zlab=="Time")
        outDat.Z       = &grow(*inDat.Z,       *addDat.Z);
    else if(outDat.Tlab=="Time")
        outDat.T       = &grow(*inDat.T,       *addDat.T);
    outDat.hdr.UCOORD  = &grow(*inDat.hdr.UCOORD, *addDat.hdr.UCOORD);
    outDat.hdr.VCOORD  = &grow(*inDat.hdr.VCOORD, *addDat.hdr.VCOORD);

    outDat.hdr.U1COORD = &grow(*inDat.hdr.U1COORD, *addDat.hdr.U1COORD);
    outDat.hdr.V1COORD = &grow(*inDat.hdr.V1COORD, *addDat.hdr.V1COORD);
    outDat.hdr.U2COORD = &grow(*inDat.hdr.U2COORD, *addDat.hdr.U2COORD);
    outDat.hdr.V2COORD = &grow(*inDat.hdr.V2COORD, *addDat.hdr.V2COORD);
}

/***************************************************************************/

func cfitsio_list_hdus(fh)
    /* DOCUMENT cfitsio_list_hdus(fh)

       DESCRIPTION

       PARAMETERS
       - fh: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    N = cfitsio_get_num_hdus(fh);
    ext = [];
    for(k=1;k<=N;k++)
    {
        cfitsio_goto_hdu, fh, k;
        tmp = cfitsio_get(fh,"EXTNAME");
        if(is_void(tmp))
            tmp="";
        grow, ext, tmp;
    }
    return ext;
}

/***************************************************************************/

func amplSaveOiData(outOiFile, oi, amberKeys=, amplTable=, quiet=)
    /* DOCUMENT amplSaveOiData(outOiFile, oi, amberKeys=, amplTable=, quiet=)

       DESCRIPTION
       oi fits file writer

       PARAMETERS
       - outOiFile: 
       - oi       : 
       - amberKeys: 
       - amplTable: 
       - quiet    : 

       Phases in radians
       
       SEE ALSO
    */
{
    yocoFileSplitName,outOiFile,d,f,e;

    if(!quiet)
        yocoLogInfo, "Saving file "+f+e;
    if(!quiet)
        yocoLogInfo, "In directory "+d;
    if(is_void(amplTable))
    {
        amplTable=1;
    }

    // Test the main data structure
    test =  *oi.VIS2DATA(1);
    if(!is_void(*oi.VIS2DATA(1)))
    {
        mainData = oi.VIS2DATA;
    }
    else if(!is_void(*oi.VISAMP(1)))
    {
        mainData = oi.VISAMP;
    }
    else if(!is_void(*oi.VISPHI(1)))
    {
        mainData = oi.VISPHI;
    }
    else if(!is_void(*oi.T3PHI(1)))
    {
        mainData = oi.T3PHI;
    }
    else
        error;
    
    nbTab = numberof(mainData);
    // Check that there is just one target (not compatible with multiple targets)
    if(nallof(oi.hdr.TARGET == oi.hdr.TARGET(1)))
        error;
    

    // create the file
    fh = cfitsio_open(outOiFile,"c");
    
    // (Try to) Follow OI FITS standard


    //////////////////////////////////////////////
    // Write  OI_TARGET table (mandatory)
    //////////////////////////////////////////////
    // test VELTYP keyword
    if(!quiet)
        yocoLogInfo,"Saving OI_TARGET";
    if(noneof(oi.hdr.VELTYP == "LSR"))
        if(noneof(oi.hdr.VELTYP == "HELIOCEN"))
            if(noneof(oi.hdr.VELTYP == "BARYCENT"))
                if(noneof(oi.hdr.VELTYP == "GEOCENTR"))
                    if(noneof(oi.hdr.VELTYP == "TOPOCENT"))
                        if(noneof(oi.hdr.VELTYP == "UNKNOWN"))
                        {
                            yocoLogWarning, "No [proper] radial velocity type inserted! Should be one of LSR, HELIOCEN, BARYCENT, GEOCENTR, TOPOCENT, UNKNOWN";
                            oi.hdr.VELTYP = "UNKNOWN";
                        }
    
    // Check radial velocity definition
    if(noneof(oi.hdr.VELDEF == "RADIO"))
        if(noneof(oi.hdr.VELDEF == "OPTICAL"))
        {
            yocoLogWarning, "No [proper] radial velocity definition inserted! Should be one of RADIO, OPTICAL";
            oi.hdr.VELTYP = "OPTICAL";
        }

    OI = oi.hdr(1);
    
    // Write the binary table
    amplCfitsio_add_bintable, fh,
        [&[int(1)],
         &array(swrite(OI.TARGET,format="%-16s"),1),
         &array(OI.RAEP0,1),
         &array(OI.DECEP0,1,1),
         &array(float(OI.EQUINOX),1),
         &array(OI.RA_ERR,1),
         &array(OI.DEC_ERR,1),
         &array(OI.SYSVEL,1),
         &array(swrite(OI.VELTYP,format="%-8s"),1),
         &array(swrite(OI.VELDEF,format="%-8s"),1),
         &array(OI.PMRA,1), &array(OI.PMDEC,1),
         &array(OI.PMRA_ERR,1),
         &array(OI.PMDEC_ERR,1),
         &array(float(OI.PARALLAX),1),
         &array(float(OI.PARA_ERR),1),
         &array(swrite(OI.SPECTYP,format="%-16s"),1)],
        ["TARGET_ID", "TARGET", "RAEP0", "DECEP0", "EQUINOX", "RA_ERR",
         "DEC_ERR", "SYSVEL", "VELTYP", "VELDEF", "PMRA", "PMDEC",
         "PMRA_ERR", "PMDEC_ERR", "PARALLAX", "PARA_ERR", "SPECTYP"],
        ["", "", "deg", "deg", "yr", "deg", "deg", "m/s", "", "",
         "deg/yr", "deg/yr", "deg/yr", "deg/yr", "deg", "deg", ""],
        "OI_TARGET";

    // Update keywords comments
    PallValues = cfitsio_get(fh, "TTYPE*", allComments, allNames, point=1);
    cfitsio_set, fh,
        "TTYPE"+swrite(format="%i",indgen(17)),
        PallValues,
        ["Index number", "Target name", "R.A. at mean equinox", "Decl. at mean equinox", "Equinox", "Error in R.A. at mean equinox", "Error in decl. at mean equinox", "Systemic radial velocity", "Reference for radial velocity (LSR, GEOCENTR, etc.)", "Definition of radial velocity (OPTICAL, RADIO)", "Proper motion in R.A.", "Proper motion in decl.", "Error of proper motion in R.A.", "Error of proper motion in decl.", "Parallax", "Error in parallax", "Spectral type"],
        ["", "", "deg", "deg", "yr", "deg", "deg", "m/s", "", "", "deg/yr", "deg/yr", "deg/yr", "deg/yr", "deg", "deg", ""];
    
    // Write the OIFITS revision (1)
    cfitsio_set, fh, "OI_REVN", int(1),
        "Revision number of the table definition", "";
    
    THE_WLEN = THE_BAND = THE_INS = THE_EXT = [];

    for(kTab=1;kTab<=nbTab;kTab++)
    {
        OI = oi.hdr(kTab);
        
        //////////////////////////////////////////////
        // Write (optional) OI_ARRAY table first,
        // for compatibility with amdlib!
        //////////////////////////////////////////////
        if(!quiet)
            yocoLogInfo,"Saving OI_ARRAY";
        if((OI.ARRNAME != string(nil))&&
           (!is_void(*OI.STA_NAME))&&
           (!is_void(*OI.STA_INDEX))&&
           (!is_void(*OI.DIAMETER))&&
           (!is_void(*OI.STAXYZ)))
        {
            amplCfitsio_add_bintable, fh,
                [&swrite(*OI.TEL_NAME,format="%-16s"), &swrite(*OI.STA_NAME,format="%-16s"), OI.STA_INDEX, OI.DIAMETER, OI.STAXYZ],
                ["TEL_NAME", "STA_NAME", "STA_INDEX", "DIAMETER", "STAXYZ"],
                ["", "", "", "m", "m"], "OI_ARRAY";
                
            // Write the OIFITS revision and instrument name
            cfitsio_set, fh, "AMPL VERSION", amplAMBER();
            cfitsio_set, fh,
                ["ARRNAME", "FRAME"],
                [OI.ARRNAME, OI.FRAME],
                ["Array name, for cross-referencing", "Coordinate frame"],
                ["", ""];
            cfitsio_set, fh, "OI_REVN", int(1),
                "Revision number of the table definition", "";
            cfitsio_set, fh,
                ["ARRAYX", "ARRAYY", "ARRAYZ"],
                [OI.ARRAYXYZ(1), OI.ARRAYXYZ(2), OI.ARRAYXYZ(3)],
                ["Coordinate frame", "Array center coordinates","Array center coordinates","Array center coordinates"],
                ["m", "m", "m"];
        }
    
        //////////////////////////////////////////////
        // Write OI_WAVELENGTH table (mandatory)
        //////////////////////////////////////////////
        if(!quiet)
            yocoLogInfo,"Saving OI_WAVELENGTH";
        
        // Check compliance with OIFITS standard
        if(is_void(*oi(kTab).EFF_WAVE))
            error, "No wavelength table!";
        if(is_void(*oi(kTab).EFF_BAND))
        {
            yocoLogWarning, "No bandwidth table! Setting to zero..";
            oi(kTab).EFF_BAND = &array(0.0,dimsof(*oi(kTab).EFF_WAVE));
        }

        // Set INSNAME value
        if(is_void(OI.INSNAME))
            OI.INSNAME = "amplSaveOiData_"+pr1(kTab);

        // Avoid multiples times the same INSNAME
        if(numberof(where((OI.INSNAME == oi.hdr.INSNAME)))>1)
            OI.INSNAME = OI.INSNAME+"_"+pr1(kTab);            

        if(!quiet)
            yocoLogInfo,OI.INSNAME;
        oldInsName = OI.INSNAME;

        // Check if the wavelength table is already present
        // in which case do not save it!
        if(is_void(THE_WLEN))
        {
            THE_WLEN = &(float(*oi(kTab).EFF_WAVE));
            THE_BAND = &(float(*oi(kTab).EFF_BAND));
            
            THE_INS = OI.INSNAME;
            THE_EXT = kTab;
            
            saveWlen = 1;

            // set values for later
            insname = OI.INSNAME;
            extver  = THE_EXT;
        }
        else
        {
            gotonext = 0;
            // Check the wavelength table already exists
            for (it=1;it<=numberof(THE_WLEN);it++)
            {
                // Check the wavelength table has the same number of elements
                N = numberof(*THE_WLEN(it));
                N2 = numberof(*oi(kTab).EFF_WAVE);
                if(N==N2)
                {
                    // Check the wavelength table is identical
                    if(allof(*THE_WLEN(it) == float(*oi(kTab).EFF_WAVE)))
                    {
                        // Do not save wavelength table
                        saveWlen = 0;
                        gotonext = 1;
                        insname = THE_INS(it);
                        extver  = THE_EXT(it);
                        
                        break;
                    }
                }
            }

            if(gotonext==0)
            {
                // if all wavelength tables are different to the current one,
                // then save it
            
                grow, THE_WLEN, &(float(*oi(kTab).EFF_WAVE));
                grow, THE_BAND, &(float(*oi(kTab).EFF_BAND));
            
                grow, THE_INS, OI.INSNAME;
                grow, THE_EXT, kTab;
            
                saveWlen = 1;

                // set values for later
                insname = OI.INSNAME;
                extver  = THE_EXT;
            
                saveWlen = 1;

                // set values for later
                insname = OI.INSNAME;
                extver  = THE_EXT;
            }
        }

        if(saveWlen)
        {
            amplCfitsio_add_bintable, fh,
                [&(float(*oi(kTab).EFF_WAVE)), &(float(*oi(kTab).EFF_BAND))],
                ["EFF_WAVE", "EFF_BAND"],
                ["m", "m"], "OI_WAVELENGTH";
            
            // Update keywords comments
            PallValues = cfitsio_get(fh, "TTYPE*", allComments, allNames, point=1);
            cfitsio_set, fh,
                "TTYPE"+swrite(format="%i",indgen(2)),
                PallValues,
                ["Effective wavelength of channel", "Effective bandpass of channel"], ["m", "m"];
            
            // Write the OIFITS revision and instrument name
            cfitsio_set, fh, "OI_REVN", int(1),
                "Revision number of the table definition", "";
            
            // Write the EXTVER number to identify table (compatibility with MIRA)
            cfitsio_set, fh, "EXTVER", extver,
                "ID number of this OI_WAVELENGTH", "";
            
            cfitsio_set, fh, "INSNAME", insname,
                "Name of detector, for cross-referencing";
        }
        
    
    
        //////////////////////////////////////////////
        // Write OI_VIS table
        //////////////////////////////////////////////
        if(oi(kTab).VISPHI!=pointer(null))
        {
            if(!quiet)
                yocoLogInfo,"Saving OI_VIS";
            //
            // bug fix: oi.STA_INDEX instead of OI.STA_INDEX
            //

            // Set number of bases
            nbBases = dimsof(*oi(kTab).VISPHI)(3);

            // Check compliance with OIFITS standard
            if(is_void(*oi(kTab).TIME))
            {
                yocoLogWarning,"No TIME table, setting all values to zero";
                oi(kTab).TIME = &array(0.0, nbBases);
            }
            if(is_void(*oi(kTab).MJD))
            {
                yocoLogWarning,"No TIME table, setting all values to zero";
                oi(kTab).MJD = &array(0.0, nbBases);
            }
            if(is_void(*oi(kTab).VISAMP))
                error, "No VISAMP table";
            if(is_void(*oi(kTab).VISAMPERR))
                error, "No VISAMPERR table";
            if(is_void(*oi(kTab).VISPHI))
                error, "No VISPHI table";
            if(is_void(*oi(kTab).VISPHIERR))
                error, "No VISPHIERR table";
            if(is_void(*oi(kTab).UCOORD))
                error, "No UCOORD table";
            if(is_void(*oi(kTab).VCOORD))
                error, "No VCOORD table";
            if(is_void(*oi(kTab).STA_INDEX))
            {
                yocoLogWarning,"No STA_INDEX table, setting random values";
                oi(kTab).STA_INDEX = &array(kTab, kTab+1, nbBases);
            }
            if(is_void(*oi(kTab).FLAG))
            {
                yocoLogWarning, "No FLAG table";
                oi(kTab).FLAG = &array(0,dimsof(*oi(kTab).VISPHI));
            }
            
            amplCfitsio_add_bintable, fh,
                [&array(int(1), nbBases),
                 oi(kTab).TIME,
                 oi(kTab).MJD,
                 &array(OI.INT_TIME, nbBases),
                 oi(kTab).VISAMP,
                 oi(kTab).VISAMPERR,
                 &(*oi(kTab).VISPHI*180./pi), // Phase given in radians as input
                 &(*oi(kTab).VISPHIERR*180./pi),
                 oi(kTab).UCOORD,
                 oi(kTab).VCOORD,
                 &(int(*oi(kTab).STA_INDEX)),
                 &(char(*oi(kTab).FLAG))],
                ["TARGET_ID", "TIME", "MJD", "INT_TIME", "VISAMP", "VISAMPERR",
                 "VISPHI", "VISPHIERR", "UCOORD", "VCOORD", "STA_INDEX", "FLAG"],
                ["", "s", "day", "s", "", "", "deg", "deg", "m", "m", "", ""],
                "OI_VIS";

            // Update keywords comments
            PallValues = cfitsio_get(fh, "TTYPE*", allComments, allNames, point=1);
            cfitsio_set, fh,
                "TTYPE"+swrite(format="%i",indgen(12)),
                PallValues,
                ["Target number as index into OI_TARGET table",
                 "UTC time of observation","Modified Julian Date",
                 "Integration time","Visibility amplitude",
                 "Error in visibility amplitude","Visibility phase",
                 "Error in visibility phase","u coordinate of the data",
                 "v coordinate of the data",
                 "Station numbers contributing to the data","Flag"],
                ["s", "day", "s", "", "", "deg", "deg", "m", "m", "", ""];
        
            // Write the OIFITS revision and instrument name
            cfitsio_set, fh, "OI_REVN", int(1),
                "Revision number of the table definition", "";
            
            // Write the EXTVER number to identify table (compatibility with MIRA)
            cfitsio_set, fh, "EXTVER", extver,
                "Reference to the corresponding OI_WAVELENGTH", "";
        
            cfitsio_set, fh,
                ["DATE-OBS", "INSNAME"],
                [OI.DATE_OBS, insname],
                ["UTC start date of observations", "Identifies corresponding OI_WAVELENGTH table"];
        
            // ARRNAME is optional in the standard
            if(OI.ARRNAME!=string(nil))
                cfitsio_set, fh, "ARRNAME", OI.ARRNAME, "Identifies corresponding OI_ARRAY";
        }
    
        //////////////////////////////////////////////
        // Write OI_VIS2 table
        //////////////////////////////////////////////
        if(oi(kTab).VIS2DATA!=pointer(null))
        {
            if(!quiet)
                yocoLogInfo,"Saving OI_VIS2";

            // Set number of bases
            nbBases = dimsof(*oi(kTab).VIS2DATA)(3);

            if(is_void(*oi(kTab).TIME))
            {
                yocoLogWarning,"No TIME table, setting all values to zero";
                oi(kTab).TIME = &array(0.0, nbBases);
            }
            if(is_void(*oi(kTab).MJD))
            {
                yocoLogWarning,"No TIME table, setting all values to zero";
                oi(kTab).MJD = &array(0.0, nbBases);
            }

            // Check compliance with OIFITS standard
            if(is_void(*oi(kTab).VIS2DATA))
                error, "No VIS2DATA table";
            if(is_void(*oi(kTab).VIS2ERR))
                error, "No VIS2ERR table";
            if(is_void(*oi(kTab).UCOORD))
                error, "No UCOORD table";
            if(is_void(*oi(kTab).VCOORD))
                error, "No VCOORD table";
            if(is_void(*oi(kTab).STA_INDEX))
            {
                yocoLogWarning,"No STA_INDEX table, setting random values";
                oi(kTab).STA_INDEX = &array(kTab, kTab+1, nbBases);
            }
            if(is_void(*oi(kTab).FLAG))
            {
                yocoLogWarning, "No FLAG table";
                oi(kTab).FLAG = &array(0,dimsof(*oi(kTab).VIS2DATA));
            }
            
            //
            // bug fix: oi.STA_INDEX instead of OI.STA_INDEX
            //
            amplCfitsio_add_bintable, fh,
                [&array(int(1), nbBases),
                 oi(kTab).TIME,
                 oi(kTab).MJD,
                 &array(OI.INT_TIME, nbBases),
                 oi(kTab).VIS2DATA,
                 oi(kTab).VIS2ERR,
                 oi(kTab).UCOORD,
                 oi(kTab).VCOORD,
                 &(int(*oi(kTab).STA_INDEX)),
                 //&(char(*oi(kTab).FLAG))],
                 &(char(*oi(kTab).FLAG))],
                ["TARGET_ID", "TIME", "MJD", "INT_TIME", "VIS2DATA", "VIS2ERR",
                 "UCOORD", "VCOORD", "STA_INDEX", "FLAG"],
                ["", "s", "day", "s", "", "", "m", "m", "", ""], "OI_VIS2";
        
            // Update keywords comments
            PallValues = cfitsio_get(fh, "TTYPE*", allComments, allNames, point=1);
            cfitsio_set, fh,
                "TTYPE"+swrite(format="%i",indgen(10)),
                PallValues,
                ["Target number as index into OI_TARGET table",
                 "UTC time of observation", "Modified Julian Date",
                 "Integration time", "Squared visibility",
                 "Error in squared visibility", "u coordinate of the data",
                 "v coordinate of the data",
                 "Station numbers contributing to the data", "Flag"],
                ["", "s", "day", "s", "", "", "m", "m", "", ""];
        
            // Write the OIFITS revision and instrument name
            cfitsio_set, fh, "OI_REVN", int(1),
                "Revision number of the table definition", "";
            // Write the EXTVER number to identify table (compatibility with MIRA)
            
            cfitsio_set, fh, "EXTVER", extver,
                "Reference to the corresponding OI_WAVELENGTH", "";
        
            cfitsio_set, fh,
                ["DATE-OBS", "INSNAME"],
                [OI.DATE_OBS, insname],
                ["UTC start date of observations", "Identifies corresponding OI_WAVELENGTH table"];
            // ARRNAME is optional in the standard
            if(OI.ARRNAME!=string(nil))
                cfitsio_set, fh, "ARRNAME", OI.ARRNAME, "Identifies corresponding OI_ARRAY";
        }
    
        //////////////////////////////////////////////
        // Write OI_T3 table
        //////////////////////////////////////////////
        if(oi(kTab).T3PHI!=pointer(null))
        {
            if(!quiet)
                yocoLogInfo,"Saving OI_T3";

            // Set number of bases
            nbBases = dimsof(*oi(kTab).T3PHI)(3);
            //
            // bug fix: need () to have only one element
            // bug fix: need t3max to write more than 1 ligne

            t3max = dimsof(*oi(kTab).T3PHI)(3);
            
            if(is_void(*oi(kTab).TIME3))
            {
                yocoLogWarning,"No TIME3 table, setting all values to zero";
                oi(kTab).TIME3 = &array(0.0, nbClos);
            }
            if(is_void(*oi(kTab).MJD3))
            {
                yocoLogWarning,"No TIME3 table, setting all values to zero";
                oi(kTab).MJD3 = &array(0.0, nbClos);
            }
            
            // Check compliance with OIFITS standard
            if(is_void(*oi(kTab).T3PHI))
                error, "No T3PHI table";
            if(is_void(*oi(kTab).T3PHIERR))
                error, "No T3PHIERR table";
            if(is_void(*oi(kTab).T3AMP))
            {
                yocoLogWarning, "No T3AMP table, setting all values to 0";
                oi(kTab).T3AMP = &array(0.0,dimsof(*oi(kTab).T3PHI));
            }
            if(is_void(*oi(kTab).T3AMPERR))
            {
                yocoLogWarning, "No T3AMPERR table, setting all values to 0";
                oi(kTab).T3AMPERR = &array(0.0,dimsof(*oi(kTab).T3PHIERR));
            }
            if(is_void(*oi(kTab).U1COORD))
                error, "No U1COORD table";
            if(is_void(*oi(kTab).V1COORD))
                error, "No V1COORD table";
            if(is_void(*oi(kTab).U2COORD))
                error, "No U2COORD table";
            if(is_void(*oi(kTab).V2COORD))
                error, "No V2COORD table";
            if(is_void(*oi(kTab).STA_INDEX3))
            {
                yocoLogWarning, "No STA_INDEX3 table. Setting to 0";
                oi(kTab).STA_INDEX3 = &array(kTab, kTab+1, nbClos);
            }
            if(is_void(*oi(kTab).FLAG3))
            {
                yocoLogWarning, "No FLAG3 table. Setting to 'f'...";
                oi(kTab).FLAG3 = &array(0,dimsof(*oi(kTab).T3PHIERR));
            }
            
            amplCfitsio_add_bintable, fh,
                [&array(int(1),t3max),
                 &(*oi(kTab).TIME3),
                 &(*oi(kTab).MJD3),
                 &array(OI.INT_TIME,t3max),
                 oi(kTab).T3AMP, oi(kTab).T3AMPERR,
                 &(*oi(kTab).T3PHI * 180./pi), // Phase in radians
                 &(*oi(kTab).T3PHIERR * 180./pi),
                 oi(kTab).U1COORD, oi(kTab).V1COORD,
                 oi(kTab).U2COORD, oi(kTab).V2COORD,
                 oi(kTab).STA_INDEX3,
                 //&(char(array((*oi(kTab).FLAG)(,sum)>0,t3max)))],
                 &(char(*oi(kTab).FLAG3))],
                ["TARGET_ID", "TIME", "MJD", "INT_TIME", "T3AMP", "T3AMPERR",
                 "T3PHI", "T3PHIERR", "U1COORD", "V1COORD", "U2COORD", "V2COORD",
                 "STA_INDEX", "FLAG"],
                ["", "s", "day", "s", "", "", "deg", "deg", "m", "m", "m", "m",
                 "", ""], "OI_T3";

            // Update keywords comments
            PallValues = cfitsio_get(fh, "TTYPE*", allComments, allNames, point=1);
            cfitsio_set, fh,
                "TTYPE"+swrite(format="%i",indgen(12)),
                PallValues,
                ["Target number as index into OI_TARGET table","UTC time of observation","Modified Julian Date", "Integration time","Triple-product amplitude","Error in triple product amplitude","Triple-product phase","Error in triple product phase","u coordinate of baseline AB of the triangle","v coordinate of baseline AB of the triangle","u coordinate of baseline BC of the triangle","v coordinate of baseline BC of the triangle","Station numbers contributing to the data","Flag"],
                ["", "s", "day", "s", "", "", "deg", "deg", "m", "m", "m", "m", "", ""];
        
            // Write the OIFITS revision and instrument name
            cfitsio_set, fh, "OI_REVN", int(1),
                "Revision number of the table definition", "";
            // Write the EXTVER number to identify table (compatibility with MIRA)
            
            cfitsio_set, fh, "EXTVER", extver,
                "Reference to the corresponding OI_WAVELENGTH", "";
        
            cfitsio_set, fh,
                ["DATE-OBS", "INSNAME"],
                [OI.DATE_OBS, insname],
                ["UTC start date of observations", "Identifies corresponding OI_WAVELENGTH table"];
        
            // ARRNAME is optional in the standard
            if(OI.ARRNAME!=string(nil))
                cfitsio_set, fh, "ARRNAME", OI.ARRNAME, "Identifies corresponding OI_ARRAY";
        }

        //////////////////////////////////////////////
        // Write AMPL_SPEC table
        //////////////////////////////////////////////
        if(oi(kTab).spec!=pointer(null))
        {
            if(!quiet)
                yocoLogInfo,"Saving AMPL_SPEC";
            //
            // bug fix: oi.STA_INDEX instead of OI.STA_INDEX
            //

            // Set number of bases
            nbBases = dimsof(*oi(kTab).spec)(3);

            // Check compliance with OIFITS standard
            if(is_void(*oi.TIME))
            {
                yocoLogWarning,"No TIME table, setting all values to zero";
                oi.TIME = &array(0.0, nbBases);
            }
            if(is_void(*oi.MJD))
            {
                yocoLogWarning,"No TIME table, setting all values to zero";
                oi.MJD = &array(0.0, nbBases);
            }
            if(is_void(*oi(kTab).spec))
                error, "No VISAMP table";
            if(is_void(*oi(kTab).specErr))
                error, "No VISAMPERR table";
            if(is_void(*oi(kTab).STA_INDEX))
            {
                yocoLogWarning,"No STA_INDEX table, setting random values";
                oi(kTab).STA_INDEX = &array(kTab, kTab+1, nbBases);
            }
            if(is_void(*oi(kTab).FLAG))
            {
                yocoLogWarning, "No FLAG table";
                oi(kTab).FLAG = &array(0,dimsof(*oi(kTab).VISPHI));
            }
            
            amplCfitsio_add_bintable, fh,
                [&array(int(1), nbBases),
                 oi(kTab).TIME,
                 oi(kTab).MJD,
                 &array(OI.INT_TIME, nbBases),
                 oi(kTab).spec,
                 oi(kTab).specErr,
                 &(int(*oi(kTab).STA_INDEX)),
                 &(char(*oi(kTab).FLAG))],
                ["TARGET_ID", "TIME", "MJD", "INT_TIME", "SPECTRUM", "SPECTRUMERR",
                 "STA_INDEX", "FLAG"],
                ["", "s", "day", "s", "e-", "e-", "", ""],
                "AMPL_SPEC";

            // Update keywords comments
            PallValues = cfitsio_get(fh, "TTYPE*", allComments, allNames, point=1);
            cfitsio_set, fh,
                "TTYPE"+swrite(format="%i",indgen(12)),
                PallValues,
                ["Target number as index into OI_TARGET table",
                 "UTC time of observation","Modified Julian Date",
                 "Integration time","Spectrum",
                 "Error in spectrum",
                 "Station numbers contributing to the data","Flag"],
                ["s", "day", "s", "e-", "e-", "", ""];
        
            // Write the OIFITS revision and instrument name
            cfitsio_set, fh, "AMPL_REVN", int(1),
                "Revision number of the table definition", "";
            cfitsio_set, fh,
                ["DATE-OBS", "INSNAME"],
                [OI.DATE_OBS, OI.INSNAME],
                ["UTC start date of observations", "Identifies corresponding OI_WAVELENGTH table"];
        
            // ARRNAME is optional in the standard
            if(OI.ARRNAME!=string(nil))
                cfitsio_set, fh, "ARRNAME", OI.ARRNAME, "Identifies corresponding OI_ARRAY";
        }

        //////////////////////////////////////////////
        // Write AMPL_VIS2 table
        //////////////////////////////////////////////
        if(oi(kTab).PHOT!=pointer(null))
        {
            if(!quiet)
                yocoLogInfo,"Saving AMPL_VIS2";

            // Set number of bases
            nbBases = dimsof(*oi(kTab).VIS2DATA)(3);

            if(is_void(*oi.TIME))
            {
                yocoLogWarning,"No TIME table, setting all values to zero";
                oi.TIME = &array(0.0, nbBases);
            }
            if(is_void(*oi.MJD))
            {
                yocoLogWarning,"No TIME table, setting all values to zero";
                oi.MJD = &array(0.0, nbBases);
            }

            // Check compliance with OIFITS standard
            if(is_void(*oi(kTab).VIS2DATA))
                error, "No VIS2DATA table";
            if(is_void(*oi(kTab).VIS2ERR))
                error, "No VIS2ERR table";
            if(is_void(*oi(kTab).UCOORD))
                error, "No UCOORD table";
            if(is_void(*oi(kTab).VCOORD))
                error, "No VCOORD table";
            if(is_void(*oi(kTab).STA_INDEX))
                error, "No STA_INDEX table";
            if(is_void(*oi(kTab).FLAG))
                error, "No FLAG table";
            if(is_void(*oi(kTab).PHOT))
                error, "No PHOT table";
            if(is_void(*oi(kTab).PHOTERR))
                error, "No PHOTERR table";
            if(is_void(*oi(kTab).COHFLUX))
                error, "No COHFLUX table";
            if(is_void(*oi(kTab).COHFERR))
                error, "No COHFERR table";
            if(is_void(*oi(kTab).OPD))
                error, "No OPD table";
            if(is_void(*oi(kTab).RAWCOHFLUX))
                error, "No RAWCOHFLUX table";
            if(is_void(*oi(kTab).RAWCOHFERR))
                error, "No RAWCOHFERR table";
            if(is_void(*oi(kTab).AO_STREHL))
                error, "No AO_STREHL table";
            if(is_void(*oi(kTab).AO_WFE))
                error, "No AO_WFE table";

            amplCfitsio_add_bintable, fh,
                [&array(int(1), nbBases),
                 oi(kTab).TIME,
                 oi(kTab).MJD,
                 &array(OI.INT_TIME, nbBases),
                 oi(kTab).VIS2DATA,
                 oi(kTab).VIS2ERR,
                 oi(kTab).UCOORD,
                 oi(kTab).VCOORD,
                 &(int(*oi(kTab).STA_INDEX)),
                 //&(char(*oi(kTab).FLAG))],
                 &(char(*oi(kTab).FLAG)),
                 oi(kTab).PHOT,
                 oi(kTab).PHOTERR,
                 oi(kTab).COHFLUX,
                 oi(kTab).COHFERR,
                 oi(kTab).OPD,
                 oi(kTab).RAWCOHFLUX,
                 oi(kTab).RAWCOHFERR,
                 oi(kTab).AO_STREHL,
                 oi(kTab).AO_WFE],
                ["TARGET_ID", "TIME", "MJD", "INT_TIME", "VIS2DATA", "VIS2ERR",
                 "UCOORD", "VCOORD", "STA_INDEX", "FLAG",
                 "PHOT","PHOTERR","COHFLUX","COHFERR","OPD",
                 "RAWCOHFLUX","RAWCOHFERR","AO_STREHL","AO_WFE"],
                ["", "s", "day", "s", "", "", "m", "m", "", "",
                 "counts","counts","counts","counts","m",
                 "counts","counts","percent","rad2"], "AMPL_VIS2";

            // Update keywords comments
            PallValues = cfitsio_get(fh, "TTYPE*", allComments, allNames, point=1);
            cfitsio_set, fh,
                "TTYPE"+swrite(format="%i",indgen(10)),
                PallValues,
                ["Target number as index into OI_TARGET table",
                 "UTC time of observation", "Modified Julian Date",
                 "Integration time", "Squared visibility",
                 "Error in squared visibility", "u coordinate of the data",
                 "v coordinate of the data",
                 "Station numbers contributing to the data", "Flag",
                 "Combined photometry of two relevant beams",
                 "Photometry RMS",
                 "Coherent flux",
                 "Coherent flux error",
                 "OPD",
                 "Raw coherent flux",
                 "Raw coherent flux error",
                 "AO streal mean","AO wavefront error"],
                ["", "s", "day", "s", "", "", "m", "m", "", "",
                 "","","","","m","","","",""];
        
            // Write the OIFITS revision and instrument name
            cfitsio_set, fh, "OI_REVN", int(1),
                "Revision number of the table definition", "";
            // Write the EXTVER number to identify table (compatibility with MIRA)
            
            cfitsio_set, fh, "EXTVER", extver,
                "Reference to the corresponding OI_WAVELENGTH", "";
        
            cfitsio_set, fh,
                ["DATE-OBS", "INSNAME"],
                [OI.DATE_OBS, insname],
                ["UTC start date of observations", "Identifies corresponding OI_WAVELENGTH table"];
            // ARRNAME is optional in the standard
            if(OI.ARRNAME!=string(nil))
                cfitsio_set, fh, "ARRNAME", OI.ARRNAME, "Identifies corresponding OI_ARRAY";
        }
    
        //////////////////////////////////////////////
        // Write AMPL_T3 table
        //////////////////////////////////////////////
        if((oi(kTab).OPD1!=pointer(null))&&(amplTable==1))
        {
            if(!quiet)
                yocoLogInfo,"Saving AMPL_T3";
            //
            // bug fix: need () to have only one element
            // bug fix: need t3max to write more than 1 ligne

            // Set number of bases
            nbBases = dimsof(*oi(kTab).T3PHI)(3);

            t3max = dimsof(*oi.T3PHI)(3);
            
            if(is_void(*oi.TIME3))
            {
                yocoLogWarning,"No TIME3 table, setting all values to zero";
                oi.TIME3 = &array(0.0, nbClos);
            }
            if(is_void(*oi.MJD3))
            {
                yocoLogWarning,"No TIME3 table, setting all values to zero";
                oi.MJD3 = &array(0.0, nbClos);
            }
            
            // Check compliance with OIFITS standard
            if(is_void(*oi(kTab).T3PHI))
                error, "No T3PHI table";
            if(is_void(*oi(kTab).T3PHIERR))
                error, "No T3PHIERR table";
            if(is_void(*oi(kTab).T3AMP))
            {
                yocoLogWarning, "No T3AMP table, setting all values to 0";
                oi(kTab).T3AMP = &array(0.0,dimsof(*oi(kTab).T3PHI));
            }
            if(is_void(*oi(kTab).T3AMPERR))
            {
                yocoLogWarning, "No T3AMPERR table, setting all values to 0";
                oi(kTab).T3AMPERR = &array(0.0,dimsof(*oi(kTab).T3PHIERR));
            }
            if(is_void(*oi(kTab).U1COORD))
                error, "No U1COORD table";
            if(is_void(*oi(kTab).V1COORD))
                error, "No V1COORD table";
            if(is_void(*oi(kTab).U2COORD))
                error, "No U2COORD table";
            if(is_void(*oi(kTab).V2COORD))
                error, "No V2COORD table";
            if(is_void(*oi(kTab).STA_INDEX3))
                yocoLogWarning, "No STA_INDEX3 table";
            if(is_void(*oi(kTab).FLAG3))
            {
                yocoLogWarning, "No FLAG3 table";
                oi(kTab).FLAG3 = &array(0,dimsof(*oi(kTab).T3PHIERR));
            }
            if(is_void(*oi(kTab).OPD1))
                error, "No OPD1 table";
            if(is_void(*oi(kTab).OPD2))
                error, "No OPD2 table";
            if(is_void(*oi(kTab).RAWT3PHI))
                error, "No RAWT3PHI table";
            if(is_void(*oi(kTab).RAWT3PHIERR))
                error, "No RAWT3PHIERR table";
            
            amplCfitsio_add_bintable, fh,
                [&array(int(1),t3max),
                 &(*oi(kTab).TIME3),
                 &(*oi(kTab).MJD3),
                 &array(OI.INT_TIME,t3max),
                 oi(kTab).T3AMP, oi(kTab).T3AMPERR,
                 &(*oi(kTab).T3PHI * 180./pi), // Phase in radians
                 &(*oi(kTab).T3PHIERR * 180./pi),
                 oi(kTab).U1COORD, oi(kTab).V1COORD,
                 oi(kTab).U2COORD, oi(kTab).V2COORD,
                 oi(kTab).STA_INDEX3,
                 //&(char(array((*oi(kTab).FLAG)(,sum)>0,t3max)))],
                 &(char(*oi.FLAG3)),
                 oi(kTab).OPD1,
                 oi(kTab).OPD2,
                 &(*oi(kTab).RAWT3PHI    *180./pi),
                 &(*oi(kTab).RAWT3PHIERR *180./pi)],
                ["TARGET_ID", "TIME", "MJD", "INT_TIME", "T3AMP", "T3AMPERR",
                 "T3PHI", "T3PHIERR", "U1COORD", "V1COORD", "U2COORD", "V2COORD",
                 "STA_INDEX", "FLAG","OPD1","OPD2","RAWT3PHI","RAWT3PHIERR"],
                ["", "s", "day", "s", "", "", "", "", "m", "m", "m", "m",
                 "", "","m","m","",""], "AMPL_T3";

            // Update keywords comments
            PallValues = cfitsio_get(fh, "TTYPE*", allComments, allNames, point=1);
            cfitsio_set, fh,
                "TTYPE"+swrite(format="%i",indgen(12)),
                PallValues,
                ["Target number as index into OI_TARGET table","UTC time of observation","Modified Julian Date", "Integration time","Triple-product amplitude","Error in triple product amplitude","Triple-product phase","Error in triple product phase","u coordinate of baseline AB of the triangle","v coordinate of baseline AB of the triangle","u coordinate of baseline BC of the triangle","v coordinate of baseline BC of the triangle","Station numbers contributing to the data","Flag","OPD1","OPD2","Raw triple-product phase","Error in raw triple-product phase"],
                ["", "s", "day", "s", "", "", "deg", "deg", "m", "m", "m", "m", "", "","m","m","",""];
        
            // Write the OIFITS revision and instrument name
            cfitsio_set, fh, "OI_REVN", int(1),
                "Revision number of the table definition", "";
            // Write the EXTVER number to identify table (compatibility with MIRA)
            
            cfitsio_set, fh, "EXTVER", extver,
                "Reference to the corresponding OI_WAVELENGTH", "";
        
            cfitsio_set, fh,
                ["DATE-OBS", "INSNAME"],
                [OI.DATE_OBS, insname],
                ["UTC start date of observations", "Identifies corresponding OI_WAVELENGTH table"];
        
            // ARRNAME is optional in the standard
            if(OI.ARRNAME!=string(nil))
                cfitsio_set, fh, "ARRNAME", OI.ARRNAME, "Identifies corresponding OI_ARRAY";
        }


        
    } // end loop on kTab

    

    //////////////////////////////////////////////
    // Add AMBER-specific keywords
    //////////////////////////////////////////////
    if(amberKeys==1)
    {
        cfitsio_rewind, fh;
        cfitsio_set, fh,
            ["HIERARCH ESO INS MODE",
             "HIERARCH ESO OBS TARG NAME",
             "DATE-OBS",
             "HIERARCH ESO DPR CATG",
             "HIERARCH ESO DPR TYPE",
             "HIERARCH ESO INS OPTI7 NAME",
             "HIERARCH ESO DEL FT STATUS",
             "HIERARCH ESO ISS CONF STATION1",
             "HIERARCH ESO ISS CONF STATION2",
             "HIERARCH ESO ISS CONF STATION3",
             "HIERARCH ESO PRO CATG",
             "HIERARCH ESO OCS DET FRAM TYPE",
             "HIERARCH ESO OBS NAME"],
            [oi.hdr.INS_MODE,
             oi.hdr.TARGET,
             oi.hdr.DATE_OBS,
             oi.hdr.catg,
             "OBJECT",
             oi.hdr.bcd,
             oi.hdr.tracker,
             (*oi.hdr.STA_NAME)(1),
             (*oi.hdr.STA_NAME)(2),
             (*oi.hdr.STA_NAME)(3),
             oi.hdr.catg+"_AVERAGED",
             "",
             oi.hdr.obName],
            ["AMBER instrument mode"];
        
        cfitsio_set, fh,
            ["EXPTIME",
             "HIERARCH ESO DET DIT",
             "HIERARCH ESO DET NDIT",
             "HIERARCH ESO ISS AIRM START",
             "HIERARCH ESO ISS AMBI FWHM START",
             "HIERARCH ESO ISS AMBI TAU0 START",
             "HIERARCH ESO INS GRAT1 WLEN",
             "HIERARCH ESO INS GRAT1 ORDER ",
             "HIERARCH ESO QC P1 OFFSETY", 
             "HIERARCH ESO QC P2 OFFSETY", 
             "HIERARCH ESO QC P3 OFFSETY",
             "MJD-OBS",
             "HIERARCH ESO OBS TPLNO",
             "HIERARCH ESO OBS ID",
             "HIERARCH ESO OCS P2VM ID",
             "RA",
             "DEC"],
            [oi.hdr.INT_TIME,
             oi.hdr.INT_TIME,
             oi.hdr.ndit,
             0,
             0,
             0,
             0,
             0,
             0,
             0,
             0,
             oi.hdr.MJD,
             oi.hdr.tplNr,
             oi.hdr.obId,
             oi.hdr.p2vmId,
             oi.hdr.RAEP0,
             oi.hdr.DECEP0],
            ["AMBER instrument mode"];


        //
        // fix: do below only when OI_VIS is available
        //
        if(oi.VISPHI!=pointer(null))
        {
            cfitsio_goto_hdu, fh, "OI_VIS";
            data = array(complex, nbWlen, nbBases);
            if(is_void(cfitsio_add_column))
                cfitsio_add_column = cfitsio_add_col;
            cfitsio_add_column, fh, data, "VISDATA";
            cfitsio_add_column, fh, data, "VISERR";
        }
    }
    
    cfitsio_close,fh;                            // close the file
}

/***************************************************************************/

func amplLoadSciFiles(&dataStruct, nightDir=, logFile=, verbose=, calibrated=, crl=, crr=)
    /* DOCUMENT amplLoadSciFiles(&dataStruct, nightDir=, logFile=, verbose=, calibrated=, crl=, crr=)

       DESCRIPTION

       PARAMETERS
       - dataStruct: 
       - nightDir  : 
       - logFile   : 
       - verbose   : 
       - calibrated: 
       - crl       : 
       - crr       : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    /* Select directory containing files to be treated */
    message = "Choose the product directory (OI_DATA)"; 
    if (amdlibFileChooser(message, nightDir) == 0)
    {
        return 0;
    }

    if(is_void(verbose))
        verbose=1;

    if(is_void(logFile))
    {
        amdlibGetLogName, nightDir, logFile;
        logFile = nightDir + string(logFile);
    }

    amdlibReadLog, logFile, logTable, titles;

    tabFiles = logTable(where(titles=="fileName"), );
    tabDates = logTable(where(titles=="date_obs"), );
    tabNr    = logTable(where(titles=="calib_nr"), );
    proTypes = logTable(where(titles=="pro_catg"), );
    tabTypes = logTable(where(titles=="obs_catg"), );
    OBJTypes = logTable(where(titles=="obs_type"), );
    tabIds   = logTable(where(titles=="P2VM_ID"), );
    tabOBsNr = logTable(where(titles=="OB_nr"), );
    tabTplNr = logTable(where(titles=="Template_nr"), );
    tabNames = logTable(where(titles=="object_name"), );

    sciObs = where((proTypes=="SCIENCE_AVERAGED"));
    if(calibrated==1)
        sciObs = where((proTypes=="SCIENCE_CALIBRATED"));


    sciFiles = tabFiles(sciObs);
    sciNames = tabNames(sciObs);
    nSciFiles = numberof(sciFiles);

    if(nSciFiles==0)
        return 0;

    amplLoadOiDatas, dataStruct, inOiFile=nightDir+sciFiles;
    
    for(kFile=1;kFile<=nSciFiles;kFile++)
    {
    }
}

/***************************************************************************/

func amplLoadOiDatas(&oi, inOiFile=, crl=, crr=, wlenRange=, freqMax=, quiet=)
    /* DOCUMENT amplLoadOiDatas(&oi, inOiFile=, crl=, crr=, wlenRange=, freqMax=, quiet=)

       DESCRIPTION
       oi fits file reader

       PARAMETERS
       - oi       : output oi data structure
       - inOiFile : input file name
       - crl      : index for cutting wavelength edges on left side
       - crr      : index for cutting wavelength edges on right side
       - wlenRange: wavelength range to keep in data, anything outside discarded
       - freqMax  : 
       - quiet    : quiet mode

       SEE ALSO
    */
{
    /* Check the inputFile */
    message = "OI DATA FILE\n Enter the AMBER  data file to load";
    if (amdlibFileChooser(message, inOiFile, 2) == 0)
    {
        yocoError,"'inputFile' not specified correctly.";
        return 0;
    }

    oi = [];
    for(k=1;k<=numberof(inOiFile);k++)
    {
        amplLoadOiData, tmpoi, inOiFile=inOiFile(k), crl=crl, crr=crr, wlenRange=wlenRange, freqMax=freqMax;
        grow,oi,tmpoi;
    }
}

/***************************************************************************/

func amplLoadOiData(&oi, inOiFile=, crl=, crr=, wlenRange=, freqMax=)
    /* DOCUMENT amplLoadOiData(&oi, inOiFile=, crl=, crr=, wlenRange=, freqMax=)

       DESCRIPTION
       oi fits file reader

       PARAMETERS
       - oi       : 
       - inOiFile : 
       - crl      : 
       - crr      : 
       - wlenRange: 
       - freqMax  : 

       SEE ALSO
    */
{
    /* Check the inputFile */
    message = "OI DATA FILE\n Enter the AMBER  data file to load";
    if (amdlibFileChooser(message, inOiFile, 2) == 0)
    {
        yocoError,"'inputFile' not specified correctly.";
        return 0;
    }

    if(numberof(inOiFile)!=1)
        error, "amplLoadOiData works on just one file. Please use amplLoadOiDatas instead.";

    yocoFileSplitName,inOiFile,d,f,e;
    if(!quiet)
        yocoLogInfo, "in", d;
    if(!quiet)
        yocoLogInfo,"Loading",f+e;
    
    amplLoadDataHdr,hdr,inputFile=inOiFile, quiet=quiet;
    
    // open fits file
    fh = cfitsio_open(inOiFile, "r");

    binTables = cfitsio_list(fh);
    nhdu      = numberof(binTables);
    
    // Test oi data revision (should be 1 in this version of the reader)
    oiRevNr = [];
    for(k=1;k<=nhdu;k++)
    {
        cfitsio_goto_hdu,fh,k;
        grow,oiRevNr,cfitsio_get(fh, "OI_REVN", allComments, allNames, point=0);
    }
    if(nallof(oiRevNr==1))
        yocoLogWarning,"This scripts handles ONLY OI FITS revision 1";

    // Get all binary table names
    binTablesNames = insNames = [];
    for(khdu=1;khdu<=nhdu;khdu++)
    {
        cfitsio_goto_hdu,fh,khdu;
        btbl = cfitsio_get(fh, "EXTNAME", allComments, allNames, point=0);
        if(is_void(btbl))
            grow,binTablesNames,"";
        else
            grow,binTablesNames,btbl;
            
        // Identify the INSNAME keyword
        insn = cfitsio_get(fh, "INSNAME", allComments, allNames, point=0);
        if(is_void(insn))
            grow,insNames,"";
        else
            grow,insNames,insn;
    }
    
    //////////////////////////////////////////////
    // First, look at the OI_TARGET table
    //////////////////////////////////////////////
    oiTargetHdu = where(binTablesNames == "OI_TARGET");
    if(numberof(oiTargetHdu) > 1)
    {
        yocoLogWarning,"more than one OI_TARGET table, non-conformant with \
OI FITS standard. Setting default to first table...";
        oiTargetHdu = oiTargetHdu(1);
    }
    else if(numberof(oiTargetHdu) == 0)
    {
        // If there is no OI_TARGET try to go on without it
        yocoLogWarning,"No OI_TARGET table! non-conformant with OI FITS\
standard. Proceeding...";
    }
    else
        oiTargetHdu = oiTargetHdu(1);

    // Search for data tables
    dataTables =  where((binTablesNames=="OI_VIS")|
                        (binTablesNames=="OI_VIS2")|
                        (binTablesNames=="AMPL_SPEC")|
                        (binTablesNames=="AMBER_SPECTRUM")|
                        (binTablesNames=="OI_T3"));
    nTab = numberof(dataTables);
    
    if(numberof(oiTargetHdu) != 0)
    {
        //////////////////////////////////////////////
        // Read  OI_TARGET table (mandatory)
        //////////////////////////////////////////////
        cfitsio_goto_hdu,fh,oiTargetHdu;
        binTable = cfitsio_read_bintable(fh, bintitle);

        tmp = yocoStrRemoveMultiple(*binTable(where(bintitle=="TARGET")(1)));
        nTarg = numberof(tmp);
        
        // Define the output oi data structure
        oi = array(amplOiData(),nTarg*nTab);
        oi.hdr = hdr;
        
        // Fill values in the structure
        oi.hdr.TARGET    = tmp;
        oi.hdr.TARGET_ID = *binTable(where(bintitle=="TARGET_ID")(1));
        oi.hdr.RAEP0     = *binTable(where(bintitle=="RAEP0")(1));
        oi.hdr.DECEP0    = *binTable(where(bintitle=="DECEP0")(1));
        oi.hdr.EQUINOX   = *binTable(where(bintitle=="EQUINOX")(1));
        oi.hdr.RA_ERR    = *binTable(where(bintitle=="RA_ERR")(1));
        oi.hdr.DEC_ERR   = *binTable(where(bintitle=="DEC_ERR")(1));
        oi.hdr.SYSVEL    = *binTable(where(bintitle=="SYSVEL")(1));
        oi.hdr.VELTYP    = *binTable(where(bintitle=="VELTYP")(1));
        oi.hdr.VELDEF    = *binTable(where(bintitle=="VELDEF")(1));
        oi.hdr.PMRA      = *binTable(where(bintitle=="PMRA")(1));
        oi.hdr.PMDEC     = *binTable(where(bintitle=="PMDEC")(1));
        oi.hdr.PMRA_ERR  = *binTable(where(bintitle=="PMRA_ERR")(1));
        oi.hdr.PMDEC_ERR = *binTable(where(bintitle=="PMDEC_ERR")(1));
        oi.hdr.PARALLAX  = *binTable(where(bintitle=="PARALLAX")(1));
        oi.hdr.PARA_ERR  = *binTable(where(bintitle=="PARA_ERR")(1));
        oi.hdr.SPECTYP   = *binTable(where(bintitle=="SPECTYP")(1));
        
        // test VELTYP keyword
        for(k=1;k<=nTarg;k++)
            if(noneof(oi(k).hdr.VELTYP == ["LSR", "HELIOCEN", "BARYCENT",
                                           "GEOCENTR", "TOPOCENT", "UNKNOWN"]))
            {
                yocoLogWarning, "No [proper] radial velocity type inserted! Should \
be one of LSR, HELIOCEN, BARYCENT, GEOCENTR, TOPOCENT, UNKNOWN. Putting default to \"UNKNOWN\".";
                oi.hdr.VELTYP = "UNKNOWN";
            }
        
        // Check radial velocity definition
        for(k=1;k<=nTarg;k++)
            if(noneof(oi(k).hdr.VELDEF == ["RADIO", "OPTICAL"]))
            {
                yocoLogWarning, "No [proper] radial velocity definition inserted!\
Should be one of RADIO, OPTICAL Putting default to \"OPTICAL\".";
                oi.hdr.VELTYP = "OPTICAL";
            }
    }
    else
    {
        //FIXME: patch for crappy MIDI data.
        // Define the output oi data structure
        oi     = array(amplOiData(),1);
        oi.hdr = hdr;
    }
    
    oiWavelengthHdu = where(binTablesNames == "OI_WAVELENGTH");
    if(numberof(oiWavelengthHdu) == 0)
        // If there is no OI_WAVELENGTH issue an error (really mandatory)
        yocoLogError,"No OI_WAVELENGTH table! non-conformant with OI FITS standard";
    
    //////////////////////////////////////////////
    // Read OI_WAVELENGTH table (mandatory)
    //////////////////////////////////////////////
    // Find corresponding oi wavelength table
    theOiWave = where((binTablesNames=="OI_WAVELENGTH"))(1);
    
    cfitsio_goto_hdu,fh,theOiWave;
    binTable = cfitsio_read_bintable(fh, bintitle);
    
    wlen = ((*binTable(where(bintitle=="EFF_WAVE")(1))));
    band = ((*binTable(where(bintitle=="EFF_BAND")(1))));
    
    w2 = wlen(sort(wlen));
    b2 = band(sort(band));

    // wlenRange overrides crl and crr
    if(!is_void(wlenRange))
    {
        wl = where(w2>wlenRange(1));
        wr = where(
                   w2<wlenRange(2));
        
        if((numberof(wl)==0)||(numberof(wr)==0))
        {
            yocoLogWarning,"Wavelength range out of range!";
            oi = [];

            return 0;
        }
        
        crl = sort(wlen)(wl)(1);
        crr = sort(wlen)(wr)(0);
        if(crr<crl)
        {
            cr2 = crl;
            crl = crr;
            crr = cr2;
        }
    }

    dejV2 = dejV = dejT3 = 0;
    for(iTab=1;iTab<=nTab;iTab++)
    {
        // Read data table
        if(!quiet)
            yocoLogInfo, "Reading " + binTablesNames(dataTables(iTab)) + 
                " table " + pr1(dataTables(iTab));
    
        cfitsio_goto_hdu, fh, dataTables(iTab);
        binTable = cfitsio_read_bintable(fh, bintitle);
        
        oi.hdr.INSNAME = insNames(dataTables(iTab));
                
        // Get common columns from all OIVIS OIVIS2 and OIT3
        wti = where(bintitle =="TARGET_ID");
        if(numberof(wti)!=0)
        {
            TARGET_ID             =  *binTable(wti(1));
            
            // Get OI_TARGET corresponding TARGET_ID
            oiTarg = where(oi.hdr.TARGET_ID == TARGET_ID(1))(1);
            
            // Get other columns
            oi(iTab).hdr.INT_TIME = (*binTable(where(bintitle =="INT_TIME")(1)))(1);
        }
        else
            yocoLogWarning,"No Target ID, this is not an oifits compliant table!";
        
        // Data table is OI VIS2
        if(binTablesNames(dataTables(iTab))=="OI_VIS2")
        {
            if(!dejV2)
            {
                dejV2 = 1;
                oi(iTab).TIME     =     binTable(where(bintitle =="TIME")(1));
                oi(iTab).MJD      =     binTable(where(bintitle =="MJD")(1));
                oi(iTab).STA_INDEX=     binTable(where(bintitle =="STA_INDEX")(1));
                oi(iTab).FLAG     = &(*(binTable(where(bintitle =="FLAG")(1)))=="T");
                oi(iTab).VIS2DATA = &((*binTable(where(bintitle =="VIS2DATA")(1)))(crl:crr,));
                oi(iTab).VIS2ERR  = &((*binTable(where(bintitle =="VIS2ERR")(1)))(crl:crr,));
                oi(iTab).UCOORD   =     binTable(where(bintitle =="UCOORD")(1));
                oi(iTab).VCOORD   =     binTable(where(bintitle =="VCOORD")(1));
            }
            // Treat the case there are several OI_VIS2 tables for 1 OI_WAVELENGTH table
            else
            {
                oi(iTab).TIME      = &grow(*(oi(iTab).TIME),
                                           *binTable(where(bintitle  =="TIME")(1)));
                oi(iTab).MJD       = &grow(*(oi(iTab).MJD),
                                           *binTable(where(bintitle  =="MJD")(1)));
                oi(iTab).STA_INDEX = &grow(*(oi(iTab).STA_INDEX),
                                           *binTable(where(bintitle  =="STA_INDEX")(1)));
                oi(iTab).FLAG      = &grow(*(oi(iTab).FLAG),
                                           *(binTable(where(bintitle =="FLAG")(1)))=="T");
                oi(iTab).VIS2DATA  = &grow(*(oi(iTab).VIS2DATA),
                                           (*binTable(where(bintitle =="VIS2DATA")(1)))(crl:crr,));
                oi(iTab).VIS2ERR   = &grow(*(oi(iTab).VIS2ERR),
                                           (*binTable(where(bintitle =="VIS2ERR")(1)))(crl:crr,));
                oi(iTab).UCOORD    = &grow(*(oi(iTab).UCOORD),
                                           *binTable(where(bintitle  =="UCOORD")(1)));
                oi(iTab).VCOORD    = &grow(*(oi(iTab).VCOORD),
                                           *binTable(where(bintitle  =="VCOORD")(1)));
            }
        }
        
        // Data table is OI VIS
        else if(binTablesNames(dataTables(iTab))=="OI_VIS")
        {
            if(!dejV)
            {
                dejV = 1;
                oi(iTab).TIME      =     binTable(where(bintitle =="TIME")(1));
                oi(iTab).MJD       =     binTable(where(bintitle =="MJD")(1));
                oi(iTab).STA_INDEX =     binTable(where(bintitle =="STA_INDEX")(1));
                oi(iTab).FLAG      = &(*(binTable(where(bintitle =="FLAG")(1)))=="T");
                oi(iTab).VISAMP    = &((*binTable(where(bintitle =="VISAMP")(1)))(crl:crr,));
                oi(iTab).VISAMPERR = &((*binTable(where(bintitle =="VISAMPERR")(1)))(crl:crr,));
                oi(iTab).VISPHI    = &(pi/180*(*binTable(where(bintitle  =="VISPHI")(1)))(crl:crr,));
                oi(iTab).VISPHIERR = &(pi/180*(*binTable(where(bintitle  =="VISPHIERR")(1)))(crl:crr,));
                oi(iTab).UCOORD    = binTable(where(bintitle  =="UCOORD")(1));
                oi(iTab).VCOORD    = binTable(where(bintitle  =="VCOORD")(1));
            }
            // Treat the case there are several OI_VIS2 tables for 1 OI_WAVELENGTH table
            else
            {
                oi(iTab).TIME      = &grow(*(oi(iTab).TIME),
                                           *binTable(where(bintitle  =="TIME")(1)));
                oi(iTab).MJD       = &grow(*(oi(iTab).MJD),
                                           *binTable(where(bintitle  =="MJD")(1)));
                oi(iTab).STA_INDEX = &grow(*(oi(iTab).STA_INDEX),
                                           *binTable(where(bintitle  =="STA_INDEX")(1)));
                oi(iTab).FLAG      = &grow(*(oi(iTab).FLAG),
                                           *(binTable(where(bintitle =="FLAG")(1)))=="T");
                oi(iTab).VISAMP  = &grow(*(oi(iTab).VISAMP),
                                         (*binTable(where(bintitle =="VISAMP")(1)))(crl:crr,));
                oi(iTab).VISAMPERR   = &grow(*(oi(iTab).VISAMPERR),
                                             (*binTable(where(bintitle =="VISAMPERR")(1)))(crl:crr,));
                oi(iTab).VISPHI  = &grow(*(oi(iTab).VISPHI),
                                         (pi/180*(*binTable(where(bintitle =="VISPHI")(1)))(crl:crr,)));
                oi(iTab).VISPHIERR   = &grow(*(oi(iTab).VISPHIERR),
                                             (pi/180*(*binTable(where(bintitle =="VISPHIERR")(1)))(crl:crr,)));
                oi(iTab).UCOORD    = &grow(*(oi(iTab).UCOORD),
                                           *binTable(where(bintitle  =="UCOORD")(1)));
                oi(iTab).VCOORD    = &grow(*(oi(iTab).VCOORD),
                                           *binTable(where(bintitle  =="VCOORD")(1)));
            }
        }
        // Data table is AMPL SPEC
        else if(binTablesNames(dataTables(iTab))=="AMPL_SPEC")
        {
            if(!dejS)
            {
                dejS = 1;
                oi(iTab).TIME      =     binTable(where(bintitle =="TIME")(1));
                oi(iTab).MJD       =     binTable(where(bintitle =="MJD")(1));
                oi(iTab).STA_INDEX =     binTable(where(bintitle =="STA_INDEX")(1));
                oi(iTab).FLAG      = &(*(binTable(where(bintitle =="FLAG")(1)))=="T");
                oi(iTab).spec    = &((*binTable(where(bintitle =="SPECTRUM")(1)))(crl:crr,));
                oi(iTab).specErr = &((*binTable(where(bintitle =="SPECTRUMERR")(1)))(crl:crr,));
                // Treat the case there are several OI_VIS2 tables for 1 OI_WAVELENGTH table
            }
            else
            {
                oi(iTab).TIME      = &grow(*(oi(iTab).TIME),
                                           *binTable(where(bintitle  =="TIME")(1)));
                oi(iTab).MJD       = &grow(*(oi(iTab).MJD),
                                           *binTable(where(bintitle  =="MJD")(1)));
                oi(iTab).STA_INDEX = &grow(*(oi(iTab).STA_INDEX),
                                           *binTable(where(bintitle  =="STA_INDEX")(1)));
                oi(iTab).FLAG      = &grow(*(oi(iTab).FLAG),
                                           *(binTable(where(bintitle =="FLAG")(1)))=="T");
                oi(iTab).spec  = &grow(*(oi(iTab).spec),
                                       (*binTable(where(bintitle =="SPECTRUM")(1)))(crl:crr,));
                oi(iTab).specErr   = &grow(*(oi(iTab).specErr),
                                           (*binTable(where(bintitle =="SPECTRUMERR")(1)))(crl:crr,));
            }
        }
        
        // Data table is AMBER SPECTRUM
        else if(binTablesNames(dataTables(iTab))=="AMBER_SPECTRUM")
        {
            write,"AMBER spectrum is being loaded...";
                
            if(!dejS)
            {
                dejS = 1;
                spec    = transpose(*binTable(where(bintitle =="SPECTRUM")(1)))(crl:crr,sum);
                specErr = transpose(*binTable(where(bintitle =="SPECTRUM_ERROR")(1)))(crl:crr,sum);
                oi(iTab).spec    = &(spec);
                oi(iTab).specErr = &(specErr);
                // Treat the case there are several OI_VIS2 tables for 1 OI_WAVELENGTH table
            }
            else
            {
                spec    = transpose(*binTable(where(bintitle == "SPECTRUM")(1)))(crl:crr,sum);
                specErr = transpose(*binTable(where(bintitle == "SPECTRUM_ERROR")(1)))(crl:crr,sum);
                oi(iTab).spec      = &grow(*(oi(iTab).spec),spec);
                oi(iTab).specErr   = &grow(*(oi(iTab).specErr),specErr);
            }
        }
        
        // Data table is OI T3
        else if(binTablesNames(dataTables(iTab))=="OI_T3")
        {
            if(!dejT3)
            {
                dejT3 = 1;
                oi(iTab).TIME3      =    binTable(where(bintitle   =="TIME")(1));
                oi(iTab).MJD3       =     binTable(where(bintitle  =="MJD")(1));
                oi(iTab).STA_INDEX3 =     binTable(where(bintitle  =="STA_INDEX")(1));
                oi(iTab).FLAG3      = &(*(binTable(where(bintitle  =="FLAG")(1)))=="T");
                oi(iTab).T3AMP    =    &((*binTable(where(bintitle  =="T3AMP")(1)))(crl:crr,));
                oi(iTab).T3AMPERR =    &((*binTable(where(bintitle  =="T3AMPERR")(1)))(crl:crr,));
                oi(iTab).T3PHI    = &(pi/180*(*binTable(where(bintitle  =="T3PHI")(1)))(crl:crr,));
                oi(iTab).T3PHIERR = &(pi/180*(*binTable(where(bintitle  =="T3PHIERR")(1)))(crl:crr,));
                oi(iTab).U1COORD  = binTable(where(bintitle  =="U1COORD")(1));
                oi(iTab).V1COORD  = binTable(where(bintitle  =="V1COORD")(1));
                oi(iTab).U2COORD  = binTable(where(bintitle  =="U2COORD")(1));
                oi(iTab).V2COORD  = binTable(where(bintitle  =="V2COORD")(1));
            }
            // Treat the case there are several OI_VIS2 tables for 1 OI_WAVELENGTH table
            else
            {
                oi(iTab).TIME3      = &grow(*(oi(iTab).TIME3),
                                            *binTable(where(bintitle  =="TIME")(1)));
                oi(iTab).MJD       = &grow(*(oi(iTab).MJD),
                                           *binTable(where(bintitle  =="MJD")(1)));
                oi(iTab).STA_INDEX3 = &grow(*(oi(iTab).STA_INDEX3),
                                            *binTable(where(bintitle  =="STA_INDEX")(1)));
                oi(iTab).FLAG3      = &grow(*(oi(iTab).FLAG3),
                                            *(binTable(where(bintitle =="FLAG")(1)))=="T");
                oi(iTab).T3AMP     = &grow(*(oi(iTab).T3AMP),
                                           (*binTable(where(bintitle =="T3AMP")(1)))(crl:crr,));
                oi(iTab).T3AMPERR  = &grow(*(oi(iTab).T3AMPERR),
                                           (*binTable(where(bintitle =="T3AMPERR")(1)))(crl:crr,));
                oi(iTab).T3PHI     = &grow(*(oi(iTab).T3PHI),
                                           pi/180*(*binTable(where(bintitle =="T3PHI")(1)))(crl:crr,));
                oi(iTab).T3PHIERR  = &grow(*(oi(iTab).T3PHIERR),
                                           pi/180*(*binTable(where(bintitle =="T3PHIERR")(1)))(crl:crr,));
                oi(iTab).U1COORD   = &grow(*(oi(iTab).U1COORD),
                                           *binTable(where(bintitle  =="U1COORD")(1)));
                oi(iTab).V1COORD   = &grow(*(oi(iTab).V1COORD),
                                           *binTable(where(bintitle  =="V1COORD")(1)));
                oi(iTab).U2COORD   = &grow(*(oi(iTab).U2COORD),
                                           *binTable(where(bintitle  =="U2COORD")(1)));
                oi(iTab).V2COORD   = &grow(*(oi(iTab).V2COORD),
                                           *binTable(where(bintitle  =="V2COORD")(1)));
            }
        }
        else
            error;
        
        //////////////////////////////////////////////
        // Read OI_WAVELENGTH table (mandatory)
        //////////////////////////////////////////////
        // Find corresponding oi wavelength table
        
        // FIXME: patch for crappy MIDI data
        // If there is only one OI_WAVELENGTH, then INSNAME
        // is not necessary to find data
        if(numberof(where(binTablesNames=="OI_WAVELENGTH"))==1)
            correspondingIns = binTablesNames=="OI_WAVELENGTH";
        else
            correspondingIns = insNames==insNames(dataTables(iTab));
            
        theOiWave = where(
                          (binTablesNames=="OI_WAVELENGTH")&
                          (correspondingIns)
                          )(1);
        
        cfitsio_goto_hdu,fh,theOiWave;
        binTable = cfitsio_read_bintable(fh, bintitle);
        
        wlen = ((*binTable(where(bintitle=="EFF_WAVE")(1))));
        band = ((*binTable(where(bintitle=="EFF_BAND")(1))));

        // Fill values in the structure
        oi(iTab).EFF_WAVE    = &(wlen(crl:crr));
        oi(iTab).EFF_BAND    = &(band(crl:crr));

        // FIXME: Filter frequencies TBD
    }
    
    cfitsio_close,fh;                            // close the file

    return oi;
}

/***************************************************************************/

func amplShowUV(oi, inputOiFile=, win=, kill=, freq=, marker=, plotNE=, limit=, titles=, sys=, color=)
    /* DOCUMENT amplShowUV(oi, inputOiFile=, win=, kill=, freq=, marker=, plotNE=, limit=, titles=, sys=, color=)

       DESCRIPTION

       PARAMETERS
       - oi         : 
       - inputOiFile: 
       - win        : 
       - kill       : 
       - freq       : 
       - marker     : 
       - plotNE     : 
       - limit      : 
       - titles     : 
       - sys        : 
       - color      : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(!is_void(color))
        col = color;
    
    if(is_void(win))
        win=1;
    
    if(is_void(freq))
        freq=0;
    
    if(is_void(plotNE))
        plotNE=1;
    
    if(kill==1)
        winkill,win;
    
    if(is_void(oi))
    {
        /* If the function showRawData, is called without argument,
           a file browser
           * popups to choose the correct file */
        message = "OIDATA FILE\nEnter the OIDATA_RAW or OIDATA_AVG file to show";
        if (amdlibFileChooser(message, inputOiFile, 2) == 0)
        {
            yocoError, "'inputOiFile' not specified correctly.";
            return 0;
        }

        /* Load the data */
        status = amplLoadOiDatas(oi,inOiFile=inputOiFile);
    }

    if(!is_void(win))
        window,win;
    
    if(!is_void(sys))
        plsys,sys;
    
    nFil = numberof(oi);
    R=0;
    
    WLEN = [];
    for(kf=1;kf<=nFil;kf++)
    {
        grow,WLEN,*oi(kf).EFF_WAVE;
    }
    WLEN = yocoStrRemoveMultiple(WLEN);
    WLEN = WLEN(sort(WLEN));
    NBWLEN = numberof(WLEN);
    
    for(kf=1;kf<=nFil;kf++)
    {
        if(strmatch(oi.hdr.ARRNAME(kf),"U"))
            diam = 8;
        else
            diam = 1.8;//*oi(kf).UCOORD;
        
        u = *oi(kf).UCOORD;
        nBas = numberof(u);
        for(kb=1;kb<=nBas;kb++)
        {
            v    = *oi(kf).VCOORD;
            
            r = abs(u,v);
            if(freq)
            {
                l = *oi(kf).EFF_WAVE;
                nbWlen = numberof(l);
                f  = r(,-) / l(-,);
                uf = u(,-) / l(-,);
                vf = v(,-) / l(-,);
            }
            else
                f = r;
            
            if(anyof(r>R))
            {
                R  = max(r);
                Rf = max(f);
            }
            
            p = span(0,2*pi,13);
            x = diam/2.*cos(p);
            y = diam/2.*sin(p);

            if(freq==1)
            {
                for(kl=1;kl<=nbWlen;kl++)
                {
                    wwlen = where(l(kl)==WLEN)(1);
                    col = get_color(1.0, double(NBWLEN-wwlen)/NBWLEN*1.5*pi);
                    plg,vf(kb,kl)+y/l(kl),uf(kb,kl)+x/l(kl),color=col;
                    plg,-vf(kb,kl)+y/l(kl),-uf(kb,kl)+x/l(kl),color=col;
                }
            }
            else if(freq==2)
            {
                for(kl=1;kl<=nbWlen;kl++)
                {
                    as = yocoAstroSI.mas*1000;
                    wwlen = where(l(kl)==WLEN)(1);
                    col = get_color(1.0, double(NBWLEN-wwlen)/NBWLEN*1.5*pi);
                    plg,as*(vf(kb,kl)+y/l(kl)), as*(uf(kb,kl)+x/l(kl)),color=col;
                    plg,as*(-vf(kb,kl)+y/l(kl)),as*(-uf(kb,kl)+x/l(kl)),color=col;
                }
            }
            else if(freq==0)
            {
                plg, v(kb)+y, u(kb)+x,color=col;
                plg,-v(kb)+y,-u(kb)+x,color=col;
            }
            else if(!is_void(marker))
            {
                plg,v(kb),u(kb),type="none",marker=marker,color=color;
                plg,-v(kb),-u(kb),type="none",marker=marker,color=color;
            }
            else
                error;
        }
    }

    plg,0,0,type="none",marker='\2';
    
    if(freq==1)
    {
        if(!is_void(limit))
            Rf = limit;
        limits,Rf,-Rf,-Rf,Rf,square=1;
        if(plotNE)
            yocoAstroPlotNorthEast,-Rf*0.9,-Rf*0.9,Rf/5;
        if(titles)
            xytitles,"U (B/!l)","V (B/!l)";
    }
    else if(freq==2)
    {
        if(!is_void(limit))
            Rf = limit;
        limits,as*Rf,as*-Rf,as*-Rf,as*Rf,square=1;
        if(plotNE)
            yocoAstroPlotNorthEast,as*-Rf*0.9,as*-Rf*0.9,as*Rf/5;
        if(titles)
            xytitles,"U (arcsec^-1^)","V (arcsec^-1^)";
    }
    else
    {
        if(!is_void(limit))
            R = limit;
        limits,R,-R,-R,R,square=1;
        if(plotNE)
            yocoAstroPlotNorthEast,-R*0.9,-R*0.9,R/5;
        if(titles)
            xytitles,"U (m)","V (m)";
    }
    if(is_void(inputOiFile))
        inputOiFile = oi.hdr.file(1)
            yocoFileSplitName,inputOiFile,d,f,e;
    hcps,d(1)+f(1)+"_UVplane.ps";

}

/***************************************************************************/

func amplShowOiData(void, inputOiFile=, projAngle=, cutAngle=, cutOpen=, color=, model=, width=, plotVis=, plotClos=, plotDPhi=, plotDVis=, type=, marker=, msize=, kill=, win=, clear=, wrap=, phaseDeg=, noDiff=, sizebar=, avgWlen=, limVis=, limClos=, limDPhi=, limDVis=, wlenRange=, logVis=)
    /* DOCUMENT amplShowOiData(inputOiFile=, projAngle=, cutAngle=, cutOpen=, color=, model=, width=, plotVis=, plotClos=, plotDPhi=, plotDVis=, type=, marker=, msize=, kill=, win=, clear=, wrap=, phaseDeg=, noDiff=, sizebar=, avgWlen=, limVis=, limClos=, limDPhi=, limDVis=, wlenRange=, logVis=)
   
       DESCRIPTION
       Shows the main contents of an OI data file. If no file is specified, a file
       browser window will prompt the user to choose one OI-fits file.

       In a first window are displayed:
       - the array configuration and uv tracks
       - the flux in all channels as a function of wavelength
       - the squared visibility for all baselines as a function of wavelength
       - the phase and closure phase as a function of wavelength

       In a second window are displayed histogrammes for the flux, the square
       visibility, the piston and the closure phase, as well as the sequence of the
       estimated piston as a function of time.
      
       PARAMETERS
       - inputOiFile: The input OI data file to plot. Can be either RAW, AVG or
       CAL OI data file. When a RAW data file is loaded, the
       script try to guess the name of an associated SEL(ection)
       file, and if it exists, load it to diplay the effect of
       selection (in purple)
       - projAngle  : 
       - cutAngle   : 
       - cutOpen    : 
       - color      : 
       - model      : 
       - width      : 
       - plotVis    : 
       - plotClos   : 
       - plotDPhi   : 
       - plotDVis   : 
       - type       : 
       - marker     : 
       - msize      : 
       - kill       : tells wether to kill graphical windows or not. This
       - win        : 
       - clear      : 
       - wrap       : 
       - phaseDeg   : 
       - noDiff     : 
       - sizebar    : 
       - avgWlen    : 
       - limVis     : 
       - limClos    : 
       - limDPhi    : 
       - limDVis    : 
       - wlenRange  : 
       - logVis     : 
       option is mainly useful for amdlibShowAllOiData.

       EXAMPLE
       amdlibShowOiData(inputOiFile="AMBER.2005-02-25T09:24:22.488_OIDATA_RAW.fits")
  
       RETURN VALUE
       Saves the plots hardcopies in ps files
  
       SEE ALSO amdlibShowAllOiData, amdlibShowRawData, amdlibShowP2vm
    */
{
    local wlen, bandWidth, time, fringeSNR, sqVis, sqVisErr,
        uvCoord, bandFlag, pistonOPDArray, sigmaPistonArray,
        diffVisAmp, diffVisAmpErr, diffVisPhase, diffVisPhaseErr,
        vis3Amp, vis3AmpErr, vis3Phase, vis3PhaseErr, cpxVis,
        cpxVisErr, visCovRI, uv1Coord, uv2Coord, fluxSumPiPj,
        fluxRatPiPj, PiMultPj, spectrum, spectrumErr, telFlux;
    

    if (is_void(click))
    {
        click = 1;
    }

    if (is_void(limVis))
    {
        limVis = [-0.1,1.1]
            }
    
    yocoLogTrace, "amdlibShowOiData()";
    if (!is_void(void))
    {
        yocoLogWarning,"void should be void !";
    }

    // legend offset
    legendOffset = [0.02,0.02];

    /* All the Yorick opened windows are killed */
    if (is_void(kill))
        kill=1;
    
    if (kill==1)
        yocoGuiWinKill;

    /* Read the default window size */
    local graphicsDpi;
    if (amdlibCheckGraphicsDpi(graphicsDpi) == 0)
    {
        yocoError, "Check function's parameters";
        return 0;
    }

    /* If the function showRawData, is called without argument, a file browser
     * popups to choose the correct file */
    message = "OIDATA FILE\nEnter the OIDATA_RAW or OIDATA_AVG file to show";
    if (amdlibFileChooser(message, inputOiFile, 2) == 0)
    {
        yocoError, "'inputOiFile' not specified correctly.";
        return 0;
    }

    /* Verbose output */
    //     yocoLogInfo,"Executing amdlibShowOiData on file:",inputOiFile;

    /* Init several variables as null */
    wlen = bandWidth = time = fringeSNR = sqVis = sqVisErr =
        uvCoord = bandFlag = pistonOPDArray = sigmaPistonArray =
        diffVisAmp = diffVisAmpErr = diffVisPhase = diffVisPhaseErr =
        vis3Amp = vis3AmpErr = vis3Phase = vis3PhaseErr = cpxVis =
        cpxVisErr = visCovRI = uv1Coord = uv2Coord = fluxSumPiPj =
        fluxRatPiPj = PiMultPj = spectrum = spectrumErr  = [];
    
    /* Load the data */
    status = amplLoadOiDatas(oi,inOiFile=inputOiFile, wlenRange=wlenRange);

    yocoFileSplitName,inputOiFile,d,f,e;

    require,"fitOmatic.i";
    // plotDataWlen(oi);
    plotDataSpFreq, oi,projAngle=projAngle,cutAngle=cutAngle,cutOpen=cutOpen,
        color=color,model=model,
        width=width,plotVis=plotVis,plotClos=plotClos,plotDPhi=plotDPhi,
        plotDVis=plotDVis,type=type,marker=marker,msize=msize,kill=kill,win=win,
        clear=clear,wrap=wrap,phaseDeg=phaseDeg,noDiff=noDiff,sizebar=sizebar,
        avgWlen=avgWlen,limVis=limVis,limClos=limClos,limDPhi=limDPhi,
        limDVis=limDVis,logVis=logVis;
    
    hcps, d(1)+f(1)+"_spFreq.ps";


    plotDataWlen(oi, plotVis=2, plotClos=1, plotDPhi=1, win=2, kill=1);
        
    plotUV(oi);
    plotUVfreq(oi);
    
}

/***************************************************************************/

// This version supersedes the one of cfitsioplugin
func amplCfitsio_add_bintable(&fh, pcol, titles, units, extname, append=)
    /* DOCUMENT amplCfitsio_add_bintable(&fh, pcol, titles, units, extname, append=)

       DESCRIPTION:
       Creat a new BINARY_TBL HDU  and write pcol as a
       binary table.
       'pcol' should be an 1-dim array of pointer (one per colum of
       the bintable). 'titles' is the array of columns titles.
       'units' is the array of columns units.
       'extname' is the name of the new HDU created.

       SEE ALSO: cfitsio_read_bintable, cfitsio_add_bintable,
       cfitsio_delete_rowlist

    */
{
    local tform,tmp;
    tform=[];

    /* check */
    if(typeof(pcol)!="pointer")
    {
        error,"pcol should be array of pointer";
    }
    
    /* number of cols */
    ncols = numberof(pcol);

    /* Loop on col to define the TFORM */
    for(i=1;i<=ncols;i++)
    {
        /* void string will kill cfitsio */
        if(typeof(*pcol(i)) =="string" &&
           is_array((tmp=where( *pcol(i)==string(0) |
                                *pcol(i)==string("")))))
        {
            (*pcol(i))(tmp)=" ";
        }
        if(numberof(*pcol(i))==1)
            pcol(i) = &([*pcol(i)]);

        grow,tform,cfitsio_set_tform(*pcol(i));
    }

    if(append==1)
        // Just append the binary table, do not create a new one
    {
        // Got to the right HDU
        cfitsio_goto_hdu,fh,extname;
        nrowstoadd = dimsof(*pcol(1))(0);
        nrows      = cfitsio_get_num_rows(fh);
        cfitsio_insert_rows, fh, nrows, nrowstoadd;
        /* Write the data */
        for(i=1;i<=ncols;i++)
        {
            amplCfitsio_write_column, fh, i, *pcol(i), firstRow=nrows+1;
        }
    }
    else
    {    
        /* Create the table with all the columns */
        cfitsio_create_tbl, fh, BINARY_TBL, 0, titles, tform, units, extname;
        
        /* Write the data */
        for(i=1;i<=ncols;i++)
        {
            toto = *pcol(i);
            if(numberof(toto)==1)
                toto = toto(1);
            amplCfitsio_write_column, fh, i, toto;
        }
    }
    
    return fh;
}

/***************************************************************************/

func amplCfitsio_write_column(&fh, colnum, data, firstRow=, firstElem=)
    /* DOCUMENT amplCfitsio_write_column(&fh, colnum, data, firstRow=, firstElem=)

       DESCRIPTION
       Write a single column in an existing BINTABLE.

       SEE ALSO cfitsio_add_column, cfitsio_read_column,
       cfitsio_read_multicolumn,
       cfitsio_read_bintable
       cfitsio_delete_col
    */
{
    /* Write the TDIM. */
    if(dimsof(data)(1)>1)
    {
        cfitsio_write_tdim,fh, colnum, dimsof(data(..,1));
    }

    /* Write the data */
    amplCfitsio_write_col, fh, colnum, data, firstRow=firstRow, firstElem=firstElem;
    
    return fh;
}

/***************************************************************************/

func amplCfitsio_write_col(&fh, colnum, arrayc, firstRow=, firstElem=)
    /* DOCUMENT amplCfitsio_write_col(&fh, colnum, arrayc, firstRow=, firstElem=)

       Write elements into an ASCII or binary table column (in the CDU).
   
       The following functions are higher level and should be
       prefered to this low-level wrapper:
   
       SEE ALSO cfitsio_add_bintable, cfitsio_read_column,
       cfitsio_read_multicolumn,
       cfitsio_read_bintable,
       cfitsio_delete_rowlist
    */
{
    if(is_void(firstRow))
        firstRow = 1;

    if(is_void(firstElem))
        firstElem = 1;
    
    /* memory allocation */
    datatype  = cfitsio_TTYPE_of(arrayc);
    nelements = long(numberof(arrayc));

    status = int(0);
    // Write column
    //FIXME: uncomment following line if the script crashes below
    //status2 = __ffpcl( fh, datatype, colnum, long(firstRow), long(firstElem), nelements, &arrayc, status);
    //FIXME: comment following line if the script crashes here
    status2 = __ffpcl( fh, datatype, colnum, long(firstRow), long(firstElem), nelements, &arrayc);
    if(status)
    {
        error, cfitsio_get_errstatus(status);
    }
}

/*************************************************************/
/** Transfer function stuff **********************************/
/*************************************************************/

func amplShowAll(inputDir=, plotSci=, DIT=)
    /* DOCUMENT amplShowAll(inputDir=, plotSci=, DIT=)

       DESCRIPTION

       PARAMETERS
       - inputDir: 
       - plotSci : 
       - DIT     : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    wkll;
    
    /* Select directory containing files to be reduced */
    message = "Choose the OIDATA directory you want to reduce";    
    if ( !amdlibFileChooser(message, inputDir, 3)  ||
         !amdlibCheckDir(inputDir, 0) )
    {
        yocoError,"'inputDir' not specified correctly.";
        return 0;
    }

    /* Verbose output */
    if(!quiet)
        yocoLogInfo, "Plotting the transfer function for directory"+
            " (overwrite="+pr1(overwrite)+"):", inputDir;

    /* Read the log of the inputDir */
    if ( !amdlibReadLogFromDir(inputDir, tab, titles) )
    {
        yocoError, "Could not read log file of 'inputDir'";
        return 0;
    }

    /* Recovering and building the necessary informations
       to associate P2VM, OBJECT, DARK and SKY */
    tabFiles = amdlibLogGet(tab, titles, "fileName");
    tabIds   = amdlibLogGet(tab, titles, "P2VM_ID");
    insMode  = amdlibLogGet(tab, titles, "insMode");
    ndit     = yocoStr2Long( amdlibLogGet(tab, titles, "NDIT") );
    dit      = yocoStr2Double( amdlibLogGet(tab, titles, "DIT") );
    tplNr    = yocoStr2Long( amdlibLogGet(tab, titles, "Template_nr") );
    obNr     = yocoStr2Long( amdlibLogGet(tab, titles, "OB_nr") );
    obsCtg   = amdlibLogGet(tab, titles, "obs_catg") ;
    obsTyp   = amdlibLogGet(tab, titles, "obs_type") ;
    proCtg   = amdlibLogGet(tab, titles, "pro_catg") ;
    dates    = amdlibLogGetDate(tab, titles);
    //_amdlibGetKwdVals_new, tabFiles,["HIERARCH AMPL DATATYPE"],valuesTab,gui=plot;

    if(is_void(DIT))
        DIT = min(dit(where(dit!=0)));
    
    inOiFiles = tabFiles(where((strmatch(proCtg, "AVERAGED")&
                                dit==DIT)));
    
    amplLoadOiDatas(oi, inOiFile=inputDir+inOiFiles);

    yocoNmCreate,1,1,3,landscape=0, height=800,width=600,
        V=[0.12,0.73,0.1,0.9];

    yocoFileReadAscii, "~/amdlibCalibDatabase.txt", data;
    
    T = V2 = V2E = ST = SV2 = SV2E = V2i = SV2i =
        NAME = SNAME = DIAM = SDIAM = [];
    for(k=1;k<=numberof(oi);k++)
    {
        v2 = *oi(k).VIS2DATA;
        if((!is_void(v2)))
        {
            t = *oi(k).MJD;
            n = oi(k).hdr.TARGET;
            v2e = *oi(k).VIS2ERR;
            U = *oi(k).UCOORD;
            V = *oi(k).VCOORD;
            w = *oi(k).EFF_WAVE;
            
            R = abs(U,V);
            x = pi * R / w;
            
            if(oi(k).hdr.catg=="CALIB")
            {
                grow,T,[t];
                grow,V2,[v2];
                grow,V2E,[v2e];
                grow,NAME,n;
            
                if(anyof(strmatch(data(1,), n)))
                {
                    diam = yocoStr2Double(data(5,where(strmatch(data(1,), n))));
                    if(diam==-1)
                        diam = 0.0;
                }
                else
                    diam = 0.0;
                
                grow,DIAM,diam;
                
                yy = 2*x * diam * mas2rad;
                Vinst = 2 * bessj1(yy) / (yy + (yy==0)) + (yy==0);
                grow,V2i, [Vinst^2];
            }
            else if(oi(k).hdr.catg=="SCIENCE")
            {
                grow,ST,[t];
                grow,SV2,[v2];
                grow,SV2E,[v2e];
                grow,SNAME,n;

                grow,SV2i,[array(1.0,dimsof(v2))];
                
                if(anyof(strmatch(data(1,), n)))
                {
                    diam = yocoStr2Double(data(5,where(strmatch(data(1,), n))));
                    grow,SDIAM,diam;
                }
                else
                    grow,SDIAM,0.0;
            }
        }
    }

    if(!is_void(T))
    {
        T = T*24.;
        V = yocoMathSignedSqrt(V2);
        Vi = yocoMathSignedSqrt(V2i);
        TR = yocoMathSignedSqrt(V2/V2i);
    }
    if(!is_void(ST))
    {
        ST = ST*24.;
        SV = yocoMathSignedSqrt(SV2);
    }

    if(plotSci)
    {
        TT = grow(T, ST);
        AV = grow(V, SV);
        NNAME = grow(NAME,SNAME);
        ADIAM = grow(DIAM,SDIAM);
    }
    else
    {
        AV = V;
        TT = T;
        NNAME = NAME;
        ADIAM = DIAM;
    }
    
    T0 = min(TT);
    T1 = max(TT);
            
    intv = abs(T0-T1);

    for(l=1;l<=3;l++)
    {
        plsys,l;
        //yocoPlotWithErrBars,v2(l),v2e(l),t(l),sizebarX=0.1;
        if(!is_void(T))
            plg,V(l,),T(l,)-T0,type="none",marker='\2',color=[192,192,192];
        if(!is_void(T))
            plg,TR(l,),T(l,)-T0,type="none",marker='\2',color="black"
        
                if(!is_void(ST)&&plotSci)
                    plg,SV(l,),ST(l,)-T0,type="none",marker='\2',color="red";
        limits;

        amdlibOverPlotLabels, NNAME+"\n"+swrite(ADIAM,format="(%1.1f mas)"),
            TT(l,)-T0,
            justify="LH",yText=median(median(AV))*3.0, plotName=(l==1);
        
        limits,-0.05*intv,1.05*intv,-min(1.,median(median(AV))*10.0)*0.05,min(1.,median(median(AV))*6.0);
    }

    plsys,1;
    pltitle,yocoStrReplace(yocoStrrtok(inputDir,"/")(2));
    yocoNmXytitles,"Time (hrs)", "Visibility",[0.02,0.02];
    hcps,inputDir+yocoStrrtok(inputDir,"/")(2)+"_Transfer_Function.ps";
}

/***************************************************************************/

func amplLoadAllRawDatas(inputDir=)
    /* DOCUMENT amplLoadAllRawDatas(inputDir=)

       DESCRIPTION

       PARAMETERS
       - inputDir: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    wkll;
    
    /* Select directory containing files to be reduced */
    message = "Choose the OIDATA directory you want to reduce";    
    if ( !amdlibFileChooser(message, inputDir, 3)  ||
         !amdlibCheckDir(inputDir, 0) )
    {
        yocoError,"'inputDir' not specified correctly.";
        return 0;
    }

    /* Read the log of the inputDir */
    if ( !amdlibReadLogFromDir(inputDir, tab, titles) )
    {
        yocoError, "Could not read log file of 'inputDir'";
        return 0;
    }

    /* Recovering and building the necessary informations
       to associate P2VM, OBJECT, DARK and SKY */
    tabFiles = amdlibLogGet(tab, titles, "fileName");
    tabIds   = amdlibLogGet(tab, titles, "P2VM_ID");
    insMode  = amdlibLogGet(tab, titles, "insMode");
    specConf = strtok(insMode,"_")(2,);
    detSetup = amdlibLogGetWindowing(tab, titles) + amdlibLogGet(tab, titles, "DIT");
    ndit     = yocoStr2Long( amdlibLogGet(tab, titles, "NDIT") );
    dit      = yocoStr2Double( amdlibLogGet(tab, titles, "DIT") );
    dates    = amdlibLogGetDate(tab, titles);
    tplNr    = yocoStr2Long( amdlibLogGet(tab, titles, "Template_nr") );
    obNr     = yocoStr2Long( amdlibLogGet(tab, titles, "OB_nr") );
    obsCtg   = amdlibLogGet(tab, titles, "obs_catg") ;
    obsTyp   = amdlibLogGet(tab, titles, "obs_type") ;
    proCtg   = amdlibLogGet(tab, titles, "pro_catg") ;
    dates    = amdlibLogGetDate(tab, titles);
    //_amdlibGetKwdVals_new, tabFiles,["HIERARCH AMPL DATATYPE"],valuesTab,gui=plot;

    /* Look for SKY, DARKs, both hot and cold ones */
    isHotDark  = ( ( amdlibLogGet(tab, titles, "obs_type") == "DARK")  ) *
        ( amdlibLogGet(tab, titles, "INS-GRIS1-NAME")      != "DARK");
    isColdDark = ( amdlibLogGet(tab, titles, "obs_type")   == "DARK") *
        ( amdlibLogGet(tab, titles, "INS-GRIS1-NAME")      == "DARK");
    isSky    =  amdlibLogGet(tab, titles, "obs_type")      == "SKY";

    /* Look for OBJECT files, but remove already processed files */
    isObject = ( amdlibLogGet(tab, titles, "obs_type") == "OBJECT") *
        ( amdlibLogGet(tab, titles, "pro_catg") == "-") ;
    

    if(is_void(DIT))
        DIT = min(dit(where(dit!=0)));

    whichFile = where((strmatch(proCtg, "-")&
                       strmatch(obsCtg , "SCIENCE")&
                       strmatch(obsTyp , "OBJECT")&
                       (dit==DIT)));
    inRawFiles = tabFiles(whichFile);

    if(!quiet)
        yocoLogInfo,inRawFiles;
    
    for(k=1;k<=numberof(inRawFiles);k++)
    {
        bestBias = amdlibFindBestBias(whichFile(k), tabIds, specConf, isHotDark, isColdDark, array(0,dimsof(isSky)), detSetup, ndit, dates);
        
        amplLoadRawData(rawDataStruct,inputRawFile=inRawFiles(k),inputBiasFile=tabFiles(bestBias),inputBPM=,
                        inputFFM=,specCalShifts=,firstFrame=,nbFrames=,plot=);
    }
    
}
