/***************************************************************************
 *                                                                         *
 *                                AMBER++                                  *
 *               FFT-based data reduction soft for AMBER                   *
 *                                                                         *
 *                      Copyright 2011, F. Millour                         *
 *                            fmillour@oca.eu                              *
 *                                                                         *
 ***************************************************************************
 *
 * This file contains yorick scripts to reduce AMBER datasets using FFTs
 *
 * Please note that this script is distributed under the GPL licence,
 * available at http://www.gnu.org/licenses/gpl.txt
 *
 * Please ACKNOWLEDGE the use of this script for any use in a publication
 * by citing Millour et al. (in prep.)
 *
 * This is a sub-script. Pease consider reading amplAMBER.i for more info.
 *
 * $Id: amplStructures.i 566 2015-02-07 16:27:40Z fmillour $
 * $URL: https://svn.oca.eu/codes/AMBERpl/trunk/yorick/amplStructures.i $
 *
 ***************************************************************************/

yocoLogInfo,"#include,\"amplStructures.i\"";


func amplStructures(&ver)
/* DOCUMENT amplStructures(&ver)

       DESCRIPTION
       Structures handling for ampl

       FUNCTIONS
   - amplStructures: this script

       SEE ALSO amplRawData, amplAMBER
    */
{
    extern MajorVersion;
    ver = yocoStr2Double(pr1(MajorVersion)+"."+
                         strpart(strtok("$Rev: 566 $",":")(2),2:-2));
    if (am_subroutine())
    {
        write, format="script version: %1.3f\n", ver;
        write,"Type \"help, amplAMBER\" to get the package version";
        help, amplStructures;
    }
    return ver;
}

maxNbTels  = 3;
maxNbBases = maxNbTels * (maxNbTels - 1) / 2;

struct amplHeaderStruct
//Structure containing common headers
{
    // General information about original file
    string  file;
    
    // All keywords, can be sometimes redundant with the following keywords
    pointer keyName;
    pointer keyVal;
    pointer keyCom;
    
    // OI_FITS common headers
    double  OI_REVN;
    string  INSNAME;
    string  INS_MODE;
    string  ARRNAME;
    string  DATE_OBS;
    string  FRAME;
    double  ARRAYXYZ(maxNbTels);
    double  MJD;
    double  INT_TIME;

    //-----
    // FIXME: just for compatibility with version 238 of amplData.i
    double  TIME;
    pointer UCOORD;
    pointer VCOORD;
    double  TIME3;
    double  MJD3;
    pointer U1COORD;
    pointer V1COORD;
    pointer U2COORD;
    pointer V2COORD;
    pointer STA_INDEX3;
    //-----
    
    // OI_TARGET table
    string TARGET;
    long   TARGET_ID;
    double RAEP0;
    double DECEP0;
    double EQUINOX;
    double RA_ERR;
    double DEC_ERR;
    double SYSVEL;
    string VELTYP;
    string VELDEF;
    double PMRA;
    double PMDEC;
    double PMRA_ERR;
    double PMDEC_ERR;
    double PARALLAX;
    double PARA_ERR;
    string SPECTYP;

    // OI_ARRAY table
    pointer TEL_NAME;
    pointer STA_NAME;
    pointer STA_INDEX;
    pointer DIAMETER;
    pointer STAXYZ;
    
    // Information about observation
    double  dit;
    long    ndit;
    string  specMode;
    long    hour;    // Hour of observation
    long    minute;  // minute of observation
    double  second;  // second of observation
    double  press;   // ambient pressure
    double  tempOut; // ambient temperature
    double  tempIn;  // ambient temperature in tunnel
    double  watVap;  // ambient relative humidity
    pointer OPL;    // Delay introduced by delay lines
    string  catg;    // Data category
    string  bcd;     // Beam commutation state
    string  tracker; // tracker on/off
    string  obName;  // OB name
    long    obId;       // OB ID
    long    tplNr;      // template number
    long    p2vmId;     // P2VM ID

    pointer AO_STREHL; // strehl ratio
    pointer AO_WFE;    // wavefront error
};

/****************************************************************/

struct amplDataStruct
//Structure containing all relevant data information
{
    string dataType;  // data type, can be any of:
    // "Raw Data", "Power Spectrum", "Cross Spectrum",
    // "Bispectrum", "Bispectrum waterfall",
    // "Power Spectrum Waterfall", "Cross Spectrum Waterfall"
    
    amplHeaderStruct hdr; // File Header
    
    // Useful information for FFT-ized dataset
    double realUpsampleX, realUpsampleY;
    
    // Raw data
    pointer dark;    // (if any) dark line measurement
    pointer phot;    // (if any) photometric measurements
    pointer fringes; // data array containing all the fringe data information
    // (can be raw fringes, FFTs, waterfall, etc. as specified in dataType)

    // Axes data
    pointer X;     // value of X axis
    string  Xlab;  // label of X axis
    string  Xunit; // Unit  of X axis
    pointer Y;     // value of Y axis
    string  Ylab;  // label of Y axis 
    string  Yunit; // Unit  of Y axis
    pointer Z;     // value of Z axis
    string  Zlab;  // label of Z axis 
    string  Zunit; // Unit  of Z axis
    pointer T;     // value of T axis
    string  Tlab;  // label of T axis 
    string  Tunit; // Unit  of T axis
    
    // masks data
    pointer mask;     // mask for work wavelengths
    pointer maskRef;  // mask for reference wavelengths
    pointer maskWlen; // wavelengths to be used for the mask
    pointer maskWlenRef; // wavelengths to be used for the mask
};

/***************************************************************************/

struct amplCalibStruct
//Structure containing all calibration information
{
    long    p2vmId;     // P2VM id (but not quite right though)
    pointer wlen;       // wavelength of the peak frequency
    
    pointer freqPeak;   // Spatial frequency of the peaks as extracted from P2VM
    pointer frgContMat; // Fringe peaks contamination matrix

    pointer kappaMat;   // Kappa (transmission) matrix
    pointer kappaFlux;  // Reference flux of P2vm calibration files

    pointer beamShape;
};

/****************************************************************/

struct amplOiData
// OI DATA structure
{
    // Header
    amplHeaderStruct hdr;
    
    // Relevant additional information
    pointer MJD;
    pointer TIME;
    pointer UCOORD;
    pointer VCOORD;
    pointer STA_INDEX;
    pointer FLAG;
    
    pointer MJD3;
    pointer TIME3;
    pointer U1COORD;
    pointer V1COORD;
    pointer U2COORD;
    pointer V2COORD;
    pointer STA_INDEX3;
    pointer FLAG3;
    
    // Spectrum table
    pointer spec;
    pointer specErr;
    
    // OI_VIS table
    pointer VISAMP;
    pointer VISAMPERR;
    pointer VISPHI;
    pointer VISPHIERR;
    
    // OI_VIS2 table
    pointer VIS2DATA;
    pointer VIS2ERR;
    
    // OI_T3 table
    pointer T3AMP;
    pointer T3AMPERR;
    pointer T3PHI;
    pointer T3PHIERR;

    // OI_WAVELENGTH
    pointer EFF_WAVE;
    pointer EFF_BAND;

    // AMPL_VIS2 
    pointer PHOT
    pointer PHOTERR
    pointer COHFLUX
    pointer COHFERR
    pointer OPD
    pointer RAWCOHFLUX
    pointer RAWCOHFERR
    pointer AO_STREHL
    pointer AO_WFE
    
    // AMPL_T3
    pointer OPD1
    pointer OPD2
    pointer RAWT3PHI
    pointer RAWT3PHIERR
};


/****************************************************************/

struct amplReductionStruct
//Structure defining datasets to be reduced and parameters for reduction
{
    // directory
    string  inputDir;
    
    // Files
    string  bpmFile;
    string  ffmFile;
    
    pointer rawFiles;
    pointer biasFiles;
    pointer noiseFiles;
    
    long    p2vmId;
    string  p2vmFile;
    string  p2vmDir;

    long    overwrite;
    long    overwriteP2VM;
    long    overwriteNoiseFFT;

    // intermediate FFT files for 1D computation
    pointer waterFiles;
    pointer noiseWaterFiles;
    pointer waterFilesNoP;
    
    // intermediate FFT files
    pointer globlWaterFiles;
    pointer chromWaterFiles;
    pointer crossWaterFiles;
    pointer noiseGloblWaterFiles;
    pointer noiseChromWaterFiles;
    
    // other track info
    double  dit;
    long    obId;
    string  detSetupId;
    string  objectName;
    string  obsCatg;
    string  dateObs;
    double  MJD;
    string  trackName;
    
    // FFT-related parameters
    long    resample;
    long    upsampleX;
    long    upsampleY;
    long    apodWidth;
    long    frameLoadBin;
    long    wlenBin;      // pixel per bin for chromatic case
    string  wlenRange;

    // frame selections and bins
    double  timeBin;
    double  cutSNR;
    long    smCoherent;
    //double  cutIntegSNR;
    //double  fracUsed;
    int     trackBinsize;
    pointer attSigmaFact;
    
    // calculation options
    //int     noFFTCalc;
    //int     noFFTNoise;
    long    quickPSD;
    long    noChromatic;
    long    noOiCalc;
    long    noVisCalc;
    long    appendRaw;

    amplDataStruct  globlWaterfall;
    amplDataStruct  noiseGloblWaterfall;
    
    // track splitting option
    long    trackSplit;
    
    // confirming plot options
    long    plotIterOPD;
    long    plotWaterfall;
    long    plotWfallKill;
    long    plotWfallClear;
    long    plotWfallTitle;
    double  plotWfallInterval;
    double  plotWfallTime0;
    long    plotOPD;
    long    plotBeamPos;
    long    plotAvePSD;
    long    plotVis;
    //int    plotFixVisRange;
    long    plotDiff;
    long    plotBispec;
    long    plotWlen;
    long    animateFFT;
    
    // hard copy production options
    long    psWaterfall;
    long    psVis;
    long    psCP;
    
    long    gui;
};

/****************************************************************/

struct amplFringeTrackStruct
// Structure handling fringe-track related variables
{
    double  mjd;
    long    sta_index1;
    long    sta_index2;
    string  sta_name1;
    string  sta_name2;
    double  ucoord;
    double  vcoord;
    double  opd;
    double  ao_strehl;
    double  ao_wfe;
    long    goodFlag;
    long    nbBinFrames; // How many frames this bin corresponds to.
    // For raw data, nbBinFrames = 1.
    long    nbGoodFrames;// How many good frames exist within this bin
    // For raw data, nbGoodFrames = goodFlag.
    double  noiseScale;
    double  integNoise;
    double  peakSNR;
    double  attFact;
    double  integPsd;
    double  igPsdSig;
    double  photPiPj;
    double  phPiPjSig;
    double  weight;
    double  sqVis;
    double  sqVisSig;
    double  sqVisSNR;
    double  vis;
    double  visSig;
};

/****************************************************************/

struct amplCPStruct
// Structure handling closure-phase related variables
{
    double  mjd;
    double  opd1;
    double  opd2;
    long    goodFlag;
    complex bispec;
    complex bsRotCorr; // bispectrum rotation correction factor
    double  amp;
    double  ampSig;
    double  phi;
    double  phiSig;
    double  weight;
};

