/***************************************************************************
 *                                                                         *
 *                              fitOmatic                                  *
 *                   Model-fitting prototyping utility                     *
 *                                                                         *
 *                      Copyright 2007, F. Millour                         *
 *                            fmillour@oca.eu                              *
 *                                                                         *
 ***************************************************************************
 *
 * Binary star ephemeris computation.
 *
 * Please note that this script is distributed under the GPL licence,
 * available at http://www.gnu.org/licenses/gpl.txt
 *
 * Please ACKNOWLEDGE the use of this script for any use in a publication
 *
 ***************************************************************************
 *
 * "@(#) $Id: binaryEphemeris.i 667 2017-02-10 13:37:52Z fmillour $"
 *
 ***************************************************************************/

func binaryEphemeris(void)
    /* DOCUMENT binaryEphemeris

       DESCRIPTION
       Binary star ephemeris computation.
       
       VERSION
       $Revision: 667 $

       REQUIRE
       bessel.i
       yoco packages (yoco.i)

       CAUTIONS

       AUTHORS
       See AUTHORS file at the root of the directory distribution

       CONTRIBUTIONS

       FUNCTIONS
       - RotateOrbit               : 
       - Rotate_2D                 : 
       - Rotate_3D                 : 
       - binaryCoordinates         : 
       - binaryEphemeris           : 
       - binaryProjectedCoordinates: 
       - getJD                     : 
       - getMJD                    : 
       - getTJD                    : 
       - keplerCoordinates         : 
       - orbit2Mass                : 
       - plotOrbit                 : 
    */
{
    version = strpart(strtok("$Revision: 667 $",":")(2),2:-2);
    if (am_subroutine())
    {
        help, binaryEphemeris;
    }
    return version;
} 

yocoLogInfo,"#include \"binaryEphemeris.i\"";

/***************************************************************************/

require, "bessel.i";

/***************************************************************************/

func getJD(year, month, day, hour, minute, second)
    /* DOCUMENT getJD(year, month, day, hour, minute, second)

       DESCRIPTION
       Get the (Astronomical) Julian Day from a date of the year

       PARAMETERS
       - year  : the year
       - month : the month
       - day   : the day of the month
       - hour  : the hour
       - minute: the minutes
       - second: the seconds

       RESULTS
       Returns the (Astronomical) Julian Day corresponding to the input date

       CAUTIONS
       if no year is input, get the JD of the current system time

       EXAMPLES
       > getJD(2006, 12, 25, 4, 18, 35)
       2.45409e+06

       SEE ALSO:
    */
{
    if(is_void(year))
    {
        return yocoAstroJulianDayNow();
    }
    
    return yocoAstroJulianDay(year, month, day, hour, minute, second);
}

/***************************************************************************/

func getMJD(year, month, day, hour, minute, second)
    /* DOCUMENT getMJD(year, month, day, hour, minute, second)

       DESCRIPTION
       Get the Modified Julian Day (MJD = JD - 2 400 000,5) from a date of the year

       PARAMETERS
       - year  : the year
       - month : the month
       - day   : the day of the month
       - hour  : the hour
       - minute: the minutes
       - second: the seconds

       RESULTS
       Returns the Modified Julian Day corresponding to the input date

       CAUTIONS
       if no year is input, get the MJD of the current system time

       EXAMPLES
       > getMJD(2006, 12, 25, 4, 18, 35)
       54094.2

       SEE ALSO:
    */
{
    return yocoAstroJulianDay(year, month, day, hour, minute, second, modified=1);
}

/***************************************************************************/

func getTJD(year, month, day, hour, minute, second)
    /* DOCUMENT getTJD(year, month, day, hour, minute, second)


       DESCRIPTION
       Get the Truncated Julian Day (TJD = JD - 2 440 000,5) from a date of the year

       PARAMETERS
       - year  : the year
       - month : the month
       - day   : the day of the month
       - hour  : the hour
       - minute: the minutes
       - second: the seconds

       RESULTS
       Returns the Truncated Julian Day corresponding to the input date

       CAUTIONS
       if no year is input, get the TJD of the current system time

       EXAMPLES
       > getTJD(2006, 12, 25, 4, 18, 35)
       14094.2

       SEE ALSO: getJD, getMJD
    */
{
    return yocoAstroJulianDay(year, month, day, hour, minute, second, truncated=1);
}

/***************************************************************************/

func keplerCoordinates(phase, eccentricity, majorAxis)
    /* DOCUMENT keplerCoordinates(phase, eccentricity, majorAxis)


       DESCRIPTION


       PARAMETERS
       - phase       : 
       - eccentricity: 
       - majorAxis   : 

       RESULTS


       CAUTIONS


       EXAMPLES


       SEE ALSO:
    */
{
    XY = [];
    phase= phase;

    // Values always positive
    eccentricity = abs(eccentricity);
    majorAxis    = abs(majorAxis);
    
    rho = majorAxis * (1 - eccentricity * cos(phase));
    theta = 2*atan(tan(phase/2.0) * yocoMathSignedSqrt((1+eccentricity)/(1-eccentricity)));

    X = rho*cos(theta);
    Y = rho*sin(theta);
    
    grow,XY,array(X,1),array(Y,1);
    return XY;
}

/***************************************************************************/

func orbit2Mass(period, majorAxis)
    /* DOCUMENT orbit2Mass(period, majorAxis)


       DESCRIPTION

       PARAMETERS
       - period   : 
       - majorAxis: 

       RESULTS


       CAUTIONS


       EXAMPLES


       SEE ALSO:
    */
{
    omega = 2*pi/period;
    totalMass = omega^2 * majorAxis^3 / yocoAstroSI.Grav;
    return totalMass;
}

/***************************************************************************/

func binaryCoordinates(time, eccentricity, majorAxis, period, &phi)
    /* DOCUMENT binaryCoordinates(time, eccentricity, majorAxis, period, &phi)


       DESCRIPTION

       PARAMETERS
       - time        : 
       - eccentricity: 
       - majorAxis   : 
       - period      : 
       - phi         : 

       RESULTS


       CAUTIONS


       EXAMPLES


       SEE ALSO:
    */
{
    omega = 2*pi/period;
    order = 100;
    phi   = omega * time;
    for(k=1;k<=order;k++)
    {
        phi = phi + 2 * bessj(k, k * eccentricity) /
            double(k) * sin(k * omega * time);
    }
    phi = (phi-pi)%(2*pi) + pi;
    
    XY = keplerCoordinates(phi,eccentricity,majorAxis);
    return XY;
}

/***************************************************************************/

func Rotate_3D(Coord, phi, psi, theta)
    /* DOCUMENT Rotate_3D(Coord, phi, psi, theta)


       DESCRIPTION

       PARAMETERS
       - Coord: 
       - phi  : 
       - psi  : 
       - theta: 

       RESULTS


       CAUTIONS


       EXAMPLES


       SEE ALSO:
    */
{
    //Compute Euler parameters (Goldstein 1980, p. 155).
    e0 = cos(0.5 * (phi + psi)) * cos(0.5 * theta);
    e1 = cos(0.5 * (phi - psi)) * sin(0.5 * theta);
    e2 = sin(0.5 * (phi - psi)) * sin(0.5 * theta);
    e3 = sin(0.5 * (phi + psi)) * cos(0.5 * theta);

    //Compute rotation matrix
    a11 = e0^2 + e1^2 + e2^2 + e3^2;
    a12 = 2 * (e1 * e2 + e0 * e3);
    a13 = 2 * (e1 * e3 - e0 * e2);
    a21 = 2 * (e1 * e2 - e0 * e3);
    a22 = e0^2 - e1^2 + e2^2 - e3^2;
    a23 = 2 * (e2 * e3 + e0 * e1);
    a31 = 2 * (e1 * e3 + e0 * e2);
    a32 = 2 * (e2 * e3 - e0 * e1);
    a33 = e0^2 - e1^2 + e2^2 - e3^2;

    rotMat = [[a11,a12,a13],
              [a21,a22,a23],
              [a31,a32,a33]];
    
    //Rotate coordinates
    Coord1 = Coord(,+)*rotMat(-,+,);
    return Coord1;
}

/***************************************************************************/

func RotateOrbit(XY_Coord, i, omega, OMEGA)
    /* DOCUMENT RotateOrbit(XY_Coord, i, omega, OMEGA)


       DESCRIPTION

       PARAMETERS
       - XY_Coord: 
       - i       : 
       - omega   : 
       - OMEGA   : 

       RESULTS


       CAUTIONS


       EXAMPLES


       SEE ALSO:
    */
{
    plot=0;
    if(plot)
        yocoGuiWinKill;
    
    d = dimsof(XY_Coord);
    d(0)++;
    Coord3 = Coord2 = Coord1 = array(0.0,d);
    Coord1(,:2) = Rotate_2D(XY_Coord,pi/2+omega)(,1,);

    if(plot)
        plg,Coord1(,2),Coord1(,1);

    Coord2 = Coord1;
    Coord2(,1) = Coord1(,1) * cos(i);
    Coord2(,3) = Coord1(,1) * sin(i);

    if(plot)
        plg,Coord2(,2),Coord2(,1);

    Coord3(,3) = Coord2(,3);
    Coord3(,:2) = Rotate_2D(Coord2(,:2),pi/2+OMEGA)(,1,);

    if(plot)
    {
        plg,Coord3(,2),Coord3(,1);
        limits,square=1;
        window,2;
        plg,Coord3(,0);
    }
    
    return Coord3;
}

/***************************************************************************/

func Rotate_2D(Coord, theta)
    /* DOCUMENT Rotate_2D(Coord, theta)


       DESCRIPTION

       PARAMETERS
       - Coord: 
       - theta: 

       RESULTS


       CAUTIONS


       EXAMPLES


       SEE ALSO:
    */
{
    rotMat = [[cos(theta), -sin(theta)],
              [sin(theta), cos(theta)]];

    Coord1 = Coord(,+)*rotMat(-,+,);
    return Coord1;
}

/***************************************************************************/

func binaryProjectedCoordinates(time, t0, eccentricity, majorAxis, period, i, omega, OMEGA, &phi)
    /* DOCUMENT binaryProjectedCoordinates(time, t0, eccentricity, majorAxis, period, i, omega, OMEGA, &phi)


       DESCRIPTION


       PARAMETER   
       -time : epoch for which the position will be calculated (in seconds)
       -t0 : epoch of the last periastron (in seconds)
       -eccentricity : eccentricity of the binary orbit (no unit)
       -majorAxis : major-axis of the orbit (in radians)
       -period : period of the orbit (in seconds)
       -i : inclination angle of the orbit compared to the line of sight (in radians)
       -omega : (in radians)
       -OMEGA : (in radians)


       RESULTS
       Return the binary separation projected on the sky-plane at "time".

       CAUTIONS


       EXAMPLES


       SEE ALSO:
    */

{
    CoordOrbit = binaryCoordinates(time-t0,eccentricity,majorAxis,period,phi);
    CoordP2    = RotateOrbit(CoordOrbit,i,omega,OMEGA);
    return CoordP2;

}


/***************************************************************************/

func plotOrbit(t0, eccentricity, majorAxis, period, i, omega, OMEGA, &CoordP2)
    /* DOCUMENT plotOrbit(t0, eccentricity, majorAxis, period, i, omega, OMEGA, &CoordP2)


       DESCRIPTION


       PARAMETER   
       -time : epoch for which the position will be calculated (in seconds)
       -t0 : epoch of the last periastron (in seconds)
       -eccentricity : eccentricity of the binary orbit (no unit)
       -majorAxis : major-axis of the orbit (in radians)
       -period : period of the orbit (in seconds)
       -i : inclination angle of the orbit compared to the line of sight (in radians)
       -omega : (in radians)
       -OMEGA : (in radians)


       RESULTS
       Return the binary separation projected on the sky-plane at "time".

       CAUTIONS


       EXAMPLES


       SEE ALSO:
    */

{
    if(is_void(win))
        win=1;
    
    time      = span(0,period,long(100/(1-eccentricity)));

    // Set nodes line coordinates
    nodeY     = span(-majorAxis*sqrt(1-eccentricity),
                     majorAxis*sqrt(1-eccentricity),10)*1.1;
    nodeX     = array(0.0,10);
    CoordNode = grow(array(nodeX,1),array(nodeY,1));

    // Set periastron-apoastron line coordinates
    periX     = span(-majorAxis / sqrt(1-eccentricity),
                     majorAxis * sqrt(1-eccentricity),10);
    periY     = array(0.0,10);
    CoordPeri = grow(array(periX,1),array(periY,1));

    // Calculate orbit's coordinates in the orbital plane
    CoordOrbit = binaryCoordinates(time-t0,eccentricity,majorAxis,period);
    
    window,win;
    plg,CoordOrbit(,2),CoordOrbit(,1);
    
    // Plot nodes line
    plg,CoordNode(,2),CoordNode(,1),width=4;

    // Plot periastron line
    plg,CoordPeri(,2),CoordPeri(,1),type="dash";
    limits,square=1;

    // Calculate orbit's coordinates in the sky plane
    CoordP2    = RotateOrbit(CoordOrbit,i,omega,OMEGA);
    
    // Plot projected orbit
    window,win+1;
    plg,CoordP2(,2),CoordP2(,1);
    
    // Plot nodes line
    CoordNode2 = RotateOrbit(CoordNode,i,omega,OMEGA);
    plg,CoordNode2(,2),CoordNode2(,1),width=4;

    // Plot periastron line
    CoordPeri2 = RotateOrbit(CoordPeri,i,omega,OMEGA);
    plg,CoordPeri2(,2),CoordPeri2(,1),type="dash";
    
    limits,square=1;
}
