/***************************************************************************
 *                                                                         *
 *                              fitOmatic                                  *
 *                   Model-fitting prototyping utility                     *
 *                                                                         *
 *                      Copyright 2007, F. Millour                         *
 *                            fmillour@oca.eu                              *
 *                                                                         *
 ***************************************************************************
 *
 * This script contains routines to compute images of standard analytical
 * models.
 *
 * Please note that this script is distributed under the GPL licence,
 * available at http://www.gnu.org/licenses/gpl.txt
 *
 * Please ACKNOWLEDGE the use of this script for any use in a publication
 *
 ***************************************************************************
 *
 * "@(#) $Id: imaModels.i 581 2015-02-10 17:28:57Z fmillour $"
 *
 ***************************************************************************/

func imaModels(void)
/* DOCUMENT imaModels

       DESCRIPTION
       Standard analytical models for visibilities computation

       REQUIRE

       FUNCTIONS
   - imaBinaryAndWWCZ            : binary and wind-wind collision zone (not working)
   - imaBinaryStar               : binary star (not used)
   - imaEllipticalGaussianDisk   : elliptical Gaussian disk
   - imaEllipticalRing           : elliptical infinitely thin ring
   - imaEllipticalSkewedRing     : 
   - imaEllipticalThickRing      : 
   - imaEllipticalUniformDisk    : elliptical uniform disk
   - imaGaussianDisk             : Gaussian disk
   - imaModelEllipticalSkewedRing: 
   - imaModelEllipticalThickRing : 
   - imaModels                   : this script
   - imaMultipleResolved         : multiple models
   - imaPinwheel                 : pinwheel model
   - imaPinwheelTempLaw          : 
   - imaRectangle                : rectangle
   - imaRing                     : ring
   - imaSquare                   : square
   - imaUniformDisk              : uniform disk
   - imaWWCZ                     : wind-wind collision zone
    */
{
    version = strpart(strtok("$Revision: 581 $",":")(2),2:-2);
    if (am_subroutine())
    {
        write, format="package version: %s\n", version;
        help, imaModels;
    } 
    return version;
} 

yocoLogInfo,"#include \"imaModels.i\"";

/***************************************************************************/

func imaBinaryAndWWCZ(Xtable, Ytable, lambda, fluxStar1, fluxStar2, type1, type2, size1, size2, sep, pa, fluxWWCZ, dstWWCZ, sizWWCZ, thickWWCZ, factWWCZ, powerWWCZ)
/* DOCUMENT imaBinaryAndWWCZ(Xtable, Ytable, lambda, fluxStar1, fluxStar2, type1, type2, size1, size2, sep, pa, fluxWWCZ, dstWWCZ, sizWWCZ, thickWWCZ, factWWCZ, powerWWCZ)

       DESCRIPTION

       PARAMETERS
   - Xtable   : X coordinates (radians)
   - Ytable   : Y coordinates (radians)
   - lambda   : 
   - fluxStar1: 
   - fluxStar2: 
   - type1    : 
   - type2    : 
   - size1    : 
   - size2    : 
   - sep      : 
   - pa       : 
   - fluxWWCZ : 
   - dstWWCZ  : 
   - sizWWCZ  : 
   - thickWWCZ: 
   - factWWCZ : 
   - powerWWCZ: 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    typei = [type1, type2];
    fwhm1 = [size1, size2];
    fwhm2 = [0,0];
    angle = [0,0];
    x0 = [0,sep*cos(pa)];
    y0 = [0,sep*sin(pa)];
    spectrumi = [fluxStar1,fluxStar2];

    imaStars = imaMultipleResolved(Xtable, Ytable, lambda, typei, fwhm1, fwhm2, angle, x0, y0, spectrumi);

    anglWWCZ = pa;
    X0WWCZ = sep*cos(pa)*dstWWCZ;
    Y0WWCZ = sep*sin(pa)*dstWWCZ;

    ima = (imaStars + fluxWWCZ(-,) * imaWWCZ(Xtable, Ytable, lambda,
                                             sizWWCZ, thickWWCZ, factWWCZ,
                                             powerWWCZ, anglWWCZ,
                                             X0WWCZ, Y0WWCZ)) /
        (1 + fluxWWCZ(-,));

    return ima;
}

/***************************************************************************/

func imaPinwheel(Xtable, Ytable, lambda, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, tailFlux, type=, fillFactor=)
/* DOCUMENT imaPinwheel(Xtable, Ytable, lambda, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, tailFlux, type=, fillFactor=)

       DESCRIPTION

       PARAMETERS
   - Xtable     : 
   - Ytable     : 
   - lambda     : 
   - rounds     : 
   - minThick   : 
   - maxThick   : 
   - totalSize  : 
   - power      : 
   - anglePhi   : 
   - inclination: 
   - angleSky   : 
   - tailFlux   : 
   - type       : 
   - fillFactor : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    if(is_void(type))
        type="g2";

    nbWlen = numberof(lambda);
    if(is_void(fillFactor))
        fillFactor = 0.5;
    thick = avg((minThick+maxThick)/2.0);
    N = 20000;

    theta = span(0,rounds*2*pi,N);

    fact = totalSize*rounds/(rounds*2*pi/(2*pi))^power;

    r = fact(-,)/2.0/rounds(-,)*(((theta+(theta==0))/(2*pi))^power(-,) - (theta==0)/(2*pi));

    dst = r*theta;

    toto = (abs((dst%(thick/fillFactor))(dif,))>
            thick/fillFactor/2);
    titi = transpose(toto);
    tutu = grow(array(0,nbWlen,1),titi);
    test = transpose(tutu)(,1);

    pr2 = where2((abs(dst)==max(abs(dst))) |
                 (abs(dst)==min(abs(dst))) |
                 // grow(0,(abs((dst%(thick(-,)/fillFactor))(dif))>
                 // thick(-,)/fillFactor/2)));
                 test)(1,);

    ruse = array(r(pr2),nbWlen);
    tetuse = array(theta(pr2),nbWlen);

    N2 = numberof(pr2);

    typei = array(type,N2);

    thick = minThick(-,) + ruse / max(r) * maxThick(-,);

    angle0 = tetuse + anglePhi(-,);
    x0 = ruse*cos(angle0);
    y0 = ruse*sin(angle0);
    fwhmx0 = array(0, N2 ,nbWlen);
    fwhmy0 = thick;

    x1 = x0 * cos(inclination(-,));
    y1 = y0;
    angle1 = atan(y1,x1);
    fwhmx1 = fwhmy0*cos(angle0)*sin(inclination(-,));
    fwhmy1 = fwhmy0;

    angle2 = transpose(angle1-angleSky(-,));
    x2 = transpose(x1*cos(angleSky(-,))+y1*sin(angleSky(-,)));
    y2 = transpose(-x1*sin(angleSky(-,))+y1*cos(angleSky(-,)));
    fwhmx2 = transpose(fwhmx1)/2;
    fwhmy2 = transpose(fwhmy1)/2;

    //spectrumi = array(1.0,nbWlen,N2);
    spectrumi = array(0.0,nbWlen,N2);
    for(k=1;k<=nbWlen;k++)
        spectrumi(k,) = span(1, tailFlux(k), N2); 

    ima = imaMultipleResolved(Xtable, Ytable,
                               typei,
                              fwhmx2, fwhmy2,
                              angle2, x2, y2, spectrumi);
    return ima;
}

/***************************************************************************/
func imaPinwheelTempLaw(Xtable, Ytable, lambda, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, powerLawDecrease, tempInit, periodSine, amplitudeSine, amortiSine, type=, fillFactor=)
/* DOCUMENT imaPinwheelTempLaw(Xtable, Ytable, lambda, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, powerLawDecrease, tempInit, periodSine, amplitudeSine, amortiSine, type=, fillFactor=)

       DESCRIPTION

       PARAMETERS
   - Xtable          : 
   - Ytable          : 
   - lambda          : 
   - rounds          : 
   - minThick        : 
   - maxThick        : 
   - totalSize       : 
   - power           : 
   - anglePhi        : 
   - inclination     : 
   - angleSky        : 
   - powerLawDecrease: 
   - tempInit        : 
   - periodSine      : 
   - amplitudeSine   : 
   - amortiSine      : 
   - type            : 
   - fillFactor      : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    if(is_void(type))
        type="r2";

    nbWlen = numberof(lambda);
    if(is_void(fillFactor))
        fillFactor = 0.5;
    thick = avg((minThick+maxThick)/2.0);
    N = 20000;

    theta = span(0,rounds*2*pi,N);

    //fact = totalSize*rounds/(rounds*2*pi/(2*pi))^power;
    fact = totalSize*rounds/(rounds*2*pi/(2*pi))^power;//totalsize : extension totale de la spirale

    r = fact(-,)/2.0/rounds(-,)*(((theta+(theta==0))/(2*pi))^power(-,) - (theta==0)/(2*pi));

    dst = r*theta;

    toto = (abs((dst%(thick/fillFactor))(dif,))>
            thick/fillFactor/2);
    titi = transpose(toto);
    tutu = grow(array(0,nbWlen,1),titi);
    test = transpose(tutu)(,1);

    pr2 = where2((abs(dst)==max(abs(dst))) |
                 (abs(dst)==min(abs(dst))) |
                 // grow(0,(abs((dst%(thick(-,)/fillFactor))(dif))>
                 // thick(-,)/fillFactor/2)));
                 test)(1,);
   pr2 = pr2(2:);
N2 = numberof(pr2);
    while(N2>1000)
    {
        if(verbose)
            write,"Dividing number of rings from "+ pr1(N2)+" to "+ pr1(N2/2)
                pr2 = pr2(::2);
        N2 = numberof(pr2);
    }


    ruse = array(r(pr2),nbWlen);
    tetuse = array(theta(pr2),nbWlen);


    typei = array(type,N2);

    thick = minThick(-,) + ruse / max(r) * maxThick(-,);

    angle0 = tetuse + anglePhi(-,);
    x0 = ruse*cos(angle0);
    y0 = ruse*sin(angle0);
    fwhmx0 = array(0, N2 ,nbWlen);
    fwhmy0 = thick;

    x1 = x0 * cos(inclination(-,));
    y1 = y0;
    angle1 = atan(y1,x1);
    fwhmx1 = fwhmy0*cos(angle0)*sin(inclination(-,));
    fwhmy1 = fwhmy0;

    angle2 = transpose(angle1-angleSky(-,));
    x2 = transpose(x1*cos(angleSky(-,))+y1*sin(angleSky(-,)));
    y2 = transpose(-x1*sin(angleSky(-,))+y1*cos(angleSky(-,)));
    fwhmx2 = transpose(fwhmx1)/2;
    fwhmy2 = transpose(fwhmy1)/2;

    //spectrumi = array(1.0,nbWlen,N2);
//   spectrumi = array(0.0,nbWlen,N2);
    spectrumi = array(0.0,nbWlen,numberof(ruse));

  //  for(k=1;k<=nbWlen;k++)
  //      spectrumi(k,) = span(1, tailFlux(k), N2); 

    r0=0.00133*4.85e-6;
     temperature_law=array(0.0,numberof(ruse(,1)));

    rAU = ruse(,1)*rad2mas/1000*1.6e3;

    powerlaw = tempInit * (ruse(,1) / r0) ^ powerLawDecrease;
    variable = (-amplitudeSine * cos(2 * pi / periodSine * (ruse(,1)-r0)) +amplitudeSine) * exp(-amortiSine * rAU);
    
      temperature_law(where(ruse(,1)>=r0)) = (powerlaw + variable)(where(ruse(,1)>=r0));

        for(k=1;k<=nbWlen;k++)
         {
            spectrumi(k, where(ruse(,1)>=r0)) = yocoAstroBBodyLambda(lambda(k), temperature_law(where(ruse(,1) >= r0)) );
         }

    ima = imaMultipleResolved(Xtable, Ytable,
                               typei,
                              fwhmx2, fwhmy2,
                              angle2, x2, y2, spectrumi);
    return ima;
}

/***************************************************************************/


func imaWWCZ(Xtable, Ytable, lambda, totalSize, thick, fact, power, angle, x0, y0)
/* DOCUMENT imaWWCZ(Xtable, Ytable, lambda, totalSize, thick, fact, power, angle, x0, y0)

       DESCRIPTION

       PARAMETERS
   - Xtable   : U coordinates (meters)
   - Ytable   : 
   - lambda   : 
   - totalSize: 
   - thick    : 
   - fact     : 
   - power    : 
   - angle    : 
   - x0       : 
   - y0       : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    nbWlen = numberof(lambda);
    fillFactor = 4;
    N2 = 10000;
    pr = span(-totalSize/2,totalSize/2,N2);
    kx0 = pr;
    ky0 = fact*(abs(kx0+(abs(kx0)<1e-10)))^power;



    dst = sign(kx0*ky0)*abs(kx0, ky0);

    kpr2 = pr(where(abs(dst) < totalSize/2));
    dst2 = dst(where(abs(dst) < totalSize/2));

    pr2 = where((dst2==max(dst2))|
                (dst2==min(dst2))|
                (abs(dst2)==min(abs(dst2)))|
                grow(0,(abs((dst2%(thick/fillFactor))(dif))>thick/fillFactor/2)));
    N = numberof(pr2);
    write,N;

    x = kpr2(pr2);
    y = fact*(abs(x+(abs(x)<1e-10)))^power;
    typei = array("g",N);
    fwhm1 = array(thick,N);
    fwhm2 = array(0.0,N);
    angle2 = array(0.0,N);
    x1 = -sin(angle)*x + cos(angle)*y;
    y1 = cos(angle)*x + sin(angle)*y;
    spectrumi = array(1.0,nbWlen,N);

    ima = imaMultipleResolved(Xtable, Ytable,
                              lambda, typei,
                              fwhm1, fwhm2,
                              angle2, x1, y1, spectrumi);
    C = shiftFourier(Xtable,Ytable,lambda,ima,x0,y0);

    return C;
}
/***************************************************************************/

func imaBinaryStar(Xtable, Ytable, lambda, sep, pa, fxRat)
/* DOCUMENT imaBinaryStar(Xtable, Ytable, lambda, sep, pa, fxRat)

       DESCRIPTION

       PARAMETERS
   - Xtable: U coordinates (meters)
   - Ytable: 
   - lambda: 
   - sep   : 
   - pa    : 
   - fxRat : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    u = Xtable(,-) / lambda(-,);
    v = Ytable(,-) / lambda(-,);

    C = (1 + fxRat(-,) * exp(-2i*pi*(u*sep(-,)*cos(pa(-,)) -
                                     v*sep(-,)*sin(pa(-,)))))/
        (1 + fxRat(-,));
    return C;
}

/***************************************************************************/

func imaSquare(Xtable, Ytable, diam, x0, y0)
/* DOCUMENT imaSquare(Xtable, Ytable, diam, x0, y0)

       DESCRIPTION

       PARAMETERS
   - Xtable: U coordinates (meters)
   - Ytable: 
   - diam  : 
   - x0    : 
   - y0    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    if(diam(1)==0)
    {
        diam2 = min(Xtable(dif));
    }
    xPrim = 2*(Xtable(,-)-x0(-,)) / diam;
    yPrim = 2*(Ytable(,-)-y0(-,)) / diam;

    i0 = ((abs(xPrim)<=1)&(abs(yPrim)<=1));
    aire = sum(i0);
    I = double(i0)/(aire+(aire==0));
    return I;
}  

/***************************************************************************/

func imaRectangle(Xtable, Ytable, diam1, diam2, angle, x0, y0)
/* DOCUMENT imaRectangle(Xtable, Ytable, diam1, diam2, angle, x0, y0)

       PARAMETERS
   - Xtable: U coordinates (meters)
   - Ytable: 
   - diam1 : 
   - diam2 : 
   - angle : 
   - x0    : 
   - y0    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    xPrim = 2*((Xtable(,-)-x0(-,))*sin(angle)+
               (Ytable(,-)-y0(-,))*cos(angle)) / diam1;
    yPrim = 2*((Xtable(,-)-x0(-,))*cos(angle)-
               (Ytable(,-)-y0(-,))*sin(angle)) / diam2;

    i0 = ((abs(xPrim)<=1)&(abs(yPrim)<=1));
    aire = sum(i0);
    I = double(i0)/(aire+(aire==0));
    return I;
}  

/***************************************************************************/

func imaGaussianDisk(Xtable, Ytable, fwhm, x0, y0)
/* DOCUMENT imaGaussianDisk(Xtable, Ytable, fwhm, x0, y0)

       DESCRIPTION

       PARAMETERS
   - Xtable: U coordinates (meters)
   - Ytable: 
   - fwhm  : 
   - x0    : 
   - y0    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    r2 = 4*log(2.0)* ((Xtable(,-) - x0(-,))^2 + (Ytable(,-) - y0(-,))^2) /
        (fwhm(-,)^2);
    sigma = 1/sqrt(pi / (4 * log(2) * fwhm));
    i0 = sigma(-,) * exp(-r2);

    return i0;
} 

/***************************************************************************/

func imaEllipticalGaussianDisk(Xtable, Ytable, fwhm1, fwhm2, angle, x0, y0)
/* DOCUMENT imaEllipticalGaussianDisk(Xtable, Ytable, fwhm1, fwhm2, angle, x0, y0)

       DESCRIPTION

       PARAMETERS
   - Xtable: U coordinates (meters)
   - Ytable: 
   - fwhm1 : 
   - fwhm2 : 
   - angle : 
   - x0    : 
   - y0    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    sigma1 = fwhm1/(2*sqrt(2*log(2)));
    sigma2 = fwhm2/(2*sqrt(2*log(2)));

    r2 = 1.0/2.0*((((Xtable-x0)*sin(angle) + (Ytable-y0)*cos(angle)) /
                   (sigma1+(sigma1==0)))^2 +
                  (((Xtable-x0)*cos(angle) - (Ytable-y0)*sin(angle)) /
                   (sigma2+(sigma2==0)))^2);

    norm = 1.0 / (2 * pi * sigma1 * sigma2);

    i0 = exp(-r2);
    i0 = i0 / sum(i0);

    return i0;
}

/***************************************************************************/

func imaRing(Xtable, Ytable, diam, x0, y0)
/* DOCUMENT imaRing(Xtable, Ytable, diam, x0, y0)

       DESCRIPTION

       PARAMETERS
   - Xtable: U coordinates (meters)
   - Ytable: 
   - diam  : 
   - x0    : 
   - y0    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    r = abs(Xtable(,-)-x0(-,), Ytable(,-)-y0(-,));

    nX = numberof(Xtable);
    pxSize = max(r)/sqrt(2)/sqrt(nX);
    I = ((r >= diam(-,)/2-pxSize) & (r <= diam(-,)/2+pxSize));
    sigma = 1/(2*pi*diam);
    I2 = sigma(-,) * I;

    return I2;
} 

/***************************************************************************/

func imaEllipticalRing(Xtable, Ytable, diam1, diam2, angle, x0, y0)
/* DOCUMENT imaEllipticalRing(Xtable, Ytable, diam1, diam2, angle, x0, y0)

       DESCRIPTION

       PARAMETERS
   - Xtable: 
   - Ytable: 
   - diam1 : 
   - diam2 : 
   - angle : 
   - x0    : 
   - y0    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    I1 = imaEllipticalUniformDisk(Xtable, Ytable, diam1, diam2, angle, x0, y0, distToCentre1, aire1);

    nX = numberof(Xtable); 
    pxSize = 4*max(Xtable)/sqrt(nX);
    pxSize1 = pxSize*diam1/diam2;
    pxSize2 = pxSize*diam2/diam1;
    
    I2 = imaEllipticalUniformDisk(Xtable, Ytable, diam1-pxSize1, diam2-pxSize2, angle, x0, y0, distToCentre2, aire2);

    I = (I1 * aire1 - I2*aire2) / ((aire1 - aire2)+((aire1 - aire2)==0));
    
    return I;
} 

/***************************************************************************/

func imaEllipticalThickRing(Xtable, Ytable, diam1, diam2, angle, x0, y0, thick)
/* DOCUMENT imaEllipticalThickRing(Xtable, Ytable, diam1, diam2, angle, x0, y0, thick)

       DESCRIPTION

       PARAMETERS
   - Xtable: 
   - Ytable: 
   - diam1 : 
   - diam2 : 
   - angle : 
   - x0    : 
   - y0    : 
   - thick : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    nX = numberof(Xtable);
    thick1 = thick*sin(atan(diam1,diam2));
    thick2 = thick*sin(atan(diam2,diam1));
    
    I1 = imaEllipticalUniformDisk(Xtable, Ytable, diam1+thick1, diam2+thick2, angle, x0, y0, distToCentre1, aire1);
    I2 = imaEllipticalUniformDisk(Xtable, Ytable, diam1-thick1, diam2-thick2, angle, x0, y0, distToCentre2, aire2);

    I = (I1 * aire1 - I2*aire2) / ((aire1 - aire2)+((aire1 - aire2)==0));
    
    return I;
} 

/***************************************************************************/

func imaEllipticalSkewedRing(Xtable, Ytable, diam1, diam2, angle, x0, y0, thick, angleSk, ampSk)
/* DOCUMENT imaEllipticalSkewedRing(Xtable, Ytable, diam1, diam2, angle, x0, y0, thick, angleSk, ampSk)

       DESCRIPTION

       PARAMETERS
   - Xtable : 
   - Ytable : 
   - diam1  : 
   - diam2  : 
   - angle  : 
   - x0     : 
   - y0     : 
   - thick  : 
   - angleSk: 
   - ampSk  : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    I1 = imaEllipticalThickRing(Xtable, Ytable, diam1, diam2, angle, x0, y0, thick);
    
    angl = atan(Xtable, Ytable);
    modul = (1.0+ampSk(-,)*cos(angl(,-)+angleSk(-,)))/2.0;

    return I1*modul;
} 

/***************************************************************************/

func imaUniformDisk(Xtable, Ytable, diam, x0, y0)
/* DOCUMENT imaUniformDisk(Xtable, Ytable, diam, x0, y0)

       DESCRIPTION

       PARAMETERS
   - Xtable: U coordinates (meters)
   - Ytable: 
   - diam  : 
   - x0    : 
   - y0    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    dimz1 = dimsof(Xtable);
    dimz2 = dimsof(lambda);
    
    I = array(0.0,[dimz1(1)+dimz2(1),dimz1(2:),dimz2(2:)]);
    
    dfsfgsd()
        
        r0 = abs(Xtable(,-)-x0(-,), Ytable(,-)-y0(-,));
    r = 2*r0 / diam(-,);
    aire = pi*diam^2/4.0;
    i0 = (r <= 1);
    I = double(i0)/aire(-,);
    return I;
} 

/***************************************************************************/

func imaEllipticalUniformDisk(Xtable, Ytable, diam1, diam2, angle, x0, y0, &distToCentre, &aire)
/* DOCUMENT imaEllipticalUniformDisk(Xtable, Ytable, diam1, diam2, angle, x0, y0, &distToCentre, &aire)

       DESCRIPTION

       PARAMETERS
   - Xtable      : U coordinates (meters)
   - Ytable      : 
   - diam1       : 
   - diam2       : 
   - angle       : 
   - x0          : 
   - y0          : 
   - distToCentre: 
   - aire        : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    dimz1 = dimsof(Xtable);
    dimz2 = dimsof(lambda);
        
    xPrim = (((Xtable(,-)-x0(-,))*sin(angle(-,))+
              (Ytable(,-)-y0(-,))*cos(angle(-,))) / (diam1+(diam1==0))(-,));
    yPrim = (((Xtable(,-)-x0(-,))*cos(angle(-,))-
              (Ytable(,-)-y0(-,))*sin(angle(-,))) / (diam2+(diam2==0))(-,));

    distToCentre = 2 * abs(xPrim, yPrim);
    i0 = (distToCentre <= 1);
    aire = sum(i0);
    I = double(i0)/(aire+(aire==0));

    return I;
} 


/***************************************************************************/

func imaMultipleResolved(Xtable, Ytable, lambda, time, typei, spec, p2, p3, p4, p5, p6, p7, p8, p9, p10, p11, p12, p13, p14, p15, p16, p17, p18, p19, p20, &corrFlux, &flux)
/* DOCUMENT imaMultipleResolved(Xtable, Ytable, lambda, time, typei, spec, p2, p3, p4, p5, p6, p7, p8, p9, p10, p11, p12, p13, p14, p15, p16, p17, p18, p19, p20, &corrFlux, &flux)

       DESCRIPTION

       PARAMETERS
   - Xtable  : U coordinates (meters)
   - Ytable  : 
   - lambda  : 
   - time    : 
   - typei   : 
   - spec    : 
   - p2      : 
   - p3      : 
   - p4      : 
   - p5      : 
   - p6      : 
   - p7      : 
   - p8      : 
   - p9      : 
   - p10     : 
   - p11     : 
   - p12     : 
   - p13     : 
   - p14     : 
   - p15     : 
   - p16     : 
   - p17     : 
   - p18     : 
   - p19     : 
   - p20     : 
   - corrFlux: 
   - flux    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    nChips = numberof(typei);
    nLambda = numberof(lambda);
    nUV = numberof(Xtable);
    bigSizei = abs(bigSizei);
    smallSizei = abs(smallSizei);
    I = 0;

    for(iChips=1;iChips<=nChips;iChips++)
    {
        if (typei(iChips)=="ud")
        {
            Ii = imaUniformDisk(
                                Xtable,Ytable,lambda, time,
                                p4(,iChips),
                                p2(,iChips),
                                p3(,iChips));
        }
        if (typei(iChips)=="ud2")
        {
            Ii = imaEllipticalUniformDisk(
                                          Xtable,Ytable,lambda, time,
                                          p4(,iChips),
                                          p5(,iChips),
                                          p6(,iChips),
                                          p2(,iChips),
                                          p3(,iChips));
        }
        else if (typei(iChips)=="g")
        {
            Ii = imaGaussianDisk(
                                 Xtable,Ytable,lambda, time,
                                 p4(,iChips),
                                 p2(,iChips),
                                 p3(,iChips));
        }
        else if (typei(iChips)=="g2") 
        {
            Ii = imaEllipticalGaussianDisk(
                                           Xtable,Ytable,lambda, time,
                                           p4(,iChips),
                                           p5(,iChips),
                                           p6(,iChips),
                                           p2(,iChips),
                                           p3(,iChips));
        }
        else if (typei(iChips)=="r")
        {
            Ii = imaRing(
                         Xtable,Ytable,lambda, time,
                         p4(,iChips),
                         p2(,iChips),
                         p3(,iChips));
        }
        else if (typei(iChips)=="r2") 
        {
            Ii = imaEllipticalRing(
                                   Xtable,Ytable,lambda, time,
                                   p4(,iChips),
                                   p5(,iChips),
                                   p6(,iChips),
                                   p2(,iChips),
                                   p3(,iChips));
        }
        else if (typei(iChips)=="sq")
        {
            Ii = imaSquare(
                           Xtable,Ytable,lambda, time,
                           p4(,iChips),
                           p2(,iChips),
                           p3(,iChips));
        }
        else if (typei(iChips)=="sq2") 
        {
            Ii = imaRectangle(
                              Xtable,Ytable,lambda, time,
                              p4(,iChips),
                              p5(,iChips),
                              p6(,iChips),
                              p2(,iChips),
                              p3(,iChips));
        }
        else if (typei(iChips)=="ful")
        {
            Ii = array(1.0/nUV, nUV, nLambda);

        }

        if (typei(iChips)=="no")
        {
            nX = numberof(Xtable);
            tol = max(Xtable)*2.1/sqrt(nX);
            Ii = imaSquare(Xtable, Ytable, tol, xi(,iChips),
                           yi(iChips))*1.102500000000023794299863766355;
        }

        I = I + spectrumi(,iChips)(-,) * Ii;
    }
    ima = I / sum(spectrumi);
    return ima;
}
    

/***************************************************************************/

func imaModelEllipticalSkewedRing(diam1, diam2, angle, x0, y0, thick, angleSk, ampSk, &x, &y, dim=, champVue=)
/* DOCUMENT imaModelEllipticalSkewedRing(diam1, diam2, angle, x0, y0, thick, angleSk, ampSk, &x, &y, dim=, champVue=)

       DESCRIPTION

       PARAMETERS
   - diam1   : 
   - diam2   : 
   - angle   : 
   - x0      : 
   - y0      : 
   - thick   : 
   - angleSk : 
   - ampSk   : 
   - x       : 
   - y       : 
   - dim     : 
   - champVue: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(dim))
        dim = DIM0;

    if(is_void(champVue))
        champVue = 1.02*max(diam1(1)+thick(1), diam2(1)+thick(1), 0.5*mas2rad);
    
    x = (array(indgen(dim)-1.,dim)/(dim-1)-0.5)*champVue(1);
    y = transpose(x);

    dimx = dimsof(x);
    nbWlen = numberof(diam1);

    img = imaEllipticalSkewedRing(ima2Grid(x),
                                  ima2Grid(y),
                                  diam1, diam2, angle,
                                  x0, y0, thick, angleSk, ampSk);
    
    img = reform(img, grow(dimx(1)+1, dimx(2:), nbWlen));
    return img;
}


/***************************************************************************/

func imaModelEllipticalThickRing(diam1, diam2, angle, x0, y0, thick, &x, &y, dim=, champVue=)
/* DOCUMENT imaModelEllipticalThickRing(diam1, diam2, angle, x0, y0, thick, &x, &y, dim=, champVue=)

       DESCRIPTION

       PARAMETERS
   - diam1   : 
   - diam2   : 
   - angle   : 
   - x0      : 
   - y0      : 
   - thick   : 
   - x       : 
   - y       : 
   - dim     : 
   - champVue: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(dim))
        dim = DIM0;

    if(is_void(champVue))
        champVue = 1.05*max(diam1(1)+thick(1), diam2(1)+thick(1), 1.0*mas2rad);
    
    x = (array(indgen(dim)-1.,dim)/(dim-1)-0.5)*champVue(1);
    y = transpose(x);

    dimx = dimsof(x);
    nbWlen = numberof(diam1);

    img = imaEllipticalThickRing(ima2Grid(x),
                                 ima2Grid(y),
                                 diam1, diam2, angle,
                                 x0, y0, thick);
    
    img = reform(img, grow(dimx(1)+1, dimx(2:), nbWlen));
    return img;
}
