/***************************************************************************
 *                                                                         *
 *                              fitOmatic                                  *
 *                   Model-fitting prototyping utility                     *
 *                                                                         *
 *                      Copyright 2007, F. Millour                         *
 *                            fmillour@oca.eu                              *
 *                                                                         *
 ***************************************************************************
 *
 * This script contains routines for a BBM-like (but with the wavelength
 * dimension) test image-reconstruction algorithm
 *
 * Please note that this script is distributed under the GPL licence,
 * available at http://www.gnu.org/licenses/gpl.txt
 *
 * Please ACKNOWLEDGE the use of this script for any use in a publication
 *
 ***************************************************************************
 *
 * "@(#) $Id: imagingStuff.i 566 2015-02-07 16:27:40Z fmillour $"
 *
 ***************************************************************************/

require, "fitOmatic.i";

/*******************************************************/
/*******************************************************/

func getTimestamp(void)
/* DOCUMENT getTimestamp

       DESCRIPTION

       PARAMETERS

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    stamp = yocoStrReplace(yocoStrReplace(timestamp(), " ", "_"), ":", "-");
    return stamp;
}
/************************************************************************/

func computeTime(inString, keyword)
/* DOCUMENT computeTime(inString, keyword)

       DESCRIPTION

       PARAMETERS
   - inString: 
   - keyword : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    Seconds = []; 
    dmonths = [31.0, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
    nmonths = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];

    for(i=1;i<=numberof(inString);i++)
    {
        if(inString(i) == "-")
            return 0;

        tab = yocoStrSplit(inString(i), "-");
        minute = yocoStr2Double(tab(2));
      
        tab2 = yocoStrSplit(tab(3), "_");
        year = yocoStr2Double(tab2(2));
        second = yocoStr2Double(tab2(1));

        tab3 = yocoStrSplit(tab(1), "_");
        month = where(nmonths==tab3(2))(1);
        day = yocoStr2Double(tab3(3));
        hour = yocoStr2Double(tab3(4));

        if(keyword="hour")
            Seconds = 3600 * hour +
                60 * minute +
                second;
        else
            Seconds = 
                365.25 * 24 * 3600 * year +
                24 * 3600 * dmonths(1:int(month)-1)(sum) +
                24 * 3600 * day +
                3600 * hour +
                60 * minute +
                second;
    }
    return Seconds;
}

/*******************************************************/ 

wkll;

/*******************************************************/

// Define fit parameters
tot = fitStruct();
// Define fit function. Default should be funcMultiple
tot.funcFit  = "funcMultiple";
// Regularization. For now only "compactness" is coded
tot.regul  = "compactness";
tot.rglFact  = 1.0 / 50.0 / mas2rad;
// Use the different observables. Available for now are squared visibilities,
// closure phases and differential phases.
tot.use_vis2 = 1;
tot.use_clos = 1;
tot.use_dPhi = 0;
tot.use_dVis = 0;
// In case to compute absolute value of closure phases. DO NOT USE !
tot.absClos  = 0;
// specify positivity of flux here
tot.absFlux  = 1;
// specify positivity of angular sizes here
tot.absSize  = 1;
// wrap allow to plot the data unwrapped (does not work so well)
wrap         = 0;
// Plot visibilities, closure phases and differential phases
plotVis      = 1;
plotClos     = 1;
plotPhi      = 0;
// od is a marker to stop fit and only plot the model on the data
od           = 0;
// For monte-carlo optimization one needs this factor
fac0         = 0.01;
fac          = fac0;
prec         = 1e-2;
// Number of wavelengths in the model
nWlen        = 5;
// Maximum size to test for fit
maxSz        = 100*mas2rad;
// Initialize field of view of parameters variations:
// i for interferometric ones, s for spectroscopic ones.
maxSpec      = 10.0;
// Maximum iterations of the simulated annealing fit
maxIT        = 10;
// maximum number of sources in the image
maxSources   = 10;
// The building block type
//blocksel     = "ud";
//blocksel     = "no";
blocksel     = "g2";

// Define data directory
dir = "/home/fmillour/ARTICLES/AMOI/NF_Betelgeuse2/DATA_2Lines/";

/*******************************************************/

// Set windows
wkll;
if(od==0)
    window,3,height=550,width=550,dpi=90,wait=1,style="work.gs";

//Load the data
if(!dej)
{
    amdlibLoadSciFiles, sci, nightDir=dir, calibrated=1, crl = 1, crr = -0;
    dej=1;
    
    for(k=1;k<=numberof(sci);k++)
    {
        // Set closure phase to absolute value of closure phase.
        if((tot.absClos==1)&&(!is_void(*(sci(k).clos))))
            *(sci(k).clos)= abs(*(sci(k).clos));

        // Remove spurious residuals in differential phases
        dphi = *(sci(k).dPhi);
        wlen = *(sci(k).wlen);
        C = exp(1i*dphi);
        for(l=1;l<=numberof(dphi(,1,1));l++)
        {
            (*(sci(k).dPhi))(l,,1) = -computeInsDiffPhi(C(l,,1), wlen);
        }
    }
}

// Store data in fit structure
tot.data = &sci;

// Plot data
plotDataSpFreq,*tot.data,projAngle=refPA,kill=1,plotVis=plotVis,plotDPhi=plotPhi,plotClos=plotClos,wrap=wrap;

/*******************************************************/ 



window,4,style="amdlibLargeBox.gs",width=650,height=450;
window,5,style="amdlibLargeBox.gs"; 

  

/********************* Paramètres ad hoc *******************/

// Reak number of iteration depends on number of sources
totalSteps = double(maxSources*maxIT);

// Get startup time to count the elapsed time
time0 = computeTime(getTimestamp());
runTime = elapsed = total = [];

/*************** Debut des iterations *********/
kMC=1;
for(kMC=1;kMC<=maxSources;kMC++)
{
    // Initialize the chi2
    refChi2 = 1e99;
    
    // Specific initialization of the blocksel is a point source
    // (hence a binary is used for initialization)
    if(blocksel == "no")
    {
        if(kMC ==1)
        {
            typei = array(blocksel,2);
            parNbWlen =  [[1,0,0,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0],
                          [nWlen,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0]];
            isVariable = [[0,0,0,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0],
                          [1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0]];
        }
        else if(kMC !=1)
        {
            grow, typei, blocksel;
            parNbWlen = grow(parNbWlen, [nWlen,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0]);
            isVariable = grow(array(0,dimsof(isVariable)), [1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0]);
            oldVal = refVal;
        }
    }
    else
    {
        if(kMC ==1)
        {
            typei = array(blocksel,1);
            parNbWlen =  [[1,0,0,1,0,0]];
            isVariable = [[0,0,0,1,0,0]];
        }
        else if(kMC !=1)
        {
            grow, typei, blocksel;
            parNbWlen = grow(parNbWlen, [nWlen,1,1,1,0,0]);
            isVariable = grow(array(0,dimsof(isVariable)), [1,1,1,1,0,0]);
            oldVal = refVal;
        }
    }
    
    // Initialization of models used
    pars  = ["Flux", "X", "Y", "FWHM1", "FWHM2", "Angle"];
    units = ["no", "mas", "mas", "mas", "mas", "deg"];
    
    // initialize the model
    initModel, tot, typei, pars, units, isVariable, parNbWlen, 0*mas2rad;

    // randomize the initial paremeters
    if(kMC ==1)
    {
        (*tot.modelVal) = random(sum(parNbWlen))*maxSz;
        (*tot.modelVal)(1) = 1;
    }
    if(kMC !=1)
    {
        newVal = grow(oldVal, random(sum(parNbWlen(,0)))*maxSz);
        (*tot.modelVal) = newVal;
    }

    // Initialize field of view of parameters variations:
    buildFov,tot,maxSpec,maxSz;

    /*******************************************************/
    // This is needed for technical reasons
    updateParams,tot,*tot.modelVal;
    params = (*tot.param);

    /*******************************************************/

    // vectorize the data
    dataLine = getDataInLine(*tot.data,
                             tot.use_vis2,
                             tot.use_clos,
                             tot.use_dVis,
                             tot.use_dPhi,
                             dataError,dataType);

    tot.dataLine = &dataLine;
    tot.dataErrLine = &dataError;
    tot.dataType = &dataType;

    // Initialize the data weights
    weight = 1.0/dataError^2;

    // Initialize additional values
    updateParVal,tot,params; 

    // Loop on several steps to do some kind of global optimization
    for(k=1;k<=maxIT;k++)
    {
        write,"test "+pr1(k);
        // Store previous chi squared
        tmpRefChi2 = refChi2;

        // This loop is interesting because sometimes the lmfit procedure comes
        //to a local minimum and iterating on it allow to converge faster.
        do
        {
            // od is a marker to stop fit and only plot the model on the data
            if(od==0)
                r = trfit(resFunc,tot,params,quiet=1);
            
            chi2 = computeChi2(tot, params, dataLine, weight, dataType);
            compar = abs(chi2-tmpRefChi2);
        
            if(chi2<tmpRefChi2)
            {
                tmpRefChi2 = chi2;
                updateParVal,tot,params;
            }
            if(od==1)
                break;
        }
        while((compar>prec)&&(chi2<tmpRefChi2));

        if(chi2<refChi2)
        {
            refChi2 = chi2;
            
            getRefModel, tot, refMod;
            refA   = *refMod.param;
            refVal = *refMod.modelVal;
            
            gamble=random();
            if(gamble<0.8)
                fac = fac0;
        
            // Write stuff
            write,model;
            nfree = numberof(dataLine)-numberof(params);
            write,"Chi2",chi2, "Red. Chi2",chi2/(nfree+(nfree==0)),
                "Red. Chi22",chi2/(nfree+(nfree==0)) * numberof(params);
            print,*tot.modelVal;
            writeParams,tot;

        
            toto = (chi2-refChi2);
            titi = 1.0/(toto+1e-9*(toto==0));
        }

        //print, refVal;
        
        plotDataSpFreq,*tot.data,projAngle=refPA,clear=1,
            plotVis=plotVis,plotDPhi=plotPhi,plotClos=plotClos,wrap=wrap;
        plotDataSpFreq,*refMod.model,color=[192,192,192],model=1, width=5 ,projAngle=refPA,
            plotVis=plotVis,plotDPhi=plotPhi,plotClos=plotClos,clear=0,wrap=wrap;
        plotDataSpFreq,*tot.model,color="black",model=1, width=5 ,projAngle=refPA,
            plotVis=plotVis,plotDPhi=plotPhi,plotClos=plotClos,clear=0,wrap=wrap;
        
        plsys,1;
        limits,,,0,0.25;
        plsys,2;
        limits,,,-pi,pi;
        // plsys,3;
        // limits,,,-pi,pi;
        
        if(od==1)
        {
            window,1;
            hcps,"~/data.ps";
            require,"chg_bbox.i";
            chg_bbox,HOME+"/data.ps";
            write,"Stopped here because od equals zero";
            break;
        }
    
    
        //plotData,*refMod.model,color="blue",projAngle=,model=1;
        //    write,fac,refA,refA*fac*random_n(numberof(refA))
        gamble=random();
        if(gamble>0.2)
        {
            tutu = 0.5*atan(titi)/pi*random();
            fac *= 1.0+tutu;
            params = (refA + refA*fac*random_n(numberof(refA))-(*tot.fov))%(2*(*tot.fov)) + (*tot.fov);
        }
        else if(gamble<0.2)
        {
            params = *tot.fov*sign(random(numberof(refA))-0.5)*random(numberof(refA));
        }

        if(fac>100)
        {
            fac = fac0;
        }
        //yorick_stats();
    } 

}

/*************** Fin des iterations *********/
