/**************************************************************************
 *                                                                        *
 *                              MIRA-Lambda                               *
 *                                                                        *
 *              Add-on script to the MIRA software to handle              *
 *                 multi-wavelength datasets, including a                 *
 *                         self-calibration step                          *
 *                                                                        *
 *                 Copyright F. Millour, MPIFR, OCA 2010                  *
 *                            fmillour@oca.eu                             *
 *                          All rights reserved                           *
 *                                                                        *
 **************************************************************************
 *
 * This script contains a self-calibration addendum to the MIRA software
 *
 * Please note that this script is distributed under the GPL licence,
 * available at http://www.gnu.org/licenses/gpl.txt
 *
 * Parts of this script come from the MIRA software. MIRA is a free
 * software under the GPL developed by E. Thiebaut
 *
 * Please ACKNOWLEDGE the use of this script for any use in a
 * publication by citing Millour et al. A&A 2011, 526, 107
 *
 ***************************************************************************
 *
 * What you will need to run this script:
 *
 * A working copy of amdlib: 3.XX, distributed at
 * http://www.jmmc.fr/data_processing_amber.htm
 *
 * the MIRA software, distributed at
 * http://www-obs.univ-lyon1.fr/labo/perso/eric.thiebaut/mira.html
 *
 ***************************************************************************/

// External includes
require, "amdlib.i";         // amdlib.i from amdlib
require, "amdlibPipeline.i"; // amdlibPipeline.i from F. Millour
require, "mira.i";           // mira.i from MIRA
require, "gauss.i";          // gauss.i from yutils
require, "lmfit.i";

// Includes from the self-cal software
require, "amplData.i";
require, "amplStructures.i";
require, "visModels.i";

/***************************************************************************/

func miral(void)
    /* DOCUMENT miral

       Add-on script to the MIRA software to handle 
       multi-wavelength datasets, including a 
       self-calibration step 

       Copyright F. Millour, MPIFR 2010, OCA 2011 
       fmillour@oca.eu 
       All rights reserved 

       FUNCTIONS
       - lff_visibility_2D           : 
       - lff_visibility_round        : 
       - mira_projected_gradient_norm: 
       - miral                       : This script
       - miral_circ_uv               : 
       - miral_clean_beam            : Compute the clean beam, from a Gaussian
       fit to the dirty beam (from MIRA)
       - miral_compute_dim           : Computes the optimal image dimension
       (2^n) given an UV coverage.
       - miral_compute_fov           : Computes the field of view (with shortest
       baseline) given a UV coverage
       - miral_compute_imgCont       : computes a continuum image 
       - miral_compute_photocentre   : compute the photocenters of an image cube
       - miral_compute_pixelsize     : Compute the optimal pixel size given a UV
       coverage (longest baseline)
       - miral_convol_beam           : Convolves an image with a beam
       - miral_get_avgWlen           : get the average wavelength from a bunch
       of data
       - miral_get_best_avgImg       : 
       - miral_get_best_medianImg    : 
       - miral_get_maxBase           : get maximum baseline
       - miral_lff                   : Low frequency filling
       - miral_plot_imgCube          : Plot image cube
       - miral_plot_imgCube_waterfall: Plot waterfall of image cubes
       - miral_polar_to_cartesian    : 
       - miral_read_imgCube          : read an image cube
       - miral_recenter_imgcube      : recenter image cube images
       - miral_reconstruct_multiwlen : multiwavelength image reconstruction
       - miral_rotate_uv             : 
       - miral_self_calibrate        : self-calibration step
       - miral_slice_wlens           : Slice wavelengths
       - miral_solve_multiRgl        : stacked regularizations (super-parameter)
       reconstructions
       - miral_write_imgCube         : image cube writing routine
       - mod                         : 
       - select_best_mu              : 

       SEE ALSO
    */
{
    version = strpart(strtok("$Revision: 703 $",":")(2),2:-2);
    if (am_subroutine())
    {
        help, miral;
    } 
    return version;
}

/***************************************************************************/

func miral_circ_uv(step, inputFile=)
    /* DOCUMENT miral_circ_uv(step, inputFile=)

       DESCRIPTION

       PARAMETERS
       - step     : 
       - inputFile: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    /* Check the inputFile */
    message = "DATA FILE\n Enter the FITS data file to read";
    if (amdlibFileChooser(message, inputFile) == 0)
    {
        yocoError,"'inputFile' not specified correctly.";
        return 0;
    }
    
    for(k=step;k<=180.;k+=step)
    {
        write,k;
        miral_rotate_uv, k, inputFile=inputFile;
    }
}

/***************************************************************************/

func miral_rotate_uv(angle, inputFile=, outputFile=)
    /* DOCUMENT miral_rotate_uv(angle, inputFile=, outputFile=)

       DESCRIPTION

       PARAMETERS
       - angle     : 
       - inputFile : 
       - outputFile: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    /* Check the inputFile */
    message = "DATA FILE\n Enter the FITS data file to read";
    if (amdlibFileChooser(message, inputFile) == 0)
    {
        yocoError,"'inputFile' not specified correctly.";
        return 0;
    }
    
    yocoFileSplitName, inputFile, d,f,e;
    if(is_void(outputFile))
        outputFile = d+f+"_"+swrite(float(angle),format="%03.1f")+e;
    
    amplLoadOiData,oi,inOiFile=inputFile;
    
    for(k=1;k<=numberof(oi);k++)
    {
        if(!is_void(*oi.UCOORD(k)))
        {
            u = *oi.UCOORD(k);
            v = *oi.VCOORD(k);

            bsl = abs(u,v);
            ang = atan(v,u) + angle*deg2rad;

            *oi.UCOORD(k) = bsl*cos(ang);
            *oi.VCOORD(k) = bsl*sin(ang);
        }
        if(!is_void(*oi.U1COORD(k)))
        {
            u = *oi.U1COORD(k);
            v = *oi.V1COORD(k);

            bsl = abs(u,v);
            ang = atan(v,u) + angle*deg2rad;

            *oi.U1COORD(k) = bsl*cos(ang);
            *oi.V1COORD(k) = bsl*sin(ang);
        }
        if(!is_void(*oi.U2COORD(k)))
        {
            u = *oi.U2COORD(k);
            v = *oi.V2COORD(k);

            bsl = abs(u,v);
            ang = atan(v,u) + angle*deg2rad;

            *oi.U2COORD(k) = bsl*cos(ang);
            *oi.V2COORD(k) = bsl*sin(ang);
        }
    }
    amplSaveOiData,outputFile,oi;
}

/***************************************************************************/

func miral_compute_photocentre(imgCub, &Xphot, &Yphot, radius=, xyc=)
    /* DOCUMENT miral_compute_photocentre(imgCub, &Xphot, &Yphot, radius=, xyc=)

       DESCRIPTION
       Computes the photocentre of an image, with the possibility to select a
       smaller radius and a first estimate of the photocentre coordinates

       PARAMETERS
       - imgCub: input image cube
       - Xphot : photocenter X
       - Yphot : photocenter Y
       - radius: radius of photocenter zone to compte for
       - xyc   : X & Y of a recognizable blob in the image

       SEE ALSO
    */
{
    nbX = dimsof(imgCub)(2);
    nbY = dimsof(imgCub)(3);
    nbWlen = dimsof(imgCub)(0);

    X = array(indgen(nbX),nbY);
    Y = transpose(array(indgen(nbY),nbX));
    Xphot = Yphot = array(0.0,nbWlen);

    if(is_void(radius))
        radius = 100*nbX*nbY;

    for(k=1;k<=nbWlen;k++)
    {
        imgTmp = imgCub(..,k);
        centroid = where2(imgTmp==max(imgTmp));
        xc = centroid(1);
        yc = centroid(2);

        // Star must not be accounted in the bias subtraction!
        if(!is_void(xyc))
        {
            xc = xyc(1);
            yc = xyc(2);
        }

        if(!is_void(radius))
        {
            R = abs(X-xc,Y-yc);
            MASK = (R < radius);

            // Barycentre
            tmp = MASK*imgTmp;
            
            XC = sum(tmp * X) / sum(tmp);
            YC = sum(tmp * Y) / sum(tmp);
        }
        else
        {
            XC = xc;
            YC = yc;
        }
        Xphot(k) = XC;
        Yphot(k) = YC;
    }
}

/***************************************************************************/ 

func miral_recenter_imgcube(imgCub, &newImgCub, radius=, xyc=, allWlen=)
    /* DOCUMENT miral_recenter_imgcube(imgCub, &newImgCub, radius=, xyc=, allWlen=)

       DESCRIPTION
       Recenter an image cube made from V^2 and CP data

       PARAMETERS
       - imgCub   : input image cube [nx, ny, nwlen]
       - newImgCub: output image cube, offseted
       - radius   : radius of the interest zone where the photocentre is computed
       - xyc      : a rough estimate of image photocentre
       - allWlen  : 

       SEE ALSO
    */
{
    if(is_void(xyc))
        xyc = [];

    if(is_void(radius))
        radius = 2.5;

    miral_compute_photocentre, imgCub,
        Xphoto, Yphoto, radius=radius, xyc=xyc;

    DIMx = dimsof(imgCub)(2);
    DIMy = dimsof(imgCub)(3);
    DIMw = dimsof(imgCub)(4);
    imgX = array(indgen(DIMx)-1,DIMy);
    imgY = transpose(array(indgen(DIMy)-1,DIMx));

    newImgCub = array(0.0,dimsof(imgCub));

    if(is_void(xyc))
        xyc = [0.0,0.0];
    for(k=1;k<=DIMw;k++)
    {
        imgTmp = imgCub(..,k);
        newx   = imgX - Xphoto(k) + DIMx/2 + xyc(1);
        newy   = imgY - Yphoto(k) + DIMy/2 + xyc(2);
        newImgCub(..,k) = interp2(imgY, imgX, imgTmp, newy, newx);
    }
}

/***************************************************************************/ 

func miral_compute_imgCont(imgCub, &imgCont, &imgCen, &imgRms, selectedFrames=, recenter=, radius=, xyc=, med=)
    /* DOCUMENT miral_compute_imgCont(imgCub, &imgCont, &imgCen, &imgRms, selectedFrames=, recenter=, radius=, xyc=, med=)

       DESCRIPTION
       Given a continuum region, compute the averaged image over many
       wavelengths

       PARAMETERS
       - imgCub        : input image cube [nx, ny, nwlen]
       - imgCont       : output continuum image
       - imgCen        : centered image cube
       - imgRms        : 
       - selectedFrames: the frames which should be used
       - recenter      : recenter (or not). Defaults 0
       - radius        : see miral_recenter_imgcube
       - xyc           : see miral_recenter_imgcube
       - med           : use median instead of average for the continuum image

       SEE ALSO: miral_recenter_imgcube
    */
{
    if(is_void(xyc))
        xyc = [0.0,0.0];

    if(recenter==1)
    {
        miral_recenter_imgcube,imgCub, imgCen, radius=radius, xyc=xyc;
        imgCub = imgCen;
    }

    DIM     = dimsof(imgCub)(2);
    nbWlen  = dimsof(imgCub)(0);
    imgCont = array(0.0,DIM,DIM);

    if(is_void(selectedFrames))
    {
        selectedFrames = indgen(nbWlen);
    }

    imgInCont = imgCub(..,selectedFrames);
    
    if(med==1)
        imgCont = median(imgInCont,0);
    else
        imgCont = imgInCont(..,avg);

    imgRms = imgInCont(..,rms);
}

/***************************************************************************/

func miral_plot_imgCube(file, win, &imgName, title=, type=, srt=, idxCont=, idxRed=, idxGreen=, idxBlue=, subtractCont=, limit=)
    /* DOCUMENT miral_plot_imgCube(file, win, &imgName, title=, type=, srt=, idxCont=, idxRed=, idxGreen=, idxBlue=, subtractCont=, limit=)

       DESCRIPTION
       Plot an image cube
       
       PARAMETERS
       - file        : input file
       - win         : window to use
       - imgName     : output image name
       - title       : title of the plot
       - type        : 
       - srt         : 
       - idxCont     : 
       - idxRed      : 
       - idxGreen    : 
       - idxBlue     : 
       - subtractCont: 
       - limit       : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(type))
        type = "cont";
            
    miral_read_imgCube,file,img,x,y,w;

    whereNonZero = where(img(sum,sum,)!=0);
    
    if(!is_void(idxCont))
        miral_compute_imgCont, img, imgCont, imgCen, selectedFrames=idxCont, recenter=, radius=, xyc=, med=1;

    if(subtractCont)
        img(..,whereNonZero) = img(..,whereNonZero) - imgCont(..,-);

    if(!is_void(srt))
        img = img^srt;
    if(!is_void(logz))
        img = log(img+(img==0));
    
    X = x*rad2mas;
    Y = y*rad2mas;
    nw = numberof(w);
    
    if(type=="water")
    {
        yocoNmCreate,win,1, style="nobox.gs", square=1,width=800,height=800, fx=1,landscape=1;
        plotImgCubeWaterfall,img;
    }
    else if(type=="tile")
    {
        ratio = 1.2;
        NN1 = int(1./ratio*ceil(sqrt(nw)));
        NN2 = int(ratio*ceil(sqrt(nw)));
        yocoNmCreate,win, NN2, NN1, style="nobox.gs", square=1,width=1200,height=800, fx=1,landscape=1,dx=0,dy=0;
        for(k=1;k<=nw;k++)
        {
            plsys,k;
            pli,img(,,k);
        }
    }
    else if(type=="cont")
    {
        fma;
        pli,img(..,idxCont)(..,avg), min(X), min(Y), max(X), max(Y);
        yocoAstroPlotNorthEast,0.9*min(X),0.9*min(Y),
            0.15*abs(max(X)-min(X)),color="white";
        limits,max(X),min(Y);
        xytitles,"!a (mas)","!d (mas)";
        palette,"heat.gp";
    }
    else if(type=="anim")
    {
        for(k=0;k<=nw;k++)
        {
            yocoNmCreate,win,1, style="work.gs", square=1,width=800,height=800, fx=1,landscape=1,wait=1;
            
            if(allof(img(..,k)==0))
                continue;
            
            fma;
            if((k==0)&&(subtractCont==1))
                pli,imgCont^(1./3.), min(X), min(Y), max(X), max(Y)// ,
                    // cmin=-0.1*max(imgCont),cmax=max(imgCont);
                else
                    pli,img(..,k), min(X), min(Y), max(X), max(Y),
                        cmin=-0.1*max(img(..,k)),cmax=max(img(..,k));
            plg,0,0,type="none",marker='\2',color="green";
            yocoAstroPlotNorthEast,0.9*min(X),0.9*min(Y),
                0.15*abs(max(X)-min(X)),color="white";
            if(is_void(limit))
                limits,max(X),min(Y);
            else
                limits,limit(1),limit(2),limit(3),limit(4);
            xytitles,"!a (mas)","!d (mas)";
            palette,"heat.gp";
            
            if(!is_void(title))
                pltitle,title;
            
            yocoFileSplitName,file,d,f,e;
            imgName=d+strpart(f,1:16)+strpart(f,-10:0)+"_ImageLambda"+swrite(format="%02ld",k);
            hcps,imgName+".ps";
            system,"convert  "+imgName+".ps "+imgName+".png";
        }
    }
    else
        error;
    
    if(!is_void(title))
        pltitle,title;
    
    yocoFileSplitName,file,d,f,e;
    imgName=d+f+"_Image";
    hcps,imgName+".ps";
    system,"convert  "+imgName+".ps "+imgName+".png";
}

/***************************************************************************/

func miral_self_calibrate(inputFiles, inputImage, outputDir, overwrite=, wlenIdx=, calibrateAmp=, useVisAmp=, pse=, kill=, gain=, plot=)
    /* DOCUMENT miral_self_calibrate(inputFiles, inputImage, outputDir, overwrite=, wlenIdx=, calibrateAmp=, useVisAmp=, pse=, kill=, gain=, plot=)

       DESCRIPTION
       Self-calibration step for MIRA-L

       PARAMETERS
       - inputFiles  : input oifits files
       - inputImage  : input image cube (the hybrid map)
       - outputDir   : output data directory (the synthetic data)
       - overwrite   : overwrite or not already-existing files
       - wlenIdx     : mask to use only selected wavelengths for self-calibration of the differential phase
       - calibrateAmp: whether or not calibrate the amplitude also
       - useVisAmp   : Use V2 or visamp to get the visibility amplitude
       - pse         : Pause between plots
       - kill        : 
       - gain        : 
       - plot        : 

       SEE ALSO
    */
{
    if(is_void(pse))
        pse = 0;

    if(is_void(gain))
        gain = 1.0;

    if(is_void(kill))
        kill=1;

    if(is_void(plot))
        plot=1;

    if(plot)
        if(kill==1)
        {
            winkill,11;
            winkill,12;
            winkill,13;
            window,13,width=3,height=3,wait=1;
            yocoNmCreate,12,3,1,landscape=1,width=800,height=600, wait=1,dx=0.1, V = [0.1,0.9,0.12,0.70],fx=1;
            if(calibrateAmp)
                yocoNmCreate,11,2,2,landscape=1,width=800,height=600, wait=1,dx=0.1, V = [0.1,0.9,0.12,0.70],fy=1,dy=0.1;
            else
                yocoNmCreate,11,1,2,landscape=1,width=800,height=600, wait=1,dx=0.1, V = [0.1,0.9,0.12,0.70],fy=1,dy=0.1;
        }
    
    if(!open(outputDir,"r",1))
    {
        yocoLogWarning,"Directory "+outputDir+" does not exist! Creating...";
        mkdir,outputDir;
    }

    // Load OIFITS files
    // amplLoadSciFiles, sci, nightDir=dataDir, calibrated=1;
    amplLoadOiDatas, sci, inOiFile=inputFiles;

    // Load input image cube
    yocoLogInfo,"Reading image cube...";
    miral_read_imgCube, inputImage, image, X, Y, W;
    dimz = dimsof(image);
    
    if(numberof(dimsof(image))==3)
        image = [image];

    // Compute FFT for each wavelength, with zero-padding
    yocoLogInfo,"FFTing the image...";
    imageFFT_INIT(image, X, Y, U_FFT, V_FFT, FFT, upSample=0);
    dimzf = dimsof(FFT);
    
    nbObs = numberof(sci);
    if(plot)
    {
        window,11,width=500,height=500,wait=1;
        fma;
        window,12,width=500,height=500,wait=1;
        fma;
    }
    
    yocoLogInfo,"Starting self-calibration...";
    ///////////////////////////
    // Self-calibrate phase
    ///////////////////////////    
    for(kbs=1; kbs<=nbObs; kbs++)
        //kbs=1;
    {
        write,"Observation nr.", kbs;

        yocoLogInfo,"Getting differential phase...";        
        // Get differential phase
        dPhi    = *sci(kbs).VISPHI;// * pi / 180;
        dPhiErr = *sci(kbs).VISPHIERR;// * pi / 180;

        if(is_void(dPhi))
            continue;
        
        // Get wavelength
        wlen    = *sci(kbs).EFF_WAVE;

        // Get bases
        if(!is_void(dPhi))
            nbBases = dimsof(dPhi)(3);
        
        U       = *sci(kbs).UCOORD;
        V       = *sci(kbs).VCOORD;

        // interpolate image cube in wavelength direction
        if(numberof(W)>1)
        {
            interpU      = U_FFT;
            interpV      = V_FFT;
            interpFFT    = array(complex,dimzf(2),dimzf(3),numberof(wlen));
            interpFFT.re = interp(FFT.re, W, wlen, 0);
            interpFFT.im = interp(FFT.im, W, wlen, 0);
        }
        // Case where there is only one wavelength (e.g. continuum image)
        else
        {
            interpU   = U_FFT;
            interpV   = V_FFT;
            interpFFT = array(FFT,numberof(wlen));
        }

        // Self-calibrate phase
        selfPhi = selfPhiErr = selfu = selfv = [];
        kCnt=0;
        for(kBas=1;kBas<=nbBases;kBas++)
        {
            write,kBas;
            kCnt++;

            grow,selfu,U(kBas);
            grow,selfv,V(kBas);
            
            // Interpolate (bilinear) FFT at UV coordinates of observation
            visModel = imageFFT_COMPUTE(U(kBas), V(kBas), wlen,
                                        interpU, interpV, interpFFT);

            // Invert phase
            visModel.im = -visModel.im;
            
            // Computes phase of model
            phiModel = atan(visModel.im, visModel.re);

            if(!is_void(dPhi))
            {
                if(plot)
                {
                    // Plot data from the model and the observation
                    window,11;
                    plsys,1;
                    fma;
                    plg, phiModel, wlen, color="green";
                    limits,,,-pi,pi;
                }
            
                // First of all, remove any residual slope or offset of
                // observed differential phase
                // Recomputes visibility with all weight same for all wlen
                visObs      = exp(1i * dPhi(,kBas));
                // Computes OPD based on phases
                pistObs     = ComputePistonAIPS(visObs(wlenIdx), wlen(wlenIdx));
                visObsNoP   = visObs * exp(-2i*pi*pistObs/wlen);
                avgPhiObs   = atan(avg(visObsNoP).im,avg(visObsNoP).re);
                visObs      = visObsNoP * exp(-1i*avgPhiObs);
                dPhi(,kBas) = atan(visObs.im, visObs.re);
            
                // Skip this step if dphi is crap (rms > 1 radian)
                if(dPhi(rms,kBas)>1.5)
                {
                    if(plot)
                    {
                        plg, dPhi(,kBas),wlen, color="red";
                    }
                    yocoLogWarning,dPhi(rms,kBas);
                    //    pause,pse;
                    continue;
                }

                if(plot)
                {
                    // Plot result
                    //plg, 2*pi*pistObs/wlen + avgPhiObs, wlen, type="dash";
                    plg, dPhi(,kBas),wlen, color=[128,128,128];
                    plg, dPhi(wlenIdx,kBas),wlen(wlenIdx), color=[128,128,128],width=10;
                }

                // Computes OPD and offset on the model visibilities
                pist   = ComputePistonAIPS(visModel(wlenIdx), wlen(wlenIdx));
                visNoP = visModel(wlenIdx) * exp(-2i*pi*pist/wlen(wlenIdx));
                avgPhi = atan(avg(visNoP).im,avg(visNoP).re);

                // Plot result
                x = 2*pi*pist/wlen + avgPhi;
                if(plot)
                {
                    plg, mod(x,-pi,pi), wlen, type="dot";
                }

                // Apply found model OPD and offset to the observed differential phase
                selfVis = visObs * exp(1i*(gain*x));
                grow,selfPhi, [atan(selfVis.im, selfVis.re)];
            
                if(plot)
                {
                    plg, selfPhi(,0), wlen, color="red";
                    pltitle,"Gray: observations\nRed: self-cal\nGreen: Image";
                    xytitles,,"!f (rad)";
                    limits,,,-pi,pi;
                }

                // Here computes error
                deltaVis = selfVis * conj(visModel);
                if(numberof(W)>1)
                    deltaPhi = atan(deltaVis.im, deltaVis.re);
                else
                    deltaPhi = median(abs(atan(deltaVis.im, deltaVis.re)));

                grow,selfPhiErr, [sqrt((dPhiErr(,kBas)^2 + deltaPhi^2)/2)];
            
                if(plot)
                {
                    window,11;
                    plsys,2;
                    plg,dPhiErr(,kBas),wlen, color="red";
                    plg,selfPhiErr(,0),wlen, color="blue";
            
                    pltitle,"Red: data error\nBlue: self-cal error";
                    xytitles,,"!s_!f_ (rad)";
                    limits,,,0,pi;
                }
            
                visSelfCal = exp(1i * selfPhi(,0));
            }

            if(plot)
            {
                window,13;
                fma;
            }
        }
        
        // Store result in data structure
        sci(kbs).VISPHI    = &selfPhi;
        sci(kbs).VISPHIERR = &selfPhiErr;
        
        // sci(kbs).UCOORD = &selfu;
        // sci(kbs).VCOORD = &selfv;
                
        // oiStruct   = sci(kbs);
        // inputFile  = sci(kbs).hdr.file;
        // yocoFileSplitName,inputFile,d,f,e;
        // outputFile = outputDir + yocoStrReplace(f, ["--","--","--","--"],
        //                                         ["-","-","-","-"]) + e;
        // Save result in oi fits file
        // amplSaveOiData(outputFile, oiStruct);
    }
    
    //////////////////////////////
    // Self-calibrate amplitude
    //////////////////////////////
    if(calibrateAmp==1)
    {
        for(kbs=1; kbs<=nbObs; kbs++)
            //kbs=1;
        {
            write,"Observation nr.", kbs;
        
            // Get Squared visibilities
            vis2    = *sci(kbs).VIS2DATA;
            vis2Err = *sci(kbs).VIS2ERR;
        
            // Get complex amplitude
            visAmp    = *sci(kbs).VISAMP;
            visAmpErr = *sci(kbs).VISAMPERR;

            if(is_void(vis2)&&is_void(visAmp))
                continue;
                
            // Decide whether to use amp from vis2 or from visamp.
            // Default is to use vis2
            if(((useVisAmp==1)||(is_void(vis2)))&&(!is_void(visAmp)))
            {
                obsAmp    = visAmp;
                obsAmpErr = visAmpErr;
                if(is_void(vis2))
                    yocoLogWarning,"No vis2, using visamp by default...";
            }
            else
            {
                obsAmp    = yocoMathSignedSqrt(vis2);
                obsAmpErr = vis2Err / (2 * (obsAmp+0.001*(obsAmp==0)));
            }
        
            // Get wavelength
            wlen    = *sci(kbs).EFF_WAVE;

            // Get bases
            if(!is_void(vis2))
                nbBases = dimsof(vis2)(3);
            
            U       = *sci(kbs).UCOORD;
            V       = *sci(kbs).VCOORD;
            
            // interpolate image cube in wavelength direction
            if(numberof(W)>1)
            {
                interpU      = U_FFT;
                interpV      = V_FFT;
                interpFFT    = array(complex,dimzf(2),dimzf(3),numberof(wlen));
                interpFFT.re = interp(FFT.re, W, wlen, 0);
                interpFFT.im = interp(FFT.im, W, wlen, 0);
            }
            // Case where there is only one wavelength (e.g. continuum image)
            else
            {
                interpU   = U_FFT;
                interpV   = V_FFT;
                interpFFT = array(FFT,numberof(wlen));
            }
            
            selfAmp = selfAmpErr = selfu = selfv = [];
            kCnt=0;
            for(kBas=1;kBas<=nbBases;kBas++)
            {
                write,kBas;
                kCnt++;

                grow,selfu,U(kBas);
                grow,selfv,V(kBas);
            
                // Interpolate (bilinear) FFT at UV coordinates of observation
                visModel = imageFFT_COMPUTE(U(kBas), V(kBas), wlen,
                                            interpU, interpV, interpFFT);
            
                // Computes amplitude of model
                ampModel = abs(visModel);

                if(plot)
                {
                    // Self-calibrate amplitudes
                    window,11;
		    fma;
                    plsys,3;
                    plg, ampModel, wlen, color="green";
                    plg, obsAmp(,kBas),wlen, color=[128,128,128];
                    plg, obsAmp(wlenIdx,kBas),wlen(wlenIdx), color=[128,128,128],width=10;
                }

                // renormalize visibility to the model value
                avgVisObs = obsAmp(wlenIdx,kBas)(avg);
                avgVisMod = ampModel(avg);
                ga        = avgVisMod / avgVisObs;
                gb        = (ga-1)*gain+1;

                grow, selfAmp, [obsAmp(,kBas) * gb];
                if(plot)
                {
                    plg, selfAmp(,0), wlen, color="red";
                }
                
                // Here computes error
                if(numberof(W)>1)
                    deltaVis          = selfAmp(,0) - ampModel;
                else
                    deltaVis          = median(abs(selfAmp(,0) - ampModel));
                
                grow,selfAmpErr, [sqrt((obsAmpErr(,kBas)^2 + deltaVis^2)/2)];

                if(plot)
                {
                    xytitles,"Wavelength (meters)","Vis.";
                    limits,,,-0.2,1.2;
                
                    window,11;
                    plsys,4;
                    plg,obsAmpErr(,kBas),wlen, color="red";
                    plg,selfAmpErr(,0),wlen, color="blue";
                    xytitles,"Wavelength (meters)","!s_Vis";
                    limits,,,0,1;

                    window,12;
                    plsys,2;
                    idxCont    = indgen(5)+10;
                    spFreq     = abs(U(kBas)/wlen, V(kBas)/wlen);

                    plg, visModel(idxCont).re, spFreq(idxCont),type="none",
                        color="red",marker='\2';
                    plg, visModel(idxCont).im, spFreq(idxCont),type="none",
                        color="blue",marker='\2';
            
                    pltitle,"Red: img real part\nBlue: img im. part";
                    xytitles,"Sp. freq. (rad^-1^)","No unit";
                    limits,0,,-1,1;

                    window,12;
                    plsys,1;
                    plg, abs(visModel(idxCont).re, visModel(idxCont).im),
                        spFreq(idxCont),type="none", color="red",marker='\2';
            
                    pltitle,"Red: image amplitude";
                    xytitles,"Sp. freq. (rad^-1^)","No unit";
                    limits,0,,-1,1;
                }

                if(!is_void(selfPhi))
                {
                    visSelfCal = selfAmp(,0) * exp(1i * selfPhi(,0));
                    if(plot)
                    {
                        window,12;
                        plsys,3;
                        plg, visSelfCal(idxCont).re, spFreq(idxCont),type="none",
                            color="magenta",marker='\2';
                        plg, visSelfCal(idxCont).im, spFreq(idxCont),type="none",
                            color="green",marker='\2';
            
                        pltitle,"Pink: self-cal real part\nGreen: self-cal im. part";
                        xytitles,"Sp. freq. (rad^-1^)","No unit";
                        limits,0,,-1,1;
                    }
                }
            }

            if(plot)
            {
                window,13;
                fma;
            }
        }

        // Store result in data structure        
        // sci(kbs).UCOORD = &selfu;
        // sci(kbs).VCOORD = &selfv;
        
        // Store result in data structure
        sci(kbs).VISAMP    = &selfAmp;
        sci(kbs).VISAMPERR = &selfAmpErr;
        
        // oiStruct   = sci(kbs);
    }
    
    for(kbs=1; kbs<=nbObs; kbs++)
        //kbs=1;
    {
        write,kbs;

        // if(kbs==34)
        //     error;

        if((*sci(kbs).VISAMP   == [])&&
           (*sci(kbs).VISPHI   == [])&&
           (*sci(kbs).VIS2DATA == [])&&
           (*sci(kbs).T3PHI    == []))
        {
            error;
        }
        // Save result in oi fits file
        inputFile  = sci(kbs).hdr.file;
        yocoFileSplitName,inputFile,d,f,e;
        outputFile = outputDir +
            yocoStrReplace(f, ["--","--","--","--"],
                           ["-","-","-","-"]) +
            "_" + pr1(kbs) + e;
        amplSaveOiData(outputFile, sci(kbs));
    }

}

/***************************************************************************/

func mod(x, limin, limax)
    /* DOCUMENT mod(x, limin, limax)

       DESCRIPTION

       PARAMETERS
       - x    : 
       - limin: 
       - limax: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    idx = where(x<limin);
    while(numberof(idx)>0)
    {
        x(idx) = x(idx) + (limax-limin);
        idx = where(x<limin);
    }
    
    idx = where(x>limax);
    while(numberof(idx)>0)
    {
        x(idx) = x(idx) - (limax-limin);
        idx = where(x>limax);
    }
    return x;
}

/***************************************************************************/

func miral_reconstruct_multiwlen(fitsFiles, &CHI2, &MU2, &IMGCUB, &REGUL, &outputImage, target=, img0=, medianImg0=, popVis=, popPhi=, popV2=, popV3=, copV2toV=, copVtoV2=, multV2toV=, binSize=, recenter=, radius=, xyc=, mu=, imgSize=, pixSize=, fovFWHM=, view=, verb=, prior=, projAngle=, regul=, regulParams=, nImg=, niter=, niterFirst=, cleanup_bad_data=, reverse=, kill=, chi2Thr=, plot=)
    /* DOCUMENT miral_reconstruct_multiwlen(fitsFiles, &CHI2, &MU2, &IMGCUB, &REGUL, &outputImage, target=, img0=, medianImg0=, popVis=, popPhi=, popV2=, popV3=, copV2toV=, copVtoV2=, multV2toV=, binSize=, recenter=, radius=, xyc=, mu=, imgSize=, pixSize=, fovFWHM=, view=, verb=, prior=, projAngle=, regul=, regulParams=, nImg=, niter=, niterFirst=, cleanup_bad_data=, reverse=, kill=, chi2Thr=, plot=)

       DESCRIPTION
       Reconstruct a multiwavelength image cube from interferometric data,
       slicing into subsets the wavelengths range and reconstructing the
       image intependently wavelength-by-wavelength.

       PARAMETERS
       - fitsFiles       : input oifits files
       - CHI2            : Output chi2
       - MU2             : 
       - IMGCUB          : 
       - REGUL           : 
       - outputImage     : output image cube
       - target          : select a target if there are several in the oifits file
       - img0            : reference (initial) image
       - medianImg0      : 
       - popVis          : do not take (=1) or take (=0) into account the OI_VIS table
       - popPhi          : 
       - popV2           : do not take (=1) or take (=0) into account the OI_VIS2 table
       - popV3           : do not take (=1) or take (=0) into account OI_T3 table
       - copV2toV        : if =1, copy the content of OI_VIS2 table to OI_VIS table (visamp)
       - copVtoV2        : if =1, copy the content of OI_VIS table to OI_VIS2
       - multV2toV       : 
       - binSize         : wavelength-bisize on which the image is reconstructed
       - recenter        : flag for centering images at each wlen step
       - radius          : see miral_recenter_imgcube
       - xyc             : see miral_recenter_imgcube
       - mu              : regularization values (see miral_solve_multiRgl)
       - imgSize         : image size (pixels)
       - pixSize         : pixel size (radians)
       - fovFWHM         : field of view FDWHM (Gaussian prior)
       - view            : plot graphics (or not) defaults 0.
       - verb            : verbosity (default 0)
       - prior           : optional prior
       - projAngle       : projection angle for 1D-imaging (See e.g. Ohnaka et al. 2011)
       - regul           : can be any of "l2l1_smoothness", "smoothness", "quadratic", "qsmooth", "entropy", "totvar"
       - regulParams     : 
       - nImg            : Number of images to reconstruct for Monte Carlo applications
       - niter           : Number of iterations in MIRA
       - niterFirst      : 
       - cleanup_bad_data: MIRA option for removing bad data
       - reverse         : Go through wavelength in reverse order
       - kill            : 
       - chi2Thr         : 
       - plot            : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(reverse))
        reverse=0;
    
    if(is_void(plot))
        plot=1;
    
    if(is_void(copV2toV))
        copV2toV=0;
    
    if(is_void(popPhi))
        popPhi=0;
    
    if(is_void(chi2Thr))
        chi2Thr=2;
    
    if(is_void(kill))
        kill=1;

    if(plot)
    {
        if(kill==1)
        {
            winkill,31;
            winkill,12;
        }
    
        yocoNmCreate,31,3,2,landscape=1,width=800,height=600,wait=1,square=1,dy=0.07;
        yocoNmCreate,12,3,2,landscape=1,width=800,height=600,wait=1,square=1,dy=0.07;
    }
    
    if(is_void(mu))
        mu = [1e6, 1e5, 1e4];

    if(mu=="random")
        MU=1;

    if(is_void(niter))
        niter=200;
    
    if(is_void(nImg))
        nImg = 1;
    
    fitsFiles = fitsFiles(sort(fitsFiles));
    
    // get the wavelength range of data and bandwidth
    miral_slice_wlens, fitsFiles, waves, bands;
    nWlen    = numberof(waves);

    if(!is_void(binSize))
    {
        // reshape wavelengths given the binsize
        nWlen = int(nWlen / binSize);
        if(nWlen<=1)
        {
            nWlen = 1;
            bands = max(waves)-min(waves);
            waves = avg(waves);
        }
        else
        {
            bands = array(avg(bands)*binSize, nWlen);
            waves = span(min(waves)+bands(1)/2, max(waves)-bands(0)/2, nWlen);
        }
    }

    // Compute optimal fov, dimension and pixel size
    if(is_void(imgSize))
    {
        imgSize = miral_compute_dim(fitsFiles);
        yocoLogWarning,"No image size given, setting it to minimum "+pr1(imgSize)+"pixels";
    }
    write,"Image size",imgSize,"pixels";

    if(is_void(pixSize))
    {
        pixSize = miral_compute_pixelsize(fitsFiles)*MIRA_MILLIARCSECOND;
        yocoLogWarning,"No pixel size given, setting it to minimum "+pr1(pixSize/MIRA_MILLIARCSECOND)+"mas";
    }
    write,"Pixel size",pixSize*rad2mas,"mas";

    fov = pixSize * imgSize;

    if(is_void(fovFWHM))
    {
        fovFWHM = fov/3.0;
    }
    write,"Effective field of view",fovFWHM*rad2mas,"mas";

    x = array(span(-fov/2,fov/2,imgSize),imgSize);
    y = transpose(x);
       
    if(is_void(img0))
        img0 = "random";

    if(is_void(prior))
    {
        xy = grow(array(x,1),array(y,1));
        // prior = miral_gauss2d(xy,[1.0,0,0,fovFWHM,fovFWHM]);
        prior = gauss2d(xy,[1.0,0,0,fovFWHM,fovFWHM]);
    }

    X = (indgen(imgSize)-imgSize/2.0)*pixSize;
    Y = (indgen(imgSize)-imgSize/2.0)*pixSize;

    // Initialize image cubes
    IMGCUB = array(0.0, imgSize, imgSize, nImg, nWlen);
    CHI2   = array(0.0, nImg, nWlen);
    MU2    = array(0.0, nImg, nWlen);
    REGUL  = array("", nImg, nWlen);
    
    /////////////////////////////////////////////////////////////////
    // Loop on N realizations
    /////////////////////////////////////////////////////////////////
    for(kImg=1;kImg<=nImg;kImg++)
    {
        write,"*************************************************************";
        write, "Treating image "+pr1(kImg)+", out of "+pr1(nImg)+ "...";

        dimzImg=dimsof(img0)(1);
        if((img0=="random")||(img0==""))
        {
            rd = random();
            if(rd<1./2.)
            {
                XX = array((indgen(imgSize)-imgSize/2.), imgSize);
                YY = transpose(XX);
                RR = abs(XX,YY);
                // Uniform disk startup
                img00 = abs((RR<random()*imgSize/2)+
                            0.1*random_n(imgSize,imgSize));
            }
            else if((rd>=1./2.)&&(rd<1.))
            {
                XX = array((indgen(imgSize)-imgSize/2.), imgSize);
                YY = transpose(XX);
                RR = abs(XX,YY);
                //Gaussian startup
                img00 = abs(gauss(RR,[1,0,random()*imgSize/2]) +
                            0.1*random_n(imgSize,imgSize));
            }
            else
            {
                // Set initial image as a random image,
                // with a brightest point in the centre.
                img00 = random(imgSize,imgSize); 
                // img0(imgSize/2,imgSize/2) += 5.0;
            }
        }
        else if(is_string(img0))
        {
            im = cfitsRead(img0);
            dimzImg=dimsof(im)(1);
            if(dimzImg==2)
                img00 = im;
            else if(dimzImg==3)
            {
                miral_read_imgCube, img0, img00, imx, imy, imw, imm, imc, imr;
            }
            else
                error;
        }
        else if(is_array(img0))
        {
            img00 = img0;
        }
        else
            error;

        dimzPri=dimsof(prior)(1);
        if(is_string(prior))
        {
            if(prior=="random")
            {
                fwhm = fovFWHM*random();
                xy = grow(array(x,1),array(y,1));
                pri00 = gauss2d(xy,[1.0,0,0,fwhm,fwhm]);
                dimzPri=dimsof(pri00)(1);
            }
            else
            {
                pr = cfitsRead(prior);
                dimzPri=dimsof(pr)(1);
                if(dimzPri==2)
                    pri00 = prior;
                else if(dimzPri==3)
                {
                    miral_read_imgCube, prior, pri00, prx, pry, prw, prm, prc, prr;
                }
                else
                    error;
            }
        }
        else
        {
            dimzPri=dimsof(prior)(1);
            pri00=prior;
        }
        
        
        if(is_void(regul))
            regul = "entropy";
        
        /* Choose a suitable regularization method: */
        if(regul=="random")
        {
            r = int(random()*6+1);
            reg = ["l2l1_smoothness", "smoothness","totvar","quadratic", "qsmooth", "entropy"];
            regul0 = reg(r);
        }
        else
            regul0 = regul;

        yocoFileSplitName,fitsFiles(1),d,f,e;

        if(niter=="random")
            niter0 = 10+int(100*random());
        else
            niter0 = niter;

        if(is_void(outputImage))
            outputName = d+f+"_"+regul0+"_"+pr1(niter)+"_"+pr1(pixSize*rad2mas)+"_"+pr1(imgSize);
        else
        {
            yocoFileSplitName,outputImage,d,f,e;
            outputName = d+f;
        }
        
        if(reverse)
        {
            if(reverse=="random")
            {
                rev=random();
                if(rev>0.5)
                {
                    wv1 = nWlen;
                    wvn = 1;
                    wvi = -1;
                }
                else
                {
                    wv1 = 1;
                    wvn = nWlen;
                    wvi = 1;
                }
            }
            else
            {
                wv1 = nWlen;
                wvn = 1;
                wvi = -1;
            }
        }
        else
        {
            wv1 = 1;
            wvn = nWlen;
            wvi = 1;
        }

        if(MU==1)
            mu = array(10^(4*random()+2),3);
            
        kWave = wv1;
        /////////////////////////////////////////////////////////////////
        // Loop on wavelengths
        /////////////////////////////////////////////////////////////////
        for(kk=1;kk<=nWlen;kk++)
        {
            if(dimzPri==2)
                pri000 = pri00;
            else if(dimzPri==3)
                pri000 = pri00(..,kWave);
            else
                pri000=prior;
            
            // Set regularization here because the prior can change with wavelength
            if(regul0=="l2l1_smoothness")
            {
                rgl = rgl_new("l2l1_smoothness");
                thres = regulParams;
                if(is_void(thres))
                    thres = 2e-2;
                rgl_config, rgl,"threshold",thres;
            }
            else if(regul0=="smoothness")
            {
                rgl = rgl_new("smoothness");
            }
            else if(regul0=="quadratic")
            {
                rgl = rgl_new("quadratic"); 
                if(!is_void(pri000))
                    rgl_config, rgl, "W", linop_new("diagonal",1.0 / (yocoMathSignedSqrt(pri000)+1e99*(pri000==0)));
            }
            else if(regul0=="qsmooth")
            {
                rgl = rgl_new("qsmooth"); 
                if(!is_void(pri000))
                    rgl_config, rgl, "prior", pri000;
            }
            else if(regul0=="entropy")
            {
                rgl_config, (rgl = rgl_new("entropy")),
                    "type", "log", "normalized", 0n, "prior", pri000;
            }
            else if(regul0=="totvar")
            {
                rgl_config, (rgl = rgl_new("totvar")),
                    "epsilon", 1e-5, "isotropic",0;//, "prior", pri000;
            }
            else
                error,"error";
            
            write,regul,regul0;
            
            REGUL(kImg,kWave) = regul0;
            
            // Write current wavelength
            write, "Treating wavelength "+pr1(kWave)+
                " ("+pr1(waves(kWave)*1e6)+ " microns), out of "+
                pr1(nWlen)+ "...";
            write,"Regularization:",regul0;

            // Load data with right wavelength and bandwidth
            if(anyof(bands==0))
                bands = 2.3e-8;
            
            mh1 = mira_new( fitsFiles,
                            monochromatic=1,
                            eff_wave=waves(kWave),eff_band=bands(kWave),
                            target=target, quiet=0,
                            cleanup_bad_data=cleanup_bad_data);

            uf0=vf0=[];
            // Cleanup bad data
            if(!is_void(mh1.vis2))
            {
                grow,uf0, mh1.vis2.u*mas2rad*1000;
                grow,vf0, mh1.vis2.v*mas2rad*1000;
            }
            if(!is_void(mh1.vis))
            {
                h_set, mh1.vis,"phierr",mh1.vis.phierr + 1e10*(mh1.vis.phierr<=0);
                // not a number, especially on errors
                phierr = mh1.vis.phierr;
                isnan = where(phierr != phierr);
                if(numberof(isnan)!=0)
                {
                    phierr(where(isnan))=1e10;
                    h_set, mh1.vis,"phierr", phierr;
                }
                grow,uf0, mh1.vis.u*mas2rad*1000;
                grow,vf0, mh1.vis.v*mas2rad*1000;
            }
            if(!is_void(mh1.vis3))
            {
                // Negative errors
                h_set, mh1.vis3,"t3phierr",
                    mh1.vis3.t3phierr + 1e10*(mh1.vis3.t3phierr<=0);
                // not a number, especially on errors
                t3phierr = mh1.vis3.t3phierr;
                isnan = where(t3phierr != t3phierr);
                if(numberof(isnan)!=0)
                {
                    t3phierr(where(isnan))=1e10;
                    h_set, mh1.vis3,"t3phierr", t3phierr;
                }
            }

            // Copy the V2 structure into visibility amplitude
            if(copV2toV==1)
            {
                // Get squared visibility
                v2dat  = mh1.vis2.vis2data;
                v2err  = mh1.vis2.vis2err;
                uv2    = mh1.vis2.u;
                vv2    = mh1.vis2.v;

                // Compute amplitude according to V2
                sqrtv2 = yocoMathSignedSqrt(v2dat);
                amp    = sqrtv2;
                amperr = v2err / (2*(sqrtv2+(sqrtv2==0)));

                // Get phase
                phi    = mh1.vis.phi;
                phierr = mh1.vis.phierr;
                uf     = mh1.vis.u;
                vf     = mh1.vis.v;

                // match amplitudes and phases
                AMP = AMPERR = PHI = PHIERR = U = V = [];
                for(k=1;k<=numberof(phi);k++)
                {
                    match = where((uf(k)==uv2)&(vf(k)==vv2));
                    if(numberof(match)!=0)
                    {
                        grow,AMP,   amp(match(1));
                        grow,AMPERR,amperr(match(1));
                        grow,PHI,   phi(k);
                        grow,PHIERR,phierr(k);
                        grow,U,uf(k);
                        grow,V,vf(k);
                    }
                }
                
                hash   = miral_polar_to_cartesian(AMP, AMPERR,
                                                  PHI, PHIERR, mh1);
                cpx = AMP * exp(1i * PHI);
                re  = cpx.re;
                im  = cpx.im;
                
                h_set_copy,mh1.vis, "re",re;
                h_set_copy,mh1.vis, "im",im;
                // h_set_copy,mh1.vis, "re",hash.re;
                // h_set_copy,mh1.vis, "im",hash.im;
                h_set_copy,mh1.vis, "wii",hash.wii;
                h_set_copy,mh1.vis, "wri",hash.wri;
                h_set_copy,mh1.vis, "wrr",hash.wrr;
                h_set_copy,mh1.vis, "amp",   AMP;
                h_set_copy,mh1.vis, "amperr",AMPERR;
                // h_set_copy,mh1.vis, "phi",PHI;
                // h_set_copy,mh1.vis, "phierr",PHIERR;
                h_set_copy,mh1.vis, "u",U;
                h_set_copy,mh1.vis, "v",V;
                h_set_copy,mh1.vis, "sgn",sign(AMP);

                if(plot)
                {
                    window,12;
                    fma;
                    plsys,1;
                    plg, cpx.re, abs(U, V),type="none",color="red",marks='\2';
                    plg, cpx.im, abs(U, V),type="none",color="blue",marks='\2';
                    limits,,,-1,1;

                    plsys,2;
                    yocoPlotWithErrBars, AMP, AMPERR, abs(U, V),type="none";
                    limits,,,-1,1;
                }
                // popV2=0;
            }
            
            if(copVtoV2==1)
            {
                vdat = mh1.vis.amp;
                v2 = vdat^2;
                v2err = 2 * mh1.vis.amperr * v2;

                h_set_copy,mh1.vis2, "re",hash.re;
                h_set_copy,mh1.vis2, "im",hash.im;
                h_set_copy,mh1.vis2, "wii",hash.wii;
                h_set_copy,mh1.vis2, "wri",hash.wri;
                h_set_copy,mh1.vis2, "wrr",hash.wrr;
                h_set_copy,mh1.vis2, "amp",amp;
                h_set_copy,mh1.vis2, "amperr",amperr;

                if(plot)
                {
                    window,34;
                    fma;
                    plg, cpx.re, abs(mh1.vis.u, mh1.vis.v),type="none",color="red";
                    plg, cpx.im, abs(mh1.vis.u, mh1.vis.v),type="none",color="blue";

                    window,33;
                    fma;
                    yocoPlotWithErrBars, amp, amperr, abs(mh1.vis.u, mh1.vis.v),type="none";
                }

                popVis=1;

            }
            
            // Pop vis table for closure-phase image reconstruction
            if(popVis==1)
                h_pop, mh1, "vis";

            // Use the phases in the image reconstruction process.
            // In this case, V2 are put into vis table (as visibilities)
            // and vis2 table is "popped"
            if(popV2==1)
            {
                h_pop, mh1, "vis2";
            }

            if(popV3==1)
            {
                h_pop, mh1, "vis3";
            }
            
            /* Configure data instance for image reconstruction parameters (DIM is
               the number of pixels along the width and height of the restored
               image; FOV is the size of the corresponding field of view in
               radians; XFORM is the name of the method to approximate the Fourier
               transform, can be "exact" or "fft", default is "exact"): */
            // mira_config, mh1, dim=imgSize, pixelsize=pixSize, xform="fft";
            mira_config, mh1, dim=imgSize, pixelsize=pixSize, xform="exact";
            
            res = miral_get_best_medianImg(IMGCUB,CHI2,MU2,chi2Thr=chi2Thr);

            if(medianImg0)
                if(nallof(res==0) && !((img0=="random") && (kk==1)))
                {
                    dimzImg=2;
                    img00 = res;
                }

            if((kImg==1)&&(kWave==wv1)&&(!is_void(niterFirst)))
            {
                nit = niterFirst;
            }
            else
            {
                nit = niter0;
            }
            
            if((img0=="random")||(img0==""))
                img000=img00;
            else if(dimzImg==2)
                img000 = img00;
            else if(dimzImg==3)
                img000 = img00(..,kWave);
            else
                error;
            
            img1 = miral_solve_multiRgl(mh1, img000, rgl, chi2,
                                        maxeval=nit, mu=mu, prior=pri000,
                                        recenter=recenter, view=view,
                                        verb=verb);
            
            if(recenter==1)
            {
                miral_recenter_imgcube,array(img1, 1), newImgCub, radius=radius, xyc=xyc;
                img1 = newImgCub(..,1);
            }

            CHI2(kImg,kWave) = chi2;
            MU2(kImg,kWave)  = mu(0);
            write,"Chi squared",chi2, "mu",mu;
            write,"*************************************************************";
            IMGCUB(..,kImg,kWave) = img1;
            
            // if(view)
            {
                if(is_void(projAngle))
                {
                    imgMed = miral_get_best_medianImg(IMGCUB,CHI2,MU2,chi2Thr=chi2Thr);
                    imgAvg = miral_get_best_avgImg(IMGCUB,CHI2,MU2,chi2Thr=chi2Thr);

                    Xmas = X*rad2mas;
                    Ymas = Y*rad2mas;

                    if(plot)
                    {
                        window,31;
                        fma;
                        plsys,1;
                        pli,img000,min(Xmas),min(Ymas),max(Ymas),max(Xmas);
                        limits,max(Xmas),min(Xmas),min(Ymas),max(Ymas);
                        pltitle,"Start image";

                        plsys,2;
                        pli,pri000,min(Xmas),min(Ymas),max(Ymas),max(Xmas);
                        limits,max(Xmas),min(Xmas),min(Ymas),max(Ymas);
                        pltitle,"Prior image";
                    
                        plsys,4;
                        pli, imgMed,min(Xmas),min(Ymas),max(Ymas),max(Xmas);
                        limits,max(Xmas),min(Xmas),min(Ymas),max(Ymas);
                        pltitle,"Median image";
                    
                        // plsys,5;
                        // pli, imgAvg,min(Xmas),min(Ymas),max(Ymas),max(Xmas);
                        // limits,max(Xmas),min(Xmas),min(Ymas),max(Ymas);
                        // pltitle,"Average image";
                    
                        plsys,6;
                        pli, img1,min(Xmas),min(Ymas),max(Ymas),max(Xmas);
                        limits,max(Xmas),min(Xmas),min(Ymas),max(Ymas);
                        pltitle,"Last image ";
                        goodChi2 = CHI2(where(CHI2!=0));
                        minChi2 = min(goodChi2);

                        plsys,5;
                        rf0 = abs(uf0,vf0);
                        plg, vf0, uf0,type="none",marker='\2';
                        plg,-vf0,-uf0,type="none",marker='\2';
                        limits,max(rf0)*1.1,-max(rf0)*1.1,
                            -max(rf0)*1.1,max(rf0)*1.1;
                        pltitle,"UV coverage";
                    
                        plt,"Test      # "+pr1(kImg)+" / "+pr1(nImg)+"\n"+
                            "Lambda    # "+pr1(kWave)+" / "+pr1(wvn)+"\n"+
                            "Lambda    = "+pr1(waves(kWave)*1e6)+" micron\n"+
                            "Npix      = "+pr1(imgSize)+" \n"+
                            "Pixel Size= "+pr1(pixSize*rad2mas)+" mas\n"+
                            "FOV       = "+pr1(fovFWHM*rad2mas)+" mas\n"+
                            "Bin Size  = "+pr1(binSize)+" \n"+
                            "Regul.      "+yocoStrReplace(regul0)+" \n"+
                            "Mu        = "+pr1(mu(0))+" \n"+
                            "Niter     = "+pr1(niter0)+" \n"+
                            "Chi2      = "+pr1(chi2)+" \n"+
                            "Min. Chi2 = "+pr1(minChi2)+" \n"+
                            "Chi2 used # "+pr1(numberof(where(goodChi2<chi2Thr*minChi2)))+" \n" ,
                            0.72,0.71,justify="LT";

                        yocoNmXytitles,["!a (mas)","freq. (as^-1^)","!a (mas)"],"!d (mas)";
                    }
                }
                else
                {
                    x2 = x*cos(-projAngle)-y*sin(-projAngle);
                    y2 = x*sin(-projAngle)+y*cos(-projAngle);
                    projIma = interp2(y, x, img1, y2, x2);
                    projIma2 = interp2(y, x, median(IMGCUB(..,:kWave),0), y2, x2);
                    if(plot)
                    {
                        window,20,width=500,height=500,wait=1;
                        fma;
                        pli, projIma;
                    
                        window,21,width=500,height=500,wait=1;
                        fma;
                        plg, projIma(,sum)/sum(projIma), x(,1)*rad2mas;
                        plg, projIma2(,sum)/sum(projIma2), x(,1)*rad2mas, color="red";
                    }
                }
            }
            kWave+=wvi;
        }
    }
    
    if(nImg!=1)
    {
        wc2 = where(CHI2(,sum)<2*min(CHI2(,sum)));

        // Recenter (or not) image on its brightest pixel(s)
        if(recenter==1)
        {
            miral_recenter_imgcube,reform(IMGCUB,[3,imgSize,imgSize,nImg*nWlen]),SIMG,radius=radius;
            SIMG = reform(SIMG,[4,imgSize,imgSize,nImg,nWlen])
                }
        else
            SIMG = IMGCUB(..,srt,);
        
        imgMed = miral_get_best_medianImg(SIMG,CHI2,MU2,chi2Thr=chi2Thr);

        if(1)
        {
            // Median image with reasonable chi2
            outputImageMed = outputName+"_IMAGE_MEDIAN";
            miral_write_imgCube, outputImageMed+"_"+
                yocoStrReplace(timestamp(),[" ",":"],["",""])+
                ".fits", imgMed, X, Y, waves;

            if(plot)
            {
                wkll;
                window,41,wait=1;
                mira_plot_image,imgMed, mh1;
                pltitle,"median";
                hcps, outputImageMed+".ps";
            }
            
            // Average image with reasonable chi2
            outputImageAvg = outputName+"_IMAGE_AVERAGE";
            miral_write_imgCube, outputImageAvg+"_"+
                yocoStrReplace(timestamp(),[" ",":"],["",""])+
                ".fits", SIMG(..,wc2,)(..,avg,), X, Y, waves;
            if(plot)
            {
                window,42,wait=1;
                mira_plot_image, SIMG(..,wc2,)(..,avg,avg), mh1;
                pltitle,"average";
                hcps, outputImageAvg+".ps";
            }
            
            // RMS plot
            outputImageRms = outputName+"_IMAGE_RMS";
            miral_write_imgCube, outputImageRms+"_"+
                yocoStrReplace(timestamp(),[" ",":"],["",""])+
                ".fits", SIMG(..,wc2,)(..,rms,), X, Y, waves;
            if(plot)
            {
                window,43,wait=1;
                mira_plot_image, SIMG(..,wc2,)(..,rms,rms), mh1;
                pltitle,"RMS";
                hcps, outputImageRms+".ps";
            
                // Animation plot with images sorted by increasing chi2
                window,44,wait=1;
                for(k=1;k<=nImg;k++)
                {
                    fma;
                    mira_plot_image, SIMG(..,k,avg), mh1;
                    pltitle,"animation";
                    // pause,100;
                }
            
                // Chi2 as a function of mu
                yocoNmCreate,20,landscape=1,width=800,height=600, V = [0.15,0.9,0.15,0.73],fx=1,fy=1;
                fma;
            }
            regs = yocoStrRemoveMultiple(REGUL);
            for(k=1;k<=numberof(regs);k++)
            {
                wr = where(REGUL==regs(k));
                if(plot)
                {
                    plg,CHI2(wr,1),MU2(wr,1),type="none",marker='\2',color=colors(k);
                    plt,regs(k),1e1,50-5*k,tosys=1,color=colors(k);
                }
            }
            if(plot)
            {
                logxy,1,1;
                xytitles,"!m","!c^2";
                limits;
            
                outputImageChi2Mu = outputName+"_CHI2_f_MU";
                hcps,outputImageChi2Mu;
            }
        }
    }
    
    if(is_void(outputImage))
        outputImage = outputName+"_"+
            yocoStrReplace(timestamp(),[" ",":"],["",""])+
            "_IMAGE.fits";
    
    if(nImg==1)
        miral_write_imgCube, outputImage, IMGCUB(,,1,), X, Y, waves, MU2, CHI2, REGUL,img00,pri00;
    else if(nWlen==1)
        miral_write_imgCube, outputImage, IMGCUB(..,1), X, Y,, MU2, CHI2, REGUL,img00,pri00;
    else
        miral_write_imgCube, outputImage, IMGCUB, X, Y, waves, MU2, CHI2, REGUL,img00,pri00;

    
    fh = cfitsio_open(outputImage,"a"); // re-open the file to modify keys

    cfitsio_goto_hdu,fh,1;
    
    cfitsio_write_key, fh, "prior", !is_void(prior),
        "Use a prior or not";
    cfitsio_write_key, fh, "img0", !is_void(img0),
        "Use a start image or not";
    
    cfitsio_write_key, fh, "imgSize", imgSize, "Image size (pixels)";
    cfitsio_write_key, fh, "pixSize", pixSize, "Pixel size (rad)";
    cfitsio_write_key, fh, "nImg", nImg, "Number of images to reconstruct for Monte Carlo applications";
    cfitsio_write_key, fh, "binSize", binSize, "Wavelength binning";

    cfitsio_write_key, fh, "niter", niter, "Number of iterations in MIRA";
    cfitsio_write_key, fh, "niter1st", niterFirst, "Number of iterations for first reconstruction";

    cfitsio_write_key, fh, "regul", regul, "can be any of \"l2l1_smoothness\", \"smoothness\", \"quadratic\", \"qsmooth\", \"entropy\", \"totvar\"";
    cfitsio_write_key, fh, "mu", mu, "regularization value (see miral_solve_multiRgl)";

    cfitsio_write_key, fh, "medImg0", medianImg0,
        "Use the median of previous reconstructions as start image";
    cfitsio_write_key, fh, "chi2Thr", chi2Thr, "Chi2 Threshold for calculating median image";

    cfitsio_write_key, fh, "popVis", popVis, "skip the use of OI_VIS table";
    cfitsio_write_key, fh, "popV2",  popV2,  "skip the use of OI_VIS2 table";
    cfitsio_write_key, fh, "popV3",  popV3,  "skip the use of OI_T3 table";
    cfitsio_write_key, fh, "popPhi", popPhi, "skip the use of VISPHI column";
    cfitsio_write_key, fh, "copV2toV", copV2toV,
        "copy the OI_VIS2 table content to OI_VIS table VISAMP";
    cfitsio_write_key, fh, "copVtoV2", copVtoV2,
        "copy the OI_VIS VISAMP table content to OI_VIS2 table";

    cfitsio_write_key, fh, "recenter", recenter, "Center image on brightest pixel";
    cfitsio_write_key, fh, "radius", radius, "Radius on which centering photocenter is computed";
    // cfitsio_write_key, fh, "xyc", xyc, "Force xy of center";
    
    cfitsio_write_key, fh, "fovFWHM", fovFWHM, "Size of Gaussian prior if selected (rad)";
    //cfitsio_write_key, fh, "projAngle", projAngle, "projection angle for 1D-imaging (See e.g. Ohnaka et al. 2011) (rad)";
    cfitsio_write_key, fh, "cleandat", cleanup_bad_data, "MIRA option for removing bad data";
    cfitsio_write_key, fh, "reverse", reverse, "Go through wavelength in reverse order";

    cfitsio_write_key, fh, "Function","miral_reconstruct_multiwlen from miral.i","What code wrote this file"
    cfitsio_write_key, fh, "Author",rdline(popen("whoami",0)),"Who wrote this file"
    
    cfitsio_close,fh;
}

/***************************************************************************/

func miral_lff(inputFiles, outputFile=, binSize=, plot=, nLFF=, useVisAmp=, csym=, freqMax=, pse=, maxBase=)
    /* DOCUMENT miral_lff(inputFiles, outputFile=, binSize=, plot=, nLFF=, useVisAmp=, csym=, freqMax=, pse=)

       DESCRIPTION
       Unfinished

       PARAMETERS
       - inputFiles: 
       - outputFile: 
       - binSize   : 
       - plot      : 
       - nLFF      : 
       - useVisAmp : 
       - csym      : 
       - freqMax   : 
       - pse       : 
       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    yocoLogInfo,"Low Frequency Filling...";

    if(is_void(plot))
        plot=1;
    
    if(is_void(binSize))
        binSize = 1;
    
    if(is_void(pse))
        pse = 0;
    
    if(plot)
    {
        winkill,56;
        window,56,wait=1,width=600,height=600;
    }
    
    if(is_void(csym))
        csym = 1;
    
    // Set the output file
    if(is_void(outputFile))
    {
        yocoFileSplitName,inputFiles(1),d,f,e;
        outputFile = d+f+"_LFF";
    }
    
    nbFiles = numberof(inputFiles);
    
    // get the wavelength range of data and bandwidth
    miral_slice_wlens, inputFiles, waves, bands;
    nWlen = numberof(waves);
    if(!is_void(binSize))
    {
        // reshape wavelengths given the binsize
        nWlen = int(nWlen / binSize);
        if(nWlen<=1)
        {
            nWlen = 1;
            bands = max(waves) - min(waves);
            waves = avg(waves);
        }
        else if(binSize!=1)
        {
            bands = array(avg(bands)*binSize, nWlen);
            waves = span(min(waves)+bands(1)/2, max(waves)-bands(0)/2, nWlen);
        }
    }
    
    nBands = numberof(waves);

    PAR = [];
    for(kBand=1;kBand<=nBands;kBand++)
    {
        // Load OIFITS files
        amplLoadOiDatas, sci, inOiFile=inputFiles,
            wlenRange=[waves(kBand)-bands(kBand)/2,
                       waves(kBand)+bands(kBand)/2],quiet=1;

        nFiles = numberof(sci);
        UF=VF=UF2=VF2=U=V=V2=V2E=VIS=VISE=W=[];
        minR = 1e99;

        // Choose whether using visibilities or squared visibilities
        for(kF=1;kF<=nFiles;kF++)
        {
            v2 = (*sci(kF).VIS2DATA);
            if(!is_void(v2))
            {
                U    = *sci(kF).UCOORD;
                V    = *sci(kF).VCOORD;
                R    = abs(U,V);
                minR = min(minR,min(R));
                W    = *sci(kF).EFF_WAVE;
                grow, UF2, (U(-,)/W(,-))(*);
                grow, VF2, (V(-,)/W(,-))(*);

                v2  = (*sci(kF).VIS2DATA)(*);
                v2e = (*sci(kF).VIS2ERR)(*);
           
                grow,V2, v2;
                grow,V2E,v2e;
            }
            else
            {
                vis = (*sci(kF).VISAMP);
                if(!is_void(vis))
                {
                    U    = *sci(kF).UCOORD;
                    V    = *sci(kF).VCOORD;
                    R    = abs(U,V);
                    minR = min(minR,min(R));
                    W    = *sci(kF).EFF_WAVE;
                    
                    grow, UF, (U(-,)/W(,-))(*);
                    grow, VF, (V(-,)/W(,-))(*);
                    
                    vis    = (*sci(kF).VISAMP)(*);
                    viserr = (*sci(kF).VISAMPERR)(*);
                    
                    grow,VIS, vis;
                    grow,VISE,viserr;
                }
            }
        }

        // Get data
        if(!is_void(V2))
        {
            vis    = V2;
            visErr = V2E;
            UF     = UF2;
            VF     = VF2;
        }
        if(useVisAmp)
        {
            vis    = sign(VIS)*VIS^2;
            visErr = 2*VISE*VIS;
        }
        
        // Get radius
        u     = UF;
        if(is_void(nLFF))
            nLFF=numberof(u);
        v     = VF;
        uv    = [u,v];
        r     = abs(u, v);
        theta = atan(v, u);
        
        if(plot)
        {
            // Plot data
            window,56,wait=1,width=600,height=600;
            fma;
            yocoPlotWithErrBars,yocoMathSignedSqrt(vis),
                visErr/yocoMathSignedSqrt(vis)/2,
                r,type="none",marks=0,marker='\1',
                msize=0.2,color="black";
            xytitles,"Frequency","visibility";

        }
      
        // Threshold for data cut
        minV = 0.5;
        if(is_void(freqMax))
            // formule savante pour savoir ou sont les basses frequences
            lf = where((r<max(r)/10.)&
                       ((vis>minV)|
                        (visErr<0.1)&
                        (r<1.15*min(r))&
                        (vis>0.)));
        else
            lf =  where(r < freqMax);
        
        if(numberof(lf)<2)
            lf = sort(r)(1:2);
        
        vLFF  = vis(lf);
        veLFF = visErr(lf);
        rLFF  = r(lf);
        uvLFF = uv(lf,);
        
        vavg = avg(vLFF);
        ravg = avg(rLFF);
        if(plot==1)
        {
            yocoPlotWithErrBars, yocoMathSignedSqrt(vLFF),
                veLFF/yocoMathSignedSqrt(vLFF)/2,
                rLFF,type="none",marks=0,marker='\2',
                msize=0.2,color="red";
            plg,yocoMathSignedSqrt(vavg),ravg,color="blue",marker='\3',marks=1,type="none";
            limits,,,0,1;
        }

        if((csym==1)||(numberof(vLFF)<3))
        {
            // 1D
            param = [(1-vavg)^(1./2.)/ravg];
            line  = transpose(UVLine(0,ravg,0,100));

            // Fit
            res = lmfit(lff_visibility_round, uvLFF, param, vLFF, 1./veLFF^2);
            if(plot==1)
            {
                plg,yocoMathSignedSqrt(lff_visibility_round(line,param)),
                    abs(line(,1),line(,2)),color="blue";
                pltitle,"Low Frequency Filling...\n"+pr1(waves(kBand)*1e6)+" microns";
            }
        }
        else
        {
            // 2D
            siz    = (1-vavg)^(1./2.)/ravg;
            param  = [siz,0.9*siz,0.1];
            
            // fit
            res = lmfit(lff_visibility_2D, uvLFF, param, vLFF, 1./veLFF^2,itmax=20);
            if(plot==1)
            {
                angle  = param(3);
                line   = transpose(UVLine(0,1.2*ravg,(angle%(2*pi)),100));
                line2  = transpose(UVLine(0,1.2*ravg,((angle%(2*pi))+pi/2),100));
                
                plg,yocoMathSignedSqrt(lff_visibility_2D(line,param)),
                    abs(line(,1),line(,2)),color="blue";
                plg,yocoMathSignedSqrt(lff_visibility_2D(line2,param)),
                    abs(line2(,1),line2(,2)),color="blue";
                pltitle,"Low Frequency Filling...\n"+pr1(waves(kBand)*1e6)+" microns";
            }
        }

        rsort  = sort(r);
        vrmin = (vis)(rsort(1:min(10,numberof(rsort))));

        grow,PAR,&param;
        //        pause,pse;
    }
    
    if(is_void(maxBase))
        maxBase = minR*0.2;
    
    // Store the result in a new oifits file
    data           = array(amplOiData(),1);
    data.EFF_WAVE  = &waves;
    data.EFF_BAND  = &bands;
    data.hdr       = sci.hdr(1);
    
    // Set the UV coordinates of data as random in a circle
    UU = VV = UU1 = VV1 = UU2 = VV2 = TIME3 = MJD3 = [];
    
    nObs    = nLFF;
    nWlen   = numberof(waves);
    
    if(0)
    {
        R   = maxBase * random(nObs);
        T   = 2*pi*(random(nObs)-0.5);
        wr  = where(R>maxBase);
        nwr = numberof(wr);
        while(nwr>0)
        {
            R(nwr) = maxBase * random(nwr);
            T(nwr) = 2*pi*(random(nwr)-0.5);
            wr     = where(R>maxBase);
            nwr    = numberof(wr);
        }
        U = R*cos(T);
        V = R*sin(T);
    }
    else
    {
        U   = 2*maxBase * (random(nObs)-0.5);
        V   = 2*maxBase * (random(nObs)-0.5);
        R   = abs(U,V);
        wr  = where(R>maxBase);
        nwr = numberof(wr);
        while(nwr>0)
        {
            U(wr) = 2*maxBase * (random(nwr)-0.5);
            V(wr) = 2*maxBase * (random(nwr)-0.5);
            R     = abs(U,V);
            wr    = where(R>maxBase);
            nwr   = numberof(wr);
        }
    }
    UV          = [U,V];
    data.UCOORD = &U;
    data.VCOORD = &V;
    
    VIS2=[];
    for(kBand=1;kBand<=nBands;kBand++)
    {
        if((csym==1)||(numberof(*PAR(kBand))==1))
            visib2 = lff_visibility_round(UV/waves(kBand),*PAR(kBand));
        else
            visib2 = lff_visibility_2D(UV/waves(kBand),*PAR(kBand));
        grow,VIS2,[visib2];
    }
    VISE   = median(veLFF)/median(vLFF) * yocoMathSignedSqrt(VIS2) * R / max(R);
    VIS2E  = median(veLFF)/median(vLFF) * VIS2 * R / max(R);
      
    if(plot==1)
    {
        R = abs(U(,-)/waves(-,),V(,-)/waves(-,));
        window,56;
        //        pla,yocoMathSignedSqrt(VIS2),R,type="none",marker='\1';
    }
        
    // Compute squared visibilities for the given UV coordinates
    data.VIS2DATA  = &transpose(VIS2);
    data.VIS2ERR   = &transpose(VIS2E);
    data.STA_INDEX = &array([1,2],dimsof(VIS2)(2));
    data.FLAG      = &array(0,dimsof(transpose(VIS2)));
    data.TIME      = &array(0,dimsof(VIS2)(2));
    
    // Save data
    for(k=1;k<=numberof(data);k++)
        amplSaveOiData(outputFile+"_"+pr1(k)+e, data(k), amberKeys=);
}

/***************************************************************************/

func miral_lff_setsize(inputFiles, outputFile=, binSize=, plot=, nLFF=, useVisAmp=, csym=, freqMax=, pse=, maxBase=,diam=,type=)
    /* DOCUMENT miral_lff(inputFiles, outputFile=, binSize=, plot=, nLFF=, useVisAmp=, csym=, freqMax=, pse=)

       DESCRIPTION
       Unfinished

       PARAMETERS
       - inputFiles: 
       - outputFile: 
       - binSize   : 
       - plot      : 
       - nLFF      : 
       - useVisAmp : 
       - csym      : 
       - freqMax   : 
       - pse       : 
       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    yocoLogInfo,"Low Frequency Filling...";

    if(is_void(plot))
        plot=1;
    
    if(is_void(binSize))
        binSize = 1;
    
    if(is_void(pse))
        pse = 0;
    
    if(plot)
    {
        winkill,56;
        window,56,wait=1,width=600,height=600;
    }
    
    if(is_void(csym))
        csym = 1;
    
    // Set the output file
    if(is_void(outputFile))
    {
        yocoFileSplitName,inputFiles(1),d,f,e;
        outputFile = d+f+"_LFF";
    }
    
    nbFiles = numberof(inputFiles);
    
    // get the wavelength range of data and bandwidth
    miral_slice_wlens, inputFiles, waves, bands;
    nWlen = numberof(waves);
    if(!is_void(binSize))
    {
        // reshape wavelengths given the binsize
        nWlen = int(nWlen / binSize);
        if(nWlen<=1)
        {
            nWlen = 1;
            bands = max(waves) - min(waves);
            waves = avg(waves);
        }
        else if(binSize!=1)
        {
            bands = array(avg(bands)*binSize, nWlen);
            waves = span(min(waves)+bands(1)/2, max(waves)-bands(0)/2, nWlen);
        }
    }
    
    nBands = numberof(waves);

    PAR = [];
    for(kBand=1;kBand<=nBands;kBand++)
    {
        // Load OIFITS files
        amplLoadOiDatas, sci, inOiFile=inputFiles,
            wlenRange=[waves(kBand)-bands(kBand)/2,
                       waves(kBand)+bands(kBand)/2],quiet=1;

        nFiles = numberof(sci);
        UF=VF=UF2=VF2=U=V=V2=V2E=VIS=VISE=W=[];
        minR = 1e99;

        // Choose whether using visibilities or squared visibilities
        for(kF=1;kF<=nFiles;kF++)
        {
            v2 = (*sci(kF).VIS2DATA);
            if(!is_void(v2))
            {
                U    = *sci(kF).UCOORD;
                V    = *sci(kF).VCOORD;
                R    = abs(U,V);
                minR = min(minR,min(R));
                W    = *sci(kF).EFF_WAVE;
                grow, UF2, (U(-,)/W(,-))(*);
                grow, VF2, (V(-,)/W(,-))(*);

                v2  = (*sci(kF).VIS2DATA)(*);
                v2e = (*sci(kF).VIS2ERR)(*);
           
                grow,V2, v2;
                grow,V2E,v2e;
            }
            else
            {
                vis = (*sci(kF).VISAMP);
                if(!is_void(vis))
                {
                    U    = *sci(kF).UCOORD;
                    V    = *sci(kF).VCOORD;
                    R    = abs(U,V);
                    minR = min(minR,min(R));
                    W    = *sci(kF).EFF_WAVE;
                    
                    grow, UF, (U(-,)/W(,-))(*);
                    grow, VF, (V(-,)/W(,-))(*);
                    
                    vis    = (*sci(kF).VISAMP)(*);
                    viserr = (*sci(kF).VISAMPERR)(*);
                    
                    grow,VIS, vis;
                    grow,VISE,viserr;
                }
            }
        }

        // Get data
        if(!is_void(V2))
        {
            vis    = V2;
            visErr = V2E;
            UF     = UF2;
            VF     = VF2;
        }
        if(useVisAmp)
        {
            vis    = sign(VIS)*VIS^2;
            visErr = 2*VISE*VIS;
        }
        
        // Get radius
        u     = UF;
        if(is_void(nLFF))
            nLFF=numberof(u);
        v     = VF;
        uv    = [u,v];
        r     = abs(u, v);
        theta = atan(v, u);
        
        if(plot)
        {
            // Plot data
            window,56,wait=1,width=600,height=600;
            fma;
            yocoPlotWithErrBars,yocoMathSignedSqrt(vis),
                visErr/yocoMathSignedSqrt(vis)/2,
                r,type="none",marks=0,marker='\1',
                msize=0.2,color="black";
            xytitles,"Frequency","visibility";

        }
      
        // Threshold for data cut
        minV = 0.25;
        if(is_void(freqMax))
            // formule savante pour savoir ou sont les basses frequences
            lf = where((r<max(r)/10.)&
                       ((vis>minV)|
                        (visErr<0.1)&
                        (r<1.15*min(r))&
                        (vis>0.)));
        else
            lf =  where(r < freqMax);
        
        if(numberof(lf)<2)
            lf = sort(r)(1:2);
        
        vLFF  = vis(lf);
        veLFF = visErr(lf);
        rLFF  = r(lf);
        uvLFF = uv(lf,);
        
        vavg = avg(vLFF);
        ravg = avg(rLFF);
        if(plot==1)
        {
            yocoPlotWithErrBars, yocoMathSignedSqrt(vLFF),
                veLFF/yocoMathSignedSqrt(vLFF)/2,
                rLFF,type="none",marks=0,marker='\2',
                msize=0.2,color="red";
            plg,yocoMathSignedSqrt(vavg),ravg,color="blue",marker='\3',marks=1,type="none";
            limits,,,0,1;
        }

            // 1D
            param = [(1-vavg)^(1./2.)/ravg];
            line  = transpose(UVLine(0,ravg,0,100));

            // No fit. plot directly on top of visibilities
            if(plot==1)
            {
                if(type=="UD")
                    vis = lff_visibility_UD(line,diam);
                else if(type=="G")
                    vis = lff_visibility_G(line,diam);
                V = abs(vis(,1).im,vis(,1).re)
                plg,yocoMathSignedSqrt(V),
                    abs(line(,1),line(,2)),color="blue";
                pltitle,"Low Frequency Filling...\n"+pr1(waves(kBand)*1e6)+" microns";
            }

        rsort  = sort(r);
        vrmin = (vis)(rsort(1:min(10,numberof(rsort))));

        grow,PAR,&param;
        //        pause,pse;
    }
    
    
    if(is_void(maxBase))
        maxBase = minR*0.8;
    
    // Store the result in a new oifits file
    data           = array(amplOiData(),1);
    data.EFF_WAVE  = &waves;
    data.EFF_BAND  = &bands;
    data.hdr       = sci.hdr(1);
    
    // Set the UV coordinates of data as random in a circle
    UU = VV = UU1 = VV1 = UU2 = VV2 = TIME3 = MJD3 = [];
    
    nObs    = nLFF;
    nWlen   = numberof(waves);
    
    if(0)
    {
        R   = maxBase * random(nObs);
        T   = 2*pi*(random(nObs)-0.5);
        wr  = where(R>maxBase);
        nwr = numberof(wr);
        while(nwr>0)
        {
            R(nwr) = maxBase * random(nwr);
            T(nwr) = 2*pi*(random(nwr)-0.5);
            wr     = where(R>maxBase);
            nwr    = numberof(wr);
        }
        U = R*cos(T);
        V = R*sin(T);
    }
    else
    {
        U   = 2*maxBase * (random(nObs)-0.5);
        V   = 2*maxBase * (random(nObs)-0.5);
        R   = abs(U,V);
        wr  = where(R>maxBase);
        nwr = numberof(wr);
        while(nwr>0)
        {
            U(wr) = 2*maxBase * (random(nwr)-0.5);
            V(wr) = 2*maxBase * (random(nwr)-0.5);
            R     = abs(U,V);
            wr    = where(R>maxBase);
            nwr   = numberof(wr);
        }
    }
    UV          = [U,V];
    data.UCOORD = &U;
    data.VCOORD = &V;
    
    VIS2=[];
    for(kBand=1;kBand<=nBands;kBand++)
    {
        if((csym==1)||(numberof(*PAR(kBand))==1))
        {    
            if(type=="UD")
                vis = lff_visibility_UD(UV/waves(kBand),diam);
            else if(type=="G")
                vis = lff_visibility_G(UV/waves(kBand),diam);
            visib2 = abs(vis(,1).im,vis(,1).re);
        }
        grow,VIS2,[visib2];
    }
    VISE   = median(veLFF)/median(vLFF) * yocoMathSignedSqrt(VIS2) * R / max(R);
    VIS2E  = median(veLFF)/median(vLFF) * VIS2 * R / max(R);
      
    if(plot==1)
    {
        R = abs(U(,-)/waves(-,),V(,-)/waves(-,));
        window,56;
        //        pla,yocoMathSignedSqrt(VIS2),R,type="none",marker='\1';
    }
        
    // Compute squared visibilities for the given UV coordinates
    data.VIS2DATA  = &transpose(VIS2);
    data.VIS2ERR   = &transpose(VIS2E);
    data.STA_INDEX = &array([1,2],dimsof(VIS2)(2));
    data.FLAG      = &array(0,dimsof(transpose(VIS2)));
    data.TIME      = &array(0,dimsof(VIS2)(2));
    
    // Save data
    for(k=1;k<=numberof(data);k++)
        amplSaveOiData(outputFile+"_"+pr1(k)+e, data(k), amberKeys=);
}

/***************************************************************************/

func lff_visibility_round(uv, param)
    /* DOCUMENT lff_visibility_round(uv, param)

       DESCRIPTION

       PARAMETERS
       - uv   : 
       - param: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    u = uv(,1);
    v = uv(,2);
    r = abs(u,v);
    return 1 - (param * r)^2;
}

func lff_visibility_UD(uv, diam)
{
    u = uv(,1);
    v = uv(,2);
    return visUniformDisk(u, v, 1, 1, diam, 0, 0)
}

func lff_visibility_G(uv, diam)
{
    u = uv(,1);
    v = uv(,2);
    return visGaussianDisk(u, v, 1, 1, diam, 0, 0)
}

/***************************************************************************/

func lff_visibility_2D(uv, params)
    /* DOCUMENT lff_visibility_2D(uv, params)

       DESCRIPTION

       PARAMETERS
       - uv    : 
       - params: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    a   = params(1);
    b   = params(2);
    ang = params(3);
    
    u  = a*uv(,1);
    v  = b*uv(,2);
    u1 = u*cos(ang)+v*sin(ang);
    v1 = u*sin(ang)-v*cos(ang);
                      
    r = abs(u1,v1);
    
    return 1 - r^2;
}


/***************************************************************************/

func miral_get_best_medianImg(IMGCUB, CHI2, MU, &okones, chi2Thr=, muThr=)
    /* DOCUMENT miral_get_best_medianImg(IMGCUB, CHI2, MU, &okones, chi2Thr=, muThr=)

       DESCRIPTION
       Get the "best image" from an image cube with several runs of
       MIRA. The best image is the median of all the images whose Chi2
       is larger than 2*min(chi2)

       PARAMETERS
       - IMGCUB : 
       - CHI2   : 
       - MU     : 
       - okones : 
       - chi2Thr: 
       - muThr  : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(chi2Thr))
        chi2Thr = 2;
    
    if(is_void(muThr))
        muThr   = 1e4;
    
    dimz    = dimsof(IMGCUB);
    imgSize = dimz(2);
    nImg    = dimz(4);
    if(dimz(1)==4)
        nWlen   = dimz(5);

    if(dimz(1)==4)
    {
        //icb = reform(IMGCUB,[3,imgSize,imgSize,nImg*nWlen]);
        icb    = IMGCUB(,,*);
        //ich  = reform(CHI2, [1,nImg*nWlen]);
        ich    = CHI2(*);
        imu    = MU(*);
        smimg  = icb(sum,sum,);
    }
    else if(dimz(1)==3)
    {
        //icb = reform(IMGCUB,[3,imgSize,imgSize,nImg*nWlen]);
        icb    = IMGCUB;
        //ich  = reform(CHI2, [1,nImg*nWlen]);
        ich    = CHI2;
        imu    = MU;
        smimg  = IMGCUB(sum,sum,);
    }
    else
        error;
    
    nonzero = where(smimg!=0);
    if(numberof(nonzero)!=0)
    {
        okones    = where((ich(nonzero) <= chi2Thr*min(ich(nonzero)))&
                          (imu(nonzero) <= muThr));
        
        if(numberof(okones)==0)
            return icb(..,1);            
        else  if(numberof(okones)==1)
            return icb(..,nonzero)(..,okones)(..,1);
        // if(numberof(okones)>=10)
        // dfsdf()

        return median(icb(..,nonzero)(..,okones),0);
    }
    else
        return 0;
}
/***************************************************************************/

func miral_get_best_avgImg(IMGCUB, CHI2, MU, &okones, &rmsImg, chi2Thr=, muThr=)
    /* DOCUMENT miral_get_best_avgImg(IMGCUB, CHI2, MU, &okones, &rmsImg, chi2Thr=, muThr=)

       DESCRIPTION
       Get the "best image" from an image cube with several runs of
       MIRA The best image is the average of all the images whose Chi2
       is larger than 2*min(chi2)

       PARAMETERS
       - IMGCUB : 
       - CHI2   : 
       - MU     : 
       - okones : 
       - rmsImg : 
       - chi2Thr: 
       - muThr  : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(chi2Thr))
        chi2Thr=2;
    
    if(is_void(muThr))
        muThr   = 1e4;
    
    dimz = dimsof(IMGCUB);
    imgSize = dimz(2);
    nImg = dimz(4);
    if(dimz(1)==4)
        nWlen = dimz(5);
    
    if(dimz(1)==4)
    {
        //icb = reform(IMGCUB,[3,imgSize,imgSize,nImg*nWlen]);
        icb    = IMGCUB(..,sum);
        //ich  = reform(CHI2, [1,nImg*nWlen]);
        ich    = CHI2(..,sum);
        imu    = MU(..,avg);
        smimg  = icb(sum,sum,..);
    }
    else if(dimz(1)==3)
    {
        //icb = reform(IMGCUB,[3,imgSize,imgSize,nImg*nWlen]);
        icb    = IMGCUB;
        //ich  = reform(CHI2, [1,nImg*nWlen]);
        ich    = CHI2;
        imu    = MU;
        smimg  = IMGCUB(sum,sum,);
    }
    nonzero = where(smimg!=0);
    if(numberof(nonzero)!=0)
    {
        okones    = where((ich(nonzero) < chi2Thr*min(ich(nonzero)))&
                          (imu(nonzero) < muThr));

        if(numberof(okones)==0)
            return icb(..,1);            
        else  if(numberof(okones)==1)
            return icb(..,nonzero)(..,okones)(..,1);
        // if(numberof(okones)>=10)
        // dfsdf()

        rmsImg = icb(..,nonzero)(..,okones)(..,rms);
        
        return icb(..,nonzero)(..,okones)(..,avg);
    }
    else
        return 0;
}

/***************************************************************************/

func miral_write_imgCube(outputFile, imgCube, X, Y, wlen, mu, chi2, regul, img0, prior)
    /* DOCUMENT miral_write_imgCube(outputFile, imgCube, X, Y, wlen, mu, chi2, regul)

       DESCRIPTION
       Writes an image cube resulting from a multiwavelebgth run of MIRA

       PARAMETERS
       - outputFile: output fits file name
       - imgCube   : input image cube
       - X         : X axis
       - Y         : Y axis
       - wlen      : Z axis
       - mu        : 
       - chi2      : 
       - regul     : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    yocoFileSplitName,outputFile,d,f,e;
    yocoLogInfo,"Writing in directory: "+d;
    yocoLogInfo,"File name: "+f+e;

    if( is_void(imgCube))
    {
        print,"WARNING: empty image cube!";
        return 0;
    }
    
    cfitsWrite,outputFile,imgCube;

    dimz = dimsof(imgCube);
    if((dimz(1)==3)&&(!is_void(wlen)))
    {
        imgSizeW = dimz(4);
        minW     = min(wlen);
        if(numberof(wlen)>=2)
            deltaW = avg(wlen(dif));
        else
            deltaW = 0;
    }
    else if(!is_void(wlen))
    {
        deltaW = 0;
        minW   = wlen;
    }
    else
        deltaW=0;
    
    imgSizeX = dimz(2);
    minX = min(X);
    deltaX = avg(X(dif));

    imgSizeY = dimsof(imgCube)(3);
    minY = min(Y);
    deltaY = avg(Y(dif));

    fh = cfitsio_open(outputFile,"a"); // re-open the file to modify keys 
    cfitsio_write_key, fh, "CRPIX1", 1,
        "X ref pixel (R.A.)"; // x reference pix
    cfitsio_write_key, fh, "CRPIX2", 1,
        "Y ref pixel (dec)"; // y reference pix
    if(!is_void(wlen))
        cfitsio_write_key, fh, "CRPIX3", 1,
            "wavelength ref pixel"; // lambda reference pix

    cfitsio_write_key, fh, "CDELT1", deltaX,
        "X increment"; // pixel increment
    cfitsio_write_key, fh, "CDELT2", deltaY,
        "Y increment"; // pixel increment
    if(!is_void(wlen))
        cfitsio_write_key, fh, "CDELT3", deltaW,
            "wavelength increment"; // wavelength increment

    cfitsio_write_key, fh, "CRVAL1", minX,
        "X minimum"; // pixel center (zero here)
    cfitsio_write_key, fh, "CRVAL2", minY,
        "Y minimum"; // pixel center (zero here)
    if(!is_void(wlen))
        cfitsio_write_key, fh, "CRVAL3", minW,
            "wavelength mimimum"; // minimum wavelength

    unit = "mas";
    if(X(dif)(avg)<1e-3)
        unit="as";
    else(X(dif)(avg)<1e-6)
            unit="rad";

    cfitsio_write_key, fh, "CUNIT1", unit,
        "X unit"; // pixel center (zero here)
    cfitsio_write_key, fh, "CUNIT2", unit ,
        "Y unit"; // pixel center (zero here)
    if(!is_void(wlen))
        cfitsio_write_key, fh, "CUNIT3", "m",
            "wavelength unit"; // minimum wavelength

    if(!is_void(mu))
    {
        // FIXME: this is to replace the cfitsioplugin bintable writing
        amplCfitsio_add_bintable, fh, [&transpose(mu),
                                       &transpose(chi2),
                                       &swrite(regul(,1),format="%-16s"),
                                       &array(img0,1),
                                       &array(prior,1)],
            ["mu", "chi2", "regul", "img0", "prior"],
            ["","","","",""],
            "INFOS";   // ad a binary table
    }
    cfitsio_close,fh; // close the file
}

/***************************************************************************/

func miral_read_imgCube(cubeFile, &imgCube, &X, &Y, &wlen, &mu, &chi2, &regul)
    /* DOCUMENT miral_read_imgCube(cubeFile, &imgCube, &X, &Y, &wlen, &mu, &chi2, &regul)

       DESCRIPTION
       Image cube reading routine

       PARAMETERS
       - cubeFile: file containing the image cube
       - imgCube : result image cube
       - X       : X axis
       - Y       : Y axis
       - wlen    : Z axis
       - mu      : 
       - chi2    : 
       - regul   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    imgCube = cfitsRead(cubeFile);

    fh = cfitsio_open(cubeFile,"r"); // open the file
    
    xref = cfitsio_read_key(fh, TDOUBLE, "CRPIX1"); // x reference pix
    xinc = cfitsio_read_key(fh, TDOUBLE, "CDELT1"); // pixel increment
    xval = cfitsio_read_key(fh, TDOUBLE, "CRVAL1"); // pixel center (zero here)
    
    yref = cfitsio_read_key(fh, TDOUBLE, "CRPIX2"); // y reference pix
    yinc = cfitsio_read_key(fh, TDOUBLE, "CDELT2"); // pixel increment
    yval = cfitsio_read_key(fh, TDOUBLE, "CRVAL2"); // pixel center (zero here)
    
    wref = cfitsio_read_key(fh, TDOUBLE, "CRPIX3"); // lambda reference pix
    winc = cfitsio_read_key(fh, TDOUBLE, "CDELT3"); // wavelength increment
    wval = cfitsio_read_key(fh, TDOUBLE, "CRVAL3"); // minimum wavelength
    
    nhdu = cfitsio_get_num_hdus(fh);
    if(nhdu==2)
    {
        cfitsio_goto_hdu,fh,2;                       // go to a HDU
        table = cfitsio_read_bintable(fh,titles);    // read the binary table
        chi2  = *(table(where(titles=="chi2")(1)));
        mu    = *(table(where(titles=="mu")(1)));
        regul = *(table(where(titles=="regul")(1)));
    }
    
    cfitsio_close,fh; // close the file

    dimz = dimsof(imgCube);
    X = (indgen(dimz(2))-xref)*xinc + xval;
    Y = (indgen(dimz(3))-yref)*yinc + yval;
    if((wref!=0)&&(numberof(dimz)==4))
        wlen = (indgen(dimz(4))-wref)*winc + wval;
    else if(!is_void(wval))
        wlen = wval;

    return imgCube;
}

/***************************************************************************/

func select_best_mu(mh1, img0, regul, &CHI2, &MU, maxeval=)
    /* DOCUMENT select_best_mu(mh1, img0, regul, &CHI2, &MU, maxeval=)

       DESCRIPTION
       Get best value for the hyperparameter after a run of 1000 image
       reconstructions. This functions needs to be updated...

       PARAMETERS
       - mh1    : 
       - img0   : 
       - regul  : 
       - CHI2   : 
       - MU     : 
       - maxeval: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    CHI2=MU=[];
    yocoNmCreate,1,landscape=1,V=[0.1,0.9,0.12,0.73],wait=1;

    for(k=1;k<=1000;k++)
    {
        mu = 10^(4*random()+2);
        
        img1 = mira_solve(mh1, img1,
                          maxeval=maxeval, xmin=1e-10,
                          normalization=1, regul=regul, mu=mu, view=view,
                          verb=verb);

	////////////////////////////////////////////////////////////
	// FIXME: 
	// FIXME: PLEASE READ THIS COMMENT!
	// FIXME: 
        // FIXME: in order to let this line of code work,
        // FIXME: insert the following line of code in mira.i at line 2204:
        // FIXME:    extern extra;
	// FIXME: 
	////////////////////////////////////////////////////////////
        chi2 = extra.data_err/extra.master.ndata;

        plg,chi2,mu,type="none";

        grow,CHI2,chi2;
        grow,MU,mu;
    }
}

/***************************************************************************/

func miral_solve_multiRgl(mh1, img0, regul, &chi2, mu=, maxeval=, prior=, recenter=, view=, verb=)
    /* DOCUMENT miral_solve_multiRgl(mh1, img0, regul, &chi2, mu=, maxeval=, prior=, recenter=, view=, verb=)

       DESCRIPTION
       cascade mira runs with decreasing regularization parameters

       PARAMETERS
       - mh1     : input hash structure
       - img0    : initial image
       - regul   : regularization
       - chi2    : 
       - mu      : superparameter
       - maxeval : maximum number of iteration steps
       - prior   : prior image
       - recenter: recenter or not the image
       - view    : show (or not) graphics during mira run
       - verb    : verbosity

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(verb))
        verb = 1;

    if(is_void(mu))
        mu = [1e12, 1e9, 1e6, 1e4, 1e3, 1e2, 1e1, 1e0];
    
    // rgl = rgl_new("l2l1_smoothness");
    // rgl_config, rgl,"threshold",2e3;
    //rgl = rgl_new("quadratic");

    if(is_void(maxeval))
        maxeval = 200;

    img1 = img0;
    for(kmu=1;kmu<=numberof(mu);kmu++)
    {
        if(recenter==1)
            img1 = mira_recenter(img1, quiet=1);
        
        img1 = mira_solve(mh1, img1,
                          maxeval=maxeval, xmin=0.0,
                          normalization=1, regul=regul, mu=mu(kmu), view=view,
                          verb=verb, zap_phase=(kmu+1)%2);

	//
        // FIXME: in order to let this line of code work,
        // insert the following line of code in mira.i at line 2204:
        //   extern extra;
	//

       chi2 = extra.data_err/extra.master.ndata;
    }

    if(recenter==1)
        img1 = mira_recenter(img1, quiet=1);

    return img1;
}

/***************************************************************************/

func miral_compute_dim(inputFiles)
    /* DOCUMENT miral_compute_dim(inputFiles)

       DESCRIPTION
       Computes optimal dimension of an image given all parameters in hands
       (baselines, wavelength, etc.)

       PARAMETERS
       - inputFiles: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if (is_void(quiet))
        quiet = 0;
    
    // Load OIFITS files
    // amplLoadSciFiles, sci, nightDir=dataDir, calibrated=1;
    amplLoadOiDatas, sci, inOiFile=inputFiles;
    nFiles = numberof(inputFiles);

    wlens = bands = u = v = [];
    for(kFile=1;kFile<=nFiles;kFile++)
    {
        if(!is_void(*sci(kFile).UCOORD))
        {
            grow,u,*sci(kFile).UCOORD;
            grow,v,*sci(kFile).VCOORD;
            grow,wlens,*sci(kFile).EFF_WAVE;
            grow,bands,*sci(kFile).EFF_BAND;
        }
    }
    
    uh = u(-,)/wlens(,-);
    vh = v(-,)/wlens(,-);
    rh = abs(uh,vh);
    rhmin = min(rh);
    rhmax = max(rh);
    fov        = miral_compute_fov(inputFiles);
    pixSize    = miral_compute_pixelsize(inputFiles);
    closestExp = round(log(fov/pixSize)/log(2));
    imgSize    = int(2^closestExp);
    return imgSize;
}

/***************************************************************************/

func miral_compute_fov(inputFiles)
    /* DOCUMENT miral_compute_fov(inputFiles)

       DESCRIPTION
       Computes the optimal field of view given a set of baselines,
       wavelength, etc.

       PARAMETERS
       - inputFiles: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if (is_void(quiet))
        quiet = 0;
    
    // Load OIFITS files
    // amplLoadSciFiles, sci, nightDir=dataDir, calibrated=1;
    amplLoadOiDatas, sci, inOiFile=inputFiles;
    nFiles = numberof(inputFiles);
    
    wlens = bands = u = v = [];
    for(kFile=1;kFile<=nFiles;kFile++)
    {
        if(!is_void(*sci(kFile).UCOORD))
        {
            grow,u,*sci(kFile).UCOORD;
            grow,v,*sci(kFile).VCOORD;
            grow,wlens,*sci(kFile).EFF_WAVE;
            grow,bands,*sci(kFile).EFF_BAND;
        }
    }
    
    uh = u(-,)/wlens(,-);
    vh = v(-,)/wlens(,-);
    rh = abs(uh,vh);
    rhmin = min(rh(where(rh!=0)));
    fov = 2.0/rhmin/MIRA_MILLIARCSECOND;

    return fov;
}

/***************************************************************************/

func miral_compute_pixelsize(inputFiles)
    /* DOCUMENT miral_compute_pixelsize(inputFiles)

       DESCRIPTION
       Computes optimal pixel size given baselines, wavelength etc.

       PARAMETERS
       - inputFiles: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if (is_void(quiet))
        quiet = 0;
    
    // Load OIFITS files
    // amplLoadSciFiles, sci, nightDir=dataDir, calibrated=1;
    amplLoadOiDatas, sci, inOiFile=inputFiles;
    nFiles = numberof(inputFiles);

    wlens = bands = u = v = [];
    for(kFile=1;kFile<=nFiles;kFile++)
    {
        if(!is_void(*sci(kFile).UCOORD))
        {
            grow,u,*sci(kFile).UCOORD;
            grow,v,*sci(kFile).VCOORD;
            grow,wlens,*sci(kFile).EFF_WAVE;
            grow,bands,*sci(kFile).EFF_BAND;
        }
    }
    
    uh = u(-,)/wlens(,-);
    vh = v(-,)/wlens(,-);
    rh = abs(uh,vh);
    rhmax = max(rh(where(rh!=0)));
    
    pixSize = .5/rhmax/2.44/MIRA_MILLIARCSECOND;
    return pixSize;
}

/***************************************************************************/

func miral_get_maxBase(mh1)
    /* DOCUMENT miral_get_maxBase(mh1)

       DESCRIPTION
       Get the maximum baseline of a dataset

       PARAMETERS
       - mh1:  input hash table

       SEE ALSO
    */
{
    uh = mh1.vis2.u;
    vh = mh1.vis2.v;
    wh = mh1.vis2.w;
    rh = abs(uh,vh)*wh;
    rhmax = max(rh);
    return rhmax;
}

/***************************************************************************/

func miral_get_avgWlen(mh1)
    /* DOCUMENT miral_get_avgWlen(mh1)

       DESCRIPTION
       Get average wavelength over the dataset

       PARAMETERS
       - mh1:  input hash table

       SEE ALSO
    */
{
    wh = mh1.vis2.w;
    return avg(wh);
}

/***************************************************************************/

func miral_clean_beam(mh1, &params)
    /* DOCUMENT miral_clean_beam(mh1, &params)

       DESCRIPTION

       PARAMETERS
       - mh1   :  input hash table
       - params: output parameters for the Gaussian fit

       SEE ALSO
    */
{
    dim     = mira_get_dim(mh1);
    pxSize  = mira_get_pixelsize(mh1);
    dirty   = mira_dirty_beam(mh1);
    maxBase = miral_get_maxBase(mh1);
    avgWlen = miral_get_avgWlen(mh1);

    interv = avgWlen/maxBase/pxSize;
    xy0    = where2(dirty==max(dirty));
    X      = array(indgen(dim),dim);
    Y      = transpose(X);
    XY     = grow(array(X,1),array(Y,1));
    params = a0 = [max(dirty), dim/2, dim/2, interv, interv, 0.0, 0.0];

    //lmfit, miral_gauss2d, XY, params, dirty;
    lmfit, gauss2d, XY, params, dirty;

    return gauss2d(XY, params);
}

/***************************************************************************/

func miral_convol_beam(img, beam)
    /* DOCUMENT miral_convol_beam(img, beam)

       DESCRIPTION
       Convolves an image with a given input beam

       PARAMETERS
       - img : input image
       - beam: input beam

       SEE ALSO
    */
{
    dimz = dimsof(img)(1);
    if(dimz==2)
        fftIm = fft(img,[1,1]);
    else if(dimz==3)
    {
        fftIm = fft(img,[1,1,0]);
    }
    fftBm = fft(beam,[1,1]);
    prod  = fftIm * fftBm;
    if(dimz==2)
        conv  = fft(prod,[-1,-1]);
    else if(dimz==3)
        conv = fft(prod,[-1,-1,0]);
    return roll(conv.re);
}

/***************************************************************************/

func miral_plot_imgCube_waterfall(imgCube, nbFrames=, offset=, width=)
    /* DOCUMENT miral_plot_imgCube_waterfall(imgCube, nbFrames=, offset=, width=)

       DESCRIPTION
       Plots a waterfall image from an image cube

       PARAMETERS
       - imgCube : the input image cube
       - nbFrames: number of frames to plot
       - offset  : what offset to put between consecutive images
       - width   : width of border around images

       SEE ALSO
    */
{
    N = dimsof(imgCube)(0);
    nx = dimsof(imgCube)(2);
    ny = dimsof(imgCube)(3);

    if(is_void(nbFrames))
        nbFrames = N;
    
    if(is_void(offset))
        offset = nx;

    hh = N / nbFrames;
    kub = imgCube(..,::hh);
    for(k=1;k<=nbFrames;k++)
    {
        add = -offset * (k-1) / double(nbFrames);
        pli,kub(..,k), add, add, add+nx, add+ny;
        pldj,add, add, add+nx, add, width=width;
        pldj,add, add, add, add+ny, width=width;
        pldj,add+nx, add, add+nx, add+ny, width=width;
        pldj,add, add+ny, add+nx, add+ny, width=width;
    }
}

/***************************************************************************/

func miral_slice_wlens(inputFiles, &waves, &bands)
    /* DOCUMENT miral_slice_wlens(inputFiles, &waves, &bands)
       DESCRIPTION
       From a set of files determine all the wavelengths
       used in a dataset

       PARAMETERS
       - inputFiles: 
       - waves     : 
       - bands     : 

       SEE ALSO
    */
{
    if (is_void(quiet))
        quiet = 0;
    
    // Load OIFITS files
    // amplLoadSciFiles, sci, nightDir=dataDir, calibrated=1;
    amplLoadOiDatas, sci, inOiFile=inputFiles;
    nFiles = numberof(inputFiles);

    wlen = *sci(1).EFF_WAVE;
    band = *sci(1).EFF_BAND;
    //wlen(1) = wlen(1) - band(1);
    // band(1) = 2. * band(1);
    //wlen(0) = wlen(0) + band(0);
    //band(0) = 2. * band(0);
    //    yocoPlotWithErrBars,wlen,band/2,indgen(numberof(wlen));

    waves = wlen;
    bands = band;
        
    return 0;
    wlens = bands = [];
    for(kFile=1;kFile<=nFiles;kFile++)
    {
        grow,wlens,*sci(kFile).EFF_WAVE;
        grow,bands,*sci(kFile).EFF_BAND;
    }

    // Get the wavelengths
    wlens = yocoStrRemoveMultiple(wlens,order);
    bands = bands(order);

    // Sort the wavelengths
    widx  = sort(wlens);
    wlens = wlens(widx);
    bands = bands(widx);

    wnz   = where(wlens!=0);
    wlens = wlens(wnz);
    bands = bands(wnz);

    minWlen = min(wlens);
    maxWlen = max(wlens);
    band    = abs((wlens)(dif));
    band    = band(where(band!=0));
    if(is_void(band))
        band=[avg(wlen)*0.1, avg(wlen)*0.1];
    
    bandwidth = median(bands);

    interv = maxWlen-minWlen;
    nWlen = int(interv/bandwidth)-1;

    if(nWlen<=1)
        nWlen = 2;
    waves = span(minWlen, maxWlen, nWlen);
    bands = array(bandwidth, nWlen);

    // FIXME: to remove
    // waves = span(8.e-6, 13e-6, 11);
    // bands = array(0.5e-6, 11);

    write,"*****************************************";
    write,"Min wlen:",minWlen, "Max wlen:",maxWlen;
    write,"Bandwidth:",bandwidth, "Nb wlen:",nWlen;
    write,"*****************************************";

    prout()
        }



/*---------------------------------------------------------------------------*/
/* UTILITIES, part of the code is from mira.i from E. Thiebaut               */


func mira_projected_gradient_norm(x, gx, xmin=, xmax=)
    /* DOCUMENT mira_projected_gradient_norm(x, gx, xmin=, xmax=)

       DESCRIPTION

       PARAMETERS
       - x   : 
       - gx  : 
       - xmin: 
       - xmax: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // FIXME: normalization not take into account
    local gp;
    if (is_void(xmin)) {
        eq_nocopy, gp, gx;
    } else {
        gp = gx*((gx < 0.0)|(x > xmin));
    }
    if (! is_void(xmax)) {
        gp *= ((gx > 0.0)|(x < xmax));
    }
    wb = where(abs(gp)>1e100);
    if(numberof(wb)!=0)
        gp(wb) = 1e100;
  
    return sqrt(sum(gp*gp));
}

func miral_polar_to_cartesian(amp, amperr, phi, phierr, what, goodman=, quiet=)
    /* DOCUMENT miral_polar_to_cartesian(amp, amperr, phi, phierr, what, goodman=, quiet=)
     *
     * Function from mira.i from E. Thiebaut with a bugfix
     *  mira.i is a free software under the GPL made by E. Thiebaut
     *
     *   Convert complex data given in polar coordinates (AMP,PHI) with
     *   their standard deviations (AMPERR,PHIERR) into cartesian
     *   coordinates (RE,IM) and associated noise model.  The result is
     *   a hash table:
     *
     *     DATA.re = real part of complex data
     *     DATA.im = imaginary part of complex data
     *     DATA.crr = variance of real part of complex data
     *     DATA.cii = variance of imaginary part of complex data
     *     DATA.cri = covariance of real and imaginary parts of complex data
     *     DATA.wrr = statistical weight for real part of residuals
     *     DATA.wii = statistical weight for imaginary part of residuals
     *     DATA.wri = statistical weight for real times imaginary parts of residuals
     *
     *   The quadratic penalty writes:
     *
     *     ERR =     DATA.wrr*(DATA.re - re)^2
     *           +   DATA.wii*(DATA.im - im)^2
     *           + 2*DATA.wri*(DATA.re - re)*(DATA.im - im);
     *
     * SEE ALSO: mira_data_penalty.
     */
{
    if(nallof(amp==amp))
    {
        amp(where(amp!=amp)) = 1.0;
    }
    if (min(amp) < 0.0)
    {
        write,"There are negative %s amplitudes!!! [FIXED]";
        // _mira_warn, swrite(format="There are negative %s amplitudes!!! [FIXED]", what);
        k = where(amp < 0.0);
        phi(k) += MIRA_PI;
        amp(k) *= -1.0;
    }
    cos_phi = cos(phi);
    sin_phi = sin(phi);
    re      = amp*cos_phi;
    im      = amp*sin_phi;
    case    = (amperr > 0.0) + 2*(phierr > 0.0);
    n0      = numberof((j0 = where(case == 0)));
    n1      = numberof((j1 = where(case == 1)));
    n2      = numberof((j2 = where(case == 2)));
    n3      = numberof((j3 = where(case == 3)));
    if (n0 || n1 || n2)
    {
        _mira_warn, swrite(format="there are %d out of %d invalid complex data",
                           (n0 + n1 + n2), numberof(case));
    }
    if (goodman)
    {
        /* Use Goodman approximation with SIGMA such that the area of ellipsoids
         * at one standard deviation are the same:
         *    PI*SIGMA^2 = PI*(RHO*SIGMA_THETA)*SIGMA_RHO
         * hence:
         *    SIGMA = sqrt(RHO*SIGMA_THETA*SIGMA_RHO)
         * If SIGMA_THETA or SIGMA_RHO is invalid, take:
         *    SIGMA = SIGMA_RHO
         *    SIGMA = RHO*SIGMA_THETA
         * accordingly.
         */
        wrr = array(double, dimsof(case));
        wri = array(double, dimsof(case));
        if (n3)
        {
            /* Valid amplitude and phase data. */
            wrr(j3) = 1.0/(amp(j3)*phierr(j3)*amperr(j3));
        }
        if (n2)
        {
            /* Only phase data. */
            if (0)
            {
                /* FIXME: requires amplitude. */
                err = amp(j2)*phierr(j2);
                j = where(err > 0.0);
                if (is_array(j))
                {
                    j2 = j2(j);
                    err = err(j);
                    wrr(j2) = 1.0/(err*err);
                }
            }
            else
            {
                err = phierr(j2);
                j = where(err > 0.0);
                if (is_array(j))
                {
                    j2 = j2(j);
                    err = err(j);
                    wrr(j2) = 1.0/(err*err);
                }
            }
        }
        if (n1)
        {
            /* Only amplitude data (FIXME: requires phase). */
            err     = amperr(j1);
            wrr(j1) = 1.0/(err*err);
        }
        wii = wrr;
    }
    else
    {
        /* Use convex quadratic local approximation. */
        wrr = wri = wii = array(double, dimsof(case));
        if (n3)
        {
            /* Valid amplitude and phase data. */
            err1 = amperr(j3);
            var1 = err1*err1;
            err2 = amp(j3)*phierr(j3);
            var2 = err2*err2;
            cs = cos_phi(j3);
            sn = sin_phi(j3);
            crr = cs*cs*var1 + sn*sn*var2;
            cri = cs*sn*(var1 - var2);
            cii = sn*sn*var1 + cs*cs*var2;

            //FIXME: bugfix when divide by zero
            nv1 = where(var1==0);
            if(numberof(nv1)!=0)
                var1(nv1) = 1.0;
            nv2 = where(var2==0);
            if(numberof(nv2)!=0)
                var2(nv2) = 1.0;
      
            a = 1.0/(var1*var2);
            wrr(j3) =  a*cii;
            wri(j3) = -a*cri;
            wii(j3) =  a*crr;
        }
        if (n2)
        {
            /* Only phase data (FIXME: requires amplitude). */
            err2 = amp(j2)*phierr(j2);
            j = where(err2 > 0.0);
            if (is_array(j))
            {
                j2 = j2(j);
                err2 = err2(j);
                var2 = err2*err2;
                cs = cos_phi(j2);
                sn = sin_phi(j2);
                a = 1.0/var2;
                wrr(j2) =  a*sn*sn;
                wri(j2) = -a*sn*cs;
                wii(j2) =  a*cs*cs;
            }
        }
        if (n1)
        {
            /* Only amplitude data (FIXME: requires phase). */
            err1 = amperr(j1);
            var1 = err1*err1;
            cs = cos_phi(j1);
            sn = sin_phi(j1);
            a = 1.0/var1;
            wrr(j1) =  a*cs*cs;
            wri(j1) =  a*cs*sn;
            wii(j1) =  a*sn*sn;
        }
    }  
    return h_new(re = re, im = im,
                 wrr = wrr, wri = wri, wii = wii);
}
