/***************************************************************************
 *                                                                         *
 *                              fitOmatic                                  *
 *                   Model-fitting prototyping utility                     *
 *                                                                         *
 *                      Copyright 2007, F. Millour                         *
 *                            fmillour@oca.eu                              *
 *                                                                         *
 ***************************************************************************
 *
 * "complex" models of wavelength-dependent objects
 *
 * Please note that this script is distributed under the GPL licence,
 * available at http://www.gnu.org/licenses/gpl.txt
 *
 * Please ACKNOWLEDGE the use of this script for any use in a publication
 *
 ***************************************************************************
 *
 * "@(#) $Id: photocentreModels.i 631 2015-11-24 18:10:52Z fmillour $"
 *
 ***************************************************************************/

func photocentreModels(void)
    /* DOCUMENT photocentreModels

       DESCRIPTION
       "complex" models of wavelength-dependent objects
       
       VERSION
       $Revision: 631 $

       REQUIRE
       fitOmatic.i
       yoco packages (yoco.i)

       CAUTIONS

       AUTHORS
       See AUTHORS file at the root of the directory distribution

       CONTRIBUTIONS

       FUNCTIONS
       - computePhotocentre: Computes the photocenter of an image cube
       - diskVelocityField : Computes the velocity field of a flat disk
       - dopplerit         : Convert kms into wavelength offset
       - expRotDiskToolbox : Rotating/expanding disk toolbox
       - expRotStarToolbox : Rotating/expanding star toolbox
       - expandingSphere   : Expanding sphere (star)
       - get_color         : Convert intensity and phase into [r,g,b]
       - imaModelExpRotDisk: Image of a rotating/expanding disk
       - imaModelExpRotStar: Image of a rotating/expanding star
       - limbDarkening     : Limb darkened disk
       - megaSpiral        : Spiral
       - photocentreModels : This script
       - rotatingSphere    : Rotating sphere (star)
       - uniformDisk       : Uniform disk
   
    */
{
    version = strpart(strtok("$Revision: 631 $",":")(2),2:-2);
    if (am_subroutine())
    {
        help, photocentreModels;
    }
    return version;
} 

yocoLogInfo,"#include \"photocentreModels.i\"";


include,"gauss.i"; // gauss.i from yutils, you may use the install script from F. Millour at https://www.oca.eu/spip.php?article563

/***************************************************************************/

func get_color(intensite, phase)
    /* DOCUMENT get_color(intensite, phase)
       
       DESCRIPTION
       Produce a [r,g,b] array given a phase and an intensity.
       
       PARAMETERS
       - intensite: 
       - phase    : 
       
       RETURN VALUES
       
       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(phase))
        phase = 0.0;
  
    if(is_void(intensite))
        intensite = 1.0;
  
    if(numberof(where(intensite>1)) != 0)
        intensite(where(intensite>1)) = 0;
    
    col = array(double,3,dimsof(phase));

    phi = 6*(phase%(2*pi))/(2*pi);
  
    col(1,) = 255 * intensite * ((phi <= 1.5) * (phi >= 0.) * 1.0 +
                                 (phi <= 3.) * (phi >= 1.5) * (3. - phi)/1.5 +
                                 (phi <= 6.) * (phi > 5.) * 1.0 +
                                 (phi <= 5.) * (phi > 4.) * (phi - 4.));
  
    col(2,) = 255 * intensite * ((phi <= 1.5) * phi / 1.5 +
                                 (phi <= 3.) * (phi > 1.5) *  1. +
                                 (phi <= 4.) * (phi > 3.) * (4. - phi));
    
    col(3,) = 255 * intensite * ((phi <= 3.) * (phi > 2.) * (phi - 2) +
                                 (phi <= 5.) * (phi > 3.) * 1.0 +
                                 (phi <= 6.) * (phi > 5.) * (6. - phi));
  
    return int(col);
}

/***************************************************************************/
           
func computePhotocentre(image)
    /* DOCUMENT computePhotocentre(image)

       DESCRIPTION

       PARAMETERS
       - image: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    nX = dimsof(image)(2);
    nY = dimsof(image)(3);
    x = indgen(nX);
    y = indgen(nY);
  
    sumIm = sum(image);
    
    xPhot = sum( image * x(,-) ) / (sumIm + (sumIm==0));
    yPhot = sum( image * y(-,) ) / (sumIm + (sumIm==0));
  
    return [xPhot,yPhot];
}
            
/***************************************************************************/
    
func dopplerit(vitesse, wlen)
    /* DOCUMENT dopplerit(vitesse, wlen)

       DESCRIPTION

       PARAMETERS
       - vitesse: 
       - wlen   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    require,"yoco.i";
    freq = 1 / wlen;
    freqDopplerisee = yocoAstroSI.c / (yocoAstroSI.c - vitesse) * freq;
    wlenDoppler = 1./freqDopplerisee;
    return wlenDoppler;
}
            
/***************************************************************************/
  
func uniformDisk(&dist, diamX=, diamY=, theta=, x0=, y0=, x=, y=)
    /* DOCUMENT uniformDisk(&dist, diamX=, diamY=, theta=, x0=, y0=, x=, y=)

       DESCRIPTION

       PARAMETERS
       - dist : 
       - diamX: 
       - diamY: 
       - theta: 
       - x0   : 
       - y0   : 
       - x    : 
       - y    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(x0))
        x0=0.0;

    if(is_void(y0))
        y0=0.0;

    if(is_void(theta))
        theta=0.0;

    if(is_void(diamX))
        diamX = 1.0;

    if(is_void(diamY))
        diamY = diamX;
    
    diamR = 200*max(diamX, diamY);

    if(is_void(x))
        x=span(-diamR,diamR,128);

    if(is_void(y))
        y = x;
    
    nPointsX = numberof(x);
    nPointsY = numberof(y);
    
    star = array(0.0,nPointsX,nPointsY);
  
    x1 = x-x0;
    y1 = y-y0;
  
    xPrim =   x1(,-) * cos(theta) + y1(-,) * sin(theta);
    yPrim = - x1(,-) * sin(theta) + y1(-,) * cos(theta);
    
    dist = abs(xPrim/diamX, yPrim/diamY);
    
    ok = dist < 0.5;
  
    inTheStar = where(ok);
  
    if(numberof(inTheStar)!=0)
        star(inTheStar) = 1;
  
    return star;  
}
            
/***************************************************************************/
            
func rotatingSphere(&validZone, vRot=, R=, oblatness=, i=, theta=, x0=, y0=, x=, y=)
    /* DOCUMENT rotatingSphere(&validZone, vRot=, R=, oblatness=, i=, theta=, x0=, y0=, x=, y=)

       DESCRIPTION

       PARAMETERS
       - validZone: 
       - vRot     : 
       - R        : 
       - oblatness: 
       - i        : 
       - theta    : 
       - x0       : 
       - y0       : 
       - x        : 
       - y        : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(x))
        x=span(-1,1,100);
    if(is_void(y))
        y = x;
    if(is_void(x0))
        x0=0.0;
    if(is_void(y0))
        y0=0.0;
    if(is_void(theta))
        theta=0.0;
    if(is_void(i))
        i = pi/2;
    if(is_void(R))
        R = 1.0;
    if(is_void(oblatness))
        oblatness = 1.0;
    if(is_void(vRot))
        vRot = 1000.0;
    
    vRad = 2 * vRot * sin(i) * ((x-x0)(,-) / oblatness * cos(theta) +
                                (y-y0)(-,) / oblatness * sin(theta)) / R;
   
    validZone =   uniformDisk(diamX = R * oblatness,
                              diamY = R / oblatness,
                              theta = theta,
                              x0 = x0, y0 = y0,
                              x = x, y = y);
   
    star = vRad * validZone;
  
    return star;
}
            
/***************************************************************************/
        
func expandingSphere(&validZone, vExp=, R=, oblatness=, i=, theta=, x0=, y0=, x=, y=)
    /* DOCUMENT expandingSphere(&validZone, vExp=, R=, oblatness=, i=, theta=, x0=, y0=, x=, y=)

       DESCRIPTION

       PARAMETERS
       - validZone: 
       - vExp     : 
       - R        : 
       - oblatness: 
       - i        : 
       - theta    : 
       - x0       : 
       - y0       : 
       - x        : 
       - y        : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(x))
        x=span(-1,1,100);
    
    if(is_void(y))
        y = x;
    
    if(is_void(x0))
        x0=0.0;
    
    if(is_void(y0))
        y0=0.0;
    
    if(is_void(theta))
        theta=0.0;
    
    if(is_void(i))
        i = pi/2;
    
    if(is_void(R))
        R = 1.0;
    
    if(is_void(oblatness))
        oblatness = 1.0;
    
    if(is_void(vExp))
        vRot = 1000.0;
  
    nPointsX = numberof(x);
    nPointsY = numberof(y);
    
    vitRad = array(0.0,nPointsX,nPointsY);

    validZone =   uniformDisk(dist, diamX=R * oblatness,
                              diamY=R / oblatness,
                              theta=theta,
                              x0=x0,y0=y0,
                              x=x,y=y);
  
    inTheStar = where(validZone);
    
    if(numberof(inTheStar)!=0)
        vitRad(inTheStar) = vExp * cos(pi * dist(inTheStar));
    
    return vitRad;  
}

/***************************************************************************/
  
func limbDarkening(&validZone, assomb=, R=, oblatness=, i=, theta=, x0=, y0=, x=, y=)
    /* DOCUMENT limbDarkening(&validZone, assomb=, R=, oblatness=, i=, theta=, x0=, y0=, x=, y=)

       DESCRIPTION

       PARAMETERS
       - validZone: 
       - assomb   : 
       - R        : 
       - oblatness: 
       - i        : 
       - theta    : 
       - x0       : 
       - y0       : 
       - x        : 
       - y        : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(x0))
        x0=0.0;
    if(is_void(y0))
        y0=0.0;
    if(is_void(theta))
        theta=0.0;
    if(is_void(i))
        i = pi/2;
    if(is_void(R))
        R = 1.0;
    if(is_void(oblatness))
        oblatness = 1.0;
    if(is_void(assomb))
        vRot = 1000.0;
    if(is_void(x))
        x = span(-1,1,100);
    if(is_void(y))
        y = x;
  
    nPointsX = numberof(x);
    nPointsY = numberof(y);
    
    flux = array(0.0,nPointsX,nPointsY);
   
    validZone = uniformDisk(dist, diamX = R * oblatness,
                            diamY = R / oblatness,
                            theta=theta,
                            x0=x0,y0=y0,
                            x=x,y=y);
   
    inTheStar = where(validZone);
    maxDist = max(dist(inTheStar));
   
    gamma = asin(dist(inTheStar)/maxDist);
    formula = (1 - assomb + assomb * cos(gamma));
    
    if(numberof(inTheStar)!=0)
        flux(inTheStar) = formula;
    
    return flux*(flux>0);  
}
  
/***************************************************************************/
  
func expRotStarToolbox(&starFlux, &starZone, &vitRad, x, y, x0, y0, R, oblatness, vRot, vExp, i, theta, totFlux, assomb)
    /* DOCUMENT expRotStarToolbox(&starFlux, &starZone, &vitRad, x, y, x0, y0, R, oblatness, vRot, vExp, i, theta, totFlux, assomb)

       DESCRIPTION

       PARAMETERS
       - starFlux : 
       - starZone : 
       - vitRad   : 
       - x        : 
       - y        : 
       - x0       : 
       - y0       : 
       - R        : 
       - oblatness: 
       - vRot     : 
       - vExp     : 
       - i        : 
       - theta    : 
       - totFlux  : 
       - assomb   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    starFlux = totFlux * limbDarkening(starZone,assomb=assomb,
                                       R=R,oblatness=oblatness,
                                       i=i,theta=theta,
                                       x0=x0,y0=y0,
                                       x=x,y=y);
   
    vitRad = expandingSphere(vExp=vExp,
                             R=R,oblatness=oblatness,
                             i=i,theta=theta,
                             x0=x0,y0=y0,
                             x=x,y=y);
    
    vitRad = vitRad + rotatingSphere(vRot=vRot,
                                     R=R,oblatness=oblatness,
                                     i=i,theta=theta,
                                     x0=x0,y0=y0,
                                     x=x,y=y);
}

/***************************************************************************/

func imaModelExpRotStar(diam, oblatness, theta, i, assomb, rapFluxCont, totFlux, vitRot, vitExp, lineCentralWlen, naturalLineWidth, obsWlen, &x, &y)
    /* DOCUMENT imaModelExpRotStar(diam, oblatness, theta, i, assomb, rapFluxCont, totFlux, vitRot, vitExp, lineCentralWlen, naturalLineWidth, obsWlen, &x, &y)

       DESCRIPTION

       PARAMETERS
       - diam            : 
       - oblatness       : 
       - theta           : 
       - i               : 
       - assomb          : 
       - rapFluxCont     : 
       - totFlux         : 
       - vitRot          : 
       - vitExp          : 
       - lineCentralWlen : 
       - naturalLineWidth: 
       - obsWlen         : 
       - x               : 
       - y               : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    oblatness2 = oblatness;
    if(1.0/oblatness2>oblatness2)
        oblatness2 = 1.0/oblatness;
    champVue  = diam/2.0*oblatness2;
    sigmaRaieLambda = abs(lineCentralWlen -
                          dopplerit(naturalLineWidth,lineCentralWlen));
    nPoints   = max(int(sqrt(lineCentralWlen/sigmaRaieLambda))/5,100);
    // nPoints   = 100;
    
    x = span(-champVue,champVue,nPoints);
    y = x;

    expRotStarToolbox,starFlux,starZone,vitRad,
        x,y,
        x0,y0,R,oblatness,
        vitRot,vitExp,i,theta,
        totFlux,assomb;
    
    colors = dopplerit(vitRad,lineCentralWlen) * double(starZone);
    
    vitRads = [];
    minVit = min(vitRad);
    maxVit = max(vitRad);
    
    vis = phi = spec = [];

    nChans = numberof(obsWlen);

    img = array(0.0,dimsof(starFlux)(2),dimsof(starFlux)(3),nChans);
    for(iChan=1;iChan<=nChans;iChan++)
    {
        profilRaieLocal = sigmaRaieLambda * sqrt(2*pi) *
            gauss(colors, [sigmaRaieLambda, obsWlen(iChan)]);
        
        imgMonochrom = (rapFluxCont * profilRaieLocal *
                        starFlux * (2 - starZone) + starFlux);
        imgMonochrom = imgMonochrom * (imgMonochrom>0);
        img(,,iChan) = imgMonochrom;
    }
    return img;
}


func diskVelocityField(x, y, taille, incl, beta, vrot, vinf, v0, gamma)
    /* DOCUMENT diskVelocityField(x, y, taille, incl, beta, vrot, vinf, v0, gamma)

       DESCRIPTION

       PARAMETERS
       - x     : 
       - y     : 
       - taille: 
       - incl  : 
       - beta  : 
       - vrot  : 
       - vinf  : 
       - v0    : 
       - gamma : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // x = (array(indgen(dim)-1.,dim)/(dim-1)-0.5)*2.*taille;
    // y = transpose(x)/cos(incl);
    ybis = y / cos(incl);

    r   = abs(x,ybis) + 1;
    phi = atan(x,ybis);
    
    vr = v0 + (vinf-v0) * (1. - 1./(r))^(gamma);

    vphi = vrot * r^(beta);
    
    vproj=(vphi*sin(phi)-vr*cos(phi))*sin(incl);

    return vproj;
}

/***************************************************************************/

func expRotDiskToolbox(&diskFlux, &vitRad, &starDisk, x, y, taille, incl, beta, vrot, vinf, v0, gamma)
    /* DOCUMENT expRotDiskToolbox(&diskFlux, &vitRad, &starDisk, x, y, taille, incl, beta, vrot, vinf, v0, gamma)

       DESCRIPTION

       PARAMETERS
       - diskFlux: 
       - vitRad  : 
       - starDisk: 
       - x       : 
       - y       : 
       - taille  : 
       - incl    : 
       - beta    : 
       - vrot    : 
       - vinf    : 
       - v0      : 
       - gamma   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    xy = grow(array(x,1),array(y,1));
    smallAxis = cos(incl) * taille;
    
    diskFlux = gauss2d(xy, [1, 0, 0, taille, smallAxis]);
    r = abs(x,y);
    starDisk = r<1;
    diskFlux = diskFlux*(1-starDisk);
    vitRad = diskVelocityField(x, y, taille, incl, beta, vrot, vinf, v0, gamma);
}

/***************************************************************************/

func imaModelExpRotDisk(fluxRaie, fluxStar, Rstar, deepStar, widthStar, distance, tailleCont, taille, incl, beta, vrot, vinf, v0, gamma, lineCentralWlen, naturalLineWidth, PAsky, obsWlen, &x, &y, &specTot, dim=, champVue=)
    /* DOCUMENT imaModelExpRotDisk(fluxRaie, fluxStar, Rstar, deepStar, widthStar, distance, tailleCont, taille, incl, beta, vrot, vinf, v0, gamma, lineCentralWlen, naturalLineWidth, PAsky, obsWlen, &x, &y, &specTot, dim=, champVue=)

       DESCRIPTION

       PARAMETERS
       - fluxRaie        : 
       - fluxStar        : 
       - Rstar           : 
       - deepStar        : 
       - widthStar       : 
       - distance        : 
       - tailleCont      : 
       - taille          : 
       - incl            : 
       - beta            : 
       - vrot            : 
       - vinf            : 
       - v0              : 
       - gamma           : 
       - lineCentralWlen : 
       - naturalLineWidth: 
       - PAsky           : 
       - obsWlen         : 
       - x               : 
       - y               : 
       - specTot         : 
       - dim             : 
       - champVue        : 

       RETURN VALUES
       te
       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    RstarRad = Rstar / (distance+(distance==0));

    if(is_void(dim))
        dim = DIM0;

    if(is_void(champVue))
        champVue = 10.*max(taille(1), tailleCont(1));
    
    x0 = (array(indgen(dim)-1.,dim)/(dim-1)-0.5)*champVue(1);
    y0 = transpose(x0);

    r = abs(x0,y0);
    t = atan(y0,x0);
    x = r*sin(t+PAsky(1));
    y = r*cos(t+PAsky(1));
    
    expRotDiskToolbox, diskFlux, vitRad, starDisk, x, y,
        taille(1), incl(1), beta(1), vrot(1), vinf(1), v0(1), gamma(1);
    
    xy = grow(array(x,1),array(y,1));
    smallAxisCont = cos(incl(1)) * tailleCont(1);

    diskFluxCont = gauss2d(xy, [1, 0, 0, tailleCont(1), smallAxisCont]);
    diskFluxCont = diskFluxCont*(1-starDisk);
    diskFluxCont = diskFluxCont / sum(diskFluxCont + allof(diskFluxCont==0));
    diskFlux     = diskFlux     / sum(diskFlux     + allof(diskFlux==0));

    starFlux     = double(starDisk) / sum(starDisk+allof(starDisk==0));
    specStar     = fluxStar(1) * gauss(obsWlen,
                                       [-deepStar(1), lineCentralWlen(1),
                                        widthStar(1), 1.0]);
    
    colors = dopplerit(vitRad,lineCentralWlen(1));
    minWlenColors = min(colors);
    maxWlenColors = max(colors);
    idxInLine = indgen(numberof(obsWlen));
    
    vis = phi = spec = [];
    
    nChans = numberof(obsWlen);

    
    img = specStar(-,-,) * starFlux(..,-) + diskFluxCont(..,-);

    specTot = array(1.0, numberof(obsWlen));
    
    //img = array(imgCont, nChans);
    for(iChan=1;iChan<=numberof(idxInLine);iChan++)
    {
        profilRaieLocal = gauss(colors,
                                [1,
                                 obsWlen(idxInLine(iChan)),
                                 naturalLineWidth(1)]);
        
        imgMonochrom = (fluxRaie(1) * profilRaieLocal * diskFlux +
                        img(,,idxInLine(iChan)));
        imgMonochrom = imgMonochrom * (imgMonochrom>0);
        specTot(idxInLine(iChan)) = sum(imgMonochrom);
        
        img(..,idxInLine(iChan)) = imgMonochrom;
    }
    img = img / (1.0+fluxStar(1));
    
    x = x0 * RstarRad(1);
    y = y0 * RstarRad(1);

    return img;
}


/***************************************************************************/ 

func megaSpiral(dim=, taille=, incl=, sigma=, phase=, period=, pow=, nspir=, ori=, rin=, rout=, star=, F=)
    /* DOCUMENT megaSpiral(dim=, taille=, incl=, sigma=, phase=, period=, pow=, nspir=, ori=, rin=, rout=, star=, F=)

       DESCRIPTION

       PARAMETERS
       - dim   : 
       - taille: 
       - incl  : 
       - sigma : 
       - phase : 
       - period: 
       - pow   : 
       - nspir : 
       - ori   : 
       - rin   : 
       - rout  : 
       - star  : 
       - F     : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{

    if (is_void(F)) F=1.;
    if (is_void(ori)) ori=0.;
    if (is_void(nspir)) nspir=1.;
    if (is_void(pow)) pow=1.;
    if (is_void(phase)) phase=0.;
    if (is_void(period)) period=taille;

    deg=pi/180.;
    incl=incl*deg;
    ori=ori*deg;

    x=(array(indgen(dim)-1.,dim)/(dim-1)-0.5)*2.*taille;

    y=transpose(x);

    xp=x*cos(ori)-y*sin(ori);
    yp=(x*sin(ori)+y*cos(ori))/cos(incl);

    r=sqrt(xp2+yp2);
    phi=atan(xp,yp);

    temp=2*(cos(nspir*(phase*deg/2.+phi/2.+pi*r/period))*2.-0.5);
    signe=abs(temp)/temp;
    map=exp(-r2/(2.*sigma2))*(1+F*signe*abs(temp)^pow);

    if (!(is_void(rin))) map=map*(r >= rin);
    if (!(is_void(rout))) map=map*(r <= rout);
    if (!(is_void(star))) map=map*(x2 + y2 > 1.);

    return map;
}
