/******************************************************
 *                                                    *
 *                     SAM-SUFFIT                     *
 *                                                    *
 *          Sparse Aperture Masking Software          *
 *      Utility For Fine Interferometric Trending     *
 *     Copyright F. Millour, MPIFR 2008, OCA 2010     *
 *                  fmillour@oca.eu                   *
 *                 All rights reserved                *
 *                                                    *
 ******************************************************/


require,"binaryEphemeris.i";
require,"amdlibPipeline.i";
require,"yeti.i";

/*************************************************************/
/* SAM masks positions and diameters from
   http://www.eso.org/sci/facilities/paranal/instruments/naco/inst/mask_datasheet.html */
/*************************************************************/

mask18HolesPos = 1.0*[[ -0.203155,     -3.87061],
                      [ -0.203155,     -4.57435],
                      [  -1.42208,     -1.75937],
                      [  -3.25047  ,  -0.703745],
                      [  -3.85992,      1.05562],
                      [  -3.85992  ,   -2.46311],
                      [    3.45362 ,     1.75936],
                      [   4.06308  ,    2.11124],
                      [   2.23470  ,  -0.351874],
                      [    2.23470  ,   -2.46311],
                      [   1.01577  ,   -3.87061],
                      [   4.06308  ,   -2.11124],
                      [    -3.25047 ,     2.11124],
                      [ -3.85992   ,   2.46311],
                      [  -0.812615  ,    2.11124],
                      [    1.01577 ,     3.16686],
                      [   2.84415  ,    2.81498],
                      [  -0.203153 ,     4.57435]];
mask18HolesDiam = 0.465;

/*************************************************************/

mask9HolesPos = transpose(Rotate_2D(transpose(1.03*[[
                                                     3.50441,     -2.60135],
                                                    [ 3.50441,      2.60135],
                                                    [ 2.00252 ,    -1.73423],
                                                    [ 0.500629 ,    -4.33558],
                                                    [ 0.500631  ,    2.60135],
                                                    [ 0.500631,      4.33558],
                                                    [ -2.50315 ,   -0.867115],
                                                    [ -4.00503  ,   -1.73423],
                                                    [  -4.00503   ,   1.73423]]),5*deg2rad));
mask9HolesDiam = 1.156;

/*************************************************************/

maskBB_9HolesPos = [[
                     -3.18399,    0.0607701],
                    [  -3.53717,      1.49530],
                    [ 0.0805017,      4.39864],
                    [   1.64462,      2.72703],
                    [   3.06355,      2.31563],
                    [   3.76908,     -2.26903],
                    [   1.53937,     -2.78780],
                    [  0.473616,     -3.81093],
                    [  -3.84958,     -2.12960]];
maskBB_9HolesDiam = 0.980;

/*************************************************************/

mask7HolesPos = transpose(Rotate_2D(transpose(
                                              1.03*[[   3.51064,     -1.99373],
                                                    [   3.51064 ,     2.49014],
                                                    [    1.56907,      1.36918],
                                                    [   1.56907 ,     3.61111],
                                                    [  -0.372507,     -4.23566],
                                                    [   -2.31408,      3.61111],
                                                    [   -4.25565,     0.248215]]),3*deg2rad));
mask7HolesDiam = 1.50;

/*************************************************************/

R = abs(mask7HolesPos(1,),mask7HolesPos(2,));
maxBase = max(R)+mask7HolesDiam/2.0;

/*************************************************************/

func samSuffit(void)
/* DOCUMENT samSuffit

       DESCRIPTION

       FUNCTIONS
   - samBatchCosmetic             : 
   - samBatchReduce               : 
   - samBatchReduceNormalNacoStuff: 
   - samComputeBS                 : 
   - samComputeV2                 : 
   - samCosmetic                  : 
   - samCreateCosmeticFile        : 
   - samCreateDARK                : 
   - samCreateFLAT                : 
   - samCreateMasters             : 
   - samCreateSKY                 : 
   - samDivideOiData              : 
   - samGauss2                    : 
   - samInfo                      : 
   - samPlotMask                  : 
   - samPlotV2fSpFreq             : 
   - samSetArrTarg                : 
   - samShow                      : 
   - samSplitUpFrame              : 
   - samSuffit                    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    version = strpart(strtok("$Revision: 552 $",":")(2),2:-2);
    if (am_subroutine())
    {
        help, samSuffit;
    }   
    return version;
}

write,"  ******************************************************\n\
   *                                                    *\n\
   *                     SAM-SUFFIT                     *"
    v="";
v="Version: "+samSuffit();
len=54;
len2= strlen(v);
strt =len/2-(len2+1.1)/2+1;
stp = len/2+(len2-1.1)/2+1;
count=1;
str="  *";
for(k=1;k<=strt-1;k++)
    str=str+" ";
str=str+v;
for(k=stp;k<=len-1;k++)
    str=str+" ";
str=str+"*";
write,str;
write,"  *                                                    *\n\
   *          Sparse Aperture Masking Software          *\n\
   *      Utility For Fine Interferometric Trending     *\n\
   *                                                    *\n\
   *               F. Millour, MPIFR 2008               *\n\
   *             fmillour@mpifr-bonn.mpg.de             *\n\
   *        All rights reserved (for the moment)        *\n\
   *                                                    *\n\
   ******************************************************\n";

/*************************************************************/

func samShow(file=)
/* DOCUMENT samShow(file=)

       DESCRIPTION

       PARAMETERS
   - file: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    amdlibFileChooser,"Enter the file you want to get information...",file;
    data = cfitsRead(file);
    med =  median(median(data));
    wkll;
    yocoNmCreate,1,2,2,square=1,width=900,height=1000;
    plsys,1;
    pli,data,cmin=0,cmax=med;
    plsys,2;
    pli,data,cmin=0,cmax=med*10;
    plsys,3;
    pli,data,cmin=0,cmax=med*100;
    plsys,4;
    pli,data,cmin=0,cmax=med*1000;
}

/*************************************************************/

func samPlotMask(maskPos, maskDiam, &U, &V, &D, &stations, fac=, color=, type=)
/* DOCUMENT samPlotMask(maskPos, maskDiam, &U, &V, &D, &stations, fac=, color=, type=)

       DESCRIPTION

       PARAMETERS
   - maskPos : 
   - maskDiam: 
   - U       : 
   - V       : 
   - D       : 
   - stations: 
   - fac     : 
   - color   : 
   - type    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    U= V = D = stations = [];
  
    if(is_void(fac))
        fac=1;
    nHoles = dimsof(maskPos)(0);
    for(k=1;k<=nHoles;k++)
    {
        for(l=k+1;l<=nHoles;l++)
        {
            par = span(-pi,pi,100);
            x1 = maskPos(1,k);
            y1 = maskPos(2,k);
            x2 = maskPos(1,l);
            y2 =  maskPos(2,l);
            x = -x1 + x2;
            y = y1 - y2;

            grow,U,fac*x;
            grow,V,fac*y;
            grow,D,fac*maskDiam;
            
            plg,fac*(y+maskDiam*cos(par)),
                fac*(x+maskDiam*sin(par)),color=color, type=type;
            plg,fac*(-y+maskDiam*cos(par)),
                fac*(-x+maskDiam*sin(par)),color=color, type=type;
          
            grow,stations,[[k,l]];
        }
    }
    limits,square=1;
}

/*************************************************************/

func samPlotV2fSpFreq(file)
/* DOCUMENT samPlotV2fSpFreq(file)

       DESCRIPTION

       PARAMETERS
   - file: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    /* Load the data */
    status = amdlibLoadOiData(wlen, bandWidth, time, fringeSNR,
                              sqVis, sqVisErr, uvCoord, bandFlag,
                              pistonOPDArray, sigmaPistonArray,
                              diffVisAmp, diffVisAmpErr,
                              diffVisPhase, diffVisPhaseErr,
                              vis3Amp, vis3AmpErr, vis3Phase, vis3PhaseErr,
                              cpxVis, cpxVisErr, visCovRI,
                              uv1Coord, uv2Coord,
                              fluxSumPiPj, fluxRatPiPj, PiMultPj,
                              spectrum, spectrumErr, inputOiFile=inputOiFile);
    
    sqVis=reform(sqVis,[1,numberof(sqVis)]);
    uvCoord = reform(uvCoord,[1,numberof(sqVis)]);
    U = uvCoord.u;
    V = uvCoord.v;
    R = abs(U,V);
    plg,sqVis,R,type="none",marker='\2';
    plg,sqVis(where(sqVis<0)),R(where(sqVis<0)),type="none",marker='\2',color="red";
    logxy,1,1;
}

/*************************************************************/

func samInfo(file)
/* DOCUMENT samInfo(file)

       DESCRIPTION

       PARAMETERS
   - file: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    extern starName;

    amdlibFileChooser,"Enter the file you want to get information...",file;
    keys = ["ESO DET WIN STARTX",
            "ESO DET WIN STARTY",
            "ESO DET WIN NX",
            "ESO DET WIN NY",
            "ESO DET DIT",
            "ESO DET NDIT",
            "ESO INS OPTI1 NAME",
            "ESO INS OPTI2 NAME",
            "ESO INS OPTI3 NAME",
            "ESO INS OPTI4 NAME",
            "ESO INS OPTI5 NAME",
            "ESO INS OPTI6 NAME",
            "ESO INS OPTI7 NAME",
            "ESO INS OPTI8 NAME",
            "ESO INS PIXSCALE",
            "ESO INS CWLEN",
            "TARGET",
            "ESO TARG NAME",
            "OBJECT",
            "ESO TEL TARG ALPHA",
            "ESO TEL TARG DELTA",
            "RA",
            "DEC",
            "ESO TEL PARANG START",
            "ESO TEL PARANG END",
            "ESO TEL TRAK STATUS",
            "ESO ADA ABSROT START",
            "ESO ADA ABSROT END",
            "MJD-OBS",
            "ESO TEL GEOELEV",
            "ESO TEL GEOLAT",
            "ESO TEL GEOLON",
            "ESO OBS TARG NAME",
            "ESO TEL TARG PMA",
            "ESO TEL TARG PMD",
            "ESO TEL TARG EQUINOX",
            "ESO TEL TARG PARALLAX",
            "ESO DET NCORRS NAME",
            "ESO DPR TYPE"
            ];
    dec = [2,2,2,2,1,2,0,0,0,0,0,0,0,0,1,1,0,0,0,0,0,1,1,1,1,0,1,1,1,1,1,1,0,1,1,1,1,0,0];
    vals = _amdlibGetKwdVals(file,keys,gui=0);
    
    write,yocoStrReplace(keys,[" ","-"],["_","_"]),vals(1,);
    for(k=1;k<=numberof(vals);k++)
    {
        //           print,vals(1,k)
        //       if((yocoStr2Double(vals(1,k))!=0)||(vals(1,k)=="0"))
        //         {
        //           if(yocoStr2Double(vals(1,k))==yocoStr2Long(vals(1,k)))
        //             symbol_set, yocoStrReplace(keys(k),[" ","-"],["_","_"]), yocoStr2Long(vals(1,k));
        //           else
        //             symbol_set, yocoStrReplace(keys(k),[" ","-"],["_","_"]), yocoStr2Double(vals(1,k));
        //         }
        //       else
        //         {
        //           symbol_set, yocoStrReplace(keys(k),[" ","-"],["_","_"]), vals(1,k);
        //         }
        
        if(dec(k)==1)
            symbol_set, yocoStrReplace(keys(k),[" ","-"],["_","_"]), yocoStr2Double(vals(1,k));
        else if(dec(k)==2)
            symbol_set, yocoStrReplace(keys(k),[" ","-"],["_","_"]), yocoStr2Long(vals(1,k));
        else
            symbol_set, yocoStrReplace(keys(k),[" ","-"],["_","_"]), vals(1,k);
    }
  
    if(TARGET!="-")
        starName = TARGET;
    else
    {
        if((OBJECT!="-")&&(OBJECT!="Object name not set"))
        {
            if(OBJECT=="LAMP")
                return 0;
            starName = OBJECT;
        }
        else
        {
            if(ESO_TARG_NAME!="-")
                starName = ESO_TARG_NAME;
            else
            {
                if((ESO_OBS_TARG_NAME!="-")&&(ESO_OBS_TARG_NAME!="No-Name"))
                    starName = ESO_OBS_TARG_NAME;
                else
                {
                    if((RA!=0)&&(RA!="-"))
                    {
                        coords = pr1(RA)+" "+pr1(DEC);
                    }
                    else if(ESO_TEL_TARG_ALPHA!="-")
                    {
                        a3 = strpart(ESO_TEL_TARG_ALPHA,-5:0);
                        a3b = strpart(ESO_TEL_TARG_ALPHA,1:-6);
                        a2 = strpart(a3b,-1:0);
                        a2b = strpart(a3b,1:-2);
                        a1 = strpart(a2b,1:0);
                        d3 = strpart(ESO_TEL_TARG_DELTA,-5:0);
                        d3b = strpart(ESO_TEL_TARG_DELTA,1:-6);
                        d2 = strpart(a3b,-1:0);
                        d2b = strpart(a3b,1:-2);
                        d1 = strpart(a2b,1:0);
                        coords = a1+" "+a2+" "+a3+" "+d1+" "+d2+" "+d3;
                    }
                    else
                    {
                        write,"Star name not set nor coordinates !!! Data useless !!!";
                        return 0;
                    }
                  
                    vals = vizQuery("HIP/hip_main",coords,"HD, Plx, e_Plx");
                    if(nallof(vals==string(nil)))
                    {
                        starName = "HD"+vals(1,0);
                        Plx = yocoStr2Double(vals(2,0));
                        e_Plx = yocoStr2Double(vals(3,0));
                        distance = 1000.0/Plx;
                        write,"In Hippacos, found",starName,distance,"Pc";
                    }
                    else
                    {
                        write,"Star not found in Hipparcos catalog, trying III/155 !!!";
                        vals = vizQuery("III/155",coords,"HD");
                        if(nallof(vals==string(nil)))
                        {
                            starName = "HD"+vals(1,0);
                            write,"In III/155, found",starName,distance,"Pc";
                        }
                        else
                        {
                            write,"Star not found in III/155 catalog, trying III/182 !!!";
                            vals = vizQuery("III/182",coords,"HD");
                            if(nallof(vals==string(nil)))
                            {
                                starName = "HD"+vals(1,0);
                                write,"In III/182, found",starName,distance,"Pc";
                            }
                            else
                            {
                                write,"Star not found in III/182 catalog, trying III/135 !!!";
                                vals = vizQuery("III/135",coords,"HD");
                                if(nallof(vals==string(nil)))
                                {
                                    starName = "HD"+vals(1,0);
                                    write,"In III/135, found",starName,distance,"Pc";
                                }
                                else
                                {
                                    write,"Star not found in any searched catalog. Setting name as coordinates";
                                    starName=coords;
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    write,starName;
    return 1;
}

/*************************************************************/

func samCosmetic(rawdata, BPM, FLAT, BIAS, GLOW, DIT, medDarkX=)
/* DOCUMENT samCosmetic(rawdata, BPM, FLAT, BIAS, GLOW, DIT, medDarkX=)

       DESCRIPTION

       PARAMETERS
   - rawdata : 
   - BPM     : 
   - FLAT    : 
   - BIAS    : 
   - GLOW    : 
   - DIT     : 
   - medDarkX: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    
    udark = BIAS + GLOW*DIT;

    dimz = dimsof(rawdata);

    if(dimz(1)==3)
        nbFrames = dimz(0);
    else
        nbFrames = 1;
    
    //  flux(-,-,..)/1.*
    flux = ((rawdata)*(1-BPM(..,-)))(avg,avg,,1);

    if(medDarkX==1)
    {
        rawdata1 = rawdata - median(rawdata,1)(-,..);
    }
    else
    {
        rawdata1 = rawdata;
    }
    rawdata=[];
    
    data = (rawdata1 - udark(..,-))/FLAT(..,-)*(1-BPM(..,-));
  
    reddata = data;
    for(k=1;k<=nbFrames;k++)
        reddata(,,k) = interpolateBadPixels(data(,,k),typeI=4);

    reddata = reddata // (10+median(reddata,1)(-,..))// *
        //                      median(reddata,2)(,-,..);
        //     reddata(1:50,..)(avg,..)(-,..);

        //render the frame square
        N = dimsof(reddata)(2);
    M = dimsof(reddata)(3);
    if(N!=M)
    {
        reddata = reddata(1:min(N,M),1:min(N,M),..);
    }
    
    //   fdark = fft(interpolateBadPixels(udark*(1-sBPM)),[1,1]);
    //   dsp0 = (abs(fdark)^2);
    //   window,w,wait=1;w++;
    //   pli,log(roll(dsp0));
  
    //   fdark = fft(interpolateBadPixels(BIAS*(1-sBPM)),[1,1]);
    //   dsp0 = (abs(fdark)^2);
    //   window,w,wait=1;w++;
    //   pli,log(roll(dsp0));
  
    //   fdark = fft(interpolateBadPixels(GLOW*(1-sBPM)),[1,1]);
    //   dsp0 = (abs(fdark)^2);
    //   window,w,wait=1;w++;
    //   pli,log(roll(dsp0));
  
    return reddata;
}

/*************************************************************/

func samCreateCosmeticFile(rawFile, BPM, FLAT, BIAS, GLOW, DIT)
/* DOCUMENT samCreateCosmeticFile(rawFile, BPM, FLAT, BIAS, GLOW, DIT)

       DESCRIPTION

       PARAMETERS
   - rawFile: 
   - BPM    : 
   - FLAT   : 
   - BIAS   : 
   - GLOW   : 
   - DIT    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    amdlibFileChooser,"Raw data file",rawFile;
    samInfo,rawFile;
    DIT = ESO_DET_DIT;
    rawdata = cfitsRead(rawFile);

    yocoFileSplitName,rawFile,d,f,e;
    outFile = d+f+"_COSM"+e;
    
    amdlibFileChooser,"Bad pixel mask",BPM;
    if(is_string(BPM))
    {
        write,"Reading file "+BPM;
        BPM = cfitsRead(BPM);
    }
    amdlibFileChooser,"Bias",BIAS;
    if(is_string(BIAS))
    {
        write,"Reading file "+BIAS;
        BIAS = cfitsRead(BIAS);
    }
    amdlibFileChooser,"Glow",GLOW;
    if(is_string(GLOW))
    {
        write,"Reading file "+GLOW;
        GLOW = cfitsRead(GLOW);
    }
    amdlibFileChooser,"Flat",FLAT;
    if(is_string(FLAT))
    {
        write,"Reading file "+FLAT;
        FLAT = cfitsRead(FLAT);
    }

    result = samCosmetic(rawdata, BPM, FLAT, BIAS, GLOW, DIT, medDarkX=);
    cfitsWrite,outFile,result;
}

/*************************************************************/

func samBatchCosmetic(fileDir)
/* DOCUMENT samBatchCosmetic(fileDir)

       DESCRIPTION

       PARAMETERS
   - fileDir: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    amdlibFileChooser,"Raw data directory",fileDir;
    amdlibGetLogName,fileDir, logName;
    amdlibReadLog,logName,vals,cols;
    
    types = vals(where(cols=="obs_type")(1),);
    names = vals(where(cols=="fileName")(1),);
    
    pouet()
        }

/*************************************************************/

func samCreateDARK(fileDir, &BPM, &BIAS, &GLOW)
/* DOCUMENT samCreateDARK(fileDir, &BPM, &BIAS, &GLOW)

       DESCRIPTION

       PARAMETERS
   - fileDir: 
   - BPM    : 
   - BIAS   : 
   - GLOW   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    amdlibFileChooser,,fileDir;
    write,"Creating dark...";
    amdlibGetLogName,fileDir, logName;
    amdlibReadLog,logName,vals,cols;
    
    types = vals(where(cols=="obs_type")(1),);
    names = vals(where(cols=="fileName")(1),);
    
    darks = where(types=="DARK");
    ND = numberof(darks);
    if(ND==0)
        error,"No suitable file found !";
    write,"Found "+pr1(ND)+" suitable files for this operation";
    
    read0 = cfitsRead(fileDir+names(darks(1)));
    nPix = dimsof(read0)(2);
    nPiy = dimsof(read0)(3);
    rawDark = array(0.0,nPix,nPiy,ND);
    rawExpTime = [];
    corrupt = ok = 0;
    for(k=1;k<=ND;k++)
    {
        yocoFileSplitName,fileDir+names(darks(k)),d,f,e;
        write,"reading "+f;
        read = cfitsRead(fileDir+names(darks(k)));
        // if(avg(read)<0)
        //         {
        //             write,"file corrupted !";
        //             corrupt++;
        //             continue;
        //         }
        //         else
        {
            ok++;
            rawDark(..,ok) = read;
            grow,rawExpTime,yocoStr2Double(_amdlibGetKwdVals(fileDir+names(darks(k)),"ESO DET DIT",gui=0)(1));
        }
    }

    rawDark = rawDark(..,1:-corrupt);
    
    N = dimsof(rawDark)(2);
    M = dimsof(rawDark)(3);

    winkill,0;
    window,0,wait=1,width=500,height=500;
    xx = span(0,max(rawExpTime),100);
    
    elapsed= array(double, 3);
    time0 = count = 0;
    glow = bias = array(0.0,N,M);
    write,"Computing...";
    for(k=1;k<=N;k++)
    {
        for(l=1;l<=M;l++)
        {
            res = yocoMathPolyFit(1, rawExpTime, rawDark(k,l,));
            bias(k,l) = res(1);
            glow(k,l) = res(2);
            
            
            count++;
            timer, elapsed;
            if(abs(elapsed(3)-time0)>2)
            {
                time0 = elapsed(3);
                write,"Finished "+swrite(format="%2.2g",count/double(N*M)*100)+"%";
                fma;
                plg,rawDark(k,l,),rawExpTime,type="none",marker='\2';
                plg,yocoMathPoly(xx,res),xx;
                xytitles,"exp. time (s)","ADU";
                limits,0,,0;
            }
        }
    }
    
    glowMed = median(median(glow));
    biasRms = bias(rms,rms);
    BPM_fromDark = (abs(glow)>3*glowMed)|
        (glow<0)|
        (abs(bias-avg(bias))>0.05*abs(biasRms));
    
    BPM = BPM_fromDark;
    
    BPM(1:50,509)=1;
    BPM(1:100,509)=1;
    BPM(,510)=1;
    BPM(,511)=1;
    BPM(,512)=1;
    BPM(,513)=1;
      
    BIAS = bias;
    GLOW = glow;

    file = fileDir+"BIAS.fits";
    write,"Writing file "+fileDir+"BIAS.fits";
    cfitsWrite,file,BIAS;
    fh = cfitsio_open(file,"a");
    keyName = "ESO DPR TYPE"//swrite(format="ESO DPR TYPE", recNb, index);
        cfitsio_write_key, fh, keyName, "REDUCED, BIAS", "Category of OI-FITS file";
    cfitsio_close,fh;
    
    write,"Writing file "+fileDir+"GLOW.fits";
    cfitsWrite,fileDir+"GLOW.fits.gz",GLOW;
    fh = cfitsio_open(file,"a");
    keyName = "ESO DPR TYPE"//swrite(format="ESO DPR TYPE", recNb, index);
        cfitsio_write_key, fh, keyName, "REDUCED, BIAS", "Category of OI-FITS file";
    cfitsio_close,fh;
    
    write,"Writing file "+fileDir+"BPM.fits";
    cfitsWrite,fileDir+"BPM.fits.gz",BPM;
    fh = cfitsio_open(file,"a");
    keyName = "ESO DPR TYPE"//swrite(format="ESO DPR TYPE", recNb, index);
        cfitsio_write_key, fh, keyName, "REDUCED, BIAS", "Category of OI-FITS file";
    cfitsio_close,fh;

    write,"Finished !";
}

/*************************************************************/

func samCreateFLAT(fileDir, BPM, BIAS, GLOW, &BPM_OUT, &FLAT)
/* DOCUMENT samCreateFLAT(fileDir, BPM, BIAS, GLOW, &BPM_OUT, &FLAT)

       DESCRIPTION

       PARAMETERS
   - fileDir: 
   - BPM    : 
   - BIAS   : 
   - GLOW   : 
   - BPM_OUT: 
   - FLAT   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    amdlibFileChooser,"Directory",fileDir;
    amdlibFileChooser,"Bad pixel mask",BPM;
    if(is_string(BPM))
    {
        write,"Reading file "+BPM;
        BPM = cfitsRead(BPM);
    }
    amdlibFileChooser,"Bias",BIAS;
    if(is_string(BIAS))
    {
        write,"Reading file "+BIAS;
        BIAS = cfitsRead(BIAS);
    }
    amdlibFileChooser,"Glow",GLOW;
    if(is_string(GLOW))
    {
        write,"Reading file "+GLOW;
        GLOW = cfitsRead(GLOW);
    }
    write,"Creating flat...";
    amdlibGetLogName,fileDir, logName;
    amdlibReadLog,logName,vals,cols;
    
    types = vals(where(cols=="obs_type")(1),);
    names = vals(where(cols=="fileName")(1),);
    
    flats = where(strmatch(types,"FLAT,LAMP")|
                  strmatch(types,"FLAT,SKY"));
    NF = numberof(flats);
    write,"Found "+pr1(NF)+" suitable files for this operation";

    keys = _amdlibGetKwdVals(fileDir+names(flats),
                             ["ESO DET DIT",
                              "ESO INS OPTI1 NAME",
                              "ESO INS OPTI2 NAME",
                              "ESO INS OPTI3 NAME",
                              "ESO INS OPTI4 NAME",
                              "ESO INS OPTI5 NAME",
                              "ESO INS OPTI6 NAME",
                              "ESO INS OPTI7 NAME"],gui=0);
    rawExpTime = yocoStr2Double(keys(,1));
    opt = keys(,2:);
    opt1 = yocoStrRemoveMultiple(opt(,1));
    n1 = numberof(opt1);
    opt2 = yocoStrRemoveMultiple(opt(,2));
    n2 = numberof(opt2);
    opt3 = yocoStrRemoveMultiple(opt(,3));
    n3 = numberof(opt3);
    opt4 = yocoStrRemoveMultiple(opt(,4));
    n4 = numberof(opt4);
    opt5 = yocoStrRemoveMultiple(opt(,5));
    n5 = numberof(opt5);
    opt6 = yocoStrRemoveMultiple(opt(,6));
    n6 = numberof(opt6);
    opt7 = yocoStrRemoveMultiple(opt(,7));
    n7 = numberof(opt7);

    for(k1=1;k1<=n1;k1++)
    {
        for(k2=1;k2<=n2;k2++)
        {
            for(k3=1;k3<=n3;k3++)
            {
                for(k4=1;k4<=n4;k4++)
                {
                    for(k5=1;k5<=n5;k5++)
                    {
                        for(k6=1;k6<=n6;k6++)
                        {
                            for(k7=1;k7<=n7;k7++)
                            {
                                test = where((opt1(k1)==opt(,1))&                            
                                             (opt2(k2)==opt(,2))&
                                             (opt3(k3)==opt(,3))&
                                             (opt4(k4)==opt(,4))&
                                             (opt5(k5)==opt(,5))&
                                             (opt6(k6)==opt(,6))&
                                             (opt7(k7)==opt(,7)));
                                if(numberof(test)==0)
                                    continue;

                                write,"proceeding for the following configuration: "+opt1(k1)+", "+opt2(k2)+", "+opt3(k3)+", "+opt4(k4)+", "+opt5(k5)+", "+opt6(k6)+", "+opt7(k7);
                                files = flats(test);
                                expTime = rawExpTime(test);
                                NF = numberof(files);
        
                                read0 = cfitsRead(fileDir+names(files(1)));
                                nPix = dimsof(read0)(2);
                                nPiy = dimsof(read0)(3);
                                rawFlat = array(0.0,nPix,nPiy,NF);
                                corrupt = ok = 0;
    
                                write,"Computing...";
                                for(k=1;k<=NF;k++)
                                {
                                    yocoFileSplitName,fileDir+names(files(k)),d,f,e;
                                    write,"reading "+f;
                                    read = cfitsRead(fileDir+names(files(k)));
                                    if(avg(read)<0)
                                    {
                                        write,"file corrupted !";
                                        corrupt++;
                                        continue;
                                    }
                                    else
                                    {
                                        ok++;
                                        rawFlat(..,ok) = read;
                                    }
                                }

                                rawFlat = rawFlat(..,1:-corrupt);
                                for(k=1;k<=NF-corrupt;k++)
                                {
                                    rawFlat(..,k) = (rawFlat(..,k) - BIAS - expTime(k)*GLOW)/expTime(k);
                                }

                                rawFlat = median(rawFlat,0);
                                avgRawFlat = avg(rawFlat*(1-BPM));
                                rawFlat = rawFlat/avgRawFlat;
                                FLAT = rawFlat;
                                BPM_OUT = BPM|(rawFlat>1.2)|(rawFlat<0.8);

                                name = fileDir+"FLAT_"+
                                    opt1(k1)+"_"+opt2(k2)+"_"+opt3(k3)+"_"+
                                    opt4(k4)+"_"+opt5(k5)+"_"+opt6(k6)+"_"+
                                    opt7(k7)+".fits.gz"
                                    write,"Writing file "+name;
                                cfitsWrite,name,FLAT;
                                
                                name = fileDir+"BPM_"+
                                    opt1(k1)+"_"+opt2(k2)+"_"+opt3(k3)+"_"+
                                    opt4(k4)+"_"+opt5(k5)+"_"+opt6(k6)+"_"+
                                    opt7(k7)+".fits.gz"
                                    write,"Writing file "+name;
                                cfitsWrite,name,BPM_OUT;
                                
                                write,"Done !";
                            }
                        }
                    }
                }
            }
        }
    }
}

/*************************************************************/

func samCreateSKY(fileDir, BPM, BIAS, GLOW, FLAT, &SKY)
/* DOCUMENT samCreateSKY(fileDir, BPM, BIAS, GLOW, FLAT, &SKY)

       DESCRIPTION

       PARAMETERS
   - fileDir: 
   - BPM    : 
   - BIAS   : 
   - GLOW   : 
   - FLAT   : 
   - SKY    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    amdlibFileChooser,"Directory",fileDir;
    amdlibFileChooser,"Bad pixel mask",BPM;
    if(is_string(BPM))
    {
        write,"Reading file "+BPM;
        BPM = cfitsRead(BPM);
    }
    amdlibFileChooser,"Bias",BIAS;
    if(is_string(BIAS))
    {
        write,"Reading file "+BIAS;
        BIAS = cfitsRead(BIAS);
    }
    amdlibFileChooser,"Glow",GLOW;
    if(is_string(GLOW))
    {
        write,"Reading file "+GLOW;
        GLOW = cfitsRead(GLOW);
    }
    amdlibFileChooser,"Flat",FLAT;
    if(is_string(FLAT))
    {
        write,"Reading file "+FLAT;
        FLAT = cfitsRead(FLAT);
    }
    write,"Creating data...";
    amdlibGetLogName,fileDir, logName;
    amdlibReadLog,logName,vals,cols;
    
    types = vals(where(cols=="obs_type")(1),);
    catg = vals(where(cols=="obs_catg")(1),);
    names = vals(where(cols=="fileName")(1),);
    
    datas = where(strmatch(types,"OBJECT")&
                  strmatch(catg,"SCIENCE"));
    NF = numberof(datas);
    write,"Found "+pr1(NF)+" suitable files for this operation";

    keys = _amdlibGetKwdVals(fileDir+names(datas),
                             ["ESO DET DIT",
                              "ESO INS OPTI1 NAME",
                              "ESO INS OPTI2 NAME",
                              "ESO INS OPTI3 NAME",
                              "ESO INS OPTI4 NAME",
                              "ESO INS OPTI5 NAME",
                              "ESO INS OPTI6 NAME",
                              "ESO INS OPTI7 NAME"],gui=0);
    rawExpTime = yocoStr2Double(keys(,1));

    opt0 = yocoStrRemoveMultiple(rawExpTime(,1));
    n0 = numberof(opt0);
    opt = keys(,2:);
    opt1 = yocoStrRemoveMultiple(opt(,1));
    n1 = numberof(opt1);
    opt2 = yocoStrRemoveMultiple(opt(,2));
    n2 = numberof(opt2);
    opt3 = yocoStrRemoveMultiple(opt(,3));
    n3 = numberof(opt3);
    opt4 = yocoStrRemoveMultiple(opt(,4));
    n4 = numberof(opt4);
    opt5 = yocoStrRemoveMultiple(opt(,5));
    n5 = numberof(opt5);
    opt6 = yocoStrRemoveMultiple(opt(,6));
    n6 = numberof(opt6);
    opt7 = yocoStrRemoveMultiple(opt(,7));
    n7 = numberof(opt7);

    for(k0=1;k0<=n0;k0++)
    {
        for(k1=1;k1<=n1;k1++)
        {
            for(k2=1;k2<=n2;k2++)
            {
                for(k3=1;k3<=n3;k3++)
                {
                    for(k4=1;k4<=n4;k4++)
                    {
                        for(k5=1;k5<=n5;k5++)
                        {
                            for(k6=1;k6<=n6;k6++)
                            {
                                for(k7=1;k7<=n7;k7++)
                                {
                                    test = where((opt1(k1)==opt(,1))&       
                                                 (opt0(k0)==rawExpTime)&
                                                 (opt2(k2)==opt(,2))&
                                                 (opt3(k3)==opt(,3))&
                                                 (opt4(k4)==opt(,4))&
                                                 (opt5(k5)==opt(,5))&
                                                 (opt6(k6)==opt(,6))&
                                                 (opt7(k7)==opt(,7)));
                                    if(numberof(test)==0)
                                        continue;

                                    write,"proceeding for the following configuration: "+pr1(opt0(k0))+", "+opt1(k1)+", "+opt2(k2)+", "+opt3(k3)+", "+opt4(k4)+", "+opt5(k5)+", "+opt6(k6)+", "+opt7(k7);
                                    files = datas(test);
                                    expTime = rawExpTime(test);
                                    NF = numberof(files);
        
                                    read0 = cfitsRead(fileDir+names(files(1)));
                                    nPix = dimsof(read0)(2);
                                    nPiy = dimsof(read0)(3);
                                    rawData = array(0.0,nPix,nPiy,NF);
                                    corrupt = ok = 0;
    
                                    write,"Computing...";
                                    for(k=1;k<=NF;k++)
                                    {
                                        yocoFileSplitName,fileDir+names(files(k)),d,f,e;
                                        write,"reading "+f;
                                        read = cfitsRead(fileDir+names(files(k)));
                                        if(0)
                                        {
                                            write,"file corrupted !";
                                            corrupt++;
                                            continue;
                                        }
                                        else
                                        {
                                            ok++;
                                            rawData(..,ok) = read;
                                        }
                                    }

                                    rawData = rawData(..,1:-corrupt);
                                    for(k=1;k<=NF-corrupt;k++)
                                    {
                                        rawData(..,k) = (rawData(..,k) - BIAS - expTime(k)*GLOW)/FLAT;
                                    }

                                    rawData = median(rawData,0);

                                    name = fileDir+"SKY_"+
                                        pr1(opt0(k0))+"_"+
                                        opt1(k1)+"_"+opt2(k2)+"_"+opt3(k3)+"_"+
                                        opt4(k4)+"_"+opt5(k5)+"_"+opt6(k6)+"_"+
                                        opt7(k7)+".fits.gz"
                                        write,"Writing file "+name;
                                    cfitsWrite,name,rawData;
                                                                
                                    write,"Done !";
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}

/*************************************************************/

func samCreateMasters(inRawCalDir)
/* DOCUMENT samCreateMasters(inRawCalDir)

       DESCRIPTION

       PARAMETERS
   - inRawCalDir: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    amdlibFileChooser,"Enter the raw master files\n(sky flats and darks) directory",inRawCalDir;
    //To generate flat and bias files from a directory with sky flats and darks
    cd,inRawCalDir;
    samCreateDARK,inRawCalDir, BPM, BIAS, GLOW;
    samCreateFLAT,inRawCalDir, BPM, BIAS, GLOW, BPM_FINAL, FLAT;
}

/*************************************************************/

func samComputeV2(dsp, U, V, Umask, Vmask, Dmask, &flux, &Ucen, &Vcen, &FCcen, &fluxCen, nIterInterp=, facDiamBckgnd=)
/* DOCUMENT samComputeV2(dsp, U, V, Umask, Vmask, Dmask, &flux, &Ucen, &Vcen, &FCcen, &fluxCen, nIterInterp=, facDiamBckgnd=)

       DESCRIPTION

       PARAMETERS
   - dsp          : 
   - U            : 
   - V            : 
   - Umask        : 
   - Vmask        : 
   - Dmask        : 
   - flux         : 
   - Ucen         : 
   - Vcen         : 
   - FCcen        : 
   - fluxCen      : 
   - nIterInterp  : 
   - facDiamBckgnd: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(nIterInterp))
        nIterInterp = 10;

    if(is_void(facDiamBckgnd))
        facDiamBckgnd = 1.5;
    
    N = numberof(Umask);
    Npix = dimsof(U)(2);
  
    r0 = 2*abs(U,V);
    r = r0 / Dmask(1)/2;
    i0 = ((r <= 1)*(V >= 0));

    //     window,w,wait=1;w++;
    //     fma;
    //     plf,i0*dsp,V,U;
  
    Ucen = U(where(i0));
    Vcen = V(where(i0));
    FCcen = dsp(where(i0));
    fluxCen = max(dsp(where(i0)));
  
    i1 = (r <= 1);
    flux = sum(i0*dsp);
    MASK = (r <= facDiamBckgnd);
  
    for(k=1;k<=N;k++)
    {      
        r0 = 2*sqrt((U-Umask(k))^2+(V-Vmask(k))^2);
        r = r0 / Dmask(k)/2;
        MASK = MASK|(r <= facDiamBckgnd);
    }
  
    MASK = MASK(::-1,::-1)|MASK;
    //   window,w,wait=1;w++;
    //   fma;
    //   pli,log(dsp*(1-MASK)+(MASK));

    bckgnd = [];
    for(k=1;k<=nIterInterp;k++)
    {
        t = int(random()*2)+3;
        grow, bckgnd, array(interpolateBadPixels(dsp*(1-MASK),typeI=t),1);
    }
    bckgnd = (median(bckgnd,0)+bckgnd(..,avg))/2.0;
    bckgnd = (roll(bckgnd(::-1,::-1),[1,1])+bckgnd)/2.0;
  
    window,w,wait=1;w++;
    fma;
    pli,log(bckgnd);
    yocoPlotColorLookupTable,min(bckgnd),max(bckgnd);
  
    window,w,wait=1;
    window,w+1,wait=1;
    FC2 = [];
    for(k=1;k<=N;k++)
    {      
        r0 = 2*sqrt((U-Umask(k))^2+(V-Vmask(k))^2);
        r = r0 / Dmask(k)/2;
        i0 = (r <= 1);
      
        R = abs(Umask(k),Vmask(k));
        fc2 = sum(i0*(dsp-bckgnd));
        grow,FC2,fc2;
      
        window,w;
        fma;
        pli,log(((dsp-bckgnd)>=0)*(1-i0)*(dsp-bckgnd)+((dsp-bckgnd)<=0)+i0),cmin=0;
      
        window,w+1;
        if(fc2/flux*N<0)
            colo="red";
        else
            colo = "black";
      
        plg,fc2/flux*N,R,type="none",marker='\2',color=colo;
      
        logxy,0,1;
        limits;
    }
    
    window,w;
    fma;
    pli,log(((dsp-bckgnd)>=0)*(dsp-bckgnd)+((dsp-bckgnd)<=0)),cmin=0;
    
    w++;
    w++;

    return FC2;
}

/*************************************************************/

func samComputeBS(fft, U, V, Umask, Vmask, Dmask, &flux)
/* DOCUMENT samComputeBS(fft, U, V, Umask, Vmask, Dmask, &flux)

       DESCRIPTION

       PARAMETERS
   - fft  : 
   - U    : 
   - V    : 
   - Umask: 
   - Vmask: 
   - Dmask: 
   - flux : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    N = numberof(Umask);
    Npix = dimsof(U)(2);
  
    r0 = 2*abs(U,V);
    r = r0 / Dmask(1)/2;
    i0 = (r <= 1);
    flux = sum(i0*dsp);
    MASK = (r <= 1.8);
    
    pause,100;   
  
    for(k=1;k<=N;k++)
    {      
        r0 = 2*sqrt((U-Umask(k))^2+(V-Vmask(k))^2);
        r = r0 / Dmask(k)/2;
        MASK = MASK|(r <= 1.8);
    }
  
    MASK = MASK(::-1,::-1)|MASK;
    window,w,wait=1;w++;
    fma;
    pli,log(dsp*(1-MASK)+(MASK));

    bckgnd = [];
    for(k=1;k<=nIterInterp;k++)
    {
        grow, bckgnd, array(interpolateBadPixels(dsp*(1-MASK),type=3),1);
    }
    bckgnd = (median(bckgnd,0)+bckgnd(..,avg))/2.0;
  
    window,w,wait=1;w++;
    fma;
    pli,log(bckgnd);
    yocoPlotColorLookupTable,min(bckgnd),max(bckgnd)

        FC2 = [];
    for(k=1;k<=N;k++)
    {      
        r0 = 2*sqrt((U-Umask(k))^2+(V-Vmask(k))^2);
        r = r0 / Dmask(k)/2;
        i0 = (r <= 1);

        R = abs(Umask(k),Vmask(k));
        fc2 = sum(i0*(dsp-bckgnd));
        grow,FC2,fc2;
      
        window,w,wait=1;
        fma;
        pli,log(((dsp-bckgnd)>=0)*(1-i0)*(dsp-bckgnd)+((dsp-bckgnd)<=0)+i0);

        window,w+1;
        plg,fc2/flux*N,R,type="none",marker='\2';
        logxy,0,1;
        limits;
      
        //pause,100;
    }
    w++;
    return FC2;
}

/*************************************************************/

// check for gussian shape of the PSF
func samGauss2(x, a)
/* DOCUMENT samGauss2(x, a)

       DESCRIPTION

       PARAMETERS
   - x: 
   - a: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    return a(1)*
        sqrt(2*pi)*a(2)*gauss(x(..,1),[a(2),a(3)])*
        sqrt(2*pi)*a(2)*gauss(x(..,2),[a(2),a(4)])+a(5);
}

/*************************************************************/

func samSplitUpFrame(dataCube, &data1, &data2, splitSz=)
/* DOCUMENT samSplitUpFrame(dataCube, &data1, &data2, splitSz=)

       DESCRIPTION

       PARAMETERS
   - dataCube: 
   - data1   : 
   - data2   : 
   - splitSz : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(splitSz))
        splitSz = 128;
    dataMed = median(dataCube,0);
    N = dimsof(dataCube)(2);
    peak1 = where2(dataMed==max(dataMed));
  
  
    a = [max(dataMed),3,peak1(1),peak1(2),avg(dataMed)];
  
    x = array(indgen(N),N);
    y = transpose(x);
    xy = grow([x],[y]);
    require,"~/CVS/fitOmatic/lmfit.i";
    weights = abs(dataMed*(dataMed>0));
    // r = lmfit(samGauss2, xy, a, dataMed, weights);
    psfSz1=a;
    resFit1=r;
    psfNoWing1 = samGauss2(xy,psfSz1);
  
    window,w,wait=1;w++;
    fma;
    pli,dataMed-psfNoWing1;
    plg,peak1(2),peak1(1),type="none",color="blue",marker='\5';

    dataMed2 = dataMed-psfNoWing1;
    peak2 = where2(dataMed2==max(dataMed2));
  
  
    a = [max(dataMed2),3,peak2(1),peak2(2),avg(dataMed2)];
  
    x = array(indgen(N),N);
    y = transpose(x);
    xy = grow([x],[y]);
    require,"~/CVS/fitOmatic/lmfit.i";
    weights = abs(dataMed2*(dataMed2>0));
    // r = lmfit(samGauss2, xy, a, dataMed2, weights);
    psfSz2=a;
    resFit2=r;
    psfNoWing2 = samGauss2(xy,psfSz2);
    
    fma;
    pli,dataMed2-psfNoWing2;
    plg,peak1(2),peak1(1),type="none",color="blue",marker='\5';
    plg,peak2(2),peak2(1),type="none",color="red",marker='\5';

    cenx1 = int(psfSz1(3));
    if(cenx1<splitSz)
        cenx1=splitSz;
    if(cenx1>N-splitSz)
        cenx1=N-splitSz;
    ceny1 = int(psfSz1(4));
    if(ceny1<splitSz)
        ceny1=splitSz;
    if(ceny1>N-splitSz)
        ceny1=N-splitSz;
    cenx2 = int(psfSz2(3));
    if(cenx2<splitSz)
        cenx2=splitSz;
    if(cenx2>N-splitSz)
        cenx2=N-splitSz;
    ceny2 = int(psfSz2(4));
    if(ceny2<splitSz)
        ceny2=splitSz;
    if(ceny2>N-splitSz)
        ceny2=N-splitSz; 

    data1 = dataCube(cenx1-splitSz+1:cenx1+splitSz,
                     ceny1-splitSz+1:ceny1+splitSz,..);
    if((abs(cenx1-cenx2)>splitSz/1.2)||(abs(ceny1-ceny2)>splitSz/1.2))
        data2 = dataCube(cenx2-splitSz+1:cenx2+splitSz,
                         ceny2-splitSz+1:ceny2+splitSz,..);
    else
    {
        write,"!!! WARNING !!! Only one fringe pattern detected";
        data2=[];
    }
    
}

/*************************************************************/

func samBatchReduce(inRawDir, inBPM, inBias, inGlow, inFlat)
/* DOCUMENT samBatchReduce(inRawDir, inBPM, inBias, inGlow, inFlat)

       DESCRIPTION

       PARAMETERS
   - inRawDir: 
   - inBPM   : 
   - inBias  : 
   - inGlow  : 
   - inFlat  : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{

    amdlibFileChooser,"Enter the bad pixels file",inBPM;
    amdlibFileChooser,"Enter the detector bias map",inBias;
    amdlibFileChooser,"Enter the detector glow map",inGlow;
    amdlibFileChooser,"Enter the flat field map",inFlat;
    amdlibFileChooser,"Enter the raw files directory",inRawDir;
  
    BPM = cfitsRead(inBPM);
    BIAS = cfitsRead(inBias);
    GLOW = cfitsRead(inGlow);
    FLAT = cfitsRead(inFlat);
  
    // Uncomment here if you want to plot the master files
    w=0;
    window,w,wait=1,width=400,height=400;w++;
    pli,FLAT,cmin=0.8,cmax=1.2;
    window,w,wait=1;w++;
    pli,BPM;
    window,w,wait=1;w++;
    pli,BIAS*(1-BPM);
    window,w,wait=1;w++;
    pli,GLOW*(1-BPM);

    // Uncomment here
    files = lsdir(inRawDir);
    file = inRawDir+files(where(strmatch(files,".fits")&(!strmatch(files,"OIVIS"))));
    nFiles = numberof(file);

    kFile=1;
    for(kFile=1;kFile<=nFiles;kFile++)
    {
        yocoFileSplitName,file(kFile),d,f,e;
        write,"File to load is "+f+e;
    
        result = samInfo(file(kFile));
      
        if(result==0)
            continue;

        //Set the output file and check if it exists
        outputFileName = d+f+"_OIVIS"+e;
        outputFileNameb = d+f+"_OIVISb"+e;
        if(open(outputFileName,"r",1))
        {
            if(strmatch(ESO_INS_OPTI4_NAME,"ollaston"))
            {
                if(open(outputFileNameb,"r",1))
                {
                    write,"File exists, skipping...";
                    continue;
                }
            }
            else
            {
                write,"File exists, skipping...";
                continue;
            }
        }
    
        sx = ESO_DET_WIN_STARTX;
        sy = ESO_DET_WIN_STARTY;
        nx = min(ESO_DET_WIN_NX,1024);
        ny = min(ESO_DET_WIN_NY,1024);

        dit = ESO_DET_DIT;
        ndit = ESO_DET_NDIT;

        usedMask = ESO_INS_OPTI3_NAME;
        wollaston = ESO_INS_OPTI4_NAME;
        filter = ESO_INS_OPTI6_NAME;
        usedCam = ESO_INS_OPTI7_NAME;
        pixelScale = ESO_INS_PIXSCALE;
        wlen = ESO_INS_CWLEN * micro2unit;
    
        write,usedMask, usedCam, pixelScale, wollaston, filter, wlen;

        if(usedMask=="18Holes")
        {
            maskPos = mask18HolesPos;
            maskDiam = mask18HolesDiam;
        }
        else if(usedMask=="7Holes")
        {
            maskPos = mask7HolesPos;
            maskDiam = mask7HolesDiam;
        }
        else if(usedMask=="9Holes")
        {
            maskPos = mask9HolesPos;
            maskDiam = mask9HolesDiam;
        }
        else if(usedMask=="BB_9Holes")
        {
            maskPos = maskBB_9HolesPos;
            maskDiam = maskBB_9HolesDiam;
        }
        else
        {
            error;
        }

        sBPM = BPM(sx:sx+nx-1,sy:sy+ny-1);
        sglow = GLOW(sx:sx+nx-1,sy:sy+ny-1);
        sbias = BIAS(sx:sx+nx-1,sy:sy+ny-1);
        sflat = FLAT(sx:sx+nx-1,sy:sy+ny-1);

        rawdata = cfitsRead(file(kFile))(1:nx,1:ny,);
        nbFrames = dimsof(rawdata)(0);
        nbPix = dimsof(rawdata)(2);
        write,dit,ndit,nbFrames;

        //         if(ndit!=nbFrames)
        //         {
        //             write,"This is averaged data. Skipping...";
        //             continue;
        //         }
      
        w=0;
        wkll;
        window,w,wait=1,width=400,height=400;
      
        reddata = samCosmetic(rawdata, sBPM, sflat, sbias, sglow, dit); 

        //     window,w,wait=1;w++;
        //     histogramme, rawdata(,,avg);

        //     window,w,wait=1;w++;
        //     fma;
        //     pli,rawdata(..,sum),cmin=0,cmax=20000;

        //     window,w,wait=1;w++;
        //     histogramme, reddata(,,avg);

        window,w,wait=1;w++;
        fma;
        pli,reddata(,,avg),cmin=0;
        yocoPlotColorLookupTable,0,max(reddata(..,avg)); 

        samSplitUpFrame,reddata,data1,data2;
        nPix = dimsof(data1)(2);

        is2 = (!is_void(data2))&&(ESO_INS_OPTI4_NAME!="empty"); 

        // free some space
        reddata = [];

        window,w,wait=1;w++;
        fma;
        pli,data1(,,avg),cmin=0;
        yocoPlotColorLookupTable,min(data1(,,avg)),max(data1(,,avg));

        if(is2)
        {
            window,w,wait=1;w++;
            fma;
            pli,data2(,,avg),cmin=0;
        }

        fdata1 = fft(data1,[1,1,0]);
        if(is2)
            fdata2 = fft(data2,[1,1,0]);

        data1 = data2 = [];

        dsp1 = roll((abs(fdata1)^2)(,,avg));
        phi1 = roll(atan(fdata1.im,fdata1.re));
        if(is2)
        {
            dsp2 = roll((abs(fdata2)^2)(,,avg));
            phi2 = roll(atan(fdata2.im,fdata2.re));
        }

        fdata1 = fdata2 = [];

        plate = 5.0/(pixelScale*mas2rad*1000)*wlen/8;
        demiPix = -5.0/(pixelScale*mas2rad*1000)/nPix*wlen/8;
        fac = 0.5;

        window,w,wait=1;w++;
        pli,log(dsp1),-plate+demiPix,-plate+demiPix,plate+demiPix,plate+demiPix;
        samPlotMask,maskPos,maskDiam,Umask,Vmask,Dmask,stations,fac=fac,color="blue";

        if(is2)
        {
            window,w,wait=1;w++;
            pli,log(dsp2),-plate+demiPix,-plate+demiPix,plate+demiPix,plate+demiPix;
            samPlotMask,maskPos,maskDiam,fac=fac,color="blue";
        }
      

        // Ug = array(span(-plate,plate,nPix),nPix);
        Ug = array(span(-plate+demiPix,plate+demiPix,nPix),nPix);
        Vg = transpose(Ug);

        FC21 = samComputeV2(dsp1,Ug,Vg,Umask,Vmask,Dmask,flux1,Ucen1,Vcen1,FCcen1,fluxCen1);
        V21i = FC21 / flux1 * numberof(Umask);
        V21Cen = FCcen1 / fluxCen1;
        V21 = grow(V21i,V21Cen);
        U = grow(Umask,Ucen1);
        V = grow(Vmask,Vcen1);

        if(is2)
        {
            FC22 = samComputeV2(dsp2,Ug,Vg,Umask,Vmask,Dmask,flux2,Ucen2,Vcen2,FCcen2,fluxCen2);
            V22i = FC22 / flux2 * numberof(Umask);
            V22Cen = FCcen2 / fluxCen2;
            V22 = grow(V22i,V21Cen);
            U2 = grow(Umask,Ucen2);
            V2 = grow(Vmask,Vcen2);
        }
      
        R = abs(U,V);
        theta = atan(V,U);

        Rcen = abs(Ucen1,Vcen1);
        thetaCen = atan(Vcen1,Ucen1);

        window,w,wait=1;w++;
        plg,V21,R,type="none",marker='\2',color="red";
        // plg_err,V21,V21err,Rmask,type="none",marker='\2';
      
        time=targetId=array(0.0,dimsof(U));
        mjd = array(MJD_OBS,dimsof(U));
        dit = array(ESO_DET_DIT*ESO_DET_NDIT,dimsof(U));
        maxStat = max(stations);

        maxStat++;
        grow,stations,[[maxStat,maxStat]];
        for(l=1;l<=numberof(FCcen1);l++)
        {
            grow,stations,[[maxStat,l+maxStat]];
        }

        //Write output file !!!

        samSetArrTarg,oiArray,oiTarget;

        amdlibSetWave, newWave, ESO_INS_CWLEN*micro2nano, 0*unit2nano;

        nbBases = numberof(V21);
        nbWlen = 1;
        nbFrames = 1;
    
        cpxDummyArray = array(complex,nbBases,nbWlen,nbFrames);
        dummyArray = array(0.0,nbBases,nbWlen,nbFrames);
        dummyTmp = array(0.0,nbBases,nbFrames);
    
        amdlibSetOiVis, newVis,
            cpxDummyArray,cpxDummyArray,dummyArray,
            dummyArray,dummyArray, 
            dummyArray, dummyArray,
            dummyTmp,
            dummyTmp, dummyTmp,
            dummyTmp, dummyTmp,
            dummyTmp, dummyTmp, stations;

        amdlibSetOiVis2, newVis2, array(V21,1,1), array(0.0,numberof(V21),1,1),
            targetId, time, mjd, dit, U, V, stations;

        write, "writing file "+outputFileName;
        status = amdlibWriteOiFile(outputFileName,
                                   &obsInsCfg, &oiArray, 
                                   &oiTarget, &newWave, &newPhot, &newVis, 
                                   &newVis2, &newTriple3, pointer(nil),
                                   pointer(nil));
        if (status == amdlibFAILURE)
        {
            yocoGuiErrorBox,"Could not save result file\n";
        }
    
        outFh = cfitsio_open(outputFileName, "a", overwrite=1);
        cfitsio_goto_hdu,outFh,1;
        if(strmatch(ESO_INS_OPTI4_NAME,"ollaston"))
        {
            cfitsio_set, outFh,
                ["WOLASTON"],
                ["WOLLASTON1"],
                ["Presence of a Wollaston ?"];
        }
        cfitsio_set, outFh,
            ["HIERARCH ESO INS MODE","WRITENBY"],
            ["DUMMY","samSuffit"],
            ["For compatibility with amdlib","a SAM pipeline written by F. Millour"];
        cfitsio_set, outFh, "HIERARCH ESO PRO CATG", "SCIENCE_REDUCED",
            "Computed by samSuffit";
    
        cfitsio_close_file, outFh;
    
        amdlibLoadOiData,inputOiFile=outputFileName; 

        if(is2)
        {
            //Write output file !!!

            nbBases = numberof(V22);
            nbWlen = 1;
            nbFrames = 1;
    
            cpxDummyArray = array(complex,nbBases,nbWlen,nbFrames);
            dummyArray = array(0.0,nbBases,nbWlen,nbFrames);
            dummyTmp = array(0.0,nbBases,nbFrames);
    
            amdlibSetOiVis, newVis,
                cpxDummyArray,cpxDummyArray,dummyArray,
                dummyArray,dummyArray, 
                dummyArray, dummyArray,
                dummyTmp,
                dummyTmp, dummyTmp,
                dummyTmp, dummyTmp,
                dummyTmp, dummyTmp, stations;

            amdlibSetOiVis2, newVis2, array(V22,1,1), array(0.0,numberof(V22),1,1),
                targetId, time, mjd, dit, U, V, stations;

            write, "writing file "+outputFileNameb;
            status = amdlibWriteOiFile(outputFileNameb,
                                       &obsInsCfg, &oiArray, 
                                       &oiTarget, &newWave, &newPhot, &newVis, 
                                       &newVis2, &newTriple3, pointer(nil),
                                       pointer(nil));
            if (status == amdlibFAILURE)
            {
                yocoGuiErrorBox,"Could not save result file\n";
            }
    
            outFh = cfitsio_open(outputFileNameb, "a", overwrite=1);
            cfitsio_goto_hdu,outFh,1;
    
            cfitsio_set, outFh,
                ["HIERARCH ESO INS MODE","WRITENBY","WOLLASTON"],
                ["DUMMY","samSuffit","WOLLASTON2"],
                ["For compatibility with amdlib","a SAM pipeline written by F. Millour","Wollaston is present"];
    
            cfitsio_close_file, outFh;
    
            amdlibLoadOiData,inputOiFile=outputFileNameb;
        }
      
    }
    write,"Batch reducing finished. You can now go to calibrate your data.";
}

/*************************************************************/

func samSetArrTarg(&oiArray, &oiTarget)
/* DOCUMENT samSetArrTarg(&oiArray, &oiTarget)

       DESCRIPTION

       PARAMETERS
   - oiArray : 
   - oiTarget: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    /**********************************************/
    /* Set up OI ARRAY table */
    /**********************************************/
    oiArray = amdlibOI_ARRAY();
    //oiArray.thisPtr = oiArray;
    oiArray.nbStations = max(stations);
    
    arrayName = strchar("VLT_UT4_NACO_SAM");
    n = numberof(arrayName);
    m = numberof(oiArray.arrayName);
    grow,arrayName,array(char(0),m-n);
    oiArray.arrayName = arrayName;

    coordinateFrame = strchar("GEODESIC");
    n = numberof(coordinateFrame);
    m = numberof(oiArray.coordinateFrame);
    grow,coordinateFrame,array(char(0),m-n);
    oiArray.coordinateFrame = coordinateFrame;

    oiArray.arrayCenterCoordinates=[ESO_TEL_GEOLAT, ESO_TEL_GEOLON, ESO_TEL_GEOELEV];
    
    /**********************************************/
    /* Elements array */
    /**********************************************/

    elements = array(amdlibOI_ARRAY_ELEMENT(),max(stations));
    
    for(k=1;k<=numberof(maskPos(1,));k++)
    {
        maskName = strchar("mask"+pr1(k));
        n = numberof(maskName);
        m = numberof(elements(k).telescopeName);
        grow,maskName,array(char(0),m-n);
        elements(k).telescopeName = maskName;
        elements(k).stationName = maskName;
        elements(k).stationIndex = k;
        elements(k).elementDiameter = maskDiam;
        elements(k).stationCoordinates = grow(maskPos(,k),0);  
    }

    p=0;
    pixSize = min(abs(Rcen(dif)));
    maskName = strchar("center"+pr1(p));
    n = numberof(maskName);
    m = numberof(elements(k).telescopeName);
    if(m>n)
        grow,maskName,array(char(0),m-n);
    else
        maskName = maskName(:m);
    elements(k).telescopeName = maskName;
    elements(k).stationName = maskName;
    elements(k).stationIndex = k;
    elements(k).elementDiameter = pixSize;
    elements(k).stationCoordinates = [0,0,0];  
    k++;
    
    pixSize = min(abs(Rcen(dif)));
    for(l=1;l<=numberof(Rcen);l++)
    {
        p++;
        maskName = strchar("C"+pr1(p));
        n = numberof(maskName);
        m = numberof(elements(k).telescopeName);
        if(m>n)
            grow,maskName,array(char(0),m-n);
        else
            maskName = maskName(:m);
        elements(k).telescopeName = maskName;
        elements(k).stationName = maskName;
        elements(k).stationIndex = k;
        elements(k).elementDiameter = pixSize;
        elements(k).stationCoordinates = [Ucen1(p),Vcen1(p),0];
        k++;
    }
    oiArray.element=&elements;

    /**********************************************/
    /* Set up OI TARGET table */
    /**********************************************/
    oiTarget = amdlibOI_TARGET();
    oiTarget.nbTargets=1;
    targElem = amdlibOI_TARGET_ELEMENT();
    targElem.equinox=ESO_TEL_TARG_EQUINOX;
    targElem.pmRa=ESO_TEL_TARG_PMA
        targElem.pmDec=ESO_TEL_TARG_PMD
        targElem.parallax=ESO_TEL_TARG_PARALLAX
        targElem.raEp0 = RA;
    targElem.decEp0 = DEC;

    sName = strchar(starName);
    n = numberof(sName);
    m = numberof(targElem.targetName);
    if(m>n)
        grow,sName,array(char(0),m-n);
    else
        sName = sName(:min(n,m));
    targElem.targetName=sName;

    oiTarget.element = &targElem;
}

/***************************************************************************/

func samDivideOiData(void, inputSciFile=, inputCalFile=, calDiam=, calDiamErr=, outputFile=, gui=)
/* DOCUMENT samDivideOiData(inputSciFile=, inputCalFile=, calDiam=, calDiamErr=, outputFile=, gui=)
   
       DESCRIPTION
       Calibrate an OI_DATA file from a science star with another from
       a calibrator. You can use the result for quick look to the data
       quality but probably not for a final scientifically usable result.
    
       PARAMETERS
   - inputSciFile: input OI-FITS file relative to an observation. 
   - inputCalFile: input OI-FITS file relative to a calibraton star.
   - calDiam     : calibration star expected diameter (in mas)
   - calDiamErr  : error on the calibration star diameter
       (in mas, for error computation on the visibilities)
   - outputFile  : name of the file where resulting IO-FITS file 
   - gui         : 
       is stored

       RETURN VALUE
       a CAL_OI_DATA file you can use for quick look to the data quality.

       CAUTION
       Please note this method for calibration (using 1 star and 1 calibrator)
       is subject to many systematics not taken into account here. Please use
       it with caution.
       The case of an over resolved calibration star with non-zero closure phase 
       is NOT taken into account

       REMARKS
       A full calibration method is still under development. It will use
       basically all the night data and will estimate more reliable error
       bars than this method.
    
       SEE ALSO 
    */
{
    if(is_void(gui))
        gui=1;
    
    
    yocoLogTrace,"samDivideOiData()";

    if (is_void(calDiam))
    {
        keybd_focus, 1;
        yocoGuiInfoBox,"No diameter !\nType in a diameter for the "+
            "calibration star (mas)",click=0;
        calDiam = 0.0;
        n= read(prompt="Your diameter >> ", format="%f", calDiam);
        keybd_focus, 0;
    }
    
    if (!is_void(void))
    {
        yocoLogWarning,"void should be void !";
    }

    /* If the function is called without argument, a file browser popups to
     * choose the correct file */
    if (amdlibFileChooser("Enter the science star OI file to load", 
                          inputSciFile) == 0)
    {
        return 0;
    }
    /* If the function is called without argument, a file browser popups to
     * choose the correct file */
    if (amdlibFileChooser("Enter the calibration star OI file to load", 
                          inputCalFile) == 0)
    {
        return 0;
    }
    
    
    // Read the science star OI_VIS file
    yocoLogInfo, "Reading OI file:",inputSciFile;

    /* DUMBLY READING TWICE THE FILE */
    _amdlibReadVis,sci_oiWave, sci_amberPhot, sci_oiVis,
        sci_oiVis2, sci_oiVis3, sci_amberOpd, sci_amberInsCfg,
        sci_oiArray, sci_oiTarget, sci_amberSpectrum,
        inputOiFile=inputSciFile;
    
    if (!amdlibLoadOiData(sci_wlen, sci_bandWidth, sci_time, sci_fringeSNR,
                          sci_sqVis, sci_sqVisErr, sci_uvCoord,
                          sci_bandFlag, sci_pistonOPDArray,
                          sci_sigmaPistonArray, sci_diffVisAmp,
                          sci_diffVisAmpErr, sci_diffVisPhase,
                          sci_diffVisPhaseErr, sci_vis3Amp,
                          sci_vis3AmpErr, sci_vis3Phase, sci_vis3PhaseErr,
                          sci_cpxVis, sci_cpxVisErr, sci_visCovRI,
                          sci_uv1Coord, sci_uv2Coord, sci_fluxSumPiPj,
                          sci_fluxRatPiPj, sci_PiMultPj,
                          sci_spectrum, sci_spectrumErr,
                          inputOiFile=inputSciFile))
    {
        yocoError, "Could not read : " + inputSciFile;
        return 0;
    }
    
    sci_nbFrames = dimsof(sci_sqVis)(2);
    sci_nbWlen = dimsof(sci_sqVis)(3);
    sci_nbBases = dimsof(sci_sqVis)(4);
    
    sci_MJD = (*sci_oiVis.table).dateObsMJD;
    sci_targetId = (*sci_oiVis.table).targetId;
    sci_time = (*sci_oiVis.table).time;
    sci_DIT = (*sci_oiVis.table).expTime;
    sci_stations = (*sci_oiVis.table).stationIndex;
        
    if (sci_nbBases == 3)
    {
        sci_stations3 = (*sci_oiVis3.table).stationIndex;
        sci_U1 = sci_uv1Coord.u;
        sci_V1 = sci_uv1Coord.v;
        sci_U2 = sci_uv2Coord.u;
        sci_V2 = sci_uv2Coord.v;
    }
        
    sci_U  = sci_uvCoord.u;
    sci_V  = sci_uvCoord.v;
    
    if (sci_nbFrames != 1)
    {
        yocoError, "Selected file has more than 1 frame, could not calibrate " +                   "it ...", "Check you perform frame selection first on your" +                   " file";
        return 0;
    }

    
    // Read the calibration star OI_VIS file
    yocoLogInfo, "Reading file:", inputCalFile;
    
    /* DUMBLY READING TWICE THE FILE */
    _amdlibReadVis,cal_oiWave, cal_amberPhot, cal_oiVis,
        cal_oiVis2, cal_oiVis3, cal_amberOpd, cal_amberInsCfg,
        cal_oiArray, cal_oiTarget, cal_amberSpectrum,
        inputOiFile=inputCalFile;

    if (!amdlibLoadOiData(cal_wlen, cal_bandWidth, cal_time,
                          cal_fringeSNR, cal_sqVis, cal_sqVisErr,
                          cal_uvCoord, cal_bandFlag, cal_pistonOPDArray,
                          cal_sigmaPistonArray, cal_diffVisAmp,
                          cal_diffVisAmpErr, cal_diffVisPhase,
                          cal_diffVisPhaseErr, cal_vis3Amp, cal_vis3AmpErr,
                          cal_vis3Phase, cal_vis3PhaseErr, cal_cpxVis,
                          cal_cpxVisErr, cal_visCovRI, cal_uv1Coord,
                          cal_uv2Coord, cal_fluxSumPiPj,
                          cal_fluxRatPiPj, cal_PiMultPj,
                          cal_spectrum, cal_spectrumErr,
                          inputOiFile=inputCalFile))
    {
        yocoError, "Cannot read : " + inputCalFile;
        return 0;
    }
        
    cal_nbFrames = dimsof(cal_sqVis)(2);
    cal_nbWlen = dimsof(cal_sqVis)(3);
    cal_nbBases = dimsof(cal_sqVis)(4);

    
    // Merge wavelength tables using only overlapping zones
    calWlenIdx = where((min(sci_wlen)<= cal_wlen) &
                       (max(sci_wlen)>= cal_wlen));
    sciWlenIdx = where((min(cal_wlen)<= sci_wlen) &
                       (max(cal_wlen)>= sci_wlen));

    sci_nbWlen = numberof(sciWlenIdx);
    cal_nbWlen = numberof(calWlenIdx);
    
    // Test wavelength match
    if(nallof(sci_wlen(sciWlenIdx)==cal_wlen(calWlenIdx)))
    {
        yocoError, "Wavelength tables mismatch. exiting...";
        return 0;
    }

    // Test if files contain 1 frame
    if (cal_nbFrames != 1)
    {
        yocoError, "Selected file has more than 1 frame, could not calibrate " +                   "it ...", "Check you perform frame selection first on your" +                   " file";
        return 0;
    }

    // Test number of baselines
    if(sci_nbBases!=cal_nbBases)
    {
        yocoError, "Calibration file and science file number of baselines mismatch("+pr1(cal_nbBases)+" vs "+pr1(sci_nbBases)+"). exiting...";
        return 0;
    }

    // Test wavelength array dimensions
    if(sci_nbWlen!=cal_nbWlen)
    {
        yocoError, "Calibration file and science file number of wavelength mismatch ("+pr1(cal_nbWlen)+" vs "+pr1(sci_nbWlen)+"). exiting...";
        return 0;
    }

    /* Treatment itself */

    // Define wavelength table for both datasets
    wlen = sci_wlen(sciWlenIdx);
    bandwidth = sci_bandWidth(sciWlenIdx);
    
    // Get uv coordinates
    sci_U = sci_uvCoord.u;
    sci_V = sci_uvCoord.v;
    if (sci_nbBases==3)
    {
        sci_U1 = sci_uv1Coord.u;
        sci_V1 = sci_uv1Coord.v;
        sci_U2 = sci_uv2Coord.u;
        sci_V2 = sci_uv2Coord.v;
    }
    
    cal_U = cal_uvCoord.u;
    cal_V = cal_uvCoord.v;

    /* Compute expected visibility of the calibration star */
    C_Cal = amdlibComputeVisUniformDisk(cal_U, cal_V, calDiam*mas2rad, 
                                        wlen);
    visCal = abs(C_Cal);
    phiCal = pi * (C_Cal<0);
    vis2Cal = visCal^2;
    
    /* Calibrate squared visibilities */
    vis2CalCorrected = cal_sqVis(,calWlenIdx,) / (vis2Cal + (vis2Cal == 0));
    vis2SciCalibrated = sci_sqVis(,sciWlenIdx,) / (vis2CalCorrected + (vis2CalCorrected == 0));

    /* Calibrate differential visibilities */
    visDiffCal = visCal / visCal(, avg)(, -);
    visDiffCalCorrected = cal_diffVisAmp(,calWlenIdx,) / (visDiffCal + (visDiffCal == 0));
    visDiffSciCalibrated = sci_diffVisAmp(,sciWlenIdx,) / (visDiffCalCorrected + (visDiffCalCorrected == 0));
    
    // Warn user about potential problems in phases calibration
    yocoLogWarning,"!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!";
    yocoLogWarning,"!!!          ATTENTION           !!!";
    yocoLogWarning,"!!! The case of a fully resolved !!!";
    yocoLogWarning,"!!!    calibration star is NOT   !!!";
    yocoLogWarning,"!!!      taken into account      !!!";
    yocoLogWarning,"!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!";
    
    /* Calibrate differential phases */
    phiDiffSciCalibrated =
        sci_diffVisPhase(,sciWlenIdx,) -
        cal_diffVisPhase(,calWlenIdx,);
    
    /* Calibrate closure phase, if any */
    if (sci_nbBases==3)
        closSciCalibrated =
            sci_vis3Phase(,sciWlenIdx,) -
            cal_vis3Phase(,calWlenIdx,);
    
    /* Compute error bars */
    /* FIXME: not written for differential visibility stuff ! */
    if (!is_void(calDiamErr))
    {
        VISMAX = amdlibComputeVisUniformDisk(cal_U, cal_V,
                                             (calDiam-calDiamErr)*mas2rad, 
                                             wlen*nano2unit);
        VISMIN = amdlibComputeVisUniformDisk(cal_U, cal_V,
                                             (calDiam+calDiamErr)*mas2rad, 
                                             wlen*nano2unit);
        vis2CalDelta = abs(VISMAX^2-VISMIN^2);
        vis2SciCalibratedError =
            sqrt((sci_sqVisErr(,sciWlenIdx,)^2 * vis2Cal^2 * cal_sqVis(,calWlenIdx,)^2 +
                  sci_sqVis(,sciWlenIdx,)^2 * vis2CalDelta^2 * cal_sqVis(,calWlenIdx,)^2 +
                  sci_sqVis(,sciWlenIdx,)^2 * vis2Cal^2 * cal_sqVisErr(,calWlenIdx,)^2)/
                 ((cal_sqVis+(cal_sqVis==0))(,calWlenIdx,)^4));
    }
    else
    {
        vis2SciCalibratedError =
            sqrt((sci_sqVisErr(,sciWlenIdx,)^2 * vis2Cal^2 * cal_sqVis(,calWlenIdx,)^2 +
                  sci_sqVis(,sciWlenIdx,)^2 * vis2Cal^2 * cal_sqVisErr(,calWlenIdx,)^2)/
                 (cal_sqVis(,calWlenIdx,)^4));
    }
        
    // Closure phase error
    if (sci_nbBases==3)
        closSciCalibratedError = sqrt(sci_vis3PhaseErr(,sciWlenIdx,)^2 +
                                      cal_vis3PhaseErr(,calWlenIdx,)^2);

    // Differential phase error
    phiDiffSciCalibratedError = sqrt(sci_diffVisPhaseErr(,sciWlenIdx,)^2 +
                                     cal_diffVisPhaseErr(,calWlenIdx,)^2);

    // FIXME: Differential visibility still not OK in amdlib, so put the errors to 0 !
    visDiffSciCalibratedError = array(0,dimsof(visDiffSciCalibrated));

    cpxVis = sigma2CpxVis = array(amdlibCOMPLEX,dimsof(visDiffSciCalibrated));
    visCovRI = fringeSNR = array(0,dimsof(visDiffSciCalibrated));
    vis3Amp = vis3AmpErr = array(0,dimsof(closSciCalibrated));
    fluxSumPiPj = fluxSumPiPjErr = fluxRatPiPj = fluxRatPiPjErr = PiMultPj = 
        array(0,dimsof(vis2SciCalibrated));
    
    /* Get back structures from the yorick arrays. */
    
    amdlibSetOiVis, newVis, cpxVis, sigma2CpxVis, visCovRI, 
        visDiffSciCalibrated, visDiffSciCalibratedError, 
        phiDiffSciCalibrated, phiDiffSciCalibratedError,
        fringeSNR, sci_targetId, sci_time, sci_MJD, sci_DIT,
        sci_U, sci_V, sci_stations;
    
    amdlibSetOiVis2, newVis2, 
        vis2SciCalibrated, vis2SciCalibratedError,
        sci_targetId, sci_time, sci_MJD, sci_DIT,
        sci_U, sci_V, sci_stations;
    
    if (sci_nbBases == 3)
    {
        amdlibSetOiVis3, newTriple3, 
            vis3Amp, vis3AmpErr, closSciCalibrated, closSciCalibratedError,
            sci_targetId, sci_time, sci_MJD, sci_DIT,
            sci_U1, sci_U2, sci_V1, sci_V2, sci_stations3; 
    }
    
    // amdlibSetPhotometry, newPhot, sci_amberPhot, sci_nbWlen,
    // fluxSumPiPj, fluxSumPiPjErr,
    // fluxRatPiPj, fluxRatPiPjErr, PiMultPj;
    
    amdlibSetWave, newWave, wlen*1e9, bandwidth;

    /* Construct output OI file name. */
    if (is_void(outputFile))
    {
        yocoFileSplitName,inputSciFile,scidir,sciname,sciext;
        yocoFileSplitName,inputCalFile,caldir,calname,calext;
        if (strmatch(calext,".gz"))
        {
            calext = sum(yocoStrSplit(calext,".gz"));
            recompressFile = "yes";
        }
    
        outputFile = yocoStrSplit(inputSciFile,".fits")(1) +
            "__DIV__" +
            calname+calext;
    }
     

    yocoLogInfo, "Writing file", outputFile;
    
    /* Save OI-FITS resulting file. */
    status = amdlibWriteOiFile(outputFile,
                               &sci_obsInsCfg, &sci_oiArray, 
                               &sci_oiTarget, &newWave, &newPhot, &newVis, 
                               &newVis2, &newTriple3, &newOpd,
                               &newSpec, errMsg);
    
    
    if (status == amdlibFAILURE)
    {
        yocoError, "Could not write OI files", errMsg, 3;
        return 0;
    }
    
    /* Copy all keywords that could be useful for use afterwards */
    if(_amdlibDumpAllKeywords( 1, inputSciFile, outputFile) == 0)
        yocoError, "Could not write OI files", errMsg, 3;
    
    /* Append specific keywords relative to the calibration recipe */
    _amdlibAppendCalibKeywords, outputFile, inputSciFile, inputCalFile;
    
    outFh = cfitsio_open(outputFile, "a", overwrite=1);
    cfitsio_goto_hdu,outFh,1;
    cfitsio_set, outFh, "HIERARCH ESO PRO CATG", "SCIENCE_CALIBRATED",
        "By amdlibDivideOiData";
    cfitsio_close_file, outFh;
    
    // Rezip file if needed
    if (recompressFile=="yes")
    {
        yocoLogInfo,"Recompressing file";
        if (yocoSystem("gzip -f "+outputFile) != 0)
        {
            return 0;
        }
    }
    yocoLogTrace,"amdlibDivideOiData done";
    return 1;
} 





/*************************************************************/
/** The running part of the code *****************************/
/*************************************************************/

/* initialize directory here */
//dir = "/home/fmillour/ARTICLES/AMOI/NF_NACOSAM/";
// amdlibFileChooser,,fileDir;
//BPMf  = dir+"COSMETIC/BPM.fits.gz";
//BIASf = dir+"COSMETIC/BIAS.fits.gz";
//GLOWf = dir+"COSMETIC/GLOW.fits.gz";
//FLATf = dir+"COSMETIC/FLAT.fits.gz";
//samBatchReduce(inRawDir, BPMf, BIASf, GLOWf, FLATf) 

/*************************************************************/

func samBatchReduceNormalNacoStuff(inRawDir, inBPM, inBias, inGlow, inFlat)
/* DOCUMENT samBatchReduceNormalNacoStuff(inRawDir, inBPM, inBias, inGlow, inFlat)

       DESCRIPTION

       PARAMETERS
   - inRawDir: 
   - inBPM   : 
   - inBias  : 
   - inGlow  : 
   - inFlat  : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{

    amdlibFileChooser,"Enter the bad pixels file",inBPM;
    amdlibFileChooser,"Enter the detector bias map",inBias;
    amdlibFileChooser,"Enter the detector glow map",inGlow;
    amdlibFileChooser,"Enter the flat field map",inFlat;
    amdlibFileChooser,"Enter the raw files directory",inRawDir;
  
    BPM = cfitsRead(inBPM);
    BIAS = cfitsRead(inBias);
    GLOW = cfitsRead(inGlow);
    FLAT = cfitsRead(inFlat);
  
    // Uncomment here if you want to plot the master files
    w=0;
    window,w,wait=1,width=400,height=400;w++;
    pli,FLAT,cmin=0.8,cmax=1.2;
    window,w,wait=1;w++;
    pli,BPM;
    window,w,wait=1;w++;
    pli,BIAS*(1-BPM);
    window,w,wait=1;w++;
    pli,GLOW*(1-BPM);

    // Uncomment here
    files = lsdir(inRawDir);
    file = inRawDir+files(where(strmatch(files,".fits")&(!strmatch(files,"OIVIS"))));
    nFiles = numberof(file);

    kFile=1;
    for(kFile=1;kFile<=nFiles;kFile++)
    {
        yocoFileSplitName,file(kFile),d,f,e;
        write,"File to load is "+f+e;
    
        result = samInfo(file(kFile));
      
        if(result==0)
        {
            // jdhc()
            continue;
        }
        
        //Set the output file and check if it exists
        if(strmatch(e,".gz"))
            e = ".fits";
        outputFileName = d+f+"_OIVIS"+e;
        outputFileNameb = d+f+"_OIVISb"+e;
        if(open(outputFileName,"r",1))
        {
            if(strmatch(ESO_INS_OPTI4_NAME,"ollaston"))
            {
                if(open(outputFileNameb,"r",1))
                {
                    write,"File exists, skipping...";
                    continue;
                }
            }
            else
            {
                write,"File exists, skipping...";
                continue;
            }
        }
        
        sx = ESO_DET_WIN_STARTX;
        sy = ESO_DET_WIN_STARTY;
        nx = min(ESO_DET_WIN_NX,1024);
        ny = min(ESO_DET_WIN_NY,1024);

        dit = ESO_DET_DIT;
        ndit = ESO_DET_NDIT;

        usedMask = ESO_INS_OPTI3_NAME;
        wollaston = ESO_INS_OPTI4_NAME;
        filter = ESO_INS_OPTI6_NAME;
        usedCam = ESO_INS_OPTI7_NAME;
        pixelScale = ESO_INS_PIXSCALE;
        
        wlen = ESO_INS_CWLEN * micro2unit;
        
        write,usedMask, usedCam, pixelScale, wollaston, filter, wlen;
        
        if(usedMask=="18Holes")
        {
            error;
        }
        else if(usedMask=="7Holes")
        {
            error;
        }
        else if(usedMask=="9Holes")
        {
            error;
        }
        else if(usedMask=="BB_9Holes")
        {
            error;
        }
        else
        {
            maskPos = [[0.0,0.0],[0.0,0.0]];
            maskDiam = 16.0;
        }

        sBPM = BPM(sx:sx+nx-1,sy:sy+ny-1);
        sglow = GLOW(sx:sx+nx-1,sy:sy+ny-1);
        sbias = BIAS(sx:sx+nx-1,sy:sy+ny-1);
        sflat = FLAT(sx:sx+nx-1,sy:sy+ny-1);

        rawdata = cfitsRead(file(kFile))(1:nx,1:ny,);
        nbFrames = dimsof(rawdata)(0);
        nbPix = dimsof(rawdata)(2);
        write,dit,ndit,nbFrames;

        //         if(ndit!=nbFrames)
        //         {
        //             write,"This is averaged data. Skipping...";
        //             continue;
        //         }
      
        w=0;
        wkll;
        window,w,wait=1,width=400,height=400;
      
        reddata = samCosmetic(rawdata, sBPM, sflat, sbias, sglow, dit);

        window,w,wait=1;w++;
        fma;
        pli,reddata(,,avg),cmin=0;
        yocoPlotColorLookupTable,0,max(reddata(..,avg)); 

        samSplitUpFrame,reddata,data1,data2, splitSz=32;
        nPix = dimsof(data1)(2);

            
        data1 = data1 - data1(1:nPix/5,1:nPix/5,)(avg,avg,)(-,-,);
        
        // free some space
        reddata = [];

        window,w,wait=1;w++;
        fma;
        pli,data1(,,avg),cmin=0;
        yocoPlotColorLookupTable,min(data1(,,avg)),max(data1(,,avg));

        fdata1 = fft(data1,[1,1,0]);

        data1 = data2 = [];

        dsp1 = roll((abs(fdata1)^2)(,,avg));
        phi1 = roll(atan(fdata1.im,fdata1.re));

        fdata1 = fdata2 = [];

        plate = 5.0/(pixelScale*mas2rad*1000)*wlen/8;
        demiPix = -5.0/(pixelScale*mas2rad*1000)/nPix*wlen/8;
        fac = 0.5;

        window,w,wait=1;w++;
        pli,log(dsp1),-plate+demiPix,-plate+demiPix,plate+demiPix,plate+demiPix;
        samPlotMask,maskPos,maskDiam,Umask,Vmask,Dmask,stations,fac=fac,color="blue";

            
        // Ug = array(span(-plate,plate,nPix),nPix);
        Ug = array(span(-plate+demiPix,plate+demiPix,nPix),nPix);
        Vg = transpose(Ug);
        Ug = -Ug;

        FC21 = samComputeV2(dsp1,Ug,Vg,Umask,Vmask,Dmask,flux1,Ucen1,Vcen1,FCcen1,fluxCen1, nIterInterp=10,facDiamBckgnd=1.4);
        V21Cen = FCcen1 / fluxCen1;
        V21 = V21Cen;
        U = Ucen1;
        V = Vcen1;
      
        R = abs(U,V);
        theta = atan(V,U);

        Rcen = abs(Ucen1,Vcen1);
        thetaCen = atan(Vcen1,Ucen1);

        window,w,wait=1;w++;
        plg,V21,R,type="none",marker='\2',color="red";
        // plg_err,V21,V21err,Rmask,type="none",marker='\2';
      
        time=targetId=array(0.0,dimsof(U));
        mjd = array(MJD_OBS,dimsof(U));
        dit = array(ESO_DET_DIT*ESO_DET_NDIT,dimsof(U));
        maxStat = max(stations);

        maxStat++;
        grow,stations,[[maxStat,maxStat]];
        for(l=1;l<=numberof(FCcen1);l++)
        {
            grow,stations,[[maxStat,l+maxStat]];
        }

        //Write output file !!!

        samSetArrTarg,oiArray,oiTarget;

        amdlibSetWave, newWave, ESO_INS_CWLEN*micro2nano, 0*unit2nano;

        nbBases = numberof(V21);
        nbWlen = 1;
        nbFrames = 1;
    
        cpxDummyArray = array(complex,nbBases,nbWlen,nbFrames);
        dummyArray = array(0.0,nbBases,nbWlen,nbFrames);
        dummyTmp = array(0.0,nbBases,nbFrames);
    
        amdlibSetOiVis, newVis,
            cpxDummyArray,cpxDummyArray,dummyArray,
            dummyArray,dummyArray, 
            dummyArray, dummyArray,
            dummyTmp,
            dummyTmp, dummyTmp,
            dummyTmp, dummyTmp,
            dummyTmp, dummyTmp, stations;

        amdlibSetOiVis2, newVis2, array(V21,1,1), array(0.0,numberof(V21),1,1),
            targetId, time, mjd, dit, U, V, stations;

        write, "writing file "+outputFileName;
        status = amdlibWriteOiFile(outputFileName,
                                   &obsInsCfg, &oiArray, 
                                   &oiTarget, &newWave, &newPhot, &newVis, 
                                   &newVis2, &newTriple3, pointer(nil),
                                   pointer(nil));
        if (status == amdlibFAILURE)
        {
            yocoGuiErrorBox,"Could not save result file\n";
        }
    
        outFh = cfitsio_open(outputFileName, "a", overwrite=1);
        cfitsio_goto_hdu,outFh,1;
        if(strmatch(ESO_INS_OPTI4_NAME,"ollaston"))
        {
            cfitsio_set, outFh,
                ["WOLASTON"],
                ["WOLLASTON1"],
                ["Presence of a Wollaston ?"];
        }
        cfitsio_set, outFh,
            ["HIERARCH ESO INS MODE","WRITENBY"],
            ["DUMMY","samSuffit"],
            ["For compatibility with amdlib","a SAM pipeline written by F. Millour"];
        cfitsio_set, outFh, "HIERARCH ESO PRO CATG", "SCIENCE_REDUCED",
            "Computed by samSuffit";
    
        cfitsio_close_file, outFh;
    
        amdlibLoadOiData,inputOiFile=outputFileName;

    }
    write,"Batch reducing finished. You can now go to calibrate your data.";
}
