/***************************************************************************
 *                                                                         *
 *                               fitOmatic                                 *
 *                   Model-fitting prototyping utility                     *
 *                                                                         *
 *                       Copyright 2007, F. Millour                        *
 *                            fmillour@oca.eu                              *
 *                                                                         *
 ***************************************************************************
 *
 * This script contains test functions for analytical models
 *
 * Please note that this script is distributed under the GPL licence,
 * available at http://www.gnu.org/licenses/gpl.txt
 *
 * Please ACKNOWLEDGE the use of this script for any use in a publication
 *
 ***************************************************************************
 * 
 * "@(#) $Id: testVisModels.i 670 2017-02-10 14:05:56Z fmillour $"
 *
 ***************************************************************************/

func testVisModels(void)
    /* DOCUMENT testVisModels

       DESCRIPTION
       Standard analytical models for visibilities computation

       REQUIRE
       - bessel.i

       AUTHORS
       - Florentin Millour (fmillour@mpifr-bonn.mpg.de)
   
       FUNCTIONS
       - testAll2D    : 
       - testAllSimple: 
       - testGammaVel : 
       - testImage    : 
       - testMultiple : 
       - testPinwheel : 
       - testProfile  : 
       - testSkewed   : 
       - testThickRing: 
       - testVisModels: 
       - testWWCZ     : 
    */
{
    version = strpart(strtok("$Revision: 670 $",":")(2),2:-2);
    if (am_subroutine())
    {
        write, format="package version: %s\n", version;
        help, testVisModels;
    }   
    return version;
} 

yocoLogInfo,"#include \"testVisModels.i\"";

/***************************************************************************/

require,"visModels.i";
require,"imaModels.i";

/***************************************************************************/

func testMultiple(void)
    /* DOCUMENT testMultiple

       DESCRIPTION 

       PARAMETERS

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    wavelength = 2.1*micro2unit;
    fov = 25*mas2rad;
    nPoints = 257;
    basemax =  min(wavelength)/fov*nPoints;
    UVTable = UVGrid(basemax, nPoints)/2;
    Utable = UVTable(1,);
    Vtable = UVTable(2,);

    typei = ["ud","g","no","r","sq","ud2","g2","r2","sq2"];
    nS = numberof(typei);
    fwhm1 = array(10.0,1,nS) * mas2rad;
    fwhm2 = array(5.0,1,nS) * mas2rad;
    angle = array(67.0,1,nS) * deg2rad;
    //     x0 = (span(-100,100,nS))*mas2rad;
    x0 = array(0.0,1,nS);
    y0 = array(0.0,1,nS);
    spectrumi = [[1.0],[1.0],[0.0001],[0.1],[1.0],[1.0],[1.0],[0.1],[1.0]];
    
    typei = ["ud","ud"];
    nS = numberof(typei);
    fwhm1 = array(10.0,1,nS) * mas2rad;
    fwhm2 = array(5.0,1,nS) * mas2rad;
    angle = array(67.0,1,nS) * deg2rad;
    //     x0 = (span(-100,100,nS))*mas2rad;
    x0 = [[0.0],[10.0]] * mas2rad;
    y0 = array(0.0,1,nS);
    spectrumi = [[1.0],[2.0]];
  
    vis = visMultipleResolved(Utable, Vtable, wavelength, typei, fwhm1, fwhm2, angle, x0, y0, spectrumi);
    //   ima = grid2ima2(UVTable;
    amp = grid2ima(abs(vis));
    phi = grid2ima(atan(vis.im,vis.re));
    uvIma = grid2ima2(UVTable);
    spFreq = uvIma / wavelength;
  
    //    yocoGuiWinKill;
    window,1,width=600,height=450;
    plf,amp,spFreq(1,,),spFreq(2,,) ;
    yocoPlotColorLookupTable,0,max(amp),orient=1;
    window,2;
    plf,phi,spFreq(1,,),spFreq(2,,);
    yocoPlotColorLookupTable,min(phi),max(phi),orient=1;

    maxX = wavelength * nPoints / basemax * rad2mas/2;
    xScales = array(span(-maxX,maxX,nPoints),nPoints);

    vis2D = amp * exp(1i * phi);
    N = numberof(vis2D);
    //fftVis = 1.0/fov^2*fft(vis2D,[-1,-1]);
    fftVis = 1.0/N*fft(vis2D,[-1,-1]);
    image = roll(abs(fftVis));
    image_m = image/max(image);
  

    palette,"gray.gp";

    XYTable = UVGrid(fov/2, nPoints);
    Xtable = XYTable(1,);
    Ytable = XYTable(2,);
    image2 = grid2ima(imaMultipleResolved(Xtable, Ytable, wavelength, time, typei, fwhm1, fwhm2, angle, x0, y0, spectrumi));
    x = grid2ima(Xtable);
    y =  grid2ima(Ytable);
    window,4;
    plf,image2,y*rad2mas,x*rad2mas,cmin=0,cmax=max(image2); 
    plc,image2,y*rad2mas,x*rad2mas,levs=max(image2)*[0.5,0.1,0.01],color="red";
    palette,"gray.gp";
    yocoPlotColorLookupTable,0,max(image2),orient=1;

    
    window,3;
    plf,image,xScales,transpose(xScales),cmin=0,cmax=max(image);
    plc,image,xScales,transpose(xScales),levs=max(image2)*[0.5,0.1,0.01],color="red";
    palette,"gray.gp";
    yocoPlotColorLookupTable,0,max(image),orient=1;
    
    window,5;
    fact = 1e14;
    plf,image2-image,y*rad2mas,x*rad2mas,cmin=-fact*0.1,cmax=fact*0.1;
    yocoPlotColorLookupTable,min(image2-image),max(image2-image),orient=1;
    
    palette,"gray.gp";
    N = numberof(xScales);
    i1 = sum(image)// * xScales(dif)(avg)^2*mas2rad^2;
        i2 = sum(image2)//* xScales(dif)(avg)^2*mas2rad^2;
        write,i1,i2,i1/i2;
    //sdfvs()
} 

/***************************************************************************/

func testGammaVel(void)
    /* DOCUMENT testGammaVel

       DESCRIPTION 

       PARAMETERS

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    wkll;
    nbWlen = 2;
    wavelength = span(2.05,2.45,nbWlen)*micro2unit;
    // wavelength = span(1.05,2.45,nbWlen)*micro2unit;
    //wavelength = 2.45*micro2unit;
    fov = 15;
    nPoints = 400;
    basemax = int(nPoints/fov*500);
    UVTable = UVGrid(basemax, nPoints);
    Utable = UVTable(1,);
    Vtable = UVTable(2,);

    pixelSize = 0.1*mas2rad;

    fluxWR = array(3.0,nbWlen);
    fluxO = array(1.0,nbWlen);
    pa = 100;
    sep = 3.65;
    
    dstWWCZ = 0.8;
    sizWWCZ = 10.0;
    thickWWCZ = 0.3;
    kWWCZ = 0.25;
    pWWCZ = 2.0;
    fluxWWCZ = spanl(0.05,1,nbWlen);
    //fluxWWCZ = array(1.05,nbWlen);
  
    typei = ["g","ud"];
    fwhm1 = [1.2,0.47] * mas2rad;
    fwhm2 = [0,0] * mas2rad;
    angle = [90,0]*deg2rad;
    x0 = [0,sep*cos(pa*deg2rad)]*mas2rad;
    y0 = [0,sep*sin(pa*deg2rad)]*mas2rad;
    spectrumi = [fluxWR,fluxO];
  
    visStars = visMultipleResolved(Utable, Vtable, wavelength, typei, fwhm1, fwhm2, angle, x0, y0, spectrumi);

    
    
    anglWWCZ = pa*deg2rad;
    X0WWCZ = sep*cos(pa*deg2rad)*dstWWCZ*mas2rad;
    Y0WWCZ = sep*sin(pa*deg2rad)*dstWWCZ*mas2rad;

    vis = visStars + fluxWWCZ(-,) * visWWCZ(Utable, Vtable, wavelength,
                                            sizWWCZ, thickWWCZ, kWWCZ, pWWCZ,
                                            anglWWCZ, X0WWCZ, Y0WWCZ);
       
    //    yocoGuiWinKill;
    //   ima = grid2ima2(UVTable);
    for(k=1;k<=nbWlen;k++)
    {
        amp = grid2ima(abs(visGaussianDisk(Utable, Vtable, wavelength(k), pixelSize, 0, 0)*vis(,k)));
        phi = grid2ima(atan(vis(,k).im,vis(,k).re));
        uvIma = grid2ima2(UVTable);
        spFreq = uvIma / wavelength(k);
        
        window,1;
        fma;
        plf,amp,spFreq(2,,),spFreq(1,,) ;
        palette,"gray.gp";
    
        window,2;
        fma;
        plf,phi,spFreq(2,,),spFreq(1,,);
        palette,"gray.gp";

        maxX = wavelength(k) * nPoints / basemax * rad2mas/2;
        xScales = array(span(-maxX,maxX,nPoints),nPoints);
  
        vis2D = amp * exp(1i * phi);
        fftVis = fft(vis2D,[-1,-1]);
        image = roll(abs(fftVis));
        image = image/max(image);
  
        window,3;
        fma;
        plf,image,xScales,transpose(xScales);
        plc,image,xScales,transpose(xScales),levs=[0.5],color="red";

        palette,"gray.gp";
        // color_bar,span(0,1,10),vert=1;
        pause,100;
    }
} 

/***************************************************************************/

func testPinwheel(type=)
    /* DOCUMENT testPinwheel(type=)

       DESCRIPTION
       Pinwheel test...

       PARAMETERS
       - type: 

       RESULTS 

       CAUTIONS
       dirty development phase

       EXAMPLES

    

       SEE ALSO:
    */
{
    if(is_void(type))
        type="g2";
    type="r2";

    fillFactor = 40.0;
    //fillFactor = 2.0;
    
    wkll;
    nbWlen = 1;
    wavelength = 2.1*micro2unit;
    nPoints = 256;
    basemax = 730.;
    UVTable = UVGrid(basemax, nPoints);
    Utable = UVTable(1,);
    Vtable = UVTable(2,);

    // WR 118 parameters
    rounds   = array(1.0,nbWlen);
    minThick = array(0.0,nbWlen)*mas2rad;
    maxThick = array(21.5,nbWlen)*mas2rad;
    fact     = array(13.4,nbWlen)*mas2rad;
    power    = array(1,nbWlen);
    anglP    = array(0.0,nbWlen)*deg2rad;
    anglS    = array(8.0,nbWlen)*deg2rad;
    incl     = array(2.6,nbWlen)*deg2rad;
    tailFlux = array(0.0,nbWlen)*deg2rad;

    // WR 104 parameters
    // rounds   = array(1.0,nbWlen);
    // minThick = array(1.0,nbWlen)*mas2rad;
    // maxThick = array(100.5,nbWlen)*mas2rad;
    // fact     = array(100.0,nbWlen)*mas2rad;
    // power    = array(1.0,nbWlen);
    // anglP    = array(30.0,nbWlen)*deg2rad;
    // anglS    = array(60.0,nbWlen)*deg2rad;
    // incl     = array(0.0,nbWlen)*deg2rad;
    // tailFlux = array(0.0,nbWlen)*deg2rad;

    //    yocoGuiWinKill;
    //        for(incl=360*deg2rad;incl>=0;incl-=5*deg2rad)
    //     for(anglP=320*deg2rad;anglP>=0;anglP-=5*deg2rad)
    //     {
    vis = visPinwheel(Utable, Vtable, wavelength, time,rounds, minThick, maxThick, fact, power, anglP, incl, anglS, tailFlux, type=type,fillFactor=fillFactor);
        
    amp = grid2ima(abs(vis));
    phi = grid2ima(atan(vis.im,vis.re));
    uvIma = grid2ima2(UVTable);
    spFreq = uvIma / wavelength;
        
    window,1,wait=1;
    fma;
    plf,amp,spFreq(2,,),spFreq(1,,) ;
    //plf,log(amp),spFreq(1,,),spFreq(2,,) ;
    palette,"gray.gp";
    xytitles,"Sp. Freq (cycles/rad)", "Sp. Freq (cycles/rad)";
    hcps,"~/testPinWheelCisibMap2.ps";
    hcps,"~/pinWheelRot_"+pr1(anglP*rad2deg)+"_"+pr1(incl)+".ps";
        
    window,2,wait=1;
    fma;
    plf,phi,spFreq(2,,),spFreq(1,,);
    palette,"gray.gp";
        
        
    maxX = wavelength * nPoints / basemax * rad2mas / 4;
    xScales = array(span(-maxX,maxX,nPoints),nPoints);
        
    vis2D = amp * exp(1i * phi);
    fftVis = fft(vis2D,[-1,-1]);
    image = roll(abs(fftVis));
    image = image/max(image);
    //     image = grid2ima(imaPinwheel(Utable, Vtable, wavelength, rounds, minThick, maxThick, fact, power, angl, X0, Y0));
        
    window,7,wait=1;
    fma;
    // plf,log(image),xScales,transpose(xScales);
    plf,image,xScales,transpose(xScales);
    // plc,image,xScales,transpose(xScales),levs=[0.5],color="red";
    xytitles,"!a (mas)","!d (mas)",[0.02,0.02];
        
    palette,"yarg.gp";
    yocoPlotColorLookupTable,0,1,orient=1;

    //hcps,"~/testPinwheel_Image2.ps";
    //         pouet()
    //     }

    x = xScales(,1)*mas2rad;
    include,"miral.i";
    // miral_write_imgCube,HOME+"WR104_Image.fits",image,x,x,1.6e-6;
    miral_write_imgCube,HOME+"WR118_Image.fits",image,x,x,wavelength;
    
        
    basemax = 130;
    //  return 0;
    line = UVLine(0, basemax, 0, nPoints);
    vis = visPinwheel(line(1,), line(2,), wavelength, time, rounds, minThick, maxThick, fact, power, anglP, incl, anglS,tailFlux, type=type,fillFactor=fillFactor);

    window,10,style="amdlibLargeBox.gs",width=600,height=500,wait=1;
    plg,abs(vis)(,1),abs(line(1,),line(2,)),type="dash";

    
    line = UVLine(0, basemax, (45)*deg2rad, nPoints);
    vis = visPinwheel(line(1,), line(2,), wavelength, time, rounds, minThick, maxThick, fact, power, anglP, incl, anglS,tailFlux,fillFactor=fillFactor);
    plg,abs(vis)(,1),abs(line(1,),line(2,)),type="dot";
    
    line = UVLine(0, basemax, (90)*deg2rad, nPoints);
    vis = visPinwheel(line(1,), line(2,), wavelength, time, rounds, minThick, maxThick, fact, power, anglP, incl, anglS,tailFlux,fillFactor=fillFactor);
    plg,abs(vis)(,1),abs(line(1,),line(2,)),type="dashdot";

    
    vis = visGaussianDisk(line(1,), line(2,), wavelength, time, fact/2.0, X0, Y0);
    plg,abs(vis)(,1),abs(line(1,),line(2,)),type="solid";
    xytitles,"Baseline length (m)", "Visibility";
    hcps,"~/testPinWheelVisibCut2.ps";
    
    window,9,wait=1;
    x = reform(xScales,[1,numberof(xScales)])*mas2rad/2;
    y = reform(transpose(xScales),[1,numberof(xScales)])*mas2rad/2;
    ima = imaPinwheel(x, y, wavelength, rounds, minThick, maxThick, fact, power, anglP, incl, anglS,tailFlux, type=type,fillFactor=fillFactor);
    
    ima2 = grid2ima(ima);
    plf,ima2,transpose(xScales),xScales;
    
    xytitles,"!a (mas)","! d (mas)",[0.02,0.02];
    // hcps,"~/testPinwheel_3ConsecutiveHourAngles.ps";

    //  pouet()
}

/***************************************************************************/

func testWWCZ(void)
    /* DOCUMENT testWWCZ

       DESCRIPTION 

       PARAMETERS

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    wkll;
    nbWlen = 1;
    wavelength = 2.1*micro2unit;
    basemax = 1300;
    nPoints = 200;
    UVTable = UVGrid(basemax, nPoints);
    Utable = UVTable(1,);
    Vtable = UVTable(2,);

    siz = array(10.0,nbWlen)*mas2rad;
    thick = array(0.5,nbWlen)*mas2rad;
    k = array(100000000.37,nbWlen);
    p = array(2.0,nbWlen);
    angl = array(5.0,nbWlen)*deg2rad;
    X0 = array(0.0,nbWlen)*mas2rad;
    Y0 = array(0.0,nbWlen)*mas2rad;

    vis = visWWCZ(Utable, Vtable, wavelength, siz, thick, k, p, angl, X0, Y0);
        
    amp = grid2ima(abs(vis));
    phi = grid2ima(atan(vis.im,vis.re));
    uvIma = grid2ima2(UVTable);
    spFreq = uvIma / wavelength;
  
    //    yocoGuiWinKill;
    window,1;
    plf,amp,spFreq(2,,),spFreq(1,,) ;
    //plf,log(amp),spFreq(1,,),spFreq(2,,) ;
    palette,"gray.gp";
    window,2;
    plf,phi,spFreq(2,,),spFreq(1,,);
    palette,"gray.gp";

    maxX = wavelength * nPoints / basemax * rad2mas/2;
    xScales = array(span(-maxX,maxX,nPoints),nPoints);
  
    vis2D = amp * exp(1i * phi);
    fftVis = fft(vis2D,[-1,-1]);
    image = roll(abs(fftVis));
    image = image/max(image);
  
    window,3;
    // plf,log(image),xScales,transpose(xScales);
    plf,image,xScales,transpose(xScales);
    plc,image,xScales,transpose(xScales),levs=[0.5],color="red";

    palette,"gray.gp";
    color_bar,span(0,1,10),vert=1;
} 

/***************************************************************************/

func testAllSimple(funcName, wlen=, fwhm=, x0=, y0=, basemax=)
    /* DOCUMENT testAllSimple(funcName, wlen=, fwhm=, x0=, y0=, basemax=)

       DESCRIPTION
       tests basic visibilities functions, plotting the Fourier-space and image space alltogether

       PARAMETERS
       - funcName:  visEllipticalRing, visEllipticalUniformDisk, visRectangle or visEllipticalGaussianDisk
       - wlen    : 
       - fwhm    : 
       - x0      : 
       - y0      : 
       - basemax : 

       EXAMPLES
       Possible tests are:
       testAllSimple,"Ring";
       testAllSimple,"UniformDisk";
       testAllSimple,"Square";
       testAllSimple,"GaussianDisk"; 

       SEE ALSO:testAll2D
    */
{
    //    yocoGuiWinKill;
    
    if(is_void(wlen))
        wlen = 2.1*micro2unit;
    fov = 1000*mas2rad;
    nPoints = 200;
    
    if(is_void(basemax))
        basemax =  min(wlen)/fov*nPoints;
    
    UVTable = UVGrid(basemax, nPoints)/2;
    Utable = UVTable(1,);
    Vtable = UVTable(2,);

    if(is_void(fwhm))
        fwhm = 400 * mas2rad;
    if(is_void(x0))
        x0 = 0*mas2rad;
    if(is_void(y0))
        y0 = 0*mas2rad;
    
    visFuncName = symbol_def("vis"+funcName);
    
    vis = visFuncName(Utable, Vtable, wlen, fwhm, x0, y0);
    //   ima = grid2ima2(UVTable);
    amp = grid2ima(abs(vis));
    phi = grid2ima(atan(vis.im,vis.re));
    uvIma = grid2ima2(UVTable);
    spFreq = uvIma / wlen;

    //    yocoGuiWinKill;
    window,1;
    plf,amp,spFreq(1,,),spFreq(2,,) ;
    window,2;
    plf,phi,spFreq(1,,),spFreq(2,,);

    maxX = wlen * nPoints / basemax * rad2mas/2;
    xScales = array(span(-maxX,maxX,nPoints),nPoints);
  
    vis2D = amp * exp(1i * phi);
    fftVis = fft(vis2D,[-1,-1]);
    image = roll(abs(fftVis));
    image = image/max(image);
  
    window,3;
    plf,image,xScales,transpose(xScales);
    plc,image,xScales,transpose(xScales),levs=[0.5,0.1,0.01],color="red";

    palette,"gray.gp";

    imaFuncName = symbol_def("ima"+funcName);
    XYTable = UVGrid(fov/2, nPoints);
    Xtable = XYTable(1,);
    Ytable = XYTable(2,);
    image2 = grid2ima(imaFuncName(Xtable, Ytable, wlen,
                                  fwhm, x0, y0));
    x = grid2ima(Xtable);
    y =  grid2ima(Ytable);
    window,4;
    plf,image2,y*rad2mas,x*rad2mas; 
    plc,image2,y*rad2mas,x*rad2mas,levs=[0.5,0.1,0.01],color="red";  
    palette,"gray.gp";

    line = UVLine(0,basemax,0,nPoints);
    vis = visFuncName(line(1,), line(2,), wlen, fwhm, x0, y0);
    window,5;
    plg,abs(vis(,1)),abs(line(1,),line(2,));
    
    window,6;
    plg,abs(vis(,1))^2,abs(line(1,),line(2,));
}

/***************************************************************************/

func testAll2D(funcName)
    /* DOCUMENT testAll2D(funcName)

       DESCRIPTION
       tests basic visibilities functions, plotting the Fourier-space and image space alltogether

       PARAMETERS
       - funcName: visEllipticalRing, visEllipticalUniformDisk, visRectangle or visEllipticalGaussianDisk

       EXAMPLES
       Possible tests are:
       testAll2D,"EllipticalRing";
       testAll2D,"EllipticalSkewedRing";
       testAll2D,"EllipticalUniformDisk";
       testAll2D,"Rectangle";
       testAll2D,"EllipticalGaussianDisk"; 

       SEE ALSO:testAllSimple
    */
{
    wavelength = 2.1*micro2unit;
    fov        = 1500*mas2rad;
    nPoints    = 200;
    basemax    = min(wavelength)/fov*nPoints;
    UVTable    = UVGrid(basemax, nPoints)/2;
    Utable     = UVTable(1,);
    Vtable     = UVTable(2,);

    fwhm1   = 4  * 160 * mas2rad;
    fwhm2   = 1  * 160 * mas2rad;
    angle   = 10 * deg2rad;
    x0      = 10 * mas2rad;
    y0      = 5  * mas2rad;
    thick   = 20 * mas2rad;
    angleSk = angle;
    minSk   = 1.0;

    visFuncName = symbol_def("vis"+funcName);
        
    vis = visFuncName(Utable, Vtable, wavelength, fwhm1, fwhm2, angle, x0, y0) * visGaussianDisk(Utable, Vtable, wavelength, 1.5*fov/nPoints, x0, y0);
    amp = grid2ima(abs(vis));
    phi = grid2ima(atan(vis.im,vis.re));
    uvIma = grid2ima2(UVTable);
    spFreq = uvIma / wavelength;
    maxSp = max(spFreq);
        
    //    yocoGuiWinKill;
    window,1;
    //    plf,amp,spFreq(1,,),spFreq(2,,);
    pli,amp,min(spFreq(1,,)),min(spFreq(2,,)),max(spFreq(1,,)),max(spFreq(2,,));
    limits,maxSp,-maxSp,square=1;
     
    //pli,amp;
    window,2;
    //    plf,phi,spFreq(1,,),spFreq(2,,);
    pli,phi,min(spFreq(1,,)),min(spFreq(2,,)),max(spFreq(1,,)),max(spFreq(2,,))
        limits,maxSp,-maxSp,square=1;
    //pli,phi;
        
    maxX = wavelength * nPoints / basemax * rad2mas/2;
    xScales = UVGrid(maxX,nPoints)(1,);
    yScales = UVGrid(maxX,nPoints)(2,);
        
    vis2D = amp * exp(1i * phi);
    fftVis = fft(vis2D,[-1,-1]);
    image = roll(abs(fftVis));
    image_m = image/max(image);
        
    window,3;
    // plf,image_m,yScales,xScales;
    // plc,image_m,yScales,xScales,levs=[0.5,0.1,0.01],color="red";
    pli,image_m,min(xScales),min(yScales),max(xScales),max(yScales);
    limits,maxX,-maxX,square=1
        //pli,image_m;
        palette,"gray.gp";
    
    imaFuncName = symbol_def("ima"+funcName);
    Xtable = ima2Grid(xScales)*mas2rad;
    Ytable = ima2Grid(yScales)*mas2rad;
    image2 = grid2ima(imaFuncName(Xtable, Ytable, wavelength,
                                  fwhm1, fwhm2, angle, x0, y0));
    x = grid2ima(Xtable);
    y = grid2ima(Ytable);
    window,4;
    //plf,image2,yScales,xScales; 
    //plc,image2,yScales,xScales,levs=max(image2)*[0.5,0.1,0.01],color="red";
    pli,image2,min(xScales),min(yScales),max(xScales),max(yScales);
    limits,maxX,-maxX,square=1;

    palette,"gray.gp";
    write,max(image),max(image2);

    asdfs()
        }

// testAll2D,"EllipticalRing";

// // testAll2D,"EllipticalSkewedRing";
//testAll2D,"EllipticalUniformDisk";
// // testAll2D,"Rectangle";
// // testAll2D,"EllipticalGaussianDisk"; 
//sf()






/***************************************************************************/

func testSkewed(void)
    /* DOCUMENT testSkewed

       DESCRIPTION
       tests basic visibilities functions, plotting the Fourier-space and image space alltogether

       PARAMETERS
       - funcName: visEllipticalRing, visEllipticalUniformDisk, visRectangle or visEllipticalGaussianDisk

       EXAMPLES
       Possible tests are:
       testAll2D,"EllipticalRing";
       testAll2D,"EllipticalSkewedRing";
       testAll2D,"EllipticalUniformDisk";
       testAll2D,"Rectangle";
       testAll2D,"EllipticalGaussianDisk"; 

       SEE ALSO:testAllSimple
    */
{
    //    yocoGuiWinKill;
    wavelength = 2.1*micro2unit;
    fov        = 1500*mas2rad;
    nPoints    = 200;
    basemax    =  min(wavelength)/fov*nPoints;
    UVTable    = UVGrid(basemax, nPoints)/2;
    Utable     = UVTable(1,);
    Vtable     = UVTable(2,);
    uvIma = grid2ima2(UVTable);
    spFreq = uvIma / wavelength;
    maxSp = max(spFreq);
        
    maxX = abs(wavelength * nPoints / basemax * rad2mas/2);
    xScales = array(span(-maxX,maxX,nPoints),nPoints);
    yScales = transpose(xScales);
    Xtable = ima2Grid(xScales)*mas2rad;
    Ytable = ima2Grid(yScales)*mas2rad;

    fwhm1   = 3*160 * mas2rad;
    fwhm2   = 3*160 * mas2rad;
    angle   = 10*deg2rad;
    x0      = 0*mas2rad;
    y0      = 0*mas2rad;
    thick   = 5*160*mas2rad;
    angleSk = angle+30*deg2rad;
    ampSk   = 0.5;

    vis = visEllipticalSkewedRing(Utable, Vtable,wavelength,[],fwhm1, fwhm2, angle, x0, y0, thick, angleSk, ampSk);
    amp = grid2ima(abs(vis));
    phi = grid2ima(atan(vis.im,vis.re));
    uvIma = grid2ima2(UVTable);
    spFreq = uvIma / wavelength;
    maxSp = max(spFreq);
    vis2D = amp * exp(1i * phi);
    fftVis = fft(vis2D,[-1,-1]);
    image = roll(abs(fftVis));
    image_m = image/max(image);
    
    window,4;
    //plf,image2,y*rad2mas,x*rad2mas; 
    // plc,image2,y*rad2mas,x*rad2mas,levs=max(image2)*[0.5,0.1,0.01],color="red";  
    pli,image,min(xScales),min(yScales),max(xScales),max(yScales)
        limits,max(xScales),-max(xScales),square=1;

        
    palette,"gray.gp";
    //write,max(image),max(image2);

    kc()
        } 


//testSkewed






/***************************************************************************/

func testThickRing(void)
    /* DOCUMENT testThickRing

       DESCRIPTION
       tests basic visibilities functions for a thick ring model, plotting the Fourier-space and image space alltogether

       SEE ALSO:testAllSimple
    */
{
    wavelength = 2.1*micro2unit;
    fov = 100*mas2rad;
    nPoints = 200;
    basemax =  min(wavelength)/fov*nPoints;
    UVTable = UVGrid(basemax, nPoints)/2;
    Utable = UVTable(1,);
    Vtable = UVTable(2,);

    fwhm1 = 40 * mas2rad;
    fwhm2 = 20 * mas2rad;
    thickness = 5*mas2rad;
    thickType = "g";
    angle = 30*deg2rad;
    x0 = 10*mas2rad;
    y0 = 5*mas2rad;

    vis = visThickEllipticalRing(Utable, Vtable, wavelength, fwhm1, fwhm2, angle, thickness, thickType, x0, y0);
    amp = grid2ima(abs(vis));
    phi = grid2ima(atan(vis.im,vis.re));
    uvIma = grid2ima2(UVTable);
    spFreq = uvIma / wavelength;

    //    yocoGuiWinKill;
    window,1;
    plf,amp,spFreq(2,,),spFreq(1,,);
    window,2;
    plf,phi,spFreq(2,,),spFreq(1,,);

    maxX = wavelength * nPoints / basemax * rad2mas/2;
    xScales = array(span(-maxX,maxX,nPoints),nPoints);
  
    vis2D = amp * exp(1i * phi);
    fftVis = fft(vis2D,[-1,-1]);
    image = roll(abs(fftVis));
    image_m = image/max(image);
    
    window,3;
    plf,image_m,xScales,transpose(xScales);
    plc,image_m,xScales,transpose(xScales),levs=[0.5,0.1,0.01],color="red";
    palette,"gray.gp";
} 

   

/***************************************************************************/ 

  

func testImage(imageFile, useFFT=)
    /* DOCUMENT testImage(imageFile, useFFT=)

       DESCRIPTION 

       PARAMETERS
       - imageFile: 
       - useFFT   : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    if(is_void(imageFile))
    {
        // Please uncomment this if you don't have an image to load
        fwhm1=4*mas2rad;
        fwhm2=7*mas2rad;
        angle = 57*deg2rad;
        x0 = y0 = 0;
        lambda = 2.1e-6;
        N = 101;
        XYtable = UVGrid(6*mas2rad,N);
        XYtable2 = grid2ima2(XYtable);
        ima = imaEllipticalUniformDisk(XYtable(1,), XYtable(2,), lambda, fwhm1, fwhm2, angle, x0, y0);
        ima2 = array(grid2ima(ima),1);
    }
    else
    {
        ima2 = array(cfitsRead("AAra.fit"),1);
        lambda = 2.1e-6;
        N = dimsof(ima2)(2);
        XYtable = UVGrid(8*mas2rad,N);
        XYtable2 = grid2ima2(XYtable);
    }
  
    window,0;
    fma;
    plf,ima2(,,1),XYtable2(1,,),XYtable2(2,,); 

    UVtable = UVGrid(101,40);
    if(useFFT==0)
    {
        vis = imageDFT(UVtable(1,), UVtable(2,), lambda, ima2,
                       array( XYtable2(1,,),1),array(XYtable2(2,,),1));
    }
    else
    {
        vis = imageFFT(UVtable(1,), UVtable(2,), lambda, ima2,
                       array(XYtable2(1,,),1),array(XYtable2(2,,),1));
    }
    
    amp = abs(vis);
    phi = atan(vis.im,vis.re);
    UVtable2 = grid2ima2(UVtable);
    amp2 = grid2ima(amp);
    phi2 = grid2ima(phi);
    window,1,height=500,width=600;
    fma;
    plf,amp2,UVtable2(1,,)/lambda,UVtable2(2,,)/lambda;
    window,2,height=500,width=600;
    fma;
    plf,phi2,UVtable2(1,,)/lambda,UVtable2(2,,)/lambda;
    yocoPlotColorLookupTable,min(phi2),max(phi2),orient=1;

    visRef = visEllipticalUniformDisk(UVtable(1,), UVtable(2,), lambda, fwhm1, fwhm2, angle, x0, y0);
    ampRef = grid2ima(abs(visRef));
    phiRef = grid2ima(atan(visRef.im,visRef.re));
    window,3,height=500,width=600;
    fma;
    plf,ampRef,UVtable2(1,,)/lambda,UVtable2(2,,)/lambda;
    window,4,height=500,width=600;
    fma;
    plf,phiRef,UVtable2(1,,)/lambda,UVtable2(2,,)/lambda;
    yocoPlotColorLookupTable,min(phiRef),max(phiRef),orient=1;
    
    window,5,height=500,width=600;
    fma;
    plf,ampRef-amp2,UVtable2(1,,)/lambda,UVtable2(2,,)/lambda;
    yocoPlotColorLookupTable,min(ampRef-amp2),max(ampRef-amp2),orient=1;
    window,6,height=500,width=600;
    fma;
    plf,phiRef-phi2,UVtable2(1,,)/lambda,UVtable2(2,,)/lambda;
    yocoPlotColorLookupTable,min(phiRef-phi2),max(phiRef-phi2),orient=1;

} 



/***************************************************************************/ 

  

func testProfile(void)
    /* DOCUMENT testProfile

       DESCRIPTION 

       PARAMETERS
       - imageFile : 
       - useFFT    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    wavelength = 2.1*micro2unit;
    fov = 1500*mas2rad;
    nPoints = 200;
    basemax =  min(wavelength)/fov*nPoints;
    UVTable = UVGrid(basemax, nPoints)/2;
    Utable = UVTable(1,);
    Vtable = UVTable(2,);

    width = 300.0*mas2rad;
    truncation = 400.0*mas2rad;
    
    data = yocoFileReadAscii("/home/fmillour/ggcar_mod2_i_mas_d2300_lam21232.txt");
    xProfile = yocoMathMovingAvg(yocoStr2Double(data(1,3:))*1e-6,5);
    yProfile = yocoMathMovingAvg(yocoStr2Double(data(2,3:)),5);

    //     xProfile = span(0,truncation,50);
    //     yProfile = yocoMathGauss(xProfile, [width, 0.0]) * width * sqrt(2*pi);
    
    majToMinAxes = 0.5;
    angle = 20.0;
    x0 = 0*mas2rad;
    y0 = 0*mas2rad;
    
    vis = visEllipticalProfile(Utable, Vtable, wavelength, yProfile, xProfile, majToMinAxes, angle, x0, y0);

    amp = grid2ima(abs(vis));
    phi = grid2ima(atan(vis.im,vis.re));
    uvIma = grid2ima2(UVTable);
    spFreq = uvIma / wavelength;

    //    yocoGuiWinKill;
    window,1;
    plf,amp,spFreq(2,,),spFreq(1,,);
    yocoPlotColorLookupTable,min(amp), max(amp), orient=1;
    window,2;
    plf,phi,spFreq(2,,),spFreq(1,,);
    yocoPlotColorLookupTable,min(phi), max(phi), orient=1;

    maxX = wavelength * nPoints / basemax * rad2mas/2;
    xScales = array(span(-maxX,maxX,nPoints),nPoints);
  
    vis2D = amp * exp(1i * phi);
    fftVis = fft(vis2D,[-1,-1]);
    image = roll(abs(fftVis));
    image_m = image/max(image);
    
    window,3;
    plf,image_m,xScales,transpose(xScales);
    plc,image_m,xScales,transpose(xScales),levs=[0.5,0.1,0.01],color="red";
    palette,"gray.gp";
    yocoPlotColorLookupTable,min(image_m), max(image_m), orient=1;

}

