/***************************************************************************
 *                                                                         *
 *                               fitOmatic                                 *
 *                   Model-fitting prototyping utility                     *
 *                                                                         *
 *                      Copyright 2007, F. Millour                         *
 *                            fmillour@oca.eu                              *
 *                                                                         *
 ***************************************************************************
 *
 * This script contains utilities dedicated to model-fitting
 *
 * Please note that this script is distributed under the GPL licence,
 * available at http://www.gnu.org/licenses/gpl.txt
 *
 * Please ACKNOWLEDGE the use of this script for any use in a publication
 *
 ***************************************************************************/

func visModelFitUtils(void)
    /* DOCUMENT visModelFitUtils

       DESCRIPTION
       This script contains utilities dedicated to model-fitting.
       Here you have a list of functions contained in this script.

       FUNCTIONS
       - Ima                   : 
       - Multiple              : 
       - buildFov              : build simmulated annealing field of view
       - computeAllObservables : computes observables
       - computeChi2           : compute chi squared from reiduals
       - computeInsBSAmp       : compute bispectrum amplitude
       - computeInsClos        : compute closure phase
       - computeInsDVis        : compute differential visibility
       - computeInsDiffPhi     : compute differential phase
       - computeInsSpec        : compute spectrum
       - computeInsSqVis       : compute squared visibility
       - computeInsVis         : compute instrumental visibility
       - computeRes            : same as before (?)
       - fitFunc               : the fit function
       - fitLoadSciFiles       : 
       - getConvFact           : get conversion factor from a string
       - getDataInLine         : flatten out data (for fit function)
       - getParamsErrors       : compute parameters errors
       - getRefModel           : get reference model
       - getWlenRange          : 
       - initModel             : inits the model values
       - miral_write_imgCube   : 
       - modulo                : phase modulo (right way)
       - performObsNight       : ???
       - plotAllDataAbsAndDif  : plot everything and even more
       - plotClosSpFreq        : plot closure phases vs sp freq
       - plotDataSpFreq        : plot all observables a sa function of sp freq
       - plotDataWlen          : plots data vs wavelengths
       - plotModelImageMultiple: plot images of models
       - plotMyLUT             : 
       - plotParamsCut         : plot parameters cuts
       - plotParamsCut2D       : 
       - plotParamsWlen        : plot parameters as a function of wavelength
       - plotPhaseSpFreq       : plot phases vs sp freq
       - plotResSpFreq         : 
       - plotSpectrum          : 
       - plotVisSpFreq         : plot visibilities as a function of spatial frequencies
       - plotVisUVFreq         : 
       - regul_compactness     : regularization (not yet used)
       - resFunc               : residuals computation function
       - unwrap                : unwraps
       - updateParVal          : same as before but on all parameters (including fixed ones)
       - updateParams          : update parameters values
       - vis2GaussFWHM         : 
       - vis2UD_Diam           : 
       - visModelFitUtils      : This script
       - writeParams           : write parameters in std out

       SEE ALSO
    */
{
    version = strpart(strtok("$Revision: 688 $",":")(2),2:-2);
    if (am_subroutine())
    {
        write, format="package version: %s\n", version;
        help, visModelFitUtils;
    }   
    return version;
}

require,"amdlib.i";
yocoLogInfo,"#include \"visModelFitUtils.i\"";


/***************************************************************************/

func modulo(a, mod, cen)
    /* DOCUMENT modulo(a, mod, cen)

       DESCRIPTION
       Returns the modulo of a number

       PARAMETERS
       - a  : input number
       - mod: modulo interval
       - cen: modulo center of the interval

       RETURN VALUES
       mod(a)
       
       EXAMPLES
       > mod = 2*pi;
       > cen = 0;
       > a = 3*pi;
       > modulo(a,mod,cen)

       SEE ALSO
    */
{
    // Set modulo center to 0 as default
    if(is_void(cen))
        cen = 0.0;

    // Calculate
    A = a - cen - mod / 2;
    
    pos = (A >0)*(A % mod - mod/2 + cen);
    neg = (A<=0)*(A % mod + mod/2 + cen);

    return pos+neg;
}

/***************************************************************************/

struct hdr
/* DOCUMENT fitOiData

   DESCRIPTION
   Simplified OI DATA structure, to store AMBER datas
   
   SEE ALSO:
*/
{
    string TARGET;
    string file;
};

/***************************************************************************/

struct fitOiData
/* DOCUMENT fitOiData
   
   DESCRIPTION
   Simplified OI DATA structure, to store AMBER datas
   
   SEE ALSO:
*/
{
    hdr hdr;
    pointer TIME;
    
    pointer VIS2DATA;
    pointer VIS2ERR;
    
    pointer VISPHI;
    pointer VISPHIERR;
    
    pointer VISAMP;
    pointer VISAMPERR;
    
    pointer T3PHI;
    pointer T3PHIERR;
    
    pointer spec;
    pointer specErr;
    
    pointer EFF_WAVE;
    pointer EFF_BAND;
    
    pointer UCOORD;
    pointer VCOORD;
    pointer U1COORD;
    pointer U2COORD;
    pointer V1COORD;
    pointer V2COORD;
};

/*******************************************************/

struct fitStruct
/* DOCUMENT fitStruct
   
   DESCRIPTION
   Structure containing all useful information to perform the fit
   
   SEE ALSO:
*/
{
    // Fit function defined with 2 input parameters: fit data structure
    // and parameters
    string funcFit;
    // Real data structure
    pointer data;
    // Model data structure
    pointer model;

    // The real data squeezed in one line
    pointer dataLine;
    // Each element is identified by type ("V2", "CP", "DP", "DV", etc.)
    pointer dataType;
    // Data error
    pointer dataErrLine;
    // Model data also squeezed
    pointer modelLine;
    
    // Number of sources (useful with visMultipleResolved / funcMultiple)
    int nSources;
    // Type of source for each source
    pointer typei;

    //Number of wavelengths for each parameter
    pointer parNbWlen;
    // Parameter is variable or fixed
    pointer parVar;
    // Parameters values
    pointer param;
    // FFT of the parameters in case of image reconstruction
    pointer paramFFT;
    // FFT of the parameters in case of image reconstruction
    pointer paramMaxUV;
    // FFT of the parameters in case of image reconstruction
    pointer paramFlux;
    // Parameter error
    pointer paramErr;
    // FIXME:?
    pointer modelVal;
    // FIXME: Field of view?
    pointer fov;
    // For the parameters: name, unit & wlen
    pointer parName;
    pointer parUnit;
    pointer parWlen;
    // Image pixel size
    double pixSize ;

    // Use squared visibility in the fit, if 1 use V, if 2 use V^2
    int use_vis2;
    // Use spectrum in the fit
    int use_spec;
    // Use closure phase in the fit
    int use_clos;
    // Use bispectrum amplitude in the fit
    int use_bsamp;
    // Use differential visibility in the fit
    int use_dVis;
    // Use false differential visibility (sqrt(V^2 / avg(V^2)) in the fit
    int use_fdVis;
    // Use differential phase in the fit
    int use_dPhi;
    
    // Set closure phases to their absolute value
    int absClos;
    // Prohibit negative fluxes
    int absFlux;
    // Prohibit negative sizes
    int absSize;
    
    // Regularization type
    string regul;
    // Regularization value
    double rglFact;
};

callCounter = 0;

/*******************************************************/

func getConvFact(parUnit)
    /* DOCUMENT getConvFact(parUnit)

       DESCRIPTION
       Returns the conversion factor depending on the input unit

       PARAMETERS
       - parUnit: string containing the desciption of the unit of interest

       SEE ALSO
    */
{
    N = numberof(parUnit);
    convFact = array(double, N);
    for(k=1;k<=N;k++)
    {
        if(parUnit(k) == "mas")
            convFact(k) = rad2mas;
        else if(parUnit(k) == "degrees")
            convFact(k) = rad2deg;
        else if(parUnit(k) == "deg")
            convFact(k) = rad2deg;
        else if(parUnit(k) == "rsun")
            convFact(k) = 1.0/yocoAstroSI.Rsun;
        else if(parUnit(k) == "micron")
            convFact(k) = unit2micro;
        else if(parUnit(k) == "kpc")
            convFact(k) = 1.0/yocoAstroSI.kpc;
        else if(parUnit(k) == "kms")
            convFact(k) = 0.001;
        else
            convFact(k) = 1.0;
    }
    return convFact;
}

/*******************************************************/

func getParamsErrors(tot, symmetric=, nChi2=, maxIterInterval=, maxInterval=, stopAtZero=)
    /* DOCUMENT getParamsErrors(tot, symmetric=, nChi2=, maxIterInterval=, maxInterval=, stopAtZero=)

       DESCRIPTION
       Computes parameters errors by 

       PARAMETERS
       - tot            : input fit structure
       - symmetric      : 
       - nChi2          : 
       - maxIterInterval: 
       - maxInterval    : 
       - stopAtZero     : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(symmetric))
        symmetric=1;
    
    if(is_void(maxIterInterval))
        maxIterInterval = 40;
    
    if(is_void(stopAtZero))
        stopAtZero = 0;
    
    if(is_void(nChi2))
        nChi2 = 50;
    
    // vectorize the data
    dataLine0 = getDataInLine(*tot.data,
                              tot.use_vis2,
                              tot.use_clos,
                              tot.use_bsamp,
                              tot.use_dVis,
                              tot.use_dPhi,
                              tot.use_spec,
                              dataError,dataType, dataWl, dataU1, dataV1,
                              dataU2, dataV2, dataTime);

    nbO = numberof(dataLine0);

    // Get relevant information to build the covariance matrix
    u1  = (dataU1(-,)==dataU1(,-));
    u2  = (dataU2(-,)==dataU2(,-));
    v1  = (dataV1(-,)==dataV1(,-));
    v2  = (dataV2(-,)==dataV2(,-));
    // Select only squared visibilities
    typ = ((dataType=="V2")(-,)&(dataType=="V2")(,-));

    wdt = 50.e-8;
    att = exp(-(dataWl(-,) - dataWl(,-))^2 / (wdt^2));
    
    errMat = dataError(-,) * dataError(,-);
    com    = // att *
        (u1 & u2 & v1 & v2 & typ);
    
    // covMat    = com * errMat + (com==0) * diag(dataError^2);
    // invCovMat = LUsolve(covMat);
        
    // wkll;
    // window,0;
    // pli, log(covMat+(covMat==0));

    // window,1;
    // pli, invCovMat;

    // window,2;
    // pli, att;

    // window,3;
    // plg,diag(invCovMat);
    // plg,1./dataError^2,color="red";
    // logxy,,1;
        
    // Initialize the data weights
    // weight = invCovMat;
    weight0 = 1.0/dataError^2;
    
    refChi2 = computeChi2(tot, params, dataLine0, weight0, dataType);


    write,refChi2;
    refVals   = (*tot.modelVal);
    refParams = (*tot.param);
    convFact  = getConvFact(*tot.parUnit);
    nParams   = numberof(refVals);
    write,"Reference parameters: ", *tot.parName, refVals*convFact, *tot.parUnit;
    
    wx = int(sqrt(nParams))+1;
    wy = wx;
    yocoNmCreate,45,wx,wy,square=0,style="boxed.gs",
        V = [0.15,0.9,0.12,0.73],
        landscape=1,dx=0.05,dy=0.07, fx=1,fy=1,
        height=750, width=950, wait=1, dpi=100;
    window,46,width=450,height=450, wait=1;
    
    ERRORS = ERRP = ERRM = [];    
    for(kCol=1;kCol<=nParams;kCol++)
    {
        window,45, wait=1;
        plsys,kCol;

        // First of all, find the right range for the parameter to plot
        write,"Treating the following parameter: "+
            (*tot.parName)(kCol) + " ("+(*tot.parUnit)(kCol)+")";

        CHI2 = refChi2;
        VAL  = refVals(kCol);
        // CHI2 = VAL = [];
        if(is_void(maxInterval))
        {
            // Lower limit
            write,"Finding lower limit";
            (*tot.modelVal) = refVals;
            val0 = refVals(kCol);
            iter = 0;
            do
            {
                iter++;
                if(refVals(kCol) - val0 == 0)
                    val0 = val0 - 0.05 / convFact(kCol);
            
                sn = sign(val0);
                val0 = val0 - 0.5*abs(refVals(kCol) - val0);
                (*tot.modelVal)(kCol) = val0;
                updateParams,tot,*tot.modelVal;
                params = (*tot.param);
            
                chi2 = computeChi2(tot, params, dataLine, weight, dataType);
                grow, VAL, val0;
                grow, CHI2, chi2;
            
            }
            while((chi2<3*refChi2)&&(iter<maxIterInterval));
            // get lower limit to a limit of 3 times the parameter value
            lowLim = val0;
            
            // Upper limit
            write,"Finding upper limit";
            (*tot.modelVal) = refVals;
            val0 = refVals(kCol);
            iter = 0;
            do
            {
                iter++;
                if(refVals(kCol) - val0==0)
                    val0 = val0 + 0.05 / convFact(kCol);

                sn = sign(val0);
                val0 = val0 + 0.5*abs(refVals(kCol) - val0);
                (*tot.modelVal)(kCol) = val0;
                updateParams,tot,*tot.modelVal;
                params = (*tot.param);
            
                chi2 = computeChi2(tot, params, dataLine, weight, dataType);
                grow, VAL, val0;
                grow, CHI2, chi2;
            }
            while((chi2 <= 2*refChi2)&&(iter < maxIterInterval));
            // get lower limit to a limit of +3 times the parameter value
            upLim = val0;
            
            // Treat special cases like angles
            hardUpLim = 10*abs(refVals(kCol));
            if(hardUpLim < 1./convFact(kCol)/1000)
            {
                yocoLogWarning,"Current parameter value equal zero. Setting hard limit to unity."
                    hardUpLim = 50.0/convFact(kCol);
            }
            if(((*tot.parUnit)(kCol)=="deg") && ((upLim - lowLim)>2*pi))
            {
                yocoLogWarning,"Angle interval too large! Reducing to [-pi,pi]...";
                hardUpLim = pi;
                lowLim = -pi;
                upLim  =  pi;
            }
            
            // Treat special cases like outliers
            if(abs(upLim - refVals(kCol)) > hardUpLim)
            {
                yocoLogWarning,"Upper limit beyond reasonable value! Reducing to 10 times the current value..."
                    upLim  =  refVals(kCol) + hardUpLim;
            }
            if(abs(lowLim - refVals(kCol)) > hardUpLim)
            {
                yocoLogWarning,"Lower limit beyond reasonable value! Reducing to 10 times the current value..."
                    lowLim  =  refVals(kCol) - hardUpLim;
            }
            //  CHI2 = CHI2(where(abs(VAL-refVals(kCol))<hardUpLim));
            // VAL  = VAL(where(abs(VAL-refVals(kCol))<hardUpLim));

            write,lowLim, upLim;
        }
        else
        {
            lowLim = refVals(kCol)-maxInterval/2;
            upLim  = refVals(kCol)+maxInterval/2;
        }
        
        write,"interval found: ",lowLim, upLim;
        CHI2 = CHI2(sort(VAL));
        VAL  = VAL(sort(VAL));

        // Compute cuts along each parameter axis
        (*tot.modelVal) = refVals;
        par = span(lowLim, upLim, nChi2);
        plsys,kCol;
        for(i=1;i<=nChi2;i++)
        {
            (*tot.modelVal)(kCol) = par(i);
            updateParams,tot,*tot.modelVal;
            params = (*tot.param);
            
            chi2 = computeChi2(tot, params, dataLine, weight, dataType);
            grow,CHI2,chi2;
            grow,VAL,par(i);
            
            CHI2 = CHI2(sort(VAL));
            VAL  = VAL(sort(VAL));
            
            //write,chi2, par(i);
            window,46,wait=1;
            fma;
            plg,CHI2,VAL;
            limits,,,0,3*min(CHI2);
        }

        // Treat the case where we want parameters positive
        if(stopAtZero==1)
        {
            wpos = where(VAL>0);
            CHI2 = CHI2(wpos);
            VAL  = VAL(wpos);
        }
        
        // Compute errors
        // Find higher boundary
        k = where(CHI2==refChi2)(1);
        do
        {
            if(k>=numberof(CHI2))
                break;
            k++;
        }
        while(CHI2(k) < 2*refChi2);
        high   = VAL(k);
        // Find lower boundary
        k = where(CHI2==refChi2)(1);
        do
        {
            if(k<=1)
                break;
            k--;
        }
        while(CHI2(k) < 2*refChi2);
        low    = VAL(k);
        deltaPlus  = abs(refVals(kCol)-high);
        deltaMoins = abs(refVals(kCol)-low);
        delta  = abs(high-low)/2;

        // Chi^2 plot
        window,45, wait=1;
        plsys,kCol;
        vw = viewport();
        plt,(*tot.parName)(kCol) + " ("+(*tot.parUnit)(kCol)+"):\n"+
            swrite(refVals(kCol)*convFact(kCol),format="%.3g")+
            " +"+swrite(deltaPlus*convFact(kCol),format="%.2g")+
            "-"+swrite(deltaMoins*convFact(kCol),format="%.2g")+
            "\n+-"+swrite(delta*convFact(kCol),format="%.2g"),
            vw(1)+(vw(2)-vw(1))/2.,vw(3)-0.02,height=12,justify="CT";
        xytitles, , "!c^2",[0.03,0.03];
        plg,CHI2,VAL*convFact(kCol);
        pldj,min(VAL)*convFact(kCol),refChi2,
            max(VAL)*convFact(kCol),refChi2,type="dash";
        pldj,min(VAL)*convFact(kCol),2*refChi2,
            max(VAL)*convFact(kCol),2*refChi2,type="dash";
        
        pldj,refVals(kCol)*convFact(kCol),0,
            refVals(kCol)*convFact(kCol),max(CHI2);
        
        limits,,,0;
        write,"Value found for "+(*tot.parName)(kCol) + ": "+
            pr1(refVals(kCol)*convFact(kCol))+
            " +"+pr1(deltaPlus*convFact(kCol))+
            "-"+pr1(deltaMoins*convFact(kCol))+
            " +-"+pr1(delta*convFact(kCol))+" "+(*tot.parUnit)(kCol)+ " ("+pr1(delta/refVals(kCol)*100)+"%)\n";

        grow, ERRORS, delta;
        grow, ERRP, deltaPlus;
        grow, ERRM, deltaMoins;
    }

    tot.paramErr = &ERRORS;
    if(symmetric==0)
        tot.paramErr = &[ERRM,ERRP];
    
}

/*******************************************************/

func plotParamsCut(&toot, iP, minP, maxP, nP, win=, plot=)
    /* DOCUMENT plotParamsCut(&toot, iP, minP, maxP, nP, win=, plot=)

       DESCRIPTION
       Computes parameters errors by cuts in the Chi2 space

       PARAMETERS
       - toot: input fit structure
       - iP  : 
       - minP: 
       - maxP: 
       - nP  : 
       - win : 
       - plot: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(plot))
        plot=1;
    
    if(plot)
    {
        if(is_void(win))
            win=33;
        window,win,wait=1;
    }
    *toot.parVar = ((*tot.parNbWlen)>0);
    params0        = params = *toot.param;

    name = *tot.parName;
    Units = *tot.parUnit;
    fac  = getConvFact(Units);
    
    //    write,params0;
    fov            = *toot.fov;
    paramsErr      = *toot.paramErr;
    dataLine       = *toot.dataLine;
    dataErrLine    = *toot.dataErrLine;
    if(dimsof(dataErrLine)(1)==1)
        dataWeight = 1.0/dataErrLine^2;
    else
        dataWeight = dataErrLine;
    dataType       = *toot.dataType;
    pwl            = *toot.parNbWlen;
    ns             = toot.nSources;
    np             = dimsof(pwl)(2);
    var            = *toot.parVar;
    val            = *toot.modelVal;
    Units          = *toot.parUnit;
    name           = *toot.parName;
    wlen           = *toot.parWlen;

    if(iP<=0)
    {
        iP  = sum(pwl) + iP;
        iP2 = np + iP;
    }
        
    kountt = akountt = 1;
    
    // Loop on sources
    for(k=1;k<=ns;k++)
    {
        // Loop on parameters
        for(l=1;l<=np;l++)
        {
            // Parameter with no entry, skip.
            if(pwl(l,k)!=0)
            {
                for(m=1;m<=pwl(l,k);m++)
                {
                    if(kountt==iP)
                    {
                        akountt0 = akountt;
                        // write,pr1(k),
                        //     name(kountt),
                        //     swrite(format="%f",val(kountt)),
                        //     pr1(wlen(kountt)*unit2micro);
                    }
                    if(var(l,k)==1)
                    {
                        akountt++;
                    }
                    kountt++;
                }
            }
        }
    } 
    
    chi2Ref = computeChi2(toot, params, dataLine, dataWeight, dataType);

    //fac = getConvFact(units(iP2))(1);
    fac     = rad2mas;
    
    C2 = P = array(0.0,nP);
    
    coord = span(minP,maxP,nP)
        for(k=1;k<=nP;k++)
        {
            (*toot.param)(akountt0) = coord(k);
            params(akountt0) = coord(k);
            C2(k) = computeChi2(toot, params, dataLine, dataWeight, dataType);
            P(k) = coord(k);
        
            if(plot)
            {
                fma;
                plg,C2(where(C2!=0)),P(where(C2!=0))*fac;
                limits;
                limits,,,0;
                xytitles,name(akountt0)+" ("+Units(akountt0)+")","!c^2";
            }
        }
    
    if(plot)
    {
        plg,C2,P*fac;
        limits,,,0;
    
        yocoPlotVertLine,params0(akountt0)*fac,type="dash";
        yocoPlotHorizLine,2*chi2Ref,type="dash";
    }

    zok = where(C2<2*chi2Ref);
    if(numberof(zok)!=0)
    {
        pok = P(zok)*fac;
        write,params0(akountt0)*fac,"+-",(pok(0)-pok(1))/2;
    }
    else
        write,"Too bad!";
    
    // Restore imitial value
    //(*toot.param)(akountt0) = params0(akountt0);
    updateParVal,toot,params0;

    return C2;
}

/*******************************************************/

func plotParamsCut2D(&toot, iP1, minP1, maxP1, nP1, iP2, minP2, maxP2, nP2, &img, win=, plot=, cmax=)
    /* DOCUMENT plotParamsCut2D(&toot, iP1, minP1, maxP1, nP1, iP2, minP2, maxP2, nP2, &img, win=, plot=, cmax=)

       DESCRIPTION

       PARAMETERS
       - toot : 
       - iP1  : 
       - minP1: 
       - maxP1: 
       - nP1  : 
       - iP2  : 
       - minP2: 
       - maxP2: 
       - nP2  : 
       - img  : 
       - win  : 
       - plot : 
       - cmax : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(win))
        win=34;

    window,win,wait=1,style="work.gs";


    name  = *toot.parName;
    Units = *toot.parUnit;
    fac   = getConvFact(Units);
    
    toot.param   = toot.modelVal;
    params0      = params = *toot.param;
    x0 = params0(iP1);
    y0 = params0(iP2);
    dataLine       = *toot.dataLine;
    dataErrLine    = *toot.dataErrLine;
    if(dimsof(dataErrLine)(1)==1)
        dataWeight = 1.0/dataErrLine^2;
    else
        dataWeight = dataErrLine;
    dataType       = *toot.dataType;
    updateParVal,toot,params0; 
    chi2Ref = computeChi2(toot, params0, dataLine, dataWeight, dataType);
    write,chi2Ref;

    pwl            = *toot.parNbWlen;
    ns             = toot.nSources;
    np             = dimsof(pwl)(2);
    var            = *toot.parVar;

    if(iP1<=0)
    {
        iP1  = sum(pwl) + iP1;
        iP1b = np + iP1;
    }
    if(iP2<=0)
    {
        iP2  = sum(pwl) + iP2;
        iP2b = np + iP2;
    }
        
    // Get the correct index for parameters value
    kountt = akountt = 1;
    
    // Loop on sources
    for(k=1;k<=ns;k++)
    {
        // Loop on parameters
        for(l=1;l<=np;l++)
        {
            // Parameter with no entry, skip.
            if(pwl(l,k)!=0)
            {
                for(m=1;m<=pwl(l,k);m++)
                {
                    if(kountt==iP2)
                    {
                        akontt0 = akountt;
                    }
                    if(var(l,k)==1)
                    {
                        akountt++;
                    }
                    kountt++;
                }
            }
        }
    } 

    //fac = getConvFact(Units(iP2))(1);
    fac     = rad2mas;
       
    img = array(0.0,nP1,nP2);

    coord = span(minP2,maxP2,nP2)
        for(k=1;k<=nP2;k++)
        {
            (*toot.param)(iP2) = coord(k);
            params(iP2)        = coord(k);
            img(,k) = plotParamsCut(toot, iP1, minP1, maxP1, nP1, win=win-1,plot=plot);
        
            window,win,width=500,height=500;
            fma;
            pli,img,minP1*fac,minP2*fac,maxP1*fac,maxP2*fac;
            xytitles,name(iP1)+" ("+Units(iP1)+")",name(iP2)+" ("+Units(iP2)+")";
            plg,y0*fac,x0*fac,marker='\2',marks=0,color="red",type="none";
        
            // // Restore imitial value
            // (*toot.param)(akontt0) = params0(akontt0);
            // params(akontt0)        = params0(akontt0);
        }
    (*toot.param)(iP2) = params0(iP2);
    updateParVal,toot,params0;

    if(is_void(cmax))
        cmax = 7;
    
    window,win;
    fma;
    mn = min(img);
    pli,img,minP1*fac,minP2*fac,maxP1*fac,maxP2*fac,cmin=mn,cmax=cmax*mn;
    X = array(span(minP1*fac,maxP1*fac,nP1),nP2);
    Y = transpose(array(span(minP2*fac,maxP2*fac,nP2),nP1));
    plc,img,Y,X,levs=[2*mn,3*mn,4*mn],color="red";
    xytitles,name(iP1)+" ("+Units(iP1)+")",name(iP2)+" ("+Units(iP2)+")";
    plg,y0*fac,x0*fac,marker='\2',marks=0,color="red",type="none";
    yocoPlotColorLookupTable,mn,cmax*mn,sup=">",levels=[mn,2*mn,3*mn,4*mn],levColor="red",orient=1;
    palette,"gray.gp";
    
}

/*******************************************************/

func plotParamsWlen(toot, paramNames=, clear=, kill=, win=)
    /* DOCUMENT plotParamsWlen(toot, paramNames=, clear=, kill=, win=)

       DESCRIPTION
       Plots vector parameters as a function of wavelengths

       PARAMETERS
       - toot      : input structure
       - paramNames: 
       - clear     : 
       - kill      : 
       - win       : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // set default graphical window
    if(is_void(win))
        win = 5;
    
    // By default kill any present window
    if(is_void(kill))
        kill = 1;
    
    // By default clear window (fma)
    if(is_void(clear))
        clear = 1;
    
    // Read values from the data structure
    paramsErr = *toot.paramErr;
    var       = *toot.parVar;
    val       = *toot.modelVal;
    Units      = *toot.parUnit;
    name      = *toot.parName;
    wlen      = *toot.parWlen;
    pwl       = *toot.parNbWlen;
    ns        = toot.nSources;
    np        = dimsof(pwl)(2);
              
    // Get the useful wavelengths
    usedWlens = yocoStrRemoveMultiple(wlen);
    uN        = numberof(usedWlens);
    fluxes    = fluxesErr = array(0.0,ns,uN);
            
    kountt    = akountt = 1;
    
    
    nPars      = numberof(where(pwl));
    // Set windowing parameters
    sqWinSize1 = sqrt(nPars);
    fracWin = 2.0;
    sqX = int(max(1,floor(sqWinSize1/sqrt(fracWin))));
    sqY = int(max(1,floor(sqWinSize1*sqrt(fracWin))));
    
    while(sqX*sqY<nPars)
        sqX++;

    // Create window with subwindows
    if(kill==1)
    {
        winkill,win;
        yocoNmCreate,win, sqX, sqY,
            fx=1, dx=0.08,
            fy=1, dy=0.04,
            V = [0.15,0.9,0.12,0.73], landscape=1,
            height=750, width=950, wait=1, dpi=100;
    }
    else
        window,win;

    if(clear==1)
        fma;
    
    TMP = [];
    // Loop on sources
    for(k=1;k<=ns;k++)
    {
        // Loop on parameters
        for(l=1;l<=np;l++)
        {
            // Parameter with no entry, skip.
            if(pwl(l,k)!=0)
            {
                // Loop on parameter wavelengths
                for(m=1;m<=pwl(l,k);m++)
                {
                    // If parameter without unit, then no unit conversion
                    if(Units(kountt)=="no")
                    {
                        tmp = "";
                        fac=1;
                    }
                    // Otherwise get the conversion factor
                    else
                    {
                        fac = getConvFact(Units(kountt))(1);
                        tmp = Units(kountt);
                    }

                    if(name(kountt)=="flux")
                    {
                        fluxes(k,where(usedWlens==wlen(kountt))) = val(kountt);
                    }
                    
                    if(!is_void(paramsErr))
                    {
                        err = swrite(format="%f",paramsErr(kountt)*fac);
                        if(dimsof(paramsErr)(1)==2)
                            errm = swrite(format="%f",paramsErr(kountt,2)*fac);
                        
                        if(name(kountt)=="flux")
                        {
                            fluxesErr(k,where(usedWlens==wlen(kountt))) = paramsErr(akountt);
                        }
                    }
                    else
                        err = errm = "";
                    
                    tmp2 = [];
                    tmp2 = grow(pr1(k),name(kountt),swrite(format="%f",val(kountt)*fac),err,tmp,pr1(wlen(kountt)*unit2micro),errm);
                    grow,TMP,array(tmp2,1);
                    kountt++;
                }
            }
        }
    }

    //Determine observations wlens
    nObs = numberof(*toot.data);
    minWlen = 999;
    maxWlen = 0;
    for(k=1;k<=nObs;k++)
    {
        wlen = *(*toot.data)(k).EFF_WAVE;
        if(min(wlen)<minWlen)
            minWlen = min(wlen);
        if(max(wlen)>maxWlen)
            maxWlen = max(wlen);
    }

    WLEN = span(minWlen,maxWlen,1000)*1e6;

    kount = 0;
    while(numberof(TMP)!=0)
    {
        kount++;
        toUse = (TMP(1,1)==TMP(1,))&
            (TMP(2,1)==TMP(2,))&
            (TMP(5,1)==TMP(5,));
        toNotUse = !toUse;
        val   = yocoStr2Double(TMP(3,where(toUse)));
        Units = TMP(5,where(toUse))(1);
        if(Units!="")
        {
            Units = "\n("+Units+")";
        }
        err  = yocoStr2Double(TMP(4,where(toUse)));
        errm = yocoStr2Double(TMP(7,where(toUse)));
        wlen = yocoStr2Double(TMP(6,where(toUse)));
        plsys,kount;
        if(numberof(val)==1)
            plt,pr1(val)+Units,avg(WLEN),0,tosys=1,height=30,justify="CH";
        else
        {
            plg,spline(val,wlen,WLEN),WLEN;
            plg,val,wlen,type="none",marks=1,marker='\2',color="red";
            if(!is_void(paramsErr))
            {
                if(dimsof(paramsErr)(1)==2)
                    pleb,val,wlen,dy=[err,errm],color="red";
                else
                    yocoPlotWithErrBars,val,err,wlen,type="none",marks=0;
            }
        }
        xytitles,,TMP(1,where(toUse))(1)+": "+TMP(2,where(toUse))(1)+Units,[0.02,0.02];
        limits,min(WLEN),max(WLEN),0;
        
        TMP = TMP(,where(toNotUse));
    }
    yocoNmXytitles,"Wavelength (!mm)",,[0.00,0.02];
}

/*******************************************************/

func computeInsVis(corrFlux, flux)
    /* DOCUMENT computeInsVis(corrFlux, flux)

       DESCRIPTION
       Computes "instrumental" visibilities from the input flux
       & correlated flux

       PARAMETERS
       - corrFlux: 
       - flux    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    return  abs(corrFlux) / abs(flux);
}

/*******************************************************/

func computeInsSqVis(corrFlux, flux)
    /* DOCUMENT computeInsSqVis(corrFlux, flux)

       DESCRIPTION
       Computes "instrumental" squared visibilities from the input flux
       & correlated flux

       PARAMETERS
       - corrFlux: 
       - flux    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(anyof(abs(corrFlux)>1e150))
        corrFlux(where(abs(corrFlux)>1e150)) = 1e150;
    
    if(anyof(abs(flux)>1e150))
        flux(where(abs(flux)>1e150)) = 1e150;
    
    return abs(corrFlux)^2 / abs(flux+(flux==0))^2;
}

/*******************************************************/

func computeInsDiffPhi(C, lambda, channelRefMat=)
    /* DOCUMENT computeInsDiffPhi(C, lambda, channelRefMat=)

       DESCRIPTION
       Computes "instrumental" differential phase from the input
       correlated flux. An optionnal matrix allows one to use
       versatile definition of the differential phase calculation

       PARAMETERS
       - C            : input complex coherent flux [nlambda]
       - lambda       : input wavelength
       - channelRefMat: reference channel matrix (0 for work channel,
       1 for reference channel)

       RETURN VALUES
       Returns the differential phase, as computed in AMBER, MIDI or
       MATISSE

       CAUTIONS
       If channelRefMat is empty, it will considered as the unit
       matrix, i.e. the default is to compute AMBER-like differential
       phase.

       SEE ALSO
       ComputePistonAIPS
    */
{
    nbWlen = numberof(lambda);

    // By default, the channelRefMat matrix is the unit matrix
    if(is_void(channelRefMat))
        channelRefMat = 1-unit(nbWlen);
    
    // Compute the piston or OPD based on the coherent flux
    pist = ComputePistonAIPS(C, lambda);

    // Correct the coherent flux from the computed OPD
    Ccorr = C * exp(-2i*pi*pist/lambda);

    // Calculate the reference channel
    Cref  = (Ccorr(,-)*channelRefMat)(avg,);

    // Calculate the reference channel
    Cwork  = (Ccorr(,-)*(1-channelRefMat))(avg,);

    // Compute the interspectrum
    W     = Ccorr*conj(Cref);

    // Get the differential phase, taking into account SIGFPE
    if(allof((W.im<1e299)&(W.re<1e299)))
        diffPhi = atan(W.im,W.re);
    else
        diffPhi = array(0.0,dimsof(W.im));

    // Return differential phase
    return -diffPhi;
}

/*******************************************************/

func computeInsDVis(corrFlux, flux, lambda, channelRefMat=)
    /* DOCUMENT computeInsDVis(corrFlux, flux, lambda, channelRefMat=)

       DESCRIPTION
       Computes "instrumental" differential visibility from the input flux
       & correlated flux. An optionnal matrix allows one to use versatile
       definition of the differential phase calculation

       PARAMETERS
       - corrFlux     : 
       - flux         : 
       - lambda       : 
       - channelRefMat: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    vis = abs(corrFlux) / abs(flux);
    nbWlen = numberof(lambda);
    
    if(is_void(channelRefMat))
        channelRefMat = 1.0;//-Units(nbWlen);

    visRef = (vis(,-)*channelRefMat)(avg,);
    if(noneof(visRef==0))
        dVis = vis / visRef;
    else
        dVis = (visRef==0) * vis / (visRef + visRef==0);
    
    return dVis;
}

/*******************************************************/

func computeInsSpec(flux, lambda, &continuum, channelRefMat=, plagesCont=, contType=, contParams=)
    /* DOCUMENT computeInsSpec(flux, lambda, &continuum, channelRefMat=, plagesCont=, contType=, contParams=)

       DESCRIPTION
       Computes "instrumental" spectrum from the input flux. An optionnal
       matrix allows one to use versatile definition of the normalization of
       the spectrum

       PARAMETERS
       - flux         : input spectrum
       - lambda       : input wavelength table
       - continuum    : continuum value
       - channelRefMat: Reference matrix
       - plagesCont   : Continuum wavelengths
       - contType     : type of interpolation for continuum estimate
       - contParams   : Parameters of interpolation function

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(contParams))
        contParams = 3;
    
    if(is_void(contType))
        contType="median";
    
    nbWlen = numberof(lambda);
    

    // Compute continuum
    if(contType=="polynom")
    {
        isCont = array(0,nbWlen);
        for(k=1;k<=dimsof(plagesCont)(3);k++)
        {
            isCont = isCont |
                (min(plagesCont(,k))<=lambda)&
                (max(plagesCont(,k))>=lambda);
        }
        wc = where(isCont);
        
        cont = yocoMathPolyFit(contParams,lambda(wc)-avg(lambda(wc)),flux(wc));
        continuum = yocoMathPoly(lambda-avg(lambda(wc)),cont);
    }
    if(contType=="median")
    {
        continuum = median(flux);
    }
    
    spec = flux / continuum;
    return spec;
}

/*******************************************************/

func computeInsClos(C)
    /* DOCUMENT computeInsClos(C)

       DESCRIPTION
       Computes "instrumental" closure phase from the input correlated flux.

       PARAMETERS
       - C: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // BS = C(1,..)*C(2,..)*C(3,..);
    BS = C(1,..)*C(2,..)*conj(C(3,..));
    // BS = C(1,..)*conj(C(2,..))*C(3,..);
    // BS = conj(C(1,..))*C(2,..)*C(3,..);
    // BS = conj(C(1,..)*C(2,..))*C(3,..);
    // BS = C(1,..)*conj(C(2,..)*C(3,..));
    // BS = conj(C(1,..))*C(2,..)*conj(C(3,..));
    // BS = conj(C(1,..)*C(2,..)*C(3,..));
    clos = atan(BS.im,BS.re);
    return -clos;
}

/*******************************************************/

func computeInsBSAmp(C)
    /* DOCUMENT computeInsBSAmp(C)

       DESCRIPTION
       Computes "instrumental" bispectrum amplitude from the input correlated
       flux & flux.

       PARAMETERS
       - C: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    BS = C(1,..)*C(2,..)*conj(C(3,..));
    amp = abs(BS);
    return amp;
}

/*******************************************************/

func performObsNight(stations, nightLength, obsFreq, ra, dec)
    /* DOCUMENT performObsNight(stations, nightLength, obsFreq, ra, dec)

       DESCRIPTION

       PARAMETERS
       - stations   : 
       - nightLength: 
       - obsFreq    : 
       - ra         : 
       - dec        : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    uvTable = u = v = w = theta = base = [];
    
    Nconfigs = dimsof(stations)(3);
    
    stardata = amdlibSTARVIS(0);
    
    stardata.date = pr1(day) + "/" + pr1(month) + "/" + pr1(year);
    
    // Detemine observed star's coordinates
    stardata.ra = ra;
    stardata.dec = dec;
    
    _amdlibGetVLTILocation, loc;
    
    nPoints = int(nightLength / obsFreq);

    for(ks=1;ks<=Nconfigs;ks++)
    {
        stat = stations(,ks);
        orig = [];
        Bases = amdlibComputeBaseVect(orig, stat);

        nbBases = numberof(Bases(1, ));
        
        for(ke=1;ke<=nPoints;ke++)
        {
            // Consider the star is always observed around meridian
            stardata.lst = ra - nightLength/2.0 + ke * obsFreq + random_n() * obsFreq;

            u = v = w = theta = base = [];
            
            for (iBase=1; iBase <= nbBases; iBase++)
            {
                bvect = Bases(,iBase);
                
                //Calcul du vecteur de base dans l'espace reciproque
                _amdlibComputeUvwCoord, stardata, bvect;
                
                if (iBase == 1)
                {
                    u = array(stardata.u, 1);
                    v = array(stardata.v, 1);
                    w = array(stardata.w, 1);
                    theta = array(stardata.theta, 1);
                    base = array(stardata.base, 1);
                }
                else
                {
                    grow, u, stardata.u;
                    grow, v, stardata.v;
                    grow, w, stardata.w;
                    grow, theta, stardata.theta;
                    grow, base, stardata.base;
                }
            }
            grow, uvTable, array([u,v],1);
        }
    }
    uvTable2 = transpose(uvTable, [1,2]);
    return uvTable2;
} 

/*******************************************************/

func writeParams(&toot, writeFracFlux=)
    /* DOCUMENT writeParams(&toot, writeFracFlux=)

       DESCRIPTION

       PARAMETERS
       - toot         : 
       - writeFracFlux: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    paramsErr = *toot.paramErr;
    var       = *toot.parVar;
    val       = *toot.modelVal;
    name      = *toot.parName;
    wlen      = *toot.parWlen;
    pwl       = *toot.parNbWlen;
    ns        =  toot.nSources;
    np        =  dimsof(pwl)(2);

    usedWlens = yocoStrRemoveMultiple(wlen);
    uN        = numberof(usedWlens);
    fluxes    = fluxesErr = array(0.0,ns,uN);
    
    kountt    = akountt = 1;
    
    DISPLAY_DATA       = [];
    // Loop on sources
    for(k=1;k<=ns;k++)
    {
        Units = *toot.parUnit;
        // Loop on parameters
        for(l=1;l<=np;l++)
        {
            // Parameter with no entry, skip.
            if(pwl(l,k)!=0)
            {
                for(m=1;m<=pwl(l,k);m++)
                {
                    if(Units(kountt)=="no")
                    {
                        unitDisp = "";
                        fac=1;
                    }
                    else
                    {
                        fac = getConvFact(Units(kountt));
                        unitDisp = Units(kountt);
                    }
                    
                    err = "";

                    tmp2 = [];
                    tmp2 = grow(pr1(k),name(kountt),swrite(format="%f",val(kountt)*fac)+err+" "+unitDisp,pr1(wlen(kountt)*unit2micro));
                    grow,DISPLAY_DATA,array(tmp2,1);

                    
                    kountt++;
                }
            }
        }
    }
    
    /* max width of each column and output format */
    width = strlen(DISPLAY_DATA)(,max);
    nh = numberof(width);
    format = "%"+pr1(width(1))+"s: "+
        "%"+pr1(width(2))+"s | "+
        "%"+pr1(width(3))+"s | "+
        "%"+pr1(width(4))+"s | %5s | \n";
    format2 = "%"+pr1(width(1)+width(2)+2)+"s | "+
        "%"+pr1(width(3))+"s | "+
        "%"+pr1(width(4))+"s | %5s |\n";

    sep2 = "";
    sepW = sum(width)+17;
    while(strlen(sep2)<sepW)
        sep2+="-";
  
    kountt = akountt = 1;
    write,sep2;
    write,format=format2,"Par.","val.","wlen","Fixed";
    fxed = ["N","Y"];
    types = ALL_MODELS.type; // see visModels.i
    trad  = ALL_MODELS.description; // see visModels.i
    for(k=1;k<=ns;k++)
    {
        sep = "-( "+trad(where(types==(*tot.typei)(k)))(1)+" )";
        while(strlen(sep)<sepW)
            sep+="-";
        
        sepi = "  ,"+sum(array("-",strlen(trad(where(types==(*tot.typei)(k)))(1))))+".";
        sepj = "  `"+sum(array("-",strlen(trad(where(types==(*tot.typei)(k)))(1))))+"'";
        
        write,sepi;  
        write,sep;
        write,sepj;  
        for(l=1;l<=np;l++)
        {
            // Fixed param, do not change
            if(pwl(l,k)!=0)
            {
                for(m=1;m<=pwl(l,k);m++)
                {
                    if(Units(kountt)=="no")
                    {
                        unitDisp = "";
                        fac=1;
                    }
                    else
                    {
                        fac = getConvFact(Units(kountt));
                        unitDisp = Units(kountt);
                    }
                        
                    err = "";
                        
                    write,format=format,pr1(k),
                        name(kountt),
                        swrite(format="%f",val(kountt)*fac)+err+" "+unitDisp,
                        pr1(wlen(kountt)*unit2micro),fxed(int(var(l,k)));
                        
                    if(name(kountt)=="x")
                    {
                        xTmp = val(kountt);
                        if(err!="")
                        {
                            xerrTmp = paramsErr(akountt-1);
                        }
                    }
                    else if(name(kountt)=="y")
                    {
                        yTmp = val(kountt);
                        
                        R = abs(xTmp,yTmp);
                        T = rad2deg*atan(xTmp,yTmp);
                        
                        if((err!="")&&
                           (abs(xTmp)<1e99)&&
                           (abs(xerrTmp)<1e99)&&
                           (abs(yTmp)<1e99))
                        {
                            yerrTmp = paramsErr(akountt-1);
                            Rerr = sqrt(((xTmp*xerrTmp)+(yTmp*yerrTmp))^2/
                                        ((xTmp^2+yTmp^2)+
                                         ((xTmp^2+yTmp^2)==0)));
                            
                            tmpp = ((xTmp^2+yTmp^2)^2);
                            Terr = rad2deg*sqrt((((xTmp*yerrTmp)-
                                                  (yTmp*xerrTmp))^2) /
                                                (tmpp+(tmpp==0))
                                                );
                            Rwerr = " +/- "+swrite(format="%2.2g",Rerr*fac);
                            Twerr = " +/- "+swrite(format="%2.2g",Terr);
                        }
                        else
                        {
                            Rwerr = "";
                            Twerr = "";
                        }
                        
                        write,"R =", swrite(format="%f",R*fac)+Rwerr+" "+unitDisp+",",
                            "Theta =", swrite(format="%f",T)+Twerr+" degrees";
                    }
                    
                    kountt++;
                }
            }
        }
    }
    write,sep2;

    if(writeFracFlux==1)
    {
        wh = where(usedWlens!=wlen(where(pwl==1))(1));
        // Loop on sources to write the fractional flux of each source
        for(k=1;k<=ns;k++)
        {
            // Loop on parameters
            for(l=1;l<=uN;l++)
            {
                if(fluxes(k,l)==0)
                {
                    idx  = where((fluxes(k,)!=0));
                    if(numberof(idx)==1)
                    {
                        idx2 = where((fluxes(k,)==0));
                        fluxes(k,idx2) = fluxes(k,idx);
                    }
                }
            }
        }
        sumFluxes = fluxes(sum,);
        if(!is_void(paramsErr))
        {
            sumFluxesVar = (fluxesErr^2)(sum,);
        }
        // Loop on sources
        for(k=1;k<=ns;k++)
        {
            write,k,":",fluxes(k,wh)/sumFluxes(wh)*100,"%, at",usedWlens(wh)*1e6,"microns";
            write,"\n";
        }
    }
}

/*******************************************************/

func buildFov(&toot, sFov, iFov)
    /* DOCUMENT buildFov(&toot, sFov, iFov)

       DESCRIPTION

       PARAMETERS
       - toot: 
       - sFov: 
       - iFov: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(sFov))
    {
        sFov = 10.0;
    }
  
    if(is_void(iFov))
    {
        iFov = 100*mas2rad;
    }

    iFov = double(iFov);
    sFov = double(sFov);
  
    ns = toot.nSources;
    kountt = akountt = 1;

    nb = *toot.parNbWlen;
    iv = *toot.parVar;
    typei = *toot.typei;
    fov = [];
    nPar = symbol_def(toot.funcFit)(toot);
  
    for(k=1;k<=ns;k++)
    {
        for(l=1;l<=nPar;l++)
        {
            // Fixed param, do not change
            if(nb(l,k)==0)
            {
                //               grow,modelVal,oldParVal(kountt);
                //               kountt++;
            }
            else if(nb(l,k)==1)
            {
                if(iv(l,k)==1)
                {
                    if(l==1)
                        grow,fov,sFov;
                    else
                        grow,fov,iFov;
                }
            }
            else
            {
                if(iv(l,k)==1)
                {
                    if(l==1)
                        grow,fov,array(sFov,+nb(l,k));
                    else
                        grow,fov,array(iFov,+nb(l,k));
                    kountt+=nb(l,k);
                }
                else
                    kountt+=nb(l,k);
            }
        }
    }
    
    if(numberof(*toot.fov)==numberof(fov))
        *toot.fov  = fov;
    else
        toot.fov    = &fov;
    
}

/*******************************************************/

func updateParams(&toot, modelVal)
    /* DOCUMENT updateParams(&toot, modelVal)

       DESCRIPTION

       PARAMETERS
       - toot    : 
       - modelVal: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    *toot.modelVal = modelVal;
    ns = toot.nSources;
    kountt = akountt = 1;

    nb = *toot.parNbWlen;
    iv = *toot.parVar;
    typei = *toot.typei;
    params = [];
    nPar = symbol_def(toot.funcFit)(toot);
    
    for(k=1;k<=ns;k++)
    {
        for(l=1;l<=nPar;l++)
        {
            // Fixed param, do not change
            if(nb(l,k)==0)
            {
                //               grow,modelVal,oldParVal(kountt);
                //               kountt++;
            }
            else if(nb(l,k)==1)
            {
                if(iv(l,k)==1)
                {
                    grow,params,modelVal(kountt);
                    kountt++;
                }
                else
                    kountt++;
            }
            else
            {
                if(iv(l,k)==1)
                {
                    grow,params,modelVal(kountt:kountt+nb(l,k)-1);
                    kountt+=nb(l,k);
                }
                else
                    kountt+=nb(l,k);
            }
        }
    }
    
    if(numberof(*toot.modelVal)==numberof(modelVal))
        *toot.param  = params;
    else
        toot.param    = &params;
}

/*******************************************************/

func updateParVal(&toot, params)
    /* DOCUMENT updateParVal(&toot, params)

       DESCRIPTION

       PARAMETERS
       - toot  : 
       - params: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    ns = toot.nSources;
    kountt = akountt = 1;

    toot.param = &params;
    oldParVal  = *toot.modelVal;
    nb = *toot.parNbWlen;
    iv = *toot.parVar;
    typei = *toot.typei;
    modelVal = [];
    nPar = symbol_def(toot.funcFit)(toot);
    
    for(k=1;k<=ns;k++)
    {
        for(l=1;l<=nPar;l++)
        {
            // Fixed param, do not change
            if(nb(l,k)==0)
            {
                // Do nothing
                // grow,modelVal,oldParVal(kountt);
                // kountt++;
            }
            else if(nb(l,k)==1)
            {
                if(iv(l,k)==1)
                {
                    grow,modelVal,params(akountt);
                    akountt++;
                }
                else
                {
                    grow,modelVal,oldParVal(kountt);
                }
                kountt++;
            }
            else
            {
                if(iv(l,k)==1)
                {
                    grow,modelVal,params(akountt:akountt+nb(l,k)-1);
                    akountt+=nb(l,k);
                }
                else
                {
                    grow,modelVal,oldParVal(kountt:kountt+nb(l,k)-1);
                }
                kountt+=nb(l,k);
            }
        }
    }
    
    if(numberof(*toot.modelVal) == numberof(modelVal))
        *toot.modelVal  = modelVal;
    else
        toot.modelVal    = &modelVal;

    if(toot.absFlux==1)
    {
        (*toot.modelVal)(where(strmatch(*toot.parName, "flux", 1))) =
            abs((*toot.modelVal)(where(strmatch(*toot.parName, "flux", 1))));
    }
    if(toot.absSize==1)
    {
        if(anyof((*toot.modelVal)(where(strmatch(*toot.parName,"FWHM", 1)|
                                        strmatch(*toot.parName,"size",1)|
                                        strmatch(*toot.parName,"thick",1)|
                                        strmatch(*toot.parName,"taille",1)))))
        {
            (*toot.modelVal)(where(strmatch(*toot.parName,"FWHM", 1)|
                                   strmatch(*toot.parName,"size",1)|
                                   strmatch(*toot.parName,"thick",1)|
                                   strmatch(*toot.parName,"taille",1))) =
                abs((*toot.modelVal)(where(strmatch(*toot.parName,"FWHM", 1)|
                                           strmatch(*toot.parName,"size",1)|
                                           strmatch(*toot.parName,"thick",1)|
                                           strmatch(*toot.parName,"taille",1))));
        }
    }
}

/*******************************************************/

func getDataInLine(data, use_vis2, use_clos, use_bsamp, use_dVis, use_dPhi, use_spec, &errInLine, &dataType, &wlenInLine, &U1inLine, &V1inLine, &U2inLine, &V2inLine, &timeInLine)
    /* DOCUMENT getDataInLine(data, use_vis2, use_clos, use_bsamp, use_dVis, use_dPhi, use_spec, &errInLine, &dataType, &wlenInLine, &U1inLine, &V1inLine, &U2inLine, &V2inLine, &timeInLine)

       DESCRIPTION
       "flattens" an observation to simplify computations

       PARAMETERS
       - data      : 
       - use_vis2  : 
       - use_clos  : 
       - use_bsamp : 
       - use_dVis  : 
       - use_dPhi  : 
       - use_spec  : 
       - errInLine : 
       - dataType  : 
       - wlenInLine: 
       - U1inLine  : 
       - V1inLine  : 
       - U2inLine  : 
       - V2inLine  : 
       - timeInLine: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    datInLine = errInLine = dataType = wlenInLine =
        U1inLine = V1inLine = U2inLine = V2inLine = timeInLine = [];
    
    nObs = numberof(data);
    for(ko=1;ko<=nObs;ko++)
    {
        if(use_spec)
        {
            sp = *(data.spec)(ko);
            if(!is_void(sp))
            {
                se   = (*(data.specErr)(ko));
                wl   = (*(data.EFF_WAVE)(ko));
                time = (*(data.TIME)(ko));

                grow, wlenInLine, double(wl);
                grow, U1inLine, array(0.0,numberof(wl));
                grow, U2inLine, array(0.0,numberof(wl));
                grow, V1inLine, array(0.0,numberof(wl));
                grow, V2inLine, array(0.0,numberof(wl));
                grow, timeInLine, array(time(1),numberof(wl));
                grow, datInLine, sp;
                grow, errInLine, se;
                grow, dataType, array("SP",numberof(sp));
            }
        }
        
        if(use_clos)
        {
            ctmp = *(data.T3PHI)(ko);
            if(!is_void(ctmp))
            {
                if(dimsof(ctmp)(1)>=2)
                    nbClos = dimsof(ctmp)(3);
                else
                    nbClos = numberof(ctmp);
                
                for(j=1;j<=nbClos;j++)
                {
                    c    = (ctmp)(,j);
                    ce   = (*(data.T3PHIERR)(ko))(,j);
                    wl   = (*(data.EFF_WAVE)(ko));
                    u1   = (*(data.U1COORD)(ko));
                    v1   = (*(data.V1COORD)(ko));
                    u2   = (*(data.U2COORD)(ko));
                    v2   = (*(data.V2COORD)(ko));
                    time = (*(data.TIME3)(ko));
                    
                    grow, wlenInLine, double(wl);
                    grow, U1inLine, array(u1(j),numberof(wl));
                    grow, V1inLine, array(v1(j),numberof(wl));
                    grow, U2inLine, array(u2(j),numberof(wl));
                    grow, V2inLine, array(v2(j),numberof(wl));
                    grow, timeInLine, array(time(j),numberof(wl));
                    grow, datInLine, c;
                    grow, errInLine, ce;
                    grow, dataType, array("CP",numberof(c));
                }
            }
        }
        
        if(use_bsamp)
        {
            ctmp = *(data.T3AMP)(ko);
            if(!is_void(ctmp))
            {
                nbBsamp = dimsof(ctmp)(3);
                for(j=1;j<=nbBsamp;j++)
                {
                    c    = (ctmp)(,j);
                    ce   = (*(data.T3AMPERR)(ko))(,j);
                    wl   = (*(data.EFF_WAVE)(ko));
                    u1   = (*(data.U1COORD)(ko));
                    v1   = (*(data.V1COORD)(ko));
                    u2   = (*(data.U2COORD)(ko));
                    v2   = (*(data.V2COORD)(ko));
                    time = (*(data.TIME3)(ko));
                    
                    grow, wlenInLine, double(wl);
                    grow, U1inLine, array(u1(j),numberof(wl));
                    grow, V1inLine, array(v1(j),numberof(wl));
                    grow, U2inLine, array(u2(j),numberof(wl));
                    grow, V2inLine, array(v2(j),numberof(wl));
                    grow, timeInLine, array(time(j),numberof(wl));
                    grow, datInLine, c;
                    grow, errInLine, ce;
                    grow, dataType, array("CP",numberof(c));
                }
            }
        }
        
        vis2 = *data(ko).VIS2DATA;
        if(!is_void(vis2) && use_vis2)
        {
            if(dimsof(vis2)(1)>=2)
                nbBases = dimsof(vis2)(3);
            else
                nbBases = numberof(vis2);
            for(j=1;j<=nbBases;j++)
            {
                v2   = vis2(..,j);
                v2e  = (*((data).VIS2ERR)(ko))(..,j);
                wl   = (*(data.EFF_WAVE)(ko));
                u    = (*(data.UCOORD)(ko));
                v    = (*(data.VCOORD)(ko));
                time = (*(data.TIME)(ko));

                grow, wlenInLine, double(wl);
                grow, U1inLine, array(u(j),numberof(wl));
                grow, V1inLine, array(v(j),numberof(wl));
                grow, U2inLine, array(0.0,numberof(wl));
                grow, V2inLine, array(0.0,numberof(wl));
                grow, timeInLine, array(time(j),numberof(wl));
                grow, datInLine, v2;
                grow, errInLine, v2e;
                grow, dataType, array("V2",numberof(v2));
            }
        }

        dVis = *data(ko).VISAMP;
        if(!is_void(dVis) && use_dVis)
        {
            nbBases = dimsof(dVis)(3);
            for(j=1;j<=nbBases;j++)
            {
                dv   = dVis(,j);
                dve  = (*(data.VISAMPERR)(ko))(,j);
                wl   = (*(data.EFF_WAVE)(ko));
                u    = (*(data.UCOORD)(ko));
                v    = (*(data.VCOORD)(ko));
                time = (*(data.TIME)(ko));

                grow, wlenInLine, double(wl);
                grow, U1inLine, array(u(j),numberof(wl));
                grow, V1inLine, array(v(j),numberof(wl));
                grow, U2inLine, array(0.0,numberof(wl));
                grow, V2inLine, array(0.0,numberof(wl));
                grow, timeInLine, array(time(j),numberof(wl));
                grow, datInLine, dv;
                grow, errInLine, dve;
                grow, dataType, array("DV",numberof(dv));
            }
        }
        
        dPhi = *data(ko).VISPHI;
        if(!is_void(dPhi) && use_dPhi)
        {
            nbBases = dimsof(dPhi)(3);
            for(j=1;j<=nbBases;j++)
            {
                dp   = (*(data.VISPHI)(ko))(,j);
                dpe  = (*(data.VISPHIERR)(ko))(,j);
                wl   = (*(data.EFF_WAVE)(ko));
                u    = (*(data.UCOORD)(ko));
                v    = (*(data.VCOORD)(ko));
                time = (*(data.TIME)(ko));

                grow, wlenInLine, double(wl);
                grow, U1inLine, array(u(j),numberof(wl));
                grow, V1inLine, array(v(j),numberof(wl));
                grow, U2inLine, array(0.0,numberof(wl));
                grow, V2inLine, array(0.0,numberof(wl));
                grow, timeInLine, array(time(j),numberof(wl));
                grow, datInLine, dp;
                grow, errInLine, dpe;
                grow, dataType, array("DP",dimsof(dp));
            }
        }
    }
    
    return datInLine;
}

/*******************************************************/

func fitFunc(&toot, a, updatePar=)
    /* DOCUMENT fitFunc(&toot, a, updatePar=)

       DESCRIPTION

       PARAMETERS
       - toot     : 
       - a        : 
       - updatePar: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    
    // FIXME: problem of memory management here
    if(updatePar)
        updateParVal, toot, a;
            
    res = symbol_def(toot.funcFit)(toot,a);
            
    model     = *toot.model;
    use_vis2  = toot.use_vis2;
    use_clos  = toot.use_clos;
    use_bsamp  = toot.use_bsamp;
    use_dVis  = toot.use_dVis;
    use_dPhi  = toot.use_dPhi;
    use_spec  = toot.use_spec;
    
    nObs      = numberof(model);
    iv        = *toot.parVar;
    wlens     = *toot.parWlen;
    modelVal  = *toot.modelVal;
    typei     = *toot.typei;
    nS        = toot.nSources;

    nPar = symbol_def(toot.funcFit)(toot);
    
    
    modInLine = getDataInLine(model,
                              use_vis2, use_clos, use_bsamp,
                              use_dVis, use_dPhi, use_spec,
                              errors, dataType);
    
    
    return modInLine;
}

/*******************************************************/

func resFunc(&toot, a)
    /* DOCUMENT resFunc(&toot, a)

       DESCRIPTION

       PARAMETERS
       - toot: 
       - a   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    dataLine    = *toot.dataLine;
    dataErrLine = *toot.dataErrLine;
    if(dimsof(dataErrLine)(1)==1)
        dataWeight = 1.0/dataErrLine^2;
    else
        dataWeight = dataErrLine;
    
    dataType    = *toot.dataType;
    res         = computeRes(toot, a, dataLine, dataWeight, dataType);

    return res;
}

/*******************************************************/

func computeRes(&tot, a, dataLine, dataWeight, dataType)
    /* DOCUMENT computeRes(&tot, a, dataLine, dataWeight, dataType)

       DESCRIPTION

       PARAMETERS
       - tot       : 
       - a         : 
       - dataLine  : 
       - dataWeight: 
       - dataType  : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    mod = fitFunc(tot,a);
    res = [];

    // Initialize residuals vector
    rsv = array(0.0,numberof(dataType));
    
    // Squared visibilities
    wv2 = where(dataType=="V2");
    if(numberof(wv2)!=0)
    {
        rsv(wv2) = (dataLine(wv2)-mod(wv2));
    }
    
    // Closure phases
    wcp = where(dataType=="CP");
    if(numberof(wcp)!=0)
    {
        W = exp(1i*dataLine(wcp)) * exp(-1i*mod(wcp));
        cpd = atan(W.im,W.re);
        
        //         W = exp(1i*dataLine(wcp)) - exp(1i*mod(wcp));
        //         cpd = abs(W.im,W.re);

        rsv(wcp) = cpd;
    }

    // Differential phases
    wdp = where(dataType=="DP");
    if(numberof(wdp)!=0)
    {
        W = exp(1i*dataLine(wdp)) * exp(-1i*mod(wdp));
        phd = atan(W.im,W.re);
        rsv(wdp) = phd;
    }

    // Differential visibilities
    wdv = where(dataType=="DV");
    if(numberof(wdv)!=0)
    {
        rsv(wdv) = dataLine(wdv)-mod(wdv);
    }

    // Spectrum
    wsp = where(dataType=="SP");
    if(numberof(wsp)!=0)
    {
        rsv(wsp) = dataLine(wsp)-mod(wsp);
    }

    return rsv;
}

/*******************************************************/

func regul_compactness(tot)
    /* DOCUMENT regul_compactness(tot)

       DESCRIPTION

       PARAMETERS
       - tot: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    kountt = 1;
    val  = *tot.modelVal;
    
    ns = tot.nSources;
    name = *tot.parName;
    pwl  = *tot.parNbWlen;
    np = dimsof(pwl)(2);

    RGL = [];
    for(k=1;k<=ns;k++)
    {
        // Loop on parameters
        for(l=1;l<=np;l++)
        {
            if(pwl(l,k)!=0)
            {
                if(name(kountt)=="X")
                {
                    xTmp = val(kountt);
                }
                else if(name(kountt)=="Y")
                {
                    yTmp = val(kountt);
                    grow, RGL, abs(xTmp,yTmp);
                }
                else if(strmatch(name(kountt), "FWHM"))
                {
                    grow, RGL, abs(val(kountt));
                }
                
                kountt+=pwl(l,k);
            }
        }
    }
    if(is_void(RGL))
        RGL = 0.0;
    
    return RGL;
}

/*******************************************************/

func computeChi2(&tot, a, dataLine, dataWeight, dataType)
    /* DOCUMENT computeChi2(&tot, a, dataLine, dataWeight, dataType)
       
       DESCRIPTION
       
       PARAMETERS
       - tot       : 
       - a         : 
       - dataLine  : 
       - dataWeight: 
       - dataType  : 
       
       RETURN VALUES
       
       CAUTIONS
       
       EXAMPLES
       
       SEE ALSO
    */
{
    if(dimsof(dataWeight)(1)==1)
        vec=1;
    else
        vec=0;
    
    res = computeRes(tot, a, dataLine, dataWeight, dataType);

    // Get rid of Nans
    wr         = where(res==res);
    res        = res(wr);
    dataLine   = dataLine(wr);
    dataWeight = dataWeight(wr);
    dataType   = dataType(wr);

    if(vec==1)
        chi2 = sum(res^2 * dataWeight);
    else
        chi2 = (res(+) * dataWeight(,+))(+) * res(+);

    return chi2;
}

/*******************************************************/

func plotModelImageMultiple(&toot, a, &IMAGE, &X0, &Y0, &WLEN, fov=, Np=, plotLog=, plotPow=, win=, clear=, kill=, useModelWlen=, wlen=, projAngle=, project=, cmin=, cmax=, normByMax=)
    /* DOCUMENT plotModelImageMultiple(&toot, a, &IMAGE, &X0, &Y0, &WLEN, fov=, Np=, plotLog=, plotPow=, win=, clear=, kill=, useModelWlen=, wlen=, projAngle=, project=, cmin=, cmax=, normByMax=)

       DESCRIPTION
       From a model & data structure, plots the image of the model

       PARAMETERS
       - toot        : the fit structure
       - a           : the fit parameters
       - IMAGE       : the image (output)
       - X0          : 
       - Y0          : 
       - WLEN        : 
       - fov         : the image field of view
       - Np          : the image number of pixels (default 256)
       - plotLog     : whether to plot the image in log scales
       - plotPow     : plot image with sqrt or power scale
       - win         : the window where to plot (default 1)
       - clear       : whether to clear the window (with fma, default 1)
       - kill        : whether to kill and create the window (default 1)
       - useModelWlen: 
       - wlen        : 
       - projAngle   : 
       - project     : 
       - cmin        : 
       - cmax        : 
       - normByMax   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    local FLUX, X, Y, FWHM1, FWHM2, ANGLE,
        ADD7, ADD8, ADD9, ADD10, ADD11, ADD12, ADD13, ADD14, ADD15,
        ADD16, ADD17, ADD18, ADD19, ADD20, pars, units, typei;
        
    pars  = ["flux", "x", "y", "fwhm1", "fwhm2", "angle", "add7", "add8", "add9", "add10", "add11", "add12", "add13", "add14", "add15", "add16", "add17", "add18", "add19", "add20"];

    if(is_void(kill))
        kill=1;
    
    if(!is_void(projAngle))
        projAngle = pi/2-projAngle;
    
    if(is_void(win))
        win=1;
    
    if(is_void(Np))
        Np = 256;
    
    if(is_void(cmin))
        cmin = 0.;
    
    if(is_void(cmax))
        cmax = 1.;
    
    if(kill==1)
        winkill,win;

    if(clear==1)
    {
        window,win;
        fma;
    }
    
    mod  = *toot.model;
    data = *toot.data;

    if(is_void(useModelWlen))
    {
        useModelWlen=1;
    }
    
    if(is_void(mod))
    {
        mod            = array(amplOiData(),dimsof(data));
        mod.EFF_WAVE   = data.EFF_WAVE;
        mod.UCOORD     = data.UCOORD;
        mod.VCOORD     = data.VCOORD;
        mod.U1COORD    = data.U1COORD;
        mod.U2COORD    = data.U2COORD;
        mod.V1COORD    = data.V1COORD;
        mod.V2COORD    = data.V2COORD;
        mod.hdr.TARGET = data.hdr.TARGET;
        mod.hdr.file   = data.hdr.file;
        mod.TIME       = data.TIME;
        mod.TIME3      = data.TIME3;
        mod.EFF_BAND   = data.EFF_BAND;
    }
        
    nObs  = numberof(data);
    nb    = *toot.parNbWlen;
    wlens = *toot.parWlen;
    
    usedwlen = yocoStrRemoveMultiple(wlens);
    usedwlen = usedwlen(where(usedwlen!=0));
    //usedwlen = usedwlen(where(usedwlen!=wlens(where(nb==1))(1)));

    
    if(is_void(wlen))
    {
        if(useModelWlen==1)
            wlen = usedwlen;
        else
        {
            wlen = [];
            for(w=1;w<=numberof(data);w++)
            {
                grow,wlen,*data(w).EFF_WAVE;
            }
            wlen = yocoStrRemoveMultiple(wlen);
        }
    }
    
    nw     = numberof(wlen);
    nbWlen = numberof(wlen);

    if(nw==0)
        nw=1;

    // Determine the number of windows to use
    nWin = nw;
    // Create the window
    if(kill==1)
    {
        sqWinSize1 = sqrt(nWin);

        fracWin = 0.50;
        sqX = int(max(1,floor(sqWinSize1/sqrt(fracWin))));
        sqY = int(max(1,floor(sqWinSize1*sqrt(fracWin))));
        
        while(sqX*sqY<nWin)
            sqX++;

        if(sqY==1)
            fx=1;
        
        winkill,win;
        //yocoNmCreate,win,sqX,sqY,dy=0.02,dx=0.06,height=800,width=800,wait=1, V = [0.12,0.73,0.1,0.9], dpi=65;
        yocoNmCreate, win, sqX, sqY, fx=fx, fy=,
            square=1, wait=1, dx=0.025, dy=0.025,
            style="work.gs", height=750, width=950, landscape=1,
            V = [0.15,0.85,0.12,0.73], dpi=100;

        if(PLOT_FFT)
            yocoNmCreate, win+1, sqX, sqY, fx=(sqY==1), fy=,
                square=1, wait=1, dx=0.025, dy=0.025,
                style="work.gs", height=750, width=950, landscape=1,
                V = [0.15,0.85,0.12,0.73], dpi=100;
    }
    else
    {
        window, win, wait=1;
        //yocoNmCreate,win,1,nWin,dy=0.0,fx=0,height=1000,width=800,wait=1;
    }
    
    if(clear==1)
        fma;
        
    modelVal = *toot.modelVal;
    typei    = *toot.typei;
    nS       = toot.nSources;
        
    nPar = symbol_def(toot.funcFit)(toot);
    modInLine = [];
    
    parsI  = strcase(1,pars);
    
    for(l=1;l<=nPar;l++)
    {
        symbol_set, parsI(l),array(0.0, nbWlen, 1);
    }
    
    iKountt = 0;
    kountt = 1;
    for(gfsk=1;gfsk<=nS;gfsk++)
    {
        for(l=1;l<=nPar;l++)
        {
            iKountt += nbWlen;
            nbElem = numberof(symbol_def(parsI(l)));
            
            if(iKountt>nbElem)
                symbol_set, parsI(l),
                    grow(symbol_def(parsI(l)),
                         array(0.0,dimsof(symbol_def(parsI(l)))));
            
            // allocate values
            knt = nb(l,gfsk);
            
            if(knt==0)
            {
                tmp = symbol_def(parsI(l));
                tmp(..,gfsk) = array(0.0, nbWlen,1);
                symbol_set,parsI(l),tmp;
            }
            else if(knt==1)
            {
                value = modelVal(kountt);
                tmp = symbol_def(parsI(l));
                tmp(..,gfsk) = array(value, nbWlen,1);
                symbol_set,parsI(l),tmp;
            }
            else
            {
                wl = wlens(kountt:kountt+knt-1);
                value = modelVal(kountt:kountt+knt-1);
                tmp = symbol_def(parsI(l));
                tmp(..,gfsk) = array(spline(value, wl, wlen),1);
                symbol_set,parsI(l),tmp;
            }
            
            kountt+=knt;
        }
    }
    for(l=1;l<=nPar;l++)
        symbol_set,parsI(l),
            symbol_def(parsI(l))(,:nS);
    
    if(is_void(fov))
        fov = 1.5*max(abs(FWHM1, FWHM2, X, Y));

    xy = UVGrid(fov/2*rad2mas,Np);
    
    XX = xy(1,);
    YY = xy(2,);
    xx = grid2ima(XX);
    yy = grid2ima(YY);
    X0 = xx(,1);
    Y0 = yy(1,);
    
    IMAGE = AMP = [];
    for(imi=1;imi<=nw;imi++)
    {
        basemax = wlen(imi)*Np/fov/2;
        UVTable = UVGrid(basemax, Np);
        
        U       = UVTable(1,);
        V       = UVTable(2,);
        if((!is_void(projAngle)))
        {
            if((project==1))
            {
                U2 =  U * cos(projAngle) + V * sin(projAngle);
                V  = -U * sin(projAngle) + V * cos(projAngle);
                U = U2;
            }
        }
        time = (*data.TIME(1))(1);

        CC = visMultipleResolved(U, V, wlen(imi), time, typei,
                                 transpose(array(FLUX(imi,),1)),
                                 transpose(array(X(imi,),1)),
                                 transpose(array(Y(imi,),1)),
                                 transpose(array(FWHM1(imi,),1)),
                                 transpose(array(FWHM2(imi,),1)),
                                 transpose(array(ANGLE(imi,),1)),
                                 transpose(array(ADD7(imi,),1)),
                                 transpose(array(ADD8(imi,),1)),
                                 transpose(array(ADD9(imi,),1)),
                                 transpose(array(ADD10(imi,),1)),
                                 transpose(array(ADD11(imi,),1)),
                                 transpose(array(ADD12(imi,),1)),
                                 transpose(array(ADD13(imi,),1)),
                                 transpose(array(ADD14(imi,),1)),
                                 transpose(array(ADD15(imi,),1)),
                                 transpose(array(ADD16(imi,),1)),
                                 transpose(array(ADD17(imi,),1)),
                                 transpose(array(ADD18(imi,),1)),
                                 transpose(array(ADD19(imi,),1)),
                                 transpose(array(ADD20(imi,),1)))  *
            visGaussianDisk(U, V, wlen(imi), time, 3.5*fov/Np, 0, 0);
        
        amp    = grid2ima(abs(CC(,1)));
        phi    = grid2ima(atan(CC(,1).im,CC(,1).re));
        vis2D  = amp * exp(1i * phi);
        fftVis = fft(roll(vis2D),[-1,-1]);
        
        // XYTable = UVGrid(fov, Np);

        
        image  = roll(abs(fftVis));
        image  = image/avg(image+allof(image==0));
        
        grow,IMAGE,array(image,1);
        grow,AMP,array(amp,1);
    }
    
    // Normalize by peak value of image
    normByMax=1;
    if(normByMax==1)
    {
        IMAGE = IMAGE / IMAGE(max,max,)(-,-,);
    }
    
    for(imi=1;imi<=nw;imi++)
    {
        window,win;
        plsys,imi;

        if((!is_void(projAngle))&&(project==1))
        {
            plg,IMAGE(,sum,imi),transpose(yy)(avg,);
        }
        else
        {
            if(plotLog==1)
            {
                pli,log(abs(IMAGE(..,imi)+(IMAGE(..,imi)==0))),
                    min(xx),min(yy),max(xx),max(yy),cmin=cmin, cmax=cmax;
                plotMyLUT,cmin,cmax,orient=1,plotLog=1;
            }
            else if(!is_void(plotPow))
            {
                // pli,(abs(IMAGE+(IMAGE==0)))^plotPow,
                //    max(x),min(y),min(x),max(y),cmin=0;
                
                pli,(abs(IMAGE(..,imi)+(IMAGE(..,imi)==0)))^plotPow,
                    min(xx),min(yy),max(xx),max(yy), cmin=cmin, cmax=cmax;
                plotMyLUT,cmin,cmax,orient=1,plotPow=plotPow;
                
                // plf,(abs(IMAGE+(IMAGE==0)))^plotPow,-xx,yy,cmin=cmin, cmax=cmax;
            }
            else
            {
                pli,abs(IMAGE(..,imi)),
                    min(xx),min(yy),max(xx),max(yy), cmin=cmin, cmax=cmax;
                plotMyLUT,cmin,cmax,orient=1;
            }
            limits, max(xx), min(xx),square=1;
            yocoAstroPlotNorthEast,
                -fov/2*rad2mas*0.9,
                -fov/2*rad2mas*0.9,
                0.3*fov/2*rad2mas,
                sepChar=0.35*fov/2*rad2mas,color="white";
            //plf,IMAGE,y,x,cmin=0;

        }
        
        if(!is_void(projAngle))
        {
            pldj,fov/2*rad2mas*cos(projAngle),
                fov/2*rad2mas*sin(projAngle),
                -fov/2*rad2mas*cos(projAngle),
                -fov/2*rad2mas*sin(projAngle), color="green";
        }          
        plt,swrite(wlen(imi)*1e6, format="%3.3f")+"!mm",fov/2.5*rad2mas,fov/2.5*rad2mas,tosys=imi,color="red",font="courierB";

        if(PLOT_FFT)
        {
            // FT plot
            window,win+1;
            plsys,imi;

            pli,(abs(AMP(..,imi)+(IMAGE(..,imi)==0))),
                min(xx),min(yy),max(xx),max(yy), cmin=0, cmax=1;
        }
    }
    
    yocoNmXytitles,"!a (mas)","!d (mas)";
    yocoPlotColorLookupTable,cmin,cmax,orient=1;

    //    cfkj()
    return 0;
}

func blackmann(x,xmax)
{
    return 0.42 - 0.5 * cos(2*pi*x/xmax) + 0.08 * cos(4*pi*x/xmax);
}

/*******************************************************/

func unwrap(vector, center=, wrapValue=)
    /* DOCUMENT unwrap(vector, center=, wrapValue=)

       SEE ALSO:
    */
{
    // Set default value to angle wrapping
    if(is_void(wrapValue))
        wrapValue = 2*pi;

    // Compute one-by-one difference
    gradient = vector(dif,..);
    
    // unwrap on one direction
    lst = where(gradient > wrapValue/2);
    if(is_array(lst))
        gradient(lst) -= wrapValue;

    // unwrap on the other direction
    lst = where(gradient < -wrapValue/2);
    if(is_array(lst))
        gradient(lst) += wrapValue;

    // Recenter values to zero average gradient
    if (!is_void(center))
        gradient = gradient - gradient(avg,..)(-,..);
    
    // Integrate back gradient values to get unwrapped signal
    unwrapped = gradient(cum,..) + vector(1,..)(-,..);
    
    return unwrapped;
}

/*******************************************************/

func plotVisUVFreq(dataStruct, width=, type=, marker=, msize=, marks=, visType=, normVis=)
    /* DOCUMENT plotVisUVFreq(dataStruct, width=, type=, marker=, msize=, marks=, visType=, normVis=)

       DESCRIPTION
       From a data structure, plots visibility as a function of spatial frequency

       PARAMETERS
       - dataStruct: input data structure
       - width     : 
       - type      : 
       - marker    : 
       - msize     : 
       - marks     : 
       - visType   : 
       - normVis   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(msize))
        msize=0.2;
    
    if(is_void(visType))
        visType=2;
    
    if(is_void(freqUnit))
        freqUnit="cycles/arcsec";
    
    // plot the data as a function of spatial frequency
    nFiles = numberof(dataStruct);

    for(kFile=1;kFile<=nFiles;kFile++)
    {
        wlen = *dataStruct.EFF_WAVE(kFile);
        
        u =  (*dataStruct.UCOORD(kFile));
        v =  (*dataStruct.VCOORD(kFile));

        if(is_void(u))
            continue;
        
        A = atan(u,v);
        B = sign(A) * abs(u,v);
        
        if(!is_void(projAngle))
            factP = sin(projAngle+A);
        else
            factP = array(1.0,30000);

        if(is_void(color))
        {
            if(anyof((wlen<1.3e-6)&(wlen>0.9e-6)))
                col = "blue";
            else if(anyof((wlen<1.8e-6)&(wlen>1.6e-6)))
                col = "green";
            else if(anyof((wlen<2.4e-6)&(wlen>2.1e-6)))
                col = "red";
            else
                col=[128,128,128];
        }
        else if(color=="obs")
        {
            if(kFile>=5)
                col="cyan";
            else
                col = colors(kFile);
        }
        else
            col=color;

        if(anyof(visType==[1,2]))
        {
            vis2    = *dataStruct.VIS2DATA(kFile);
            if(is_void(vis2))
                continue;
            vis2Err = *dataStruct.VIS2ERR(kFile);
        }
        else if(anyof(visType==3))
        {
            vis2    = *dataStruct.VISAMP(kFile);
            vis2Err = *dataStruct.VISAMPERR(kFile);
        }
        
        if(!is_void(vis2))
        {
            if(dimsof(vis2)(1)==1)
            {
                vis2    = transpose([vis2]);
                vis2Err = transpose([vis2Err]);
            }
            
            nbBases = dimsof(vis2)(3);
            for(kBas=1;kBas<=nbBases;kBas++)
            {
                if(color=="bases")
                {
                    col = colors(kBas);
                }
                cut1 = cutAngle + cutOpen/2;
                cut2 = cutAngle - cutOpen/2;
                // write, A, modulo(A,pi),
                //     cut1, modulo(cut1, pi, cutAngle),
                //     cut2, modulo(cut2, pi, cutAngle);
                if((modulo(A,pi, cutAngle)(kBas)<modulo(cut1, pi, cutAngle))&&
                   (modulo(A,pi, cutAngle)(kBas)>modulo(cut2, pi, cutAngle)))
                {
                    // Compute spatial frequency
                    spFreq = B(kBas) / wlen * factP(kBas) * mas2rad*1000.0;

                    // Choose wether plotting visibility or squared visibility
                    if(visType==1)
                    {
                        V     = yocoMathSignedSqrt(vis2(,kBas,1));
                        V2Err = vis2Err(,kBas,1);
                        VErr  = 0.5* V2Err / (V+(V==0));
                        V2Err = VErr;
                        V2    = V;
                    }
                    else if(anyof(visType==[2,3]))
                    {
                        V2    = vis2(,kBas,1);
                        V2Err = vis2Err(,kBas,1);
                    }

                    if(normVis==1)
                    {
                        if(numberof(V2)>1)
                        {
                            V2Err  = V2Err/median(V2);
                            V2     = V2/median(V2)-1;
                        }
                    }

                    if(avgWlen==1)
                    {
                        // Moyenne bourin
                        if(numberof(V2)>1)
                        {
                            V2     = median(V2);
                            V2Err  = median(V2Err);
                        }
                        if(numberof(spFreq)>1)
                        {
                            spFreq = median(spFreq);
                            wlen   = median(wlen);
                        }
                    }
                    
                    // Model plot: no error bars
                    if(model==1)
                    {
                        if(!is_void(marker)||(numberof(V2)==1))
                        {
                            if(numberof(V2)==1)
                                marker = 6;
                            plmk,V2+factY*spFreq, factX*spFreq + factW*wlen + offW,
                                color=col, marker=marker,msize=msize;
                            plmk,V2-factY*spFreq,-factX*spFreq + factW*wlen + offW,
                                color=col, marker=marker,msize=msize;
                        }
                        else
                        {
                            plg,V2+factY*spFreq, factX*spFreq + factW*wlen + offW,
                                color=col,width=width, type=type,
                                marker=marker, marks=marks;
                            plg,V2-factY*spFreq,-factX*spFreq + factW*wlen + offW,
                                color=col,width=width, type=type,
                                marker=marker, marks=marks;
                        }
                    }
                    // Data plot: error bars
                    else
                    {
                        yocoPlotWithErrBars, V2+factY*spFreq, V2Err,
                            factX*spFreq + factW*wlen + offW,color=col,
                            width=width, type=type,
                            marker=marker, msize=msize, sizebar=sizebar;
                        yocoPlotWithErrBars, V2-factY*spFreq, V2Err,
                            -factX*spFreq + factW*wlen + offW,color=col,
                            width=width, type=type,
                            marker=marker, msize=msize, sizebar=sizebar;
                    }
                }
            }
        }
    }
}

/*******************************************************/

func plotVisSpFreq(dataStruct, projAngle=, cutAngle=, cutOpen=, color=, model=, width=, type=, marker=, msize=, marks=, visType=, factY=, factX=, factW=, offW=, normVis=, sizebar=, avgWlen=, legendHeight=)
    /* DOCUMENT plotVisSpFreq(dataStruct, projAngle=, cutAngle=, cutOpen=, color=, model=, width=, type=, marker=, msize=, marks=, visType=, factY=, factX=, factW=, offW=, normVis=, sizebar=, avgWlen=, legendHeight=)

       DESCRIPTION
       From a data structure, plots visibility as a function of spatial frequency

       PARAMETERS
       - dataStruct  : input data structure
       - projAngle   : 
       - cutAngle    : 
       - cutOpen     : 
       - color       : 
       - model       : 
       - width       : 
       - type        : 
       - marker      : 
       - msize       : 
       - marks       : 
       - visType     : 
       - factY       : 
       - factX       : 
       - factW       : 
       - offW        : 
       - normVis     : 
       - sizebar     : 
       - avgWlen     : 
       - legendHeight: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(legendHeight))
        legendHeight=1.0;
    
    if(is_void(visType))
        visType=2;

    if(is_void(msize))
        msize=0.2;

    // if(is_void(marker))
    //     marker='\4';
    
    if(is_void(factY))
        factY = 0.0;
    
    if(is_void(factX))
        factX = 1.0;
    
    if(is_void(factW))
        factW = 0.0;
    
    if(is_void(offW))
        offW = 0.0;
    
    if(is_void(cutAngle))
        cutAngle=0;
    
    if(is_void(cutOpen))
        cutOpen = pi - 1e-9;
    
    if(is_void(freqUnit))
        freqUnit="cycles/arcsec";
    
    // plot the data as a function of spatial frequency
    nFiles = numberof(dataStruct);

    spFreqMax = 0.0;
    WLEN = [];
    for(kFile=1;kFile<=nFiles;kFile++)
    {
        wlen = *dataStruct.EFF_WAVE(kFile);
        grow,WLEN,wlen;
        u    =  (*dataStruct.UCOORD(kFile));
        v    =  (*dataStruct.VCOORD(kFile));

        if(is_void(u))
            continue;

        A = atan(u,v);
        B = sign(A) * abs(u,v);
        
        if(!is_void(projAngle))
            factP = sin(projAngle+A);
        else
            factP = array(1.0,30000);
        
        vis2 = *dataStruct.VIS2DATA(kFile);
        if(!is_void(vis2))
        {
            if(dimsof(vis2)(1)==1)
                vis2=transpose([vis2]);
            
            nbBases = dimsof(vis2)(3);
            for(kBas=1;kBas<=nbBases;kBas++)
            {
                cut1 = cutAngle + cutOpen/2;
                cut2 = cutAngle - cutOpen/2;
                // write, A, modulo(A,pi),
                //     cut1, modulo(cut1, pi, cutAngle),
                //     cut2, modulo(cut2, pi, cutAngle);
                if((modulo(A,pi, cutAngle)(kBas)<modulo(cut1, pi, cutAngle))&&
                   (modulo(A,pi, cutAngle)(kBas)>modulo(cut2, pi, cutAngle)))
                {                
                    spFreq = abs(B(kBas) / wlen * factP(kBas) * mas2rad*1000.0);
                    if(numberof(spFreq)>1)
                        spFreqDif = min(spFreq(dif));
                    else
                        spFreqDif = spFreq/100;
                    if(max(spFreq)>spFreqMax)
                        spFreqMax = max(spFreq);
                }
            }
        }
    }
    WLEN   = yocoStrRemoveMultiple(WLEN);
    WLEN   = WLEN(sort(WLEN));
    NBWLEN = numberof(WLEN);
    
    if(is_void(sizebar))
        sizebar = spFreqDif/2.;

    for(kFile=1;kFile<=nFiles;kFile++)
    {
        wlen    = *dataStruct.EFF_WAVE(kFile);
        nbWlen  = numberof(wlen);
        theWlen = [];
        for(i=1;i<=nbWlen;i++)
            grow, theWlen, where(wlen(i)==WLEN)(1);
        
        u =  (*dataStruct.UCOORD(kFile));
        v =  (*dataStruct.VCOORD(kFile));

        if(is_void(u))
            continue;
        
        A = atan(u,v);
        B = sign(A) * abs(u,v);
        
        if(!is_void(projAngle))
            factP = sin(projAngle+A);
        else
            factP = array(1.0,30000);

        if(is_void(color))
        {
            // Plot colors as a function of wavelength
            colorBar = span(0.5*spFreqMax,0.7*spFreqMax,NBWLEN);
            xCol     = WLEN*1e6;
            xColUnit = "!mm";
            kol = get_color(array(1.0,NBWLEN),
                            double(NBWLEN-indgen(NBWLEN))/(NBWLEN-1)*4./3.*pi);
        }
        else if(color=="obs")
        {
            colorBar = span(0.5*spFreqMax,0.7*spFreqMax,nFiles);
            xCol = indgen(nFiles);
            xColUnit = "Obs.";
            kol = get_color(array(1.0,nFiles),
                            double(nFiles-indgen(nFiles))/(nFiles-1)*4./3.*pi);
        }
        else if(color=="bases")
        {
            colorBar = span(0.5*spFreqMax,0.7*spFreqMax,nbBases);
            xCol = indgen(nbBases);
            xColUnit = "Base";
            kol = get_color(array(1.0,nbBases),
                            double(nbBases-indgen(nbBases))/(nbBases-1)*4./3.*pi);
        }
        else
        {
            kol=color;
        }
        
        if(anyof(visType==[1,2]))
        {
            vis2    = *dataStruct.VIS2DATA(kFile);
            if(is_void(vis2))
                continue;
            vis2Err = *dataStruct.VIS2ERR(kFile);
        }
        else if(anyof(visType==3))
        {
            vis2    = *dataStruct.VISAMP(kFile);
            vis2Err = *dataStruct.VISAMPERR(kFile);
        }
        
        if(!is_void(vis2))
        {
            if(dimsof(vis2)(1)==1)
            {
                vis2    = transpose([vis2]);
                vis2Err = transpose([vis2Err]);
            }
            
            nbBases = dimsof(vis2)(3);
            for(kBas=1;kBas<=nbBases;kBas++)
            {
                if(color=="bases")
                {
                    col = kol(,kBas);
                }
                else if(color=="obs")
                {
                    col = kol(,kFile);
                }
                else
                    col=kol;
                
                cut1 = cutAngle + cutOpen/2;
                cut2 = cutAngle - cutOpen/2;
                // write, A, modulo(A,pi),
                //     cut1, modulo(cut1, pi, cutAngle),
                //     cut2, modulo(cut2, pi, cutAngle);
                if((modulo(A,pi, cutAngle)(kBas)<modulo(cut1, pi, cutAngle))&&
                   (modulo(A,pi, cutAngle)(kBas)>modulo(cut2, pi, cutAngle)))
                {
                    // Compute spatial frequency
                    spFreq = B(kBas) / wlen * factP(kBas) * mas2rad*1000.0;

                    // Choose wether plotting visibility or squared visibility
                    if(visType==1)
                    {
                        V     = yocoMathSignedSqrt(vis2(,kBas,1));
                        V2Err = vis2Err(,kBas,1);
                        VErr  = 0.5* V2Err / (V+(V==0));
                        V2Err = VErr;
                        V2    = V;
                    }
                    else if(anyof(visType==[2,3]))
                    {
                        V2    = vis2(,kBas,1);
                        V2Err = vis2Err(,kBas,1);
                    }

                    if(normVis==1)
                    {
                        if(numberof(V2)>1)
                        {
                            V2Err  = V2Err/median(V2);
                            V2     = V2/median(V2)-1;
                        }
                    }

                    if(avgWlen==1)
                    {
                        // Moyenne bourin
                        if(numberof(V2)>1)
                        {
                            V2     = median(V2);
                            V2Err  = median(V2Err);
                        }
                        if(numberof(spFreq)>1)
                        {
                            spFreq = median(spFreq);
                            wlen   = median(wlen);
                        }
                    }
                    
                    // Get rid of Nans
                    wr     = where((V2==V2)&(V2Err==V2Err));
                    V2     = V2(wr);
                    V2Err  = V2Err(wr);
                    spFreq = spFreq(wr);
                    wlen   = wlen(wr);
                    
                    if(!is_void(V2))
                    {
                        // Model plot: no error bars
                        if(model==1)
                        {
                            if(!is_void(marker)||(numberof(V2)==1))
                            {
                                if(numberof(V2)==1)
                                    marker = 6;

                                if(is_void(color))
                                {
                                    for(iwlen=1;iwlen<=nbWlen;iwlen++)
                                    {
                                        col = kol(,theWlen(iwlen));
                                
                                        plmk,V2(iwlen)+factY*spFreq(iwlen),
                                            factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                            color=col, marker=marker,msize=msize;
                                        plmk,V2(iwlen)-factY*spFreq(iwlen),
                                            -factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                            color=col, marker=marker,msize=msize;
                                    }
                                }
                                else
                                {
                                    plmk,V2+factY*spFreq,
                                        factX*spFreq + factW*wlen + offW,
                                        color=col, marker=marker,msize=msize;
                                    plmk,V2-factY*spFreq,
                                        -factX*spFreq + factW*wlen + offW,
                                        color=col, marker=marker,msize=msize;
                                }
                            }
                            else
                            {
                                if(is_void(color))
                                {
                                    col = kol(,theWlen(iwlen));
                                    for(iwlen=1;iwlen<=nbWlen;iwlen++)
                                    {
                                        plg,V2(iwlen:)+factY*spFreq(iwlen:),
                                            factX*spFreq(iwlen:) + factW*wlen(iwlen:) + offW,
                                            color=col,width=width, type=type,
                                            marker=marker, marks=marks;
                                    
                                        plg,V2(iwlen:)-factY*spFreq(iwlen:),
                                            -factX*spFreq(iwlen:) + factW*wlen(iwlen:) + offW,
                                            color=col,width=width, type=type,
                                            marker=marker, marks=marks;
                                    }
                                }
                                else
                                {                                
                                    plg,V2+factY*spFreq,
                                        factX*spFreq + factW*wlen + offW,
                                        color=col,width=width, type=type,
                                        marker=marker, marks=marks;
                                
                                    plg,V2-factY*spFreq,
                                        -factX*spFreq + factW*wlen + offW,
                                        color=col,width=width, type=type,
                                        marker=marker, marks=marks;
                                }
                            }
                        }
                        // Data plot: error bars
                        else
                        {
                            for(iwlen=1;iwlen<=nbWlen;iwlen++)
                            {
                                if(is_void(color))
                                {
                                    col = kol(,theWlen(iwlen));
                                }

                                yocoPlotWithErrBars, V2(iwlen)+factY*spFreq(iwlen),
                                    V2Err(iwlen),
                                    factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                    color=col,
                                    width=width, type=type,
                                    marker=marker, msize=msize, sizebar=sizebar;
                                yocoPlotWithErrBars, V2(iwlen)-factY*spFreq(iwlen),
                                    V2Err(iwlen),
                                    -factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                    color=col,
                                    width=width, type=type,
                                    marker=marker, msize=msize, sizebar=sizebar;
                            
                            }
                        }
                    }
                }
            }
        }
    }

    if(!model)
    {
        for(k=1;k<=numberof(colorBar);k++)
        {
            col = kol(,k);
            plmk,legendHeight,colorBar(k),color=col,marker=4,msize=0.3,width=10;
            plt,swrite(xCol(1),xColUnit,format="%2.2f %s"),min(colorBar),legendHeight-0.02,tosys=1,justify="CT",height=12;
            plt,swrite(xCol(0),xColUnit,format="%2.2f %s"),max(colorBar),legendHeight-0.02,tosys=1,justify="CT",height=12;
        }
        // png,HOME+"toto.png"
        // opriut()
    }
}

/*******************************************************/

func plotClosSpFreq(dataStruct, projAngle=, cutAngle=, cutOpen=, color=, model=, width=, type=, marker=, msize=, marks=, factY=, factX=, factW=, offW=, wrap=, phaseDeg=, avgWlen=, sizebar=)
    /* DOCUMENT plotClosSpFreq(dataStruct, projAngle=, cutAngle=, cutOpen=, color=, model=, width=, type=, marker=, msize=, marks=, factY=, factX=, factW=, offW=, wrap=, phaseDeg=, avgWlen=, sizebar=)

       DESCRIPTION

       PARAMETERS
       - dataStruct: 
       - projAngle : 
       - cutAngle  : 
       - cutOpen   : 
       - color     : 
       - model     : 
       - width     : 
       - type      : 
       - marker    : 
       - msize     : 
       - marks     : 
       - factY     : 
       - factX     : 
       - factW     : 
       - offW      : 
       - wrap      : 
       - phaseDeg  : 
       - avgWlen   : 
       - sizebar   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    
    if(is_void(msize))
        msize=0.2;
    
    if(is_void(factY))
        factY = 0.0;
    
    if(is_void(factX))
        factX = 1.0;
    
    if(is_void(factW))
        factW = 0.0;
    
    if(is_void(offW))
        offW = 0.0;
    
    if(is_void(cutAngle))
        cutAngle=0;
    
    if(is_void(cutOpen))
        cutOpen=pi - 1e-9;
    
    if(is_void(freqUnit))
        freqUnit="cycles/arcsec";
    
    // plot the data as a function of spatial frequency
    nFiles = numberof(dataStruct); 

    spFreqMax = 0.0;
    WLEN = [];
    for(kFile=1;kFile<=nFiles;kFile++)
    {
        wlen = *dataStruct.EFF_WAVE(kFile);
        grow,WLEN,wlen;
        u1 =  (*dataStruct.U1COORD(kFile));

        // If u1 is void then abort and go to next file
        if(is_void(u1))
            continue;

        v1 =  (*dataStruct.V1COORD(kFile));
        u2 =  (*dataStruct.U2COORD(kFile));
        v2 =  (*dataStruct.V2COORD(kFile));
        
        u3 = u1 + u2;
        v3 = v1 + v2;
        u = grow([u1],[u2],[u3]);
        v = grow([v1],[v2],[v3]);
        
        A  = atan(u,v);
        B = sign(A) * abs(u,v);
        
        if(!is_void(projAngle))
            factP = sin(projAngle+A);
        else
            factP = array(1.0,1000,1000);

        clos = (*dataStruct.T3PHI(kFile));
        if(!is_void(clos))
        {
            cut1 = cutAngle + cutOpen/2;
            cut2 = cutAngle - cutOpen/2;
            // write, A, modulo(A,pi),
            //     cut1, modulo(cut1, pi, cutAngle),
            //     cut2, modulo(cut2, pi, cutAngle);
            if(anyof((modulo(A,pi, cutAngle)(kBas)<modulo(cut1, pi, cutAngle))&
                     (modulo(A,pi, cutAngle)(kBas)>modulo(cut2, pi, cutAngle))))
            {
                if(!model)
                    closErr = *dataStruct.T3PHIERR(kFile); 
                if(phaseDeg==1)
                {
                    clos = clos*180/pi;
                    if(!model)
                        closErr = closErr*180/pi;
                }
            
                // Maximum baseline is the closure phase baseline
                xb = abs(B)(mxx);
                Bb = B(xb);
                Ab = factP(xb);
            
                // Other case: geometric average of the three beselines
                // is the closure phase baseline
                // Bb = (B(1)*B(2)*B(3))^(1.0/3.0);
                // Bb = avg(B);
                // Ab = avg(factP);
            
                spFreq = Bb / wlen * Ab * mas2rad*1000.0;
                if(numberof(spFreq)>1)
                    spFreqDif = min(spFreq(dif));
                else
                    spFreqDif = spFreq/100;
                
                if(max(spFreq)>spFreqMax)
                    spFreqMax = max(spFreq);
            }
        }
    }
    WLEN = yocoStrRemoveMultiple(WLEN);
    WLEN = WLEN(sort(WLEN));
    NBWLEN = numberof(WLEN);
    
    if(is_void(sizebar)) 
        sizebar = spFreqDif/2.;
    
    for(kFile=1;kFile<=nFiles;kFile++)
    {
        wlen = *dataStruct.EFF_WAVE(kFile);
        nbWlen = numberof(wlen);
        theWlen = [];
        for(i=1;i<=nbWlen;i++)
            grow, theWlen, where(wlen(i)==WLEN)(1);
        
        u1 =  (*dataStruct.U1COORD(kFile));
        if(is_void(u1))
            continue;
        v1 =  (*dataStruct.V1COORD(kFile));
        u2 =  (*dataStruct.U2COORD(kFile));
        v2 =  (*dataStruct.V2COORD(kFile));
        
        u3 = u1 + u2;
        v3 = v1 + v2;
        u = grow([u1],[u2],[u3]);
        v = grow([v1],[v2],[v3]);
        
        A = atan(u,v);
        B = sign(A) * abs(u,v);
        
        if(!is_void(projAngle))
            factP = sin(projAngle+A);
        else
            factP = array(1.0,1000,1000);

        if(is_void(color))
        {
            // Plot colors as a function of wavelength
            colorBar = span(0.7*spFreqMax,0.9*spFreqMax,NBWLEN);
            xCol     = WLEN*1e6;
            kol      = get_color(array(1.0,NBWLEN),
                                 double(NBWLEN-indgen(NBWLEN))/(NBWLEN-1)*4./3.*pi);
        }
        else if(color=="obs")
        {
            colorBar = span(0.7*spFreqMax,0.9*spFreqMax,nFiles);
            xCol = indgen(nFiles);
            kol = get_color(array(1.0,nFiles),
                            double(nFiles-indgen(nFiles))/(nFiles-1)*4./3.*pi);
        }
        else if(color=="bases")
        {
            colorBar = span(0.7*spFreqMax,0.9*spFreqMax,nbBases);
            xCol = indgen(nbBases);
            kol = get_color(array(1.0,nbBases),
                            double(nbBases-indgen(nbBases))/(nbBases-1)*4./3.*pi);
        }
        else
            kol=color;
        
        clos    = (*dataStruct.T3PHI(kFile));
        if(!model)
            closErr = *dataStruct.T3PHIERR(kFile);
        
        if(!is_void(clos))
        {
            if(dimsof(clos)(1)==1)
            {
                clos    = transpose([clos]);
                if(!model)
                    closErr = transpose([closErr]);
            }
            
            nbClos = dimsof(clos)(3);
            for(kClos=1;kClos<=nbClos;kClos++)
            {
                if(color=="bases")
                {
                    col = kol(,kBas);
                }
                else if(color=="obs")
                {
                    col = kol(,kFile);
                }
                
                cut1 = cutAngle + cutOpen/2;
                cut2 = cutAngle - cutOpen/2;
                // write, A, modulo(A,pi),
                //     cut1, modulo(cut1, pi, cutAngle),
                //     cut2, modulo(cut2, pi, cutAngle);
                if(anyof((modulo(A,pi, cutAngle)(kClos)<modulo(cut1, pi, cutAngle))&
                         (modulo(A,pi, cutAngle)(kClos)>modulo(cut2, pi, cutAngle))))
                {
                    // Compute spatial frequency
                    
                    // Maximum baseline is the closure phase baseline
                    xb = abs(B)(kClos,mxx);
                    Bb = B(kClos,xb);
                    Ab = factP(kClos,xb);

                    // Other case: geometric average of the three beselines
                    // is the closure phase baseline
                    // Bb = (B(1)*B(2)*B(3))^(1.0/3.0);
                    // Bb = avg(B);
                    // Ab = avg(factP);
                    
                    spFreq = Bb / wlen * Ab * mas2rad*1000.0;

                    // Unwrap phase
                    if(wrap)
                    {
                        CLOS    = unwrap(clos(,kClos,1));
                        if(anyof(CLOS<=0))
                            CLOS=CLOS+2*pi;
                    }
                    else
                        CLOS    = clos(,kClos,1);

                    // Convert angle into degrees (default is radians)
                    if(phaseDeg==1)
                    {
                        CLOS    = clos*180/pi;
                        if(!model)
                            closErr = closErr*180/pi;
                    }
                    
                    if(!model)
                        CErr = (closErr)(,kClos,1);
                    
                    if(avgWlen==1)
                    {
                        // Moyenne bourin
                        if(numberof(CLOS)>1)
                        {
                            CLOS   = median(CLOS);
                            if(!model)
                                CErr   = median(CErr);
                            spFreq = median(spFreq);
                            wlen   = median(wlen);
                        }
                    }
                    
                    // Get rid of Nans
                    wr     = where((CLOS==CLOS));
                    CLOS   = CLOS(wr);
                    if(!model)
                        CErr   = CErr(wr);
                    spFreq = spFreq(wr);
                    wlen   = wlen(wr);
                    
                    if(model==1)
                    {
                        if(!is_void(marker)||(numberof(CLOS)==1))
                        {
                            if(numberof(CLOS)==1)
                                marker = 6;
                            
                            if(is_void(color))
                            {
                                for(iwlen=1;iwlen<=nbWlen;iwlen++)
                                {
                                    col = kol(,theWlen(iwlen));
                                    
                                    plmk,CLOS(iwlen)+factY*spFreq(iwlen),
                                        factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                        color=col, marker=marker,msize=msize;
                                    plmk,CLOS(iwlen)-factY*spFreq(iwlen),
                                        -factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                        color=col, marker=marker,msize=msize;
                                }
                            }
                            else
                            {
                                col = color;
                                plmk,CLOS+factY*spFreq,
                                    factX*spFreq + factW*wlen + offW,
                                    color=col, marker=marker,msize=msize;
                                plmk,CLOS-factY*spFreq,
                                    -factX*spFreq + factW*wlen + offW,
                                    color=col, marker=marker,msize=msize;
                            }
                        }
                        else
                        {
                            if(is_void(color))
                            {
                                for(iwlen=1;iwlen<=nbWlen;iwlen++)
                                {
                                    col = kol(,theWlen(iwlen));
                                
                                    plg, CLOS(iwlen)+factY*spFreq(iwlen),
                                        factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                        color=col,width=width, type=type,
                                        marker=marker, marks=marks;
                            
                                    plg, CLOS(iwlen)-factY*spFreq(iwlen),
                                        -factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                        color=col,width=width, type=type,
                                        marker=marker, marks=marks;
                                }
                            }
                            else
                            { 
                                col = color;
                                                          
                                plg,CLOS+factY*spFreq,
                                    factX*spFreq + factW*wlen + offW,
                                    color=col,width=width, type=type,
                                    marker=marker, marks=marks;
                                
                                plg,CLOS-factY*spFreq,
                                    -factX*spFreq + factW*wlen + offW,
                                    color=col,width=width, type=type,
                                    marker=marker, marks=marks;
                            }
                        }
                    
		    }
                    // Data plot: error bars
                    else
		    {
                        for(iwlen=1;iwlen<=nbWlen;iwlen++)
			{
                            if(is_void(color))
                                col = kol(,theWlen(iwlen));
                            else
                                col = color;
                            
                            yocoPlotWithErrBars, CLOS(iwlen)+factY*spFreq(iwlen),
                                CErr(iwlen),
                                factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                color=col,
                                width=width, type=type, marker=marker,msize=msize,
                                sizebar=sizebar;
                            
                            yocoPlotWithErrBars, CLOS(iwlen)-factY*spFreq(iwlen),
                                CErr(iwlen),
                                -factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                color=col,
                                width=width, type=type,
                                marker=marker,msize=msize,sizebar=sizebar;
			}
		    }
		}
	    }
	}
    }
}

/*******************************************************/

func plotPhaseSpFreq(dataStruct, projAngle=, cutAngle=, cutOpen=, color=, model=, width=, type=, marker=, msize=, marks=, factY=, factX=, factW=, offW=, sizebar=, avgWlen=, legendHeight=, wrap=, phaseDeg=, freqUnit=)
    /* DOCUMENT plotPhaseSpFreq(dataStruct, projAngle=, cutAngle=, cutOpen=, color=, model=, width=, type=, marker=, msize=, marks=, factY=, factX=, factW=, offW=, sizebar=, avgWlen=, legendHeight=, wrap=, phaseDeg=, freqUnit=)

       DESCRIPTION
       From a data structure, plots differential phases as a function of spatial frequency

       PARAMETERS
       - dataStruct  : input data structure
       - projAngle   : 
       - cutAngle    : 
       - cutOpen     : 
       - color       : 
       - model       : 
       - width       : 
       - type        : 
       - marker      : 
       - msize       : 
       - marks       : 
       - factY       : 
       - factX       : 
       - factW       : 
       - offW        : 
       - sizebar     : 
       - avgWlen     : 
       - legendHeight: 
       - wrap        : 
       - phaseDeg    : 
       - freqUnit    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(legendHeight))
        legendHeight=1.0;

    if(is_void(msize))
        msize=0.2;
    
    if(is_void(cutAngle))
        cutAngle=0;
    
    if(is_void(cutOpen))
        cutOpen = pi - 1e-9;
    
    if(is_void(factY))
        factY = 0.0;
    
    if(is_void(factX))
        factX = 1.0;
    
    if(is_void(factW))
        factW = 0.0;
    
    if(is_void(offW))
        offW = 0.0;
    
    if(is_void(freqUnit))
        freqUnit="cycles/arcsec";
    
    // plot the data as a function of spatial frequency
    nFiles = numberof(dataStruct);

    spFreqMax = 0.0;
    WLEN = [];
    for(kFile=1;kFile<=nFiles;kFile++)
    {
        wlen = *dataStruct.EFF_WAVE(kFile);
        grow,WLEN,wlen;
        u    =  (*dataStruct.UCOORD(kFile));
        v    =  (*dataStruct.VCOORD(kFile));

        if(is_void(u))
            continue;

        A = atan(u,v);
        B = sign(A) * abs(u,v);
        
        if(!is_void(projAngle))
            factP = sin(projAngle+A);
        else
            factP = array(1.0,30000);
        
        dPhi = *dataStruct.VISPHI(kFile);
        if(!is_void(dPhi))
        {
            if(dimsof(dPhi)(1)==1)
                dPhi=transpose([dPhi]);
            
            nbBases = dimsof(dPhi)(3);
            for(kBas=1;kBas<=nbBases;kBas++)
            {
                cut1 = cutAngle + cutOpen/2;
                cut2 = cutAngle - cutOpen/2;
                // write, A, modulo(A,pi),
                //     cut1, modulo(cut1, pi, cutAngle),
                //     cut2, modulo(cut2, pi, cutAngle);
                if((modulo(A,pi, cutAngle)(kBas) < modulo(cut1, pi, cutAngle))&&
                   (modulo(A,pi, cutAngle)(kBas) > modulo(cut2, pi, cutAngle)))
                {                
                    spFreq = abs(B(kBas) / wlen * factP(kBas) * mas2rad*1000.0);
                    spFreqDif = min(spFreq(dif));
                    if(max(spFreq)>spFreqMax)
                        spFreqMax = max(spFreq);
                }
            }
        }
    }
    WLEN   = yocoStrRemoveMultiple(WLEN);
    WLEN   = WLEN(sort(WLEN));
    NBWLEN = numberof(WLEN);
    
    if(is_void(sizebar))
        sizebar = spFreqDif/2.;

    for(kFile=1;kFile<=nFiles;kFile++)
    {
        wlen = *dataStruct.EFF_WAVE(kFile);
        nbWlen = numberof(wlen);
        theWlen = [];
        for(i=1;i<=nbWlen;i++)
            grow, theWlen, where(wlen(i)==WLEN)(1);
        
        u =  (*dataStruct.UCOORD(kFile));
        v =  (*dataStruct.VCOORD(kFile));

        if(is_void(u))
            continue;
        
        A = atan(u,v);
        B = sign(A) * abs(u,v);
        
        if(!is_void(projAngle))
            factP = sin(projAngle+A);
        else
            factP = array(1.0,30000);

        if(is_void(color))
        {
            // Plot colors as a function of wavelength
            colorBar = span(0.7*spFreqMax,0.9*spFreqMax,NBWLEN);
            xCol     = WLEN*1e6;
            kol = get_color(array(1.0,NBWLEN),
                            double(NBWLEN-indgen(NBWLEN))/(NBWLEN-1)*4./3.*pi);
        }
        else if(color=="obs")
        {
            colorBar = span(0.7*spFreqMax,0.9*spFreqMax,nFiles);
            xCol = indgen(nFiles);
            kol = get_color(array(1.0,nFiles),
                            double(nFiles-indgen(nFiles))/(nFiles-1)*4./3.*pi);
        }
        else if(color=="bases")
        {
            colorBar = span(0.7*spFreqMax,0.9*spFreqMax,nbBases);
            xCol = indgen(nbBases);
            kol = get_color(array(1.0,nbBases),
                            double(nbBases-indgen(nbBases))/(nbBases-1)*4./3.*pi);
        }
        else
        {
            kol=color;
        }
        
        
        dPhi    = *dataStruct.VISPHI(kFile);
        if(is_void(dPhi))
            continue;
        dPhiErr = *dataStruct.VISPHIERR(kFile);
        
        if(!is_void(dPhi))
        {
            if(dimsof(dPhi)(1)==1)
            {
                dPhi    = transpose([dPhi]);
                dPhiErr = transpose([dPhiErr]);
            }
            
            nbBases = dimsof(dPhi)(3);
            for(kBas=1;kBas<=nbBases;kBas++)
            {
                if(color=="bases")
                {
                    col = kol(,kBas);
                }
                else if(color=="obs")
                {
                    col = kol(,kFile);
                }
                else
                    col=kol;
                
                cut1 = cutAngle + cutOpen/2;
                cut2 = cutAngle - cutOpen/2;
                // write, A, modulo(A,pi),
                //     cut1, modulo(cut1, pi, cutAngle),
                //     cut2, modulo(cut2, pi, cutAngle);
                if((modulo(A,pi, cutAngle)(kBas)<modulo(cut1, pi, cutAngle))&&
                   (modulo(A,pi, cutAngle)(kBas)>modulo(cut2, pi, cutAngle)))
                {
                    // Compute spatial frequency
                    spFreq = B(kBas) / wlen * factP(kBas) * mas2rad*1000.0;


                    DP    = dPhi(,kBas,1);
                    DPErr = dPhiErr(,kBas,1);

                    // Convert angle into degrees (default is radians)
                    if(phaseDeg==1)
                    {
                        DP    = DP*180/pi;
                        if(!model)
                            DPErr = DPErr*180/pi;
                    }

                    if(avgWlen==1)
                    {
                        // Moyenne bourin
                        if(numberof(DP)>1)
                        {
                            DP     = median(DP);
                            DPErr  = median(DPErr);
                        }
                        if(numberof(spFreq)>1)
                        {
                            spFreq = median(spFreq);
                            wlen   = median(wlen);
                        }
                    }
                    
                    // Get rid of Nans
                    wr     = where((DP==DP)&(DPErr==DPErr));
                    DP     = DP(wr);
                    DPErr  = DPErr(wr);
                    spFreq = spFreq(wr);
                    wlen   = wlen(wr);
                    
                    // Model plot: no error bars
                    if(model==1)
                    {
                        if(!is_void(marker)||(numberof(DP)==1))
                        {
                            if(numberof(DP)==1)
                                marker = 6;

                            if(is_void(color))
                            {
                                for(iwlen=1;iwlen<=nbWlen;iwlen++)
                                {
                                    col = kol(,theWlen(iwlen));
                                
                                    plmk,DP(iwlen)+factY*spFreq(iwlen),
                                        factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                        color=col, marker=marker,msize=msize;
                                    plmk,DP(iwlen)-factY*spFreq(iwlen),
                                        -factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                        color=col, marker=marker,msize=msize;
                                }
                            }
                            else
                            {
                                plmk,DP+factY*spFreq,
                                    factX*spFreq + factW*wlen + offW,
                                    color=col, marker=marker,msize=msize;
                                plmk,DP-factY*spFreq,
                                    -factX*spFreq + factW*wlen + offW,
                                    color=col, marker=marker,msize=msize;
                            }
                        }
                        else
                        {
                            if(is_void(color))
                            {
                                col = kol(,theWlen(iwlen));
                                for(iwlen=1;iwlen<=nbWlen;iwlen++)
                                {
                                    plg,DP(iwlen:)+factY*spFreq(iwlen:),
                                        factX*spFreq(iwlen:) + factW*wlen(iwlen:) + offW,
                                        color=col,width=width, type=type,
                                        marker=marker, marks=marks;
                                    
                                    plg,DP(iwlen:)-factY*spFreq(iwlen:),
                                        -factX*spFreq(iwlen:) + factW*wlen(iwlen:) + offW,
                                        color=col,width=width, type=type,
                                        marker=marker, marks=marks;
                                }
                            }
                            else
                            {                                
                                plg,DP+factY*spFreq,
                                    factX*spFreq + factW*wlen + offW,
                                    color=col,width=width, type=type,
                                    marker=marker, marks=marks;
                                
                                plg,DP-factY*spFreq,
                                    -factX*spFreq + factW*wlen + offW,
                                    color=col,width=width, type=type,
                                    marker=marker, marks=marks;
                            }
                        }
                    }
                    // Data plot: error bars
                    else
                    {
                        for(iwlen=1;iwlen<=nbWlen;iwlen++)
                        {
                            if(is_void(color))
                            {
                                col = kol(,theWlen(iwlen));
                            }
                                
                            yocoPlotWithErrBars, DP(iwlen)+factY*spFreq(iwlen),
                                DPErr(iwlen),
                                factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                color=col,
                                width=width, type=type,
                                marker=marker, msize=msize, sizebar=sizebar;
                            yocoPlotWithErrBars, DP(iwlen)-factY*spFreq(iwlen),
                                DPErr(iwlen),
                                -factX*spFreq(iwlen) + factW*wlen(iwlen) + offW,
                                color=col,
                                width=width, type=type,
                                marker=marker, msize=msize, sizebar=sizebar;
                        }
                    }
                }
            }
        }
    }

    if(!model)
        for(k=1;k<=numberof(colorBar);k++)
        {
            col = kol(,k);
            plg,legendHeight,colorBar(k),color=col,type="none",marker=marker,msize=msize;
            plt,pr1(xCol(1)),min(colorBar),legendHeight-0.02,tosys=1,justify="CT",height=8;
            plt,pr1(xCol(0)),max(colorBar),legendHeight-0.02,tosys=1,justify="CT",height=8;

        }
}


/*******************************************************/

func plotAllDataAbsAndDif(data, model=, width=, color=, clear=, factX=, factY=, factW=, maxSpFreq=, minWlen=, maxWlen=, sizebar=, type=, cutOpen=, cutAngle=)
    /* DOCUMENT plotAllDataAbsAndDif(data, model=, width=, color=, clear=, factX=, factY=, factW=, maxSpFreq=, minWlen=, maxWlen=, sizebar=, type=, cutOpen=, cutAngle=)

       DESCRIPTION
       This is a mix of plotDataSpFreq & plotDataWlen. See Millour et al. 2011
       for an example of use

       PARAMETERS
       - data     : 
       - model    : 
       - width    : 
       - color    : 
       - clear    : 
       - factX    : 
       - factY    : 
       - factW    : 
       - maxSpFreq: 
       - minWlen  : 
       - maxWlen  : 
       - sizebar  : 
       - type     : 
       - cutOpen  : 
       - cutAngle : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    
    localType = type;
    
    if(is_void(model))
        model=0;
    // Get wavelength range of dataset
    getWlenRange, data, minWlen0, maxWlen0;
    
    nObs = numberof(data);
    allWlen = allBase = [];
    for(k=1;k<=nObs;k++)
    {
        grow,allBase,abs((*data.UCOORD(k)),(*data.VCOORD(k)));
        grow,allBase,abs((*data.U1COORD(k)),(*data.V1COORD(k)));
        grow,allBase,abs((*data.U2COORD(k)),(*data.V2COORD(k)));
        grow,allBase,abs((*data.U1COORD(k))+(*data.U2COORD(k)),
                         (*data.V1COORD(k))+(*data.V2COORD(k)));
        grow,allWlen,*data.EFF_WAVE(k);
    }
    maxBase = max(allBase);
    
    allWlen = yocoStrRemoveMultiple(allWlen);
    
    if(is_void(localType))
        localType = "solid";
    
    if(is_void(minWlen))
        minWlen = min(allWlen);
    if(is_void(maxWlen))
        maxWlen = max(allWlen);

    BW = (maxWlen-minWlen);
    avgWlen = (maxWlen+minWlen)/2.0;
    R = avgWlen/BW;

    if(is_void(factW))
        factW = 1e6;
    if(is_void(msize))
        msize=0.2;
    if(is_void(factX))
        factX  = avgWlen/maxBase*factW/1000;
    if(is_void(factY))
        factY  = 10*BW*factW;
    if(is_void(maxSpFreq))
        maxSpFreq = maxBase/minWlen*mas2rad*1000;

    yocoNmCreate,1,2,2,dy=0.1,dx=0.1,wait=1,style="boxed.gs",square=0,landscape=1,width=900,height=750,fx=1,
        V = [0.15,0.9,0.12,0.73];
    
    //    fsdfa()
    if(clear==1)
        fma;

    plsys,1;
    plotVisSpFreq, data, avgWlen=0, model=model,
        width=width, color=color, msize=msize, sizebar=maxSpFreq/200;
    
    limits,0,maxSpFreq+15,0,1;
    xytitles,"Spatial Frequency (cycles/as.)","Squared visibility",[0.02,0.02];

    plsys,2;
    plotClosSpFreq, data, avgWlen=0, phaseDeg=0, model=model,
        width=width, color=color, msize=msize, sizebar=maxSpFreq/200;
    
    limits,0,maxSpFreq+15,-180*pi/180,180*pi/180;
    xytitles,"Sp. Freq. longest baseline (cycles/as.)","Closure phase (rad)",[0.02,0.02];

    yocoNmCreate,20,3,1,dy=0.0,dx=-0.1,wait=1,style="nobox.gs",square=1,
        landscape=1,width=900,height=750,fx=1,
        V = [0.15,0.9,0.12,0.73],landscape=1;
    
    if(clear==1)
        fma;
    
    plsys,1;
    
    vp = viewport();
    xyfac = (vp(2)-vp(1)) / (vp(4)-vp(3));
    
    left      = minWlen*factW;
    leftLim   = left - BW/7.5*factW;
    rightWlen = maxWlen*factW;
    right     = maxWlen*factW+factX*maxSpFreq;
    rightLim  = right + 2*BW/6*factW;
    bottom    = 0;
    bottomLim = bottom - 0.004*maxSpFreq;
    top       = factY*maxSpFreq;
    topLim    = top + 0.001*maxSpFreq;

    plotVisSpFreq, data, model=1,
        factX=factX, type=localType,
        factY=factY,
        factW=factW,
        normVis=1, width=width, color=color, cutOpen=cutOpen, cutAngle=cutAngle;

    if(model==0)
        plotVisSpFreq, data, avgWlen=1, model=model,
            factX=factX, factY=factY, factW=factW, offW=-(right-left)/3.6,
            width=1, sizebar=(right-left)/200,
            normVis=1, color=color, msize=msize, cutOpen=cutOpen, cutAngle=cutAngle;
    
    facX = (rightLim - leftLim) / (topLim - bottomLim) / xyfac;
    
    yocoPlotArrow,left,bottom,left,1,facX=facX, arlng=0.4,width=1;
    plt,"!dV^2",(minWlen+0.05*BW)*factW,0.8,tosys=count,justify="LH";
    plt,"0",(minWlen-0.05*BW)*factW,0.,tosys=count,justify="RH";
    plt,"1",(minWlen-0.05*BW)*factW,1,tosys=count,justify="RH";
        
    yocoPlotArrow,left,bottom,rightWlen,bottom,facX=facX, arlng=0.4,width=1;
    plt,"!l (!mm)",avgWlen*factW,.2,tosys=count;
    plt,swrite(minWlen*factW, format="%1.3f"),(minWlen)*factW,-.2,tosys=count,justify="CT";
    plt,swrite(maxWlen*factW, format="%1.3f"),(maxWlen)*factW,-.2,tosys=count,justify="CT";
    
    yocoPlotArrow,rightWlen+BW/10*factW,0,right+BW/10*factW,top, facX=facX, arlng=0.4,width=1;
    plt,"Sp. Freq.\n(cyc./as.)",rightWlen+BW/7*factW+(right-rightWlen)/2,top/2,tosys=count, justify="LH";
    plt,"0",rightWlen+BW/7*factW,0., tosys=count, justify="LA";
    plt,swrite(maxSpFreq, format="%1.0f"),right+BW/7*factW,top, tosys=count, justify="LT";
    
    pltitle,"Differential visibility - 1";

    limits,leftLim, rightLim, bottomLim, topLim;

    



    plsys,2;
    plotClosSpFreq, data, model=1, phaseDeg=0,
        factY=factY,
        factX=factX,
        factW=factW, width=width, color=color, type=localType, cutOpen=cutOpen, cutAngle=cutAngle;

    if(model==0)
        plotClosSpFreq, data, avgWlen=1, model=model, phaseDeg=0,
            factX=factX, factY=factY, factW=factW, offW=-(right-left)/3.6,
            width=1, sizebar=(right-left)/200, color=color, msize=msize, cutOpen=cutOpen, cutAngle=cutAngle;
    
    
    yocoPlotArrow,left,bottom,left,1,facX=facX, arlng=0.4,width=1;
    plt,"!y (rad)",(minWlen+0.05*BW)*factW,0.8,tosys=count,justify="LH";
    plt,"0",(minWlen-0.05*BW)*factW,0.,tosys=count,justify="RH";
    plt,"1",(minWlen-0.05*BW)*factW,1,tosys=count,justify="RH";
        
    yocoPlotArrow,left,bottom,rightWlen,bottom,facX=facX, arlng=0.4,width=1;
    plt,"!l (!mm)",avgWlen*factW,.2,tosys=count;
    plt,swrite(minWlen*factW, format="%1.3f"),(minWlen)*factW,-.2,tosys=count,justify="CT";
    plt,swrite(maxWlen*factW, format="%1.3f"),(maxWlen)*factW,-.2,tosys=count,justify="CT";
    
    yocoPlotArrow,rightWlen+BW/10*factW,0,right+BW/10*factW,top, facX=facX, arlng=0.4,width=1;
    plt,"Sp. Freq.\n(cyc./as.)",rightWlen+BW/7*factW+(right-rightWlen)/2,top/2,tosys=count, justify="LH";
    plt,"0",rightWlen+BW/7*factW,0., tosys=count, justify="LA";
    plt,swrite(maxSpFreq, format="%1.0f"),right+BW/7*factW,top, tosys=count, justify="LT";
    
    pltitle,"Closure phase";

    limits,leftLim, rightLim, bottomLim, topLim;


    plsys,3;
    plotPhaseSpFreq, data, model=1, phaseDeg=0,
        factY=factY,
        factX=factX,
        factW=factW, width=width, color=color, type=localType, cutOpen=cutOpen, cutAngle=cutAngle;

    if(model==0)
        plotPhaseSpFreq, data, avgWlen=1, model=model, phaseDeg=0,
            factX=factX, factY=factY, factW=factW, offW=-(right-left)/3.6,
            width=1, sizebar=(right-left)/200, color=color, msize=msize, cutOpen=cutOpen, cutAngle=cutAngle;

    yocoPlotArrow,left,bottom,left,1,facX=facX, arlng=0.4,width=2;
    plt,"!d!f (rad)",(minWlen+0.05*BW)*factW,0.8,tosys=count,justify="LH";
    plt,"0",(minWlen-0.05*BW)*factW,0.,tosys=count,justify="RH";
    plt,"1",(minWlen-0.05*BW)*factW,1,tosys=count,justify="RH";
        
    yocoPlotArrow,left,bottom,rightWlen,bottom,facX=facX, arlng=0.4,width=1;
    plt,"!l (!mm)",avgWlen*factW,.2,tosys=count;
    plt,swrite(minWlen*factW, format="%1.3f"),(minWlen)*factW,-.2,tosys=count,justify="CT";
    plt,swrite(maxWlen*factW, format="%1.3f"),(maxWlen)*factW,-.2,tosys=count,justify="CT";
    
    yocoPlotArrow,rightWlen+BW/10*factW,0,right+BW/10*factW,top, facX=facX, arlng=0.4,width=1;
    plt,"Sp. Freq.\n(cyc./as.)",rightWlen+BW/7*factW+(right-rightWlen)/2,top/2,tosys=count, justify="LH";
    plt,"0",rightWlen+BW/7*factW,0., tosys=count, justify="LA";
    plt,swrite(maxSpFreq, format="%1.0f"),right+BW/7*factW,top, tosys=count, justify="LT";
    
    pltitle,"Differential phase";
    
    limits,leftLim, rightLim, bottomLim, topLim;
}

/*******************************************************/

func plotDataSpFreq(dataStruct, projAngle=, cutAngle=, cutOpen=, color=, model=, width=, plotVis=, plotClos=, plotDPhi=, plotDVis=, plotSpec=, type=, marker=, msize=, kill=, win=, clear=, wrap=, phaseDeg=, noDiff=, sizebar=, avgWlen=, limVis=, limClos=, limDPhi=, limDVis=, limFreq=, logVis=, resid=, plotLegends=, legendHeight=)
    /* DOCUMENT plotDataSpFreq(dataStruct, projAngle=, cutAngle=, cutOpen=, color=, model=, width=, plotVis=, plotClos=, plotDPhi=, plotDVis=, plotSpec=, type=, marker=, msize=, kill=, win=, clear=, wrap=, phaseDeg=, noDiff=, sizebar=, avgWlen=, limVis=, limClos=, limDPhi=, limDVis=, limFreq=, logVis=, resid=, plotLegends=, legendHeight=)

       DESCRIPTION
       Plots data as a function of spatial frequencies

       PARAMETERS
       - dataStruct  : The data structure
       - projAngle   : 
       - cutAngle    : 
       - cutOpen     : 
       - color       : 
       - model       : 
       - width       : 
       - plotVis     : 
       - plotClos    : 
       - plotDPhi    : 
       - plotDVis    : 
       - plotSpec    : 
       - type        : 
       - marker      : 
       - msize       : 
       - kill        : 
       - win         : 
       - clear       : 
       - wrap        : 
       - phaseDeg    : 
       - noDiff      : 
       - sizebar     : 
       - avgWlen     : 
       - limVis      : 
       - limClos     : 
       - limDPhi     : 
       - limDVis     : 
       - limFreq     : 
       - logVis      : 
       - resid       : 
       - plotLegends : 
       - legendHeight: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(msize))
        msize=0.2;
    
    if(is_void(noDiff))
        diffPlt = "Diff. ";
    else
        diffPlt = "";
        
    if(is_void(plotVis))
        plotVis = 1;
    
    if(is_void(plotClos))
        plotClos = 1;

    if(is_void(plotDPhi))
        plotDPhi = 0;

    if(is_void(plotDVis))
        plotDVis = 0;

    if(is_void(plotSpec))
        plotSpec = 0;

    if(is_void(win))
        win=1;

    if(is_void(kill))
        kill=1;
    
    if(is_void(plotLegends))
        plotLegends=1;
    
    if(resid==1)
    {
        ksys=-1;
        plusKsys=2;
        nWin = 2*numberof(where([plotVis,plotClos,plotDPhi,plotDVis,plotSpec]));
    }
    else
    {
        ksys=0;
        plusKsys=1;
        nWin = numberof(where([plotVis,plotClos,plotDPhi,plotDVis,plotSpec]));
    }

    if(nWin==0)
    {
        yocoLogWarning, "Nothing to plot!";
        return 0;
    }
    
    if(kill==1)
    {
        winkill,win;

        // if(nWin==1)
        //     fx=1;
        // else
        fx = [];
        
        // Add residuals plots
        if(resid==1)
        {
            ry = reform(array([1.,0.4],nWin/2),[1,nWin]);
            dy = reform(array([0.001,0.015],nWin/2),[1,nWin]);
            yocoNmCreate,win,1,nWin,fx=,height=750,width=950,wait=1,
                landscape=1,V = [0.15,0.9,0.12,0.73],dpi=100,ry=ry,dy=dy;
        }
        else
        {
            yocoNmCreate,win,1,nWin,fx=fx,height=750,width=950,wait=1,
                landscape=1,V = [0.15,0.9,0.12,0.43],dpi=100,dy=0.015;
        }
    }
    else
        //yocoNmCreate,win,1,1,dy=0.065,fx=1,height=750,width=950,wait=1, landscape=1;
        window,win, wait=1;

    if(clear==1)
        fma;

    
    // Spectrum plot
    if(plotSpec)
    {
        ksys+=plusKsys;
        plsys,ksys;
    
        plotSpectrum, dataStruct, color=color,
            model=model, width=width, type=type, marker=marker;
        
        // xytitles, "Wavelength", "Spectrum", [0.01,0.025];

        if(anyof(limSpec))
            limits,,,limSpec(1),limSpec(2);
    }
    
    // Squared visibilitiy plot
    if(plotVis)
    {
        ksys+=plusKsys;
        plsys,ksys;
    
        plotVisSpFreq, dataStruct, projAngle=projAngle,
            cutAngle=cutAngle, cutOpen=cutOpen, color=color,
            model=model, width=width, type=type, marker=marker,
            msize=msize, visType=plotVis, sizebar=sizebar, avgWlen=avgWlen,
            legendHeight=legendHeight;
                
        // Set graph limits and xy axes names
        //limits,0,,-0.1,1.1;
        limits,0;

        if(plotLegends==1)
        {
            if(plotVis==1)
                yAxis = "Vis.";
            else if(plotVis==2)
                yAxis = "V^2";
            if(!is_void(projAngle))
                xytitles,// "Sp. Freq. (B._proj._/!l, cycles/arcsec.)"
                    ,
                    yAxis,[.01,0.025];
            else
                xytitles,// "Sp. Freq. (cycles/arcsec.)"
                    ,
                    yAxis,[.01,0.025];
        }   
               
        if(logVis)
            logxy,,1;

        if(anyof(limVis))
            limits,,,limVis(1),limVis(2);

        if(anyof(limFreq))
            limits,limFreq(1),limFreq(2);
        
        yocoPlotHorizLine,[0,1],type="dash";
    }
        
        
    // Closure phase plot
    if(plotClos)
    {
        ksys+=plusKsys;
        plsys,ksys;

        if(anyof(limClos))
            limits,,,limClos(1),limClos(2);
        if(anyof(limFreq))
            limits,limFreq(1),limFreq(2);
    
        plotClosSpFreq, dataStruct, projAngle=projAngle,
            cutAngle=cutAngle, cutOpen=cutOpen, color=color,
            model=model, width=width, type=type, marker=marker,
            msize=msize, sizebar=sizebar,
            wrap=wrap, phaseDeg=phaseDeg, avgWlen=avgWlen;

        if(anyof(limClos))
            limits,0,,limClos(1),limClos(2);
        else
            limits,0,,-4,4;

        yocoPlotHorizLine,0,type="dot";
        if(phaseDeg==1)
            yocoPlotHorizLine,[-180,180],type="dash";
        else
            yocoPlotHorizLine,[-pi,pi],type="dash";
            
        
        if(phaseDeg==1)
            phaseUnit = "^o^";
        else
            phaseUnit = "rad";
            
        if(!is_void(projAngle))
            xytitles,
                // "Sp. Freq. largest baseline (B._proj._/!l, cycles/arcsec.)"
                ,
                "Clos. !f ("+phaseUnit+")",[0.01,0.025];
        else
            xytitles,// "Sp. Freq. largest baseline (cycles/arcsec.)"
                ,
                "Clos. !f ("+phaseUnit+")",[0.01,0.025];
    }
        
    // Differential phase plot
    if(plotDPhi)
    {
        ksys+=plusKsys;
        plsys,ksys;
    
        plotPhaseSpFreq, dataStruct, projAngle=projAngle,
            cutAngle=cutAngle, cutOpen=cutOpen, color=color,
            model=model, width=width, type=type, marker=marker,
            msize=msize, 
            wrap=wrap, phaseDeg=phaseDeg;
        
        limits,0;
        if(phaseDeg==1)
            phaseUnit = "^o^";
        else
            phaseUnit = "rad";
        
        if(!is_void(projAngle))
            xytitles,// "Sp. Freq. (B._proj._/!l, cycles/arcsec.)"
                ,
                diffPlt+"!f ("+phaseUnit+")",[0.01,0.025];
        else
            xytitles,// "Sp. Freq. (cycles/arcsec.)"
                ,
                diffPlt+"!f ("+phaseUnit+")",[0.01,0.025];

        if(anyof(limDPhi))
            limits,,,limDPhi(1),limDPhi(2);

        if(anyof(limFreq))
            limits,limFreq(1),limFreq(2);
    }
    
    
    // Differential visibility plot
    if(plotDVis)
    {
        ksys+=plusKsys;
        plsys,ksys;
    
        plotVisSpFreq, dataStruct, projAngle=projAngle,
            cutAngle=cutAngle, cutOpen=cutOpen, color=color,
            model=model, width=width, type=type, marker=marker,
            msize=msize, visType=3, sizebar=sizebar, avgWlen=avgWlen;

        limits,0;
        if(phaseDeg==1)
            phaseUnit = "^o^";
        else
            phaseUnit = "rad";
            
        if(!is_void(projAngle))
            xytitles,
                // "Sp. Freq. largest baseline (B._proj._/!l, cycles/arcsec.)"
                ,
                "Diff. Vis.",[0.01,0.025];
        else
            xytitles,// "Sp. Freq. largest baseline (cycles/arcsec.)"
                ,
                "Diff. Vis.",[0.01,0.025];

        if(anyof(limDVis))
            limits,,,limDVis(1),limDVis(2);

        if(anyof(limFreq))
            limits,limFreq(1),limFreq(2);
    }
    //    yocoNmXytitles, "Sp. Freq. (cycles/arcsec.)";
}

/*******************************************************/

func plotResSpFreq(dataStruct, modelStruct, projAngle=, cutAngle=, cutOpen=, color=, model=, width=, plotVis=, plotClos=, plotDPhi=, plotDVis=, type=, marker=, msize=, kill=, win=, clear=, wrap=, phaseDeg=, noDiff=, sizebar=, avgWlen=, limVis=, limClos=, limDPhi=, limDVis=, limSpec=, limFreq=, logVis=)
    /* DOCUMENT plotResSpFreq(dataStruct, modelStruct, projAngle=, cutAngle=, cutOpen=, color=, model=, width=, plotVis=, plotClos=, plotDPhi=, plotDVis=, type=, marker=, msize=, kill=, win=, clear=, wrap=, phaseDeg=, noDiff=, sizebar=, avgWlen=, limVis=, limClos=, limDPhi=, limDVis=, limSpec=, limFreq=, logVis=)

       DESCRIPTION
       Plots data residuals as a function of spatial frequencies

       PARAMETERS
       - dataStruct : The data structure
       - modelStruct: 
       - projAngle  : 
       - cutAngle   : 
       - cutOpen    : 
       - color      : 
       - model      : 
       - width      : 
       - plotVis    : 
       - plotClos   : 
       - plotDPhi   : 
       - plotDVis   : 
       - type       : 
       - marker     : 
       - msize      : 
       - kill       : 
       - win        : 
       - clear      : 
       - wrap       : 
       - phaseDeg   : 
       - noDiff     : 
       - sizebar    : 
       - avgWlen    : 
       - limVis     : 
       - limClos    : 
       - limDPhi    : 
       - limDVis    : 
       - limSpec    : 
       - limFreq    : 
       - logVis     : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(msize))
        msize=0.2;
    
    if(is_void(noDiff))
        diffPlt = "Diff. ";
    else
        diffPlt = "";
        
    if(is_void(plotVis))
        plotVis = 1;
    
    if(is_void(plotClos))
        plotClos = 1;

    if(is_void(plotDPhi))
        plotDPhi = 0;

    if(is_void(plotDVis))
        plotDVis = 0;

    if(is_void(plotSpec))
        plotSpec = 0;

    if(is_void(win))
        win=1;

    if(is_void(kill))
        kill=1;
    
    nWin = 2*numberof(where([plotVis,plotClos,plotDPhi,plotDVis]));
    if(kill==1)
    {
        winkill,win;
        yocoNmCreate,win,1,nWin,dy=0.0,fx=,height=750,width=950,wait=1, landscape=1,V=[0.15,0.9,0.12,0.73],dpi=100;
    }
    else
        //yocoNmCreate,win,1,1,dy=0.065,fx=1,height=750,width=950,wait=1, landscape=1;
        window,win, wait=1;

    if(clear==1)
        fma;

    ksys=0;
    tmpStruct = array(amplOiData, dimsof(dataStruct));

    
    // Spectrum plot
    if(plotSpec)
    {
        ksys+=2;
        plsys,ksys;

        // Compute residuals for spectrum
        for(k=1;k<=numberof(dataStruct);k++)
        {
            tmp = *dataStruct(k).specErr;
            if(is_void(tmp))
                continue;
            dat = *dataStruct(k).spec;
            dam = median(dat);
            mod = *modelStruct(k).spec;
            tmpStruct(k).spec =
                &((dat/dam - mod/median(mod)) /
                  (tmp/dam + (tmp==0)));
            tmpStruct(k).specErr  = dataStruct(k).specErr;
            tmpStruct(k).EFF_WAVE = dataStruct(k).EFF_WAVE;
        }
    
        plotSpectrum, tmpStruct, color=color,
            model=1, width=width, type=type, marker=marker;
        
        //xytitles, "Wavelength", "Spectrum", [0.01,0.025];

        if(anyof(limSpec))
            limits,,,limSpec(1),limSpec(2);
    }
    
    // Squared visibilitiy plot
    if(plotVis)
    {
        ksys+=2;
        plsys,ksys;

        // Compute residuals for squared visibilities
        for(k=1;k<=numberof(dataStruct);k++)
        {
            tmp = *dataStruct(k).VIS2ERR;
            if(is_void(tmp))
                continue;
            dat = *dataStruct(k).VIS2DATA;
            mod = *modelStruct(k).VIS2DATA;
            tmpStruct(k).VIS2DATA =
                &((dat - mod) /
                  (tmp + (tmp==0)));
            tmpStruct(k).VIS2ERR  = dataStruct(k).VIS2ERR;
            tmpStruct(k).EFF_WAVE = dataStruct(k).EFF_WAVE;
            tmpStruct(k).UCOORD  = dataStruct(k).UCOORD;
            tmpStruct(k).VCOORD  = dataStruct(k).VCOORD;
        }
    
        plotVisSpFreq, tmpStruct, projAngle=projAngle,
            cutAngle=cutAngle, cutOpen=cutOpen, color=color,
            model=model, width=width, type=type, marker=marker,
            msize=msize, visType=plotVis, sizebar=sizebar, avgWlen=avgWlen;
                
        // Set graph limits and xy axes names
        //limits,0,,-0.1,1.1;
        limits,0;
                
        yAxis = "Res. (!s)";
        if(!is_void(projAngle))
            xytitles,// "Sp. Freq. (B._proj._/!l, cycles/arcsec.)"
                ,
                yAxis,[0.01,0.025];
        else
            xytitles,// "Sp. Freq. (cycles/arcsec.)"
                ,
                yAxis,[0.01,0.025];
        
        if(logVis)
            logxy,,1;

        if(anyof(limVis))
            limits,,,limVis(1),limVis(2);

        if(anyof(limFreq))
            limits,limFreq(1),limFreq(2);

        yocoPlotHorizLine,0;
        yocoPlotHorizLine,[3,-3],type="dash";
    }
        
        
    // Closure phase plot
    if(plotClos)
    {
        ksys+=2;
        plsys,ksys;

        // Closure phase residual
        for(k=1;k<=numberof(dataStruct);k++)
        {
            if(!is_void(*dataStruct(k).T3PHI))
            {
                faz = exp(1i * (*dataStruct(k).T3PHI - *modelStruct(k).T3PHI));
                tmpStruct(k).T3PHI    = &(atan(faz.im,faz.re) / *dataStruct(k).T3PHIERR);
                tmpStruct(k).T3PHIERR = dataStruct(k).T3PHIERR;
                tmpStruct(k).EFF_WAVE = dataStruct(k).EFF_WAVE;
                tmpStruct(k).UCOORD   = dataStruct(k).UCOORD;
                tmpStruct(k).VCOORD   = dataStruct(k).VCOORD;
                tmpStruct(k).U1COORD  = dataStruct(k).U1COORD;
                tmpStruct(k).V1COORD  = dataStruct(k).V1COORD;
                tmpStruct(k).U2COORD  = dataStruct(k).U2COORD;
                tmpStruct(k).V2COORD  = dataStruct(k).V2COORD;
                tmpStruct(k).T3AMP    = dataStruct(k).T3AMP;
                tmpStruct(k).T3AMPERR = dataStruct(k).T3AMPERR;
            }
        }
    
        plotClosSpFreq, tmpStruct, projAngle=projAngle,
            cutAngle=cutAngle, cutOpen=cutOpen, color=color,
            model=model, width=width, type=type, marker=marker,
            msize=msize,
            wrap=wrap, phaseDeg=0, avgWlen=avgWlen;
        
        limits,0;
        if(phaseDeg==1)
            phaseUnit = "^o^";
        else
            phaseUnit = "rad";
            
        if(!is_void(projAngle))
            xytitles,
                // "Sp. Freq. largest baseline (B._proj._/!l, cycles/arcsec.)"
                ,
                "Res. (!s)",[0.01,0.025];
        else
            xytitles,// "Sp. Freq. largest baseline (cycles/arcsec.)"
                ,
                "Res. (!s)",[0.01,0.025];

        if(anyof(limClos))
            limits,,,limClos(1),limClos(2);

        if(anyof(limFreq))
            limits,limFreq(1),limFreq(2); 

        yocoPlotHorizLine,0;
        yocoPlotHorizLine,[3,-3],type="dash";

    }
        
    // Differential phase plot
    if(plotDPhi)
    {
        ksys+=2;
        plsys,ksys;

        // Differential phase residual
        for(k=1;k<=numberof(dataStruct);k++)
        {
            if(!is_void(*dataStruct(k).VISPHI))
            {
                faz = exp(1i * (*dataStruct(k).VISPHI - *modelStruct(k).VISPHI));
                tmpStruct(k).VISPHI    = &(atan(faz.im,faz.re) / *dataStruct(k).VISPHIERR);
                tmpStruct(k).VISPHIERR = dataStruct(k).VISPHIERR;
                tmpStruct(k).EFF_WAVE  = dataStruct(k).EFF_WAVE;
                tmpStruct(k).UCOORD    = dataStruct(k).UCOORD;
                tmpStruct(k).VCOORD    = dataStruct(k).VCOORD;
            }
        }
    
        plotPhaseSpFreq, tmpStruct, projAngle=projAngle,
            cutAngle=cutAngle, cutOpen=cutOpen, color=color,
            model=model, width=width, type=type, marker=marker,
            msize=msize, 
            wrap=wrap, phaseDeg=phaseDeg, avgWlen=avgWlen;
        
        limits,0;
        
        if(!is_void(projAngle))
            xytitles,// "Sp. Freq. (B._proj._/!l, cycles/arcsec.)"
                ,
                "Res. (!s)",[0.01,0.025];
        else
            xytitles,// "Sp. Freq. (cycles/arcsec.)"
                ,
                "Res. (!s)",[0.01,0.025];

        if(anyof(limDPhi))
            limits,,,limDPhi(1),limDPhi(2);

        if(anyof(limFreq))
            limits,limFreq(1),limFreq(2);
        
        yocoPlotHorizLine,0;
        yocoPlotHorizLine,[3,-3],type="dash";

        // prout();
    }
    
    
    // Differential visibility plot
    if(plotDVis)
    {
        ksys+=2;
        plsys,ksys;
    
        // Compute residuals for squared visibilities
        for(k=1;k<=numberof(dataStruct);k++)
        {
            tmp = *dataStruct(k).VISAMPERR;
            if(is_void(tmp))
                continue;
            
            tmpStruct(k).VISAMP =
                &((*dataStruct(k).VISAMP - *modelStruct(k).VISAMP) /
                  (tmp + (tmp==0)));
            tmpStruct(k).VISAMPERR = dataStruct(k).VISAMPERR;
            tmpStruct(k).EFF_WAVE  = dataStruct(k).EFF_WAVE;
            tmpStruct(k).UCOORD    = dataStruct(k).UCOORD;
            tmpStruct(k).VCOORD    = dataStruct(k).VCOORD;
        }

        if(anyof(limVis))
            limits,,,limVis(1),limVis(2);

        if(anyof(limFreq))
            limits,limFreq(1),limFreq(2);
    
        plotVisSpFreq, tmpStruct, projAngle=projAngle,
            cutAngle=cutAngle, cutOpen=cutOpen, color=color,
            model=model, width=width, type=type, marker=marker,
            msize=msize, visType=3, sizebar=sizebar, avgWlen=avgWlen;
        
        // Set graph limits and xy axes names
        //limits,0,,-0.1,1.1;
        limits,0;
            
        yAxis = "Res. (!s)";
        if(!is_void(projAngle))
            xytitles,// "Sp. Freq. (B._proj._/!l, cycles/arcsec.)"
                ,
                yAxis,[0.01,0.025];
        else
            xytitles,// "Sp. Freq. (cycles/arcsec.)"
                ,
                yAxis,[0.01,0.025];

        if(anyof(limDVis))
            limits,,,limDVis(1),limDVis(2);

        if(anyof(limFreq))
            limits,limFreq(1),limFreq(2);
    }
    yocoNmXytitles, "Sp. Freq. (cycles/arcsec.)";

    yocoPlotHorizLine,0;
    yocoPlotHorizLine,[3,-3],type="dash";
}

/*******************************************************/

func plotSpectrum(dataStruct, color=, model=, width=, type=, marker=, marks=, normalize=)
    /* DOCUMENT plotSpectrum(dataStruct, color=, model=, width=, type=, marker=, marks=, normalize=)

       DESCRIPTION

       PARAMETERS
       - dataStruct: 
       - color     : 
       - model     : 
       - width     : 
       - type      : 
       - marker    : 
       - marks     : 
       - normalize : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    
    if(is_void(msize))
        msize = 0.2;
    
    if(is_void(normalize))
        normalize = "median";
    
    // plot the data as a function of spatial frequency
    nFiles = numberof(dataStruct);
    
    for(k=1;k<=nFiles;k++)
    {
        wlen = *dataStruct.EFF_WAVE(k);
        
        spec    =  *dataStruct.spec(k);
        specErr =  *dataStruct.specErr(k);
        
        if(!is_void(spec))
        {
            if(normalize=="median")
            {
                norm = median(spec);
                norm = norm+(norm==0);
            }
            else
                error;
            
            if(is_void(color))
            {
                col = ["red", "green", "blue"];
            }
            else
                col = [color, color, color]; 
            
            if(!is_void(specErr)&&(noneof(specErr==0))&&(!model))
            {
                yocoPlotWithErrBars, spec/norm, specErr/norm, wlen*1e6, color=col(1),marker=marker,msize=msize;
            }
            else
                plg, spec/norm, wlen*1e6, color=col(1);
        }
    }
    
}

/*******************************************************/

func plotDataWlen(dataStruct, projAngle=, color=, model=, width=, plotVis=, plotClos=, plotDPhi=, plotDVis=, plotSpec=, limVis=, limClos=, limDPhi=, limSpec=, logVis=, type=, marker=, marks=, kill=, win=, clear=, phaseDeg=, msize=, idxData=)
    /* DOCUMENT plotDataWlen(dataStruct, projAngle=, color=, model=, width=, plotVis=, plotClos=, plotDPhi=, plotDVis=, plotSpec=, limVis=, limClos=, limDPhi=, limSpec=, logVis=, type=, marker=, marks=, kill=, win=, clear=, phaseDeg=, msize=, idxData=)

       DESCRIPTION
       From a data structure, plots the data as a function of wavelength

       PARAMETERS
       - dataStruct: The data structure to plot
       - projAngle : a UV coordinates projection angle
       - color     : 
       - model     : 
       - width     : 
       - plotVis   : 
       - plotClos  : 
       - plotDPhi  : 
       - plotDVis  : 
       - plotSpec  : 
       - limVis    : 
       - limClos   : 
       - limDPhi   : 
       - limSpec   : 
       - logVis    : 
       - type      : 
       - marker    : 
       - marks     : 
       - kill      : 
       - win       : 
       - clear     : 
       - phaseDeg  : 
       - msize     : 
       - idxData   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    colors= ["red", "green", "blue", "cyan", "magenta", "yellow", "black"];
    
    if(is_void(msize))
        msize=0.2;
    
    if(is_void(plotVis))
        plotVis=2;
    if(is_void(plotClos))
        plotClos=1;
    if(is_void(plotDPhi))
        plotDPhi=1;
    if(is_void(plotDVis))
        plotDVis=1;
    if(is_void(win))
        win=1;
    
    // plot the data as a function of wavelengths
    nFiles = numberof(dataStruct);
    if(is_void(idxData))
        idxData = indgen(nFiles);

    nWin = 0;
    for(kFile=1;kFile<=numberof(idxData);kFile++)
    {
        if(plotVis)
        {
            if(!is_void(*dataStruct.VIS2DATA(idxData(kFile))))
                nWin++;
        }
        if(plotDPhi)
        {  
            if(!is_void(*dataStruct.VISPHI(idxData(kFile))))
                nWin++;
        }
        if(plotDVis)
        {  
            if(!is_void(*dataStruct.VISAMP(idxData(kFile))))
                nWin++;
        }
        if(plotClos)
        {
            if(!is_void(*dataStruct.T3PHI(idxData(kFile))))
                nWin++;
        }
        if(plotSpec)
        {
            if(!is_void(*dataStruct.spec(idxData(kFile))))
                nWin++;
        }
    }

    sqWinSize1 = sqrt(nWin);

    fracWin = 2.00;
    sqX = int(max(1,floor(sqWinSize1/sqrt(fracWin))));
    sqY = int(max(1,floor(sqWinSize1*sqrt(fracWin))));

    if(plotClos==1)
        V=[0.15,0.9,0.42,0.73];
    else
        V=[0.15,0.9,0.27,0.73];
        
    while(sqX*sqY<nWin)
        sqX++;
    
    if(kill==1)
    {
        
        winkill,win;
        //yocoNmCreate,win,sqX,sqY,dy=0.02,dx=0.06,height=800,width=800,wait=1, V = [0.12,0.73,0.1,0.9], dpi=65;
        yocoNmCreate,win,sqX,sqY,dy=0.03,dx=0.06,fx=1,height=750,width=950,wait=1, landscape=1,V=V,dpi=100;
    }
    else
    {
        window, win, wait=1;
    }
    
    if(clear==1)
        fma;
    
    ksys = 0;
    for(kFile=1;kFile<=numberof(idxData);kFile++)
    {
        wlen = *dataStruct.EFF_WAVE(idxData(kFile));
        // Wavelength in microns
        wlen = wlen*1e6;
        minWlen = min(wlen);
        maxWlen = max(wlen);

        // Set colors of plots
        if(is_void(color))
        {
            if(anyof((wlen<1.3e-6)&(wlen>0.9e-6)))
                col = "blue";
            else if(anyof((wlen<1.8e-6)&(wlen>1.6e-6)))
                col = "green";
            else if(anyof((wlen<2.4e-6)&(wlen>2.0e-6)))
                col = "red";
            else
                col=[128,128,128];
        }
        else
            col = array(color,10000);
        
        /////////////////////
        // V2 plot
        /////////////////////
        vis2 = *dataStruct.VIS2DATA(idxData(kFile));
        if(!is_void(vis2))
        {
            u =  (*dataStruct.UCOORD(idxData(kFile)));
            if(!is_void(u))
            {
                v =  (*dataStruct.VCOORD(idxData(kFile)));
                A = atan(u,v);
                B = abs(u,v);
                
                // Squared visibilitiy plot
                if(plotVis)
                {
                    ksys++;
                    plsys,ksys;
                    
                    dimz = dimsof(vis2);
                    if(dimz(1)==1)
                        nbBases=1;
                    else
                        nbBases = dimsof(vis2)(3);
                    for(lb=1;lb<=nbBases;lb++)
                    {
                        // Choose whether plotting visibility or squared visibility
                        if(plotVis==1)
                        {
                            V = yocoMathSignedSqrt((vis2)(,lb));
                            V2Err = (*dataStruct.VIS2ERR(idxData(kFile)))(,lb);
                            VErr = 0.5* V2Err / (V+(V==0));
                            V2Err = VErr;
                            V2 = V;
                        }
                        else if(plotVis==2)
                        {
                            V2 = (vis2)(,lb);
                            V2Err = (*dataStruct.VIS2ERR(idxData(kFile)))(,lb);
                        }
                        
                        // Get rid of Nans
                        wr     = where((V2==V2)&(V2Err==V2Err));
                        V2     = V2(wr);
                        V2Err  = V2Err(wr);
                        wlen   = wlen(wr);
                        
                        // Model plot: no error bars
                        if(model==1)
                        {
                            if(mk==1)
                            {
                                plmk, V2+lb-1, wlen,
                                    color=col(lb), marker=marker,msize=msize;
                            }
                            else
                            {
                                plg,V2+lb-1, wlen,
                                    color=col(lb),width=width, type=type,
                                    marker=marker, marks=marks;
                            }
                        }
                        // Data plot: error bars
                        else /*MAJ : addition of lb-1 on the V2 to give an offset to the visibility*/
                        {
                            yocoPlotWithErrBars, V2+lb-1, V2Err,
                                wlen,
                                marker=marker, marks=marks,msize=msize;
                            //  plt,pr1(idxData(kFile)), min(wlen)+0.1*(max(wlen)-min(wlen)), -0.2,tosys=ksys;//,color=col(lb),width=width, type=type;
                        }

                        yocoPlotHorizLine, lb-1, type="dash";
                        
                        plt,swrite(B(lb), A(lb)*180/pi, format="BL: %2.2f m, PA: %2.2f ^o"),
                        min(wlen)+0.1*(max(wlen)-min(wlen)),
                        lb-1+0.5,tosys=ksys;
                        
                        // Set graph limits and xy axes names
                        limits,,,-0.1,nbBases+0.1;

                        if(plotVis==1)
                            yAxis = "V";
                        else if(plotVis==2)
                            yAxis = "V^2";
                    
                        xytitles,,yAxis,[0.02,0.025];
                    }
                    yocoPlotHorizLine, lb-1, type="dash";
                }
            }
        }

        /////////////////////////////////////
        // Differential visibility plot
        /////////////////////////////////////
        /*MAJ : There wasn't the section to plot the differential visibility*/
        dVis = *dataStruct.VISAMP(idxData(kFile));
        if(!is_void(dVis))
        {     
            u =  (*dataStruct.UCOORD(idxData(kFile)));
            if(!is_void(u))
            {
                v =  (*dataStruct.VCOORD(idxData(kFile)));
                
                // Differential visibility plot
                if(plotDVis)
                {     
                    ksys++;
                    plsys,ksys;
                    dimz = dimsof(dVis);
                    if(dimz(1)==1)
                        nbBases=1;
                    else
                        nbBases = dimsof(dVis)(3);

                    maxDV = 0;
                    for(l=1;l<=nbBases;l++)
                    {
                
                        // Store differential visibility
                        DV = (dVis)(,l);
                        DVErr = (*dataStruct.VISAMPERR(idxData(kFile)))(,l);
                        
                        // Get rid of Nans
                        wr     = where((DV==DV)&(DVErr==DVErr));
                        DV     = DV(wr);
                        DVErr  = DVErr(wr);
                        wlen   = wlen(wr);

                        if(!is_void(DV))
                        {
                            // Model plot: no error bars
                            if(model==1)
                            {
                                if(mk==1)
                                {
                                    plmk,DV+l-1, wlen,
                                        color=col(l), marker=marker,msize=msize;
                                }
                                else
                                {
                                    plg,DV+l-1, wlen,
                                        color=col(l),width=width, type=type,
                                        marker=marker, marks=marks;
                                }
                            }
                            // Data plot: error bars
                            else
                            {
                                yocoPlotWithErrBars,DV+l-1,DVErr,
                                    wlen,color=col(l),width=width, type=type,
                                    marker=marker, marks=marks,msize=msize;

                                //plt,pr1(idxData(kFile)), min(wlen)+0.1*(max(wlen)-min(wlen)), -0.2,tosys=ksys;
                            }
                            yocoPlotHorizLine, l-1, type="dash";
                            
                            maxDV = min(max(maxDV,max(DV+l-1)),6);
                        }
                    }
                    yocoPlotHorizLine, l-1, type="dash";
                    
                    
                    limits,,,0,1.1*maxDV;
                    if(sqX>3)
                        xytitles,,"DV",[0.02,0.025];
                    else
                        xytitles,,"Diff. Vis.",[0.02,0.025];
                        
                }
            }
        }

        /////////////////////////////////////
        // Closure phase plot
        /////////////////////////////////////
        clos =(*dataStruct.T3PHI(idxData(kFile)));
        if(!is_void(clos))
        {
            if(phaseDeg)
            {
                clos = clos*180/pi;
                fac  = 100;
            }
            else
                fac=1;
            
            u1 =  (*dataStruct.U1COORD(idxData(kFile)));
            if(!is_void(u))
            {
                u2 =  (*dataStruct.U2COORD(idxData(kFile)));
                v1 =  (*dataStruct.V1COORD(idxData(kFile)));
                v2 =  (*dataStruct.V2COORD(idxData(kFile)));
                
                if(plotClos)
                {
                    ksys++;
                    plsys,ksys;
                
                    nbClos = dimsof(clos)(3);
                    maxC= minC=0;
                    if(phaseDeg)
                        maxP = 360;
                    else
                        maxP = 6;
                    for(l=1;l<=nbClos;l++)
                    {
                
                        C    = clos(,l);
                        CErr = (*dataStruct.T3PHIERR(idxData(kFile)))(,l);
            
                        if(phaseDeg)
                            CErr = CErr*180/pi;
                        
                        maxC = min(max(maxC,max(C+fac*(l-1))),maxP);
                        minC = max(min(minC,min(C+fac*(l-1))),-maxP);
                        
                        if(model==1)
                        {
                            if(mk==1)
                            {
                                plmk,C, wlen,
                                    color=col(1), marker=marker,msize=msize;
                            }
                            else
                            {
                                plg,C, wlen,
                                    color=col(1),width=width, type=type,
                                    marker=marker, marks=marks;
                            }

                
                        }
                        else
                        {
                            yocoPlotWithErrBars, C, CErr,
                                wlen,color=col(1),width=width, type=type,
                                marker=marker, marks=marks,msize=msize;
                            //plt,pr1(idxData(kFile)), min(wlen)+0.1*(max(wlen)-min(wlen)), 1.3*minC,tosys=ksys;
                        }
                    
                        yocoPlotHorizLine, fac*(l-1), type="dash";
                        
                        if(is_void(limClos))
                            limClos = [1.1*minC,1.1*maxC];
                    
                        limits,,,limClos(1),limClos(2);
                        limits,,,-180,180;
                        
                        if(sqX>3)
                            xytitles,,"CP",[0.02,0.025];
                        else
                            xytitles,,"Clos !f",[0.02,0.025];
                    }
                }
            }
        }
        
        /////////////////////////////////////
        // Differential phase plot
        /////////////////////////////////////
        dPhi = *dataStruct.VISPHI(idxData(kFile));
        if(!is_void(dPhi))
        {
            if(phaseDeg)
            {
                dPhi = dPhi*180/pi;
                fac  = 100;
            }
            else
                fac=1;
                    
            u =  (*dataStruct.UCOORD(idxData(kFile)));
            if(!is_void(u))
            {
                v =  (*dataStruct.VCOORD(idxData(kFile)));
                
                // Differential phase plot
                if(plotDPhi)
                {     
                    ksys++;
                    plsys,ksys;
                    nbBases = dimsof(dPhi)(3);
                    maxDP= minDP=0;
                    for(l=1;l<=nbBases;l++)
                    {
                        DP = (dPhi)(,l);
                        DPErr = (*dataStruct.VISPHIERR(idxData(kFile)))(,l);

                        if(phaseDeg)
                            DPErr = DPErr*180/pi;
                        
                        if(model==1)
                        {
                            if(mk==1)
                            {
                                plmk,DP+fac*(l-1), wlen,
                                    color=col(l), marker=marker,msize=msize;
                            }
                            else
                            {
                                plg,DP+fac*(l-1), wlen,
                                    color=col(l),width=width, type=type,
                                    marker=marker, marks=marks;
                            }
                        }
                        else
                        {
                            yocoPlotWithErrBars,DP+fac*(l-1),DPErr,
                                wlen,color=col(l),width=width, type=type,
                                marker=marker, marks=marks,msize=msize;
                        }
                    
                        yocoPlotHorizLine, fac*(l-1), type="dash";

                        if(phaseDeg)
                            maxP = 360;
                        else
                            maxP = 6;
                        
                        maxDP = min(max(maxDP,max(DP+fac*(l-1))),maxP);
                        minDP = max(min(minDP,min(DP+fac*(l-1))),-maxP);
                    }
                    
                    
                    // plt,pr1(idxData(kFile)), min(wlen)+0.1*(max(wlen)-min(wlen)), 1.3*minDP,tosys=ksys;
                    
                    if(is_void(limDPhi))
                        limDPhi = [1.1*minDP,1.1*maxDP];
                    
                    limits,,,limDPhi(1),limDPhi(2);
                        limits,,,-140,320;
                    // limits,,,-pi,pi;
                    
                    if(sqX>3)
                        xytitles,,"DP",[0.02,0.025];
                    else
                        xytitles,,"Diff. !f",[0.02,0.025];
                }
            }
        }
        
        /////////////////////////////////////
        // Spectrum plot
        /////////////////////////////////////
        spec = *dataStruct.spec(idxData(kFile));
        if(!is_void(spec))
        {
            // Spectrum plot
            if(plotSpec)
            {      
                ksys++;
                plsys,ksys;
                
                SP = spec;
                SPErr = (*dataStruct.specErr(idxData(kFile)));
                if(model==1)
                {
                    if(mk==1)
                    {
                        plmk,SP/median(SP), wlen,
                            color=col(1), marker=marker,msize=msize;
                    }
                    else
                    {
                        plg,SP/median(SP), wlen,
                            color=col(1),width=width, type=type,
                            marker=marker, marks=marks,msize=msize;
                    }
                }
                else
                {
                    yocoPlotWithErrBars,SP/median(SP),SPErr/median(SP),
                        wlen,color=col(l),width=width, type=type,
                        marker=marker, marks=marks,msize=msize;
                    // plt,pr1(idxData(kFile)), min(wlen)+0.1*(max(wlen)-min(wlen)),3,tosys=ksys;
                }
                
                // limits,,,-pi,pi;
                if(is_void(limSpec))
                    limSpec = [0,1.1*max(SP/median(SP))];
                limits,,,limSpec(1),limSpec(2);
                xytitles,,"Spectrum",[0.02,0.025];
            }
        }
    }
    yocoNmXytitles,"Wavelength (!mm)",,[0.02,0.025];
    // yocoNmLimits,,,-.5,4.5;
}

/*******************************************************/

func initModel(&toot, typei, isvariable, parNbWlen, value, ifov, sfov)
    /* DOCUMENT initModel(&toot, typei, isvariable, parNbWlen, value, ifov, sfov)

       DESCRIPTION

       PARAMETERS
       - toot      : 
       - typei     : 
       - isvariable: 
       - parNbWlen : 
       - value     : 
       - ifov      : 
       - sfov      : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if((toot.funcFit)!=string(nil))
        nPars = symbol_def(toot.funcFit)(toot);
    
    if(is_void(value))
        value = array(1e-8,int(sum(parNbWlen+(parNbWlen==0))));

    /******************************************/
    
    if(numberof(*toot.parVar)==numberof(isvariable))
        *toot.parVar = isvariable;
    else
        toot.parVar = &isvariable;
        
    if(numberof(*toot.parNbWlen)==numberof(parNbWlen))
        *toot.parNbWlen = parNbWlen;
    else
        toot.parNbWlen = &parNbWlen;
        
    if((numberof(*toot.typei)==numberof(typei))&&
       !is_void(typei))
        *toot.typei = typei;
    else
        toot.typei = &typei;
        
    /******************************************/
    
    // Get wavelength range of dataset
    data = *toot.data;
    getWlenRange,data,minWlen, maxWlen;
    
    ns = numberof(typei);
    toot.nSources = ns;
    
    minWlen = 1e99;
    maxWlen = -1e99;
    no = numberof(data);
    for(f=1;f<=no;f++)
    {
        wlen = *data(f).EFF_WAVE;
        wlen2 = wlen(where(wlen!=0));
        if(minWlen>min(wlen2))
            minWlen = min(wlen2);
        if(maxWlen<max(wlen2))
            maxWlen = max(wlen2);
    }

    nPar = numberof(pars);
    param = modelVal = parName = parUnit = wlens = [];

    // parNbWlen gives the number of spectral channels to use per parameter:
    // 0 for dummy, 1 for constant, 2 for a line between extreme wavelengths,
    // etc. The wavelength range is log-equally sampled with the different
    // wavelengths
    
    // isVariable is an array with same dimension of parNbWlen telling if
    // the group of parameters is variable or fixed

    // Loop on N sources
    for(k=1;k<=ns;k++)
    {
        // Loop on N parameters
        for(l=1;l<=nPar;l++)
        {
            // Append values only if 
            if(parNbWlen(l,k)!=0)
            {
                if(parNbWlen(l,k)>1)
                    // Here is the real wavelength sampling of the parameters
                    grow,wlens,spanl(minWlen,maxWlen,parNbWlen(l,k));
                else
                    grow,wlens,(minWlen+maxWlen)/2.0;
                
                // Decide whether using random values or not
                if((pars(l)=="flux")&&(!is_void(sfov)))
                    totoVal = sfov * random(parNbWlen(l,k));
                else if((pars(l)!="flux")&&(pars(l)!="x")&&
                        (pars(l)!="y")&&(!is_void(ifov)))
                    totoVal = ifov * random(parNbWlen(l,k));
                else if((pars(l)!="flux")&&(!is_void(ifov)))
                    totoVal = ifov * random_n(parNbWlen(l,k));
                else
                    totoVal = array(value,parNbWlen(l,k));
                
                if(isvariable(l,k)!=0)
                    grow,param,totoVal;
                grow,modelVal,totoVal;

                idxDescr = where(typei(k)==ALL_MODELS.type)(1);
                pars  = *ALL_MODELS.params(idxDescr);
                units = *ALL_MODELS.units(idxDescr);
                
                if(numberof(pars)>=l)
                {
                    grow,parName,array(pars(l),parNbWlen(l,k));
                    grow,parUnit,array(units(l),parNbWlen(l,k));
                }
                else
                {
                    grow,parName,"no";
                    grow,parUnit,"no";
                }
            }
        }
    }
    if(numberof(*toot.param)==numberof(param))
        *toot.param    = param;
    else
    {
        toot.param   = &param;
    }
        
    if(numberof(*toot.parName)==numberof(parName))
        *toot.parName  = parName;
    else
    {
        toot.parName = &parName;
    }
    
    if(numberof(*toot.parUnit)==numberof(parUnit))
        *toot.parUnit  = parUnit;
    else
    {
        toot.parUnit = &parUnit;
    }
    
    if(numberof(*toot.modelVal)==numberof(modelVal))
        *toot.modelVal = modelVal;
    else
        toot.modelVal  = &modelVal;
    
    if(numberof(*toot.parWlen)==numberof(wlens))
        *toot.parWlen  = wlens;
    else
    {
        toot.parWlen    = &wlens;
    }

    write,"Model Wavelengths";
    print,wlens;
}

/*******************************************************/

func getWlenRange(&data, &minWlen, &maxWlen)
    /* DOCUMENT getWlenRange(&data, &minWlen, &maxWlen)

       DESCRIPTION

       PARAMETERS
       - data   : 
       - minWlen: 
       - maxWlen: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // Determine wavelength interval
    minWlen = 1e99;
    maxWlen = -1e99;
    no = numberof(data);
    for(f=1;f<=no;f++)
    {
        wlen = *data(f).EFF_WAVE;
        wlen2 = wlen(where(wlen!=0));
        if(minWlen>min(wlen2))
            minWlen = min(wlen2);
        if(maxWlen<max(wlen2))
            maxWlen = max(wlen2);
    }
}
/*******************************************************/

func getRefModel(tot, &refMod)
    /* DOCUMENT getRefModel(tot, &refMod)

       DESCRIPTION

       PARAMETERS
       - tot   : 
       - refMod: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // Copy structures without passing pointers
    refMod           = fitStruct();
    refMod.data      = tot.data;
    initModel, refMod, *tot.typei,
        *tot.parVar, *tot.parNbWlen, 0.0;
    *refMod.modelVal = *tot.modelVal;
    *refMod.param    = *tot.param;
    dataLine = getDataInLine(*tot.data,
                             tot.use_vis2,
                             tot.use_clos,
                             tot.use_bsamp,
                             tot.use_dVis,
                             tot.use_dPhi, tot.use_spec,
                             dataError,dataType);
    refMod.dataLine    = &dataLine;
    refMod.dataErrLine = &dataError;
    refMod.dataType    = &dataType;
    refMod.funcFit     = tot.funcFit;
    refMod.use_vis2    = tot.use_vis2;
    refMod.use_clos    = tot.use_clos;
    refMod.use_bsamp   = tot.use_bsamp;
    refMod.use_dPhi    = tot.use_dPhi;
    refMod.use_dVis    = tot.use_dVis;
    refMod.use_spec    = tot.use_spec;
    refMod.absClos     = tot.absClos;
    refMod.absFlux     = tot.absFlux;
    refMod.absSize     = tot.absSize;
    resFunc, refMod, *tot.param;
}

/*******************************************************/

func computeAllObservables(&toot, &mod, Umod, Vmod, wlenMod, cf, fl)
    /* DOCUMENT computeAllObservables(&toot, &mod, Umod, Vmod, wlenMod, cf, fl)

       DESCRIPTION

       PARAMETERS
       - toot   : 
       - mod    : 
       - Umod   : 
       - Vmod   : 
       - wlenMod: 
       - cf     : 
       - fl     : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    data = *toot.data;

    for(iObs=1;iObs<=numberof(data);iObs++)
    {
    
        // Compute spectrum
        if(!is_void(*(data.spec)(iObs))&&toot.use_spec)
        {
            wl = *(data.EFF_WAVE)(iObs);

            wlMatch = wlOrder = [];
            for(kWl=1;kWl<=numberof(wlenMod);kWl++)
                if(anyof(wl==wlenMod(kWl)))
                {
                    grow, wlOrder, where(wl==wlenMod(kWl));
                    grow, wlMatch, kWl;
                }
            
            spec = computeInsSpec(fl(wlMatch)(wlOrder), wl);
            
            if(numberof(*(mod.spec)(iObs))==numberof(spec))
                *(mod.spec)(iObs) = spec;
            else
                (mod.spec)(iObs) = &spec;
            
            specErr = array(0.0,dimsof(spec));
            if(numberof(*(mod.specErr)(iObs))==numberof(specErr))
                *(mod.specErr)(iObs) = specErr;
            else
                (mod.specErr)(iObs) = &specErr;
        }
        
        // Compute closure phase
        if(!is_void(*(data.T3PHI)(iObs))&&toot.use_clos)
        {
            wl = *(data.EFF_WAVE)(iObs);
            wlMatch=wlOrder=[];
            for(kWl=1;kWl<=numberof(wlenMod);kWl++)
            {
                if(anyof(wl==wlenMod(kWl)))
                {
                    grow, wlOrder, where(wl==wlenMod(kWl));
                    grow, wlMatch, kWl;
                }
            }
            
            u1 = (*(data.U1COORD)(iObs));
            v1 = (*(data.V1COORD)(iObs));
            u2 = (*(data.U2COORD)(iObs));
            v2 = (*(data.V2COORD)(iObs));
            u3 = u1 + u2;
            v3 = v1 + v2;
            CCC = array(complex, 3, numberof(wl), numberof(u1));
            for(iClos=1;iClos<=numberof(u1);iClos++)
            {
                CCC(1,,iClos) = cf(where((Umod==u1(iClos))&(Vmod==v1(iClos)))(1),wlMatch)(wlOrder) /
                    fl(wlMatch)(wlOrder);
                CCC(2,,iClos) = cf(where((Umod==u2(iClos))&(Vmod==v2(iClos)))(1),wlMatch)(wlOrder) /
                    fl(wlMatch)(wlOrder);
                CCC(3,,iClos) = cf(where(
                                         (swrite(format="%3.3f", Umod)==
                                          swrite(format="%3.3f", u3(iClos)))&
                                         (swrite(format="%3.3f", Vmod)==
                                          swrite(format="%3.3f", v3(iClos)))
                                         )(1),wlMatch)(wlOrder) /
                    fl(wlMatch)(wlOrder);
            }
            
            clos = [computeInsClos(CCC)];

            if(toot.absClos==1)
                clos = abs(clos);
            
            if(numberof(*(mod.T3PHI)(iObs))==numberof(clos))
                *(mod.T3PHI)(iObs) = clos;
            else
                (mod.T3PHI)(iObs) = &clos;
            
            closErr = array(transpose(array(0.0,dimsof(clos))),1);
            if(numberof(*(mod.T3PHIERR)(iObs))==numberof(closErr))
                *(mod.T3PHIERR)(iObs) = closErr;
            else
                (mod.T3PHIERR)(iObs) = &closErr;
        }
        
        
        // Compute Bispectrum amplitude
        if(!is_void(*(data.T3AMP)(iObs))&&toot.use_bsamp)
        {
            wl = *(data.EFF_WAVE)(iObs);
            wlMatch=wlOrder=[];
            for(kWl=1;kWl<=numberof(wlenMod);kWl++)
            {
                if(anyof(wl==wlenMod(kWl)))
                {
                    grow, wlOrder, where(wl==wlenMod(kWl));
                    grow, wlMatch, kWl;
                }
            }
            
            u1 = (*(data.U1COORD)(iObs));
            v1 = (*(data.V1COORD)(iObs));
            u2 = (*(data.U2COORD)(iObs));
            v2 = (*(data.V2COORD)(iObs));
            u3 = u1 + u2;
            v3 = v1 + v2;
            CCC = array(complex, 3, numberof(wl), numberof(u1));
            for(iBsamp=1;iBsamp<=numberof(u1);iBsamp++)
            {
                CCC(1,,iBsamp) = cf(where((Umod==u1(iBsamp))&(Vmod==v1(iBsamp)))(1),wlMatch)(wlOrder) /
                    fl(1,wlMatch)(wlOrder);
                CCC(2,,iBsamp) = cf(where((Umod==u2(iBsamp))&(Vmod==v2(iBsamp)))(1),wlMatch)(wlOrder) /
                    fl(1,wlMatch)(wlOrder);
                CCC(3,,iBsamp) = cf(where(
                                          (swrite(format="%3.3f", Umod)==
                                           swrite(format="%3.3f", u3(iBsamp)))&
                                          (swrite(format="%3.3f", Vmod)==
                                           swrite(format="%3.3f", v3(iBsamp)))
                                          )(1),wlMatch)(wlOrder) /
                    fl(1,wlMatch)(wlOrder);
            }
            
            bsamp = [computeInsBSAmp(CCC)];
            
            if(numberof(*(mod.T3AMP)(iObs))==numberof(bsamp))
            {
                *(mod.T3AMP)(iObs) = bsamp;
            }
            else
            {
                (mod.T3AMP)(iObs) = &bsamp;
            }
            
            bsampErr = array(transpose(array(0.0,dimsof(bsamp))),1);
            if(numberof(*(mod.T3AMPERR)(iObs))==numberof(bsampErr))
                *(mod.T3AMPERR)(iObs) = bsampErr;
            else
            {
                (mod.T3AMPERR)(iObs) = &bsampErr;
            }
        }

        vis2 = *(data.VIS2DATA)(iObs);
        // Compute squared visibility
        if(!is_void(vis2)&&toot.use_vis2)
        {
            wl = *(data.EFF_WAVE)(iObs);
            if(dimsof(vis2)(1)>=2)
                nbBases = dimsof(vis2)(3);
            else
                nbBases = numberof(vis2);
          
            wlMatch=wlOrder=[];
            for(kWl=1;kWl<=numberof(wlenMod);kWl++)
                if(anyof(wl==wlenMod(kWl)))
                {
                    grow, wlOrder, where(wl==wlenMod(kWl));
                    grow, wlMatch, kWl;
                }
            vis2 = [];
            for(jBas=1;jBas<=nbBases;jBas++)
            {
                u1 = (*(data.UCOORD)(iObs))(jBas);
                v1 = (*(data.VCOORD)(iObs))(jBas);
                grow, vis2,
                    array(computeInsSqVis(cf(where((Umod==u1)&(Vmod==v1))(1),wlMatch)(wlOrder),
                                          fl(wlMatch)(wlOrder)), 1);
            }
        
            //vis2    = array(transpose(vis2),1);
            vis2Err = array(0.0,dimsof(vis2));
            
            if(numberof(*(mod.VIS2DATA)(iObs))==numberof(vis2))
                *(mod.VIS2DATA)(iObs) = vis2;
            else
            {
                (mod.VIS2DATA)(iObs) = &vis2;
            }
        
            if(numberof(*(mod.VIS2ERR)(iObs))==numberof(vis2Err))
                *(mod.VIS2ERR)(iObs) = vis2Err;
            else
            {
                (mod.VIS2ERR)(iObs) = &vis2Err;
            }
        }

        visamp = *(data.VISAMP)(iObs);
        // Compute [differential] visibility
        // toot.use_dVis = 1 : differential visibility
        // toot.use_dVis = 2 : visibility
        if(!is_void(visamp)&&toot.use_dVis)
        {
            wl      = *(data.EFF_WAVE)(iObs);
            nbBases = dimsof(visamp)(3);
            wlMatch = wlOrder = [];
            
            for(kWl=1;kWl<=numberof(wlenMod);kWl++)
            {
                if(anyof(wl==wlenMod(kWl)))
                {
                    grow, wlOrder, where(wl==wlenMod(kWl));
                    grow, wlMatch, kWl;
                }
            }
            
            dVis = dVisErr = [];
            for(jBas=1;jBas<=nbBases;jBas++)
            {
                u1 = (*(data.UCOORD)(iObs))(jBas);
                v1 = (*(data.VCOORD)(iObs))(jBas);

                if(toot.use_dVis == 1) 
                    grow, dVis,
                        array(computeInsDVis(cf(where((Umod==u1)&(Vmod==v1))(1),
                                                wlMatch)(wlOrder),
                                             fl(1,wlMatch)(wlOrder),
                                             wlenMod(wlMatch)(wlOrder)), 1);

                else if(toot.use_dVis == 2)
                    grow, dVis,
                        array(computeInsVis(cf(where((Umod==u1)&(Vmod==v1))(1),
                                               wlMatch)(wlOrder),
                                            fl(1,wlMatch)(wlOrder)), 1);
                else
                    error;
            }
        
            // dVis    = array(transpose(dVis),1);
            dVisErr = array(0.0,dimsof(dVis));
        
            if(numberof(*(mod.VISAMP)(iObs))==numberof(dVis))
                *(mod.VISAMP)(iObs) = dVis;
            else
                (mod.VISAMP)(iObs)  = &dVis;
        
            if(numberof(*(mod.VISAMPERR)(iObs))==numberof(dVisErr))
                *(mod.VISAMPERR)(iObs) = dVisErr;
            else
                (mod.VISAMPERR)(iObs)  = &dVisErr;
        }

        visphi = *(data.VISPHI)(iObs);
        // Compute differential phase
        if(!is_void(visphi)&&toot.use_dPhi)
        {
            wl = *(data.EFF_WAVE)(iObs);
            nbBases = dimsof(visphi)(3);
            wlMatch = wlOrder = [];
            for(kWl=1;kWl<=numberof(wlenMod);kWl++)
            {
                if(anyof(wl==wlenMod(kWl)))
                {
                    grow, wlOrder, where(wl==wlenMod(kWl));
                    grow, wlMatch, kWl;
                }
            }
            dPhi = dPhiErr = [];
            for(jBas=1;jBas<=nbBases;jBas++)
            {
                u1 = (*(data.UCOORD)(iObs))(jBas);
                v1 = (*(data.VCOORD)(iObs))(jBas);
                
                grow, dPhi,
                    array(computeInsDiffPhi(cf(where((Umod==u1)&(Vmod==v1))(1),wlMatch)(wlOrder),
                                            wlenMod(wlMatch)(wlOrder)), 1);
            }
            
            // dPhi    = array(transpose(dPhi),1);
            dPhiErr = array(0.0,dimsof(dPhi));
        
            if(numberof(*(mod.VISPHI)(iObs))==numberof(dPhi))
                *(mod.VISPHI)(iObs) = dPhi;
            else
                (mod.VISPHI)(iObs) = &dPhi;
        
            if(numberof(*(mod.VISPHIERR)(iObs))==numberof(dPhiErr))
                *(mod.VISPHIERR)(iObs) = dPhiErr;
            else
            {
                (mod.VISPHIERR)(iObs) = &dPhiErr;
            }
        }
    }
}

/*******************************************************/

func funcIma(&toot, ima)
    /* DOCUMENT funcIma(&toot, ima)

       DESCRIPTION
       Fit function for imaging (image as input parameter)

       PARAMETERS
       - toot: 
       - ima : 
       
       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    //FIXME: THIS IS A MEMORY DEBUGGING. See also next FIXME: 
    // ysa = yorick_stats();
    
    // Initialization of models used
    extern pars, units;
    local FLUX, X, Y, PIX, data, mod, typei;
    
    pars  = ["pix"];
    units = ["no"];
    
    if(is_void(ima))
        return "Nothine there!";
        
    // updateParVal,toot,a;
  
    data = *toot.data;
    mod  = *toot.model;

        
    // If there is no model, initialize structure
    if(is_void(mod))
    {
        mod            = array(amplOiData(),dimsof(data));
        mod.EFF_WAVE   = data.EFF_WAVE;
        mod.UCOORD     = data.UCOORD;
        mod.VCOORD     = data.VCOORD;
        mod.U1COORD    = data.U1COORD;
        mod.V2COORD    = data.V2COORD;
        mod.V1COORD    = data.V1COORD;
        mod.U2COORD    = data.U2COORD;
        mod.hdr.TARGET = data.hdr.TARGET;
        mod.hdr.file   = data.hdr.file;
        mod.TIME       = data.TIME;
        mod.TIME3      = data.TIME3;
        mod.EFF_BAND   = data.EFF_BAND;
    }

    // Initialize dimensions
    nObs  = numberof(data);
    nb    = *toot.parNbWlen;
    wlens = *toot.parWlen;
    
    modelVal = *toot.modelVal;
    
    typei = *toot.typei;
    nS    = toot.nSources;

    nPar = symbol_def(toot.funcFit)(toot);
    modInLine = [];

    // Get data sorted as a single vector
    dataLine = getDataInLine(data,
                             toot.use_vis2, toot.use_clos,toot.use_bsamp,
                             toot.use_dVis, toot.use_dPhi, toot.use_spec,
                             errInLine, dataType, wlenInLine,
                             U1inLine, V1inLine, U2inLine, V2inLine,
                             timeInLine);

    // Sort out wavelengths to avoid redundancy
    wlenUse = yocoStrRemoveMultiple(wlenInLine);
    
    // Sort out UV coordinates to avoid also here redundancy
    U1use = V1use = U2use = V2use = U3use = V3use = [];
    for(ku=1;ku<=numberof(U1inLine);ku++)
    {
        if(noneof((U1use==U1inLine(ku)) & (V1use==V1inLine(ku))))
        {
            grow, U1use, U1inLine(ku);
            grow, V1use, V1inLine(ku);
        }
    }
    
    U2Order = where(U2inLine!=0);
    if(numberof(U2Order))
    {
        U3us = U1inLine + U2inLine;
        V3us = V1inLine + V2inLine;
    
        for(ku=1;ku<=numberof(U2inLine(U2order));ku++)
        {
            if(noneof((U2use==U2inLine(U2order)(ku)) &
                      (V2use==V2inLine(U2order)(ku))))
            {
                grow, U2use, U2inLine(ku);
                grow, V2use, V2inLine(ku);
            }
        
            if(noneof((U3use==U3us(ku)) &
                      (V3use==V3us(ku))))
            {
                grow, U3use, U3us(ku);
                grow, V3use, V3us(ku);
            }
        }
        U1use = grow(U1use, U2use, U3use);
        V1use = grow(V1use, V2use, V3use);
    }

    // Re-define dimensions
    nbWlen  = numberof(wlenUse);
    nbBases = numberof(U1use);

    getWlenRange, data, minWlen, maxWlen;
    nWlenIma = dimsof(ima)(0);
    if(nWlenIma>1)
        imaWlen  = span(minWlen,maxWlen,nWlenIma);
    else
        imaWlen = avg([minWlen, maxWlen]);
        
    // Compute FFT
    CC = imageFFT(U1use, V1use, wlenUse, imaWlen, ima, Xtable, Ytable, cf, fl, U_FFT, V_FFT, FFT);
    toot.paramFFT   = &FFT;
    toot.paramMaxUV = &U_FFT(max,max,,);
    toot.paramFlux  = &fl;
    
    computeAllObservables, toot, mod, U1use, V1use, wlenUse, cf, fl;

    // if(numberof(U1use)>2000)
    //     stop()

    if(numberof(*toot.model) == numberof(mod))
        *toot.model = mod;
    else
    {
        toot.model  = &mod;
    }
    
    //FIXME: next debugging step
    // ysb = yorick_stats();
    // ysa
    //  ysb
    // ysb-ysa;
    // write,"end"
        
    return 0;
}

/*******************************************************/

func funcMultiple(&toot, a)
    /* DOCUMENT funcMultiple(&toot, a)

       DESCRIPTION

       PARAMETERS
       - toot: 
       - a   : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    //FIXME: THIS IS A MEMORY DEBUGGING. See also next FIXME: 
    // ysa = yorick_stats();
    
    // Initialization of models used
    extern pars, units;
    local FLUX, X, Y, FWHM1, FWHM2, ANGLE,
        ADD7, ADD8, ADD9, ADD10, ADD11, ADD12, ADD13, ADD14, ADD15,
        ADD16, ADD17, ADD18, ADD19, ADD20, data, mod, typei;
    
    pars  = ["flux", "x", "y", "fwhm1", "fwhm2", "angle", "add7", "add8", "add9", "add10", "add11", "add12", "add13", "add14", "add15", "add16", "add17", "add18", "add19", "add20"];
    units = ["no", "mas", "mas", "mas", "mas", "deg", "mas", "no", "no", "no", "no", "no", "no", "no", "no", "no", "no", "no", "no", "no"];
    
    if(is_void(a))
        return numberof(pars);
        
    updateParVal,toot,a;
  
    data = *toot.data;
    mod  = *toot.model;

    if(is_void(mod))
    {
        mod            = array(amplOiData(),dimsof(data));
        mod.EFF_WAVE   = data.EFF_WAVE;
        mod.UCOORD     = data.UCOORD;
        mod.VCOORD     = data.VCOORD;
        mod.U1COORD    = data.U1COORD;
        mod.V2COORD    = data.V2COORD;
        mod.V1COORD    = data.V1COORD;
        mod.U2COORD    = data.U2COORD;
        mod.hdr.TARGET = data.hdr.TARGET;
        mod.hdr.file   = data.hdr.file;
        mod.TIME       = data.TIME;
        mod.TIME3      = data.TIME3;
        mod.EFF_BAND   = data.EFF_BAND;
    }

    nObs      = numberof(data);
    nb        = *toot.parNbWlen;
    wlens     = *toot.parWlen;
    modelVal  = *toot.modelVal;
    typei     = *toot.typei;
    nS        =  toot.nSources;
        
    nPar      = symbol_def(toot.funcFit)(toot);
    modInLine = [];

    dataLine  = getDataInLine(data,
                              toot.use_vis2, toot.use_clos,toot.use_bsamp,
                              toot.use_dVis, toot.use_dPhi, toot.use_spec,
                              errInLine, dataType, wlenInLine,
                              U1inLine, V1inLine, U2inLine, V2inLine,
                              timeInLine);
    
    wlenUse = yocoStrRemoveMultiple(wlenInLine,order);

    U1use = V1use = U2use = V2use = U3use = V3use = timeUse = [];
    for(ku=1;ku<=numberof(U1inLine);ku++)
    {
        if(noneof((U1use==U1inLine(ku)) & (V1use==V1inLine(ku))))
        {
            grow, U1use, U1inLine(ku);
            grow, V1use, V1inLine(ku);
            grow, timeUse, timeInLine(ku);
        }
    }
    
    U2Order = where(U2inLine!=0);
    if(numberof(U2Order))
    {
        U3us = U1inLine + U2inLine;
        V3us = V1inLine + V2inLine;
    
        for(ku=1;ku<=numberof(U2inLine(U2order));ku++)
        {
            if(noneof((U2use==U2inLine(U2order)(ku)) &
                      (V2use==V2inLine(U2order)(ku))))
            {
                grow, U2use, U2inLine(ku);
                grow, V2use, V2inLine(ku);
                grow, timeUse, timeInLine(ku);
            }
        
            if(noneof((U3use==U3us(ku)) &
                      (V3use==V3us(ku))))
            {
                grow, U3use, U3us(ku);
                grow, V3use, V3us(ku);
                grow, timeUse, timeInLine(ku);
            }
        }
        U1use = grow(U1use, U2use, U3use);
        V1use = grow(V1use, V2use, V3use);
    }
    
    // U1use = yocoStrRemoveMultiple(U1inLine, order);
    // V1use = V1inLine(order);
    
    // U2use = U2inLine(order);
    // V2use = V2inLine(order);
    // U2Order = where(U2use!=0);
    // if(numberof(U2Order)!=0)
    // {
    //     U2use = U2use(U2Order);
    //     V2use = V2use(U2Order);
    //     U3use = U1use(U2Order) + U2use;
    //     V3use = V1use(U2Order) + V2use;

    //     U1use = grow(U1use, U2use, U3use);
    //     V1use = grow(V1use, V2use, V3use);
    // }
    
    nbWlen = numberof(wlenUse);
    nbBases = numberof(U1use);
    
    parsI  = strcase(1,pars);
    
    for(l=1;l<=nPar;l++)
    {
        symbol_set, parsI(l),array(0.0, nbWlen, 1);
    }
    
    iKountt = 0;
    kountt = 1;
    for(gfsk=1;gfsk<=nS;gfsk++)
    {
        for(l=1;l<=nPar;l++)
        {
            iKountt += nbWlen;
            nbElem = numberof(symbol_def(parsI(l)));
            
            if(iKountt>nbElem)
            {
                symbol_set, parsI(l),
                    grow(symbol_def(parsI(l)),
                         array(0.0,dimsof(symbol_def(parsI(l)))));
            }
            
            // allocate values
            knt = nb(l,gfsk);
            
            if(knt==0)
            {
                tmp = symbol_def(parsI(l));
                tmp(..,gfsk) = array(0.0, nbWlen,1);
                symbol_set,parsI(l),tmp;
            }
            else if(knt==1)
            {
                value = modelVal(kountt);
                tmp = symbol_def(parsI(l));
                tmp(..,gfsk) = array(value, nbWlen,1);
                symbol_set,parsI(l),tmp;
            }
            else
            {
                wl           = wlens(kountt:kountt+knt-1);
                value        = modelVal(kountt:kountt+knt-1);
                tmp          = symbol_def(parsI(l));
                tmp(..,gfsk) = array(spline(value, wl, wlenUse),1);
                symbol_set,parsI(l),tmp;
            }
            
            kountt+=knt;
        }
    }
    for(l=1;l<=nPar;l++)
        symbol_set,parsI(l),
            symbol_def(parsI(l))(,:nS); 

    CC = visMultipleResolved(U1use, V1use, wlenUse, timeUse, typei,
                             FLUX, X, Y, FWHM1, FWHM2, ANGLE,
                             ADD7, ADD8, ADD9, ADD10, ADD11, ADD12,
                             ADD13, ADD14, ADD15, ADD16, ADD17, ADD18,
                             ADD19, ADD20, cf, fl);
    
    computeAllObservables, toot, mod, U1use, V1use, wlenUse, cf, fl;

    // if(numberof(U1use)>2000)
    //     stop()

    if(numberof(*toot.model) == numberof(mod))
        *toot.model = mod;
    else
        toot.model  = &mod;
    
    //FIXME: next debugging step
    // ysb = yorick_stats();
    // ysa
    //  ysb
    // ysb-ysa;
    // write,"end"
        
    return 0;
}

/***************************************************************************/

func fitLoadSciFiles(&dataStruct, nightDir=, logFile=, verbose=, calibrated=, crl=, crr=)
    /* DOCUMENT fitLoadSciFiles(&dataStruct, nightDir=, logFile=, verbose=, calibrated=, crl=, crr=)

       DESCRIPTION
       Load only the files marked as "SCIENCE" and fill in an oidata structure

       PARAMETERS
       - dataStruct: 
       - nightDir  : Data directory name
       - logFile   : 
       - verbose   : 
       - calibrated: 
       - crl       : 
       - crr       : 

       RESULTS 

       CAUTIONS
       Still in development !

       EXAMPLES 

       SEE ALSO:
    */
{
    VIS2 = VIS2_ERR =
        DVIS = DVIS_ERR =
        CLOS = CLOS_ERR =
        PHI = PHI_ERR =
        WLEN = TIME = NAMES =
        ORIG_V2 = ORIG_V2_ERR = FILENAMES =
        BANDWIDTH = U = V = U1 = V1 = U2 = V2 = dataStruct = [];
    
    /* Select directory containing files to be treated */
    message = "Choose the product directory (OI_DATA)"; 
    if (amdlibFileChooser(message, nightDir) == 0)
    {
        return 0;
    }

    if(is_void(verbose))
        verbose=1;

    if(is_void(logFile))
    {
        amdlibGetLogName, nightDir, logFile;
        logFile = nightDir + string(logFile);
    }

    amdlibReadLog, logFile, logTable, titles;

    tabFiles = logTable(where(titles=="fileName"), );
    tabDates = logTable(where(titles=="date_obs"), );
    tabNr    = logTable(where(titles=="calib_nr"), );
    proTypes = logTable(where(titles=="pro_catg"), );
    tabTypes = logTable(where(titles=="obs_catg"), );
    OBJTypes = logTable(where(titles=="obs_type"), );
    tabIds   = logTable(where(titles=="P2VM_ID"), );
    tabOBsNr = logTable(where(titles=="OB_nr"), );
    tabTplNr = logTable(where(titles=="Template_nr"), );
    tabNames = logTable(where(titles=="object_name"), );

    sciObs = where((proTypes=="SCIENCE_AVERAGED"));
    if(calibrated==1)
        sciObs = where((proTypes=="SCIENCE_CALIBRATED"));


    sciFiles = tabFiles(sciObs);
    sciNames = tabNames(sciObs);
    nSciFiles = numberof(sciFiles);

    if(nSciFiles==0)
        return 0;
    
    inputOiFile=nightDir+"/"+sciFiles;
        
    require,"amplAMBER.i";
    amplLoadOiDatas, dataStruct, inOiFile=inputOiFile;
}


func vis2GaussFWHM(vis, BL, lambda)
    /* DOCUMENT vis2GaussFWHM(vis, BL, lambda)

       DESCRIPTION

       PARAMETERS
       - vis   : 
       - BL    : 
       - lambda: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    return sqrt(-log(vis) / (pi^2 * (BL/lambda)^2) * 4.0*log(2.0));
}

func vis2UD_Diam(vis, BL, lambda)
    /* DOCUMENT vis2UD_Diam(vis, BL, lambda)

       DESCRIPTION

       PARAMETERS
       - vis   : 
       - BL    : 
       - lambda: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    
}


/***************************************************************************/

func plotMyLUT(minCLUT, maxCLUT, orient=, sys=, unit=, sup=, inf=, levels=, levColor=, nLabs=, nTicks=, palette=, offset=, plotLog=, plotPow=)
    /* DOCUMENT plotMyLUT(minCLUT, maxCLUT, orient=, sys=, unit=, sup=, inf=, levels=, levColor=, nLabs=, nTicks=, palette=, offset=, plotLog=, plotPow=)

       DESCRIPTION
       Plot a color lookup table in a color bar at the side or behind the
       viewport.

       REQUIRE
       - style.i

       PARAMETERS
       - minCLUT : min value of the color lookup table
       - maxCLUT : max value of the color lookup table
       - orient  : bar orientation; 0=horizontal, 1=vertical (default)
       - sys     : gist-system for which the table has to be plotted
       - unit    : Put a unit to the labels
       - sup     : Put a sign for the top label
       - inf     : Put a sign for the bottom label
       - levels  : Number of levels if there are e.g. contour plots
       together with the image
       - levColor: Levels colors
       - nLabs   : number of labels on the color bar
       - nTicks  : Number of ticks on the color bar
       - palette : 
       - offset  : 
       - plotLog : 
       - plotPow : 
    
       EXAMPLES
       > window, 0, style="amdlib1horiz.gs", height=500, width=600, legends=0;
       > yocoPlotColorLookupTable, 0, 100,orient=1
       > winkill, 0
    */
{
    require, "style.i";

    if(is_void(unit))
        unit = "";

    if(is_void(sup))
        sup = "";

    if(is_void(inf))
        inf = "";
    
    if(is_void(offset))
        offset = 0.0;
    
    get_style, landscape, systems, legends, clegends;
    vportTmp = systems.viewport();
    if(is_void(sys))
    {
        vport = grow(vportTmp(1,min),
                     vportTmp(2,max),
                     vportTmp(3,min),
                     vportTmp(4,max))
            }
    else
        vport = vportTmp(,sys);

    if(is_void(nLabs))
        nLabs = 6;

    if(is_void(nTicks))
        nTicks = 2*(nLabs-1)+1;
    
    levs = span(minCLUT, maxCLUT,  250); 
    n = numberof(levs) + 1;

    if(is_void(palette))
        colors= span(1, n, n);
    else
        colors = char(palette);

    n = dimsof(colors)(0);
    

    dx = dy = 0.0;
    if (orient == 1)
    {
        x      = (vport(2)-offset + [0.022,0.042])(-:1:n+1,);
        xLabs  = (vport(2)-offset + 0.042)(-:1:nLabs);
        xTicks = (vport(2)-offset + 0.042)(-:1:nTicks);
        x0     = x(1, 1);
        x1     = x(1, 0);
        dx     = 0.005;
        if(plotLog)
        {
            y      = spanl(min(vport(3, )), max(vport(4, )), n+1)(, -:1:2);
            yLabs  = spanl(min(vport(3, )), max(vport(4, )), nLabs);
            yTicks = spanl(min(vport(3, )), max(vport(4, )), nTicks);
        }
        else if(plotPow)
        {
            y0 = span(minCLUT,maxCLUT,n+1)(, -:1:2)^(1./plotPow);
            y1 = (y0-min(y0)) / (max(y0)-min(y0));
            y  = min(vport(3, )) + (max(vport(4, ))-min(vport(3, )))*y1;
                                    
            yLabs  = span(min(vport(3, )), max(vport(4, )), nLabs);
            yTicks = span(min(vport(3, )), max(vport(4, )), nTicks);
            
            // y0 = span(minCLUT,maxCLUT,nLabs)^(1./plotPow);
            // y1 = (y0-min(y0)) / (max(y0)-min(y0));
            // yLabs  = min(vport(3, )) + (max(vport(4, ))-min(vport(3, )))*y1;
            
            // y0 = span(minCLUT,maxCLUT,nTicks)^(1./plotPow);
            // y1 = (y0-min(y0)) / (max(y0)-min(y0));
            // yTicks = min(vport(3, )) + (max(vport(4, ))-min(vport(3, )))*y1;

        }
        else
        {
            y      = span(min(vport(3, )), max(vport(4, )), n+1)(, -:1:2);
            yLabs  = span(min(vport(3, )), max(vport(4, )), nLabs);
            yTicks = span(min(vport(3, )), max(vport(4, )), nTicks);
        }
        y0     = y(1, 1);
        y1     = y(0, 1);
    }
    else
    {
        y      = (vport(3)-offset-[0.045, 0.065])(-:1:n+1, );
        yLabs  = (vport(3)-offset-0.065)(-:1:nLabs);
        yTicks = (vport(3)-offset-0.065)(-:1:nTicks);
        y0     = y(1, 1);
        y1     = y(1, 0);
        dy     = -0.005;
        x      = span(min(vport(1, )), max(vport(2, )), n+1)(, -:1:2);
        xLabs  = span(min(vport(1, )), max(vport(2, )), nLabs);
        xTicks = span(min(vport(1, )), max(vport(2, )), nTicks);
        x0     = x(1, 1);
        x1     = x(0, 1);
    }

    sys = plsys(0);
    plf, [colors], y, x, edges=0;

    plg, [y0, y0, y1, y1], [x0, x1, x1, x0], closed=1, marks=0, width=1, type=1;
        
    if(!is_void(levels))
    {
        pldj, array(x0,numberof(levels)), y0+levels/(maxCLUT-minCLUT)*(y1-y0),
            array(x1,numberof(levels)), y0+levels/(maxCLUT-minCLUT)*(y1-y0),
            color=levColor;
    }
    
    plsys, sys;

    labs = swrite(format="%.3g", span(minCLUT, maxCLUT, nLabs));
    labs(0) = sup+labs(0);
    labs(1) = inf+labs(1);

    plsys, 0;
    pldj, xTicks, yTicks, xTicks + dx, yTicks + dy;
    
    plsys, sys;
    plt1, labs+unit, xLabs + 2 * dx, yLabs + 2 * dy, justify=(orient?"LH":"CT");
}


/***************************************************************************/

func miral_write_imgCube(outputFile, imgCube, X, Y, wlen, mu, chi2, regul)
    /* DOCUMENT miral_write_imgCube(outputFile, imgCube, X, Y, wlen, mu, chi2, regul)

       DESCRIPTION
       Writes an image cube resulting from a multiwavelebgth run of MIRA

       PARAMETERS
       - outputFile: output fits file name
       - imgCube   : input image cube
       - X         : X axis
       - Y         : Y axis
       - wlen      : Z axis
       - mu        : 
       - chi2      : 
       - regul     : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    yocoFileSplitName,outputFile,d,f,e;
    yocoLogInfo,"Writing in directory: "+d;
    yocoLogInfo,"File name: "+f+e;
    cfitsWrite,outputFile,imgCube;

    dimz = dimsof(imgCube);
    if((dimz(1)==3)&&(!is_void(wlen)))
    {
        imgSizeW = dimz(4);
        minW     = min(wlen);
        if(numberof(wlen)>=2)
            deltaW = avg(wlen(dif));
        else
            deltaW = 0;
    }
    else if(!is_void(wlen))
    {
        deltaW = 0;
        minW   = wlen;
    }
    
    imgSizeX = dimz(2);
    minX = min(X);
    deltaX = avg(X(dif));

    imgSizeY = dimsof(imgCube)(3);
    minY = min(Y);
    deltaY = avg(Y(dif));

    fh = cfitsio_open(outputFile,"a"); // re-open the file to modify keys 
    cfitsio_write_key, fh, "CRPIX1", 1,
        "X ref pixel (R.A.)"; // x reference pix
    cfitsio_write_key, fh, "CRPIX2", 1,
        "Y ref pixel (dec)"; // y reference pix
    if(!is_void(wlen))
        cfitsio_write_key, fh, "CRPIX3", 1,
            "wavelength ref pixel"; // lambda reference pix

    cfitsio_write_key, fh, "CDELT1", deltaX,
        "X increment (rad)"; // pixel increment
    cfitsio_write_key, fh, "CDELT2", deltaY,
        "Y increment (rad)"; // pixel increment
    if(!is_void(wlen))
        cfitsio_write_key, fh, "CDELT3", deltaW,
            "wavelength increment (m)"; // wavelength increment

    cfitsio_write_key, fh, "CRVAL1", minX,
        "X minimum (rad)"; // pixel center (zero here)
    cfitsio_write_key, fh, "CRVAL2", minY,
        "Y minimum (rad)"; // pixel center (zero here)
    if(!is_void(wlen))
        cfitsio_write_key, fh, "CRVAL3", minW,
            "wavelength mimimum (m)"; // minimum wavelength

    if(!is_void(mu))
    {
        // FIXME: this is to replace the cfitsioplugin bintable writing
        require,"amplData.i";
        amplCfitsio_add_bintable, fh, [&mu(,1),&chi2(,1),&swrite(regul(,1),format="%-16s")], ["mu", "chi2", "regul"],["","",""], "INFOS";   // ad a binary table
    }
    
}
