/***************************************************************************
 *                                                                         *
 *                              fitOmatic                                  *
 *                   Model-fitting prototyping utility                     *
 *                    Copyright 2007-2012, F. Millour                      *
 *                            fmillour@oca.eu                              *
 *                                                                         *
 ***************************************************************************
 *
 * This script is a library of routines to compute visibilities of standard
 * analytical models.
 *
 * Please note that this script is distributed under the GPL licence,
 * available at http://www.gnu.org/licenses/gpl.txt
 *
 * Please ACKNOWLEDGE the use of this script for any use in a publication
 *
 ***************************************************************************
 * 
 * "@(#) $Id: visModels.i 668 2017-02-10 13:44:08Z fmillour $"
 *
 ***************************************************************************/

func visModels(void)
    /* DOCUMENT visModels

       DESCRIPTION
       This function computes any combination of several functions
       which are interfaced in this script. It outputs visibilities
       (as return value), correlated flux and total flux. Potentially
       up to 20 parameters can be set with this script.
       Standard analytical models for visibilities computation,
       plus tentative models for specific cases.

       REQUIRE
       - bessel.i
   
       FUNCTIONS
       - UVGrid                   : griding uv plane. Inspired from JP Berger work
       - UVLine                   : line in the uv plane. Inspired from JP Berger work
       - UVTable2Bases            : from (U,V) coordinates to baselengths
       - corrFxPoints             : 
       - grid2ima                 : uv plane grid to image (suitable for pli). Inspired from JP Berger work
       - grid2ima2                : uv plane grid to image (suitable for plf). Inspired from JP Berger work
       - growModels               : add one model structure to another
       - ima2Grid                 : transform an image to a grid
       - imageDFT                 : Discrete Fourier Transform of a model image
       - imageFFT                 : Fast Fourier transform of a model image
       - imageFFT_COMPUTE         : In the case only stretching, rotating and zomming is necessary, the previous function splits into 2.
       - imageFFT_INIT            : In the case only stretching, rotating and zomming is necessary, the previous function splits into 2.
       - impact_read              : read a CMFGEN output data file
       - shiftFourier             : Apply a image-space shift to the Fourier-transform of the image
       - visBinaryAndWWCZ         : model of a wind-wind collision zone and a binary star
       - visBinaryStar            : Binary star model
       - visDilutePinwheel        : 
       - visDiskGprofile          : 
       - visDoublePinwheel        : 
       - visEllipticalGaussianDisk: Elliptical gaussian disk model
       - visEllipticalProfile     : return the Hankel visibility function from a given profile
       - visEllipticalRing        : Elliptical ring model
       - visEllipticalSkewedRing  : 
       - visEllipticalThickRing   : 
       - visEllipticalUniformDisk : Elliptical uniform disk model
       - visExpDisk               : 
       - visExponential           : 
       - visGamWCZ                : 
       - visGaussianDisk          : gaussian disk model
       - visLimbLinear0           : 
       - visLimbLinear1           : 
       - visLineBiGaussian        : 
       - visLineDisk              : 
       - visLorentzianDisk        : 
       - visModelExpRotDisk       : 
       - visModelExpRotStar       : 
       - visModelGasDisk          : 
       - visModelGasPlusDustDisk  : 
       - visModels                : this script
       - visMultipleResolved      : Tentative multi-models
       - visOrbit                 : Model with orbital elements motion
       - visOrbit2                : Model with two components orbiting each other
       - visPinwheel              : Pinwheel model
       - visPinwheelTempLaw       : Pinwheel model with temperature law (not working yet)
       - visPowerDisk             : 
       - visPowerLaw              : 
       - visProfile               : 
       - visRectangle             : rectangle ("pixel") model
       - visRing                  : circular ring model
       - visSkewedRing            : skewed ring model
       - visSquare                : square ("pixel") model
       - visThickEllipticalRing   : ring with a thickness
       - visThickEllipticalRing2  : ring with a thickness second flavour
       - visThickRing             : a thick ring model
       - visUniformDisk           : circular uniform disk
       - visWWCZ                  : Tentative model of a wind-wind collision zone
       
    */
{
    version = strpart(strtok("$Revision: 668 $",":")(2),2:-2);
    if (am_subroutine())
    {
        write, format="package version: %s\n", version;
        help, visModels;
    }   
    return version;
} 

/***************************************************************************/

require, "amdlib.i";
require, "bessel.i";
require, "digit2.i";

yocoLogInfo,"#include \"visModels.i\"";

/***************************************************************************/

min2deg     = 1./60.;
sec2deg     = min2deg / 60.;
mas2deg     = sec2deg / 1000;
deg2rad     = pi/180.;
rad2deg     = 180./pi;
mas2rad     = mas2deg * deg2rad;
sec2rad     = sec2deg * deg2rad;
rad2mas     = rad2deg / mas2deg;

unit2kilo = milli2unit = 1e-3;
unit2mega = micro2unit = 1e-6;
unit2giga = nano2unit  = 1e-9;
unit2tera = pico2unit  = 1e-12;
  
kilo2unit = unit2milli = 1e3;
mega2unit = unit2micro = 1e6;
giga2unit = unit2nano  = 1e9;
tera2unit = unit2pico  = 1e12;

DIM0 = 64;

/*******************************************************/

struct modelDescription
{
    string type;
    string description;
    string function;
    pointer params;
    pointer units;
}; 

ALL_MODELS = [];

// Define the way elongated structures are described,
// Use AXES_DEF keyword to specify it, e.g. AXES_DEF=USE_INCLINATION
// Provide major and minor axes
USE_MAJ_MIN_AXES = 0;
// Provide major axis and flattening ratio
USE_FLATTENING   = 1;
// Provide major axis and inclination
USE_INCLINATION  = 2;

// Point source
grow, ALL_MODELS, array(modelDescription(),1);
ALL_MODELS.description(0) = "Point source";
ALL_MODELS.type(0)        = "no";
ALL_MODELS.params(0)      = &["flux", "x", "y"];
ALL_MODELS.units(0)       = &["no", "mas", "mas"];

// Fully resolved background
grow, ALL_MODELS, array(modelDescription(),1);
ALL_MODELS.description(0) = "Fully Resolved structure";
ALL_MODELS.type(0)        = "ful";
ALL_MODELS.params(0)      = &["flux"];
ALL_MODELS.units(0)       = &["no"];

/*******************************************************/

func impact_read(file, &lambda, &p, &Ip)
    /* DOCUMENT impact_read(file, &lambda, &p, &Ip)

       DESCRIPTION
       Reading routine for Gamma Vel WR Model, made from the impact_read.pro
       from O. Chesneau

       PARAMETERS
       - file  : input file
       - lambda: wavelength
       - p     : impact parameter
       - Ip    : specific intensity
   
       SEE ALSO
    */
{
    // Read whole ascii file
    dat = yocoFileReadAscii(file,nbCols=4);

    // Number of frequencies
    ncf = yocoStr2Long(dat(1,1));
    // Number of points in the profile
    np  = yocoStr2Long(dat(2,1));
    // Flux data
    pIp = Ip = array(double,ncf,np);
    tmp = array(double,np);

    // frequency array
    // frequences en 10^15Hz
    f = yocoStr2Double(dat(1,2:ncf+1));

    // corresponding velocity array
    clight  = 2.99794e5;
    lambda  = 0.299794e+04 / f;  // frequency -> wavelength (A) conversion

    // impact parameter array
    p = yocoStr2Double(dat(,ncf+2:ncf+1+int(ceil(np/4.))));
    p = reform(p,[1,numberof(p)])(1:np);

    // p.I(p) array
    tmp = yocoStr2Double(dat(:3,ncf+1+int(ceil(np/4.))+1:));
    ttmp = reform(tmp,[1,numberof(tmp)]);
    while(anyof((ttmp(2:)==0)&(ttmp(:-1)==0)))
    {
        ttmp = ttmp(where(!((ttmp(2:)==0)&(ttmp(:-1)==0))));
    }

    pIp = reform(ttmp,[2,np,ncf]);
    Ip  = pIp / (p+(p==0))(,-);

}

/*******************************************************/

func growModels(&models, type)
    /* DOCUMENT growModels(&models, type)

       DESCRIPTION
       add one model structure to another

       PARAMETERS
       - models: model structure
       - type  : type of the model
    */
{
    if(is_void(models))
    {
        grow, models, array(modelDescription(),1); 
        models.type(0)        = type;
        return 1;
    }
    else if(noneof(models.type==type))
    {
        grow, models, array(modelDescription(),1);
        models.type(0)        = type;
        return 1;
    }
    return 0;
}

/***************************************************************************/

func imageFFT_INIT(image, Xtable, Ytable, &U_FFT, &V_FFT, &FFT, &Flux, upSample=)
    /* DOCUMENT imageFFT_INIT(image, Xtable, Ytable, &U_FFT, &V_FFT, &FFT, &Flux, upSample=)
   
       DESCRIPTION
       Returns the FFT of an image at the (U,V) coordinates. Known
       problems are interpolation artifacts when the UV points are
       much more denser than the FFT sampling

       PARAMETERS
       - image   : input image
       - Xtable  : X coordinates of the image
       - Ytable  : Y coordinates of the image
       - U_FFT   : 
       - V_FFT   : 
       - FFT     : 
       - Flux    : 
       - upSample: 
       
       RESULTS
       FFT interpolated at points (U,V)

       CAUTIONS
       BE CAREFUL! number of wavelengths are not altered by this function!

       EXAMPLES
       See testImage function

       SEE ALSO: imageFFT_COMPUTE, imageFFT
    */
{
    // Define variables as local to avoid modification to external variables
    local image2, XtableP, YtableP, dimzXP, dimzYP;
    extern _imgWkspc;
    
    // Take into account the case where Xtable and Ytable are
    // 1-dimension arrays, and transform them into 2D arrays
    if((dimsof(Xtable)(1)==1)&&(dimsof(Ytable)(1)==1))
    {
        nx = numberof(Xtable);
        ny = numberof(Ytable);
        nw = dimsof(image)(0);
                
        Xtable = array(Xtable, ny, nw);
        Ytable = transpose(array(Ytable, nx, nw),[1,2]);
    }

    // Define default values for upsampling
    if(is_void(upSample))
        upSample = 0;
    
    // Get dimension of arrays
    dimzX = dimsof(image)(2);
    dimzY = dimsof(image)(3);
    nbWlen = dimsof(image)(4);

    // Initialize arrays
    U_FFT = V_FFT = FFT = [];
    image2 = image;
    XtableP = Xtable;
    YtableP = Ytable;

    // Take into account the cases of odd arrays
    if(((dimzX%2)==0)||((dimzY%2)==0))
    {
        if((dimzX%2)==0)
            dimzXP = dimzX+1;
        else
            dimzXP = dimzX;
      
        if((dimzY%2)==0)
            dimzYP = dimzY+1;
        else
            dimzYP = dimzY;
          
        image2 = array(0.0,dimzXP,dimzYP,nbWlen);
      
        for(k=1;k<=nbWlen;k++)
        {
            YtableP = array(span(min(Ytable(,,k)), max(Ytable(,,k)),
                                 dimzYP), dimzXP);
            XtableP = transpose(array(span(min(Xtable(,,k)), max(Xtable(,,k)),
                                           dimzXP), dimzYP));

            image2(,,k) = interp2(XtableP,YtableP,
                                  image(,,k),Xtable(,,k),Ytable(,,k));
        }
        dimzX=dimzXP;
        dimzY=dimzYP;
        image=image2;
    }

    // Produce a square image with power of two for fft by means
    // of zero-padding
    b = upSample;
    Nx = int(ceil(log(dimzX)/log(2)+b));
    Ny = int(ceil(log(dimzY)/log(2)+b));
    N = 2^max(Nx,Ny);
    zeroPaddedImage = array(0.0,N,N,nbWlen);
    zeroPaddedImage(N/2-(dimzX)/2+1:N/2+dimzX/2+1,
                    N/2-(dimzY)/2+1:N/2+dimzY/2+1,) = image;
    image = zeroPaddedImage;

    // Compute the Fast-Fourier Transform of the input image, and
    // re-center it (roll)
    FFT = array(complex, dimsof(image));
    for(k=1;k<=nbWlen;k++)
    {
        if(is_void(_imgWkspc))
            _imgWkspc = fft_setup(dimsof(image(..,k)));
        if((*_imgWkspc(1))(1)!=N)
            _imgWkspc = fft_setup(dimsof(image(..,k)));
        
        FFT0 = fft(roll(image(..,k)),[1,1], setup=_imgWkspc);
        FFT(..,k) = roll(FFT0);
    }
    // Free memory !
    Flux = image(sum,sum,);
    image = [];
    
    // Normalize FFT to 1 for zero-frequency
    FFT = FFT/((Flux+(Flux==0))(-,-,));
    
    // Compute spatial frequencies
    intervX = (max(Xtable)-min(Xtable))*double(N)/double(dimzX);
    intervY = (max(Ytable)-min(Ytable))*double(N)/double(dimzY);
    dimzX = dimzY = N;
    a = -10 - 2*b + 6; // ???
    V = array(span(-0.5*(N+1+a)/intervX,0.5*(N-1+a)/intervX,N),N);
    U = transpose(array(span(-0.5*(N+1+a)/intervY,0.5*(N-1+a)/intervY,N),N));

    U_FFT = U;
    V_FFT = V;

    return 1;
}

/***************************************************************************/

func imageFFT_COMPUTE(Utable, Vtable, lambda, U_FFT, V_FFT, FFT, &cf, &fl, inputFlux=)
    /* DOCUMENT imageFFT_COMPUTE(Utable, Vtable, lambda, U_FFT, V_FFT, FFT, &cf, &fl, inputFlux=)
   
       DESCRIPTION
       Returns the Discrete Fourier Transform of an image at the (U,V)
       coordinates. Use this method if you have few (U,V) points, since
       it is then faster than fft
   
       PARAMETERS
       - Utable   : U coordinates (meters)
       - Vtable   : V coordinates (meters)
       - lambda   : wavelength (meters)
       - U_FFT    : 
       - V_FFT    : 
       - FFT      : 
       - cf       : 
       - fl       : 
       - inputFlux: 

       RESULTS
       DFT at points (U,V)

       CAUTIONS
       BE CAREFUL! FFT, U_FFT and V_FFT are supposed to be conformed
       with numberof(lambda) as the wavelength dimension.

       EXAMPLES
       See testImage function

       SEE ALSO: imageFFT_INIT, imageFFT
    */
{
    // Get number of wavelengths and various other dimensions
    nbWlen = numberof(lambda);
    dimzu  = dimsof(Utable);
    dimzv  = dimsof(Vtable);
    
    if(is_void(inputFLux))
        inputFlux = array(1.0,nbWLen);

    // Initialize vis array
    vis = array(complex,numberof(Utable), nbWlen);

    // Compute spatial frequencies
    Uobs = reform(Utable, numberof(Utable))(..,-)/lambda(-,);
    Vobs = reform(Vtable, numberof(Vtable))(..,-)/lambda(-,);

    // Interpolate visibilities
    for(k=1;k<=nbWlen;k++)
    {
        vis(,k).re = interp2(Uobs(,k), Vobs(,k),
                             FFT(,,k).re, U_FFT, V_FFT);
        vis(,k).im = interp2(Uobs(,k), Vobs(,k),
                             FFT(,,k).im, U_FFT, V_FFT);
    }
    
    dimzu(1)++;
    vis = reform(vis,grow(dimzu,nbWlen));
    cf = vis * inputFlux(-,);
    fl = inputFlux;
    
    return vis;
}

/***************************************************************************/

func imageFFT(Utable, Vtable, lambda, imaWlen, image, Xtable, Ytable, &cf, &fl, &U_FFT, &V_FFT, &FFT, upSample=, interpFirst=)
    /* DOCUMENT imageFFT(Utable, Vtable, lambda, imaWlen, image, Xtable, Ytable, &cf, &fl, &U_FFT, &V_FFT, &FFT, upSample=, interpFirst=)
   
       DESCRIPTION
       Returns the Fast Fourier Transform of an image at the (U,V)
       coordinates. Use this method if you have few (U,V) points, since
       it is then faster than fft

       PARAMETERS
       - Utable     : U coordinates (meters)
       - Vtable     : V coordinates (meters)
       - lambda     : wavelength (meters)
       - imaWlen    : 
       - image      : input image
       - Xtable     : X coordinates of the image
       - Ytable     : Y coordinates of the image
       - cf         : (output) coherent flux
       - fl         : (output) flux
       - U_FFT      : (output) U coordinates of the 2D map
       - V_FFT      : (output) V coordinates of the 2D map
       - FFT        : (output) 2D FFT map
       - upSample   : upsampling for zero padding
       - interpFirst: 

       RESULTS
       DFT at points (U,V)

       CAUTIONS
       Very slow when a lot of (U,V) points are used.

       EXAMPLES
       See testImage function

       SEE ALSO:
    */
{
    // local U_FFT, V_FFT, FFT;
    interpFirst=1;

    if(interpFirst)
    {
        if(numberof(imaWlen)>1)
            ima2 = interp(image, imaWlen, lambda, 0);
        else
            ima2 = array(image(..,1),numberof(lambda));
        
        Xtable2 = array(Xtable, numberof(lambda));
        Ytable2 = array(Ytable, numberof(lambda));
    }
    else
    {
        ima2 = image;
        Xtable2 = array(Xtable, numberof(imaWlen));
        Ytable2 = array(Ytable, numberof(imaWlen));
    }
    
    imageFFT_INIT, ima2, Xtable2, Ytable2, U_FFT, V_FFT, FFT, flux, upSample=upSample;

    if(!interpFirst)
    {
        if(numberof(imaWlen)==1)
        {
            FFT    = array(complex, grow(dimsof(FFT)(:-1),numberof(lambda)));
            FFT.re = array(FFT.re(..,1), numberof(lambda));
            FFT.im = array(FFT.im(..,1), numberof(lambda));
        }
        else
        {
            FFT    = array(complex, grow(dimsof(FFT)(:-1),numberof(lambda)));
            FFT.re = interp(FFT.re, imaWlen, lambda, 0);
            FFT.im = interp(FFT.im, imaWlen, lambda, 0);
        }
    }
    
    vis = imageFFT_COMPUTE(Utable, Vtable, lambda, U_FFT, V_FFT, FFT, cf, fl, inputFlux=flux);

    return vis;
}

/***************************************************************************/

func imageDFT(Utable, Vtable, lambda, image, Xtable, Ytable)
    /* DOCUMENT imageDFT(Utable, Vtable, lambda, image, Xtable, Ytable)
   
       DESCRIPTION
       Returns the Discrete Fourier Transform of an image at the (U,V)
       coordinates. Use this method if you have few (U,V) points, since
       it is then faster than fft

       PARAMETERS
       - Utable: U coordinates (meters)
       - Vtable: V coordinates (meters)
       - lambda: wavelength (meters)
       - image : input image
       - Xtable: X coordinates of the image
       - Ytable: Y coordinates of the image

       RESULTS
       DFT at points (U,V)

       CAUTIONS
       Very slow when a lot of (U,V) points are used.

       EXAMPLES
       See testImage function

       SEE ALSO:
    */
{
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);

    C_out = array(complex,dimsof(u));
    C_out = (image(-,,,) * exp(- 2i*pi*(u(,-,-,)*Xtable(-,,,)+
                                        v(,-,-,)*Ytable(-,,,))))(,sum,sum,) /
        (image(-,,,))(,sum,sum,);
    
    return C_out;
}

/***************************************************************************/

func shiftFourier(Utable, Vtable, lambda, C_in, x0, y0, &C_out)
    /* DOCUMENT shiftFourier(Utable, Vtable, lambda, C_in, x0, y0, &C_out)
       
       DESCRIPTION
       Apply a image-space shift to the Fourier-transform of the image
    
       PARAMETERS
       - Utable: U coordinates (meters)
       - Vtable: V coordinates (meters)
       - lambda: wavelength (meters)
       - C_in  : the input array of the complex values of the Fourier transform
       - x0    : The x shift to apply
       - y0    : the y shift to apply
       - C_out : the returned complex values of the shifted Fourier transform
    */
{
    if(is_void(x0))
        x0 = 0;
    if(is_void(y0))
        y0 = 0;
  
    u = pi*Utable(,-) / lambda(-,);
    v = pi*Vtable(,-) / lambda(-,);
  
    C_out = C_in * exp(- 2i*(u*x0(-,)+v*y0(-,)));
    return C_out;
}

/***************************************************************************/

func visBinaryAndWWCZ(Utable, Vtable, lambda, time, fluxStar1, fluxStar2, type1, type2, size1, size2, sep, pa, fluxWWCZ, dstWWCZ, sizWWCZ, thickWWCZ, factWWCZ, powerWWCZ)
    /* DOCUMENT visBinaryAndWWCZ(Utable, Vtable, lambda, time, fluxStar1, fluxStar2, type1, type2, size1, size2, sep, pa, fluxWWCZ, dstWWCZ, sizWWCZ, thickWWCZ, factWWCZ, powerWWCZ)

       DESCRIPTION 
       model of a wind-wind collision zone and a binary star

       PARAMETERS
       - Utable   : U coordinates (meters)
       - Vtable   : V coordinates (meters)
       - lambda   : wavelength (meters)
       - time     : 
       - fluxStar1: 
       - fluxStar2: 
       - type1    : 
       - type2    : 
       - size1    : 
       - size2    : 
       - sep      : 
       - pa       : 
       - fluxWWCZ : 
       - dstWWCZ  : 
       - sizWWCZ  : 
       - thickWWCZ: 
       - factWWCZ : 
       - powerWWCZ: 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    typei = [type1, type2];
    fwhm1 = [size1, size2];
    fwhm2 = [0,0];
    angle = [0,0];
    x0 = [0,sep*cos(pa)];
    y0 = [0,sep*sin(pa)];
    spectrumi = [fluxStar1,fluxStar2];

    visStars = visMultipleResolved(Utable, Vtable, lambda, time, typei, spectrumi, x0, y0, fwhm1, fwhm2, angle);
    
    anglWWCZ = pa;
    X0WWCZ = sep*cos(pa)*dstWWCZ;
    Y0WWCZ = sep*sin(pa)*dstWWCZ;

    vis = (visStars + fluxWWCZ(-,) * visWWCZ(Utable, Vtable, lambda, time,
                                             sizWWCZ, thickWWCZ, factWWCZ,
                                             powerWWCZ, anglWWCZ,
                                             X0WWCZ, Y0WWCZ)) /
        (1 + fluxWWCZ(-,));
    
    return vis;
}

/***************************************************************************/

func visDilutePinwheel(Utable, Vtable, lambda, time, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, tailFlux, diluteFlux, diluteSize, type=, fillFactor=)
    /* DOCUMENT visDilutePinwheel(Utable, Vtable, lambda, time, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, tailFlux, diluteFlux, diluteSize, type=, fillFactor=)

       DESCRIPTION

       PARAMETERS
       - Utable     : 
       - Vtable     : 
       - lambda     : 
       - time       : 
       - rounds     : 
       - minThick   : 
       - maxThick   : 
       - totalSize  : 
       - power      : 
       - anglePhi   : 
       - inclination: 
       - angleSky   : 
       - tailFlux   : 
       - diluteFlux : 
       - diluteSize : 
       - type       : 
       - fillFactor : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    return (diluteFlux(-,) *
            visGaussianDisk(Utable, Vtable, lambda, time, diluteSize) +
            (1-diluteFlux(-,)) *
            visPinwheel(Utable, Vtable, lambda, time, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, tailFlux, type=, fillFactor=fillFactor));
    //return visPinwheel(Utable, Vtable, lambda, time, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, tailFlux, type=, fillFactor=)
}

/***************************************************************************/

// Pinwheel
if(growModels(ALL_MODELS, "pwhl"))
{
    ALL_MODELS.description(0) = "Simple pinwheel";
    ALL_MODELS.params(0) = &["flux", "rounds", "minThick", "maxThick", "totalSize", "power", "anglePhi", "inclination", "angleSky", "tailFlux"];
    ALL_MODELS.units(0) = &["no", "no", "mas", "mas", "mas", "no", "degrees", "degrees", "degrees", "no"];
}    

func visPinwheel(Utable, Vtable, lambda, time, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, tailFlux, type=, fillFactor=, verbose=)
    /* DOCUMENT visPinwheel(Utable, Vtable, lambda, time, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, tailFlux, type=, fillFactor=, verbose=)

       DESCRIPTION
       Pinwheel model

       PARAMETERS
       - Utable     : 
       - Vtable     : 
       - lambda     : 
       - time       : 
       - rounds     : 
       - minThick   : 
       - maxThick   : 
       - totalSize  : 
       - power      : 
       - anglePhi   : 
       - inclination: 
       - angleSky   : 
       - tailFlux   : 
       - type       : 
       - fillFactor : 
       - verbose    : 

       RESULTS 

       CAUTIONS

       EXAMPLES 

       SEE ALSO:
    */
{
    if(is_void(type))
        type="r2";

    rounds = abs(rounds);
    
    nbWlen = numberof(lambda);
    if(is_void(fillFactor))
        fillFactor = 10.0;
    thick = avg((minThick+maxThick)/2.0);
    N = 20000;
    
    theta = span(0,rounds*2*pi,N);

    fact = totalSize*rounds/(rounds*2*pi/(2*pi))^power;
    
    r = fact(-,)/rounds(-,)*(((theta+(theta==0))/(2*pi))^power(-,) - (theta==0)/(2*pi));

    dst = r*theta;

    toto = (abs((dst%(thick/fillFactor))(dif,))>
            thick/fillFactor/2);

    titi = transpose(toto);

    tutu = grow(array(0,nbWlen,1),titi);

    test = transpose(tutu)(,1);
    
    pr2 = where2((abs(dst)==max(abs(dst))) |
                 (abs(dst)==min(abs(dst))) |
                 test)(1,);
    pr2 = pr2(2:);
    N2 = numberof(pr2);
    while(N2>1000)
    {
        if(verbose)
            write,"Dividing number of rings from "+ pr1(N2)+" to "+ pr1(N2/2)
                pr2 = pr2(::2);
        N2 = numberof(pr2);
    }

    ruse   = array(r(pr2),nbWlen);
    tetuse = array(theta(pr2),nbWlen);
        
    typei  = array(type,N2);
    
    thick  = minThick(-,) + ruse / (max(r) + (max(r)==0))* maxThick(-,);
    
    angle0 = tetuse + anglePhi(-,);
    x0     = ruse*cos(angle0);
    y0     = ruse*sin(angle0);
    fwhmx0 = array(0, N2 ,nbWlen);
    fwhmy0 = thick;

    x1     = x0 * cos(inclination(-,));
    y1     = y0;
    angle1 = atan(x1,y1);
    fwhmx1 = fwhmy0*cos(angle0)*sin(inclination(-,));
    fwhmy1 = fwhmy0;
    
    angle2 = transpose(angle1+angleSky(-,));
    x2     = transpose( x1*cos(angleSky(-,))+y1*sin(angleSky(-,)));
    y2     = transpose(-x1*sin(angleSky(-,))+y1*cos(angleSky(-,)));
    fwhmx2 = transpose(fwhmx1);
    fwhmy2 = transpose(fwhmy1);
    
    spectrumi = array(0.0,nbWlen,N2);
    for(k=1;k<=nbWlen;k++)
        spectrumi(k,) = span(1, tailFlux(k), N2);

    vis = visMultipleResolved(Utable, Vtable,
                              lambda, time, typei,
                              spectrumi,
                              x2, y2, fwhmx2, fwhmy2,
                              angle2);

    return vis;
}

/***************************************************************************/

// Pinwheel
if(growModels(ALL_MODELS, "2pwhl"))
{
    ALL_MODELS.description(0) = "Double pinwheel";
    ALL_MODELS.params(0) = &["flux", "rounds", "minThick", "maxThick", "totalSize", "power", "anglePhi", "inclination", "angleSky", "tailFlux"];
    ALL_MODELS.units(0) = &["no", "no", "mas", "mas", "mas", "no", "degrees", "degrees", "degrees", "no"];
}    

func visDoublePinwheel(Utable, Vtable, lambda, time, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, tailFlux, type=, fillFactor=, verbose=)
    /* DOCUMENT visDoublePinwheel(Utable, Vtable, lambda, time, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, tailFlux, type=, fillFactor=, verbose=)

       DESCRIPTION
       Double Pinwheel model

       PARAMETERS
       - Utable     : 
       - Vtable     : 
       - lambda     : 
       - time       : 
       - rounds     : 
       - minThick   : 
       - maxThick   : 
       - totalSize  : 
       - power      : 
       - anglePhi   : 
       - inclination: 
       - angleSky   : 
       - tailFlux   : 
       - type       : 
       - fillFactor : 
       - verbose    : 

       RESULTS 

       CAUTIONS

       EXAMPLES 

       SEE ALSO:
    */
{
    if(is_void(type))
        type="r2";
    
    if(is_void(fillFactor))
        fillFactor = 1.0;
    
    vis1 = visPinwheel(Utable, Vtable, lambda, time, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, tailFlux, type=, fillFactor=fillFactor, verbose=);

    vis2 = visPinwheel(-Utable, -Vtable, lambda, time, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, tailFlux, type=, fillFactor=fillFactor, verbose=);

    vis = (vis1 + vis2)/2;
    
    return vis;
}

/***************************************************************************/

func visPinwheelTempLaw(Utable, Vtable, lambda, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, powerLawDecrease, tempInit, periodSine, amplitudeSine, amortiSine, type=, fillFactor=, verbose=)
    /* DOCUMENT visPinwheelTempLaw(Utable, Vtable, lambda, rounds, minThick, maxThick, totalSize, power, anglePhi, inclination, angleSky, powerLawDecrease, tempInit, periodSine, amplitudeSine, amortiSine, type=, fillFactor=, verbose=)

       DESCRIPTION
       Pinwheel model with temperature law (not working yet)

       PARAMETERS
       - Utable          : 
       - Vtable          : 
       - lambda          : 
       - rounds          : nb de tours
       - minThick        : 
       - maxThick        : 
       - totalSize       : Extension totale de la spirale. Noté S dans le schema p L51 de Florentin et al A&A 2009
       - power           : 
       - anglePhi        : 
       - inclination     : 
       - angleSky        : 
       - powerLawDecrease: 
       - tempInit        : 
       - periodSine      : 
       - amplitudeSine   : 
       - amortiSine      : 
       - type            : 
       - fillFactor      : 
       - verbose         : 

       RESULTS 

       CAUTIONS

       EXAMPLES 

       SEE ALSO:
    */
{

    if(is_void(type))
        type="r2";
    
    nbWlen = numberof(lambda);
    print,"nbWlen",nbWlen
        print,lambda

        if(is_void(fillFactor))
            fillFactor = 10.0;//var donnant par unite de mas le nb d´anneau le long de la spirale
    // nb d´anneaux / tour ou mas
    thick = avg((minThick+maxThick)/2.0);//
    N = 20000;
    
    theta = span(0,rounds*2*pi,N);

    //fact = totalSize*rounds/(rounds*2*pi/(2*pi))^power;//
    fact = totalSize*rounds/(rounds*2*pi/(2*pi))^power;//totalsize : extension totale de la spirale
    // si pas spirale d´archimede. Taille * le nb de tours
    // si pas spirale d´archimede, normalisation /rapport à la loi de puissance à faire.
    // regarder ce que ça donne quand power est changé : plus archimede
    //avec power=1 (spirale d´archimede) 
   
    r = fact(-,)/rounds(-,)*(((theta+(theta==0))/(2*pi))^power(-,) - (theta==0)/(2*pi));

    // r -> distance au centre
    // totalsize/rounds=pas de la spirale == 2.pi.a sachant que r=a.theta
    //note:
    // x(-,) : reajuste dimension de tableau : fact = 1D -> r=2D
    // pour faire un tableau à 2 dim / rapport à theta regarder la tête du tableau
    // pour pouvoir construire pinwhel avec 1D en lambda    
    //1 r par lambda
    //pas utilisé

    dst = r*theta;
    //abscisse curviligne

    // La suite determine sur la spirale ou vont se trouver les anneaux sur l´abscisse curviligne (dst) :
    toto = (abs((dst%(thick/fillFactor))(dif,))>
            thick/fillFactor/2);//defini les positions des anneaux
    //Thick:min and maxthick taille mini et max des anneaux
    //Thick epaisseur moyenne: - d anneaux si anneaux plus gros
    // Fillfactor : nb d anneau par taille moyenne d anneau


    //reformatage du tableau avec des bonnes dimensions    
    titi = transpose(toto);
    tutu = grow(array(0,nbWlen,1),titi);//grow : augmente dimensions
    test = transpose(tutu)(,1);
    
    pr2 = where2((abs(dst)==max(abs(dst))) |
                 (abs(dst)==min(abs(dst))) |
                 test)(1,);
    // | comparaison c est 'ou' element par element
    // || comparaison elemtn global
    pr2 = pr2(2:); //index ou vont se trouver les anneaux sur les 20000 elements

    //


    N2 = numberof(pr2);
    while(N2>1000)
    {
        if(verbose)
            write,"Dividing number of rings from "+ pr1(N2)+" to "+ pr1(N2/2)
                pr2 = pr2(::2);
        N2 = numberof(pr2);
    }

    ruse = array(r(pr2),nbWlen);// r utilisé pour positionner les anneaux
    // ce dont on a besoin pour la loi de temperature    

    tetuse = array(theta(pr2),nbWlen);// theta utilisé pour positionner les anneaux
        
    typei = array(type,N2);
    
    thick = minThick(-,) + ruse / (max(r) + (max(r)==0))* maxThick(-,);
    
    angle0 = tetuse + anglePhi(-,);
    x0 = ruse*cos(angle0);
    y0 = ruse*sin(angle0);
    fwhmx0 = array(0, N2 ,nbWlen);
    fwhmy0 = thick;

    x1 = x0 * cos(inclination(-,));
    y1 = y0;
    angle1 = atan(x1,y1);
    fwhmx1 = fwhmy0*cos(angle0)*sin(inclination(-,));
    fwhmy1 = fwhmy0;
    
    angle2 = transpose(angle1+angleSky(-,));
    x2 = transpose( x1*cos(angleSky(-,))+y1*sin(angleSky(-,)));
    y2 = transpose(-x1*sin(angleSky(-,))+y1*cos(angleSky(-,)));
    fwhmx2 = transpose(fwhmx1);
    fwhmy2 = transpose(fwhmy1);
    
    spectrumi = array(0.0,nbWlen,numberof(ruse));

    //temperature initiale
    //Tsub=1200.0;
    // Les temp des 2 premiers tours sont cte. pour le premier T=Tsub
    //Ttour1=Tsub;
    // Les temp des 2 premiers tours sont cte. pour le premier T=650K
    //Ttour2=1000.0;
    //voir Harries et al. 2004. p570

    // en radian, distance à partir de laquelle il ya de la poussière	
    r0=0.00133*4.85e-6;
    // initialisation du tableau de temperatures
    temperature_law=array(0.0,numberof(ruse(,1)));

    // Parametres en dur pour WR104
    // powerLawDecrease = -0.4;
    // tempInit      = 1200;
    // periodSine    = fact/rounds(1);
    // amplitudeSine = 200.;
    // amortiSine    = 0.005;

    rAU = ruse(,1)*rad2mas/1000*1.6e3;


    powerlaw = tempInit * (ruse(,1) / r0) ^ powerLawDecrease;
    variable = (-amplitudeSine * cos(2 * pi / periodSine * (ruse(,1)-r0)) +amplitudeSine) * exp(-amortiSine * rAU);
    
    temperature_law(where(ruse(,1)>=r0)) = (powerlaw + variable)(where(ruse(,1)>=r0));
    //temperature_law(where(ruse(,1)>=r0)) = ( variable)(where(ruse(,1)>=r0));
    //temperature_law(where(ruse(,1)>=r0)) = ( powerlaw)(where(ruse(,1)>=r0));

    window,33,wait=1;
    plg,temperature_law, rAU;

    print,"lambda",lambda; 
    for(k=1;k<=nbWlen;k++)
    {
        spectrumi(k, where(ruse(,1)>=r0)) = yocoAstroBBodyLambda(lambda(k), temperature_law(where(ruse(,1) >= r0)) );
    }
    //(temperature_law(r,lamda(k)))

    // pouet()
    
    vis = visMultipleResolved(Utable, Vtable,
                              lambda, time, typei,
                              spectrumi,
                              x2, y2, fwhmx2, fwhmy2,
                              angle2);

    return vis;
}

//visPinwheel = visPinwheelTempLaw;
    
/***************************************************************************/

func visWWCZ(Utable, Vtable, lambda, time, totalSize, thick, fact, power, angle, x0, y0)
    /* DOCUMENT visWWCZ(Utable, Vtable, lambda, time, totalSize, thick, fact, power, angle, x0, y0)

       DESCRIPTION
       Tentative model of a wind-wind collision zone

       PARAMETERS
       - Utable   : U coordinates (meters)
       - Vtable   : V coordinates (meters)
       - lambda   : wavelength (meters)
       - time     : 
       - totalSize: 
       - thick    : 
       - fact     : 
       - power    : 
       - angle    : 
       - x0       : 
       - y0       : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    nbWlen = numberof(lambda);
    fillFactor = 10;
    N2 = 10000;
    pr = span(-totalSize/2,totalSize/2,N2);
    kx0 = pr;
    ky0 = fact*(abs(kx0))^power;

    dkx0 = kx0(dif);
    dky0 = ky0(dif);
    ds = sqrt(dkx0^2+dky0^2);
    dst = ds(cum);
    dst = dst - max(dst)/2;

    //     dst = sign(kx0*ky0)*sqrt(kx0^2+ky0^2);

    kpr2 = pr(where(abs(dst) < totalSize/2));
    dst2 = dst(where(abs(dst) < totalSize/2));

    pr2 = where((dst2==max(dst2))|
                (dst2==min(dst2))|
                (abs(dst2)==min(abs(dst2)))|
                grow(0,(abs((dst2%(thick/fillFactor))(dif))>thick/fillFactor/2)));
    N = numberof(pr2);

    x = kpr2(pr2);
    y = fact*(abs(x+(abs(x)<1e-10)))^power;
    typei = array("g",N);
    fwhm1 = array(thick,N);
    fwhm2 = array(0.0,nbWlen,N);
    angle2 = array(0.0,nbWlen,N);
    x1 = transpose(array(-sin(angle)*x + cos(angle)*y,nbWlen));
    y1 = transpose(array(cos(angle)*x + sin(angle)*y,nbWlen));
    spectrumi = array(1.0,nbWlen,N);
  
    vis = visMultipleResolved(Utable, Vtable,
                              lambda, time, typei, spectrumi, x1, y1,
                              fwhm1, fwhm2,
                              angle2);

    C = shiftFourier(Utable,Vtable,lambda,vis,x0,y0);

    return C;
}

/************************************************************************/
        
if(growModels(ALL_MODELS, "orb"))
{
    ALL_MODELS.description(0) = "Orbital-moving target";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "diam", "type", "t0", "eccentricity", "majorAxis", "period", "i", "omega", "OMEGA"];
    ALL_MODELS.units(0)       = &["no", "mas", "mas","mas", "no", "d", "no", "mas", "d", "deg", "deg", "deg"];
}  

func visOrbit(Utable, Vtable, lambda, time, size, type, t0, eccentricity, majorAxis, period, i, omega, OMEGA, x0, y0, &flx)
    /* DOCUMENT visOrbit(Utable, Vtable, lambda, time, size, type, t0, eccentricity, majorAxis, period, i, omega, OMEGA, x0, y0, &flx)

       DESCRIPTION
       Model with orbital elements motion

       PARAMETERS
       - Utable      : 
       - Vtable      : 
       - lambda      : 
       - time        : 
       - size        : 
       - type        : 
       - t0          : 
       - eccentricity: 
       - majorAxis   : 
       - period      : 
       - i           : 
       - omega       : 
       - OMEGA       : 
       - x0          : 
       - y0          : 
       - flx         : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // write,Utable(1), Vtable(1),diam(1),x0(1)*rad2mas,y0(1)*rad2mas;
    
    // Calculate Kepler coordinates
    CoordCenR = binaryProjectedCoordinates(time/3600/24,
                                           t0(1),
                                           eccentricity(1),
                                           majorAxis(1),
                                           period(1),
                                           i(1),
                                           omega(1),
                                           OMEGA(1));

    kuv=stophere=0;
    C_cen = [];
    // Loop on UV coordinates as the time can be different
    while(stophere!=1)
    {
        kuv++;
        if(numberof(time)==1)
        {
            // Get plane-of-sky coordinates
            x1 = CoordCenR(1,2);
            y1 = CoordCenR(1,1);
            kuv = [];
            stophere = 1;
        }
        else
        {
            x1 = CoordCenR(,2);
            y1 = CoordCenR(,1);
        }

        //        write,x1(1)*rad2mas,y1(1)*rad2mas;
        
        // Define what to put at the position of the calculated coordinates
        g    = 1;
        ud   = 2;
        wr11 = 3;
        
        if(type(1)%4 <= g)
        {
            grow,C_cen,
                [visGaussianDisk(Utable(kuv), Vtable(kuv), lambda, time(kuv),
                                 size, x1(kuv), y1(kuv))];
            flx = [];
        }
        else if(type(1)%4 <= ud)
        {
            grow,C_cen,
                [visUniformDisk(Utable(kuv), Vtable(kuv), lambda, time(kuv),
                                size, x1(kuv), y1(kuv))];
            flx = [];
        }
        else
        {    
            dst          = 368.; // Distance in parsecs
            dir          = HOME+"/ARTICLES/AMOI/NF_GAMMA_VEL/WR_Model/";
            fileDessart  = dir+"wr11_1_tau_PxIP_nearir.dat";
            extern dessart_lam, dessart_p, dessart_Ip;
            if(is_void(dessart_lam))
                impact_read, fileDessart, dessart_lam, dessart_p, dessart_Ip;
            pap   = 30*size(1) / mas2rad * dessart_p * yocoAstroSI.Rsun / yocoAstroSI.pc / dst;
            grow,C_cen,
                [visProfile(Utable(kuv),
                            Vtable(kuv),
                            lambda,
                            time(kuv),
                            dessart_Ip,
                            pap,dessart_lam/1e10,
                            x1(kuv),
                            y1(kuv),
                            flx)];
            flx   = flx / size(1)^2 * 1e2/1.89330;
            window,7;
            plg,flx;
        }
        if(kuv == numberof(time))
            stophere=1;
    }
    //     if(numberof(time)!=1)
    // prout()
    
    //    info,C_cen;
    if(numberof(time)!=1)
        C_cen = transpose(C_cen);
    //    info,C_cen;
    //    write,"************"
    C = shiftFourier(Utable,Vtable,lambda,C_cen,x0,y0);
    
    return C;
}

/************************************************************************/
        
if(growModels(ALL_MODELS, "gam_wcz"))
{
    ALL_MODELS.description(0) = "Wind collision zone of Gamma Vel";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "zoom", "t0", "eccentricity", "majorAxis", "period", "i", "omega", "OMEGA"];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "no", "d", "no", "mas", "d", "deg", "deg", "deg"];
}  

func visGamWCZ(Utable, Vtable, lambda, time, zoom, t0, eccentricity, majorAxis, period, i, omega, OMEGA, x0, y0)
    /* DOCUMENT visGamWCZ(Utable, Vtable, lambda, time, t0, eccentricity, majorAxis, period, i, omega, OMEGA, x0, y0)

       DESCRIPTION
       Gamma Vel model + wind-wind collision zone

       PARAMETERS
       - Utable      : 
       - Vtable      : 
       - lambda      : 
       - time        : 
       - t0          : 
       - eccentricity: 
       - majorAxis   : 
       - period      : 
       - i           : 
       - omega       : 
       - OMEGA       : 
       - x0          : 
       - y0          : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // Parameters of the old simulation from Astrid
    // old = 1;
    // wcz_params0 = [50120.4, 0.337, 247.7, 67, 65, 78.53, 3.57];
    // Mwr = 9;
    // Mo  = 28.5;
    // dir = HOME+"/ARTICLES/AMOI/NF_GAMMA_VEL/Hydro_Model/florentin_final/";
    // fileLamberts1 = dir+"flux_20s_"+pr1(int(iFile1))+".fits";
    // fileLamberts2 = dir+"flux_20s_"+pr1(int(iFile2))+".fits";
    // n_phi = 10;
    // o_phi = 3;

    // Parameters of the new simulation from Astrid
    old = 0;
    wcz_params0 = [50120.4, 0.337, 245, 67, 65, 78.53, 3.57];
    Mwr = 9;
    Mo  = 28.5;
    dir = HOME+"/ARTICLES/AMOI/NF_GAMMA_VEL/Hydro_Model/flux_florentin_new/";
    n_phi = 100
    o_phi = 0;
    
    // New values
    T02           = wcz_params0(1);
    eccentricity2 = wcz_params0(2);
    OMEGA2        = wcz_params0(3) * yocoAstroSI.deg;
    omega2        = wcz_params0(4) * yocoAstroSI.deg;
    i2            = wcz_params0(5) * yocoAstroSI.deg;
    period2       = wcz_params0(6);
    majorAxis2    = wcz_params0(7) * yocoAstroSI.mas;

    //    write,T02, eccentricity2,majorAxis2,period2,i2,omega2,OMEGA2;
    //    write,t0(1), eccentricity(1),majorAxis(1),period(1),i(1),omega(1),OMEGA(1);

    // Coordinates of the simulated WCZ
    CoordCenR0 = binaryProjectedCoordinates(time/3600/24,
                                            T02,
                                            eccentricity2,
                                            majorAxis2,
                                            period2,
                                            i2,
                                            omega2,
                                            OMEGA2,
                                            phi2);

    // Coordinates of actual stars
    CoordCenR = binaryProjectedCoordinates(time/3600/24,
                                           t0(1),
                                           eccentricity(1),
                                           majorAxis(1),
                                           period(1),
                                           i(1),
                                           omega(1),
                                           OMEGA(1),
                                           phi);

    

    phi  /= 2*pi;
    if(phi(1)<0)
        phi+=1;
    phi2 /= 2*pi;
    if(phi2(1)<0)
        phi2+=1;
    //    write,phi,modulo(phi,1),phi2,modulo(phi2,1);

    fracp = phi2(1)-int(phi2(1));
    
    //write,int(iFile1),int(iFile2);
    if(old == 1)
    {
        iFile1 = fracp*n_phi+o_phi;
        iFile2 = fracp*n_phi+o_phi+1;
        
        fileLamberts1 = dir+"flux_20s_"+pr1(int(iFile1))+".fits";
        fileLamberts2 = dir+"flux_20s_"+pr1(int(iFile2))+".fits";
    }
    if(old == 0)
    {
        iFile1 = int(fracp*20);
        iFile2 = int(fracp*20+1);
        if(iFile2>n_phi+o_phi-1)
            iFile2 = o_phi;
        iFile1 = double(iFile1) / 20.
        iFile2 = double(iFile2) / 20.
        
        fileLamberts1 = dir+"flux_2.2sabs_phi"+swrite(iFile1,format="%0.2f")+".fits";
        fileLamberts2 = dir+"flux_2.2sabs_phi"+swrite(iFile2,format="%0.2f")+".fits";
    }
    
    // dir = HOME+"/ARTICLES/AMOI/NF_GAMMA_VEL/Hydro_Model/florentin_final/";
    // fileLamberts1 = dir+"flux_20s_"+pr1(int(iFile1))+".fits";
    // fileLamberts2 = dir+"flux_20s_"+pr1(int(iFile2))+".fits";
    

    require,"miral.i";
    miral_read_imgCube, fileLamberts1,ima1,x1,y1;
    miral_read_imgCube, fileLamberts2,ima2,x2,y2;

    // if(old == 0)
    // {
    //     x1 = x1*mas2rad;
    //     y1 = y1*mas2rad;
    // }
    

    // Get the expected position of the O star in the simulation
    yO2  = CoordCenR0(1)*Mwr/(Mwr+Mo);
    xO2  = CoordCenR0(2)*Mwr/(Mwr+Mo);
    pxO2 = xO2 / x1(dif)(avg);
    pyO2 = yO2 / y1(dif)(avg);
    ima1 = roll(ima1,[lround(pyO2),lround(pxO2)]);
    ima2 = roll(ima2,[lround(pyO2),lround(pxO2)]);

    //    write,pxO2,pyO2;
    
    fraci = iFile1 - int(iFile1);
    ima = [(ima1 * (1-fraci) + ima2 * fraci) / 2.0];
    wl = 2.2e-6;

    dim = dimsof(ima1);
    dimx = dim(2);
    dimy = dim(3);
    
    Xtable = array(x1*zoom(1),dimy,1);
    Ytable = [transpose(array(y1*zoom(1),dimx))];

    vis = imageFFT(Utable, Vtable, lambda, wl, ima, Xtable, Ytable);
    return vis;
}

/************************************************************************/
        
if(growModels(ALL_MODELS, "orb2"))
{
    ALL_MODELS.description(0) = "Orbital-moving binary";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "flux ratio", "diam1", "diam2", "type1", "type2", "t0", "eccentricity", "majorAxis", "period", "i", "omega", "OMEGA"];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "no", "mas", "mas", "no", "no", "d", "no", "mas", "d", "deg", "deg", "deg"];
}  

func visOrbit2(Utable, Vtable, lambda, time, fxRat, size1, size2, type1, type2, t0, eccentricity, majorAxis, period, i, omega, OMEGA, x0, y0)
    /* DOCUMENT visOrbit2(Utable, Vtable, lambda, time, fxRat, size1, size2, type1, type2, t0, eccentricity, majorAxis, period, i, omega, OMEGA, x0, y0)

       DESCRIPTION
       Model with two components orbiting each other

       PARAMETERS
       - Utable      : 
       - Vtable      : 
       - lambda      : 
       - time        : 
       - fxRat       : 
       - size1       : 
       - size2       : 
       - type1       : 
       - type2       : 
       - t0          : 
       - eccentricity: 
       - majorAxis   : 
       - period      : 
       - i           : 
       - omega       : 
       - OMEGA       : 
       - x0          : 
       - y0          : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    vis1 = visOrbit( Utable,  Vtable, lambda, time, size1, type1, t0, eccentricity, 0,         period, i, omega, OMEGA, x0, y0, flx1);
    vis2 = visOrbit(-Utable, -Vtable, lambda, time, size2, type2, t0, eccentricity, majorAxis, period, i, omega, OMEGA, x0, y0, flx2);

    C_centered = vis1 * flx1 * (1-fxRat) + vis2 * flx2 * fxRat;
    flx = (flx1 * (1-fxRat) + flx2 * fxRat);
    C_centered = C_centered / flx;
          
    C = shiftFourier(Utable,Vtable,lambda,C_centered,x0,y0);

    return C;
}

/************************************************************************/

func visProfile(Utable, Vtable, lambda, time, yProfile, xProfile, lProfile, x0, y0, &flu_x)
    /* DOCUMENT visProfile(Utable, Vtable, lambda, time, yProfile, xProfile, lProfile, x0, y0, &flu_x)

       DESCRIPTION

       PARAMETERS
       - Utable     : 
       - Vtable     : 
       - lambda     : 
       - time       : 
       - yProfile   : 
       - xProfile   : 
       - lProfile   : 
       - x0         : 
       - y0         : 
       - flu_x      : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{

    //     write,"*******************************"
    //     write,x0*rad2mas,y0*rad2mas;
    //     write,abs(x0,y0)*rad2mas,rad2deg*atan(x0,y0);
    
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);

    if(dimsof(yProfile)(1)==2)
        ypr = interp(yProfile,lProfile,lambda,2);
    else
        ypr = yProfile;

    //flu_x = ypr(sum,);

    N = dimsof(yProfile)(2);
    C = array(0.0,dimsof(u));
    FLX = array(0.0,dimsof(lambda));
    
    for(k=2;k<=N;k++)
    {
        x1p  = xProfile(k-1);
        x2p  = xProfile(k);
        maj  = (x1p+x2p)/2;
        thk  = (x2p-x1p)/2;        
        flux = ypr(k,);
        
        Ctmp = visThickRing(Utable, Vtable, lambda, time, maj, thk,x0,y0,"ud", flx);
        //Ftmp = visThickRing(0, 0, lambda, time, maj, thk,0,0,"ud", flx);
        C   = C + flux(-,) * Ctmp * flx;
        FLX = FLX + flux * flx;
    }
    
    C_centered = C / FLX(-,);
    flu_x = FLX;
    C = C_centered;//shiftFourier(Utable,Vtable,lambda,C_centered,x0,y0);
    
    return C;
}


/***************************************************************************/
             
if(growModels(ALL_MODELS, "ldl"))
{
    ALL_MODELS.description(0) = "Circular linear limb-darkened disk";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "diam", "ulam"];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "no"];
}

func visLimbLinear1(Utable, Vtable, lambda, time, diam, x0, y0, epsilon)
    /* DOCUMENT visLimbLinear1(Utable, Vtable, lambda, time, diam, x0, y0, epsilon)

       DESCRIPTION
       Linear limb darkening model
       According to Hanbury Brown et al. 1974

       PARAMETERS
       - Utable : 
       - Vtable : 
       - lambda : 
       - time   : 
       - diam   : 
       - x0     : 
       - y0     : 
       - epsilon: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
  
    r = abs(u, v);

    alpha = 1 - epsilon;
    beta = epsilon;
    x     = pi * abs(diam)(-,) * r;

    // Avoid floating points
    xnez = (x+(x==0));

    // Eq. 4 of H. Brown et al. 1974
    C_centered = (x==0) + 
        (x!=0) * (alpha(-,) / 2. + beta(-,) / 3.)^-1 *
        (alpha(-,) * bessj(1, x) / xnez +
         beta(-,) * sqrt(pi/2) * bessj(3./2., xnez) / (xnez^(3./2.)));

    C = shiftFourier(Utable, Vtable, lambda, C_centered, x0, y0);

    return C;
}

visLimbLinear = visLimbLinear1;

/***************************************************************************/
             
if(growModels(ALL_MODELS, "ldl"))
{
    ALL_MODELS.description(0) = "Circular linear limb-darkened disk";
    ALL_MODELS.params(0)         = &["flux", "x", "y", "diam", "ulam"];
    ALL_MODELS.units(0)          = &["no", "mas", "mas", "mas", "no"];
}

func visLimbLinear0(Utable, Vtable, lambda, time, diam, x0, y0, epsilon)
    /* DOCUMENT visLimbLinear0(Utable, Vtable, lambda, time, diam, x0, y0, epsilon)

       DESCRIPTION
       Linear limb darkening model (not working?)

       PARAMETERS
       - Utable : 
       - Vtable : 
       - lambda : 
       - time   : 
       - diam   : 
       - x0     : 
       - y0     : 
       - epsilon: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    R        = abs(Utable, Vtable);
    maxR     = max(R);
    minW     = min(lambda)/maxR;
    upSample = 10;
    nData    = max(int(diam * upSample / minW), 1)(1);
    
    xProfile     = span(0,diam,nData);
    x2           = span(0,1,nData);
    yProfile     = 1.0 - epsilon(,-) * (1 - sqrt(1 - x2(-,)^2));
    majToMinAxes = 1.0;
    angle        = 0.0;
    
    vis = visEllipticalProfile(Utable, Vtable, lambda, time, yProfile, xProfile, majToMinAxes, angle, x0, y0);
    return vis;
}

/***************************************************************************/
             
func visEllipticalProfile(Utable, Vtable, lambda, time, yProfile, xProfile, majToMinAxes, angle, x0, y0)
    /* DOCUMENT visEllipticalProfile(Utable, Vtable, lambda, time, yProfile, xProfile, majToMinAxes, angle, x0, y0)

       DESCRIPTION
       Computes the visibility of a Hankel function

       PARAMETERS
       - Utable      : U coordinates (meters)
       - Vtable      : V coordinates (meters)
       - lambda      : wavelength (meters)
       - time        : 
       - yProfile    : yProfile must have [nb(xProfile), nb(lambda)] dimension
       - xProfile    : the coordinates of each profile point (radians)
       - majToMinAxes: 
       - angle       : 
       - x0          : 
       - y0          : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);

    N = dimsof(yProfile)(2);
    C = array(0.0,dimsof(u));

    for(k=1;k<=N;k++)
    {
        majorAxis = 2* xProfile(k,);
        minorAxis = 2* majToMinAxes * xProfile(k,);
        flux      = yProfile(k,);
        C = C + flux(-,) * 
            visEllipticalRing(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0);
    }
    
    C = C / yProfile(sum,)(-,);
            
    return C;
}

/***************************************************************************/
            
if(growModels(ALL_MODELS, "pwrdsk2"))
{
    ALL_MODELS.description(0) = "Elliptical Power law disk";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "majAxis", "minAxis", "pow", "P.A."];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "mas", "no", "deg"];
    if(AXES_DEF==USE_FLATTENING)
    {
        ALL_MODELS.params(0)      = &["flux", "x", "y", "majAxis", "flattening", "pow", "P.A."];
        ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "no", "no", "deg"];
    }
    else if(AXES_DEF==USE_INCLINATION)
    {
        ALL_MODELS.params(0)      = &["flux", "x", "y", "majAxis", "incl", "pow", "P.A."];
        ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "deg", "no", "deg"];
    }
}
if(growModels(ALL_MODELS, "pwrdsk"))
{
    ALL_MODELS.description(0) = "Circular power law disk";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "rin", "pow"];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "no"];
}

func visPowerDisk(Utable, Vtable, lambda, time, majAxis, minAxis, pow, PA, x0, y0)
    /* DOCUMENT visPowerDisk(Utable, Vtable, lambda, time, majAxis, minAxis, pow, PA, x0, y0)

       DESCRIPTION
       Power law disk

       PARAMETERS
       - Utable : 
       - Vtable : 
       - lambda : 
       - time   : 
       - majAxis: 
       - minAxis: 
       - pow    : 
       - PA     : 
       - x0     : 
       - y0     : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    extern AXES_DEF;
    R        = abs(Utable, Vtable);
    maxR     = max(R);
    minR     = min(R);
    minW     = min(lambda)/maxR;
    maxW     = max(lambda)/minR/1.;
    maxW     = 500*mas2rad;
    upSample = 0.5;
    rmax     = maxW;
    nData    = max(int(upSample * maxW / minW), 1)(1);
    nData    = 500;
    
    // FIXME: write,nData, majAxis, rmax;
    
    xProfile     = span(abs(majAxis),rmax,nData);
    x2           = span(1.0,abs(rmax(-)/majAxis),nData);
    yProfile     = x2(,-)^pow(-,);
    
    majToMinAxes = majAxis / minAxis;
    if     (AXES_DEF==USE_FLATTENING)
        majToMinAxes = minAxis;
    else if(AXES_DEF==USE_INCLINATION)
        majToMinAxes = cos(minAxis);
    
    angle        = PA;

    vis = // nData/2./0.675 * 
        visEllipticalProfile(Utable, Vtable, lambda, time, yProfile, xProfile, majToMinAxes, angle, x0, y0);

    // if(numberof(Utable)!=65536)
    //     prout()
    
    return vis;
}

/***************************************************************************/
           
if(growModels(ALL_MODELS, "expdsk2"))
{
    ALL_MODELS.description(0) = "Elliptical Exponential law disk";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "majAxis", "minAxis", "pow", "P.A."];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "mas", "no", "deg"];
    if(AXES_DEF==USE_FLATTENING)
    {
        ALL_MODELS.params(0)      = &["flux", "x", "y", "majAxis", "flattening", "pow", "P.A."];
        ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "no", "no", "deg"];
    }
    else if(AXES_DEF==USE_INCLINATION)
    {
        ALL_MODELS.params(0)      = &["flux", "x", "y", "majAxis", "incl", "pow", "P.A."];
        ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "deg", "no", "deg"];
    }
}         
if(growModels(ALL_MODELS, "expdsk"))
{
    ALL_MODELS.description(0) = "Circular Exponential law disk";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "rin", "pow"];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "no"];
}

func visExpDisk(Utable, Vtable, lambda, time, majAxis, minAxis, pow, PA, x0, y0)
    /* DOCUMENT visExpDisk(Utable, Vtable, lambda, time, majAxis, minAxis, pow, PA, x0, y0)

       DESCRIPTION
       Exponential law disk

       PARAMETERS
       - Utable : 
       - Vtable : 
       - lambda : 
       - time   : 
       - majAxis: 
       - minAxis: 
       - pow    : 
       - PA     : 
       - x0     : 
       - y0     : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    extern AXES_DEF;
    R        = abs(Utable, Vtable);
    maxR     = max(R);
    minR     = min(R);
    minW     = min(lambda)/maxR;
    maxW     = max(lambda)/minR/1.;
    maxW     = 500*mas2rad;
    upSample = 0.5;
    rmax     = maxW;
    nData    = max(int(upSample * maxW / minW), 1)(1);
    nData    = 500;
    
    xProfile     = span(abs(majAxis),rmax,nData);
    x2           = span(1.0,abs(rmax(-)/majAxis),nData);
    yProfile     = exp(x2(,-)*pow(-,));
    
    majToMinAxes = majAxis / minAxis;
    if     (AXES_DEF==USE_FLATTENING)
        majToMinAxes = minAxis;
    else if(AXES_DEF==USE_INCLINATION)
        majToMinAxes = cos(minAxis);
    
    angle        = PA;
    
    vis = visEllipticalProfile(Utable, Vtable, lambda, time, yProfile, xProfile, majToMinAxes, angle, x0, y0);
    return vis;
}

/***************************************************************************/
             
func visDiskGprofile(Utable, Vtable, lambda, time, majAxis, pow, minAxis, PA, x0, y0)
    /* DOCUMENT visDiskGprofile(Utable, Vtable, lambda, time, majAxis, pow, minAxis, PA, x0, y0)

       DESCRIPTION
       Gaussian disk profile

       PARAMETERS
       - Utable : 
       - Vtable : 
       - lambda : 
       - time   : 
       - majAxis: 
       - pow    : 
       - minAxis: 
       - PA     : 
       - x0     : 
       - y0     : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    extern AXES_DEF;
    R            = abs(Utable, Vtable);
    maxR         = max(R);
    minR         = min(R);
    minW         = min(lambda)/maxR;
    maxW         = max(lambda)/minR;
    upSample     = 10;
    nData        = max(int(maxW / minW), 1)(1);
    rmax         = maxW*upSample;
    
    xProfile     = span(majAxis,rmax,nData);
    x2           = span(.0,rmax/majAxis-1,nData);
    yProfile     = exp(x2^2);
    
    majToMinAxes = majAxis / minAxis;
    if     (AXES_DEF==USE_FLATTENING)
        majToMinAxes = minAxis;
    else if(AXES_DEF==USE_INCLINATION)
        majToMinAxes = cos(minAxis);
    
    angle        = PA;
    
    vis = visEllipticalProfile(Utable, Vtable, lambda, time, yProfile, xProfile, majToMinAxes, angle, x0, y0);
    return vis;
}

/***************************************************************************/
             
func visBinaryStar(Utable, Vtable, lambda, time, sep, pa, fxRat)
    /* DOCUMENT visBinaryStar(Utable, Vtable, lambda, time, sep, pa, fxRat)

       DESCRIPTION
       Binary star model

       PARAMETERS
       - Utable: U coordinates (meters)
       - Vtable: V coordinates (meters)
       - lambda: wavelength (meters)
       - time  : 
       - sep   : 
       - pa    : 
       - fxRat : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    pa = pa+pi/2;
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
  
    C = (1 + fxRat(-,) * exp(-2i*pi*(u*sep(-,)*cos(pa(-,)) -
                                     v*sep(-,)*sin(pa(-,)))))/
        (1 + fxRat(-,));

    return C;
}

/***************************************************************************/

// Square
if(growModels(ALL_MODELS, "sq"))
{
    ALL_MODELS.description(0) = "Square";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "size"];
    ALL_MODELS.units(0)          = &["no", "mas", "mas", "mas"];
}

func visSquare(Utable, Vtable, lambda, time, diam, x0, y0)
    /* DOCUMENT visSquare(Utable, Vtable, lambda, time, diam, x0, y0)

       DESCRIPTION
       square ("pixel") model

       PARAMETERS
       - Utable: U coordinates (meters)
       - Vtable: V coordinates (meters)
       - lambda: wavelength (meters)
       - time  : 
       - diam  : 
       - x0    : 
       - y0    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
    
    C_centered = yocoMathSinc(pi * u * diam(-,))*yocoMathSinc(pi * v * diam(-,));
    C = shiftFourier(Utable,Vtable,lambda,C_centered,x0,y0);
    return C;
}         

/***************************************************************************/

// Rectangle
if(growModels(ALL_MODELS, "sq2"))
{
    ALL_MODELS.description(0) = "Rectangle";
    ALL_MODELS.params(0) = &["flux", "x", "y", "size1", "size2", "angle"];
    ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "mas", "deg"];
    if(AXES_DEF==USE_FLATTENING)
    {
        ALL_MODELS.params(0) = &["flux", "x", "y", "size", "flatng", "angle"];
        ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "no", "deg"];
    }
    else if(AXES_DEF==USE_INCLINATION)
    {
        ALL_MODELS.params(0) = &["flux", "x", "y", "size", "incl", "angle"];
        ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "deg", "deg"];
    }
}

func visRectangle(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0)
    /* DOCUMENT visRectangle(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0)

       DESCRIPTION
       rectangle ("pixel") model

       PARAMETERS
       - Utable   : U coordinates (meters)
       - Vtable   : V coordinates (meters)
       - lambda   : wavelength (meters)
       - time     : 
       - majorAxis: (milliarcsecond) array of n(lambda)
       - minorAxis: (milliarcsecond) array of n(lambda)
       - angle    : 
       - x0       : 
       - y0       : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    extern AXES_DEF;

    if     (AXES_DEF==USE_FLATTENING)
        minorAxis = majorAxis / minorAxis;
    else if(AXES_DEF==USE_INCLINATION)
        minorAxis = majorAxis * cos(minorAxis);
        
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);

    uPrim = (u*cos(angle(-,))+v*sin(angle(-,))) * majorAxis(-,);
    vPrim = (u*sin(angle(-,))-v*cos(angle(-,))) * minorAxis(-,);
    
    C_centered = yocoMathSinc(pi * uPrim)*yocoMathSinc(pi * vPrim);
    C = shiftFourier(Utable,Vtable,lambda,C_centered,x0,y0);
    return C;
}         

/***************************************************************************/

  
if(growModels(ALL_MODELS, "l"))
{
    ALL_MODELS.description(0) = "Circular lorentzian";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "fwhm"];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas"];
}

func visLorentzianDisk(Utable, Vtable, lambda, time, fwhm, x0, y0)
    /* DOCUMENT visLorentzianDisk(Utable, Vtable, lambda, time, fwhm, x0, y0)

       DESCRIPTION
       Lorentz profile disk

       PARAMETERS
       - Utable: U coordinates (meters)
       - Vtable: V coordinates (meters)
       - lambda: wavelength (meters)
       - time  : 
       - fwhm  : 
       - x0    : 
       - y0    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
    
    r = pi * abs(u, v) * fwhm(-,);
    C_centered = exp(-r);
  
    C = shiftFourier(Utable,Vtable,lambda,C_centered,x0,y0);

    return C;
}    

/***************************************************************************/

if(growModels(ALL_MODELS, "g"))
{
    ALL_MODELS.description(0) = "Circular Gaussian disk";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "fwhm"];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas"];
}
        
func visGaussianDisk(Utable, Vtable, lambda, time, fwhm, x0, y0)
    /* DOCUMENT visGaussianDisk(Utable, Vtable, lambda, time, fwhm, x0, y0)

       DESCRIPTION
       Gauss profile disk

       PARAMETERS
       - Utable: U coordinates (meters)
       - Vtable: V coordinates (meters)
       - lambda: wavelength (meters)
       - time  : 
       - fwhm  : 
       - x0    : 
       - y0    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
    
    
    r2 = (pi^2) * (u^2 + v^2) * (fwhm(-,)^2)/(4.0*log(2.0));
    C_centered = exp(-r2);
  
    C = shiftFourier(Utable,Vtable,lambda,C_centered,x0,y0);

    return C;
}
            
/***************************************************************************/

if(growModels(ALL_MODELS, "gl"))
{
    ALL_MODELS.description(0) = "Gauss. disk with gauss. line profile";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "fwhm", "wlenLine", "speedLine", "widthLine", "fluxLine"];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "micron", "m/s", "micron", "no"];
}

if(growModels(ALL_MODELS, "udl"))
{
    ALL_MODELS.description(0) = "Unif. disk with gauss. line profile";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "fwhm", "wlenLine", "speedLine", "widthLine", "fluxLine"];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "micron", "m/s", "micron", "no"];
}

func visLineDisk(Utable, Vtable, lambda, time, flux, fwhm, x0, y0, wlenLine, speedLine, widthLine, fluxLine, type, &specTot)
    /* DOCUMENT visLineDisk(Utable, Vtable, lambda, time, flux, fwhm, x0, y0, wlenLine, speedLine, widthLine, fluxLine, type, &specTot)

       DESCRIPTION
       different types of profiles with with Gaussian line profile

       PARAMETERS
       - Utable   : 
       - Vtable   : 
       - lambda   : 
       - time     : 
       - flux     : 
       - fwhm     : 
       - x0       : 
       - y0       : 
       - wlenLine : 
       - speedLine: 
       - widthLine: 
       - fluxLine : 
       - type     : 
       - specTot  : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    if(is_void(type))
        error;
    
    specTot = flux + gauss(lambda,
                           [fluxLine(1),
                            wlenLine(1) * yocoAstroSI.c / (yocoAstroSI.c - speedLine(1)),
                            widthLine(1)/(2*sqrt(2*log(2)))]);
    
    if(type=="g")
        C = visGaussianDisk(Utable, Vtable, lambda, time, fwhm, x0, y0);
    else if(type=="ud")
        C = visUniformDisk(Utable, Vtable, lambda, time, fwhm, x0, y0);
    
    // if(numberof(flux)!=1)
    // prout()
            
    return C;
}
            
/***************************************************************************/

if(growModels(ALL_MODELS, "ggl"))
{
    ALL_MODELS.description(0) = "Bigauss. with gauss. line profile";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "fwhm", "dx", "dy", "wlenLine", "speedLine", "widthLine", "dephtLine"];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "mas", "mas", "micron", "m/s", "micron", "no"];
}

func visLineBiGaussian(Utable, Vtable, lambda, time, flux, fwhm, x0, y0, dx, dy, wlenLine, speedLine, widthLine, dephtLine, &specTot)
    /* DOCUMENT visLineBiGaussian(Utable, Vtable, lambda, time, flux, fwhm, x0, y0, dx, dy, wlenLine, speedLine, widthLine, dephtLine, &specTot)

       DESCRIPTION
       Double Gaussian disk profile

       PARAMETERS
       - Utable   : 
       - Vtable   : 
       - lambda   : 
       - time     : 
       - flux     : 
       - fwhm     : 
       - x0       : 
       - y0       : 
       - dx       : 
       - dy       : 
       - wlenLine : 
       - speedLine: 
       - widthLine: 
       - dephtLine: 
       - specTot  : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    G1 = visLineGaussianDisk(Utable, Vtable, lambda, time, flux, fwhm, x0+dx/2, y0+dy/2, wlenLine, speedLine, widthLine, dephtLine, specTot1);
    
    G2 = visLineGaussianDisk(Utable, Vtable, lambda, time, flux, fwhm, x0-dx/2, y0-dy/2, wlenLine, -speedLine, widthLine, dephtLine, specTot2);
    
    specTot = specTot1+specTot2;
    specTot = specTot + (specTot==0);

    C = (G1 * specTot1(-,) + G2 * specTot2(-,)) / specTot(-,);
    
    return C;
}
            
/***************************************************************************/

if(growModels(ALL_MODELS, "g2"))
{
    ALL_MODELS.description(0) = "Elliptical Gaussian disk";
    ALL_MODELS.params(0) = &["flux", "x", "y", "fwhm1", "fwhm2", "angle"];
    ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "mas", "deg"];
    if(AXES_DEF==USE_FLATTENING)
    {
        ALL_MODELS.params(0) = &["flux", "x", "y", "fwhm", "flatng", "angle"];
        ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "no", "deg"];
    }
    else if(AXES_DEF==USE_INCLINATION)
    {
        ALL_MODELS.params(0) = &["flux", "x", "y", "fwhm", "incl", "angle"];
        ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "deg", "deg"];
    }
}

func visEllipticalGaussianDisk(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0)
    /* DOCUMENT visEllipticalGaussianDisk(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0)

       DESCRIPTION
       Elliptical Gaussian disk

       PARAMETERS
       - Utable   : U coordinates (meters)
       - Vtable   : V coordinates (meters)
       - lambda   : wavelength (meters)
       - time     : 
       - majorAxis: 
       - minorAxis: 
       - angle    : 
       - x0       : 
       - y0       : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    extern AXES_DEF;

    if     (AXES_DEF==USE_FLATTENING)
        minorAxis = majorAxis / minorAxis;
    else if(AXES_DEF==USE_INCLINATION)
        minorAxis = majorAxis * cos(minorAxis);
        
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
    
    if(numberof(where(abs(majorAxis)>1e100))>0)
        majorAxis(where(abs(majorAxis)>1e100)) = 1e100;
    
    if(numberof(where(abs(minorAxis)>1e100))>0)
        minorAxis(where(abs(minorAxis)>1e100)) = 1e100;
    
    if(numberof(where(abs(angle)>1e10))>0)
        angle(where(abs(angle)>1e10)) = 1e10;
    
    r2 = (pi^2) * (((u*sin(angle(-,))+v*cos(angle(-,))) * majorAxis(-,))^2 +
                   ((u*cos(angle(-,))-v*sin(angle(-,))) * minorAxis(-,))^2)/
        (4.0*log(2.0));
  
    C_centered = exp(-r2);
    C = shiftFourier(Utable,Vtable,lambda,C_centered,x0,y0);
    return C;
}

/***************************************************************************/

types = ["rg","rud","rr"];
descr = ["Circular Gaussian ring",
         "Circular Uniform ring",
         "Circular Ring ring"];
for(k=1;k<=3;k++)
{
    if(growModels(ALL_MODELS, types(k)))
    {
        ALL_MODELS.description(0) = descr(k);
        ALL_MODELS.params(0)      = &["flux", "x", "y", "fwhm", "thickfwhm"];
        ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas", "mas", "deg", "mas"];
    }
}

func visThickRing(Utable, Vtable, lambda, time, diamR, diamG, x0, y0, type, &flux)
    /* DOCUMENT visThickRing(Utable, Vtable, lambda, time, diamR, diamG, x0, y0, type, &flux)
       
       DESCRIPTION
       a thick ring model

       PARAMETERS
       - Utable: U coordinates (meters)
       - Vtable: V coordinates (meters)
       - lambda: wavelength (meters)
       - time  : 
       - diamR : 
       - diamG : 
       - x0    : 
       - y0    : 
       - type  : 
       - flux  : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{

    return visThickEllipticalRing2(Utable, Vtable, lambda, time, diamR, diamR, 0.0, 2*diamG, x0, y0, flux);
}        

/***************************************************************************/
  
if(growModels(ALL_MODELS, "r"))
{
    ALL_MODELS.description(0) = "Circular ring";
    ALL_MODELS.params(0)      = &["flux", "x", "y", "diam"];
    ALL_MODELS.units(0)       = &["no", "mas", "mas", "mas"];
}

func visRing(Utable, Vtable, lambda, time, diam, x0, y0)
    /* DOCUMENT visRing(Utable, Vtable, lambda, time, diam, x0, y0)

       DESCRIPTION

       PARAMETERS
       - Utable: U coordinates (meters)
       - Vtable: V coordinates (meters)
       - lambda: wavelength (meters)
       - time  : 
       - diam  : 
       - x0    : 
       - y0    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
  
    r = sqrt(u^2+v^2);
    
    C_centered = bessj(0, pi * r * diam(-,));
    C = shiftFourier(Utable,Vtable,lambda,C_centered,x0,y0);
    return C;
}        

/***************************************************************************/
  
if(growModels(ALL_MODELS, "r2"))
{
    ALL_MODELS.description(0) = "Elliptical ring";
    ALL_MODELS.params(0) = &["flux", "x", "y", "majorAxis", "minorAxis", "angle"];
    ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "mas", "deg"];
    if(AXES_DEF==USE_FLATTENING)
    {
        ALL_MODELS.params(0) = &["flux", "x", "y", "diam", "flatng", "angle"];
        ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "no", "deg"];
    }
    else if(AXES_DEF==USE_INCLINATION)
    {
        ALL_MODELS.params(0) = &["flux", "x", "y", "diam", "incl", "angle"];
        ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "deg", "deg"];
    }
}

func visEllipticalRing(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0)
    /* DOCUMENT visEllipticalRing(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0)

       DESCRIPTION

       PARAMETERS
       - Utable   : U coordinates (meters)
       - Vtable   : V coordinates (meters)
       - lambda   : wavelength (meters)
       - time     : 
       - majorAxis: 
       - minorAxis: 
       - angle    : 
       - x0       : 
       - y0       : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    extern AXES_DEF;

    // Change the way the model is described.
    // By default major and minr axes are provided as input parameters
    // Change minor axis to match a flattening ratio
    if     (AXES_DEF==USE_FLATTENING)
        minorAxis = majorAxis / minorAxis;
    // Change minor axis to match an inclination angle
    else if(AXES_DEF==USE_INCLINATION)
        minorAxis = majorAxis * cos(minorAxis);
        
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
  
    r = sqrt(((u*sin(angle(-,))+v*cos(angle(-,))) * majorAxis(-,))^2 +
             ((u*cos(angle(-,))-v*sin(angle(-,))) * minorAxis(-,))^2);
    
    C_centered = bessj(0,pi * r);
    C = shiftFourier(Utable,Vtable,lambda,C_centered,x0,y0);
    return C;
}        

/***************************************************************************/
  
types = ["rg2","rud2","rr2"];
descr = ["Elliptical Gaussian ring",
         "Elliptical Uniform ring",
         "Elliptical Ring ring"];
for(k=1;k<=3;k++)
{
    if(growModels(ALL_MODELS, types(k)))
    {
        //Elliptical Circular Gaussian ring
        ALL_MODELS.description(0) = descr(k);
        ALL_MODELS.params(0) = &["flux", "x", "y", "majorAxis",
                                 "minorAxis", "angle", "thickfwhm"];
        ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas",
                                 "mas", "deg", "mas"];
        if(AXES_DEF==USE_FLATTENING)
        {
            ALL_MODELS.params(0)     = &["flux", "x", "y", "diam",
                                         "flatng", "angle", "thickfwhm"];
            ALL_MODELS.units(0)      = &["no", "mas", "mas", "mas",
                                         "no", "deg", "mas"];
        }
        if(AXES_DEF==USE_INCLINATION)
        {
            ALL_MODELS.params(0)     = &["flux", "x", "y", "diam",
                                         "incl", "angle", "thickfwhm"];
            ALL_MODELS.units(0)      = &["no", "mas", "mas", "mas",
                                         "deg", "deg", "mas"];
        }
    }
}

func visThickEllipticalRing(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, thickness, thickType, x0, y0)
    /* DOCUMENT visThickEllipticalRing(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, thickness, thickType, x0, y0)

       DESCRIPTION

       PARAMETERS
       - Utable   : U coordinates (meters)
       - Vtable   : V coordinates (meters)
       - lambda   : wavelength (meters)
       - time     : 
       - majorAxis: 
       - minorAxis: 
       - angle    : 
       - thickness: 
       - thickType: 
       - x0       : 
       - y0       : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    extern AXES_DEF;

    // Change the way the model is described.
    // By default major and minr axes are provided as input parameters
    // Change minor axis to match a flattening ratio
    if     (AXES_DEF==USE_FLATTENING)
        minorAxis = majorAxis / minorAxis;
    // Change minor axis to match an inclination angle
    else if(AXES_DEF==USE_INCLINATION)
        minorAxis = majorAxis * cos(minorAxis);
    
    if(is_void(thickType))
        thickType = "g";
    
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
  
    r = sqrt(((u*sin(angle(-,))+v*cos(angle(-,))) * majorAxis(-,))^2 +
             ((u*cos(angle(-,))-v*sin(angle(-,))) * minorAxis(-,))^2);
    
    C_centered = bessj(0,pi * r);
    C_shifted = shiftFourier(Utable,Vtable,lambda,C_centered,x0,y0);
    
    if(thickType=="g")
        visF = visGaussianDisk;
    else if(thickType=="ud")
        visF = visUniformDisk;
    else if(thickType=="r")
        visF = visRing;
    else
        error;
    
    C = C_shifted * visF(Utable, Vtable, lambda, time, thickness, 0.0, 0.0);
    return C;
}

/***************************************************************************/
  

func visThickEllipticalRing2(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, thickness, x0, y0, &flux)
    /* DOCUMENT visThickEllipticalRing2(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, thickness, x0, y0, &flux)

       DESCRIPTION

       PARAMETERS
       - Utable   : U coordinates (meters)
       - Vtable   : V coordinates (meters)
       - lambda   : wavelength (meters)
       - time     : 
       - majorAxis: 
       - minorAxis: 
       - angle    : 
       - thickness: 
       - x0       : 
       - y0       : 
       - flux     : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    extern AXES_DEF;

    // Change the way the model is described.
    // By default major and minr axes are provided as input parameters
    // Change minor axis to match a flattening ratio
    if     (AXES_DEF==USE_FLATTENING)
        minorAxis = majorAxis / minorAxis;
    // Change minor axis to match an inclination angle
    else if(AXES_DEF==USE_INCLINATION)
        minorAxis = majorAxis * cos(minorAxis);
    
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
  
    r = sqrt(((u*sin(angle(-,))+v*cos(angle(-,))) * majorAxis(-,))^2 +
             ((u*cos(angle(-,))-v*sin(angle(-,))) * minorAxis(-,))^2);

    r1 = majorAxis - thickness/2;
    r2 = majorAxis + thickness/2;
    
    a = visUniformDisk(Utable, Vtable, lambda, time, r1, 0.0, 0.0);
    b = visUniformDisk(Utable, Vtable, lambda, time, r2, 0.0, 0.0);

    alpha = pi*r1^2;
    beta =  pi*r2^2;
    flux =  beta - alpha;

    C = (alpha * a - beta * b) / flux;
    C = shiftFourier(Utable,Vtable,lambda,C,x0,y0);
    
    return C;
}

/***************************************************************************/
  
if(growModels(ALL_MODELS, "ud"))
{
    ALL_MODELS.description(0) = "Circular uniform disk";
    ALL_MODELS.params(0)         = &["flux", "x", "y", "diam"];
    ALL_MODELS.units(0)          = &["no", "mas", "mas", "mas"];
}

func visUniformDisk(Utable, Vtable, lambda, time, diam, x0, y0)
    /* DOCUMENT visUniformDisk(Utable, Vtable, lambda, time, diam, x0, y0)

       DESCRIPTION

       PARAMETERS
       - Utable: U coordinates (meters)
       - Vtable: V coordinates (meters)
       - lambda: wavelength (meters)
       - time  : 
       - diam  : 
       - x0    : 
       - y0    : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    // write,Utable(1), Vtable(1),diam(1),x0(1)*rad2mas,y0(1)*rad2mas;
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
  
    r = abs(u,v);
    
    C_centered = (r==0)+(diam(-,)==0)+
        2*bessj(1,pi * r * diam(-,))/(pi * (r+(r==0)) * (diam(-,)+(diam(-,)==0)));
    C = shiftFourier(Utable, Vtable, lambda, C_centered, x0, y0);
    return C;
}

/***************************************************************************/
             
if(growModels(ALL_MODELS, "pwr"))
{
    ALL_MODELS.description(0) = "Power law";
    ALL_MODELS.params(0)         = &["flux", "x", "y", "pow"];
    ALL_MODELS.units(0)          = &["no", "mas", "mas", "no"];
}

func visPowerLaw(Utable, Vtable, lambda, time, pow, x0, y0)
    /* DOCUMENT visPowerLaw(Utable, Vtable, lambda, time, pow, x0, y0)

       DESCRIPTION

       PARAMETERS
       - Utable: 
       - Vtable: 
       - lambda: 
       - time  : 
       - pow   : 
       - x0    : 
       - y0    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    require,"gamma.i";

    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
    r        = abs(u, v);
    
    mag  = exp(lngamma(pow(-,)+1)) * r^(-pow(-,)+1);
    if(anyof(-pow(-,)+1>0))
        mag0 = exp(lngamma(pow(-,)+1)) * 0.0^(-pow(-,)+1);
    else
        mag0=0;
    
    mag = mag/(mag0+(mag0==0));

    phi = -pi * (pow+1)/2.;
    
    vis = mag * exp(1i*phi(-,));

    C = shiftFourier(Utable,Vtable,lambda,vis,x0,y0);
    
    return C;
}

/***************************************************************************/
             
if(growModels(ALL_MODELS, "exp"))
{
    ALL_MODELS.description(0) = "Exponential";
    ALL_MODELS.params(0)         = &["flux", "x", "y", "k0"];
    ALL_MODELS.units(0)          = &["no", "mas", "mas", "no"];
}

func visExponential(Utable, Vtable, lambda, time, k0, x0, y0)
    /* DOCUMENT visExponential(Utable, Vtable, lambda, time, k0, x0, y0)

       DESCRIPTION

       PARAMETERS
       - Utable: 
       - Vtable: 
       - lambda: 
       - time  : 
       - k0    : 
       - x0    : 
       - y0    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    R = abs(Utable, Vtable);
    r = R(,-) / lambda(-,);
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
    //r = abs(u, v);
    
    vis = 1./k0(-,)^2 / (r^2 + 1./k0(-,)^2);

    C = shiftFourier(Utable,Vtable,lambda,vis,x0,y0);
    
    return C;
}
  
/***************************************************************************/
  
if(growModels(ALL_MODELS, "ud2"))
{
    ALL_MODELS.description(0) = "Elliptical uniform disk";
    ALL_MODELS.params(0) = &["flux", "x", "y", "majorAxis", "minorAxis", "angle"];
    ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "mas", "deg"];
    if(AXES_DEF==USE_FLATTENING)
    {
        ALL_MODELS.params(0) = &["flux", "x", "y", "diam", "flatng", "angle"];
        ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "no", "deg"];
    }
    if(AXES_DEF==USE_INCLINATION)
    {
        ALL_MODELS.params(0) = &["flux", "x", "y", "diam", "incl", "angle"];
        ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "deg", "deg"];
    }
}

func visEllipticalUniformDisk(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0)
    /* DOCUMENT visEllipticalUniformDisk(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0)

       DESCRIPTION

       PARAMETERS
       - Utable   : U coordinates (meters)
       - Vtable   : V coordinates (meters)
       - lambda   : wavelength (meters)
       - time     : 
       - majorAxis: 
       - minorAxis: 
       - angle    : 
       - x0       : 
       - y0       : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    extern AXES_DEF;

    // Change the way the model is described.
    // By default major and minr axes are provided as input parameters
    // Change minor axis to match a flattening ratio
    if     (AXES_DEF==USE_FLATTENING)
        minorAxis = majorAxis / minorAxis;
    // Change minor axis to match an inclination angle
    else if(AXES_DEF==USE_INCLINATION)
        minorAxis = majorAxis * cos(minorAxis);
    
    u = Utable(,-) / lambda(-,);
    v = Vtable(,-) / lambda(-,);
  
    r = sqrt(((u*sin(angle(-,))+v*cos(angle(-,))) * majorAxis(-,))^2 +
             ((u*cos(angle(-,))-v*sin(angle(-,))) * minorAxis(-,))^2);
  
    C_centered = ((r==0)+
                  2*bessj(1,pi * r)/(pi * (r+(r==0))));
    C = shiftFourier(Utable,Vtable,lambda,C_centered,x0,y0);
    return C;
}
  
/***************************************************************************/

if(growModels(ALL_MODELS, "skwr2"))
{
    ALL_MODELS.description(0) = "Elliptical Skewed ring";
    ALL_MODELS.type(0)        = "skwr2";
    ALL_MODELS.params(0) = &["flux", "x", "y", "majorAxis", "minorAxis", "angle", "thickdiam", "skewangle", "skewstrength"];
    ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas",     "mas",       "deg",   "mas",       "deg",       "no"];
    if(AXES_DEF==USE_FLATTENING)
    {
        ALL_MODELS.params(0) = &["flux", "x", "y", "diam", "flatng", "angle", "thickdiam", "skewangle", "skewstrength"];
        ALL_MODELS.units(0)  = &["no", "mas", "mas", "mas", "no", "deg", "mas", "deg", "no"];
    }
    if(AXES_DEF==USE_INCLINATION)
    {
        ALL_MODELS.params(0) = &["flux", "x", "y", "diam", "incl", "angle", "thickdiam", "skewangle", "skewstrength"];
        ALL_MODELS.units(0) = &["no", "mas", "mas", "mas", "deg", "deg", "mas", "deg", "no"];
    }
}

func visEllipticalSkewedRing(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0, thick, angleSk, ampSk)
    /* DOCUMENT visEllipticalSkewedRing(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0, thick, angleSk, ampSk)

       DESCRIPTION

       PARAMETERS
       - Utable   : 
       - Vtable   : 
       - lambda   : 
       - time     : 
       - majorAxis: 
       - minorAxis: 
       - angle    : 
       - x0       : 
       - y0       : 
       - thick    : 
       - angleSk  : 
       - ampSk    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    extern AXES_DEF;

    // Change the way the model is described.
    // By default major and minr axes are provided as input parameters
    // Change minor axis to match a flattening ratio
    if     (AXES_DEF==USE_FLATTENING)
        minorAxis = majorAxis / minorAxis;
    // Change minor axis to match an inclination angle
    else if(AXES_DEF==USE_INCLINATION)
        minorAxis = majorAxis * cos(minorAxis);

    nbWlen = numberof(lambda);
    
    imageCube = imaModelEllipticalSkewedRing(majorAxis, minorAxis, angle, x0, y0, thick, angleSk, ampSk, x, y, dim=, champVue=);
    
    Xtable = array(x,nbWlen);
    Ytable = array(y,nbWlen);
    
    vis = imageFFT(Utable, Vtable, lambda, lambda, imageCube, Xtable, Ytable);
    return vis;
} 
  
/***************************************************************************/

if(growModels(ALL_MODELS, "skwr"))
{
    ALL_MODELS.description(0) = "Skewed ring";
    ALL_MODELS.type(0)        = "skwr";
    ALL_MODELS.params(0) = &["flux", "x", "y", "size", "thickdiam", "skewangle", "skewstrength"];
    ALL_MODELS.units(0)  = &["no", "mas","mas","mas",  "mas",       "deg",       "no"];
}

func visSkewedRing(Utable, Vtable, lambda, time, size, x0, y0, thick, angleSk, ampSk)
    /* DOCUMENT visSkewedRing(Utable, Vtable, lambda, time, size, x0, y0, thick, angleSk, ampSk)

       DESCRIPTION

       PARAMETERS
       - Utable : 
       - Vtable : 
       - lambda : 
       - time   : 
       - size   : 
       - x0     : 
       - y0     : 
       - thick  : 
       - angleSk: 
       - ampSk  : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    nbWlen = numberof(lambda);
    
    imageCube = imaModelEllipticalSkewedRing(size, size, 0.0, x0, y0, thick, angleSk, ampSk, x, y, dim=, champVue=);
    
    Xtable = array(x,nbWlen);
    Ytable = array(y,nbWlen);
    
    vis = imageFFT(Utable, Vtable, lambda, lambda, imageCube, Xtable, Ytable);
    return vis;
} 
  
/***************************************************************************/

func visEllipticalThickRing(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0, thick)
    /* DOCUMENT visEllipticalThickRing(Utable, Vtable, lambda, time, majorAxis, minorAxis, angle, x0, y0, thick)

       DESCRIPTION

       PARAMETERS
       - Utable   : 
       - Vtable   : 
       - lambda   : 
       - time     : 
       - majorAxis: 
       - minorAxis: 
       - angle    : 
       - x0       : 
       - y0       : 
       - thick    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    imageCube = imaModelEllipticalThickRing(majorAxis, minorAxis, angle, x0, y0, thick, lambda, x, y, dim=, champVue=);

    nbWlen = numberof(lambda);
    
    Xtable = array(x,nbWlen);
    Ytable = array(y,nbWlen);
    
    vis = imageFFT(Utable, Vtable, lambda, time, lambda, imageCube, Xtable, Ytable);
    return vis;
} 

  
/***************************************************************************/

if(growModels(ALL_MODELS, "exprotdsk"))
{
    ALL_MODELS.description(0) = "Rotating/expanding disk, units in SI";
    ALL_MODELS.params(0) = &["fluxRaie", "fluxStar", "Rstar", "deepStar", "widthStar", "distance", "sizeCont", "size", "incl", "beta", "vrot", "vinf", "v0", "gamma", "lineCentralWlen", "naturalLineWidth", "PAsky"];
    ALL_MODELS.units(0) = &["no", "no", "rsun", "no", "micron", "kpc", "rstar", "rstar", "degrees", "no", "kms", "kms", "kms", "no", "micron", "micron", "degrees"];
}

func visModelExpRotDisk(Utable, Vtable, lambda, time, fluxRaie, fluxStar, Rstar, deepStar, widthStar, distance, sizeCont, size, incl, beta, vrot, vinf, v0, gamma, lineCentralWlen, naturalWidthLine, skyPA, &specTot)
    /* DOCUMENT visModelExpRotDisk(Utable, Vtable, lambda, time, fluxRaie, fluxStar, Rstar, deepStar, widthStar, distance, sizeCont, size, incl, beta, vrot, vinf, v0, gamma, lineCentralWlen, naturalWidthLine, skyPA, &specTot)

       DESCRIPTION

       PARAMETERS
       - Utable          : U table (m)
       - Vtable          : V table (m)
       - lambda          : Wavelength table (m)
       - time            : time table (s)
       - fluxRaie        : 
       - fluxStar        : 
       - Rstar           : 
       - deepStar        : 
       - widthStar       : 
       - distance        : 
       - sizeCont        : 
       - size            : 
       - incl            : 
       - beta            : 
       - vrot            : 
       - vinf            : 
       - v0              : 
       - gamma           : 
       - lineCentralWlen : 
       - naturalWidthLine: 
       - skyPA           : 
       - specTot         : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // Rotate-expand disk, metric units
    imageCube = imaModelExpRotDisk(fluxRaie(1), fluxStar(1), Rstar(1), deepStar(1), widthStar(1), distance(1), sizeCont(1), size(1), incl(1), beta(1), vrot(1), vinf(1), v0(1), gamma(1), lineCentralWlen(1), naturalWidthLine(1), skyPA(1), lambda, x, y, specTot);
    
    nbWlen = numberof(lambda);
    
    Xtable = array(x,nbWlen);
    Ytable = array(transpose(x),nbWlen);
    
    vis = imageFFT(Utable, Vtable, lambda, lambda, imageCube, Xtable, Ytable);
    return vis;
} 

  
/***************************************************************************/

if(growModels(ALL_MODELS, "disk"))
{
    ALL_MODELS.description(0) = "Rotating/expanding disk, angular units";
    ALL_MODELS.params(0) = &["fluxStar", "Rstar", "deepStar", "widthStar", "fluxRaie", "sizeCont", "sizeLine", "beta", "vrot", "vinf", "v0", "gamma", "lineCentralWlen", "naturalLineWidth", "incl", "PAsky"];
    ALL_MODELS.units(0) = &["no", "mas", "no", "micron", "no", "mas", "mas", "no", "kms", "kms", "kms", "no", "micron", "micron", "degrees", "degrees"];
}

func visModelGasPlusDustDisk(Utable, Vtable, lambda, time, fluxStar, RstarRad, deepStar, widthStar, fluxGas, fwhmContRad, fwhmLineRad, beta, vrot, vinf, v0, gamma, lineCentralWlen, naturalLineWidth, fluxDust, diamDust, thickDust, incl, PAsky, &specTot2)
    /* DOCUMENT visModelGasPlusDustDisk(Utable, Vtable, lambda, time, fluxStar, RstarRad, deepStar, widthStar, fluxGas, fwhmContRad, fwhmLineRad, beta, vrot, vinf, v0, gamma, lineCentralWlen, naturalLineWidth, fluxDust, diamDust, thickDust, incl, PAsky, &specTot2)

       DESCRIPTION
       Model used in the paper HD62623 Millour et al. A&A 2011

       PARAMETERS
       - Utable          : 
       - Vtable          : 
       - lambda          : 
       - time            : 
       - fluxStar        : 
       - RstarRad        : 
       - deepStar        : 
       - widthStar       : 
       - fluxGas         : 
       - fwhmContRad     : 
       - fwhmLineRad     : 
       - beta            : 
       - vrot            : 
       - vinf            : 
       - v0              : 
       - gamma           : 
       - lineCentralWlen : 
       - naturalLineWidth: 
       - fluxDust        : 
       - diamDust        : 
       - thickDust       : 
       - incl            : 
       - PAsky           : 
       - specTot2        : 
       
       RETURN VALUES
       
       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // Rotate-expand disk, angular units
    distance = 1*yocoAstroSI.pc;
    Rstar    = RstarRad * distance;
    fwhmCont = fwhmContRad * distance / Rstar;
    fwhmLine = fwhmContRad * distance / Rstar;
    
    nbWlen   = numberof(lambda);
    
    bigDiamRing   = array(diamDust(1), nbWlen);
    smallDiamRing = cos(incl(1)) * bigDiamRing;
    amgleRing     = array(PAsky(1), nbWlen);
    angleSk       = array(PAsky(1)+pi/2, nbWlen);
    ampSk         = array(sign(incl(1))*(0.5-0.5*cos(2*incl(1))), nbWlen);

    spec  = [array(fluxGas(1), nbWlen),          fluxDust];
    p2    = [array(fluxStar(1), nbWlen),         array(0.0,nbWlen)];
    p3    = [array(Rstar(1), nbWlen),            array(0.0,nbWlen)];
    p4    = [array(deepStar(1), nbWlen),         bigDiamRing];
    p5    = [array(widthStar(1), nbWlen),        smallDiamRing];
    p6    = [array(distance(1), nbWlen),         amgleRing];
    p7    = [array(fwhmCont(1), nbWlen),         array(thickDust(1),nbWlen)];
    p8    = [array(fwhmLine(1), nbWlen),         angleSk];
    p9    = [array(incl(1), nbWlen),             ampSk];
    p10   = [array(beta(1), nbWlen),             array(0.0,nbWlen)];
    p11   = [array(vrot(1), nbWlen),             array(0.0,nbWlen)];
    p12   = [array(vinf(1), nbWlen),             array(0.0,nbWlen)];
    p13   = [array(v0(1), nbWlen),               array(0.0,nbWlen)];
    p14   = [array(gamma(1), nbWlen),            array(0.0,nbWlen)];
    p15   = [array(lineCentralWlen(1), nbWlen),  array(0.0,nbWlen)];
    p16   = [array(naturalLineWidth(1), nbWlen), array(0.0,nbWlen)];
    p17   = [array(PAsky(1), nbWlen),            array(0.0,nbWlen)];
    p18   = [];
    p19   = [];
    p20   = [];
    typei = ["exprotdsk", "skwr2"];
    
    vis = visMultipleResolved(Utable, Vtable, lambda, time, typei, spec, p2, p3, p4, p5, p6, p7, p8, p9, p10, p11, p12, p13, p14, p15, p16, p17, p18, p19, p20, corrFlux, flux);

    specTot2 = flux;

    return vis;
} 

  
/***************************************************************************/

if(growModels(ALL_MODELS, "gdisk"))
{
    ALL_MODELS.description(0) = "Rotating/expanding gas+dust disk";
    ALL_MODELS.type(0)        = "gdisk";
    ALL_MODELS.params(0) = &["fluxStar", "Rstar", "deepStar", "widthStar", "fluxGas", "fwhmContRad", "fwhmLineRad", "beta", "vrot", "vinf", "v0", "gamma", "lineCentralWlen", "naturalLineWidth", "fluxDust", "diamDust", "thickDust", "incl", "PAsky"];
    ALL_MODELS.units(0) = &["no", "mas", "no", "micron", "no", "mas", "mas", "no", "kms", "kms", "kms", "no", "micron", "micron", "no", "mas", "mas", "degrees", "degrees"];
}

func visModelGasDisk(Utable, Vtable, lambda, time, fluxStar, RstarRad, deepStar, widthStar, fluxGas, fwhmContRad, fwhmLineRad, beta, vrot, vinf, v0, gamma, lineCentralWlen, naturalLineWidth, incl, PAsky, &specTot2)
    /* DOCUMENT visModelGasDisk(Utable, Vtable, lambda, time, fluxStar, RstarRad, deepStar, widthStar, fluxGas, fwhmContRad, fwhmLineRad, beta, vrot, vinf, v0, gamma, lineCentralWlen, naturalLineWidth, incl, PAsky, &specTot2)

       DESCRIPTION
       Model used in the paper HD62623 Millour et al. A&A 2011

       PARAMETERS
       - Utable          : 
       - Vtable          : 
       - lambda          : 
       - time            : 
       - fluxStar        : 
       - RstarRad        : 
       - deepStar        : 
       - widthStar       : 
       - fluxGas         : 
       - fwhmContRad     : 
       - fwhmLineRad     : 
       - beta            : 
       - vrot            : 
       - vinf            : 
       - v0              : 
       - gamma           : 
       - lineCentralWlen : 
       - naturalLineWidth: 
       - incl            : 
       - PAsky           : 
       - specTot2        : 
       
       RETURN VALUES
       
       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    // Rotate-expand gas+dust disk
    distance = 1*yocoAstroSI.pc;
    Rstar    = RstarRad * distance;
    fwhmCont = fwhmContRad * distance / Rstar;
    fwhmLine = fwhmContRad * distance / Rstar;
    
    nbWlen   = numberof(lambda);
    
    spec  = [array(fluxGas(1), nbWlen)];
    p2    = [array(fluxStar(1), nbWlen)];
    p3    = [array(Rstar(1), nbWlen)];
    p4    = [array(deepStar(1), nbWlen)];
    p5    = [array(widthStar(1), nbWlen)];
    p6    = [array(distance(1), nbWlen)];
    p7    = [array(fwhmCont(1), nbWlen)];
    p8    = [array(fwhmLine(1), nbWlen)];
    p9    = [array(incl(1), nbWlen)];
    p10   = [array(beta(1), nbWlen)];
    p11   = [array(vrot(1), nbWlen)];
    p12   = [array(vinf(1), nbWlen)];
    p13   = [array(v0(1), nbWlen)];
    p14   = [array(gamma(1), nbWlen)];
    p15   = [array(lineCentralWlen(1), nbWlen)];
    p16   = [array(naturalLineWidth(1), nbWlen)];
    p17   = [array(PAsky(1), nbWlen)];
    p18   = [];
    p19   = [];
    p20   = [];
    typei = ["exprotdsk"];
    
    vis = visMultipleResolved(Utable, Vtable, lambda, time, typei, spec, p2, p3, p4, p5, p6, p7, p8, p9, p10, p11, p12, p13, p14, p15, p16, p17, p18, p19, p20, corrFlux, flux);

    specTot2 = flux;

    return vis;
} 

  
/***************************************************************************/

func visModelExpRotStar(Utable, Vtable, lambda, time, diam, oblatness, theta, i, assomb, rapFluxCont, totFlux, vitRot, vitExp, lineCentralWlen, naturalLineWidth)
    /* DOCUMENT visModelExpRotStar(Utable, Vtable, lambda, time, diam, oblatness, theta, i, assomb, rapFluxCont, totFlux, vitRot, vitExp, lineCentralWlen, naturalLineWidth)

       DESCRIPTION

       PARAMETERS
       - Utable          : 
       - Vtable          : 
       - lambda          : 
       - time            : 
       - diam            : 
       - oblatness       : 
       - theta           : 
       - i               : 
       - assomb          : 
       - rapFluxCont     : 
       - totFlux         : 
       - vitRot          : 
       - vitExp          : 
       - lineCentralWlen : 
       - naturalLineWidth: 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    require, "visModels.i";
    imageCube = imaModelExpRotStar(diam, oblatness, theta, i, assomb, rapFluxCont, totFlux, vitRot, vitExp, lineCentralWlen, naturalLineWidth, lambda, x, y);
        
    nbWlen = numberof(lambda);
    
    Xtable = array(x,numberof(x),nbWlen);
    Ytable = array(transpose(array(x,numberof(x))),nbWlen);
    
    vis = imageFFT(Utable, Vtable, lambda, time, lambda, imageCube, Xtable, Ytable);
    return vis;
} 

      
            
/***************************************************************************/
  
func visMultipleResolved(Utable, Vtable, lambda, time, typei, spec, p2, p3, p4, p5, p6, p7, p8, p9, p10, p11, p12, p13, p14, p15, p16, p17, p18, p19, p20, &corrFlux, &flux)
    /* DOCUMENT visMultipleResolved(Utable, Vtable, lambda, time, typei, spec, p2, p3, p4, p5, p6, p7, p8, p9, p10, p11, p12, p13, p14, p15, p16, p17, p18, p19, p20, &corrFlux, &flux)
       
       DESCRIPTION
       This function computes any combination of several functions
       which are interfaced in this script. It outputs visibilities
       (as return value), correlated flux and total flux. Potentially
       up to 20 parameters can be set with this script.
       
       PAPERS
       Here is a (non-exhaustive) list of papers that made use of fitOmatic:
       Domiciano et al. 2008 A&A, 489, 5       : Uniform disks
       Meilland et al.  2008 A&A, 488, 67      : Point sources and uniform disks
       Millour et a.    2009 A&A, 507, 317     : Gaussian disk, resolved component
       Millour et al.   2009 A&A, 506, 49      : Rings, pinwheel
       Chiavassa et al. 2010 A&A, 511, 51      : UD, point sources, resolved source
       Millour et al.   2011 A&A, 526, 107     : Gaz+dust rotating disk
       Chesneau et al.  2014 A&A, 563, 71      : Uniform disk, Gaussian disk, point source
       Chesneau et al.  2014 A&A, 569, 3       : Uniform disks, resolved component
       Chesneau et al.  2014 A&A, 569, 4       : Uniform disks, point sources
       Fabas et al. submitted                  : Uniform disks, Gaussian disks
       Lamberts et al. in prep                 : point sources
       Sanchez et al. SPIE2016 (beauty contest): point sources, uniform disk, skewed ring, Gaussian
       
       The following parameters are relevant for the following models:
       ____________________________________________________________________
       | Type    |  Model    |spec| p2  | p3  | p4  | p5  | p6  |  p7 | p8  |
       |---------|-----------|----|-----|-----|-----|-----|-----|-----|-----|
       | no      |point      |flux| x0  | y0  |  -  |  -  |  -  |  -  |  -  |
       | ful     |resolved   |flux| -   |-    |  -  |  -  |  -  |  -  |  -  |
       | g       |Gauss      |flux| x0  | y0  |FWHM |  -  |  -  |  -  |  -  |
       | g2      |eGauss     |flux| x0  | y0  |FWHM1|FWHM2|angle|  -  |  -  |
       | l       |Lorentzian |flux| x0  | y0  |fwhm |  -  |  -  |  -  |  -  |
       | ud      |UD         |flux| x0  | y0  |diam |  -  |  -  |  -  |  -  |
       | ud2     |eUD        |flux| x0  | y0  |diam1|diam2|angle|  -  |  -  |
       | ldl     |limbUD     |flux| x0  | y0  |diam |ulam |  -  |  -  |  -  |
       | ldl     |LDD lin    |flux| x0  | y0  |size |epsil|     |     |     |
       | ld3     |limbUDpow  |flux| x0  | y0  |diam |ulam |vlam |wlam |  -  |
       | r       |ring       |flux| x0  | y0  |diam |  -  |  -  |  -  |  -  |
       | r2      |eRing      |flux| x0  | y0  |diam1|diam2|angle|  -  |  -  |
       | rg      |g ring     |flux| x0  | y0  |diam |  -  |  -  |FWHM |  -  |
       | rg2     |g eRing    |flux| x0  | y0  |diam1|diam2|angle|FWHM |  -  |
       | rud     |UD ring    |flux| x0  | y0  |diam |  -  |  -  |diam3|  -  |
       | rud2    |UD eRing   |flux| x0  | y0  |diam1|diam2|angle|diam3|  -  |
       | rr      |r ring     |flux| x0  | y0  |diam |  -  |  -  |diam3|  -  |
       | rr2     |r eRing    |flux| x0  | y0  |diam1|diam2|angle|diam3|  -  |
       | skwr    |skwedring  |flux| x0  | y0  |size |thick|angSk|ampSk|  -  |
       | pwr     |power law  |flux| x0  | y0  |pow  |  -  |  -  |  -  |  -  |
       | exp     |exponential|flux| x0  | y0  |k0   |  -  |  -  |  -  |  -  |_____
       | sq      |square     |flux| x0  | y0  |size |  -  |  -  |  -  |  -  | p9  |________
       | sq2     |rectangle  |flux| x0  | y0  |size1|size2|angle|  -  |  -  |-----| p10    |
       | skwr2   |e_skwedring|flux| x0  | y0  |size1|size2|angle|thick|angSk|ampSk|--------|_________________________________
       | pwhl    |pinwheel   |flux|round|mnThk|mxThk|sz   |pw   |phi  |  i  |omega|tail    | p11    | p12         | p13      |
       | 2pwhl   |double pwhl|flux|round|mnThk|mxThk|sz   |pw   |phi  |  i  |omega|tail    |--------|-------------|----------|
       | pwhltl  |pinwheel   |flux|round|mnThk|mxThk|sz   |pw   |phi  |  i  |omega|PowerLaw|tempInit|amplitudeSine|amortiSine|___________________________________________
       |         |temp Law   |    |     |     |     |     |     |     |     |Decr | -      |  -     |  -          | -        | p14   | p15   | p16    | p17   |p18 | p19 |
       | orb     |orbitin obj|flux| x0  | y0  |diam |type | t0  | ecc |majAx| per | i      | omega  |OMEGA        |          |-------|-------|--------|-------|----|-----|
       | orb2    |orbitin bin|flux| x0  | y0  |fxRat|diam1|diam2|type1|type2| M1  | M2     | t0     |eccentricity | maj Axis |period | i     | omega  | OMEGA |    |     |
       | disk    |rotgasdisk |fxS |rStar|deepS|wdtS |fxLin|szCnt|szLne|beta |vrot |vinf    |v0      |gamma        |lineCen   |lineWdt|incl   |pasky   |       |    |     |
       |exprotdsk|rotgasdskSI|fxRa|fxStr|rStar|deepS|wdtS |dst  |szCnt|szLne|incl |beta    |vrot    |vinf         |v0        |gamma  |lineCen|lineWdt |pasky  |    |     |
       | gdisk   |gasdustdisk|fxS |rStar|deepS|wdtS |fxG  |szCnt|szLin|beta |vrot |vinf    |v0      |gamma        |lineCen   |lineWdt|fxDust |FWHMDust|thkDust|incl|pasky|
       |_________|___________|____|_____|_____|_____|_____|_____|_____|_____|_____|________|________|_____________|__________|_______|_______|________|_______|____|_____|

       EXAMPLES 

       SEE ALSO:
    */
{
    nChips  = numberof(typei);
    nBase   = dimsof(Utable)(2);
    nLambda = numberof(lambda);
    corrFluxTmp = 0;
        
    for(iChips=1;iChips<=nChips;iChips++)
    {
        // Call Uniform disk
        if (typei(iChips)=="ud")
        {
            Ci = visUniformDisk(
                                Utable,Vtable,lambda, time,
                                p4(,iChips),
                                p2(,iChips),
                                p3(,iChips));
            
            // if(numberof(Ci)==65536)
            // {
            //     if(p2(1,iChips)!=0)
            //     {
            //         c = grid2ima(Ci);
            //         yocoGuiWinKill;
            //         pli,abs(c),cmin=0,cmax=1;
            //         window,2;
            //         pli,atan(c.im,c.re);
            //         frwer();
            //     }
            // }
            
            // if(numberof(Ci)!=65536)
            // {
            //     if(p2(1,iChips)!=0)
            //     {
            //         yocoGuiWinKill;
            //         plg,transpose(Ci.re)(*),color="blue";
            //         plg,transpose(Ci.im)(*),color="red";
            //         write,Utable,Vtable;
            //         // write,lambda,time;
            //         frwer();
            //     }
            // }
        }
        else if (typei(iChips)=="udl")
        {
            Ci = visLineDisk(
                             Utable,Vtable,lambda, time,
                             spec(,iChips), //fluxRaie,
                             p4(,iChips),   // FWHM
                             p2(,iChips),   // x0
                             p3(,iChips),   // y0
                             p5(,iChips),   // line wlen
                             p6(,iChips),   // line speed
                             p7(,iChips),   // line width
                             p8(,iChips),   // line depht
                             "ud",
                             specTot);
            spec(,iChips) = specTot;
        }
        // Call elliptical uniform disk
        else if (typei(iChips)=="ud2")
            Ci = visEllipticalUniformDisk(
                                          Utable,Vtable,lambda, time,
                                          p4(,iChips),
                                          p5(,iChips),
                                          p6(,iChips),
                                          p2(,iChips),
                                          p3(,iChips));
        else if (typei(iChips)=="ldl")
            Ci = visLimbLinear(
                               Utable,Vtable,lambda, time,
                               p4(,iChips),
                               p2(,iChips),
                               p3(,iChips),
                               p5(,iChips));
        else if (typei(iChips)=="pwr")
            Ci = visPowerLaw(
                             Utable,Vtable,lambda, time,
                             p4(,iChips),
                             p2(,iChips),
                             p3(,iChips));
        else if (typei(iChips)=="exp")
            Ci = visExponential(
                                Utable,Vtable,lambda, time,
                                p4(,iChips),
                                p2(,iChips),
                                p3(,iChips));
        else if (typei(iChips)=="orb")
        {
            Ci = visOrbit(
                          Utable,Vtable,lambda, time,
                          p4(,iChips),  // size
                          p5(,iChips),  // type
                          p6(,iChips),  // t0
                          p7(,iChips),  // e
                          p8(,iChips),  // maj ax
                          p9(,iChips),  // period
                          p10(,iChips), // i
                          p11(,iChips), // w
                          p12(,iChips), // W
                          p2(,iChips),  // x0
                          p3(,iChips),  // y0
                          specTot);
            if(p5(1,iChips)>=3)
            {
                spec(,iChips) = spec(1,iChips)*specTot;
            }
            
            // if(numberof(Ci)==65536)
            // {
            //     c = grid2ima(Ci);
            //     yocoGuiWinKill;
            //     pli,abs(c),cmin=0,cmax=1;
            //     window,2;
            //     pli,atan(c.im,c.re);
            //     frwer();
            // }
            
            // if(numberof(Ci)!=65536)
            // {
            //     yocoGuiWinKill;
            //     plg,transpose(Ci.re)(*);
            //     plg,transpose(Ci.im)(*),color="red";
            //     write,Utable,Vtable;
            //         // write,lambda,time;
            //     frwer();
            // }
        }
        else if (typei(iChips)=="gam_wcz")
        {
            Ci = visGamWCZ(
                           Utable,Vtable,lambda, time,
                           p4(,iChips),  // zoom
                           p5(,iChips),  // t0
                           p6(,iChips),  // e
                           p7(,iChips),  // maj ax
                           p8(,iChips),  // period
                           p9(,iChips), // i
                           p10(,iChips), // w
                           p11(,iChips), // W
                           p2(,iChips),  // x0
                           p3(,iChips))  // y0
                }
        else if (typei(iChips)=="orb2")
        {
            Ci = visOrbit2(
                           Utable,Vtable,lambda, time,
                           p4(,iChips),
                           p5(,iChips),
                           p6(,iChips),
                           p7(,iChips),
                           p8(,iChips),
                           p9(,iChips),
                           p10(,iChips),
                           p11(,iChips),
                           p12(,iChips),
                           p13(,iChips),
                           p14(,iChips),
                           p15(,iChips),
                           p2(,iChips),
                           p3(,iChips),
                           specTot);
            if(p5(1,iChips)==3)
            {
                spec(,iChips) = spec(1,iChips)*specTot;
            }
        }
        else if (typei(iChips)=="l")
            Ci = visLorentzianDisk(
                                   Utable,Vtable,lambda, time,
                                   p4(,iChips),
                                   p2(,iChips),
                                   p3(,iChips));
        else if (typei(iChips)=="pwrdsk")
            Ci = visPowerDisk(
                              Utable,Vtable,lambda, time,
                              p4(,iChips),
                              0.0,
                              p5(,iChips),
                              0.0,
                              p2(,iChips),
                              p3(,iChips));
        else if (typei(iChips)=="expdsk")
            Ci = visExpDisk(
                            Utable,Vtable,lambda, time,
                            p4(,iChips),
                            0.0,
                            p5(,iChips),
                            0.0,
                            p2(,iChips),
                            p3(,iChips));
        else if (typei(iChips)=="pwrdsk2")
            Ci = visPowerDisk(
                              Utable,Vtable,lambda, time,
                              p4(,iChips),
                              p5(,iChips),
                              p6(,iChips),
                              p7(,iChips),
                              p2(,iChips),
                              p3(,iChips));
        else if (typei(iChips)=="expdsk2")
            Ci = visExpDisk(
                            Utable,Vtable,lambda, time,
                            p4(,iChips),
                            p5(,iChips),
                            p6(,iChips),
                            p7(,iChips),
                            p2(,iChips),
                            p3(,iChips));
        else if (typei(iChips)=="g")
            Ci = visGaussianDisk(
                                 Utable,Vtable,lambda, time,
                                 p4(,iChips),
                                 p2(,iChips),
                                 p3(,iChips));
        else if (typei(iChips)=="gl")
        {
            Ci = visLineDisk(
                             Utable,Vtable,lambda, time,
                             spec(,iChips), //fluxRaie,
                             p4(,iChips),   // FWHM
                             p2(,iChips),   // x0
                             p3(,iChips),   // y0
                             p5(,iChips),   // line wlen
                             p6(,iChips),   // line speed
                             p7(,iChips),   // line width
                             p8(,iChips),   // line depht
                             "g",
                             specTot);
            spec(,iChips) = specTot;
        }
        else if (typei(iChips)=="ggl")
        {
            Ci = visLineBiGaussian(
                                   Utable,Vtable,lambda, time,
                                   spec(,iChips), //fluxRaie,
                                   p4(,iChips),   // FWHM
                                   p2(,iChips),   // x0
                                   p3(,iChips),   // y0
                                   p5(,iChips),   // dx
                                   p6(,iChips),   // dy
                                   p7(,iChips),   // line wlen
                                   p8(,iChips),   // line speed
                                   p9(,iChips),   // line width
                                   p10(,iChips),  // line depht
                                   specTot);
            spec(,iChips) = specTot;
        }
        else if (typei(iChips)=="g2")      
            Ci = visEllipticalGaussianDisk(
                                           Utable,Vtable,lambda, time,
                                           p4(,iChips),
                                           p5(,iChips),
                                           p6(,iChips),
                                           p2(,iChips),
                                           p3(,iChips));
        else if (typei(iChips)=="r")
            Ci = visRing(
                         Utable,Vtable,lambda, time,
                         p4(,iChips),
                         p2(,iChips),
                         p3(,iChips));
        else if (typei(iChips)=="r2")      
            Ci = visEllipticalRing(
                                   Utable,Vtable,lambda, time,
                                   p4(,iChips),
                                   p5(,iChips),
                                   p6(,iChips),
                                   p2(,iChips),
                                   p3(,iChips));
        else if (typei(iChips)=="rg")
            Ci = visThickEllipticalRing(
                                        Utable, Vtable, lambda, time,
                                        p4(,iChips),
                                        p4(,iChips),
                                        0.0,
                                        p5(,iChips),
                                        "g",
                                        p2(,iChips),
                                        p3(,iChips));
        else if (typei(iChips)=="rg2")
            Ci = visThickEllipticalRing(
                                        Utable, Vtable, lambda, time,
                                        p4(,iChips),
                                        p5(,iChips),
                                        p6(,iChips),
                                        p7(,iChips),
                                        "g",
                                        p2(,iChips),
                                        p3(,iChips));
        else if (typei(iChips)=="rud")
            Ci = visThickEllipticalRing(
                                        Utable, Vtable, lambda, time,
                                        p4(,iChips),
                                        p4(,iChips),
                                        0.0,
                                        p5(,iChips),
                                        "ud",
                                        p2(,iChips),
                                        p3(,iChips));
        else if (typei(iChips)=="rud2")
            Ci = visThickEllipticalRing(
                                        Utable, Vtable, lambda, time,
                                        p4(,iChips),
                                        p5(,iChips),
                                        p6(,iChips),
                                        p7(,iChips),
                                        "ud",
                                        p2(,iChips),
                                        p3(,iChips));
        else if (typei(iChips)=="rr")
            Ci = visThickEllipticalRing(
                                        Utable, Vtable, lambda, time,
                                        p4(,iChips),
                                        p4(,iChips),
                                        0.0,
                                        p5(,iChips),
                                        "r",
                                        p2(,iChips),
                                        p3(,iChips));
        else if (typei(iChips)=="rr2")
            Ci = visThickEllipticalRing(
                                        Utable, Vtable, lambda, time,
                                        p4(,iChips),
                                        p5(,iChips),
                                        p6(,iChips),
                                        p7(,iChips),
                                        "r",
                                        p2(,iChips),
                                        p3(,iChips));
        else if (typei(iChips)=="sq")
            Ci = visSquare(
                           Utable,Vtable,lambda, time,
                           p4(,iChips),
                           p2(,iChips),
                           p3(,iChips));
        else if (typei(iChips)=="pwhl")      
            Ci = visPinwheel(Utable, Vtable, lambda, time,
                             p2(,iChips),// rounds,
                             p3(,iChips),// minThick,
                             p4(,iChips),// maxThick,
                             p5(,iChips),// totalSize,
                             p6(,iChips),// power,
                             p7(,iChips),// anglePhi,
                             p8(,iChips),// inclination,
                             p9(,iChips),// angleSky,
                             p10(,iChips) // tailFlux
                             );
        else if (typei(iChips)=="2pwhl")      
            Ci = visDoublePinwheel(Utable, Vtable, lambda, time,
                                   p2(,iChips),// rounds,
                                   p3(,iChips),// minThick,
                                   p4(,iChips),// maxThick,
                                   p5(,iChips),// totalSize,
                                   p6(,iChips),// power,
                                   p7(,iChips),// anglePhi,
                                   p8(,iChips),// inclination,
                                   p9(,iChips),// angleSky,
                                   p10(,iChips) // tailFlux
                                   );
        else if (typei(iChips)=="pwhltl")      
            Ci = visPinwheelTempLaw(Utable, Vtable, lambda, time,
                                    p2(,iChips),// rounds,
                                    p3(,iChips),// minThick,
                                    p4(,iChips),// maxThick,
                                    p5(,iChips),// totalSize,
                                    p6(,iChips),// power,
                                    p7(,iChips),// anglePhi,
                                    p8(,iChips),// inclination,
                                    p9(,iChips),// angleSky,
                                    p10(,iChips), //powerLawDecrease
                                    p11(,iChips),// tempInit
                                    p12(,iChips),// amplitudeSine
                                    p13(,iChips)// amortiSine
                                    );
        else if (typei(iChips)=="skwr")      
            Ci = visSkewedRing(Utable, Vtable, lambda, time,
                               p4(,iChips),// majorAxis,
                               p2(,iChips),// x0,
                               p3(,iChips),// y0,
                               p5(,iChips),// thick,
                               p6(,iChips),// angleSk,
                               p7(,iChips)// ampSk
                               );
        else if (typei(iChips)=="skwr2")      
            Ci = visEllipticalSkewedRing(Utable, Vtable, lambda, time,
                                         p4(,iChips),// majorAxis,
                                         p5(,iChips),// minorAxis,
                                         p6(,iChips),// angle,
                                         p2(,iChips),// x0,
                                         p3(,iChips),// y0,
                                         p7(,iChips),// thick,
                                         p8(,iChips),// angleSk,
                                         p9(,iChips)// ampSk
                                         );
        else if (typei(iChips)=="exprotdsk")
        {
            Ci = visModelExpRotDisk(Utable, Vtable, lambda, time,
                                    spec(,iChips), //fluxRaie,
                                    p2(,iChips), //fluxStar,
                                    p3(,iChips), //Rstar,
                                    p4(,iChips), //deepStar,
                                    p5(,iChips), //widthStar,
                                    p6(,iChips), //distance,
                                    p7(,iChips), //sizeCont,
                                    p8(,iChips), //size,
                                    p9(,iChips), //incl,
                                    p10(,iChips), //beta,
                                    p11(,iChips), //vrot,
                                    p12(,iChips), //vinf,
                                    p13(,iChips), //v0,
                                    p14(,iChips), //gamma,
                                    p15(,iChips), //lineCentralWlen,
                                    p16(,iChips), //naturalLineWidth,
                                    p17(,iChips), //PAsky
                                    specTot);
            spec(,iChips) = specTot;
        }
        else if (typei(iChips)=="disk")      
            Ci = visModelGasDisk(Utable, Vtable, lambda, time,
                                 spec(,iChips), //fluxStar,
                                 p2(,iChips), //Rstar,
                                 p3(,iChips), //deepStar,
                                 p4(,iChips), //widthStar,
                                 p5(,iChips), //fluxGas,
                                 p6(,iChips), //sizeCont,
                                 p7(,iChips), //sizeLine,
                                 p8(,iChips), //beta,
                                 p9(,iChips), //vrot,
                                 p10(,iChips), //vinf,
                                 p11(,iChips), //v0,
                                 p12(,iChips), //gamma,
                                 p13(,iChips), //lineCentralWlen,
                                 p14(,iChips), //naturalLineWidth,
                                 p15(,iChips), //incl
                                 p16(,iChips), //PAsky
                                 specTot);
        else if (typei(iChips)=="gdisk")
        {
            Ci = visModelGasPlusDustDisk(Utable, Vtable, lambda, time,
                                         spec(,iChips), //fluxStar,
                                         p2(,iChips), //Rstar,
                                         p3(,iChips), //deepStar,
                                         p4(,iChips), //widthStar,
                                         p5(,iChips), //fluxGas,
                                         p6(,iChips), //sizeCont,
                                         p7(,iChips), //sizeLine,
                                         p8(,iChips), //beta,
                                         p9(,iChips), //vrot,
                                         p10(,iChips), //vinf,
                                         p11(,iChips), //v0,
                                         p12(,iChips), //gamma,
                                         p13(,iChips), //lineCentralWlen,
                                         p14(,iChips), //naturalLineWidth,
                                         p15(,iChips), //fluxDust,
                                         p16(,iChips), //dismDust,
                                         p17(,iChips), //thickDust
                                         p18(,iChips), //incl
                                         p19(,iChips), //PAsky
                                         specTot);
            spec(,iChips) = specTot(1,);
        }
        else if (typei(iChips)=="sq2")      
            Ci = visRectangle(
                              Utable,Vtable,lambda, time,
                              p4(,iChips), // majorAxis
                              p5(,iChips), // minorAxis
                              p6(,iChips), // angle
                              p2(,iChips), // x0
                              p3(,iChips)  // y0
                              );
        else if (typei(iChips)=="no")
        {
            Ci_centered = array(1.0,numberof(Utable),numberof(lambda));
            Ci = shiftFourier(Utable, Vtable, lambda, Ci_centered,
                              p2(,iChips),
                              p3(,iChips));
        }
        else if (typei(iChips)=="ful")
        {
            Ci = array(0.0,numberof(Utable),numberof(lambda));
        }
        else
            error;

        corrFluxTmp = corrFluxTmp + spec(,iChips)(-,) * Ci;
    }

    corrFlux = corrFluxTmp;
    flux     = spec(,sum)(-,);
    vis      = corrFluxTmp / (spec(,sum)(-,)+(spec(,sum)(-,)==0));

    return vis;
}
            
/***************************************************************************/
             
func UVLine(basemin, basemax, angle, nPoints)
    /* DOCUMENT UVLine(basemin, basemax, angle, nPoints)

       DESCRIPTION

       PARAMETERS
       - basemin: 
       - basemax: 
       - angle  : 
       - nPoints: 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    if(is_void(basemin))
        basemin = 0.0000000000001;
    if(!is_void(nPoints))
        N = nPoints;
    else
        N = 10;
    
    UVtable=array(0.0,2,N);
      
    x=span(basemin,basemax,N);
    
    UVtable(1,)=x*sin(angle);
    UVtable(2,)=x*cos(angle);
    
    return UVtable;
}
            
/***************************************************************************/
             
func UVGrid(basemax, nPoints)
    /* DOCUMENT UVGrid(basemax, nPoints)

       DESCRIPTION

       PARAMETERS
       - basemax: 
       - nPoints: 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    if(!is_void(nPoints))
        N = nPoints;
    else
        N = 100;
    
    Utable = array(span(-basemax,basemax,N),N);
    Vtable = transpose(Utable);

    uTab = Utable(*);
    vTab = Vtable(*);
    
    UVtable = array(0.0,2,N*N);
  
    UVtable(1,) = uTab;
    UVtable(2,) = vTab;
             
    return UVtable;
}
            
/***************************************************************************/
             
func grid2ima(lintab, which)
    /* DOCUMENT grid2ima(lintab, which)

       DESCRIPTION

       PARAMETERS
       - lintab: 
       - which : 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    N = sqrt(dimsof(lintab)(2));
    if(int(N)!=N)
        write,"Not a NxN image";
    N=int(N);
    
    return reform(lintab,[2,N,N]);
}
            
/***************************************************************************/
             
func grid2ima2(twocoltab)
    /* DOCUMENT grid2ima2(twocoltab)

       DESCRIPTION

       PARAMETERS
       - twocoltab: 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
             
    N=sqrt(dimsof(twocoltab)(3));
    if(int(N)!=N) write,"Not a NxN image";
    N=int(N);
    ima=array(0.0,2,N,N);
             
    for (i=1;i<=N;i++)
    {
        for (j=1;j<=N;j++)
        {
            ima(,,i) = twocoltab(,(i-1)*N+1:i*N);
        }
    }
    return ima;
}
            
/***************************************************************************/
             
func ima2Grid(inIma)
    /* DOCUMENT ima2Grid(inIma)

       DESCRIPTION

       PARAMETERS
       - inIma: 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    d = dimsof(inIma)(2:);
    N=1;
    for(k=1;k<=numberof(d);k++)
        N *= d(k);
    return reform(inIma,[1,N]);
}
            
/***************************************************************************/
            
func UVTable2Bases(UVTable)
    /* DOCUMENT UVTable2Bases(UVTable)

       DESCRIPTION

       PARAMETERS
       - UVTable: 

       RESULTS 

       CAUTIONS 

       EXAMPLES 

       SEE ALSO:
    */
{
    bases = sqrt(UVTable(1,)^2 + UVTable(2,)^2);
    return bases;  
}
            
/***************************************************************************/
  
func corrFxPoints(Utable, Vtable, lambda, time, p4, p5, p6)
    /* DOCUMENT corrFxPoints(Utable, Vtable, lambda, time, p4, p5, p6)

       DESCRIPTION

       PARAMETERS
       - Utable: 
       - Vtable: 
       - lambda: 
       - time  : 
       - p4    : 
       - p5    : 
       - p6    : 

       RETURN VALUES

       CAUTIONS

       EXAMPLES

       SEE ALSO
    */
{
    nChips  = numberof(typei);
    nBase   = dimsof(Utable)(2);
    nLambda = numberof(lambda);
    C = 0;
        
    for(iChips=1;iChips<=nChips;iChips++)
    {
        Ci_centered = array(1.0,numberof(Utable),numberof(lambda));
        Ci = shiftFourier(Utable, Vtable, lambda, Ci_centered, p4(iChips), p5(iChips));
        
        C = C + p6(,iChips)(-,) * Ci;
    }
    
    return C;
}
