pro extract_visdata, file=file, oivis=oivis_input, $
 oiwavelength=oiwavelength_input, oitarget=oitarget_input,visdata  ,$
 status=status

;+
; NAME:
;     extract_uvdata
; PURPOSE:
;     To extract uvdata into 'usuable' IDL variables for plotting,
;     inspecting data, etc.  Can either be used to read directly from
;     FITS file or by passing the required OITABLES. Will also allow 
;     selection parameters
;
; CALLING SEQUENCE (Examples):
;     extract_visdata, visdata, file=fits_file, $
;             eff_wave =[1.8,2.3], target_id =[1,2]
;           or
;     extract_visdata, visdata, oivis= oivistable,$
;             oiwavelength=oiwavetable, oitarget=oitargettable,$
;              insname=['IOTA_IONIC3']
;
; INPUTS:
;
;   KEYWORDS:

;   One must pass EITHER 
;     FILE      - Name of fits file to read. Must be in
;
;            or All
;   OIWAVELENGTH Contains information on the observing bandpasses.
;      OIVIS     Contains visibility amplitude measurements.
;      OITARGET  Contains information on the target names
;
;    -
;
; OPTIONAL INPUTS:
;
;     Various keywords can be used to select subsets of the data that
;     pass certain selection criteria.  This section will be expanded
;     and new capabilities are added.
;
;     /STATUS    Print out status messages as data is being Extracted
;
; OUTPUTS:
;
;      VISDATA   A structure which contains in a simple format all the
;                requested data.   Does not use pointers and so all
;                data can be easily searched and plotted in IDL.
;                See define_vis2data.pro for detailed description of
;                this structure
;
; RESTRICTIONS:
;      None.
;
; PROCEDURE:
;      Call from command line or in a program.
;
;
;      The Data Exchange Standard for Optical (Visible/IR) Interferometry
;      is being maintained by the IAU Working group on Optical Interferometry
;      and the current version of the standard can be found at :
;            http://www.mrao.cam.ac.uk/~jsy1001/exchange/
;
; EXAMPLE:


;
; PROCEDURES USED:
;	This routine is part of the IDL Optical Interferometry (IOI) Library
;	 (more information at www.astro.lsa.umich.edu/~monnier).
;       The routines of the IOI Library generically 
;       require an up-to-date Astrolib library in order to read and write binary
;       FITS tables, among other things. The IDL Astronomy User's Library can
;       be found at http://idlastro.gsfc.nasa.gov/homepage.html.
;	
;
; MODIFICATION HISTORY:
;     v0.0 2003Jul04    J.D. Monnier    Initiated
;     v0.1 2005May30    J.D. Monnier	Added target names.
;     v0.2 2006Oct22    J.D. Monnier	Added sfu
;     v0.3 2009Mar22    J.D. Monnier    Added, ra, dec, equi
;          2013Mar06    G.H. Schaefer   Create extract_visdata.pro
;
;     To do: A.  Add Revision Checking (once multiple revisions exist)  
;	     B.  Add /gui or /interactive options to do fancy
;                selections.
;            C.  Better options for SORTING by TIME, baseline, etc.
;            D.  Better error checking
;  *** Please write monnier@umich.edu with bug reports. ***
;-

if (keyword_set(file) eq 1) then begin
  read_oidata, file, oiarray,oitarget,oiwavelength,oivis,oivis2,oit3
endif else begin
  oiwavelength = oiwavelength_input
  oivis =oivis_input
  oitarget=oitarget_input
endelse

; Add error checking.
if n_elements(oivis) eq 0 then begin
  print,'NO DATA'
  visdata=-1
  return
endif


; First approach [maybe too slow and memory hog for big data sets!]
;    make big long array.
;    apply cuts at the end using a WHERE statements.
;

lastarrname = ' '
;init
define_visdata,visdata_unit
insnames = strtrim( oiwavelength.insname,2)
vis_insnames =strtrim(oivis.insname,2)
for i=0,n_elements(oivis)-1 do begin
  oiwave_index =  (where(insnames eq vis_insnames(i)))(0)
   nwave = oiwavelength(oiwave_index).nwave
       ; Not supposed to ever crash in the above line since
       ; oidata standard requires wavelength tables for each insnames

if (keyword_set(status) eq 1) then begin
   if (oivis(i).arrname ne lastarrname) then begin
       pe= strtrim(string(fix(100.*i/n_elements(oivis))),2)
       print, pe+'% Complete:  Beginning extraction of ARRNAME : ',oivis(i).arrname 
       lastarrname = oivis(i).arrname
   endif
endif

   vis_new = replicate(visdata_unit,nwave)
   
   vis_new(*).oi_revn = oivis(i).oi_revn
   vis_new(*).date_obs= oivis(i).date_obs
   vis_new(*).arrname = strtrim(oivis(i).arrname,2)
   vis_new(*).insname = strtrim(oivis(i).insname,2)
   vis_new.eff_wave   = *oiwavelength(oiwave_index).eff_wave
   vis_new.eff_band   = *oiwavelength(oiwave_index).eff_band
   vis_new.wave_id    = indgen(nwave)
   vis_new(*).target_id  = oivis(i).target_id
   tin=where( oitarget.target_id eq oivis(i).target_id,ct ) ; NO ERRORS!!
    if ct eq 0 then begin
        print,' Target ID not in OITARGET table!!'
	stop
        return
    endif
   vis_new(*).target     = oitarget(tin).target
   vis_new(*).ra 	 = oitarget(tin).raep0
   vis_new(*).dec	 = oitarget(tin).decep0
   vis_new(*).equinox    = oitarget(tin).equinox
   vis_new(*).time       = oivis(i).time
   vis_new(*).mjd        = oivis(i).mjd
   vis_new(*).int_time   = oivis(i).int_time
   vis_new.visamp       =*oivis(i).visamp
   vis_new.visamperr        =*oivis(i).visamperr
   vis_new.visdata       =*oivis(i).visamp
   vis_new.viserr        =*oivis(i).visamperr
   vis_new.visphi       =*oivis(i).visphi
   vis_new.visphierr        =*oivis(i).visphierr
   vis_new(*).ucoord     = oivis(i).ucoord
   vis_new(*).vcoord     = oivis(i).vcoord
   vis_new(*).sta_index  = oivis(i).sta_index
   vis_new.flag          =*oivis(i).flag
  
; do baseline geometry calc at the end..

   if (i eq 0) then visdata=vis_new else visdata=concat_oitable(visdata,vis_new)
   
endfor
  
; Geometry
   visdata.u = visdata.ucoord/visdata.eff_wave
   visdata.v = visdata.vcoord/visdata.eff_wave
   ri2at, visdata.vcoord, visdata.ucoord, amp, theta ; not a bug (angle E of No)
   visdata.baseline = amp
   visdata.pa       = theta
   visdata.sfu = visdata.baseline/visdata.eff_wave
end


