;*******************************************************************************
; File: ameeba.pro
;
; Description:
; ------------
; Main container of IDL scripts for AMOEBA software. AMOEBA is closely
; related to CHAMELEON wrt shared common blocks, procedures, and structure.
; AMOEBA will fit models of stellar systems to data from interferometry,
; spectroscopy, photometry, etc.
;
; Block directory:
; ----------------
; Block 1: list_starcomps,list_bincomps,list_modcomp,list_modcomps
;          list_poscomps,list_velcomps,list_magcomps,
;	   list_wavelengths,list_filters,
;	   list_summary_amoeba
;
; Block 2: load_interferometry,load_astrometry,load_spectroscopy,load_photometry
;	   set_parallax,set_k2,add_component
;
; Block 3: storenight,loadnight,freememory
;
; Block 4: setfluxes,fitwaveparms,adjustfluxes,adjustmasses
;
; Block 5: readmodel,calcmodel,readimage,readimages,marquardtdata
;
; Block 6: mark32cha,mark32psn,psn2tex,pti2tex,vel2tex,roger2vel,
;	   compile_speckle
;
;************************************************************************Block 1
pro list_starcomps,components
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
components=''
if n_elements(star_model) eq 0 then begin
	if n_params() eq 0 then $
		print,'***Error(LIST_COMPONENTS): star_model not defined!'
	return
endif
if num_star() gt 0 then components=star_model.component
if n_params() eq 0 then print,components
;
end
;-------------------------------------------------------------------------------
pro list_bincomps,components
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
components=''
if n_elements(binary_model) eq 0 then begin
	if n_params() eq 0 then $
		print,'***Error(LIST_COMPONENTS): binary_model not defined!'
	return
endif
if num_binary() gt 0 then components=binary_model.component
if n_params() eq 0 then print,components
;
end
;-------------------------------------------------------------------------------
pro list_modcomp,component
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then return
if n_elements (component) eq 0 then return
;
print,'_______________________________________________________________'
index=where(binary_model.component eq component,count)
if count gt 0 then begin
	tags=tag_names(binary_model)
	for i=0,n_elements(tags)-1 do begin
		if n_elements(binary_model[index].(i)) eq 1 then begin
			result=size(binary_model[index].(i))
			if result[n_elements(result)-2] eq 5 then $
			print,tags[i],'=',binary_model[index].(i), $
				  ' +/- ',binary_error[index].(i) else $
			print,tags[i],'=',binary_model[index].(i)
		endif else begin
			print,tags[i],'=',binary_model[index].(i)
			blank=' '
			for j=0,strlen(tags[i])-6 do blank=blank+' '
			print,blank+' +/- ',binary_error[index].(i)
		endelse
	endfor
endif
tcount=count
index=where(star_model.component eq component,count)
if count gt 0 then begin
	tags=tag_names(star_model)
	for i=0,n_elements(tags)-1 do begin
		if n_elements(star_model[index].(i)) eq 1 then begin
			result=size(star_model[index].(i))
			if result[n_elements(result)-2] eq 5 then $
			print,tags[i],'=',star_model[index].(i), $
		          	  ' +/- ',star_error[index].(i) else $
			print,tags[i],'=',star_model[index].(i)
		endif else begin
			print,tags[i],'=',star_model[index].(i)
			blank=' '
                        for j=0,strlen(tags[i])-6 do blank=blank+' '
			print,blank+' +/- ',star_error[index].(i)
		endelse
	endfor
endif
tcount=tcount+count
if tcount eq 0 then begin
	tags=tag_names(gen_model)
	for i=0,n_elements(tags)-1 do begin
		print,tags[i],'=',gen_model.(i)
		blank=' '
                for j=0,strlen(tags[i])-6 do blank=blank+' '
		print,blank+' +/- ',gen_error.(i)
	endfor
endif
print,'______________________________***______________________________'
;
end
;-------------------------------------------------------------------------------
pro list_modcomps
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
help,gen_model,/structure
if n_elements(gen_model) eq 0 then return
;
for i=0,num_binary()-1 do begin
        help,binary_model[i],/structure
        hak,mesg='Hit any key to continue...'
endfor
for i=0,num_star()-1 do begin
        help,star_model[i],/structure
        if i lt num_star()-1 then hak,mesg='Hit any key to continue...' $
                else print,'Info complete.'
endfor
;
end
;-------------------------------------------------------------------------------
pro list_poscomps,components
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
components=''
if n_elements(positions) eq 0 then begin
	print,'***Error(LIST_POSCOMPS): no data!'
	return
endif
components=[positions.component]
components=components[uniq(components,sort(components))]
;
if n_params() eq 0 then print,components
;
end
;-------------------------------------------------------------------------------
pro list_velcomps,components
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
components=''
if n_elements(velocities) eq 0 then begin
	print,'***Error(LIST_VELCOMPS): no data!'
	return
endif
components=[velocities.component]
components=components[uniq(components,sort(components))]
;
if n_params() eq 0 then print,components
;
end
;-------------------------------------------------------------------------------
pro list_magcomps,components
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
components=''
if n_elements(magnitudes) eq 0 then begin
	print,'***Error(LIST_MAGCOMPS): no data!'
	return
endif
components=[magnitudes.component]
components=components[uniq(components,sort(components))]
;
if n_params() eq 0 then print,components
;
end
;-------------------------------------------------------------------------------
pro list_filters,filters
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
filters=''
if n_elements(magnitudes) eq 0 then begin
        print,'***Error(LIST_WAVELENGTHS): no data!'
        return
endif
filters=[magnitudes.filter]
filters=filters[uniq(filters,sort(filters))]
;
if n_params() eq 0 then print,filters
;
end
;-------------------------------------------------------------------------------
pro list_summary_amoeba
;
common ModelFit,parameters,ds_options
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common AuxData,parallaxes,k2,vsini
;
print,''
print,'_________________________________________________________'
if n_elements(scans) ne 0 then begin
	print,'Interferometry loaded, weight=',ds_options.i
	print,'    Number of nights:',n_elements(GeoInfo.date)
endif
if n_elements(positions) ne 0 then begin
	print,'Astrometry loaded, weight=',ds_options.a
	print,'    Number of positions:',n_elements(positions)
	list_poscomps,poscomps
	print,'    Components: ',poscomps
endif
if n_elements(velocities) ne 0 then begin
	print,'Spectroscopy loaded, weight=',ds_options.s
	print,'    Number of velocities:',n_elements(velocities)
	list_velcomps,velcomps
	print,'    Components: ',velcomps
endif
if n_elements(magnitudes) ne 0 then begin
	print,'Photometry loaded, weight=',ds_options.p
	print,'    Number of magnitudes:',n_elements(magnitudes)
endif
if n_elements(parallaxes) ne 0 then begin
	print,'Parallax set, weight=',ds_options.px
endif
if n_elements(k2) ne 0 then begin
	print,'K2 set, weight=',ds_options.k2
endif
if n_elements(vsini) ne 0 then begin
	print,'Vsin(i) set, weight=',ds_options.vsini
endif
print,'___________________________***___________________________'
;
end
;************************************************************************Block 2
pro load_interferometry,chafiles
;
; Procedure to load multiple .cha scan data files and store each in
; memory using storenight. Expand GenConfig and GeoParms into arrays of
; structures, geninfo and geoinfo, respectively, to have configuration
; data readily available.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,scantable,bgtable,stationtable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common StarBase,StarTable,Notes
;
freememory
;
num_files=n_elements(chafiles)
if num_files eq 0 then begin
	print,'***Error(LOAD_INTERFEROMETRY): no files specified!'
	return
endif
;
files=strcompress(chafiles,/remove_all)
first=1
j=0
;
for i=0,num_files-1 do begin
	result=findfile(files[i],count=fcount)
	if fcount ne 0 then begin
		if strpos(files[i],'.xdr') ge 0 then begin
			get_xdr,files[i],/nostore
			get_stationtable
;			stars=startable.name
;			get_startable,startable.starid
;			startable.name=stars
		endif else begin
			hds_open,files[i],'READ',status
			if status ne 0 then return
			get_sysconfig,no_alloc=(first eq 0)
			get_stationtable,update=0
			get_scantable
			get_scans
			list_stars,stars
			get_startable,stars
			hds_close
		endelse
		if first then begin
			first=0
			bufferinfo=replicate(nightinfo(),num_files)
			GeoInfo=replicate(GeoParms,num_files)
;			GenInfo=replicate(GenConfig,num_files)
;			We do not always have enough memory for the above
			GenInfo=replicate(allocgenconfig(/geninfo),num_files)
			table=startable
			sttbl=stationtable
		endif else begin
			table=merge_startable(table,startable)
			sttbl=mergetable(sttbl,stationtable)
		endelse
		GeoInfo[j]=GeoParms
;		GenInfo(j)=GenConfig
		G=GenInfo[j]
		struct_assign,GenConfig,G
		GenInfo[j]=G
		calcastrom
		storenight
		j=j+1
	endif else print,'***Error(LOAD_INTERFEROMETRY): file not found: ', $
			files[i],'!'
endfor
if n_elements(table) eq 0 then return else StarTable=table
;
; Get names and diameters for FKV and BSC stars in the startable
starids=startable.starid
stars=startable.name
rename_starids,'fkv-bsc'
index=where(startable.bsc ne 0,count)
get_diameter
t=startable
read_catalogs
rename_bsc
if count gt 0 then stars[index]=startable[index].name
startable=t
startable.starid=starids
startable.name=stars
;
index=uniq(sttbl.stationid,sort(sttbl.stationid))
stationtable=sttbl[index]
;
plotinit=init_plot('amoeba','pt')
;
end
;-------------------------------------------------------------------------------
pro load_astrometry,files
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
num_files=n_elements(files)
if num_files eq 0 then begin
	print,'***Error(LOAD_ASTROMETRY): no files specified!'
	return
endif
;
for i=0,n_elements(files)-1 do begin
	c=''
	y=0.d0
	r=0.d0
	t=0.d0
	a=0.d0
	b=0.d0
	p=0.d0
	status=dc_read_free(files[i],c,y,r,t,a,b,p, $
		/col,ignore=['!'],resize=[1,2,3,4,5,6,7])
	if i eq 0 then begin
		component=c
		rho=r
		theta=t
		emajor=a
		eminor=b
		pa=p
		jy=y
	endif else begin
		component=[component,c]
		rho=[rho,r]
		theta=[theta,t]
		emajor=[emajor,a]
		eminor=[eminor,b]
		pa=[pa,p]
		jy=[jy,y]
	endelse
endfor
;
num_scan=n_elements(jy)
positions=replicate(location(),num_scan)
;
RAD=180/pi_circle
;
positions.component=component
positions.rho=rho
positions.theta=theta/RAD	; Note internal units of radians!
positions.emajor=emajor		; Semi-major axis of uncertainty ellipse [mas]
positions.eminor=eminor		; Semi-minor axis of uncertainty ellipse [mas]
positions.pa=pa/RAD		; Note internal units of radians!
positions.jy=jy
;
positions.jd=jy2jd(jy)-2440000d0
jd2date,positions.jd+2440000d0,y,m,d
positions.date=constrictordate(y,m,d)
;
; Code to fake positions
; RAD=180/3.141592653d0
; a=109.d0
; e=0.2d0
; i=32.d0/RAD
; w=99.d0/RAD
; n=30.d0/RAD
; p=974.d0
; t=5675.d0
; o_parms=dblarr(8)
; o_parms(0)=a
; o_parms(1)=e
; o_parms(2)=i
; o_parms(3)=w
; o_parms(4)=n
; o_parms(5)=p
; o_parms(6)=t
; o_parms(7)=1
; jd=positions.jd
; xy=true2app(jd,o_parms,rho,theta)
; positions.theta=theta
; positions.rho=rho
;
print,'Positions read.'
end
;-------------------------------------------------------------------------------
pro load_spectroscopy,files
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
num_files=n_elements(files)
if num_files eq 0 then begin
	print,'***Error[LOAD_SPECTROSCOPY]: no files specified!'
	return
endif
;
for i=0,n_elements(files)-1 do begin
	c=''
	hjd=0.d0
	v=0.d0
	e=0.d0
	o=''
	status=dc_read_free(files[i],c,hjd,v,e,o, $
		/col,ignore=['!'],resize=[1,2,3,4,5])
	if i eq 0 then begin
		component=c
		jd=hjd
		value=v
		error=e
		comment=o
	endif else begin
		component=[component,c]
		jd=[jd,hjd]
		value=[value,v]
		error=[error,e]
		comment=[comment,o]
	endelse
endfor
;
num_scan=n_elements(jd)
velocities=replicate(velocity(),num_scan)
;
RAD=180/pi_circle
;
velocities.component=component
velocities.value=value
velocities.error=error
velocities.jd=jd-2440000d0
velocities.comment=comment
;
jd2date,velocities.jd+2440000d0,y,m,d
velocities.date=constrictordate(y,m,d)
;
print,'Velocities read.'
end
;-------------------------------------------------------------------------------
pro load_photometry,files
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
num_files=n_elements(files)
if num_files eq 0 then begin
	print,'***Error[LOAD_PHOTOMETRY]: no files specified!'
	return
endif
;
for i=0,n_elements(files)-1 do begin
	c=''
	f=''
	hjd=0.d0
	m=0.d0
	e=0.d0
	status=dc_read_free(files[i],c,f,hjd,m,e, $
		/col,ignore=['!'],resize=[1,2,3,4,5])
	if i eq 0 then begin
		component=c
		filter=f
		jd=hjd
		value=m
		error=e
	endif else begin
		component=[component,c]
		filter=[filter,f]
		jd=[jd,hjd]
		value=[value,m]
		error=[error,e]
	endelse
endfor
;
num_scan=n_elements(jd)
magnitudes=replicate(magnitude(),num_scan)
;
RAD=180/pi_circle
;
magnitudes.component=component
magnitudes.filter=filter
magnitudes.value=value
magnitudes.error=error
magnitudes.jd=jd-2440000d0
;
jd2date,magnitudes.jd+2440000d0,y,m,d
magnitudes.date=constrictordate(y,m,d)
;
print,'Magnitudes read.'
end
;-------------------------------------------------------------------------------
pro set_parallax,value,error,weight
;
; Units of parallax are mas. Reset to "no parallax" if no inputs defined.
;
common AuxData,parallaxes,k2,vsini
common ModelFit,parameters,ds_options
;
if n_params() eq 1 then begin
	print,'***Error[SET_PARALLAX]: please provide value AND error!'
	return
endif
if n_elements(weight) eq 0 then weight=1 
if n_elements(parallaxes) ne 0 and weight eq 0 then return
if n_params() eq 0 then ds_options.px=0 else begin
	parallaxes=alloc_parallax()
	parallaxes.value=value
	parallaxes.error=error
	ds_options.px=weight
endelse
;
end
;-------------------------------------------------------------------------------
pro set_k2,value,error,weight
;
common AuxData,parallaxes,k2,vsini
common ModelFit,parameters,ds_options
;
if n_params() eq 1 then begin
	print,'***Error[SET_K2]: please provide value AND error!'
	ds_options.k2=0
	return
endif
if n_params() eq 0 then ds_options.k2=0 else begin
	k2=alloc_k2()
	k2.value=value
	k2.error=error
	if n_elements(weight) eq 0 then weight=1 
	ds_options.k2=weight
endelse
;
end
;-------------------------------------------------------------------------------
pro set_vsini,value,error,weight
;
common AuxData,parallaxes,k2,vsini
common ModelFit,parameters,ds_options
;
if n_params() eq 1 then begin
	print,'***Error[SET_VSINI]: please provide value AND error!'
	ds_options.vsini=0
	return
endif
if n_params() eq 0 then ds_options.vsini=0 else begin
	vsini=alloc_vsini()
	vsini.value=value
	vsini.error=error
	if n_elements(weight) eq 0 then weight=1 
	ds_options.vsini=weight
endelse
;
end
;-------------------------------------------------------------------------------
pro add_component,component,origin,reference
;
; e.g. add_component,'B-C','A-C','A-B' does B-C=A-C-(A-B)
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
n_old=n_elements(positions)
index=where(positions.component eq origin,n_new)
new_positions=replicate(location(),n_old+n_new)
new_positions[0:n_old-1].component=positions.component
new_positions[0:n_old-1].date=positions.date
new_positions[0:n_old-1].jy=positions.jy
new_positions[0:n_old-1].jd=positions.jd
new_positions[0:n_old-1].rho=positions.rho
new_positions[0:n_old-1].theta=positions.theta
new_positions[0:n_old-1].rhom=positions.rhom
new_positions[0:n_old-1].thetam=positions.thetam
new_positions[0:n_old-1].emajor=positions.emajor
new_positions[0:n_old-1].eminor=positions.eminor
new_positions[0:n_old-1].pa=positions.pa
;
RAD=180/!pi
jd0=2440000.d0
;
for i=0,n_new-1 do begin
	p=reform(binarypos(jd0+positions[index[i]].jd,reference))
	rar=p[0]*sin(p[1]/RAD)
	dcr=p[0]*cos(p[1]/RAD)
	rao=positions[index[i]].rho*sin(positions[index[i]].theta)
	dco=positions[index[i]].rho*cos(positions[index[i]].theta)
	dra=rao-rar
	ddc=dco-dcr
	new_positions[n_old+i].component=component
	new_positions[n_old+i].rho=sqrt(dra^2+ddc^2)
	new_positions[n_old+i].theta=atan(dra,ddc)
	new_positions[n_old+i].date=positions[index[i]].date
	new_positions[n_old+i].jy=positions[index[i]].jy
	new_positions[n_old+i].jd=positions[index[i]].jd
	new_positions[n_old+i].emajor=positions[index[i]].emajor
	new_positions[n_old+i].eminor=positions[index[i]].eminor
	new_positions[n_old+i].pa=positions[index[i]].pa
endfor
;
positions=new_positions
;
end	
;************************************************************************Block 3
pro storenight,mode
;
; Store one night's data (scans) in memory (or on disk if too big). 
; This data can be retrieved later with loadnight.
; mode=10:	Store new night
; mode=11:	Overwrite existing night
;
; Note that GenInfo and bufferinfo already have to have an entry for
; this currently loaded night before this function can be called.
;
common LocalAmoebaBuffer,BUFFERSIZE,BUFFERLIMIT
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; Update the complex triple data
set_compltriple
;
; Determine which index of bufferinfo is in use
cdate=checkdate()
arrayid=GeoParms.systemid
configid=GenConfig.configid
j=where(GenInfo.date eq cdate $
    and GeoInfo.SystemId eq arrayid $
    and GenInfo.ConfigId eq configid)
j=j[0]
;
if n_elements(mode) eq 0 then mode=10L
mode=long(mode)
;
BUFFERLIMIT=200000000L
if n_elements(BUFFERSIZE) eq 0 then BUFFERSIZE=0
;
if sizeof(scans)+BUFFERSIZE gt BUFFERLIMIT then begin
;
	bufferinfo[j].xdr=1
	save,scans,genconfig,filename=cdate+arrayid+configid+'.xdr'
;
endif else begin
;
	bufferinfo[j].xdr=0
;
	if mode ne 11 then $
	BUFFERSIZE=BUFFERSIZE+sizeof(scans)+sizeof(genconfig)
;
	bufferinfo[j].nigc.numsid=genconfig.numsid
	bufferinfo[j].nigc.numoutbeam=genconfig.numoutbeam
	bufferinfo[j].nigc.maxtriple=n_elements(genconfig.triplenumchan)
	bufferinfo[j].nigc.maxbaseline=n_elements(genconfig.baselineid[*,0])
	bufferinfo[j].nigc.maxchannel=n_elements(genconfig.wavelength[*,0])
	bufferinfo[j].nigc.maxconfig=n_elements(genconfig.config)
;
	n=0
	fields=tag_names(genconfig)
	for i=0,n_elements(fields)-1 do begin
		data=genconfig.(i)
		dims=size(data)
		if dims[dims[0]+1] eq 7 then begin
                        sl=strlen(data)
                        index=where(sl eq 0,count)
                        if count gt 0 then data[index]=' '
			bufferinfo[j].nigc.maxlen(n)=max(strlen(data))
			n=n+1
		endif
		bufferinfo[j].nigc.size(i,0:n_elements(dims)-1)=dims
		status=linknload(!external_lib,'nightbuffer', $
			dims,data,cdate+arrayid+configid,mode)
		if status then begin
			print,'***Error(STORENIGHT): error storing GenConfig!'
			return
		endif
	endfor
;
	bufferinfo[j].nisc.numscan=n_elements(scans)
;
	n=0
	fields=tag_names(scans)
	for i=0,n_elements(fields)-1 do begin
		data=scans.(i)
		dims=size(data)
		if dims[dims[0]+1] eq 7 then begin
                        sl=strlen(data)
                        index=where(sl eq 0,count)
                        if count gt 0 then data[index]=' '
			bufferinfo[j].nisc.maxlen(n)=max(strlen(data))
			n=n+1
		endif
		bufferinfo[j].nisc.size(i,0:n_elements(dims)-1)=dims
		status=linknload(!external_lib,'nightbuffer', $
			dims,data,cdate+arrayid+configid,mode)
		if status then begin
			print,'***Error(STORENIGHT): error storing scans!'
			return
		endif
	endfor
;
endelse
;
end
;-------------------------------------------------------------------------------
pro loadnight,datum,arrayid,configid,force=force
;
; Load specified night from memory into internal data structure scans.
; Datum can be a combination of date (YYYY-MM-DD) and config string, but
; the latter must not contain any blanks.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
datum=datum[0]
words=nameparse(datum)
datum=words[0]
if n_elements(words) eq 3 then begin
	arrayid=words[1]
	configid=words[2]
endif
if n_elements(arrayid) eq 1 then arrayid=arrayid[0]
if n_elements(configid) eq 1 then configid=configid[0]
;
; Force=1 to force reading from buffer
if n_elements(force) eq 0 then force=1 else force=force gt 0
if not force and datum eq Date then return
;
; ConfigId enables storing different configurations in the same night
if n_elements(configid) eq 0 then configid='STANDARD'
;
Date=datum
;
j=where(GenInfo.date eq Date $ 
    and GeoInfo.SystemId eq arrayid $
    and GenInfo.configid eq configid,count)
if count eq 0 then begin
        set=Date+' '+arrayid+' '+configid
	print,'***Error(LOADNIGHT): this night was not stored before: '+set+'!'
	return
endif
j=j[0]
;
; This is now buffered
; GenConfig=GenInfo(j)
GeoParms=GeoInfo[j]
;
mode=20L
;
if bufferinfo[j].xdr then begin
;
; This restores both scans and GenConfig
restore,date+arrayid+configid+'.xdr'
;
endif else begin
;
GenConfig=allocGenConfig(bufferinfo[j].nigc.numsid, $
			 bufferinfo[j].nigc.numoutbeam, $
			 bufferinfo[j].nigc.maxtriple, $
			 bufferinfo[j].nigc.maxbaseline, $
			 bufferinfo[j].nigc.maxchannel, $
			 bufferinfo[j].nigc.maxconfig)
;
n=0
fields=tag_names(GenConfig)
for i=0,n_elements(fields)-1 do begin
	data=GenConfig.(i)
	dims=size(data)
	if dims[dims[0]+1] eq 7 then begin
		blank=''
		for k=0,bufferinfo[j].nigc.maxlen(n)-1 do blank=blank+' '
		data[*]=blank
		n=n+1
	endif
	status=linknload(!external_lib,'nightbuffer', $
		dims,data,Date+arrayid+configid,mode)
	if status then begin
		print,'***Error(LOADNIGHT): error loading data!'
		return
	endif
	if dims[dims[0]+1] eq 7 then data=strtrim(data,2)
	GenConfig.(i)=data
endfor
;
; Note: scan() must return dimensions compatible with GenConfig!
num_scans=bufferinfo[j].nisc.numscan
scans=replicate(scan(),num_scans)
;
n=0
fields=tag_names(scans)
for i=0,n_elements(fields)-1 do begin
	data=scans.(i)
	dims=size(data)
;	dims=bufferinfo(j).nisc.size(i,*)
;	data=make_array(size=dims(where(dims ne 0)))
	if dims[dims[0]+1] eq 7 then begin
		blank=''
		for k=0,bufferinfo[j].nisc.maxlen(n)-1 do blank=blank+' '
		data[*]=blank
		n=n+1
	endif
	status=linknload(!external_lib,'nightbuffer', $
		dims,data,Date+arrayid+configid,mode)
	if status then begin
		print,'***Error(LOADNIGHT): error loading data!'
		return
	endif
	if dims[dims[0]+1] eq 7 then data=strtrim(data,2)
	scans.(i)=data
endfor
;
endelse
;
!qiet=1 & get_scantable & !qiet=0
;
Date=GeoParms.Date
SystemId=GeoParms.SystemId
;
end
;-------------------------------------------------------------------------------
pro freememory
;
; Clear the data buffer memory.
;
common LocalAmoebaBuffer,BUFFERSIZE,BUFFERLIMIT
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
ddims=0L
ddata=' '
ddate=' '
mode=0L
;
status=linknload(!external_lib,'nightbuffer',ddims,ddata,ddate,mode)
BUFFERSIZE=0
;
; Remove disk buffer files
if n_elements(bufferinfo) ne 0 then begin
	index=where(bufferinfo.xdr eq 1,count)
	if count gt 0 then begin
		files=geninfo[index].date $
		     +geoinfo[index].systemid $
		     +geninfo[index].configid+'.xdr'
		for i=0,n_elements(files)-1 do spawn,'rm -f '+files[i],r
	endif
endif
;
end
;************************************************************************Block 4
pro setfluxes
;
; Obsolete!
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
star_model.fluxes=10^(star_model.magnitudes/(-2.5))
return
;
num_wave=n_elements(gen_model.wavelengths)
for j=0,num_wave-1 do begin
	fluxes=10^(star_model.magnitudes[j]/(-2.5))
	star_model.fluxes[j]=fluxes/total(fluxes)
endfor
;
end
;-------------------------------------------------------------------------------
pro fitwaveparms
;
; Obsolete!
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
num_wave=n_elements(gen_model.wavelengths)
;
; Stars:
degree=num_wave-1
ld_fitcoeffs=dblarr(degree+1,num_star())
flux_fitcoeffs=dblarr(degree+1,num_star())
for i=0,num_star()-1 do begin
	ld_fitcoeffs[*,i]= $
		poly_fit(gen_model.wavelengths,star_model[i].ld_coeffs,degree)
	flux_fitcoeffs[*,i]= $
		poly_fit(gen_model.wavelengths,star_model[i].fluxes,degree)
endfor
star_model.ld_fitcoeffs=ld_fitcoeffs
star_model.flux_fitcoeffs=flux_fitcoeffs
;
; Binaries:
; if num_binary() gt 0 then begin
; 	dm_fitcoeffs=dblarr(degree+1,num_binary())
; 	for i=0,num_binary()-1 do begin
; 		dm_fitcoeffs(*,i)= $
; 			poly_fit(gen_model.wavelengths, $
; 				binary_model(i).magdifference,degree)
; 	endfor
; 	binary_model.dm_fitcoeffs=dm_fitcoeffs
; endif
;
end
;-------------------------------------------------------------------------------
pro adjustfluxes,component,flux
;
; Obsolete!
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if componentparse(component,comp_1,comp_2) eq -1 then return
if strlen(comp_1) eq 1 then s1=where(star_model.component eq comp_1) else s1=-1
if strlen(comp_2) eq 1 then s2=where(star_model.component eq comp_2) else s2=-1
s1=s1[0] & s2=s2[0]
;
i=where(binary_model.component eq component)
magdifference=binary_model[i].magdifference
num_wave=n_elements(gen_model.wavelengths)
f1=dblarr(num_wave)
f2=dblarr(num_wave)
;
for j=0,num_wave-1 do begin
	f2[j]=flux[j]/(1+10^(magdifference[j]/2.5))
	f1[j]=flux[j]-f2[j]
endfor
if s1 ne -1 then star_model[s1].flux=f1 else adjustfluxes,comp_1,f1
if s2 ne -1 then star_model[s2].flux=f2 else adjustfluxes,comp_1,f2
;
end
;-------------------------------------------------------------------------------
pro adjustmasses,component,mass
;
; Obsolete!
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if componentparse(component,comp_1,comp_2) eq -1 then return
if strlen(comp_1) eq 1 then s1=where(star_model.component eq comp_1) else s1=-1
if strlen(comp_2) eq 1 then s2=where(star_model.component eq comp_2) else s2=-1
s1=s1[0] & s2=s2[0]
;
i=where(binary_model.component eq component)
massratio=binary_model[i].massratio
;
m1=mass/(1+massratio)
m2=mass-m1
if s1 ne -1 then star_model[s1].mass=m1 else adjustmasses,comp_1,m1
if s2 ne -1 then star_model[s2].mass=m2 else adjustmasses,comp_1,m2
;
end
;************************************************************************Block 5
pro readmodel,modelfile
;
; This procedure sets up a hierarchical model format and reads
; a file with the model information. It also initializes the two
; structures identical to the model structures, which hold the
; parameter error information.
;
common StarBase,StarTable,Notes
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
; Read command file into string array
;
commands=''
status=dc_read_fixed(modelfile,commands,resize=[1],format='(a90)',/col)
if status ne 0 then begin
	print,'***Error(READMODEL): error reading file or file not found!'
	return
endif
; commands=strupcase(commands)
;
; Allocate arrays
;
num_binaries=10
num_stars=2*num_binaries
index=where(strpos(strupcase(commands),'WAVELENGTHS') ne -1,count)
if count ge 1 then for i=0,count-1 do r=execute(commands[index[i]]) else begin
	print,'***Error(READMODEL): Check definition of WAVELENGTHS!'
	return
endelse
;
; General parameters
num_wave=n_elements(wavelengths)
starid=''
rv=0.d0					; radial velocity [km/s]
;
; Star parameters
name    	=strarr(num_stars)	; A, B, C, D,...
wmc		=strarr(num_stars)	; WMC designation, e.g. 'Aa'
mode		=intarr(num_stars)+1	; Limb darkening and model mode
model		=strarr(num_stars)	; Model atmosphere file
mass		=dblarr(num_stars)+1	; solar masses
diameter	=dblarr(num_stars)	; Diameter in mas
hole     	=dblarr(num_stars)	; Inner hole radius in mas (accretion disks)
ratio		=dblarr(num_stars)+1	; Minor axis/major axis
pa		=dblarr(num_stars)	; Major axis position angle, 0=N
omegabr    	=dblarr(num_stars)	; omega (rotation) / omega breakup  
tilt       	=dblarr(num_stars)	; inclination of rot. axis, 90=eq.on
rotation	=dblarr(num_stars)	; ratio of axial to orbital rate
gr		=dblarr(num_stars)	; exponent in gravity darkening law
albedo		=dblarr(num_stars)	; bolometric albedo
teff		=dblarr(num_stars)	; effective temperature
alpha		=dblarr(num_stars)	; accretion disk: T ~ r^alpha
logg		=dblarr(num_stars)	; log(g)
spot		=dblarr(4,num_stars)
magnitudes	=dblarr(num_wave,num_stars)
; lines   	=dblarr(num_wave,num_stars)
; widths		=dblarr(num_wave,num_stars)
; fluxes		=dblarr(num_wave,num_stars)
; coeffs		=dblarr(num_wave,num_stars)
;
; Binary parameters
component	=strarr(num_binaries)	; A-B, AB-C, AB-CD,...
method		=intarr(num_binaries)
wdmode		=intarr(num_binaries)
semimajoraxis	=dblarr(num_binaries)	; mas
inclination	=dblarr(num_binaries)	; degrees
ascendingnode	=dblarr(num_binaries)	; degrees
eccentricity	=dblarr(num_binaries)
periastron	=dblarr(num_binaries)	; degrees
apsidalmotion	=dblarr(num_binaries)	; degrees/year
epoch		=dblarr(num_binaries)
period		=dblarr(num_binaries)	; days
rho             =dblarr(num_binaries)	; mas
theta           =dblarr(num_binaries)	; degrees
;
; Execute commands
;
for i=0,n_elements(commands)-1 do r=execute(commands[i])
;
; Allocate structures, arrays, store information, and remove zero entries
;
; General model parameters
gen_model=alloc_gen_model(num_wave)
wavelengths=double(wavelengths)
si=sort(wavelengths)
gen_model.wavelengths=wavelengths[si]
gen_error=gen_model
if total(strlen(starid)) gt 0 then begin
	gen_model.starid=starid
	gen_model.rv=rv
	if n_elements(StarTable) eq 0 then get_startable,starid
	index=where(StarTable.starid eq starid,count)
	if count eq 0 then begin
		addstar,starid
		index=where(StarTable.starid eq starid)
	endif
	gen_model.ra=StarTable[index[0]].ra
	gen_model.dec=StarTable[index[0]].dec
	set_parallax,StarTable[index[0]].px*1000,StarTable[index[0]].pxe,0
endif
;
; Star model parameters
star_struct=alloc_star_struct(num_wave)
star_model=replicate(star_struct,num_stars)
star_error=star_model
if total(strlen(starid)) gt 0 then begin
	star_model.component		=name
	star_model.wmc			=wmc
	star_model.mode			=mode
	star_model.model		=model
	star_model.mass			=mass
	star_model.diameter		=diameter
	star_model.hole     		=hole
	star_model.ratio		=ratio
	star_model.pa			=pa
	star_model.omegabr		=omegabr
	star_model.tilt   		=tilt
	star_model.rotation		=rotation
	star_model.gr			=gr
	star_model.albedo		=albedo
	star_model.teff			=teff
	star_model.alpha		=alpha
	star_model.logg			=logg
	star_model.spot			=spot
;	star_model.magnitudes		=reform(magnitudes(si,*),num_wave,num_stars)
	star_model.magnitudes		=reform(magnitudes[si,*])
;
	star_error.component		=name
;
	star_model=star_model[where(strlen(star_model.component) ne 0)]
	star_error=star_error[where(strlen(star_model.component) ne 0)]
endif
;
; Binary model parameters
binary_struct=alloc_binary_struct()
binary_model=replicate(binary_struct,num_binaries)
binary_error=binary_model
index=where(epoch gt 0,count)
if count gt 0 then epoch[index]=epoch[index]-2440000.d0
if total(strlen(component)) gt 0 then begin
	binary_model.component		=component
	binary_model.mode		=method
	binary_model.wdmode		=wdmode
	binary_model.semimajoraxis	=semimajoraxis
	binary_model.inclination	=inclination
	binary_model.ascendingnode	=ascendingnode
	binary_model.eccentricity	=eccentricity
	binary_model.periastron		=periastron
	binary_model.apsidalmotion	=apsidalmotion
	binary_model.epoch		=epoch
	binary_model.period		=period
	binary_model.rho		=rho
	binary_model.theta		=theta
;
	binary_error.component		=component
;
	c=fltarr(num_binaries)
	for i=0,num_binaries-1 do $
	 for j=1,n_tags(binary_model)-1 do c[i]=c[i]+total(binary_model[i].(j))
	binary_model=binary_model[where(c ne 0)]
	binary_error=binary_error[where(c ne 0)]
endif
;
if checkmodel() eq -1 then return
;
; if total(strlen(binary_model.component)) ne 0 then begin
; 	adjustfluxes,topbincomp(),dblarr(num_wave)+1
; 	adjustmasses,topbincomp(),1.d0
; endif
; setfluxes
; fitwaveparms
;
print,'Model file read and checked successfully.'
;
end
;-------------------------------------------------------------------------------
pro calcmodel
;
; Main model data computation function for interferometric, spectroscopic,
; astrometric, and photometric data. The common block variable ds_options
; specifies by weight which data sets are requested.
;
common MarquardtFit,fit_options
common ModelFit,parameters,ds_options
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Tables,scantable,bgtable,stationtable
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AuxData,parallaxes,k2,vsini
common RandomModel,RandomSeed
common LocalRocheVis,LDinit
;
RAD=180/pi_circle
RandomSeed=0
LDinit=1
;
if n_elements(ds_options) eq 0 then begin
	print,'***Error(CALCMODEL): no data weights specified!'
	return
endif
if n_elements(gen_model) eq 0 then begin
	print,'***Error(CALCMODEL): no model!'
	return
endif
if n_elements(fit_options) eq 0 then fit_options=alloc_fit_options()
;
; Section for visibility data...................................................
;
if ds_options.i ne 0 then begin
;
if n_elements(GenInfo) eq 0 then begin
	print,'***Error(CALCMODEL): no data (GENINFO undefined)!'
	return
endif
;
if n_elements(GeoInfo) eq 0 then begin
	print,'***Error(CALCMODEL): no data (GEOINFO undefined)!'
	return
endif
;
; The model is always computed for all data
nights=GenInfo.date
arrays=GeoInfo.systemid
confgs=GenInfo.configid
;
; Do one night at a time
;
FOR n=0,n_elements(nights)-1 DO BEGIN
;
; Load a night
night=where(GenInfo.date eq nights[n] $
        and GeoInfo.systemid eq arrays[n] $
        and GenInfo.configid eq confgs[n],count)
if count eq 0 and n_elements(GenInfo.date) eq 1 then night=0
if strupcase(!version.os) ne 'WIN32' then $
loadnight,GenInfo[night[0]].date,GeoInfo[night[0]].systemid,GenInfo[night[0]].configid
;
; Select data for model star
index0=where(scans.starid eq gen_model.starid and scantable.code eq 1,NS0)
if NS0 eq 0 then begin
 print,'***Error(CALCMODEL): no data selected for ',gen_model.starid,' on ', $
	nights[n]+' '+confgs[n],'!'
 return
endif
;
time=abs(scans[index0].time)
;
; Baseline visibilities
;
for ob=0,GenConfig.NumOutBeam-1 do begin
;
; Set up grid of wavelengths for band pass integrations
; Note that currently all functions in filter.pro use [nm] as units for lambda
lambda_grid=system_config(SystemId,'GRID')
if string(lambda_grid[0]) eq 'SPECTROMETER' then begin
	lambda_grid=genconfig.wavelength[0:genconfig.numspecchan[ob]-1,ob]*1e9
endif else begin
	index=where(lambda_grid*1e-9 le max(genconfig.wavelength[*,ob]+genconfig.chanwidth[*,ob]/2) $
		and lambda_grid*1e-9 ge min(genconfig.wavelength[*,ob]-genconfig.chanwidth[*,ob]/2),count)
	if count gt 0 then lambda_grid=lambda_grid[index]
endelse
lambda=lambda_grid/1d9	; convert to SI units
num_lambda=n_elements(lambda)
;
nc=GenConfig.NumSpecChan[ob]
vis=reform(complexarr(nc,NS0,/nozero),nc,NS0)
pwr=reform(complexarr(nc,NS0,/nozero),nc,NS0)
;
for bl=0,GenConfig.NumBaseline[ob]-1 do begin
;	This is to speed up the computation in special circumstances
	e=reform(scans[*].vissqcerr(ob,0:GenConfig.NumSpecChan[ob]-1,bl) > 0)
	if GenConfig.NumSpecChan[ob] gt 1 then e=total(e,1)
	index=where(scans.starid eq gen_model.starid $
		and scantable.code eq 1 $
		and e gt 0,NS)
	if NS gt 0 then begin
	time=abs(scans[index].time)
	vis=reform(complexarr(nc,NS,/nozero),nc,NS)
	pwr=reform(complexarr(nc,NS,/nozero),nc,NS)
;
	um=[scans[index].uvw(ob,0,bl,0)*GenConfig.Wavelength[0,ob]]
	vm=[scans[index].uvw(ob,0,bl,1)*GenConfig.Wavelength[0,ob]]
	nu=1/lambda
	u=nu#um
	v=nu#vm
	vis_grid=reform(modelvis(time,lambda,fluxes,u,v),num_lambda,NS)
;
;	Bandpass integration
	if num_lambda eq genconfig.numspecchan[ob] then begin
;		High R spectrometer
		for sc=0,NS-1 do vis[*,sc]=vis_grid[*,sc]/fluxes
	endif else begin
		for ch=0,nc-1 do begin
			filters=system_config(SystemId,'FILTERS')
			if n_elements(filters) gt 1 then filter=filters[ch,ob] $
						    else filter=filters[0]
			if filter[0] eq 'generic_c' then $
				tm=generic_c(lambda_grid,ch,ob) $
						 else $
				tm=call_function(filter,lambda_grid)
			if NS eq 1 then begin
				tm=reform(tm,1,num_lambda)
				lam=reform(lambda,1,num_lambda)
			endif else lam=lambda
			vis[ch,*]=(tm#vis_grid)/total(tm*fluxes)
			pwr_grid=reform(abs(vis_grid)^2,num_lambda,NS)
			pwr[ch,*]=(tm^2#pwr_grid)/total(tm^2*fluxes^2)
			pwr[ch,*]=((lam^2*tm^2)#pwr_grid)/total(tm^2*fluxes^2*lam^2)
		endfor
	endelse
	dimensions= $
	size(scans[index].complexvis(ob,0:GenConfig.NumSpecChan[ob]-1,bl))
	scans[index].complexvis(ob,0:GenConfig.NumSpecChan[ob]-1,bl)= $
		reform(vis,dimensions[1:dimensions[0]])
	scans[index].complexweight(ob,0:GenConfig.NumSpecChan[ob]-1,bl)=100.0
;
	dimensions= $
	size(scans[index].vissqm(ob,0:GenConfig.NumSpecChan[ob]-1,bl))
;	Use amplitude for NPOI band pass integration (ABCD method)
	scans[index].vissqm(ob,0:GenConfig.NumSpecChan[ob]-1,bl)= $
		reform(abs(vis)^2,dimensions[1:dimensions[0]])
	scans[index].visphasem(ob,0:GenConfig.NumSpecChan[ob]-1,bl)= $
		reform(cphase(vis),dimensions[1:dimensions[0]])
;	Use power for VINCI band pass integration (scanning interferometers)
	if strpos(instrument_id(systemid),'VINCI') ge 0 then $
	scans[index].vissqm(ob,0:GenConfig.NumSpecChan[ob]-1,bl)= $
		reform(pwr,dimensions[1:dimensions[0]])
;	Self-cal (NPOI only) ?
	if fit_options.v and system_id(systemid) eq 'NPOI' then begin
	for sc=0,n_elements(index)-1 do begin
		v=scans[index[sc]].vissqc(ob,0:GenConfig.NumSpecChan[ob]-1,bl)
		e=scans[index[sc]].vissqcerr(ob,0:GenConfig.NumSpecChan[ob]-1,bl)
		m=scans[index[sc]].vissqm(ob,0:GenConfig.NumSpecChan[ob]-1,bl)
		eindex=where(e gt 0,ecount)
		if ecount gt 0 then begin
			nf=100
			f=findgen(nf+1)/100+(1.0-nf/200.) & r=f
			for i=0,nf do r[i]=total(((v[eindex]*f[i]-m[eindex])/e[eindex])^2)
			i=where(r eq min(r)) & i=i[0]
			scans[index[sc]].vissqc(ob,0:GenConfig.NumSpecChan[ob]-1,bl)= $
			scans[index[sc]].vissqc(ob,0:GenConfig.NumSpecChan[ob]-1,bl)*f[i] 
		endif
	endfor
	endif
	endif	; NS > 0
endfor
endfor
;
; Triple visibilities
;
NS=NS0
index=index0
time=abs(scans[index].time)
for tr=0,GenConfig.NumTriple-1 do begin
;
;	This is to speed up the computation in special circumstances
	e=reform(scans[*].tripleampcerr(tr,0:GenConfig.TripleNumChan[tr]-1) > 0)
	if GenConfig.TripleNumChan[tr] gt 1 then e=total(e,1)
	index=where(scans.starid eq gen_model.starid $
		and scantable.code eq 1 $
		and e gt 0,NS)
	if NS gt 0 then begin
;
	nc=GenConfig.TripleNumChan[tr]
 	vis=reform(complexarr(3,nc,NS,/nozero),3,nc,NS)
	fBaseMatrix=intarr(GenConfig.NumSid,3)
	fBaseFactor=fltarr(3)+1
	for l=0,2 do begin
		ob=GenConfig.TripleBeam[l,tr]
		bl=GenConfig.TripleBase[l,tr]
		j1=where(GenConfig.StationId eq strmid(GenConfig.BaselineId[bl,ob],0,3))
		j2=where(GenConfig.StationId eq strmid(GenConfig.BaselineId[bl,ob],4,3))
		fBaseMatrix[j1,l]=+1
		fBaseMatrix[j2,l]=-1
		if l gt 0 then fBaseFactor[l]=-total(fBaseMatrix[*,0]*fBaseMatrix[*,l])
	endfor
	for l=0,2 do begin
		ob=GenConfig.TripleBeam[l,tr]
		bl=GenConfig.TripleBase[l,tr]
		ch=GenConfig.TripleChan[0:GenConfig.TripleNumChan[tr]-1,l,tr]
		lambda_grid=system_config(SystemId,'GRID')
		if string(lambda_grid[0]) eq 'SPECTROMETER' then begin
			lambda_grid=genconfig.wavelength[ch,ob]*1e9
		endif else begin
			jndex=where(lambda_grid*1e-9 le max(genconfig.wavelength[ch,ob] $
							   +genconfig.chanwidth[ch,ob]/2) $
				and lambda_grid*1e-9 ge min(genconfig.wavelength[ch,ob] $
							   -genconfig.chanwidth[ch,ob]/2),count)
			if count gt 0 then lambda_grid=lambda_grid[jndex]
		endelse
		lambda=lambda_grid/1d9
		num_lambda=n_elements(lambda)
		um=[scans[index].uvw(ob,0,bl,0)*GenConfig.Wavelength[0,ob]]
		vm=[scans[index].uvw(ob,0,bl,1)*GenConfig.Wavelength[0,ob]]
		nu=1/lambda
		u=nu#um
		v=nu#vm
		vis_grid=reform(modelvis(time,lambda,fluxes,u,v),num_lambda,NS)
		if num_lambda eq n_elements(ch) then begin
			for sc=0,NS-1 do vis[l,*,sc]=vis_grid[*,sc]/fluxes
		endif else begin
		for j=0,nc-1 do begin
			filters=system_config(SystemId,'FILTERS')
			if n_elements(filters) gt 1 then filter=filters[ch[j],ob] $
						    else filter=filters[0]
			if filter eq 'generic_c' then $
				tm=generic_c(lambda_grid,ch[j],ob) $
						 else $
				tm=call_function(filter,lambda_grid)
			if NS eq 1 then tm=reform(tm,1,num_lambda)
			vis[l,j,*]=(tm#vis_grid)/total(tm*fluxes)
		endfor
		endelse
		if fBaseFactor[l] lt 0 then $
			vis[l,*,*]=conj(vis[l,*,*])
	endfor
;
;	Form complex triple products
	result=reform(float(vis[0,*,*]*vis[1,*,*]*vis[2,*,*]),1,1,NC,NS)
	if NC eq 1 then result=reform(result,1,NS)
	if NC eq 1 and NS eq 1 then result=reform(result)
	scans[index].compltriplem(tr,0,0:NC-1)=result
	result=reform(imaginary(vis[0,*,*]*vis[1,*,*]*vis[2,*,*]),1,1,NC,NS)
	if NC eq 1 then result=reform(result,1,NS)
	if NC eq 1 and NS eq 1 then result=reform(result)
	scans[index].compltriplem(tr,1,0:NC-1)=result
;
;	Compute amp and phase
	result=reform(sqrt(scans[index].compltriplem(tr,0,0:NC-1)^2 $
		     +scans[index].compltriplem(tr,1,0:NC-1)^2),1,NC,NS)
	if NC eq 1 then result=reform(result,1,NS)
	if NC eq 1 and NS eq 1 then result=reform(result)
	scans[index].tripleampm(tr,0:NC-1)=result
	result=reform(atan(scans[index].compltriplem(tr,1,0:NC-1), $
		      scans[index].compltriplem(tr,0,0:NC-1)),1,NC,NS)
	if NC eq 1 then result=reform(result,1,NS)
	if NC eq 1 and NS eq 1 then result=reform(result)
	scans[index].triplephasem(tr,0:NC-1)= result
;
;	Self-cal (NPOI only) ?
	if fit_options.v and system_id(systemid) eq 'NPOI' then begin
	for sc=0,n_elements(index)-1 do begin
		v=scans[index[sc]].tripleampc(tr,*)
		e=scans[index[sc]].tripleampcerr(tr,*)
		m=scans[index[sc]].tripleampm(tr,*)
		eindex=where(e gt 0,ecount)
		if ecount gt 0 then begin
			nf=80
			f=findgen(nf+1)/100+(1.0-nf/200.) & r=f
			for i=0,nf do r[i]=total(((v[eindex]*f[i]-m[eindex])/e[eindex])^2)
			i=where(r eq min(r)) & i=i[0]
			scans[index[sc]].compltriple(tr,*,*)= $
			scans[index[sc]].compltriple(tr,*,*)*f[i] 
			scans[index[sc]].tripleampc(tr,*)= $
			scans[index[sc]].tripleampc(tr,*)*f[i]
		endif
	endfor
	endif
;
;	Unwrap the data phase by comparing it to the model
	for iscan=0,n_elements(index)-1 do begin
		repeat begin
		wrapped=where((scans[index[iscan]].triplephasec(tr,*)- $
			       scans[index[iscan]].triplephasem(tr,*) $
					gt pi_circle) $
			  and (scans[index[iscan]].triplephasecerr(tr,*) gt 0),count)
		tcount=count
		if count gt 0 then scans[index[iscan]].triplephasec(tr,wrapped)=$
			scans[index[iscan]].triplephasec(tr,wrapped)-2*pi_circle
		wrapped=where((scans[index[iscan]].triplephasec(tr,*)- $
			       scans[index[iscan]].triplephasem(tr,*) $
					lt -pi_circle) $
			  and (scans[index[iscan]].triplephasecerr(tr,*) gt 0),count)
		tcount=tcount+count
		if count gt 0 then scans[index[iscan]].triplephasec(tr,wrapped)=$
			scans[index[iscan]].triplephasec(tr,wrapped)+2*pi_circle
		endrep until tcount eq 0
	endfor
;
	endif ; NS > 1
endfor
;
; Update night stored with new model data
if strupcase(!version.os) ne 'WIN32' then storenight,11
;
ENDFOR
;
endif
;
; Section for positions.........................................................
;
if ds_options.a ne 0 then begin
;
if n_elements(positions) eq 0 then begin
	print,'***Error(CALCMODEL): no position data!'
	return
endif
;
; Assume visual (speckle) band if not specified
lambda=2.2e-6
lambda=fltarr(1)+gen_model.wavelengths[n_elements(gen_model.wavelengths)-1]*1e-6
lambda=fltarr(1)+550e-9
;
p=modelpos(positions.jd)
list_poscomps,components
for i=0,n_elements(components)-1 do begin
	index=where(positions.component eq components[i],count)
	pos1=dblarr(count,2)
	if componentparse(components[i],comp_1,comp_2) ne 0 then return
	tflux=0.d0
	for k=0,strlen(comp_1)-1 do begin
		j=where(strmid(comp_1,k,1) eq star_model.component) & j=j[0]
		flux=stellarflux(star_model[j],lambda) & flux=total(flux)
		pos1=pos1+reform(p[index,j,*]*flux,count,2)
		tflux=tflux+flux
	endfor
	pos1=pos1/tflux
	pos2=dblarr(count,2)
	tflux=0.d0
	for k=0,strlen(comp_2)-1 do begin
		j=where(strmid(comp_2,k,1) eq star_model.component) & j=j[0]
		flux=stellarflux(star_model[j],lambda) & flux=total(flux)
		pos2=pos2+reform(p[index,j,*]*flux,count,2)
		tflux=tflux+flux
	endfor
	pos2=pos2/tflux
	dx=pos2[*,0]-pos1[*,0]
	dy=pos2[*,1]-pos1[*,1]
	rhom=sqrt(dx^2+dy^2)
	thetam=atan(dx,dy)
	if count eq 1 then begin
		positions[index].rhom=rhom[0]
		positions[index].thetam=thetam[0]
	endif else begin
		positions[index].rhom=rhom
		positions[index].thetam=thetam
	endelse
endfor
;
if 0 then begin
for i=0,num_binary()-1 do begin
	index=where(positions.component eq binary_model[i].component,count)
	if count gt 0 then begin
		o_parms=dblarr(7)
		o_parms[0]=binary_model[i].semimajoraxis
		o_parms[1]=binary_model[i].eccentricity
		o_parms[2]=binary_model[i].inclination/RAD
		o_parms[3]=binary_model[i].periastron/RAD
		o_parms[4]=binary_model[i].ascendingnode/RAD
		o_parms[5]=binary_model[i].period
		o_parms[6]=binary_model[i].epoch
		jd=positions[index].jd
		xy=true2app(jd,o_parms,rho,theta)
		positions[index].rhom=rho
		positions[index].thetam=theta
	endif
endfor
endif
;
endif
;
; Section for velocities........................................................
if ds_options.s ne 0 then begin
;
if n_elements(velocities) eq 0 then begin
        print,'***Error(CALCMODEL): no velocity data!'
        return
endif
;
v=modelvel(velocities.jd)
list_velcomps,components
for i=0,n_elements(components)-1 do begin
	j=where(components[i] eq star_model.component) & j=j[0]
	index=where(velocities.component eq components[i])
	velocities[index].valuem=v[index,j]
endfor
;
endif
;
; Section for magnitudes........................................................
;
if ds_options.p ne 0 then begin
;
if n_elements(magnitudes) eq 0 then begin
        print,'***Error(CALCMODEL): no magnitude data!'
        return
endif
;
filters=unique(magnitudes.filter)
;
vega_filter=['U','B','V','R','I']
vega_colors=[0.03,0.03,0.03,0.07,0.10]	; from Johnson et al.
vega_colors=[0.026,0.026,0.026,0.066,0.096]	; Deane version 2
; Get Vega zero points with model T=9500 K, d=3.24 mas, px=129 mas, (mass=2)
vega_zp=[-22.7879,-22.4989,-21.9033,-21.1903,-20.5634]-vega_colors
; Get Vega zero points with model T=9442 K, d=3.24 mas, px=128.9 mas, (mass=2.375)
vega_zp=[-22.7633,-22.4774,-21.8882,-21.1780,-20.5523]-vega_colors
;
for i=0,n_elements(filters)-1 do begin
;
case filters[i] of
	'U': 	begin
		lambda=findgen(250)*2+401
		tm=johnson_u(lambda)
		zp=vega_zp[0]
		end
	'B': 	begin
		lambda=findgen(250)*2+401
		tm=johnson_b(lambda)
		zp=vega_zp[1]
		end
	'V': 	begin
		lambda=findgen(250)*2+401
		tm=johnson_v(lambda)
		zp=vega_zp[2]
		end
	'R': 	begin
		lambda=findgen(250)*2+401
		tm=johnson_r(lambda)
		zp=vega_zp[3]
		end
	'I': 	begin
		lambda=findgen(250)*2+401
		tm=johnson_i(lambda)
		zp=vega_zp[4]
		end
endcase
lambda=lambda*1d-9
index=where(magnitudes.filter eq filters[i])
jd=magnitudes[index].jd
flux=modelflux(jd,lambda)
;
magnitudes[index].valuem=-2.5*alog10(total(flux*tm)/total(tm))-zp
;
endfor
;
endif
;
; Section for auxilliary data...................................................
;
; Parallax
if ds_options.px ne 0 then begin
if n_elements(parallaxes) eq 0 then begin
        print,'***Error(CALCMODEL): no parallax data!'
        return
endif
j=0
for i=0,n_elements(binary_model.component)-1 do begin
	if binary_model[i].mode eq 1 then begin
		parallaxes.valuem[j]=modelpx(binary_model[i].component)
		j=j+1
	endif
endfor
endif
;
; K2
if ds_options.k2 ne 0 then begin
if n_elements(k2) eq 0 then begin
        print,'***Error(CALCMODEL): no k2 data!'
        return
endif
k2.valuem=modelk('B')
endif
;
; V sin(i)
;
end
;-------------------------------------------------------------------------------
pro readimage,fitsfile,cellsize=cellsize
;
; Read a FITS file with an image into OYSTER's effective temperature Pearl image.
; Effective temperature and log(g) maps should be found in the image extensions,
; if not the procedure will ask them to be defined using XROI.
;
; Cellsize in arcseconds.
;
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,db,cb,dm,rm,cm,cc,rt,rg,rf
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common StarBase,StarTable,Notes
;
threshold=0.01	; Set pixels to zero below this threshold
;
cmmap=mrdfits[fitsfile,0,header]
cmmap[where(cmmap lt max(cmmap)*threshold)]=0
cmmap=cmmap/total(cmmap)
;
nx=n_elements(cmmap[*,0])
ny=n_elements(cmmap[0,*])
imsze=[nx,ny]
;
index=where(strpos(header,'RA---SIN') ge 0,exists)
if exists then begin
	words=nameparse(header[index[0]+2])
	cdelt1=abs(float(words[2]))*3600000
endif else cdelt1=1.0
index=where(strpos(header,'DEC--SIN') ge 0,exists)
if exists then begin
	words=nameparse(header[index[0]+2])
	cdelt2=abs(float(words[2]))*3600000
endif else cdelt2=1.0
if cdelt1 ne cdelt2 then begin
	print,'***Error(GET_IMAGE): cell size in RA and Dec different!'
	return
endif
if n_elements(cellsize) eq 0 then cellsze=cdelt1 else cellsze=cellsize
print,'Cell size = ',cellsze
;
cm=pearlimage(imsze,cellsze,/relax)
cm.map=cmmap
;
; Initialize default model for this type of image
index=where(strpos(header,'WAVE') ge 0,exists)
if not exists then begin
;	print,'***Error(READIMAGE): wavelength axis undefined in header!'
;	read,wavelength,prompt='Please enter min and max wavelengths: '
	wavelength=1.0	; For this type of image wavelength is not used
endif else begin
	words=nameparse(header[index[0]+1])
	crval3=float(words[2])
	words=nameparse(header[index[0]+2])
	cdelt3=float(words[2])
	words=nameparse(header[index[0]+3])
	crpix3=float(words[2])
	wavelength=crval3
endelse
if n_elements(gen_model) ne 0 then starid=gen_model.starid
num_wave=1
gen_model=alloc_gen_model(num_wave)
gen_error=gen_model
if n_elements(startable) eq 1 then gen_model.starid=startable.starid $
			      else gen_model.starid='CCCNNNN'
if n_elements(starid) ne 0 then gen_model.starid=starid
gen_model.wavelengths=wavelength
star_model=alloc_star_struct(num_wave)
star_error=star_model
star_model[0].component='A'
star_model[0].mode=8
star_model[0].diameter=1
binary_struct=alloc_binary_struct()
binary_model=replicate(binary_struct,1)
;
; Check if map is AIPS/MX ICL map
index=where(strpos(header,"IMCLASS='ICL") ge 0,exists)
if exists then begin
	rad=180/!pi
	mas2rad=rad*3600000.0
	cclist=mrdfits[fitsfile,1,header]
	cm.map=0
	for i=0,n_elements(cclist)-1 do begin
		j=where(abs(cm.x-cclist[i].deltax/rad) lt cellsze/mas2rad $
		    and abs(cm.y-cclist[i].deltay/rad) lt cellsze/mas2rad,count)
		if count eq 0 then begin
			print,'***Error: could not assign CC!'
			return
		endif
		cm.map[j]=cm.map[j]+cclist[i].flux
	endfor
endif else begin
; 	Read FITS extensions for eff. temp. and log(g) region maps
	etmap=mrdfits[fitsfile,1,header]
	lgmap=mrdfits[fitsfile,2,header]
endelse
;
if n_elements(etmap) le 1 then begin
	print,'------------------------------------------------------------------'
	print,'This type of image requires the definition of Teff/log(g) regions.'
	print,'Please use XROI to define regions of interest, close (X) when done.'
	print,'The largest regions will be processed first, smaller ones override.'
	print,'------------------------------------------------------------------'
	scale_factor=4
	xroi,bytscl(rebin(cm.map,imsze*scale_factor),top=200),roi_geometry=roigeo,regions_out=regout,/block
	if n_elements(roigeo) eq 0 then return
;
	print,'Now enter Teff/log(g) for each region in order of decreasing size.'
	si=reverse(sort(roigeo.area))
	roigeo=roigeo[si]
	regout=regout[si]
	etmap=fltarr(nx,ny)
	lgmap=fltarr(nx,ny)
	nr=n_elements(regout)
	for ireg=0,nr-1 do begin
		read,teff,logg,prompt='Please enter Teff,log(g) for region '+string(ireg,format='(i1)')+': '
		regout[ireg]->scale,fltarr(2)+1./float(scale_factor)
		mask=regout[ireg]->computemask(dimension=[nx,ny])
		etmap[where(mask eq 255)]=teff
		lgmap[where(mask eq 255)]=logg
;		Make sure there are no pixels without initialization
		if ireg eq 0 then begin
			etmap[*]=teff
			lgmap[*]=logg
		endif
	endfor
	ext_header=["XTENSION= 'IMAGE' / IMAGE extension", $
		    "BITPIX  = -32 /  Number of bits per data pixel", $
		    "NAXIS   = 2 / Number of data axes", $
		    "NAXIS1  = "+string(nx)+" /", $
		    "NAXIS2  = "+string(ny)+" /", $
		    "PCOUNT  = 0 / No Group Parameters", $
		    "GCOUNT  = 1 / One Data Group", $
		    "EXTNAME = 'Teff' /", $
		    "END     "]
		
	writefits,fitsfile,etmap,ext_header,/append
	ext_header[where(strpos(ext_header,'EXTNAME') ge 0)]="EXTNAME = 'log(g)' /"
	writefits,fitsfile,lgmap,ext_header,/append
;
;	Obsolete
	if 0 then begin
	window,xsize=nx,ysize=ny,/free
	tvscl,cm.map
	print,'Please define regions with same Teff!'
	etmap=fltarr(nx,ny)
	lgmap=fltarr(nx,ny)
	nr=20
	rt=fltarr(nr) & rt[0]=5000
	rg=fltarr(nr)
	nr=0
	repeat begin
		read,teff,prompt='Please enter Teff for region '+string(nr+1,format='(i2)')+': '
		if teff ne 0 then begin
			etmap=set_region(etmap,teff)
			rt[nr]=teff
			nr=nr+1
		endif
	endrep until teff eq 0
	nc=16
	rf=fltarr(nc,nr)+1
	rg[0]=5
	cm.teff=rt[0]
	endif
endif
;
; Here we have a valid eff. temp. map, now we use it to initialize the rest
etlg=string(etmap,format='(i5)')+' '+string(lgmap,format='(f3.1)')
uetlg=unique(etlg)
nuetlg=n_elements(uetlg)
rt=fltarr(nuetlg)
rg=rt
for i=0,nuetlg-1 do begin
	words=nameparse(uetlg[i])
	rt[i]=float(words[0])
	rg[i]=float(words[1])
endfor
cm.teff=etmap
cm.logg=lgmap
;
; Read convolving beam, if any
; a=0.
; read,a,b,c,prompt='Please enter major, minor axis [mas] and PA of restoring beam: '
; if a ne 0 then cb=cleankernel(cm,a) else cb=0
;
end
;-------------------------------------------------------------------------------
pro plot_image
;
common PearlImage,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,db,cb,dm,rm,cm,cc,rt,rg,rf
;
tvscl,cm.map
;
end
;-------------------------------------------------------------------------------
pro readimages,fitsfile,cellsize=cellsize,wavelengths=wavelengths
;
; Read a FITS file containing an image cube to be used for modeling.
; Cellsize [mas], wavelengths [m]. Wavelengths should be specified
; if not in the header of the FITS file (WAVE keyword). 
; Instead of a cube, the images can be specified using wildcards,
; and in that case they will be assembled into a cube. Here one also has the
; option to load a model file first, and the wavelengths specified there
; take the place of the wavelengths required by this routine.
;
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
; Check if multiple files were given, one image per wavelength
files=findfile(fitsfile)
if strlen(files[0]) eq 0 then begin
	print,'***Error: file does not exist!'
	return
endif
if n_elements(files) gt 1 then begin
	channel_image=mrdfits[files[0],0,header]
	nx=n_elements(channel_image[*,0])
	ny=n_elements(channel_image[0,*])
	nz=n_elements(files)
	index=where(strpos(header,'WAVE') ge 0,exists)
	if not exists then begin
		if n_elements(wavelengths) eq 0 then begin
			print,'Wavelength axis undefined in header!'
			if n_elements(gen_model) eq 0 then begin
				print,'***Error: no wavelengths defined!'
				return
			endif
			channel_wavelengths=gen_model.wavelengths*1e-6
			print,'Wavelengths taken from model.'
		endif else begin
			channel_wavelengths=wavelengths
		endelse
		if nz ne n_elements(channel_wavelengths) then begin
			print,'***Error: different number of files and wavelengths!'
			return
		endif
	endif
	channel_images=fltarr(nx,ny,nz)
	for i=0,nz-1 do begin
		channel_images[*,*,i]=mrdfits[files[i],0,header]
		index=where(strpos(header,'WAVE') ge 0,exists)
		if exists then begin
			words=nameparse(header[index[0]+1])
			crval3=float(words[2])
			channel_wavelengths[i]=crval3
		endif
	endfor
endif else begin
	channel_images=mrdfits[fitsfile,0,header]
endelse
;
nx=n_elements(channel_images[*,0,0])
ny=n_elements(channel_images[0,*,0])
nz=n_elements(channel_images[0,0,*])
imsze=[nx,ny]
;
cdelt1=0
cdelt2=0
index=where(strpos(header,'RA---TAN') ge 0,exists)
if not exists then begin
	print,'Warning(READIMAGES): RA axis undefined in header!'
endif else begin
	words=nameparse(header[index[0]+2])
	cdelt1=abs(float(words[2]))*(180/!pi)*3600000	; convert radians in mas
endelse
;
index=where(strpos(header,'DEC--TAN') ge 0,exists)
if not exists then begin
	print,'Warning(READIMAGES): DEC axis undefined in header!'
endif else begin
	words=nameparse(header[index[0]+2])
	cdelt2=abs(float(words[2]))*(180/!pi)*3600000	; convert radians in mas
endelse
if cdelt1 ne cdelt2 then begin
	print,'***Error(READIMAGES): cell size in RA and Dec different!'
	return
endif
;
if n_elements(cellsize) eq 0 then images_cellsize=cdelt1 $
			     else images_cellsize=cellsize
;
index=where(strpos(header,'WAVE') ge 0,exists)
if not exists and n_elements(files) eq 1 then begin
	print,'***Error(READIMAGES): wavelength axis undefined in header!'
	return
endif
if exists and n_elements(files) eq 1 then begin
	words=nameparse(header[index[0]+1])
	crval3=float(words[2])
	words=nameparse(header[index[0]+2])
	cdelt3=float(words[2])
	words=nameparse(header[index[0]+3])
	crpix3=float(words[2])
	channel_wavelengths=findgen(nz)*cdelt3+crval3-(crpix3-1)*cdelt3
endif
;
; Initialize default model for this type of image
flag=0
if n_elements(gen_model) eq 0 then begin
	flag=1
endif else begin
	if n_elements(gen_model.wavelengths) $
	ne n_elements(channel_wavelengths) then flag=1
endelse
if flag then begin
	if n_elements(gen_model) ne 0 then starid=gen_model.starid
	wavelengths=[min(channel_wavelengths),max(channel_wavelengths)]*1e6
	num_wave=n_elements(wavelengths)
	gen_model=alloc_gen_model(num_wave)
	gen_error=gen_model
	if n_elements(startable) eq 1 then gen_model.starid=startable.starid $
				      else gen_model.starid='CCCNNNN'
	if n_elements(starid) ne 0 then gen_model.starid=starid
	gen_model.wavelengths=wavelengths	; model file wavelengths [mu]
	star_model=alloc_star_struct(num_wave)
	star_error=star_model
	star_model[0].component='A'
	star_model[0].mode=13
	binary_struct=alloc_binary_struct()
	binary_model=replicate(binary_struct,1)
endif
;
end
;-------------------------------------------------------------------------------
pro checkimages
;
; Check that images exist for the entire wavelength range specified in genconfig
;
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
nx=n_elements(channel_images[*,0,0])
ny=n_elements(channel_images[0,*,0])
;
for i=0,genconfig.numoutbeam-1 do begin
index=where(scans.vissqerr[i,*,*] gt 0,count)
if count gt 0 then begin
;
minw=min(channel_wavelengths)
maxw=max(channel_wavelengths)
minl=min(genconfig.wavelength[0:genconfig.numspecchan[i]-1,i])
maxl=max(genconfig.wavelength[0:genconfig.numspecchan[i]-1,i])
;
nz=n_elements(channel_images[0,0,*])
if minl lt minw then begin
	print,'Warning: min.lambda cube/genconfig): ',minw*1e6,minl*1e6
	channel_wavelengths=[minl,channel_wavelengths]
	images=channel_images
	channel_images=fltarr(nx,ny,nz+1)
	channel_images[*,*,0]=images[*,*,0]
	channel_images[*,*,1:nz]=images
endif
nz=n_elements(channel_images[0,0,*])
if maxl gt maxw then begin
	print,'Warning: max.lambda cube/genconfig): ',maxw*1e6,maxl*1e6
	channel_wavelengths=[channel_wavelengths,maxl]
	images=channel_images
	channel_images=fltarr(nx,ny,nz+1)
	channel_images[*,*,0:nz-1]=images
	channel_images[*,*,nz]=images[*,*,nz-1]
endif
;
endif
endfor
;
end
;-------------------------------------------------------------------------------
pro plotimages,compress=compress,magnify=magnify
;
; Display a GUI to brwose through images of a cube. Compress will scale the
; dynamic range, magnify will enlrage the images.
;
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
;
labels=string(channel_wavelengths*1e6,format='(f6.3)')
;
; Find center
nx=n_elements(channel_images[*,0,0])
ny=n_elements(channel_images[0,*,0])
index=where(channel_images[*,*,0] eq max(channel_images[*,*,0]))
index=index[0]
j=index/nx
i=index mod nx
if n_elements(compress) eq 0 then compress=1.0
if n_elements(magnify) eq 0 then magnify=1 else magnify=fix(magnify)
m=magnify
images=channel_images[(i-(nx/2)/m) > 0:(i+(nx/2)/m) < nx, $
		      (j-(ny/2)/m) > 0:(j+(ny/2)/m) < nx,*)^(1./compress)
plot_images,images,labels
;
end
;-------------------------------------------------------------------------------
pro sedimages,sedfile,split=split
;
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
;
if n_elements(sedfile) eq 0 then begin
	print,'***Error: please specify filename!'
	return
endif
;
flux=total(total(channel_images,1),1)
l=channel_wavelengths
;
if keyword_set(split) then begin
	r=poly_fit(l*1e6,flux,1,yfit)
	v=medianve(flux-yfit,e)
	index=where(abs(flux-yfit) lt 2*e)
	r=poly_fit(l[index]*1e6,flux[index],2)
	yfit=poly(l*1e6,r)
	plot,l,flux,psym=0
	oplot,l,yfit,psym=0
	f=yfit
	save,l,f,filename=sedfile+'.A.xdr'
	f=abs(flux-yfit)
	save,l,f,filename=sedfile+'.B.xdr'
	return
endif
;
save,l,f,filename=sedfile
;
end
;-------------------------------------------------------------------------------
pro marquardtdata,y,ysig,ymod
;
; Return a `flat' array of the combined observed or model data sets, as
; requested per ds_options. Please note that this procedures relies
; on error bars of unobserved quantities to be negative since it does
; not check boundary information in GenConfig.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ModelFit,parameters,ds_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common AuxData,parallaxes,k2,vsini
;
y=0.d0
ysig=0.d0
ymod=0.d0
;
; Interferometry
if ds_options.i ne 0 then begin
	nights=geninfo.date
	arrays=geoinfo.systemid
	confgs=geninfo.configid
	wf=1/sqrt(ds_options.i)
	for n=0,n_elements(nights)-1 do begin
		night=where(GenInfo.date eq nights[n] $
			and GeoInfo.systemid eq arrays[n] $
			and GenInfo.configid eq confgs[n])
		loadnight,GenInfo[night[0]].date,GeoInfo[night[0]].systemid,GenInfo[night[0]].configid
;
; 		Scan selection
		index=where(scans.starid eq gen_model.starid,NS)
		if ds_options.t le 0 then begin
			ndata=n_elements(scans[index].vissqc)
			y=[y,reform(scans[index].vissqc,ndata)]
			ysig=[ysig,reform(scans[index].vissqcerr,ndata)*wf]
			ymod=[ymod,reform(scans[index].vissqm,ndata)]
		endif
		if ds_options.t ge 0 then begin
			ndata=n_elements(scans[index].compltriplem)
;			y=[y,reform(scans(index).compltriple,ndata)]
;			ysig=[ysig,reform(scans(index).compltripleerr,ndata)*wf]
;			ymod=[ymod,reform(scans(index).compltriplem,ndata)]
;			New algorithm appropriate for Buscher's Theorem
			ct=reform(scans[index].compltriple)
			cm=reform(scans[index].compltriplem)
			cp=reform(scans[index].triplephasec)
			ce=reform(scans[index].compltripleerr)
			if genconfig.numtriple ge 2 then begin
				ctr=ct[*,0,*,*]
				cti=ct[*,1,*,*]
				ct[*,0,*,*]=ctr*cos(-cp)-cti*sin(-cp)
				ct[*,1,*,*]=cti*cos(-cp)+ctr*sin(-cp)
				cmr=cm[*,0,*,*]
				cmi=cm[*,1,*,*]
				cm[*,0,*,*]=cmr*cos(-cp)-cmi*sin(-cp)
				cm[*,1,*,*]=cmi*cos(-cp)+cmr*sin(-cp)
			endif else begin
				ctr=reform(ct[0,*,*])
				cti=reform(ct[1,*,*])
				ct[0,*,*]=ctr*cos(-cp)-cti*sin(-cp)
				ct[1,*,*]=cti*cos(-cp)+ctr*sin(-cp)
				cmr=cm[0,*,*]
				cmi=cm[1,*,*]
				cm[0,*,*]=cmr*cos(-cp)-cmi*sin(-cp)
				cm[1,*,*]=cmi*cos(-cp)+cmr*sin(-cp)
			endelse
			y=[y,reform(ct,ndata)]
			ysig=[ysig,reform(ce,ndata)*wf]
			ymod=[ymod,reform(cm,ndata)]
		endif
	endfor
endif
;
; Astrometry
if ds_options.a ne 0 then begin
	wf=1/sqrt(ds_options.a)
	y_ra=positions.rho*sin(positions.theta)
	y_dc=positions.rho*cos(positions.theta)
	ymod_ra=positions.rhom*sin(positions.thetam)
	ymod_dc=positions.rhom*cos(positions.thetam)
	y=[y,dblarr(n_elements(positions))]
	ddc=ymod_dc-y_dc
	dra=ymod_ra-y_ra
	phi=atan(dra,ddc)-positions.pa
	if ds_options.c ge 1 then phi[*]=0
	eminor=positions.eminor
	emajor=positions.emajor
	if ds_options.c eq 2 then begin
		eminor[*]=1
		emajor[*]=1
	endif
	e=sqrt(emajor^2-eminor^2)/emajor
	ysig=[ysig,sqrt(eminor^2/(1-(e*cos(phi))^2))*wf]
	ymod=[ymod,sqrt(dra^2+ddc^2)]
endif
;
; Spectroscopy
if ds_options.s ne 0 then begin
	wf=1/sqrt(ds_options.s)
	y=[y,velocities.value]
	ysig=[ysig,velocities.error*wf]
	ymod=[ymod,velocities.valuem]
endif
;
; Photometry
if ds_options.p ne 0 then begin
	wf=1/sqrt(ds_options.p)
	y=[y,magnitudes.value]
	ysig=[ysig,magnitudes.error*wf]
	ymod=[ymod,magnitudes.valuem]
endif
;
; Auxilliary data
; Parallax
if ds_options.px ne 0 then begin
	wf=1/sqrt(ds_options.px)
	index=where(binary_model.mode eq 1,count)
	if count gt 0 then begin
		y=[y,parallaxes.value[index]]
		ysig=[ysig,parallaxes.error[index]*wf]
		ymod=[ymod,parallaxes.valuem[index]]
	endif
endif
; K2
if ds_options.k2 ne 0 then begin
	wf=1/sqrt(ds_options.k2)
	y=[y,k2.value]
	ysig=[ysig,k2.error*wf]
	ymod=[ymod,k2.valuem]
endif
; V sin(i)
 ds_options.vsini=0
if ds_options.vsini ne 0 then begin
	wf=1/sqrt(ds_options.vsini)
	y=[y,vsini.value]
	ysig=[ysig,vsini.error*wf]
	ymod=[ymod,vsini.valuem]
endif
;
; Edit the data
index=where(ysig gt 0,count)
if count gt 0 then begin
	y=y[index]
	ysig=ysig[index]
	ymod=ymod[index]
endif else begin
	print,'***Error(MARQUARDTDATA): no valid data found!'
	y=0
	ysig=0
	ymod=0
endelse
;
end
;************************************************************************Block 6
pro mark32cha,files
;
num_files=n_elements(files)
if num_files eq 0 then begin
	print,'***Error(MARK32CHA): no files specified!'
	return
endif
;
for i=0,n_elements(files)-1 do begin
	get_mark3data,files[i],status
	if status eq 0 then begin
	dotpos=strpos(files[i],'.')
	filename=strmid(files[i],0,dotpos)+'.cha'
	result=findfile(filename,count=fcount)
	if fcount ne 0 then hds_open,filename,'UPDATE',status else $
		hds_new,filename,'DataSet','CHAMELEON',status
	if status ne 0 then begin
		print,'***Error(MARK32CHA): file problem!'
		clearstatus
		return
	endif
	put_SysConfig
	put_scans
	toplevel
	dat_annul
	endif
endfor
;
end
;-------------------------------------------------------------------------------
pro mark32psn,file
;
status=dc_read_free(file,r,t,a,b,p,d,s,x,y,/col,ignore=['!','$TEXT_IN_NUMERIC'])
openw,unit,'star.psn',/get_lun
;
parsemark3,d,y,m,d
jy=jd2jy(julian(y,m,d))
for i=0,n_elements(d)-1 do begin
	printf,unit,'A-B',jy[i],r[i],t[i],a[i],b[i],p[i], $
		format='(a,2x,f9.4,2x,f6.2,1x,f7.2,2x,f5.2,1x,f4.2,2x,f7.2)'
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro psn2tex,component,noload=noload
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ModelFit,parameters,ds_options
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
if n_elements(noload) eq 0 then noload=1
;
if n_elements(component) eq 0 then begin
	print,'***Error(PSN2TEX): component undefined!'
	return
endif
if n_elements(positions) eq 0 then begin
	print,'***Error(PSN2TEX): no astrometry!'
	return
endif
if n_elements(gen_model) eq 0 then begin
	print,'***Error(PSN2TEX): no model!'
	return
endif
index=where(positions.component eq component,count)
if count eq 0 then begin
	print,'***Error(PSN2TEX): no data for this component!'
	return
endif
;
openw,unit,'psn.tex',/get_lun
;
printf,unit,'\documentclass{/home/cah/aastex502/aastex}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{lcccrrrrrrr}'
printf,unit,'\tablecaption{Observation and result log for ' $
	    +gen_model.starid+'}\label{table1}'
printf,unit,'\tablehead{'
printf,unit,'\colhead{}&'
printf,unit,'\colhead{}&'
printf,unit,'\colhead{Number of} &'
printf,unit,'\colhead{$\rho$}&'
printf,unit,'\colhead{$\theta$}&'
printf,unit,'\colhead{$\sigma_{\rm maj}$}&'
printf,unit,'\colhead{$\sigma_{\rm min}$}&'
printf,unit,'\colhead{$\varphi$}&'
printf,unit,'\colhead{O--C$_\rho$}&'
printf,unit,'\colhead{O--C$_\theta$}'
printf,unit,'\\'
printf,unit,'\colhead{UT Date}&'
printf,unit,'\colhead{Julian Year}&'
printf,unit,'\colhead{visibilities}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\\'
printf,unit,'\colhead{(1)}&'
printf,unit,'\colhead{(2)}&'
printf,unit,'\colhead{(3)}&'
printf,unit,'\colhead{(4)}&'
printf,unit,'\colhead{(5)}&'
printf,unit,'\colhead{(6)}&'
printf,unit,'\colhead{(7)}&'
printf,unit,'\colhead{(8)}&'
printf,unit,'\colhead{(9)}&'
printf,unit,'\colhead{(10)}'
printf,unit,'}'
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
RAD=180/!pi
x=positions.rho*sin(positions.theta)
y=positions.rho*cos(positions.theta)
xm=positions.rhom*sin(positions.thetam)
ym=positions.rhom*cos(positions.thetam)
oc=sqrt((x-xm)^2+(y-ym)^2)
oca=atan(x-xm,y-ym)*RAD
;
printf,unit,'\startdata'
;
for i=0,n_elements(positions)-1 do begin
	if positions[i].component eq component then begin
	date=positions[i].date
	parsedate,date,y,m,d
	month=months[long(strmid(date,5,2))-1]
	day=strmid(date,8,2)
	if positions[i].jy lt 1996 then filename=mark3date(y,m,d)+'.cha' else $
					filename=constrictordate(y,m,d)+'.cha'
	if not noload then begin
	load_interferometry,filename
	ds_options.i=1
	ds_options.a=0
	ds_nights=date
	marquardtdata,y,ysig,ymod
	num_vis=n_elements(y)
	endif else begin
	num_vis=0
	endelse
	printf,unit,month+' '+day+'\dotfill', $
		positions[i].jy,num_vis, $
		positions[i].rho,positions[i].theta*RAD, $
		positions[i].emajor,positions[i].eminor,positions[i].pa*RAD, $
;		oc(i),oca(i), $
		positions[i].rho-positions[i].rhom, $
		((positions[i].theta-positions[i].thetam)*RAD) mod 360, $
		format='(a,"&",f9.3,"&",i4,"&",f6.2,"&",f7.2,"&",f6.2,"&",f6.2,"&",f5.1,"&",f6.2,"&",f6.1,"\\")'
	endif
endfor
;
printf,unit,'\enddata'
;
printf,unit,'\end{deluxetable}'
printf,unit,'\end{document}'
;
free_lun,unit
return
;
; Version not using aastex
printf,unit,'\documentstyle[12pt]{article}'
printf,unit,'\renewcommand{\baselinestretch}{1.0}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{Observation and result log for '+gen_model.starid+'}'
printf,unit,'\begin{tabular}{lcccrrrrrr} \tableline \tableline'
printf,unit,'&&&'+ $
	    '\multicolumn{1}{c}{Number of}&'+ $
	    '\multicolumn{1}{c}{$\rho$}&'+ $
	    '\multicolumn{1}{c}{$\theta$}&'+ $
            '\multicolumn{1}{c}{$\sigma_{\rm maj}$}&'+ $
	    '\multicolumn{1}{c}{$\sigma_{\rm min}$}&'+ $
            '\multicolumn{1}{c}{$\varphi$}&'+ $
	    '\multicolumn{1}{c}{O-C}\\'
printf,unit,'\multicolumn{1}{c}{UT Date}&&\multicolumn{1}{c}{Julian Year}&'+ $
	    '\multicolumn{1}{c}{visibilities}&'+ $
	    '\multicolumn{1}{c}{(mas)}&'+ $
	    '\multicolumn{1}{c}{(deg)}&'+ $
	    '\multicolumn{1}{c}{(mas)}&'+ $
	    '\multicolumn{1}{c}{(mas)}&'+ $
	    '\multicolumn{1}{c}{(deg)}&'+ $
	    '\multicolumn{1}{c}{(mas)}\\'
printf,unit,'\multicolumn{1}{c}{(1)}&&'+ $
	    '\multicolumn{1}{c}{(2)}&'+ $
	    '\multicolumn{1}{c}{(3)}&'+ $
	    '\multicolumn{1}{c}{(4)}&'+ $
	    '\multicolumn{1}{c}{(5)}&'+ $
	    '\multicolumn{1}{c}{(6)}&'+ $
	    '\multicolumn{1}{c}{(7)}&'+ $
	    '\multicolumn{1}{c}{(8)}&'+ $
	    '\multicolumn{1}{c}{(9)}\\ \tableline'
printf,unit,'\multicolumn{2}{l}{'+month+' '+day+'\dotfill}', $
		positions[i].jy,num_vis, $
		positions[i].rho,positions[i].theta*RAD, $
		positions[i].emajor,positions[i].eminor,positions[i].pa*RAD, $
		oc[i], $
		format='(a,"&",f9.4,"&",i4,"&",f5.2,"&",f7.2,"&",f6.3,"&",f6.3,"&",f5.1,"&",f5.3,"\\")'
printf,unit,'\tableline \tableline'
printf,unit,'\end{tabular}'
printf,unit,'\end{table}'
printf,unit,'\end{document}'
;
end
;-------------------------------------------------------------------------------
pro psn3tex
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ModelFit,parameters,ds_options
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
if n_elements(positions) eq 0 then begin
	print,'***Error(PSN3TEX): no astrometry!'
	return
endif
if n_elements(gen_model) eq 0 then begin
	print,'***Error(PSN3TEX): no model!'
	return
endif
;
dates=unique(positions.jy)
comps=unique(positions.component)
num=n_elements(dates)
;
openw,unit,'psn.tex',/get_lun
;
printf,unit,'\documentclass{/home/cah/aastex502/aastex}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{lcccccrrrrrr}'
printf,unit,'\tablecaption{Observation and result log for ' $
	    +gen_model.starid+'}\label{table1}'
printf,unit,'\tablehead{'
printf,unit,'\colhead{}&'
printf,unit,'\colhead{}&'
printf,unit,'\colhead{} &'
printf,unit,'\colhead{$\sigma_{\rm maj}$}&'
printf,unit,'\colhead{$\sigma_{\rm min}$}&'
printf,unit,'\colhead{$\varphi$}&'
printf,unit,'\colhead{$\rho_{\rm '+comps[0]+'}$}&'
printf,unit,'\colhead{$\theta_{\rm '+comps[0]+'}$}&'
printf,unit,'\colhead{O--C}&'
printf,unit,'\colhead{$\rho_{\rm '+comps[1]+'}$}&'
printf,unit,'\colhead{$\theta_{\rm '+comps[1]+'}$}&'
printf,unit,'\colhead{O--C}'
printf,unit,'\\'
printf,unit,'\colhead{UT Date}&'
printf,unit,'\colhead{Julian Year}&'
printf,unit,'\colhead{$N_{\rm vis.}$}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}'
printf,unit,'\\'
printf,unit,'\colhead{(1)}&'
printf,unit,'\colhead{(2)}&'
printf,unit,'\colhead{(3)}&'
printf,unit,'\colhead{(4)}&'
printf,unit,'\colhead{(5)}&'
printf,unit,'\colhead{(6)}&'
printf,unit,'\colhead{(7)}&'
printf,unit,'\colhead{(8)}&'
printf,unit,'\colhead{(9)}&'
printf,unit,'\colhead{(19)}&'
printf,unit,'\colhead{(11)}&'
printf,unit,'\colhead{(12)}'
printf,unit,'}'
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
RAD=180/!pi
;
printf,unit,'\startdata'
;
for i=0,n_elements(dates)-1 do begin
	i0=where(positions.jy eq dates[i])
	i1=where(positions.jy eq dates[i] and positions.component eq comps[0],i1c)
	i2=where(positions.jy eq dates[i] and positions.component eq comps[1],i2c)
	i0=i0[0]
	i1=i1[0]
	i2=i2[0]
;
	if i1c gt 0 then begin
		rho1=string(positions[i1].rho,format='(f3.1)')
		theta1=string(positions[i1].theta*RAD,format='(f5.1)')
		x=positions[i1].rho*sin(positions[i1].theta)
		y=positions[i1].rho*cos(positions[i1].theta)
		xm=positions[i1].rhom*sin(positions[i1].thetam)
		ym=positions[i1].rhom*cos(positions[i1].thetam)
		oc1=string(sqrt((x-xm)^2+(y-ym)^2),format='(f4.2)')
	endif else begin
		rho1=''
		theta1=''
		oc1=''
	endelse
;
	if i2c gt 0 then begin
		rho2=string(positions[i2].rho,format='(f5.1)')
		theta2=string(positions[i2].theta*RAD,format='(f5.1)')
		x=positions[i2].rho*sin(positions[i2].theta)
		y=positions[i2].rho*cos(positions[i2].theta)
		xm=positions[i2].rhom*sin(positions[i2].thetam)
		ym=positions[i2].rhom*cos(positions[i2].thetam)
		oc2=string(sqrt((x-xm)^2+(y-ym)^2),format='(f4.2)')
	endif else begin
		rho2=''
		theta2=''
		oc2=''
	endelse
;
	parsedate,positions[i0].date,y,m,d
	filename=constrictordate(y,m,d)+'.cha'
	load_interferometry,filename
	ds_options.i=1
	ds_options.a=0
	ds_nights=date
	marquardtdata,y,ysig,ymod
	num_vis=n_elements(y)
	month=months[long(strmid(date,5,2))-1]
	day=strmid(date,8,2)
	printf,unit,month+' '+day+'\dotfill', $
		positions[i0].jy,num_vis, $
		positions[i0].emajor,positions[i0].eminor,positions[i0].pa*RAD, $
		rho1,theta1,oc1,rho2,theta2,oc2, $
		format='(a,"&",f8.3,"&",i4,"&",f4.2,"&",f4.2,"&",f5.1,"&",a,"&",a,"&",a,"&",a,"&",a,"&",a,"\\")'
endfor
;
printf,unit,'\enddata'
;
printf,unit,'\end{deluxetable}'
printf,unit,'\end{document}'
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro pti2tex
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ModelFit,parameters,ds_options
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
if n_elements(scans) eq 0 then begin
	print,'***Error(PTI2TEX): no interferometry!
	return
endif
if n_elements(gen_model) eq 0 then begin
	print,'***Error(PTI2TEX): no model!
	return
endif
;
openw,unit,'pti.tex',/get_lun
;
printf,unit,'\documentclass[12pt]{article}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{PTI Observation log for '+gen_model.starid+'}'
printf,unit,'\begin{tabular}{lcccrr} \tableline \tableline'
printf,unit,'&&&'+ $
	    '\multicolumn{1}{c}{Number of}&'+ $
	    '\multicolumn{1}{c}{$\rho_\mathrm{eph.}$}&'+ $
	    '\multicolumn{1}{c}{$\theta_\mathrm{eph.}$}\\'
printf,unit,'\multicolumn{1}{c}{UT Date}&&\multicolumn{1}{c}{Julian Year}&'+ $
	    '\multicolumn{1}{c}{visibilities}&'+ $
	    '\multicolumn{1}{c}{(mas)}&'+ $
	    '\multicolumn{1}{c}{(deg)}\\'
printf,unit,'\multicolumn{1}{c}{(1)}&&'+ $
	    '\multicolumn{1}{c}{(2)}&'+ $
	    '\multicolumn{1}{c}{(3)}&'+ $
	    '\multicolumn{1}{c}{(4)}&'+ $
	    '\multicolumn{1}{c}{(5)}\\ \tableline'
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
parsedate,date,y,m,d
jd=julian(y,m,d)+scans.time/86400
jd2date,jd,y,m,d
nites=unique(constrictordate(y,m,d))
dates=constrictordate(y,m,d)
;
for i=0,n_elements(nites)-1 do begin
	index=where(dates eq nites[i],num_vis)
	jy=jd2jy(avg(jd[index]))
	pos=binarypos(jy2jd(jy))
	month=months[long(strmid(nites[i],5,2))-1]
	day=strmid(nites[i],8,2)
	printf,unit,'\multicolumn{2}{l}{'+month+' '+day+'\dotfill}', $
		jy,num_vis, $
		pos[0],pos[1], $
		format='(a,"&",f9.4,"&",i4,"&",f5.2,"&",f7.2,"\\")'
endfor
;
printf,unit,'\tableline \tableline'
printf,unit,'\end{tabular}'
printf,unit,'\label{table}'
printf,unit,'\end{table}'
printf,unit,'\end{document}'
;
free_lun,unit
return
;
; Version not using aastex
printf,unit,'\documentstyle[12pt]{article}'
printf,unit,'\renewcommand{\baselinestretch}{1.0}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{PTI Observation log for '+gen_model.starid+'}'
printf,unit,'\begin{tabular}{lcccrr} \tableline \tableline'
printf,unit,'&&&'+ $
	    '\multicolumn{1}{c}{Number of}&'+ $
	    '\multicolumn{1}{c}{$\rho_\mathrm{eph.}$}&'+ $
	    '\multicolumn{1}{c}{$\theta_\mathrm{eph.}$}\\'
printf,unit,'\multicolumn{1}{c}{UT Date}&&\multicolumn{1}{c}{Julian Year}&'+ $
	    '\multicolumn{1}{c}{visibilities}&'+ $
	    '\multicolumn{1}{c}{(mas)}&'+ $
	    '\multicolumn{1}{c}{(deg)}\\'
printf,unit,'\multicolumn{1}{c}{(1)}&&'+ $
	    '\multicolumn{1}{c}{(2)}&'+ $
	    '\multicolumn{1}{c}{(3)}&'+ $
	    '\multicolumn{1}{c}{(4)}&'+ $
	    '\multicolumn{1}{c}{(5)}\\ \tableline'
;
printf,unit,'\multicolumn{2}{l}{'+month+' '+day+'\dotfill}', $
		jy,num_vis, $
		pos[0],pos[1], $
		format='(a,"&",f9.4,"&",i4,"&",f5.2,"&",f7.2,"\\")'
;
printf,unit,'\tableline \tableline'
printf,unit,'\end{tabular}'
printf,unit,'\label{table}'
printf,unit,'\end{table}'
printf,unit,'\end{document}'
;
end
;-------------------------------------------------------------------------------
pro vel2tex
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(velocities) eq 0 then begin
	print,'***Error(VEL2TEX): no astrometry!
	return
endif
;
openw,unit,'vel.tex',/get_lun
;
printf,unit,'\documentclass{/home/cah/aastex502/aastex}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{rllrrrrr}'
printf,unit,'\tablecaption{Radial velocity result log for '+gen_model.starid+'}\label{table1}'
;
printf,unit,'\tablehead{'
;
printf,unit,'\colhead{}&'
printf,unit,'\colhead{Julian}&'
printf,unit,'\colhead{}&'
printf,unit,'\multicolumn{2}{c}{Primary}&'
printf,unit,'\multicolumn{2}{c}{Secondary}&'
printf,unit,'\colhead{}'
printf,unit,'\\'
printf,unit,'\colhead{Date}&'
printf,unit,'\colhead{Year}&'
printf,unit,'\colhead{MJD}&'
printf,unit,'\colhead{[km/s]}&'
printf,unit,'\colhead{$(O-C)$}&'
printf,unit,'\colhead{[km/s]}&'
printf,unit,'\colhead{$(O-C)$}&'
printf,unit,'\colhead{Observatory}'
printf,unit,'}'
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
jds=unique(velocities.jd)
;
printf,unit,'\startdata'
;
for i=0,n_elements(jds)-1 do begin
	i_a=where(velocities.jd eq jds[i] and velocities.component eq 'A',a_c)
	i_b=where(velocities.jd eq jds[i] and velocities.component eq 'B',b_c)
	if a_c eq 1 then index=i_a else index=i_b
	parsedate,velocities[index].date,y,m,d
	month=months[long(strmid(velocities[index].date,5,2))-1]
	day=strmid(velocities[index].date,8,2)
	if a_c+b_c eq 2 then begin
	printf,unit,month+' '+day+'\dotfill', $
		jd2jy(velocities[index].jd+2440000.d0), $
		velocities[index].jd+39999.5d0, $
		velocities[i_a].value, $
		velocities[i_a].value-velocities[i_a].valuem, $
		velocities[i_b].value, $
		velocities[i_b].value-velocities[i_b].valuem, $
		velocities[index].comment, $
		format='(a,"&",f9.4,"&",f9.2,"&$",f5.1,"$&$",f5.2,"$&$",f5.1,"$&$",f5.2,"$&",a,"\\")'
	endif else if a_c eq 1 then begin
	printf,unit,month+' '+day+'\dotfill', $
		jd2jy(velocities[index].jd+2440000.d0), $
		velocities[i_a].jd+39999.5d0, $
		velocities[i_a].value, $
		velocities[i_a].value-velocities[i_a].valuem, $
		velocities[index].comment, $
		format='(a,"&",f9.4,"&",f9.2,"&$",f5.1,"$&$",f5.2,"$&","&","&",a,"\\")'
	endif else begin
	printf,unit,month+' '+day+'\dotfill', $
		jd2jy(velocities[index].jd+2440000.d0), $
		velocities[index].jd+39999.5d0, $
		velocities[i_b].value, $
		velocities[i_b].value-velocities[i_b].valuem, $
		velocities[index].comment, $
		format='(a,"&",f9.4,"&",f9.2,"&","&","&$",f5.1,"$&$",f5.2,"$&",a,"\\")'
	endelse
endfor
;
printf,unit,'\enddata'
;
printf,unit,'\end{deluxetable}'
printf,unit,'\end{document}'
;
free_lun,unit
return
;
; Version not using aastex
printf,unit,'\documentstyle[12pt]{article}'
printf,unit,'\renewcommand{\baselinestretch}{1.0}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{Observation and result log for '+gen_model.starid+'}'
printf,unit,'\begin{tabular}{lllrrrrrrr}'
;
printf,unit,'&'+ $
	    '\multicolumn{1}{c}{Julian}&'+ $
	    '&'+ $
	    '\multicolumn{3}{c}{Primary}&'+ $
	    '\multicolumn{3}{c}{Secondary}\\'
printf,unit,'\multicolumn{1}{c}{Date}&'+ $
	    '\multicolumn{1}{c}{Year}&'+ $
	    '\multicolumn{1}{c}{MJD}&'+ $
	    '\multicolumn{1}{c}{[km/s]}&'+ $
	    '\multicolumn{1}{c}{$\pm$}&'+ $
	    '\multicolumn{1}{c}{$(O-C)$}&'+ $
	    '\multicolumn{1}{c}{[km/s]}&'+ $
	    '\multicolumn{1}{c}{$\pm$}&'+ $
	    '\multicolumn{1}{c}{$(O-C)$}&'+ $
            '\multicolumn{1}{c}{Observatory}\\ \tableline'
;
printf,unit,month+' '+day+'\dotfill', $
		jd2jy(velocities[index[i]].jd+2440000.d0), $
		velocities[index[i]].jd+39999.5d0, $
		velocities[index[i]].value,velocities[index[i]].error, $
		velocities[index[i]].value-velocities[index[i]].valuem, $
		velocities[jndex].value,velocities[jndex].error, $
		velocities[jndex].value-velocities[jndex].valuem, $
		velocities[index[i]].comment, $
		format='(a,"&",f9.4,"&",f9.2,"&",f5.1,"&",f5.2,"&",f5.2,"&",f5.1,"&",f5.2,"&",f5.2,"&",a,"\\")'
;
printf,unit,'\tableline \tableline'
printf,unit,'\end{tabular}'
printf,unit,'\end{table}'
printf,unit,'\end{document}'
;
end
;-------------------------------------------------------------------------------
pro roger2vel,rfile,uw=uw
;
; Below is Roger's data-file of observation for 93 Leo.  The first column is
; *modified* JD, then observed velocity, then (where given) the weight. The
; final column is a code for the symbol on the plot when the default is not
; selectd, and can of course be ignored.
;
; I have put at the top of the file the information which Roger uses to
; combine the different sets.  He identifies each source of measurements by
; a symbol on the plot, and refers to each source by a code.  The code is
; also used for making global corrections or weightings in the
; combined solution.
;
; In the pre-file below, the first number is the observation code as
; given in the main datafile.  The next number, if it is small, is the
; weight which is to be applied globally to the observations corresponding
; to that code; if it is 100 or more, then subtract 100 and you have a
; universal correction which is to be applied to all velocities from that
; source.
;
; codes=[202,203,204,205,206,218,219,303]
; wghts=[.25,.07,0.0,.25,.10,0.5,0.5,1.0]
;
codes=[202,203,204,205,218,219,303]
wghts=[0.6,1.0,1.3,1.0,0.5,0.5,1.2]
offss=[0.5,0.5,0.0,.25,0.0,0.5,0.0]
;
mjd=0.d0
rv=0.0
w=''
c=0
s=dc_read_fixed(rfile,mjd,rv,w,c,/col,format='(f11.2,f9.1,a6,i4)',ignore='!')
;
index=where(strlen(strcompress(w,/remove)) eq 0,count)
if count gt 0 then w[index]='1.0'
w=float(w)
;
for i=0,n_elements(codes)-1 do begin
	index=where(c eq codes[i],count)
	if count gt 0 then begin
		w[index]=w[index]*wghts[i]
		rv[index]=rv[index]+offss[i]
	endif
endfor
index=where(w gt 0,count)
if count eq 0 then return
mjd=mjd[index]
rv=rv[index]
w=w[index]
;
jd=mjd+2400000.5
;
if n_elements(uw) eq 0 then uw=1
s=uw/sqrt(w)
;
openw,unit,rfile+'.vel',/get_lun
for i=0,n_elements(rv)-1 do printf,unit,'A ',jd[i],rv[i],s[i], $
	format='(a,2x,f10.2,2x,f6.2,2x,f5.2)'
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro compile_speckle,date,rho,theta
;
; With orbital elements stored in the startable, compute current separations
; and select all those double stars with separations less than 200 mas.
;
common StarBase,StarTable,Notes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
rad=180/pi_circle
;
; get_wds
;
if n_elements(StarTable) eq 0 then begin
	print,'***Error(COMPILE_SPECKLE): no StarTable!'
	return
endif
if n_elements(date) eq 0 then begin
	print,'***Error(COMPILE_SPECKLE): no date!'
	return
endif
n=n_elements(StarTable)
if n eq 0 then return
o_parms=dblarr(8,n)
rho=dblarr(n)
theta=rho
;
o_parms[0,*]=StarTable.a
o_parms[1,*]=StarTable.e
o_parms[2,*]=StarTable.i/rad
o_parms[2,*]=StarTable.i/rad
o_parms[3,*]=StarTable.o/rad
o_parms[5,*]=StarTable.n/rad
o_parms[6,*]=StarTable.p
o_parms[7,*]=StarTable.t
o_parms[7,*]=StarTable.t
;
parsedate,date,y,m,d
jd=julian(y,m,d)
;
for i=0,n-1 do begin
	xy=true2app(jd,o_parms[*,i],r,t,StarTable[i].ra,StarTable[i].dec)
	rho[i]=r
	theta[i]=t
endfor
;
index=where(rho lt 0.150,count)
if count gt 0 then begin
	StarTable=StarTable[index]
	rho=rho[index]
	theta=theta[index]
endif else print,'No stars found!'
;
end
;-------------------------------------------------------------------------------
pro compile_astrometry
;
common StarBase,StarTable,Notes
;
get_bsc
index=where(startable.dec gt -20 and startable.mv lt 4 and startable.mv gt 1)
startable=startable[index]
get_diameter
index=where(startable.diameter lt 3 and startable.bflag ne 'B')
startable=startable[index]
reduceobslist,15
selectup
compileobslist
;
end
;-------------------------------------------------------------------------------
