;----------------------------------------------------------------------------
pro average
; modified bias correction
;
; Averages PointData. Applies background correction. Stores the data in scans,
; and also puts a copy into the buffer (after clearing it) for use by AMOEBA.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,Date,SystemId,MetroConfig,GeoParms,GenConfig,GeoInfo,GenInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime,FDLPos,FDLPosErr, $
        DelayJitter,DelayJitterErr,NATJitter,NATJitterErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        MetroPos,MetroPosErr,VacDelay,VacDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common MetroData,Sol0,SolN,Jscan,MetroTime, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ModelFit,parameters,ds_options
common mybias,coeffa0,coeffa1,coeffb0,coeffa0sig,coeffa1sig,coeffb0sig,coeffchisq,coeffchisq2
;
print,"mw average"
;
; Check presence of data
if checkdata([1,2,6,7,8,10,11]) ne 0 then return
;
; Minimum number of data points needed for a scan
MINPOINTS=10
;
NS=n_elements(ScanTable)
;
; Allocate and initialize arrays for ScanData if not done before or uptodate
if n_elements(scans) ne NS then begin
        scans=replicate(scan(),NS)
        s=size(scans(0))
        if s(n_elements(s)-2) ne 8 then begin
                print,'***Error(AVERAGE): GenConfig incomplete!
                scans=-1
                return
        endif
        scans.starid=ScanTable.StarId
        if NS gt 1 then scans.iscan=lindgen(NS)+1 else scans.iscan=1
        bufferinfo=replicate(nightinfo(),1)
        ds_options.i=1
endif

; Set calibration factors
case SystemId of
        'Mark_III':normfactor=pi_circle^2/2
        'NPOI'    :normfactor=pi_circle^2/8/(1-1/sqrt(2))
              else:begin
                   print,'***Error(AVERAGE): unknown system: ',SystemId,'!'
                   return
                   end
endcase
if GenConfig.NumTriple gt 0 then $
      triple_factor=fltarr(NS,GenConfig.NumTriple,max(GenConfig.NumSpecChan))+1
;
; Choose the method for computing error bars, N, G, or P (see below)
UncertaintyEstimate='G'
;
; Background rate, photon rate, visibilities
;
print,'Averaging rates and squared visibilities...'
;
for i=0,GenConfig.NumOutBeam-1 do begin
for j=0,GenConfig.NumSpecChan(i)-1 do begin
for k=0,n_elements(Iscan)-1 do begin
;
  kk=Iscan(k)-1 ; subtract 1 since arrays start with 0!
  StartTime=ScanTable(kk).StartTime
  StopTime=ScanTable(kk).StopTime
;
; Check background measurement
  index=where(BGTable.ScanId eq ScanTable(kk).ScanId,bg_count)
  if bg_count gt 0 then begin
    bgr=bgscans(index).Rate(i,j)
    bgre=bgscans(index).RateErr(i,j)
    bgt=bgscans(index).Time
    bg_index=where(bgre gt 0,bg_count)
  endif
;
; Check photonrate
  PR=PhotonRate(i,j,Rec0(k):RecN(k))
  PRE=PhotonRateErr(i,j,Rec0(k):RecN(k))
  pr_index=where(PRE gt 0,NumPoint)
;
; Catch frozen channels with the back ground rate = photonrate
  if NumPoint gt 0 and bg_count gt 0 then $
  if avg(PR(pr_index))-avg(bgr(bg_index)) eq 0 then NumPoint=0
;
; Only average scans with background and sufficient number of points
;
  IF NumPoint ge MINPOINTS AND bg_count gt 0 THEN BEGIN
;
;     Average the background data
;
      bgr=bgr(bg_index)
      bgre=bgre(bg_index)
      bgwt=1/bgre^2
      bgt=bgt(bg_index)
      sum_bgwt=total(bgwt)
      scans(kk).BackgndRate(i,j)=total(bgr*bgwt)/sum_bgwt
      scans(kk).BackgndErr(i,j)=1/sqrt(sum_bgwt)
      index_l=where(bgt lt StartTime,count_l)
      index_r=where(bgt gt StopTime,count_r)
;
;     Average photonrate and squared visibilities
;
      wt=1/PRE(pr_index)^2
      wt(*)=1
      sum_wt=total(wt)
      scans(kk).PhotonRate(i,j)=total(PR(pr_index)*wt)/sum_wt $
                               -scans(kk).BackgndRate(i,j)
      case UncertaintyEstimate of
      'N':begin
;         Normal distribution
          scans(kk).PhotonRateErr(i,j)=1/sqrt(sum_wt)
          end
      'G':begin
;         Gaussian error propagation
          scans(kk).PhotonRateErr(i,j)=stdev(PR(pr_index))/sqrt(NumPoint)
          end
      'P':begin
;         Poisson noise theoretical estimate
          scans(kk).PhotonRateErr(i,j)=sqrt(abs(scans(kk).PhotonRate(i,j)) $
                  /(ScanTable(kk).NumCoh*ScanTable(kk).NumIncoh*NumPoint))
          end
      endcase
;
      bgpoints=fltarr(NumPoint)
      if (count_l gt 0) and (count_r gt 0) then begin
        x=(findgen(NumPoint)/NumPoint*(StopTime-StartTime)+StartTime)/3600
        coeffs=polyfitw(bgt/3600,bgr,bgwt,1)
        bgpoints=poly(x,coeffs)
      endif else begin
        bgpoints(*)=scans(kk).BackgndRate(i,j)
      endelse
      calfactor=avg(PR(pr_index)-bgpoints)^2    ; Used until 1999-07-09
      calfactors=(PR(pr_index)-bgpoints)^2
      index=where(calfactors ne 0,NumPoint)
      if NumPoint ge MINPOINTS then begin
        calfactors=calfactors(index)
        pr_index=pr_index(index)
        bias=(avg(PR(pr_index))/10)/normfactor  ; Approximate bias correction
     bias=(coeffa0(i,j)*avg(PR(pr_index))^coeffa1(i,j))/normfactor
     bias=avg(PR(pr_index))*avg(PR(pr_index))*bias
      endif
;
      for l=0,GenConfig.NumBaseline(i)-1 do begin
;
        if NumPoint ge MINPOINTS then begin
          VSQ=VisSq(i,j,l,Rec0(k):RecN(k))-bias & VSQ=VSQ(pr_index)/calfactors
          VSQE=VisSqErr(i,j,l,Rec0(k):RecN(k)) & VSQE=VSQE(pr_index)/calfactors
          index=where(VSQE gt 0,count)
        endif else count=0

        if count gt MINPOINTS then begin
          wt=1/VSQE(index)^2
          wt(*)=1
          sum_wt=total(wt)
          scans(kk).VisSq(i,j,l)= $
                        (total(VSQ(index)*wt)/sum_wt)*normfactor
          case SystemId of
          'Mark_III':scans(kk).VisSqErr(i,j,l)= $
                        (stdev(VSQ(index))/sqrt(count))*normfactor
          'NPOI'  :begin
                   case UncertaintyEstimate of
               'N':begin
;                  Normal distribution
                   scans(kk).VisSqErr(i,j,l)= $
                        (1/sqrt(sum_wt))*normfactor
                   end
               'G':begin
;                  Gaussian error propagation
                   scans(kk).VisSqErr(i,j,l)= $
                        (stdev(VSQ(pr_index))/sqrt(NumPoint))*normfactor
                   end
               'P':begin
;                  Estimate for Poisson noise, photonrate fluctuations neglected
                   scans(kk).VisSqErr(i,j,l)=sqrt( $
                        pi_circle^4 $
                        /(4*ScanTable(kk).NumCoh*ScanTable(kk).NumIncoh*count) $
                        *(1/scans(kk).PhotonRate(i,j)^2 $
                          +(4/pi_circle^2)*abs(scans(kk).VisSq(i,j,l)) $
                                              /scans(kk).PhotonRate(i,j)))
                   end
                   endcase
                   end
          endcase
        endif else scans(kk).VisSqErr(i,j,l)=-1
;
        scans(kk).VisSqE(i,j,l)=scans(kk).VisSq(i,j,l)
        scans(kk).VisSqEErr(i,j,l)=scans(kk).VisSqErr(i,j,l)
        scans(kk).VisSqC(i,j,l)=scans(kk).VisSq(i,j,l)
        scans(kk).VisSqCErr(i,j,l)=scans(kk).VisSqErr(i,j,l)
        scans(kk).VisSqEC(i,j,l)=scans(kk).VisSq(i,j,l)
        scans(kk).VisSqECErr(i,j,l)=scans(kk).VisSqErr(i,j,l)
;
;
        for tr=0,GenConfig.NumTriple-1 do begin
         for bl=0,2 do begin
          if (GenConfig.TripleBeam(bl,tr) eq i) and $
             (GenConfig.TripleBase(bl,tr) eq l) then begin
           index=where( $
            GenConfig.TripleChan(0:GenConfig.TripleNumChan(tr)-1,bl,tr) eq j,n)
           if n gt 0 then triple_factor(kk,tr,index)= $
                          triple_factor(kk,tr,index)/sqrt(calfactor/normfactor)
          endif
         endfor
        endfor
;
      endfor
;
  ENDIF ELSE BEGIN
;
      if bg_count eq 0 then begin
        print,'Warning(AVERAGE): no BackgndRate for OB=',i+1, $
            ', CH=',j+1,', scan=',kk+1, $
            ', Star: ',ScanTable(kk).starid, $
            format='(a,i1,a,i2,a,i3,a,a)'
        scans(kk).BackgndErr(i,j)=-1
      endif
;     if NumPoint eq 0 then $
;     print,'Warning(AVERAGE): no valid photonrates for scan ',kk+1, $
;         ', output beam ',i+1,', channel ',j+1, $
;         format='(a,i3,a,i1,a,i2)'
;
      scans(kk).PhotonRateErr(i,j)=-1
      for l=0,GenConfig.NumBaseline(i)-1 do begin
        scans(kk).VisSqErr(i,j,l)=-1
        scans(kk).VisSqEErr(i,j,l)=-1
        scans(kk).VisSqCErr(i,j,l)=-1
        scans(kk).VisSqECErr(i,j,l)=-1
      endfor
      for t=0,GenConfig.NumTriple-1 do begin
       for l=0,2 do begin
        if (GenConfig.TripleBeam(l,t) eq i) then begin
          index=where( $
              GenConfig.TripleChan(0:GenConfig.TripleNumChan(t)-1,l,t) eq j,n)
          if n gt 0 then begin
            triple_factor(kk,t,index)=0
            scans(kk).TripleAmpErr(t,index)=-1
            scans(kk).TripleAmpEErr(t,index)=-1
            scans(kk).TripleAmpCErr(t,index)=-1
            scans(kk).TripleAmpECErr(t,index)=-1
            scans(kk).TriplePhaseErr(t,index)=-1
            scans(kk).TriplePhaseCErr(t,index)=-1
          endif
         endif
        endfor
      endfor
;
  ENDELSE
;
endfor
endfor
endfor
;
; Triple products
;
print,'Averaging triple products...'
;
for t=0,GenConfig.NumTriple-1 do begin
for j=0,GenConfig.TripleNumChan(t)-1 do begin
for k=0,n_elements(Iscan)-1 do begin
;
kk=Iscan(k)-1
i_r=where(  TripleAmpErr(t,j,Rec0(k):RecN(k)) gt 0,count_r)
i_i=where(TriplePhaseErr(t,j,Rec0(k):RecN(k)) gt 0,count_i)
if count_r ne count_i then begin
  print,'***Error(AVERAGE): inconsistent triple editing!'
  return
endif
if count_r gt MINPOINTS and triple_factor(kk,t,j) ne 0 then begin
  CTR=ComplTriple(t,0,j,Rec0(k):RecN(k))
  CTI=ComplTriple(t,1,j,Rec0(k):RecN(k))
  CTER=ComplTripleErr(t,0,j,Rec0(k):RecN(k))
  CTEI=ComplTripleErr(t,1,j,Rec0(k):RecN(k))
  wtr=1/CTER(i_r)^2
  wtr(*)=1
  wti=1/CTEI(i_i)^2
  wti(*)=1
  sum_wtr=total(wtr)
  sum_wti=total(wti)
  scans(kk).ComplTriple(t,0,j)=total(CTR(i_r)*wtr)/sum_wtr*triple_factor(kk,t,j)
  scans(kk).ComplTriple(t,1,j)=total(CTI(i_i)*wti)/sum_wti*triple_factor(kk,t,j)
  case UncertaintyEstimate of
  'N':begin
;     Normal distribution
      scans(kk).ComplTripleErr(t,0,j)=1/sqrt(sum_wtr)*triple_factor(kk,t,j)
      scans(kk).ComplTripleErr(t,1,j)=1/sqrt(sum_wti)*triple_factor(kk,t,j)
      end
  'G':begin
;     Gaussian error propagation
      scans(kk).ComplTripleErr(t,0,j)= $
        stdev(CTR(i_r))/sqrt(count_r)*triple_factor(kk,t,j)
      scans(kk).ComplTripleErr(t,1,j)= $
        stdev(CTI(i_i))/sqrt(count_i)*triple_factor(kk,t,j)
      end
  'P':begin
;     Not available, do normal distribution instead
      scans(kk).ComplTripleErr(t,0,j)=1/sqrt(sum_wtr)*triple_factor(kk,t,j)
      scans(kk).ComplTripleErr(t,1,j)=1/sqrt(sum_wti)*triple_factor(kk,t,j)
      end
  endcase
;
; Compute amplitude and phase...
  scans(kk).TripleAmp(t,j)=sqrt(scans(kk).ComplTriple(t,0,j)^2+ $
                                scans(kk).ComplTriple(t,1,j)^2)
  if (scans(kk).ComplTriple(t,0,j) eq 0) and  $
     (scans(kk).ComplTriple(t,1,j) eq 0) then $
       scans(kk).TriplePhase(t,j)=0 $
  else scans(kk).TriplePhase(t,j)= $
                atan(scans(kk).ComplTriple(t,1,j), $
                     scans(kk).ComplTriple(t,0,j))
; ...and their errors
  scans(kk).TripleAmpErr(t,j)= $
    sqrt((scans(kk).ComplTriple(t,0,j)*scans(kk).ComplTripleErr(t,0,j))^2 $
        +(scans(kk).ComplTriple(t,1,j)*scans(kk).ComplTripleErr(t,1,j))^2) $
            /scans(kk).TripleAmp(t,j)
  if scans(kk).TripleAmp(t,j) eq 0 then scans(kk).TriplePhaseErr(t,j)=-1 $
  else scans(kk).TriplePhaseErr(t,j)=scans(kk).TripleAmpErr(t,j)/ $
                                     scans(kk).TripleAmp(t,j)
; Initialize derived variables
  scans(kk).TripleAmpC(t,j)=scans(kk).TripleAmp(t,j)
  scans(kk).TripleAmpCErr(t,j)=scans(kk).TripleAmpErr(t,j)
  scans(kk).TripleAmpE(t,j)=scans(kk).TripleAmp(t,j)
  scans(kk).TripleAmpEErr(t,j)=scans(kk).TripleAmpErr(t,j)
  scans(kk).TripleAmpEC(t,j)=scans(kk).TripleAmp(t,j)
  scans(kk).TripleAmpECErr(t,j)=scans(kk).TripleAmpErr(t,j)
  scans(kk).TriplePhaseC(t,j)=scans(kk).TriplePhase(t,j)
  scans(kk).TriplePhaseCErr(t,j)=scans(kk).TriplePhaseErr(t,j)
endif else begin
  scans(kk).ComplTripleErr(t,0,j)=-1
  scans(kk).ComplTripleErr(t,1,j)=-1
  scans(kk).TripleAmpErr(t,j)=-1
  scans(kk).TripleAmpCErr(t,j)=-1
  scans(kk).TripleAmpEErr(t,j)=-1
  scans(kk).TripleAmpECErr(t,j)=-1
  scans(kk).TriplePhaseErr(t,j)=-1
  scans(kk).TriplePhaseCErr(t,j)=-1
endelse
;
endfor
endfor
endfor
;
;
; Time, delays, NAT data
;
print,'Averaging delays...'
;
for k=0,n_elements(Iscan)-1 do begin
;
; Time
        kk=Iscan(k)-1
        t=PointTime(Rec0(k):RecN(k))
        index=where(t ge 0,count)
        if count gt 0 then scans(kk).Time=avg(t(index)) $
                else scans(kk).Time=-avg(abs(t))
;
; DelayJitter
for i=0,GenConfig.NumOutBeam-1 do begin
for l=0,GenConfig.NumBaseline(i)-1 do begin
        v=DelayJitter(i,l,Rec0(k):RecN(k))
        ve=DelayJitterErr(i,l,Rec0(k):RecN(k))
        index=where(ve gt 0,count)
        if count gt MINPOINTS then begin
                scans(kk).DelayJitter(i,l)=median(v(index))
                scans(kk).DelayJitterErr(i,l)=1e-6
        endif else scans(kk).DelayJitterErr(i,l)=-1e-6
endfor
endfor
;
; NATJitter, delay positions
for i=0,GenConfig.NumSid-1 do begin
;
        v=NATJitter(i,Rec0(k):RecN(k))
        ve=NATJitterErr(i,Rec0(k):RecN(k))
        index=where(ve gt 0,count)
        if count gt MINPOINTS then begin
                scans(kk).NATJitter(i)=median(v(index))
                scans(kk).NATJitterErr(i)=0.01
        endif else scans(kk).NATJitterErr(i)=-1
;
        v=FDLPos(i,Rec0(k):RecN(k))
        ve=FDLPosErr(i,Rec0(k):RecN(k))
        index=where(ve gt 0,count)
        if count gt MINPOINTS then begin
                coeffs=polyfit(t(index)/3600,v(index),2,vft)
                scans(kk).FDLPos(i)=polynom(scans(kk).Time/3600,coeffs)
                scans(kk).FDLPosErr(i)=stdev(v(index)-vft)
        endif else begin
                scans(kk).FDLPos(i)=0
                scans(kk).FDLPosErr(i)=-1e-6
        endelse
;
        IF n_elements(GrpDelay) gt 1 and n_elements(MetroPos) gt 1 then BEGIN
        IF n_elements(GrpDelay(0,*)) ne n_elements(PointTime) $
        or n_elements(MetroPos(0,*)) ne n_elements(PointTime) then BEGIN
                if i eq 0 then $
                print,'Warning(AVERAGE): incompatible delay corrections!'
        ENDIF ELSE BEGIN
        d=FDLPos(i,Rec0(k):RecN(k))*(i ne GenConfig.RefStation-1) $
         -GrpDelay(i,Rec0(k):RecN(k))*1d-6 $
         -MetroPos(i,Rec0(k):RecN(k))*1d-6*(i ne GenConfig.RefStation-1)
        ferr=FDLPosErr(i,Rec0(k):RecN(k))
        gerr=GrpDelayErr(i,Rec0(k):RecN(k))
        merr=MetroPosErr(i,Rec0(k):RecN(k))
        index=where(ferr gt 0 and gerr gt 0 and merr gt 0,count)
        if count gt MINPOINTS then begin
                coeffs=polyfit(t(index)/3600,d(index),2,dft)
                scans(kk).GrpDelay(i)=polynom(scans(kk).Time/3600,coeffs)
                scans(kk).GrpDelayErr(i)=stdev(d(index)-dft)
                if i eq GenConfig.RefStation-1 then $
                        scans(kk).GrpDelayErr(i)=1e-6
        endif else begin
                scans(kk).GrpDelay(i)=avg(d)
                scans(kk).GrpDelayErr(i)=-1e-6
        endelse
;
        d=FDLPos(i,Rec0(k):RecN(k))*(i ne GenConfig.RefStation-1) $
         -DryDelay(i,Rec0(k):RecN(k))*1d-6 $
         -MetroPos(i,Rec0(k):RecN(k))*1d-6*(i ne GenConfig.RefStation-1)
        derr=DryDelayErr(i,Rec0(k):RecN(k))
        index=where(ferr gt 0 and derr gt 0 and merr gt 0,count)
        if count gt MINPOINTS then begin
                coeffs=polyfit(t(index)/3600,d(index),2,dft)
                scans(kk).DryDelay(i)=polynom(scans(kk).Time/3600,coeffs)
                scans(kk).DryDelayErr(i)=stdev(d(index)-dft)
                if i eq GenConfig.RefStation-1 then $
                        scans(kk).DryDelayErr(i)=1e-6
        endif else begin
                scans(kk).DryDelay(i)=avg(d)
                scans(kk).DryDelayErr(i)=-1e-6
        endelse
;
        d=FDLPos(i,Rec0(k):RecN(k))*(i ne GenConfig.RefStation-1) $
         -WetDelay(i,Rec0(k):RecN(k))*1d-6 $
         -MetroPos(i,Rec0(k):RecN(k))*1d-6*(i ne GenConfig.RefStation-1)
        werr=WetDelayErr(i,Rec0(k):RecN(k))
        index=where(ferr gt 0 and werr gt 0 and merr gt 0,count)
        if count gt MINPOINTS then begin
                coeffs=polyfit(t(index)/3600,d(index),2,dft)
                scans(kk).WetDelay(i)=poly(scans(kk).Time/3600,coeffs)
                scans(kk).WetDelayErr(i)=stdev(d(index)-dft)
                if i eq GenConfig.RefStation-1 then $
                        scans(kk).WetDelayErr(i)=1e-6
        endif else begin
                scans(kk).WetDelay(i)=avg(d)
                scans(kk).WetDelayErr(i)=-1e-6
        endelse
        ENDELSE
;
        if n_elements(Sol0) ne 0 and n_elements(SolN) ne 0 then begin
        if Sol0(kk) ge 0 and SolN(kk) ge 0 then begin
                x=ParX(i,Sol0(kk):SolN(kk))
                xerr=ParXErr(i,Sol0(kk):SolN(kk))
                y=ParY(i,Sol0(kk):SolN(kk))
                yerr=ParYErr(i,Sol0(kk):SolN(kk))
                z=ParZ(i,Sol0(kk):SolN(kk))
                zerr=ParZErr(i,Sol0(kk):SolN(kk))
                scans(kk).ParX(i)=avg(x)
                scans(kk).ParY(i)=avg(y)
                scans(kk).ParZ(i)=avg(z)
                if SolN(kk)-Sol0(kk)+1 ge 2 then begin
                        scans(kk).ParXErr(i)=stdev(double(x))
                        scans(kk).ParYErr(i)=stdev(double(y))
                        scans(kk).ParZErr(i)=stdev(double(z))
                endif else begin
                        scans(kk).ParXErr(i)=ParXErr(Sol0(kk))
                        scans(kk).ParYErr(i)=ParYErr(Sol0(kk))
                        scans(kk).ParZErr(i)=ParZErr(Sol0(kk))
                endelse
        endif else begin
                scans(kk).ParX(i)=0
                scans(kk).ParXErr(i)=-1
                scans(kk).ParY(i)=0
                scans(kk).ParYErr(i)=-1
                scans(kk).ParZ(i)=0
                scans(kk).ParZErr(i)=-1
        endelse
        endif
        ENDIF
;
endfor
;
endfor
;
calcastrom
calcvis
;
; Store a copy in the AMOEBA buffer after clearing it
freememory
storenight
;
print,'Finished averaging.'
print,'mw average'
;
end
