function eraDat,iy,im,id,fd,deltat


;  int eraDat(int iy, int im, int id, double fd, double *deltat )
;  /*
;  **  - - - - - - -
;  **   e r a D a t
;  **  - - - - - - -
;  **
;  **  For a given UTC date, calculate delta(AT) = TAI-UTC.
;  **
;  **     :------------------------------------------:
;  **     :                                          :
;  **     :                 IMPORTANT                :
;  **     :                                          :
;  **     :  A new version of this function must be  :
;  **     :  produced whenever a new leap second is  :
;  **     :  announced.  There are four items to     :
;  **     :  change on each such occasion:           :
;  **     :                                          :
;  **     :  1) A new line must be added to the set  :
;  **     :     of statements that initialize the    :
;  **     :     array "changes".                     :
;  **     :                                          :
;  **     :  2) The constant IYV must be set to the  :
;  **     :     current year.                        :
;  **     :                                          :
;  **     :  3) The "Latest leap second" comment     :
;  **     :     below must be set to the new leap    :
;  **     :     second date.                         :
;  **     :                                          :
;  **     :  4) The "This revision" comment, later,  :
;  **     :     must be set to the current date.     :
;  **     :                                          :
;  **     :  Change (2) must also be carried out     :
;  **     :  whenever the function is re-issued,     :
;  **     :  even if no leap seconds have been       :
;  **     :  added.                                  :
;  **     :                                          :
;  **     :  Latest leap second:  2015 June 30       :
;  **     :                                          :
;  **     :__________________________________________:
;  **
;  **  Given:
;  **     iy     int      UTC:  year (Notes 1 and 2)
;  **     im     int            month (Note 2)
;  **     id     int            day (Notes 2 and 3)
;  **     fd     double         fraction of day (Note 4)
;  **
;  **  Returned:
;  **     deltat double   TAI minus UTC, seconds
;  **
;  **  Returned (function value):
;  **            int      status (Note 5):
;  **                       1 = dubious year (Note 1)
;  **                       0 = OK
;  **                      -1 = bad year
;  **                      -2 = bad month
;  **                      -3 = bad day (Note 3)
;  **                      -4 = bad fraction (Note 4)
;  **                      -5 = internal error (Note 5)
;  **
;  **  Notes:
;  **
;  **  1) UTC began at 1960 January 1.0 (JD 2436934.5) and it is improper
;  **     to call the function with an earlier date.  If this is attempted,
;  **     zero is returned together with a warning status.
;  **
;  **     Because leap seconds cannot, in principle, be predicted in
;  **     advance, a reliable check for dates beyond the valid range is
;  **     impossible.  To guard against gross errors, a year five or more
;  **     after the release year of the present function (see the constant
;  **     IYV) is considered dubious.  In this case a warning status is
;  **     returned but the result is computed in the normal way.
;  **
;  **     For both too-early and too-late years, the warning status is +1.
;  **     This is distinct from the error status -1, which signifies a year
;  **     so early that JD could not be computed.
;  **
;  **  2) If the specified date is for a day which ends with a leap second,
;  **     the UTC-TAI value returned is for the period leading up to the
;  **     leap second.  If the date is for a day which begins as a leap
;  **     second ends, the UTC-TAI returned is for the period following the
;  **     leap second.
;  **
;  **  3) The day number must be in the normal calendar range, for example
;  **     1 through 30 for April.  The "almanac" convention of allowing
;  **     such dates as January 0 and December 32 is not supported in this
;  **     function, in order to avoid confusion near leap seconds.
;  **
;  **  4) The fraction of day is used only for dates before the
;  **     introduction of leap seconds, the first of which occurred at the
;  **     end of 1971.  It is tested for validity (0 to 1 is the valid
;  **     range) even if not used;  if invalid, zero is used and status -4
;  **     is returned.  For many applications, setting fd to zero is
;  **     acceptable;  the resulting error is always less than 3 ms (and
;  **     occurs only pre-1972).
;  **
;  **  5) The status value returned in the case where there are multiple
;  **     errors refers to the first error detected.  For example, if the
;  **     month and day are 13 and 32 respectively, status -2 (bad month)
;  **     will be returned.  The "internal error" status refers to a
;  **     case that is impossible but causes some compilers to issue a
;  **     warning.
;  **
;  **  6) In cases where a valid result is not available, zero is returned.
;  **
;  **  References:
;  **
;  **  1) For dates from 1961 January 1 onwards, the expressions from the
;  **     file ftp://maia.usno.navy.mil/ser7/tai-utc.dat are used.
;  **
;  **  2) The 5ms timestep at 1961 January 1 is taken from 2.58.1 (p87) of
;  **     the 1992 Explanatory Supplement.
;  **
;  **  Called:
;  **     eraCal2jd    Gregorian calendar to JD
;  **
;  **  Copyright (C) 2013-2015, NumFOCUS Foundation.
;  **  Derived, with permission, from the SOFA library.  See notes at end of file.
;  */
  
;  /* Release year for this version of eraDat */
  IYV = 2015l;
  
;  /* Reference dates (MJD) and drift rates (s/day), pre leap seconds */
drift = [ $
  [ 37300.0d, 0.0012960d ],$
  [ 37300.0d, 0.0012960d ],$
  [ 37300.0d, 0.0012960d ],$
  [ 37665.0d, 0.0011232d ],$
  [ 37665.0d, 0.0011232d ],$
  [ 38761.0d, 0.0012960d ],$
  [ 38761.0d, 0.0012960d ],$
  [ 38761.0d, 0.0012960d ],$
  [ 38761.0d, 0.0012960d ],$
  [ 38761.0d, 0.0012960d ],$
  [ 38761.0d, 0.0012960d ],$
  [ 38761.0d, 0.0012960d ],$
  [ 39126.0d, 0.0025920d ],$
  [ 39126.0d, 0.0025920d ] $
];
drift=rotate(drift,4)
;/* Number of Delta(AT) expressions before leap seconds were introduced */
 NERA1 =n_elements(drift) / 2;

;/* Dates and Delta(AT)s */
;static const struct {
;int iyear, month;
;double delat;
changes_m = [ $
[ 1960d,  1d,  1.4178180d ],$
[ 1961,  1,  1.4228180d ],$
[ 1961,  8,  1.3728180d ],$
[ 1962,  1,  1.8458580d ],$
[ 1963, 11,  1.9458580d ],$
[ 1964,  1,  3.2401300d ],$
[ 1964,  4,  3.3401300d ],$
[ 1964,  9,  3.4401300d ],$
[ 1965,  1,  3.5401300d ],$
[ 1965,  3,  3.6401300d ],$
[ 1965,  7,  3.7401300d ],$
[ 1965,  9,  3.8401300d ],$
[ 1966,  1,  4.3131700d ],$
[ 1968,  2,  4.2131700d ],$
[ 1972,  1, 10.0d       ],$
[ 1972,  7, 11.0d       ],$
[ 1973,  1, 12.0d       ],$
[ 1974,  1, 13.0d       ],$
[ 1975,  1, 14.0d       ],$
[ 1976,  1, 15.0d       ],$
[ 1977,  1, 16.0d       ],$
[ 1978,  1, 17.0d       ],$
[ 1979,  1, 18.0d       ],$
[ 1980,  1, 19.0d       ],$
[ 1981,  7, 20.0d       ],$
[ 1982,  7, 21.0d       ],$
[ 1983,  7, 22.0d       ],$
[ 1985,  7, 23.0d       ],$
[ 1988,  1, 24.0d       ],$
[ 1990,  1, 25.0d       ],$
[ 1991,  1, 26.0d       ],$
[ 1992,  7, 27.0d       ],$
[ 1993,  7, 28.0d       ],$
[ 1994,  7, 29.0d       ],$
[ 1996,  1, 30.0d       ],$
[ 1997,  7, 31.0d       ],$
[ 1999,  1, 32.0d       ],$
[ 2006,  1, 33.0d       ],$
[ 2009,  1, 34.0d       ],$
[ 2012,  7, 35.0d       ],$
[ 2015,  7, 36.0d       ]$
];

changes={strc, iyear:changes_m[0,*], month:changes_m[1,*],delat:changes_m[2,*]}

;/* Number of Delta(AT) changes */
NDAT = n_elements(changes_m[0,*]); / n_elements(changes_m[0,*]);

;/* Miscellaneous local variables */
j=0 & i=0 & m=0;
da=0d & djm0=0d & djm=0d;


;/* Initialize the result to zero. */
deltat = 0.0d
da = 0.0d;

;/* If invalid fraction of a day, set error status and give up. */
if (fd lt 0.0 or fd gt 1.0) then return, -4;

;/* Convert the date into an MJD. */
j = eraCal2jd(iy, im, id, djm0, djm);

;/* If invalid year, month, or day, give up. */
if (j lt 0) then return, j;

;/* If pre-UTC year, set warning status and give up. */
if (iy lt changes.iyear[0]) then return, 1;

;/* If suspiciously late year, set warning status but proceed. */
if (iy gt IYV + 5) then j = 1;

;/* Combine year and month to form a date-ordered integer... */
m = 12l*iy + im;

;/* ...and use it to find the preceding table entry. */
for i = NDAT-1,0,-1 do begin 
  if (m ge (12d * changes.iyear[i] + changes.month[i])) then break;
endfor

;/* Prevent underflow warnings. */
if (i lt 0) then return, -5;

;/* Get the Delta(AT). */
da = changes.delat[i];

;/* If pre-1972, adjust for drift. */
if (i lt NERA1) then da += (djm + fd - drift[i,0]) * drift[i,1];

;/* Return the Delta(AT) value. */
deltat = da;

;/* Return the status. */
return, j;


;/*----------------------------------------------------------------------




end