function eraEpv00,date1,date2,pvh,pvb
;  int eraEpv00(double date1, double date2,
;  double pvh[2][3], double pvb[2][3])
;  /*
;  **  - - - - - - - - -
;  **   e r a E p v 0 0
;  **  - - - - - - - - -
;  **
;  **  Earth position and velocity, heliocentric and barycentric, with
;  **  respect to the Barycentric Celestial Reference System.
;  **
;  **  Given:
;  **     date1,date2  double        TDB date (Note 1)
;  **
;  **  Returned:
;  **     pvh          double[2][3]  heliocentric Earth position/velocity
;  **     pvb          double[2][3]  barycentric Earth position/velocity
;  **
;  **  Returned (function value):
;  **                  int           status: 0 = OK
;  **                                       +1 = warning: date outside
;  **                                            the range 1900-2100 AD
;  **
;  **  Notes:
;  **
;  **  1) The TDB date date1+date2 is a Julian Date, apportioned in any
;  **     convenient way between the two arguments.  For example,
;  **     JD(TDB)=2450123.7 could be expressed in any of these ways, among
;  **     others:
;  **
;  **            date1          date2
;  **
;  **         2450123.7           0.0       (JD method)
;  **         2451545.0       -1421.3       (J2000 method)
;  **         2400000.5       50123.2       (MJD method)
;  **         2450123.5           0.2       (date & time method)
;  **
;  **     The JD method is the most natural and convenient to use in cases
;  **     where the loss of several decimal digits of resolution is
;  **     acceptable.  The J2000 method is best matched to the way the
;  **     argument is handled internally and will deliver the optimum
;  **     resolution.  The MJD method and the date & time methods are both
;  **     good compromises between resolution and convenience.  However,
;  **     the accuracy of the result is more likely to be limited by the
;  **     algorithm itself than the way the date has been expressed.
;  **
;  **     n.b. TT can be used instead of TDB in most applications.
;  **
;  **  2) On return, the arrays pvh and pvb contain the following:
;  **
;  **        pvh[0][0]  x       }
;  **        pvh[0][1]  y       } heliocentric position, AU
;  **        pvh[0][2]  z       }
;  **
;  **        pvh[1][0]  xdot    }
;  **        pvh[1][1]  ydot    } heliocentric velocity, AU/d
;  **        pvh[1][2]  zdot    }
;  **
;  **        pvb[0][0]  x       }
;  **        pvb[0][1]  y       } barycentric position, AU
;  **        pvb[0][2]  z       }
;  **
;  **        pvb[1][0]  xdot    }
;  **        pvb[1][1]  ydot    } barycentric velocity, AU/d
;  **        pvb[1][2]  zdot    }
;  **
;  **     The vectors are with respect to the Barycentric Celestial
;  **     Reference System.  The time unit is one day in TDB.
;  **
;  **  3) The function is a SIMPLIFIED SOLUTION from the planetary theory
;  **     VSOP2000 (X. Moisson, P. Bretagnon, 2001, Celes. Mechanics &
;  **     Dyn. Astron., 80, 3/4, 205-213) and is an adaptation of original
;  **     Fortran code supplied by P. Bretagnon (private comm., 2000).
;  **
;  **  4) Comparisons over the time span 1900-2100 with this simplified
;  **     solution and the JPL DE405 ephemeris give the following results:
;  **
;  **                                RMS    max
;  **           Heliocentric:
;  **              position error    3.7   11.2   km
;  **              velocity error    1.4    5.0   mm/s
;  **
;  **           Barycentric:
;  **              position error    4.6   13.4   km
;  **              velocity error    1.4    4.9   mm/s
;  **
;  **     Comparisons with the JPL DE406 ephemeris show that by 1800 and
;  **     2200 the position errors are approximately double their 1900-2100
;  **     size.  By 1500 and 2500 the deterioration is a factor of 10 and
;  **     by 1000 and 3000 a factor of 60.  The velocity accuracy falls off
;  **     at about half that rate.
;  **
;  **  5) It is permissible to use the same array for pvh and pvb, which
;  **     will receive the barycentric values.
;  **
;  **  Copyright (C) 2013-2015, NumFOCUS Foundation.
;  **  Derived, with permission, from the SOFA library.  See notes at end of file.
;  */
;  {
;  /*
;  ** Matrix elements for orienting the analytical model to DE405.
;  **
;  ** The corresponding Euler angles are:
;  **
;  **                       d  '  "
;  **   1st rotation    -  23 26 21.4091 about the x-axis  (obliquity)
;  **   2nd rotation    +         0.0475 about the z-axis  (RA offset)
;  **
;  ** These were obtained empirically, by comparisons with DE405 over
;  ** 1900-2100.
;  */
  am12 =  0.000000211284d
  am13 = -0.000000091603d
  am21 = -0.000000230286d
  am22 =  0.917482137087d
  am23 = -0.397776982902d
  am32 =  0.397776982902d
  am33 =  0.917482137087d;
  
;  /*
;  ** ----------------------
;  ** Ephemeris Coefficients
;  ** ----------------------
;  **
;  ** The ephemeris consists of harmonic terms for predicting (i) the Sun
;  ** to Earth vector and (ii) the Solar-System-barycenter to Sun vector
;  ** respectively.  The coefficients are stored in arrays which, although
;  ** 1-demensional, contain groups of three.  Each triplet of
;  ** coefficients is the amplitude, phase and frequency for one term in
;  ** the model, and each array contains the number of terms called for by
;  ** the model.
;  **
;  ** There are eighteen such arrays, named as follows:
;  **
;  **     array         model      power of T      component
;  **
;  **      e0x      Sun-to-Earth        0              x
;  **      e0y      Sun-to-Earth        0              y
;  **      e0z      Sun-to-Earth        0              z
;  **
;  **      e1x      Sun-to-Earth        1              x
;  **      e1y      Sun-to-Earth        1              y
;  **      e1z      Sun-to-Earth        1              z
;  **
;  **      e2x      Sun-to-Earth        2              x
;  **      e2y      Sun-to-Earth        2              y
;  **      e2z      Sun-to-Earth        2              z
;  **
;  **      s0x      SSB-to-Sun          0              x
;  **      s0y      SSB-to-Sun          0              y
;  **      s0z      SSB-to-Sun          0              z
;  **
;  **      s1x      SSB-to-Sun          1              x
;  **      s1y      SSB-to-Sun          1              y
;  **      s1z      SSB-to-Sun          1              z
;  **
;  **      s2x      SSB-to-Sun          2              x
;  **      s2y      SSB-to-Sun          2              y
;  **      s2z      SSB-to-Sun          2              z
;  */
  
;  /* Sun-to-Earth, T^0, X */
e0x = [ $
  0.9998292878132d+00, 0.1753485171504d+01, 0.6283075850446d+01,$
  0.8352579567414d-02, 0.1710344404582d+01, 0.1256615170089d+02,$
  0.5611445335148d-02, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.1046664295572d-03, 0.1667225416770d+01, 0.1884922755134d+02,$
  0.3110842534677d-04, 0.6687513390251d+00, 0.8399684731857d+02,$
  0.2552413503550d-04, 0.5830637358413d+00, 0.5296909721118d+00,$
  0.2137207845781d-04, 0.1092330954011d+01, 0.1577343543434d+01,$
  0.1680240182951d-04, 0.4955366134987d+00, 0.6279552690824d+01,$
  0.1679012370795d-04, 0.6153014091901d+01, 0.6286599010068d+01,$
  0.1445526946777d-04, 0.3472744100492d+01, 0.2352866153506d+01,$
  $
  0.1091038246184d-04, 0.3689845786119d+01, 0.5223693906222d+01,$
  0.9344399733932d-05, 0.6073934645672d+01, 0.1203646072878d+02,$
  0.8993182910652d-05, 0.3175705249069d+01, 0.1021328554739d+02,$
  0.5665546034116d-05, 0.2152484672246d+01, 0.1059381944224d+01,$
  0.6844146703035d-05, 0.1306964099750d+01, 0.5753384878334d+01,$
  0.7346610905565d-05, 0.4354980070466d+01, 0.3981490189893d+00,$
  0.6815396474414d-05, 0.2218229211267d+01, 0.4705732307012d+01,$
  0.6112787253053d-05, 0.5384788425458d+01, 0.6812766822558d+01,$
  0.4518120711239d-05, 0.6087604012291d+01, 0.5884926831456d+01,$
  0.4521963430706d-05, 0.1279424524906d+01, 0.6256777527156d+01,$
  $
  0.4497426764085d-05, 0.5369129144266d+01, 0.6309374173736d+01,$
  0.4062190566959d-05, 0.5436473303367d+00, 0.6681224869435d+01,$
  0.5412193480192d-05, 0.7867838528395d+00, 0.7755226100720d+00,$
  0.5469839049386d-05, 0.1461440311134d+01, 0.1414349524433d+02,$
  0.5205264083477d-05, 0.4432944696116d+01, 0.7860419393880d+01,$
  0.2149759935455d-05, 0.4502237496846d+01, 0.1150676975667d+02,$
  0.2279109618501d-05, 0.1239441308815d+01, 0.7058598460518d+01,$
  0.2259282939683d-05, 0.3272430985331d+01, 0.4694002934110d+01,$
  0.2558950271319d-05, 0.2265471086404d+01, 0.1216800268190d+02,$
  0.2561581447555d-05, 0.1454740653245d+01, 0.7099330490126d+00,$
  $
  0.1781441115440d-05, 0.2962068630206d+01, 0.7962980379786d+00,$
  0.1612005874644d-05, 0.1473255041006d+01, 0.5486777812467d+01,$
  0.1818630667105d-05, 0.3743903293447d+00, 0.6283008715021d+01,$
  0.1818601377529d-05, 0.6274174354554d+01, 0.6283142985870d+01,$
  0.1554475925257d-05, 0.1624110906816d+01, 0.2513230340178d+02,$
  0.2090948029241d-05, 0.5852052276256d+01, 0.1179062909082d+02,$
  0.2000176345460d-05, 0.4072093298513d+01, 0.1778984560711d+02,$
  0.1289535917759d-05, 0.5217019331069d+01, 0.7079373888424d+01,$
  0.1281135307881d-05, 0.4802054538934d+01, 0.3738761453707d+01,$
  0.1518229005692d-05, 0.8691914742502d+00, 0.2132990797783d+00,$
  $
  0.9450128579027d-06, 0.4601859529950d+01, 0.1097707878456d+02,$
  0.7781119494996d-06, 0.1844352816694d+01, 0.8827390247185d+01,$
  0.7733407759912d-06, 0.3582790154750d+01, 0.5507553240374d+01,$
  0.7350644318120d-06, 0.2695277788230d+01, 0.1589072916335d+01,$
  0.6535928827023d-06, 0.3651327986142d+01, 0.1176985366291d+02,$
  0.6324624183656d-06, 0.2241302375862d+01, 0.6262300422539d+01,$
  0.6298565300557d-06, 0.4407122406081d+01, 0.6303851278352d+01,$
  0.8587037089179d-06, 0.3024307223119d+01, 0.1672837615881d+03,$
  0.8299954491035d-06, 0.6192539428237d+01, 0.3340612434717d+01,$
  0.6311263503401d-06, 0.2014758795416d+01, 0.7113454667900d-02,$
  
  0.6005646745452d-06, 0.3399500503397d+01, 0.4136910472696d+01,$
  0.7917715109929d-06, 0.2493386877837d+01, 0.6069776770667d+01,$
  0.7556958099685d-06, 0.4159491740143d+01, 0.6496374930224d+01,$
  0.6773228244949d-06, 0.4034162934230d+01, 0.9437762937313d+01,$
  0.5370708577847d-06, 0.1562219163734d+01, 0.1194447056968d+01,$
  0.5710804266203d-06, 0.2662730803386d+01, 0.6282095334605d+01,$
  0.5709824583726d-06, 0.3985828430833d+01, 0.6284056366286d+01,$
  0.5143950896447d-06, 0.1308144688689d+01, 0.6290189305114d+01,$
  0.5088010604546d-06, 0.5352817214804d+01, 0.6275962395778d+01,$
  0.4960369085172d-06, 0.2644267922349d+01, 0.6127655567643d+01,$
  
  0.4803137891183d-06, 0.4008844192080d+01, 0.6438496133249d+01,$
  0.5731747768225d-06, 0.3794550174597d+01, 0.3154687086868d+01,$
  0.4735947960579d-06, 0.6107118308982d+01, 0.3128388763578d+01,$
  0.4808348796625d-06, 0.4771458618163d+01, 0.8018209333619d+00,$
  0.4115073743137d-06, 0.3327111335159d+01, 0.8429241228195d+01,$
  0.5230575889287d-06, 0.5305708551694d+01, 0.1336797263425d+02,$
  0.5133977889215d-06, 0.5784230738814d+01, 0.1235285262111d+02,$
  0.5065815825327d-06, 0.2052064793679d+01, 0.1185621865188d+02,$
  0.4339831593868d-06, 0.3644994195830d+01, 0.1726015463500d+02,$
  0.3952928638953d-06, 0.4930376436758d+01, 0.5481254917084d+01,$
  $
  0.4898498111942d-06, 0.4542084219731d+00, 0.9225539266174d+01,$
  0.4757490209328d-06, 0.3161126388878d+01, 0.5856477690889d+01,$
  0.4727701669749d-06, 0.6214993845446d+00, 0.2544314396739d+01,$
  0.3800966681863d-06, 0.3040132339297d+01, 0.4265981595566d+00,$
  0.3257301077939d-06, 0.8064977360087d+00, 0.3930209696940d+01,$
  0.3255810528674d-06, 0.1974147981034d+01, 0.2146165377750d+01,$
  0.3252029748187d-06, 0.2845924913135d+01, 0.4164311961999d+01,$
  0.3255505635308d-06, 0.3017900824120d+01, 0.5088628793478d+01,$
  0.2801345211990d-06, 0.6109717793179d+01, 0.1256967486051d+02,$
  0.3688987740970d-06, 0.2911550235289d+01, 0.1807370494127d+02,$
  $
  0.2475153429458d-06, 0.2179146025856d+01, 0.2629832328990d-01,$
  0.3033457749150d-06, 0.1994161050744d+01, 0.4535059491685d+01,$
  0.2186743763110d-06, 0.5125687237936d+01, 0.1137170464392d+02,$
  0.2764777032774d-06, 0.4822646860252d+00, 0.1256262854127d+02,$
  0.2199028768592d-06, 0.4637633293831d+01, 0.1255903824622d+02,$
  0.2046482824760d-06, 0.1467038733093d+01, 0.7084896783808d+01,$
  0.2611209147507d-06, 0.3044718783485d+00, 0.7143069561767d+02,$
  0.2286079656818d-06, 0.4764220356805d+01, 0.8031092209206d+01,$
  0.1855071202587d-06, 0.3383637774428d+01, 0.1748016358760d+01,$
  0.2324669506784d-06, 0.6189088449251d+01, 0.1831953657923d+02,$
  $
  0.1709528015688d-06, 0.5874966729774d+00, 0.4933208510675d+01,$
  0.2168156875828d-06, 0.4302994009132d+01, 0.1044738781244d+02,$
  0.2106675556535d-06, 0.3800475419891d+01, 0.7477522907414d+01,$
  0.1430213830465d-06, 0.1294660846502d+01, 0.2942463415728d+01,$
  0.1388396901944d-06, 0.4594797202114d+01, 0.8635942003952d+01,$
  0.1922258844190d-06, 0.4943044543591d+00, 0.1729818233119d+02,$
  0.1888460058292d-06, 0.2426943912028d+01, 0.1561374759853d+03,$
  0.1789449386107d-06, 0.1582973303499d+00, 0.1592596075957d+01,$
  0.1360803685374d-06, 0.5197240440504d+01, 0.1309584267300d+02,$
  0.1504038014709d-06, 0.3120360916217d+01, 0.1649636139783d+02,$
  $
  0.1382769533389d-06, 0.6164702888205d+01, 0.7632943190217d+01,$
  0.1438059769079d-06, 0.1437423770979d+01, 0.2042657109477d+02,$
  0.1326303260037d-06, 0.3609688799679d+01, 0.1213955354133d+02,$
  0.1159244950540d-06, 0.5463018167225d+01, 0.5331357529664d+01,$
  0.1433118149136d-06, 0.6028909912097d+01, 0.7342457794669d+01,$
  0.1234623148594d-06, 0.3109645574997d+01, 0.6279485555400d+01,$
  0.1233949875344d-06, 0.3539359332866d+01, 0.6286666145492d+01,$
  0.9927196061299d-07, 0.1259321569772d+01, 0.7234794171227d+01,$
  0.1242302191316d-06, 0.1065949392609d+01, 0.1511046609763d+02,$
  0.1098402195201d-06, 0.2192508743837d+01, 0.1098880815746d+02,$
  $
  0.1158191395315d-06, 0.4054411278650d+01, 0.5729506548653d+01,$
  0.9048475596241d-07, 0.5429764748518d+01, 0.9623688285163d+01,$
  0.8889853269023d-07, 0.5046586206575d+01, 0.6148010737701d+01,$
  0.1048694242164d-06, 0.2628858030806d+01, 0.6836645152238d+01,$
  0.1112308378646d-06, 0.4177292719907d+01, 0.1572083878776d+02,$
  0.8631729709901d-07, 0.1601345232557d+01, 0.6418140963190d+01,$
  0.8527816951664d-07, 0.2463888997513d+01, 0.1471231707864d+02,$
  0.7892139456991d-07, 0.3154022088718d+01, 0.2118763888447d+01,$
  0.1051782905236d-06, 0.4795035816088d+01, 0.1349867339771d+01,$
  0.1048219943164d-06, 0.2952983395230d+01, 0.5999216516294d+01,$
  $
  0.7435760775143d-07, 0.5420547991464d+01, 0.6040347114260d+01,$
  0.9869574106949d-07, 0.3695646753667d+01, 0.6566935184597d+01,$
  0.9156886364226d-07, 0.3922675306609d+01, 0.5643178611111d+01,$
  0.7006834356188d-07, 0.1233968624861d+01, 0.6525804586632d+01,$
  0.9806170182601d-07, 0.1919542280684d+01, 0.2122839202813d+02,$
  0.9052289673607d-07, 0.4615902724369d+01, 0.4690479774488d+01,$
  0.7554200867893d-07, 0.1236863719072d+01, 0.1253985337760d+02,$
  0.8215741286498d-07, 0.3286800101559d+00, 0.1097355562493d+02,$
  0.7185178575397d-07, 0.5880942158367d+01, 0.6245048154254d+01,$
  0.7130726476180d-07, 0.7674871987661d+00, 0.6321103546637d+01,$
  $
  0.6650894461162d-07, 0.6987129150116d+00, 0.5327476111629d+01,$
  0.7396888823688d-07, 0.3576824794443d+01, 0.5368044267797d+00,$
  0.7420588884775d-07, 0.5033615245369d+01, 0.2354323048545d+02,$
  0.6141181642908d-07, 0.9449927045673d+00, 0.1296430071988d+02,$
  0.6373557924058d-07, 0.6206342280341d+01, 0.9517183207817d+00,$
  0.6359474329261d-07, 0.5036079095757d+01, 0.1990745094947d+01,$
  0.5740173582646d-07, 0.6105106371350d+01, 0.9555997388169d+00,$
  0.7019864084602d-07, 0.7237747359018d+00, 0.5225775174439d+00,$
  0.6398054487042d-07, 0.3976367969666d+01, 0.2407292145756d+02,$
  0.7797092650498d-07, 0.4305423910623d+01, 0.2200391463820d+02,$
  $
  0.6466760000900d-07, 0.3500136825200d+01, 0.5230807360890d+01,$
  0.7529417043890d-07, 0.3514779246100d+01, 0.1842262939178d+02,$
  0.6924571140892d-07, 0.2743457928679d+01, 0.1554202828031d+00,$
  0.6220798650222d-07, 0.2242598118209d+01, 0.1845107853235d+02,$
  0.5870209391853d-07, 0.2332832707527d+01, 0.6398972393349d+00,$
  0.6263953473888d-07, 0.2191105358956d+01, 0.6277552955062d+01,$
  0.6257781390012d-07, 0.4457559396698d+01, 0.6288598745829d+01,$
  0.5697304945123d-07, 0.3499234761404d+01, 0.1551045220144d+01,$
  0.6335438746791d-07, 0.6441691079251d+00, 0.5216580451554d+01,$
  0.6377258441152d-07, 0.2252599151092d+01, 0.5650292065779d+01,$
  $
  0.6484841818165d-07, 0.1992812417646d+01, 0.1030928125552d+00,$
  0.4735551485250d-07, 0.3744672082942d+01, 0.1431416805965d+02,$
  0.4628595996170d-07, 0.1334226211745d+01, 0.5535693017924d+00,$
  0.6258152336933d-07, 0.4395836159154d+01, 0.2608790314060d+02,$
  0.6196171366594d-07, 0.2587043007997d+01, 0.8467247584405d+02,$
  0.6159556952126d-07, 0.4782499769128d+01, 0.2394243902548d+03,$
  0.4987741172394d-07, 0.7312257619924d+00, 0.7771377146812d+02,$
  0.5459280703142d-07, 0.3001376372532d+01, 0.6179983037890d+01,$
  0.4863461189999d-07, 0.3767222128541d+01, 0.9027992316901d+02,$
  0.5349912093158d-07, 0.3663594450273d+01, 0.6386168663001d+01,$
  $
  0.5673725607806d-07, 0.4331187919049d+01, 0.6915859635113d+01,$
  0.4745485060512d-07, 0.5816195745518d+01, 0.6282970628506d+01,$
  0.4745379005326d-07, 0.8323672435672d+00, 0.6283181072386d+01,$
  0.4049002796321d-07, 0.3785023976293d+01, 0.6254626709878d+01,$
  0.4247084014515d-07, 0.2378220728783d+01, 0.7875671926403d+01,$
  0.4026912363055d-07, 0.2864103423269d+01, 0.6311524991013d+01,$
  0.4062935011774d-07, 0.2415408595975d+01, 0.3634620989887d+01,$
  0.5347771048509d-07, 0.3343479309801d+01, 0.2515860172507d+02,$
  0.4829494136505d-07, 0.2821742398262d+01, 0.5760498333002d+01,$
  0.4342554404599d-07, 0.5624662458712d+01, 0.7238675589263d+01,$
  $
  0.4021599184361d-07, 0.5557250275009d+00, 0.1101510648075d+02,$
  0.4104900474558d-07, 0.3296691780005d+01, 0.6709674010002d+01,$
  0.4376532905131d-07, 0.3814443999443d+01, 0.6805653367890d+01,$
  0.3314590480650d-07, 0.3560229189250d+01, 0.1259245002418d+02,$
  0.3232421839643d-07, 0.5185389180568d+01, 0.1066495398892d+01,$
  0.3541176318876d-07, 0.3921381909679d+01, 0.9917696840332d+01,$
  0.3689831242681d-07, 0.4190658955386d+01, 0.1192625446156d+02,$
  0.3890605376774d-07, 0.5546023371097d+01, 0.7478166569050d-01,$
  0.3038559339780d-07, 0.6231032794494d+01, 0.1256621883632d+02,$
  0.3137083969782d-07, 0.6207063419190d+01, 0.4292330755499d+01,$
  $
  0.4024004081854d-07, 0.1195257375713d+01, 0.1334167431096d+02,$
  0.3300234879283d-07, 0.1804694240998d+01, 0.1057540660594d+02,$
  0.3635399155575d-07, 0.5597811343500d+01, 0.6208294184755d+01,$
  0.3032668691356d-07, 0.3191059366530d+01, 0.1805292951336d+02,$
  0.2809652069058d-07, 0.4094348032570d+01, 0.3523159621801d-02,$
  0.3696955383823d-07, 0.5219282738794d+01, 0.5966683958112d+01,$
  0.3562894142503d-07, 0.1037247544554d+01, 0.6357857516136d+01,$
  0.3510598524148d-07, 0.1430020816116d+01, 0.6599467742779d+01,$
  0.3617736142953d-07, 0.3002911403677d+01, 0.6019991944201d+01,$
  0.2624524910730d-07, 0.2437046757292d+01, 0.6702560555334d+01,$
  $
  0.2535824204490d-07, 0.1581594689647d+01, 0.3141537925223d+02,$
  0.3519787226257d-07, 0.5379863121521d+01, 0.2505706758577d+03,$
  0.2578406709982d-07, 0.4904222639329d+01, 0.1673046366289d+02,$
  0.3423887981473d-07, 0.3646448997315d+01, 0.6546159756691d+01,$
  0.2776083886467d-07, 0.3307829300144d+01, 0.1272157198369d+02,$
  0.3379592818379d-07, 0.1747541251125d+01, 0.1494531617769d+02,$
  0.3050255426284d-07, 0.1784689432607d-01, 0.4732030630302d+01,$
  0.2652378350236d-07, 0.4420055276260d+01, 0.5863591145557d+01,$
  0.2374498173768d-07, 0.3629773929208d+01, 0.2388894113936d+01,$
  0.2716451255140d-07, 0.3079623706780d+01, 0.1202934727411d+02,$
  $
  0.3038583699229d-07, 0.3312487903507d+00, 0.1256608456547d+02,$
  0.2220681228760d-07, 0.5265520401774d+01, 0.1336244973887d+02,$
  0.3044156540912d-07, 0.4766664081250d+01, 0.2908881142201d+02,$
  0.2731859923561d-07, 0.5069146530691d+01, 0.1391601904066d+02,$
  0.2285603018171d-07, 0.5954935112271d+01, 0.6076890225335d+01,$
  0.2025006454555d-07, 0.4061789589267d+01, 0.4701116388778d+01,$
  0.2012597519804d-07, 0.2485047705241d+01, 0.6262720680387d+01,$
  0.2003406962258d-07, 0.4163779209320d+01, 0.6303431020504d+01,$
  0.2207863441371d-07, 0.6923839133828d+00, 0.6489261475556d+01,$
  0.2481374305624d-07, 0.5944173595676d+01, 0.1204357418345d+02,$
  $
  0.2130923288870d-07, 0.4641013671967d+01, 0.5746271423666d+01,$
  0.2446370543391d-07, 0.6125796518757d+01, 0.1495633313810d+00,$
  0.1932492759052d-07, 0.2234572324504d+00, 0.1352175143971d+02,$
  0.2600122568049d-07, 0.4281012405440d+01, 0.4590910121555d+01,$
  0.2431754047488d-07, 0.1429943874870d+00, 0.1162474756779d+01,$
  0.1875902869209d-07, 0.9781803816948d+00, 0.6279194432410d+01,$
  0.1874381139426d-07, 0.5670368130173d+01, 0.6286957268481d+01,$
  0.2156696047173d-07, 0.2008985006833d+01, 0.1813929450232d+02,$
  0.1965076182484d-07, 0.2566186202453d+00, 0.4686889479442d+01,$
  0.2334816372359d-07, 0.4408121891493d+01, 0.1002183730415d+02,$
  $
  0.1869937408802d-07, 0.5272745038656d+01, 0.2427287361862d+00,$
  0.2436236460883d-07, 0.4407720479029d+01, 0.9514313292143d+02,$
  0.1761365216611d-07, 0.1943892315074d+00, 0.1351787002167d+02,$
  0.2156289480503d-07, 0.1418570924545d+01, 0.6037244212485d+01,$
  0.2164748979255d-07, 0.4724603439430d+01, 0.2301353951334d+02,$
  0.2222286670853d-07, 0.2400266874598d+01, 0.1266924451345d+02,$
  0.2070901414929d-07, 0.5230348028732d+01, 0.6528907488406d+01,$
  0.1792745177020d-07, 0.2099190328945d+01, 0.6819880277225d+01,$
  0.1841802068445d-07, 0.3467527844848d+00, 0.6514761976723d+02,$
  0.1578401631718d-07, 0.7098642356340d+00, 0.2077542790660d-01,$
  $
  0.1561690152531d-07, 0.5943349620372d+01, 0.6272439236156d+01,$
  0.1558591045463d-07, 0.7040653478980d+00, 0.6293712464735d+01,$
  0.1737356469576d-07, 0.4487064760345d+01, 0.1765478049437d+02,$
  0.1434755619991d-07, 0.2993391570995d+01, 0.1102062672231d+00,$
  0.1482187806654d-07, 0.2278049198251d+01, 0.1052268489556d+01,$
  0.1424812827089d-07, 0.1682114725827d+01, 0.1311972100268d+02,$
  0.1380282448623d-07, 0.3262668602579d+01, 0.1017725758696d+02,$
  0.1811481244566d-07, 0.3187771221777d+01, 0.1887552587463d+02,$
  0.1504446185696d-07, 0.5650162308647d+01, 0.7626583626240d-01,$
  0.1740776154137d-07, 0.5487068607507d+01, 0.1965104848470d+02,$
  $
  0.1374339536251d-07, 0.5745688172201d+01, 0.6016468784579d+01,$
  0.1761377477704d-07, 0.5748060203659d+01, 0.2593412433514d+02,$
  0.1535138225795d-07, 0.6226848505790d+01, 0.9411464614024d+01,$
  0.1788140543676d-07, 0.6189318878563d+01, 0.3301902111895d+02,$
  0.1375002807996d-07, 0.5371812884394d+01, 0.6327837846670d+00,$
  0.1242115758632d-07, 0.1471687569712d+01, 0.3894181736510d+01,$
  0.1450977333938d-07, 0.4143836662127d+01, 0.1277945078067d+02,$
  0.1297579575023d-07, 0.9003477661957d+00, 0.6549682916313d+01,$
  0.1462667934821d-07, 0.5760505536428d+01, 0.1863592847156d+02,$
  0.1381774374799d-07, 0.1085471729463d+01, 0.2379164476796d+01,$
  $
  0.1682333169307d-07, 0.5409870870133d+01, 0.1620077269078d+02,$
  0.1190812918837d-07, 0.1397205174601d+01, 0.1149965630200d+02,$
  0.1221434762106d-07, 0.9001804809095d+00, 0.1257326515556d+02,$
  0.1549934644860d-07, 0.4262528275544d+01, 0.1820933031200d+02,$
  0.1252138953050d-07, 0.1411642012027d+01, 0.6993008899458d+01,$
  0.1237078905387d-07, 0.2844472403615d+01, 0.2435678079171d+02,$
  0.1446953389615d-07, 0.5295835522223d+01, 0.3813291813120d-01,$
  0.1388446457170d-07, 0.4969428135497d+01, 0.2458316379602d+00,$
  0.1019339179228d-07, 0.2491369561806d+01, 0.6112403035119d+01,$
  0.1258880815343d-07, 0.4679426248976d+01, 0.5429879531333d+01,$
  $
  0.1297768238261d-07, 0.1074509953328d+01, 0.1249137003520d+02,$
  0.9913505718094d-08, 0.4735097918224d+01, 0.6247047890016d+01,$
  0.9830453155969d-08, 0.4158649187338d+01, 0.6453748665772d+01,$
  0.1192615865309d-07, 0.3438208613699d+01, 0.6290122169689d+01,$
  0.9835874798277d-08, 0.1913300781229d+01, 0.6319103810876d+01,$
  0.9639087569277d-08, 0.9487683644125d+00, 0.8273820945392d+01,$
  0.1175716107001d-07, 0.3228141664287d+01, 0.6276029531202d+01,$
  0.1018926508678d-07, 0.2216607854300d+01, 0.1254537627298d+02,$
  0.9500087869225d-08, 0.2625116459733d+01, 0.1256517118505d+02,$
  0.9664192916575d-08, 0.5860562449214d+01, 0.6259197520765d+01,$
  $
  0.9612858712203d-08, 0.7885682917381d+00, 0.6306954180126d+01,$
  0.1117645675413d-07, 0.3932148831189d+01, 0.1779695906178d+02,$
  0.1158864052160d-07, 0.9995605521691d+00, 0.1778273215245d+02,$
  0.9021043467028d-08, 0.5263769742673d+01, 0.6172869583223d+01,$
  0.8836134773563d-08, 0.1496843220365d+01, 0.1692165728891d+01,$
  0.1045872200691d-07, 0.7009039517214d+00, 0.2204125344462d+00,$
  0.1211463487798d-07, 0.4041544938511d+01, 0.8257698122054d+02,$
  0.8541990804094d-08, 0.1447586692316d+01, 0.6393282117669d+01,$
  0.1038720703636d-07, 0.4594249718112d+00, 0.1550861511662d+02,$
  0.1126722351445d-07, 0.3925550579036d+01, 0.2061856251104d+00,$
  $
  0.8697373859631d-08, 0.4411341856037d+01, 0.9491756770005d+00,$
  0.8869380028441d-08, 0.2402659724813d+01, 0.3903911373650d+01,$
  0.9247014693258d-08, 0.1401579743423d+01, 0.6267823317922d+01,$
  0.9205062930950d-08, 0.5245978000814d+01, 0.6298328382969d+01,$
  0.8000745038049d-08, 0.3590803356945d+01, 0.2648454860559d+01,$
  0.9168973650819d-08, 0.2470150501679d+01, 0.1498544001348d+03,$
  0.1075444949238d-07, 0.1328606161230d+01, 0.3694923081589d+02,$
  0.7817298525817d-08, 0.6162256225998d+01, 0.4804209201333d+01,$
  0.9541469226356d-08, 0.3942568967039d+01, 0.1256713221673d+02,$
  0.9821910122027d-08, 0.2360246287233d+00, 0.1140367694411d+02,$
  $
  0.9897822023777d-08, 0.4619805634280d+01, 0.2280573557157d+02,$
  0.7737289283765d-08, 0.3784727847451d+01, 0.7834121070590d+01,$
  0.9260204034710d-08, 0.2223352487601d+01, 0.2787043132925d+01,$
  0.7320252888486d-08, 0.1288694636874d+01, 0.6282655592598d+01,$
  0.7319785780946d-08, 0.5359869567774d+01, 0.6283496108294d+01,$
  0.7147219933778d-08, 0.5516616675856d+01, 0.1725663147538d+02,$
  0.7946502829878d-08, 0.2630459984567d+01, 0.1241073141809d+02,$
  0.9001711808932d-08, 0.2849815827227d+01, 0.6281591679874d+01,$
  0.8994041507257d-08, 0.3795244450750d+01, 0.6284560021018d+01,$
  0.8298582787358d-08, 0.5236413127363d+00, 0.1241658836951d+02,$
  $
  0.8526596520710d-08, 0.4794605424426d+01, 0.1098419223922d+02,$
  0.8209822103197d-08, 0.1578752370328d+01, 0.1096996532989d+02,$
  0.6357049861094d-08, 0.5708926113761d+01, 0.1596186371003d+01,$
  0.7370473179049d-08, 0.3842402530241d+01, 0.4061219149443d+01,$
  0.7232154664726d-08, 0.3067548981535d+01, 0.1610006857377d+03,$
  0.6328765494903d-08, 0.1313930030069d+01, 0.1193336791622d+02,$
  0.8030064908595d-08, 0.3488500408886d+01, 0.8460828644453d+00,$
  0.6275464259232d-08, 0.1532061626198d+01, 0.8531963191132d+00,$
  0.7051897446325d-08, 0.3285859929993d+01, 0.5849364236221d+01,$
  0.6161593705428d-08, 0.1477341999464d+01, 0.5573142801433d+01,$
  $
  0.7754683957278d-08, 0.1586118663096d+01, 0.8662240327241d+01,$
  0.5889928990701d-08, 0.1304887868803d+01, 0.1232342296471d+02,$
  0.5705756047075d-08, 0.4555333589350d+01, 0.1258692712880d+02,$
  0.5964178808332d-08, 0.3001762842062d+01, 0.5333900173445d+01,$
  0.6712446027467d-08, 0.4886780007595d+01, 0.1171295538178d+02,$
  0.5941809275464d-08, 0.4701509603824d+01, 0.9779108567966d+01,$
  0.5466993627395d-08, 0.4588357817278d+01, 0.1884211409667d+02,$
  0.6340512090980d-08, 0.1164543038893d+01, 0.5217580628120d+02,$
  0.6325505710045d-08, 0.3919171259645d+01, 0.1041998632314d+02,$
  0.6164789509685d-08, 0.2143828253542d+01, 0.6151533897323d+01,$
  $
  0.5263330812430d-08, 0.6066564434241d+01, 0.1885275071096d+02,$
  0.5597087780221d-08, 0.2926316429472d+01, 0.4337116142245d+00,$
  0.5396556236817d-08, 0.3244303591505d+01, 0.6286362197481d+01,$
  0.5396615148223d-08, 0.3404304703662d+01, 0.6279789503410d+01,$
  0.7091832443341d-08, 0.8532377803192d+00, 0.4907302013889d+01,$
  0.6572352589782d-08, 0.4901966774419d+01, 0.1176433076753d+02,$
  0.5960236060795d-08, 0.1874672315797d+01, 0.1422690933580d-01,$
  0.5125480043511d-08, 0.3735726064334d+01, 0.1245594543367d+02,$
  0.5928241866410d-08, 0.4502033899935d+01, 0.6414617803568d+01,$
  0.5249600357424d-08, 0.4372334799878d+01, 0.1151388321134d+02,$
  $
  0.6059171276087d-08, 0.2581617302908d+01, 0.6062663316000d+01,$
  0.5295235081662d-08, 0.2974811513158d+01, 0.3496032717521d+01,$
  0.5820561875933d-08, 0.1796073748244d+00, 0.2838593341516d+00,$
  0.4754696606440d-08, 0.1981998136973d+01, 0.3104930017775d+01,$
  0.6385053548955d-08, 0.2559174171605d+00, 0.6133512519065d+01,$
  0.6589828273941d-08, 0.2750967106776d+01, 0.4087944051283d+02,$
  0.5383376567189d-08, 0.6325947523578d+00, 0.2248384854122d+02,$
  0.5928941683538d-08, 0.1672304519067d+01, 0.1581959461667d+01,$
  0.4816060709794d-08, 0.3512566172575d+01, 0.9388005868221d+01,$
  0.6003381586512d-08, 0.5610932219189d+01, 0.5326786718777d+01,$
  $
  0.5504225393105d-08, 0.4037501131256d+01, 0.6503488384892d+01,$
  0.5353772620129d-08, 0.6122774968240d+01, 0.1735668374386d+03,$
  0.5786253768544d-08, 0.5527984999515d+01, 0.1350651127443d+00,$
  0.5065706702002d-08, 0.9980765573624d+00, 0.1248988586463d+02,$
  0.5972838885276d-08, 0.6044489493203d+01, 0.2673594526851d+02,$
  0.5323585877961d-08, 0.3924265998147d+01, 0.4171425416666d+01,$
  0.5210772682858d-08, 0.6220111376901d+01, 0.2460261242967d+02,$
  0.4726549040535d-08, 0.3716043206862d+01, 0.7232251527446d+01,$
  0.6029425105059d-08, 0.8548704071116d+00, 0.3227113045244d+03,$
  0.4481542826513d-08, 0.1426925072829d+01, 0.5547199253223d+01,$
  $
  0.5836024505068d-08, 0.7135651752625d-01, 0.7285056171570d+02,$
  0.4137046613272d-08, 0.5330767643283d+01, 0.1087398597200d+02,$
  0.5171977473924d-08, 0.4494262335353d+00, 0.1884570439172d+02,$
  0.5694429833732d-08, 0.2952369582215d+01, 0.9723862754494d+02,$
  0.4009158925298d-08, 0.3500003416535d+01, 0.6244942932314d+01,$
  0.4784939596873d-08, 0.6196709413181d+01, 0.2929661536378d+02,$
  0.3983725022610d-08, 0.5103690031897d+01, 0.4274518229222d+01,$
  0.3870535232462d-08, 0.3187569587401d+01, 0.6321208768577d+01,$
  0.5140501213951d-08, 0.1668924357457d+01, 0.1232032006293d+02,$
  0.3849034819355d-08, 0.4445722510309d+01, 0.1726726808967d+02,$
  $
  0.4002383075060d-08, 0.5226224152423d+01, 0.7018952447668d+01,$
  0.3890719543549d-08, 0.4371166550274d+01, 0.1491901785440d+02,$
  0.4887084607881d-08, 0.5973556689693d+01, 0.1478866649112d+01,$
  0.3739939287592d-08, 0.2089084714600d+01, 0.6922973089781d+01,$
  0.5031925918209d-08, 0.4658371936827d+01, 0.1715706182245d+02,$
  0.4387748764954d-08, 0.4825580552819d+01, 0.2331413144044d+03,$
  0.4147398098865d-08, 0.3739003524998d+01, 0.1376059875786d+02,$
  0.3719089993586d-08, 0.1148941386536d+01, 0.6297302759782d+01,$
  0.3934238461056d-08, 0.1559893008343d+01, 0.7872148766781d+01,$
  0.3672471375622d-08, 0.5516145383612d+01, 0.6268848941110d+01,$
  $
  0.3768911277583d-08, 0.6116053700563d+01, 0.4157198507331d+01,$
  0.4033388417295d-08, 0.5076821746017d+01, 0.1567108171867d+02,$
  0.3764194617832d-08, 0.8164676232075d+00, 0.3185192151914d+01,$
  0.4840628226284d-08, 0.1360479453671d+01, 0.1252801878276d+02,$
  0.4949443923785d-08, 0.2725622229926d+01, 0.1617106187867d+03,$
  0.4117393089971d-08, 0.6054459628492d+00, 0.5642198095270d+01,$
  0.3925754020428d-08, 0.8570462135210d+00, 0.2139354194808d+02,$
  0.3630551757923d-08, 0.3552067338279d+01, 0.6294805223347d+01,$
  0.3627274802357d-08, 0.3096565085313d+01, 0.6271346477544d+01,$
  0.3806143885093d-08, 0.6367751709777d+00, 0.1725304118033d+02,$
  $
  0.4433254641565d-08, 0.4848461503937d+01, 0.7445550607224d+01,$
  0.3712319846576d-08, 0.1331950643655d+01, 0.4194847048887d+00,$
  0.3849847534783d-08, 0.4958368297746d+00, 0.9562891316684d+00,$
  0.3483955430165d-08, 0.2237215515707d+01, 0.1161697602389d+02,$
  0.3961912730982d-08, 0.3332402188575d+01, 0.2277943724828d+02,$
  0.3419978244481d-08, 0.5785600576016d+01, 0.1362553364512d+02,$
  0.3329417758177d-08, 0.9812676559709d-01, 0.1685848245639d+02,$
  0.4207206893193d-08, 0.9494780468236d+00, 0.2986433403208d+02,$
  0.3268548976410d-08, 0.1739332095686d+00, 0.5749861718712d+01,$
  0.3321880082685d-08, 0.1423354800666d+01, 0.6279143387820d+01,$
  $
  0.4503173010852d-08, 0.2314972675293d+00, 0.1385561574497d+01,$
  0.4316599090954d-08, 0.1012646782616d+00, 0.4176041334900d+01,$
  0.3283493323850d-08, 0.5233306881265d+01, 0.6287008313071d+01,$
  0.3164033542343d-08, 0.4005597257511d+01, 0.2099539292909d+02,$
  0.4159720956725d-08, 0.5365676242020d+01, 0.5905702259363d+01,$
  0.3565176892217d-08, 0.4284440620612d+01, 0.3932462625300d-02,$
  0.3514440950221d-08, 0.4270562636575d+01, 0.7335344340001d+01,$
  0.3540596871909d-08, 0.5953553201060d+01, 0.1234573916645d+02,$
  0.2960769905118d-08, 0.1115180417718d+01, 0.2670964694522d+02,$
  0.2962213739684d-08, 0.3863811918186d+01, 0.6408777551755d+00,$
  $
  0.3883556700251d-08, 0.1268617928302d+01, 0.6660449441528d+01,$
  0.2919225516346d-08, 0.4908605223265d+01, 0.1375773836557d+01,$
  0.3115158863370d-08, 0.3744519976885d+01, 0.3802769619140d-01,$
  0.4099438144212d-08, 0.4173244670532d+01, 0.4480965020977d+02,$
  0.2899531858964d-08, 0.5910601428850d+01, 0.2059724391010d+02,$
  0.3289733429855d-08, 0.2488050078239d+01, 0.1081813534213d+02,$
  0.3933075612875d-08, 0.1122363652883d+01, 0.3773735910827d+00,$
  0.3021403764467d-08, 0.4951973724904d+01, 0.2982630633589d+02,$
  0.2798598949757d-08, 0.5117057845513d+01, 0.1937891852345d+02,$
  0.3397421302707d-08, 0.6104159180476d+01, 0.6923953605621d+01,$
  $
  0.3720398002179d-08, 0.1184933429829d+01, 0.3066615496545d+02,$
  0.3598484186267d-08, 0.3505282086105d+01, 0.6147450479709d+01,$
  0.3694594027310d-08, 0.2286651088141d+01, 0.2636725487657d+01,$
  0.2680444152969d-08, 0.1871816775482d+00, 0.6816289982179d+01,$
  0.3497574865641d-08, 0.3143251755431d+01, 0.6418701221183d+01,$
  0.3130274129494d-08, 0.2462167316018d+01, 0.1235996607578d+02,$
  0.3241119069551d-08, 0.4256374004686d+01, 0.1652265972112d+02,$
  0.2601960842061d-08, 0.4970362941425d+01, 0.1045450126711d+02,$
  0.2690601527504d-08, 0.2372657824898d+01, 0.3163918923335d+00,$
  0.2908688152664d-08, 0.4232652627721d+01, 0.2828699048865d+02,$
  $
  0.3120456131875d-08, 0.3925747001137d+00, 0.2195415756911d+02,$
  0.3148855423384d-08, 0.3093478330445d+01, 0.1172006883645d+02,$
  0.3051044261017d-08, 0.5560948248212d+01, 0.6055599646783d+01,$
  0.2826006876660d-08, 0.5072790310072d+01, 0.5120601093667d+01,$
  0.3100034191711d-08, 0.4998530231096d+01, 0.1799603123222d+02,$
  0.2398771640101d-08, 0.2561739802176d+01, 0.6255674361143d+01,$
  0.2384002842728d-08, 0.4087420284111d+01, 0.6310477339748d+01,$
  0.2842146517568d-08, 0.2515048217955d+01, 0.5469525544182d+01,$
  0.2847674371340d-08, 0.5235326497443d+01, 0.1034429499989d+02,$
  0.2903722140764d-08, 0.1088200795797d+01, 0.6510552054109d+01,$
  $
  0.3187610710605d-08, 0.4710624424816d+01, 0.1693792562116d+03,$
  0.3048869992813d-08, 0.2857975896445d+00, 0.8390110365991d+01,$
  0.2860216950984d-08, 0.2241619020815d+01, 0.2243449970715d+00,$
  0.2701117683113d-08, 0.6651573305272d-01, 0.6129297044991d+01,$
  0.2509891590152d-08, 0.1285135324585d+01, 0.1044027435778d+02,$
  0.2623200252223d-08, 0.2981229834530d+00, 0.6436854655901d+01,$
  0.2622541669202d-08, 0.6122470726189d+01, 0.9380959548977d+01,$
  0.2818435667099d-08, 0.4251087148947d+01, 0.5934151399930d+01,$
  0.2365196797465d-08, 0.3465070460790d+01, 0.2470570524223d+02,$
  0.2358704646143d-08, 0.5791603815350d+01, 0.8671969964381d+01,$
  $
  0.2388299481390d-08, 0.4142483772941d+01, 0.7096626156709d+01,$
  0.1996041217224d-08, 0.2101901889496d+01, 0.1727188400790d+02,$
  0.2687593060336d-08, 0.1526689456959d+01, 0.7075506709219d+02,$
  0.2618913670810d-08, 0.2397684236095d+01, 0.6632000300961d+01,$
  0.2571523050364d-08, 0.5751929456787d+00, 0.6206810014183d+01,$
  0.2582135006946d-08, 0.5595464352926d+01, 0.4873985990671d+02,$
  0.2372530190361d-08, 0.5092689490655d+01, 0.1590676413561d+02,$
  0.2357178484712d-08, 0.4444363527851d+01, 0.3097883698531d+01,$
  0.2451590394723d-08, 0.3108251687661d+01, 0.6612329252343d+00,$
  0.2370045949608d-08, 0.2608133861079d+01, 0.3459636466239d+02,$
  $
  0.2268997267358d-08, 0.3639717753384d+01, 0.2844914056730d-01,$
  0.1731432137906d-08, 0.1741898445707d+00, 0.2019909489111d+02,$
  0.1629869741622d-08, 0.3902225646724d+01, 0.3035599730800d+02,$
  0.2206215801974d-08, 0.4971131250731d+01, 0.6281667977667d+01,$
  0.2205469554680d-08, 0.1677462357110d+01, 0.6284483723224d+01,$
  0.2148792362509d-08, 0.4236259604006d+01, 0.1980482729015d+02,$
  0.1873733657847d-08, 0.5926814998687d+01, 0.2876692439167d+02,$
  0.2026573758959d-08, 0.4349643351962d+01, 0.2449240616245d+02,$
  0.1807770325110d-08, 0.5700940482701d+01, 0.2045286941806d+02,$
  0.1881174408581d-08, 0.6601286363430d+00, 0.2358125818164d+02,$
  $
  0.1368023671690d-08, 0.2211098592752d+01, 0.2473415438279d+02,$
  0.1720017916280d-08, 0.4942488551129d+01, 0.1679593901136d+03,$
  0.1702427665131d-08, 0.1452233856386d+01, 0.3338575901272d+03,$
  0.1414032510054d-08, 0.5525357721439d+01, 0.1624205518357d+03,$
  0.1652626045364d-08, 0.4108794283624d+01, 0.8956999012000d+02,$
  0.1642957769686d-08, 0.7344335209984d+00, 0.5267006960365d+02,$
  0.1614952403624d-08, 0.3541213951363d+01, 0.3332657872986d+02,$
  0.1535988291188d-08, 0.4031094072151d+01, 0.3852657435933d+02,$
  0.1593193738177d-08, 0.4185136203609d+01, 0.2282781046519d+03,$
  0.1074569126382d-08, 0.1720485636868d+01, 0.8397383534231d+02,$
  $
  0.1074408214509d-08, 0.2758613420318d+01, 0.8401985929482d+02,$
  0.9700199670465d-09, 0.4216686842097d+01, 0.7826370942180d+02,$
  0.1258433517061d-08, 0.2575068876639d+00, 0.3115650189215d+03,$
  0.1240303229539d-08, 0.4800844956756d+00, 0.1784300471910d+03,$
  0.9018345948127d-09, 0.3896756361552d+00, 0.5886454391678d+02,$
  0.1135301432805d-08, 0.3700805023550d+00, 0.7842370451713d+02,$
  0.9215887951370d-09, 0.4364579276638d+01, 0.1014262087719d+03,$
  0.1055401054147d-08, 0.2156564222111d+01, 0.5660027930059d+02,$
  0.1008725979831d-08, 0.5454015785234d+01, 0.4245678405627d+02,$
  0.7217398104321d-09, 0.1597772562175d+01, 0.2457074661053d+03,$
  $
  0.6912033134447d-09, 0.5824090621461d+01, 0.1679936946371d+03,$
  0.6833881523549d-09, 0.3578778482835d+01, 0.6053048899753d+02,$
  0.4887304205142d-09, 0.3724362812423d+01, 0.9656299901946d+02,$
  0.5173709754788d-09, 0.5422427507933d+01, 0.2442876000072d+03,$
  0.4671353097145d-09, 0.2396106924439d+01, 0.1435713242844d+03,$
  0.5652608439480d-09, 0.2804028838685d+01, 0.8365903305582d+02,$
  0.5604061331253d-09, 0.1638816006247d+01, 0.8433466158131d+02,$
  0.4712723365400d-09, 0.8979003224474d+00, 0.3164282286739d+03,$
  0.4909967465112d-09, 0.3210426725516d+01, 0.4059982187939d+03,$
  0.4771358267658d-09, 0.5308027211629d+01, 0.1805255418145d+03,$
  $
  0.3943451445989d-09, 0.2195145341074d+01, 0.2568537517081d+03,$
  0.3952109120244d-09, 0.5081189491586d+01, 0.2449975330562d+03,$
  0.3788134594789d-09, 0.4345171264441d+01, 0.1568131045107d+03,$
  0.3738330190479d-09, 0.2613062847997d+01, 0.3948519331910d+03,$
  0.3099866678136d-09, 0.2846760817689d+01, 0.1547176098872d+03,$
  0.2002962716768d-09, 0.4921360989412d+01, 0.2268582385539d+03,$
  0.2198291338754d-09, 0.1130360117454d+00, 0.1658638954901d+03,$
  0.1491958330784d-09, 0.4228195232278d+01, 0.2219950288015d+03,$
  0.1475384076173d-09, 0.3005721811604d+00, 0.3052819430710d+03,$
  0.1661626624624d-09, 0.7830125621203d+00, 0.2526661704812d+03,$
  $
  0.9015823460025d-10, 0.3807792942715d+01, 0.4171445043968d+03 ];
  
  
  
  
  
  
 ; /* Sun-to-Earth, T^0, Y */
 e0y = [ $
  0.9998921098898d+00, 0.1826583913846d+00, 0.6283075850446d+01,$
  -0.2442700893735d-01, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.8352929742915d-02, 0.1395277998680d+00, 0.1256615170089d+02,$
  0.1046697300177d-03, 0.9641423109763d-01, 0.1884922755134d+02,$
  0.3110841876663d-04, 0.5381140401712d+01, 0.8399684731857d+02,$
  0.2570269094593d-04, 0.5301016407128d+01, 0.5296909721118d+00,$
  0.2147389623610d-04, 0.2662510869850d+01, 0.1577343543434d+01,$
  0.1680344384050d-04, 0.5207904119704d+01, 0.6279552690824d+01,$
  0.1679117312193d-04, 0.4582187486968d+01, 0.6286599010068d+01,$
  0.1440512068440d-04, 0.1900688517726d+01, 0.2352866153506d+01,$
  $
  0.1135139664999d-04, 0.5273108538556d+01, 0.5223693906222d+01,$
  0.9345482571018d-05, 0.4503047687738d+01, 0.1203646072878d+02,$
  0.9007418719568d-05, 0.1605621059637d+01, 0.1021328554739d+02,$
  0.5671536712314d-05, 0.5812849070861d+00, 0.1059381944224d+01,$
  0.7451401861666d-05, 0.2807346794836d+01, 0.3981490189893d+00,$
  0.6393470057114d-05, 0.6029224133855d+01, 0.5753384878334d+01,$
  0.6814275881697d-05, 0.6472990145974d+00, 0.4705732307012d+01,$
  0.6113705628887d-05, 0.3813843419700d+01, 0.6812766822558d+01,$
  0.4503851367273d-05, 0.4527804370996d+01, 0.5884926831456d+01,$
  0.4522249141926d-05, 0.5991783029224d+01, 0.6256777527156d+01,$
  $
  0.4501794307018d-05, 0.3798703844397d+01, 0.6309374173736d+01,$
  0.5514927480180d-05, 0.3961257833388d+01, 0.5507553240374d+01,$
  0.4062862799995d-05, 0.5256247296369d+01, 0.6681224869435d+01,$
  0.5414900429712d-05, 0.5499032014097d+01, 0.7755226100720d+00,$
  0.5463153987424d-05, 0.6173092454097d+01, 0.1414349524433d+02,$
  0.5071611859329d-05, 0.2870244247651d+01, 0.7860419393880d+01,$
  0.2195112094455d-05, 0.2952338617201d+01, 0.1150676975667d+02,$
  0.2279139233919d-05, 0.5951775132933d+01, 0.7058598460518d+01,$
  0.2278386100876d-05, 0.4845456398785d+01, 0.4694002934110d+01,$
  0.2559088003308d-05, 0.6945321117311d+00, 0.1216800268190d+02,$
  $
  0.2561079286856d-05, 0.6167224608301d+01, 0.7099330490126d+00,$
  0.1792755796387d-05, 0.1400122509632d+01, 0.7962980379786d+00,$
  0.1818715656502d-05, 0.4703347611830d+01, 0.6283142985870d+01,$
  0.1818744924791d-05, 0.5086748900237d+01, 0.6283008715021d+01,$
  0.1554518791390d-05, 0.5331008042713d-01, 0.2513230340178d+02,$
  0.2063265737239d-05, 0.4283680484178d+01, 0.1179062909082d+02,$
  0.1497613520041d-05, 0.6074207826073d+01, 0.5486777812467d+01,$
  0.2000617940427d-05, 0.2501426281450d+01, 0.1778984560711d+02,$
  0.1289731195580d-05, 0.3646340599536d+01, 0.7079373888424d+01,$
  0.1282657998934d-05, 0.3232864804902d+01, 0.3738761453707d+01,$
  $
  0.1528915968658d-05, 0.5581433416669d+01, 0.2132990797783d+00,$
  0.1187304098432d-05, 0.5453576453694d+01, 0.9437762937313d+01,$
  0.7842782928118d-06, 0.2823953922273d+00, 0.8827390247185d+01,$
  0.7352892280868d-06, 0.1124369580175d+01, 0.1589072916335d+01,$
  0.6570189360797d-06, 0.2089154042840d+01, 0.1176985366291d+02,$
  0.6324967590410d-06, 0.6704855581230d+00, 0.6262300422539d+01,$
  0.6298289872283d-06, 0.2836414855840d+01, 0.6303851278352d+01,$
  0.6476686465855d-06, 0.4852433866467d+00, 0.7113454667900d-02,$
  0.8587034651234d-06, 0.1453511005668d+01, 0.1672837615881d+03,$
  0.8068948788113d-06, 0.9224087798609d+00, 0.6069776770667d+01,$
  $
  0.8353786011661d-06, 0.4631707184895d+01, 0.3340612434717d+01,$
  0.6009324532132d-06, 0.1829498827726d+01, 0.4136910472696d+01,$
  0.7558158559566d-06, 0.2588596800317d+01, 0.6496374930224d+01,$
  0.5809279504503d-06, 0.5516818853476d+00, 0.1097707878456d+02,$
  0.5374131950254d-06, 0.6275674734960d+01, 0.1194447056968d+01,$
  0.5711160507326d-06, 0.1091905956872d+01, 0.6282095334605d+01,$
  0.5710183170746d-06, 0.2415001635090d+01, 0.6284056366286d+01,$
  0.5144373590610d-06, 0.6020336443438d+01, 0.6290189305114d+01,$
  0.5103108927267d-06, 0.3775634564605d+01, 0.6275962395778d+01,$
  0.4960654697891d-06, 0.1073450946756d+01, 0.6127655567643d+01,$
  $
  0.4786385689280d-06, 0.2431178012310d+01, 0.6438496133249d+01,$
  0.6109911263665d-06, 0.5343356157914d+01, 0.3154687086868d+01,$
  0.4839898944024d-06, 0.5830833594047d-01, 0.8018209333619d+00,$
  0.4734822623919d-06, 0.4536080134821d+01, 0.3128388763578d+01,$
  0.4834741473290d-06, 0.2585090489754d+00, 0.7084896783808d+01,$
  0.5134858581156d-06, 0.4213317172603d+01, 0.1235285262111d+02,$
  0.5064004264978d-06, 0.4814418806478d+00, 0.1185621865188d+02,$
  0.3753476772761d-06, 0.1599953399788d+01, 0.8429241228195d+01,$
  0.4935264014283d-06, 0.2157417556873d+01, 0.2544314396739d+01,$
  0.3950929600897d-06, 0.3359394184254d+01, 0.5481254917084d+01,$
  $
  0.4895849789777d-06, 0.5165704376558d+01, 0.9225539266174d+01,$
  0.4215241688886d-06, 0.2065368800993d+01, 0.1726015463500d+02,$
  0.3796773731132d-06, 0.1468606346612d+01, 0.4265981595566d+00,$
  0.3114178142515d-06, 0.3615638079474d+01, 0.2146165377750d+01,$
  0.3260664220838d-06, 0.4417134922435d+01, 0.4164311961999d+01,$
  0.3976996123008d-06, 0.4700866883004d+01, 0.5856477690889d+01,$
  0.2801459672924d-06, 0.4538902060922d+01, 0.1256967486051d+02,$
  0.3638931868861d-06, 0.1334197991475d+01, 0.1807370494127d+02,$
  0.2487013269476d-06, 0.3749275558275d+01, 0.2629832328990d-01,$
  0.3034165481994d-06, 0.4236622030873d+00, 0.4535059491685d+01,$
  $
  0.2676278825586d-06, 0.5970848007811d+01, 0.3930209696940d+01,$
  0.2764903818918d-06, 0.5194636754501d+01, 0.1256262854127d+02,$
  0.2485149930507d-06, 0.1002434207846d+01, 0.5088628793478d+01,$
  0.2199305540941d-06, 0.3066773098403d+01, 0.1255903824622d+02,$
  0.2571106500435d-06, 0.7588312459063d+00, 0.1336797263425d+02,$
  0.2049751817158d-06, 0.3444977434856d+01, 0.1137170464392d+02,$
  0.2599707296297d-06, 0.1873128542205d+01, 0.7143069561767d+02,$
  0.1785018072217d-06, 0.5015891306615d+01, 0.1748016358760d+01,$
  0.2324833891115d-06, 0.4618271239730d+01, 0.1831953657923d+02,$
  0.1709711119545d-06, 0.5300003455669d+01, 0.4933208510675d+01,$
  $
  0.2107159351716d-06, 0.2229819815115d+01, 0.7477522907414d+01,$
  0.1750333080295d-06, 0.6161485880008d+01, 0.1044738781244d+02,$
  0.2000598210339d-06, 0.2967357299999d+01, 0.8031092209206d+01,$
  0.1380920248681d-06, 0.3027007923917d+01, 0.8635942003952d+01,$
  0.1412460470299d-06, 0.6037597163798d+01, 0.2942463415728d+01,$
  0.1888459803001d-06, 0.8561476243374d+00, 0.1561374759853d+03,$
  0.1788370542585d-06, 0.4869736290209d+01, 0.1592596075957d+01,$
  0.1360893296167d-06, 0.3626411886436d+01, 0.1309584267300d+02,$
  0.1506846530160d-06, 0.1550975377427d+01, 0.1649636139783d+02,$
  0.1800913376176d-06, 0.2075826033190d+01, 0.1729818233119d+02,$
  $
  0.1436261390649d-06, 0.6148876420255d+01, 0.2042657109477d+02,$
  0.1220227114151d-06, 0.4382583879906d+01, 0.7632943190217d+01,$
  0.1337883603592d-06, 0.2036644327361d+01, 0.1213955354133d+02,$
  0.1159326650738d-06, 0.3892276994687d+01, 0.5331357529664d+01,$
  0.1352853128569d-06, 0.1447950649744d+01, 0.1673046366289d+02,$
  0.1433408296083d-06, 0.4457854692961d+01, 0.7342457794669d+01,$
  0.1234701666518d-06, 0.1538818147151d+01, 0.6279485555400d+01,$
  0.1234027192007d-06, 0.1968523220760d+01, 0.6286666145492d+01,$
  0.1244024091797d-06, 0.5779803499985d+01, 0.1511046609763d+02,$
  0.1097934945516d-06, 0.6210975221388d+00, 0.1098880815746d+02,$
  $
  0.1254611329856d-06, 0.2591963807998d+01, 0.1572083878776d+02,$
  0.1158247286784d-06, 0.2483612812670d+01, 0.5729506548653d+01,$
  0.9039078252960d-07, 0.3857554579796d+01, 0.9623688285163d+01,$
  0.9108024978836d-07, 0.5826368512984d+01, 0.7234794171227d+01,$
  0.8887068108436d-07, 0.3475694573987d+01, 0.6148010737701d+01,$
  0.8632374035438d-07, 0.3059070488983d-01, 0.6418140963190d+01,$
  0.7893186992967d-07, 0.1583194837728d+01, 0.2118763888447d+01,$
  0.8297650201172d-07, 0.8519770534637d+00, 0.1471231707864d+02,$
  0.1019759578988d-06, 0.1319598738732d+00, 0.1349867339771d+01,$
  0.1010037696236d-06, 0.9937860115618d+00, 0.6836645152238d+01,$
  $
  0.1047727548266d-06, 0.1382138405399d+01, 0.5999216516294d+01,$
  0.7351993881086d-07, 0.3833397851735d+01, 0.6040347114260d+01,$
  0.9868771092341d-07, 0.2124913814390d+01, 0.6566935184597d+01,$
  0.7007321959390d-07, 0.5946305343763d+01, 0.6525804586632d+01,$
  0.6861411679709d-07, 0.4574654977089d+01, 0.7238675589263d+01,$
  0.7554519809614d-07, 0.5949232686844d+01, 0.1253985337760d+02,$
  0.9541880448335d-07, 0.3495242990564d+01, 0.2122839202813d+02,$
  0.7185606722155d-07, 0.4310113471661d+01, 0.6245048154254d+01,$
  0.7131360871710d-07, 0.5480309323650d+01, 0.6321103546637d+01,$
  0.6651142021039d-07, 0.5411097713654d+01, 0.5327476111629d+01,$
  $
  0.8538618213667d-07, 0.1827849973951d+01, 0.1101510648075d+02,$
  0.8634954288044d-07, 0.5443584943349d+01, 0.5643178611111d+01,$
  0.7449415051484d-07, 0.2011535459060d+01, 0.5368044267797d+00,$
  0.7421047599169d-07, 0.3464562529249d+01, 0.2354323048545d+02,$
  0.6140694354424d-07, 0.5657556228815d+01, 0.1296430071988d+02,$
  0.6353525143033d-07, 0.3463816593821d+01, 0.1990745094947d+01,$
  0.6221964013447d-07, 0.1532259498697d+01, 0.9517183207817d+00,$
  0.5852480257244d-07, 0.1375396598875d+01, 0.9555997388169d+00,$
  0.6398637498911d-07, 0.2405645801972d+01, 0.2407292145756d+02,$
  0.7039744069878d-07, 0.5397541799027d+01, 0.5225775174439d+00,$
  $
  0.6977997694382d-07, 0.4762347105419d+01, 0.1097355562493d+02,$
  0.7460629558396d-07, 0.2711944692164d+01, 0.2200391463820d+02,$
  0.5376577536101d-07, 0.2352980430239d+01, 0.1431416805965d+02,$
  0.7530607893556d-07, 0.1943940180699d+01, 0.1842262939178d+02,$
  0.6822928971605d-07, 0.4337651846959d+01, 0.1554202828031d+00,$
  0.6220772380094d-07, 0.6716871369278d+00, 0.1845107853235d+02,$
  0.6586950799043d-07, 0.2229714460505d+01, 0.5216580451554d+01,$
  0.5873800565771d-07, 0.7627013920580d+00, 0.6398972393349d+00,$
  0.6264346929745d-07, 0.6202785478961d+00, 0.6277552955062d+01,$
  0.6257929115669d-07, 0.2886775596668d+01, 0.6288598745829d+01,$
  $
  0.5343536033409d-07, 0.1977241012051d+01, 0.4690479774488d+01,$
  0.5587849781714d-07, 0.1922923484825d+01, 0.1551045220144d+01,$
  0.6905100845603d-07, 0.3570757164631d+01, 0.1030928125552d+00,$
  0.6178957066649d-07, 0.5197558947765d+01, 0.5230807360890d+01,$
  0.6187270224331d-07, 0.8193497368922d+00, 0.5650292065779d+01,$
  0.5385664291426d-07, 0.5406336665586d+01, 0.7771377146812d+02,$
  0.6329363917926d-07, 0.2837760654536d+01, 0.2608790314060d+02,$
  0.4546018761604d-07, 0.2933580297050d+01, 0.5535693017924d+00,$
  0.6196091049375d-07, 0.4157871494377d+01, 0.8467247584405d+02,$
  0.6159555108218d-07, 0.3211703561703d+01, 0.2394243902548d+03,$
  $
  0.4995340539317d-07, 0.1459098102922d+01, 0.4732030630302d+01,$
  0.5457031243572d-07, 0.1430457676136d+01, 0.6179983037890d+01,$
  0.4863461418397d-07, 0.2196425916730d+01, 0.9027992316901d+02,$
  0.5342947626870d-07, 0.2086612890268d+01, 0.6386168663001d+01,$
  0.5674296648439d-07, 0.2760204966535d+01, 0.6915859635113d+01,$
  0.4745783120161d-07, 0.4245368971862d+01, 0.6282970628506d+01,$
  0.4745676961198d-07, 0.5544725787016d+01, 0.6283181072386d+01,$
  0.4049796869973d-07, 0.2213984363586d+01, 0.6254626709878d+01,$
  0.4248333596940d-07, 0.8075781952896d+00, 0.7875671926403d+01,$
  0.4027178070205d-07, 0.1293268540378d+01, 0.6311524991013d+01,$
  $
  0.4066543943476d-07, 0.3986141175804d+01, 0.3634620989887d+01,$
  0.4858863787880d-07, 0.1276112738231d+01, 0.5760498333002d+01,$
  0.5277398263530d-07, 0.4916111741527d+01, 0.2515860172507d+02,$
  0.4105635656559d-07, 0.1725805864426d+01, 0.6709674010002d+01,$
  0.4376781925772d-07, 0.2243642442106d+01, 0.6805653367890d+01,$
  0.3235827894693d-07, 0.3614135118271d+01, 0.1066495398892d+01,$
  0.3073244740308d-07, 0.2460873393460d+01, 0.5863591145557d+01,$
  0.3088609271373d-07, 0.5678431771790d+01, 0.9917696840332d+01,$
  0.3393022279836d-07, 0.3814017477291d+01, 0.1391601904066d+02,$
  0.3038686508802d-07, 0.4660216229171d+01, 0.1256621883632d+02,$
  $
  0.4019677752497d-07, 0.5906906243735d+01, 0.1334167431096d+02,$
  0.3288834998232d-07, 0.9536146445882d+00, 0.1620077269078d+02,$
  0.3889973794631d-07, 0.3942205097644d+01, 0.7478166569050d-01,$
  0.3050438987141d-07, 0.1624810271286d+01, 0.1805292951336d+02,$
  0.3601142564638d-07, 0.4030467142575d+01, 0.6208294184755d+01,$
  0.3689015557141d-07, 0.3648878818694d+01, 0.5966683958112d+01,$
  0.3563471893565d-07, 0.5749584017096d+01, 0.6357857516136d+01,$
  0.2776183170667d-07, 0.2630124187070d+01, 0.3523159621801d-02,$
  0.2922350530341d-07, 0.1790346403629d+01, 0.1272157198369d+02,$
  0.3511076917302d-07, 0.6142198301611d+01, 0.6599467742779d+01,$
  $
  0.3619351007632d-07, 0.1432421386492d+01, 0.6019991944201d+01,$
  0.2561254711098d-07, 0.2302822475792d+01, 0.1259245002418d+02,$
  0.2626903942920d-07, 0.8660470994571d+00, 0.6702560555334d+01,$
  0.2550187397083d-07, 0.6069721995383d+01, 0.1057540660594d+02,$
  0.2535873526138d-07, 0.1079020331795d-01, 0.3141537925223d+02,$
  0.3519786153847d-07, 0.3809066902283d+01, 0.2505706758577d+03,$
  0.3424651492873d-07, 0.2075435114417d+01, 0.6546159756691d+01,$
  0.2372676630861d-07, 0.2057803120154d+01, 0.2388894113936d+01,$
  0.2710980779541d-07, 0.1510068488010d+01, 0.1202934727411d+02,$
  0.3038710889704d-07, 0.5043617528901d+01, 0.1256608456547d+02,$
  $
  0.2220364130585d-07, 0.3694793218205d+01, 0.1336244973887d+02,$
  0.3025880825460d-07, 0.5450618999049d-01, 0.2908881142201d+02,$
  0.2784493486864d-07, 0.3381164084502d+01, 0.1494531617769d+02,$
  0.2294414142438d-07, 0.4382309025210d+01, 0.6076890225335d+01,$
  0.2012723294724d-07, 0.9142212256518d+00, 0.6262720680387d+01,$
  0.2036357831958d-07, 0.5676172293154d+01, 0.4701116388778d+01,$
  0.2003474823288d-07, 0.2592767977625d+01, 0.6303431020504d+01,$
  0.2207144900109d-07, 0.5404976271180d+01, 0.6489261475556d+01,$
  0.2481664905135d-07, 0.4373284587027d+01, 0.1204357418345d+02,$
  0.2674949182295d-07, 0.5859182188482d+01, 0.4590910121555d+01,$
  $
  0.2450554720322d-07, 0.4555381557451d+01, 0.1495633313810d+00,$
  0.2601975986457d-07, 0.3933165584959d+01, 0.1965104848470d+02,$
  0.2199860022848d-07, 0.5227977189087d+01, 0.1351787002167d+02,$
  0.2448121172316d-07, 0.4858060353949d+01, 0.1162474756779d+01,$
  0.1876014864049d-07, 0.5690546553605d+01, 0.6279194432410d+01,$
  0.1874513219396d-07, 0.4099539297446d+01, 0.6286957268481d+01,$
  0.2156380842559d-07, 0.4382594769913d+00, 0.1813929450232d+02,$
  0.1981691240061d-07, 0.1829784152444d+01, 0.4686889479442d+01,$
  0.2329992648539d-07, 0.2836254278973d+01, 0.1002183730415d+02,$
  0.1765184135302d-07, 0.2803494925833d+01, 0.4292330755499d+01,$
  $
  0.2436368366085d-07, 0.2836897959677d+01, 0.9514313292143d+02,$
  0.2164089203889d-07, 0.6127522446024d+01, 0.6037244212485d+01,$
  0.1847755034221d-07, 0.3683163635008d+01, 0.2427287361862d+00,$
  0.1674798769966d-07, 0.3316993867246d+00, 0.1311972100268d+02,$
  0.2222542124356d-07, 0.8294097805480d+00, 0.1266924451345d+02,$
  0.2071074505925d-07, 0.3659492220261d+01, 0.6528907488406d+01,$
  0.1608224471835d-07, 0.4774492067182d+01, 0.1352175143971d+02,$
  0.1857583439071d-07, 0.2873120597682d+01, 0.8662240327241d+01,$
  0.1793018836159d-07, 0.5282441177929d+00, 0.6819880277225d+01,$
  0.1575391221692d-07, 0.1320789654258d+01, 0.1102062672231d+00,$
  $
  0.1840132009557d-07, 0.1917110916256d+01, 0.6514761976723d+02,$
  0.1760917288281d-07, 0.2972635937132d+01, 0.5746271423666d+01,$
  0.1561779518516d-07, 0.4372569261981d+01, 0.6272439236156d+01,$
  0.1558687885205d-07, 0.5416424926425d+01, 0.6293712464735d+01,$
  0.1951359382579d-07, 0.3094448898752d+01, 0.2301353951334d+02,$
  0.1569144275614d-07, 0.2802103689808d+01, 0.1765478049437d+02,$
  0.1479130389462d-07, 0.2136435020467d+01, 0.2077542790660d-01,$
  0.1467828510764d-07, 0.7072627435674d+00, 0.1052268489556d+01,$
  0.1627627337440d-07, 0.3947607143237d+01, 0.6327837846670d+00,$
  0.1503498479758d-07, 0.4079248909190d+01, 0.7626583626240d-01,$
  $
  0.1297967708237d-07, 0.6269637122840d+01, 0.1149965630200d+02,$
  0.1374416896634d-07, 0.4175657970702d+01, 0.6016468784579d+01,$
  0.1783812325219d-07, 0.1476540547560d+01, 0.3301902111895d+02,$
  0.1525884228756d-07, 0.4653477715241d+01, 0.9411464614024d+01,$
  0.1451067396763d-07, 0.2573001128225d+01, 0.1277945078067d+02,$
  0.1297713111950d-07, 0.5612799618771d+01, 0.6549682916313d+01,$
  0.1462784012820d-07, 0.4189661623870d+01, 0.1863592847156d+02,$
  0.1384185980007d-07, 0.2656915472196d+01, 0.2379164476796d+01,$
  0.1221497599801d-07, 0.5612515760138d+01, 0.1257326515556d+02,$
  0.1560574525896d-07, 0.4783414317919d+01, 0.1887552587463d+02,$
  $
  0.1544598372036d-07, 0.2694431138063d+01, 0.1820933031200d+02,$
  0.1531678928696d-07, 0.4105103489666d+01, 0.2593412433514d+02,$
  0.1349321503795d-07, 0.3082437194015d+00, 0.5120601093667d+01,$
  0.1252030290917d-07, 0.6124072334087d+01, 0.6993008899458d+01,$
  0.1459243816687d-07, 0.3733103981697d+01, 0.3813291813120d-01,$
  0.1226103625262d-07, 0.1267127706817d+01, 0.2435678079171d+02,$
  0.1019449641504d-07, 0.4367790112269d+01, 0.1725663147538d+02,$
  0.1380789433607d-07, 0.3387201768700d+01, 0.2458316379602d+00,$
  0.1019453421658d-07, 0.9204143073737d+00, 0.6112403035119d+01,$
  0.1297929434405d-07, 0.5786874896426d+01, 0.1249137003520d+02,$
  $
  0.9912677786097d-08, 0.3164232870746d+01, 0.6247047890016d+01,$
  0.9829386098599d-08, 0.2586762413351d+01, 0.6453748665772d+01,$
  0.1226807746104d-07, 0.6239068436607d+01, 0.5429879531333d+01,$
  0.1192691755997d-07, 0.1867380051424d+01, 0.6290122169689d+01,$
  0.9836499227081d-08, 0.3424716293727d+00, 0.6319103810876d+01,$
  0.9642862564285d-08, 0.5661372990657d+01, 0.8273820945392d+01,$
  0.1165184404862d-07, 0.5768367239093d+01, 0.1778273215245d+02,$
  0.1175794418818d-07, 0.1657351222943d+01, 0.6276029531202d+01,$
  0.1018948635601d-07, 0.6458292350865d+00, 0.1254537627298d+02,$
  0.9500383606676d-08, 0.1054306140741d+01, 0.1256517118505d+02,$
  $
  0.1227512202906d-07, 0.2505278379114d+01, 0.2248384854122d+02,$
  0.9664792009993d-08, 0.4289737277000d+01, 0.6259197520765d+01,$
  0.9613285666331d-08, 0.5500597673141d+01, 0.6306954180126d+01,$
  0.1117906736211d-07, 0.2361405953468d+01, 0.1779695906178d+02,$
  0.9611378640782d-08, 0.2851310576269d+01, 0.2061856251104d+00,$
  0.8845354852370d-08, 0.6208777705343d+01, 0.1692165728891d+01,$
  0.1054046966600d-07, 0.5413091423934d+01, 0.2204125344462d+00,$
  0.1215539124483d-07, 0.5613969479755d+01, 0.8257698122054d+02,$
  0.9932460955209d-08, 0.1106124877015d+01, 0.1017725758696d+02,$
  0.8785804715043d-08, 0.2869224476477d+01, 0.9491756770005d+00,$
  $
  0.8538084097562d-08, 0.6159640899344d+01, 0.6393282117669d+01,$
  0.8648994369529d-08, 0.1374901198784d+01, 0.4804209201333d+01,$
  0.1039063219067d-07, 0.5171080641327d+01, 0.1550861511662d+02,$
  0.8867983926439d-08, 0.8317320304902d+00, 0.3903911373650d+01,$
  0.8327495955244d-08, 0.3605591969180d+01, 0.6172869583223d+01,$
  0.9243088356133d-08, 0.6114299196843d+01, 0.6267823317922d+01,$
  0.9205657357835d-08, 0.3675153683737d+01, 0.6298328382969d+01,$
  0.1033269714606d-07, 0.3313328813024d+01, 0.5573142801433d+01,$
  0.8001706275552d-08, 0.2019980960053d+01, 0.2648454860559d+01,$
  0.9171858254191d-08, 0.8992015524177d+00, 0.1498544001348d+03,$
  $
  0.1075327150242d-07, 0.2898669963648d+01, 0.3694923081589d+02,$
  0.9884866689828d-08, 0.4946715904478d+01, 0.1140367694411d+02,$
  0.9541835576677d-08, 0.2371787888469d+01, 0.1256713221673d+02,$
  0.7739903376237d-08, 0.2213775190612d+01, 0.7834121070590d+01,$
  0.7311962684106d-08, 0.3429378787739d+01, 0.1192625446156d+02,$
  0.9724904869624d-08, 0.6195878564404d+01, 0.2280573557157d+02,$
  0.9251628983612d-08, 0.6511509527390d+00, 0.2787043132925d+01,$
  0.7320763787842d-08, 0.6001083639421d+01, 0.6282655592598d+01,$
  0.7320296650962d-08, 0.3789073265087d+01, 0.6283496108294d+01,$
  0.7947032271039d-08, 0.1059659582204d+01, 0.1241073141809d+02,$
  $
  0.9005277053115d-08, 0.1280315624361d+01, 0.6281591679874d+01,$
  0.8995601652048d-08, 0.2224439106766d+01, 0.6284560021018d+01,$
  0.8288040568796d-08, 0.5234914433867d+01, 0.1241658836951d+02,$
  0.6359381347255d-08, 0.4137989441490d+01, 0.1596186371003d+01,$
  0.8699572228626d-08, 0.1758411009497d+01, 0.6133512519065d+01,$
  0.6456797542736d-08, 0.5919285089994d+01, 0.1685848245639d+02,$
  0.7424573475452d-08, 0.5414616938827d+01, 0.4061219149443d+01,$
  0.7235671196168d-08, 0.1496516557134d+01, 0.1610006857377d+03,$
  0.8104015182733d-08, 0.1919918242764d+01, 0.8460828644453d+00,$
  0.8098576535937d-08, 0.3819615855458d+01, 0.3894181736510d+01,$
  $
  0.6275292346625d-08, 0.6244264115141d+01, 0.8531963191132d+00,$
  0.6052432989112d-08, 0.5037731872610d+00, 0.1567108171867d+02,$
  0.5705651535817d-08, 0.2984557271995d+01, 0.1258692712880d+02,$
  0.5789650115138d-08, 0.6087038140697d+01, 0.1193336791622d+02,$
  0.5512132153377d-08, 0.5855668994076d+01, 0.1232342296471d+02,$
  0.7388890819102d-08, 0.2443128574740d+01, 0.4907302013889d+01,$
  0.5467593991798d-08, 0.3017561234194d+01, 0.1884211409667d+02,$
  0.6388519802999d-08, 0.5887386712935d+01, 0.5217580628120d+02,$
  0.6106777149944d-08, 0.3483461059895d+00, 0.1422690933580d-01,$
  0.7383420275489d-08, 0.5417387056707d+01, 0.2358125818164d+02,$
  $
  0.5505208141738d-08, 0.2848193644783d+01, 0.1151388321134d+02,$
  0.6310757462877d-08, 0.2349882520828d+01, 0.1041998632314d+02,$
  0.6166904929691d-08, 0.5728575944077d+00, 0.6151533897323d+01,$
  0.5263442042754d-08, 0.4495796125937d+01, 0.1885275071096d+02,$
  0.5591828082629d-08, 0.1355441967677d+01, 0.4337116142245d+00,$
  0.5397051680497d-08, 0.1673422864307d+01, 0.6286362197481d+01,$
  0.5396992745159d-08, 0.1833502206373d+01, 0.6279789503410d+01,$
  0.6572913000726d-08, 0.3331122065824d+01, 0.1176433076753d+02,$
  0.5123421866413d-08, 0.2165327142679d+01, 0.1245594543367d+02,$
  0.5930495725999d-08, 0.2931146089284d+01, 0.6414617803568d+01,$
  $
  0.6431797403933d-08, 0.4134407994088d+01, 0.1350651127443d+00,$
  0.5003182207604d-08, 0.3805420303749d+01, 0.1096996532989d+02,$
  0.5587731032504d-08, 0.1082469260599d+01, 0.6062663316000d+01,$
  0.5935263407816d-08, 0.8384333678401d+00, 0.5326786718777d+01,$
  0.4756019827760d-08, 0.3552588749309d+01, 0.3104930017775d+01,$
  0.6599951172637d-08, 0.4320826409528d+01, 0.4087944051283d+02,$
  0.5902606868464d-08, 0.4811879454445d+01, 0.5849364236221d+01,$
  0.5921147809031d-08, 0.9942628922396d-01, 0.1581959461667d+01,$
  0.5505382581266d-08, 0.2466557607764d+01, 0.6503488384892d+01,$
  0.5353771071862d-08, 0.4551978748683d+01, 0.1735668374386d+03,$
  $
  0.5063282210946d-08, 0.5710812312425d+01, 0.1248988586463d+02,$
  0.5926120403383d-08, 0.1333998428358d+01, 0.2673594526851d+02,$
  0.5211016176149d-08, 0.4649315360760d+01, 0.2460261242967d+02,$
  0.5347075084894d-08, 0.5512754081205d+01, 0.4171425416666d+01,$
  0.4872609773574d-08, 0.1308025299938d+01, 0.5333900173445d+01,$
  0.4727711321420d-08, 0.2144908368062d+01, 0.7232251527446d+01,$
  0.6029426018652d-08, 0.5567259412084d+01, 0.3227113045244d+03,$
  0.4321485284369d-08, 0.5230667156451d+01, 0.9388005868221d+01,$
  0.4476406760553d-08, 0.6134081115303d+01, 0.5547199253223d+01,$
  0.5835268277420d-08, 0.4783808492071d+01, 0.7285056171570d+02,$
  $
  0.5172183602748d-08, 0.5161817911099d+01, 0.1884570439172d+02,$
  0.5693571465184d-08, 0.1381646203111d+01, 0.9723862754494d+02,$
  0.4060634965349d-08, 0.3876705259495d+00, 0.4274518229222d+01,$
  0.3967398770473d-08, 0.5029491776223d+01, 0.3496032717521d+01,$
  0.3943754005255d-08, 0.1923162955490d+01, 0.6244942932314d+01,$
  0.4781323427824d-08, 0.4633332586423d+01, 0.2929661536378d+02,$
  0.3871483781204d-08, 0.1616650009743d+01, 0.6321208768577d+01,$
  0.5141741733997d-08, 0.9817316704659d-01, 0.1232032006293d+02,$
  0.4002385978497d-08, 0.3656161212139d+01, 0.7018952447668d+01,$
  0.4901092604097d-08, 0.4404098713092d+01, 0.1478866649112d+01,$
  $
  0.3740932630345d-08, 0.5181188732639d+00, 0.6922973089781d+01,$
  0.4387283718538d-08, 0.3254859566869d+01, 0.2331413144044d+03,$
  0.5019197802033d-08, 0.3086773224677d+01, 0.1715706182245d+02,$
  0.3834931695175d-08, 0.2797882673542d+01, 0.1491901785440d+02,$
  0.3760413942497d-08, 0.2892676280217d+01, 0.1726726808967d+02,$
  0.3719717204628d-08, 0.5861046025739d+01, 0.6297302759782d+01,$
  0.4145623530149d-08, 0.2168239627033d+01, 0.1376059875786d+02,$
  0.3932788425380d-08, 0.6271811124181d+01, 0.7872148766781d+01,$
  0.3686377476857d-08, 0.3936853151404d+01, 0.6268848941110d+01,$
  0.3779077950339d-08, 0.1404148734043d+01, 0.4157198507331d+01,$
  $
  0.4091334550598d-08, 0.2452436180854d+01, 0.9779108567966d+01,$
  0.3926694536146d-08, 0.6102292739040d+01, 0.1098419223922d+02,$
  0.4841000253289d-08, 0.6072760457276d+01, 0.1252801878276d+02,$
  0.4949340130240d-08, 0.1154832815171d+01, 0.1617106187867d+03,$
  0.3761557737360d-08, 0.5527545321897d+01, 0.3185192151914d+01,$
  0.3647396268188d-08, 0.1525035688629d+01, 0.6271346477544d+01,$
  0.3932405074189d-08, 0.5570681040569d+01, 0.2139354194808d+02,$
  0.3631322501141d-08, 0.1981240601160d+01, 0.6294805223347d+01,$
  0.4130007425139d-08, 0.2050060880201d+01, 0.2195415756911d+02,$
  0.4433905965176d-08, 0.3277477970321d+01, 0.7445550607224d+01,$
  $
  0.3851814176947d-08, 0.5210690074886d+01, 0.9562891316684d+00,$
  0.3485807052785d-08, 0.6653274904611d+00, 0.1161697602389d+02,$
  0.3979772816991d-08, 0.1767941436148d+01, 0.2277943724828d+02,$
  0.3402607460500d-08, 0.3421746306465d+01, 0.1087398597200d+02,$
  0.4049993000926d-08, 0.1127144787547d+01, 0.3163918923335d+00,$
  0.3420511182382d-08, 0.4214794779161d+01, 0.1362553364512d+02,$
  0.3640772365012d-08, 0.5324905497687d+01, 0.1725304118033d+02,$
  0.3323037987501d-08, 0.6135761838271d+01, 0.6279143387820d+01,$
  0.4503141663637d-08, 0.1802305450666d+01, 0.1385561574497d+01,$
  0.4314560055588d-08, 0.4812299731574d+01, 0.4176041334900d+01,$
  $
  0.3294226949110d-08, 0.3657547059723d+01, 0.6287008313071d+01,$
  0.3215657197281d-08, 0.4866676894425d+01, 0.5749861718712d+01,$
  0.4129362656266d-08, 0.3809342558906d+01, 0.5905702259363d+01,$
  0.3137762976388d-08, 0.2494635174443d+01, 0.2099539292909d+02,$
  0.3514010952384d-08, 0.2699961831678d+01, 0.7335344340001d+01,$
  0.3327607571530d-08, 0.3318457714816d+01, 0.5436992986000d+01,$
  0.3541066946675d-08, 0.4382703582466d+01, 0.1234573916645d+02,$
  0.3216179847052d-08, 0.5271066317054d+01, 0.3802769619140d-01,$
  0.2959045059570d-08, 0.5819591585302d+01, 0.2670964694522d+02,$
  0.3884040326665d-08, 0.5980934960428d+01, 0.6660449441528d+01,$
  $
  0.2922027539886d-08, 0.3337290282483d+01, 0.1375773836557d+01,$
  0.4110846382042d-08, 0.5742978187327d+01, 0.4480965020977d+02,$
  0.2934508411032d-08, 0.2278075804200d+01, 0.6408777551755d+00,$
  0.3966896193000d-08, 0.5835747858477d+01, 0.3773735910827d+00,$
  0.3286695827610d-08, 0.5838898193902d+01, 0.3932462625300d-02,$
  0.3720643094196d-08, 0.1122212337858d+01, 0.1646033343740d+02,$
  0.3285508906174d-08, 0.9182250996416d+00, 0.1081813534213d+02,$
  0.3753880575973d-08, 0.5174761973266d+01, 0.5642198095270d+01,$
  0.3022129385587d-08, 0.3381611020639d+01, 0.2982630633589d+02,$
  0.2798569205621d-08, 0.3546193723922d+01, 0.1937891852345d+02,$
  $
  0.3397872070505d-08, 0.4533203197934d+01, 0.6923953605621d+01,$
  0.3708099772977d-08, 0.2756168198616d+01, 0.3066615496545d+02,$
  0.3599283541510d-08, 0.1934395469918d+01, 0.6147450479709d+01,$
  0.3688702753059d-08, 0.7149920971109d+00, 0.2636725487657d+01,$
  0.2681084724003d-08, 0.4899819493154d+01, 0.6816289982179d+01,$
  0.3495993460759d-08, 0.1572418915115d+01, 0.6418701221183d+01,$
  0.3130770324995d-08, 0.8912190180489d+00, 0.1235996607578d+02,$
  0.2744353821941d-08, 0.3800821940055d+01, 0.2059724391010d+02,$
  0.2842732906341d-08, 0.2644717440029d+01, 0.2828699048865d+02,$
  0.3046882682154d-08, 0.3987793020179d+01, 0.6055599646783d+01,$
  $
  0.2399072455143d-08, 0.9908826440764d+00, 0.6255674361143d+01,$
  0.2384306274204d-08, 0.2516149752220d+01, 0.6310477339748d+01,$
  0.2977324500559d-08, 0.5849195642118d+01, 0.1652265972112d+02,$
  0.3062835258972d-08, 0.1681660100162d+01, 0.1172006883645d+02,$
  0.3109682589231d-08, 0.5804143987737d+00, 0.2751146787858d+02,$
  0.2903920355299d-08, 0.5800768280123d+01, 0.6510552054109d+01,$
  0.2823221989212d-08, 0.9241118370216d+00, 0.5469525544182d+01,$
  0.3187949696649d-08, 0.3139776445735d+01, 0.1693792562116d+03,$
  0.2922559771655d-08, 0.3549440782984d+01, 0.2630839062450d+00,$
  0.2436302066603d-08, 0.4735540696319d+01, 0.3946258593675d+00,$
  $
  0.3049473043606d-08, 0.4998289124561d+01, 0.8390110365991d+01,$
  0.2863682575784d-08, 0.6709515671102d+00, 0.2243449970715d+00,$
  0.2641750517966d-08, 0.5410978257284d+01, 0.2986433403208d+02,$
  0.2704093466243d-08, 0.4778317207821d+01, 0.6129297044991d+01,$
  0.2445522177011d-08, 0.6009020662222d+01, 0.1171295538178d+02,$
  0.2623608810230d-08, 0.5010449777147d+01, 0.6436854655901d+01,$
  0.2079259704053d-08, 0.5980943768809d+01, 0.2019909489111d+02,$
  0.2820225596771d-08, 0.2679965110468d+01, 0.5934151399930d+01,$
  0.2365221950927d-08, 0.1894231148810d+01, 0.2470570524223d+02,$
  0.2359682077149d-08, 0.4220752950780d+01, 0.8671969964381d+01,$
  $
  0.2387577137206d-08, 0.2571783940617d+01, 0.7096626156709d+01,$
  0.1982102089816d-08, 0.5169765997119d+00, 0.1727188400790d+02,$
  0.2687502389925d-08, 0.6239078264579d+01, 0.7075506709219d+02,$
  0.2207751669135d-08, 0.2031184412677d+01, 0.4377611041777d+01,$
  0.2618370214274d-08, 0.8266079985979d+00, 0.6632000300961d+01,$
  0.2591951887361d-08, 0.8819350522008d+00, 0.4873985990671d+02,$
  0.2375055656248d-08, 0.3520944177789d+01, 0.1590676413561d+02,$
  0.2472019978911d-08, 0.1551431908671d+01, 0.6612329252343d+00,$
  0.2368157127199d-08, 0.4178610147412d+01, 0.3459636466239d+02,$
  0.1764846605693d-08, 0.1506764000157d+01, 0.1980094587212d+02,$
  $
  0.2291769608798d-08, 0.2118250611782d+01, 0.2844914056730d-01,$
  0.2209997316943d-08, 0.3363255261678d+01, 0.2666070658668d+00,$
  0.2292699097923d-08, 0.4200423956460d+00, 0.1484170571900d-02,$
  0.1629683015329d-08, 0.2331362582487d+01, 0.3035599730800d+02,$
  0.2206492862426d-08, 0.3400274026992d+01, 0.6281667977667d+01,$
  0.2205746568257d-08, 0.1066051230724d+00, 0.6284483723224d+01,$
  0.2026310767991d-08, 0.2779066487979d+01, 0.2449240616245d+02,$
  0.1762977622163d-08, 0.9951450691840d+00, 0.2045286941806d+02,$
  0.1368535049606d-08, 0.6402447365817d+00, 0.2473415438279d+02,$
  0.1720598775450d-08, 0.2303524214705d+00, 0.1679593901136d+03,$
  $
  0.1702429015449d-08, 0.6164622655048d+01, 0.3338575901272d+03,$
  0.1414033197685d-08, 0.3954561185580d+01, 0.1624205518357d+03,$
  0.1573768958043d-08, 0.2028286308984d+01, 0.3144167757552d+02,$
  0.1650705184447d-08, 0.2304040666128d+01, 0.5267006960365d+02,$
  0.1651087618855d-08, 0.2538461057280d+01, 0.8956999012000d+02,$
  0.1616409518983d-08, 0.5111054348152d+01, 0.3332657872986d+02,$
  0.1537175173581d-08, 0.5601130666603d+01, 0.3852657435933d+02,$
  0.1593191980553d-08, 0.2614340453411d+01, 0.2282781046519d+03,$
  0.1499480170643d-08, 0.3624721577264d+01, 0.2823723341956d+02,$
  0.1493807843235d-08, 0.4214569879008d+01, 0.2876692439167d+02,$
  $
  0.1074571199328d-08, 0.1496911744704d+00, 0.8397383534231d+02,$
  0.1074406983417d-08, 0.1187817671922d+01, 0.8401985929482d+02,$
  0.9757576855851d-09, 0.2655703035858d+01, 0.7826370942180d+02,$
  0.1258432887565d-08, 0.4969896184844d+01, 0.3115650189215d+03,$
  0.1240336343282d-08, 0.5192460776926d+01, 0.1784300471910d+03,$
  0.9016107005164d-09, 0.1960356923057d+01, 0.5886454391678d+02,$
  0.1135392360918d-08, 0.5082427809068d+01, 0.7842370451713d+02,$
  0.9216046089565d-09, 0.2793775037273d+01, 0.1014262087719d+03,$
  0.1061276615030d-08, 0.3726144311409d+01, 0.5660027930059d+02,$
  0.1010110596263d-08, 0.7404080708937d+00, 0.4245678405627d+02,$
$
  0.7217424756199d-09, 0.2697449980577d-01, 0.2457074661053d+03,$
  0.6912003846756d-09, 0.4253296276335d+01, 0.1679936946371d+03,$
  0.6871814664847d-09, 0.5148072412354d+01, 0.6053048899753d+02,$
  0.4887158016343d-09, 0.2153581148294d+01, 0.9656299901946d+02,$
  0.5161802866314d-09, 0.3852750634351d+01, 0.2442876000072d+03,$
  0.5652599559057d-09, 0.1233233356270d+01, 0.8365903305582d+02,$
  0.4710812608586d-09, 0.5610486976767d+01, 0.3164282286739d+03,$
  0.4909977500324d-09, 0.1639629524123d+01, 0.4059982187939d+03,$
  0.4772641839378d-09, 0.3737100368583d+01, 0.1805255418145d+03,$
  0.4487562567153d-09, 0.1158417054478d+00, 0.8433466158131d+02,$
  $
  0.3943441230497d-09, 0.6243502862796d+00, 0.2568537517081d+03,$
  0.3952236913598d-09, 0.3510377382385d+01, 0.2449975330562d+03,$
  0.3788898363417d-09, 0.5916128302299d+01, 0.1568131045107d+03,$
  0.3738329328831d-09, 0.1042266763456d+01, 0.3948519331910d+03,$
  0.2451199165151d-09, 0.1166788435700d+01, 0.1435713242844d+03,$
  0.2436734402904d-09, 0.3254726114901d+01, 0.2268582385539d+03,$
  0.2213605274325d-09, 0.1687210598530d+01, 0.1658638954901d+03,$
  0.1491521204829d-09, 0.2657541786794d+01, 0.2219950288015d+03,$
  0.1474995329744d-09, 0.5013089805819d+01, 0.3052819430710d+03,$
  0.1661939475656d-09, 0.5495315428418d+01, 0.2526661704812d+03,$
  
  0.9015946748003d-10, 0.2236989966505d+01, 0.4171445043968d+03 ];
  
  
  ;-------------------------------------------------------------
  ;-------------------------------------------------------------
  ;-------------------------------------------------------------
  ;-------------------------------------------------------------
  ;/* Sun-to-Earth, T^0, Z */
  e0z =  [ $
  0.2796207639075d-05, 0.3198701560209d+01, 0.8433466158131d+02,$
  0.1016042198142d-05, 0.5422360395913d+01, 0.5507553240374d+01,$
  0.8044305033647d-06, 0.3880222866652d+01, 0.5223693906222d+01,$
  0.4385347909274d-06, 0.3704369937468d+01, 0.2352866153506d+01,$
  0.3186156414906d-06, 0.3999639363235d+01, 0.1577343543434d+01,$
  0.2272412285792d-06, 0.3984738315952d+01, 0.1047747311755d+01,$
  0.1645620103007d-06, 0.3565412516841d+01, 0.5856477690889d+01,$
  0.1815836921166d-06, 0.4984507059020d+01, 0.6283075850446d+01,$
  0.1447461676364d-06, 0.3702753570108d+01, 0.9437762937313d+01,$
  0.1430760876382d-06, 0.3409658712357d+01, 0.1021328554739d+02,$
  $
  0.1120445753226d-06, 0.4829561570246d+01, 0.1414349524433d+02,$
  0.1090232840797d-06, 0.2080729178066d+01, 0.6812766822558d+01,$
  0.9715727346551d-07, 0.3476295881948d+01, 0.4694002934110d+01,$
  0.1036267136217d-06, 0.4056639536648d+01, 0.7109288135493d+02,$
  0.8752665271340d-07, 0.4448159519911d+01, 0.5753384878334d+01,$
  0.8331864956004d-07, 0.4991704044208d+01, 0.7084896783808d+01,$
  0.6901658670245d-07, 0.4325358994219d+01, 0.6275962395778d+01,$
  0.9144536848998d-07, 0.1141826375363d+01, 0.6620890113188d+01,$
  0.7205085037435d-07, 0.3624344170143d+01, 0.5296909721118d+00,$
  0.7697874654176d-07, 0.5554257458998d+01, 0.1676215758509d+03,$
  $
  0.5197545738384d-07, 0.6251760961735d+01, 0.1807370494127d+02,$
  0.5031345378608d-07, 0.2497341091913d+01, 0.4705732307012d+01,$
  0.4527110205840d-07, 0.2335079920992d+01, 0.6309374173736d+01,$
  0.4753355798089d-07, 0.7094148987474d+00, 0.5884926831456d+01,$
  0.4296951977516d-07, 0.1101916352091d+01, 0.6681224869435d+01,$
  0.3855341568387d-07, 0.1825495405486d+01, 0.5486777812467d+01,$
  0.5253930970990d-07, 0.4424740687208d+01, 0.7860419393880d+01,$
  0.4024630496471d-07, 0.5120498157053d+01, 0.1336797263425d+02,$
  0.4061069791453d-07, 0.6029771435451d+01, 0.3930209696940d+01,$
  0.3797883804205d-07, 0.4435193600836d+00, 0.3154687086868d+01,$
  $
  0.2933033225587d-07, 0.5124157356507d+01, 0.1059381944224d+01,$
  0.3503000930426d-07, 0.5421830162065d+01, 0.6069776770667d+01,$
  0.3670096214050d-07, 0.4582101667297d+01, 0.1219403291462d+02,$
  0.2905609437008d-07, 0.1926566420072d+01, 0.1097707878456d+02,$
  0.2466827821713d-07, 0.6090174539834d+00, 0.6496374930224d+01,$
  0.2691647295332d-07, 0.1393432595077d+01, 0.2200391463820d+02,$
  0.2150554667946d-07, 0.4308671715951d+01, 0.5643178611111d+01,$
  0.2237481922680d-07, 0.8133968269414d+00, 0.8635942003952d+01,$
  0.1817741038157d-07, 0.3755205127454d+01, 0.3340612434717d+01,$
  0.2227820762132d-07, 0.2759558596664d+01, 0.1203646072878d+02,$
  $
  0.1944713772307d-07, 0.5699645869121d+01, 0.1179062909082d+02,$
  0.1527340520662d-07, 0.1986749091746d+01, 0.3981490189893d+00,$
  0.1577282574914d-07, 0.3205017217983d+01, 0.5088628793478d+01,$
  0.1424738825424d-07, 0.6256747903666d+01, 0.2544314396739d+01,$
  0.1616563121701d-07, 0.2601671259394d+00, 0.1729818233119d+02,$
  0.1401210391692d-07, 0.4686939173506d+01, 0.7058598460518d+01,$
  0.1488726974214d-07, 0.2815862451372d+01, 0.2593412433514d+02,$
  0.1692626442388d-07, 0.4956894109797d+01, 0.1564752902480d+03,$
  0.1123571582910d-07, 0.2381192697696d+01, 0.3738761453707d+01,$
  0.9903308606317d-08, 0.4294851657684d+01, 0.9225539266174d+01,$
  $
  0.9174533187191d-08, 0.3075171510642d+01, 0.4164311961999d+01,$
  0.8645985631457d-08, 0.5477534821633d+00, 0.8429241228195d+01,$
  -0.1085876492688d-07, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.9264309077815d-08, 0.5968571670097d+01, 0.7079373888424d+01,$
  0.8243116984954d-08, 0.1489098777643d+01, 0.1044738781244d+02,$
  0.8268102113708d-08, 0.3512977691983d+01, 0.1150676975667d+02,$
  0.9043613988227d-08, 0.1290704408221d+00, 0.1101510648075d+02,$
  0.7432912038789d-08, 0.1991086893337d+01, 0.2608790314060d+02,$
  0.8586233727285d-08, 0.4238357924414d+01, 0.2986433403208d+02,$
  0.7612230060131d-08, 0.2911090150166d+01, 0.4732030630302d+01,$
  $
  0.7097787751408d-08, 0.1908938392390d+01, 0.8031092209206d+01,$
  0.7640237040175d-08, 0.6129219000168d+00, 0.7962980379786d+00,$
  0.7070445688081d-08, 0.1380417036651d+01, 0.2146165377750d+01,$
  0.7690770957702d-08, 0.1680504249084d+01, 0.2122839202813d+02,$
  0.8051292542594d-08, 0.5127423484511d+01, 0.2942463415728d+01,$
  0.5902709104515d-08, 0.2020274190917d+01, 0.7755226100720d+00,$
  0.5134567496462d-08, 0.2606778676418d+01, 0.1256615170089d+02,$
  0.5525802046102d-08, 0.1613011769663d+01, 0.8018209333619d+00,$
  0.5880724784221d-08, 0.4604483417236d+01, 0.4690479774488d+01,$
  0.5211699081370d-08, 0.5718964114193d+01, 0.8827390247185d+01,$
  $
  0.4891849573562d-08, 0.3689658932196d+01, 0.2132990797783d+00,$
  0.5150246069997d-08, 0.4099769855122d+01, 0.6480980550449d+02,$
  0.5102434319633d-08, 0.5660834602509d+01, 0.3379454372902d+02,$
  0.5083405254252d-08, 0.9842221218974d+00, 0.4136910472696d+01,$
  0.4206562585682d-08, 0.1341363634163d+00, 0.3128388763578d+01,$
  0.4663249683579d-08, 0.8130132735866d+00, 0.5216580451554d+01,$
  0.4099474416530d-08, 0.5791497770644d+01, 0.4265981595566d+00,$
  0.4628251220767d-08, 0.1249802769331d+01, 0.1572083878776d+02,$
  0.5024068728142d-08, 0.4795684802743d+01, 0.6290189305114d+01,$
  0.5120234327758d-08, 0.3810420387208d+01, 0.5230807360890d+01,$
  $
  0.5524029815280d-08, 0.1029264714351d+01, 0.2397622045175d+03,$
  0.4757415718860d-08, 0.3528044781779d+01, 0.1649636139783d+02,$
  0.3915786131127d-08, 0.5593889282646d+01, 0.1589072916335d+01,$
  0.4869053149991d-08, 0.3299636454433d+01, 0.7632943190217d+01,$
  0.3649365703729d-08, 0.1286049002584d+01, 0.6206810014183d+01,$
  0.3992493949002d-08, 0.3100307589464d+01, 0.2515860172507d+02,$
  0.3320247477418d-08, 0.6212683940807d+01, 0.1216800268190d+02,$
  0.3287123739696d-08, 0.4699118445928d+01, 0.7234794171227d+01,$
  0.3472776811103d-08, 0.2630507142004d+01, 0.7342457794669d+01,$
  0.3423253294767d-08, 0.2946432844305d+01, 0.9623688285163d+01,$
  $
  0.3896173898244d-08, 0.1224834179264d+01, 0.6438496133249d+01,$
  0.3388455337924d-08, 0.1543807616351d+01, 0.1494531617769d+02,$
  0.3062704716523d-08, 0.1191777572310d+01, 0.8662240327241d+01,$
  0.3270075600400d-08, 0.5483498767737d+01, 0.1194447056968d+01,$
  0.3101209215259d-08, 0.8000833804348d+00, 0.3772475342596d+02,$
  0.2780883347311d-08, 0.4077980721888d+00, 0.5863591145557d+01,$
  0.2903605931824d-08, 0.2617490302147d+01, 0.1965104848470d+02,$
  0.2682014743119d-08, 0.2634703158290d+01, 0.7238675589263d+01,$
  0.2534360108492d-08, 0.6102446114873d+01, 0.6836645152238d+01,$
  0.2392564882509d-08, 0.3681820208691d+01, 0.5849364236221d+01,$
  $
  0.2656667254856d-08, 0.6216045388886d+01, 0.6133512519065d+01,$
  0.2331242096773d-08, 0.5864949777744d+01, 0.4535059491685d+01,$
  0.2287898363668d-08, 0.4566628532802d+01, 0.7477522907414d+01,$
  0.2336944521306d-08, 0.2442722126930d+01, 0.1137170464392d+02,$
  0.3156632236269d-08, 0.1626628050682d+01, 0.2509084901204d+03,$
  0.2982612402766d-08, 0.2803604512609d+01, 0.1748016358760d+01,$
  0.2774031674807d-08, 0.4654002897158d+01, 0.8223916695780d+02,$
  0.2295236548638d-08, 0.4326518333253d+01, 0.3378142627421d+00,$
  0.2190714699873d-08, 0.4519614578328d+01, 0.2908881142201d+02,$
  0.2191495845045d-08, 0.3012626912549d+01, 0.1673046366289d+02,$
  $
  0.2492901628386d-08, 0.1290101424052d+00, 0.1543797956245d+03,$
  0.1993778064319d-08, 0.3864046799414d+01, 0.1778984560711d+02,$
  0.1898146479022d-08, 0.5053777235891d+01, 0.2042657109477d+02,$
  0.1918280127634d-08, 0.2222470192548d+01, 0.4165496312290d+02,$
  0.1916351061607d-08, 0.8719067257774d+00, 0.7737595720538d+02,$
  0.1834720181466d-08, 0.4031491098040d+01, 0.2358125818164d+02,$
  0.1249201523806d-08, 0.5938379466835d+01, 0.3301902111895d+02,$
  0.1477304050539d-08, 0.6544722606797d+00, 0.9548094718417d+02,$
  0.1264316431249d-08, 0.2059072853236d+01, 0.8399684731857d+02,$
  0.1203526495039d-08, 0.3644813532605d+01, 0.4558517281984d+02,$
  $
  0.9221681059831d-09, 0.3241815055602d+01, 0.7805158573086d+02,$
  0.7849278367646d-09, 0.5043812342457d+01, 0.5217580628120d+02,$
  0.7983392077387d-09, 0.5000024502753d+01, 0.1501922143975d+03,$
  0.7925395431654d-09, 0.1398734871821d-01, 0.9061773743175d+02,$
  0.7640473285886d-09, 0.5067111723130d+01, 0.4951538251678d+02,$
  0.5398937754482d-09, 0.5597382200075d+01, 0.1613385000004d+03,$
  0.5626247550193d-09, 0.2601338209422d+01, 0.7318837597844d+02,$
  0.5525197197855d-09, 0.5814832109256d+01, 0.1432335100216d+03,$
  0.5407629837898d-09, 0.3384820609076d+01, 0.3230491187871d+03,$
  0.3856739119801d-09, 0.1072391840473d+01, 0.2334791286671d+03,$
  $
  0.3856425239987d-09, 0.2369540393327d+01, 0.1739046517013d+03,$
  0.4350867755983d-09, 0.5255575751082d+01, 0.1620484330494d+03,$
  0.3844113924996d-09, 0.5482356246182d+01, 0.9757644180768d+02,$
  0.2854869155431d-09, 0.9573634763143d+00, 0.1697170704744d+03,$
  0.1719227671416d-09, 0.1887203025202d+01, 0.2265204242912d+03,$
  0.1527846879755d-09, 0.3982183931157d+01, 0.3341954043900d+03,$
  0.1128229264847d-09, 0.2787457156298d+01, 0.3119028331842d+03 ];
  
  
       
  
 ; /* Sun-to-Earth, T^1, X */
  e1x =  [ $
  0.1234046326004d-05, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.5150068824701d-06, 0.6002664557501d+01, 0.1256615170089d+02,$
  0.1290743923245d-07, 0.5959437664199d+01, 0.1884922755134d+02,$
  0.1068615564952d-07, 0.2015529654209d+01, 0.6283075850446d+01,$
  0.2079619142538d-08, 0.1732960531432d+01, 0.6279552690824d+01,$
  0.2078009243969d-08, 0.4915604476996d+01, 0.6286599010068d+01,$
  0.6206330058856d-09, 0.3616457953824d+00, 0.4705732307012d+01,$
  0.5989335313746d-09, 0.3802607304474d+01, 0.6256777527156d+01,$
  0.5958495663840d-09, 0.2845866560031d+01, 0.6309374173736d+01,$
  0.4866923261539d-09, 0.5213203771824d+01, 0.7755226100720d+00,$
  $
  0.4267785823142d-09, 0.4368189727818d+00, 0.1059381944224d+01,$
  0.4610675141648d-09, 0.1837249181372d-01, 0.7860419393880d+01,$
  0.3626989993973d-09, 0.2161590545326d+01, 0.5753384878334d+01,$
  0.3563071194389d-09, 0.1452631954746d+01, 0.5884926831456d+01,$
  0.3557015642807d-09, 0.4470593393054d+01, 0.6812766822558d+01,$
  0.3210412089122d-09, 0.5195926078314d+01, 0.6681224869435d+01,$
  0.2875473577986d-09, 0.5916256610193d+01, 0.2513230340178d+02,$
  0.2842913681629d-09, 0.1149902426047d+01, 0.6127655567643d+01,$
  0.2751248215916d-09, 0.5502088574662d+01, 0.6438496133249d+01,$
  0.2481432881127d-09, 0.2921989846637d+01, 0.5486777812467d+01,$
  $
  0.2059885976560d-09, 0.3718070376585d+01, 0.7079373888424d+01,$
  0.2015522342591d-09, 0.5979395259740d+01, 0.6290189305114d+01,$
  0.1995364084253d-09, 0.6772087985494d+00, 0.6275962395778d+01,$
  0.1957436436943d-09, 0.2899210654665d+01, 0.5507553240374d+01,$
  0.1651609818948d-09, 0.6228206482192d+01, 0.1150676975667d+02,$
  0.1822980550699d-09, 0.1469348746179d+01, 0.1179062909082d+02,$
  0.1675223159760d-09, 0.3813910555688d+01, 0.7058598460518d+01,$
  0.1706491764745d-09, 0.3004380506684d+00, 0.7113454667900d-02,$
  0.1392952362615d-09, 0.1440393973406d+01, 0.7962980379786d+00,$
  0.1209868266342d-09, 0.4150425791727d+01, 0.4694002934110d+01,$
  $
  0.1009827202611d-09, 0.3290040429843d+01, 0.3738761453707d+01,$
  0.1047261388602d-09, 0.4229590090227d+01, 0.6282095334605d+01,$
  0.1047006652004d-09, 0.2418967680575d+01, 0.6284056366286d+01,$
  0.9609993143095d-10, 0.4627943659201d+01, 0.6069776770667d+01,$
  0.9590900593873d-10, 0.1894393939924d+01, 0.4136910472696d+01,$
  0.9146249188071d-10, 0.2010647519562d+01, 0.6496374930224d+01,$
  0.8545274480290d-10, 0.5529846956226d-01, 0.1194447056968d+01,$
  0.8224377881194d-10, 0.1254304102174d+01, 0.1589072916335d+01,$
  0.6183529510410d-10, 0.3360862168815d+01, 0.8827390247185d+01,$
  0.6259255147141d-10, 0.4755628243179d+01, 0.8429241228195d+01,$
  $
  0.5539291694151d-10, 0.5371746955142d+01, 0.4933208510675d+01,$
  0.7328259466314d-10, 0.4927699613906d+00, 0.4535059491685d+01,$
  0.6017835843560d-10, 0.5776682001734d-01, 0.1255903824622d+02,$
  0.7079827775243d-10, 0.4395059432251d+01, 0.5088628793478d+01,$
  0.5170358878213d-10, 0.5154062619954d+01, 0.1176985366291d+02,$
  0.4872301838682d-10, 0.6289611648973d+00, 0.6040347114260d+01,$
  0.5249869411058d-10, 0.5617272046949d+01, 0.3154687086868d+01,$
  0.4716172354411d-10, 0.3965901800877d+01, 0.5331357529664d+01,$
  0.4871214940964d-10, 0.4627507050093d+01, 0.1256967486051d+02,$
  0.4598076850751d-10, 0.6023631226459d+01, 0.6525804586632d+01,$
  $
  0.4562196089485d-10, 0.4138562084068d+01, 0.3930209696940d+01,$
  0.4325493872224d-10, 0.1330845906564d+01, 0.7632943190217d+01,$
  0.5673781176748d-10, 0.2558752615657d+01, 0.5729506548653d+01,$
  0.3961436642503d-10, 0.2728071734630d+01, 0.7234794171227d+01,$
  0.5101868209058d-10, 0.4113444965144d+01, 0.6836645152238d+01,$
  0.5257043167676d-10, 0.6195089830590d+01, 0.8031092209206d+01,$
  0.5076613989393d-10, 0.2305124132918d+01, 0.7477522907414d+01,$
  0.3342169352778d-10, 0.5415998155071d+01, 0.1097707878456d+02,$
  0.3545881983591d-10, 0.3727160564574d+01, 0.4164311961999d+01,$
  0.3364063738599d-10, 0.2901121049204d+00, 0.1137170464392d+02,$
  $
  0.3357039670776d-10, 0.1652229354331d+01, 0.5223693906222d+01,$
  0.4307412268687d-10, 0.4938909587445d+01, 0.1592596075957d+01,$
  0.3405769115435d-10, 0.2408890766511d+01, 0.3128388763578d+01,$
  0.3001926198480d-10, 0.4862239006386d+01, 0.1748016358760d+01,$
  0.2778264787325d-10, 0.5241168661353d+01, 0.7342457794669d+01,$
  0.2676159480666d-10, 0.3423593942199d+01, 0.2146165377750d+01,$
  0.2954273399939d-10, 0.1881721265406d+01, 0.5368044267797d+00,$
  0.3309362888795d-10, 0.1931525677349d+01, 0.8018209333619d+00,$
  0.2810283608438d-10, 0.2414659495050d+01, 0.5225775174439d+00,$
  0.3378045637764d-10, 0.4238019163430d+01, 0.1554202828031d+00,$
  $
  0.2558134979840d-10, 0.1828225235805d+01, 0.5230807360890d+01,$
  0.2273755578447d-10, 0.5858184283998d+01, 0.7084896783808d+01,$
  0.2294176037690d-10, 0.4514589779057d+01, 0.1726015463500d+02,$
  0.2533506099435d-10, 0.2355717851551d+01, 0.5216580451554d+01,$
  0.2716685375812d-10, 0.2221003625100d+01, 0.8635942003952d+01,$
  0.2419043435198d-10, 0.5955704951635d+01, 0.4690479774488d+01,$
  0.2521232544812d-10, 0.1395676848521d+01, 0.5481254917084d+01,$
  0.2630195021491d-10, 0.5727468918743d+01, 0.2629832328990d-01,$
  0.2548395840944d-10, 0.2628351859400d-03, 0.1349867339771d+01 ];

    
  
;  /* Sun-to-Earth, T^1, Y */
  e1y =  [ $
  0.9304690546528d-06, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.5150715570663d-06, 0.4431807116294d+01, 0.1256615170089d+02,$
  0.1290825411056d-07, 0.4388610039678d+01, 0.1884922755134d+02,$
  0.4645466665386d-08, 0.5827263376034d+01, 0.6283075850446d+01,$
  0.2079625310718d-08, 0.1621698662282d+00, 0.6279552690824d+01,$
  0.2078189850907d-08, 0.3344713435140d+01, 0.6286599010068d+01,$
  0.6207190138027d-09, 0.5074049319576d+01, 0.4705732307012d+01,$
  0.5989826532569d-09, 0.2231842216620d+01, 0.6256777527156d+01,$
  0.5961360812618d-09, 0.1274975769045d+01, 0.6309374173736d+01,$
  0.4874165471016d-09, 0.3642277426779d+01, 0.7755226100720d+00,$
  $
  0.4283834034360d-09, 0.5148765510106d+01, 0.1059381944224d+01,$
  0.4652389287529d-09, 0.4715794792175d+01, 0.7860419393880d+01,$
  0.3751707476401d-09, 0.6617207370325d+00, 0.5753384878334d+01,$
  0.3559998806198d-09, 0.6155548875404d+01, 0.5884926831456d+01,$
  0.3558447558857d-09, 0.2898827297664d+01, 0.6812766822558d+01,$
  0.3211116927106d-09, 0.3625813502509d+01, 0.6681224869435d+01,$
  0.2875609914672d-09, 0.4345435813134d+01, 0.2513230340178d+02,$
  0.2843109704069d-09, 0.5862263940038d+01, 0.6127655567643d+01,$
  0.2744676468427d-09, 0.3926419475089d+01, 0.6438496133249d+01,$
  0.2481285237789d-09, 0.1351976572828d+01, 0.5486777812467d+01,$
  $
  0.2060338481033d-09, 0.2147556998591d+01, 0.7079373888424d+01,$
  0.2015822358331d-09, 0.4408358972216d+01, 0.6290189305114d+01,$
  0.2001195944195d-09, 0.5385829822531d+01, 0.6275962395778d+01,$
  0.1953667642377d-09, 0.1304933746120d+01, 0.5507553240374d+01,$
  0.1839744078713d-09, 0.6173567228835d+01, 0.1179062909082d+02,$
  0.1643334294845d-09, 0.4635942997523d+01, 0.1150676975667d+02,$
  0.1768051018652d-09, 0.5086283558874d+01, 0.7113454667900d-02,$
  0.1674874205489d-09, 0.2243332137241d+01, 0.7058598460518d+01,$
  0.1421445397609d-09, 0.6186899771515d+01, 0.7962980379786d+00,$
  0.1255163958267d-09, 0.5730238465658d+01, 0.4694002934110d+01,$
  $
  0.1013945281961d-09, 0.1726055228402d+01, 0.3738761453707d+01,$
  0.1047294335852d-09, 0.2658801228129d+01, 0.6282095334605d+01,$
  0.1047103879392d-09, 0.8481047835035d+00, 0.6284056366286d+01,$
  0.9530343962826d-10, 0.3079267149859d+01, 0.6069776770667d+01,$
  0.9604637611690d-10, 0.3258679792918d+00, 0.4136910472696d+01,$
  0.9153518537177d-10, 0.4398599886584d+00, 0.6496374930224d+01,$
  0.8562458214922d-10, 0.4772686794145d+01, 0.1194447056968d+01,$
  0.8232525360654d-10, 0.5966220721679d+01, 0.1589072916335d+01,$
  0.6150223411438d-10, 0.1780985591923d+01, 0.8827390247185d+01,$
  0.6272087858000d-10, 0.3184305429012d+01, 0.8429241228195d+01,$
  $
  0.5540476311040d-10, 0.3801260595433d+01, 0.4933208510675d+01,$
  0.7331901699361d-10, 0.5205948591865d+01, 0.4535059491685d+01,$
  0.6018528702791d-10, 0.4770139083623d+01, 0.1255903824622d+02,$
  0.5150530724804d-10, 0.3574796899585d+01, 0.1176985366291d+02,$
  0.6471933741811d-10, 0.2679787266521d+01, 0.5088628793478d+01,$
  0.5317460644174d-10, 0.9528763345494d+00, 0.3154687086868d+01,$
  0.4832187748783d-10, 0.5329322498232d+01, 0.6040347114260d+01,$
  0.4716763555110d-10, 0.2395235316466d+01, 0.5331357529664d+01,$
  0.4871509139861d-10, 0.3056663648823d+01, 0.1256967486051d+02,$
  0.4598417696768d-10, 0.4452762609019d+01, 0.6525804586632d+01,$
  $
  0.5674189533175d-10, 0.9879680872193d+00, 0.5729506548653d+01,$
  0.4073560328195d-10, 0.5939127696986d+01, 0.7632943190217d+01,$
  0.5040994945359d-10, 0.4549875824510d+01, 0.8031092209206d+01,$
  0.5078185134679d-10, 0.7346659893982d+00, 0.7477522907414d+01,$
  0.3769343537061d-10, 0.1071317188367d+01, 0.7234794171227d+01,$
  0.4980331365299d-10, 0.2500345341784d+01, 0.6836645152238d+01,$
  0.3458236594757d-10, 0.3825159450711d+01, 0.1097707878456d+02,$
  0.3578859493602d-10, 0.5299664791549d+01, 0.4164311961999d+01,$
  0.3370504646419d-10, 0.5002316301593d+01, 0.1137170464392d+02,$
  0.3299873338428d-10, 0.2526123275282d+01, 0.3930209696940d+01,$
  $
  0.4304917318409d-10, 0.3368078557132d+01, 0.1592596075957d+01,$
  0.3402418753455d-10, 0.8385495425800d+00, 0.3128388763578d+01,$
  0.2778460572146d-10, 0.3669905203240d+01, 0.7342457794669d+01,$
  0.2782710128902d-10, 0.2691664812170d+00, 0.1748016358760d+01,$
  0.2711725179646d-10, 0.4707487217718d+01, 0.5296909721118d+00,$
  0.2981760946340d-10, 0.3190260867816d+00, 0.5368044267797d+00,$
  0.2811672977772d-10, 0.3196532315372d+01, 0.7084896783808d+01,$
  0.2863454474467d-10, 0.2263240324780d+00, 0.5223693906222d+01,$
  0.3333464634051d-10, 0.3498451685065d+01, 0.8018209333619d+00,$
  0.3312991747609d-10, 0.5839154477412d+01, 0.1554202828031d+00,$
  $
  0.2813255564006d-10, 0.8268044346621d+00, 0.5225775174439d+00,$
  0.2665098083966d-10, 0.3934021725360d+01, 0.5216580451554d+01,$
  0.2349795705216d-10, 0.5197620913779d+01, 0.2146165377750d+01,$
  0.2330352293961d-10, 0.2984999231807d+01, 0.1726015463500d+02,$
  0.2728001683419d-10, 0.6521679638544d+00, 0.8635942003952d+01,$
  0.2484061007669d-10, 0.3468955561097d+01, 0.5230807360890d+01,$
  0.2646328768427d-10, 0.1013724533516d+01, 0.2629832328990d-01,$
  0.2518630264831d-10, 0.6108081057122d+01, 0.5481254917084d+01,$
  0.2421901455384d-10, 0.1651097776260d+01, 0.1349867339771d+01,$
  0.6348533267831d-11, 0.3220226560321d+01, 0.8433466158131d+02 ];
  
  
  
  
;  /* Sun-to-Earth, T^1, Z */
 e1z =  [ $
  0.2278290449966d-05, 0.3413716033863d+01, 0.6283075850446d+01,$
  0.5429458209830d-07, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.1903240492525d-07, 0.3370592358297d+01, 0.1256615170089d+02,$
  0.2385409276743d-09, 0.3327914718416d+01, 0.1884922755134d+02,$
  0.8676928342573d-10, 0.1824006811264d+01, 0.5223693906222d+01,$
  0.7765442593544d-10, 0.3888564279247d+01, 0.5507553240374d+01,$
  0.7066158332715d-10, 0.5194267231944d+01, 0.2352866153506d+01,$
  0.7092175288657d-10, 0.2333246960021d+01, 0.8399684731857d+02,$
  0.5357582213535d-10, 0.2224031176619d+01, 0.5296909721118d+00,$
  0.3828035865021d-10, 0.2156710933584d+01, 0.6279552690824d+01,$
  $
  0.3824857220427d-10, 0.1529755219915d+01, 0.6286599010068d+01,$
  0.3286995181628d-10, 0.4879512900483d+01, 0.1021328554739d+02 ];
  
    
;  /* Sun-to-Earth, T^2, X */
 e2x =  [ $
  -0.4143818297913d-10, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.2171497694435d-10, 0.4398225628264d+01, 0.1256615170089d+02,$
  0.9845398442516d-11, 0.2079720838384d+00, 0.6283075850446d+01,$
  0.9256833552682d-12, 0.4191264694361d+01, 0.1884922755134d+02,$
  0.1022049384115d-12, 0.5381133195658d+01, 0.8399684731857d+02 ];
  
;  /* Sun-to-Earth, T^2, Y */
e2y =  [ $
  0.5063375872532d-10, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.2173815785980d-10, 0.2827805833053d+01, 0.1256615170089d+02,$
  0.1010231999920d-10, 0.4634612377133d+01, 0.6283075850446d+01,$
  0.9259745317636d-12, 0.2620612076189d+01, 0.1884922755134d+02,$
  0.1022202095812d-12, 0.3809562326066d+01, 0.8399684731857d+02 ];
  
;  /* Sun-to-Earth, T^2, Z */
e2z =  [ $
  0.9722666114891d-10, 0.5152219582658d+01, 0.6283075850446d+01,$
  -0.3494819171909d-11, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.6713034376076d-12, 0.6440188750495d+00, 0.1256615170089d+02 ];
  
  

  
 ; /* SSB-to-Sun, T^0, X */
 s0x =  [ $
  0.4956757536410d-02, 0.3741073751789d+01, 0.5296909721118d+00,$
  0.2718490072522d-02, 0.4016011511425d+01, 0.2132990797783d+00,$
  0.1546493974344d-02, 0.2170528330642d+01, 0.3813291813120d-01,$
  0.8366855276341d-03, 0.2339614075294d+01, 0.7478166569050d-01,$
  0.2936777942117d-03, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.1201317439469d-03, 0.4090736353305d+01, 0.1059381944224d+01,$
  0.7578550887230d-04, 0.3241518088140d+01, 0.4265981595566d+00,$
  0.1941787367773d-04, 0.1012202064330d+01, 0.2061856251104d+00,$
  0.1889227765991d-04, 0.3892520416440d+01, 0.2204125344462d+00,$
  0.1937896968613d-04, 0.4797779441161d+01, 0.1495633313810d+00,$
  $
  0.1434506110873d-04, 0.3868960697933d+01, 0.5225775174439d+00,$
  0.1406659911580d-04, 0.4759766557397d+00, 0.5368044267797d+00,$
  0.1179022300202d-04, 0.7774961520598d+00, 0.7626583626240d-01,$
  0.8085864460959d-05, 0.3254654471465d+01, 0.3664874755930d-01,$
  0.7622752967615d-05, 0.4227633103489d+01, 0.3961708870310d-01,$
  0.6209171139066d-05, 0.2791828325711d+00, 0.7329749511860d-01,$
  0.4366435633970d-05, 0.4440454875925d+01, 0.1589072916335d+01,$
  0.3792124889348d-05, 0.5156393842356d+01, 0.7113454667900d-02,$
  0.3154548963402d-05, 0.6157005730093d+01, 0.4194847048887d+00,$
  0.3088359882942d-05, 0.2494567553163d+01, 0.6398972393349d+00,$
  $
  0.2788440902136d-05, 0.4934318747989d+01, 0.1102062672231d+00,$
  0.3039928456376d-05, 0.4895077702640d+01, 0.6283075850446d+01,$
  0.2272258457679d-05, 0.5278394064764d+01, 0.1030928125552d+00,$
  0.2162007057957d-05, 0.5802978019099d+01, 0.3163918923335d+00,$
  0.1767632855737d-05, 0.3415346595193d-01, 0.1021328554739d+02,$
  0.1349413459362d-05, 0.2001643230755d+01, 0.1484170571900d-02,$
  0.1170141900476d-05, 0.2424750491620d+01, 0.6327837846670d+00,$
  0.1054355266820d-05, 0.3123311487576d+01, 0.4337116142245d+00,$
  0.9800822461610d-06, 0.3026258088130d+01, 0.1052268489556d+01,$
  0.1091203749931d-05, 0.3157811670347d+01, 0.1162474756779d+01,$
  $
  0.6960236715913d-06, 0.8219570542313d+00, 0.1066495398892d+01,$
  0.5689257296909d-06, 0.1323052375236d+01, 0.9491756770005d+00,$
  0.6613172135802d-06, 0.2765348881598d+00, 0.8460828644453d+00,$
  0.6277702517571d-06, 0.5794064466382d+01, 0.1480791608091d+00,$
  0.6304884066699d-06, 0.7323555380787d+00, 0.2243449970715d+00,$
  0.4897850467382d-06, 0.3062464235399d+01, 0.3340612434717d+01,$
  0.3759148598786d-06, 0.4588290469664d+01, 0.3516457698740d-01,$
  0.3110520548195d-06, 0.1374299536572d+01, 0.6373574839730d-01,$
  0.3064708359780d-06, 0.4222267485047d+01, 0.1104591729320d-01,$
  0.2856347168241d-06, 0.3714202944973d+01, 0.1510475019529d+00,$
  $
  0.2840945514288d-06, 0.2847972875882d+01, 0.4110125927500d-01,$
  0.2378951599405d-06, 0.3762072563388d+01, 0.2275259891141d+00,$
  0.2714229481417d-06, 0.1036049980031d+01, 0.2535050500000d-01,$
  0.2323551717307d-06, 0.4682388599076d+00, 0.8582758298370d-01,$
  0.1881790512219d-06, 0.4790565425418d+01, 0.2118763888447d+01,$
  0.2261353968371d-06, 0.1669144912212d+01, 0.7181332454670d-01,$
  0.2214546389848d-06, 0.3937717281614d+01, 0.2968341143800d-02,$
  0.2184915594933d-06, 0.1129169845099d+00, 0.7775000683430d-01,$
  0.2000164937936d-06, 0.4030009638488d+01, 0.2093666171530d+00,$
  0.1966105136719d-06, 0.8745955786834d+00, 0.2172315424036d+00,$
  $
  0.1904742332624d-06, 0.5919743598964d+01, 0.2022531624851d+00,$
  0.1657399705031d-06, 0.2549141484884d+01, 0.7358765972222d+00,$
  0.1574070533987d-06, 0.5277533020230d+01, 0.7429900518901d+00,$
  0.1832261651039d-06, 0.3064688127777d+01, 0.3235053470014d+00,$
  0.1733615346569d-06, 0.3011432799094d+01, 0.1385174140878d+00,$
  0.1549124014496d-06, 0.4005569132359d+01, 0.5154640627760d+00,$
  0.1637044713838d-06, 0.1831375966632d+01, 0.8531963191132d+00,$
  0.1123420082383d-06, 0.1180270407578d+01, 0.1990721704425d+00,$
  0.1083754165740d-06, 0.3414101320863d+00, 0.5439178814476d+00,$
  0.1156638012655d-06, 0.6130479452594d+00, 0.5257585094865d+00,$
  $
  0.1142548785134d-06, 0.3724761948846d+01, 0.5336234347371d+00,$
  0.7921463895965d-07, 0.2435425589361d+01, 0.1478866649112d+01,$
  0.7428600285231d-07, 0.3542144398753d+01, 0.2164800718209d+00,$
  0.8323211246747d-07, 0.3525058072354d+01, 0.1692165728891d+01,$
  0.7257595116312d-07, 0.1364299431982d+01, 0.2101180877357d+00,$
  0.7111185833236d-07, 0.2460478875808d+01, 0.4155522422634d+00,$
  0.6868090383716d-07, 0.4397327670704d+01, 0.1173197218910d+00,$
  0.7226419974175d-07, 0.4042647308905d+01, 0.1265567569334d+01,$
  0.6955642383177d-07, 0.2865047906085d+01, 0.9562891316684d+00,$
  0.7492139296331d-07, 0.5014278994215d+01, 0.1422690933580d-01,$
  $
  0.6598363128857d-07, 0.2376730020492d+01, 0.6470106940028d+00,$
  0.7381147293385d-07, 0.3272990384244d+01, 0.1581959461667d+01,$
  0.6402909624032d-07, 0.5302290955138d+01, 0.9597935788730d-01,$
  0.6237454263857d-07, 0.5444144425332d+01, 0.7084920306520d-01,$
  0.5241198544016d-07, 0.4215359579205d+01, 0.5265099800692d+00,$
  0.5144463853918d-07, 0.1218916689916d+00, 0.5328719641544d+00,$
  0.5868164772299d-07, 0.2369402002213d+01, 0.7871412831580d-01,$
  0.6233195669151d-07, 0.1254922242403d+01, 0.2608790314060d+02,$
  0.6068463791422d-07, 0.5679713760431d+01, 0.1114304132498d+00,$
  0.4359361135065d-07, 0.6097219641646d+00, 0.1375773836557d+01,$
  $
  0.4686510366826d-07, 0.4786231041431d+01, 0.1143987543936d+00,$
  0.3758977287225d-07, 0.1167368068139d+01, 0.1596186371003d+01,$
  0.4282051974778d-07, 0.1519471064319d+01, 0.2770348281756d+00,$
  0.5153765386113d-07, 0.1860532322984d+01, 0.2228608264996d+00,$
  0.4575129387188d-07, 0.7632857887158d+00, 0.1465949902372d+00,$
  0.3326844933286d-07, 0.1298219485285d+01, 0.5070101000000d-01,$
  0.3748617450984d-07, 0.1046510321062d+01, 0.4903339079539d+00,$
  0.2816756661499d-07, 0.3434522346190d+01, 0.2991266627620d+00,$
  0.3412750405039d-07, 0.2523766270318d+01, 0.3518164938661d+00,$
  0.2655796761776d-07, 0.2904422260194d+01, 0.6256703299991d+00,$
  $
  0.2963597929458d-07, 0.5923900431149d+00, 0.1099462426779d+00,$
  0.2539523734781d-07, 0.4851947722567d+01, 0.1256615170089d+02,$
  0.2283087914139d-07, 0.3400498595496d+01, 0.6681224869435d+01,$
  0.2321309799331d-07, 0.5789099148673d+01, 0.3368040641550d-01,$
  0.2549657649750d-07, 0.3991856479792d-01, 0.1169588211447d+01,$
  0.2290462303977d-07, 0.2788567577052d+01, 0.1045155034888d+01,$
  0.1945398522914d-07, 0.3290896998176d+01, 0.1155361302111d+01,$
  0.1849171512638d-07, 0.2698060129367d+01, 0.4452511715700d-02,$
  0.1647199834254d-07, 0.3016735644085d+01, 0.4408250688924d+00,$
  0.1529530765273d-07, 0.5573043116178d+01, 0.6521991896920d-01,$
  $
  0.1433199339978d-07, 0.1481192356147d+01, 0.9420622223326d+00,$
  0.1729134193602d-07, 0.1422817538933d+01, 0.2108507877249d+00,$
  0.1716463931346d-07, 0.3469468901855d+01, 0.2157473718317d+00,$
  0.1391206061378d-07, 0.6122436220547d+01, 0.4123712502208d+00,$
  0.1404746661924d-07, 0.1647765641936d+01, 0.4258542984690d-01,$
  0.1410452399455d-07, 0.5989729161964d+01, 0.2258291676434d+00,$
  0.1089828772168d-07, 0.2833705509371d+01, 0.4226656969313d+00,$
  0.1047374564948d-07, 0.5090690007331d+00, 0.3092784376656d+00,$
  0.1358279126532d-07, 0.5128990262836d+01, 0.7923417740620d-01,$
  0.1020456476148d-07, 0.9632772880808d+00, 0.1456308687557d+00,$
  $
  0.1033428735328d-07, 0.3223779318418d+01, 0.1795258541446d+01,$
  0.1412435841540d-07, 0.2410271572721d+01, 0.1525316725248d+00,$
  0.9722759371574d-08, 0.2333531395690d+01, 0.8434341241180d-01,$
  0.9657334084704d-08, 0.6199270974168d+01, 0.1272681024002d+01,$
  0.1083641148690d-07, 0.2864222292929d+01, 0.7032915397480d-01,$
  0.1067318403838d-07, 0.5833458866568d+00, 0.2123349582968d+00,$
  0.1062366201976d-07, 0.4307753989494d+01, 0.2142632012598d+00,$
  0.1236364149266d-07, 0.2873917870593d+01, 0.1847279083684d+00,$
  0.1092759489593d-07, 0.2959887266733d+01, 0.1370332435159d+00,$
  0.8912069362899d-08, 0.5141213702562d+01, 0.2648454860559d+01,$
  $
  0.9656467707970d-08, 0.4532182462323d+01, 0.4376440768498d+00,$
  0.8098386150135d-08, 0.2268906338379d+01, 0.2880807454688d+00,$
  0.7857714675000d-08, 0.4055544260745d+01, 0.2037373330570d+00,$
  0.7288455940646d-08, 0.5357901655142d+01, 0.1129145838217d+00,$
  0.9450595950552d-08, 0.4264926963939d+01, 0.5272426800584d+00,$
  0.9381718247537d-08, 0.7489366976576d-01, 0.5321392641652d+00,$
  0.7079052646038d-08, 0.1923311052874d+01, 0.6288513220417d+00,$
  0.9259004415344d-08, 0.2970256853438d+01, 0.1606092486742d+00,$
  0.8259801499742d-08, 0.3327056314697d+01, 0.8389694097774d+00,$
  0.6476334355779d-08, 0.2954925505727d+01, 0.2008557621224d+01,$
  $
  0.5984021492007d-08, 0.9138753105829d+00, 0.2042657109477d+02,$
  0.5989546863181d-08, 0.3244464082031d+01, 0.2111650433779d+01,$
  0.6233108606023d-08, 0.4995232638403d+00, 0.4305306221819d+00,$
  0.6877299149965d-08, 0.2834987233449d+01, 0.9561746721300d-02,$
  0.8311234227190d-08, 0.2202951835758d+01, 0.3801276407308d+00,$
  0.6599472832414d-08, 0.4478581462618d+01, 0.1063314406849d+01,$
  0.6160491096549d-08, 0.5145858696411d+01, 0.1368660381889d+01,$
  0.6164772043891d-08, 0.3762976697911d+00, 0.4234171675140d+00,$
  0.6363248684450d-08, 0.3162246718685d+01, 0.1253008786510d-01,$
  0.6448587520999d-08, 0.3442693302119d+01, 0.5287268506303d+00,$
  $
  0.6431662283977d-08, 0.8977549136606d+00, 0.5306550935933d+00,$
  0.6351223158474d-08, 0.4306447410369d+01, 0.5217580628120d+02,$
  0.5476721393451d-08, 0.3888529177855d+01, 0.2221856701002d+01,$
  0.5341772572619d-08, 0.2655560662512d+01, 0.7466759693650d-01,$
  0.5337055758302d-08, 0.5164990735946d+01, 0.7489573444450d-01,$
  0.5373120816787d-08, 0.6041214553456d+01, 0.1274714967946d+00,$
  0.5392351705426d-08, 0.9177763485932d+00, 0.1055449481598d+01,$
  0.6688495850205d-08, 0.3089608126937d+01, 0.2213766559277d+00,$
  0.5072003660362d-08, 0.4311316541553d+01, 0.2132517061319d+00,$
  0.5070726650455d-08, 0.5790675464444d+00, 0.2133464534247d+00,$
  $
  0.5658012950032d-08, 0.2703945510675d+01, 0.7287631425543d+00,$
  0.4835509924854d-08, 0.2975422976065d+01, 0.7160067364790d-01,$
  0.6479821978012d-08, 0.1324168733114d+01, 0.2209183458640d-01,$
  0.6230636494980d-08, 0.2860103632836d+01, 0.3306188016693d+00,$
  0.4649239516213d-08, 0.4832259763403d+01, 0.7796265773310d-01,$
  0.6487325792700d-08, 0.2726165825042d+01, 0.3884652414254d+00,$
  0.4682823682770d-08, 0.6966602455408d+00, 0.1073608853559d+01,$
  0.5704230804976d-08, 0.5669634104606d+01, 0.8731175355560d-01,$
  0.6125413585489d-08, 0.1513386538915d+01, 0.7605151500000d-01,$
  0.6035825038187d-08, 0.1983509168227d+01, 0.9846002785331d+00,$
  $
  0.4331123462303d-08, 0.2782892992807d+01, 0.4297791515992d+00,$
  0.4681107685143d-08, 0.5337232886836d+01, 0.2127790306879d+00,$
  0.4669105829655d-08, 0.5837133792160d+01, 0.2138191288687d+00,$
  0.5138823602365d-08, 0.3080560200507d+01, 0.7233337363710d-01,$
  0.4615856664534d-08, 0.1661747897471d+01, 0.8603097737811d+00,$
  0.4496916702197d-08, 0.2112508027068d+01, 0.7381754420900d-01,$
  0.4278479042945d-08, 0.5716528462627d+01, 0.7574578717200d-01,$
  0.3840525503932d-08, 0.6424172726492d+00, 0.3407705765729d+00,$
  0.4866636509685d-08, 0.4919244697715d+01, 0.7722995774390d-01,$
  0.3526100639296d-08, 0.2550821052734d+01, 0.6225157782540d-01,$
  $
  0.3939558488075d-08, 0.3939331491710d+01, 0.5268983110410d-01,$
  0.4041268772576d-08, 0.2275337571218d+01, 0.3503323232942d+00,$
  0.3948761842853d-08, 0.1999324200790d+01, 0.1451108196653d+00,$
  0.3258394550029d-08, 0.9121001378200d+00, 0.5296435984654d+00,$
  0.3257897048761d-08, 0.3428428660869d+01, 0.5297383457582d+00,$
  0.3842559031298d-08, 0.6132927720035d+01, 0.9098186128426d+00,$
  0.3109920095448d-08, 0.7693650193003d+00, 0.3932462625300d-02,$
  0.3132237775119d-08, 0.3621293854908d+01, 0.2346394437820d+00,$
  0.3942189421510d-08, 0.4841863659733d+01, 0.3180992042600d-02,$
  0.3796972285340d-08, 0.1814174994268d+01, 0.1862120789403d+00,$
  $
  0.3995640233688d-08, 0.1386990406091d+01, 0.4549093064213d+00,$
  0.2875013727414d-08, 0.9178318587177d+00, 0.1905464808669d+01,$
  0.3073719932844d-08, 0.2688923811835d+01, 0.3628624111593d+00,$
  0.2731016580075d-08, 0.1188259127584d+01, 0.2131850110243d+00,$
  0.2729549896546d-08, 0.3702160634273d+01, 0.2134131485323d+00,$
  0.3339372892449d-08, 0.7199163960331d+00, 0.2007689919132d+00,$
  0.2898833764204d-08, 0.1916709364999d+01, 0.5291709230214d+00,$
  0.2894536549362d-08, 0.2424043195547d+01, 0.5302110212022d+00,$
  0.3096872473843d-08, 0.4445894977497d+01, 0.2976424921901d+00,$
  0.2635672326810d-08, 0.3814366984117d+01, 0.1485980103780d+01,$
  $
  0.3649302697001d-08, 0.2924200596084d+01, 0.6044726378023d+00,$
  0.3127954585895d-08, 0.1842251648327d+01, 0.1084620721060d+00,$
  0.2616040173947d-08, 0.4155841921984d+01, 0.1258454114666d+01,$
  0.2597395859860d-08, 0.1158045978874d+00, 0.2103781122809d+00,$
  0.2593286172210d-08, 0.4771850408691d+01, 0.2162200472757d+00,$
  0.2481823585747d-08, 0.4608842558889d+00, 0.1062562936266d+01,$
  0.2742219550725d-08, 0.1538781127028d+01, 0.5651155736444d+00,$
  0.3199558469610d-08, 0.3226647822878d+00, 0.7036329877322d+00,$
  0.2666088542957d-08, 0.1967991731219d+00, 0.1400015846597d+00,$
  0.2397067430580d-08, 0.3707036669873d+01, 0.2125476091956d+00,$
  $
  0.2376570772738d-08, 0.1182086628042d+01, 0.2140505503610d+00,$
  0.2547228007887d-08, 0.4906256820629d+01, 0.1534957940063d+00,$
  0.2265575594114d-08, 0.3414949866857d+01, 0.2235935264888d+00,$
  0.2464381430585d-08, 0.4599122275378d+01, 0.2091065926078d+00,$
  0.2433408527044d-08, 0.2830751145445d+00, 0.2174915669488d+00,$
  0.2443605509076d-08, 0.4212046432538d+01, 0.1739420156204d+00,$
  0.2319779262465d-08, 0.9881978408630d+00, 0.7530171478090d-01,$
  0.2284622835465d-08, 0.5565347331588d+00, 0.7426161660010d-01,$
  0.2467268750783d-08, 0.5655708150766d+00, 0.2526561439362d+00,$
  0.2808513492782d-08, 0.1418405053408d+01, 0.5636314030725d+00,$
  $ 
  0.2329528932532d-08, 0.4069557545675d+01, 0.1056200952181d+01,$
  0.9698639532817d-09, 0.1074134313634d+01, 0.7826370942180d+02 ];
  
  
  

  
  
 ; /* SSB-to-Sun, T^0, Y */
 s0y =  [ $
  0.4955392320126d-02, 0.2170467313679d+01, 0.5296909721118d+00,$
  0.2722325167392d-02, 0.2444433682196d+01, 0.2132990797783d+00,$
  0.1546579925346d-02, 0.5992779281546d+00, 0.3813291813120d-01,$
  0.8363140252966d-03, 0.7687356310801d+00, 0.7478166569050d-01,$
  0.3385792683603d-03, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.1201192221613d-03, 0.2520035601514d+01, 0.1059381944224d+01,$
  0.7587125720554d-04, 0.1669954006449d+01, 0.4265981595566d+00,$
  0.1964155361250d-04, 0.5707743963343d+01, 0.2061856251104d+00,$
  0.1891900364909d-04, 0.2320960679937d+01, 0.2204125344462d+00,$
  0.1937373433356d-04, 0.3226940689555d+01, 0.1495633313810d+00,$
  $
  0.1437139941351d-04, 0.2301626908096d+01, 0.5225775174439d+00,$
  0.1406267683099d-04, 0.5188579265542d+01, 0.5368044267797d+00,$
  0.1178703080346d-04, 0.5489483248476d+01, 0.7626583626240d-01,$
  0.8079835186041d-05, 0.1683751835264d+01, 0.3664874755930d-01,$
  0.7623253594652d-05, 0.2656400462961d+01, 0.3961708870310d-01,$
  0.6248667483971d-05, 0.4992775362055d+01, 0.7329749511860d-01,$
  0.4366353695038d-05, 0.2869706279678d+01, 0.1589072916335d+01,$
  0.3829101568895d-05, 0.3572131359950d+01, 0.7113454667900d-02,$
  0.3175733773908d-05, 0.4535372530045d+01, 0.4194847048887d+00,$
  0.3092437902159d-05, 0.9230153317909d+00, 0.6398972393349d+00,$
  $
  0.2874168812154d-05, 0.3363143761101d+01, 0.1102062672231d+00,$
  0.3040119321826d-05, 0.3324250895675d+01, 0.6283075850446d+01,$
  0.2699723308006d-05, 0.2917882441928d+00, 0.1030928125552d+00,$
  0.2134832683534d-05, 0.4220997202487d+01, 0.3163918923335d+00,$
  0.1770412139433d-05, 0.4747318496462d+01, 0.1021328554739d+02,$
  0.1377264209373d-05, 0.4305058462401d+00, 0.1484170571900d-02,$
  0.1127814538960d-05, 0.8538177240740d+00, 0.6327837846670d+00,$
  0.1055608090130d-05, 0.1551800742580d+01, 0.4337116142245d+00,$
  0.9802673861420d-06, 0.1459646735377d+01, 0.1052268489556d+01,$
  0.1090329461951d-05, 0.1587351228711d+01, 0.1162474756779d+01,$
  $
  0.6959590025090d-06, 0.5534442628766d+01, 0.1066495398892d+01,$
  0.5664914529542d-06, 0.6030673003297d+01, 0.9491756770005d+00,$
  0.6607787763599d-06, 0.4989507233927d+01, 0.8460828644453d+00,$
  0.6269725742838d-06, 0.4222951804572d+01, 0.1480791608091d+00,$
  0.6301889697863d-06, 0.5444316669126d+01, 0.2243449970715d+00,$
  0.4891042662861d-06, 0.1490552839784d+01, 0.3340612434717d+01,$
  0.3457083123290d-06, 0.3030475486049d+01, 0.3516457698740d-01,$
  0.3032559967314d-06, 0.2652038793632d+01, 0.1104591729320d-01,$
  0.2841133988903d-06, 0.1276744786829d+01, 0.4110125927500d-01,$
  0.2855564444432d-06, 0.2143368674733d+01, 0.1510475019529d+00,$
  $
  0.2765157135038d-06, 0.5444186109077d+01, 0.6373574839730d-01,$
  0.2382312465034d-06, 0.2190521137593d+01, 0.2275259891141d+00,$
  0.2808060365077d-06, 0.5735195064841d+01, 0.2535050500000d-01,$
  0.2332175234405d-06, 0.9481985524859d-01, 0.7181332454670d-01,$
  0.2322488199659d-06, 0.5180499361533d+01, 0.8582758298370d-01,$
  0.1881850258423d-06, 0.3219788273885d+01, 0.2118763888447d+01,$
  0.2196111392808d-06, 0.2366941159761d+01, 0.2968341143800d-02,$
  0.2183810335519d-06, 0.4825445110915d+01, 0.7775000683430d-01,$
  0.2002733093326d-06, 0.2457148995307d+01, 0.2093666171530d+00,$
  0.1967111767229d-06, 0.5586291545459d+01, 0.2172315424036d+00,$
  $
  0.1568473250543d-06, 0.3708003123320d+01, 0.7429900518901d+00,$
  0.1852528314300d-06, 0.4310638151560d+01, 0.2022531624851d+00,$
  0.1832111226447d-06, 0.1494665322656d+01, 0.3235053470014d+00,$
  0.1746805502310d-06, 0.1451378500784d+01, 0.1385174140878d+00,$
  0.1555730966650d-06, 0.1068040418198d+01, 0.7358765972222d+00,$
  0.1554883462559d-06, 0.2442579035461d+01, 0.5154640627760d+00,$
  0.1638380568746d-06, 0.2597913420625d+00, 0.8531963191132d+00,$
  0.1159938593640d-06, 0.5834512021280d+01, 0.1990721704425d+00,$
  0.1083427965695d-06, 0.5054033177950d+01, 0.5439178814476d+00,$
  0.1156480369431d-06, 0.5325677432457d+01, 0.5257585094865d+00,$
  $
  0.1141308860095d-06, 0.2153403923857d+01, 0.5336234347371d+00,$
  0.7913146470946d-07, 0.8642846847027d+00, 0.1478866649112d+01,$
  0.7439752463733d-07, 0.1970628496213d+01, 0.2164800718209d+00,$
  0.7280277104079d-07, 0.6073307250609d+01, 0.2101180877357d+00,$
  0.8319567719136d-07, 0.1954371928334d+01, 0.1692165728891d+01,$
  0.7137705549290d-07, 0.8904989440909d+00, 0.4155522422634d+00,$
  0.6900825396225d-07, 0.2825717714977d+01, 0.1173197218910d+00,$
  0.7245757216635d-07, 0.2481677513331d+01, 0.1265567569334d+01,$
  0.6961165696255d-07, 0.1292955312978d+01, 0.9562891316684d+00,$
  0.7571804456890d-07, 0.3427517575069d+01, 0.1422690933580d-01,$
  $
  0.6605425721904d-07, 0.8052192701492d+00, 0.6470106940028d+00,$
  0.7375477357248d-07, 0.1705076390088d+01, 0.1581959461667d+01,$
  0.7041664951470d-07, 0.4848356967891d+00, 0.9597935788730d-01,$
  0.6322199535763d-07, 0.3878069473909d+01, 0.7084920306520d-01,$
  0.5244380279191d-07, 0.2645560544125d+01, 0.5265099800692d+00,$
  0.5143125704988d-07, 0.4834486101370d+01, 0.5328719641544d+00,$
  0.5871866319373d-07, 0.7981472548900d+00, 0.7871412831580d-01,$
  0.6300822573871d-07, 0.5979398788281d+01, 0.2608790314060d+02,$
  0.6062154271548d-07, 0.4108655402756d+01, 0.1114304132498d+00,$
  0.4361912339976d-07, 0.5322624319280d+01, 0.1375773836557d+01,$
  $
  0.4417005920067d-07, 0.6240817359284d+01, 0.2770348281756d+00,$
  0.4686806749936d-07, 0.3214977301156d+01, 0.1143987543936d+00,$
  0.3758892132305d-07, 0.5879809634765d+01, 0.1596186371003d+01,$
  0.5151351332319d-07, 0.2893377688007d+00, 0.2228608264996d+00,$
  0.4554683578572d-07, 0.5475427144122d+01, 0.1465949902372d+00,$
  0.3442381385338d-07, 0.5992034796640d+01, 0.5070101000000d-01,$
  0.2831093954933d-07, 0.5367350273914d+01, 0.3092784376656d+00,$
  0.3756267090084d-07, 0.5758171285420d+01, 0.4903339079539d+00,$
  0.2816374679892d-07, 0.1863718700923d+01, 0.2991266627620d+00,$
  0.3419307025569d-07, 0.9524347534130d+00, 0.3518164938661d+00,$
  $
  0.2904250494239d-07, 0.5304471615602d+01, 0.1099462426779d+00,$
  0.2471734511206d-07, 0.1297069793530d+01, 0.6256703299991d+00,$
  0.2539620831872d-07, 0.3281126083375d+01, 0.1256615170089d+02,$
  0.2281017868007d-07, 0.1829122133165d+01, 0.6681224869435d+01,$
  0.2275319473335d-07, 0.5797198160181d+01, 0.3932462625300d-02,$
  0.2547755368442d-07, 0.4752697708330d+01, 0.1169588211447d+01,$
  0.2285979669317d-07, 0.1223205292886d+01, 0.1045155034888d+01,$
  0.1913386560994d-07, 0.1757532993389d+01, 0.1155361302111d+01,$
  0.1809020525147d-07, 0.4246116108791d+01, 0.3368040641550d-01,$
  0.1649213300201d-07, 0.1445162890627d+01, 0.4408250688924d+00,$
  $
  0.1834972793932d-07, 0.1126917567225d+01, 0.4452511715700d-02,$
  0.1439550648138d-07, 0.6160756834764d+01, 0.9420622223326d+00,$
  0.1487645457041d-07, 0.4358761931792d+01, 0.4123712502208d+00,$
  0.1731729516660d-07, 0.6134456753344d+01, 0.2108507877249d+00,$
  0.1717747163567d-07, 0.1898186084455d+01, 0.2157473718317d+00,$
  0.1418190430374d-07, 0.4180286741266d+01, 0.6521991896920d-01,$
  0.1404844134873d-07, 0.7654053565412d-01, 0.4258542984690d-01,$
  0.1409842846538d-07, 0.4418612420312d+01, 0.2258291676434d+00,$
  0.1090948346291d-07, 0.1260615686131d+01, 0.4226656969313d+00,$
  0.1357577323612d-07, 0.3558248818690d+01, 0.7923417740620d-01,$
  $
  0.1018154061960d-07, 0.5676087241256d+01, 0.1456308687557d+00,$
  0.1412073972109d-07, 0.8394392632422d+00, 0.1525316725248d+00,$
  0.1030938326496d-07, 0.1653593274064d+01, 0.1795258541446d+01,$
  0.1180081567104d-07, 0.1285802592036d+01, 0.7032915397480d-01,$
  0.9708510575650d-08, 0.7631889488106d+00, 0.8434341241180d-01,$
  0.9637689663447d-08, 0.4630642649176d+01, 0.1272681024002d+01,$
  0.1068910429389d-07, 0.5294934032165d+01, 0.2123349582968d+00,$
  0.1063716179336d-07, 0.2736266800832d+01, 0.2142632012598d+00,$
  0.1234858713814d-07, 0.1302891146570d+01, 0.1847279083684d+00,$
  0.8912631189738d-08, 0.3570415993621d+01, 0.2648454860559d+01,$
  $
  0.1036378285534d-07, 0.4236693440949d+01, 0.1370332435159d+00,$
  0.9667798501561d-08, 0.2960768892398d+01, 0.4376440768498d+00,$
  0.8108314201902d-08, 0.6987781646841d+00, 0.2880807454688d+00,$
  0.7648364324628d-08, 0.2499017863863d+01, 0.2037373330570d+00,$
  0.7286136828406d-08, 0.3787426951665d+01, 0.1129145838217d+00,$
  0.9448237743913d-08, 0.2694354332983d+01, 0.5272426800584d+00,$
  0.9374276106428d-08, 0.4787121277064d+01, 0.5321392641652d+00,$
  0.7100226287462d-08, 0.3530238792101d+00, 0.6288513220417d+00,$
  0.9253056659571d-08, 0.1399478925664d+01, 0.1606092486742d+00,$
  0.6636432145504d-08, 0.3479575438447d+01, 0.1368660381889d+01,$
  $
  0.6469975312932d-08, 0.1383669964800d+01, 0.2008557621224d+01,$
  0.7335849729765d-08, 0.1243698166898d+01, 0.9561746721300d-02,$
  0.8743421205855d-08, 0.3776164289301d+01, 0.3801276407308d+00,$
  0.5993635744494d-08, 0.5627122113596d+01, 0.2042657109477d+02,$
  0.5981008479693d-08, 0.1674336636752d+01, 0.2111650433779d+01,$
  0.6188535145838d-08, 0.5214925208672d+01, 0.4305306221819d+00,$
  0.6596074017566d-08, 0.2907653268124d+01, 0.1063314406849d+01,$
  0.6630815126226d-08, 0.2127643669658d+01, 0.8389694097774d+00,$
  0.6156772830040d-08, 0.5082160803295d+01, 0.4234171675140d+00,$
  0.6446960563014d-08, 0.1872100916905d+01, 0.5287268506303d+00,$
  $
  0.6429324424668d-08, 0.5610276103577d+01, 0.5306550935933d+00,$
  0.6302232396465d-08, 0.1592152049607d+01, 0.1253008786510d-01,$
  0.6399244436159d-08, 0.2746214421532d+01, 0.5217580628120d+02,$
  0.5474965172558d-08, 0.2317666374383d+01, 0.2221856701002d+01,$
  0.5339293190692d-08, 0.1084724961156d+01, 0.7466759693650d-01,$
  0.5334733683389d-08, 0.3594106067745d+01, 0.7489573444450d-01,$
  0.5392665782110d-08, 0.5630254365606d+01, 0.1055449481598d+01,$
  0.6682075673789d-08, 0.1518480041732d+01, 0.2213766559277d+00,$
  0.5079130495960d-08, 0.2739765115711d+01, 0.2132517061319d+00,$
  0.5077759793261d-08, 0.5290711290094d+01, 0.2133464534247d+00,$
  $
  0.4832037368310d-08, 0.1404473217200d+01, 0.7160067364790d-01,$
  0.6463279674802d-08, 0.6038381695210d+01, 0.2209183458640d-01,$
  0.6240592771560d-08, 0.1290170653666d+01, 0.3306188016693d+00,$
  0.4672013521493d-08, 0.3261895939677d+01, 0.7796265773310d-01,$
  0.6500650750348d-08, 0.1154522312095d+01, 0.3884652414254d+00,$
  0.6344161389053d-08, 0.6206111545062d+01, 0.7605151500000d-01,$
  0.4682518370646d-08, 0.5409118796685d+01, 0.1073608853559d+01,$
  0.5329460015591d-08, 0.1202985784864d+01, 0.7287631425543d+00,$
  0.5701588675898d-08, 0.4098715257064d+01, 0.8731175355560d-01,$
  0.6030690867211d-08, 0.4132033218460d+00, 0.9846002785331d+00,$
  $
  0.4336256312655d-08, 0.1211415991827d+01, 0.4297791515992d+00,$
  0.4688498808975d-08, 0.3765479072409d+01, 0.2127790306879d+00,$
  0.4675578609335d-08, 0.4265540037226d+01, 0.2138191288687d+00,$
  0.4225578112158d-08, 0.5237566010676d+01, 0.3407705765729d+00,$
  0.5139422230028d-08, 0.1507173079513d+01, 0.7233337363710d-01,$
  0.4619995093571d-08, 0.9023957449848d-01, 0.8603097737811d+00,$
  0.4494776255461d-08, 0.5414930552139d+00, 0.7381754420900d-01,$
  0.4274026276788d-08, 0.4145735303659d+01, 0.7574578717200d-01,$
  0.5018141789353d-08, 0.3344408829055d+01, 0.3180992042600d-02,$
  0.4866163952181d-08, 0.3348534657607d+01, 0.7722995774390d-01,$
  $
  0.4111986020501d-08, 0.4198823597220d+00, 0.1451108196653d+00,$
  0.3356142784950d-08, 0.5609144747180d+01, 0.1274714967946d+00,$
  0.4070575554551d-08, 0.7028411059224d+00, 0.3503323232942d+00,$
  0.3257451857278d-08, 0.5624697983086d+01, 0.5296435984654d+00,$
  0.3256973703026d-08, 0.1857842076707d+01, 0.5297383457582d+00,$
  0.3830771508640d-08, 0.4562887279931d+01, 0.9098186128426d+00,$
  0.3725024005962d-08, 0.2358058692652d+00, 0.1084620721060d+00,$
  0.3136763921756d-08, 0.2049731526845d+01, 0.2346394437820d+00,$
  0.3795147256194d-08, 0.2432356296933d+00, 0.1862120789403d+00,$
  0.2877342229911d-08, 0.5631101279387d+01, 0.1905464808669d+01,$
  $
  0.3076931798805d-08, 0.1117615737392d+01, 0.3628624111593d+00,$
  0.2734765945273d-08, 0.5899826516955d+01, 0.2131850110243d+00,$
  0.2733405296885d-08, 0.2130562964070d+01, 0.2134131485323d+00,$
  0.2898552353410d-08, 0.3462387048225d+00, 0.5291709230214d+00,$
  0.2893736103681d-08, 0.8534352781543d+00, 0.5302110212022d+00,$
  0.3095717734137d-08, 0.2875061429041d+01, 0.2976424921901d+00,$
  0.2636190425832d-08, 0.2242512846659d+01, 0.1485980103780d+01,$
  0.3645512095537d-08, 0.1354016903958d+01, 0.6044726378023d+00,$
  0.2808173547723d-08, 0.6705114365631d-01, 0.6225157782540d-01,$
  0.2625012866888d-08, 0.4775705748482d+01, 0.5268983110410d-01,$
  $
  0.2572233995651d-08, 0.2638924216139d+01, 0.1258454114666d+01,$
  0.2604238824792d-08, 0.4826358927373d+01, 0.2103781122809d+00,$
  0.2596886385239d-08, 0.3200388483118d+01, 0.2162200472757d+00,$
  0.3228057304264d-08, 0.5384848409563d+01, 0.2007689919132d+00,$
  0.2481601798252d-08, 0.5173373487744d+01, 0.1062562936266d+01,$
  0.2745977498864d-08, 0.6250966149853d+01, 0.5651155736444d+00,$
  0.2669878833811d-08, 0.4906001352499d+01, 0.1400015846597d+00,$
  0.3203986611711d-08, 0.5034333010005d+01, 0.7036329877322d+00,$
  0.3354961227212d-08, 0.6108262423137d+01, 0.4549093064213d+00,$
  0.2400407324558d-08, 0.2135399294955d+01, 0.2125476091956d+00,$
  $
  0.2379905859802d-08, 0.5893721933961d+01, 0.2140505503610d+00,$
  0.2550844302187d-08, 0.3331940762063d+01, 0.1534957940063d+00,$
  0.2268824211001d-08, 0.1843418461035d+01, 0.2235935264888d+00,$
  0.2464700891204d-08, 0.3029548547230d+01, 0.2091065926078d+00,$
  0.2436814726024d-08, 0.4994717970364d+01, 0.2174915669488d+00,$
  0.2443623894745d-08, 0.2645102591375d+01, 0.1739420156204d+00,$
  0.2318701783838d-08, 0.5700547397897d+01, 0.7530171478090d-01,$
  0.2284448700256d-08, 0.5268898905872d+01, 0.7426161660010d-01,$
  0.2468848123510d-08, 0.5276280575078d+01, 0.2526561439362d+00,$
  0.2814052350303d-08, 0.6130168623475d+01, 0.5636314030725d+00,$
  $
  0.2243662755220d-08, 0.6631692457995d+00, 0.8886590321940d-01,$
  0.2330795855941d-08, 0.2499435487702d+01, 0.1056200952181d+01,$
  0.9757679038404d-09, 0.5796846023126d+01, 0.7826370942180d+02 ];
  
  

  
;  /* SSB-to-Sun, T^0, Z */
  s0z =  [ $
  0.1181255122986d-03, 0.4607918989164d+00, 0.2132990797783d+00,$
  0.1127777651095d-03, 0.4169146331296d+00, 0.5296909721118d+00,$
  0.4777754401806d-04, 0.4582657007130d+01, 0.3813291813120d-01,$
  0.1129354285772d-04, 0.5758735142480d+01, 0.7478166569050d-01,$
  -0.1149543637123d-04, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.3298730512306d-05, 0.5978801994625d+01, 0.4265981595566d+00,$
  0.2733376706079d-05, 0.7665413691040d+00, 0.1059381944224d+01,$
  0.9426389657270d-06, 0.3710201265838d+01, 0.2061856251104d+00,$
  0.8187517749552d-06, 0.3390675605802d+00, 0.2204125344462d+00,$
  0.4080447871819d-06, 0.4552296640088d+00, 0.5225775174439d+00,$
  $
  0.3169973017028d-06, 0.3445455899321d+01, 0.5368044267797d+00,$
  0.2438098615549d-06, 0.5664675150648d+01, 0.3664874755930d-01,$
  0.2601897517235d-06, 0.1931894095697d+01, 0.1495633313810d+00,$
  0.2314558080079d-06, 0.3666319115574d+00, 0.3961708870310d-01,$
  0.1962549548002d-06, 0.3167411699020d+01, 0.7626583626240d-01,$
  0.2180518287925d-06, 0.1544420746580d+01, 0.7113454667900d-02,$
  0.1451382442868d-06, 0.1583756740070d+01, 0.1102062672231d+00,$
  0.1358439007389d-06, 0.5239941758280d+01, 0.6398972393349d+00,$
  0.1050585898028d-06, 0.2266958352859d+01, 0.3163918923335d+00,$
  0.1050029870186d-06, 0.2711495250354d+01, 0.4194847048887d+00,$
  $
  0.9934920679800d-07, 0.1116208151396d+01, 0.1589072916335d+01,$
  0.1048395331560d-06, 0.3408619600206d+01, 0.1021328554739d+02,$
  0.8370147196668d-07, 0.3810459401087d+01, 0.2535050500000d-01,$
  0.7989856510998d-07, 0.3769910473647d+01, 0.7329749511860d-01,$
  0.5441221655233d-07, 0.2416994903374d+01, 0.1030928125552d+00,$
  0.4610812906784d-07, 0.5858503336994d+01, 0.4337116142245d+00,$
  0.3923022803444d-07, 0.3354170010125d+00, 0.1484170571900d-02,$
  0.2610725582128d-07, 0.5410600646324d+01, 0.6327837846670d+00,$
  0.2455279767721d-07, 0.6120216681403d+01, 0.1162474756779d+01,$
  0.2375530706525d-07, 0.6055443426143d+01, 0.1052268489556d+01,$
  $
  0.1782967577553d-07, 0.3146108708004d+01, 0.8460828644453d+00,$
  0.1581687095238d-07, 0.6255496089819d+00, 0.3340612434717d+01,$
  0.1594657672461d-07, 0.3782604300261d+01, 0.1066495398892d+01,$
  0.1563448615040d-07, 0.1997775733196d+01, 0.2022531624851d+00,$
  0.1463624258525d-07, 0.1736316792088d+00, 0.3516457698740d-01,$
  0.1331585056673d-07, 0.4331941830747d+01, 0.9491756770005d+00,$
  0.1130634557637d-07, 0.6152017751825d+01, 0.2968341143800d-02,$
  0.1028949607145d-07, 0.2101792614637d+00, 0.2275259891141d+00,$
  0.1024074971618d-07, 0.4071833211074d+01, 0.5070101000000d-01,$
  0.8826956060303d-08, 0.4861633688145d+00, 0.2093666171530d+00,$
  $
  0.8572230171541d-08, 0.5268190724302d+01, 0.4110125927500d-01,$
  0.7649332643544d-08, 0.5134543417106d+01, 0.2608790314060d+02,$
  0.8581673291033d-08, 0.2920218146681d+01, 0.1480791608091d+00,$
  0.8430589300938d-08, 0.3604576619108d+01, 0.2172315424036d+00,$
  0.7776165501012d-08, 0.3772942249792d+01, 0.6373574839730d-01,$
  0.8311070234408d-08, 0.6200412329888d+01, 0.3235053470014d+00,$
  0.6927365212582d-08, 0.4543353113437d+01, 0.8531963191132d+00,$
  0.6791574208598d-08, 0.2882188406238d+01, 0.7181332454670d-01,$
  0.5593100811839d-08, 0.1776646892780d+01, 0.7429900518901d+00,$
  0.4553381853021d-08, 0.3949617611240d+01, 0.7775000683430d-01,$
  $
  0.5758000450068d-08, 0.3859251775075d+01, 0.1990721704425d+00,$
  0.4281283457133d-08, 0.1466294631206d+01, 0.2118763888447d+01,$
  0.4206935661097d-08, 0.5421776011706d+01, 0.1104591729320d-01,$
  0.4213751641837d-08, 0.3412048993322d+01, 0.2243449970715d+00,$
  0.5310506239878d-08, 0.5421641370995d+00, 0.5154640627760d+00,$
  0.3827450341320d-08, 0.8887314524995d+00, 0.1510475019529d+00,$
  0.4292435241187d-08, 0.1405043757194d+01, 0.1422690933580d-01,$
  0.3189780702289d-08, 0.1060049293445d+01, 0.1173197218910d+00,$
  0.3226611928069d-08, 0.6270858897442d+01, 0.2164800718209d+00,$
  0.2893897608830d-08, 0.5117563223301d+01, 0.6470106940028d+00,$
  $
  0.3239852024578d-08, 0.4079092237983d+01, 0.2101180877357d+00,$
  0.2956892222200d-08, 0.1594917021704d+01, 0.3092784376656d+00,$
  0.2980177912437d-08, 0.5258787667564d+01, 0.4155522422634d+00,$
  0.3163725690776d-08, 0.3854589225479d+01, 0.8582758298370d-01,$
  0.2662262399118d-08, 0.3561326430187d+01, 0.5257585094865d+00,$
  0.2766689135729d-08, 0.3180732086830d+00, 0.1385174140878d+00,$
  0.2411600278464d-08, 0.3324798335058d+01, 0.5439178814476d+00,$
  0.2483527695131d-08, 0.4169069291947d+00, 0.5336234347371d+00,$
  0.7788777276590d-09, 0.1900569908215d+01, 0.5217580628120d+02 ];
  

  
 ; /* SSB-to-Sun, T^1, X */
  s1x =  [ $
  -0.1296310361520d-07, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.8975769009438d-08, 0.1128891609250d+01, 0.4265981595566d+00,$
  0.7771113441307d-08, 0.2706039877077d+01, 0.2061856251104d+00,$
  0.7538303866642d-08, 0.2191281289498d+01, 0.2204125344462d+00,$
  0.6061384579336d-08, 0.3248167319958d+01, 0.1059381944224d+01,$
  0.5726994235594d-08, 0.5569981398610d+01, 0.5225775174439d+00,$
  0.5616492836424d-08, 0.5057386614909d+01, 0.5368044267797d+00,$
  0.1010881584769d-08, 0.3473577116095d+01, 0.7113454667900d-02,$
  0.7259606157626d-09, 0.3651858593665d+00, 0.6398972393349d+00,$
  0.8755095026935d-09, 0.1662835408338d+01, 0.4194847048887d+00,$
  $
  0.5370491182812d-09, 0.1327673878077d+01, 0.4337116142245d+00,$
  0.5743773887665d-09, 0.4250200846687d+01, 0.2132990797783d+00,$
  0.4408103140300d-09, 0.3598752574277d+01, 0.1589072916335d+01,$
  0.3101892374445d-09, 0.4887822983319d+01, 0.1052268489556d+01,$
  0.3209453713578d-09, 0.9702272295114d+00, 0.5296909721118d+00,$
  0.3017228286064d-09, 0.5484462275949d+01, 0.1066495398892d+01,$
  0.3200700038601d-09, 0.2846613338643d+01, 0.1495633313810d+00,$
  0.2137637279911d-09, 0.5692163292729d+00, 0.3163918923335d+00,$
  0.1899686386727d-09, 0.2061077157189d+01, 0.2275259891141d+00,$
  0.1401994545308d-09, 0.4177771136967d+01, 0.1102062672231d+00,$
  $
  0.1578057810499d-09, 0.5782460597335d+01, 0.7626583626240d-01,$
  0.1237713253351d-09, 0.5705900866881d+01, 0.5154640627760d+00,$
  0.1313076837395d-09, 0.5163438179576d+01, 0.3664874755930d-01,$
  0.1184963304860d-09, 0.3054804427242d+01, 0.6327837846670d+00,$
  0.1238130878565d-09, 0.2317292575962d+01, 0.3961708870310d-01,$
  0.1015959527736d-09, 0.2194643645526d+01, 0.7329749511860d-01,$
  0.9017954423714d-10, 0.2868603545435d+01, 0.1990721704425d+00,$
  0.8668024955603d-10, 0.4923849675082d+01, 0.5439178814476d+00,$
  0.7756083930103d-10, 0.3014334135200d+01, 0.9491756770005d+00,$
  0.7536503401741d-10, 0.2704886279769d+01, 0.1030928125552d+00,$
  $
  0.5483308679332d-10, 0.6010983673799d+01, 0.8531963191132d+00,$
  0.5184339620428d-10, 0.1952704573291d+01, 0.2093666171530d+00,$
  0.5108658712030d-10, 0.2958575786649d+01, 0.2172315424036d+00,$
  0.5019424524650d-10, 0.1736317621318d+01, 0.2164800718209d+00,$
  0.4909312625978d-10, 0.3167216416257d+01, 0.2101180877357d+00,$
  0.4456638901107d-10, 0.7697579923471d+00, 0.3235053470014d+00,$
  0.4227030350925d-10, 0.3490910137928d+01, 0.6373574839730d-01,$
  0.4095456040093d-10, 0.5178888984491d+00, 0.6470106940028d+00,$
  0.4990537041422d-10, 0.3323887668974d+01, 0.1422690933580d-01,$
  0.4321170010845d-10, 0.4288484987118d+01, 0.7358765972222d+00,$
  $
  0.3544072091802d-10, 0.6021051579251d+01, 0.5265099800692d+00,$
  0.3480198638687d-10, 0.4600027054714d+01, 0.5328719641544d+00,$
  0.3440287244435d-10, 0.4349525970742d+01, 0.8582758298370d-01,$
  0.3330628322713d-10, 0.2347391505082d+01, 0.1104591729320d-01,$
  0.2973060707184d-10, 0.4789409286400d+01, 0.5257585094865d+00,$
  0.2932606766089d-10, 0.5831693799927d+01, 0.5336234347371d+00,$
  0.2876972310953d-10, 0.2692638514771d+01, 0.1173197218910d+00,$
  0.2827488278556d-10, 0.2056052487960d+01, 0.2022531624851d+00,$
  0.2515028239756d-10, 0.7411863262449d+00, 0.9597935788730d-01,$
  0.2853033744415d-10, 0.3948481024894d+01, 0.2118763888447d+01 ];
  
  

  
  
 ; /* SSB-to-Sun, T^1, Y */
 s1y =  [ $
  0.8989047573576d-08, 0.5840593672122d+01, 0.4265981595566d+00,$
  0.7815938401048d-08, 0.1129664707133d+01, 0.2061856251104d+00,$
  0.7550926713280d-08, 0.6196589104845d+00, 0.2204125344462d+00,$
  0.6056556925895d-08, 0.1677494667846d+01, 0.1059381944224d+01,$
  0.5734142698204d-08, 0.4000920852962d+01, 0.5225775174439d+00,$
  0.5614341822459d-08, 0.3486722577328d+01, 0.5368044267797d+00,$
  0.1028678147656d-08, 0.1877141024787d+01, 0.7113454667900d-02,$
  0.7270792075266d-09, 0.5077167301739d+01, 0.6398972393349d+00,$
  0.8734141726040d-09, 0.9069550282609d-01, 0.4194847048887d+00,$
  0.5377371402113d-09, 0.6039381844671d+01, 0.4337116142245d+00,$
  $
  0.4729719431571d-09, 0.2153086311760d+01, 0.2132990797783d+00,$
  0.4458052820973d-09, 0.5059830025565d+01, 0.5296909721118d+00,$
  0.4406855467908d-09, 0.2027971692630d+01, 0.1589072916335d+01,$
  0.3101659310977d-09, 0.3317677981860d+01, 0.1052268489556d+01,$
  0.3016749232545d-09, 0.3913703482532d+01, 0.1066495398892d+01,$
  0.3198541352656d-09, 0.1275513098525d+01, 0.1495633313810d+00,$
  0.2142065389871d-09, 0.5301351614597d+01, 0.3163918923335d+00,$
  0.1902615247592d-09, 0.4894943352736d+00, 0.2275259891141d+00,$
  0.1613410990871d-09, 0.2449891130437d+01, 0.1102062672231d+00,$
  0.1576992165097d-09, 0.4211421447633d+01, 0.7626583626240d-01,$
  $
  0.1241637259894d-09, 0.4140803368133d+01, 0.5154640627760d+00,$
  0.1313974830355d-09, 0.3591920305503d+01, 0.3664874755930d-01,$
  0.1181697118258d-09, 0.1506314382788d+01, 0.6327837846670d+00,$
  0.1238239742779d-09, 0.7461405378404d+00, 0.3961708870310d-01,$
  0.1010107068241d-09, 0.6271010795475d+00, 0.7329749511860d-01,$
  0.9226316616509d-10, 0.1259158839583d+01, 0.1990721704425d+00,$
  0.8664946419555d-10, 0.3353244696934d+01, 0.5439178814476d+00,$
  0.7757230468978d-10, 0.1447677295196d+01, 0.9491756770005d+00,$
  0.7693168628139d-10, 0.1120509896721d+01, 0.1030928125552d+00,$
  0.5487897454612d-10, 0.4439380426795d+01, 0.8531963191132d+00,$
  $
  0.5196118677218d-10, 0.3788856619137d+00, 0.2093666171530d+00,$
  0.5110853339935d-10, 0.1386879372016d+01, 0.2172315424036d+00,$
  0.5027804534813d-10, 0.1647881805466d+00, 0.2164800718209d+00,$
  0.4922485922674d-10, 0.1594315079862d+01, 0.2101180877357d+00,$
  0.6155599524400d-10, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.4447147832161d-10, 0.5480720918976d+01, 0.3235053470014d+00,$
  0.4144691276422d-10, 0.1931371033660d+01, 0.6373574839730d-01,$
  0.4099950625452d-10, 0.5229611294335d+01, 0.6470106940028d+00,$
  0.5060541682953d-10, 0.1731112486298d+01, 0.1422690933580d-01,$
  0.4293615946300d-10, 0.2714571038925d+01, 0.7358765972222d+00,$
  $
  0.3545659845763d-10, 0.4451041444634d+01, 0.5265099800692d+00,$
  0.3479112041196d-10, 0.3029385448081d+01, 0.5328719641544d+00,$
  0.3438516493570d-10, 0.2778507143731d+01, 0.8582758298370d-01,$
  0.3297341285033d-10, 0.7898709807584d+00, 0.1104591729320d-01,$
  0.2972585818015d-10, 0.3218785316973d+01, 0.5257585094865d+00,$
  0.2931707295017d-10, 0.4260731012098d+01, 0.5336234347371d+00,$
  0.2897198149403d-10, 0.1120753978101d+01, 0.1173197218910d+00,$
  0.2832293240878d-10, 0.4597682717827d+00, 0.2022531624851d+00,$
  0.2864348326612d-10, 0.2169939928448d+01, 0.9597935788730d-01,$
  0.2852714675471d-10, 0.2377659870578d+01, 0.2118763888447d+01 ];
  
 
  
  
 ; /* SSB-to-Sun, T^1, Z */
  s1z =  [ $
  0.5444220475678d-08, 0.1803825509310d+01, 0.2132990797783d+00,$
  0.3883412695596d-08, 0.4668616389392d+01, 0.5296909721118d+00,$
  0.1334341434551d-08, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.3730001266883d-09, 0.5401405918943d+01, 0.2061856251104d+00,$
  0.2894929197956d-09, 0.4932415609852d+01, 0.2204125344462d+00,$
  0.2857950357701d-09, 0.3154625362131d+01, 0.7478166569050d-01,$
  0.2499226432292d-09, 0.3657486128988d+01, 0.4265981595566d+00,$
  0.1937705443593d-09, 0.5740434679002d+01, 0.1059381944224d+01,$
  0.1374894396320d-09, 0.1712857366891d+01, 0.5368044267797d+00,$
  0.1217248678408d-09, 0.2312090870932d+01, 0.5225775174439d+00,$
  $
  0.7961052740870d-10, 0.5283368554163d+01, 0.3813291813120d-01,$
  0.4979225949689d-10, 0.4298290471860d+01, 0.4194847048887d+00,$
  0.4388552286597d-10, 0.6145515047406d+01, 0.7113454667900d-02,$
  0.2586835212560d-10, 0.3019448001809d+01, 0.6398972393349d+00 ];
  
;  /* SSB-to-Sun, T^2, X */
  s2x =  [ $
  0.1603551636587d-11, 0.4404109410481d+01, 0.2061856251104d+00,$
  0.1556935889384d-11, 0.4818040873603d+00, 0.2204125344462d+00,$
  0.1182594414915d-11, 0.9935762734472d+00, 0.5225775174439d+00,$
  0.1158794583180d-11, 0.3353180966450d+01, 0.5368044267797d+00,$
  0.9597358943932d-12, 0.5567045358298d+01, 0.2132990797783d+00,$
  0.6511516579605d-12, 0.5630872420788d+01, 0.4265981595566d+00,$
  0.7419792747688d-12, 0.2156188581957d+01, 0.5296909721118d+00,$
  0.3951972655848d-12, 0.1981022541805d+01, 0.1059381944224d+01,$
  0.4478223877045d-12, 0.0000000000000d+00, 0.0000000000000d+00 ];
  
;  /* SSB-to-Sun, T^2, Y */
 s2y =  [ $
  0.1609114495091d-11, 0.2831096993481d+01, 0.2061856251104d+00,$
  0.1560330784946d-11, 0.5193058213906d+01, 0.2204125344462d+00,$
  0.1183535479202d-11, 0.5707003443890d+01, 0.5225775174439d+00,$
  0.1158183066182d-11, 0.1782400404928d+01, 0.5368044267797d+00,$
  0.1032868027407d-11, 0.4036925452011d+01, 0.2132990797783d+00,$
  0.6540142847741d-12, 0.4058241056717d+01, 0.4265981595566d+00,$
  0.7305236491596d-12, 0.6175401942957d+00, 0.5296909721118d+00,$
  -0.5580725052968d-12, 0.0000000000000d+00, 0.0000000000000d+00,$
  0.3946122651015d-12, 0.4108265279171d+00, 0.1059381944224d+01 ];
  
 ; /* SSB-to-Sun, T^2, Z */
 s2z=  [ $
  0.3749920358054d-12, 0.3230285558668d+01, 0.2132990797783d+00,$
  0.2735037220939d-12, 0.6154322683046d+01, 0.5296909721118d+00 ];
  
  
  
  
 ; /* Pointers to coefficient arrays, in x,y,z sets */ 
  ce0 = [ ptr_new([e0x], /No_Copy),  ptr_new([e0y], /No_Copy),  ptr_new([e0z], /No_Copy) ];
  ce1 = [ ptr_new([e1x], /No_Copy), ptr_new([e1y], /No_Copy), ptr_new([e1z], /No_Copy)];
  ce2 = [ ptr_new([e2x], /No_Copy), ptr_new([e2y], /No_Copy), ptr_new([e2z], /No_Copy) ];
  cs0 = [ ptr_new([s0x], /No_Copy), ptr_new([s0y], /No_Copy), ptr_new([s0z], /No_Copy) ]
  cs1 = [ ptr_new([s1x], /No_Copy), ptr_new([s1y], /No_Copy), ptr_new([s1z], /No_Copy) ]
  cs2 = [ ptr_new([s2x], /No_Copy), ptr_new([s2y], /No_Copy), ptr_new([s2z], /No_Copy) ];
  coeffs=0d
  
  
;  /* Numbers of terms for each component of the model, in x,y,z sets */
   ne0 = [n_elements(e0x) /3 , n_elements(e0y)/3 , n_elements(e0z) /3 ]
   ne1 = [n_elements(e1x) /3 , n_elements(e1y) /3,n_elements(e1z) /3 ]
   ne2 = [n_elements(e2x) /3 , n_elements(e2y)/3 , n_elements(e2z) /3  ]
   ns0 = [n_elements(s0x) /3, n_elements(s0y) /3 ,n_elements(s0z) /3 ]
  ns1 = [n_elements(s1x) /3, n_elements(s1y) /3  , n_elements(s1z)/3 ]
  ns2 = [n_elements(s2x) /3 ,n_elements(s2y)/3 , n_elements(s2z) /3 ]
  
  nterms=0
  
  
;  /* Miscellaneous */
  jstat=0 & i=0 & j=0;
 t=0d & t2=0d &  xyz=0d & xyzd=0d & a=0d & b=0d & c=0d & ct=0d & p=0d & cp=0d
  ph=dblarr(3) & vh=dblarr(3) & pb=dblarr(3) & vb=dblarr(3) & x=0d & y=0d & z=0d;
  
  
@erfam
;  /* Time since reference epoch, Julian years. */

  t = ((date1 - ERFA_DJ00) + date2) / ERFA_DJY;
  t2 = t*t;  
  
  
;  /* Set status. */
  jstat = 1
  if abs(t) le 100.0 then jstat = 0;  
  
  
  
;  /* X then Y then Z. */
for i = 0,2 do begin
  
 ; /* Initialize position and velocity component. */
  xyz = 0.0d;
  xyzd = 0.0d;
  
;  /* ------------------------------------------------ */
;  /* Obtain component of Sun to Earth ecliptic vector */
;  /* ------------------------------------------------ */
  
;  /* Sun to Earth, T^0 terms. */
  coeffs = *ce0[i];
  nterms = ne0[i];
  for j = 0,nterms-1 do begin
  a = coeffs[j*3];
  b = coeffs[j*3+1];
  c = coeffs[j*3+2];
  p = b + c*t;
  xyz  += a*cos(p);
  ;xyz  =xyz+ a*cos(p);
  xyzd -= a*c*sin(p);
  ;xyzd =xyzd- a*c*sin(p);
  endfor

;/* Sun to Earth, T^1 terms. */
coeffs = *ce1[i];
nterms = ne1[i];
for j = 0,nterms-1 do begin
a = coeffs[j*3];
b = coeffs[j*3+1];
c = coeffs[j*3+2];
ct = c*t;
p = b + ct;
cp = cos(p);
xyz  += a*t*cp;
xyzd += a*( cp - ct*sin(p) );
endfor

;/* Sun to Earth, T^2 terms. */
coeffs = *ce2[i];
nterms = ne2[i];
for j = 0,nterms-1 do begin
a = coeffs[j*3];
b = coeffs[j*3+1];
c = coeffs[j*3+2];
ct = c*t;
p = b + ct;
cp = cos(p);
xyz  += a*t2*cp;
xyzd += a*t*( 2.0d*cp - ct*sin(p) );
endfor

;/* Heliocentric Earth position and velocity component. */
ph[i] = xyz;
vh[i] = xyzd / ERFA_DJY;

;/* ------------------------------------------------ */
;/* Obtain component of SSB to Earth ecliptic vector */
;/* ------------------------------------------------ */

;/* SSB to Sun, T^0 terms. */
coeffs = *cs0[i];
nterms = ns0[i];
for j = 0,nterms-1 do begin
a = coeffs[j*3];
b = coeffs[j*3+1];
c = coeffs[j*3+2];
p = b + c*t;
xyz  += a*cos(p);
xyzd -= a*c*sin(p);
endfor

;/* SSB to Sun, T^1 terms. */
coeffs = *cs1[i];
nterms = ns1[i];
for j = 0,nterms-1 do begin
a = coeffs[j*3];
b = coeffs[j*3+1];
c = coeffs[j*3+2];
ct = c*t;
p = b + ct;
cp = cos(p);
xyz  += a*t*cp;
xyzd += a*(cp - ct*sin(p));
endfor

;/* SSB to Sun, T^2 terms. */
coeffs = *cs2[i];
nterms = ns2[i];
for j = 0,nterms-1 do begin
a = coeffs[j*3];
b = coeffs[j*3+1];
c = coeffs[j*3+2];
ct = c*t;
p = b + ct;
cp = cos(p);
xyz  += a*t2*cp;
xyzd += a*t*(2.0d*cp - ct*sin(p));
endfor

;/* Barycentric Earth position and velocity component. */
pb[i] = xyz;
vb[i] = xyzd / ERFA_DJY;

;/* Next Cartesian component. */
endfor
  
 



;/* Rotate from ecliptic to BCRS coordinates. */

x = ph[0];
y = ph[1];
z = ph[2];
pvh[0,0] =      x + am12*y + am13*z;
pvh[0,1] = am21*x + am22*y + am23*z;
pvh[0,2] =          am32*y + am33*z;

x = vh[0];
y = vh[1];
z = vh[2];
pvh[1,0] =      x + am12*y + am13*z;
pvh[1,1] = am21*x + am22*y + am23*z;
pvh[1,2] =          am32*y + am33*z;

x = pb[0];
y = pb[1];
z = pb[2];
pvb[0,0] =      x + am12*y + am13*z;
pvb[0,1] = am21*x + am22*y + am23*z;
pvb[0,2] =          am32*y + am33*z;

x = vb[0];
y = vb[1];
z = vb[2];
pvb[1,0] =      x + am12*y + am13*z;
pvb[1,1] = am21*x + am22*y + am23*z;
pvb[1,2] =          am32*y + am33*z;

;/* Return the status. */
return, jstat


  

end