function eraJd2cal,dj1,dj2,iy,im,id,fd

;  int eraJd2cal(double dj1, double dj2,
;  int *iy, int *im, int *id, double *fd)
;  /*
;  **  - - - - - - - - - -
;  **   e r a J d 2 c a l
;  **  - - - - - - - - - -
;  **
;  **  Julian Date to Gregorian year, month, day, and fraction of a day.
;  **
;  **  Given:
;  **     dj1,dj2   double   Julian Date (Notes 1, 2)
;  **
;  **  Returned (arguments):
;  **     iy        int      year
;  **     im        int      month
;  **     id        int      day
;  **     fd        double   fraction of day
;  **
;  **  Returned (function value):
;  **               int      status:
;  **                           0 = OK
;  **                          -1 = unacceptable date (Note 3)
;  **
;  **  Notes:
;  **
;  **  1) The earliest valid date is -68569.5 (-4900 March 1).  The
;  **     largest value accepted is 1e9.
;  **
;  **  2) The Julian Date is apportioned in any convenient way between
;  **     the arguments dj1 and dj2.  For example, JD=2450123.7 could
;  **     be expressed in any of these ways, among others:
;  **
;  **            dj1             dj2
;  **
;  **         2450123.7           0.0       (JD method)
;  **         2451545.0       -1421.3       (J2000 method)
;  **         2400000.5       50123.2       (MJD method)
;  **         2450123.5           0.2       (date & time method)
;  **
;  **  3) In early eras the conversion is from the "proleptic Gregorian
;  **     calendar";  no account is taken of the date(s) of adoption of
;  **     the Gregorian calendar, nor is the AD/BC numbering convention
;  **     observed.
;  **
;  **  Reference:
;  **
;  **     Explanatory Supplement to the Astronomical Almanac,
;  **     P. Kenneth Seidelmann (ed), University Science Books (1992),
;  **     Section 12.92 (p604).
;  **
;  **  Copyright (C) 2013-2015, NumFOCUS Foundation.
;  **  Derived, with permission, from the SOFA library.  See notes at end of file.
;  */
  
;  /* Minimum and maximum allowed JD */
 DJMIN = -68569.5d;
 DJMAX = 1d9;
  
  jd=0l & l=0l & n=0l & i=0l & k=0l
  dj=0d & d1=0d & d2=0d & f1=0d & f2=0d & f=0d & d=0d
  
;  /* Verify date is acceptable. */
  dj = dj1 + dj2;
  if (dj lt DJMIN or dj gt DJMAX) then return, -1;
  
;  /* Copy the date, big then small, and re-align to midnight. */
  if (dj1 ge dj2)  then begin
  d1 = dj1;
  d2 = dj2;
endif else begin
d1 = dj2;
d2 = dj1;
endelse
d2 -= 0.5d;

;/* Separate day and fraction. */
f1 = (d1 mod 1.0d);
f2 = (d2 mod 1.0d);
f = ((f1 + f2) mod 1.0d);
if (f lt 0.0) then f += 1.0d;
d = floor(d1 - f1,/l64) + floor(d2 - f2,/l64) + floor(f1 + f2 - f,/l64);
jd = long(floor(d,/l64) + 1L);

;;/* Express day in Gregorian calendar. */
l = jd + 68569L;
n = (4L * l) / 146097L;
l -= (146097L * n + 3L) / 4L;
i = (4000L * (l + 1L)) / 1461001L;
l -= (1461L * i) / 4L - 31L;
k = (80L * l) / 2447L;
id = long(l - (2447L * k) / 80L);
l = k / 11L;
im = long(k + 2L - 12L * l);
iy = long(100L * (n - 49L) + i + l);
fd = f;

return, 0;


;/*----------------------------------------------------------------------



end