pro eraRefco,phpa,tc,rh,wl,refa,refb

;  void eraRefco(double phpa, double tc, double rh, double wl,
;  double *refa, double *refb)
;  /*
;  **  - - - - - - - - -
;  **   e r a R e f c o
;  **  - - - - - - - - -
;  **
;  **  Determine the constants A and B in the atmospheric refraction model
;  **  dZ = A tan Z + B tan^3 Z.
;  **
;  **  Z is the "observed" zenith distance (i.e. affected by refraction)
;  **  and dZ is what to add to Z to give the "topocentric" (i.e. in vacuo)
;  **  zenith distance.
;  **
;  **  Given:
;  **    phpa   double    pressure at the observer (hPa = millibar)
;  **    tc     double    ambient temperature at the observer (deg C)
;  **    rh     double    relative humidity at the observer (range 0-1)
;  **    wl     double    wavelength (micrometers)
;  **
;  **  Returned:
;  **    refa   double*   tan Z coefficient (radians)
;  **    refb   double*   tan^3 Z coefficient (radians)
;  **
;  **  Notes:
;  **
;  **  1) The model balances speed and accuracy to give good results in
;  **     applications where performance at low altitudes is not paramount.
;  **     Performance is maintained across a range of conditions, and
;  **     applies to both optical/IR and radio.
;  **
;  **  2) The model omits the effects of (i) height above sea level (apart
;  **     from the reduced pressure itself), (ii) latitude (i.e. the
;  **     flattening of the Earth), (iii) variations in tropospheric lapse
;  **     rate and (iv) dispersive effects in the radio.
;  **
;  **     The model was tested using the following range of conditions:
;  **
;  **       lapse rates 0.0055, 0.0065, 0.0075 deg/meter
;  **       latitudes 0, 25, 50, 75 degrees
;  **       heights 0, 2500, 5000 meters ASL
;  **       pressures mean for height -10% to +5% in steps of 5%
;  **       temperatures -10 deg to +20 deg with respect to 280 deg at SL
;  **       relative humidity 0, 0.5, 1
;  **       wavelengths 0.4, 0.6, ... 2 micron, + radio
;  **       zenith distances 15, 45, 75 degrees
;  **
;  **     The accuracy with respect to raytracing through a model
;  **     atmosphere was as follows:
;  **
;  **                            worst         RMS
;  **
;  **       optical/IR           62 mas       8 mas
;  **       radio               319 mas      49 mas
;  **
;  **     For this particular set of conditions:
;  **
;  **       lapse rate 0.0065 K/meter
;  **       latitude 50 degrees
;  **       sea level
;  **       pressure 1005 mb
;  **       temperature 280.15 K
;  **       humidity 80%
;  **       wavelength 5740 Angstroms
;  **
;  **     the results were as follows:
;  **
;  **       ZD       raytrace     eraRefco   Saastamoinen
;  **
;  **       10         10.27        10.27        10.27
;  **       20         21.19        21.20        21.19
;  **       30         33.61        33.61        33.60
;  **       40         48.82        48.83        48.81
;  **       45         58.16        58.18        58.16
;  **       50         69.28        69.30        69.27
;  **       55         82.97        82.99        82.95
;  **       60        100.51       100.54       100.50
;  **       65        124.23       124.26       124.20
;  **       70        158.63       158.68       158.61
;  **       72        177.32       177.37       177.31
;  **       74        200.35       200.38       200.32
;  **       76        229.45       229.43       229.42
;  **       78        267.44       267.29       267.41
;  **       80        319.13       318.55       319.10
;  **
;  **      deg        arcsec       arcsec       arcsec
;  **
;  **     The values for Saastamoinen's formula (which includes terms
;  **     up to tan^5) are taken from Hohenkerk and Sinclair (1985).
;  **
;  **  3) A wl value in the range 0-100 selects the optical/IR case and is
;  **     wavelength in micrometers.  Any value outside this range selects
;  **     the radio case.
;  **
;  **  4) Outlandish input parameters are silently limited to
;  **     mathematically safe values.  Zero pressure is permissible, and
;  **     causes zeroes to be returned.
;  **
;  **  5) The algorithm draws on several sources, as follows:
;  **
;  **     a) The formula for the saturation vapour pressure of water as
;  **        a function of temperature and temperature is taken from
;  **        Equations (A4.5-A4.7) of Gill (1982).
;  **
;  **     b) The formula for the water vapour pressure, given the
;  **        saturation pressure and the relative humidity, is from
;  **        Crane (1976), Equation (2.5.5).
;  **
;  **     c) The refractivity of air is a function of temperature,
;  **        total pressure, water-vapour pressure and, in the case
;  **        of optical/IR, wavelength.  The formulae for the two cases are
;  **        developed from Hohenkerk & Sinclair (1985) and Rueger (2002).
;  **
;  **     d) The formula for beta, the ratio of the scale height of the
;  **        atmosphere to the geocentric distance of the observer, is
;  **        an adaption of Equation (9) from Stone (1996).  The
;  **        adaptations, arrived at empirically, consist of (i) a small
;  **        adjustment to the coefficient and (ii) a humidity term for the
;  **        radio case only.
;  **
;  **     e) The formulae for the refraction constants as a function of
;  **        n-1 and beta are from Green (1987), Equation (4.31).
;  **
;  **  References:
;  **
;  **     Crane, R.K., Meeks, M.L. (ed), "Refraction Effects in the Neutral
;  **     Atmosphere", Methods of Experimental Physics: Astrophysics 12B,
;  **     Academic Press, 1976.
;  **
;  **     Gill, Adrian E., "Atmosphere-Ocean Dynamics", Academic Press,
;  **     1982.
;  **
;  **     Green, R.M., "Spherical Astronomy", Cambridge University Press,
;  **     1987.
;  **
;  **     Hohenkerk, C.Y., & Sinclair, A.T., NAO Technical Note No. 63,
;  **     1985.
;  **
;  **     Rueger, J.M., "Refractive Index Formulae for Electronic Distance
;  **     Measurement with Radio and Millimetre Waves", in Unisurv Report
;  **     S-68, School of Surveying and Spatial Information Systems,
;  **     University of New South Wales, Sydney, Australia, 2002.
;  **
;  **     Stone, Ronald C., P.A.S.P. 108, 1051-1058, 1996.
;  **
;  **  Copyright (C) 2013-2015, NumFOCUS Foundation.
;  **  Derived, with permission, from the SOFA library.  See notes at end of file.
;  */
  
  optic=0;
  p=0d & t=0d & r=0d & w=0d & ps=0d & pw=0d & tk=0d& wlsq=0d & gamma=0d & beta=0d &
  
 ; /* Decide whether optical/IR or radio case:  switch at 100 microns. */
  optic = ( wl le 100.0 );
  
 ; /* Restrict parameters to safe values. */
  t = max( [tc, -150.0d] );
  t = min( [t, 200.0d] );
  p = max( [phpa, 0.0d] );
  p = min( [p, 10000.0d] );
  r = max( [rh, 0.0] );
  r = min( [r, 1.0d ]);
  w = max( [wl, 0.1d] );
  w = min( [w, 1d6] );
  
;  /* Water vapour pressure at the observer. */
  if ( p gt 0.0 ) then begin
    expo=( ( 0.7859d + 0.03477d*t ) / ( 1.0d + 0.00412d*t ) ) 
  ps = 10d^expo *  $
  ( 1.0d + p * ( 4.5d-6 + 6d-10*t*t )  );
  pw = r * ps / ( 1.0d - (1.0d -r)*ps/p );
endif else begin
pw = 0.0d;
endelse

;/* Refractive index minus 1 at the observer. */
tk = t + 273.15d;
if ( optic ) then begin
wlsq = w * w;
gamma = ( ( 77.53484d-6 +  $
( 4.39108d-7 + 3.666d-9/wlsq ) / wlsq ) * p $
- 11.2684d-6*pw ) / tk;
endif else begin
gamma = ( 77.6890d-6*p - ( 6.3938d-6 - 0.375463d/tk ) * pw ) / tk;
endelse

;/* Formula for beta from Stone, with empirical adjustments. */
beta = 4.4474d-6 * tk;
if ( ~optic ) then beta -= 0.0074d * pw * beta;

;/* Refraction constants from Green. */
refa = gamma * ( 1.0d - beta );
refb = - gamma * ( beta - gamma / 2.0d );

;/* Finished. */


;/*----------------------------------------------------------------------


end