function eraUtctai,utc1,utc2,tai1,tai2

;  int eraUtctai(double utc1, double utc2, double *tai1, double *tai2)
;  /*
;  **  - - - - - - - - - -
;  **   e r a U t c t a i
;  **  - - - - - - - - - -
;  **
;  **  Time scale transformation:  Coordinated Universal Time, UTC, to
;  **  International Atomic Time, TAI.
;  **
;  **  Given:
;  **     utc1,utc2  double   UTC as a 2-part quasi Julian Date (Notes 1-4)
;  **
;  **  Returned:
;  **     tai1,tai2  double   TAI as a 2-part Julian Date (Note 5)
;  **
;  **  Returned (function value):
;  **                int      status: +1 = dubious year (Note 3)
;  **                                  0 = OK
;  **                                 -1 = unacceptable date
;  **
;  **  Notes:;
;  **
;  **  1) utc1+utc2 is quasi Julian Date (see Note 2), apportioned in any
;  **     convenient way between the two arguments, for example where utc1
;  **     is the Julian Day Number and utc2 is the fraction of a day.
;  **
;  **  2) JD cannot unambiguously represent UTC during a leap second unless
;  **     special measures are taken.  The convention in the present
;  **     function is that the JD day represents UTC days whether the
;  **     length is 86399, 86400 or 86401 SI seconds.  In the 1960-1972 era
;  **     there were smaller jumps (in either direction) each time the
;  **     linear UTC(TAI) expression was changed, and these "mini-leaps"
;  **     are also included in the ERFA convention.
;  **
;  **  3) The warning status "dubious year" flags UTCs that predate the
;  **     introduction of the time scale or that are too far in the future
;  **     to be trusted.  See eraDat for further details.
;  **
;  **  4) The function eraDtf2d converts from calendar date and time of day
;  **     into 2-part Julian Date, and in the case of UTC implements the
;  **     leap-second-ambiguity convention described above.
;  **
;  **  5) The returned TAI1,TAI2 are such that their sum is the TAI Julian
;  **     Date.
;  **
;  **  Called:
;  **     eraJd2cal    JD to Gregorian calendar
;  **     eraDat       delta(AT) = TAI-UTC
;  **     eraCal2jd    Gregorian calendar to JD
;  **
; **  References:
;  **
;  **     McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
;  **     IERS Technical Note No. 32, BKG (2004)
;  **
;  **     Explanatory Supplement to the Astronomical Almanac,
;  **     P. Kenneth Seidelmann (ed), University Science Books (1992)
;  **
;  **  Copyright (C) 2013-2015, NumFOCUS Foundation.
;  **  Derived, with permission, from the SOFA library.  See notes at end of file.
;  */
  
  big1=0;
  iy=0 & im=0 & id=0 & j=0 & iyt=0 & imt=0 & idt=0;
  u1=0d & u2=0d & fd=0d & dat0=0d & dat12=0d & w=0d & dat24=0d & dlod=0d & dleap=0d & z1=0d & z2=0d & a2=0d;
  @erfam
; /* Seconds per day. */
;ERFA_DAYSEC=86400.0d
  
;  /* Put the two parts of the UTC into big-first order. */
  big1 = ( utc1 gt utc2 );
  if ( big1 ) then begin
  u1 = utc1;
  u2 = utc2;
  endif else begin 
  u1 = utc2;
  u2 = utc1;
  endelse

;/* Get TAI-UTC at 0h today. */
j = eraJd2cal(u1, u2, iy, im, id, fd);
if  j then return, j;
j = eraDat(iy, im, id, 0.0d, dat0);
if ( j lt 0 ) then return, j;

;/* Get TAI-UTC at 12h today (to detect drift). */
j = eraDat(iy, im, id, 0.5d, dat12);
if ( j lt 0 ) then return, j;

;/* Get TAI-UTC at 0h tomorrow (to detect jumps). */
j = eraJd2cal(u1+1.5d, u2-fd, iyt, imt, idt, w);
if ( j ) then return, j;
j = eraDat(iyt, imt, idt, 0.0d, dat24);
if ( j lt 0 ) then return, j;

;/* Separate TAI-UTC change into per-day (DLOD) and any jump (DLEAP). */
dlod = 2.0d * (dat12 - dat0);
dleap = dat24 - (dat0 + dlod);

;/* Remove any scaling applied to spread leap into preceding day. */
fd *= (ERFA_DAYSEC+dleap)/ERFA_DAYSEC;

;/* Scale from (pre-1972) UTC seconds to SI seconds. */
fd *= (ERFA_DAYSEC+dlod)/ERFA_DAYSEC;

;/* Today's calendar date to 2-part JD. */
if ( eraCal2jd(iy, im, id, z1, z2) ) then return, -1;

;/* Assemble the TAI result, preserving the UTC split and order. */
a2 = z1 - u1;
a2 += z2;
a2 += fd + dat0/ERFA_DAYSEC;
if ( big1 ) then begin
tai1 = u1;
tai2 = a2;
endif else begin
tai1 = a2;
tai2 = u1;
endelse

;/* Status. */
return, j;


;;/*----------------------------------------------------------------------





end