;
; This file is part of MIA.
;
; MIA is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
; $Id: calibrate_disk.pro,v 1.3 2003/09/09 09:56:17 koehler Exp $
;
; calibrate_disk.pro
; Created:     Tue Jul 15 17:05:52 2003 by Koehler@sun47
; Last change: Tue Dec 16 11:22:41 2003
;
; PURPOSE:
;	calibrate the calibrators
;
; CALLING SEQUENCE:
;
;
; INPUT:
;
;
; OUTPUT:
;
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; cartesic method, i.e. Christoph's method
;
pro project_baseline, t, de, stations, base_len, base_pa
  ; t = hourangle in hours, de = Declination in degrees
  ; base_len is given in meters
  ; base_pa in degrees

  t  = t * !PI/12.
  de = de* !PI/180.
  sinH = sin(t)
  cosH = cos(t)

  ; from ESO exposure time calculator:
  ;longitude=  70.40479659;
  latitude = -24.62794830

  sin_la = sin(latitude * !PI/180.)
  cos_la = cos(latitude * !PI/180.)
  ;
  ; find out the baseline used
  ;
  case stations of
  "U1-U2": begin & BaselineLen=  56.569 & BaselinePA=  26.016 & end
  "U1-U3": begin & BaselineLen= 102.434 & BaselinePA=  32.369 & end
  "U1-U4": begin & BaselineLen= 130.231 & BaselinePA=  60.396 & end
  "U2-U3": begin & BaselineLen=  46.635 & BaselinePA=  40.082 & end
  "U2-U4": begin & BaselineLen=  89.443 & BaselinePA=  81.321 & end
  "U3-U4": begin & BaselineLen=  62.463 & BaselinePA= 110.803 & end
      else: message,"Unknown baseline "+stations
  endcase

  ; coordinates of baseline in horizontal system
  BPAr  = BaselinePA * !PI/180.
  BaseN = BaselineLen * cos(BPAr)
  BaseE = BaselineLen * sin(BPAr)


  ; rotate around y-axis into equatorial system
  r1 = BaseN * (-sin_la)          ; + c * cos_la
  r2 = BaseE
  r3 = BaseN * (-cos_la)

  ; rotate into star-centered system
  a_E =   r1 * sinH + r2 * cosH
  b_N = (-r1 * cosH + r2 * sinH) * sin(de) - r3 * cos(de)
  ; length along line of sight doesn't matter

  base_len = sqrt( a_E * a_E + b_N * b_N );
  base_pa  = atan(a_E, b_N) * 180./!PI;
end

FUNCTION calibrate_disk, filename, diameter, lambda
  ; filename = MIDI filename string (will use first file)
  ; diameter = diameter in mas
  ; lambda   = array of wavelengths in micron

  filelist = oirFileList(filename[0])
  im = obj_new('imagedata', filelist[0])
  he = im->prihead()
  objectname= STRCOMPRESS(he->getpar('OBS TARG NAME'),/remove_all)
  esolen = StrCompress(he->getpar('HIERARCH ESO ISS PBL12 START'),/remove_all)
  stat1  = StrCompress(he->getpar('HIERARCH ESO ISS CONF STATION1'),/remove_all)
  stat2  = StrCompress(he->getpar('HIERARCH ESO ISS CONF STATION2'),/remove_all)
  date   = StrCompress(he->getpar('DATE-OBS'),/remove_all)
  lst    = StrCompress(he->getpar('LST'),/remove_all)
  RA     = StrCompress(he->getpar('RA') ,/remove_all)
  Dec    = StrCompress(he->getpar('DEC'),/remove_all)
  obj_destroy,im

  if (stat1 gt stat2) then begin
      tmp = stat1 & stat1 = stat2 & stat2 = tmp
  endif

  print, "object:              ", objectname
  print, "Baseline:            ", stat1, " - ", stat2
  print, "ESO's baseline len.: ", esolen
  print, "Date of observation: ", date
  print, "Local Sid Time:      ", LST
  print, "Right Ascension:     ", RA
  print, "Declination:         ", Dec

  project_baseline, (LST/3600. - RA/15.), Dec, stat1+"-"+stat2, base_len, base_pa

  print, "Our baseline length: ", strcompress(string(base_len),/remove_all), $
    ", p.a.: ", base_pa

  diarad = diameter/1000./3600./180. * !PI
  print, "Diameter ", diameter, "mas = ", diarad, "rad"

  x  = !pi * base_len * diarad / (lambda / 1e6)
  vis = abs(2. * beselj(x,1) / x)

  print,"      lambda      x_bessel     visibility"
  for i=0,N_elements(x)-1 do print, lambda[i], x[i], vis[i]

  return, vis
end
