pro auto_ephemeris, EL, time, RAsep, DECsep, noprint=noprint

; Plot the PA and sep over a specified number of years into the future.
;
; Enter P,T,e,a,i,Omega,omega for each solution
; Enter starting year and number of year into the future from which to
; compute the ephemeris.
; 
; Calling sequence:
;     auto_ephemeris, EL, time, RAsep, DECsep [, /noprint]
;
; set noprint if no printed output to screen is desired
; default is to print RAsep and DECsep to screen
;
; Equations:  
; Orbital elements (P,T,e) and a set of observations (ti,xi,yi) are known
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; The four Thiele-Innes elements are solved through linear least squares fit:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
;
; Essentially, the equations one needs to solve are:
;	E - esinE = 2Pi/P * (t - T)
;	x = A*(cos(Ei) - e) + F*(sqrt(1 - e^2)*sin(Ei))
;	y = B*(cos(Ei) - e) + G*(sqrt(1 - e^2)*sin(Ei))
;
; Paramers:
;	Period:	period (P)
;	Tperi: 	time of periastron passage (T)
;	ecc:	eccentricity (e)
;	major:	semi-major axis (in mas) (a)
;	inc:	inclination (i)
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xd:	RA - data
;	yd:	DEC - data
;	xf:	RA - fit (determined from orbital elements for time of obs.)
;	yf:	DEC - fit (determined from orbital elements for time of obs.)
;	A,B,F,G: Thiele-Innes elements
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; INPUT:
;       P,T,e,a,i,Omega,omega
;       tstart and trange
;	
; Prompted inputs:
;	all of the orbital elements: P,T,e
;
; OUTPUT: 
;	plot of data points and orbital solutions
;
; Modifications to be added:
; 	- error bars on data points
;	- line connecting measurement to claculated point
;
; 2006 July 19: Modify EPHEMERIS.PRO to work with entire set of 
;     orbital elements (P,T,e,a,i,Omega,omega).  This avoids needing
;     to input data file for the least-squares fitting.
;
; 2006 August 30: automate procedure so that it can be called on the command
;     line.  Take parameters and times as input to program.  

!P.charsize=1.0
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.0	;scale factor for adjusting size of plotting symbols


; Set color table
;  red=1, green=2, blue=3

;tvlct, [255,0,0], [0,255,0], [0,0,255], 1

ans_print = keyword_set(noprint)


; Obtain initial values for P,T,e,a,i,W,w

period = EL(0)
Tperi = EL(1)
ecc = EL(2)
major = EL(3)
inc = EL(4)
W_cap = EL(5)
w_low = EL(6)

; angles are already in radians

calc_vbfit, EL, time, PA, sep

xarr = sep*cos(PA)	; x coordinate
yarr = sep*sin(PA)	; y coordinate

tnum = n_elements(time)

if (ans_print ne 1.0) then begin
;print, "Period (years):", period
print, "Date RAsep DECsep:"
for i=0,tnum-1 do print,time(i),yarr(i),xarr(i)

print, "Date PA sep:"
for i=0,tnum-1 do print,time(i),PA(i)*180d/!dpi,1.0,sep(i),2.0
endif

RAsep = yarr
DECsep = xarr


plot_title=""
;print, "Enter plot title:"
;read, plot_title

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z), /fill

if (ans_print ne 1.0) then begin
;print, "Do you want to save plot as a postscript file?  (Type yes or no):"
;pans = ""
;read,pans
pans = 'yes'
; Set plotting device to ps if answer is yes
if (pans eq "yes") then begin
	set_plot, 'ps'
	device, /Color, Bits_per_pixel=8, filename='temp.ps' 
endif

		maxx = max(yarr)
		minx = min(yarr)
		maxy = max(xarr)
		miny = min(xarr)

;		ex = 0d
;		print, "x: max min ", maxx,minx
;		print, "y: max min ", maxy,miny
;		print, "By what value would you like to extend x and y axes?"
;		read, ex

ex = 0.10*max([abs(maxx-minx),abs(maxy-miny)])

plot, yarr, xarr, psym=8, XSTYLE=1, YSTYLE=1, ISOTROPIC=1, $
	xrange=[maxx+ex,minx-ex], yrange=[miny-ex,maxy+ex], $
	xtitle="E-W sep (mas)", ytitle="N-S sep (mas)", $
	title = plot_title

; set plotting device back to terminal if previously saving to ps file
if (pans eq "yes") then begin 
	device, /close
	set_plot, 'x'
endif
endif

end
