pro auto_fit_velocity_sb1, time, V1, dV1, EL, elfix, ELerr = ELerr, showplot=showplot,nskip=nskip;, toler=toler

; Same as fit_velocity_sb1.pro, except enter data as time,V1,dV1
; rather than as a data file.
; 
; /showplot = show plots and print results to screen
; toler: set tolerance for chi2 agreement
;
; Compute the spectroscopic orbital elements through a Newton-Raphson 
; technique.  Works for SB1!!!
;
; The program requires an initial guess that can be calculated from using
; the points from the fitting curves.
;
; Calls the following routines:
;     calc_deriv_SB1.pro
;     calc_Ei.pro
;     solve_trans.pro
;     calc_sb1fit.pro - for plotting model orbits
;
; Equations:  
; Initial estimates (P0,T0,e0,K1_0,omega0,Vsys0) and  
; a set of observations (ti,V1i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axis (a1sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
; The fitted velocities (V1_i) are determined from the orbital elements:
;       V1 = K1*[e*cos(omega) + cos(nu + omega)] 
;
; Procedure:
; Minimize chi squared between measured velocities and fitted velocities:
;	chi^2 = sum[(V1data - V1fit)^2/sigmaV1^2]
; where the sums run over all data points
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,K1,omega,Vsys) and set to zero. Solve for the value of
; the orbital elements that minimize chi^2.
;
; Since the equations for V1 are non-linear, cannot solve for 
; the orbital elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (V1fit,V2fit) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) + dx/dw|(w-w0)
;	       + dx/d(K1)|(K1 - K1|0) + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(V1) and the | represents the value evaluated for the
; initial estimates of the orbital elements. Now, V1fit and V2fit 
; are linear in (P,T,e,K1,w,Vsys), so the partial derivatives
; can easily be taken and solved by setting up a matrix and using 
; Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;       Vsys:    systemic velocity
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  V1  errV1
;		      where V1 is the radial velocity of the primary
;                     and errV1 is the measurement error
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; Prompted inputs:
;	initial estimates for all the orbital elements: 
;             P, T, e, K1, omega, Vsys
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 -> solves for all
;                   0 0 0 1 0 0 -> solves only for a1 of primary
; Notes: make sure P,T are given in same units
;        make sure K1,V1 are given in same units
;
; OUTPUT:
;	best fit values of orbital elements: P,T,e,a1sini,omega,Vsys
;       and corresponding errors
;
; Notes on units:
;    P, T, and time should be in same units (days or years)
;    K1 and V1 should be in same units (km/s)
;
; Began 23 June 2005
;
; Version 1: Orignal copy
; Version 2: 
;    - Include Marquadt Method of adjusting diagonal elements of
;      the alpha matrix to ensure convergence. 5 July 2005
;    - Fixed non-convergence problems   7 July 2005
;        + do not remove P wrap around effects from dV/dP
;          leave (t-T)/P in full form; net effect is that adjustments
;          smaller
;        + fixed primary/secondary indices in setting dV1/dVsys and dV2/dVsys
;    - Designed for SB1's only!!!
;
; 31 May 2006: Save unmodified program as version 2
; 31 May 2006: Fit SB2 parameters in terms of K1
;              (as opposed to a1sini)


; If keyword showplot is set, then show orbit plots on screen
; keyword_set returns True (1) if showplot is defined, False (0) if undefined
ans_plot = keyword_set(showplot)

; Check if we are outputting the array of parameter errors
ans_err = keyword_set(ELerr)

; Do we need to end loop if solution is not found?
ans_skip = keyword_set(nskip)
if (ans_skip eq 0) then nskip=1d6

;; If keyword showplot is set, then show orbit plots on screen
;ans_tol = keyword_set(toler)
;if(ans_tol) then ctol = toler else ctol = 0.0

num = n_elements(time)

; Obtain values for P,T,e,a1sini,omega,Vsys

period = EL(0)
Tperi = EL(1)
ecc = EL(2)
K1 = EL(3)
omega = EL(4)
Vsys = EL(5)

;convert omega to radians
omega = omega*!dpi/180

k=0
for i=0, 4 do if (elfix(i) eq 0) then k=k+1

nEl = n_elements(elfix)
mfit = nEl - k 		; number of elements to improve

; determine indices of elements for which to vary
elvar = where(elfix ne 0)
    
EL = [period,Tperi,ecc,K1,omega,Vsys]
ELadj = EL
elLabel = ['P','T','e','K1','omega','Vsys']

; plot model and data of initial guess

tnum = 1000.0
tmin = min(time)
tmax = max(time)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL, tarr, V1mod

vmax = max(V1mod)
vmin = min(V1mod)

if (ans_plot) then begin
   plot,time,V1,psym=2,yrange=[vmin,vmax]
   oplot,tarr,V1mod
endif

count = 0

delta_chi = 1.0		; set initially to begin loop

lambda = 0.001   ; Marquardt method to ensure convergence
lflag = 0

while ((delta_chi gt 0.001) and (count lt nskip)) do begin

    ; Determine errors in orbital elements
    ; set up matrices for error determination
    ; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

    ; set up covariance matrix and column matrix
    ; alpha = dblarr(6,6)
    ; beta = dblarr(6)

    calc_deriv_SB1, EL, elfix, mfit, time, V1, dV1, V1_f, alpha, beta

    ; Determine chi squared

    chi2old = total((V1 - V1_f)^2/dV1^2)

    invmat = invert(alpha, stat, /double)

    if (ans_plot) then print,"stat (0 successful):",stat

    ;determine errors:

    ELerr = dblarr(mfit)

    for i=0, mfit-1 do ELerr(i) = sqrt(invmat(i,i))

    ; adjust alpha matrix by Marquardt parameter lambda

    for i=0, mfit-1 do alpha(i,i) = alpha(i,i)*(1.0 + lambda)

    if (mfit eq 1) then begin
        delta_el = dblarr(1)
        delta_el(0) = beta/alpha
    endif else delta_el = cramer(alpha,beta)   ; adjustments

;    print,'alpha',alpha
;    print,'beta',beta

    ; adjust orbital parameters
    
    for i=0, mfit-1 do ELadj(elvar(i)) = EL(elvar(i)) + delta_el(i)

    if (ans_plot) then begin
    k=0
    print,"Orbital Element     Adjustment"
    for i=0, nEL - 1 do begin
        if (elfix(i) eq 0) then begin
            if (i eq 4) then print, elLabel(i), ': ', El(i)*180/!dpi $
            else print, elLabel(i), ': ', El(i)
        endif else begin
            if (i eq 4) then $
              print, elLabel(i), ': ', El(i)*180/!dpi, delta_el(k)*180/!dpi $
            else print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor

    print,"Chi2 of current solution:",chi2old

    plot,time,V1,psym=2,yrange=[vmin,vmax]
    oplot,time,V1_f,psym=7
    for l=0, num-1 do oplot,[time(l),time(l)],[V1(l),V1_f(l)]

    endif

    calc_deriv_SB1, ELadj, elfix, mfit, time, V1, dV1, V1_f, alpha, beta

    ; Determine chi squared

    chi2new = total((V1 - V1_f)^2/dV1^2)

    if (ans_plot) then print,"chi2 of next modification:",chi2new

    if (chi2new le chi2old) then begin
        EL = ELadj
	chi2 = chi2new
        delta_chi = chi2old - chi2new
        lambda = lambda/10.0
        if (ans_plot) then print, 'lambda', lambda
    endif else begin
	if (ans_plot) then begin
           print, "next adjustment does not improve chi2"
           print, "increasing lambda by 10.0"
        endif
        lambda = lambda*10.0
        if (ans_plot) then print, 'lambda', lambda
        delta_chi = 1.0
    endelse

    count = count+1

    ; do not exit unless lambda is less than 0.001

;    if (lambda gt 0.001) and (delta_chi lt 0.001) then delta_chi = 1.0
;    print,'lambda',lambda
;    print,'delta_chi',delta_chi

    ;stop,'Type .cont to continue'

endwhile

lambda = 0.0

if (ans_plot) then print, "Final values:"

period = EL(0)
Tperi = EL(1)
ecc = EL(2)
K1 = EL(3)
omega = EL(4)
Vsys = EL(5)

;; Determine the velocity amplitudes
;K1 = (2.0*!dpi/(period*sec))*(a1sini*km/sqrt(1 - ecc^2))

; Determine final error matrix:

calc_deriv_SB1, EL, elfix, mfit, time, V1, dV1, V1_f, alpha, beta

invmat = invert(alpha, stat, /double)

;determine errors:

chi2 = total((V1 - V1_f)^2/dV1^2)

; degrees of freedom
dof = double(num - mfit)

; reduced chi squared:
chi2red = chi2/dof

ELerr = dblarr(mfit)
for i=0,mfit-1 do ELerr(i) = sqrt(chi2red)*sqrt(invmat(i,i))

if (ans_plot) then begin
k=0
print,"Orbital Element     Error     "
for i=0, nEL - 1 do begin
    if (elfix(i) eq 0) then begin
        if (i eq 4) then print, elLabel(i), ': ', El(i)*180/!dpi $
        else print, elLabel(i), ': ', El(i)
    endif else begin
        if (i eq 4) then $
          print, elLabel(i), ': ', El(i)*180/!dpi, Elerr(k)*180/!dpi $
        else print, elLabel(i), ': ', El(i), Elerr(k)
        k=k+1
    endelse
endfor
;print,'K1:',K1
print, "Final chi2:", chi2
print, "Reduced chi2:", chi2red
print,'Number of iterations:',count

print,EL

; plot best-fit orbit and data

tnum = 1000.0
V1mod = dblarr(tnum)
tarr = dblarr(tnum)
tmin = min(time)
tmax = max(time)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL, tarr, V1mod

vmax = max(V1mod)
vmin = min(V1mod)

; plot data and best-fit positions
plot,time,V1,psym=2,yrange=[vmin,vmax]
oplot,time,V1_f,psym=7
for l=0, num-1 do oplot,[time(l),time(l)],[V1(l),V1_f(l)]

; plot best-fit orbit
oplot,tarr,V1mod

endif

;re-set omega to degrees for exit

EL(4)=EL(4)*180.0/!dpi

if (ans_err) then ELerr(4)=ELerr(4)*180.0/!dpi

end
