pro auto_hartkopf_grid, time, theta, dtheta, rho, drho, theta_f, rho_f, EL, $
    mass_constraint=mass_constraint
;pro auto_hartkopf_grid, time, xarr, dx, yarr, dy, xfit, yfit, EL

; Same test_chi.pro but data is fed directly to program 
; (time, PA, dPA, sep, dsep)
; Search parameters are read from file auto_hartkopf_mc_par
;
; Add mass_constraint = [Mtot, dMtot, parallax, yd] if you want to force
; total mass to stay within specified range.  If period is in years
; then yd = 1.0d, if period is in days then yd = 1.0d/365.25.
;
; This is a copy of HARTKOPF_E_V1.PRO that uses a variation of 1 in
; chi2, rather than in the reduced chi2, to compute errors.
;
; Compute the orbital elements of a binary system, using the least squares
; technique described by Hartkopf, McAlister, Franz 1989, AJ, 98, 1014
;
; Procedure: Given (P,T,e) and a set of observations (ti,xi,yi), the eccentric
; anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; Four Thiele-Innes elements determined from least squares solution of 
; equations:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
; 
; Paramers:
;	P:	period
;	T: 	time of periastron passage
;	e:	eccentricity
;	a:	semi-major axis (in mas)
;	i:	inclination
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xi:	RA
;	yi:	DEC
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orientations, they are just rotated by 90 deg with respect to each other.
;
; INPUT: 
;	data_file: file containing data points in the format:
;		      time  theta  dtheta rho drho 
;		      where theta is the position angle and rho the separation
;		      dtheta and drho are the errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; Prompted inputs:
;	period (P)
;	time of periastron passage (T)
;	eccentricity (e)
;       epoch: search for T within P/2 of epoch
;
; OUTPUT: 
;       result_chi:  file containing all solutions within Delta(chi2)=1
;                    format: P T e chi2
;
;       xyresiduals: file containing (Observed-Calculated) residuals
;                    format:
;                    date     dx       xO-C   xOC/dx dy      yO-C   yOC/dy
;                    where dx and dy are the measurement uncertainties
;                          xO-C and yO-C are the (data - fit) values
;                          xOC/dx and yOC/dy are the ratio of the two
;
; Calls the following routines:
;     calc_AFBG.pro
;     calc_Ei.pro
;     solve_trans.pro
;     linfith.pro
;     aiWw.pro
;     calc_deriv_vb.pro
;
; Began: May 2002
; 31 August 2011: Move subroutines at end to separate files
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

close,1
close,2

num = n_elements(time)

; convert data points to x and y coordinates
xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:
dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

; Are mass constraints defined?
; keyword_set returns True (1) if mass_constraint is defined, False (0) if undefined

ans_mass = keyword_set(mass_constraint)

if (ans_mass) then begin

   Mcrit = mass_constraint(0)
   dMcrit = mass_constraint(1)
   parallax = mass_constraint(2)
   yd = mass_constraint(3)

   Mcrit_high = Mcrit + dMcrit
   Mcrit_low  = Mcrit - dMcrit

endif

;ans=''
;for i=0, num-1 do print,yarr(i),dy(i),xarr(i),dx(i)
;print,'Hit enter to continue'
;read,ans

; Initial values for P,T,e
P0 = 0d
T0 = 0d
e0 = 0d
; Only search for orbital solutions where the time of periaston passage
; is within a cycle of the period from the current epoch (ex. 2000).
; This program will only search for solutions of the time of
; periastron passage within +/- P/2 of the chosen epoch.
epoch=0d
; Determine searching ranges and step sizes for P,T,e."
; step is the incremental size for which to vary given parameter"
; min is the lower value for the searching range, ie: Pi = P-minP"
; max is the higher value for the searching range, ie: Pf = P+maxP"
Pstep = 0d
minP = 0d
maxP = 0d
Tstep = 0d
minT = 0d
maxT = 0d
estep = 0d
mine = 0d
maxe = 0d

; Read in search parameters from auto_hartkopf_mc_par
; Format of auto_hartkopf_mc_par:
; # Initial Estimates
; Period: 200.0
; Tperi:  2000.0
; ecc:    0.5
; epoch:  2000.0
; # Search Ranges and step sizes: step min max (search goes from P-minP to P+maxP
; P: 10.0 100.0 500.0
; T: 10.0 1950.0 2050.0
; e: 0.05 0.0 0.95

tempstring = ''

openr,1,'auto_hartkopf_grid_par'

; Read in one line of header
readf,1,tempstring
; Read in initial estimates for P, T, e:
; P
readf,1,tempstring
sepvalues = strsplit(tempstring,/extract)
P0 = double(sepvalues(1))
; T
readf,1,tempstring
sepvalues = strsplit(tempstring,/extract)
T0 = double(sepvalues(1))
; e
readf,1,tempstring
sepvalues = strsplit(tempstring,/extract)
e0 = double(sepvalues(1))
; Read in epoch for Tperi:
readf,1,tempstring
sepvalues = strsplit(tempstring,/extract)
epoch = double(sepvalues(1))
; Read in one line of header
readf,1,tempstring
; Read in search ranges (step min max)
; Search goes from P-minP to P+maxP
; Read in Pstep size, minP, maxP for search in P:
readf,1,tempstring
sepvalues = strsplit(tempstring,/extract)
Pstep = double(sepvalues(1))
minP = double(sepvalues(2))
maxP = double(sepvalues(3))
; Read in Tstep size, minT, maxT for search in T:
readf,1,tempstring
sepvalues = strsplit(tempstring,/extract)
Tstep = double(sepvalues(1))
minT = double(sepvalues(2))
maxT = double(sepvalues(3))
; Read in estep size, mine, maxe for search in e:
readf,1,tempstring
sepvalues = strsplit(tempstring,/extract)
estep = double(sepvalues(1))
mine = double(sepvalues(2))
maxe = double(sepvalues(3))

close,1

; open file "result_chi" to save chi squared values
openw,2,"result_chi"

; degrees of freedom
dof = 2d*num - 7d

; Create initial model:
			
period = P0
Tperi = T0
ecc =e0

; Determine fitted points (xfit,yfit) by computing the Thiele Innes
; elements A,F,B,G

calc_AFBG, time, xarr, dx, yarr, dy, period, Tperi, ecc, $
	   A_cap, F_cap, B_cap, G_cap, xfit, yfit

; Determine chi squared

chi2 = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

; reduced chi squared:
chi2red = chi2/dof

chimin = chi2
chiredmin = chi2red
bestpar = [period,Tperi,ecc,chi2]
bestABFG = [A_cap,B_cap,F_cap,G_cap]
xbest = xfit
ybest = yfit

; begin modifying P,T,e

;(the 0.000001d added to the upper limit is to account for errors in precision)

for period = (P0 - minP) , (P0 + maxP + 0.000001d), Pstep do begin

   for Tperi = (T0 - minT) , (T0 + maxT + 0.000001d), Tstep do begin

      ; only search for orbital solutions where the time of periaston
      ; passage is within a cycle of the period from the current
      ; epoch (ex. 2000)

      timediff = abs(epoch - Tperi)

      if (timediff le (period/2d)) then begin

         for ecc = (e0 - mine), (e0 + maxe + 0.000001d), estep do begin

	    ; Calculate Thiele-Innes elements and fitted points
	    ; (A,F,B,G;xfit,yfit)	

            calc_AFBG, time, xarr, dx, yarr, dy, $
                       period, Tperi, ecc, $
                       A_cap, F_cap, B_cap, G_cap, xfit, yfit
	
	    ; Determine chi squared

            chi2 = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

	    ; reduced chi squared:
            chi2red = chi2/dof

;	    print, "chi2", chi2

            if (chi2 le chimin) then begin

               if (ans_mass) then begin

                  ; Check to make sure mass falls within critical values

                  ; determine a,i,Omega,omega (using AIWW.PRO)
                  aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low

                  ; calculate mass of system
                  mass = major^3/parallax^3/(yd*period)^2

                  if (mass le Mcrit_high) and (mass ge Mcrit_low) then begin

                     chimin = chi2
                     chiredmin = chi2red
                     bestpar = [period,Tperi,ecc,chi2]
;;		     print,"Best fit P T e chi^2"
;;		     print, bestpar
;;		     printf,2, bestpar
                     bestABFG = [A_cap,B_cap,F_cap,G_cap]
                     xbest = xfit
                     ybest = yfit

		     ;plot data and fitted points

                     xmin = min(xarr)
                     xmax = max(xarr)
                     ymin = min(yarr)
                     ymax = max(yarr)

                     xrange = xmax - xmin
                     yrange = ymax - ymin

                     plot, xarr, yarr, psym=4, XSTYLE=1, YSTYLE=1, ISOTROPIC=1,$
                           xrange=[xmin-0.05*xrange,xmax+0.05*xrange],$
                           yrange=[ymin-0.05*yrange,ymax+0.05*yrange]
                     oplot, xfit, yfit, psym=5 

                     wait,0.2
                  endif 
               endif else begin

                  ; Otherwise just save best fit values

                  chimin = chi2
                  chiredmin = chi2red
                  bestpar = [period,Tperi,ecc,chi2]
;;		  print,"Best fit P T e chi^2"
;;		  print, bestpar
;;		  printf,2, bestpar
                  bestABFG = [A_cap,B_cap,F_cap,G_cap]
                  xbest = xfit
                  ybest = yfit

		  ;plot data and fitted points
                  
                  xmin = min(xarr)
                  xmax = max(xarr)
                  ymin = min(yarr)
                  ymax = max(yarr)

                  xrange = xmax - xmin
                  yrange = ymax - ymin


                  plot, xarr, yarr, psym=4, XSTYLE=1, YSTYLE=1, ISOTROPIC=1, $
                        xrange=[xmin-0.05*xrange,xmax+0.05*xrange],$
                        yrange=[ymin-0.05*yrange,ymax+0.05*yrange]
                  oplot, xfit, yfit, psym=5 
                  
                  wait,0.2

               endelse
            endif
;;          endif else begin
	
;;          if (chi2 le (chimin + 1d)) then begin
;;                print,"P T e chi^2"
;;	          print,period,Tperi,ecc,chi2
;;	          printf,2,period,Tperi,ecc,chi2
;;		
;;           endif
;;           endelse
         endfor
      endif
   endfor
endfor

close,2

print,"Best fit P T e chi^2"
print, bestpar

print,"reduced chi2:", chiredmin

print, "A B F G:", bestABFG

period=bestpar(0)
Tperi = bestpar(1)
ecc = bestpar(2)
A_cap = bestABFG(0)
B_cap = bestABFG(1)
F_cap = bestABFG(2)
G_cap = bestABFG(3)

xfit = xbest
yfit = ybest

; determine a,i,Omega,omega (using AIWW.PRO)

aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low

print, "a i Omega omega:", major, (180/!dpi)*inc, $
(180/!dpi)*W_cap, (180/!dpi)*w_low

EL = [period,Tperi,ecc,major,inc,W_cap,w_low]

; Determine errors in orbital elements
; set up matrices for error determination
; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

; set up covariance matrix and column matrix
alpha = dblarr(7,7)
beta = dblarr(7)
;For each element, enter 0 to hold element fixed, 1 to vary
elfix = [1.0,1.0,1.0,1.0,1.0,1.0,1.0]
indfix = where(elfix eq 1.0, mfit)

;calc_err, EL, time, theta, rho, dtheta, drho, alpha, beta
calc_deriv_vb, EL, elfix, mfit, time, theta, rho, dtheta, drho, theta_f, rho_f, alpha, beta

;print,"covariance matrix and right-hand side vector:"
;print,"alpha"
;print,alpha
;print,"beta"
;print,beta	

invmat = invert(alpha, stat, /double)

;print,"stat (0 successful):",stat

;print,"invmat"
;print,invmat

;determine errors:

ELerr = dblarr(7)

for i=0,6 do ELerr(i) = sqrt(chiredmin)*sqrt(invmat(i,i))

print, "P T e a i Omega omega"
print,EL(0),El(1),EL(2),EL(3),180*EL(4)/!dpi,180*EL(5)/!dpi,180*EL(6)/!dpi
print, "Errors determined from covariance matrix:"
print,Elerr(0),Elerr(1),ELerr(2),ELerr(3),180*ELerr(4)/!dpi, $
	180*ELerr(5)/!dpi,180*ELerr(6)/!dpi

; perform gauss-jordan elimination to invert alpha and find solution vector B

gaussj, alpha, beta

;print,"after gauss-jordan elimination:"
;print,"alpha"
;print,alpha
;print,"beta"
;print,beta	

; print O-C residuals (Observed - Calculated)

xOC = xarr - xfit
yOC = yarr - yfit

openw,2,'xyresiduals'
printf,2,"O-C residuals (data - fit)"
printf,2,"date     dx       xO-C   xOC/dx dy      yO-C   yOC/dy"
;printf,2,"date   dx   xdata - xfit   xOC/dx   dy   ydata - yfit   yOC/dy"
for i=0, num-1 do printf,2,format='(f8.3,f7.3,f8.3,f7.3,f7.3,f8.3,f7.3)', $
	time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
close,2

;for i=0, num-1 do print, "x",xarr(i)," dx",dx(i)," y",yarr(i)," dy",dy(i)

;plot data and fitted points

;print, "Do you want to save plot as a postscript file?  (Type yes or no):"
;answer = ""
;read,answer
;; Set plotting device to ps if answer is yes
;if (answer eq "yes") then begin
;	mydevice=!d.name
;	set_plot, 'ps'
;	device, /Color, Bits_per_pixel=8, filename='temp.ps' 
;endif
;
;plot, xarr, yarr, psym=4, XSTYLE=1, YSTYLE=1, ISOTROPIC=1, $
;	xrange=[-50,+150], yrange=[-150,+50]
;oplot, xfit, yfit, psym=5 
;
;; set plotting device back to terminal if previously saving to ps file
;if (answer eq "yes") then begin 
;	device, /close
;	set_plot, mydevice
;endif
 
end

