pro calc_deriv_SB2, ELarr, elfix, mfit, time, V1_d, V2_d, err_V1, err_V2, V1_f, V2_f, chimat, colmat

; Notes for future reference:
; Equivalent to the function of calc_err for the VB case
; For doing a joint fit - treat K1, K2, a("), i separately 
;                       - rename to calc_deriv_SBj  
; -endnotes-

; Set up matrices for error determination of orbital elements
; (derivatives)
;
; Add option of fixing values of orbital elements
;
; INPUT:
;	ELarr: array of orbital elements [P,T,e,K1,K2,omega,Vsys]
;       elfix: 7-element array: 0 if value to be held fixed
;                               1 if value to be free
;       mfit: number of parameters for which to solve (number of 1's in elfix)
;	Ei: eccentric anomalies
;	time: time of observation
;	V1_d: array containing velocity measurements for primary (in km/s) 
;	V2_d: array containing velocity measurements for secondary (in km/s)
;	err_V1: error in V1_d
;	err_V2: error in V2_d
;
; OUTPUT:
;       V1_f: fit-value of primary velocity
;       V2_f: fit- value of secondary velocity
;	chimat: 7x7 covariance matrix
;	colmat: column matrix containing differences between data and fit
;
; Calls the following routines:
;     calc_Ei.pro
;     solve_trans.pro
;
; Equations:  
; Initial estimates (P0,T0,e0,K1|0,K2|0,omega0,Vsys) and  
; a set of observations (ti,V1i,V2i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axes (a1sini, a2sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
;       a2sini = (P/2pi)*K2*(1 - e^2)^1/2
; The fitted velocities (V1_i, V2_i) are determined from the orbital elements:
;       V1 =  K1*[e*cos(omega) + cos(nu + omega)] + Vsys
;       V2 = -K2*[e*cos(omega) + cos(nu + omega)] + Vsys
;
; Procedure:
; Minimize chi squared between measured velocities and fitted velocities:
;	chi^2 = sum[(V1data - V1fit)^2/sigmaV1^2 
;		    + (V2data - V2fit)^2/sigmaV2^2]
; where the sums run over all data points
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,K1,K2,omega,Vsys) and set to zero. Solve for the value of
; the orbital elements that minimize chi^2.
;
; Since the equations for V1 and V2 are non-linear, cannot solve for 
; the orbital elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (V1fit,V2fit) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) + dx/dw|(w-w0)
;	       + dx/d(K1)|(K1 - K1|0) + dx/d(K2)|(K2 - K2|0)
;              + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(V1,V2) and the | represents the value evaluated for the
; initial estimates of the orbital elements. Now, V1fit and V2fit 
; are linear in (P,T,e,K1,K2,w,Vsys), so the partial derivatives
; can easily be taken and solved by setting up a matrix and using 
; Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;       K2:      velocity amplitude of secondary (in km/s) (K2)
;	inc:	 inclination (i)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V2_d:    velocity of secondary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;	V2_f:    V2 - fit (determined from orbital elements for time of obs.)
;       Vsys:    systemic velocity
;
; 23 June 2005: Begin writing program
; 7 July 2005: Fixed non-convergence problems 
;    + do not remove P wrap around effects from dV/dP
;      leave (t-T)/P in full form; net effect is that adjustments smaller
;    + fixed primary/secondary indices in setting dV1/dVsys and dV2/dVsys
; 31 May 2006: Save unmodified program as version 1
; 31 May 2006: Fit SB2 parameters in terms of K1 and K2
;              (as opposed to a1sini and a2sini)

; Define orbital element parameters

period = ELarr(0)
Tperi = ELarr(1)
ecc = ELarr(2)
K1 = ELarr(3)
K2 = ELarr(4)
omega = ELarr(5)
Vsys = ELarr(6)

EL = dblarr(mfit)
i=0
for k=0, mfit-1 do begin
	if (elfix(k) ne 0) then EL(k) = ELarr(i)
	i = i+1
endfor

num = n_elements(time)

; Determine the eccentric anomalies Ei: 
; (Ei: array to the eccentric anomaly)

calc_Ei, time, period, Tperi, ecc, Ei

; Determine true anomalies:

nu = 2*atan(sqrt((1+ecc)/(1-ecc))*tan(Ei/2.0))

;print,'Ei',Ei
;print,'nu',nu

; check sign of true anomaly

cosnu = cos(nu)
sinnu = sin(nu)

actualcosnu = (cos(Ei) - ecc)/(1 - ecc*cos(Ei))
actualsinnu = sqrt(1 -ecc^2)*sin(Ei)/(1-ecc*cos(Ei))

;print,'sinnnu actualsinnu cosnu actualcosnu'

for i=0, num-1 do begin
;    print,nu(i)*180.0/!dpi
;    print, sinnu(i), actualsinnu(i), cosnu(i), actualcosnu(i)
    if (cosnu(i)/abs(cosnu(i)) ne actualcosnu(i)/abs(actualcosnu(i))) and $
       (sinnu(i)/abs(sinnu(i)) ne actualsinnu(i)/abs(actualsinnu(i))) then $
       nu(i) = nu(i) + !dpi
;    print,sin(nu(i)), actualsinnu(i), cos(nu(i)),actualcosnu(i)
;    Strange!  It seems that nu is ALWAYS in the proper quadrant ???
endfor

; Determine (V1_f, V2_f) velocities from orbital elements and times of 
; observations

V1_f = K1*(ecc*cos(omega) + cos(nu + omega))

V2_f = -K2*(ecc*cos(omega) + cos(nu + omega))

;print,'V1_f',V1_f
;print,'V2_f',V2_f

; Calculate derivatives evaluated at initial conditions 
;	dx/dEL = (dx/dP,dx/dT,dx/de,dx/dK1,dx/dK2,dx/dw.dx/dVsys)
;	dy/dEL = (dy/dP,dy/dT,dy/de,dy/dK2,dy/dK2,dy/dw,dy/dVsys)
; where (x,y) -> (V1,V2) and
; dEL = (P,T,e,K1,K2,omega,Vsys)

;initialize derivative arrays 

dV1_del = dblarr(mfit,num)	;7: number of orbital elements
dV2_del = dblarr(mfit,num)	;num: number of data points

chimat = dblarr(mfit,mfit) 	;matrix for minimizing chi^2
colmat = dblarr(mfit)		;column matrix for minimizing chi^2
delta_el = dblarr(mfit)		;solution to matrix equation

timediff=dblarr(num)

for i=0, num-1 do begin

	; reduce time difference to same epoch
	; (fraction of period covered at time t)

	fracP = (time(i) - Tperi)/period - fix((time(i) - Tperi)/period)

	;(remove effects of positive and negative time -> arbitrary zero-point
	; for Tperi...... fracP goes from 0.0 to 1.0)

;	if (fracP lt 0) then fracP = 1.0d + fracP	

	timediff(i) = fracP

endfor

;print,'dEi_dP = -2*!dpi*(time - Tperi)/period^2/(1.0 - ecc*cos(Ei))'
;print,-2*!dpi*(time - Tperi)/period^2/(1.0 - ecc*cos(Ei))
;print,'dEi_dP = -2*!dpi*timediff/period/(1.0 - ecc*cos(Ei))'
;print,-2*!dpi*timediff/period/(1.0 - ecc*cos(Ei))
;print,'dEi_dP = -(Ei - ecc*sin(Ei))/(1.0 - ecc*cos(Ei))/period'
;print,-(Ei - ecc*sin(Ei))/(1.0 - ecc*cos(Ei))/period
;print, ' '


;gamma = 2*!dpi/period*sin(nu + omega)*sqrt(1.0 - ecc^2)/(1.0 - ecc*cos(Ei))^2
;
;print,'gamma',gamma
;print,'gamma*K1',gamma*K1
;print,'gamma*K2',gamma*K2

;dEi_dP = -2*!dpi*timediff/period/(1.0 - ecc*cos(Ei))
dEi_dP = -2*!dpi*(time - Tperi)/period^2/(1.0 - ecc*cos(Ei))
;dEi_dP = -(Ei - ecc*sin(Ei))/(1.0 - ecc*cos(Ei))/period
dEi_dT = -2*!dpi/period/(1.0 - ecc*cos(Ei))
dEi_de = sin(Ei)/(1.0 - ecc*cos(Ei))

dnu_dP = sqrt(1.0 - ecc^2)/(1.0 - ecc*cos(Ei))*dEi_dP
dnu_dT = sqrt(1.0 - ecc^2)/(1.0 - ecc*cos(Ei))*dEi_dT
dnu_de =((1.0 - ecc^2)*dEi_de + sin(Ei))/sqrt(1.0 - ecc^2)/(1.0 - ecc*cos(Ei))

dV1_dP = -K1*sin(nu + omega)*dnu_dP
dV1_dT = -K1*sin(nu + omega)*dnu_dT
dV1_de =  K1*(cos(omega) - sin(nu + omega)*dnu_de)
dV1_dw = -K1*(ecc*sin(omega) + sin(nu + omega))
dV1_dK1 = V1_f/K1
dV1_dK2 = dblarr(num)
dV1_dVsys = dblarr(num) + 1.0

dV2_dP =  K2*sin(nu + omega)*dnu_dP
dV2_dT =  K2*sin(nu + omega)*dnu_dT
dV2_de = -K2*(cos(omega) - sin(nu + omega)*dnu_de)
dV2_dw =  K2*(ecc*sin(omega) + sin(nu + omega))
dV2_dK2 = V2_f/K2
dV2_dK1 = dblarr(num)
dV2_dVsys = dblarr(num) + 1.0

;dV1_dEL = (dx_dP,dx_dT,dx_de,dx_dK1,dx_dK2,dx_dw,dx_dVsys)
;dV2_dEL = (dy_dP,dy_dT,dy_de,dy_dK1,dy_dK2,dy_dw,dy_dVsys)

k=0
if (elfix(0) ne 0) then begin
	dV1_del(k,*) = dV1_dP
	k=k+1
endif
if (elfix(1) ne 0) then begin
	dV1_del(k,*) = dV1_dT
	k=k+1
endif
if (elfix(2) ne 0) then begin
	dV1_del(k,*) = dV1_de
	k=k+1
endif
if (elfix(3) ne 0) then begin
	dV1_del(k,*) = dV1_dK1
	k=k+1
endif
if (elfix(4) ne 0) then begin
	dV1_del(k,*) = dV1_dK2
	k=k+1
endif
if (elfix(5) ne 0) then begin
	dV1_del(k,*) = dV1_dw
	k=k+1
endif
if (elfix(6) ne 0) then begin
	dV1_del(k,*) = dV1_dVsys
	k=k+1
endif

k=0
if (elfix(0) ne 0) then begin
	dV2_del(k,*) = dV2_dP
	k=k+1
endif
if (elfix(1) ne 0) then begin
	dV2_del(k,*) = dV2_dT
	k=k+1
endif
if (elfix(2) ne 0) then begin
	dV2_del(k,*) = dV2_de
 	k=k+1
endif
if (elfix(3) ne 0) then begin
	dV2_del(k,*) = dV2_dK1
	k=k+1
endif
if (elfix(4) ne 0) then begin
	dV2_del(k,*) = dV2_dK2
	k=k+1
endif
if (elfix(5) ne 0) then begin
	dV2_del(k,*) = dV2_dw
	k=k+1
endif
if (elfix(6) ne 0) then begin
	dV2_del(k,*) = dV2_dVsys
	k=k+1
endif

; Set up matrix for minimizing chi squared
; Set up column matrix too
; (col) = (chimat)(delta_el)

diff_V1 = dblarr(num)	  ; array for [V1(data) - V1(fit)]
diff_V2 = dblarr(num)     ; array for [V2(data) - V2(fit)]

; correct velocities for Vsys
V1_f = V1_f + Vsys
V2_f = V2_f + Vsys

;print,'Vsys',Vsys
;print,'Vsys included:'
;print,'V1_f',V1_f
;print,'V2_f',V2_f
;print,'-V1_f/P',-V1_f/period
;print,'-V2_f/P',-V2_f/period
;
;print,'V1_d',V1_d
;print,'V2_d',V2_d
;
;print,'err_V1',err_V1
;print,'err_V2',err_V2

for k=0, num-1 do begin

	; weight derivative for each data point by corresponding 
	; measurement error

	diff_V1(k) = (V1_d(k)-V1_f(k))
	diff_V2(k) = (V2_d(k)-V2_f(k))

	; the ## operator performs typical matrix multiplication
	; the * multiplies individual elements (no summing)

	for i=0, mfit-1 do begin

		for j=0, mfit-1 do begin

			chimat(i,j) = chimat(i,j) $
				+ dV1_del(i,k)*dV1_del(j,k)/err_V1(k)^2 $
				+ dV2_del(i,k)*dV2_del(j,k)/err_V2(k)^2
		endfor

		colmat(i) = colmat(i) + diff_V1(k)*dV1_del(i,k)/err_V1(k)^2 $
			    + diff_V2(k)*dV2_del(i,k)/err_V2(k)^2
	endfor		
endfor

;print,'diff_V1',diff_V1
;print,'diff_V2',diff_V2
;
;print,'dV1_del',dV1_del
;print,'dV2_del',dV2_del
;
;print,'colmat',colmat
;print,'chimat',chimat

;stop,'Type .cont to continue'

end
